c                                                                1/15/81
c***********************************************************************
c  odrv -- driver for sparse matrix reordering routines
c***********************************************************************
        subroutine  odrv
     *     (n, ia,ja,a, p,ip, nsp,isp, path, flag)
c
c  description
c
c    odrv finds a minimum degree ordering of the rows and columns of a
c    symmetric matrix m stored in (ia,ja,a) format (see below).  for the
c    reordered matrix, the work and storage required to perform gaussian
c    elimination is (usually) significantly less.
c
c    if only the nonzero entries in the upper triangle of m are being
c    stored, then odrv symmetrically reorders (ia,ja,a), (optionally)
c    with the diagonal entries placed first in each row.  this is to
c    ensure that if m(i,j) will be in the upper triangle of m with
c    respect to the new ordering, then m(i,j) is stored in row i (and
c    thus m(j,i) is not stored);  whereas if m(i,j) will be in the
c    strict lower triangle of m, then m(j,i) is stored in row j (and
c    thus m(i,j) is not stored).
c
c
c  storage of sparse matrices
c
c    the nonzero entries of the matrix m are stored row-by-row in the
c    array a.  to identify the individual nonzero entries in each row,
c    we need to know in which column each entry lies.  these column
c    indices are stored in the array ja;  i.e., if  a(k) = m(i,j),  then
c    ja(k) = j.  to identify the individual rows, we need to know where
c    each row starts.  these row pointers are stored in the array ia;
c    i.e., if m(i,j) is the first nonzero entry (stored) in the i-th row
c    and  a(k) = m(i,j),  then  ia(i) = k.  moreover, ia(n+1) points to
c    the first location following the last element in the last row.
c    thus, the number of entries in the i-th row is  ia(i+1) - ia(i),
c    the nonzero entries in the i-th row are stored consecutively in
c
c            a(ia(i)),  a(ia(i)+1),  ..., a(ia(i+1)-1),
c
c    and the corresponding column indices are stored consecutively in
c
c            ja(ia(i)), ja(ia(i)+1), ..., ja(ia(i+1)-1).
c
c    since the coefficient matrix is symmetric, only the nonzero entries
c    in the upper triangle need be stored.  for example, the matrix
c
c             ( 1  0  2  3  0 )
c             ( 0  4  0  0  0 )
c         m = ( 2  0  5  6  0 )
c             ( 3  0  6  7  8 )
c             ( 0  0  0  8  9 )
c
c    could be stored as
c
c            \ 1  2  3  4  5  6  7  8  9 10 11 12 13
c         ---+--------------------------------------
c         ia \ 1  4  5  8 12 14
c         ja \ 1  3  4  2  1  3  4  1  3  4  5  4  5
c          a \ 1  2  3  4  2  5  6  3  6  7  8  8  9
c
c    or (symmetrically) as
c
c            \ 1  2  3  4  5  6  7  8  9
c         ---+--------------------------
c         ia \ 1  4  5  7  9 10
c         ja \ 1  3  4  2  3  4  4  5  5
c          a \ 1  2  3  4  5  6  7  8  9          .
c
c
c  parameters
c
c    n    - order of the matrix
c
c    ia   - integer one-dimensional array containing pointers to delimit
c           rows in ja and a;  dimension = n+1
c
c    ja   - integer one-dimensional array containing the column indices
c           corresponding to the elements of a;  dimension = number of
c           nonzero entries in (the upper triangle of) m
c
c    a    - real one-dimensional array containing the nonzero entries in
c           (the upper triangle of) m, stored by rows;  dimension =
c           number of nonzero entries in (the upper triangle of) m
c
c    p    - integer one-dimensional array used to return the permutation
c           of the rows and columns of m corresponding to the minimum
c           degree ordering;  dimension = n
c
c    ip   - integer one-dimensional array used to return the inverse of
c           the permutation returned in p;  dimension = n
c
c    nsp  - declared dimension of the one-dimensional array isp;  nsp
c           must be at least  3n+4k,  where k is the number of nonzeroes
c           in the strict upper triangle of m
c
c    isp  - integer one-dimensional array used for working storage;
c           dimension = nsp
c
c    path - integer path specification;  values and their meanings are -
c             1  find minimum degree ordering only
c             2  find minimum degree ordering and reorder symmetrically
c                  stored matrix (used when only the nonzero entries in
c                  the upper triangle of m are being stored)
c             3  reorder symmetrically stored matrix as specified by
c                  input permutation (used when an ordering has already
c                  been determined and only the nonzero entries in the
c                  upper triangle of m are being stored)
c             4  same as 2 but put diagonal entries at start of each row
c             5  same as 3 but put diagonal entries at start of each row
c
c    flag - integer error flag;  values and their meanings are -
c               0    no errors detected
c              9n+k  insufficient storage in md
c             10n+1  insufficient storage in odrv
c             11n+1  illegal path specification
c
c
c  conversion from real to double precision
c
c    change the real declarations in odrv and sro to double precision
c    declarations.
c
c-----------------------------------------------------------------------
c
        integer  ia(1), ja(1),  p(1), ip(1),  isp(1),  path,  flag,
     *     v, l, head,  tmp, q
        real  a(1)
c...    double precision  a(1)
        logical  dflag
c
c----initialize error flag and validate path specification
        flag = 0
        if (path.lt.1 .or. 5.lt.path)  go to 111
c
c----allocate storage and find minimum degree ordering
        if ((path-1) * (path-2) * (path-4) .ne. 0)  go to 1
          max = (nsp-n)/2
          v    = 1
          l    = v     +  max
          head = l     +  max
          next = head  +  n
          if (max.lt.n)  go to 110
c
          call  md
     *       (n, ia,ja, max,isp(v),isp(l), isp(head),p,ip, isp(v), flag)
          if (flag.ne.0)  go to 100
c
c----allocate storage and symmetrically reorder matrix
   1    if ((path-2) * (path-3) * (path-4) * (path-5) .ne. 0)  go to 2
          tmp = (nsp+1) -      n
          q   = tmp     - (ia(n+1)-1)
          if (q.lt.1)  go to 110
c
          dflag = path.eq.4 .or. path.eq.5
          call sro
     *       (n,  ip,  ia, ja, a,  isp(tmp),  isp(q),  dflag)
c
   2    return
c
c ** error -- error detected in md
 100    return
c ** error -- insufficient storage
 110    flag = 10*n + 1
        return
c ** error -- illegal path specified
 111    flag = 11*n + 1
        return
        end
c***********************************************************************
c***********************************************************************
c  md -- minimum degree algorithm (based on element model)
c***********************************************************************
        subroutine  md
     *     (n, ia,ja, max, v,l, head,last,next, mark, flag)
c
c  description
c
c    md finds a minimum degree ordering of the rows and columns of a
c    symmetric matrix m stored in (ia,ja,a) format.
c
c
c  additional parameters
c
c    max  - declared dimension of the one-dimensional arrays v and l;
c           max must be at least  n+2k,  where k is the number of
c           nonzeroes in the strict upper triangle of m
c
c    v    - integer one-dimensional work array;  dimension = max
c
c    l    - integer one-dimensional work array;  dimension = max
c
c    head - integer one-dimensional work array;  dimension = n
c
c    last - integer one-dimensional array used to return the permutation
c           of the rows and columns of m corresponding to the minimum
c           degree ordering;  dimension = n
c
c    next - integer one-dimensional array used to return the inverse of
c           the permutation returned in last;  dimension = n
c
c    mark - integer one-dimensional work array (may be the same as v);
c           dimension = n
c
c    flag - integer error flag;  values and their meanings are -
c             0      no errors detected
c             11n+1  insufficient storage in md
c
c
c  definitions of internal parameters
c
c    ---------+---------------------------------------------------------
c    v(s)     \ value field of list entry
c    ---------+---------------------------------------------------------
c    l(s)     \ link field of list entry  (0 => end of list)
c    ---------+---------------------------------------------------------
c    l(vi)    \ pointer to element list of uneliminated vertex vi
c    ---------+---------------------------------------------------------
c    l(ej)    \ pointer to boundary list of active element ej
c    ---------+---------------------------------------------------------
c    head(d)  \ vj => vj head of d-list d
c             \  0 => no vertex in d-list d
c
c
c             \                  vi uneliminated vertex
c             \          vi in ek           \       vi not in ek
c    ---------+-----------------------------+---------------------------
c    next(vi) \ undefined but nonnegative   \ vj => vj next in d-list
c             \                             \  0 => vi tail of d-list
c    ---------+-----------------------------+---------------------------
c    last(vi) \ (not set until mdp)         \ -d => vi head of d-list d
c             \-vk => compute degree        \ vj => vj last in d-list
c             \ ej => vi prototype of ej    \  0 => vi not in any d-list
c             \  0 => do not compute degree \
c    ---------+-----------------------------+---------------------------
c    mark(vi) \ mark(vk)                    \ nonnegative tag < mark(vk)
c
c
c             \                   vi eliminated vertex
c             \      ei active element      \           otherwise
c    ---------+-----------------------------+---------------------------
c    next(vi) \ -j => vi was j-th vertex    \ -j => vi was j-th vertex
c             \       to be eliminated      \       to be eliminated
c    ---------+-----------------------------+---------------------------
c    last(vi) \  m => size of ei = m        \ undefined
c    ---------+-----------------------------+---------------------------
c    mark(vi) \ -m => overlap count of ei   \ undefined
c             \       with ek = m           \
c             \ otherwise nonnegative tag   \
c             \       < mark(vk)            \
c
c-----------------------------------------------------------------------
c
        integer  ia(1), ja(1),  v(1), l(1),  head(1), last(1), next(1),
     *     mark(1),  flag,  tag, dmin, vk,ek, tail
        equivalence  (vk,ek)
c
c----initialization
        tag = 0
        call  mdi
     *     (n, ia,ja, max,v,l, head,last,next, mark,tag, flag)
        if (flag.ne.0)  return
c
        k = 0
        dmin = 1
c
c----while  k < n  do
   1    if (k.ge.n)  go to 4
c
c------search for vertex of minimum degree
   2      if (head(dmin).gt.0)  go to 3
            dmin = dmin + 1
            go to 2
c
c------remove vertex vk of minimum degree from degree list
   3      vk = head(dmin)
          head(dmin) = next(vk)
          if (head(dmin).gt.0)  last(head(dmin)) = -dmin
c
c------number vertex vk, adjust tag, and tag vk
          k = k+1
          next(vk) = -k
          last(ek) = dmin - 1
          tag = tag + last(ek)
          mark(vk) = tag
c
c------form element ek from uneliminated neighbors of vk
          call  mdm
     *       (vk,tail, v,l, last,next, mark)
c
c------purge inactive elements and do mass elimination
          call  mdp
     *       (k,ek,tail, v,l, head,last,next, mark)
c
c------update degrees of uneliminated vertices in ek
          call  mdu
     *       (ek,dmin, v,l, head,last,next, mark)
c
          go to 1
c
c----generate inverse permutation from permutation
   4    do 5 k=1,n
          next(k) = -next(k)
   5      last(next(k)) = k
c
        return
        end
c
c***********************************************************************
c  mdi -- initialization
c***********************************************************************
        subroutine  mdi
     *     (n, ia,ja, max,v,l, head,last,next, mark,tag, flag)
        integer  ia(1), ja(1),  v(1), l(1),  head(1), last(1), next(1),
     *     mark(1), tag,  flag,  sfs, vi,dvi, vj
c
c----initialize degrees, element lists, and degree lists
        do 1 vi=1,n
          mark(vi) = 1
          l(vi) = 0
   1      head(vi) = 0
        sfs = n+1
c
c----create nonzero structure
c----for each nonzero entry a(vi,vj) in strict upper triangle
        do 3 vi=1,n
          jmin = ia(vi)
          jmax = ia(vi+1) - 1
          if (jmin.gt.jmax)  go to 3
          do 2 j=jmin,jmax
            vj = ja(j)
            if (vi.ge.vj)  go to 2
              if (sfs.ge.max)  go to 101
c
c------enter vj in element list for vi
              mark(vi) = mark(vi) + 1
              v(sfs) = vj
              l(sfs) = l(vi)
              l(vi) = sfs
              sfs = sfs+1
c
c------enter vi in element list for vj
              mark(vj) = mark(vj) + 1
              v(sfs) = vi
              l(sfs) = l(vj)
              l(vj) = sfs
              sfs = sfs+1
   2        continue
   3      continue
c
c----create degree lists and initialize mark vector
        do 4 vi=1,n
          dvi = mark(vi)
          next(vi) = head(dvi)
          head(dvi) = vi
          last(vi) = -dvi
          if (next(vi).gt.0)  last(next(vi)) = vi
   4      mark(vi) = tag
c
        return
c
c ** error -- insufficient storage
 101    flag = 9*n + vi
        return
        end
c
c***********************************************************************
c  mdm -- form element from uneliminated neighbors of vk
c***********************************************************************
        subroutine  mdm
     *     (vk,tail, v,l, last,next, mark)
        integer  vk, tail,  v(1), l(1),   last(1), next(1),   mark(1),
     *     tag, s,ls,vs,es, b,lb,vb, blp,blpmax
        equivalence  (vs, es)
c
c----initialize tag and list of uneliminated neighbors
        tag = mark(vk)
        tail = vk
c
c----for each vertex/element vs/es in element list of vk
        ls = l(vk)
   1    s = ls
        if (s.eq.0)  go to 5
          ls = l(s)
          vs = v(s)
          if (next(vs).lt.0)  go to 2
c
c------if vs is uneliminated vertex, then tag and append to list of
c------uneliminated neighbors
            mark(vs) = tag
            l(tail) = s
            tail = s
            go to 4
c
c------if es is active element, then ...
c--------for each vertex vb in boundary list of element es
   2        lb = l(es)
            blpmax = last(es)
            do 3 blp=1,blpmax
              b = lb
              lb = l(b)
              vb = v(b)
c
c----------if vb is untagged vertex, then tag and append to list of
c----------uneliminated neighbors
              if (mark(vb).ge.tag)  go to 3
                mark(vb) = tag
                l(tail) = b
                tail = b
   3          continue
c
c--------mark es inactive
            mark(es) = tag
c
   4      go to 1
c
c----terminate list of uneliminated neighbors
   5    l(tail) = 0
c
        return
        end
c
c***********************************************************************
c  mdp -- purge inactive elements and do mass elimination
c***********************************************************************
        subroutine  mdp
     *     (k,ek,tail, v,l, head,last,next, mark)
        integer  ek, tail,  v(1), l(1),  head(1), last(1), next(1),
     *     mark(1),  tag, free, li,vi,lvi,evi, s,ls,es, ilp,ilpmax
c
c----initialize tag
        tag = mark(ek)
c
c----for each vertex vi in ek
        li = ek
        ilpmax = last(ek)
        if (ilpmax.le.0)  go to 12
        do 11 ilp=1,ilpmax
          i = li
          li = l(i)
          vi = v(li)
c
c------remove vi from degree list
          if (last(vi).eq.0)  go to 3
            if (last(vi).gt.0)  go to 1
              head(-last(vi)) = next(vi)
              go to 2
   1          next(last(vi)) = next(vi)
   2        if (next(vi).gt.0)  last(next(vi)) = last(vi)
c
c------remove inactive items from element list of vi
   3      ls = vi
   4      s = ls
          ls = l(s)
          if (ls.eq.0)  go to 6
            es = v(ls)
            if (mark(es).lt.tag)  go to 5
              free = ls
              l(s) = l(ls)
              ls = s
   5        go to 4
c
c------if vi is interior vertex, then remove from list and eliminate
   6      lvi = l(vi)
          if (lvi.ne.0)  go to 7
            l(i) = l(li)
            li = i
c
            k = k+1
            next(vi) = -k
            last(ek) = last(ek) - 1
            go to 11
c
c------else ...
c--------classify vertex vi
   7        if (l(lvi).ne.0)  go to 9
              evi = v(lvi)
              if (next(evi).ge.0)  go to 9
                if (mark(evi).lt.0)  go to 8
c
c----------if vi is prototype vertex, then mark as such, initialize
c----------overlap count for corresponding element, and move vi to end
c----------of boundary list
                  last(vi) = evi
                  mark(evi) = -1
                  l(tail) = li
                  tail = li
                  l(i) = l(li)
                  li = i
                  go to 10
c
c----------else if vi is duplicate vertex, then mark as such and adjust
c----------overlap count for corresponding element
   8              last(vi) = 0
                  mark(evi) = mark(evi) - 1
                  go to 10
c
c----------else mark vi to compute degree
   9              last(vi) = -ek
c
c--------insert ek in element list of vi
  10        v(free) = ek
            l(free) = l(vi)
            l(vi) = free
  11      continue
c
c----terminate boundary list
  12    l(tail) = 0
c
        return
        end
c
c***********************************************************************
c  mdu -- update degrees of uneliminated vertices in ek
c***********************************************************************
        subroutine  mdu
     *     (ek,dmin, v,l, head,last,next, mark)
        integer  ek, dmin,  v(1), l(1),  head(1), last(1), next(1),
     *     mark(1),  tag, vi,evi,dvi, s,vs,es, b,vb, ilp,ilpmax,
     *     blp,blpmax
        equivalence  (vs, es)
c
c----initialize tag
        tag = mark(ek) - last(ek)
c
c----for each vertex vi in ek
        i = ek
        ilpmax = last(ek)
        if (ilpmax.le.0)  go to 11
        do 10 ilp=1,ilpmax
          i = l(i)
          vi = v(i)
          if (last(vi))  1, 10, 8
c
c------if vi neither prototype nor duplicate vertex, then merge elements
c------to compute degree
   1        tag = tag + 1
            dvi = last(ek)
c
c--------for each vertex/element vs/es in element list of vi
            s = l(vi)
   2        s = l(s)
            if (s.eq.0)  go to 9
              vs = v(s)
              if (next(vs).lt.0)  go to 3
c
c----------if vs is uneliminated vertex, then tag and adjust degree
                mark(vs) = tag
                dvi = dvi + 1
                go to 5
c
c----------if es is active element, then expand
c------------check for outmatched vertex
   3            if (mark(es).lt.0)  go to 6
c
c------------for each vertex vb in es
                b = es
                blpmax = last(es)
                do 4 blp=1,blpmax
                  b = l(b)
                  vb = v(b)
c
c--------------if vb is untagged, then tag and adjust degree
                  if (mark(vb).ge.tag)  go to 4
                    mark(vb) = tag
                    dvi = dvi + 1
   4              continue
c
   5          go to 2
c
c------else if vi is outmatched vertex, then adjust overlaps but do not
c------compute degree
   6        last(vi) = 0
            mark(es) = mark(es) - 1
   7        s = l(s)
            if (s.eq.0)  go to 10
              es = v(s)
              if (mark(es).lt.0)  mark(es) = mark(es) - 1
              go to 7
c
c------else if vi is prototype vertex, then calculate degree by
c------inclusion/exclusion and reset overlap count
   8        evi = last(vi)
            dvi = last(ek) + last(evi) + mark(evi)
            mark(evi) = 0
c
c------insert vi in appropriate degree list
   9      next(vi) = head(dvi)
          head(dvi) = vi
          last(vi) = -dvi
          if (next(vi).gt.0)  last(next(vi)) = vi
          if (dvi.lt.dmin)  dmin = dvi
c
  10      continue
c
  11    return
        end
c***********************************************************************
c***********************************************************************
c  sro -- symmetric reordering of sparse symmetric matrix
c***********************************************************************
        subroutine  sro
     *     (n, ip, ia,ja,a, q, r, dflag)
c
c  description
c
c    the nonzero entries of the matrix m are assumed to be stored
c    symmetrically in (ia,ja,a) format (i.e., not both m(i,j) and m(j,i)
c    are stored if i ne j).
c
c    sro does not rearrange the order of the rows, but does move
c    nonzeroes from one row to another to ensure that if m(i,j) will be
c    in the upper triangle of m with respect to the new ordering, then
c    m(i,j) is stored in row i (and thus m(j,i) is not stored);  whereas
c    if m(i,j) will be in the strict lower triangle of m, then m(j,i) is
c    stored in row j (and thus m(i,j) is not stored).
c
c
c  additional parameters
c
c    q     - integer one-dimensional work array;  dimension = n
c
c    r     - integer one-dimensional work array;  dimension = number of
c            nonzero entries in the upper triangle of m
c
c    dflag - logical variable;  if dflag = .true., then store nonzero
c            diagonal elements at the beginning of the row
c
c-----------------------------------------------------------------------
c
        integer  ip(1),  ia(1), ja(1),  q(1), r(1)
        real  a(1),  ak
c...    double precision  a(1),  ak
        logical  dflag
c
c
c--phase 1 -- find row in which to store each nonzero
c----initialize count of nonzeroes to be stored in each row
        do 1 i=1,n
  1       q(i) = 0
c
c----for each nonzero element a(j)
        do 3 i=1,n
          jmin = ia(i)
          jmax = ia(i+1) - 1
          if (jmin.gt.jmax)  go to 3
          do 2 j=jmin,jmax
c
c--------find row (=r(j)) and column (=ja(j)) in which to store a(j) ...
            k = ja(j)
            if (ip(k).lt.ip(i))  ja(j) = i
            if (ip(k).ge.ip(i))  k = i
            r(j) = k
c
c--------... and increment count of nonzeroes (=q(r(j)) in that row
  2         q(k) = q(k) + 1
  3       continue
c
c
c--phase 2 -- find new ia and permutation to apply to (ja,a)
c----determine pointers to delimit rows in permuted (ja,a)
        do 4 i=1,n
          ia(i+1) = ia(i) + q(i)
  4       q(i) = ia(i+1)
c
c----determine where each (ja(j),a(j)) is stored in permuted (ja,a)
c----for each nonzero element (in reverse order)
        ilast = 0
        jmin = ia(1)
        jmax = ia(n+1) - 1
        j = jmax
        do 6 jdummy=jmin,jmax
          i = r(j)
          if (.not.dflag .or. ja(j).ne.i .or. i.eq.ilast)  go to 5
c
c------if dflag, then put diagonal nonzero at beginning of row
            r(j) = ia(i)
            ilast = i
            go to 6
c
c------put (off-diagonal) nonzero in last unused location in row
  5         q(i) = q(i) - 1
            r(j) = q(i)
c
  6       j = j-1
c
c
c--phase 3 -- permute (ja,a) to upper triangular form (wrt new ordering)
        do 8 j=jmin,jmax
  7       if (r(j).eq.j)  go to 8
            k = r(j)
            r(j) = r(k)
            r(k) = k
            jak = ja(k)
            ja(k) = ja(j)
            ja(j) = jak
            ak = a(k)
            a(k) = a(j)
            a(j) = ak
            go to 7
  8       continue
c
        return
        end
c                                                                1/15/81
c***********************************************************************
c  sdrv -- driver for sparse symmetric positive definite matrix routines
c***********************************************************************
        subroutine sdrv
     *     (n, p,ip, ia,ja,a, b, z, nsp,isp,rsp,esp, path, flag)
c
c  description
c
c    sdrv solves sparse symmetric positive definite systems of linear
c    equations.  the solution process is divided into three stages --
c
c      ssf - the coefficient matrix m is factored symbolically to
c            determine where fillin will occur during the numeric
c            factorization.
c
c      snf - m is factored numerically into the product ut-d-u, where
c            d is diagonal and u is unit upper triangular.
c
c      sns - the linear system  mx = b  is solved using the ut-d-u
c            factorization from snf.
c
c    for several systems with the same coefficient matrix, ssf and snf
c    need be done only once (for the first system);  then sns is done
c    once for each additional right-hand side.  for several systems
c    whose coefficient matrices have the same nonzero structure, ssf
c    need be done only once (for the first system);  then snf and sns
c    are done once for each additional system.
c
c
c  storage of sparse matrices
c
c    the nonzero entries of the matrix m are stored row-by-row in the
c    array a.  to identify the individual nonzero entries in each row,
c    we need to know in which column each entry lies.  these column
c    indices are stored in the array ja;  i.e., if  a(k) = m(i,j),  then
c    ja(k) = j.  to identify the individual rows, we need to know where
c    each row starts.  these row pointers are stored in the array ia;
c    i.e., if m(i,j) is the first nonzero entry (stored) in the i-th row
c    and  a(k) = m(i,j),  then  ia(i) = k.  moreover, ia(n+1) points to
c    the first location following the last element in the last row.
c    thus, the number of entries in the i-th row is  ia(i+1) - ia(i),
c    the nonzero entries in the i-th row are stored consecutively in
c
c            a(ia(i)),  a(ia(i)+1),  ..., a(ia(i+1)-1),
c
c    and the corresponding column indices are stored consecutively in
c
c            ja(ia(i)), ja(ia(i)+1), ..., ja(ia(i+1)-1).
c
c    since the coefficient matrix is symmetric, only the nonzero entries
c    in the upper triangle need be stored, for example, the matrix
c
c             ( 1  0  2  3  0 )
c             ( 0  4  0  0  0 )
c         m = ( 2  0  5  6  0 )
c             ( 3  0  6  7  8 )
c             ( 0  0  0  8  9 )
c
c    could be stored as
c
c            \ 1  2  3  4  5  6  7  8  9 10 11 12 13
c         ---+--------------------------------------
c         ia \ 1  4  5  8 12 14
c         ja \ 1  3  4  2  1  3  4  1  3  4  5  4  5
c          a \ 1  2  3  4  2  5  6  3  6  7  8  8  9
c
c    or (symmetrically) as
c
c            \ 1  2  3  4  5  6  7  8  9
c         ---+--------------------------
c         ia \ 1  4  5  7  9 10
c         ja \ 1  3  4  2  3  4  4  5  5
c          a \ 1  2  3  4  5  6  7  8  9          .
c
c
c  reordering the rows and columns of m
c
c    a symmetric permutation of the rows and columns of the coefficient
c    matrix m (e.g., which reduces fillin or enhances numerical
c    stability) must be specified.  the solution z is returned in the
c    original order.
c
c    to specify the trivial ordering (i.e., the identity permutation),
c    set  p(i) = ip(i) = i,  i=1,...,n.  in this case, p and ip can be
c    the same array.
c
c    if a nontrivial ordering (i.e., not the identity permutation) is
c    specified and m is stored symmetrically (i.e., not both m(i,j) and
c    m(j,i) are stored for i ne j), then odrv should be called (with
c    path = 3 or 5) to symmetrically reorder (ia,ja,a) before calling
c    sdrv.  this is to ensure that if m(i,j) will be in the upper
c    triangle of m with respect to the new ordering, then m(i,j) is
c    stored in row i (and thus m(j,i) is not stored);  whereas if m(i,j)
c    will be in the strict lower triangle of m, then m(j,i) is stored in
c    row j (and thus m(i,j) is not stored).
c
c
c  parameters
c
c    n    - number of variables/equations
c
c    p    - integer one-dimensional array specifying a permutation of
c           the rows and columns of m;  dimension = n
c
c    ip   - integer one-dimensional array containing the inverse of the
c           permutation specified in p;  i.e., ip(p(i)) = i, i=1,...,n;
c           dimension = n
c
c    ia   - integer one-dimensional array containing pointers to delimit
c           rows in ja and a;  dimension = n+1
c
c    ja   - integer one-dimensional array containing the column indices
c           corresponding to the elements of a;  dimension = number of
c           nonzero entries in m stored
c
c    a    - real one-dimensional array containing the nonzero entries in
c           the coefficient matrix m, stored by rows;  dimension =
c           number of nonzero entries in m stored
c
c    b    - real one-dimensional array containing the right-hand side b;
c           b and z can be the same array;  dimension = n
c
c    z    - real one-dimensional array containing the solution x;  z and
c           b can be the same array;  dimension = n
c
c    nsp  - declared dimension of the one-dimensional arrays isp and
c           rsp;  nsp must be (substantially) larger than  3n+2k,  where
c           k = number of nonzero entries in the upper triangle of m
c
c    isp  - integer one-dimensional array used for working storage;  isp
c           and rsp should be equivalenced;  dimension = nsp
c
c    rsp  - real one-dimensional array used for working storage;  rsp
c           and isp should be equivalenced;  dimension = nsp
c
c    esp  - integer variable;  if sufficient storage was available to
c           perform the symbolic factorization (ssf), then esp is set to
c           the amount of excess storage provided (negative if
c           insufficient storage was available to perform the numeric
c           factorization (snf))
c
c    path - integer path specification;  values and their meanings are -
c             1  perform ssf, snf, and sns
c             2  perform snf and sns (isp/rsp is assumed to have been
c                  set up in an earlier call to sdrv (for ssf))
c             3  perform sns only (isp/rsp is assumed to have been set
c                  up in an earlier call to sdrv (for ssf and snf))
c             4  perform ssf
c             5  perform ssf and snf
c             6  perform snf only (isp/rsp is assumed to have been set
c                  up in an earlier call to sdrv (for ssf))
c
c    flag - integer error flag;  values and their meanings are -
c               0     no errors detected
c              2n+k   duplicate entry in a  --  row = k
c              6n+k   insufficient storage in ssf  --  row = k
c              7n+1   insufficient storage in snf
c              8n+k   zero pivot  --  row = k
c             10n+1   insufficient storage in sdrv
c             11n+1   illegal path specification
c
c
c  conversion from real to double precision
c
c    change the real declarations in sdrv, snf, and sns to double
c    precision declarations;  and change the value in the data statement
c    for the integer variable ratio (in sdrv) from 1 to 2.
c
c-----------------------------------------------------------------------
c
        integer  p(1), ip(1),  ia(1), ja(1),  isp(1), esp,  path,  flag,
     *     ratio,  q, mark, d, u, tmp,  umax
        real  a(1),  b(1),  z(1),  rsp(1)
        data  ratio/1/
c...    double precision  a(1),  b(1),  z(1),  rsp(1)
c...    data  ratio/2/
c
c----validate path specification
        if (path.lt.1 .or. 6.lt.path)  go to 111
c
c----allocate storage and factor m symbolically to determine fill-in
        iju   = 1
        iu    = iju     +  n
        jl    = iu      + n+1
        ju    = jl      +  n
        q     = (nsp+1) -  n
        mark  = q       -  n
        jumax = mark    - ju
c
        if ((path-1) * (path-4) * (path-5) .ne. 0)  go to 1
          if (jumax.le.0)  go to 110
          call ssf
     *       (n,  p, ip,  ia, ja,  isp(iju), isp(ju), isp(iu), jumax,
     *        isp(q), isp(mark), isp(jl), flag)
          if (flag.ne.0)  go to 100
c
c----allocate storage and factor m numerically
   1    il   = ju      + isp(iju+(n-1))
        tmp  = ((il-1)+(ratio-1)) / ratio  +  1
        d    = tmp     + n
        u    = d       + n
        umax = (nsp+1) - u
        esp  = umax    - (isp(iu+n)-1)
c
        if ((path-1) * (path-2) * (path-5) * (path-6) .ne. 0)  go to 2
          if (umax.le.0)  go to 110
          call snf
     *       (n,  p, ip,  ia, ja, a,
     *        rsp(d),  isp(iju), isp(ju), isp(iu), rsp(u), umax,
     *        isp(il),  isp(jl),  flag)
          if (flag.ne.0)  go to 100
c
c----solve system of linear equations  mx = b
   2    if ((path-1) * (path-2) * (path-3) .ne. 0)  go to 3
          if (umax.le.0)  go to 110
          call sns
     *       (n,  p,  rsp(d), isp(iju), isp(ju), isp(iu), rsp(u),  z, b,
     *        rsp(tmp))
c
   3    return
c
c ** error -- error detected in ssf, snf, or sns
 100    return
c ** error -- insufficient storage
 110    flag = 10*n + 1
        return
c ** error -- illegal path specification
 111    flag = 11*n + 1
        return
        end
c
c
c***********************************************************************
c  internal documentation for ssf, snf, and sns
c***********************************************************************
c
c  compressed storage of sparse matrices
c
c    the strict upper triangular portion of the matrix u is stored in
c    (ia,ja,a) format using the arrays iu, ju, and u, except that an
c    additional array iju is used to reduce the storage required for ju
c    by allowing some sequences of column indices to correspond to more
c    than one row.  for i < n, iju(i) is the index in ju of the first
c    entry for the i-th row;  iju(n) is the number of entries in ju.
c    thus, the number of entries in the i-th row is  iu(i+1) - iu(i),
c    the nonzero entries of the i-th row are stored consecutively in
c
c        u(iu(i)),   u(iu(i)+1),   ..., u(iu(i+1)-1),
c
c    and the corresponding column indices are stored consecutively in
c
c        ju(iju(i)), ju(iju(i)+1), ..., ju(iju(i)+iu(i+1)-iu(i)-1).
c
c    compression in ju occurs in two ways.  first, if a row i was merged
c    into row k, and the number of elements merged in from (the tail
c    portion of) row i is the same as the final length of row k, then
c    the kth row and the tail of row i are identical and iju(k) points
c    to the start of the tail.  second, if some tail portion of the
c    (k-1)st row is identical to the head of the kth row, then iju(k)
c    points to the start of that tail portion.  for example, the nonzero
c    structure of the strict upper triangular part of the matrix
c
c             ( d 0 0 0 x x x )
c             ( 0 d 0 x x 0 0 )
c             ( 0 0 d 0 x x 0 )
c         u = ( 0 0 0 d x x 0 )
c             ( 0 0 0 0 d x x )
c             ( 0 0 0 0 0 d x )
c             ( 0 0 0 0 0 0 d )
c
c    would be stored as
c
c             \ 1  2  3  4  5  6  7  8
c         ----+------------------------
c          iu \ 1  4  6  8 10 12 13 13
c          ju \ 5  6  7  4  5  6
c         iju \ 1  4  5  5  2  3  6           .
c
c    the diagonal entries of u are equal to one and are not stored.
c
c
c  additional parameters
c
c    d     - real one-dimensional array containing the reciprocals of
c            the diagonal entries of the matrix d;  dimension = n
c
c    iju   - integer one-dimensional array containing pointers to the
c            start of each row in ju;  dimension = n
c
c    iu    - integer one-dimensional array containing pointers to
c            delimit rows in u;  dimension = n+1
c
c    ju    - integer one-dimensional array containing the column indices
c            corresponding to the elements of u;  dimension = jumax
c
c    jumax - declared dimension of the one-dimensional array ju;  jumax
c            must be at least the size of u minus compression (iju(n)
c            after the call to ssf)
c
c    u     - real one-dimensional array containing the nonzero entries
c            in the strict upper triangle of u, stored by rows;
c            dimension = umax
c
c    umax  - declared dimension of the one-dimensional array u;  umax
c            must be at least the number of nonzero entries in the
c            strict upper triangle of m plus fillin (iu(n+1)-1 after the
c            call to ssf)
c
c
c***********************************************************************
c  ssf --  symbolic ut-d-u factorization of sparse symmetric matrix
c***********************************************************************
        subroutine  ssf
     *     (n, p,ip, ia,ja, iju,ju,iu,jumax, q, mark, jl, flag)
c
c  additional parameters
c
c    q     - integer one-dimensional work array;  dimension = n
c
c    mark  - integer one-dimensional work array;  dimension = n
c
c    jl    - integer one-dimensional work array;  dimension = n
c
c
c  definitions of internal parameters (during k-th stage of elimination)
c
c    q contains an ordered linked list representation of the nonzero
c      structure of the k-th row of u --
c        q(k) is the first column with a nonzero entry
c        q(i) is the next column with a nonzero entry after column i
c      in either case, q(i) = n+1 indicates the end of the list
c
c    jl contains lists of rows to be merged into uneliminated rows --
c        i ge k => jl(i) is the first row to be merged into row i
c        i lt k => jl(i) is the row following row i in some list of rows
c      in either case, jl(i) = 0 indicates the end of a list
c
c    mark(i) is the last row stored in ju for which u(mark(i),i) ne 0
c
c    jumin and juptr are the indices in ju of the first and last
c      elements in the last row saved in ju
c
c    luk is the number of nonzero entries in the k-th row
c
c-----------------------------------------------------------------------
c
        integer  p(1), ip(1),  ia(1), ja(1),  iju(1), ju(1), iu(1),
     *     q(1),  mark(1),  jl(1),  flag,  tag, vj, qm
        logical  clique
c
c----initialization
        jumin = 1
        juptr = 0
        iu(1) = 1
        do 1 k=1,n
          mark(k) = 0
   1      jl(k) = 0
c
c----for each row k
        do 18 k=1,n
          luk = 0
          q(k) = n+1
c
          tag = mark(k)
          clique = .false.
          if (jl(k).ne.0)  clique = jl(jl(k)).eq.0
c
c------initialize nonzero structure of k-th row to row p(k) of m
          jmin = ia(p(k))
          jmax = ia(p(k)+1) - 1
          if (jmin.gt.jmax)  go to 4
          do 3 j=jmin,jmax
            vj = ip(ja(j))
            if (vj.le.k)  go to 3
c
              qm = k
   2          m = qm
              qm = q(m)
              if (qm.lt.vj)  go to 2
              if (qm.eq.vj)  go to 102
                luk = luk+1
                q(m) = vj
                q(vj) = qm
                if (mark(vj).ne.tag)  clique = .false.
c
   3        continue
c
c------if exactly one row is to be merged into the k-th row and there is
c------a nonzero entry in every column in that row in which there is a
c------nonzero entry in row p(k) of m, then do not compute fill-in, just
c------use the column indices for the row which was to have been merged
   4      if (.not.clique)  go to 5
            iju(k) = iju(jl(k)) + 1
            luk = iu(jl(k)+1) - (iu(jl(k))+1)
            go to 17
c
c------modify nonzero structure of k-th row by computing fill-in
c------for each row i to be merged in
   5      lmax = 0
          iju(k) = juptr
c
          i = k
   6      i = jl(i)
          if (i.eq.0)  go to 10
c
c--------merge row i into k-th row
            lui = iu(i+1) - (iu(i)+1)
            jmin = iju(i) +  1
            jmax = iju(i) + lui
            qm = k
c
            do 8 j=jmin,jmax
              vj = ju(j)
   7          m = qm
              qm = q(m)
              if (qm.lt.vj)  go to 7
              if (qm.eq.vj)  go to 8
                luk = luk+1
                q(m) = vj
                q(vj) = qm
                qm = vj
   8          continue
c
c--------remember length and position in ju of longest row merged
            if (lui.le.lmax)  go to 9
              lmax = lui
              iju(k) = jmin
c
   9        go to 6
c
c------if the k-th row is the same length as the longest row merged,
c------then use the column indices for that row
  10      if (luk.eq.lmax)  go to 17
c
c------if the tail of the last row saved in ju is the same as the head
c------of the k-th row, then overlap the two sets of column indices --
c--------search last row saved for first nonzero entry in k-th row ...
            i = q(k)
            if (jumin.gt.juptr)  go to 12
            do 11 jmin=jumin,juptr
              if (ju(jmin)-i)  11, 13, 12
  11          continue
  12        go to 15
c
c--------... and then test whether tail matches head of k-th row
  13        iju(k) = jmin
            do 14 j=jmin,juptr
              if (ju(j).ne.i)  go to 15
              i = q(i)
              if (i.gt.n)  go to 17
  14          continue
            juptr = jmin - 1
c
c------save nonzero structure of k-th row in ju
  15      i = k
          jumin = juptr +  1
          juptr = juptr + luk
          if (juptr.gt.jumax)  go to 106
          do 16 j=jumin,juptr
            i = q(i)
            ju(j) = i
  16        mark(i) = k
          iju(k) = jumin
c
c------add k to row list for first nonzero element in k-th row
  17      if (luk.le.1)  go to 18
            i = ju(iju(k))
            jl(k) = jl(i)
            jl(i) = k
c
  18      iu(k+1) = iu(k) + luk
c
        flag = 0
        return
c
c ** error -- duplicate entry in a
 102    flag = 2*n + p(k)
        return
c ** error -- insufficient storage for ju
 106    flag = 6*n + k
        return
        end
c
c***********************************************************************
c  snf -- numerical ut-d-u factorization of sparse symmetric positive
c         definite matrix
c***********************************************************************
        subroutine  snf
     *     (n, p,ip, ia,ja,a, d, iju,ju,iu,u,umax, il, jl, flag)
c
c  additional parameters
c
c    il    - integer one-dimensional work array;  dimension = n
c
c    jl    - integer one-dimensional work array;  dimension = n
c
c
c  definitions of internal parameters (during k-th stage of elimination)
c
c    (d(i),i=k,n) contains the k-th row of u (expanded)
c
c    il(i) points to the first nonzero element in columns k,...,n of
c      row i of u
c
c    jl contains lists of rows to be added to uneliminated rows --
c      i ge k => jl(i) is the first row to be added to row i
c      i lt k => jl(i) is the row following row i in some list of rows
c      in either case, jl(i) = 0 indicates the end of a list
c
c-----------------------------------------------------------------------
c
        integer  p(1), ip(1),  ia(1), ja(1),  iju(1), ju(1), iu(1),
     *     umax,  il(1),  jl(1),  flag,  vj
        real  a(1),  d(1), u(1),  dk, ukidi
c...    double precision  a(1),  d(1), u(1),  dk, ukidi
c
c----check for sufficient storage for u
        if (iu(n+1)-1 .gt. umax)  go to 107
c
c----initialization
        do 1 k=1,n
          d(k) = 0
   1      jl(k) = 0
c
c----for each row k
        do 11 k=1,n
c
c------initialize k-th row with elements nonzero in row p(k) of m
   3      jmin = ia(p(k))
          jmax = ia(p(k)+1) - 1
          if (jmin.gt.jmax) go to 5
          do 4 j=jmin,jmax
            vj = ip(ja(j))
            if (k.le.vj)  d(vj) = a(j)
   4        continue
c
c------modify k-th row by adding in those rows i with u(i,k) ne 0
c------for each row i to be added in
   5      dk = d(k)
          i = jl(k)
   6      if (i.eq.0)  go to 9
            nexti = jl(i)
c
c--------compute multiplier and update diagonal element
            ili = il(i)
            ukidi = - u(ili) * d(i)
            dk = dk + ukidi * u(ili)
            u(ili) = ukidi
c
c--------add multiple of row i to k-th row ...
            jmin = ili     + 1
            jmax = iu(i+1) - 1
            if (jmin.gt.jmax)  go to 8
              mu = iju(i) - iu(i)
              do 7 j=jmin,jmax
   7            d(ju(mu+j)) = d(ju(mu+j)) + ukidi * u(j)
c
c--------... and add i to row list for next nonzero entry
              il(i) = jmin
              j = ju(mu+jmin)
              jl(i) = jl(j)
              jl(j) = i
c
   8        i = nexti
            go to 6
c
c------check for zero pivot and save diagonal element
   9      if (dk.eq.0)  go to 108
          d(k) = 1 / dk
c
c------save nonzero entries in k-th row of u ...
          jmin = iu(k)
          jmax = iu(k+1) - 1
          if (jmin.gt.jmax)  go to 11
            mu = iju(k) - jmin
            do 10 j=jmin,jmax
              jumuj = ju(mu+j)
              u(j) = d(jumuj)
  10          d(jumuj) = 0
c
c------... and add k to row list for first nonzero entry in k-th row
            il(k) = jmin
            i = ju(mu+jmin)
            jl(k) = jl(i)
            jl(i) = k
  11      continue
c
        flag = 0
        return
c
c ** error -- insufficient storage for u
 107    flag = 7*n + 1
        return
c ** error -- zero pivot
 108    flag = 8*n + k
        return
        end
c
c***********************************************************************
c  sns -- solution of sparse symmetric positive definite system of
c         linear equations  mx = b  given ut-d-u factorization of m
c***********************************************************************
        subroutine  sns
     *     (n, p, d, iju,ju,iu,u, z, b, tmp)
        integer  p(1),  iju(1), ju(1), iu(1)
        real  d(1), u(1),  z(1), b(1),  tmp(1),  tmpk, sum
c...    double precision  d(1), u(1),  z(1), b(1),  tmp(1),  tmpk, sum
c
c  additional parameters
c
c    tmp   - real one-dimensional work array;  dimension = n
c
c-----------------------------------------------------------------------
c
c----set tmp to permuted b
        do 1 k=1,n
   1      tmp(k) = b(p(k))
c
c----solve  ut d y = b  by forward substitution
        do 3 k=1,n
          tmpk = tmp(k)
          jmin = iu(k)
          jmax = iu(k+1) - 1
          if (jmin.gt.jmax)  go to 3
          mu = iju(k) - jmin
          do 2 j=jmin,jmax
   2        tmp(ju(mu+j)) = tmp(ju(mu+j)) + u(j) * tmpk
   3      tmp(k) = tmpk * d(k)
c
c----solve  u x = y  by back substitution
        k = n
        do 6 i=1,n
          sum = tmp(k)
          jmin = iu(k)
          jmax = iu(k+1) - 1
          if (jmin.gt.jmax)  go to 5
          mu = iju(k) - jmin
          do 4 j=jmin,jmax
   4        sum = sum + u(j) * tmp(ju(mu+j))
   5      tmp(k) = sum
          z(p(k)) = sum
   6      k = k-1
c
        return
        end
