/*
*				spline.c
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*/
#include <stdio.h>
#include <cdecs.h>	/* includes stdio.h, string.h, ctype.h, math.h */

/*
*				fit_qspline():
*
*	Given a boundary function f(x), a<=x<=b, and a net of points
*	x[0], .., x[n], determines a function F(x) with the properties:
*
*	1.  F(x[i]) = f(x[i]),  i = 0,..,n
*
*	2.  F(x) is quadratic on [x[i],x[i+1]], i = 0,..,n-1
*
*	3.  F(x) is continuous on [x[0],x[n]]
*
*	4.  dF/dx(x) is continuous on [x[0],x[n]]
*
*	The conditions above do not quite determine a unique function F.
*	Rather they determine a one-paramater family of functions.
*
*	The function F() is determined uniquely by the extra condition:
*
*	5.  F(x) is linear on [x[0],x[1]].
*
*
*	Representation:
*
*	We represent the function F() in the form:
*
*		F(x) = a[i](x-x[i])**2 + b[i](x-x[i]) + c[i],  x[i]<=x<=x[i+1],
*
*	where the 3n coeffs a[i],b[i],c[i] ,0<=i<n, remain to be determined.
*
*	The routine below takes as input the function f(), the number of
*	intervals  n  and the net of points x[i], 0<=i<=n, and returns
*	the coeffs a[i],b[i],c[i] in the supplied arrays a[],b[],c[].
*		
*/


/*ARGSUSED*/
EXPORT void fit_qspline(
	float	(*f)(float),
	int	n,
	float	*x,
	float	*a,
	float	*b,
	float	*c,
	float	*d)	/* d is ignored */
{
	int i;
	float h;

	for (i=0; i<=n; i++) c[i] = (*f)(x[i]);	/* Note: case i=n too */

	a[0] = 0.;
	b[0] = (c[1]-c[0])/(x[1]-x[0]);

	for (i=0; i<n; i++) {
		h = x[i+1]-x[i];
		if (i<n-1) b[i+1] = 2.*(c[i+1]-c[i])/h - b[i];
		a[i] = ((c[i+1]-c[i])/h - b[i])/h;
	}
}




/*
*				qspline():
*
*	Returns the value of the spline function  F() at a point X
*	where x[0] <= X <= x[n].   The coeffs n,x[],a[],b[],c[] are 
*	as defined in the previous subroutine.
*/

/*ARGSUSED*/
EXPORT float qspline(
	float	X,
	int	n,
	float	*x,
	float	*a,
	float	*b,
	float	*c,
	float	*d)	/* d is ignored */
{
	int i;
	float y;

	for (i=0; i<n; i++)
		if (X<=x[i+1]) break;	/* X is in ith interval */
	if (i==n) i--;

	y = X-x[i];
	return a[i]*y*y + b[i]*y + c[i];
}







/*
*				fit_cspline():
*
*	Determines a cubic polynomial F(x) which fits a given
*	boundary function f(x).The function F(x) is represented
*	as
*	   F(x) = a[i](x-x[i])**3 +b[i](x-x[i])**2 + c[i](x-x[i])
*	   + d[i], x[i]<=x<=x[i+1]
*
*
*
*/

EXPORT void fit_cspline(
       	float	(*f)(float),
       	int	n,
       	float	*x,
	float	*a,
	float	*b,
	float	*c,
	float	*d)
{
       	int i;
       	float h;

       	for (i=0;i<=n;i++) d[i] = (*f)(x[i]);

	h = x[1]-x[0];
       	a[0] = 0. ;
       	b[0] = 0. ;
       	c[0] = (d[1]-d[0])/h;

	for (i=1; i<n; i++) {
		b[i] = b[i-1] + 3*a[i-1]*h;
		c[i] = c[i-1] + 2*b[i-1]*h + 3*a[i-1]*h*h;
		h = x[i+1]-x[i];
		a[i] = (d[i+1]-d[i])/(h*h*h) - c[i]/(h*h) - b[i]/h;
	}
}






/*     
*				cspline():
*
*	Returns the value of the cubic spline function F() at a
* 	point X.The coeffs n,x[],a[],b[],c[],d[] are defined in
* 	previous subroutine.
*
*/

EXPORT float cspline(
       	float	X,
       	int	n,
       	float	*x,
	float	*a,
	float	*b,
	float	*c,
	float	*d)
{
       	int i;
       	float y;

       	for (i=0;i<n;i++)
	   	if (X<=x[i+1]) break;

       	if (i==n) i-- ;

       	y = X-x[i] ;

       	return a[i]*y*y*y +b[i]*y*y +c[i]*y +d[i];

}
