/*
*			gausselim.c
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*		Gauss Elimination Band Matrix Routines:
*
*	This is THE WORLDS CRUDEST SOLVER.   It does no pivoting or
*	checking for singular matrix and is extremely slow on a
*	sparse matrix.   It does however work on any invertible
*	matrix.   Use to verify other solution methods.
*	NOTE: Uses lots of storage.
*/


#include  <stdio.h>
#include <cdecs.h>	/* includes stdio.h, string.h, ctype.h, math.h */
#include  <vmalloc.h>




/*
*				gauss_eliminate():
*				band_gauss_eliminate();
*
*	Straight Gauss elimination Solver.   Solves the matrix quation:
*				a*x = f
*	where a is an n*n matrix, f an n-vector.   Note that a is not
*	modified by the call, but f is overwritten with the solution x.
*
*	For the band-version, only the relevant part of the matrix is
*	copied.
*
*	Returns 1 if successfull.
*	Returns -1 if memory for a copy of the matrix supllied cannot
*	be generated.
*	Returns -2 if a zero pivot is encountered.
*/

int band_gauss_eliminate(int,int,int,float**,float*);

EXPORT int gauss_eliminate(int n,float **a, float *f)
{
	return band_gauss_eliminate(n,n-1,n-1,a,f);
}







EXPORT int band_gauss_eliminate(
	int n, 		/* Matrix dimension */
	int p, int q,	/* Numbers of bands below, above diagonal */
	float **A,
	float *f)
{
	int i,j,k;
	float c;
	int low,	/* lowest row to subtract from */
	    right;	/* Furthest right column to subtract with */
	float **a;

	matrix(&a,n,n,FLOAT);
	if (a==NULL) return -1;
	for(i=0; i<n; i++) {
		right = min(i+q,n-1);
		for(j=max(i-p,0); j<=right; j++)
			a[i][j] = A[i][j];
	}
			
			/* Triangulate: */
	for(i=0; i<n-1; i++)  {
		if (fabs(a[i][i])<1.e-10) {
			free(a);
			return -2;
		}
		low = min(i+p,n-1);
		right = min(i+q,n-1);
		for (j=i+1; j<=low; j++) {
			c = a[j][i]/a[i][i];
			for (k=i; k<=right; k++) a[j][k] -= c*a[i][k];
			f[j] -= c*f[i];
		}
	}
	if (fabs(a[n-1][n-1])<1.e-10) {
		free(a);
		return -2;
	}

			/* Back Solve: */
	for (i=n-1; i>=0; i--) {
		right = min(i+q,n-1);
		for (j=i+1; j<=right; j++) f[i] -= a[i][j]*f[j];
		f[i] /= a[i][i];
	}
	free(a);


	return 1;
}









/*
*				band_triangulate():
*				band_lu_eliminate():
*
*	These pair of routines first triangulate a matrix then solve
*	a linear system using the triangulated matrix.    The matrix
*	is modified by band_triangulate(), and is replaced by its
*	LU decomposition.
*
*	band_triangulate() returns 1 on success, or 0 if a zero pivot
*	is encountered.
*	band_lu_eliminate() should always succeed and returns 1.
*/


EXPORT int band_triangulate(
	int n,		/* Matrix dimension */
	int p,		/* Numbers of bands below, above diagonal */
	int q,
	float **a)
{
	int i,j,k;
	float c;
	int low,	/* lowest row to subtract from */
	    right;	/* Furthest right column to subtract with */

			
			/* Triangulate: */
	for(i=0; i<n-1; i++)  {
		if (fabs(a[i][i])<1.e-10) return 0;
		low = min(i+p,n-1);
		right = min(i+q,n-1);
		for (j=i+1; j<=low; j++) {
			a[j][i] = c = a[j][i]/a[i][i];
			for (k=i+1; k<=right; k++) a[j][k] -= c*a[i][k];
		}
	}
	if (fabs(a[n-1][n-1])<1.e-10) return 0;
	return 1;
}






EXPORT int band_lu_eliminate(
	int   n,
	int   p,
	int   q,
	float **a,
	float *f)
{
	int i,j,low,right,n_p,n_q;

			/* Forward Eliminate: */
	n_p = n-p;

	switch (p) {
	case 1:
		for(i=0; i<n_p; i++)
			f[i+1] -= a[i+1][i]*f[i];
		break;
	case 2:
		for(i=0; i<n_p; i++) {
			f[i+1] -= a[i+1][i]*f[i];
			f[i+2] -= a[i+2][i]*f[i];
		}
		break;
	case 3:
		for(i=0; i<n_p; i++) {
			f[i+1] -= a[i+1][i]*f[i];
			f[i+2] -= a[i+2][i]*f[i];
			f[i+3] -= a[i+3][i]*f[i];
		}
		break;

	default:
		for(i=0; i<n_p; i++)  {
			low = i+p;
			for (j=i+1; j<=low; j++)
				f[j] -= a[j][i]*f[i];
		}
		break;
	}

	for (i=n_p; i<n-1; i++)  {
		for (j=i+1; j<n; j++)
			f[j] -= a[j][i]*f[i];
	}


			/* Back Solve: */
	n_q = n-q;

	for (i=n-1; i>=n_q; i--) {
		right = n-1;
		for (j=i+1; j<=right; j++) f[i] -= a[i][j]*f[j];
		f[i] /= a[i][i];
	}

	switch (q) {
	case 1:
		for(i=n_q-1; i>=0; i--) {
			f[i] -= a[i][i+1]*f[i+1];
			f[i] /= a[i][i];
		}
		break;
	case 2:
		for(i=n_q-1; i>=0; i--) {
			f[i] -= (a[i][i+1]*f[i+1] + a[i][i+2]*f[i+2]);
			f[i] /= a[i][i];
		}
		break;
	case 3:
		for(i=n_q-1; i>=0; i--) {
			f[i] -= (a[i][i+1]*f[i+1] + a[i][i+2]*f[i+2]
						  + a[i][i+3]*f[i+3]);
			f[i] /= a[i][i];
		}
		break;

	default:
		for (i=n_q-1; i>=0; i--) {
			right = i+q;
			for (j=i+1; j<=right; j++) f[i] -= a[i][j]*f[j];
			f[i] /= a[i][i];
		}
		break;
	}
	return 1;
}
