C
C                        ACCELERATED CONJUGATE GRADIENT:
C
C                Solves the system of equations  Ax = f , A > 0, using the
C        Accelerated Conjugate Gradient method.   It is assumed that A has
C        an "approximate" inverse  B > 0, which can be computed cheaply.
C        The following paramaters are required:
C                
C                n    The dimension of the matrix A and of all vectors.
C                x    The given trial solution, which may be zero.
C                f    The right hand side of the equation - nonzero!
C                nn   At most nn iterations will be performed.
C                err  The iterations cease when |r| < err*|f|, r is the residual
C                w    A vector of length 2n which is used for workspace.
C
C        NOTE:                All vectors are indexed from 1 to n.     
C
C        Two functions are also required, and must be supplied by the user:
C
C                applya(n,x,ax)    Replaces  ax  by  Ax, leaving x unchanged.
C                applyb(n,x,bx)    Replaces  bx  by  Bx, leaving x unchanged.
C
C
C                The function acconjgrad() returns the number of conjugate
C        gradient iterations used in the variable numits, and the solution
C        of the equations is in x().  When called, u should be the same
C        as x and r the same as f.
C


        SUBROUTINE acconj(n,x,u,f,r,nn,err,w,p,numdi,indi,di,numits)
        integer n,nn,numits,numdi,indi(numdi)
        real x(n),u(n),f(n),r(n),err,w(n),p(n),di(*)

        integer k
        real ff,oldrbr,rbr,s,sp
C       equivalence (r(1) , f(1)) , (u(1) , x(1))

        call ip(n,f,f,ff)
        if (ff .le. err*err)  then
C           Solution is 0
                call zfv(n,x)
                numits = 0
                return
        end if
        ff = ff*err*err 
C                                                  w = A*guess
        call applya(n,x,w,numdi,indi,di)
C                                               r = f - Ax  
        call vevmv(n,r,f,w)
C                                               p = Br      
        call applyb(n,r,p,numdi,indi,di)
C                                               oldrbr = <r,Br>
        call ip(n,r,p,oldrbr)
        if (oldrbr .le. ff) then
C                                               x is already converged
                numits = 0
                return
        end if

        do 100 k=0,nn-1
C                                                w = Ap
                call applya(n,p,w,numdi,indi,di)
C                                                s = <rBr>/<pAp>
                call ip(n,p,w,sp)
                s = oldrbr/sp
C                                                  r = r - s*Ap
                call astvtv(n,-s,w,r)
C                                                  u = u + s*p
                call astvtv(n,s,p,u)
C                                                 w = Br
                call applyb(n,r,w,numdi,indi,di)
C                                                rbr = <r,Br>
                call ip(n,r,w,rbr)
C                                                End Iteration
                if (rbr .le. ff) then
                        numits = k+1
                        return 
                end if
C                                                  s = <r,Br>/<oldr,Boldr>
                s = rbr/oldrbr
C                                                olrbr = <oldr,Boldr>
                oldrbr = rbr
C                                                 p  = Br + s*p
                call vevpst(n,p,w,s,p)
  100   continue
        numits = nn
        return 
        end



C                               inner_product(n,f,g,sp)
C
        SUBROUTINE ip(n,f,g,sp)
        integer n
        real f(n),g(n),sp

        integer i

        sp = 0.0
        do 10 i=1,n
                sp = sp + f(i)*g(i)
   10   continue
        return
        end






C                               zero_float_vector(n,x)
C
        SUBROUTINE zfv(n,x)
        integer n
        real x(n)

        integer i

        do 10 i=1,n
                x(i) = 0.0
   10   continue
        return
        end







C                               copy_vector_to_vector(n,x,y)
C
        SUBROUTINE cvtv(n,x,y)
        integer n
        real x(n),y(n)

        integer i

        do 10 i=1,n
                y(i) = x(i)
   10   continue
        return
        end







C                       vector_equals_vector_minus_vector(n,x,y,z)
C
        SUBROUTINE vevmv(n,x,y,z)
        integer n
        real x(n),y(n),z(n)

        integer i

        do 10 i=1,n
                x(i) = y(i) - z(i)
   10   continue
        return
        end







C               add_scalar_times_vector_to_vector(n,s,x,y)
C
        SUBROUTINE astvtv(n,s,x,y)
        integer n
        real s,x(n),y(n)

        integer i

        do 10 i=1,n
                y(i) = y(i) + s*x(i)
   10   continue
        return
        end







C               vector_equals_vector_plus_scalar_times_vector(n,x,y,s,z)
C
        SUBROUTINE vevpst(n,x,y,s,z)
        integer n
        real x(n),y(n),s,z(n)

        integer i

        do 10 i=1,n
                x(i) = y(i) + s*z(i)
   10   continue
        return
        end








C               add_vector_times_vector_to_vector:
C
        SUBROUTINE avtvtv(n,x,y,z)
        integer n
        real x(n),y(n),z(n)

        integer i

        do 10 i=1,n
                z(i) = z(i) + x(i)*y(i)
   10   continue
        return
        end








C                               applya()
C
C       Given a symmetric matrix in diagonal form and a vector x, 
C       computes the vector Ax.   The diagonal representation of an
C       n*n symmetric matrix consists of three quantities: numdi, 
C       indi(numdi), di(n*numdi).   Here numdi is the number of 
C       non-zero diagonals in the upper triangle, indi(i) is the
C       offset of the ith non-zero diagonal from the main diagonal
C       (thus indi(1) = 0 always, the offset of the main diagonal.
C       The elements of the ith diagonal are stored as a vector
C       beginning at  di(1+(i-1)*n) (thus the main diagonal begins
C       at d(1)).
C
        SUBROUTINE  applya(n,x,ax,numdi,indi,di)
        integer n,numdi,indi(numdi)
        real x(n),ax(n),di(*)

        integer r

        call zfv(n,ax)
        call avtvtv(n,di(1),x,ax)
        do 10 r=2,numdi
                call avtvtv(n-indi(r),di(1+(r-1)*n),x(1+indi(r)),ax)
                call avtvtv(n-indi(r),di(1+(r-1)*n),x,ax(1+indi(r)))
   10   continue
        return
        end








C                               applyb()
C
C       This is the identity transformation x -> Bx = x
C
C
        SUBROUTINE applyb(n,x,bx,numdi,indi,di)
        integer n,numdi,indi(numdi)
        real x(n),bx(n),di(*)

        call cvtv(n,x,bx)
        return
        end

