/***********************************************************************
* DOLIB/DONIO Version 0.0 (8/24/94)                                    *       *
*  Software to emulate shared memory on distributed memory environments*
* written by:                                                          *
*  Ed D'Azevedo and Charles Romine of Oak Ridge National Laboratory    *
*                                                                      *
* Questions and comments should be directed to                         *
*      efdazedo@msr.epm.ornl.gov or romine@msr.epm.ornl.gov            *
*                                                                      *
*  Please notify and acknowledge the authors in any research or        *
*  publications utilizing DOLIB/DONIO or any part of the code.         *
*                                                                      *
* NOTICE: Neither the institution nor the author make any              *
*  representations about the suitability of this software for any      *
*  purpose. This software is provided "as is", without express or      *
*  implied warranty.                                                   *
************************************************************************/


#include <stdio.h>
#include <assert.h>
#include <malloc.h>

#if !defined(FALSE)
#define FALSE 0
#endif

#if !defined(TRUE)
#define TRUE (!(FALSE))
#endif

	/* cannot inline away in actual implementation */
#undef USE_INLINE

#include "htable.h"

#if !defined(logical)
#define logical int
#endif



	
	/* given the pointer to an entry in the global list,
	   get the key */

#define GPTR_2_KEY( gptr, mykey )   \
   {     \
        DLinkCellptr iptr;   \
        HtableCellptr lptr;  \
  \
        iptr = gptr->local_iptr;  \
        assert(iptr != NULL );  \
  \
        lptr = data_doublelist( iptr );  \
        assert( lptr != NULL );  \
  \
        mykey = lptr->key;  \
        }






Htable create_htable(  ifp isequal, ifp hash_htable)
{
	Htable myhtable;
	int i;


	myhtable = (Htable) malloc(  sizeof( struct Htable_t ));
	assert( myhtable != NULL );

	myhtable->global_list = create_doublelist();

	myhtable->isequal = isequal;
	myhtable->hash_htable = hash_htable;
	
	for( i=0; i < HTABLE_SIZE; i++ ) {
		myhtable->table[i] = create_doublelist();
		};


	return( myhtable );
}



/* return HtableCellptr, use data_htable to get at the actual data */

DLinkCellptr find_htable( Htable myhtable, void *key ) 
{
	DoubleList mylist;
	int ip;
	int icount;
	DLinkCellptr iptr;
	DLinkCellptr nextptr;
	HtableCellptr myiptr;
	HtableGCellptr giptr;
	logical found;
	logical isempty;



	assert( myhtable != NULL );
	ip =  (myhtable->hash_htable( key )) % HTABLE_SIZE ;

	mylist = myhtable->table[ ip ];
	assert( mylist != NULL );

	isempty = (size_doublelist( mylist ) == 0);
	if (isempty) {
		return( NULL );
		};

	/* linear search to find entry */

	
	found = FALSE; icount = 0;
	for( iptr = head_doublelist(mylist); 
		iptr != NULL; 
		iptr = nextptr ) {

		nextptr = next_doublelist( mylist, iptr);

		myiptr = (HtableCellptr) data_doublelist(iptr);
		found = myhtable->isequal( key, myiptr->key );
		if (found) { break; };

		icount++;
		};

		

	if (!found) { 
		/* double check */
		assert(icount == size_doublelist( mylist));

		return( NULL ); 
		};
	

	/* entry found, adjust global and this linked lists */

	/* double check */

	assert( myiptr->global_iptr != NULL );

	giptr = (HtableGCellptr) data_doublelist((myiptr->global_iptr));

	assert( giptr != NULL );

	assert( giptr->local_iptr == iptr );

	movetohead_doublelist( myhtable->global_list, myiptr->global_iptr );

	assert( giptr->hashed_result == ip );
	movetohead_doublelist( myhtable->table[ip], iptr );


	return( iptr );
}


void * data_htable( Htable myhtable, HtableCellptr myiptr )
{
	assert( myhtable != NULL );
	assert( myiptr != NULL );

	return(  myiptr->pdata );
}



void insert_htable( Htable myhtable, void *key, void *pdata )
{
	HtableGCellptr giptr;
	HtableCellptr  liptr;
	int ip;

	logical isfound;

	assert( myhtable != NULL);


	/* make sure there is no duplication */
	isfound = ( find_htable(myhtable,key) != NULL );
	if (isfound) { return; /* nothing to do */ };


	ip = (myhtable->hash_htable(key)) % HTABLE_SIZE;

	giptr = (HtableGCellptr) malloc( sizeof(HtableGCell) );
	assert(giptr != NULL);

	giptr->hashed_result = ip;

	liptr = (HtableCellptr) malloc( sizeof(HtableCell) );
	assert(liptr != NULL );


	liptr->global_iptr = head_doublelist(myhtable->global_list );


	liptr->pdata = pdata;

	/* NOTE: the pointer is copied, key better not be on stack! */
	liptr->key = key;

	



	inserthead_doublelist(myhtable->global_list, giptr );
	inserthead_doublelist(myhtable->table[ip], liptr );

	giptr->local_iptr = head_doublelist( myhtable->table[ip] );
	liptr->global_iptr = head_doublelist( myhtable->global_list );


	return;
}


/* returns pdata, user should explicitly free storage */

void *deletelast_htable( Htable myhtable )
{

	HtableGCellptr giptr;
	DLinkCellptr liptr;
	HtableCellptr hptr;
	HtableCellptr hptr2;
	int ip;

	DoubleList myglist;
	DoubleList myllist;
	void *pdata;

	assert( myhtable != NULL );


	/* delete last one on the global list */

	myglist = myhtable->global_list;
	assert( myglist != NULL );

	assert( size_doublelist(myglist) > 0);

	giptr = (HtableGCellptr) deletelast_doublelist( myglist );
	assert( giptr != NULL );

	assert( giptr->local_iptr != NULL );
	liptr = giptr->local_iptr;
	assert(liptr != NULL );

	hptr = (HtableCellptr) data_doublelist(  liptr );
	assert(hptr != NULL);

	ip = (myhtable->hash_htable( hptr->key )) % HTABLE_SIZE;
	assert( ip == giptr->hashed_result );

	/* this local pointer should also be at the tail */

	myllist = myhtable->table[ip];
	assert( tail_doublelist(myllist) == liptr );

	hptr2 = (HtableCellptr) deletelast_doublelist( myllist );

	/* double check */
	assert(hptr == hptr2);

	pdata = hptr->pdata;

	/* finally deallocate storage */
	free( giptr );
	free( hptr->key ); /* NOTE: key is deallocated */
	free( hptr );



	return( pdata );
}


HtableCellptr delete_htable( Htable myhtable, void *key )
{

	HtableCellptr hptr;
	DLinkCellptr iptr;
	int ip;
	logical found;

	DLinkCellptr gptr;
	HtableGCellptr gcptr;

	assert( myhtable != NULL );

	iptr = find_htable( myhtable, key );

	found = (iptr != NULL );
	if (!found) { return( NULL); };

	ip  = (myhtable->hash_htable( key )) % HTABLE_SIZE;
	hptr = delete_doublelist( myhtable->table[ip], iptr );
	found = (hptr != NULL);

	if (found) {
		/* need to adjust global list */
	    gptr = hptr->global_iptr;
	    assert( gptr != NULL );

	    gcptr = delete_doublelist( myhtable->global_list, gptr );

	    /* double check */
	    assert( gcptr != NULL );
	    assert( gcptr->local_iptr == iptr );
	    assert( gcptr->hashed_result == ip );

	    free(gcptr);

	    /* free the "key:" filed */
	    free( hptr->key );

	    };


	return( hptr );

}


int purge_htable( Htable myhtable, void *inkey, ifp ismatch )
{
	/* traverse the global linked list and
	   purge all entries that match */

	DoubleList glist;
	DLinkCellptr iptr; 
	DLinkCellptr nextptr;
	int icount;
	int ndeletes;
	int oldsize;
	void *pdata;
	void *mykey;
	HtableGCellptr gptr;

	glist = myhtable->global_list;

	icount = 0; oldsize = size_doublelist( glist );
	ndeletes = 0;

	for( iptr = head_doublelist(glist);
		iptr != NULL;
		iptr = nextptr ) {

		nextptr = next_doublelist(glist, iptr);

                gptr = (HtableGCellptr) data_doublelist( iptr );
		assert( gptr != NULL );

                GPTR_2_KEY( gptr, mykey )
		assert( mykey != NULL );

                if (ismatch(inkey,mykey)) {

                   pdata =  delete_htable( myhtable, mykey );

                   assert( pdata != NULL );
                   free( pdata );

		   ndeletes++;

                   };


	   icount++;
	   }; /* end for */

	/* we have traversed the entire list, it better be
	   equal to original number of entries in the list */
	assert( icount == oldsize );

	return( ndeletes );
}
