/***********************************************************************
* DOLIB/DONIO Version 0.0 (8/24/94)                                    *       *
*  Software to emulate shared memory on distributed memory environments*
* written by:                                                          *
*  Ed D'Azevedo and Charles Romine of Oak Ridge National Laboratory    *
*                                                                      *
* Questions and comments should be directed to                         *
*      efdazedo@msr.epm.ornl.gov or romine@msr.epm.ornl.gov            *
*                                                                      *
*  Please notify and acknowledge the authors in any research or        *
*  publications utilizing DOLIB/DONIO or any part of the code.         *
*                                                                      *
* NOTICE: Neither the institution nor the author make any              *
*  representations about the suitability of this software for any      *
*  purpose. This software is provided "as is", without express or      *
*  implied warranty.                                                   *
************************************************************************/

#include <stdio.h>
#include <assert.h>
#include <malloc.h>

	/* must not use inline expansion from doublelist.h 
		in the implementation */

#undef USE_INLINE


#if !defined(ALLOC)
#define ALLOC( type, count ) ( (type *) malloc( sizeof(type)*(count) ) );
#endif

#if !defined(FREE)
#define FREE( iptr ) { assert( iptr != NULL ); free( iptr ); }
#endif

#if (IDEBUG > 1)
#define TRIPLE_CHECK( mylist ) \
        {  DLinkCellptr ip; DLinkCellptr ipold; int i;  \
           ip = (mylist)->head; ipold = NULL;  \
           for( i=1; i <= (mylist)->ncount; i++ ) {  \
                if (i > 1) {   \
                        assert( ip->blink == ipold );  \
                        assert( ipold != NULL );  \
                        assert( ipold->flink == ip );  \
                        };  \
                ipold = ip; ip = ip->flink;  \
                };  \
        };  
#else
#define TRIPLE_CHECK( mylist )
#endif


#if (IDEBUG > 0)
#define DOUBLE_CHECK( mylist ) \
  {   \
	if (((mylist)->ncount) == 0){   \
   \
		/* empty list */   \
   \
		assert( (mylist)->head == NULL );   \
		assert( (mylist)->tail == NULL );   \
		}   \
	else {   \
		/* non-empty list */   \
   \
		assert( (mylist)->head != NULL );   \
		assert( (mylist)->tail != NULL );   \
   \
		assert( ((mylist)->head)->blink == NULL );   \
		assert( ((mylist)->tail)->flink == NULL );   \
		};   \
	if (((mylist)->ncount) == 1 ) {   \
		assert( ((mylist)->head) == ((mylist)->tail) );   \
		};   \
	TRIPLE_CHECK( mylist ); \
  }
#else
#define DOUBLE_CHECK(mylist)
#endif


/* source to implement doubly-linked lists */

#include "doublelist.h"



DoubleList create_doublelist()
{
	DoubleList this_ptr;
	this_ptr = (DoubleList) malloc( sizeof( struct DoubleList_t ) );

	assert( this_ptr != NULL );

	this_ptr->head = NULL;
	this_ptr->tail = NULL;
	this_ptr->ncount = 0;

	return( this_ptr );
}

void destroy_doublelist( DoubleList mylist )
{
	void * data;

	assert( mylist != NULL );

	DOUBLE_CHECK( mylist );

	while (mylist->ncount > 0) {
	    data = deletelast_doublelist( mylist );
	    if (data != NULL) {
		FREE( data );
		};
	    };

	assert( (mylist->head == NULL) && 
		(mylist->tail == NULL) &&
		(mylist->ncount == 0) );

	FREE( mylist );
}


void *delete_doublelist( DoubleList mylist, DLinkCellptr iptr )
{

	/* returns the data pointer, user decide whether to deallocate it */

	DLinkCellptr flink;
	DLinkCellptr blink;
	int isempty;

	int ishead;
	int istail;

	void *data;


	assert( iptr != NULL );
	assert( mylist != NULL );

	DOUBLE_CHECK( mylist );



	flink = iptr->flink; blink = iptr->blink;
	data = iptr->data;

	ishead = (iptr == (mylist->head));
	istail = (iptr == (mylist->tail));

	if (flink != NULL) {
		flink->blink = blink;
		}
	else {
		assert( istail );
		};

	if (blink != NULL) {
		blink->flink = flink;
		}
	else {
		assert( ishead );
		};

	/* test for end conditions */

	if (ishead) {
		assert( blink == NULL );

		mylist->head = flink;
		};

	if (istail) {
		assert( flink == NULL );

		mylist->tail = blink;
		};

	mylist->ncount = (mylist->ncount) - 1;


	/* test for special case */


	isempty = ((mylist->ncount) == 0 );
	if (isempty) {
		mylist->head = NULL;
		mylist->tail = NULL;
		mylist->ncount = 0;
		};

	/* double check */

	DOUBLE_CHECK( mylist );

	/* finally deallocate storage */

	FREE( iptr );

	return(data);
}


DLinkCellptr head_doublelist( DoubleList mylist )
{
	assert( mylist != NULL );

	DOUBLE_CHECK( mylist );

	return( mylist->head );
}
		

DLinkCellptr tail_doublelist( DoubleList mylist )
{
	assert( mylist != NULL );

	DOUBLE_CHECK( mylist );

	return( mylist->tail );
}


void *deletelast_doublelist( DoubleList mylist )
{
	
	assert( mylist != NULL );

	DOUBLE_CHECK( mylist );


	return( delete_doublelist( mylist, mylist->tail ));
}
	

void inserthead_doublelist( DoubleList mylist, void *data )
{
	DLinkCellptr iptr;
	int isempty;

	assert( mylist != NULL );

	isempty = (mylist->ncount == 0);

	/* double check */
	DOUBLE_CHECK( mylist );

	iptr = ALLOC( DLinkCell, 1 );
	assert( iptr != NULL );
	
	iptr->flink = mylist->head;
	iptr->blink = NULL;
	iptr->data = data;

	if (iptr->flink != NULL) {
		(iptr->flink)->blink = iptr;
		};

	mylist->head = iptr;

	/* check for end condition */
	
	if (isempty) {
		mylist->ncount = 1;
		mylist->tail = mylist->head;
		}
	else {
		mylist->ncount = mylist->ncount +  1;
		};

	DOUBLE_CHECK( mylist );

	return;
}


DLinkCellptr next_doublelist( DoubleList mylist, DLinkCellptr iptr )
{
	assert( mylist != NULL );
	assert( iptr != NULL );

	DOUBLE_CHECK( mylist );

	return( iptr->flink );
}


DLinkCellptr prev_doublelist(DoubleList mylist, DLinkCellptr iptr )
{
        assert( mylist != NULL );
        assert( iptr != NULL );


	DOUBLE_CHECK( mylist );

        return( iptr->blink );
}

	

void *data_doublelist(  DLinkCellptr iptr )
{
        assert( iptr != NULL );


        return( iptr->data );
}

void movetohead_doublelist( DoubleList mylist, DLinkCellptr  iptr )
{
	DLinkCellptr flink;
	DLinkCellptr blink;

	assert( mylist != NULL );
	assert( iptr != NULL );

	DOUBLE_CHECK( mylist );

	/* must be non-empty list */
	assert( mylist->ncount > 0 );

	if (mylist->head == iptr) { 
		/* nothing to do */
		return;
		};


	/* if there is only 1 entry, it must be
	   already pointed to by head */

	assert( mylist->ncount > 1);


	flink = iptr->flink; blink = iptr->blink;

	/* decouple from list */
	if (flink != NULL) {
		flink->blink = blink;
		};
	if (blink != NULL) {
		blink->flink = flink;
		};

	/* check for end condition */
	if (mylist->tail == iptr) {
		mylist->tail = blink;
		};

	
	if (mylist->head != NULL) {
		(mylist->head)->blink = iptr;
		};
	



	/* make head point to this */

	iptr->blink = NULL;
	iptr->flink = mylist->head;
	mylist->head = iptr;


	DOUBLE_CHECK(mylist);

	return;
}

int size_doublelist( DoubleList mylist )
{
	assert( mylist != NULL );

	assert( mylist->ncount >= 0 );

	DOUBLE_CHECK( mylist );

	return( mylist->ncount );
}
