/***********************************************************************
*                                                                      *
* A serial library to implement most of DOLIB library calls for use    *
* for debugging in serial mode.                                        *
* written by:                                                          *
*  Ed D'Azevedo of Oak Ridge National Laboratory                       *
* Based on DOLIB, written by Ed D'Azevedo and Charles Romine           *
*  of Oak Ridge National Laboratory.                                   *
*                                                                      *
* Questions or comments should be directed to efdazedo@msr.epm.ornl.gov*
*  Please notify and acknowledge the author in any research or         *
*  publications utilizing DOLIB or any part of the code.               *
*                                                                      *
* NOTICE: Neither the institution nor the author make any              *
*  representations about the suitability of this software for any      *
*  purpose. This software is provided "as is", without express or      *
*  implied warranty.                                                   *
************************************************************************/

/* fake dolib calls for serial code */






#include <stdio.h>
#include <assert.h>
#include <malloc.h>
#include <math.h>
#include <string.h>

/*
#include "dolib.h"
*/

#include "memory.h"

#define MEMCPY( dest, src, nbytes ) { \
	memcpy( (void *) (dest), (void *) (src), (size_t) (nbytes) ); }

#define INLINE_BLOCK_COPY 1

/*
#undef INLINE_BLOCK_COPY 
*/


#define dfloat double
#define logical int
#define ALL -1


#if !defined(FALSE)
#define FALSE 0
#endif

#if !defined(TRUE)
#define TRUE (!FALSE)
#endif

#if !defined(NULL)
#define NULL 0
#endif


#define MEMALLOC( pointer, type, nbytes ) \
		{ \
		  assert( (nbytes) > 0); \
			pointer = type malloc( (nbytes) ); \
			assert( pointer != NULL ); \
		}

#define MEMFREE( pointer ) \
	{ assert( pointer != NULL ); \
		free( pointer ); \
		pointer = NULL; \
		}

/* #define malloc(n) i_malloc(n)  */

/* #define realloc(p,n) i_realloc(p,n)  */
/* #define free(p) i_free(p)  */


#if I860 || RX
/* bcopy is either not available or is very slow (in libX11.a) */
/* reverse arguments for faster memcpy */
#define BCOPY(src, dest, nbytes)  memcpy( dest, src, nbytes );

#else
#define BCOPY(src,dest,nbytes) \
	{ bcopy((char *)src, (char *) dest, (int) nbytes); }
extern void bcopy(char *src, char *dest, int nbytes);

#endif



#if !defined(MOD)
#define MOD( ival, ibase )((ival) %(ibase) )
#endif

#if !defined(max)
#define max(ix,iy) ( (ix) >(iy) ?(ix):(iy) )
#endif

#if !defined(min)
#define min(ix,iy)((ix) <(iy) ?(ix):(iy) )
#endif






#define strequal( str1, str2 ) (strcmp(str1,str2) == 0)

#define logical int
#define real float

#define MAX_GLOBAL_ARRAY 256

typedef struct {
	char           *name;
	char           *ctype;
	int             gsize;
	int             pagesize;
	int             blocksize;
	void           *pointer;
}               dolib_node;

static dolib_node Global_array[MAX_GLOBAL_ARRAY + 1];

#define VALIDIAF( IAf ) ( \
			(1 <= IAf) &&  \
			(IAf <= MAX_GLOBAL_ARRAY) &&  \
			(Global_array[ IAf ].name != NULL) &&  \
			(Global_array[ IAf ].ctype != NULL) && \
			(Global_array[ IAf ].gsize > 0) &&  \
			(Global_array[ IAf ].blocksize > 0) && \
			(Global_array[ IAf ].pointer != NULL)  \
			)

#define ISINTEGER( IAf ) (\
		(strequal(Global_array[ IAf ].ctype,"integer") || \
		 strequal(Global_array[ IAf ].ctype,"int") ) \
		)

#define ISREAL( IAf ) (\
		(strequal(Global_array[ IAf ].ctype,"real") || \
		 strequal(Global_array[ IAf ].ctype,"REAL") || \
		 strequal(Global_array[ IAf ].ctype,"float") || \
		 strequal(Global_array[ IAf ].ctype,"FLOAT") || \
		 strequal(Global_array[ IAf ].ctype,"real4") || \
		 strequal(Global_array[ IAf ].ctype,"real*8") ) \
		)

#define ISDOUBLE( IAf ) (\
		(strequal(Global_array[ IAf ].ctype,"double") || \
		 strequal(Global_array[ IAf ].ctype,"double precision") || \
		 strequal(Global_array[ IAf ].ctype,"double") || \
		 strequal(Global_array[ IAf ].ctype,"real8") || \
		 strequal(Global_array[ IAf ].ctype,"real*8") ) \
		)

#define GSIZE( IAf ) ( Global_array[ IAf ].gsize )

#define NAME( IAf ) ( Global_array[ IAf ].name )

#define POINTER( IAf ) ( Global_array[ IAf ].pointer )

#if defined(INLINE)
#define CHECK_DOUBLE( IAf, routine ) { \
		assert( VALIDIAF( IAf ) );  \
		if (!ISDOUBLE( IAf )) {  \
			printf("*** %s: expect %s to be double \n",   \
					routine, NAME( IAf ));  \
			};  \
		assert( ISDOUBLE( IAf ) );  \
		}
#else

#define CHECK_DOUBLE( IAf, routine) {  check_double( IAf, routine ); }

void 
check_double(IAf, routine)
	int             IAf;
	char           *routine;
{
	if (!VALIDIAF(IAf)) {
		printf("*** %s: invalid IAf %d\n", routine, IAf);
	};
	assert(VALIDIAF(IAf));
	if (!ISDOUBLE(IAf)) {
		printf("*** %s: expect %s to be double \n",
		       routine, NAME(IAf));
	};
	assert(ISDOUBLE(IAf));
}
#endif


#if defined(INLINE)
#define CHECK_REAL( IAf, routine ) { \
                assert( VALIDIAF( IAf ) );  \
                if (!ISREAL( IAf )) {  \
                        printf("*** %s: expect %s to be real \n",   \
                                        routine, NAME( IAf ));  \
                        };  \
                assert( ISREAL( IAf ) );  \
                }
#else

#define CHECK_REAL( IAf, routine) {  check_real( IAf, routine ); }

void 
check_real(IAf, routine)
	int             IAf;
	char           *routine;
{
	if (!VALIDIAF(IAf)) {
		printf("*** %s: invalid IAf %d\n", routine, IAf);
	};
	assert(VALIDIAF(IAf));
	if (!ISREAL(IAf)) {
		printf("*** %s: expect %s to be real \n",
		       routine, NAME(IAf));
	};
	assert(ISREAL(IAf));
}
#endif


#if defined(INLINE)
#define CHECK_INT( IAf, routine ) { \
		assert( VALIDIAF( IAf ) );  \
		if (!ISINTEGER( IAf )) {  \
			printf("*** %s: expect %s to be integer \n",   \
					routine, NAME( IAf ));  \
			};  \
		assert( ISINTEGER( IAf ) );  \
		}
#else
#define CHECK_INT( IAf, routine ) { check_int( IAf, routine ); }

void 
check_int(IAf, routine)
	int             IAf;
	char           *routine;
{
	if (!VALIDIAF(IAf)) {
		printf("invalid IAf %d in routine %s\n", IAf, routine);
	};
	assert(VALIDIAF(IAf));
	if (!ISINTEGER(IAf)) {
		printf("*** %s: expect %s to be integer \n",
		       routine, NAME(IAf));
	};
	assert(ISINTEGER(IAf));
}

#endif



void
do_destroy(IAf)
	int             IAf;
{

	assert(VALIDIAF(IAf));

	assert(Global_array[IAf].pointer != NULL);
	free(Global_array[IAf].pointer);
	Global_array[IAf].pointer = NULL;


	assert(Global_array[IAf].name != NULL);
	free(Global_array[IAf].name);
	Global_array[IAf].name = NULL;

	assert(Global_array[IAf].ctype != NULL);
	free(Global_array[IAf].ctype);
	Global_array[IAf].ctype = NULL;

	Global_array[IAf].gsize = 0;
	Global_array[IAf].pagesize = 0;
	Global_array[IAf].blocksize = 0;
}

void
do_declare(IAf, name, gsize, ctype, pagesize, blocksize)
	int            *IAf;
	char           *name;
	int             gsize;
	char           *ctype;
	int             pagesize, blocksize;
{

	int             i, ipos, nbytes;
	logical         found;
	int            *iptr;
	double         *dptr;
	real           *rptr;

	/* find a slot */

	ipos = -1;
	for (i = 1; i <= MAX_GLOBAL_ARRAY; i++) {
		if (Global_array[i].name == NULL) {
			ipos = i;
			break;
		};
	};
	found = (1 <= ipos) && (ipos <= MAX_GLOBAL_ARRAY);
	assert(found);

	*IAf = ipos;


	Global_array[ipos].name = (char *) strdup(name);
	Global_array[ipos].ctype = (char *) strdup(ctype);
	Global_array[ipos].gsize = gsize;
	Global_array[ipos].pagesize = pagesize;
	Global_array[ipos].blocksize = blocksize;

	if (ISINTEGER(ipos)) {
		/* add 1 for fortran indexing starting 1..gsize */

		nbytes = (gsize + 1) * sizeof(int);
		Global_array[ipos].pointer =
			(void *) malloc(nbytes);
		assert(Global_array[ipos].pointer != NULL);

		iptr = (int *) Global_array[ipos].pointer;
		for (i = 1; i <= gsize; i++) {
			iptr[i] = 123456;
		};
	} else if (ISREAL(ipos)) {
		/* add 1 for fortran indexing starting 1..gsize */

		nbytes = (gsize + 1) * sizeof(double);
		Global_array[ipos].pointer =
			(void *) malloc(nbytes);
		assert(Global_array[ipos].pointer != NULL);



		rptr = (real *) Global_array[ipos].pointer;
		for (i = 1; i <= gsize; i++) {
			rptr[i] = 999.999;
		};
	} else if (ISDOUBLE(ipos)) {
		/* add 1 for fortran indexing starting 1..gsize */

		nbytes = (gsize + 1) * sizeof(double);
		Global_array[ipos].pointer =
			(void *) malloc(nbytes);
		assert(Global_array[ipos].pointer != NULL);

		dptr = (double *) Global_array[ipos].pointer;
		for (i = 1; i <= gsize; i++) {
			dptr[i] = 999.999;
		};
	} else {
		/* impossible */
		assert(0);
	};

	assert(Global_array[ipos].pointer != NULL);

	if (!VALIDIAF(*IAf)) {
		printf("do_declare: invalid iaf %d, gsize %d, name %s\n",
		       (*IAf), gsize, name);
	};

	assert(VALIDIAF(*IAf));

}



int
do_bigather_C(IAf, nsize, istart, ibuf)
	int             IAf, nsize, istart;
	int            *ibuf;
{

	int             i, ip, gsize;
	int            *isrc;
	logical         isvalid;


	CHECK_INT(IAf, "do_bigather_C");

	isrc = (int *) POINTER(IAf);
	gsize = GSIZE(IAf);

        ip = istart;
        isvalid = (1 <= ip) && (ip <= gsize);
        if (!isvalid) {
            printf("** do_bigather_C: of %s\n", NAME(IAf));
            printf("** invalid index value %d, gsize %d\n", ip, gsize);
            };
        assert(isvalid);

        ip = istart + (nsize-1);
        isvalid = (1 <= ip) && (ip <= gsize);
        if (!isvalid) {
            printf("** do_bigather_C: of %s\n", NAME(IAf));
            printf("** invalid index value %d, gsize %d\n", ip, gsize);
            };
        assert(isvalid);


#if defined(INLINE_BLOCK_COPY)
	ip = istart;
	for (i = 0; i < nsize; i = i + 1) {
		ibuf[i] = isrc[ip];
		ip = ip + 1;
	};
#else

	/* use MEMCPY(dest,src,nbytes) */
	{   
	void *src; void *dest; size_t nbytes;


	ip = istart; i = 0;
	src = (void *) (&(isrc[ip]));
	dest = (void *) (&(ibuf[i]));
	nbytes = (size_t) (sizeof(int)*nsize);

	MEMCPY(dest,src,nbytes);
	};

	
#endif /* defined(INLINE_BLOCK_COPY) */

	return (IAf);
}

void
do_igather_C(IAf, nsize, list, ibuf)
	int             IAf, nsize;
	int            *list;
	int            *ibuf;
{
	int             i, ip, gsize;
	int            *isrc;
	logical         isvalid;


	CHECK_INT(IAf, "do_igather_C");

	isrc = (int *) POINTER(IAf);
	gsize = GSIZE(IAf);

	for (i = 0; i < nsize; i++) {
		ip = list[i];
		isvalid = (1 <= ip) && (ip <= gsize);
		if (!isvalid) {
			printf("** do_igather_C: of %s\n", NAME(IAf));
			printf("** invalid list[i], i %d list[i] %d\n",
			       i, list[i]);
		};
		assert(isvalid);
		ibuf[i] = isrc[ip];
	};

}

void
do_dgather_C(IAf, nsize, list, dbuf)
	int             IAf, nsize;
	int            *list;
	double         *dbuf;
{

	int             i, ip, gsize;
	logical         isvalid;
	double         *dsrc;


	CHECK_DOUBLE(IAf, "do_dgather_C");

	dsrc = (double *) POINTER(IAf);
	gsize = GSIZE(IAf);

	for (i = 0; i < nsize; i++) {

		ip = list[i];
		isvalid = (1 <= ip) && (ip <= gsize);
		if (!isvalid) {
			printf("** do_dgather_C: of %s\n", NAME(IAf));
			printf("** invalid list[i], i %d list[i] %d\n",
			       i, list[i]);
		};
		assert(isvalid);
		dbuf[i] = dsrc[ip];
	};

}

void
do_rgather_C(IAf, nsize, list, dbuf)
	int             IAf, nsize;
	int            *list;
	real           *dbuf;
{

	int             i, ip, gsize;
	logical         isvalid;
	real           *dsrc;


	CHECK_REAL(IAf, "do_rgather_C");

	dsrc = (real *) POINTER(IAf);
	gsize = GSIZE(IAf);

	for (i = 0; i < nsize; i++) {

		ip = list[i];
		isvalid = (1 <= ip) && (ip <= gsize);
		if (!isvalid) {
			printf("** do_rgather_C: of %s\n", NAME(IAf));
			printf("** invalid list[i], i %d list[i] %d\n",
			       i, list[i]);
		};
		assert(isvalid);
		dbuf[i] = dsrc[ip];
	};

}



int
do_bdgather_C(IAf, nsize, istart, dbuf)
	int             IAf, nsize, istart;
	double         *dbuf;
{

	int             i, ip, gsize;
	logical         isvalid;
	double         *dsrc;

	CHECK_DOUBLE(IAf, "do_bdgather_C");

	dsrc = (double *) POINTER(IAf);
	gsize = GSIZE(IAf);

	ip = istart;
	isvalid = (1 <= ip) && (ip <= gsize);
	if (!isvalid) {
	    printf("** do_bdgather_C: of %s\n", NAME(IAf));
	    printf("** invalid  index %d, gsize %d\n", ip, gsize);
	    };
	assert(isvalid);


        ip = istart + (nsize-1);
        isvalid = (1 <= ip) && (ip <= gsize);
        if (!isvalid) {
            printf("** do_bdgather_C: of %s\n", NAME(IAf));
            printf("** invalid  index %d, gsize %d\n", ip, gsize);
            };
        assert(isvalid);




#if defined(INLINE_BLOCK_COPY)

	ip = istart;
	for (i = 0; i < nsize; i = i + 1) {

		dbuf[i] = dsrc[ip];
		ip = ip + 1;
	};
#else

        /* use MEMCPY(dest,src,nbytes) */
        {
        void *src; void *dest; size_t nbytes;


        ip = istart; i = 0;
        src = (void *) (&(dsrc[ip]));
        dest = (void *) (&(dbuf[i]));
        nbytes = (size_t) (sizeof(double)*nsize);

        MEMCPY(dest,src,nbytes);
        };


#endif /* defined(INLINE_BLOCK_COPY) */


	return (IAf);
}



int
do_brgather_C(IAf, nsize, istart, dbuf)
	int             IAf, nsize, istart;
	real           *dbuf;
{

	int             i, ip, gsize;
	logical         isvalid;
	real           *dsrc;

	CHECK_REAL(IAf, "do_brgather_C");

	dsrc = (real *) POINTER(IAf);
	gsize = GSIZE(IAf);

	ip = istart;
	isvalid = (1 <= ip) && (ip <= gsize);
	if (!isvalid) {
		printf("** do_brgather_C: of %s\n", NAME(IAf));
		printf("** invalid  index %d, gsize %d\n", ip, gsize);
	};
	assert(isvalid);

	ip = istart + nsize - 1;
	isvalid = (1 <= ip) && (ip <= gsize);
	if (!isvalid) {
		printf("** do_brgather_C: of %s\n", NAME(IAf));
		printf("** invalid  index %d, gsize %d\n", ip, gsize);
	};
	assert(isvalid);



#if defined(INLINE_BLOCK_COPY)

	ip = istart;
	for (i = 0; i < nsize; i = i + 1) {

		dbuf[i] = dsrc[ip];
		ip = ip + 1;
	};

#else

        /* use MEMCPY(dest,src,nbytes) */
        {
        void *src; void *dest; size_t nbytes;


        ip = istart;
        src = (void *) (&(dsrc[ip]));
        dest = (void *) (&(dbuf[0]));
        nbytes = (size_t) (sizeof(real)*nsize);

        MEMCPY(dest,src,nbytes);
        };


#endif /* defined(INLINE_BLOCK_COPY) */


	return (IAf);
}


void
do_iaxpbyz_C(IAf, nsize, list, ix, ialpha, ibeta)
	int             IAf, nsize;
	int            *list;
	int            *ix;
	int             ialpha, ibeta;
{
	int             i, ip, gsize;
	logical         isvalid;
	int            *iy;
	int		iytemp;


	CHECK_INT(IAf, "do_iaxpbyz_C");

	iy = (int *) POINTER(IAf);
	gsize = GSIZE(IAf);

	for (i = 0; i < nsize; i++) {
		ip = list[i];
		isvalid = (1 <= ip) && (ip <= gsize);
		if (!isvalid) {
			printf("** do_iaxpbyz_C: of %s\n", NAME(IAf));
			printf("** invalid  index %d, gsize %d\n", ip, gsize);
		};
		assert(isvalid);

		iytemp = iy[ip];
		iy[ip] = ialpha * ix[i] + ibeta * iy[ip];
		ix[i] = iytemp;
	};

}

void
do_daxpbyz_C(IAf, nsize, list, dx, dalpha, dbeta)
	int             IAf, nsize;
	int            *list;
	double         *dx;
	double          dalpha, dbeta;
{

	int             i, ip, gsize;
	logical         isvalid;
	double         *dy;
	double		dytemp;

	CHECK_DOUBLE(IAf, "do_daxpbyz_C");

	dy = (double *) POINTER(IAf);
	gsize = GSIZE(IAf);

	for (i = 0; i < nsize; i++) {
		ip = list[i];
		isvalid = (1 <= ip) && (ip <= gsize);
		if (!isvalid) {
			printf("** do_daxpbyz_C: of %s\n", NAME(IAf));
			printf("** invalid  index %d, gsize %d \n", ip, gsize);
		};
		assert(isvalid);

		dytemp = dy[ip];
		dy[ip] = dalpha * dx[i] + dbeta * dy[ip];
		dx[i] = dytemp;
	};

}


void
do_raxpbyz_C(IAf, nsize, list, dx, dalpha, dbeta)
	int             IAf, nsize;
	int            *list;
	real           *dx;
	real            dalpha, dbeta;
{

	int             i, ip, gsize;
	logical         isvalid;
	real           *dy;
	real		dytemp;

	CHECK_REAL(IAf, "do_raxpbyz_C");

	dy = (real *) POINTER(IAf);
	gsize = GSIZE(IAf);

	for (i = 0; i < nsize; i++) {
		ip = list[i];
		isvalid = (1 <= ip) && (ip <= gsize);
		if (!isvalid) {
			printf("** do_raxpbyz_C: of %s\n", NAME(IAf));
			printf("** invalid  index %d, gsize %d \n", ip, gsize);
		};
		assert(isvalid);

		dytemp = dy[ip];
		dy[ip] = dalpha * dx[i] + dbeta * dy[ip];
		dx[i] = dytemp;
	};

}





int
do_biaxpbyz_C(IAf, nsize, istart, ix, ialpha, ibeta)
	int             IAf, nsize, istart;
	int            *ix;
	int             ialpha, ibeta;
{
	int             i, ip, gsize;
	int            *iy, iytemp;
	logical         isvalid;

	CHECK_INT(IAf, "do_biaxpbyz_C");

	iy = (int *) POINTER(IAf);
	gsize = GSIZE(IAf);

	

	ip = istart;
	isvalid = (1 <= ip) && (ip <= gsize);
	if (!isvalid) {
	    printf("** do_biaxpbyz_C: of %s\n", NAME(IAf));
	    printf("** invalid  index %d, gsize %d\n", ip, gsize);
	    };
	assert(isvalid);


        ip = istart + (nsize-1);
        isvalid = (1 <= ip) && (ip <= gsize);
        if (!isvalid) {
            printf("** do_biaxpbyz_C: of %s\n", NAME(IAf));
            printf("** invalid  index %d, gsize %d\n", ip, gsize);
            };
        assert(isvalid);




	ip = istart;
	for (i = 0; i < nsize; i++) {

		iytemp = iy[ip];
		iy[ip] = ialpha * ix[i] + ibeta * iy[ip];
		ix[i] = iytemp;

		ip = ip + 1;
	};
	return (IAf);
}


int
do_bdaxpbyz_C(IAf, nsize, istart, dx, dalpha, dbeta)
	int             IAf, nsize, istart;
	double         *dx;
	double          dalpha, dbeta;
{
	int             i, ip, gsize;
	logical         isvalid;

	double         *dy, dytemp;


	CHECK_DOUBLE(IAf, "do_bdaxpbyz_C");

	dy = (double *) POINTER(IAf);
	gsize = GSIZE(IAf);

	ip = istart;
	for (i = 0; i < nsize; i++) {
		isvalid = (1 <= ip) && (ip <= gsize);
		if (!isvalid) {
			printf("** do_bdaxpbyz_C: of %s\n", NAME(IAf));
			printf("** invalid  index %d, gsize %d\n", ip, gsize);
		};
		assert(isvalid);

		dytemp = dy[ip];
		dy[ip] = dalpha * dx[i] + dbeta * dy[ip];
		dx[i] = dytemp;

		ip = ip + 1;
	};
	return (IAf);
}





int
do_braxpbyz_C(IAf, nsize, istart, dx, dalpha, dbeta)
	int             IAf, nsize, istart;
	real           *dx;
	real            dalpha, dbeta;
{
	int             i, ip, gsize;
	logical         isvalid;

	real           *dy, dytemp;


	CHECK_REAL(IAf, "do_braxpbyz_C");

	dy = (real *) POINTER(IAf);
	gsize = GSIZE(IAf);

	ip = istart;
	isvalid = (1 <= ip) && (ip <= gsize);
	if (!isvalid) {
		printf("** do_braxpbyz_C: of %s\n", NAME(IAf));
		printf("** invalid  index %d, gsize %d\n", ip, gsize);
	};
	assert(isvalid);

	ip = istart + nsize - 1;
	isvalid = (1 <= ip) && (ip <= gsize);
	if (!isvalid) {
		printf("** do_braxpbyz_C: of %s\n", NAME(IAf));
		printf("** invalid  index %d, gsize %d\n", ip, gsize);
	};
	assert(isvalid);




	ip = istart;
	for (i = 0; i < nsize; i++) {
		dytemp = dy[ip];
		dy[ip] = dalpha * dx[i] + dbeta * dy[ip];
		dx[i] = dytemp;

		ip = ip + 1;
	};
	return (IAf);
}


void
do_iaxpby_C(IAf, nsize, list, ix, ialpha, ibeta)
	int             IAf, nsize;
	int            *list;
	int            *ix;
	int             ialpha, ibeta;
{
	int             i, ip, gsize;
	logical         isvalid;
	int            *iy;


	CHECK_INT(IAf, "do_iaxpby_C");

	iy = (int *) POINTER(IAf);
	gsize = GSIZE(IAf);

	for (i = 0; i < nsize; i++) {
		ip = list[i];
		isvalid = (1 <= ip) && (ip <= gsize);
		if (!isvalid) {
			printf("** do_iaxpby_C: of %s\n", NAME(IAf));
			printf("** invalid  index %d, gsize %d\n", ip, gsize);
		};
		assert(isvalid);

		iy[ip] = ialpha * ix[i] + ibeta * iy[ip];
	};

}

void
do_daxpby_C(IAf, nsize, list, dx, dalpha, dbeta)
	int             IAf, nsize;
	int            *list;
	double         *dx;
	double          dalpha, dbeta;
{

	int             i, ip, gsize;
	logical         isvalid;
	double         *dy;

	CHECK_DOUBLE(IAf, "do_daxpby_C");

	dy = (double *) POINTER(IAf);
	gsize = GSIZE(IAf);

	for (i = 0; i < nsize; i++) {
		ip = list[i];
		isvalid = (1 <= ip) && (ip <= gsize);
		if (!isvalid) {
			printf("** do_daxpby_C: of %s\n", NAME(IAf));
			printf("** invalid  index %d, gsize %d \n", ip, gsize);
		};
		assert(isvalid);

		dy[ip] = dalpha * dx[i] + dbeta * dy[ip];
	};

}


void
do_raxpby_C(IAf, nsize, list, dx, dalpha, dbeta)
	int             IAf, nsize;
	int            *list;
	real           *dx;
	real            dalpha, dbeta;
{

	int             i, ip, gsize;
	logical         isvalid;
	real           *dy;

	CHECK_REAL(IAf, "do_raxpby_C");

	dy = (real *) POINTER(IAf);
	gsize = GSIZE(IAf);

	for (i = 0; i < nsize; i++) {
		ip = list[i];
		isvalid = (1 <= ip) && (ip <= gsize);
		if (!isvalid) {
			printf("** do_raxpby_C: of %s\n", NAME(IAf));
			printf("** invalid  index %d, gsize %d \n", ip, gsize);
		};
		assert(isvalid);

		dy[ip] = dalpha * dx[i] + dbeta * dy[ip];
	};

}





int
do_biaxpby_C(IAf, nsize, istart, ix, ialpha, ibeta)
	int             IAf, nsize, istart;
	int            *ix;
	int             ialpha, ibeta;
{
	int             i, ip, gsize;
	int            *iy;
	logical         isvalid;

	CHECK_INT(IAf, "do_biaxpby_C");

	iy = (int *) POINTER(IAf);
	gsize = GSIZE(IAf);

        ip = istart;
        isvalid = (1 <= ip) && (ip <= gsize);
        if (!isvalid) {
            printf("** do_biaxpby_C: of %s\n", NAME(IAf));
            printf("** invalid  index %d, gsize %d\n", ip, gsize);
            };
        assert(isvalid);


        ip = istart + (nsize-1);
        isvalid = (1 <= ip) && (ip <= gsize);
        if (!isvalid) {
            printf("** do_biaxpby_C: of %s\n", NAME(IAf));
            printf("** invalid  index %d, gsize %d\n", ip, gsize);
            };
        assert(isvalid);


	ip = istart;
	for (i = 0; i < nsize; i++) {

		iy[ip] = ialpha * ix[i] + ibeta * iy[ip];
		ip = ip + 1;
	};
	return (IAf);
}


int
do_bdaxpby_C(IAf, nsize, istart, dx, dalpha, dbeta)
	int             IAf, nsize, istart;
	double         *dx;
	double          dalpha, dbeta;
{
	int             i, ip, gsize;
	logical         isvalid;

	double         *dy;


	CHECK_DOUBLE(IAf, "do_bdaxpby_C");

	dy = (double *) POINTER(IAf);
	gsize = GSIZE(IAf);

	ip = istart;
	for (i = 0; i < nsize; i++) {
		isvalid = (1 <= ip) && (ip <= gsize);
		if (!isvalid) {
			printf("** do_bdaxpby_C: of %s\n", NAME(IAf));
			printf("** invalid  index %d, gsize %d\n", ip, gsize);
		};
		assert(isvalid);

		dy[ip] = dalpha * dx[i] + dbeta * dy[ip];
		ip = ip + 1;
	};
	return (IAf);
}





int
do_braxpby_C(IAf, nsize, istart, dx, dalpha, dbeta)
	int             IAf, nsize, istart;
	real           *dx;
	real            dalpha, dbeta;
{
	int             i, ip, gsize;
	logical         isvalid;

	real           *dy;


	CHECK_REAL(IAf, "do_braxpby_C");

	dy = (real *) POINTER(IAf);
	gsize = GSIZE(IAf);

	ip = istart;
	isvalid = (1 <= ip) && (ip <= gsize);
	if (!isvalid) {
		printf("** do_braxpby_C: of %s\n", NAME(IAf));
		printf("** invalid  index %d, gsize %d\n", ip, gsize);
	};
	assert(isvalid);

	ip = istart + nsize - 1;
	isvalid = (1 <= ip) && (ip <= gsize);
	if (!isvalid) {
		printf("** do_braxpby_C: of %s\n", NAME(IAf));
		printf("** invalid  index %d, gsize %d\n", ip, gsize);
	};
	assert(isvalid);




	ip = istart;
	for (i = 0; i < nsize; i++) {
		dy[ip] = dalpha * dx[i] + dbeta * dy[ip];
		ip = ip + 1;
	};
	return (IAf);
}




int
do_biscatter_C(IAf, nsize, istart, ibuf)
	int             IAf, nsize, istart;
	int            *ibuf;
{

        int             i, ip, gsize;
        int            *isrc;
        logical         isvalid;


        CHECK_INT(IAf, "do_biscatter_C");

        isrc = (int *) POINTER(IAf);
        gsize = GSIZE(IAf);

        ip = istart;
        isvalid = (1 <= ip) && (ip <= gsize);
        if (!isvalid) {
            printf("** do_biscatter_C: of %s\n", NAME(IAf));
            printf("** invalid index value %d, gsize %d\n", ip, gsize);
            };
        assert(isvalid);

        ip = istart + (nsize-1);
        isvalid = (1 <= ip) && (ip <= gsize);
        if (!isvalid) {
            printf("** do_biscatter_C: of %s\n", NAME(IAf));
            printf("** invalid index value %d, gsize %d\n", ip, gsize);
            };
        assert(isvalid);


#if defined(INLINE_BLOCK_COPY)
        ip = istart;
        for (i = 0; i < nsize; i = i + 1) {
		isrc[ip] = ibuf[i];
                ip = ip + 1;
        };
#else

        /* use MEMCPY(dest,src,nbytes) */
        {
        void *src; void *dest; size_t nbytes;


        ip = istart; i = 0;
        src = (void *) (&(ibuf[i]));
        dest = (void *) (&(isrc[ip]));
        nbytes = (size_t) (sizeof(int)*nsize);

        MEMCPY(dest,src,nbytes);
        };


#endif /* defined(INLINE_BLOCK_COPY) */

        return (IAf);

}

int
do_bdscatter_C(IAf, nsize, istart, dbuf)
	int             IAf, nsize, istart;
	double         *dbuf;
{
	double          dalpha, dbeta;
	return (do_bdaxpby_C(IAf, nsize, istart, dbuf,
			     dalpha = (double) 1, dbeta = (double) 0));

}


int
do_brscatter_C(IAf, nsize, istart, dbuf)
	int             IAf, nsize, istart;
	real           *dbuf;
{
        int             i, ip, gsize;
        logical         isvalid;
        real           *dsrc;

        CHECK_REAL(IAf, "do_brscatter_C");

        dsrc = (real *) POINTER(IAf);
        gsize = GSIZE(IAf);

        ip = istart;
        isvalid = (1 <= ip) && (ip <= gsize);
        if (!isvalid) {
                printf("** do_brscatter_C: of %s\n", NAME(IAf));
                printf("** invalid  index %d, gsize %d\n", ip, gsize);
        };
        assert(isvalid);

        ip = istart + nsize - 1;
        isvalid = (1 <= ip) && (ip <= gsize);
        if (!isvalid) {
                printf("** do_brscatter_C: of %s\n", NAME(IAf));
                printf("** invalid  index %d, gsize %d\n", ip, gsize);
        };
        assert(isvalid);



#if defined(INLINE_BLOCK_COPY)

        ip = istart;
        for (i = 0; i < nsize; i = i + 1) {

                dsrc[ip] = dbuf[i];
                ip = ip + 1;
        };

#else

        /* use MEMCPY(dest,src,nbytes) */
        {
        void *src; void *dest; size_t nbytes;


        ip = istart;
        src = (void *) (&(dbuf[0]));
        dest = (void *) (&(dsrc[ip]));
        nbytes = (size_t) (sizeof(real)*nsize);

        MEMCPY(dest,src,nbytes);
        };


#endif /* defined(INLINE_BLOCK_COPY) */


        return (IAf);

}


void
do_iscatter_C(IAf, nsize, list, ibuf)
	int             IAf, nsize;
	int            *list;
	int            *ibuf;
{
	int             ialpha, ibeta;

	do_iaxpby_C(IAf, nsize, list, ibuf,
		      ialpha = 1, ibeta = 0);

}

void
do_dscatter_C(IAf, nsize, list, dbuf)
	int             IAf, nsize;
	int            *list;
	double         *dbuf;
{
	double          dalpha, dbeta;

	do_daxpby_C(IAf, nsize, list, dbuf,
		      dalpha = (double) 1, dbeta = (double) 0);

}



void
do_rscatter_C(IAf, nsize, list, dbuf)
	int             IAf, nsize;
	int            *list;
	real           *dbuf;
{
	real            dalpha, dbeta;

	do_raxpby_C(IAf, nsize, list, dbuf,
		      dalpha = (real) 1, dbeta = (real) 0);

}



void
do_wait(reqid)
	int             reqid;
{				/* do nothing */
}

/* cache related stuff */
void
do_enable(IAf)
	int             IAf;
{				/* do nothing */
}

void
do_disable(IAf)
	int             IAf;
{				/* do nothing */
}

int
do_setchsize(nsize)
	int             nsize;
{
	return (nsize);
}








#if AIX || IBM || RS6K || RIOS
#undef NEED_UNDERSCORE

#else
#define NEED_UNDERSCORE 1

#endif


#if !defined(USE_DOUBLE)
#define USE_DOUBLE 1
#endif


#if defined(NEED_UNDERSCORE)

#define DODESTROY dodestroy_

#define ISCATTER doiscatter_

#define DOWAIT   dowait_
#define DODECLARE dodeclare_

#define BRGATHER dobrgather_
#define BRSCATTER dobrscatter_

#define BDGATHER dobdgather_
#define BDSCATTER dobdscatter_

#define DGATHER dodgather_
#define DSCATTER dodscatter_

#define IGATHER doigather_
#define ISCATTER doiscatter_

#define RGATHER dorgather_
#define RSCATTER dorscatter_

#define BIGATHER dobigather_
#define BISCATTER dobiscatter_

#define DOSETCHSIZE dosetchsize_

#define DO_GSYNC dogsync_
#define DO_DISABLE dodisable_
#define DO_ENABLE doenable_

#define DAXPBY dodaxpby_
#define BDAXPBY dobdaxpby_

#define RAXPBY doraxpby_
#define BRAXPBY dobraxpby_

#define IAXPBY doiaxpby_
#define BIAXPBY dobiaxpby_

#define DOGSYNC dogsync_

#define DOCHECK docheck_


#define DAXPBYZ dodaxpbyz_
#define BDAXPBYZ dobdaxpbyz_
#define RAXPBYZ doraxpbyz_
#define BRAXPBYZ dobraxpbyz_
#define IAXPBYZ doiaxpbyz_
#define BIAXPBYZ dobiaxpbyz_




#else

#define DODESTROY dodestroy

#define ISCATTER doiscatter

#define DOWAIT   dowait
#define DODECLARE dodeclare

#define BRGATHER dobrgather
#define BRSCATTER dobrscatter


#define BDGATHER dobdgather
#define BDSCATTER dobdscatter

#define DGATHER dodgather
#define DSCATTER dodscatter

#define IGATHER doigather
#define ISCATTER doiscatter


#define RGATHER dorgather
#define RSCATTER dorscatter

#define BIGATHER dobigather
#define BISCATTER dobiscatter

#define DOSETCHSIZE dosetchsize

#define DO_GSYNC dogsync
#define DO_DISABLE dodisable
#define DO_ENABLE doenable

#define DAXPBY dodaxpby
#define BDAXPBY dobdaxpby

#define RAXPBY doraxpby
#define BRAXPBY dobraxpby


#define IAXPBY doiaxpby
#define BIAXPBY dobiaxpby

#define DOGSYNC dogsync


#define DOCHECK docheck


#define DAXPBYZ dodaxpbyz
#define BDAXPBYZ dobdaxpbyz
#define RAXPBYZ doraxpbyz
#define BRAXPBYZ dobraxpbyz
#define IAXPBYZ doiaxpbyz
#define BIAXPBYZ dobiaxpbyz


#endif				/* if NEED_UNDERSCORE */

/* f2c */

void
DOGSYNC()
{ 
	return;
}

void DOCHECK()
{
	return;
}





int
BDGATHER(IAf, nsize, istart, dbuf)
	int            *IAf;
	int            *nsize;
	int            *istart;
	double         *dbuf;
{
	return (do_bdgather_C(*IAf, *nsize, *istart, dbuf));
}

void
DAXPBY(IAf, nsize, list, dbuf, alpha, beta)
	int            *IAf;
	int            *nsize;
	int            *list;
	double         *dbuf;
	double         *alpha;
	double         *beta;
{
	do_daxpby_C(*IAf, *nsize, list, dbuf, *alpha, *beta);
}

void
BDAXPBY(IAf, nsize, istart, dbuf, alpha, beta)
	int            *IAf;
	int            *nsize;
	int            *istart;
	double         *dbuf;
	double         *alpha;
	double         *beta;
{
	do_bdaxpby_C(*IAf, *nsize, *istart, dbuf, *alpha, *beta);
}


void
DAXPBYZ(IAf, nsize, list, dbuf, alpha, beta)
        int            *IAf;
        int            *nsize;
        int            *list;
        double         *dbuf;
        double         *alpha;
        double         *beta;
{
        do_daxpbyz_C(*IAf, *nsize, list, dbuf, *alpha, *beta);
}

void
BDAXPBYZ(IAf, nsize, istart, dbuf, alpha, beta)
        int            *IAf;
        int            *nsize;
        int            *istart;
        double         *dbuf;
        double         *alpha;
        double         *beta;
{
        do_bdaxpbyz_C(*IAf, *nsize, *istart, dbuf, *alpha, *beta);
}

int
BRGATHER(IAf, nsize, istart, dbuf)
	int            *IAf;
	int            *nsize;
	int            *istart;
	real           *dbuf;
{
	return (do_brgather_C(*IAf, *nsize, *istart, dbuf));
}

void
RAXPBY(IAf, nsize, list, dbuf, alpha, beta)
	int            *IAf;
	int            *nsize;
	int            *list;
	real           *dbuf;
	real           *alpha;
	real           *beta;
{
	do_raxpby_C(*IAf, *nsize, list, dbuf, *alpha, *beta);
}

void
BRAXPBY(IAf, nsize, istart, dbuf, alpha, beta)
	int            *IAf;
	int            *nsize;
	int            *istart;
	real           *dbuf;
	real           *alpha;
	real           *beta;
{
	do_braxpby_C(*IAf, *nsize, *istart, dbuf, *alpha, *beta);
}


void
RAXPBYZ(IAf, nsize, list, dbuf, alpha, beta)
        int            *IAf;
        int            *nsize;
        int            *list;
        real           *dbuf;
        real           *alpha;
        real           *beta;
{
        do_raxpbyz_C(*IAf, *nsize, list, dbuf, *alpha, *beta);
}

void
BRAXPBYZ(IAf, nsize, istart, dbuf, alpha, beta)
        int            *IAf;
        int            *nsize;
        int            *istart;
        real           *dbuf;
        real           *alpha;
        real           *beta;
{
        do_braxpbyz_C(*IAf, *nsize, *istart, dbuf, *alpha, *beta);
}





void 
IAXPBY(IAf, nsize, list, ibuf, ialpha, ibeta)
        int            *IAf;
        int            *nsize;
        int            *list;
	int		*ibuf;
	int		*ialpha;
	int		*ibeta;
{
        do_iaxpby_C(*IAf, *nsize, list, ibuf, *ialpha, *ibeta);
}


void
BIAXPBY(IAf, nsize, istart, ibuf, ialpha, ibeta)
        int            *IAf;
        int            *nsize;
        int            *istart;
        int             *ibuf;
        int             *ialpha;
        int             *ibeta;
{
        do_biaxpby_C(*IAf, *nsize, *istart, ibuf, *ialpha, *ibeta);
}


void
IAXPBYZ(IAf, nsize, list, ibuf, ialpha, ibeta)
        int            *IAf;
        int            *nsize;
        int            *list;
        int             *ibuf;
        int             *ialpha;
        int             *ibeta;
{
        do_iaxpbyz_C(*IAf, *nsize, list, ibuf, *ialpha, *ibeta);
}


void
BIAXPBYZ(IAf, nsize, istart, ibuf, ialpha, ibeta)
        int            *IAf;
        int            *nsize;
        int            *istart;
        int             *ibuf;
        int             *ialpha;
        int             *ibeta;
{
        do_biaxpbyz_C(*IAf, *nsize, *istart, ibuf, *ialpha, *ibeta);
}




void
DOWAIT(reqid)
	int            *reqid;
{
	do_wait(*reqid);
}

void
DSCATTER(IAf, nsize, list, dbuf)
	int            *IAf;
	int            *nsize;
	int            *list;
	double         *dbuf;
{
	do_dscatter_C(*IAf, *nsize, list, dbuf);
}


void
RSCATTER(IAf, nsize, list, dbuf)
	int            *IAf;
	int            *nsize;
	int            *list;
	real           *dbuf;
{
	do_rscatter_C(*IAf, *nsize, list, dbuf);
}


void
ISCATTER(IAf, nsize, list, buf)
	int            *IAf;
	int            *nsize;
	int            *list;
	int            *buf;
{
	do_iscatter_C(*IAf, *nsize, list, buf);
}

void
BISCATTER(IAf, nsize, istart, buf)
	int            *IAf;
	int            *nsize;
	int            *istart;
	int            *buf;
{
	do_biscatter_C(*IAf, *nsize, *istart, buf);
}

void
BDSCATTER(IAf, nsize, istart, buf)
	int            *IAf;
	int            *nsize;
	int            *istart;
	double         *buf;
{
	do_bdscatter_C(*IAf, *nsize, *istart, buf);
}



void
BRSCATTER(IAf, nsize, istart, buf)
	int            *IAf;
	int            *nsize;
	int            *istart;
	real           *buf;
{
	do_brscatter_C(*IAf, *nsize, *istart, buf);
}


void
DODESTROY(IAf)
	int            *IAf;
{
	do_destroy(*IAf);
}

void
DODECLARE(IAf, name, gsize, ctype, pagesize, blocksize)
	int            *IAf;
	char           *name;
	int            *gsize;
	char           *ctype;
	int            *pagesize;
	int            *blocksize;
{
	do_declare(IAf, name, *gsize, ctype, *pagesize, *blocksize);
}


void
IGATHER(IAf, nsize, list, ibuf)
	int            *IAf;
	int            *nsize;
	int            *list;
	int            *ibuf;
{
	do_igather_C(*IAf, *nsize, list, ibuf);
}

void
DGATHER(IAf, nsize, list, dbuf)
	int            *IAf;
	int            *nsize;
	int            *list;
	double         *dbuf;
{
	do_dgather_C(*IAf, *nsize, list, dbuf);
}


void
RGATHER(IAf, nsize, list, dbuf)
	int            *IAf;
	int            *nsize;
	int            *list;
	real           *dbuf;
{
	do_rgather_C(*IAf, *nsize, list, dbuf);
}


int
BIGATHER(IAf, nsize, istart, ibuf)
	int            *IAf;
	int            *nsize;
	int            *istart;
	int            *ibuf;
{
	return (do_bigather_C(*IAf, *nsize, *istart, ibuf));
}



void
DO_ENABLE(IAf)
	int            *IAf;
{
	do_enable(*IAf);
}

void
DO_DISABLE(IAf)
	int            *IAf;
{
	do_disable(*IAf);
}

int
DOSETCHSIZE(nsize)
	int            *nsize;
{
	return (do_setchsize(*nsize));
}


/* C to c translation */

#if !defined(LIST_SIZE)
#define LIST_SIZE (8*1024)
#endif

#define DECLARE_STORAGE() \
	 \
        int             stack_list[LIST_SIZE]; \
        int            *newlist; 


#define GET_STORAGE(newlist,nsize) \
    { \
	logical use_stack; \
        use_stack = (nsize <= LIST_SIZE); \
        if (use_stack) { \
                newlist = (int *) (&(stack_list[0])); \
        } else { \
                MEMALLOC(newlist, (int *), (nsize * sizeof(int)) ); \
        }; \
    };

#define CONVERT_LIST(newlist,list,nsize) \
   {  \
	int i; \
 \
        /* convert from C [0:(nsize-1)] to Fortran [1:nsize]  */ \
        for (i = 0; i < nsize; i++) { \
                newlist[i] = list[i] + 1; \
        }; \
   };


#define FREE_STORAGE(newlist) \
   { \
	logical use_stack; \
 \
	use_stack = (nsize <= LIST_SIZE); \
 \
        /* deallocate storage */ \
 \
        if (use_stack) { \
                /* do nothing */ \
        } else { \
                MEMFREE(newlist); \
        }; \
   }





void do_iaxpby( Iaf, nsize, list, xvec, 
		ialpha,ibeta )

int Iaf, nsize, *list;
int *xvec, ialpha,ibeta;
{



	DECLARE_STORAGE();

	GET_STORAGE(newlist,nsize);

	CONVERT_LIST(newlist,list,nsize);

	do_iaxpby_C(Iaf, nsize, newlist, xvec,
		    ialpha, ibeta);

	FREE_STORAGE(newlist);

}


void
do_daxpby(Iaf, nsize, list, xvec,
          dalpha, dbeta)

int Iaf, nsize, *list;
dfloat *xvec, dalpha,dbeta;
{



        DECLARE_STORAGE();

        GET_STORAGE(newlist,nsize);

        CONVERT_LIST(newlist,list,nsize);

        do_daxpby_C(Iaf, nsize, newlist, xvec,
                    dalpha, dbeta);

        FREE_STORAGE(newlist);

}



void
do_raxpby(Iaf, nsize, list, xvec,
          dalpha, dbeta)

int Iaf, nsize, *list;
float *xvec, dalpha,dbeta;


{



        DECLARE_STORAGE();

        GET_STORAGE(newlist,nsize);

        CONVERT_LIST(newlist,list,nsize);

        do_raxpby_C(Iaf, nsize, newlist, xvec,
                    dalpha, dbeta);

        FREE_STORAGE(newlist);

}



void
do_biaxpby(  Iaf, nsize, idxstart,
		xvec,   alpha, beta )

int Iaf,nsize,idxstart;
int *xvec,  alpha,beta;
{

	/* convert from C [0:(nsize-1)] to Fortran [1:nsize]  */
	do_biaxpby_C( Iaf, nsize, idxstart+1, 	xvec,alpha,beta ) ;
}


void
do_bdaxpby(  Iaf, nsize, idxstart,
                xvec,   alpha, beta )

int Iaf,nsize,idxstart;
dfloat *xvec,  alpha,beta;
{

        /* convert from C [0:(nsize-1)] to Fortran [1:nsize]  */
        do_bdaxpby_C( Iaf, nsize, idxstart+1,   xvec,alpha,beta ) ;
}


void
do_braxpby(  Iaf, nsize, idxstart,
                xvec,   alpha, beta )

int Iaf,nsize,idxstart;
float *xvec,  alpha,beta;
{

        /* convert from C [0:(nsize-1)] to Fortran [1:nsize]  */
        do_braxpby_C( Iaf, nsize, idxstart+1,   xvec,alpha,beta );
}


void do_iaxpbyz( Iaf, nsize, list, xvec, 
		ialpha,ibeta )

int Iaf, nsize, *list;
int *xvec, ialpha,ibeta;
{



	DECLARE_STORAGE();

	GET_STORAGE(newlist,nsize);

	CONVERT_LIST(newlist,list,nsize);

	do_iaxpbyz_C(Iaf, nsize, newlist, xvec,
		    ialpha, ibeta);

	FREE_STORAGE(newlist);

}


void
do_daxpbyz(Iaf, nsize, list, xvec,
          dalpha, dbeta)

int Iaf, nsize, *list;
dfloat *xvec, dalpha,dbeta;
{



        DECLARE_STORAGE();

        GET_STORAGE(newlist,nsize);

        CONVERT_LIST(newlist,list,nsize);

        do_daxpbyz_C(Iaf, nsize, newlist, xvec,
                    dalpha, dbeta);

        FREE_STORAGE(newlist);

}



void
do_raxpbyz(Iaf, nsize, list, xvec,
          dalpha, dbeta)

int Iaf, nsize, *list;
float *xvec, dalpha,dbeta;


{



        DECLARE_STORAGE();

        GET_STORAGE(newlist,nsize);

        CONVERT_LIST(newlist,list,nsize);

        do_raxpbyz_C(Iaf, nsize, newlist, xvec,
                    dalpha, dbeta);

        FREE_STORAGE(newlist);

}



void
do_biaxpbyz(  Iaf, nsize, idxstart,
		xvec,   alpha, beta )

int Iaf,nsize,idxstart;
int *xvec,  alpha,beta;
{

	/* convert from C [0:(nsize-1)] to Fortran [1:nsize]  */
	do_biaxpbyz_C( Iaf, nsize, idxstart+1, 	xvec,alpha,beta ) ;
}


void
do_bdaxpbyz(  Iaf, nsize, idxstart,
                xvec,   alpha, beta )

int Iaf,nsize,idxstart;
dfloat *xvec,  alpha,beta;
{

        /* convert from C [0:(nsize-1)] to Fortran [1:nsize]  */
        do_bdaxpbyz_C( Iaf, nsize, idxstart+1,   xvec,alpha,beta ) ;
}


void
do_braxpbyz(  Iaf, nsize, idxstart,
                xvec,   alpha, beta )

int Iaf,nsize,idxstart;
float *xvec,  alpha,beta;
{

        /* convert from C [0:(nsize-1)] to Fortran [1:nsize]  */
        do_braxpbyz_C( Iaf, nsize, idxstart+1,   xvec,alpha,beta );
}



void
do_gather( Iaf, nsize, list,     xvec )
int Iaf,nsize, *list;
void *xvec;
{

	DECLARE_STORAGE();

	GET_STORAGE(newlist,nsize);

	CONVERT_LIST(newlist,list,nsize );


	if (ISINTEGER(Iaf)) {
	  do_igather_C( Iaf, nsize, newlist,   (int *) xvec );
	  }
	else if (ISREAL(Iaf)) {
	  do_rgather_C( Iaf, nsize, newlist,   (float *) xvec );
	  }
	else if (ISDOUBLE(Iaf)) {
	  do_dgather_C( Iaf, nsize, newlist,  (dfloat *) xvec );
	  }
	else {
	   assert(FALSE); /* gather for char type not implemented */
	   };

	FREE_STORAGE(newlist);
}


void
do_scatter( Iaf, nsize, list,     xvec )
int Iaf,nsize, *list;
void *xvec;
{

        DECLARE_STORAGE();

        GET_STORAGE(newlist,nsize);

        CONVERT_LIST(newlist,list,nsize );

        if (ISINTEGER(Iaf)) {
          do_iscatter_C( Iaf, nsize, newlist,   (int *) xvec );
          }
        else if (ISREAL(Iaf)) {
          do_rscatter_C( Iaf, nsize, newlist,   (float *) xvec );
          }
        else if (ISDOUBLE(Iaf)) {
          do_dscatter_C( Iaf, nsize, newlist,  (dfloat *) xvec );
          }
        else {
           assert(FALSE); /* scatter for char type not implemented */
           };


        FREE_STORAGE(newlist);
}


int do_bgather_C( Iaf, nsize, idxstart, xvec )
int Iaf, nsize, idxstart;
void *xvec;
{
	assert( VALIDIAF( Iaf ));

	if (ISINTEGER(Iaf)) {
	   return( do_bigather_C( Iaf, nsize, idxstart, (int *) xvec ) );
	   }
	else if (ISREAL( Iaf )) {
	  return( do_brgather_C( Iaf, nsize, idxstart, (float *) xvec ) );
	  }
	else if (ISDOUBLE(Iaf)) {
	  return( do_bdgather_C( Iaf, nsize, idxstart, (dfloat *) xvec ) );
	  }
	else {
	   assert( FALSE ); /* gather for char type not implemented yet */
	   };

	return( NULL );
}


void do_bscatter_C( Iaf, nsize, idxstart, xvec )
int Iaf, nsize, idxstart;
void *xvec;
{
        assert( VALIDIAF( Iaf ));

        if (ISINTEGER(Iaf)) {
           do_biscatter_C ( Iaf, nsize, idxstart, (int *) xvec );
           }
        else if (ISREAL( Iaf )) {
          do_brscatter_C ( Iaf, nsize, idxstart, (float *) xvec );
          }
        else if (ISDOUBLE(Iaf)) {
          do_bdscatter_C ( Iaf, nsize, idxstart, (dfloat *) xvec );
          }
        else {
           assert( FALSE ); /* scatter for char type not implemented yet */
           };


}



int
do_bgather( Iaf, nsize, idxstart,  xvec )
int Iaf, nsize, idxstart;
void *xvec;
{

        /* convert from C [0:(nsize-1)] to Fortran [1:nsize]  */
	return( do_bgather_C( Iaf, nsize, idxstart+1,   xvec ) );
}



void
do_bscatter( Iaf, nsize, idxstart,  xvec )
int Iaf, nsize, idxstart;
void *xvec;
{

        /* convert from C [0:(nsize-1)] to Fortran [1:nsize]  */
        do_bscatter_C( Iaf, nsize, idxstart+1,   xvec );
}


