/***********************************************************************
* DOLIB/DONIO Version 0.0 (8/24/94)                                    *       *
*  Software to emulate shared memory on distributed memory environments*
* written by:                                                          *
*  Ed D'Azevedo and Charles Romine of Oak Ridge National Laboratory    *
*                                                                      *
* Questions and comments should be directed to                         *
*      efdazedo@msr.epm.ornl.gov or romine@msr.epm.ornl.gov            *
*                                                                      *
*  Please notify and acknowledge the authors in any research or        *
*  publications utilizing DOLIB/DONIO or any part of the code.         *
*                                                                      *
* NOTICE: Neither the institution nor the author make any              *
*  representations about the suitability of this software for any      *
*  purpose. This software is provided "as is", without express or      *
*  implied warranty.                                                   *
************************************************************************/

#include <stdio.h>
#include <assert.h>

#include "ipx.h"
#include "globals.h"
#include "wait_block.h"

#if !defined(GET_SUCCESS)
#define GET_SUCCESS 0
#endif

#if !defined(DEFAULT_MAX_PAGES)
#define DEFAULT_MAX_PAGES (8*1024)
#endif

int
do_setgather(int Iaf, int nsize, int *list,
	     int nsets, int *gindex, int *start_list, int *size_list,
	     logical is_sorted, logical use_compress, void *buf)
{
#include "get_page.h"

	extern void     isort(int *pix, int n, int *piperm);

	extern IDTYPE   iget_page(int Iaf, int page_no, void **ppage_ptr);

	logical         is_integer, is_dfloat, is_char, is_real;
	int            *ibuf;
	dfloat         *dbuf;
	char           *cbuf;
	float          *rbuf;

	int            *iptr;
	dfloat         *dptr;
	char           *cptr;
	float          *rptr;

	char           *csrc;
	char           *cdest;

	logical         inorder, is_valid, isfound, spaceok, is_samepage, is_page_ready,
	                alldone, is_done;
	int             ii, iip1, buf_ptr, ihi, i, j, istart, isize, gni, offset,
	                ndone, nbytes, gsize, ipage, page_hi, page_lo, cur_page_no,
	                page_size;
	int             wait_id, max_pages;

	struct wait_node *ip;
	struct Iarray_node *anp;

	int             npages;
	int            *iperm;

	int            *pageno_array;
	int            *sorted_pageno_array;
	void          **pageptr_array;
	void          **sorted_pageptr_array;
	void           *page_ptr;

	int            *gid_array;
	int            *sorted_gid_array;

	logical        *userbuf_array;
	logical        *sorted_userbuf_array;

#define MAX_STACK_PAGES (8*1024)
	int             stack_pageno_array[MAX_STACK_PAGES];
	int             stack_gid_array[MAX_STACK_PAGES];
	logical         stack_userbuf_array[MAX_STACK_PAGES];
	void           *stack_pageptr_array[MAX_STACK_PAGES];

	logical         use_stack;
	logical         userbuf;

	/******* start execution ***********/

	INTERRUPT_CHECK();

	is_valid = ((1 <= Iaf) && (Iaf <= MAX_GLOBAL_ARRAY));
	ASSERT(is_valid,"do_setgather(): Iaf of %d out of range\n", Iaf);

	anp = Global_array[Iaf];
	ASSERT(anp != NULL,"do_setgather(): Iaf of %d indexes null entry\n", Iaf);

	page_size = anp->page_size;
	ASSERT(page_size >= 1,"do_setgather(): page_size of %d out of range\n", page_size);

	gsize = anp->gsize;
	ASSERT(gsize >= 1,"do_setgather(): gsize of %d out of range\n", gsize);

	is_valid = ((1 <= Iaf) && (Iaf <= MAX_GLOBAL_ARRAY));
	ASSERT(is_valid,"do_setgather(): Iaf of %d out of range\n", Iaf);

	is_integer = (anp->type == INTEGER);
	is_dfloat = (anp->type == REAL8);
	is_char = (anp->type == CHAR);
	is_real = (anp->type == REAL);

	is_valid = (is_integer || is_dfloat || is_char || is_real);
	ASSERT(is_valid,"do_setgather(): global array type not valid\n", 0);

	ibuf = NULL;
	dbuf = NULL;
	cbuf = NULL;
	rbuf = NULL;

	if (is_char) {
		cbuf = (char *) buf;
	} else if (is_integer) {
		ibuf = (int *) buf;
	} else if (is_dfloat) {
		dbuf = (dfloat *) buf;
	} else if (is_real) {
		rbuf = (float *) buf;
	} else {
		ASSERT(FALSE,"do_setgather(): global array type not valid\n", 0);
	}

	/* find out what pages are needed */
	npages = 0;
	if (use_compress) {
		max_pages = 1 + min(DEFAULT_MAX_PAGES, nsets);
	} else {
		max_pages = 1 + min(DEFAULT_MAX_PAGES, nsize);
	}

	use_stack = (max_pages <= MAX_STACK_PAGES);
	if (use_stack) {
		pageno_array = (int *) (&(stack_pageno_array[0]));

		gid_array = (int *) (&(stack_gid_array[0]));
		userbuf_array = (logical *) (&(stack_userbuf_array[0]));

		pageptr_array = (void **) (&(stack_pageptr_array[0]));
	} else {
		MEMALLOC(pageno_array, (int *), max_pages * sizeof(int));

		MEMALLOC(gid_array, (int *), max_pages * sizeof(int));
		MEMALLOC(userbuf_array, (logical *), 
					max_pages * sizeof(logical));

		MEMALLOC(pageptr_array, (void **), max_pages * sizeof(void *));
	}

	page_lo = -1;
	page_hi = -1;
	buf_ptr = 0;

	if (use_compress) {
		ihi = nsets;
	} else {
		ihi = nsize;
		isize = 1;
	}

	for (i = 0; i < ihi; i++) {
		if (use_compress) {
			istart = start_list[i];
			isize = size_list[i];

			gni = gindex[i];
		} else {
			gni = list[i];
		}

		is_samepage = ((page_lo <= gni) && (gni <= page_hi));
		if (!is_samepage) {

			cur_page_no = (gni - 1) / page_size;

			isfound = FALSE;
			if (!is_sorted) {
				/*
				 * if list is not sorted, there is a chance
				 * of duplicate pages, need search to be sure
				 */

				/* backward search, LRU strategy */
				for (j = npages - 1; j >= 0; j--) {
					isfound = (pageno_array[j] == cur_page_no);
					if (isfound) {
						ipage = j;
						break;
					}
				}	/* end for(j) */
			}	/* end if (!is_sorted) */
			if (!isfound) {
				/* add this page */
				npages += 1;
				spaceok = (npages <= max_pages);
				ASSERT(spaceok,"do_setgather(): npages of %d out of range\n", npages);

				ipage = npages - 1;
				pageno_array[ipage] = cur_page_no;

				/*
				 * Ed Oct 6, set pageptr_array[ipage] to null
				 * first
				 */

				pageptr_array[ipage] = NULL;

				userbuf = use_compress &&
					(isize == page_size) &&
					(MOD(gni-1,page_size)==0);

				if (userbuf) {

					/*
					 * drain data directly into user
					 * buffer
					 */

					if (is_dfloat) {
						pageptr_array[ipage] =
						    (void *) (&(dbuf[buf_ptr]));
					} else if (is_char) {
						pageptr_array[ipage] =
						    (void *) (&(cbuf[buf_ptr]));
					} else if (is_integer) {
						pageptr_array[ipage] =
						    (void *) (&(ibuf[buf_ptr]));
					} else if (is_real) {
						pageptr_array[ipage] =
						    (void *) (&(rbuf[buf_ptr]));
					} else {
						ASSERT(FALSE,"do_setgather(): global array type not valid\n", 0);	/* impossible */
					};

				};


				userbuf_array[ipage] = userbuf;
				gid_array[ipage] =
					iget_page(Iaf, cur_page_no,
					(void *) (&(pageptr_array[ipage])));
			};

			page_lo = cur_page_no * page_size + 1;
			page_hi = page_lo + page_size - 1;

			page_ptr = pageptr_array[ipage];
			ASSERT(page_ptr != NULL,"do_setgather(): page_ptr is null\n", 0);

			if (is_dfloat) {
				dptr = (dfloat *) page_ptr;
			} else if (is_integer) {
				iptr = (int *) page_ptr;
			} else if (is_char) {
				cptr = (char *) page_ptr;
			} else if (is_real) {
				rptr = (float *) page_ptr;
			} else {
				ASSERT(FALSE,"do_setgather(): global array type not valid\n", 0);	/* impossible */
			}
		}		/* end if (!is_samepage) */
		is_page_ready = (gid_array[ipage] == GET_SUCCESS);
		if (is_page_ready) {

			/* try to decouple interaction with the cache */
			/* do copying right away */

			offset = (gni - page_lo);
#if defined(DEBUG)
			ASSERT(offset == MOD(gni - 1, page_size),"do_setgather(): offset of %d is a mismatch\n", offset);
			ASSERT((0 <= offset) && (offset < page_size),"do_setgather(): offset of %d out of range\n", offset);
#endif
			if (is_dfloat) {
				csrc = (char *) (&(dptr[offset]));
				cdest = (char *) (&(dbuf[buf_ptr]));

				nbytes = isize * sizeof(dfloat);
			} else if (is_char) {
				csrc = (char *) (&(cptr[offset]));
				cdest = (char *) (&(cbuf[buf_ptr]));

				nbytes = isize * sizeof(char);
			} else if (is_integer) {
				csrc = (char *) (&(iptr[offset]));
				cdest = (char *) (&(ibuf[buf_ptr]));

				nbytes = isize * sizeof(int);
			} else if (is_real) {
				csrc = (char *) (&(rptr[offset]));
				cdest = (char *) (&(rbuf[buf_ptr]));

				nbytes = isize * sizeof(float);
			} else {
				ASSERT(FALSE,"do_setgather(): global array type not valid\n", 0);	/* impossible */
			}

			/* Ed Oct 6, avoid unnecessary copy */

			if (csrc != cdest) {
				BCOPY(csrc, cdest, nbytes);
			};
		}		/* end if (is_page_ready) */
		buf_ptr += isize;	/* always increment */
	}			/* end for(i) */

	/*
	 * Note: out standing iget_page request MUST be associated with
	 * "get_array" messages, hence no longer dependent on the cache
	 */

	for (i = 0; i < npages; i++) {
		is_done = (gid_array[i] == GET_SUCCESS);
		if (is_done) {
			return_page(Iaf, pageno_array[i],
				    pageptr_array[i], userbuf_array[i]);
			pageptr_array[i] = NULL;
		};
	};			/* end for */


	/* allocate a wait_block */

	wait_id = allocwait();

	/*
	 * check to see if all page requests are already satisfied. to avoid
	 * expensive memory allocation and copying
	 */

	ndone = 0;
	for (i = 0; i < npages; i++) {
		is_done = (gid_array[i] == GET_SUCCESS);
		if (is_done) {
			ndone += 1;
		}
	}

	alldone = (ndone == npages);

	MEMALLOC(ip, (struct wait_node *), sizeof(struct wait_node));
	Gwait_node_array[wait_id] = ip;

	ip->Iaf = Iaf;
	ip->buf = (void *) buf;

	if (alldone) {
		/* stuff fake values to avoid doing work */
		ip->nsize = 0;
		ip->nsets = 0;
		ip->use_compress = TRUE;
		ip->list = NULL;
		ip->start_list = NULL;
		ip->size_list = NULL;
		ip->gindex = NULL;

		ip->npages = 0;
		ip->pageno_array = NULL;
		ip->pageptr_array = NULL;

		ip->gid_array = NULL;
		ip->userbuf_array = NULL;

		/* Ed dec 1, correct a memory leak */
		/* may need to deallocated memory */
		/* return (wait_id);  */
		goto CLEAN_UP;
	}
	ip->use_compress = use_compress;
	ip->nsize = nsize;

	if (use_compress) {
		ip->list = NULL;/* the original index list is not stored */

		ip->nsets = nsets;

		/* reminder: bcopy( src, dest, nbytes ) */

		nbytes = sizeof(int) * nsets;
		MEMALLOC(ip->start_list, (int *), nbytes);
		BCOPY((char *) start_list, ip->start_list, nbytes);

		MEMALLOC(ip->size_list, (int *), nbytes);
		BCOPY((char *) size_list, ip->size_list, nbytes);

		MEMALLOC(ip->gindex, (int *), nbytes);
		BCOPY((char *) gindex, ip->gindex, nbytes);
	} else {

		/* make a copy of original index list */

		ip->nsets = 0;

		ip->start_list = NULL;
		ip->size_list = NULL;
		ip->gindex = NULL;

		nbytes = sizeof(int) * nsize;
		MEMALLOC(ip->list, (int *), nbytes);
		BCOPY((char *) list, (char *) ip->list, nbytes);
	};

	ip->npages = npages;
	/* prepare sorted list of pageno */

	MEMALLOC(sorted_pageno_array, (int *), npages * sizeof(int));

	MEMALLOC(sorted_gid_array, (int *), npages * sizeof(int));
	MEMALLOC(sorted_userbuf_array, (logical *), npages * sizeof(logical));

	MEMALLOC(sorted_pageptr_array, (void **), npages * sizeof(void *));

	if (!is_sorted) {
		/* perform sort */
		MEMALLOC(iperm, (int *), npages * sizeof(int));
		isort(pageno_array, npages, iperm);

		/* double check */
		for (i = 0; i < npages; i++) {
			is_valid = ((0 <= iperm[i]) && (iperm[i] < npages));
			ASSERT(is_valid,"do_setgather(): iperm[i] of %d out of range\n", iperm[i]);
		}
		for (i = 0; i < (npages - 1); i++) {
			ii = iperm[i];
			iip1 = iperm[i + 1];

			/* all pageno's should be unique, strict < test */
			inorder = (pageno_array[ii] < pageno_array[iip1]);
			ASSERT(inorder,"do_setgather(): pages not in order after sort\n", 0);
		}

		for (i = 0; i < npages; i++) {
			ii = iperm[i];

			sorted_pageno_array[i] = pageno_array[ii];

			sorted_gid_array[i] = gid_array[ii];
			sorted_userbuf_array[i] = userbuf_array[ii];

			sorted_pageptr_array[i] = pageptr_array[ii];
		}		/* end for */
	} else {

		BCOPY((char *) pageno_array,
		      (char *) sorted_pageno_array, npages * sizeof(int));

		BCOPY((char *) gid_array,
		      (char *) sorted_gid_array, npages * sizeof(int));
		BCOPY((char *) userbuf_array,
		      (char *) sorted_userbuf_array, npages * sizeof(int));

		BCOPY((char *) pageptr_array,
		      (char *) sorted_pageptr_array,
		      sizeof(void *) * npages);
	}			/* end if (!is_sorted) */

	ip->gid_array = sorted_gid_array;
	ip->userbuf_array = sorted_userbuf_array;

	ip->pageptr_array = sorted_pageptr_array;
	ip->pageno_array = sorted_pageno_array;

	/* clean up */
	if (!is_sorted) {
		MEMFREE(iperm);
	};

 CLEAN_UP:
	if (use_stack) {
		/* do nothing, space reclaimed on stack */
	} else {
		MEMFREE(pageno_array);
		MEMFREE(pageptr_array);
		MEMFREE(gid_array);
		MEMFREE(userbuf_array);
	}

	return (wait_id);
}
