/************************************************************************
 * DOLIB/DONIO Version 0.0 (8/24/94)                                    *
 *  Software to emulate shared memory on distributed memory environments*
 * written by:                                                          *
 *  Ed D'Azevedo and Charles Romine of Oak Ridge National Laboratory    *
 *                                                                      *
 * Questions and comments should be directed to                         *
 *      efdazedo@msr.epm.ornl.gov or romine@msr.epm.ornl.gov            *
 *                                                                      *
 *  Please notify and acknowledge the authors in any research or        *
 *  publications utilizing DOLIB/DONIO or any part of the code.         *
 *                                                                      *
 * NOTICE: Neither the institution nor the author make any              *
 *  representations about the suitability of this software for any      *
 *  purpose. This software is provided "as is", without express or      *
 *  implied warranty.                                                   *
 ************************************************************************/


#include "stdinc.h"
#include "message.h"

#include <stdio.h>
#include <assert.h>
#include <string.h>
#include "globals.h"


#define MEMSET memset

#include <ctype.h>

#if !defined(strequal)
#define strequal( str1, str2) ( strcmp( str1, str2) == 0)
#endif

#if !defined(MOD)
#define MOD(x,y) ( (x) % (y) )
#endif



/*
 * DODECLARE -- declare distributed array.  Allocates space across the
 * processors based on gsize.
 */

void
do_declare(int *Iaf, char *name, int gsize,
	   char *ctype, int page_size, int block_size)
{
  int             i;
  logical         is_found, is_valid;
  logical         is_integer, is_dfloat, is_char, is_real;

  struct Iarray_node *anp;
  int             total_pages;
  int             total_blocks;
  int             nblocks;
  int             nbytes;
  int             page_shift, block_shift;

  extern int      nproc;
  extern int      myid;

  int             name_len;
  char           *src;
  char           *dest;

  /* --------- */

  INTERRUPT_CHECK();

  GSYNC();
  finish(0);		/* wait for all IPX routines to complete */
  GSYNC();

  ASSERT(name != NULL,
	 "do_declare(): attempt to declare nameless global array\n", 0);
  ASSERT(ctype != NULL,
	 "do_declare(): attempt to declare typeless global array\n", 0);

  /* find an empty slot */

  for (i = 1; i <= MAX_GLOBAL_ARRAY; i++) {
    is_found = (Global_array[i] == NULL);
    if (is_found) {
      break;
    }
  }

  ASSERT(is_found, "do_declare(): empty slot not found for array %s\n", name);
  *Iaf = i;

  MEMALLOC(anp, (struct Iarray_node *), sizeof(struct Iarray_node));

  ASSERT(anp != NULL, "do_declare(): MEMALLOC failed for anp\n", 0);
  Global_array[i] = anp;

  src = name;
  name_len = strlen(name);
  ASSERT(name_len > 0,
	 "do_declare(): zero-length name for global array invalid\n", 0);

  MEMALLOC(dest, (char *), (name_len + 1));
  ASSERT(dest != NULL, "do_declare(): MEMALLOC failed for dest\n", 0);

  for (i = 0; i < name_len; i++) {
    dest[i] = src[i];
  }
  dest[name_len] = (char) 0;

  anp->name = (char *) dest;

  ASSERT(gsize > 0, "do_declare(): gsize of %d out of range\n", gsize);
  anp->gsize = gsize;

  is_integer = (strequal(ctype, "integer") ||
		strequal(ctype, "INTEGER") ||
		strequal(ctype, "int") ||
		strequal(ctype, "INT"));
  is_dfloat = (strequal(ctype, "real*8") ||
	       strequal(ctype, "REAL*8") ||
	       strequal(ctype, "double precision") ||
	       strequal(ctype, "DOUBLE PRECISION") ||
	       strequal(ctype, "double") ||
	       strequal(ctype, "DOUBLE"));
  is_char = (strequal(ctype, "char") ||
	     strequal(ctype, "CHAR") ||
	     strequal(ctype, "character") ||
	     strequal(ctype, "CHARACTER") ||
	     strequal(ctype, "byte") ||
	     strequal(ctype, "BYTE"));
  is_real = (strequal(ctype, "real") ||
	     strequal(ctype, "REAL") ||
	     strequal(ctype, "float") ||
	     strequal(ctype, "FLOAT") ||
	     strequal(ctype, "real*4") ||
	     strequal(ctype, "REAL*4"));

  is_valid = (is_integer || is_dfloat || is_char || is_real);
  ASSERT(is_valid, "do_declare(): global array type not valid\n", 0);

  if (is_integer) {
    anp->type = INTEGER;
  } else if (is_dfloat) {
    anp->type = REAL8;
  } else if (is_char) {
    anp->type = CHAR;
  } else if (is_real) {
    anp->type = REAL;
  } else {
    ASSERT(FALSE, "do_declare(): global array type not valid\n", 0);
  }
  ASSERT(page_size > 0,
	 "do_declare(): page_size of %d out of bounds\n", page_size);
  anp->page_size = page_size;
  page_shift = compute_shift(page_size);
  anp->page_shift = page_shift;

  ASSERT(block_size > 0,
	 "do_declare(): block_size of %d out of bounds\n", block_size);
  anp->block_size = block_size;
  block_shift = compute_shift(block_size);
  anp->block_shift = block_shift;

  anp->cache_is_on = DEFAULT_CACHE_FLAG;

  /* compute global and local storage requirements */

  total_pages = DIV_OR_SHIFT(gsize,page_size,page_shift);

  if (MOD_OR_SHIFT(gsize,page_size,page_shift) != 0) {
    total_pages += 1;
  }
  anp->total_pages = total_pages;

  total_blocks = DIV_OR_SHIFT(total_pages,block_size,block_shift);
  if (MOD_OR_SHIFT(total_pages,block_size,block_shift) != 0) {
    total_blocks += 1;
  }
  anp->total_blocks = total_blocks;

  nblocks = DIV_OR_SHIFT(total_blocks,nproc,proc_shift);
  if (myid < MOD_OR_SHIFT(total_blocks,nproc,proc_shift)) {
    nblocks += 1;
  }
  anp->nblocks = nblocks;

  /* allocate storage */

  /* local_data_start is array [0..(nproc-1)] of pointers */

  MEMALLOC(anp->local_data_start, (void **), sizeof(void *) * nproc);
  ASSERT(anp->local_data_start != NULL,
	 "do_declare(): MEMALLOC failed for local_data_start\n", 0);

  for (i = 0; i < nproc; i++) {
    anp->local_data_start[i] = NULL;
  }

  anp->iblock_ptr = NULL;
  anp->dblock_ptr = NULL;
  anp->cblock_ptr = NULL;
  anp->rblock_ptr = NULL;

  if (is_integer) {

    /* avoid null malloc */
    nbytes = max(1, nblocks * block_size * page_size * sizeof(int));

    /* anp->iblock_ptr = (int *) malloc( nbytes ); */
    MEMALLOC(anp->iblock_ptr, (int *), nbytes);
    ASSERT(anp->iblock_ptr != NULL,
	   "do_declare(): MEMALLOC failed for iblock_ptr\n", 0);

    anp->local_data_start[myid] = anp->iblock_ptr;

#if defined(DEBUG)
    for (i = 0; i < nblocks * block_size * page_size; i++) {
      anp->iblock_ptr[i] = -i;
    }
#endif
  } else if (is_dfloat) {

    /* avoid null malloc */
    nbytes = max(1, nblocks * block_size * page_size * sizeof(dfloat));

    /* anp->dblock_ptr = (dfloat *) malloc( nbytes ); */
    MEMALLOC(anp->dblock_ptr, (dfloat *), nbytes);
    ASSERT(anp->dblock_ptr != NULL,
	   "do_declare(): MEMALLOC failed for dblock_ptr\n", 0);

    anp->local_data_start[myid] = anp->dblock_ptr;

#if defined(DEBUG)
    for (i = 0; i < nblocks * block_size * page_size; i++) {
      anp->dblock_ptr[i] = (dfloat) - (9 * 1000 * 1000 + i);
    };
#endif
    /* end-debug */

  } else if (is_char) {

    /* avoid null malloc */
    nbytes = max(1, nblocks * block_size * page_size * sizeof(char));

    MEMALLOC(anp->cblock_ptr, (char *), nbytes);
    ASSERT(anp->cblock_ptr != NULL,
	   "do_declare(): MEMALLOC failed for cblock_ptr\n", 0);

    anp->local_data_start[myid] = anp->cblock_ptr;

#if defined(DEBUG)
    MEMSET(anp->cblock_ptr, 'A', nbytes);
#endif
  } else if (is_real) {
	  
    /* avoid null malloc */
    nbytes = max(1, nblocks * block_size * page_size * sizeof(float));
	  
    /* anp->rblock_ptr = (dfloat *) malloc( nbytes ); */
    MEMALLOC(anp->rblock_ptr, (float *), nbytes);
    ASSERT(anp->rblock_ptr != NULL,
	   "do_declare(): MEMALLOC failed for rblock_ptr\n", 0);
	  
    anp->local_data_start[myid] = anp->rblock_ptr;
	  
#if defined(DEBUG)
    for (i = 0; i < nblocks * block_size * page_size; i++) {
      anp->rblock_ptr[i] = (float) -(9 * 1000 * 1000 + i);
    }
#endif
    /* end-debug */
	  
  } else {
    ASSERT(FALSE, "do_declare(): global array type not valid\n", 0);
  }
  /* global collect on all processors */
  
  GISUM(&(anp->local_data_start[0]), nproc);
  
  /* double check */
  
  for (i = 0; i < nproc; i++) {
    ASSERT(anp->local_data_start[i] != NULL,
	   "do_declare(): local_data_start[%d] is NULL\n", i);
  }
}
