/***********************************************************************
* DOLIB/DONIO Version 0.0 (8/24/94)                                    *       *
*  Software to emulate shared memory on distributed memory environments*
* written by:                                                          *
*  Ed D'Azevedo and Charles Romine of Oak Ridge National Laboratory    *
*                                                                      *
* Questions and comments should be directed to                         *
*      efdazedo@msr.epm.ornl.gov or romine@msr.epm.ornl.gov            *
*                                                                      *
*  Please notify and acknowledge the authors in any research or        *
*  publications utilizing DOLIB/DONIO or any part of the code.         *
*                                                                      *
* NOTICE: Neither the institution nor the author make any              *
*  representations about the suitability of this software for any      *
*  purpose. This software is provided "as is", without express or      *
*  implied warranty.                                                   *
************************************************************************/


#include <stdio.h>
#include <assert.h>
#include "globals.h"


#define MAX_BGATHER_SIZE (1024*1024)
#define DEFAULT_MAXSETS (2*1024)



extern int 
bencode_list(int nitems, int index_start, int segment_size,
	     int maxsets, int *start_list, int *size_list);

#include "externs.h"

/* DO_BGATHER --- block gather, the indices are assumed to be */
/* increasing with stride one                  */

int
do_bgather_C(int Iaf, int nsize, int index_start, void *buf)
/* note index_start is FORTRAN convention, starts from 1 */
{

#define MAX_STACK_SETS (8*1024)

	int             stack_start_list[MAX_STACK_SETS];
	int             stack_size_list[MAX_STACK_SETS];
	int             stack_gindex[MAX_STACK_SETS];

	logical         use_stack;



	logical         is_valid, is_sorted, use_compress;
	logical         is_integer, is_dfloat, is_char, is_real;
	int             i, isize, gni, nbytes, nsets, maxsets, segment_size,
	                page_size, total_pages, gsize, index_end;
	int             wait_id;

	int            *start_list;
	int            *size_list;
	int            *gindex;

	struct Iarray_node *anp;
	int            *ibuf;
	dfloat         *dbuf;
	char           *cbuf;
	float          *rbuf;

	int            *list;




	INTERRUPT_CHECK();

	/* verify that Iaf is valid */

	is_valid = ((1 <= Iaf) && (Iaf <= MAX_GLOBAL_ARRAY));
	ASSERT(is_valid, "do_bgather_C(): Iaf of %d out of range\n", Iaf);

	/* verify that LIST is within bounds for Iaf */

	anp = Global_array[Iaf];
	ASSERT(anp != NULL, "do_bgather_C(): Iaf of %d indexes NULL entry\n", Iaf);

	gsize = anp->gsize;
	page_size = anp->page_size;
	total_pages = anp->total_pages;

	is_integer = (anp->type == INTEGER);
	is_dfloat = (anp->type == REAL8);
	is_char = (anp->type == CHAR);
	is_real = (anp->type == REAL);

	is_valid = (is_integer || is_dfloat || is_char || is_real);
	ASSERT(is_valid, "do_bgather_C(): global array type not valid\n", 0);

	ibuf = NULL;
	dbuf = NULL;
	cbuf = NULL;
	rbuf = NULL;

	if (is_char) {
		cbuf = (char *) buf;
		nbytes = page_size * sizeof(char);
	} else if (is_dfloat) {
		dbuf = (dfloat *) buf;
		nbytes = page_size * sizeof(dfloat);
	} else if (is_integer) {
		ibuf = (int *) buf;
		nbytes = page_size * sizeof(int);
	} else if (is_real) {
		rbuf = (float *) buf;
		nbytes = page_size * sizeof(float);
	} else {
		ASSERT(FALSE, "do_bgather_C(): global array type not valid\n", 0);
	}

	list = NULL;

	ASSERT(nsize > 0, "do_bgather_C(): nsize of %d out of range\n", nsize);

	/* check parameters */
	is_valid = ((1 <= index_start) && (index_start <= gsize));
	ASSERT(is_valid, "do_bgather_C(): index_start of %d out of range\n", index_start);

	index_end = index_start + nsize - 1;
	is_valid = ((1 <= index_end) && (index_end <= gsize));
	ASSERT(is_valid, "do_bgather_C(): index_end of %d out of range\n", index_end);

	/* force  use of compression */
	maxsets = 2 + (nsize / page_size);

	use_stack = (maxsets <= MAX_STACK_SETS);
	if (use_stack) {
		start_list = &(stack_start_list[0]);
		size_list = &(stack_size_list[0]);
		gindex = &(stack_gindex[0]);
	} else {

		MEMALLOC(start_list, (int *), maxsets * sizeof(int));
		ASSERT(start_list != NULL, "do_bgather_C(): Unable to allocate memory for start_list\n", 0);

		MEMALLOC(size_list, (int *), maxsets * sizeof(int));
		ASSERT(size_list != NULL, "do_bgather_C(): Unable to allocate memory for size_list\n", 0);

		MEMALLOC(gindex, (int *), maxsets * sizeof(int));
		ASSERT(gindex != NULL, "do_bgather_C(): Unable to allocate memory for gindex\n", 0);
	}

	/*
	 * since we are getting one-page at a time, segment_size MUST be
	 * page_size, NOT page_size*block_size
	 */

	segment_size = page_size;

	nsets = bencode_list(nsize, index_start, segment_size,
			     maxsets, start_list, size_list);

	is_valid = ((1 <= nsets) && (nsets <= maxsets));
	ASSERT(is_valid, "do_bgather_C(): nsets of %d out of range\n", nsets);

	use_compress = TRUE;
	is_sorted = TRUE;

	gni = index_start;
	for (i = 0; i < nsets; i++) {
		gindex[i] = gni;

		isize = size_list[i];
		gni += isize;
	}

	/* most of the work done in do_setgather() */

	wait_id = do_setgather(Iaf, nsize, list = NULL,
			       nsets, gindex, start_list, size_list,
			       is_sorted, use_compress,
			       buf);

	if (use_stack) {
		/* do nothing */
	} else {
		MEMFREE(start_list);
		MEMFREE(size_list);
		MEMFREE(gindex);
	}
	return (wait_id);
}
