/*
*                               i2dblk.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*       Containing function of rebuilding interface within a mesh block.
*
*/

#if defined(TWOD)

#include <intfc/int.h>

#define DEBUG_STRING "blk_intfc"

LOCAL   BBI_POINT  *crx_in_idir2d(BLK_CRX2*,int,int,int);
LOCAL   BBI_POINT  *crx_in_jdir2d(BLK_CRX2*,int,int,int);
LOCAL   BBI_POINT  *crxing_in_between2d(int*,int*,BLK_CRX2*,int);
LOCAL   bool       is_isolated_point(COMPONENT***,int,int,int,COMPONENT);
LOCAL   void       find_odd_point_and_create_bond(BLK_CRX2*,BLK_BOND*);
LOCAL   void       find_odd_point_and_create_2bonds(BLK_CRX2*,BLK_BOND*);
LOCAL   bool       create_two_point_bond(BLK_CRX2*,BLK_BOND*);
LOCAL   int        sort_crx_in_blk(BBI_POINT  *crxs[], BBI_POINT  *ord_crxs[],int);
LOCAL   void       print_blk_crx2(BLK_CRX2*,int);
LOCAL   int        reconstruct_comp2_intfc2d(BLK_CRX2*,BLK_BOND*,JC_BLK_BOND*,int,BBI_POINT  *crxs[]);
LOCAL   int        reconstruct_comp3_intfc2d(BLK_CRX2*,BLK_BOND*,int,BBI_POINT  *crxs[]);
LOCAL   int        reconstruct_mix_crx_intfc2d(BLK_CRX2*,BLK_BOND*,int,BBI_POINT  *crxs[]);
LOCAL   int        create_even_multi_bonds(BLK_CRX2*,BLK_BOND*,int,BBI_POINT  *crxs[],
                          int,BBI_POINT  *crxs2[]);  
LOCAL   int        create_odd_multi_bonds(BLK_CRX2*,BLK_BOND*,int,BBI_POINT  *crxs[],
                          int,BBI_POINT  *crxs2[]);  
LOCAL   int        two_connect_comps_have_node_create_bonds(BLK_CRX2*,BLK_BOND*,
                        int,BBI_POINT  *crxs[]);
LOCAL   int        create_join_c_2p_bond(BLK_CRX2*,BLK_BOND*,JC_BLK_BOND*);
LOCAL   void       in_topology_blk_node(int*,CURVE*,NODE**,int*);

LOCAL void print_blk_crx2(
        BLK_CRX2   *blk_crx,
        int        prt_c)
{
        int  i, j; 
        int  nc; 
      
        switch(blk_crx->num_comps)
        {
        case 2:
            printf("Blk type[%s]\n","COMP2_BLOCK"); 
        break;
        case 3:
            printf("Blk type[%s]\n","COMP3_BLOCK"); 
        break;
        default:
            printf("WARNING in print_blk_crx2\n");
            printf("num_comps inside blk = %d, Unknown type\n", blk_crx->num_comps);
        }

        printf("Blk component: \n"); 
        for(j = 1; j >= 0; j--)
            printf("Line %d: [%d] [%d]\n", j, 
                  blk_crx->comp[0][j][0], blk_crx->comp[1][j][0]); 

        printf("Blk crossings:\n"); 
        for (i = 0; i < 2; i++)
        {
            if(blk_crx->n_crx[0][i][0] != 0)
            {
                for(nc = 0; nc < blk_crx->n_crx[0][i][0]; nc++)
                {
                    printf("X dir. line[%d] : ", i);
                    print_general_vector("",Coords(blk_crx->crx[0][i][0][nc].p),2,"\n");
                }
            }
        }
        for (i = 0; i < 2; i++)
        {
            if(blk_crx->n_crx[1][0][i] != 0)
            {
                for(nc = 0; nc < blk_crx->n_crx[1][0][i]; nc++)
                {  
                    printf("Y dir. line[%d] : ", i);
                    print_general_vector("",Coords(blk_crx->crx[1][0][i][nc].p),2,"\n");
                }
            }
        }
        printf("Inside blk: # of comps[%d], # of curves[%d] # of waves[%d]\n", 
                   blk_crx->num_comps, blk_crx->num_curves, blk_crx->num_waves); 

        if(prt_c == YES)
        {
            for (i = 0; i < 2; i++)
            {
                if(blk_crx->n_crx[0][i][0] != 0)
                {
                    for(nc = 0; nc < blk_crx->n_crx[0][i][0]; nc++)
                    {
                        printf("Curve at X dir. line[%d] : ", i);
                        print_curve(Curve_of_hs(blk_crx->crx[0][i][0][nc].hs));
                    }
                }
            }
            for (i = 0; i < 2; i++)
            {
                if(blk_crx->n_crx[1][0][i] != 0)
                {
                    for(nc = 0; nc < blk_crx->n_crx[1][0][i]; nc++)
                    {
                        printf("Curve at Y dir. line[%d] : ", i);
                        print_curve(Curve_of_hs(blk_crx->crx[1][0][i][nc].hs));
                    }
                }
            }
        }
        printf("End of print BLK_CRX\n");
}

LOCAL   BBI_POINT  *crx_in_idir2d(
        BLK_CRX2   *blk_crx,
        int        j,
        int        k,
        int        nc)
{
        int ip[3],ipn[3];
        int ***ix = blk_crx->ix;
        int ***iy = blk_crx->iy;
        int ***iz = blk_crx->iz;
        BBI_POINT  *bbi;
#if defined(NAVIGATOR)
        NAV_ENTER(crx_in_idir)
#endif /* defined(NAVIGATOR) */

        ip[0] = ix[0][j][k];
        ip[1] = iy[0][j][k];
        ip[2] = iz[0][j][k];
        ipn[0] = ix[1][j][k];
        ipn[1] = iy[1][j][k];
        ipn[2] = iz[1][j][k];
        bbi = crxing_in_between2d(ip,ipn,blk_crx,nc);
#if defined(NAVIGATOR)
        NAV_LEAVE(crx_in_idir)
#endif /* defined(NAVIGATOR) */
        return bbi;
}               /*end crx_in_idir*/


LOCAL   BBI_POINT  *crx_in_jdir2d(
        BLK_CRX2 *blk_crx,
        int      k,
        int      i,
        int      nc)
{
        int ip[3],ipn[3];
        int ***ix = blk_crx->ix;
        int ***iy = blk_crx->iy;
        int ***iz = blk_crx->iz;
        BBI_POINT  *bbi;
#if defined(NAVIGATOR)
        NAV_ENTER(crx_in_jdir)
#endif /* defined(NAVIGATOR) */

        ip[0] = ix[i][0][k];
        ip[1] = iy[i][0][k];
        ip[2] = iz[i][0][k];
        ipn[0] = ix[i][1][k];
        ipn[1] = iy[i][1][k];
        ipn[2] = iz[i][1][k];
        bbi = crxing_in_between2d(ip,ipn,blk_crx,nc);
#if defined(NAVIGATOR)
        NAV_LEAVE(crx_in_jdir)
#endif /* defined(NAVIGATOR) */
        return bbi;
}               /*end crx_in_jdir*/


LOCAL   BBI_POINT  *crxing_in_between2d(
        int      *ip,
        int      *ipn,
        BLK_CRX2 *blk_crx,
        int      nc)
{

        if (ip[0] != ipn[0])
        {
            if (blk_crx->crx[0][ip[1]][ip[2]][nc].p == NULL)
            {
                screen("ERROR in crxing_in_between(), "
                       "no crossing point between:\n"
                       "(%d %d %d) and (%d %d %d)\n",
                        ip[0],ip[1],ip[2],ipn[0],ipn[1],ipn[2]);
                clean_up(ERROR);
            }
            return &blk_crx->crx[0][ip[1]][ip[2]][nc];
        }
        else if (ip[1] != ipn[1])
        {
            if (blk_crx->crx[1][ip[2]][ip[0]][nc].p == NULL)
            {
                screen("ERROR in crxing_in_between(), "
                       "no crossing point between:\n"
                       "(%d %d %d) and (%d %d %d)\n",
                        ip[0],ip[1],ip[2],ipn[0],ipn[1],ipn[2]);
                clean_up(ERROR);
            }
            return &blk_crx->crx[1][ip[2]][ip[0]][nc];
        }
        else if (ip[2] != ipn[2])
        {
            if (blk_crx->crx[2][ip[0]][ip[1]][nc].p == NULL)
            {
                screen("ERROR in crxing_in_between(), "
                       "no crossing point between:\n"
                       "(%d %d %d) and (%d %d %d)\n",
                        ip[0],ip[1],ip[2],ipn[0],ipn[1],ipn[2]);
                clean_up(ERROR);
            }
            return &blk_crx->crx[2][ip[0]][ip[1]][nc];
        }
        screen("ERROR in crxing_in_between(), "
               "Inconsistent values of ip and ipn arrays\n");
        clean_up(ERROR);
        return NULL;
}               /* end crxing_in_between */


LOCAL   bool  is_isolated_point(
        COMPONENT ***comp,
        int i,
        int j,
        int k,
        COMPONENT min_comp)
{
        bool status;
#if defined(NAVIGATOR)
        NAV_ENTER(is_isolated_point)
#endif /* defined(NAVIGATOR) */

        if (comp[(i+1)%2][j][k] == min_comp)
            status = NO;
        else if (comp[i][(j+1)%2][k] == min_comp)
            status = NO;
        else if (comp[i][j][(k+1)%2] == min_comp)
            status = NO;
        else
            status = YES;
#if defined(NAVIGATOR)
        NAV_LEAVE(is_isolated_point)
#endif /* defined(NAVIGATOR) */
        return status;
}               /*end is_isolated_point*/


EXPORT  void vol_alloc_blk_crx2d(
        BLK_CRX2      **blk_crx2)
{
        int           i, j, k; 
        stat_scalar(blk_crx2,sizeof(BLK_CRX2));
        stat_tri_array(&(*blk_crx2)->comp,2,2,2,sizeof(COMPONENT));
        stat_tri_array(&(*blk_crx2)->ix,2,2,2,sizeof(int));
        stat_tri_array(&(*blk_crx2)->iy,2,2,2,sizeof(int));
        stat_tri_array(&(*blk_crx2)->iz,2,2,2,sizeof(int));
        stat_tri_array(&(*blk_crx2)->crx,3,2,2,sizeof(BBI_POINT *));
        stat_tri_array(&(*blk_crx2)->n_crx,3,2,2,sizeof(int));
        vector(&(*blk_crx2)->crx_store,20,sizeof(BBI_POINT ));

        k = 0; 
        for(i = 0; i < 2; i++)
        {
            if(i != 0)
                k += 5; 
            (*blk_crx2)->crx[0][i][0] = (*blk_crx2)->crx_store + k;
        }

        for(i = 0; i < 2; i++)
        {
            k += 5; 
            (*blk_crx2)->crx[1][0][i] = (*blk_crx2)->crx_store + k;
        }
}               /*end vol_alloc_blk_crx2d*/

LOCAL   void find_odd_point_and_create_bond(
        BLK_CRX2    *blk_crx,
        BLK_BOND    *blk_mem)
{
        int         i,j,k;
        COMPONENT   ***comp = blk_crx->comp;
        COMPONENT   pos_comp, neg_comp;
        int         num_crx;
        BBI_POINT   *crxs[2];
        CURVE       *c;
        BOND        *newb;
        int         find = NO;

        num_crx = 0;
        for (j = 0; j < 2; j++)
        {
            if (comp[0][j][0] != comp[1][j][0])
            {
                crxs[num_crx] = crx_in_idir2d(blk_crx,j,0,0);
#if defined(DEBUG)
                if(crxs[num_crx] == NULL)
                {
                    printf("ERROR find_odd_point_and_create_bond\n");
                    printf("i's crxs[%d] = NULL\n", num_crx);
                    clean_up(ERROR); 
                }
#endif /* if defined(DEBUG) */
                num_crx++;
                break;
            }
        }

        for (i = 0; i < 2; i++)
        {
            if (comp[i][0][0] != comp[i][1][0])
            {
                crxs[num_crx] = crx_in_jdir2d(blk_crx,0,i,0);
#if defined(DEBUG)
                if(crxs[num_crx] == NULL)
                {
                    printf("ERROR find_odd_point_and_create_bond\n");
                    printf("j's crxs[%d] = NULL\n", num_crx);
                    clean_up(ERROR); 
                }
#endif /* if defined(DEBUG) */
                num_crx++;
                break;
            }
        }

#if defined(DEBUG)
        if(num_crx != 2)
        {
            printf("ERROR in find_odd_point_and_create_bond\n");
            printf("num_crx != 2\n"); 
            clean_up(ERROR);  
        }
#endif /* if defined(DEBUG) */

        if (crxs[0]->hs != crxs[1]->hs)
        {
            screen("ERROR in find_odd_point_and_create_bond(),"
                    " crxs from different surfs\n");
            clean_up(ERROR);
        }

        pos_comp = crxs[0]->hs->pos_comp;
        neg_comp = crxs[0]->hs->neg_comp;

        c = Curve_of_hs(crxs[0]->hs);

        for (k = 0; k < blk_mem->num_curves; k++)
        {
            if (c == blk_mem->curves[k])
            {
                find = YES;
                break;
            }
        }
#if defined(DEBUG)
        if(find != YES)
        {
            printf("ERROR in find_odd_point_and_create_bond()"
              " find = NO\n");
            clean_up(ERROR);  
        }
#endif /* if defined(DEBUG) */

        if(j == 0 && i == 0)
        {
#if defined(DEBUG)
            if(crxs[0]->lcomp != crxs[1]->lcomp ||
               crxs[0]->ucomp != crxs[1]->ucomp)
            {
                printf("ERROR in find_odd_point_and_create_bond()"
                  " j =0, i = 0, crxs[0]->lcomp != crxs[1]->lcomp\n");
                clean_up(ERROR);  
            }
#endif /* if defined(DEBUG) */
            if(crxs[0]->ucomp == pos_comp)
                newb = Bond(crxs[0]->p, crxs[1]->p);
            else if(crxs[0]->ucomp == neg_comp)
                newb = Bond(crxs[1]->p, crxs[0]->p);
            else
            {
                printf("ERROR In find_odd_point_and_create_bond(), 0\n");
                clean_up(ERROR);
            }
            (blk_mem->num_bonds[k])++;
            if (blk_mem->first[k] == NULL)
                blk_mem->first[k] = newb;
            if(blk_mem->cur_bonds[k] == NULL)
                blk_mem->cur_bonds[k] = newb;
            else
            {
                newb->next = blk_mem->cur_bonds[k];
                blk_mem->cur_bonds[k]->prev = newb;
                blk_mem->cur_bonds[k] = newb;
            }
            return;
        }
        if(j == 0 && i == 1)
        {
#if defined(DEBUG)
            if(crxs[0]->lcomp != crxs[1]->ucomp ||
               crxs[0]->ucomp != crxs[1]->lcomp)
            {
                printf("ERROR in find_odd_point_and_create_bond()"
                  " j =0, i = 1, crxs[0]->lcomp != crxs[1]->lcomp\n");
                clean_up(ERROR);  
            }
#endif /* if defined(DEBUG) */
            if(crxs[0]->ucomp == pos_comp)
                newb = Bond(crxs[0]->p, crxs[1]->p);
            else if(crxs[0]->ucomp == neg_comp)
                newb = Bond(crxs[1]->p, crxs[0]->p);
            else
            {
                printf("ERROR In find_odd_point_and_create_bond(), 1\n");
                clean_up(ERROR);
            }
            (blk_mem->num_bonds[k])++;
            if (blk_mem->first[k] == NULL)
                blk_mem->first[k] = newb;
            if(blk_mem->cur_bonds[k] == NULL)
                blk_mem->cur_bonds[k] = newb;
            else
            {
                newb->next = blk_mem->cur_bonds[k];
                blk_mem->cur_bonds[k]->prev = newb;
                blk_mem->cur_bonds[k] = newb;
            }
            //  print_bond(blk_mem->cur_bonds[k]);
            return;
        }
        if(j == 1 && i == 1)
        {
#if defined(DEBUG)
            if(crxs[0]->lcomp != crxs[1]->lcomp ||
               crxs[0]->ucomp != crxs[1]->ucomp)
            {
                printf("ERROR in find_odd_point_and_create_bond()"
                  " j =1, i = 1, crxs[0]->lcomp != crxs[1]->lcomp\n");
                clean_up(ERROR);  
            }
#endif /* if defined(DEBUG) */
            if(crxs[0]->ucomp == pos_comp)
                newb = Bond(crxs[1]->p, crxs[0]->p);
            else if(crxs[0]->ucomp == neg_comp)
                newb = Bond(crxs[0]->p, crxs[1]->p);
            else
            {
                printf("ERROR In find_odd_point_and_create_bond(), 2\n");
                clean_up(ERROR);
            }
            (blk_mem->num_bonds[k])++;
            if (blk_mem->first[k] == NULL)
                blk_mem->first[k] = newb;
            if(blk_mem->cur_bonds[k] == NULL)
                blk_mem->cur_bonds[k] = newb;
            else
            {
                newb->next = blk_mem->cur_bonds[k];
                blk_mem->cur_bonds[k]->prev = newb;
                blk_mem->cur_bonds[k] = newb;
            }
            //  print_bond(blk_mem->cur_bonds[k]);
            return;
        }
        if( j == 1 && i == 0)
        {
#if defined(DEBUG)
            if(crxs[0]->lcomp != crxs[1]->ucomp ||
               crxs[0]->ucomp != crxs[1]->lcomp)
            {
                printf("ERROR in find_odd_point_and_create_bond()"
                  " j =1, i = 0, crxs[0]->lcomp != crxs[1]->lcomp\n");
                clean_up(ERROR);  
            }
#endif /* if defined(DEBUG) */
            if(crxs[0]->ucomp == pos_comp)
                newb = Bond(crxs[1]->p, crxs[0]->p);
            else if(crxs[0]->ucomp == neg_comp)
                newb = Bond(crxs[0]->p, crxs[1]->p);
            else
            {
                printf("ERROR In find_odd_point_and_create_bond(), 3\n");
                clean_up(ERROR);
            }
            (blk_mem->num_bonds[k])++;
            if (blk_mem->first[k] == NULL)
                blk_mem->first[k] = newb;
            if(blk_mem->cur_bonds[k] == NULL)
                blk_mem->cur_bonds[k] = newb;
            else
            {
                newb->next = blk_mem->cur_bonds[k];
                blk_mem->cur_bonds[k]->prev = newb;
                blk_mem->cur_bonds[k] = newb;
            }
            //   print_bond(blk_mem->cur_bonds[k]);
            return;
        }
        printf("ERROR In find_odd_point_and_create_bond(),4 \n");
        clean_up(ERROR);

}

LOCAL   void find_odd_point_and_create_2bonds(
        BLK_CRX2    *blk_crx,
        BLK_BOND    *blk_mem)
{
        int         i,j, k0, k1;
        COMPONENT   ***comp = blk_crx->comp;
        COMPONENT   pos_comp, neg_comp;
        int         num_crx;
        BBI_POINT   *crxs[2];
        CURVE       **c, *c0, *c1;
        BOND        *newb;
        NODE        *node = NULL; 
        int         find = NO, n_bonds = 0;

        num_crx = 0;
        for (j = 0; j < 2; j++)
        {
            if (comp[0][j][0] != comp[1][j][0])
            {
                crxs[num_crx] = crx_in_idir2d(blk_crx,j,0,0);
#if defined(DEBUG)
                if(crxs[num_crx] == NULL)
                {
                    printf("ERROR find_odd_point_and_create_2bonds\n");
                    printf("i's crxs[%d] = NULL\n", num_crx);
                    clean_up(ERROR); 
                }
#endif /* if defined(DEBUG) */
                num_crx++;
                break;
            }
        }

        for (i = 0; i < 2; i++)
        {
            if (comp[i][0][0] != comp[i][1][0])
            {
                crxs[num_crx] = crx_in_jdir2d(blk_crx,0,i,0);
#if defined(DEBUG)
                if(crxs[num_crx] == NULL)
                {
                    printf("ERROR find_odd_point_and_create_2bonds\n");
                    printf("j's crxs[%d] = NULL\n", num_crx);
                    clean_up(ERROR); 
                }
#endif /* if defined(DEBUG) */
                num_crx++;
                break;
            }
        }

#if defined(DEBUG)
        if(num_crx != 2 || crxs[0]->hs == crxs[1]->hs)
        {
            printf("ERROR in find_odd_point_and_create_2bonds\n");
            printf("num_crx = %d, crxs[0]->hs = %d crxs[1]->hs = %d\n",
                   num_crx, crxs[0]->hs, crxs[1]->hs); 
            clean_up(ERROR);  
        }
#endif /* if defined(DEBUG) */

        pos_comp = crxs[0]->hs->pos_comp;
        neg_comp = crxs[0]->hs->neg_comp;

        c0 = Curve_of_hs(crxs[0]->hs);
        c1 = Curve_of_hs(crxs[1]->hs);
 
        if(c0->start == c1->start)
            node = c0->start;
        else if(c0->start == c1->end)
            node = c0->start;
        else if(c0->end == c1->start)
            node = c0->end;
        else if(c0->end == c1->end)
            node = c0->end;
        else
        {
            printf("ERROR In find_odd_point_and_create_2bonds()\n");
            printf("Two crossings does not meet at same node\n");
            print_curve(c0);
            print_curve(c1); 
            clean_up(ERROR);
        }

        for (k0 = 0; k0 < blk_mem->num_curves; k0++)
        {
            if (c0 == blk_mem->curves[k0])
            {
                find = YES;
                break;
            }
        }
#if defined(DEBUG)
        if(find != YES)
        {
            printf("ERROR in find_odd_point_and_create_2bonds()"
              " find = NO for c0\n");
            print_curve(c0); 
            clean_up(ERROR);
        }
#endif /* if defined(DEBUG) */

        find = NO; 
        for (k1 = 0; k1 < blk_mem->num_curves; k1++)
        {
            if (c1 == blk_mem->curves[k1])
            {
                find = YES;
                break;
            }
        }
#if defined(DEBUG)
        if(find != YES)
        {
            printf("ERROR in find_odd_point_and_create_2bonds()"
              " find = NO for c1\n");
            print_curve(c1); 
            clean_up(ERROR);
        }
#endif /* if defined(DEBUG) */
       
        for( c = node->in_curves; c && *c; c++)
        {
            if(c0 == *c)
            {
                newb = Bond(crxs[0]->p, node->posn); 
                n_bonds++; 
                (blk_mem->num_bonds[k0])++;
                if (blk_mem->first[k0] == NULL)
                    blk_mem->first[k0] = newb;
                if(blk_mem->cur_bonds[k0] == NULL)
                    blk_mem->cur_bonds[k0] = newb;
                else
                {
                    newb->next = blk_mem->cur_bonds[k0];
                    blk_mem->cur_bonds[k0]->prev = newb;
                    blk_mem->cur_bonds[k0] = newb;
                }
            }
            if(c1 == *c)
            {
                newb = Bond(crxs[1]->p, node->posn); 
                n_bonds++; 
                (blk_mem->num_bonds[k1])++;
                if (blk_mem->first[k1] == NULL)
                    blk_mem->first[k1] = newb;
                if(blk_mem->cur_bonds[k1] == NULL)
                    blk_mem->cur_bonds[k1] = newb;
                else
                {
                    newb->next = blk_mem->cur_bonds[k1];
                    blk_mem->cur_bonds[k1]->prev = newb;
                    blk_mem->cur_bonds[k1] = newb;
                }
            }
        } 

        for( c = node->out_curves; c && *c; c++)
        {
            if(c0 == *c)
            {
                newb = Bond(node->posn, crxs[0]->p); 
                n_bonds++; 
                (blk_mem->num_bonds[k0])++;
                if (blk_mem->first[k0] == NULL)
                    blk_mem->first[k0] = newb;
                if(blk_mem->cur_bonds[k0] == NULL)
                    blk_mem->cur_bonds[k0] = newb;
                else
                {
                    newb->next = blk_mem->cur_bonds[k0];
                    blk_mem->cur_bonds[k0]->prev = newb;
                    blk_mem->cur_bonds[k0] = newb;
                }
            }
            if(c1 == *c)
            {
                newb = Bond(node->posn, crxs[1]->p); 
                n_bonds++; 
                (blk_mem->num_bonds[k1])++;
                if (blk_mem->first[k1] == NULL)
                    blk_mem->first[k1] = newb;
                if(blk_mem->cur_bonds[k1] == NULL)
                    blk_mem->cur_bonds[k1] = newb;
                else
                {
                    newb->next = blk_mem->cur_bonds[k1];
                    blk_mem->cur_bonds[k1]->prev = newb;
                    blk_mem->cur_bonds[k1] = newb;
                }
            }
        } 
#if defined(DEBUG)
        if(n_bonds != 2)
        {
            printf("ERROR in find_odd_point_and_create_2bonds()\n");
            printf("num of created bonds = %d, not 2\n", n_bonds);
            clean_up(ERROR);  
        }
#endif /* if defined(DEBUG) */

        
}


LOCAL   bool create_two_point_bond(
        BLK_CRX2     *blk_crx,
        BLK_BOND     *blk_mem)
{
        int          i,j,k;
        COMPONENT    min_comp = blk_crx->pos_comp;
        COMPONENT    ***comp = blk_crx->comp;
        COMPONENT    pos_comp, neg_comp;
        int          num_isolated = 0;
        int          num_crx;
        BBI_POINT    *crxs[4], *ord_crxs[4];
        CURVE        *c;
        BOND         *newb, *newb2;
        POINT        tmpp;
        int          find = NO;

        /* Search for isolated points */
        for (i = 0; i < 2; i++)
        {
            for (j = 0; j < 2; j++)
            {
                if (comp[i][j][0] == min_comp)
                {
                    if (is_isolated_point(comp,i,j,0,min_comp) == YES)
                    {
                        num_isolated++;
                    }
                }
            }
        }

        if(num_isolated == 0)
        {
            num_crx = 0;
            for (j = 0; j < 2; j++)
            {
                if (comp[0][j][0] != comp[1][j][0])
                {
                    crxs[num_crx] = crx_in_idir2d(blk_crx,j,0,0);
#if defined(DEBUG)
                    if(crxs[num_crx] == NULL)
                    {
                        printf("ERROR in create_two_point_bond\n");
                        printf("i's crxs[%d] = NULL\n", num_crx);
                        clean_up(ERROR);  
                    }
#endif /*if defined(DEBUG)  */
                    num_crx++;
                }
            }
            if(num_crx == 2)
            {
#if defined(DEBUG)
                if(crxs[0] != crx_in_idir2d(blk_crx,0,0,0))
                {
                    printf("ERROR in create_two_point_bond\n");
                    printf("crxs[0] != crx_in_idir2d(blk_crx,0,0)\n");
                    clean_up(ERROR);  
                }
#endif /* if defined(DEBUG) */
                if (crxs[0]->hs != crxs[1]->hs)
                {
                    screen("ERROR in create_two_point_bond(),"
                        " crxs from different surfs\n");
                    clean_up(ERROR);
                }

                c = Curve_of_hs(crxs[0]->hs);
                for (k = 0; k < blk_mem->num_curves; k++)
                {
                    if (c == blk_mem->curves[k])
                    {
                        find = YES;
                        break;
                    }
                }
#if defined(DEBUG)
                if(find != YES)
                {
                    printf("ERROR in create_two_point_bond(),"
                            "find = NO\n");
                    clean_up(ERROR); 
                }
#endif /* if defined(DEBUG) */

                pos_comp = crxs[0]->hs->pos_comp;
                neg_comp = crxs[0]->hs->neg_comp;
#if defined(DEBUG)
                if(crxs[0]->lcomp != crxs[1]->lcomp ||
                   crxs[0]->ucomp != crxs[1]->ucomp)
                {
                    printf("ERROR in create_two_point_bond()"
                       "crxs[0]->lcomp != crxs[1]->lcomp\n");
                    clean_up(ERROR);  
                }
#endif /* if defined(DEBUG) */
                if(crxs[0]->ucomp == pos_comp)
                    newb = Bond(crxs[0]->p, crxs[1]->p);
                else if(crxs[0]->ucomp == neg_comp)
                    newb = Bond(crxs[1]->p, crxs[0]->p);
                else
                {
                    printf("ERROR In create_two_point_bond(), 0\n");
                    clean_up(ERROR);
                }
                (blk_mem->num_bonds[k])++;
                if (blk_mem->first[k] == NULL)
                    blk_mem->first[k] = newb;
                if(blk_mem->cur_bonds[k] == NULL)
                    blk_mem->cur_bonds[k] = newb;
                else
                {
                    newb->next = blk_mem->cur_bonds[k];
                    blk_mem->cur_bonds[k]->prev = newb;
                    blk_mem->cur_bonds[k] = newb;
                }
                //  print_bond(blk_mem->cur_bonds[k]);
                return FUNCTION_SUCCEEDED;
            }


            if(num_crx != 0)
            {
                printf("ERROR In create_two_point_bond()\n");
                printf("num_crx =%d, should be 0\n", num_crx);
                clean_up(ERROR);
            }

            for (i = 0; i < 2; i++)
            {
                if (comp[i][0][0] != comp[i][1][0])
                {
                    crxs[num_crx] = crx_in_jdir2d(blk_crx,0,i,0);
#if defined(DEBUG)
                    if(crxs[num_crx] == NULL)
                    {
                        printf("ERROR in create_two_point_bond\n");
                        printf("j's crxs[%d] = NULL\n", num_crx);
                        clean_up(ERROR);  
                    }
#endif /*if defined(DEBUG)  */
                    num_crx++;
                }
            }
            if(num_crx == 2)
            {
#if defined(DEBUG)
                if(crxs[0] != crx_in_jdir2d(blk_crx,0,0,0))
                {
                    printf("ERROR in create_two_point_bond\n");
                    printf("crxs[0] != crx_in_jdir2d(blk_crx,0,0)\n");
                    clean_up(ERROR);  
                }
#endif /* if defined(DEBUG) */
                if (crxs[0]->hs != crxs[1]->hs)
                {
                    screen("ERROR in create_two_point_bond(),"
                        " crxs from different surfs\n");
                    clean_up(ERROR);
                }

                c = Curve_of_hs(crxs[0]->hs);
                for (k = 0; k < blk_mem->num_curves; k++)
                {
                    if (c == blk_mem->curves[k])
                    {
                        find = YES;
                        break;
                    }
                }
#if defined(DEBUG)
                if(find != YES)
                {
                    printf("ERROR in create_two_point_bond(),"
                            "find = NO 2\n");
                    clean_up(ERROR); 
                }
#endif /* if defined(DEBUG) */

                pos_comp = crxs[0]->hs->pos_comp;
                neg_comp = crxs[0]->hs->neg_comp;
#if defined(DEBUG)
                if(crxs[0]->lcomp != crxs[1]->lcomp ||
                   crxs[0]->ucomp != crxs[1]->ucomp)
                {
                    printf("ERROR in create_two_point_bond()"
                       "crxs[0]->lcomp != crxs[1]->lcomp 2\n");
                    clean_up(ERROR);  
                }
#endif /* if defined(DEBUG) */
                if(crxs[0]->ucomp == pos_comp)
                    newb = Bond(crxs[1]->p, crxs[0]->p);
                else if(crxs[0]->ucomp == neg_comp)
                    newb = Bond(crxs[0]->p, crxs[1]->p);
                else
                {
                    printf("ERROR In create_two_point_bond(), 1\n");
                    clean_up(ERROR);
                }
                (blk_mem->num_bonds[k])++;
                if (blk_mem->first[k] == NULL)
                    blk_mem->first[k] = newb;
                if(blk_mem->cur_bonds[k] == NULL)
                    blk_mem->cur_bonds[k] = newb;
                else
                {
                    newb->next = blk_mem->cur_bonds[k];
                    blk_mem->cur_bonds[k]->prev = newb;
                    blk_mem->cur_bonds[k] = newb;
                }
                //  print_bond(blk_mem->cur_bonds[k]);
                return FUNCTION_SUCCEEDED;
            }

            screen("ERROR in create_two_point_bond(),"
                   " crxs from unkown  surfs\n");
            clean_up(ERROR);
        }
        else  /* two bonds in one blk */
        {

            num_crx = 0;
            for (j = 0; j < 2; j++)
            {
                if (comp[0][j][0] != comp[1][j][0])
                {
                    crxs[num_crx] = crx_in_idir2d(blk_crx,j,0,0);
#if defined(DEBUG)
                    if(crxs[num_crx] == NULL)
                    {
                        printf("ERROR: IN create_two_point_bond(), NULL crx1\n");
                        clean_up(ERROR);
                    }
#endif /* if defined(DEBUG) */ 
                    num_crx++;
                }
            }
            for (i = 0; i < 2; i++)
            {
                if (comp[i][0][0] != comp[i][1][0])
                {
                    crxs[num_crx] = crx_in_jdir2d(blk_crx,0,i,0);
#if defined(DEBUG)
                    if(crxs[num_crx] == NULL)
                    {
                        printf("ERROR: IN create_two_point_bond(), NULL crx2\n");
                        clean_up(ERROR);
                    }
#endif /* if defined(DEBUG) */ 
                    num_crx++;
                }
            }

            if(num_crx != 4)
            {
                printf("ERROR: IN create_two_point_bond(), should be 4 crxs, but not\n");
                clean_up(ERROR);
            }


            if( NO == sort_crx_in_blk(crxs, ord_crxs,4))
            {
                printf("ERROR: IN create_two_point_bond(), not sort\n");
                clean_up(ERROR);
            }

            c = Curve_of_hs(crxs[0]->hs);
            for (k = 0; k < blk_mem->num_curves; k++)
            {
                if (c == blk_mem->curves[k])
                {
                    find = YES;
                    break;
                }
            }
            if(! find)
            {
                printf("ERROR: IN create_two_point_bond(), should find k\n");
                clean_up(ERROR);
            }
            newb2 = newb = Bond(ord_crxs[0]->p, ord_crxs[1]->p);
            (blk_mem->num_bonds[k])++;
            if (blk_mem->first[k] == NULL)
                blk_mem->first[k] = newb;
            if(blk_mem->cur_bonds[k] == NULL)
                blk_mem->cur_bonds[k] = newb;
            else
            {
                newb->next = blk_mem->cur_bonds[k];
                blk_mem->cur_bonds[k]->prev = newb;
                blk_mem->cur_bonds[k] = newb;
            }
            //  print_bond(blk_mem->cur_bonds[k]);

            newb = Bond(ord_crxs[2]->p, ord_crxs[3]->p);
            (blk_mem->num_bonds[k])++;
            if (blk_mem->first[k] == NULL)
                blk_mem->first[k] = newb;
            if(blk_mem->cur_bonds[k] == NULL)
                blk_mem->cur_bonds[k] = newb;
            else
            {
                newb->next = blk_mem->cur_bonds[k];
                blk_mem->cur_bonds[k]->prev = newb;
                blk_mem->cur_bonds[k] = newb;
            }
            //  print_bond(blk_mem->cur_bonds[k]);

            if(cross_bonds(newb2, blk_mem->cur_bonds[k], &tmpp))
            {
                printf("ERROR in create_two_point_bond(), two bonds crossing\n");
                clean_up(ERROR);
            }

            /*
            printf("WARNING: IN create_two_point_bond() num_isolated = %d\n",
                    num_isolated);
            for(i = 0; i < num_crx; i++)
                printf("ord_pt<%d> <%g, %g>\n",i,
                        Coords(ord_crxs[i]->p)[0], Coords(ord_crxs[i]->p)[1]);
            */
            return FUNCTION_SUCCEEDED;
            clean_up(ERROR);
        }
}

LOCAL int sort_crx_in_blk(
        BBI_POINT    *crxs[],
        BBI_POINT    *ord_crxs[],
        int          num_crx)
{
        CURVE       *c;
        BOND        *b;
        int         i, num;
        int         found = NO;

        c = Curve_of_hs(crxs[0]->hs);
        for(i = 1; i < num_crx; i++)
        {
            if(Curve_of_hs(crxs[i]->hs) != c)
            {
                printf("ERROR in sort_crx_in_blk, crx[%d] not on same curve\n",i);
                clean_up(ERROR);
            }
        }
        num = 0;
        b = c->first;
        while (b != NULL)
        {
            if(b == c->first)
            {
                for(i = 0; i < num_crx; i++)
                {
                    if(crxs[i]->p == b->start)
                    {
                        ord_crxs[num++] = crxs[i];
                        break;
                    }
                }
            }
            for(i = 0; i < num_crx; i++)
            {
                if(crxs[i]->p == b->end)
                {
                    ord_crxs[num++] = crxs[i];
                    break;
                }
            }
            if(num == num_crx)
               break;
            b = b->next;
        }
        if(num != num_crx)
        {
            printf("ERROR in sort_crx_in_blk, crx not find on curve");
            printf(" num = %d, num_crx = %d\n", num, num_crx);
            return NO;
            clean_up(ERROR);
        }
        return YES;
}

/* 
 * NOTE in BLK_CRX2, "*c" variable in BBI_POINT  variable crx[i][j][k],
 * is not used. Only hypersurface is saved.
 */ 
EXPORT  int reconstruct_blk_intfc2d(
        BLK_CRX2    *blk_crx,
        BLK_BOND    *blk_mem,
        JC_BLK_BOND *jc_blk_bond)
{
        int         i, j;
        COMPONENT   ***comp = blk_crx->comp;
        int         num_pos,num_crx, nc;
        BBI_POINT   *crxs[20];

        DEBUG_ENTER(reconstruct_blk_intfc2d)

        num_crx = 0;
        for(j = 0; j < 2; j++)
        {
            if(blk_crx->n_crx[0][j][0] != 0)
            {
                for(nc = 0; nc < blk_crx->n_crx[0][j][0]; nc++)
                {
                    crxs[num_crx] = crx_in_idir2d(blk_crx,j,0,nc);
                    if (crxs[num_crx] == NULL)
                    {
                        printf("ERROR in reconstruct_blk_intfc2d, crxs NULL 1\n");
                        return FUNCTION_FAILED;
                    }
                    num_crx++;
                }
            }
        }
        for (i = 0; i < 2; i++)
        {
            if(blk_crx->n_crx[1][0][i] != 0)
            {
                for(nc = 0; nc < blk_crx->n_crx[1][0][i]; nc++)
                {
                    crxs[num_crx] = crx_in_jdir2d(blk_crx,0,i,nc);
                    if (crxs[num_crx] == NULL)
                    {
                        printf("ERROR in reconstruct_blk_intfc2d, crxs NULL 2\n");
                        return FUNCTION_FAILED;
                    }
                    num_crx++;
                }
            }
        }

#if defined(DEBUG)
        /***
        if((blk_crx->blkic[1] == 67 && blk_crx->blkic[0] == 8) || 
            (blk_crx->blkic[1] == 67 && blk_crx->blkic[0] == 1))
        {
            printf("ic[%d %d] construct intfc\n",
                     blk_crx->blkic[0], blk_crx->blkic[1]);
            print_blk_crx2(blk_crx, NO);
        }
        ***/
#endif /* if defined(DEBUG) */


        if (num_crx == 0)
        {
            /* No interface, but ONFRONT, this happens */
            // blk_mem->num_bonds = 0;
            DEBUG_LEAVE(reconstruct_blk_intfc2d)
            return FUNCTION_SUCCEEDED;
        }

        /* _assign_2dblk_type */ 
        (*blk_crx->assign_2dblk_type)(blk_crx, num_crx, crxs); 

        if(blk_crx->bdry)
        {
            if(! reconstruct_mix_crx_intfc2d(blk_crx, blk_mem, num_crx, crxs))
            {
                printf("ERROR in reconstruct_blk_intfc2d\n");
                printf("reconstruct_multi_crx_intfc2d failed\n");
                print_blk_crx2(blk_crx, YES); 
                clean_up(ERROR);  
            }
            /*
            if(blk_crx->m_crx == YES || blk_crx->blk_type == COMP3_BLOCK ||
               blk_crx->blk_type == COMP1_BLOCK)
            {
                printf("m_crx or comp3_block\n");
                print_blk_crx2(blk_crx, NO); 
            }
            */
            DEBUG_LEAVE(reconstruct_blk_intfc2d)
            return FUNCTION_SUCCEEDED;
        }
        switch (blk_crx->blk_type)
        {
        case COMP2_BLOCK:
            if(! reconstruct_comp2_intfc2d(blk_crx, blk_mem, jc_blk_bond, num_crx, crxs))
            {
                printf("ERROR in reconstruct_blk_intfc2d\n");
                printf("reconstruct_comp2_intfc2d failed\n");
                print_blk_crx2(blk_crx, YES); 
                clean_up(ERROR);  
            }
        break;
        case COMP3_BLOCK:
            if(! reconstruct_comp3_intfc2d(blk_crx, blk_mem, num_crx, crxs))
            {
                printf("ERROR in reconstruct_blk_intfc2d()\n");
                printf("Implement 3comp intfc new case\n");  
                print_blk_crx2(blk_crx, YES); 
                clean_up(ERROR);  
            }
        }

        DEBUG_LEAVE(reconstruct_blk_intfc2d)
        return FUNCTION_SUCCEEDED;
}

/* THIS function is not a real multi_crx intfc 
 * reconstruction implementation. The multi crxs
 * appears only because the tracked curves and
 * the boundary curves happen to be on the same grid line.
 * Only the tracked curves are reconstructed to be grid based.
 */
LOCAL int reconstruct_mix_crx_intfc2d(
        BLK_CRX2    *blk_crx,
        BLK_BOND    *blk_mem,
        int         num_crx, 
        BBI_POINT   *crxs[])
{
        DEBUG_ENTER(reconstruct_mix_crx_intfc2d)

        BBI_POINT   *tmpcrx[20];
        int         i, n_trk;

        n_trk = 0;
        for(i = 0; i < num_crx; i++)
        {
            if((*blk_crx->trk_wv_val)(Curve_of_hs(crxs[i]->hs)) == YES)
            {
                tmpcrx[n_trk] = crxs[i];
                n_trk++;
            }
        }
        if(n_trk == 0)
            return YES;

        if(n_trk%2 == 1)
        {
            if(! create_odd_multi_bonds(blk_crx, blk_mem, 
                      num_crx, crxs, n_trk, tmpcrx))
            {
                printf("ERROR in reconstruct_mix_crx_intfc2d\n");
                printf("n_trk = %d is odd, implement\n", n_trk);
                print_blk_crx2(blk_crx, NO);
                clean_up(ERROR);  
            }
        }
        else
        {
            if(n_trk != 2)
            {
                printf("ERROR in reconstruct_multi_crx_intfc2d\n");
                printf("num_tracked_crx = %d is even > 2, implement\n", n_trk);
                print_blk_crx2(blk_crx, YES);
                clean_up(ERROR);  
            }
            if(! create_even_multi_bonds(blk_crx, blk_mem, 
                      num_crx, crxs, n_trk, tmpcrx))
            {
                printf("ERROR in reconstruct_multi_crx_intfc2d\n");
                printf("Number of Crx = %d\n", num_crx);
                printf("There is either no two connected comps found or"
                       " no node inside the blk\n");
                print_blk_crx2(blk_crx, YES);
            }
        }

        DEBUG_LEAVE(reconstruct_mix_crx_intfc2d)
        return FUNCTION_SUCCEEDED; 
}

LOCAL void in_topology_blk_node(
        int         *blkic,
        CURVE       *c,
        NODE        **node,
        int         *num)
{
        RECT_GRID   *gr; 
        int         ic[MAXD];
     
        gr = &topological_grid(c->interface); 
        (*num) = 0; 
        node[0] = node[1] = NULL;  
      
        if(FUNCTION_FAILED == rect_in_which(Coords(c->start->posn),ic,gr))
        {
            printf("ERROR in_the_blk_node, rect_in_which failed\n");
            clean_up(ERROR);  
        } 
        if(ic[0] == blkic[0] && ic[1] == blkic[1])
        {
            node[(*num)] = c->start;
            (*num)++; 
        }
 
        if(FUNCTION_FAILED == rect_in_which(Coords(c->end->posn),ic,gr))
        {
            printf("ERROR in_the_blk_node, rect_in_which failed 2\n");
            clean_up(ERROR);  
        } 
        if(ic[0] == blkic[0] && ic[1] == blkic[1])
        {
            node[(*num)] = c->end;
            (*num)++; 
        }
}

LOCAL  int create_odd_multi_bonds(
        BLK_CRX2    *blk_crx,
        BLK_BOND    *blk_mem,
        int         num_crx, 
        BBI_POINT   *crxs[],
        int         n_trk, 
        BBI_POINT   *trk_crxs[])
{
        CURVE       *cc, *c[10], **cur;
        NODE        *node = NULL; 
        static NODE **innodes = NULL; 
        int         i, j, nnum, find_comm = NO; 
        int         k[5] = {-1, -1, -1, -1, -1}, n_bonds = 0;
        BOND        *newb;

        if(innodes == NULL)
            vector(&innodes,2,sizeof(NODE*)); 

        if(num_crx < 2)
        {
            printf("ERROR in create_odd_multi_bonds\n");
            printf("num_crx = %d, which is less than 2\n", num_crx);
            printf("With mixed tracked curve and boundary crxings, "
                   "the number should be greater than 2\n");   
            clean_up(ERROR); 
        }        

        if(n_trk > 1)
        {
            int one_node = YES; 
            if(n_trk != 3)
            {
                printf("ERROR in create_odd_multi_bonds\n");
                printf("num_crx = %d\n", num_crx);
                printf("The number of tracked crxings = %d, "
                   "the max.  number should be no greater than 3\n", n_trk);   
                clean_up(ERROR); 
            }
            for(i = 0; i < n_trk; i++)
            {
                for(j = i+1; j < n_trk; j++)
                {
                    if((Curve_of_hs(crxs[i]->hs)->start != 
                        Curve_of_hs(crxs[j]->hs)->start &&
                        Curve_of_hs(crxs[i]->hs)->start !=
                        Curve_of_hs(crxs[j]->hs)->end) &&
                       (Curve_of_hs(crxs[i]->hs)->end !=
                        Curve_of_hs(crxs[j]->hs)->start &&
                        Curve_of_hs(crxs[i]->hs)->end !=
                        Curve_of_hs(crxs[j]->hs)->end))
                    {
                        one_node = NO;  
                    }
                }    
            }    
            if(one_node != YES)
            {
                printf("ERROR in create_odd_multi_bonds\n");
                printf("contains more than one tracked nodes\n");
                printf("Implementing this case\n");
                print_blk_crx2(blk_crx, YES);
                clean_up(ERROR); 
            }
        }

        c[0] = Curve_of_hs(trk_crxs[0]->hs);
        in_topology_blk_node(blk_crx->blkic,c[0],innodes,&nnum); 
        if(nnum != 1)
        {
            printf("ERROR in create_odd_multi_bonds\n");
            printf("Does not found the in blk[%d %d] node\n",
                 blk_crx->blkic[0],blk_crx->blkic[1]);
            print_blk_crx2(blk_crx, YES);
            clean_up(ERROR);  
        }
        node = innodes[0];

        for(i = 0; i < num_crx; i++)
        {
            if(trk_crxs[0] == crxs[i]) continue;
            cc = Curve_of_hs(crxs[i]->hs);
            if(cc->start == node ||
               cc->start == node)
            {
                find_comm = YES; 
                break;
            }
            if(cc->end == node ||
               cc->end == node)
            {
                find_comm = YES; 
                break;
            }
        }

        if(node == NULL || find_comm == NO)
        {
            printf("ERROR in create_odd_multi_bonds\n");
            printf("Blk[%d %d] Does not found the common node %d\n",
                blk_crx->blkic[0],blk_crx->blkic[1], nnum);
            if(innodes[0] != NULL)
                print_node(innodes[0]); 
            if(innodes[1] != NULL)
                print_node(innodes[1]); 

            print_blk_crx2(blk_crx, NO);
            clean_up(ERROR);  
        }

        for(i = 0; i < n_trk; i++)
        {
            for (j = 0; j < blk_mem->num_curves; j++)
            {
                cc = Curve_of_hs(trk_crxs[i]->hs);
                if (cc == blk_mem->curves[j])
                {
                    k[i] = j;
                    break;
                }
            }
        }       
        for(cur = node->in_curves; cur && *cur; cur++)
        {
            for(i = 0; i < n_trk; i++)
            {
                cc = Curve_of_hs(trk_crxs[i]->hs);
                if(cc == *cur)
                {
                    newb = Bond(trk_crxs[i]->p, node->posn);
                    n_bonds++;
                    (blk_mem->num_bonds[k[i]])++;
                    if (blk_mem->first[k[i]] == NULL)
                        blk_mem->first[k[i]] = newb;
                    if(blk_mem->cur_bonds[k[i]] == NULL)
                        blk_mem->cur_bonds[k[i]] = newb;
                    else
                    {
                        newb->next = blk_mem->cur_bonds[k[i]];
                        blk_mem->cur_bonds[k[i]]->prev = newb;
                        blk_mem->cur_bonds[k[i]] = newb;
                    }
                }
            }
        }

        for(cur = node->out_curves; cur && *cur; cur++)
        {
            for(i = 0; i < n_trk; i++)
            {
                cc = Curve_of_hs(trk_crxs[i]->hs);
                if(cc == *cur)
                {
                    newb = Bond(node->posn, trk_crxs[i]->p);
                    n_bonds++;
                    (blk_mem->num_bonds[k[i]])++;
                    if (blk_mem->first[k[i]] == NULL)
                        blk_mem->first[k[i]] = newb;
                    if(blk_mem->cur_bonds[k[i]] == NULL)
                        blk_mem->cur_bonds[k[i]] = newb;
                    else
                    {
                        newb->next = blk_mem->cur_bonds[k[i]];
                        blk_mem->cur_bonds[k[i]]->prev = newb;
                        blk_mem->cur_bonds[k[i]] = newb;
                    }
                }
            }
        }
        
        if(n_bonds != n_trk)
        {
            printf("ERROR create_odd_multi_bonds, n_trk = %d\n", n_trk);       
            printf("num of bonds created = %d\n", n_bonds);       
            print_blk_crx2(blk_crx, YES);
            clean_up(ERROR); 
      
        }
        return YES; 
} 

LOCAL  int create_even_multi_bonds(
        BLK_CRX2    *blk_crx,
        BLK_BOND    *blk_mem,
        int         num_crx, 
        BBI_POINT   *crxs[],
        int         n_trk, 
        BBI_POINT   *trk_crxs[])
{
        BBI_POINT    *ord_crxs[12];
        int          k, find = NO;
        BOND         *newb; 
        CURVE        *c; 

        switch(n_trk)
        {
        case 2:
            if(trk_crxs[0]->hs != trk_crxs[1]->hs)
            {
                printf("ERROR in create_even_multi_bonds\n");
                printf("# tracked crxs = %d, but belong to different"
                       " curves, implement\n", n_trk);
                print_blk_crx2(blk_crx, YES);
                clean_up(ERROR);
            }
            if(NO == sort_crx_in_blk(trk_crxs,ord_crxs,2)) 
            {
                printf("ERROR: IN create_even_multi_bonds(), not sort\n");
                clean_up(ERROR); 
            }
            c = Curve_of_hs(trk_crxs[0]->hs);
            for (k = 0; k < blk_mem->num_curves; k++)
            {
                if (c == blk_mem->curves[k])
                {
                    find = YES;
                    break;
                }
            }
#if defined(DEBUG)
            if(find == NO)
            {
                printf("ERROR: create_even_multi_bonds(), could find k\n");
                clean_up(ERROR);
            }
#endif /* if defined(DEBUG) */
            newb = Bond(ord_crxs[0]->p, ord_crxs[1]->p);
            (blk_mem->num_bonds[k])++;
            if (blk_mem->first[k] == NULL)
                blk_mem->first[k] = newb;
            if(blk_mem->cur_bonds[k] == NULL)
                blk_mem->cur_bonds[k] = newb;
            else
            {
                newb->next = blk_mem->cur_bonds[k];
                blk_mem->cur_bonds[k]->prev = newb;
                blk_mem->cur_bonds[k] = newb;
            }
        break;  
        default:
            printf("ERROR in create_even_multi_bonds\n");
            printf("# tracked crxs = %d, implement\n", n_trk);
            print_blk_crx2(blk_crx, YES);
            clean_up(ERROR);
        }
        return YES; 
} 

LOCAL  int reconstruct_comp3_intfc2d(
        BLK_CRX2    *blk_crx,
        BLK_BOND    *blk_mem,
        int         num_crx, 
        BBI_POINT   *crxs[])
{
        BBI_POINT   *tmpcrx[20];
        int         i, n_trk;

        DEBUG_ENTER(reconstruct_comp3_intfc2d)

        n_trk = 0;
        for(i = 0; i < num_crx; i++)
        {
            if((*blk_crx->trk_wv_val)(Curve_of_hs(crxs[i]->hs)) == YES)
            {
                tmpcrx[n_trk] = crxs[i];
                n_trk++;
            }
        }
        if(n_trk == 0)
            return YES;

        printf("ERROR in reconstruct_comp3_intfc2d\n");
        printf("Mixed with tracked and untracked crxings, n_trk = %d\n",n_trk);
        print_blk_crx2(blk_crx, YES);
        clean_up(ERROR); 

        switch(n_trk)
        {
        case 1:
        break; 
        /*
        case 3:
            if(! two_connect_comps_have_node_create_bonds(
                    blk_crx, blk_mem, num_crx, crxs))
            {
                printf("ERROR in reconstruct_comp3_intfc2d\n");
                printf("Number of Crx = %d\n", num_crx);
                printf("There is either no two connected comps found or"
                       " no node inside the blk\n"); 
                print_blk_crx2(blk_crx, YES);
            }
        break;
        */
        default:
            printf("ERROR in reconstruct_comp3_intfc2d\n");
            printf("Number of Crx = %d\n", num_crx);
            printf("implement this case\n");
            print_blk_crx2(blk_crx, YES);
            clean_up(ERROR);    
        }

        DEBUG_LEAVE(reconstruct_comp3_intfc2d)
        return FUNCTION_SUCCEEDED; 
}

/* The configuration of this crossing blk should be 
 * like this: 
 *    --------------X1--------------
 *    |             |              |
 *    |             |              |
 *    |             N-------------X2
 *    |             |              |
 *    |             |              |
 *    --------------X3--------------
 *    X1, x2 and X3 are three crossings
 *    N is the node joining the curves passing 
 *    through these crossings.
 *    We do not worry about wave types there. 
 */
LOCAL int two_connect_comps_have_node_create_bonds(
        BLK_CRX2    *blk_crx,
        BLK_BOND    *blk_mem,
        int         num_crx, 
        BBI_POINT   *crxs[])
{
        NODE        *node = NULL;
        CURVE       **cc, *c[3]; 
        BOND        *newb;
        int         k[3] = {-1, -1, -1}; 
        int         joined, i, j; 
        int         n_bonds = 0;

        c[0] = Curve_of_hs(crxs[0]->hs);
        c[1] = Curve_of_hs(crxs[1]->hs);
        c[2] = Curve_of_hs(crxs[2]->hs);

        /* First test if they share the same node */
        if(c[0]->start == c[1]->start)
            node = c[0]->start;
        else if(c[0]->start == c[1]->end)
            node = c[0]->start;
        else if(c[0]->end == c[1]->start)
            node = c[0]->end;
        else if(c[0]->end == c[1]->end)
            node = c[0]->end;
        else
        {
            printf("ERROR In two_connect_comps_have_node_create_bonds()\n");
            printf("Two crossings does not meet at same node\n");
            print_curve(c[0]);
            print_curve(c[1]);
            clean_up(ERROR);
        }
        
        if(c[2]->start == node)
        {
            NULL;
        }
        else if(c[2]->end == node)
        {
            NULL; 
        }
        else
        {
            printf("ERROR In two_connect_comps_have_node_create_bonds()\n");
            printf("Two crossings  meet at same node\n");
            printf("But the third crossing does not meet\n"); 
            print_curve(c[0]);
            print_curve(c[1]);
            print_curve(c[2]);
            clean_up(ERROR);
        }

        /* Test if there is a pair of components are directly connected */
        /* This test seems to be redundant, but do it anyway */
        joined = NO;
        for(i = 0; i < 2; i++)
        {
            if(blk_crx->comp[0][i][0] == blk_crx->comp[1][i][0])
            {
                joined = YES;
                break; 
            }
        }
        if(joined == NO)
        {
            for(i = 0; i < 2; i++)
            {
                if(blk_crx->comp[i][0][0] == blk_crx->comp[i][1][0])
                {
                    joined = YES;
                    break; 
                }
            }
        }
        if(joined == NO)
        {
            printf("ERROR In two_connect_comps_have_node_create_bonds()\n");
            printf("There is not connected same component pairs\n");
            print_blk_crx2(blk_crx, YES); 
            clean_up(ERROR);  
        }

        for(i = 0; i < 3; i++)
        {
            for (j = 0; j < blk_mem->num_curves; j++)
            {
                if (c[i] == blk_mem->curves[j])
                {
                    k[i] = j; 
                    break;
                }
            }
        }        

        for(i = 0; i < 3; i++)
        {
            if(k[i] == -1)
            {
                printf("ERROR In two_connect_comps_have_node_create_bonds()\n");
                printf("curve[%d] does not have a curve curve number\n", i);
                print_curve(c[i]); 
                print_blk_crx2(blk_crx, YES); 
                clean_up(ERROR);  
            }
        }

        for( cc = node->in_curves; cc && *cc; cc++)
        {
            for(i = 0; i < 3; i++)
            {
                if(c[i] == *c)
                {
                    newb = Bond(crxs[i]->p, node->posn);
                    n_bonds++;
                    (blk_mem->num_bonds[k[i]])++;
                    if (blk_mem->first[k[i]] == NULL)
                        blk_mem->first[k[i]] = newb;
                    if(blk_mem->cur_bonds[k[i]] == NULL)
                        blk_mem->cur_bonds[k[i]] = newb;
                    else
                    {
                        newb->next = blk_mem->cur_bonds[k[i]];
                        blk_mem->cur_bonds[k[i]]->prev = newb;
                        blk_mem->cur_bonds[k[i]] = newb;
                    }
                }
            }
        }

        for( cc = node->out_curves; cc && *cc; cc++)
        {
            for(i = 0; i < 3; i++)
            {
                if(c[i] == *c)
                {
                    newb = Bond(node->posn, crxs[i]->p);
                    n_bonds++;
                    (blk_mem->num_bonds[k[i]])++;
                    if (blk_mem->first[k[i]] == NULL)
                        blk_mem->first[k[i]] = newb;
                    if(blk_mem->cur_bonds[k[i]] == NULL)
                        blk_mem->cur_bonds[k[i]] = newb;
                    else
                    {
                        newb->next = blk_mem->cur_bonds[k[i]];
                        blk_mem->cur_bonds[k[i]]->prev = newb;
                        blk_mem->cur_bonds[k[i]] = newb;
                    }
                }
            }
        }
#if defined(DEBUG)
        if(n_bonds != 3)
        {
            printf("ERROR in two_connect_comps_have_node_create_bonds()\n");
            printf("num of created bonds = %d, not 2\n", n_bonds);
            clean_up(ERROR);
        }
#endif /* if defined(DEBUG) */
        return YES; 
}

LOCAL  int reconstruct_comp2_intfc2d(
        BLK_CRX2    *blk_crx,
        BLK_BOND    *blk_mem,
        JC_BLK_BOND *jc_blk_bond,
        int         num_crx, 
        BBI_POINT   *crxs[])
{
        int         i, j;
        COMPONENT   ***comp = blk_crx->comp;
        /* BLOCK blk_bin; */
        int         num_pos;

        DEBUG_ENTER(reconstruct_comp2_intfc2d)

        blk_crx->pos_comp = crxs[0]->hs->pos_comp;
        blk_crx->neg_comp = crxs[0]->hs->neg_comp;

        // blk_mem->num_bonds = 0;
        num_pos = 0;
        for (i = 0; i < 2; i++)
        {
            for (j = 0; j < 2; j++)
            {
                if (comp[i][j][0] == blk_crx->pos_comp)
                    num_pos++;
            }
        }

        switch(num_pos)
        {
        case 1:
            if(blk_crx->num_curves == 1)
                find_odd_point_and_create_bond(blk_crx, blk_mem);
            else if(blk_crx->num_curves == 2 && 
                    blk_crx->num_waves == 2)
            {
                /* THE two CRXs belongs to two curves (different type)  respectively. 
                 * There must be a node to connect these two crxs. 
                 */ 
                find_odd_point_and_create_2bonds(blk_crx, blk_mem);
            }
            else
            {
                if(blk_crx->bdry == YES)
                    find_odd_point_and_create_2bonds(blk_crx, blk_mem);
                else
                {
                    printf("ERROR in reconstruct_comp2_intfc2d\n");
                    printf("Case 1: number_curves inside one blk = %d\n", blk_crx->num_curves);
                    printf("This is an possible case that two same type curves joining\n");  
                    printf("Number of waves inside = %d\n", blk_crx->num_waves);  
                    print_blk_crx2(blk_crx,YES);
                    clean_up(ERROR); 
                }
            }
        break;
        case 3:
            blk_crx->neg_comp = crxs[0]->hs->pos_comp;
            blk_crx->pos_comp = crxs[0]->hs->neg_comp;
            num_pos = 0;
            for (i = 0; i < 2; i++)
            {
                 for (j = 0; j < 2; j++)
                 {
                     if (comp[i][j][0] == blk_crx->pos_comp)
                         num_pos++;
                 }
            }
            if(num_pos != 1)
            {
                printf("ERROR in reconstruct_comp2_intfc2d()\n");
                printf("component case num_pos = %d, not 1\n", num_pos);
                print_blk_crx2(blk_crx, YES);
                clean_up(ERROR);  
            }
            if(blk_crx->num_curves == 1)
                find_odd_point_and_create_bond(blk_crx, blk_mem);
            else if(blk_crx->num_curves == 2 && 
                    blk_crx->num_waves == 2)
            {
                printf("WARNING in reconstruct_comp2_intfc2d()\n");
                printf("WARNING: case 3 implementing two curves case\n"); 
                print_blk_crx2(blk_crx, YES);
                find_odd_point_and_create_2bonds(blk_crx, blk_mem);
                clean_up(ERROR); 
            }
            else
            {
                printf("ERROR in reconstruct_comp2_intfc2d\n");
                printf("Case 3: number_curves inside one blk = %d\n", blk_crx->num_curves);
                printf("Number of waves inside = %d\n", blk_crx->num_waves);  
                printf("This is an possible case that two same type curves joining\n");  
                print_blk_crx2(blk_crx, YES);
                clean_up(ERROR);
            }           
        break;
        case 2:
            if(blk_crx->num_curves == 1)
            {
                if(! create_two_point_bond(blk_crx, blk_mem))
                {
                     return FUNCTION_FAILED;
                }
            }
            else if(blk_crx->num_curves == 2)
            {
                if(! create_join_c_2p_bond(blk_crx, blk_mem, jc_blk_bond))
                {
                    printf("ERROR create_join_c_2p_bond failed\n"); 
                    clean_up(ERROR); 
                } 
            }
            else
            {
                printf("ERROR in reconstruct_comp2_intfc2d\n");
                printf("Case 2: number_curves inside one blk = %d\n", blk_crx->num_curves);
                printf("Number of waves inside = %d\n", blk_crx->num_waves);  
                printf("This is an impossible case\n");
                print_blk_crx2(blk_crx, YES);
                clean_up(ERROR);
            }
        break;
        default:
            printf("ERROR in reconstruct_blk_intfc2d()\n");
            printf("Unkown component case num_pos = %d\n", num_pos);
            clean_up(ERROR);
        break;
        }

        DEBUG_LEAVE(reconstruct_comp2_intfc2d)
        return FUNCTION_SUCCEEDED;
}

LOCAL int create_join_c_2p_bond(
        BLK_CRX2    *blk_crx,
        BLK_BOND    *blk_mem,
        JC_BLK_BOND *jc_blk_bond)
{
        int          i,j,k;
        COMPONENT    min_comp = blk_crx->pos_comp;
        COMPONENT    ***comp = blk_crx->comp;
        COMPONENT    pos_comp, neg_comp;
        int          num_isolated = 0;
        int          num_crx;
        BBI_POINT    *crxs[4], *ord_crxs[4];
        BOND         *newb, *newb2;
        POINT        tmpp;
        JC_BLK_BOND  *tmpjc_bond, *tmpjcb; 

        /* Search for isolated points */
        for (i = 0; i < 2; i++)
        {
            for (j = 0; j < 2; j++)
            {
                if (comp[i][j][0] == min_comp)
                {
                    if (is_isolated_point(comp,i,j,0,min_comp) == YES)
                    {
                        num_isolated++;
                    }
                }
            }
        }

        if(num_isolated == 0)
        {
            num_crx = 0;
            for (j = 0; j < 2; j++)
            {
                if (comp[0][j][0] != comp[1][j][0])
                {
                    crxs[num_crx] = crx_in_idir2d(blk_crx,j,0,0);
#if defined(DEBUG)
                    if(crxs[num_crx] == NULL)
                    {
                        printf("ERROR in create_join_c_2p_bond\n");
                        printf("i's crxs[%d] = NULL\n", num_crx);
                        clean_up(ERROR);
                    }
#endif /*if defined(DEBUG)  */
                    num_crx++;
                }
            }
            if(num_crx == 2)
            {
#if defined(DEBUG)
                if(crxs[0] != crx_in_idir2d(blk_crx,0,0,0))
                {
                    printf("ERROR in create_join_c_2p_bond\n");
                    printf("crxs[0] != crx_in_idir2d(blk_crx,0,0)\n");
                    clean_up(ERROR);
                }
                if (crxs[0]->hs == crxs[1]->hs)
                {
                    screen("ERROR in create_join_c_2p_bond(),"
                        " crxs from same surfs\n");
                    clean_up(ERROR);
                }
#endif /* if defined(DEBUG) */

                pos_comp = crxs[0]->hs->pos_comp;
                neg_comp = crxs[0]->hs->neg_comp;
#if defined(DEBUG)
                if(crxs[0]->lcomp != crxs[1]->lcomp ||
                   crxs[0]->ucomp != crxs[1]->ucomp)
                {
                    printf("ERROR in create_join_c_2p_bond()"
                       "crxs[0]->lcomp != crxs[1]->lcomp\n");
                    clean_up(ERROR);
                }
#endif /* if defined(DEBUG) */
                if(crxs[0]->ucomp == pos_comp)
                    newb = Bond(crxs[0]->p, crxs[1]->p);
                else if(crxs[0]->ucomp == neg_comp)
                    newb = Bond(crxs[1]->p, crxs[0]->p);
                else
                {
                    printf("ERROR In create_join_c_2p_bond(), 0\n");
                    clean_up(ERROR);
                }
                scalar(&tmpjc_bond,sizeof(JC_BLK_BOND)); 
                tmpjc_bond->prev = tmpjc_bond->next = NULL; 
                tmpjc_bond->jc_bond = newb;
                tmpjc_bond->c0 = Curve_of_hs(crxs[0]->hs);
                tmpjc_bond->c1 = Curve_of_hs(crxs[1]->hs); 

                tmpjcb = jc_blk_bond;
                while(tmpjcb->next)
                    tmpjcb = tmpjcb->next;
                tmpjcb->next = tmpjc_bond;
                tmpjc_bond->prev = tmpjcb;

                return FUNCTION_SUCCEEDED;
            }
        }
        else
        {
            printf("ERROR create_join_c_2p_bond\n");
            printf("Implement num_isolated = %d case\n",
                 num_isolated); 
            clean_up(ERROR); 
        }

        return FUNCTION_FAILED;
}

#if defined(CONSERVATIVE_ALG)

EXPORT int prt_pt_corresp_curve(
        CURVE    *c,
        int      prt_diff_only)
{
        BOND     *b = c->first;
        int      num = 0; 

        printf("\t Curve#[%llu]'s construction curv_id[%d]:\n",
               curve_number(c), c->curv_id);

        if(prt_diff_only == YES)
        {
            if(b->start->curv_id != c->curv_id)
            {
                printf("[%14.12f %14.12f] indx = %d, corresp C %d crx[%d]\n",
                 Coords(b->start)[0],Coords(b->start)[1],
                 b->start->indx, b->start->curv_id, b->start->crx);
                num++;
            }
        }
        else
           printf("[%14.12f %14.12f] indx = %d, corresp C %d crx[%d]\n",
             Coords(b->start)[0],Coords(b->start)[1],
             b->start->indx, b->start->curv_id, b->start->crx);
       while(b)
       {
            if(prt_diff_only == YES)
            {
                if(b->end->curv_id != c->curv_id)
                {
                    printf("[%14.12f %14.12f] indx = %d, corresp C %d crx[%d]\n",
                     Coords(b->end)[0],Coords(b->end)[1],
                     b->end->indx, b->end->curv_id, b->end->crx);
                    num++;
                }
            }
            else
               printf("[%14.12f %14.12f] indx = %d, corresp C %d crx[%d]\n",
                 Coords(b->end)[0],Coords(b->end)[1],
                 b->end->indx, b->end->curv_id, b->end->crx);
           b = b->next;
       }
       return num;
}

#endif /* if defined(CONSERVATIVE_ALG) */

#endif /* if defined(TWOD) */
