/*
*				hnpt.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains n-point stencil code for finite difference solution
*	of the hyperbolic equations within a single component of a tracked
*	interface problem.
*	Point sources in the region are found and passed appropriately.
*/


#include <hyp/hlocaldecs.h>

#define DEBUG_HYP_NPT

LOCAL	int	gmax[MAXD], vsten_rad, is_vec_method;
LOCAL	int	dim;
LOCAL   int	lbuf[MAXD], ubuf[MAXD];
LOCAL	float	h[MAXD];
LOCAL	float	L[MAXD], VL[MAXD], VU[MAXD];

LOCAL   bool Debug;

	/* Redefine rect grid macros for local efficiency */
	/* these will have to be modified to support variable mesh grids */
#undef cell_center
#define cell_center(indx,i,gr)	(L[i] + ((indx) + 0.5)*h[i])

	/* LOCAL Function Declarations */
LOCAL   COMPONENT       Find_rect_comp(int,int*,Wave*);
LOCAL	COMPONENT	set_stencil_comps_and_crosses(int,Stencil*);
LOCAL	bool	interior_sten_reg(int,int,Stencil*);
LOCAL	void	copy_states(int,CRX_SIDE,int,int,Stencil*,CRXING*);
LOCAL   void    find_outer_states(int,int,Stencil*,CRX_SIDE);
LOCAL	void	set_dirichlet_bdry_sten_states(int,HYPER_SURF*,Stencil*,
                                               CRX_SIDE);
LOCAL	void	set_neumann_bdry_sten_states(int,int,int,COMPONENT,CRXING*,
					     Stencil*,CRX_SIDE);
LOCAL	void	update_reg_grid_state(float,float,int,int*,int,
				      Stencil*,POINT*,COMPONENT);
#if defined(TWOD) || defined(THREED)
LOCAL   void    set_static_coords(int**,POINT*,int,int,RECT_GRID*);
#endif /* defined(TWOD) || defined(THREED) */

#if defined(CONSERVATIVE_ALG) && defined(TWOD)

LOCAL  COMPONENT set_unsplit_stencil_comps_and_crosses(int,int,int*,Stencil*);
LOCAL   bool     unsplit_interior_sten_reg(int,int,int,int*,Stencil*);
LOCAL   void     find_out_states_in_other_dir(int,int,int,int*,Stencil*);
LOCAL   void     unsplit_update_reg_grid_state(float*,float,int,int*,int,
                       Stencil*,POINT*,COMPONENT);
LOCAL   void     set_neumann_bdry_sten_Tstates(int,int,float*,int,COMPONENT,CRXING*,
                                             Stencil*);
LOCAL   void     copy_Tstates(int,float*,int,COMPONENT,Stencil*,CRXING*);
LOCAL   COMPONENT  Find_rect_comp2(int*,Wave*);  
LOCAL   int      crds_of_ic(float*,Wave*,int*);

LOCAL  float     hndir; 

#endif /* #if defined(CONSERVATIVE_ALG) && defined(TWOD) */



EXPORT	void set_hyp_npt_globals(
	Wave		*wave)
{
	RECT_GRID	*r_grid;
	int		i;

	r_grid = wave->rect_grid;
	dim = r_grid->dim;
	for (i = 0; i < dim; ++i)
	{
	    gmax[i] = r_grid->gmax[i];
	    h[i] = r_grid->h[i];
	    L[i] = r_grid->L[i];
	    VL[i] = r_grid->VL[i];
	    VU[i] = r_grid->VU[i];
	    lbuf[i] = r_grid->lbuf[i];
	    ubuf[i] = r_grid->ubuf[i];
	}
	vsten_rad = vsten_radius(wave);
	is_vec_method = is_vector_method(wave);
}		/*end set_hyp_npt_globals*/

/*
*	The following globals are shared between hyp_npt() and
*	the subroutine update_reg_grid_state().  They are used
*	to pass quantities that do not change inside the inner loop
*	of the update step.  The are passed as globals for efficiency.
*/

LOCAL	COMPONENT	ext_comp;
LOCAL	float		hdir, dir[MAXD];
LOCAL	GRID_DIRECTION	prev_side, next_side;
LOCAL	INTERFACE	*intfc = NULL;
LOCAL	int		endpt;


/*
*			hyp_npt():
*
*	Single direction sweep for a n-point scheme.
*/

EXPORT void hyp_npt(
	int		swp_num,
	int		*iperm,	    /* sweep based permutation of coord dir */
	float		ds,
	float		dt,
	Wave		*wave,
	Wave		*newwave,
	Front		*fr,
	Front		*newfr,	    /* newfr needed if hlw is to support */
		    		    /*	changing topologies	     */
	COMPONENT	max_comp)
{
	POINT		Basep;
	int 		i, idirs[MAXD];
	int		imin[3], imax[3];
	static	Stencil	*sten = NULL;
	static	int	num_pts;
#if defined(TWOD) || defined(THREED)
	static	int	**icoords = NULL;
	RECT_GRID	*gr = wave->rect_grid;
#endif /* defined(TWOD) || defined(THREED) */

	for (i = 0; i < dim; ++i)
	{
	    idirs[i] = iperm[(i+swp_num)%dim];
	    dir[i] = 0.0;
	}
	dir[idirs[0]] = 1.0;
	debug_print("hyp_npt","Entered hyp_npt(), swp_num = %d, dir = %d\n",
			swp_num,idirs[0]);

	intfc = fr->interf;
	ext_comp = exterior_component(intfc);

	if (sten == NULL)
	{
	    /* This assumes that the stencil size is constant. */
	    num_pts = wave->npts_sten;
	    endpt = stencil_radius(wave);

	    sten = alloc_stencil(num_pts,fr);
#if defined(TWOD) || defined(THREED)
	    icoords = sten->icoords;
#endif /* defined(TWOD) || defined(THREED) */
	}

	hdir = ds;		/* grid spacing in sweep direction */
	switch (idirs[0])
	{
	case 0:	/*X SWEEP */
	    prev_side = WEST;
	    next_side = EAST;
	    if (debugging("hyp_npt"))
	    	(void) printf("prev_side = WEST, next_side = EAST\n");
	    break;
	case 1: /*Y SWEEP */
	    prev_side = SOUTH;
	    next_side = NORTH;
	    if (debugging("hyp_npt"))
	    	(void) printf("prev_side = SOUTH, next_side = NORTH\n");
	    break;
	case 2: /* Z SWEEP */
	    prev_side = LOWER;
	    next_side = UPPER;
	    if (debugging("hyp_npt"))
	    	(void) printf("prev_side = LOWER, next_side = UPPER\n");
	    break;
	}

	sten->fr   = fr;	sten->newfr   = newfr;
	sten->wave = wave;	sten->newwave = newwave;

	set_sweep_limits(wave,swp_num,idirs,imin,imax);

	switch (dim)
	{
#if defined(ONED)
	case 1:
	{
	    int	i0;
	    int	i0max;
	    int	i0min;

	    i0min = imin[0];	i0max = imax[0]; 
	    sten->reg_stencil = NO;
	    for (i0 = i0min; i0 < i0max; ++i0) 
	    {
	    	update_reg_grid_state(ds,dt,swp_num,iperm,
				      i0,sten,&Basep,max_comp);
	    }
	    break;
	}
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	{
	    int	i0, i1;
	    int	i0max, i1max;
	    int	i0min, i1min;

	    i0min = imin[0];	i0max = imax[0]; 
	    i1min = imin[1];	i1max = imax[1]; 
	    for (i1 = i1min; i1 < i1max; ++i1)
	    {
	    	set_static_coords(icoords,&Basep,idirs[1],i1,gr);
	    	sten->reg_stencil = NO;
	    	for (i0 = i0min; i0 < i0max; ++i0) 
	    	{
	    	    update_reg_grid_state(ds,dt,swp_num,iperm,
	    	    		          i0,sten,&Basep,max_comp);
	    	}
	    }
	    break;
	}
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	{
	    int	i0, i1, i2;
	    int	i0max, i1max, i2max;
	    int	i0min, i1min, i2min;

	    i0min = imin[0];	i0max = imax[0]; 
	    i1min = imin[1];	i1max = imax[1]; 
	    i2min = imin[2];	i2max = imax[2];
	    for (i2 = i2min; i2 < i2max; ++i2)
	    {
	    	set_static_coords(icoords,&Basep,idirs[2],i2,gr);
	    	for (i1 = i1min; i1 < i1max; ++i1)
	    	{
	    	    set_static_coords(icoords,&Basep,idirs[1],i1,gr);
		    sten->reg_stencil = NO;
		    for (i0 = i0min; i0 < i0max; ++i0) 
		    {
			update_reg_grid_state(ds,dt,swp_num,iperm,i0,
					      sten,&Basep,max_comp);
		    }
		}
	    }
	    break;
	}
#endif /* defined(THREED) */
	}
	debug_print("hyp_npt","Leaving hyp_npt(), dir = %d\n",idirs[0]);
}		/*end hyp_npt*/


#if defined(TWOD) || defined(THREED)
/*
*			set_static_coords():
*/

/*ARGSUSED*/
LOCAL	void set_static_coords(
	int		**icoords,
	POINT		*basep,
	int		idir,
	int		is,
	RECT_GRID	*gr)
{
	int		i;

	for (i = -endpt; i <= endpt; ++i)
	    icoords[i][idir] = is;
	Coords(basep)[idir] = cell_center(is,idir,gr);
}		/*end set_static_coords*/
#endif /* defined(TWOD) || defined(THREED) */


/*
*			interior_sten_reg():
*
*	If called after a vector solver, decides whether the stencil at a given
*	point was regular for the vector solver.  In other words, decides
*	whether solution needs to be recomputed at a given point.
*/

LOCAL	bool interior_sten_reg(
	int		is,
	int		idir,
	Stencil		*sten)
{
	Wave		*wave = sten->wave;
	Front		*newfr = sten->newfr;
	int 		i;
	int 		**icoords = sten->icoords, icrds[MAXD];
	COMPONENT 	cmp;
	COMPONENT	new_comp = sten->newcomp, *comp = sten->comp;
	CRXING		*cross;

	if (ComponentIsFlowSpecified(new_comp,newfr))
	    return NO;
	for (i = -endpt; i <= endpt; ++i)
	    if ((equivalent_comps(comp[i],new_comp,newfr->interf) == NO) ||
		(sten->nc[i] != 0))
	    return NO;
	if (endpt >= vsten_rad)
	    return YES;

	for (i = 0; i < dim; ++i)
	    icrds[i] = icoords[0][i];
	for (i = endpt+1; i <= vsten_rad; ++i)
	{
	    icrds[idir] = is - i;
	    cmp = Find_rect_comp(idir,icrds,wave);
	    cross = __Rect_crossing(icrds,next_side,wave);
	    if ((equivalent_comps(cmp,new_comp,newfr->interf) == NO) ||
		(cross != NULL))
	    	return NO;
		
	    icrds[idir] = is + i;
	    cmp = Find_rect_comp(idir,icrds,wave);
	    cross = __Rect_crossing(icrds,prev_side,wave);
	    if ((equivalent_comps(cmp,new_comp,newfr->interf) == NO) ||
		(cross != NULL))
		return NO;
	}
	return YES;
}		/*end interior_sten_reg*/

/*
*			update_reg_grid_state():
*
*	Computes the solution at a given point based on at stencil of
*	surrounding points.
*/

LOCAL	void update_reg_grid_state(
	float		ds,
	float		dt,
	int		swp_num,
	int		*iperm,
	int		is,
	Stencil		*sten,
	POINT		*basep,
	COMPONENT	max_comp)
{
	Wave		*wave = sten->wave;
	Wave		*newwave = sten->newwave;
	Front		*fr = sten->fr;
	Front		*newfr = sten->newfr;
	COMPONENT	*comp;
	COMPONENT	new_comp;		/* mid comp wrt newfr */
	HYPER_SURF	*hs;
	int		i, j, index, idir = iperm[swp_num];
	int		**icoords = sten->icoords;

#if defined(DEBUG_HYP_NPT)
	int		deb_hyp_npt = NO;

static	char fname[3][11] = {"xhyp_npt()","yhyp_npt()","zhyp_npt()"};
	if (debugging("hyp_npt"))			deb_hyp_npt = YES;
	else if (debugging("xhyp_npt") && idir == 0)	deb_hyp_npt = YES;
	else if (debugging("yhyp_npt") && idir == 1)	deb_hyp_npt = YES;
	else if (debugging("zhyp_npt") && idir == 2)	deb_hyp_npt = YES;
	else						deb_hyp_npt = NO;
#endif /* defined(DEBUG_HYP_NPT) */

	for (i = -endpt; i <= endpt; ++i)
	    icoords[i][idir] = is + i;
	sten->prev_reg_stencil = sten->reg_stencil;

		/* Find components and crosses*/

	new_comp = set_stencil_comps_and_crosses(idir,sten);
	comp = sten->comp;

#if defined(DEBUG_HYP_NPT)
	if (deb_hyp_npt)
	{
	    (void) printf("\n%s - ",fname[idir]);
	    print_int_vector("icoords[0] = ",icoords[0],dim,"\n");
	    (void) printf("\n comps: ");
	    for (i = -endpt; i <= endpt; ++i)
	    	(void) printf("comp[%d] %d ",i,comp[i]);
	    (void) printf("new_comp %d\n",new_comp);
	}
#endif /* defined(DEBUG_HYP_NPT) */

	if (is_vec_method)
	{
	    if (interior_sten_reg(is,idir,sten))
	    	return;		/*Don't overwrite vector solution*/
	    else
	    	sten->prev_reg_stencil = NO;
	}

	for (j = 0; j < dim; ++j)
	{
	    if (j == idir)
	    {
	    	Coords(sten->p[0])[idir] = cell_center(is,idir,wave->rect_grid);
		for (i = 1; i <= endpt; ++i)
		{
		    Coords(sten->p[-i])[idir] = Coords(sten->p[0])[idir]-i*hdir;
		    Coords(sten->p[i])[idir]  = Coords(sten->p[0])[idir]+i*hdir;
		}
	    }
	    else
	    {
	    	for (i = -endpt; i <= endpt; ++i)
		    Coords(sten->p[i])[j] = Coords(basep)[j];
	    }
	}
	sten->reg_stencil = YES;
	if (RegionIsFlowSpecified(Rect_state(icoords[0],newwave),
			          Rect_state(icoords[0],wave),
			          Coords(sten->p[0]),new_comp,comp[0],fr))
	{
	    sten->reg_stencil = NO;
	    return;
	}

	if ((new_comp > max_comp) && (swp_num > 0))
	{
	    sten->reg_stencil = NO;
	    assign(Rect_state(icoords[0],newwave),
	    	   Rect_state(icoords[0],wave),fr->sizest);
	    return;
	}

	if (new_comp > max_comp)
	{
	    sten->reg_stencil = NO;
	    nearest_intfc_state_and_pt(Coords(sten->p[0]),new_comp,newfr,fr,
				       sten->worksp[0],Coords(sten->p[0]),&hs);
	    assign(Rect_state(icoords[0],newwave),sten->worksp[0],fr->sizest);
	    return;
	}

		/* Find mid state */
	if (equivalent_comps(comp[0],new_comp,newfr->interf) == YES)
	    sten->st[0] = Rect_state(icoords[0],wave);
	else 
	{
            if (!nearest_crossing_state_with_comp(icoords[0],
                        Coords(sten->p[0]),new_comp,
                        wave_tri_soln(newwave)->tri_grid,&sten->st[0]))
            {
	        nearest_intfc_state_and_pt(Coords(sten->p[0]),new_comp,fr,
				       newfr,sten->worksp[0],
				       Coords(sten->p[0]),&hs);
	        sten->st[0] = sten->worksp[0];
            }
	    sten->reg_stencil = NO;
	}
	assign(left_state(sten->p[0]),sten->st[0],fr->sizest);
	assign(right_state(sten->p[0]),sten->st[0],fr->sizest);

	find_outer_states(is,idir,sten,PREV);
	find_outer_states(is,idir,sten,NEXT);

#if defined(DEBUG_HYP_NPT)
	if (deb_hyp_npt)
	{
	    (void) printf("STENCIL: "); print_Stencil(sten);
	    for (i = -endpt; i <= endpt; ++i)
	    {
	    	(void) printf("state[%d]:	",i);
	    	(*fr->print_state)(sten->st[i]);
	    }
	}
#endif /* defined(DEBUG_HYP_NPT) */

 		/* Check for point source/sink in icoords */

	index = (wave->num_point_sources) ?
	    is_source_block(wave,newfr->interf,new_comp,icoords[0]) : -1;

 		/* Update state */

	npt_solver(ds,dt,Rect_state(icoords[0],newwave),dir,swp_num,iperm,
		   &index,sten,wave);

#if defined(DEBUG_HYP_NPT)
	if (deb_hyp_npt)
	{
	    (void) printf("ANSWER: ");
	    (*fr->print_state)(Rect_state(icoords[0],newwave));
	}
#endif /* defined(DEBUG_HYP_NPT) */
}		/*end update_reg_grid_state*/

/*
*			set_stencil_comps_and_crosses():
*
*	This function loads the components of each point and any crosses
*	that occur in the interior of the stencil.  For the crosses,
*	each stencil entry contains the list of crosses lying on its interior
*	side, and these lists are ordered towards the outer points.
*	Thus sten->crx[0] is meaningless, and sten->crx[1] contains the list
*	of crosses (if any) in order FROM point 0 TO point 1.  Periodic
*	boundaries are essentially invisible (cf Find_rect_comp).
*/

LOCAL	COMPONENT set_stencil_comps_and_crosses(
	int		idir,
	Stencil		*sten)
{
	Wave		*wave = sten->wave;
	Wave		*newwave = sten->newwave;
	int		**icoords = sten->icoords;
	TRI_GRID	*tg = wave_tri_soln(wave)->tri_grid;
	int		i, *nc = sten->nc;
	CRXING		***crx = sten->crx;
	COMPONENT	*comp = sten->comp;

	sten->newcomp = Rect_comp(icoords[0],newwave);
	comp[0] = Rect_comp(icoords[0],wave);
	crx[0][0] = NULL;
	nc[0] = 0;
	sten->hs[0] = NULL;/*TODO REMOVE*/

	for (i = 1; i <= endpt; ++i)
	{
	    sten->hs[-i] = sten->hs[i] = NULL;/*TODO REMOVE*/
	    comp[-i] = Find_rect_comp(idir,icoords[-i],wave);
	    nc[-i] = crossings_in_direction(crx[-i],icoords[-i+1],prev_side,tg);
	    comp[i] = Find_rect_comp(idir,icoords[i],wave);
	    nc[i] = crossings_in_direction(crx[i],icoords[i-1],next_side,tg);
	}
	return sten->newcomp;
}		/*set_stencil_comps_and_crosses*/

/*
*			Find_rect_comp():
*
*	Finds the component of a given point using the macro Rect_comp.
*/

LOCAL COMPONENT	Find_rect_comp(
	int		idir,
	int		*icoords,
	Wave		*wave)
{
	int		is, imax;
	
	is = icoords[idir];
	imax = gmax[idir] + ubuf[idir];

	if ((is < -lbuf[idir]) || (is >= imax))
	    return ext_comp;
	else
	    return Rect_comp(icoords,wave);
		
}		/*Find_rect_comp*/

/*
*			find_outer_states():
*
*	Loads appropriate states for the outer indices of a stencil on either
*	the prev or next sides.  If an interface is crossed, an appropriate
*	state is picked off the interface, and copied out to the end of the
*	stencil (cf copy_states).  Neumann boundaries are also handled
*	appropriately (cf set_neumann_bdry_sten_states).
*/

LOCAL	void find_outer_states(
	int	 is,
	int	 idir,
	Stencil	 *sten,
	CRX_SIDE side)
{
	Wave	   *wave = sten->wave;
	Front	   *fr = sten->fr;
	Front	   *newfr = sten->newfr;
	COMPONENT  *comp = sten->comp, new_comp = sten->newcomp;
	int	   **icoords = sten->icoords;
	int	   i, indx;
	CRXING	   *crx;
	HYPER_SURF *hs;
	CRXING	   *new_crx[MAX_NUM_CRX];

	crx = NULL;
	for  (i = 1; i <= endpt; ++i)
	{
	    indx = i * side;
	    if (equivalent_comps(comp[indx],new_comp,newfr->interf) == YES)
	    {
	        sten->st[indx] = Rect_state(icoords[indx],wave);
	        continue;
	    }
	    sten->reg_stencil = NO;
	    crx = sten->crx[indx][0];
	    if (crx != NULL)
	    {
	        /*there is a cross between indx and in_one*/

	        (void) crossings_in_direction(new_crx,icoords[(i-1)*side],
	                               (side == PREV) ? prev_side : next_side,
	                               wave_tri_soln(sten->newwave)->tri_grid);

	        hs = crx->hs;
		if ((new_crx[0] == NULL) ||
		    (wave_type(hs) != wave_type(new_crx[0]->hs)))
		{
		   /* crx has moved out of stencil OR
		    * another curve has moved between indx and the
		    * old crx over the course of the time step. */

	            copy_states(i,side,endpt,new_comp,sten,crx);
	            return;
		}
	        if ((wave_type(hs) == NEUMANN_BOUNDARY) &&
	            (fr->neumann_bdry_state))
	        {
	            set_neumann_bdry_sten_states(i,is,idir,new_comp,
	                                         crx,sten,side);
	            return;
	        }
	        else if (wave_type(hs) == DIRICHLET_BOUNDARY)
	        {
	            set_dirichlet_bdry_sten_states(i,hs,sten,side);
	            return;
	        }
	        else
	        {  
	            /* crx is interior */

	            copy_states(i,side,endpt,new_comp,sten,crx);
	            return;
	        }
	    }
	    else
	    {
	        float		   *coords;
	        static const float OFFSET = 0.01; /*TOLERANCE*/

	        coords = Coords(sten->p[indx]);
	        if (is+indx < -lbuf[idir])
	            coords[idir] = VL[idir] + hdir*OFFSET;
	        else if (is+indx >= gmax[idir]+ubuf[idir])
	            coords[idir] = VU[idir] - hdir*OFFSET;
	        sten->st[indx] = sten->worksp[indx];
                if (!nearest_crossing_state_with_comp(icoords[indx],
                        Coords(sten->p[indx]),new_comp,
                        wave_tri_soln(sten->newwave)->tri_grid,
                        &sten->st[indx]))
                {
	            nearest_intfc_state_and_pt(coords,new_comp,
	                                   sten->fr,sten->newfr,
					   sten->st[indx],
					   Coords(sten->p[indx]),
	                                   &sten->hs[indx]);
                }
	        copy_states(i,side,endpt,new_comp,sten,crx);
	        return;
	    }
	}
}		/*end find_outer_states*/


/*
*			set_neumann_bdry_sten_states():
*
*	Sets stencil states using reflection through a Neumann boundary.
*/

LOCAL	void	set_neumann_bdry_sten_states(
	int	  i,
	int	  is,
	int	  idir,
	COMPONENT new_comp,
	CRXING	  *nbdry,
	Stencil	  *sten,
	CRX_SIDE  side)
{
    	HYPER_SURF *hs = nbdry->hs;
	Wave	   *wave = sten->wave;
	Front	   *fr = sten->fr;
	CRXING	   *rcrx;
	float 	   coords[MAXD], coordsref[MAXD], n[MAXD];
	int	   j, indx, ri;
	size_t	   sizest = fr->sizest;

	for (; i <= endpt; ++i)
	{
	    indx = i * side;
	    if ((*fr->neumann_bdry_state)(Coords(sten->p[indx]),new_comp,
					  nbdry->pt,hs,fr,
					  (POINTER)wave,sten->worksp[indx]))
	    {
	        sten->st[indx] = sten->worksp[indx];
	        for (j = 0; j < dim; ++j)
	    	    Coords(sten->p[indx])[j] = Coords(nbdry->pt)[j];
	        assign(left_state(sten->p[indx]),sten->st[indx],sizest);
	        assign(right_state(sten->p[indx]),sten->st[indx],sizest);
	    }
	    else 
	    {
	        if (sten->crx[indx][0] != nbdry)
	    	    copy_states(i,side,endpt,new_comp,sten,NULL);
	        else
	    	    copy_states(i,side,endpt,new_comp,sten,nbdry);
	        return;
	    }

	    if (is_bdry_hs(hs))
	    {
	        /* TODO:
		 * Currently this section of set_neumann_bdry_sten_states() is
	         * only implemented for rectangular Neumann boundaries
	         * as indicated by the above test.  Interior
	         * or oblique boundaries do not use this boundary
	         * state function. */

	        /* Have set the first state, now need to check for another
	         * cross on the other side of the Neumann boundary. Some
	         * technical issues arise due to the way neumann_bdry_state()
	         * reflects coords.
	         */

	        if (i == endpt)
		    break;
	        ri = (side == PREV) ? -indx - 2*is - 1 :
			              -indx + 2*(gmax[idir] - is) - 1;
	        if (side == PREV)
	        {
	    	    rcrx = (ri >= 0) ? sten->crx[ri+1][0] :
				       sten->crx[ri][sten->nc[ri]-1];
	        }
	        else if (side == NEXT)
	        {
	    	    rcrx = (ri <= 0) ? sten->crx[ri-1][0] :
	    			       sten->crx[ri][sten->nc[ri]-1];
	        }
	        if (rcrx != NULL) 
	        {
	    	    indx = (++i) * side;
	    	    for (j = 0; j < dim; ++j) 
	    	        coords[j] = Coords(rcrx->pt)[j];
	    	    if ((reflect_pt_about_Nbdry(coords,coordsref,n,new_comp,
					        nbdry->hs,fr))
		        && 
		        (*fr->neumann_bdry_state)(coordsref,new_comp,nbdry->pt,
					          hs,fr,(POINTER)wave,
					          sten->worksp[indx]))
		    {
		        sten->st[indx] = sten->worksp[indx];
		        for (j = 0; j < dim; ++j)
		    	    Coords(sten->p[indx])[j] = coordsref[i];
		        assign(left_state(sten->p[indx]),sten->st[indx],sizest);
		        assign(right_state(sten->p[indx]),sten->st[indx],sizest);
		    }
		    copy_states(i,side,endpt,new_comp,sten,NULL);
		    return;
	        }
	    }
	}
}		/*end set_neumann_bdry_sten_states*/

/*
*			set_dirichlet_bdry_sten_states():
*
*	Sets stencil states through a Dirichlet boundary.
*/

LOCAL	void	set_dirichlet_bdry_sten_states(
	int	   i,
	HYPER_SURF *hs,
	Stencil	   *sten,
	CRX_SIDE   side)
{
	Wave		*wave = sten->wave;
	Front		*fr = sten->fr;
	int		indx;
	size_t		sizest = fr->sizest;

	for (; i <= endpt; ++i)
	{
	    indx = i * side;
	    sten->st[indx] = sten->worksp[indx];
	    evaluate_dirichlet_boundary_state(Coords(sten->p[indx]),hs,fr,wave,
						     sten->st[indx]);
	    assign(left_state(sten->p[indx]),sten->st[indx],sizest);
	    assign(right_state(sten->p[indx]),sten->st[indx],sizest);
	}
}		/*end set_dirichlet_bdry_sten_states*/

/*
*			copy_states():
*
*	We have crossed a non-reflecting curve and are ready to simply copy an
*       appropriate state out to the end of the stencil.  If the cross is not
*	NULL, then it is assumed that the start state has not been set, and
*	an appropriate state is loaded and then copied.
*/

LOCAL	void copy_states(
	int	 start,
	CRX_SIDE side,
	int	 endpt,
	int	 new_comp,
	Stencil	 *sten,
	CRXING	 *cross)
{
	int		i, j, indx, in_one;	
	size_t		sizest = sten->fr->sizest;

	if (cross != NULL)
	{
	    indx = start*side;
	    sten->st[indx] = state_with_comp(cross->pt,cross->hs,new_comp);
	    if (sten->st[indx] != NULL)
	    {
	        for (j = 0; j < dim; ++j)
	    	    Coords(sten->p[indx])[j] = Coords(cross->pt)[j];
	        assign(left_state(sten->p[indx]),sten->st[indx],sizest);
	        assign(right_state(sten->p[indx]),sten->st[indx],sizest);
	    }
	    else
	    {
	        sten->st[indx] = sten->worksp[indx];
	        nearest_intfc_state_and_pt(Coords(sten->p[indx]),new_comp,
					   sten->fr,sten->newfr,sten->st[indx],
					   Coords(sten->p[indx]),
					   &sten->hs[indx]);
	    }
	}

	for (i = start+1; i <= endpt; ++i)
	{
	    indx = i*side;
	    in_one = indx - side;
	    sten->st[indx] = sten->st[in_one];
	    for (j = 0; j < dim; ++j)
	    	Coords(sten->p[indx])[j] = Coords(sten->p[in_one])[j];
	    assign(left_state(sten->p[indx]),sten->st[indx],sizest);
	    assign(right_state(sten->p[indx]),sten->st[indx],sizest);
	}
	return;
}		/*end copy_states*/



#if defined(TWOD)

/*                  h_symmetry_states_printout
*
*       This function print out the front states of two symmetric
*       points w.r.t. the center line on the interface. It's for
*       the symmetric testing of the single mode interface.
*       First we find the two bonds which cross these two symmetric
*       vertical lines, then locate the coordinates of the two points
*       of the intersections.  The left and right states are printed
*       on these two points, in order to check the symmetry on the front.
*/

LOCAL   int	 h_symmetry_states_printout(
        Front           *front,
        Wave		*wave)
{
        int             i, k, dim = wave->rect_grid->dim;
        int             wave_type;
        int             imin[MAXD], imax[MAXD];
        HYPER_SURF      *hs;
        CURVE           *c;
        BOND            *b;
        float           lp[2], rp[2], s0, s1, s2, s3;
        float           dh[MAXD], sgn_x_l, sgn_x_r;
        Locstate        sl_l, sl_r, sr_l, sr_r;
        INTERFACE       *interf = front->interf;
        COMPONENT       pcomp = positive_component(hs);
        COMPONENT       ncomp = negative_component(hs);

        alloc_state(intfc,&sl_l,front->sizest);
        alloc_state(intfc,&sl_r,front->sizest);
        alloc_state(intfc,&sr_l,front->sizest);
        alloc_state(intfc,&sr_r,front->sizest);

        for (i = 0; i < dim; i++)
        {
            dh[i] = wave->rect_grid->h[i];
            imin[i] = 0;        imax[i] = wave->rect_grid->gmax[i];
            if (wave->rect_grid->lbuf[i] == 0) imin[i] += 1;
            if (wave->rect_grid->ubuf[i] == 0) imax[i] -= 1;
        }

        printf("dh[0] = %lf, dh[1] = %lf\n",dh[0],dh[1]);

        if (make_bond_comp_lists(interf) == FUNCTION_FAILED)
        {
                (void) printf("WARNING in make_bond_lists(), ");
                (void) printf("make_bond_listsd() failed\n");
                return FUNCTION_FAILED;
        }

        k = 0;      

        lp[0] = ((imin[0] + imax[0]) / 4) * (dh[0]);

        rp[0] = (((imin[0] + imax[0]) * 3 / 4) - 1) * (dh[0]);


        (void) next_bond(interf,NULL,NULL);

        while (next_bond(interf,&b,&c))
        {
                if(wave_type(c) < FIRST_SCALAR_PHYSICS_WAVE_TYPE) continue;

	        hs = Hyper_surf(c);
	       	pcomp = positive_component(hs);
         	ncomp = negative_component(hs);

                s0 = (Coords(b->end)[0] - lp[0]);
                s1 = (Coords(b->start)[0] - lp[0]);
                s2 = (Coords(b->end)[0] - rp[0]);
                s3 = (Coords(b->start)[0] - rp[0]);

                sgn_x_l = s0 * s1;
                sgn_x_r = s2 * s3;

                if(sgn_x_l > 0.0 && sgn_x_r > 0.0) continue;

                if(s0 == 0.0)
                {
                  lp[1] = Coords(b->end)[1];

                  k = k + 1;
                }

                if(sgn_x_l < 0.0)
                {
                  lp[1] = ((Coords(b->end)[1] - Coords(b->start)[1]) *
                  (Coords(b->start)[0] - lp[0]) /
                  (Coords(b->end)[0] - Coords(b->start)[0])) +
                   Coords(b->start)[1];

                  k = k + 1;
                }

                if(s2 == 0.0)
                {
                  rp[1] = Coords(b->end)[1];

                  k = k + 1;
                }

                if(sgn_x_r < 0.0)
                {
                  rp[1] = ((Coords(b->end)[1] - Coords(b->start)[1]) *
                  (Coords(b->start)[0] - rp[0]) /
                  (Coords(b->end)[0] - Coords(b->start)[0])) +
                   Coords(b->start)[1];

                  k = k + 1;
                }

        }

        printf("left point = (%lf,%lf)\n",lp[0],lp[1]);

        printf("right point = (%lf,%lf)\n",rp[0],rp[1]);

        printf("k = %d\n",k);

        wave->hyp_soln(lp,ncomp,hs,NEGATIVE_SIDE,front,wave,sl_l,NULL);

        wave->hyp_soln(lp,pcomp,hs,POSITIVE_SIDE,front,wave,sl_r,NULL);

        wave->hyp_soln(rp,ncomp,hs,NEGATIVE_SIDE,front,wave,sr_l,NULL);

        wave->hyp_soln(rp,pcomp,hs,POSITIVE_SIDE,front,wave,sr_r,NULL);

        (*front->print_state)(sl_l);
        (*front->print_state)(sl_r);
        (*front->print_state)(sr_l);
        (*front->print_state)(sr_r);

}
#endif /* defined(TWOD) */

#if defined(CONSERVATIVE_ALG) && defined(TWOD)

LIB_LOCAL void hyp_unsplit_npt(
        int             swp_num,
        int             *iperm,     /* sweep based permutation of coord dir */
        float           *dh,
        float           dt,
        Wave            *wave,
        Wave            *newwave,
        Wave            *swp_wv, 
        Front           *fr,
        Front           *newfr,     /* newfr needed if hlw is to support */
                                    /*  changing topologies          */
        COMPONENT       max_comp)
{
        POINT           Basep;
        int             i, idirs[MAXD];
        int             imin[3], imax[3];
        static  Stencil *sten = NULL;
        static  int     num_pts;
        static  int     **icoords = NULL;
        RECT_GRID       *gr = wave->rect_grid;

        /* SET LOCAL GLOBAL */
        dim = gr->dim;
        intfc = fr->interf;
        ext_comp = exterior_component(intfc);

        for (i = 0; i < dim; ++i)
            idirs[i] = iperm[(i+swp_num)%dim];

        /* SET LOCAL GLOBAL */
        hdir = dh[idirs[0]];     /* grid spacing in sweep direction */
        hndir = dh[idirs[1]];    /* in the other direction */
        if (sten == NULL)
        {
            /* This assumes that the stencil size is constant. */
            num_pts = wave->npts_sten;
            endpt = stencil_radius(wave); // = 2

            sten = alloc_stencil(num_pts,fr);
#if defined(TWOD) || defined(THREED)
            icoords = sten->icoords;
#endif /* defined(TWOD) || defined(THREED) */
        }

        switch (idirs[0])
        {
        case 0: /*X SWEEP */
            prev_side = WEST;
            next_side = EAST;
            if (debugging("hyp_npt"))
                (void) printf("prev_side = WEST, next_side = EAST\n");
            break;
        case 1: /*Y SWEEP */
            prev_side = SOUTH;
            next_side = NORTH;
            if (debugging("hyp_npt"))
                (void) printf("prev_side = SOUTH, next_side = NORTH\n");
            break;
        case 2: /* Z SWEEP */
            prev_side = LOWER;
            next_side = UPPER;
            if (debugging("hyp_npt"))
                (void) printf("prev_side = LOWER, next_side = UPPER\n");
            break;
        }

        sten->fr   = fr;        sten->newfr   = newfr;
        sten->wave = wave;      sten->newwave = newwave;
        sten->swap_wv = swp_wv; 

        // if not enough, call set_unsplit_sweep_limits()
        set_sweep_limits(wave,swp_num,idirs,imin,imax);
        if (gr->lbuf[idirs[1]] > 0)
            imin[1] += 1;
        if (gr->ubuf[idirs[1]] > 0)
            imax[1] -= 1;
        if(swp_num == 1)
        {
            if (gr->lbuf[idirs[1]] > 0)
                imin[1] += 2; // if set_unsplit_sweep_limits(), should be 1 here.
            if (gr->ubuf[idirs[1]] > 0)
                imax[1] -= 2; // if set_unsplit_sweep_limits(), should be 1 here.
        }
#if defined(DEBUG)
        /*
        {
            static const char *var[] = {"ix","iy","iz"};
            int kk;

            (void) printf("\nUnsplit_npt Sweep limits\n");
            (void) printf("\tdir = %d, ",idirs[0]);
            print_int_vector("idirs = ",idirs,dim," ");
            (void) printf(", swp_num = %d\n",swp_num);
            for (kk = 0; kk < dim; ++kk)
                (void) printf("\t\t%2d <= %s < %d\n",
                        imin[kk],var[idirs[kk]],imax[kk]);
            (void) printf("\n");
        }
        */
#endif /* if defined(DEBUG) */

        switch (dim)
        {
#if defined(TWOD)
        case 2:
        {
            int i0, i1;
            int i0max, i1max;
            int i0min, i1min;

            i0min = imin[0];    i0max = imax[0];
            i1min = imin[1];    i1max = imax[1];
            for (i1 = i1min; i1 < i1max; ++i1)
            {
                set_static_coords(icoords,&Basep,idirs[1],i1,gr);
                sten->reg_stencil = NO;
                for (i0 = i0min; i0 < i0max; ++i0)
                {
                    unsplit_update_reg_grid_state(dh,dt,swp_num,iperm,
                                          i0,sten,&Basep,max_comp);
                }
            }
            break;
        }
#endif /* defined(TWOD) */
        }

}

LOCAL        int        deb_hyp_npt = NO;
LOCAL   void unsplit_update_reg_grid_state(
        float           *dh,
        float           dt,
        int             swp_num,
        int             *iperm,
        int             is,
        Stencil         *sten,
        POINT           *basep,
        COMPONENT       max_comp)
{
        Wave            *wave = sten->wave;
        Wave            *newwave = sten->newwave;
        Front           *fr = sten->fr;
        Front           *newfr = sten->newfr;
        COMPONENT       *comp;
        COMPONENT       new_comp;     /* mid comp wrt newfr */
        HYPER_SURF      *hs;
        int             i, j, index, idir = iperm[swp_num];
        int             **icoords = sten->icoords;

#if defined(DEBUG_HYP_NPT)
        // int             deb_hyp_npt = NO;

static  char fname[3][11] = {"xhyp_npt()","yhyp_npt()","zhyp_npt()"};
        if (debugging("hyp_npt"))                       deb_hyp_npt = YES;
        else if (debugging("xhyp_npt") && idir == 0)    deb_hyp_npt = YES;
        else if (debugging("yhyp_npt") && idir == 1)    deb_hyp_npt = YES;
        else if (debugging("zhyp_npt") && idir == 2)    deb_hyp_npt = YES;
        else                                            deb_hyp_npt = NO;
#endif /* defined(DEBUG_HYP_NPT) */

        for (i = -endpt; i <= endpt; ++i)
            icoords[i][idir] = is + i;
        sten->prev_reg_stencil = sten->reg_stencil;

                /* Find components and crosses*/

        new_comp = set_unsplit_stencil_comps_and_crosses(idir,swp_num,iperm,sten);
        comp = sten->comp;

#if defined(DEBUG_HYP_NPT)
        if(debugging("check_mass_conservation"))
        {
            if(icoords[0][1] == 0) deb_hyp_npt = YES;  
            if(icoords[0][0] == 0) deb_hyp_npt = YES;  
            if(icoords[0][0] == 39) deb_hyp_npt = YES;  
        }
        // if(icoords[0][0] == 35 && icoords[0][1] == 3) deb_hyp_npt = YES;  
        if (deb_hyp_npt && icoords[0][0] == 35 && icoords[0][1] == 3)
        {
            int side; 
            (void) printf("\n%s - ",fname[idir]);
            print_int_vector("icoords[0] = ",icoords[0],dim,"\n");
            (void) printf("comps: ");
            for (i = -endpt; i <= endpt; ++i)
                (void) printf("comp[%d] %d ",i,comp[i]);
            (void) printf("new_comp %d\n",new_comp);
            for(side = 0; side <= 1; side++) 
            {
                for (i = -1; i <= 1; ++i)
                    printf("Tcomp[%d][%d] %d ",side, i,sten->Tcomp[side][i]);
                printf("\n"); 
            }
        }
#endif /* defined(DEBUG_HYP_NPT) */

#if defined(DEBUG_HYP_NPT)
        if(deb_hyp_npt == YES)
            sten->prev_reg_stencil = NO;
        else
#endif /* defined(DEBUG_HYP_NPT) */ 
        if (is_vec_method)
        {
            if (unsplit_interior_sten_reg(is,idir,swp_num,iperm,sten))
                return;  /*Don't overwrite vector solution*/
            else
                sten->prev_reg_stencil = NO;
        }

        for (j = 0; j < dim; ++j)
        {
            if (j == idir)
            {
                Coords(sten->p[0])[idir] = cell_center(is,idir,wave->rect_grid);
                for (i = 1; i <= endpt; ++i)
                {
                    Coords(sten->p[-i])[idir] = Coords(sten->p[0])[idir]-i*hdir;
                    Coords(sten->p[i])[idir]  = Coords(sten->p[0])[idir]+i*hdir;
                }
            }
            else
            {
                for (i = -endpt; i <= endpt; ++i)
                    Coords(sten->p[i])[j] = Coords(basep)[j];
            }
        }
        sten->reg_stencil = YES;
        if (RegionIsFlowSpecified(Rect_state(icoords[0],newwave),
                                  Rect_state(icoords[0],wave),
                                  Coords(sten->p[0]),new_comp,comp[0],fr))
        {
            sten->reg_stencil = NO;
            return;
        }

        if ((new_comp > max_comp) && (swp_num > 0))
        {
            sten->reg_stencil = NO;
            assign(Rect_state(icoords[0],newwave),
                   Rect_state(icoords[0],wave),fr->sizest);
            return;
        }

        if (new_comp > max_comp)
        {
            sten->reg_stencil = NO;
            nearest_intfc_state_and_pt(Coords(sten->p[0]),new_comp,newfr,fr,
                                       sten->worksp[0],Coords(sten->p[0]),&hs);
            assign(Rect_state(icoords[0],newwave),sten->worksp[0],fr->sizest);
            return;
        }

                /* Find mid state */
        if (equivalent_comps(comp[0],new_comp,newfr->interf) == YES)
            sten->st[0] = Rect_state(icoords[0],wave);
        else
        {
            nearest_intfc_state_and_pt(Coords(sten->p[0]),new_comp,fr,
                                       newfr,sten->worksp[0],
                                       Coords(sten->p[0]),&hs);
            sten->st[0] = sten->worksp[0];
            sten->reg_stencil = NO;
        }
        assign(left_state(sten->p[0]),sten->st[0],fr->sizest);
        assign(right_state(sten->p[0]),sten->st[0],fr->sizest);

        find_outer_states(is,idir,sten,PREV);
        find_outer_states(is,idir,sten,NEXT);

        find_out_states_in_other_dir(is,idir,swp_num,iperm,sten); 

#if defined(DEBUG_HYP_NPT)
        if (deb_hyp_npt && icoords[0][0] == 35 && icoords[0][1] == 3)
        {
            int side; 
            (void) printf("STENCIL: "); print_Stencil(sten);
            for (i = -endpt; i <= endpt; ++i)
            {
                (void) printf("state[%d]:       ",i);
                (*fr->print_state)(sten->st[i]);
            }
            /* 
            (void) printf("The state in the other dir\n");
            for(side = 1; side >= 0; side--)
            {
                for(i = -1; i <= 1; i++)
                {
                    (void) printf("Tstate[%d][%d]:   ",side, i);
                    (*fr->print_state)(sten->Tst[side][i]);
                }
            }
            */
        }
#endif /* defined(DEBUG_HYP_NPT) */
                /* Check for point source/sink in icoords */

        index = (wave->num_point_sources) ?
            is_source_block(wave,newfr->interf,new_comp,icoords[0]) : -1;

                /* Update state */

        // unsplit_point_FD
        unsplit_npt_solver(dh,dt,Rect_state(icoords[0],newwave),dir,
              swp_num,iperm,&index,sten,wave);

#if defined(DEBUG_HYP_NPT)
        if (deb_hyp_npt && icoords[0][0] == 35 && icoords[0][1] == 3)
        {
            (void) printf("\n%s - ",fname[idir]);
            print_int_vector("icoords[0] = ",icoords[0],dim,"\n");
            (void) printf("ANSWER: ");
            (*fr->print_state)(Rect_state(icoords[0],newwave));
        }
#endif /* defined(DEBUG_HYP_NPT) */
}


LOCAL void find_out_states_in_other_dir(
        int             is,
        int             idir,
        int             swp_num,
        int             *iperm,
        Stencil         *sten)
{
        Wave       *wave = sten->wave;
        Front      *fr = sten->fr;
        Front      *newfr = sten->newfr;
        COMPONENT  **tcomp = sten->Tcomp, new_comp = sten->newcomp;
        int        **icoords = sten->icoords, ic[MAXD], ictmp[MAXD];
        float      crds[MAXD];  
        int        i, j, indx, indir;
        int        in_domain, face; 
        CRXING     *crx;
        HYPER_SURF *hs, *Ths;
        CRXING     *new_crx[MAX_NUM_CRX];
        CRXING     *old_crx[MAX_NUM_CRX];
        int        side, sign; 
        GRID_DIRECTION  *npgdir, *pgdir, gdir[2][2] = {SOUTH, NORTH,
                                      WEST, EAST};
        int        error = NO;

        indir = iperm[(swp_num+1)%dim];  
        if(iperm[swp_num%dim] == 0)
            npgdir = gdir[1];
        else
            npgdir = gdir[0];
 
        if(indir == 0)
            pgdir = gdir[1];
        else
            pgdir = gdir[0];

        crx = NULL;

        /* For the purpose when this orthogonal. dir. 
         * passes the passive boundary, see copy_Tstates() 
         */
        for(side = 0; side <=1; side++)
        {
            for(j = -1; j <= 1; j++)
            {
                assign(sten->Tworksp[side][j], sten->st[j], fr->sizest);  
                sten->Tset[side][j] = NO; 
            }
        }

        /* First check in the orthogonal direction. */
        for(side = 0; side <= 1; side++)
        {
            if(side == 0) sign = -1;
            else sign = 1; 
            for(j = -1; j <= 1; j++)
            {
                for (i = 0; i < dim; ++i)
                    ic[i] = icoords[0][i];
                ic[idir] += j; ic[indir] += sign*1;

                in_domain = crds_of_ic(crds,sten->wave,ic); 
                if (equivalent_comps(tcomp[side][j],new_comp,newfr->interf) == YES)
                {
                    if(in_domain == NO)
                    {
                        printf("ERROR find_out_states_in_other_dir\n");
                        printf("NOT in domain ic:[");
                        for(i = 0; i < dim; i++)
                            printf("%3d ", ic[i]); 
                        printf("] Normal dir[%d]\n", iperm[swp_num%dim]); 
                        printf("Compute ic:[");
                        for(i = 0; i < dim; i++)
                            printf("%3d ", icoords[0][i]); 
                        clean_up(ERROR); 
                    }
                    sten->Tst[side][j] = Rect_state(ic,wave);
                    sten->Tset[side][j] = YES;
                    continue;
                }
                sten->reg_stencil = NO;

                // Is it better to use the closest one? 
                // old_crx[0] = __Rect_crossing(icoords[j],pgdir[side],
                //                wave_tri_soln(sten->wave)->tri_grid);
                old_crx[0] = NULL;  
                crossings_in_direction(old_crx, icoords[j],pgdir[side],
                     wave_tri_soln(sten->wave)->tri_grid);
                
                if(old_crx[0] != NULL)
                {
                    /* There is a cross berween icoords[j] and its upper line one */
                    crossings_in_direction(new_crx, icoords[j],pgdir[side],
                         wave_tri_soln(sten->newwave)->tri_grid);

                    hs = old_crx[0]->hs;
                    if ((new_crx[0] == NULL) ||
                        (wave_type(hs) != wave_type(new_crx[0]->hs)))
                    {
                       /* crx has moved out of stencil OR
                        * another curve has moved between indx and the
                        * old crx over the course of the time step. */

                        // copy_Tstates(j,crds,side,new_comp,sten,crx);
                        copy_Tstates(j,crds,side,new_comp,sten,old_crx[0]);
                    }
                    if ((wave_type(hs) == NEUMANN_BOUNDARY) &&
                        (fr->neumann_bdry_state))
                    {
                        set_neumann_bdry_sten_Tstates(j,side,crds,idir,new_comp,
                                                 old_crx[0],sten);
                    }                
                    else if(wave_type(hs) == DIRICHLET_BOUNDARY) 
                    {
                        // set_dirichlet_bdry_sten_states(i,hs,sten,side);
                        sten->Tst[side][j] = sten->Tworksp[side][j];
                        sten->Tset[side][j] = YES; 
                        evaluate_dirichlet_boundary_state(crds,hs,sten->fr,
                              sten->wave,sten->Tst[side][j]);
                    }
                    else
                    {
                        /* crx is interior */ 
                        copy_Tstates(j,crds,side,new_comp,sten,old_crx[0]);
                    }
                }
                else
                {
                    if(in_domain == YES)
                    {
                        copy_Tstates(j,crds,side,new_comp,sten,NULL);
                        /*
                        sten->Tst[side][j] = sten->Tworksp[side][j];
                        nearest_intfc_state_and_pt(crds,new_comp,sten->fr,
                               sten->newfr,sten->Tst[side][j],crds,&Ths);
                        sten->Tset[side][j] = YES; 
                        */
                    }
                    // copy_Tstates(j,crds,side,new_comp,sten,NULL);
                }
            }
        }

        face = 0; 
        /* Check in the sweep direction */
        for(j = -1; j <= 1; j+=2)
        {
            for(side = 0; side <= 1; side++)
            {
                if(side == 0) sign = -1;
                else sign = 1;
                for (i = 0; i < dim; ++i)
                    ictmp[i] = ic[i] = icoords[0][i];
                ic[indir] += sign*1;
                ictmp[indir] += sign*1; ictmp[idir] += j;
       
                if(sten->Tset[side][j] == YES)
                    continue; 

                in_domain = crds_of_ic(crds,sten->wave,ictmp);
                if (equivalent_comps(tcomp[side][j],new_comp,newfr->interf) == YES)
                {
                    if(in_domain == NO)
                    {
                        printf("ERROR find_out_states_in_other_dir, sweep\n");
                        printf("NOT in domain ic:[");
                        for(i = 0; i < dim; i++)
                            printf("%3d ", ic[i]);
                        printf("] Normal dir[%d]\n", iperm[swp_num%dim]);
                        printf("Compute ic:[");
                        for(i = 0; i < dim; i++)
                            printf("%3d ", icoords[0][i]);
                        clean_up(ERROR);
                    }
                    sten->Tst[side][j] = Rect_state(ictmp,wave);
                    sten->Tset[side][j] = YES;

                    continue;
                }

                old_crx[0] = NULL;  
                crossings_in_direction(old_crx, ic,npgdir[face],
                     wave_tri_soln(sten->wave)->tri_grid);

                if(old_crx[0] != NULL)
                {
                    /* There is a cross berween icoords[j] and its upper line one */

                    hs = old_crx[0]->hs;
                    crossings_in_direction(new_crx, ic,npgdir[face],
                         wave_tri_soln(sten->newwave)->tri_grid);
                    if ((new_crx[0] == NULL) ||
                        (wave_type(hs) != wave_type(new_crx[0]->hs)))
                    {
                       /* crx has moved out of stencil OR
                        * another curve has moved between indx and the
                        * old crx over the course of the time step. */

                        copy_Tstates(j,crds,side,new_comp,sten,old_crx[0]);
                    }
                    if ((wave_type(hs) == NEUMANN_BOUNDARY) &&
                        (fr->neumann_bdry_state))
                    {
                        set_neumann_bdry_sten_Tstates(j,side,crds,idir,new_comp,
                                                 old_crx[0],sten);
                        if(deb_hyp_npt == YES)
                        {
                            printf("Tst[%d][%d] set by neumann[%d,%d] in %s\n",
                                  side,j,ic[0],ic[1], grid_dir(npgdir[face])); 
                        }
                    }
                    else if(wave_type(hs) == DIRICHLET_BOUNDARY)
                    {
                        sten->Tst[side][j] = sten->Tworksp[side][j];
                        sten->Tset[side][j] = YES;
                        evaluate_dirichlet_boundary_state(crds,hs,sten->fr,
                              sten->wave,sten->Tst[side][j]);
                    }
                    else
                    {
                        /* crx is interior */
                        copy_Tstates(j,crds,side,new_comp,sten,old_crx[0]);
                    }
                }
            }
            face++; 
        }

        /* Last way to set the unset point state */
        for(side = 0; side <= 1; side++)
        {
            if(side == 0) sign = -1;
            else sign = 1;
            for(j = -1; j <= 1; j++)
            {
                if(sten->Tset[side][j] == YES)
                    continue; 
                for (i = 0; i < dim; ++i)
                    ic[i] = icoords[0][i];
                ic[idir] += j; ic[indir] += sign*1;
                in_domain = crds_of_ic(crds,sten->wave,ic);        
                copy_Tstates(j,crds,side,new_comp,sten,NULL);
            }
        }


        for(side = 0; side <=1; side++)
        {
            for(j = -1; j <= 1; j++)
            {
                if(sten->Tset[side][j] == NO)
                {
                    error = YES; 
                    break; 
                }
            }
        }
 
        if(error == YES)
        {
            for(side = 0; side <=1; side++)
            {
                for(j = -1; j <= 1; j++)
                {
                    if(sten->Tset[side][j] == NO)
                    {
                        printf("ERROR find_out_states_in_other_dir, Tst[%d][%d] not set\n",
                            side, j);
                    }
                }
            }
            printf("Normal dir[%d]\n", iperm[swp_num%dim]); 
            print_int_vector("icoords[0] = ",icoords[0],dim,"\n");
            (void) printf("\n comps: ");
            for (i = -endpt; i <= endpt; ++i)
                (void) printf("comp[%d] %d ",i,sten->comp[i]);
            (void) printf("new_comp %d\n",new_comp);
            for(side = 0; side <= 1; side++)
            {
                for (i = -1; i <= 1; ++i)
                    printf("Tcomp[%d][%d] %d ",side, i,sten->Tcomp[side][i]);
                printf("\n");
            }
            (void) printf("STENCIL: "); print_Stencil(sten);
            clean_up(ERROR); 
        }
}

LOCAL int crds_of_ic(
        float       *crds,
        Wave        *wave, 
        int         *ic)  
{
        int         i, in_domain = YES; 
        RECT_GRID   *rgr = wave->rect_grid;  

        for(i = 0; i < dim; i++)
        {
            if((ic[i] < -lbuf[i]) || (ic[i] >= (gmax[i] + ubuf[i])))
            in_domain = NO; 
        }
        if(in_domain == YES)
        {
            for (i = 0; i < dim; ++i)
                crds[i] = Rect_coords(ic,wave)[i];
            return in_domain; 
        }

        for (i = 0; i < dim; ++i)
           crds[i] = rgr->L[i] + 
                (ic[i]+0.5)*rgr->h[i];
        return in_domain; 
}

/*
*                       set_neumann_bdry_sten_Tstates():
*
*       Sets stencil states using reflection through a Neumann boundary.
*       Is it complete? Compare with set_neumann_bdry_sten_states(). 
*/

LOCAL   void    set_neumann_bdry_sten_Tstates(
        int       i,
        int       side,
        float     *crds,
        int       idir,
        COMPONENT new_comp,
        CRXING    *nbdry,
        Stencil   *sten)
{
        HYPER_SURF *hs = nbdry->hs;
        Wave       *wave = sten->wave;
        Front      *fr = sten->fr;
        CRXING     *rcrx;
        float      coords[MAXD], coordsref[MAXD], n[MAXD];
        int        j, indx, ri;
        size_t     sizest = fr->sizest;

        if ((*fr->neumann_bdry_state)(crds,new_comp,nbdry->pt,hs,fr,
                                (POINTER)wave,sten->Tworksp[side][i]))
        {
            sten->Tst[side][i] = sten->Tworksp[side][i];
            sten->Tset[side][i] = YES;
        }
        else
        {
            printf("ERROR set_neumann_bdry_sten_Tstates, the state is not set\n");  
            clean_up(ERROR);  
        }
}


/*      copy_Tstates() 
 *  Copy single state for the other direction.
 */
LOCAL   void copy_Tstates(
        int        start,
        float      *coords,
        int        side,
        COMPONENT  new_comp,
        Stencil    *sten,
        CRXING     *cross)
{
        int             i, j, in_one;
        size_t          sizest = sten->fr->sizest;
        HYPER_SURF      *Ths; 
        if(cross != NULL)
        {
            sten->Tst[side][start] = state_with_comp(cross->pt,cross->hs,new_comp);

            // Have not allocated point for it yet.
            if(sten->Tst[side][start] == NULL)
            {
                if(is_passive_boundary(cross->hs))
                    sten->Tst[side][start] = sten->Tworksp[side][start]; 
                else
                {
                    printf("ERROR: copy_Tstates, Tst[%d][%d] NULL\n", side, start);
                    printf("center icoords[%d, %d]\n", sten->icoords[0][0], sten->icoords[0][1]);
                    print_vector("Coords", 2, coords, "%g "); 
                    print_hypersurface(cross->hs);  
                    clean_up(ERROR); 
                }
            }
            // Have not allocated point for it yet.
            sten->Tset[side][start] = YES; 
        }
        else
        {
            sten->Tst[side][start] = sten->Tworksp[side][start]; 
            nearest_intfc_state_and_pt(coords,new_comp,sten->fr,
                    sten->newfr,sten->Tst[side][start],coords,&Ths);
            if((wave_type(Ths) == NEUMANN_BOUNDARY) &&
                        (sten->fr->neumann_bdry_state))
            {
                if ((*sten->fr->neumann_bdry_state)(coords,new_comp,NULL,Ths,sten->fr,
                                (POINTER)(sten->wave),sten->Tworksp[side][start]))
                {
                    sten->Tst[side][start] = sten->Tworksp[side][start];
                }
                else
                {
                    printf("ERROR in copy_Tstates\n"); 
                    printf("ERROR set_neumann_bdry_sten_Tstates, the state is not set\n");
                    clean_up(ERROR);
                }
            }
            else if(wave_type(Ths) == DIRICHLET_BOUNDARY)
            {
                sten->Tst[side][start] = sten->Tworksp[side][start];
                evaluate_dirichlet_boundary_state(coords,Ths,sten->fr,
                      sten->wave,sten->Tst[side][start]);
            }
            sten->Tset[side][start] = YES;
        }
}

/*
*                       unsplit_interior_sten_reg():
*
*       If called after a vector solver, decides whether the stencil at a given
*       point was regular for the vector solver.  In other words, decides
*       whether solution needs to be recomputed at a given point.
*       For the unsplit muscl, for updating the states in one direction, the stencil is like:
*
*       x      X      X      X      x
*       .      |      |      |      . 
*       .      |      |      |      .
*       X------X------X------X------X
*       -2     -1     0      1      2
*       .      |      |      |      .
*       .      |      |      |      .
*       x      X      X      X      x
*
*      Capital Xs are the points actually used to update point[0]. Small xs are not
*      used, but they are checked. Because of the way sweep_3l_comp_segments()
*      checks the component. 
*/

LOCAL   bool unsplit_interior_sten_reg(
        int             is,
        int             idir,
        int             swp_num,
        int             *iperm,
        Stencil         *sten)
{
        Wave            *wave = sten->wave;
        Front           *newfr = sten->newfr;
        int             sign, i, j, indir, side;
        int             **icoords = sten->icoords, icrds[MAXD], ictmp[MAXD];
        COMPONENT       cmp;
        COMPONENT       new_comp = sten->newcomp, *comp = sten->comp;
        CRXING          *cross;
        GRID_DIRECTION  *npgdir, *pgdir, gdir[2][2] = {NORTH, SOUTH,
                                      EAST, WEST};

        if (ComponentIsFlowSpecified(new_comp,newfr))
            return NO;

        for (i = -endpt; i <= endpt; ++i)
        {
            if ((equivalent_comps(comp[i],new_comp,newfr->interf) == NO) ||
                (sten->nc[i] != 0))
            {
                return NO;
            }
        }

        if(iperm[swp_num%dim] == 0)
            npgdir = gdir[1];
        else
            npgdir = gdir[0];

        indir = iperm[(swp_num+1)%dim];
        if(indir == 0)
            pgdir = gdir[1];
        else
            pgdir = gdir[0];

        for (i = 0; i < dim; ++i)
            ictmp[i] = icrds[i] = icoords[0][i];

        /* Check the components of the upper and lower rows */  
        for(side = 0; side <=1; side++)
        {
            if(side == 0)
                ictmp[indir] = icrds[indir] - 1; 
            else
                ictmp[indir] = icrds[indir] + 1; 
            for (i = -endpt; i <= endpt; ++i)
            {
                ictmp[idir] = is - i; 
                cmp = Find_rect_comp2(ictmp,wave); 
                if (equivalent_comps(cmp,new_comp,newfr->interf) == NO)
                    return NO;
            }
            for (i = 0; i < dim; ++i)
                ictmp[i] = icoords[0][i];
        }
         
        /* Check the crossings in the other direction */
        for(i = -endpt; i <= endpt; i++)
        {
            icrds[idir] = is - i;
            // nearest_crossing2d
            if(__Rect_crossing(icrds,pgdir[0],wave)!= NULL ||
               __Rect_crossing(icrds,pgdir[1],wave)!= NULL)
            {
                /*
                printf("Catched in dir[%d] ndir[%d], ic[%d, %d] griddir[%d, %d]\n",
                  idir, indir, icoords[0][0], icoords[0][1], pgdir[0], pgdir[1]);
                */
                return NO;
            }
        }
        
        /* Check the crossings in the sweep direction */
        for(j = -1; j <= 1; j+=2)
        {
            for(side = 0; side <= 1; side++)
            {
                if(side == 0) sign = -1;
                else sign = 1;
                for (i = 0; i < dim; ++i)
                    icrds[i] = icoords[0][i];
                icrds[indir] += sign*1;
                if(__Rect_crossing(icrds,npgdir[0],wave)!= NULL ||
                   __Rect_crossing(icrds,npgdir[1],wave)!= NULL)
                {
                    return NO; 
                }
            }
        }

        if (endpt >= vsten_rad)
            return YES;

	/* Should add ?? */
        for (i = 0; i < dim; ++i)
            icrds[i] = icoords[0][i];

        for (i = endpt+1; i <= vsten_rad; ++i)
        {
            icrds[idir] = is - i;
            cmp = Find_rect_comp(idir,icrds,wave);
            cross = __Rect_crossing(icrds,next_side,wave);
            if ((equivalent_comps(cmp,new_comp,newfr->interf) == NO) ||
                (cross != NULL))
                return NO;

            icrds[idir] = is + i;
            cmp = Find_rect_comp(idir,icrds,wave);
            cross = __Rect_crossing(icrds,prev_side,wave);
            if ((equivalent_comps(cmp,new_comp,newfr->interf) == NO) ||
                (cross != NULL))
                return NO;
        }
        return YES;
}               /*end unsplit_interior_sten_reg*/


/*
*                       set_unsplit_stencil_comps_and_crosses():
*
*       This function loads the components of each point and any crosses
*       that occur in the interior of the stencil.  For the crosses,
*       each stencil entry contains the list of crosses lying on its interior
*       side, and these lists are ordered towards the outer points.
*       Thus sten->crx[0] is meaningless, and sten->crx[1] contains the list
*       of crosses (if any) in order FROM point 0 TO point 1.  Periodic
*       boundaries are essentially invisible (cf Find_rect_comp).
*/

LOCAL   COMPONENT set_unsplit_stencil_comps_and_crosses(
        int             idir,
        int             swp_num,
        int             *iperm,
        Stencil         *sten)
{
        Wave            *wave = sten->wave;
        Wave            *newwave = sten->newwave;
        int             **icoords = sten->icoords, ic[MAXD];
        TRI_GRID        *tg = wave_tri_soln(wave)->tri_grid;
        int             i, *nc = sten->nc;
        CRXING          ***crx = sten->crx;
        COMPONENT       *comp = sten->comp;

        int             indir, j; 
        GRID_DIRECTION  *pgdir, gdir[2][2] = {NORTH, SOUTH,
                                      EAST, WEST};


        sten->newcomp = Rect_comp(icoords[0],newwave);
        comp[0] = Rect_comp(icoords[0],wave);
        crx[0][0] = NULL;
        nc[0] = 0;
        sten->hs[0] = NULL;/*TODO REMOVE*/

        for (i = 1; i <= endpt; ++i)
        {
            sten->hs[-i] = sten->hs[i] = NULL;/*TODO REMOVE*/
            comp[-i] = Find_rect_comp(idir,icoords[-i],wave);
            nc[-i] = crossings_in_direction(crx[-i],icoords[-i+1],prev_side,tg);
            comp[i] = Find_rect_comp(idir,icoords[i],wave);
            nc[i] = crossings_in_direction(crx[i],icoords[i-1],next_side,tg);
        }

        indir = iperm[(swp_num+1)%dim];
        if(indir == 0)
            pgdir = gdir[1];
        else
            pgdir = gdir[0];

        for(j = -1; j <= 1; j++)
        {
            for (i = 0; i < dim; ++i)
                ic[i] = icoords[0][i];
            ic[idir] += j; 
            ic[indir] += 1; 
            sten->Tcomp[1][j] = Find_rect_comp2(ic,wave);
        }
        for(j = -1; j <= 1; j++)
        {
            for (i = 0; i < dim; ++i)
                ic[i] = icoords[0][i];
            ic[idir] += j; 
            ic[indir] -= 1; 
            sten->Tcomp[0][j] = Find_rect_comp2(ic,wave);
        }

        return sten->newcomp;
}               /*set_unsplit_stencil_comps_and_crosses*/

LOCAL COMPONENT Find_rect_comp2(
        int             *ic,
        Wave            *wave)
{
        int             imax0, imax1;

        imax0 = gmax[0] + ubuf[0];
        imax1 = gmax[1] + ubuf[1];

        if ((ic[0] < -lbuf[0]) || (ic[0] >= imax0) ||
            (ic[1] < -lbuf[1]) || (ic[1] >= imax1))
            return ext_comp;
        else
            return Rect_comp(ic,wave);

}               /*Find_rect_comp2*/

#endif /* #if defined(CONSERVATIVE_ALG) && defined(TWOD) */




