/*
*				gstglobs.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains global variables for the use in the gas dynamics code.
*	All globals in this file are local to the file, but can be
*	evaluated or set by calls to the functions in this file.
*
*
*	TODO:  Introduce a physics structure to pass these parameters
*
*	Contains
*
*		g_sizest()
*		set_coord_sys()
*		g_set_sizeof_state()
*		set_composition_type()
*		g_obstacle_state()
*		return_obst_state()
*/

#include <gdecs/gdecs.h>

typedef struct {
	AVISC	Avisc;	       /* Global defaults for artifical parameters */
	float	alpha;	       /* geometry factor */
	GRAVITY	*grav_data;	  /* gravitational force,  NULL if no gravity */
	int	composition_type; /* generic type of materials in flow */
	size_t	sizest;	       /* size of flow state structure */
	int	nfloats;       /* number of floating point variables in state*/
	int	n_comps;       /* number of possible components in a state*/
	GEOMETRY geometry;     /* geometry flag for coordinate system */
	int	dim;	       /* number of space dimensions in the flow */
	Gas_param **params_list;/* List of EOS params used in simulation */
#if defined(ONED)
	int	state_type;
#endif /* defined(ONED) */
} FLOW_PARAMS;

	/* LOCAL function declarations */
LOCAL const float *astrophysical_gravity(const float*,GRAVITY*);
LOCAL const float *radial_gravity(const float*,GRAVITY*);
LOCAL const float *time_dependent_gravity(const float,GRAVITY*);

LOCAL	FLOW_PARAMS Fparams = {
	{
	    NULL,  /* hyp_method */
	    NO,    /* use_lin_av */
	    NO,	   /* use_lapidus_av */
	    NO,	   /* use_upwind_av */
	    YES,   /* use_msf */
	    0.0,   /* linear_visc_coef */
	    0.0,   /* SEE NOTE!!! lapidus_visc_coef */
	    0.0,   /* upwind_visc_coef */
	    2.0,   /* msf_ieta */
	    0.25,  /* min_shock_jump */
	    1e-06, /* min_sp_vol_jump */
	    0.0,   /* heat_cond */
	    1.0,   /* SEE NOTE!!! sp_coef */
	    0.0,   /* char_speed_cutoff */
	    0.0    /* dynamic surface tension */
	}, /* Avisc */
	0.0,			/* alpha */
	NULL,			/* grav_data */
	PURE_NON_REACTIVE,	/* composition_type */
	sizeof(Gas),		/* sizest */
	MAXD+2,			/* nfloats */
	1,			/* n_comps */
	RECTANGULAR,		/* geometry */
	MAXD,			/* dim */
	NULL,			/*params_list */
#if defined(ONED)
	GAS_STATE,		/* state_type */
#endif /* defined(ONED) */
};

EXPORT size_t g_sizest(void)
{
	return Fparams.sizest;
}		/*end g_sizest*/

EXPORT int g_nfloats(void)
{
	return Fparams.nfloats;
}		/*end g_nfloats*/

EXPORT	void g_set_sizeof_state(
	CHART		*chart,
	size_t		sizest,
	int		nfloats)
{
	Fparams.sizest = sizest;
	Fparams.nfloats = nfloats;
	if (chart != NULL)
	{
	    chart->wave->sizest = sizest;
	    chart->wave->nfloats = nfloats;
	    chart->front->sizest = sizest;
	    chart->front->nfloats = nfloats;
	}
}		/*end g_set_sizeof_state*/

EXPORT void set_composition_type(
	int		type)
{
	Fparams.composition_type = type;
}		/*end set_composition_type*/

EXPORT int g_composition_type(void)
{
	return Fparams.composition_type;
}		/*end g_composition_type*/


EXPORT void g_obstacle_state(
	Locstate	state,
	size_t		sizest)
{
	if (sizest != 0)
	{
	    zero_scalar(state,sizest);
	    Dens(state) = MACH_EPS;/*Allow divide by rho*/
	    Params(state) = NULL;
	    set_type_of_state(state,OBSTACLE_STATE);
	}
}		/*end g_obstacle_state*/

EXPORT Locstate return_obst_state(void)
{
	static	Locstate obst_state = NULL;

	if (obst_state == NULL)
	{
	    scalar(&obst_state,Fparams.sizest);
	    g_obstacle_state(obst_state,Fparams.sizest);
	}

	return obst_state;
}		/*end return_obst_state*/


EXPORT void set_gravity(
	GRAVITY		*grav_data)
{
	F_USER_INTERFACE *fuh;
	struct Table    *T;
	int		i;

	Fparams.grav_data = grav_data;
	if (grav_data == NULL || grav_data->type == NO_GRAVITY)
	    return;
	
	for (i = 0; i < MAXD; i++)
	{
	    fuh = f_user_hook(i+1);
	    fuh->_reflect_state = g_reflect_and_stratify_state;
	}
	for (T = interface_table_list(); T != NULL; T = T->next)
	{
	    if (interface_type(T->interface) != PHYSICAL_INTERFACE)
	    	continue;
	    f_user_interface(T->interface)._reflect_state =
	    	g_reflect_and_stratify_state;
	}
}		/*end set_gravity*/

EXPORT	bool is_gravity(void)
{
	if (Fparams.grav_data == NULL)
	    return NO;
	else if (Fparams.grav_data->type == NO_GRAVITY)
	    return NO;
	else
	    return YES;
}		/*end is_gravity*/

EXPORT const float *gravity(
	const float *coords,
	const float time)
{
	static float no_g[3] = {0.0, 0.0, 0.0};
	if ((Fparams.grav_data==NULL) || (Fparams.grav_data->type==NO_GRAVITY))
	{
	    return (const float*) no_g;
	}
	switch (Fparams.grav_data->type)
	{
	case NO_GRAVITY:
	case CONSTANT_GRAVITY:
	    return (const float*) Fparams.grav_data->g;
	case TIME_DEPENDENT_GRAVITY:
	    return time_dependent_gravity(time,Fparams.grav_data);
	case ASTROPHYSICAL_GRAVITY:
	    return astrophysical_gravity(coords,Fparams.grav_data);
	case RADIAL_GRAVITY:
	    return radial_gravity(coords,Fparams.grav_data);
	case USER_DEFINED:
	    return user_defined_gravity(coords,time,Fparams.grav_data);
	default:
	    screen("ERROR in gravity(), invalid gravity type %d\n",
		   Fparams.grav_data->type);
	    clean_up(ERROR);
	}
	return (const float*) no_g;
}		/*end gravity*/

EXPORT	void	eval_gravity(
	const float *coords,
	const float time,
	float	    *grav)
{
	const float *g = gravity(coords,time);
	int   i, dim = Fparams.dim;
	for (i = 0; i < dim; i++)
	    grav[i] = g[i];
}		/*end eval_gravity*/

LOCAL const float *time_dependent_gravity(
	const float time,
	GRAVITY	    *grav_data)
{
	float	*g = grav_data->g;
	float	**g_of_t;
	float	tau;
	int	i, dim, num_time_points;
	static	float last_time;
	static	int last_n;
	static	bool first = YES;

	if (first == YES)
	{
	    last_time = -HUGE_VAL;
	    last_n = 0;
	}
	if (time == last_time)
	    return (const float*) g;
	last_time = time;
	dim = grav_data->dim;
	num_time_points = grav_data->num_time_points;
	g_of_t = grav_data->g_of_t;

	if (time < g_of_t[0][0])
	{
	    last_n = 0;
	    for (i = 0; i < dim; i++)
		g[i] = g_of_t[0][i+1];
	    return (const float*) g;
	}
	if (time > g_of_t[num_time_points-1][0])
	{
	    last_n = num_time_points-1;
	    for (i = 0; i < dim; i++)
		g[i] = g_of_t[num_time_points-1][i+1];
	    return (const float*) g;
	}
	for (i = last_n; i < (num_time_points-1); i++)
	    if ((g_of_t[i][0] <= time) && (time < g_of_t[i+1][0]))
		break;
	if (i == (num_time_points-1))
	{
	    for (i = 0; i < (num_time_points-1); i++)
	        if ((g_of_t[i][0] <= time) && (time < g_of_t[i+1][0]))
		break;
	}
	if (i == (num_time_points-1))
	{
	    screen("ERROR in time_dependent_gravity(), can't find time %g\n",
		   time);
	}
	last_n = i;
	tau = (time - g_of_t[last_n][0]) /
	      (g_of_t[last_n+1][0] - g_of_t[last_n][0]);
	for (i = 0; i < dim; i++)
	{
	    g[i] = (1.0 - tau)*g_of_t[last_n][i+1] + tau*g_of_t[last_n+1][i+1];
	}
	return (const float*) g;
}		/*end time_dependent_gravity*/

LOCAL const float *astrophysical_gravity(
	const float *coords,
	GRAVITY     *grav_data)
{
	float	*g = grav_data->g;
	float	*cen = grav_data->center;
	float	G = grav_data->G;
	float	M = grav_data->M;
	int	i, dim = grav_data->dim;
	float	r;

	for (i = 0; i < dim; i++)
	    g[i] = cen[i] - coords[i];
	r = mag_vector(g,dim);
	for (i = 0; i < dim; i++)
	    g[i] = G*M*g[i]/(r*r*r);
	return (const float*)g;
}		/*end astrophysical_gravity*/

LOCAL const float *radial_gravity(
	const float *coords,
	GRAVITY     *grav_data)
{
	float	*g = grav_data->g;
	float	*cen = grav_data->center;
	float	G = grav_data->G;
	int	i, dim = grav_data->dim;
	float	r;

	for (i = 0; i < dim; i++)
	    g[i] = coords[i] - cen[i];
	r = mag_vector(g,dim);
	for (i = 0; i < dim; i++)
	    g[i] = G*g[i]/r;
	return (const float*)g;
}		/*end radial_gravity*/

EXPORT	bool source_terms_exist(void)
{
	if (is_gravity() == YES)
	    return YES;
#if defined(ROTATIONAL_SYMMETRY)
	if (rotational_symmetry() > 0.0)
	    return YES;
#endif /* defined(ROTATIONAL_SYMMETRY) */
	return NO;
}		/*end source_terms_exist*/

EXPORT void set_default_artificial_viscosity(
	AVISC		*avisc)
{
	if ((avisc != NULL) && (avisc != &Fparams.Avisc))
		Fparams.Avisc = *avisc;
	Fparams.Avisc.sp_coef =
		lapidus_stability_factor(Fparams.Avisc.lapidus_visc_coef);
}		/*end set_default_artificial_viscosity*/


EXPORT void default_artificial_viscosity(
	AVISC		*avisc)
{
	if (avisc == NULL)
	    return;
	if (avisc != &Fparams.Avisc)
	    *avisc = Fparams.Avisc;
	avisc->sp_coef = lapidus_stability_factor(avisc->lapidus_visc_coef);
}		/*end default_artificial_viscosity*/


EXPORT	void use_artificial_dissipation(
	AVISC		*avisc)
{
	Gas_param	**params_list;
	int		i, nprms;

	default_artificial_viscosity(avisc);
	nprms = return_params_list(&params_list);
	for (i = 0; i < nprms; i++)
	{
		if (use_lapidus_artificial_viscosity(params_list[i]->avisc))
			use_lapidus_artificial_viscosity(*avisc) = YES;
		if (use_linear_artificial_viscosity(params_list[i]->avisc))
			use_linear_artificial_viscosity(*avisc) = YES;
		if (use_upwind_artificial_viscosity(params_list[i]->avisc))
			use_upwind_artificial_viscosity(*avisc) = YES;
		if (use_muscl_slope_flattening(params_list[i]->avisc))
			use_muscl_slope_flattening(*avisc) = YES;
	}
	return;
}		/*end use_artificial_dissipation*/


/*
*                       set_coord_sys():
*
*	The coordinate remap is visible in the gas code through the
*	coord_system() function.  This function initializes the remap
*	identifier (Fparams.geometry) returned by coord_system().
*	It must be called before coord_system() is used. 
*
*	Input:  remap		- coordinate remap
*	Output: none		- Fparams.geometry is set
*
*/

EXPORT void set_coord_sys(
	int		remap,
	int		dim)
{
	Fparams.dim = dim;
	switch (remap)
	{
	case IDENTITY_REMAP:
	    Fparams.geometry = RECTANGULAR;
	    break;
	case CYLINDRICAL_REMAP:
	    Fparams.geometry = CYLINDRICAL;
	    break;
	case SPHERICAL_REMAP:
	    Fparams.geometry = SPHERICAL;
	    break;
	default:
	    screen("ERROR in set_coord_sys(), Illegal remap %d "
		   "in set_coord_sys\n",remap);
	    clean_up(ERROR);
	    break;
	}
	switch (Fparams.geometry)
	{
	case SPHERICAL:
	    Fparams.alpha = 2.0;
	    break;
	case CYLINDRICAL:
	    Fparams.alpha = 1.0;
	    break;
	case RECTANGULAR:
	default:
	    Fparams.alpha = 0.0;
	}
}		/*end set_coord_sys*/

/*
*		coord_system():
*
*	This function is used to make the coordinate remap visible
*	throughout the gas code.  It can only be called after
*	set_coord_sys() (which is called in init_physics).
*/

EXPORT GEOMETRY coord_system(void)
{
 	if (debugging("coord_system"))
	    screen("coord_system called\n");
	return Fparams.geometry;
}		/*end coord_system*/


EXPORT float rotational_symmetry(void)
{
	return Fparams.alpha;
}		/*end rotational_symmetry*/


EXPORT GEOMETRY Geometry(
	float		*aa)
{
	if (aa != NULL)
	    *aa = Fparams.alpha;
	return Fparams.geometry;
}		/*end Geometry*/
