/*
*				glayer.h
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Declarations and definitions for layer statistics and extrema
*/

#if !defined(_GLAYER_H)
#define _GLAYER_H

#include <gdecs/gdecs.h>

#define	DEBUG_STRING	"glayer"

enum {
	MIN_POS_ID   = FIRST_PHYSICS_MESSAGE_ID+1,
	MIN_COMP_ID,
	MIN_STATE_ID,
	MAX_POS_ID,
	MAX_COMP_ID,
	MAX_STATE_ID,

	G_IMIN_ID     = FIRST_PHYSICS_MESSAGE_ID+10,
	G_IMAX_ID,
	G_HMIN_ID,
	G_HMAX_ID,
	LAYER_SUM_ID,
	INTFC_SUM_ID,
	LAYER_FRAC_ID,

/* This ID should be last, because it uses subsequent ID's 
   in the case of parallel processors: */
	L_MIN_MAX_ID = LAYER_FRAC_ID
};

/* mapping from (i,j) entry of a symmetric matrix (j >= i) 
   to an array of dimension MAXD*(MAXD-1)/2 */
#define sym_index(i,j) (i*MAXD-i*(i+1)/2+j)

enum intfc_geometry { planar, spherical };

typedef struct {
    float 		d;		/* density */
    float 		v[MAXD];	/* velocity */
    float 		p;		/* pressure */
    float               k;              /* kinetic energy */
    float 		e;		/* internal energy */
    float		frac;		/* layer fraction */
    int			count;		/* number of pts summed within layer */
} Big_State;


typedef struct {
    float		dd;             /* density*density */
    float		de;             /* internal energy density */
    float 		dv[MAXD];       /* momentum */
    float		dkv[MAXD];      /* kinetic energy flux */
    float		dev[MAXD];      /* internal energy flux */
    float 		pv[MAXD];       /* pressure work */
    float 		vv[MAXD*(MAXD+1)/2];   /* velocity covariances  */
                                               /*   XX, XY, XZ, YY, YZ, ZZ */
    float 		dvv[MAXD*(MAXD+1)/2];  /* momentum flux tensor  */
                                               /*   XX, XY, XZ, YY, YZ, ZZ */
} Turbulence_moments;


typedef struct {
    OUTPUT_DATA         odata;
    FILE		*out_min, *out_max, *out_amp;
    FILE		*out_min1, *out_max1, *out_amp1;
    FILE		*out_min5, *out_max5, *out_amp5;
    bool		do_01, do_05;
    char                *min_fname, *max_fname, *amp_fname;
    char		*min_dname, *max_dname, *amp_dname;
    enum intfc_geometry geom;
    float 		origin[3];
    float		rfactor;
    float		pos[2][3];       /* 0 <--> min; 1 <--> max; */
    int			index_at_min, index_at_max;
} Intfc_extrema;

#define Intfc_extrema_data(odata)	(Intfc_extrema*)odata

typedef struct {	 

    /* quantities that are summed at interfaces within each layer: */

    float nor[MAXD];	 /* gradient of characteristic function, */
    float vdotn;         /* interface velocity dotted with normal, */
    float pre[MAXD];     /* pressure times normal */
    float pvdotn;        /* pressure times velocity dotted with normal */

    float nor_fac;       /* For 2D, number of points of intersection of 
			  * contact surface with x axis at a particular z.
			  * For 3D, total length of lines of intersection of
			  * contact surface with (x,y) plane at particular z */

    int count;           /* Number of occurrences of intersection of contact
			  * surface with x axis or (x,y) plane at a particular
			  * z.  Redundant for 2D, but needed as a flag for 3D
			  * to avoid division by 0 during normalization. */
} Intfc_data;


typedef struct {
    OUTPUT_DATA	odata;
    char	column_header[Gets_BUF_SIZE];

    enum intfc_geometry geom;           /* layer geometry */
    float 		origin[MAXD];   /* needed for spherical geometry */
    
    float 	h_min;         /* range of height or radius over which to */
    float 	h_max;         /* compute intfc stats */
    			       
    int		n_layers;      /* number of layers ranging from
                                * h_min to h_max inclusive */

    float 	dh;            /* (h_max - h_min)/(n_layers - 1) */
    			       
    Intfc_data *data;    /* array containing the data for each layer */
	                                   
} Intfc_stats;


typedef struct {
    OUTPUT_DATA         odata;
    FILE		**bst_out;  	/* array of output files of */
    char		**bst_fname;    /* layer-averaged Big_States, one */
    char		**bst_dname;    /* for each distinct material, with */
                                        /* associated file and dir names */

    bool		include_turb;
    FILE		**turb_out;     /* array of output files of */
    char                **turb_fname;   /* layer-averaged Turbulent_moments, */
    char		**turb_dname;   /* one for each material, with */
                                        /* associated file and dir names */

    enum intfc_geometry geom;           /* layer geometry */
    int			n_params;	/* number of distinct materials */
    float 		origin[MAXD];   /* needed for spherical geometry */
    float		rfactor;        /* sub-grid refinement factor */

    char	bst_col_header[Gets_BUF_SIZE];
    char	turb_col_header[Gets_BUF_SIZE];
    
    float 	h_min;         /* range of height or radius over which to */
    float 	h_max;         /* compute layer stats */
    			       
    int		n_layers;      /* number of layers ranging from
                                * h_min to h_max inclusive */

    float 	dh;            /* (h_max - h_min)/(n_layers - 1) */
    			       
    Big_State   *bst_data;    /* arrays containing the data for each layer */
    Turbulence_moments* turb_data;
	                                   
} Layer_stats;


#if defined(TWOD) || defined(THREED)

IMPORT	void accumulate_state_in_layer(Wave*,Front*,float,Big_State*,
				        Turbulence_moments*,float,const float*,
					bool,bool,bool);
IMPORT	void accumulate_state_totals(const Big_State*,const Turbulence_moments*,
				     Big_State*,Turbulence_moments*);
IMPORT	void normalize_state_totals(Big_State*,Turbulence_moments*,int);

#endif  /* defined(TWOD) || defined(THREED) */

#endif /* !defined(_GLAYER_H) */
