/*
*				ginitintfc.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains initialization routines for gas dynamics.
*
*	The main routines in this file that are accessed through function
*	pointers are:
*
*	g_init_interface(), called from init() in dinit.c.
*
*	g_init_cauchy_data_pointers(), g_initializer(), g_intfc_initializer(),
*	g_restart_initializer(), and g_restart_intfc_initializer(), called
*	from init_states() in dinit.c.
*/

#include <ginit/ginit.h>

#if defined(ONED)
struct _DENSITY_STEP_WALL_DATA {
	OUTPUT_DATA odata;

	COMPONENT   *pcomp;
	Locstate    *exact_wstate;
	Locstate    *pstate;
	Locstate    *wave_head, *wave_tail;
	bool        inside_wave;
	bool        isforward;
	float       c, s;
	float       wave_tol;
	float       *exact_incomingWs, *exact_outgoingWs;
	float       *wave_head_time, *wave_tail_time, *wave_mid_time;
	float       Plast;
	float       *px;
	float       *t;
	float       DX;
	int         num_exact_wall_states;
	int         NumAllocWallEvents;
	int         num_events;
	int         num_samples;
	size_t      sizest;
};
typedef struct _DENSITY_STEP_WALL_DATA DENSITY_STEP_WALL_DATA;
#endif /* defined(ONED) */

	/* LOCAL Function Declarations */
LOCAL	void	free_ambient_comp_type(COMP_TYPE*);
LOCAL	void	g_bdry_state_initializer(int,int,HYPER_SURF*,INIT_DATA*,
					 INIT_PHYSICS*);
LOCAL	void	g_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*);
LOCAL	void	g_FM_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL	void	g_Dual_cell_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,POLYGON*);
LOCAL	void	g_intfc_initializer(POINT*,HYPER_SURF_ELEMENT*,HYPER_SURF*,
				    Locstate,Locstate,INIT_DATA*);
LOCAL	void	get_state_exterior(float*,Locstate,COMP_TYPE*,
				   HYPER_SURF*,INTERFACE*,INIT_DATA*,int);
LOCAL	void	get_state_obstacle(float*,Locstate,COMP_TYPE*,
				   HYPER_SURF*,INTERFACE*,INIT_DATA*,int);
LOCAL	void	init_ambient(INIT_DATA*,INIT_PHYSICS*);
LOCAL	void	init_tri_grid_test(INIT_DATA*,INIT_PHYSICS*);
LOCAL	void	print_comp_types(INTERFACE*);
LOCAL	void	print_component_type(const char*,COMP_TYPE*,const char*);
LOCAL	void	prompt_for_boundary_flags(INIT_DATA*,INIT_PHYSICS*,
					  const Prompt_type*);
LOCAL	void	prompt_for_contact_wall_params(const IO_TYPE*,Front*);
LOCAL	void	prompt_for_rect_boundary_flags(INIT_DATA*,INIT_PHYSICS*,
					       const Prompt_type*);
LOCAL	void	set_exterior_comp_type(COMP_TYPE*);
LOCAL	void	set_passive_curve_flags(Front*);

#if defined(ONED)
LOCAL	DENSITY_STEP_WALL_DATA *alloc_DENSITY_STEP_WALL_DATA(int,int,Front*);
LOCAL	void    expand_num_wall_events(DENSITY_STEP_WALL_DATA*,Front*);
LOCAL	void	g_1d_bdry_state_initializer(int,int,POINT*,INIT_DATA*,
					    INIT_PHYSICS*);
LOCAL	void	init_oned_density_step(INIT_DATA*,INIT_PHYSICS*);
LOCAL	void	prompt_for_bdry_flags1d(INIT_DATA*,INIT_PHYSICS*,
					const Prompt_type*);
LOCAL	void	print_density_step_wall_data(Grid*,Wave*,Front*,Printplot*,
                                             OUTPUT_DATA*,bool);
#endif /* defined(ONED) */

#if defined(TWOD)
LOCAL	void	g_2d_bdry_state_initializer(int,int,CURVE*,INIT_DATA*,
					    INIT_PHYSICS*);
LOCAL	void	init_bow_shock(INIT_DATA*,INIT_PHYSICS*);
LOCAL	void	init_expanding_shock(INIT_DATA*,INIT_PHYSICS*);
LOCAL	void	prompt_for_bdry_flags2d(INIT_DATA*,INIT_PHYSICS*,
					const Prompt_type*);
LOCAL	void	prompt_for_contact_wall_node_params(const IO_TYPE*,Front*);
#endif /* defined(TWOD) */

#if defined(THREED)
LOCAL	void	g_3d_bdry_state_initializer(int,int,SURFACE*,INIT_DATA*,
					    INIT_PHYSICS*);
LOCAL	void	prompt_for_bdry_flags3d(INIT_DATA*,INIT_PHYSICS*,
					const Prompt_type*);
#endif /* defined(THREED) */

#if defined(ROTATIONAL_SYMMETRY)
LOCAL	void	prompt_for_cylindrical_pencil(INIT_DATA*,INIT_PHYSICS*);
#endif /* defined(ROTATIONAL_SYMMETRY) */

LOCAL 	void    set_time_dep_pres(BOUNDARY_STATE*,float,float,float,
			float,float);
LOCAL   void    prompt_for_adherence_coeff(HYPER_SURF *hs);

LOCAL	void	add_vortex_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL	void	vortex_evolution_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL	void	vortex_evolution_initializer_interp(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL	void	vortex_evolution_initializer_least_sqr(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL	void	init_Shu_Osher_problem(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL	void	init_db_Mach_state(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*);

LOCAL	void	sine_evolution_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL	void	twod_riemann_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL	void	Burgers_evolution_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL	void	Burgers_evolution_initializer_13_quad(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL   float   ff_Burgers(float*);
//LOCAL   float   ff_g_sin(float*,int);
// LOCAL   float   ff_vort(float*,int);
LOCAL   float   ff_Shu_Osher(float*,int,float);
LOCAL   float   ff_2d_riemann(float*,int);
LOCAL   float   ff_shock_vort(float*,float*,int);
LOCAL   void    add_accurate_vortex_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL   void    accurate_magnetic_vortex_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL   void    accurate_magnetic_vortex_initializer_2(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL   void    accurate_magnetic_vortex_initializer_3(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL   void    accurate_magnetic_vortex_initializer_4(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL   void    accurate_magnetic_rotor_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL   void    accurate_Orszag_T_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL   void    accurate_magnetic_blast_MHD_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL   void    accurate_magnetic_decay_alfven_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);
LOCAL   void    field_loop_advection_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,TRI*);

LOCAL   double   vh_val_ver2(double,int);
LOCAL   void    dual_cell_accurate_magnetic_vortex_initializer(float*,COMPONENT,Locstate,
                             INTERFACE*,INIT_DATA*,POLYGON*);
LOCAL   void    dual_cell_delay_alfven_initializer(float*,COMPONENT,Locstate,
                             INTERFACE*,INIT_DATA*,POLYGON*);
LOCAL   void    dual_cell_magnetic_rotor_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,POLYGON*);
LOCAL   void    dual_cell_magnetic_Orszag_T_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,POLYGON*);
LOCAL   void    dual_cell_blast_MHD_initializer(float*,COMPONENT,Locstate,INTERFACE*,INIT_DATA*,POLYGON*);
LOCAL   void    dual_cell_field_loop_advection_initializer(float*,COMPONENT,Locstate,
                             INTERFACE*,INIT_DATA*,POLYGON*);
LOCAL   double  Az_field_loop_adv(double*);  


// LOCAL   void    trans_matrix(double**,int,int,double**);

/*
*			g_init_interface():
*
*	Initializes the interface points, constraints, and the components.
*	Any choice of  a component which is a union of connected components
*	of domain\interface is allowed; the the union of the components must 
*	equal domain\interface.
*
*	The function pointer ip->init_interface is set to be g_init_interface
*	in set_basic_phys_parameters().
*/

/*ARGSUSED*/
EXPORT void g_init_interface(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	Front	  *front = ip->root->front;
	CROSS	  *cross;
	INTERFACE *intfc = front->interf;
	int	  max_n_comps = max_num_comps();
	int       ptOffset = g_init_data(init)->_promptForBoundaryFlagsOffset;
	const Prompt_type *ptypes = bdry_wave_type_prompt_type(init);
        bool      status;

	debug_print("init","Entered g_init_interface()\n");

	exclude_comp(COMPOBST,intfc);

	init_problem_type(ip);
	if (restart_io_type(init) == NULL)
	{
#if defined(ROTATIONAL_SYMMETRY)
	    if (coord_system() == CYLINDRICAL) 
	        prompt_for_cylindrical_pencil(init,ip);
#endif /* defined(ROTATIONAL_SYMMETRY) */

	    /* Set initial interface */
	    prompt_for_problem_specific_data(init,ip);

#if defined(ROTATIONAL_SYMMETRY) && defined(TWOD)
	    if (coord_system() == CYLINDRICAL) 
	    	make_vertical_axis_curve(init,ip);
#endif /* defined(ROTATIONAL_SYMMETRY) && defined(TWOD) */

	    if (debugging("init")) 
	    {
	        (void) output();
	        (void) printf("\t\t\tINTERFACE BEFORE set_boundary()\n\n");
	        print_interface(intfc);
	    }

	    /* Set Boundary Information */
	    prompt_for_rect_boundary_flags(init,ip,ptypes);

	    /* TODO Unify 2 and 3 D */
	    if (front->rect_grid->dim == 3)
	    	clip_front_to_rect_boundary_type(front);

	    if (set_boundary(intfc,front->rect_grid,FIRST_DYNAMIC_COMPONENT,
			     grid_tolerance(front->rect_grid)) !=
			     FUNCTION_SUCCEEDED)
	    {
	    	screen("ERROR in g_init_interface(), set_boundary() failed\n");
	    	print_interface(intfc);
	    	clean_up(ERROR);
	    }
	    set_passive_curve_flags(front);
	    /*
	     * Second pass over boundary curves.
	     * Only wave types with index >= ptOffset are prompted for.
	     * See bdry_wave_type_prompt_type().
	     */
	    prompt_for_boundary_flags(init,ip,ptypes+ptOffset);
	    if (debugging("reverse"))
	    {
	    	CURVE **c;
	    	for (c = front->interf->curves; c && *c; c++)
	    	    if (wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE)
			invert_curve(*c);
	    }

	    if (debugging("init")) 
	    {
	        (void) output();
	        (void) printf("\t\t\tINTERFACE after set_boundary()\n\n");
	        print_interface(intfc);
	    }

	}
	prompt_for_contact_wall_params(restart_io_type(init),front);

	if (max_component(intfc) >= max_n_comps) 
	{
	    screen("ERROR in g_init_interface(), max_component(intfc) = %d "
	           ">= max_n_comps = %d "
	           "in g_init_interface()\n",max_component(intfc),
		   max_n_comps);
	    clean_up(ERROR);
	}
	set_exterior_comp_type(comp_type(exterior_component(intfc)));

	    /* Check for intersections */

#if defined(THREED)
	if (debugging("plot3d") && intfc->dim == 3)
	    gview_plot_interface("plot-intfc",intfc);
#endif /* defined(THREED) */

        if(Tracking_algorithm(front) == GRID_FREE_TRACKING || 
           front->rect_grid->dim == 2)
        {
	    status = intersections(intfc,&cross,NO); 
            status = pp_min_status(status);
	    if (status == FUNCTION_FAILED) 
	    {
	        screen("ERROR in g_init_interface(), "
	               "intersections() failed\n\n\n");
	        (void) print_interface(intfc);
	        clean_up(ERROR);
    	    }

	    if (cross != NULL)
	    {
	        screen("ERROR in g_init_interface(), interface tangled\n");
	        (void) print_number_of_tangles("initial ",intfc,cross);
	        (void) print_intersections(cross,intfc);
	        clean_up(ERROR);
    	    }
        }

	set_params_list(intfc);

	if (debugging("init")) 
	{
	    (void) output();
	    (void) printf("\t\t\tINITIAL INTERFACE:\n\n");
	    print_interface(intfc);
	}
	debug_print("init","Left g_init_interface()\n");
}		/*end g_init_interface*/

/*ARGSUSED*/
EXPORT int g_prompt_for_wave_type(
	const char	*mesg,
	INTERFACE	*intfc,
	INIT_PHYSICS	*ip)
{
	int   w_type;
	char  s[Gets_BUF_SIZE];

	if (mesg == NULL)
	    mesg = "";

	screen("Possible choices for wave types are\n"
	       "\t Forward shock wave (f), \n"
	       "\t Forward sound wave leading edge (fl), \n"
	       "\t Forward sound wave trailing edge (ft), \n"
	       "\t Backward shock wave (b), \n"
	       "\t Backward sound wave leading edge (bl), \n"
	       "\t Backward sound wave trailing edge (bt), \n"
	       "\t Contact (c), \n"
	       "\t Neumann boundary (n), \n"
	       "\t Dirichlet boundary (d), \n"
	       "\t Periodic boundary (p), \n"
	       "\t Passive boundary (pa), \n");
	if (supports_riemann_problem_waves(ip))
	    screen("\t Riemann problem wave (rpw), \n");
	screen("\t Unspecified wave type (u, default), \n"
	       "Enter the wave type%s%s: ",(strlen(mesg) != 0) ? " " : "",mesg);
	(void) Gets(s);
	w_type = read_wave_type_from_string(s,intfc);
	(void) printf("wave type = %s\n",wave_type_as_string(w_type,intfc));
	return w_type;
}		/*end g_prompt_for_wave_type*/


EXPORT	void	g_init_problem_type(
	INIT_PHYSICS	*ip)
{
	int       dim = ip->root->front->rect_grid->dim;
	Prob_type *probtype;
#if defined(ONED)
	static Prob_type Probtype1d[] =
	{
	    {{"an ambient state test","AM",2,{AMBIENT_STATE_TEST}},
	        init_ambient},
	    {"a oned test", "ONED", 1, {ONED_TEST}, init_multi_layer},
	    {"a density step", "DS", 2, {DENSITY_STEP}, init_oned_density_step},
	    {{NULL, NULL, 0, {UNKNOWN_PROBLEM_TYPE}}, NULL}
	};
#endif /* defined(ONED) */
#if defined(TWOD)
	static Prob_type Probtype2d[] =
	{
	    {{"an ambient state test","AM",2,{AMBIENT_STATE_TEST}},
	        init_ambient},
	    {{"a trigrid test", "TRI", 3, {TRI_GRID_TEST}}, init_tri_grid_test},
	    {{"a plane front", "P", 1, {PLANE_FRONT}}, init_multi_layer},
	    {{"a bowshock", "BO", 2, {BOWSHOCK}}, init_bow_shock},
#if defined(FULL_PHYSICS)
	    {{"a Meshkov instability problem","M",1,{MESHKOV}},init_meshkov},
	    {{"a shock diffraction", "D", 1,
		{SHOCK_DIFFRACTION}}, init_shock_diffraction},
	    {{"a shock transmission", "T", 1,
		{SHOCK_TRANSMISSION}}, init_shock_transmission},
	    {{"a ramp reflection problem", "RR", 2,
		{RAMP_REFLECTION}}, NULL},
	    {{"a contact-contact interaction", "CC", 2,
		{CC_NODE_TEST}}, init_CC_interaction},
	    {{"a Richtmyer linear theory", "RL", 2,
		{RICHTMYER_LINEAR_THEORY}}, init_multi_layer},
#endif /* defined(FULL_PHYSICS) */
	    {{"an astrophysical jet",  "AJ", 2,
		{ASTROPHYSICAL_JET}}, init_injection_inlet_jet},
	    {{"an injection inlet jet",  "IJ", 2,
		{INJECTION_INLET}}, init_injection_inlet_jet},
	    {{"a gas injection jet",  "FJ", 2,
		{INJECTION_INLET}}, init_fuel_injection_jet},
	    {{"a neutrino booster colapse",  "NB", 2,
                {NE_BOOSTER}}, init_neutrino_booster_detector},
	    {{"a Supernova simulation", "SN", 2,
		{SUPERNOVA}}, init_el_riem_prob},
	    {{"an imploding elliptical shock", "IMP", 3,
		{IMPLOSION}}, init_el_riem_prob},
	    {{"a shock running over an expanding ramp", "X", 1,
		{EXPANDING_SHOCK}}, init_expanding_shock},
	    {{"a random surface instability problem", "RS", 2,
		{RANDOM_SURFACE}}, init_random_surface},
	    {{"a shocked thermal layer", "STL", 2,
		{SHOCKED_THERMAL_LAYER}}, init_multi_layer},
	    {{"a Richtmyer-Meshkov instability problem", "RM", 2,
		{RICHTMYER_MESHKOV}}, init_multi_layer},
	    {{"a Rayleigh-Taylor instability problem", "RT", 2,
		{RAYLEIGH_TAYLOR}}, init_multi_layer},
	    {{"a bubbles and drops problem", "BD", 2,
		{BUBBLES_DROPS}}, init_multi_layer},
	    {{"an expanding shells", "ES", 2,
		{EXPANDING_SHELLS}}, init_multi_layer},
	    {{"shock jet interaction", "SJ", 2,
		{SHOCK_JET}}, init_multi_layer},
	    {{"a Radial Rayleigh-Taylor instability problem",
		"Radial Rayleigh Taylor", 22,
		{RADIAL_RAYLEIGH_TAYLOR}}, init_el_riem_prob},
	    {{"a Kelvin-Helmholtz instability problem", "KH", 2,
		{KELVIN_HELMHOLTZ}}, init_kelvin_helmholtz},
	    {{NULL, NULL, 0, {UNKNOWN_PROBLEM_TYPE}}, NULL}
	};
#endif /* defined(TWOD) */
#if defined(THREED)
	static	Prob_type Probtype3d[] =
	{
	    {{"an ambient state test","AM",2,{AMBIENT_STATE_TEST}},
	        init_ambient},
	    {{"a random surface instability problem", "RS", 2,
		{RANDOM_SURFACE}}, init_random_surface},
	    {{"a shocked thermal layer", "STL", 2,
		{SHOCKED_THERMAL_LAYER}}, init_multi_layer},
	    {{"a Richtmyer-Meshkov instability problem", "RM", 2,
		{RICHTMYER_MESHKOV}}, init_multi_layer},
	    {{"a Rayleigh-Taylor instability problem", "RT", 2,
		{RAYLEIGH_TAYLOR}}, init_multi_layer},
	    {{"a bubbles and drops problem", "BD", 2,
		{BUBBLES_DROPS}}, init_multi_layer},
	    {{"an expanding shells", "ES", 2,
		{EXPANDING_SHELLS}}, init_multi_layer},
	    {{"shock jet interaction", "SJ", 2,
		{SHOCK_JET}}, init_multi_layer},
	    {{"a Radial Rayleigh-Taylor instability problem",
		"Radial Rayleigh Taylor", 22,
		{RADIAL_RAYLEIGH_TAYLOR}}, init_el_riem_prob},
	    {{"a Kelvin-Helmholtz instability problem", "KH", 2,
		{KELVIN_HELMHOLTZ}}, init_kelvin_helmholtz},
	    {{"an imploding elliptical shock", "IMP", 3,
		{IMPLOSION}}, init_el_riem_prob},
	    {{"an injection inlet jet",  "IJ", 2,
		{INJECTION_INLET}}, init_injection_inlet_jet},
	    {{NULL, NULL, 0, {UNKNOWN_PROBLEM_TYPE}}, NULL}
	};
#endif /* defined(THREED) */

	switch (dim)
	{
#if defined(ONED)
	case 1:
	    probtype = Probtype1d;
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    probtype = Probtype2d;
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    probtype = Probtype3d;
	    break;
#endif /* defined(THREED) */
	default:
	    probtype = NULL;
	    break;
	}

	prompt_for_problem_type(ip,probtype);
}		/* end g_init_problem_type */

EXPORT void	prompt_for_problem_type(
	INIT_PHYSICS    *ip,
	Prob_type 	*Probtype)
{
        char            s[Gets_BUF_SIZE];
        int             i;

	screen("\nRequest problem type.  Current choices are\n");
	for (i = 0; Probtype[i].ptype.prompt != NULL; i++)
	{
	    screen("\t\t");
	    if (Probtype[i+1].ptype.prompt == NULL)
		screen("or ");
	    screen("%s (%s)",Probtype[i].ptype.prompt,Probtype[i].ptype.select);
	    screen("%s\n",(Probtype[i+1].ptype.prompt == NULL) ? "." : ",");
	}
	screen("\tEnter choice here: ");
	(void) Gets(s);

	problem_type(ip) = UNKNOWN_PROBLEM_TYPE;
     	for (i = 0; Probtype[i].ptype.prompt != NULL; i++)
	{
	    if (strncasecmp(s,Probtype[i].ptype.select,
			    Probtype[i].ptype.ncmp)==0)
	    {
	    	problem_type(ip) = prt_problem_type(ip->prt) =
	    		Probtype[i].ptype.type.itype;
	    	g_iphys(ip)->_prompt_for_problem_specific_data =
				Probtype[i].ppsd;
		break;
	    }
	}

	if (problem_type(ip) == UNKNOWN_PROBLEM_TYPE)
	{
	    screen("ERROR in prompt_for_problem_type(), "
	           "unrecognized problem type\n");
	    clean_up(ERROR);
	}
}		/*end prompt_for_problem_type*/

LOCAL void prompt_for_rect_boundary_flags(
	INIT_DATA	  *init,
	INIT_PHYSICS	  *ip,
	const Prompt_type *ptypes)
{
	PP_GRID    *pp_grid = ip->root->front->pp_grid;
	INTERFACE  *intfc = ip->root->front->interf;
	char	   mesg[256];
	int	   i, j, dim = intfc->dim;
	int	   b_type;
	int        me[3], *G;
	static const char *direction[3] = { "x", "y", "z"};
	static const char *side[3][2] = { {"left", "right"},
				          {"lower", "upper"},
				          {"bottom", "top"}
				        };

	G = pp_grid->gmax;
	find_Cartesian_coordinates(pp_mynode(),pp_grid,me);
	for (i = 0; i < dim; i++)
	{
	    for (j = 0; j < 2; j++)
	    {
	        if (rect_boundary_type(intfc,i,j) != UNKNOWN_BOUNDARY_TYPE)
		    continue;
	        (void) sprintf(mesg,"for the %s boundary in the %s direction",
	    		       side[i][j],direction[i]);
	        b_type = prompt_for_bdry_wave_type(init,mesg,ptypes);
		rect_boundary_type(intfc,i,j) = b_type;
	        g_bdry_state_initializer(i,j,NULL,init,ip);
		if (((me[i]>0) && (j==0)) || ((me[i]<(G[i]-1)) && (j==1)))
		    rect_boundary_type(intfc,i,j) = SUBDOMAIN_BOUNDARY;
	        if ((b_type == SUBDOMAIN_BOUNDARY) && (j == 0))
	        {
	            j++;
	            (void) printf("Boundary type for the %s boundary ",
	        	   side[i][1]);
	            (void) printf("in the %s direction:",direction[i]);
		    if (me[i] < (G[i]-1))
	                (void) printf("SUBDOMAIN_BOUNDARY\n");
		    else
	                (void) printf("PERIODIC_BOUNDARY\n");
	            rect_boundary_type(intfc,i,j) = SUBDOMAIN_BOUNDARY;
	            g_bdry_state_initializer(i,j,NULL,init,ip);
	        }
	    }
	}
}		/*end prompt_for_rect_boundary_flags*/

LOCAL void prompt_for_boundary_flags(
	INIT_DATA	  *init,
	INIT_PHYSICS	  *ip,
	const Prompt_type *ptypes)
{
	INTERFACE	*intfc = ip->root->front->interf;

	switch (intfc->dim)
	{
#if defined(ONED)
	case 1:
	    prompt_for_bdry_flags1d(init,ip,ptypes);
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    prompt_for_bdry_flags2d(init,ip,ptypes);
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    prompt_for_bdry_flags3d(init,ip,ptypes);
	    break;/* TODO */
#endif /* defined(THREED) */
	}
}		/*end prompt_for_boundary_flags*/


#if defined(ONED)

/*
*
*			prompt_for_bdry_flags1d():
*
*	Loops over boundary points for which the wave type has not
*	been defined and prompts for the wave type.  Also calls
*
*		(*bdry_state_initializer)(idir,iside,Hyper_surf(p),init,ip);
*
*	which is expected to set boundary_state(point) or
*	boundary_state_function(point) appropriately.
*/

LOCAL void prompt_for_bdry_flags1d(
	INIT_DATA	  *init,
	INIT_PHYSICS	  *ip,
	const Prompt_type *ptypes)
{
	char		mesg[256];
	int		iside, idir;
	int		w_type;
	INTERFACE	*intfc = ip->root->front->interf;
	POINT		*p,**pp;
	float 		h = ip->root->front->rect_grid->h[0];
	static const float BOUNDARY_TOL = 0.001;  /* TOLERANCE */

#define is_left_boundary(p)						\
	(is_bdry(p) &&							\
	 fabs((p)->interface->table->rect_grid.L[0] - Coords(p)[0]) 	\
	 						< BOUNDARY_TOL*h)
#define is_right_boundary(p)						\
	(is_bdry(p) &&							\
	 fabs((p)->interface->table->rect_grid.U[0] - Coords(p)[0])	\
	 						< BOUNDARY_TOL*h)

	if (ptypes == NULL) return;

	idir = 0;
	for (pp = intfc->points; pp && (*pp); pp++)
	{
	    p = *pp;
	    if (!is_bdry(p)) continue;
	    if (wave_type(p) != ERROR)
	    	continue;
	    if (is_left_boundary(p))
		iside = 0;
	    else if (is_right_boundary(p))
		iside = 1;
	    else
	    {
		iside = -1;
	    	screen("ERROR in prompt_for_bdry_flags1d(), "
	    	       "invalid boundary\n");
	    	clean_up(ERROR);
	    }
	    w_type = rect_boundary_type(intfc,idir,iside);
	    if ((w_type != UNKNOWN_BOUNDARY_TYPE) &&
		(w_type != MIXED_TYPE_BOUNDARY))
	    {
		switch (w_type)
		{
		case REFLECTION_BOUNDARY:
	            wave_type(p) = SUBDOMAIN_BOUNDARY;
		    break;
		case NO_SLIP_NEUMANN_BOUNDARY:
	            wave_type(p) = NEUMANN_BOUNDARY;
                    prompt_for_adherence_coeff(Hyper_surf(p));
                    /*
		    no_slip(Hyper_surf(p)) = YES;
		    adherence_coeff(Hyper_surf(p)) = 1.0;
                    */
		    break;
		default:
	            wave_type(p) = w_type;
		    break;
		}
	        if (w_type == DIRICHLET_BOUNDARY)
	    	    bstate_index(p) = iside;
	    }
	    else
	    {
	    	(void) sprintf(mesg, "for the boundary point (%g)",
	    		       Coords(p)[0]);
	    	wave_type(p) = prompt_for_bdry_wave_type(init,mesg,ptypes);
	    	if (wave_type(p) == UNKNOWN_BOUNDARY_TYPE)
	    	{
	    	    screen("ERROR in prompt_for_bdry_flags1d(), "
	    	           "Unknown wave type\n");
	    	    clean_up(ERROR);
	    	}
	    }
	    g_bdry_state_initializer(idir,iside,Hyper_surf(p),init,ip);
	}
#undef is_left_boundary
#undef is_right_boundary
}		/*end prompt_for_bdry_flags1d*/

#endif /* defined(ONED) */


#if defined(TWOD)

/*
*
*			prompt_for_bdry_flags2d():
*
*	Loops over boundary curve for which the wave type has not
*	been defined and prompts for the wave type.  Also calls
*	g_bdry_state_initializer() which is expected to set the boundary state
*	data for the curve appropriately.  Finally, the node types of the
*	boundary nodes are set.
*/

LOCAL void prompt_for_bdry_flags2d(
	INIT_DATA	  *init,
	INIT_PHYSICS	  *ip,
	const Prompt_type *ptypes)
{
	Front		*front = ip->root->front;
	char		mesg[256];
	int		passive, fixed, bdry_type1, bdry_type2;
	int		iside, idir;
	int		w_type;
	INTERFACE	*intfc = front->interf;
	RECT_GRID	*tgr = &topological_grid(intfc);
	CURVE		*c,**pc;
	NODE		**n;

	if (ptypes == NULL)
	    return;
		
	for ((void) next_curve(intfc,NULL); next_curve(intfc,&c); )
	{
	    if (!is_bdry(c))
		continue;
	    if (wave_type(c) != ERROR)
	    	continue;
	    (void) rect_bdry_side_for_curve(&idir,&iside,c,tgr);
	    w_type = rect_boundary_type(intfc,idir,iside);
	    if ((w_type != UNKNOWN_BOUNDARY_TYPE) &&
		(w_type != MIXED_TYPE_BOUNDARY))
	    {
		switch (w_type)
		{
		case REFLECTION_BOUNDARY:
	            wave_type(c) = SUBDOMAIN_BOUNDARY;
		    break;
		case NO_SLIP_NEUMANN_BOUNDARY:
	            wave_type(c) = NEUMANN_BOUNDARY;
                    prompt_for_adherence_coeff(Hyper_surf(c));
                    /*
		    no_slip(Hyper_surf(c)) = YES;
		    adherence_coeff(Hyper_surf(c)) = 1.0;
                    */
		    break;
		default:
	            wave_type(c) = w_type;
		    break;
		}
	        if (w_type == DIRICHLET_BOUNDARY)
	        {
	    	    bstate_index(c) = 2*idir + iside;
	        }
	    }
	    else if (is_excluded_comp(negative_component(c),intfc) &&
	    		is_excluded_comp(positive_component(c),intfc))
	    {
	        wave_type(c) = PASSIVE_BOUNDARY;
	    }
	    else
	    {
	    	(void) sprintf(mesg,
			       "for the boundary from (%g, %g) to (%g, %g)",
			       Coords(c->start->posn)[0],
			       Coords(c->start->posn)[1],
			       Coords(c->end->posn)[0],
			       Coords(c->end->posn)[1]);
		wave_type(c) = prompt_for_bdry_wave_type(init,mesg,ptypes);
		if (wave_type(c) == NO_SLIP_NEUMANN_BOUNDARY)
		{
		    wave_type(c) = NEUMANN_BOUNDARY;
                    prompt_for_adherence_coeff(Hyper_surf(c));
                    /*
		    no_slip(Hyper_surf(c)) = YES;
		    adherence_coeff(Hyper_surf(c)) = 1.0;
                    */
		}
		if (wave_type(c) == UNKNOWN_BOUNDARY_TYPE)
		{
		    screen("ERROR in prompt_for_bdry_flags2d(), "
		           "Unknown wave type\n");
		    clean_up(ERROR);
		}
	    }
	    g_bdry_state_initializer(idir,iside,Hyper_surf(c),init,ip);
	}

	/* The node_type of a boundary node is set to PASSIVE_NODE if only
	 * passive curves meet there. If there is a physical curve at the
	 * node, the node type will be set to one of DIRICHLET_NODE,
	 * NEUMANN_NODE or SUBDOMAIN_NODE depending on the wave type of the
	 * boundary at that node.  If there is no physical curve at the node,
	 * it is a FIXED_NODE.
	 */

	for (n = intfc->nodes; *n; n++)
	{
	    if (node_type(*n) != ERROR)
		continue;
	    bdry_type1 = bdry_type2 = ERROR;
	    passive = YES;
	    fixed = YES;
	    if ((*n)->in_curves != NULL)
	    {
	    	for (pc = (*n)->in_curves; *pc; pc++)
	    	{
	    	    if (wave_type(*pc) != PASSIVE_BOUNDARY)
	    	    	passive = NO;
	    	    if (wave_type(*pc) >= FIRST_PHYSICS_WAVE_TYPE)
	    	    	fixed = NO;
		    else if (is_bdry(*pc))
		    {
	    	        if (bdry_type1 == ERROR)
	    	    	    bdry_type1 = wave_type(*pc);
	    	        else if (bdry_type2 == ERROR)
	    	    	    bdry_type2 = wave_type(*pc);
		    }
	    	}
	    }
	    if ((*n)->out_curves != NULL)
	    {
	    	for (pc = (*n)->out_curves; *pc; pc++)
	    	{
	    	    if (wave_type(*pc) != PASSIVE_BOUNDARY)
	    	    	passive = NO;
	    	    if (wave_type(*pc) >= FIRST_PHYSICS_WAVE_TYPE)
	    	    	fixed = NO;
		    else if (is_bdry(*pc))
		    {
	    	        if (bdry_type1 == ERROR)
	    	    	    bdry_type1 = wave_type(*pc);
	    	        else if (bdry_type2 == ERROR)
	    	    	    bdry_type2 = wave_type(*pc);
		    }
	    	}
	    }

	    if (is_bdry(*n) && (bdry_type1 != bdry_type2))
	    	fixed = YES;
			
	    if ((bdry_type1 == SUBDOMAIN_BOUNDARY) ||
		(bdry_type2 == SUBDOMAIN_BOUNDARY))
	    	node_type(*n) = SUBDOMAIN_NODE;
	    else if (passive)
	    	node_type(*n) = PASSIVE_NODE;
	    else if (fixed)
	    	node_type(*n) = FIXED_NODE;
	    else if (bdry_type1 == DIRICHLET_BOUNDARY)
	    	node_type(*n) = DIRICHLET_NODE;
	    else if (bdry_type1 == NEUMANN_BOUNDARY)
	    	node_type(*n) = NEUMANN_NODE;
	    else if (bdry_type1 == SUBDOMAIN_BOUNDARY)
	    	node_type(*n) = SUBDOMAIN_NODE;
	}
}		/*end prompt_for_bdry_flags2d*/
#endif /* defined(TWOD) */

#if defined(THREED)

/*
*
*			prompt_for_bdry_flags3d():
*
*	Loops over boundary surfaces for which the wave type has not
*	been defined and prompts for the wave type.  Also calls
*	g_bdry_state_initializer() which is expected to set the boundary state
*	data for the surface appropriately.  Finally, the hsbdry types of the
*	hypersurface boundaries are set.
*/

LOCAL void prompt_for_bdry_flags3d(
	INIT_DATA	  *init,
	INIT_PHYSICS	  *ip,
	const Prompt_type *ptypes)
{
	Front		*front = ip->root->front;
	char		mesg[256];
	float		pbar[3];
	int		passive, fixed, bdry_type1, bdry_type2;
	int		iside, idir;
	int		w_type;
	INTERFACE	*intfc = front->interf;
	RECT_GRID	*tgr = &topological_grid(intfc);
	SURFACE		**s;
	CURVE		**c;

	if (ptypes == NULL)
	    return;
		
	for (s = intfc->surfaces; s && *s; s++)
	{
	    if (!is_bdry(*s))
		continue;
	    if (wave_type(*s) != ERROR)
	    	continue;
	    rect_bdry_side_for_hyper_surf(&idir,&iside,Hyper_surf(*s),tgr);
	    w_type = rect_boundary_type(intfc,idir,iside);
	    if ((w_type != UNKNOWN_BOUNDARY_TYPE) &&
		(w_type != MIXED_TYPE_BOUNDARY))
	    {
	        wave_type(*s) = (w_type == REFLECTION_BOUNDARY) ?
				 SUBDOMAIN_BOUNDARY : w_type;
		switch (w_type)
		{
		case REFLECTION_BOUNDARY:
	            wave_type(*s) = SUBDOMAIN_BOUNDARY;
		    break;
		case NO_SLIP_NEUMANN_BOUNDARY:
	            wave_type(*s) = NEUMANN_BOUNDARY;
		    no_slip(Hyper_surf(*s)) = YES;
                    prompt_for_adherence_coeff(Hyper_surf(*s));
                    /*
		    adherence_coeff(Hyper_surf(*s)) = 1.0;
                    */
		    break;
		default:
	            wave_type(*s) = w_type;
		    break;
		}
	        if (w_type == DIRICHLET_BOUNDARY)
	        {
	    	    bstate_index(*s) = 2*idir + iside;
	        }
	    }
	    else if (is_excluded_comp(negative_component(*s),intfc) &&
	    		is_excluded_comp(positive_component(*s),intfc))
	    {
	        wave_type(*s) = PASSIVE_BOUNDARY;
	    }
	    else
	    {
		int   i;
		float *h = front->rect_grid->h;
		average_position_of_surface(pbar,*s);
		for (i = 0; i < 3; i++)
		{
		    if (fabs(pbar[i]) < 0.001*h[i]) /*TOLERANCE*/
			pbar[i] = 0.0;
		}
	    	(void) sprintf(mesg,
			       "for the boundary with average position\n\t"
			       "(%g, %g, %g)",pbar[0],pbar[1],pbar[1]);
		wave_type(*s) = prompt_for_bdry_wave_type(init,mesg,ptypes);
		if (wave_type(*s) == NO_SLIP_NEUMANN_BOUNDARY)
		{
	            wave_type(*s) = NEUMANN_BOUNDARY;
		    no_slip(Hyper_surf(*s)) = YES;
                    prompt_for_adherence_coeff(Hyper_surf(*s));
                    /*
		    adherence_coeff(Hyper_surf(*s)) = 1.0;
                    */
		}
		if (wave_type(*s) == UNKNOWN_BOUNDARY_TYPE)
		{
		    screen("ERROR in prompt_for_bdry_flags3d(), "
		           "Unknown wave type\n");
		    clean_up(ERROR);
		}
	    }
	    g_bdry_state_initializer(idir,iside,Hyper_surf(*s),init,ip);
	}

	/* The hsbdry_type of a boundary curve is set to PASSIVE_HSBDRY if only
	 * passive surfaces meet there. If there is a physical surface at the
	 * curve, the hsbdry_type type will be set to one of DIRICHLET_HSBDRY,
	 * NEUMANN_HSBDRY or SUBDOMAIN_HSBDRY depending on the wave type of the
	 * boundary at that curve.  If there is no physical surface at the
	 * curve, it is a FIXED_HSBDRY.
	 */

	for (c = intfc->curves; c && *c; c++)
	{
	    if (hsbdry_type(*c) != ERROR)
		continue;
	    bdry_type1 = bdry_type2 = ERROR;
	    passive = YES;
	    fixed = YES;
	    for (s = (*c)->neg_surfaces; s && *s; s++)
	    {
	    	if (wave_type(*s) != PASSIVE_BOUNDARY)
	    	    passive = NO;
	    	if (wave_type(*s) >= FIRST_PHYSICS_WAVE_TYPE)
	    	    fixed = NO;
	    	else if (bdry_type1 == ERROR)
	    	    bdry_type1 = wave_type(*s);
	    	else if (bdry_type2 == ERROR)
	    	    bdry_type2 = wave_type(*s);
	    }
	    for (s = (*c)->neg_surfaces; s && *s; s++)
	    {
	    	if (wave_type(*s) != PASSIVE_BOUNDARY)
	    	    passive = NO;
	    	if (wave_type(*s) >= FIRST_PHYSICS_WAVE_TYPE)
	    	    fixed = NO;
	    	else if (bdry_type1 == ERROR)
	    	    bdry_type1 = wave_type(*s);
	    	else if (bdry_type2 == ERROR)
	    	    bdry_type2 = wave_type(*s);
	    }

	    if (is_bdry(*c) && (bdry_type1 != bdry_type2))
	    	fixed = YES;
			
	    if (passive)
	    	hsbdry_type(*c) = PASSIVE_HSBDRY;
	    else if (fixed)
	    	hsbdry_type(*c) = FIXED_HSBDRY;
	    else if (bdry_type1 == DIRICHLET_HSBDRY)
	    	hsbdry_type(*c) = DIRICHLET_HSBDRY;
	    else if (bdry_type1 == NEUMANN_HSBDRY)
	    	hsbdry_type(*c) = NEUMANN_HSBDRY;
	    else if (bdry_type1 == SUBDOMAIN_HSBDRY)
	    	hsbdry_type(*c) = SUBDOMAIN_HSBDRY;
	}
}		/*end prompt_for_bdry_flags3d*/
#endif /* defined(THREED) */

/*ARGSUSED*/
EXPORT	int g_prompt_for_bdry_wave_type(
	INIT_DATA         *init,
	const char        *mesg,
	const Prompt_type *ptypes)
{
	int		  i, n;
	char		  s[Gets_BUF_SIZE];
	const Prompt_type *ptype;

	if (ptypes == NULL)
	    return UNKNOWN_BOUNDARY_TYPE;

	if (mesg == NULL)
	    mesg = "";
	for (n = 0, ptype = ptypes; ptype->prompt != NULL; ptype++, n++);
	n--;
	screen("\nEnter the boundary type -- ");
	for (i = 0; i < n; i++)
	{
	    if (i > 0 && i%3 == 0)
	    	screen("\n                           ");
	    screen("%s, ",ptypes[i].prompt);
	}
	if (i > 0 && i%3 == 0)
	    screen("\n                           ");
	screen("or %s --\n",ptypes[n++].prompt);
	screen("\t%s: ",mesg);
	(void) Gets(s);

	for (i = 0; i < n; i++)
	{
	    if (strncasecmp(s,ptypes[i].select,ptypes[i].ncmp) == 0 ||
	    	strncasecmp(s,ptypes[i].prompt,ptypes[i].ncmp) == 0)
	    	return ptypes[i].type.itype;
	}
	return UNKNOWN_BOUNDARY_TYPE;
}		/*end g_prompt_for_bdry_wave_type*/


/*ARGSUSED*/
LOCAL	void prompt_for_contact_wall_params(
	const IO_TYPE *io_type,
	Front         *front)
{
	switch (front->rect_grid->dim)
	{
#if defined(TWOD)
	case 2:
	    prompt_for_contact_wall_node_params(io_type,front);
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    break; /* TODO */
#endif /* defined(THREED) */
	}
}		/*end prompt_for_contact_wall_params*/


#if defined(TWOD)
/*ARGSUSED*/
LOCAL	void prompt_for_contact_wall_node_params(
	const IO_TYPE *io_type,
	Front         *front)
{
	INTERFACE *intfc = front->interf;
	char	  s[Gets_BUF_SIZE];
	CWNP	  *cwnp;
	CURVE	  *c;
	int	  dim = intfc->dim;
	bool	  is_neumann;
	int	  i, j;

	is_neumann = NO;
	for (i = 0; i < dim; i++)
	{
	    for (j = 0; j < 2; j++)
	    {
	    	if (rect_boundary_type(intfc,i,j) == NEUMANN_BOUNDARY)
		    is_neumann = YES;
	    }
	}
	if (is_neumann == NO)
	{
	    (void) next_curve(intfc,NULL);
	    while (next_curve(intfc,&c))
	    {
	    	if (wave_type(c) == NEUMANN_BOUNDARY)
	    	{
	    	    is_neumann = YES;
	    	    break;
	    	}
	    }
	}
	is_neumann = pp_max_status(is_neumann);
	if (is_neumann == NO)
	    return;

	cwnp = contact_wall_node_params(intfc);

/**
	if (io_type != NULL)
	{
	    set_use_normal_D_extend(cwnp->adjust);
	    return;
	}
**/

	screen("Insert wall normal at wall contact nodes (dflt = %s): ",
		(cwnp->adjust == YES) ? "yes" : "no");
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    if (s[0] == 'y' || s[0] == 'Y')
	    	cwnp->adjust = YES;
	    else if (s[0] == 'n' || s[0] == 'N')
	    	cwnp->adjust = NO;
	}
	set_use_normal_D_extend(cwnp->adjust);
	if (cwnp->adjust == NO)
	    return;

	screen("Enter first time step to begin adjustment (dflt = %d): ",
		cwnp->first_adjust_step);
	(void) Gets(s);
	if (s[0] != '\0')
	    (void) sscanf(s,"%d",&cwnp->first_adjust_step);
	screen("Enter the first real time to begin adjustment (dflt = %g): ",
		cwnp->first_adjust_time);
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    (void) sscan_float(s,&cwnp->first_adjust_time);
	}
	screen("Enter the scaled wall bond length (dflt = %g): ",
		cwnp->wall_bond_len);
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    (void) sscan_float(s,&cwnp->wall_bond_len);
	}
}		/*end prompt_for_contact_wall_node_params*/


LOCAL	void init_bow_shock(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	Front     *front = ip->root->front;
	float     angle1, angle2, thickness, dist_from_left;
	float     rad[MAXD];
	float     surf_ten;
	float     *L, *U;
	float     center[MAXD], theta, delta_theta;
	float     coords[MAXD];
	int	  type;
	int       *gmax;
	int       i, num_points;
	NODE      *ns, *ne;
	CURVE     *cur, *ramp;
	RECT_GRID *rect_grid = front->rect_grid;
	int       dim = rect_grid->dim;
	Gas_param *params;

	if (dim != 2)
	{
	    screen("ERROR in init_bow_shock(), dim = %d != 2 not supported\n",
		   dim);
	    clean_up(ERROR);
	}
	ramp = NULL;

	screen("Enter the ramp angles (degrees),\n");
	screen("\tthickness, and distance to inlet: ");
	(void) Scanf("%f %f %f %f\n",&angle1,&angle2,&thickness,
		     &dist_from_left);
	screen("Enter height of bowshock at exit: ");
	(void) Scanf("%f\n",&rad[1]);

	angle1 = radians(angle1);	angle2 = radians(angle2);

	type = prompt_for_wave_type("",front->interf,ip);
	params = init_eos_params(init,ip,"",YES);
	prompt_for_ambient_state(comp_type(COMPA),params," ahead",front,init);
	prompt_for_ambient_state(comp_type(COMPB),params," behind",front,init);

	surf_ten = prompt_for_surface_tension(type,"of the bow wave ");

	set_obstacle_comp_type(comp_type(COMPOBST),front);

	gmax = rect_grid->gmax;
	L = rect_grid->L;
	U = rect_grid->U;
	make_ramp(NORMAL,angle1,angle2,thickness,dist_from_left,
	    	  COMPOBST,COMPB,rect_grid,&ramp);
	center[0] = U[0];
	center[1] = L[1];
	rad[0] = U[0] - Coords(ramp->end->posn)[0];
	ns = ramp->end;
	node_type(ns) = ATTACHED_B_NODE;
	coords[0] = U[0];	coords[1] = rad[1] + L[1];
	ne = make_node(Point(coords));
	node_type(ne) = DIRICHLET_NODE;
	cur = make_curve(COMPA,COMPB,ns,ne);
	wave_type(cur) = type;
	start_status(cur) = INCIDENT;
	end_status(cur) = INCIDENT;
	surface_tension(cur) = surf_ten;
	num_points = 2*max(gmax[0],gmax[1]);
	delta_theta = .5*PI / num_points;
	for (i = 1; i <= num_points; i++)
	{
	    theta = i * delta_theta;
	    coords[0] = center[0] - rad[0] * cos(theta);
	    coords[1] = center[1] - rad[1] * sin(theta);
	    if (insert_point_in_bond(Point(coords),cur->last,cur) !=
		FUNCTION_SUCCEEDED)
	    {
	        screen("ERROR in init_bow_shock(), "
		       "insert_point_in_bond() failed\n");
	        clean_up(ERROR);
	    }
	}
}		/*end init_bow_shock*/

LOCAL	void init_expanding_shock(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	Front		*front = ip->root->front;
	INTERFACE	*intfc = front->interf;
	float		*L,*U;
	float		x;
	float		angle1, angle2;
	float		thickness, dist_from_left;
	float		M;
	float		rho, pr;
	float		coords0[MAXD], coords1[MAXD];
	float		s_n[MAXD];
	char		s[Gets_BUF_SIZE];
	POINT		*p, Pcross;
	BOND		*b1, *b2;
	NODE		*ns, *ne;
	CURVE		*ramp, *incident;
	CURVE		**split_curves;
	RECT_GRID	*rect_grid = front->rect_grid;
	Gas_param	*params;
	bool		sav_scss = interpolate_states_at_split_curve_node();
	int		constant_comp;

	ramp = NULL;
	L = rect_grid->L;
	U = rect_grid->U;
	s_n[0] = -1.;	s_n[1] = 0;

	screen("Enter the ramp angles (degrees),\n");
	screen("\tthickness, and distance to outlet: ");
	(void) Scanf("%f %f %f %f\n",&angle1,&angle2,&thickness,
		     &dist_from_left);
	screen("Enter the distance of the shock from the inlet: ");
	(void) Scanf("%f\n",&x);

	x = U[0] - x;
	angle1 = radians(angle1);	angle2 = radians(angle2);

	params = init_eos_params(init,ip,"",YES);
	screen("Enter the ahead state -- rho,pr: ");
	(void) Scanf("%f %f\n",&rho,&pr);
	screen("Enter the incident Mach number: ");
	(void) Scanf("%f\n",&M);

	set_ambient_comp_type(comp_type(COMPA),front);
	screen("Is the flow ahead of the wave constant (dflt = no): ");
	(void) Gets(s);
	constant_comp = (s[0] == 'y' || s[0] == 'Y') ? YES : NO;
	set_ambient_comp_type(comp_type(COMPB),front);
	set_obstacle_comp_type(comp_type(COMPOBST),front);
	init_shock_states(rho,pr,M,s_n,params,Ambient(comp_type(COMPA)),
			  Ambient(comp_type(COMPB)));
	if (constant_comp == YES)
	    (void)SetConstantFlowRegion(COMPA,Ambient(comp_type(COMPA)),intfc);
	make_ramp(NORMAL,angle1,angle2,thickness,dist_from_left,COMPOBST,COMPB,
		  rect_grid,&ramp);

	coords0[0] = x;	coords0[1] = L[1];
	coords1[0] = x;	coords1[1] = U[1];
	b1 = Bond(Point(coords0),Point(coords1));
	b2 = ramp->first;
	if (!cross_bonds(b1,b2,&Pcross))
	{
	    screen("ERROR in init_expanding_shock(), "
		   "the incident shock is not on ramp\n");
	    clean_up(ERROR);
	}
	p = Point(Coords(&Pcross));
	set_interpolate_states_at_split_curve_node(NO);
	split_curves = split_curve(p,ramp->first,ramp,COMPOBST,COMPB,COMPOBST,
				   COMPA);
	set_interpolate_states_at_split_curve_node(sav_scss);
	wave_type(split_curves[0]) = NEUMANN_BOUNDARY;
	start_status(split_curves[0]) = FIXED;
	end_status(split_curves[0]) = FIXED;
	wave_type(split_curves[1]) = NEUMANN_BOUNDARY;
	start_status(split_curves[1]) = FIXED;
	end_status(split_curves[1]) = FIXED;
	ns = split_curves[0]->end;
	node_type(ns) = NEUMANN_NODE;
	ne = make_node(Point(coords1));
	node_type(ne) = NEUMANN_NODE;
	incident = make_curve(COMPA,COMPB,ns,ne);
	wave_type(incident) = BACKWARD_SHOCK_WAVE;
	start_status(incident) = INCIDENT;
	end_status(incident) = INCIDENT;

}		/*end init_expanding_shock*/
#endif /* defined(TWOD) */

/*
*			set_passive_curve_flags():
*
*	The wave_type of a boundary curve corresponds to the
*	boundary data type unless it borders on an OBSTACLE
*	region, in which case it is PASSIVE_BOUNDARY.  Here we
*	set the flags for the passive curves.
*/

LOCAL void set_passive_curve_flags(
	Front		*front)
{
	INTERFACE	*intfc = front->interf;

	switch (intfc->dim)
	{
#if defined(TWOD)
	case 2:
	{
	    CURVE *c;
	    for ((void) next_curve(intfc,NULL); next_curve(intfc,&c); )
	    {
	    	if (!is_bdry(c))
	    	    continue;
	    	if (comp_type(positive_component(c))->type != OBSTACLE)
	    	    continue;
	    	wave_type(c) = PASSIVE_BOUNDARY;
	    	start_status(c) = PASSIVE;
	    	end_status(c) = PASSIVE;
	    	continue;
	    }
	}
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    /* TODO: 3d */
	    break;
#endif /* defined(THREED) */
	}
}		/*end set_passive_curve_flags*/


LOCAL void g_bdry_state_initializer(
	int		idir,
	int		iside,
	HYPER_SURF	*hs,
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	INTERFACE	*intfc = ip->root->front->interf;
	RECT_GRID	*gr = computational_grid(intfc);
	COMPONENT	comp = NO_COMP;
	float		coords[MAXD];
	int		i, dim = gr->dim;
	float		*L = gr->L, *U = gr->U;

	if (hs == NULL)
	{
	    if (rect_boundary_type(intfc,idir,iside) == DIRICHLET_BOUNDARY)
	    {
	    	for (i = 0; i < dim; i++)
	    	    coords[i] = 0.5*(L[i] + U[i]);
	    	coords[idir] = (iside == 0) ? L[idir] : U[idir];
		comp = nearest_interior_comp(YES,FIRST_DYNAMIC_COMPONENT,
	    				         coords,intfc);
	    }
	    (void) prompt_for_boundary_state(
	    		rect_boundary_type(intfc,idir,iside),NULL,
	    		coords,comp,2*idir+iside,hs,init,ip);
	}
	else
	{
	    switch (dim)
	    {
#if defined(ONED)
	    case 1:
	    	g_1d_bdry_state_initializer(idir,iside,Point_of_hs(hs),init,ip);
		break;
#endif /* defined(ONED) */
#if defined(TWOD)
	    case 2:
	    	g_2d_bdry_state_initializer(idir,iside,Curve_of_hs(hs),init,ip);
		break;
#endif /* defined(TWOD) */
#if defined(THREED)
	    case 3:
	    	g_3d_bdry_state_initializer(idir,iside,Surface_of_hs(hs),
					    init,ip);
	    	break;
#endif /* defined(THREED) */
	    }
	}
}		/*end g_bdry_state_initializer*/


EXPORT	int g_prompt_for_boundary_state(
	int		w_type,
	const char	*name,
	float		*coords,
	COMPONENT	comp,
	int		index,
	HYPER_SURF	*hs,
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	INTERFACE	*intfc = ip->root->front->interf;
	BOUNDARY_STATE	Bstate;
	static	Locstate tmpst = NULL;
	char		s[Gets_BUF_SIZE];
	int		i, dim = intfc->dim;
	size_t		sizest = size_of_state(intfc);
	COMP_TYPE	*ctype;

	if (tmpst == NULL)
	    alloc_state(intfc,&tmpst,sizest);

	switch (w_type) 
	{
	case NEUMANN_BOUNDARY:
	case NO_SLIP_NEUMANN_BOUNDARY:
	case PASSIVE_BOUNDARY:
	case REFLECTION_BOUNDARY:
	case SUBDOMAIN_BOUNDARY:
	case MIXED_TYPE_BOUNDARY:
	    return -1;
	case DIRICHLET_BOUNDARY:
	    screen("Specify the %s boundary state"
	           "\n\ttime-independent boundary state set by "
	           "ambient state (A, default),"
	           "\n\tflow-through boundary conditions (FT),"
	           "\n\tflow-through boundary with constant pressure (FP), or"
	           "\n\tflow-through boundary with time dependent pressure (FD), or"
	           "\n\trandom velocity inlet (R),"
	           "\n\tpreset boundary state (P), or"
	           "\n\ttime-independent state specified by the user (U)."
	           "\nEnter choice here: ",(name == NULL) ? "Dirichlet" : name);
	    (void) Gets(s);
	    if (s[0] == '\0')
		(void) strcpy(s,"A");
	    if (strncasecmp(s,"FT",2) == 0)
	    {
		Bstate._boundary_state = NULL;
		Bstate._boundary_state_function = flow_through_boundary_state;
		Bstate._boundary_state_function_name =
				strdup("flow_through_boundary_state");
		Bstate._boundary_state_data = NULL;
		Bstate._fprint_boundary_state_data = 
				f_fprint_boundary_state_data;
		index = add_bstate_to_list(&Bstate,intfc,index);
	    }
	    else if (strncasecmp(s,"FP",2) == 0)
	    {
		Bstate._boundary_state = tmpst;
		Bstate._boundary_state_function =
		    constant_pressure_flow_through_boundary_state;
		Bstate._boundary_state_function_name =
			strdup("constant_pressure_flow_through_boundary_state");
		Bstate._boundary_state_data = NULL;
		Bstate._fprint_boundary_state_data = 
				f_fprint_boundary_state_data;
		set_ambient_boundary_state(tmpst,coords,comp,intfc,init);
		set_state(tmpst,TGAS_STATE,tmpst);
		screen("Enter rho, pr: ");
		(void) Scanf("%f %f\n",&Dens(tmpst),&Press(tmpst));
		Dens(tmpst)=density(tmpst);
		screen("Density = %"FFMT" for pressure = %"FFMT"\n"
		       ,Dens(tmpst),Press(tmpst));

		set_state(tmpst,GAS_STATE,tmpst);
		index = add_bstate_to_list(&Bstate,intfc,index);
	    }
	    else if (strncasecmp(s,"FD",2) == 0) /*reformat*/
	    {
	        float tw,tp,tc,prb,prp;
		Bstate._boundary_state_function = 
		    time_dep_pressure_flow_through_boundary_state;
		Bstate._boundary_state_function_name =
                    strdup("time_dep_pressure_flow_through_boundary_state");
		Bstate._boundary_state_data = NULL;
		Bstate._fprint_boundary_state_data = 
				g_fprint_tdp_boundary_state_data;
		set_ambient_boundary_state(tmpst,coords,comp,intfc,init);
		set_type_of_state(tmpst,TGAS_STATE);
		screen("Enter warming-up time, peak time, cooling-down time,"
		       "\n\tbase pressure, peak pressure: ");
		(void) Scanf("%f %f %f %f %f",&tw,&tp,&tc,&prb,&prp);
		(void) getc(stdin);/*get trailing newline*/
		screen("\n");
		(void) set_time_dep_pres(&Bstate,tw,tp,tc,prb,prp);
		Press(tmpst) = prb;
		Dens(tmpst) = density(tmpst);
		screen("Density = %"FFMT" for pressure = %"FFMT"\n"
		       ,Dens(tmpst),Press(tmpst));
		Press(tmpst) = prp;
		Dens(tmpst) = density(tmpst);
		screen("Density = %"FFMT" for pressure = %"FFMT"\n"
		       ,Dens(tmpst),Press(tmpst));
		Press(tmpst) = prb;
		Dens(tmpst) = density(tmpst);

		set_state(tmpst,GAS_STATE,tmpst);
		Bstate._boundary_state = tmpst;
		index = add_bstate_to_list(&Bstate,intfc,index);
	    }
	    else if (strncasecmp(s,"U",1) == 0)
	    {
	    	(void) strcpy(s,"n");
	    	ctype = comp_type(comp);
	    	if (ctype->type == AMBIENT) 
	    	{
	    	    screen("Use adjacent ambient state params (y,n(deflt)?: ");
	    	    (void) Gets(s);
	    	}
	    	if (s[0] == 'y' || s[0] == 'Y')
		    Set_params(tmpst,Ambient(ctype));
		else
		{
		    Init_params(tmpst,init_eos_params(init,ip,"",YES));
#if defined(COMBUSTION_CODE)
		    prompt_for_burning(&Params(tmpst),"");
#endif /* defined(COMBUSTION_CODE) */
		}

		set_type_of_state(tmpst,TGAS_STATE);

		screen("Enter rho, pr, ");
#if defined(COMBUSTION_CODE)
		if (Composition_type(tmpst) == ZND)
		{
		    screen("react, ");
		}
#endif /* defined(COMBUSTION_CODE) */
		for (i = 0; i < dim-1; i++) screen("v[%d], ",i);
		    screen("v[%d]: ",i);

		(void) Scanf("%f %f",&Dens(tmpst),&Press(tmpst));
#if defined(COMBUSTION_CODE)
		if (Composition_type(tmpst) == ZND)
		{
		    (void) Scanf("%f",&React(tmpst));
		}
#endif /* defined(COMBUSTION_CODE) */
		for (i = 0; i < dim; i++)
		    (void) Scanf("%f",&Vel(tmpst)[i]);
		(void) getc(stdin);/*get trailing newline*/
		screen("\n");
		Dens(tmpst)=density(tmpst);
		screen("Density = %"FFMT" for pressure = %"FFMT"\n"
		       ,Dens(tmpst),Press(tmpst));

		set_state(tmpst,GAS_STATE,tmpst);
		Bstate._boundary_state = tmpst;
		Bstate._boundary_state_function = g_fixed_boundary_state;
		Bstate._boundary_state_function_name = 
		    strdup("g_fixed_boundary_state");
		Bstate._boundary_state_data = NULL;
		Bstate._fprint_boundary_state_data = 
				f_fprint_boundary_state_data;
		index = add_bstate_to_list(&Bstate,intfc,index);
		break;
	    }
	    else if (strncasecmp(s,"R",1) == 0)
	    {
	    	index = prompt_for_random_flow_inlet(coords,comp,hs,
						     intfc,index,init);
	    }
	    else if (strncasecmp(s,"A",1) == 0)
	    {
	        set_ambient_boundary_state(tmpst,coords,comp,intfc,init);
	    	Bstate._boundary_state_function = g_fixed_boundary_state;
		Bstate._boundary_state_function_name =
		    strdup("g_fixed_boundary_state");
		Bstate._boundary_state = tmpst;
		Bstate._boundary_state_data = NULL;
		Bstate._fprint_boundary_state_data = 
				f_fprint_boundary_state_data;
		index = add_bstate_to_list(&Bstate,intfc,index);
	    }
	    return index;
	case UNKNOWN_BOUNDARY_TYPE:
	    return index;
	default:
	    screen("ERROR in g_prompt_for_boundary_state(), "
	           "non-bdry wave type - %d\n",w_type);
	    clean_up(ERROR);
	    break;
	}
	return index;
}		/*end g_prompt_for_boundary_state*/

EXPORT	void	set_ambient_boundary_state(
	Locstate	bdry_state,
	float		*coords,
	COMPONENT	comp,
	INTERFACE	*intfc,
	INIT_DATA	*init)
{
	COMP_TYPE	*ctype;

	ctype = comp_type(comp);

	Get_state(coords,bdry_state,ctype,NULL,intfc,init,GAS_STATE);
}		/*end set_ambient_boundary_state*/


#if defined(ONED)  
LOCAL void g_1d_bdry_state_initializer(
	int		idir,
	int		iside,
	POINT		*p,
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	INTERFACE *intfc = ip->root->front->interf;
	float	  coords[MAXD];
	int       w_type;

	w_type = rect_boundary_type(intfc,idir,iside);
	if ((w_type != UNKNOWN_BOUNDARY_TYPE) &&
	    (w_type != MIXED_TYPE_BOUNDARY))
	    return;
	coords[0] = Coords(p)[0];
	bstate_index(p) = prompt_for_boundary_state(wave_type(p),NULL,coords,
						    positive_component(p),-1,
						    Hyper_surf(p),init,ip);
}		/*end g_1d_bdry_state_initializer*/
#endif /* defined(ONED) */


#if defined(TWOD)
LOCAL void g_2d_bdry_state_initializer(
	int		idir,
	int		iside,
	CURVE		*c,
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	INTERFACE	*intfc = ip->root->front->interf;
	float		coords[MAXD];
	int		i, dim = intfc->dim;
	int       w_type;

	start_status(c) = FIXED;	end_status(c) = FIXED;
	w_type = rect_boundary_type(intfc,idir,iside);
	if ((w_type!=UNKNOWN_BOUNDARY_TYPE) && (w_type!=MIXED_TYPE_BOUNDARY))
	    return;
	for (i = 0; i < dim; i++)
	{
	    coords[i] = 0.5*(Coords(c->start->posn)[i]+Coords(c->end->posn)[i]);
	}
	bstate_index(c) = prompt_for_boundary_state(wave_type(c),NULL,coords,
						    positive_component(c),-1,
						    Hyper_surf(c),init,ip);
}		/*end g_2d_bdry_state_initializer*/
#endif /* defined(TWOD) */

#if defined(THREED)
LOCAL void g_3d_bdry_state_initializer(
	int		idir,
	int		iside,
	SURFACE		*s,
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	Front     *front = ip->root->front;
	INTERFACE *intfc = front->interf;
	float	  coords[3];
	int       w_type;
	int       i;
	float     *h = front->rect_grid->h;

	w_type = rect_boundary_type(intfc,idir,iside);
	if ((w_type != UNKNOWN_BOUNDARY_TYPE) &&
	    (w_type != MIXED_TYPE_BOUNDARY))
	    return;
	average_position_of_surface(coords,s);
	for (i = 0; i < 3; i++)
	{
	    if (fabs(coords[i]) < 0.001*h[i]) /*TOLERANCE*/
	        coords[i] = 0.0;
	}
	bstate_index(s) = prompt_for_boundary_state(wave_type(s),NULL,coords,
						    positive_component(s),-1,
						    Hyper_surf(s),init,ip);
}		/*end g_3d_bdry_state_initializer*/
#endif /* defined(THREED) */

/*
*			g_init_cauchy_data_pointers():
*
*	Initializes function-pointer array handling the cauchy data.
*
*/

EXPORT void g_init_cauchy_data_pointers(
	INIT_PHYSICS	*ip,
	bool		got_intfc_from_file)
{
	debug_print("init","Entered init_cauchy_data()\n");

	if (got_intfc_from_file == YES)
	{
	    ip->restart_initializer = g_restart_initializer;
	    ip->restart_intfc_initializer = g_restart_intfc_initializer;
	    ip->restart_pt_source_initializer = NULL;
	    ip->initializer = NULL;
	    ip->intfc_initializer = NULL;
	    ip->pt_source_initializer = NULL;
	}
	else
	{
	    ip->initializer = g_initializer;
	    ip->FM_initializer = g_FM_initializer;
	    ip->Dual_cell_initializer = g_Dual_cell_initializer;
	    ip->intfc_initializer = g_intfc_initializer;
	    ip->pt_source_initializer = NULL;
	    ip->restart_initializer = NULL;
	    ip->restart_intfc_initializer = NULL;
	    ip->pt_source_initializer = NULL;
	}
	debug_print("init","Left init_cauchy_data()\n");
}		/*end g_init_cauchy_data_pointers*/


/* ARGSUSED */
LOCAL void g_initializer(
	float		*coords,
	COMPONENT	comp,
	Locstate	state,
	INTERFACE	*intfc,
	INIT_DATA	*init)
{
	int i, dim = intfc->dim;
	int stype = type_of_state(init);

	debug_print("init_states","Entered g_initializer()\n");
	if (debugging("init_states"))
	{
  	    (void) printf("comp = %d, ",comp);
	    print_component_type("type = ",comp_type(comp),", ");
  	    for (i = 0; i < dim; i++)
  	    	(void) printf(" coords[%d] = %g ",i,coords[i]);
	    (void) printf("state - ");
	}

	Get_state(coords,state,comp_type(comp),NULL,intfc,init,stype);

        // FOR 2d DG
        dg_Dens(state)[0] = Dens(state);
        dg_Energy(state)[0] = Energy(state);
        dg_Mom(state)[0][0] = Mom(state)[0];
        dg_Mom(state)[1][0] = Mom(state)[1];
        for(i = 1; i < MAX_N_COEF; i++)
        {
            // printf("MAX_N_COEF = %d\n", MAX_N_COEF);
            dg_Dens(state)[i] = 0.0;
            dg_Energy(state)[i] = 0.0;
            dg_Mom(state)[0][i] = 0.0;
            dg_Mom(state)[1][i] = 0.0;
        }

	if (debugging("init_states"))
	{
	    print_gas_state(state);
	}
        // TMP
        // printf("coords[%g %g] dens = %g\n", coords[0], coords[1], Dens(state));

        if(debugging("Shu_Osher"))
        {
            init_Shu_Osher_problem(coords,comp,state,intfc,init,NULL);
        }

        if(debugging("shock_vortex"))
        {
            add_vortex_initializer(coords,comp,state,intfc,init,NULL); 
        }

        if(debugging("v_evo"))
        {
            vortex_evolution_initializer(coords,comp,state,intfc,init,NULL); 
        }

        if(debugging("db_Mach"))
        {
            /**
            printf("coords[%g %g] state:, cos(60) = %g, sin(60) = %g\n", 
                  coords[0], coords[1], cos(radians(60.0)),
                  sin(radians(60.0)));
            printf("den, u, v, press (%g %g %g %g)\n", Dens(state),
                    vel(0, state),   vel(1, state), pressure(state));
            **/
            init_db_Mach_state(coords,comp,state,intfc,init);

            // printf("after reset: den, u, v, press (%g %g %g %g)\n", Dens(state),
            //         vel(0, state),   vel(1, state), pressure(state));

            dg_Dens(state)[0] = Dens(state);
            dg_Energy(state)[0] = Energy(state);
            dg_Mom(state)[0][0] = Mom(state)[0];
            dg_Mom(state)[1][0] = Mom(state)[1];
            for(i = 1; i < MAX_N_COEF; i++)
            {
                // printf("MAX_N_COEF = %d\n", MAX_N_COEF);
                dg_Dens(state)[i] = 0.0;
                dg_Energy(state)[i] = 0.0;
                dg_Mom(state)[0][i] = 0.0;
                dg_Mom(state)[1][i] = 0.0;
            }
        } 

	debug_print("init_states","Left g_initializer()\n");
}		/*end g_initializer*/

LOCAL void g_Dual_cell_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        POLYGON         *poly)
{
        if(debugging("mag_acc_vert"))
        {
            /// accurate test 
            dual_cell_accurate_magnetic_vortex_initializer(coords,comp,state,intfc,init,poly);
            return;
        }

        if(debugging("decay_alfven"))
        {
            dual_cell_delay_alfven_initializer(coords,comp,state,intfc,init,poly);
            // printf("WARNING: implemented decay_alfven in g_Dual_cell_initializer(), need test\n");
            // clean_up(ERROR); 
            return;
        }

        if(debugging("rotor"))
        {
            dual_cell_magnetic_rotor_initializer(coords,comp,state,intfc,init,poly);
            return;
        }

        if(debugging("Orszag_T"))
        {
            dual_cell_magnetic_Orszag_T_initializer(coords,comp,state,intfc,init,poly);
            return;
        }

        if(debugging("blast_MHD"))
        {
            dual_cell_blast_MHD_initializer(coords,comp,state,intfc,init,poly);
            return;
        }
        if(debugging("field_loop"))
        {
            // dual_cell_field_loop_advection_initializer(coords,comp,state,intfc,init,poly);
            return;
        }

        printf("ERROR: problem ??? need to implement init in  g_Dual_cell_initializer\n");
        clean_up(ERROR);
}

LOCAL void dual_cell_blast_MHD_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        POLYGON         *poly)
{
        RECT_GRID       *gr = computational_grid(intfc);
        double           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        double           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        double           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        double          *pcrds[MAX_N_POLY_SIDE], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[10][MAX_N_COEF], mulrhs[10][MAX_N_COEF], dens[10], tmp;
        double           t = 0.0, sqrt_area;
        int             debug_flag = NO, i, j, k, indx, side;
        double a = 0.065130102902216, b = 0.869739794195568;
        double c = 0.312865496004875, d = 0.638444188569809;
        double e = 0.048690315425316, f = 0.260345966079038;
        double g = 0.479308067841923, third;
        double w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[30][8];
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        int           Lobatto_N = 7;
        double        q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987;
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905;
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905;
        }
        else
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: dual_cell_blast_MHD_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }

        third = 1.0/3.0;

        for(side = 0; side < poly->n_sides; side++)
        {
            for(indx = 0; indx < N_COEF_EDGE; indx++)
                polyg_side_dgB(poly)[side][indx] = 0.0;
            pcrds[side] = Coords(Point_of_polyg(poly)[side]);
        }

        for(side = 0; side < poly->n_sides; side++)
        {
            // TMP
            if(side == 3 && poly->id == 381)
                debug_flag = YES;
            else if(side == 0 && poly->id == 407)
                debug_flag = YES;
            else
                debug_flag = NO;
            if(YES == debug_flag)
                printf("\n\ndual cell %d in dual_cell_blast_MHD_initializer\n\n", poly->id);
            /// END::: TMP
            
            for(i = 0; i < 2; i++)
                v2[i] = polyg_side_vector(poly)[side][i];
            nor[0] = v2[1];
            nor[1] = -v2[0];

#if defined(__MPI__)
            if(YES == debug_flag)
                printf("side = %d, nor[%g, %g]\n\n", side, nor[0], nor[1]);
#endif

            for(k = 0; k < Lobatto_N; k++)//use Lobatto points(include vertices)
            {
                for(i = 0; i < 2; i++)
                    qcrds[i] = (pcrds[(side+1)%poly->n_sides][i] + pcrds[side][i])/2.0 +
                           (pcrds[(side+1)%poly->n_sides][i] - pcrds[side][i])/2.0*q[k];

                for(j = 5; j < N_EQN; j++)
                    v_conU[side][j] = ff_accurate_magnetic_blast(NULL, qcrds,j,YES);

                Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];

                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    //printf("q[%d]=%Lf vh_val_ver2=%g\n",indx,q[k],vh_val_ver2(q[k],indx));
                    polyg_side_dgB(poly)[side][indx] += qw[k]*Bn[k]*vh_val_ver2(q[k],indx);//rhs

                    // TMP
#if defined(__MPI__)
                    if(YES == debug_flag && indx == 1)
                    {
                        double temp_a;
                        temp_a = (qw[k])*Bn[k]*vh_val_ver2((q[k]),indx);
                        printf("Lobatto[%d](%g,%g), Bn=%16.14g, add %16.14g, qw = %g, vh = %g\n",
                                  k, qcrds[0], qcrds[1], 1.0*Bn[k], 1.0*temp_a, 
                                  qw[k], vh_val_ver2((q[k]),indx));
                    }
#endif
                    // END::TMP
                }
            }

            // TMP
            if(YES == debug_flag )
            {
                printf("dg_Bn %15.14g, %15.14g of side %d\n\n", polyg_side_dgB(poly)[side][0],
                      polyg_side_dgB(poly)[side][1], side);
            }
            // END::: TMP
            
            for(indx = 0; indx < N_COEF_EDGE; indx++)
            {
                polyg_side_dgB(poly)[side][indx] = polyg_side_dgB(poly)[side][indx]*(2.0*indx+1.0)/2.0;
            }
        }

        tmp = 0.0;
        for(side = 0; side < poly->n_sides; side++)
            tmp += polyg_side_dgB(poly)[side][0]*polyg_length_side(poly)[side];

        if(debug_flag == YES || fabs(tmp) > 1.0e-10)
        {
            printf("polygon in dual_cell_blast_MHD_initializer()\n");
            printf("divB = %14.13g\n", tmp);
        }
}



LOCAL void dual_cell_magnetic_rotor_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        POLYGON         *poly)
{           
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[MAX_N_POLY_SIDE], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[10][MAX_N_COEF], mulrhs[10][MAX_N_COEF], dens[10], tmp;
        float           t = 0.0, sqrt_area;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[30][8];
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        int           Lobatto_N = 7;
        double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987;
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905;
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905;
        }
        else
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: dual_cell_magnetic_rotor_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }

        third = 1.0/3.0;

        for(side = 0; side < poly->n_sides; side++)
        {
            for(indx = 0; indx < N_COEF_EDGE; indx++)
                polyg_side_dgB(poly)[side][indx] = 0.0;
            pcrds[side] = Coords(Point_of_polyg(poly)[side]);
        }

        for(side = 0; side < poly->n_sides; side++)
        {
            for(i = 0; i < 2; i++)
                v2[i] = polyg_side_vector(poly)[side][i];
            nor[0] = v2[1];
            nor[1] = -v2[0];

            for(k = 0; k < Lobatto_N; k++)//use Lobatto points(include vertices)
            {
                for(i = 0; i < 2; i++)
                    qcrds[i] = (pcrds[(side+1)%poly->n_sides][i] + pcrds[side][i])/2.0 +
                           (pcrds[(side+1)%poly->n_sides][i] - pcrds[side][i])/2.0*q[k];

                for(j = 5; j < N_EQN; j++)
                    v_conU[side][j] = ff_accurate_magnetic_rotor(NULL, qcrds,j,YES);

                Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];

                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    //printf("q[%d]=%Lf vh_val_ver2=%g\n",indx,q[k],vh_val_ver2(q[k],indx));
                    polyg_side_dgB(poly)[side][indx] += qw[k]*Bn[k]*vh_val_ver2(q[k],indx);//rhs
                }
            }

            for(indx = 0; indx < N_COEF_EDGE; indx++)
            {
                polyg_side_dgB(poly)[side][indx] = polyg_side_dgB(poly)[side][indx]*(2.0*indx+1.0)/2.0;
            }
        }

        tmp = 0.0;
        for(side = 0; side < poly->n_sides; side++)
            tmp += polyg_side_dgB(poly)[side][0]*polyg_length_side(poly)[side];

        if(debug_flag == YES || fabs(tmp) > 1.0e-10)
        {
            printf("polygon in dual_cell_magnetic_rotor_initializer()\n");
            printf("divB = %14.13g\n", tmp);
        }
}


LOCAL void dual_cell_magnetic_Orszag_T_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        POLYGON         *poly)
{           
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[MAX_N_POLY_SIDE], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[10][MAX_N_COEF], mulrhs[10][MAX_N_COEF], dens[10], tmp;
        float           t = 0.0, sqrt_area;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[30][8];
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        int           Lobatto_N = 7;
        double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987;
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905;
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905;
        }
        else
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: dual_cell_magnetic_Orszag_T_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }

        third = 1.0/3.0;

        for(side = 0; side < poly->n_sides; side++)
        {
            for(indx = 0; indx < N_COEF_EDGE; indx++)
                polyg_side_dgB(poly)[side][indx] = 0.0;
            pcrds[side] = Coords(Point_of_polyg(poly)[side]);
        }

        for(side = 0; side < poly->n_sides; side++)
        {
            for(i = 0; i < 2; i++)
                v2[i] = polyg_side_vector(poly)[side][i];
            nor[0] = v2[1];
            nor[1] = -v2[0];

            for(k = 0; k < Lobatto_N; k++)//use Lobatto points(include vertices)
            {
                for(i = 0; i < 2; i++)
                    qcrds[i] = (pcrds[(side+1)%poly->n_sides][i] + pcrds[side][i])/2.0 +
                           (pcrds[(side+1)%poly->n_sides][i] - pcrds[side][i])/2.0*q[k];

                for(j = 5; j < N_EQN; j++)
                    v_conU[side][j] = ff_accurate_Orszag_T(NULL, qcrds,j,YES);

                Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];

                for(indx = 0; indx < N_COEF_EDGE; indx++)
                    polyg_side_dgB(poly)[side][indx] += qw[k]*Bn[k]*vh_val_ver2(q[k],indx);//rhs
            }

            for(indx = 0; indx < N_COEF_EDGE; indx++)
                polyg_side_dgB(poly)[side][indx] = polyg_side_dgB(poly)[side][indx]*(2.0*indx+1.0)/2.0;
        }

        tmp = 0.0;
        for(side = 0; side < poly->n_sides; side++)
            tmp += polyg_side_dgB(poly)[side][0]*polyg_length_side(poly)[side];

        if(debug_flag == YES || fabs(tmp) > 1.0e-11)
        {
            printf("ERROR: polygon in dual_cell_magnetic_Orszag_T_initializer()\n");
            printf("divB = %14.13g\n", tmp);
        }
}


LOCAL void dual_cell_accurate_magnetic_vortex_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        POLYGON         *poly)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[MAX_N_POLY_SIDE], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[10][MAX_N_COEF], mulrhs[10][MAX_N_COEF], dens[10], tmp;
        float           t = 0.0, sqrt_area;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[30][8];
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        int           Lobatto_N = 7;
        double        q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        // Lobatto quadrature
        if(Lobatto_N == 4)
        {
            q[0] = -1.0; q[1] = -sqrt(5.0)/5.0;
            q[2] =  sqrt(5.0)/5.0; q[3] = 1.0;
            qw[0] =  1.0/6.0; qw[1] = 5.0/6.0;
            qw[2] =  5.0/6.0; qw[3] = 1.0/6.0;

            // Gaussian
            // q[0] = -0.86113631159405257522; q[1] = -0.33998104358485626480;
            // q[2] =  0.33998104358485626480; q[3] = 0.86113631159405257522;
            // qw[0] =  0.34785484513745385737; qw[1] = 0.65214515486254614263;
            // qw[2] =  0.65214515486254614263; qw[3] = 0.34785484513745385737;
        }
        else if(Lobatto_N == 5)
        {
            q[0] = -1.0; q[1] = -sqrt(21.0)/7.0;
            q[2] = 0.0;  q[3] =  sqrt(21.0)/7.0; q[4] = 1.0;
            qw[0] =  0.1; qw[1] = 49.0/90.0; qw[2] = 32.0/45.0;
            qw[3] =  49.0/90.0; qw[4] = 0.1;
        }
        else if(Lobatto_N == 6)
        {
            q[0] = -1.0; q[1] = -sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));
            q[2] = -sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));  q[3] =  sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));
            q[4] =  sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));     q[5] = 1.0;
            qw[0] =  1.0/15.0; qw[1] = 1.0/30.0*(14.0-sqrt(7.0));
            qw[2] = 1.0/30.0*(14.0+sqrt(7.0)); qw[3] =  1.0/30.0*(14.0+sqrt(7.0));
            qw[4] = 1.0/30.0*(14.0-sqrt(7.0));       qw[5] = 1.0/15.0;
        }
        else if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987;
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905;
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905;
        }
        else
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: dual_cell_accurate_magnetic_vortex_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }

        third = 1.0/3.0;

        for(side = 0; side < poly->n_sides; side++)
        {
            for(indx = 0; indx < N_COEF_EDGE; indx++)
                polyg_side_dgB(poly)[side][indx] = 0.0;
            pcrds[side] = Coords(Point_of_polyg(poly)[side]);
        }

        for(side = 0; side < poly->n_sides; side++)
        {
#if !defined(__MPI__)
            if(poly->id == 250 && side ==0)
                debug_flag = YES;    
            else if(poly->id == 220 && side == 3)
                debug_flag = YES;
            else 
                debug_flag = NO;
            if(YES == debug_flag)
                printf("\n\ndual cell %d dual_cell_accurate_magnetic_vortex_initializer()\n", poly->id);
#endif 
            for(i = 0; i < 2; i++)
                v2[i] = polyg_side_vector(poly)[side][i];
            nor[0] = v2[1];
            nor[1] = -v2[0];

#if !defined(__MPI__)
            if(YES == debug_flag)
                printf("side = %d, nor[%g, %g]\n\n", side, nor[0], nor[1]);
#endif 

            for(k = 0; k < Lobatto_N; k++)//use Lobatto points(include vertices)
            {
                for(i = 0; i < 2; i++)
                    qcrds[i] = (pcrds[(side+1)%poly->n_sides][i] + pcrds[side][i])/2.0 +
                           (pcrds[(side+1)%poly->n_sides][i] - pcrds[side][i])/2.0*q[k];

                for(j = 5; j < N_EQN; j++)
                    v_conU[side][j] = ff_accurate_magnetic_vort(NULL, qcrds,j,YES);

                Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];

                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    //printf("q[%d]=%Lf vh_val_ver2=%g\n",indx,q[k],vh_val_ver2(q[k],indx));
                    polyg_side_dgB(poly)[side][indx] += qw[k]*Bn[k]*vh_val_ver2(q[k],indx);//rhs

#if !defined(__MPI__)
                    if(YES == debug_flag && indx == 1)
                    {
                        double temp_a; 
                        temp_a = qw[k]*Bn[k]*vh_val_ver2(q[k],indx);
                        printf("Lobatto[%d] crds(%g, %g), Bn=%15.14g, B(%g, %g)\n",
                              k, qcrds[0], qcrds[1], Bn[k], v_conU[side][5], v_conU[side][6]);
                        printf("[%d] add %16.14g, qw = %g, vh = %g\n",
                              k, temp_a, qw[k], vh_val_ver2(q[k],indx));
                    }
#endif 
                }
            }

            for(indx = 0; indx < N_COEF_EDGE; indx++)
            {
                polyg_side_dgB(poly)[side][indx] = polyg_side_dgB(poly)[side][indx]*(2.0*indx+1.0)/2.0;
            }
#if !defined(__MPI__)
            if(YES == debug_flag)
                printf("   dual [%d], side %d, bn(%e %e)\n", poly->id, side, 
                       polyg_side_dgB(poly)[side][0], polyg_side_dgB(poly)[side][1]);
#endif
        }/// END:: for(side = 0; side < poly->n_sides; side++)

        tmp = 0.0;
        for(side = 0; side < poly->n_sides; side++)
            tmp += polyg_side_dgB(poly)[side][0]*polyg_length_side(poly)[side];

        if(debug_flag == YES || fabs(tmp) > 1.0e-10)
        {
            printf("polygon in dual_cell_accurate_magnetic_vortex_initializer()\n");
            printf("divB = %14.13g\n", tmp);
        }
}

LOCAL void dual_cell_field_loop_advection_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        POLYGON         *poly)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[MAX_N_POLY_SIDE], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[10][MAX_N_COEF], mulrhs[10][MAX_N_COEF], dens[10], tmp;
        float           t = 0.0, sqrt_area;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[30][8];
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        int           Lobatto_N = 7;
        double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        // Lobatto quadrature
        if(Lobatto_N == 4)
        {
            q[0] = -1.0; q[1] = -sqrt(5.0)/5.0;
            q[2] =  sqrt(5.0)/5.0; q[3] = 1.0;
            qw[0] =  1.0/6.0; qw[1] = 5.0/6.0;
            qw[2] =  5.0/6.0; qw[3] = 1.0/6.0;

            // Gaussian
            // q[0] = -0.86113631159405257522; q[1] = -0.33998104358485626480;
            // q[2] =  0.33998104358485626480; q[3] = 0.86113631159405257522;
            // qw[0] =  0.34785484513745385737; qw[1] = 0.65214515486254614263;
            // qw[2] =  0.65214515486254614263; qw[3] = 0.34785484513745385737;
        }
        else if(Lobatto_N == 5)
        {
            q[0] = -1.0; q[1] = -sqrt(21.0)/7.0;
            q[2] = 0.0;  q[3] =  sqrt(21.0)/7.0; q[4] = 1.0;
            qw[0] =  0.1; qw[1] = 49.0/90.0; qw[2] = 32.0/45.0;
            qw[3] =  49.0/90.0; qw[4] = 0.1;
        }
        else if(Lobatto_N == 6)
        {
            q[0] = -1.0; q[1] = -sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));
            q[2] = -sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));  q[3] =  sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));
            q[4] =  sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));     q[5] = 1.0;
            qw[0] =  1.0/15.0; qw[1] = 1.0/30.0*(14.0-sqrt(7.0));
            qw[2] = 1.0/30.0*(14.0+sqrt(7.0)); qw[3] =  1.0/30.0*(14.0+sqrt(7.0));
            qw[4] = 1.0/30.0*(14.0-sqrt(7.0));       qw[5] = 1.0/15.0;
        }
        else if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987;
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905;
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905;
        }
        else
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: dual_cell_field_loop_advection_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }

        third = 1.0/3.0;

        for(side = 0; side < poly->n_sides; side++)
        {
            for(indx = 0; indx < N_COEF_EDGE; indx++)
                polyg_side_dgB(poly)[side][indx] = 0.0;
            pcrds[side] = Coords(Point_of_polyg(poly)[side]);
        }

        for(side = 0; side < poly->n_sides; side++)
        {
            for(i = 0; i < 2; i++)
                v2[i] = polyg_side_vector(poly)[side][i];
            nor[0] = v2[1];
            nor[1] = -v2[0];

            for(k = 0; k < Lobatto_N; k++)//use Lobatto points(include vertices)
            {
                for(i = 0; i < 2; i++)
                    qcrds[i] = (pcrds[(side+1)%poly->n_sides][i] + pcrds[side][i])/2.0 +
                           (pcrds[(side+1)%poly->n_sides][i] - pcrds[side][i])/2.0*q[k];

                for(j = 5; j < N_EQN; j++)
                    // v_conU[side][j] = ff_accurate_magnetic_vort(NULL, qcrds,j,YES);
                    v_conU[side][j] = ff_accurate_field_loop_advection(NULL, qcrds,j,YES);

                Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];

                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    //printf("q[%d]=%Lf vh_val_ver2=%g\n",indx,q[k],vh_val_ver2(q[k],indx));
                    polyg_side_dgB(poly)[side][indx] += qw[k]*Bn[k]*vh_val_ver2(q[k],indx);//rhs
                }
            }

            for(indx = 0; indx < N_COEF_EDGE; indx++)
            {
                polyg_side_dgB(poly)[side][indx] = polyg_side_dgB(poly)[side][indx]*(2.0*indx+1.0)/2.0;
            }
        }

        tmp = 0.0;
        for(side = 0; side < poly->n_sides; side++)
            tmp += polyg_side_dgB(poly)[side][0]*polyg_length_side(poly)[side];

        if(debug_flag == YES || fabs(tmp) > 1.0e-10)
        {
            printf("ERROR: polygon in dual_cell_field_loop_advection_initializer()\n");
            printf("divB = %14.13g\n", tmp);
            clean_up(ERROR);
        }
}

LOCAL void dual_cell_delay_alfven_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        POLYGON         *poly)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[MAX_N_POLY_SIDE], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[10][MAX_N_COEF], mulrhs[10][MAX_N_COEF], dens[10], tmp;
        float           t = 0.0, sqrt_area;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[30][8];
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        int           Lobatto_N = 7;
        double        q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        // Lobatto quadrature
        if(Lobatto_N == 4)
        {
            q[0] = -1.0; q[1] = -sqrt(5.0)/5.0;
            q[2] =  sqrt(5.0)/5.0; q[3] = 1.0;
            qw[0] =  1.0/6.0; qw[1] = 5.0/6.0;
            qw[2] =  5.0/6.0; qw[3] = 1.0/6.0;

            // Gaussian
            // q[0] = -0.86113631159405257522; q[1] = -0.33998104358485626480;
            // q[2] =  0.33998104358485626480; q[3] = 0.86113631159405257522;
            // qw[0] =  0.34785484513745385737; qw[1] = 0.65214515486254614263;
            // qw[2] =  0.65214515486254614263; qw[3] = 0.34785484513745385737;
        }
        else if(Lobatto_N == 5)
        {
            q[0] = -1.0; q[1] = -sqrt(21.0)/7.0;
            q[2] = 0.0;  q[3] =  sqrt(21.0)/7.0; q[4] = 1.0;
            qw[0] =  0.1; qw[1] = 49.0/90.0; qw[2] = 32.0/45.0;
            qw[3] =  49.0/90.0; qw[4] = 0.1;
        }
        else if(Lobatto_N == 6)
        {
            q[0] = -1.0; q[1] = -sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));
            q[2] = -sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));  q[3] =  sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));
            q[4] =  sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));     q[5] = 1.0;
            qw[0] =  1.0/15.0; qw[1] = 1.0/30.0*(14.0-sqrt(7.0));
            qw[2] = 1.0/30.0*(14.0+sqrt(7.0)); qw[3] =  1.0/30.0*(14.0+sqrt(7.0));
            qw[4] = 1.0/30.0*(14.0-sqrt(7.0));       qw[5] = 1.0/15.0;
        }
        else if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987;
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905;
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905;
        }
        else
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: dual_cell_delay_alfven_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }

        third = 1.0/3.0;

        for(side = 0; side < poly->n_sides; side++)
        {
            for(indx = 0; indx < N_COEF_EDGE; indx++)
                polyg_side_dgB(poly)[side][indx] = 0.0;
            pcrds[side] = Coords(Point_of_polyg(poly)[side]);
        }

        for(side = 0; side < poly->n_sides; side++)
        {
            for(i = 0; i < 2; i++)
                v2[i] = polyg_side_vector(poly)[side][i];
            nor[0] = v2[1];
            nor[1] = -v2[0];

            for(k = 0; k < Lobatto_N; k++)//use Lobatto points(include vertices)
            {
                for(i = 0; i < 2; i++)
                    qcrds[i] = (pcrds[(side+1)%poly->n_sides][i] + pcrds[side][i])/2.0 +
                           (pcrds[(side+1)%poly->n_sides][i] - pcrds[side][i])/2.0*q[k];

                for(j = 5; j < N_EQN; j++)
                    // v_conU[side][j] = ff_accurate_magnetic_vort(NULL, qcrds,j,YES);
                    v_conU[side][j] = ff_accurate_magnetic_decay_alfven(NULL, qcrds,j,YES);

                Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];

                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    //printf("q[%d]=%Lf vh_val_ver2=%g\n",indx,q[k],vh_val_ver2(q[k],indx));
                    polyg_side_dgB(poly)[side][indx] += qw[k]*Bn[k]*vh_val_ver2(q[k],indx);//rhs
                }
            }

            for(indx = 0; indx < N_COEF_EDGE; indx++)
            {
                polyg_side_dgB(poly)[side][indx] = polyg_side_dgB(poly)[side][indx]*(2.0*indx+1.0)/2.0;
            }
        }

        tmp = 0.0;
        for(side = 0; side < poly->n_sides; side++)
            tmp += polyg_side_dgB(poly)[side][0]*polyg_length_side(poly)[side];

        if(debug_flag == YES || fabs(tmp) > 1.0e-10)
        {
            printf("polygon in dual_cell_delay_alfven_initializer()\n");
            printf("divB = %14.13g\n", tmp);
        }
}


LOCAL void g_FM_initializer(
	float		*coords,
	COMPONENT	comp,
	Locstate	state,
	INTERFACE	*intfc,
	INIT_DATA	*init,
        TRI         *tri)
{
	int i, j, dim = intfc->dim;
	int stype = type_of_state(init);

	debug_print("init_states","Entered g_initializer()\n");
	if (debugging("init_states"))
	{
  	    (void) printf("comp = %d, ",comp);
	    print_component_type("type = ",comp_type(comp),", ");
  	    for (i = 0; i < dim; i++)
  	    	(void) printf(" coords[%d] = %g ",i,coords[i]);
	    (void) printf("state - ");
	}

	Get_state(coords,state,comp_type(comp),NULL,intfc,init,stype);
        if(tri == NULL)
        {
            // printf("tri == NULL\n");
            return; 
        }

        dg_Dens(state)[0] = Dens(state);
        dg_Energy(state)[0] = Energy(state);
        for(i = 0; i < dim; i++)
        {
            dg_Mom(state)[i][0] = Mom(state)[i];
            dg_B(state)[i][0] = Mag(state)[i];
        }
        for(i = 1; i < MAX_N_COEF; i++)
        {
            // printf("MAX_N_COEF = %d\n", MAX_N_COEF);
            dg_Dens(state)[i] = 0.0;
            dg_Energy(state)[i] = 0.0;
            for(j = 0; j < dim; j++)
            {
                dg_Mom(state)[j][i] = 0.0;
                dg_B(state)[j][i] = 0.0;
            }
        }

	if (debugging("init_states"))
	{
	    print_gas_state(state);
	}
        // TMP
        // printf("coords[%g %g] dens = %g\n", coords[0], coords[1], Dens(state));

        if(debugging("Shu_Osher"))
        {
            init_Shu_Osher_problem(coords,comp,state,intfc,init,tri);
        }
        if(debugging("mag_acc_vert"))
        {
            /// accurate test 
            // accurate_magnetic_vortex_initializer(coords,comp,state,intfc,init,tri);
            // accurate_magnetic_vortex_initializer_2(coords,comp,state,intfc,init,tri);
            // test for DG //Huijing 
            // printf("In g_FM_initializer() before accurate_magnetic_vortex_initializer_3()\n%g %g\n",coords[0],coords[1]);
            // printf("%g %g\n",coords[0],coords[1]);
            // accurate_magnetic_vortex_initializer_3(coords,comp,state,intfc,init,tri);
            accurate_magnetic_vortex_initializer_4(coords,comp,state,intfc,init,tri);
            // printf("In g_FM_initializer() after accurate_magnetic_vortex_initializer_3()\n");
        }
        if(debugging("rotor"))
        {
            /// shock test 
            accurate_magnetic_rotor_initializer(coords,comp,state,intfc,init,tri);
        }
        if(debugging("Orszag_T"))
        {
            /// shock test 
            accurate_Orszag_T_initializer(coords,comp,state,intfc,init,tri);
        }
        if(debugging("blast_MHD"))
        {
            /// shock test 
            /// Since each of the physical state is piecewise const. We do not need to do reconstruction initially.
            /// And the energy is initialized with total energy.
            accurate_magnetic_blast_MHD_initializer(coords,comp,state,intfc,init,tri);
        }
        if(debugging("decay_alfven"))
        {
            /// This function is modified from supporting FV scheme to central DG on overlapping grid.
            accurate_magnetic_decay_alfven_initializer(coords,comp,state,intfc,init,tri);
        }
        if(debugging("field_loop"))
        {
            field_loop_advection_initializer(coords,comp,state,intfc,init,tri);
        }
        if(debugging("acc_vert"))
        {
            /// accurate test 
            accurate_magnetic_vortex_initializer(coords,comp,state,intfc,init,tri);
            //add_accurate_vortex_initializer(coords,comp,state,intfc,init,tri);
        }
        if(debugging("shock_vort"))
        {
            add_vortex_initializer(coords,comp,state,intfc,init,tri); 
        }

        if(debugging("v_evo"))
        {
            vortex_evolution_initializer(coords,comp,state,intfc,init,tri); 
            // vortex_evolution_initializer_interp(coords,comp,state,intfc,init,tri); 
            // vortex_evolution_initializer_least_sqr(coords,comp,state,intfc,init,tri); 
        }

        if(debugging("g_sine") || debugging("mag_g_sine"))
        {
            sine_evolution_initializer(coords,comp,state,intfc,init,tri);
        }

        if(debugging("twod_riemann"))
        {
            twod_riemann_initializer(coords,comp,state,intfc,init,tri);
        }

        if(debugging("Burgers"))
        {
            if(MAX_N_COEF == 6)
                Burgers_evolution_initializer(coords,comp,state,intfc,init,tri);
            else if(MAX_N_COEF == 10)
                Burgers_evolution_initializer_13_quad(coords,comp,state,intfc,init,tri);
            else
            {
                printf("ERROR: g_FM_initializer, implement Burgers MAX_N_COEF = %d\n",
                        MAX_N_COEF);
                clean_up(ERROR);
            }
        }

        if(debugging("db_Mach"))
        {
            /**
            printf("coords[%g %g] state:, cos(60) = %g, sin(60) = %g\n", 
                  coords[0], coords[1], cos(radians(60.0)),
                  sin(radians(60.0)));
            printf("den, u, v, press (%g %g %g %g)\n", Dens(state),
                    vel(0, state),   vel(1, state), pressure(state));
            **/
            init_db_Mach_state(coords,comp,state,intfc,init);

            // printf("after reset: den, u, v, press (%g %g %g %g)\n", Dens(state),
            //         vel(0, state),   vel(1, state), pressure(state));

            dg_Dens(state)[0] = Dens(state);
            dg_Energy(state)[0] = Energy(state);
            dg_Mom(state)[0][0] = Mom(state)[0];
            dg_Mom(state)[1][0] = Mom(state)[1];
            for(i = 1; i < MAX_N_COEF; i++)
            {
                // printf("MAX_N_COEF = %d\n", MAX_N_COEF);
                dg_Dens(state)[i] = 0.0;
                dg_Energy(state)[i] = 0.0;
                dg_Mom(state)[0][i] = 0.0;
                dg_Mom(state)[1][i] = 0.0;
            }
        } 

        if(debugging("Mach_step"))
        {
            set_state(state,TGAS_STATE,state);
            Dens(state) = 1.4;
            Press(state) = 1.0;
            Vel(state)[0] = 0.0; Vel(state)[1] = 3.0;
            set_state(state,GAS_STATE,state);

            dg_Dens(state)[0] = Dens(state);
            dg_Energy(state)[0] = Energy(state);
            dg_Mom(state)[0][0] = Mom(state)[0];
            dg_Mom(state)[1][0] = Mom(state)[1];
            for(i = 1; i < MAX_N_COEF; i++)
            {
                dg_Dens(state)[i] = 0.0;
                dg_Energy(state)[i] = 0.0;
                dg_Mom(state)[0][i] = 0.0;
                dg_Mom(state)[1][i] = 0.0;
            }
        }

	debug_print("init_states","Left g_initializer()\n");
}		/*end g_initializer*/

// y-1/6 = -sqrt(3)/3 (x-1)
LOCAL void init_db_Mach_state(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init)
{
        float y, vn, v[3];
        float angle = 60.0; // 60.0
        y = 1.0/6.0 -sqrt(3)/3.0*(coords[0]-1.0);
        // y = 1.0/4.0;

        if(coords[1] > y)
        {
        // pre-shock
            set_state(state,TGAS_STATE,state);
            Dens(state) = 1.4;
            Press(state) = 1.0;
            Vel(state)[0] = Vel(state)[1] = 0.0;
        }
        else
        {
        // post-shock
            set_state(state,TGAS_STATE,state);
            v[0] = Vel(state)[0] = 8.25*cos(radians(angle));
            v[1] = Vel(state)[1] = 8.25*sin(radians(angle));
            Press(state) = 116.5;
            Dens(state) = 8.0;
            // printf("post shock velocity %16.15g, %16.15g\n", v[0], v[1]);
        }

        set_state(state,GAS_STATE,state); 
}

LOCAL void add_vortex_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[3], dbcent[MAXD];
        float           Gam, S, gam = 1.4;
        double   rhs[4][MAX_N_COEF], mulrhs[4][MAX_N_COEF], dens[7];
        float         area = fg_area(tri), t = 0.0;
        double      **mass_inv, **Lmass_matrix;
        int     debug_flag = NO, i, j, k, indx;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
          w3 = 0.175615257433204,  w4 = 0.077113760890257;
        float crds[13][2];
        double     *lcent = fg_centroid(tri);
        static double **tmp_mass_inv = NULL;

        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;

        // printf("Entered add_vortex_initializer\n");
        // print_rectangular_grid(gr);   

        // Gam = gruneisen_gamma(state);

        Lmass_matrix = tri->Lmass_matrix;
        if(tri->mass_inv != NULL)
            mass_inv = tri->mass_inv;
        else
        {
            mass_inv = tmp_mass_inv;
            inverse_matrix(Lmass_matrix,MAX_N_COEF,mass_inv);
        }

        for(i = 0; i < 4; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
            dbcent[i] = fg_centroid(tri)[i];

        if(dbcent[1] > 0.5)
        {
            Dens(state) = dg_Dens(state)[0] = ff_shock_vort(dbcent, dbcent, 0);    
            Mom(state)[0] = dg_Mom(state)[0][0] = ff_shock_vort(dbcent, dbcent, 1);    
            Mom(state)[1] = dg_Mom(state)[1][0] = ff_shock_vort(dbcent, dbcent, 2);    
            Energy(state) = dg_Energy(state)[0] = ff_shock_vort(dbcent, dbcent, 3);    
            for(indx = 1; indx < MAX_N_COEF; indx++)
            {
                dg_Dens(state)[indx] = 0.0;
                dg_Mom(state)[0][indx] = 0.0;
                dg_Mom(state)[1][indx] = 0.0;
                dg_Energy(state)[indx] = 0.0;
            }
            // printf("print post shock state\n");
            // verbose_print_state("post shock state", state);
            // clean_up(0);
            return;
        }

        tri_quadrature_13_pts(pcrds[0], pcrds[1], pcrds[2], crds);

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = w1*ff_shock_vort(dbcent, crds[0],j)*vh_val(crds[0],lcent,k) +
                      w2*(ff_shock_vort(dbcent, crds[1],j)*vh_val(crds[1],lcent,k) +
                          ff_shock_vort(dbcent, crds[2],j)*vh_val(crds[2],lcent,k) +
                          ff_shock_vort(dbcent, crds[3],j)*vh_val(crds[3],lcent,k)) +
                      w3*(ff_shock_vort(dbcent, crds[4],j)*vh_val(crds[4],lcent,k) +
                          ff_shock_vort(dbcent, crds[5],j)*vh_val(crds[5],lcent,k) +
                          ff_shock_vort(dbcent, crds[6],j)*vh_val(crds[6],lcent,k)) +
                      w4*(ff_shock_vort(dbcent, crds[7],j)*vh_val(crds[7],lcent,k) +
                          ff_shock_vort(dbcent, crds[8],j)*vh_val(crds[8],lcent,k) +
                          ff_shock_vort(dbcent, crds[9],j)*vh_val(crds[9],lcent,k) +
                          ff_shock_vort(dbcent, crds[10],j)*vh_val(crds[10],lcent,k) +
                          ff_shock_vort(dbcent, crds[11],j)*vh_val(crds[11],lcent,k) +
                          ff_shock_vort(dbcent, crds[12],j)*vh_val(crds[12],lcent,k));
            }
        }

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Lmass_matrix[0][indx];
        }
        Dens(state) /= Lmass_matrix[0][0];
        Mom(state)[0] /= Lmass_matrix[0][0];
        Mom(state)[1] /= Lmass_matrix[0][0];
        Energy(state) /= Lmass_matrix[0][0];

        // TMP
        /**
        if(dbcent[1] > 0.5)
        {
            printf("print post shock state\n");
            verbose_print_state("post shock state", state);
            clean_up(0);
        }
        **/ 

        return;
        /////////////////////

        // cent[0] = gr->L[0] + 0.5*(gr->U[0]-gr->L[0]);
        // cent[1] = gr->L[1] + 0.125*(gr->U[1]-gr->L[1]);
        cent[0] = 0.5;
        cent[1] = 0.25;
        rc = 0.05;

        v2[0] = coords[0]-cent[0];
        v2[1] = coords[1]-cent[1];
        r = sqrt(sqr(v2[0]) + sqr(v2[1])); 
        tau = r/rc;

        // if(r > MACH_EPS*1000.0)
        {
            norv2[0] = v2[0]/r;
            norv2[1] = v2[1]/r;
            r_angle = angle(norv2[0],norv2[1]);
            costheta = cos(r_angle);
            sintheta = sin(r_angle);
            v = ep*tau*exp(alpha*(1-sqr(tau)))*sintheta;
            u = -ep*tau*exp(alpha*(1-sqr(tau)))*costheta;
            dT = -Gam*sqr(ep)*exp(2.0*alpha*(1.0-sqr(tau)))/(4.0*alpha*(Gam+1.0));
        }
        // else
        //     dT = -Gam*sqr(ep)*exp(2.0*alpha)/(4.0*alpha*(Gam+1.0)); 

        set_state(state,TGAS_STATE,state);
        Vel(state)[0] += u;
        Vel(state)[1] += v;
        T = temperature(state);
        S = log(Press(state)/(pow(Dens(state), gam)));
  
        nT = T + dT;

        Dens(state) = pow(exp(S)/nT, 1.0/(1.0-gam));
        Press(state) = nT*Dens(state);

        set_state(state,GAS_STATE,state);

        // FOR 2d DG
        dg_Dens(state)[0] = Dens(state);
        dg_Energy(state)[0] = Energy(state);
        dg_Mom(state)[0][0] = Mom(state)[0];
        dg_Mom(state)[1][0] = Mom(state)[1];
 
        /*
        printf("coords[%g %g]\n", coords[0], coords[1]);
        printf("cent[%g %g], rc %g vec[%g %g], cos %g, sin %g\n", 
                  cent[0], cent[1], rc, v2[0], v2[1], costheta, sintheta); 
        printf("pert u(%g %g), T = %g\n", u, v, T);
        clean_up(0);
        */
}

LOCAL void init_Shu_Osher_problem(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           r2, r, ep = 5.0;
        float           du[2], dT, T, S, nT, nS, nP;
        float           gam = 1.4, P;
        float           bar[3];
        int             i, j, k, indx;
        POINT           *p[3];
        float           *pcrds[3], dbcent[MAXD];
        float           emid0[3], emid1[3], emid2[3], *pemid[3], conu[4];
        double     *cent = fg_centroid(tri);
        double     rhs[4][MAX_N_COEF], mulrhs[4][MAX_N_COEF], dens[7];
        float           area = fg_area(tri), v = 0.0, t = 0.0;
        double     **mass_inv, **Lmass_matrix;
        int             debug_flag = NO;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
          w3 = 0.175615257433204,  w4 = 0.077113760890257;
        float crds[13][2];
        static double **tmp_mass_inv = NULL;

        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;

        // gam = adiabatic_gamma(state);

        Lmass_matrix = tri->Lmass_matrix;
        if(tri->mass_inv != NULL)
            mass_inv = tri->mass_inv;
        else
        {
            mass_inv = tmp_mass_inv;
            inverse_matrix(Lmass_matrix,MAX_N_COEF,mass_inv);
        }

        for(i = 0; i < 4; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        tri_quadrature_13_pts(pcrds[0], pcrds[1], pcrds[2], crds);

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = w1*ff_Shu_Osher(crds[0],j, 0.0)*vh_val(crds[0],cent,k) +
                      w2*(ff_Shu_Osher(crds[1],j, 0.0)*vh_val(crds[1],cent,k) +
                          ff_Shu_Osher(crds[2],j, 0.0)*vh_val(crds[2],cent,k) +
                          ff_Shu_Osher(crds[3],j, 0.0)*vh_val(crds[3],cent,k)) +
                      w3*(ff_Shu_Osher(crds[4],j, 0.0)*vh_val(crds[4],cent,k) +
                          ff_Shu_Osher(crds[5],j, 0.0)*vh_val(crds[5],cent,k) +
                          ff_Shu_Osher(crds[6],j, 0.0)*vh_val(crds[6],cent,k)) +
                      w4*(ff_Shu_Osher(crds[7],j, 0.0)*vh_val(crds[7],cent,k) +
                          ff_Shu_Osher(crds[8],j, 0.0)*vh_val(crds[8],cent,k) +
                          ff_Shu_Osher(crds[9],j, 0.0)*vh_val(crds[9],cent,k) +
                          ff_Shu_Osher(crds[10],j, 0.0)*vh_val(crds[10],cent,k) +
                          ff_Shu_Osher(crds[11],j, 0.0)*vh_val(crds[11],cent,k) +
                          ff_Shu_Osher(crds[12],j, 0.0)*vh_val(crds[12],cent,k));
            }
        }

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0; 
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Lmass_matrix[0][indx];
        }
        Dens(state) /= Lmass_matrix[0][0];
        Mom(state)[0] /= Lmass_matrix[0][0];
        Mom(state)[1] /= Lmass_matrix[0][0];
        Energy(state) /= Lmass_matrix[0][0];

        if(MAX_N_COEF ==  6)
            map_poly_SV_to_CV_p2(tri);
        else
            map_poly_SV_to_CV_p3(tri);        
        return;
        /////////////////////

        for(i = 0; i < 2; i++)
        {
            emid0[i] = 0.5*(pcrds[0][i] + pcrds[1][i]);
            emid1[i] = 0.5*(pcrds[1][i] + pcrds[2][i]);
            emid2[i] = 0.5*(pcrds[2][i] + pcrds[0][i]);
        }
        pemid[0] = emid0;
        pemid[1] = emid1;
        pemid[2] = emid2;

        // vertices
        for(i = 0; i < 3; i++)
        {
            set_state(state,TGAS_STATE,state);
            Vel(state)[0] = Vel(state)[1] = 0.0;
            if(pcrds[i][1] < -4.0)
            {
                Dens(state) = 3.857143;
                Press(state) = 10.333333;
                Vel(state)[1] = 2.629369;
            }
            else
            {
                Dens(state) = 1.0 + 0.2*sin(5.0*pcrds[i][1]);
                Press(state) = 1.0;
            }

            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            conu[1] = Mom(state)[0];
            conu[2] = Mom(state)[1];
            conu[3] = Energy(state);
            for(j = 0; j < 4; j++)
            {
                for(k = 0; k < MAX_N_COEF; k++)
                    rhs[j][k] += 0.05*(conu[j]*vh_val(pcrds[i],cent,k));
            }
            //
            if(debug_flag == YES)
            {
                printf("vertice[%d] state[%g %g %g %g], du[%g %g]\n",
                    i, conu[0], conu[1], conu[2], conu[3], du[0], du[1]);
            }
        }

        // edge mid points
        for(i = 0; i < 3; i++)
        {
            set_state(state,TGAS_STATE,state);
            Vel(state)[0] = Vel(state)[1] = 0.0;
            if(pemid[i][1] < -4.0)
            {
                Dens(state) = 3.857143;
                Press(state) = 10.333333;
                Vel(state)[1] = 2.629369;
            }
            else
            {
                Dens(state) = 1.0 + 0.2*sin(5.0*pemid[i][1]);
                Press(state) = 1.0;
            }

            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            conu[1] = Mom(state)[0];
            conu[2] = Mom(state)[1];
            conu[3] = Energy(state);
            for(j = 0; j < 4; j++)
            {
                for(k = 0; k < MAX_N_COEF; k++)
                    rhs[j][k] += 2.0/15.0*(conu[j]*vh_val(pemid[i],cent,k));
            }
        }

        // tri centriod
        for(i = 0; i < 2; i++)
            dbcent[i] = cent[i];
        set_state(state,TGAS_STATE,state);
        Vel(state)[0] = Vel(state)[1] = 0.0;
        if(dbcent[1] < -4.0)
        {
            Dens(state) = 3.857143;
            Press(state) = 10.333333;
            Vel(state)[1] = 2.629369;
        }
        else
        {
            Dens(state) = 1.0 + 0.2*sin(5.0*dbcent[1]);
            Press(state) = 1.0;
        }

        set_state(state,GAS_STATE,state);
        conu[0] = Dens(state);
        conu[1] = Mom(state)[0];
        conu[2] = Mom(state)[1];
        conu[3] = Energy(state);

        for(j = 0; j < 4; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
                rhs[j][k] += 9.0/20.0*(conu[j]*vh_val(dbcent,cent,k));
        }

        for(j = 0; j < 4; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < 4; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Lmass_matrix[0][indx];
        }
        Dens(state) /= Lmass_matrix[0][0];
        Mom(state)[0] /= Lmass_matrix[0][0];
        Mom(state)[1] /= Lmass_matrix[0][0];
        Energy(state) /= Lmass_matrix[0][0];
}

/**
7 point quadrature which has precision 4.
Johnson, pp340
**/

LOCAL void vortex_evolution_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           r2, r, ep = 5.0;
        float           du[2], dT, T, S, nT, nS, nP;
        float           gam = 1.4, P;
        float           bar[3];
        int             i, j, k, indx;
        POINT   *p[3];
        float   *pcrds[3], dbcent[MAXD];
        float   emid0[3], emid1[3], emid2[3], *pemid[3], conu[4];
        double   *cent = fg_centroid(tri);
        double   rhs[4][MAX_N_COEF], mulrhs[4][MAX_N_COEF], dens[7];
        float   area = fg_area(tri), v = 0.0;
        double      **mass_inv, **Lmass_matrix;
        int     debug_flag = NO;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
          w3 = 0.175615257433204,  w4 = 0.077113760890257;
        float crds[13][2];
        static double **tmp_mass_inv = NULL;

        third = 1.0/3.0;

        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        // gam = adiabatic_gamma(state);

        Lmass_matrix = tri->Lmass_matrix;
        if(tri->mass_inv != NULL)
            mass_inv = tri->mass_inv;
        else
        {
            mass_inv = tmp_mass_inv;
            inverse_matrix(Lmass_matrix,MAX_N_COEF,mass_inv);
        }

        for(i = 0; i < 4; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        tri_quadrature_13_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = w1*ff_vort(crds[0],j)*vh_val(crds[0],cent,k) +
                      w2*(ff_vort(crds[1],j)*vh_val(crds[1],cent,k) +
                          ff_vort(crds[2],j)*vh_val(crds[2],cent,k) +
                          ff_vort(crds[3],j)*vh_val(crds[3],cent,k)) +
                      w3*(ff_vort(crds[4],j)*vh_val(crds[4],cent,k) +
                          ff_vort(crds[5],j)*vh_val(crds[5],cent,k) +
                          ff_vort(crds[6],j)*vh_val(crds[6],cent,k)) +
                      w4*(ff_vort(crds[7],j)*vh_val(crds[7],cent,k) +
                          ff_vort(crds[8],j)*vh_val(crds[8],cent,k) +
                          ff_vort(crds[9],j)*vh_val(crds[9],cent,k) +
                          ff_vort(crds[10],j)*vh_val(crds[10],cent,k) +
                          ff_vort(crds[11],j)*vh_val(crds[11],cent,k) +
                          ff_vort(crds[12],j)*vh_val(crds[12],cent,k));
            }
        }

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Lmass_matrix[0][indx];
        }
        Dens(state) /= Lmass_matrix[0][0];
        Mom(state)[0] /= Lmass_matrix[0][0];
        Mom(state)[1] /= Lmass_matrix[0][0];
        Energy(state) /= Lmass_matrix[0][0];

        return;
        ////////////////////////

        for(i = 0; i < 2; i++)
        {
            emid0[i] = 0.5*(pcrds[0][i] + pcrds[1][i]);
            emid1[i] = 0.5*(pcrds[1][i] + pcrds[2][i]);
            emid2[i] = 0.5*(pcrds[2][i] + pcrds[0][i]);
        }
        pemid[0] = emid0;
        pemid[1] = emid1;
        pemid[2] = emid2;

        // vertices
        for(i = 0; i < 3; i++)
        {
            set_state(state,TGAS_STATE,state);
            Dens(state) = 1.0;
            Press(state) = 1.0;
            Vel(state)[0] = Vel(state)[1] = v;
            T = 1.0;
            // S = log(Press(state)/(pow(Dens(state), gam)));
            S = (Press(state)/(pow(Dens(state), gam)));

            bar[0] = pcrds[i][0] - 5.0;
            bar[1] = pcrds[i][1] - 5.0;

            r2 = sqr(bar[0]) + sqr(bar[1]);
            du[0] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(-bar[1]);
            du[1] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(bar[0]);
            dT = -1.0*(gam - 1.0)*sqr(ep)/(8.0*gam*sqr(PI))*exp(1.0-r2);
            nT = T + dT;

            // Dens(state) = pow(exp(S)/nT, 1.0/(1.0-gam));
            dens[i] = Dens(state) = pow(nT/S, 1.0/(gam-1.0));
            Press(state) = nT*Dens(state);

            Vel(state)[0] += du[0];
            Vel(state)[1] += du[1];

            nS = Press(state)/pow(Dens(state), gam);
            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            conu[1] = Mom(state)[0];
            conu[2] = Mom(state)[1];
            conu[3] = Energy(state);
            for(j = 0; j < 4; j++)
            {
                for(k = 0; k < MAX_N_COEF; k++)
                    rhs[j][k] += 0.05*(conu[j]*vh_val(pcrds[i],cent,k));
            }
            // 
            if(debug_flag == YES)
            {
                printf("vertice[%d] state[%g %g %g %g], du[%g %g]\n",
                    i, conu[0], conu[1], conu[2], conu[3], du[0], du[1]);
            }
        }
         
        // edge mid points
        for(i = 0; i < 3; i++)
        {
            set_state(state,TGAS_STATE,state);
            Dens(state) = 1.0;
            Press(state) = 1.0;
            Vel(state)[0] = Vel(state)[1] = v;
            T = 1.0;
            // S = log(Press(state)/(pow(Dens(state), gam)));
            S = (Press(state)/(pow(Dens(state), gam)));

            bar[0] = pemid[i][0] - 5.0;
            bar[1] = pemid[i][1] - 5.0;

            r2 = sqr(bar[0]) + sqr(bar[1]);
            du[0] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(-bar[1]);
            du[1] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(bar[0]);
            dT = -1.0*(gam - 1.0)*sqr(ep)/(8.0*gam*sqr(PI))*exp(1.0-r2);
            nT = T + dT;

            // Dens(state) = pow(exp(S)/nT, 1.0/(1.0-gam));
            dens[i+3] = Dens(state) = pow(nT/S, 1.0/(gam-1.0));
            Press(state) = nT*Dens(state);

            Vel(state)[0] += du[0];
            Vel(state)[1] += du[1];

            nS = Press(state)/pow(Dens(state), gam);
            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            conu[1] = Mom(state)[0];
            conu[2] = Mom(state)[1];
            conu[3] = Energy(state);
            for(j = 0; j < 4; j++)
            {
                for(k = 0; k < MAX_N_COEF; k++)
                    rhs[j][k] += 2.0/15.0*(conu[j]*vh_val(pemid[i],cent,k));
            }
        }


        // tri centriod
        for(i = 0; i < 2; i++) 
            dbcent[i] = cent[i];
        set_state(state,TGAS_STATE,state);
        Dens(state) = 1.0;
        Press(state) = 1.0;
        Vel(state)[0] = Vel(state)[1] = v;
        T = 1.0;
        // S = log(Press(state)/(pow(Dens(state), gam)));
        S = (Press(state)/(pow(Dens(state), gam)));

        bar[0] = dbcent[0] - 5.0;
        bar[1] = dbcent[1] - 5.0;

        r2 = sqr(bar[0]) + sqr(bar[1]);
        du[0] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(-bar[1]);
        du[1] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(bar[0]);
        dT = -1.0*(gam - 1.0)*sqr(ep)/(8.0*gam*sqr(PI))*exp(1.0-r2);
        nT = T + dT; 

        // Dens(state) = pow(exp(S)/nT, 1.0/(1.0-gam));
        dens[6] = Dens(state) = pow(nT/S, 1.0/(gam-1.0)); 
        Press(state) = nT*Dens(state);

        Vel(state)[0] += du[0];
        Vel(state)[1] += du[1];

        nS = Press(state)/pow(Dens(state), gam);
        set_state(state,GAS_STATE,state);
        nP = pressure(state);
        conu[0] = Dens(state);
        conu[1] = Mom(state)[0];
        conu[2] = Mom(state)[1];
        conu[3] = Energy(state);

        for(j = 0; j < 4; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
                rhs[j][k] += 9.0/20.0*(conu[j]*vh_val(dbcent,cent,k));
        }

        for(j = 0; j < 4; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < 4; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];  
            dg_Mom(state)[0][indx] = mulrhs[1][indx];  
            dg_Mom(state)[1][indx] = mulrhs[2][indx];  
            dg_Energy(state)[indx] = mulrhs[3][indx];  
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Lmass_matrix[0][indx];
        }
        Dens(state) /= Lmass_matrix[0][0];
        Mom(state)[0] /= Lmass_matrix[0][0];
        Mom(state)[1] /= Lmass_matrix[0][0];
        Energy(state) /= Lmass_matrix[0][0];

        if(debug_flag == YES)
        {
            printf("vortex_evolution_initializer coords[%g %g]\n", coords[0], coords[1]);
            printf("pert u(%g %g), dT = %g\n", du[0], du[1], dT);
            printf("entropy change = %g, gamma = %g, S = %g, T = %g, rho = %g, nP = %g\n\n",
                 (S - nS), gam, S, T, Dens(state), nP);
            g_verbose_print_state(state);
            // con_u_at_pt(state, dbcent, cent, conu); 
            printf("cent-dens: %14.13g, vertex: ", conu[0]);
            for(i = 0; i < 3; i++)
            {
                // con_u_at_pt(state, pcrds[i], cent, conu); 
                printf("%g, ", conu[0]);
            }
            printf("\n");
            printf("pt_dens(%g, %g, %g, %g, %g, %g, %g)\n", dens[0], dens[1], 
                      dens[2], dens[3], dens[4],dens[5],dens[6]);
            printf("total mass %g, Proj mass %g\n\n", rhs[0][0], Dens(state)*area);
            // clean_up(0);
        }
}

/**
7 point quadrature which has precision 4.
Johnson, pp340
Now use 13 point quadrature which has precision 6.
**/
LOCAL void sine_evolution_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           r2, r, ep = 5.0;
        float           du[2], dT, T, S, nT, nS, nP;
        float           gam = 1.4, P;
        float           bar[3];
        int             i, j, k, indx;
        POINT   *p[3];
        float   *pcrds[3], dbcent[MAXD];
        float   emid0[3], emid1[3], emid2[3], *pemid[3], conu[8];
        double   *cent = fg_centroid(tri);
        double   rhs[8][MAX_N_COEF], mulrhs[8][MAX_N_COEF], dens[8];
        float   area = fg_area(tri), v = 0.0, t = 0.0;
        double      **mass_inv, **Lmass_matrix;
        int     debug_flag = NO;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
          w3 = 0.175615257433204,  w4 = 0.077113760890257;
        float crds[13][2];
        static double **tmp_mass_inv = NULL;

        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;

        // gam = adiabatic_gamma(state);

        Lmass_matrix = tri->Lmass_matrix;
        if(tri->mass_inv != NULL)
            mass_inv = tri->mass_inv;
        else
        {
            mass_inv = tmp_mass_inv;
            inverse_matrix(Lmass_matrix,MAX_N_COEF,mass_inv);
        }

        for(i = 0; i < 8; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        tri_quadrature_13_pts(pcrds[0], pcrds[1], pcrds[2], crds);

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = w1*ff_g_sin(crds[0],j, 0.0)*vh_val(crds[0],cent,k) +
                      w2*(ff_g_sin(crds[1],j, 0.0)*vh_val(crds[1],cent,k) +
                          ff_g_sin(crds[2],j, 0.0)*vh_val(crds[2],cent,k) +
                          ff_g_sin(crds[3],j, 0.0)*vh_val(crds[3],cent,k)) +
                      w3*(ff_g_sin(crds[4],j, 0.0)*vh_val(crds[4],cent,k) +
                          ff_g_sin(crds[5],j, 0.0)*vh_val(crds[5],cent,k) +
                          ff_g_sin(crds[6],j, 0.0)*vh_val(crds[6],cent,k)) +
                      w4*(ff_g_sin(crds[7],j, 0.0)*vh_val(crds[7],cent,k) +
                          ff_g_sin(crds[8],j, 0.0)*vh_val(crds[8],cent,k) +
                          ff_g_sin(crds[9],j, 0.0)*vh_val(crds[9],cent,k) +
                          ff_g_sin(crds[10],j, 0.0)*vh_val(crds[10],cent,k) +
                          ff_g_sin(crds[11],j, 0.0)*vh_val(crds[11],cent,k) +
                          ff_g_sin(crds[12],j, 0.0)*vh_val(crds[12],cent,k));
            }
        }

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);

        if(N_EQN == 4)
        {
            for(indx = 0; indx < MAX_N_COEF; indx++)
            {
                dg_Dens(state)[indx] = mulrhs[0][indx];
                dg_Mom(state)[0][indx] = mulrhs[1][indx];
                dg_Mom(state)[1][indx] = mulrhs[2][indx];
                dg_Energy(state)[indx] = mulrhs[3][indx];
            }
        }
        else if(N_EQN == 8)
        {
            for(indx = 0; indx < MAX_N_COEF; indx++)
            {
                dg_Dens(state)[indx] = mulrhs[0][indx];
                dg_Mom(state)[0][indx] = mulrhs[1][indx];
                dg_Mom(state)[1][indx] = mulrhs[2][indx];
                dg_Energy(state)[indx] = mulrhs[3][indx];
                dg_Mom(state)[2][indx] = 0.0;
                dg_B(state)[0][indx] = dg_B(state)[1][indx] = 
                                           dg_B(state)[2][indx] = 0.0;
            }
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        if(N_EQN == 8)
        {
            Mom(state)[2] = 0.0;
            Mag(state)[0] = 0.0;
            Mag(state)[1] = 0.0;
            Mag(state)[2] = 0.0;
        }

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Lmass_matrix[0][indx];
        }
        Dens(state) /= Lmass_matrix[0][0];
        Mom(state)[0] /= Lmass_matrix[0][0];
        Mom(state)[1] /= Lmass_matrix[0][0];
        Energy(state) /= Lmass_matrix[0][0];

        if(N_EQN == 8)
        {
            for(i=0; i < 3; i++)
                fg_side_B(tri)[i] = 0.0;
        }
        return;
        //////////////////



        for(i = 0; i < 2; i++)
        {
            emid0[i] = 0.5*(pcrds[0][i] + pcrds[1][i]);
            emid1[i] = 0.5*(pcrds[1][i] + pcrds[2][i]);
            emid2[i] = 0.5*(pcrds[2][i] + pcrds[0][i]);
        }
        pemid[0] = emid0;
        pemid[1] = emid1;
        pemid[2] = emid2;

        // vertices
        for(i = 0; i < 3; i++)
        {
            set_state(state,TGAS_STATE,state);
            Vel(state)[0] = 1.0; Vel(state)[1] = -0.7;
            Dens(state) = 1.0 + 0.5*sin(pcrds[i][0] + pcrds[i][1] - (Vel(state)[0] + Vel(state)[1])*t);
            Press(state) = 1.0;

            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            conu[1] = Mom(state)[0];
            conu[2] = Mom(state)[1];
            conu[3] = Energy(state);
            for(j = 0; j < 4; j++)
            {
                for(k = 0; k < MAX_N_COEF; k++)
                    rhs[j][k] += 0.05*(conu[j]*vh_val(pcrds[i],cent,k));
            }
            // 
            if(debug_flag == YES)
            {
                printf("vertice[%d] state[%g %g %g %g], du[%g %g]\n",
                    i, conu[0], conu[1], conu[2], conu[3], du[0], du[1]);
            }
        }
         
        // edge mid points
        for(i = 0; i < 3; i++)
        {
            set_state(state,TGAS_STATE,state);
            Vel(state)[0] = 1.0; Vel(state)[1] = -0.7;
            Dens(state) = 1.0 + 0.5*sin(pemid[i][0] + pemid[i][1] - (Vel(state)[0] + Vel(state)[1])*t);
            Press(state) = 1.0;

            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            conu[1] = Mom(state)[0];
            conu[2] = Mom(state)[1];
            conu[3] = Energy(state);
            for(j = 0; j < 4; j++)
            {
                for(k = 0; k < MAX_N_COEF; k++)
                    rhs[j][k] += 2.0/15.0*(conu[j]*vh_val(pemid[i],cent,k));
            }
        }


        // tri centriod
        for(i = 0; i < 2; i++) 
            dbcent[i] = cent[i];
        set_state(state,TGAS_STATE,state);
        Vel(state)[0] = 1.0; Vel(state)[1] = -0.7;
        Dens(state) = 1.0 + 0.5*sin(dbcent[0] + dbcent[1] - (Vel(state)[0] + Vel(state)[1])*t);
        Press(state) = 1.0;

        set_state(state,GAS_STATE,state);
        conu[0] = Dens(state);
        conu[1] = Mom(state)[0];
        conu[2] = Mom(state)[1];
        conu[3] = Energy(state);

        for(j = 0; j < 4; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
                rhs[j][k] += 9.0/20.0*(conu[j]*vh_val(dbcent,cent,k));
        }

        for(j = 0; j < 4; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < 4; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];  
            dg_Mom(state)[0][indx] = mulrhs[1][indx];  
            dg_Mom(state)[1][indx] = mulrhs[2][indx];  
            dg_Energy(state)[indx] = mulrhs[3][indx];  
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Lmass_matrix[0][indx];
        }
        Dens(state) /= Lmass_matrix[0][0];
        Mom(state)[0] /= Lmass_matrix[0][0];
        Mom(state)[1] /= Lmass_matrix[0][0];
        Energy(state) /= Lmass_matrix[0][0];
}

/**
7 point quadrature which has precision 4.
Johnson, pp340
u(x,y,0) = 1/4 + 1/2sin(pi(x+y))
**/
LOCAL void Burgers_evolution_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           r2, r, ep = 5.0;
        float           du[2], dT, T, S, nT, nS, nP;
        float           gam = 1.4, P;
        float           bar[3];
        int             i, j, k, indx;
        POINT   *p[3];
        float   *pcrds[3], dbcent[MAXD];
        float   emid0[3], emid1[3], emid2[3], *pemid[3], conu[4];
        double   *cent = fg_centroid(tri);
        double   rhs[4][MAX_N_COEF], mulrhs[4][MAX_N_COEF], dens[7];
        float   area = fg_area(tri), v = 0.0, t = 0.0;
        double      **mass_inv, **Lmass_matrix;
        int     debug_flag = NO;
        static double **tmp_mass_inv = NULL;

        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        // gam = adiabatic_gamma(state);

        Lmass_matrix = tri->Lmass_matrix;
        if(tri->mass_inv != NULL)
            mass_inv = tri->mass_inv;
        else
        {
            mass_inv = tmp_mass_inv;
            inverse_matrix(Lmass_matrix,MAX_N_COEF,mass_inv);
        }

        // Taylor expansion version
        /**
        for(i = 0; i < 2; i++)
            dbcent[i] = cent[i];
        dg_Dens(state)[0] = 0.25 + 0.5*sin((dbcent[0] + dbcent[1])*PI);
        dg_Dens(state)[1] = 0.5*cos((dbcent[0] + dbcent[1])*PI)*PI;  
        dg_Dens(state)[2] = 0.5*cos((dbcent[0] + dbcent[1])*PI)*PI;  
        dg_Dens(state)[3] = -0.5*sin((dbcent[0] + dbcent[1])*PI)*PI*PI/2.0;  
        dg_Dens(state)[4] = -0.5*sin((dbcent[0] + dbcent[1])*PI)*PI*PI;  
        dg_Dens(state)[5] = -0.5*sin((dbcent[0] + dbcent[1])*PI)*PI*PI/2.0;  
        // sin(PI*y); 
        dg_Dens(state)[0] = 0.25 + 0.5*sin((dbcent[1])*PI);
        dg_Dens(state)[1] = 0.0;  
        dg_Dens(state)[2] = 0.5*cos((dbcent[1])*PI)*PI;  
        dg_Dens(state)[3] = 0.0;  
        dg_Dens(state)[4] = -0.0;  
        dg_Dens(state)[5] = -0.5*sin((dbcent[1])*PI)*PI*PI/2.0;  
        // End sin(PI*y);
        Dens(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
        Dens(state) /= Lmass_matrix[0][0];
        return;
        **/
        // End Taylor expansion version

        for(i = 0; i < 4; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
        {
            emid0[i] = 0.5*(pcrds[0][i] + pcrds[1][i]);
            emid1[i] = 0.5*(pcrds[1][i] + pcrds[2][i]);
            emid2[i] = 0.5*(pcrds[2][i] + pcrds[0][i]);
        }
        pemid[0] = emid0;
        pemid[1] = emid1;
        pemid[2] = emid2;

        // vertices
        for(i = 0; i < 3; i++)
        {
            Dens(state) = 0.25 + 0.5*sin((pcrds[i][0] + pcrds[i][1])*PI);

            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            for(j = 0; j < N_EQN; j++)
            {
                for(k = 0; k < MAX_N_COEF; k++)
                    rhs[j][k] += 0.05*(conu[j]*vh_val(pcrds[i],cent,k));
            }
            // 
            if(debug_flag == YES)
            {
                printf("vertice[%d] state[%g], du[%g %g]\n",
                    i, conu[0], du[0], du[1]);
            }
        }
         
        // edge mid points
        for(i = 0; i < 3; i++)
        {
            Dens(state) = 0.25 + 0.5*sin((pemid[i][0] + pemid[i][1])*PI);
            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            for(j = 0; j < N_EQN; j++)
            {
                for(k = 0; k < MAX_N_COEF; k++)
                    rhs[j][k] += 2.0/15.0*(conu[j]*vh_val(pemid[i],cent,k));
            }
        }


        // tri centriod
        for(i = 0; i < 2; i++) 
            dbcent[i] = cent[i];
        Dens(state) = 0.25 + 0.5*sin((dbcent[0] + dbcent[1])*PI);
        set_state(state,GAS_STATE,state);
        conu[0] = Dens(state);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
                rhs[j][k] += 9.0/20.0*(conu[j]*vh_val(dbcent,cent,k));
        }

        for(j = 0; j < 4; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];  
        }

        // Compute average soln
        Dens(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];

        Dens(state) /= Lmass_matrix[0][0];
}

LOCAL float ff_Burgers(
	float *crds)
{
	return (0.25 + 0.5*sin((crds[0] + crds[1])*PI));
}

/**
13 point quadrature
u(x,y,0) = 1/4 + 1/2sin(pi(x+y))
**/
LOCAL void Burgers_evolution_initializer_13_quad(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           r2, r, ep = 5.0;
        float           du[2], dT, T, S, nT, nS, nP;
        float           gam = 1.4, P;
        float           bar[3];
        int             i, j, k, indx;
        POINT   *p[3];
        float   *pcrds[3], dbcent[MAXD];
        float   emid0[3], emid1[3], emid2[3], *pemid[3], conu[4];
        double   *cent = fg_centroid(tri);
        double   rhs[4][MAX_N_COEF], mulrhs[4][MAX_N_COEF], dens[7];
        float   area = fg_area(tri), v = 0.0, t = 0.0;
        double      **mass_inv, **Lmass_matrix;
        int     debug_flag = NO;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
          w3 = 0.175615257433204,  w4 = 0.077113760890257;
        float crds[13][2];
        static double **tmp_mass_inv = NULL;

        third = 1.0/3.0; 
        // gam = adiabatic_gamma(state);

        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        Lmass_matrix = tri->Lmass_matrix;
        if(tri->mass_inv != NULL)
            mass_inv = tri->mass_inv;
        else
        {
            mass_inv = tmp_mass_inv;
            inverse_matrix(Lmass_matrix,MAX_N_COEF,mass_inv);
        }

        // Taylor expansion version
        /**
        for(i = 0; i < 2; i++)
            dbcent[i] = cent[i];
        dg_Dens(state)[0] = 0.25 + 0.5*sin((dbcent[0] + dbcent[1])*PI);
        dg_Dens(state)[1] = 0.5*cos((dbcent[0] + dbcent[1])*PI)*PI;  
        dg_Dens(state)[2] = 0.5*cos((dbcent[0] + dbcent[1])*PI)*PI;  
        dg_Dens(state)[3] = -0.5*sin((dbcent[0] + dbcent[1])*PI)*PI*PI/2.0;  
        dg_Dens(state)[4] = -0.5*sin((dbcent[0] + dbcent[1])*PI)*PI*PI;  
        dg_Dens(state)[5] = -0.5*sin((dbcent[0] + dbcent[1])*PI)*PI*PI/2.0;  
        // sin(PI*y); 
        // dg_Dens(state)[0] = 0.25 + 0.5*sin((dbcent[1])*PI);
        // dg_Dens(state)[1] = 0.0;  
        // dg_Dens(state)[2] = 0.5*cos((dbcent[1])*PI)*PI;  
        // dg_Dens(state)[3] = 0.0;  
        // dg_Dens(state)[4] = -0.0;  
        // dg_Dens(state)[5] = -0.5*sin((dbcent[1])*PI)*PI*PI/2.0;  
        // End sin(PI*y);
        Dens(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
        Dens(state) /= Lmass_matrix[0][0];
        return;
        // End Taylor expansion version
        **/

        for(i = 0; i < 4; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        crds[0][0] = (pcrds[0][0] + pcrds[1][0] + pcrds[2][0]) * third;
        crds[0][1] = (pcrds[0][1] + pcrds[1][1] + pcrds[2][1]) * third;

        crds[1][0] = b*pcrds[0][0] + a*(pcrds[1][0] + pcrds[2][0]);
        crds[1][1] = b*pcrds[0][1] + a*(pcrds[1][1] + pcrds[2][1]);
        crds[2][0] = b*pcrds[1][0] + a*(pcrds[0][0] + pcrds[2][0]);
        crds[2][1] = b*pcrds[1][1] + a*(pcrds[0][1] + pcrds[2][1]);
        crds[3][0] = b*pcrds[2][0] + a*(pcrds[0][0] + pcrds[1][0]);
        crds[3][1] = b*pcrds[2][1] + a*(pcrds[0][1] + pcrds[1][1]);

        crds[4][0] = g*pcrds[0][0] + f*(pcrds[1][0] + pcrds[2][0]);
        crds[4][1] = g*pcrds[0][1] + f*(pcrds[1][1] + pcrds[2][1]);
        crds[5][0] = g*pcrds[1][0] + f*(pcrds[0][0] + pcrds[2][0]);  
        crds[5][1] = g*pcrds[1][1] + f*(pcrds[0][1] + pcrds[2][1]);  
        crds[6][0] = g*pcrds[2][0] + f*(pcrds[1][0] + pcrds[0][0]);
        crds[6][1] = g*pcrds[2][1] + f*(pcrds[1][1] + pcrds[0][1]);

        crds[7][0] = c*pcrds[0][0] + d*pcrds[1][0] + e*pcrds[2][0];
        crds[7][1] = c*pcrds[0][1] + d*pcrds[1][1] + e*pcrds[2][1];

        crds[8][0] = d*pcrds[0][0] + c*pcrds[1][0] + e*pcrds[2][0];
        crds[8][1] = d*pcrds[0][1] + c*pcrds[1][1] + e*pcrds[2][1];
 
        crds[9][0] = d*pcrds[0][0] + e*pcrds[1][0] + c*pcrds[2][0];
        crds[9][1] = d*pcrds[0][1] + e*pcrds[1][1] + c*pcrds[2][1];

        crds[10][0] = e*pcrds[0][0] + d*pcrds[1][0] + c*pcrds[2][0];
        crds[10][1] = e*pcrds[0][1] + d*pcrds[1][1] + c*pcrds[2][1];

        crds[11][0] = e*pcrds[0][0] + c*pcrds[1][0] + d*pcrds[2][0];
        crds[11][1] = e*pcrds[0][1] + c*pcrds[1][1] + d*pcrds[2][1];

        crds[12][0] = c*pcrds[0][0] + e*pcrds[1][0] + d*pcrds[2][0];
        crds[12][1] = c*pcrds[0][1] + e*pcrds[1][1] + d*pcrds[2][1];

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = w1*ff_Burgers(crds[0])*vh_val(crds[0],cent,k) +
                      w2*(ff_Burgers(crds[1])*vh_val(crds[1],cent,k) + 
                          ff_Burgers(crds[2])*vh_val(crds[2],cent,k) +
                          ff_Burgers(crds[3])*vh_val(crds[3],cent,k)) +
                      w3*(ff_Burgers(crds[4])*vh_val(crds[4],cent,k) + 
                          ff_Burgers(crds[5])*vh_val(crds[5],cent,k) +
                          ff_Burgers(crds[6])*vh_val(crds[6],cent,k)) +
                      w4*(ff_Burgers(crds[7])*vh_val(crds[7],cent,k) + 
                          ff_Burgers(crds[8])*vh_val(crds[8],cent,k) +
                          ff_Burgers(crds[9])*vh_val(crds[9],cent,k) +
                          ff_Burgers(crds[10])*vh_val(crds[10],cent,k) + 
                          ff_Burgers(crds[11])*vh_val(crds[11],cent,k) + 
                          ff_Burgers(crds[12])*vh_val(crds[12],cent,k));
            }
        }

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);

        for(indx = 0; indx < MAX_N_COEF; indx++)
            dg_Dens(state)[indx] = mulrhs[0][indx];  

        // Compute average soln
        Dens(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];

        Dens(state) /= Lmass_matrix[0][0];
}

// init by interpolation at vertices and midpoints
LOCAL void vortex_evolution_initializer_interp(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           r2, r, ep = 5.0;
        float           du[2], dT, T, S, nT, nS, nP;
        float           gam = 1.4, P;
        float           bar[3];
        int             i, j, k, indx;
        POINT   *p[3];
        float   *pcrds[3], dbcent[MAXD];
        float   emid0[3], emid1[3], emid2[3], *pemid[3], conu[4];
        double   *cent = fg_centroid(tri);
        double   rhs[4][6], mulrhs[4][6], dens[7];
        float   area = fg_area(tri);
        int     debug_flag = NO;
        static double   **Lmass_matrix = NULL, **mass_inv;
           
        if(Lmass_matrix == NULL)
        {
            matrix(&(Lmass_matrix), 6, 6, sizeof(double));
            matrix(&(mass_inv), 6, 6, sizeof(double));
        } 

        // gam = adiabatic_gamma(state);

        if(fabs(coords[0] - 5.0) < 0.5 &&
           fabs(coords[1] - 5.0) < 0.5)
            debug_flag = YES; 

        if(tri->id == 26)
            debug_flag = YES; 

        for(i = 0; i < 4; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
        {
            emid0[i] = 0.5*(pcrds[0][i] + pcrds[1][i]);
            emid1[i] = 0.5*(pcrds[1][i] + pcrds[2][i]);
            emid2[i] = 0.5*(pcrds[2][i] + pcrds[0][i]);
        }
        pemid[0] = emid0;
        pemid[1] = emid1;
        pemid[2] = emid2;

        // vertices
        for(i = 0; i < 3; i++)
        {
            set_state(state,TGAS_STATE,state);
            Dens(state) = 1.0;
            Press(state) = 1.0;
            Vel(state)[0] = Vel(state)[1] = 1.0;
            T = 1.0;
            // S = log(Press(state)/(pow(Dens(state), gam)));
            S = (Press(state)/(pow(Dens(state), gam)));

            bar[0] = pcrds[i][0] - 5.0;
            bar[1] = pcrds[i][1] - 5.0;

            r2 = sqr(bar[0]) + sqr(bar[1]);
            du[0] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(-bar[1]);
            du[1] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(bar[0]);
            dT = -1.0*(gam - 1.0)*sqr(ep)/(8.0*gam*sqr(PI))*exp(1.0-r2);
            nT = T + dT;

            // Dens(state) = pow(exp(S)/nT, 1.0/(1.0-gam));
            dens[i] = Dens(state) = pow(nT/S, 1.0/(gam-1.0));
            Press(state) = nT*Dens(state);

            Vel(state)[0] += du[0];
            Vel(state)[1] += du[1];

            nS = Press(state)/pow(Dens(state), gam);
            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            conu[1] = Mom(state)[0];
            conu[2] = Mom(state)[1];
            conu[3] = Energy(state);
            for(j = 0; j < 4; j++)
                rhs[j][i] = conu[j];
            Lmass_matrix[i][0] = 1.0;
            Lmass_matrix[i][1] = pcrds[i][0] - cent[0];
            Lmass_matrix[i][2] = pcrds[i][1] - cent[1];
            Lmass_matrix[i][3] = sqr(pcrds[i][0] - cent[0]);
            Lmass_matrix[i][4] = (pcrds[i][0] - cent[0])*(pcrds[i][1] - cent[1]);
            Lmass_matrix[i][5] = sqr(pcrds[i][1] - cent[1]);
        }
         
        // edge mid points
        for(i = 0; i < 3; i++)
        {
            set_state(state,TGAS_STATE,state);
            Dens(state) = 1.0;
            Press(state) = 1.0;
            Vel(state)[0] = Vel(state)[1] = 1.0;
            T = 1.0;
            // S = log(Press(state)/(pow(Dens(state), gam)));
            S = (Press(state)/(pow(Dens(state), gam)));

            bar[0] = pemid[i][0] - 5.0;
            bar[1] = pemid[i][1] - 5.0;

            r2 = sqr(bar[0]) + sqr(bar[1]);
            du[0] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(-bar[1]);
            du[1] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(bar[0]);
            dT = -1.0*(gam - 1.0)*sqr(ep)/(8.0*gam*sqr(PI))*exp(1.0-r2);
            nT = T + dT;

            // Dens(state) = pow(exp(S)/nT, 1.0/(1.0-gam));
            dens[i+3] = Dens(state) = pow(nT/S, 1.0/(gam-1.0));
            Press(state) = nT*Dens(state);

            Vel(state)[0] += du[0];
            Vel(state)[1] += du[1];

            nS = Press(state)/pow(Dens(state), gam);
            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            conu[1] = Mom(state)[0];
            conu[2] = Mom(state)[1];
            conu[3] = Energy(state);
            for(j = 0; j < 4; j++)
            {
                rhs[j][i+3] = conu[j];
            }
            Lmass_matrix[i+3][0] = 1.0;
            Lmass_matrix[i+3][1] = pemid[i][0] - cent[0];
            Lmass_matrix[i+3][2] = pemid[i][1] - cent[1];
            Lmass_matrix[i+3][3] = sqr(pemid[i][0] - cent[0]);
            Lmass_matrix[i+3][4] = (pemid[i][0] - cent[0])*(pemid[i][1] - cent[1]);
            Lmass_matrix[i+3][5] = sqr(pemid[i][1] - cent[1]);
        }

        inverse_matrix(Lmass_matrix,6,mass_inv);

        for(i = 0; i < 4; i++)
            matrix_vec_mult(mass_inv, rhs[i], 6, 6, mulrhs[i]);

        for(indx = 0; indx < 6; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];  
            dg_Mom(state)[0][indx] = mulrhs[1][indx];  
            dg_Mom(state)[1][indx] = mulrhs[2][indx];  
            dg_Energy(state)[indx] = mulrhs[3][indx];  
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*tri->Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*tri->Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*tri->Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*tri->Lmass_matrix[0][indx];
        }
        Dens(state) /= tri->Lmass_matrix[0][0];
        Mom(state)[0] /= tri->Lmass_matrix[0][0];
        Mom(state)[1] /= tri->Lmass_matrix[0][0];
        Energy(state) /= tri->Lmass_matrix[0][0];

        if(debug_flag == YES)
        {
            printf("vortex_evolution_initializer coords[%g %g], tri(%d)\n",
                         coords[0], coords[1], tri->id);
            printf("pert u(%g %g), dT = %g\n", du[0], du[1], dT);
            printf("entropy change = %g, gamma = %g, S = %g, T = %g, rho = %g, nP = %g\n\n",
                 (S - nS), gam, S, T, Dens(state), nP);
            g_verbose_print_state(state);
            // con_u_at_pt(state, dbcent, cent, conu); 
            printf("cent-dens: %14.13g, vertex: ", conu[0]);
            for(i = 0; i < 3; i++)
            {
                // con_u_at_pt(state, pcrds[i], cent, conu); 
                printf("%g, ", conu[0]);
            }
            printf("\n");
            printf("pt_dens(%g, %g, %g, %g, %g, %g, %g)\n", dens[0], dens[1], 
                      dens[2], dens[3], dens[4],dens[5],dens[6]);
            printf("total mass %g, Proj mass %g\n", rhs[0][0], Dens(state)*area);
            // clean_up(0);
        }
}

// init by approx. at vertices, midpoints and centroid
LOCAL void vortex_evolution_initializer_least_sqr(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           r2, r, ep = 5.0;
        float           du[2], dT, T, S, nT, nS, nP;
        float           gam = 1.4, P;
        float           bar[3];
        int             i, j, k, indx;
        POINT   *p[3];
        float   *pcrds[3], dbcent[MAXD];
        float   emid0[3], emid1[3], emid2[3], *pemid[3], conu[4];
        double   *cent = fg_centroid(tri);
        double   rhs[4][7], mulrhs[4][7], mulrhs2[4][7], dens[7];
        float   area = fg_area(tri);
        int     debug_flag = NO;
        static double   **Lmass_matrix = NULL, **mass_tran, **AA, **inv;
           
        if(Lmass_matrix == NULL)
        {
            matrix(&(Lmass_matrix), 7, 7, sizeof(double));
            matrix(&(mass_tran), 7, 7, sizeof(double));
            matrix(&(AA), 7, 7, sizeof(double));
            matrix(&(inv), 7, 7, sizeof(double));
        } 

        // gam = adiabatic_gamma(state);

        if(fabs(coords[0] - 5.0) < 0.5 &&
           fabs(coords[1] - 5.0) < 0.5)
            debug_flag = YES; 

        if(tri->id == 26)
            debug_flag = YES; 

        for(i = 0; i < 4; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
        {
            emid0[i] = 0.5*(pcrds[0][i] + pcrds[1][i]);
            emid1[i] = 0.5*(pcrds[1][i] + pcrds[2][i]);
            emid2[i] = 0.5*(pcrds[2][i] + pcrds[0][i]);
        }
        pemid[0] = emid0;
        pemid[1] = emid1;
        pemid[2] = emid2;

        // vertices
        for(i = 0; i < 3; i++)
        {
            set_state(state,TGAS_STATE,state);
            Dens(state) = 1.0;
            Press(state) = 1.0;
            Vel(state)[0] = Vel(state)[1] = 1.0;
            T = 1.0;
            // S = log(Press(state)/(pow(Dens(state), gam)));
            S = (Press(state)/(pow(Dens(state), gam)));

            bar[0] = pcrds[i][0] - 5.0;
            bar[1] = pcrds[i][1] - 5.0;

            r2 = sqr(bar[0]) + sqr(bar[1]);
            du[0] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(-bar[1]);
            du[1] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(bar[0]);
            dT = -1.0*(gam - 1.0)*sqr(ep)/(8.0*gam*sqr(PI))*exp(1.0-r2);
            nT = T + dT;

            // Dens(state) = pow(exp(S)/nT, 1.0/(1.0-gam));
            dens[i] = Dens(state) = pow(nT/S, 1.0/(gam-1.0));
            Press(state) = nT*Dens(state);

            Vel(state)[0] += du[0];
            Vel(state)[1] += du[1];

            nS = Press(state)/pow(Dens(state), gam);
            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            conu[1] = Mom(state)[0];
            conu[2] = Mom(state)[1];
            conu[3] = Energy(state);
            for(j = 0; j < 4; j++)
                rhs[j][i] = conu[j];
            Lmass_matrix[i][0] = 1.0;
            Lmass_matrix[i][1] = pcrds[i][0] - cent[0];
            Lmass_matrix[i][2] = pcrds[i][1] - cent[1];
            Lmass_matrix[i][3] = sqr(pcrds[i][0] - cent[0]);
            Lmass_matrix[i][4] = (pcrds[i][0] - cent[0])*(pcrds[i][1] - cent[1]);
            Lmass_matrix[i][5] = sqr(pcrds[i][1] - cent[1]);
        }
         
        // edge mid points
        for(i = 0; i < 3; i++)
        {
            set_state(state,TGAS_STATE,state);
            Dens(state) = 1.0;
            Press(state) = 1.0;
            Vel(state)[0] = Vel(state)[1] = 1.0;
            T = 1.0;
            // S = log(Press(state)/(pow(Dens(state), gam)));
            S = (Press(state)/(pow(Dens(state), gam)));

            bar[0] = pemid[i][0] - 5.0;
            bar[1] = pemid[i][1] - 5.0;

            r2 = sqr(bar[0]) + sqr(bar[1]);
            du[0] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(-bar[1]);
            du[1] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(bar[0]);
            dT = -1.0*(gam - 1.0)*sqr(ep)/(8.0*gam*sqr(PI))*exp(1.0-r2);
            nT = T + dT;

            // Dens(state) = pow(exp(S)/nT, 1.0/(1.0-gam));
            dens[i+3] = Dens(state) = pow(nT/S, 1.0/(gam-1.0));
            Press(state) = nT*Dens(state);

            Vel(state)[0] += du[0];
            Vel(state)[1] += du[1];

            nS = Press(state)/pow(Dens(state), gam);
            set_state(state,GAS_STATE,state);
            conu[0] = Dens(state);
            conu[1] = Mom(state)[0];
            conu[2] = Mom(state)[1];
            conu[3] = Energy(state);
            for(j = 0; j < 4; j++)
            {
                rhs[j][i+3] = conu[j];
            }
            Lmass_matrix[i+3][0] = 1.0;
            Lmass_matrix[i+3][1] = pemid[i][0] - cent[0];
            Lmass_matrix[i+3][2] = pemid[i][1] - cent[1];
            Lmass_matrix[i+3][3] = sqr(pemid[i][0] - cent[0]);
            Lmass_matrix[i+3][4] = (pemid[i][0] - cent[0])*(pemid[i][1] - cent[1]);
            Lmass_matrix[i+3][5] = sqr(pemid[i][1] - cent[1]);
        }

        // tri centriod
        for(i = 0; i < 2; i++)
            dbcent[i] = cent[i];
        set_state(state,TGAS_STATE,state);
        Dens(state) = 1.0;
        Press(state) = 1.0;
        Vel(state)[0] = Vel(state)[1] = 1.0;
        T = 1.0;
        // S = log(Press(state)/(pow(Dens(state), gam)));
        S = (Press(state)/(pow(Dens(state), gam)));

        bar[0] = dbcent[0] - 5.0;
        bar[1] = dbcent[1] - 5.0;

        r2 = sqr(bar[0]) + sqr(bar[1]);
        du[0] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(-bar[1]);
        du[1] = ep/(2.0*PI)*exp(0.5*(1.0-r2))*(bar[0]);
        dT = -1.0*(gam - 1.0)*sqr(ep)/(8.0*gam*sqr(PI))*exp(1.0-r2);
        nT = T + dT;

        // Dens(state) = pow(exp(S)/nT, 1.0/(1.0-gam));
        dens[6] = Dens(state) = pow(nT/S, 1.0/(gam-1.0));
        Press(state) = nT*Dens(state);

        Vel(state)[0] += du[0];
        Vel(state)[1] += du[1];

        nS = Press(state)/pow(Dens(state), gam);
        set_state(state,GAS_STATE,state);
        nP = pressure(state);
        conu[0] = Dens(state);
        conu[1] = Mom(state)[0];
        conu[2] = Mom(state)[1];
        conu[3] = Energy(state);
        for(j = 0; j < 4; j++)
            rhs[j][6] = conu[j];

        Lmass_matrix[6][0] = 1.0;
        Lmass_matrix[6][1] = 0.0;
        Lmass_matrix[6][2] = 0.0;
        Lmass_matrix[6][3] = 0.0;
        Lmass_matrix[6][4] = 0.0;
        Lmass_matrix[6][5] = 0.0;

        trans_matrix(Lmass_matrix, 7, 6, mass_tran);
        matrix_matrix_mult(mass_tran, Lmass_matrix, 7, 7, AA); 

        inverse_matrix(AA,6,inv);

        for(i = 0; i < 4; i++)
            matrix_vec_mult(mass_tran, rhs[i], 6, 7, mulrhs2[i]);

        for(i = 0; i < 4; i++)
            matrix_vec_mult(inv, mulrhs2[i], 6, 6, mulrhs[i]);

        for(indx = 0; indx < 6; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];  
            dg_Mom(state)[0][indx] = mulrhs[1][indx];  
            dg_Mom(state)[1][indx] = mulrhs[2][indx];  
            dg_Energy(state)[indx] = mulrhs[3][indx];  
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*tri->Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*tri->Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*tri->Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*tri->Lmass_matrix[0][indx];
        }
        Dens(state) /= tri->Lmass_matrix[0][0];
        Mom(state)[0] /= tri->Lmass_matrix[0][0];
        Mom(state)[1] /= tri->Lmass_matrix[0][0];
        Energy(state) /= tri->Lmass_matrix[0][0];

        if(debug_flag == YES)
        {
            printf("vortex_evolution_initializer coords[%g %g], tri(%d)\n",
                         coords[0], coords[1], tri->id);
            printf("pert u(%g %g), dT = %g\n", du[0], du[1], dT);
            printf("entropy change = %g, gamma = %g, S = %g, T = %g, rho = %g, nP = %g\n\n",
                 (S - nS), gam, S, T, Dens(state), nP);
            g_verbose_print_state(state);
            // con_u_at_pt(state, dbcent, cent, conu); 
            printf("cent-dens: %14.13g, vertex: ", conu[0]);
            for(i = 0; i < 3; i++)
            {
                // con_u_at_pt(state, pcrds[i], cent, conu); 
                printf("%g, ", conu[0]);
            }
            printf("\n");
            printf("pt_dens(%g, %g, %g, %g, %g, %g, %g)\n", dens[0], dens[1], 
                      dens[2], dens[3], dens[4],dens[5],dens[6]);
            printf("total mass %g, Proj mass %g\n", rhs[0][0], Dens(state)*area);
            // clean_up(0);
        }
}

/* ARGSUSED */
LOCAL void g_intfc_initializer(
	POINT		   *p,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	   *hs,
	Locstate	   lstate,
	Locstate	   rstate,
	INIT_DATA	   *init)
{
	COMPONENT      pcomp = NO_COMP, ncomp = NO_COMP;
	COMP_TYPE      *nct, *pct;
	INTERFACE      *intfc = hs->interface;
	int            stype = type_of_state(init);
	int	       typel, typer;
	int	       dim = intfc->dim;

	debug_print("init_states","Entered g_intfc_initializer()\n");

	if (debugging("init_states"))
	{
	    (void) printf("hs = %llu, hse = %llu ",
	    	          hypersurface_number(hs),
			  hypersurface_element_number(hse,intfc));
	    print_general_vector("Coords(p) = ",Coords(p),dim,"\n");
	}

	ncomp = negative_component(hs);
	nct = comp_type(ncomp);
	pcomp = positive_component(hs);
	pct = comp_type(pcomp);
	switch(wave_type(hs))
	{
	case FORWARD_SOUND_WAVE_LE:
	case BACKWARD_SOUND_WAVE_TE:
	    /* Ambient side == right */
	    Get_state(Coords(p),rstate,pct,hs,intfc,init,stype);
	    copy_state(lstate,rstate);
	    goto leave;

	case FORWARD_SOUND_WAVE_TE:
	case BACKWARD_SOUND_WAVE_LE:
	    /* Ambient side == left */
	    Get_state(Coords(p),lstate,nct,hs,intfc,init,stype);
	    copy_state(rstate,lstate);
	    goto leave;

	case REFLECTION_BOUNDARY:
	case SUBDOMAIN_BOUNDARY:
	case PASSIVE_BOUNDARY:
	    obstacle_state(intfc,lstate,size_of_state(intfc));
	    obstacle_state(intfc,rstate,size_of_state(intfc));
	    goto leave;

	default:
	    break;
	}

	typel = nct->type;
	if (typel != EXTERIOR)
	{
	    switch(typel)
	    {
	    case PRANDTL_MEYER_WAVE:
	    case UNTRACKED_SHOCK:
#if defined(TWOD)
	    	if (dim == 2)
	    	{
	    	    CURVE *c = Curve_of_hs(hs);
	            BOND *b = Bond_of_hse(hse);
		    if (p == c->start->posn)
			Get_state(Coords(b->end),lstate,nct,hs,
				  intfc,init,stype);
		    else if (p == c->end->posn)
			Get_state(Coords(b->start),lstate,nct,hs,
				  intfc,init,stype);
		    else
			Get_state(Coords(p),lstate,nct,hs,intfc,init,stype);
		}
		else
#endif /* defined(TWOD) */
		    Get_state(Coords(p),lstate,nct,hs,
			      intfc,init,stype);
		break;
#if defined(TWOD)
	    case BUBBLE:
	    	debug_print("init_states","BUBBLE\n");
		break;
#endif /* defined(TWOD) */
	    default:
		Get_state(Coords(p),lstate,nct,hs,intfc,init,stype);
	    }
	}

	typer = pct->type;
	if (typer != EXTERIOR)
	{
	    switch(typer)
	    {
	    case PRANDTL_MEYER_WAVE:
	    case UNTRACKED_SHOCK:
	    	if (dim == 2)
	    	{
	    	    CURVE *c = Curve_of_hs(hs);
		    BOND *b = Bond_of_hse(hse);
		    if (p == c->start->posn)
		        Get_state(Coords(b->end),rstate,pct,hs,
				  intfc,init,stype);
		    else if (p == c->end->posn)
			Get_state(Coords(b->start),rstate,pct,hs,
				  intfc,init,stype);
		    else
			Get_state(Coords(p),rstate,pct,hs,intfc,init,stype);
		}
		else
		    Get_state(Coords(p),rstate,pct,hs,intfc,init,stype);
		break;
#if defined(TWOD)
	    case BUBBLE:
	    	debug_print("init_states","BUBBLE\n");
		break;
#endif /* defined(TWOD) */
	    default:
		Get_state(Coords(p),rstate,pct,hs,intfc,init,stype);
	    }
	}

	if ((typel == EXTERIOR) || (typel == OBSTACLE))
	{
	    switch (wave_type(hs)) 
	    {
	    case DIRICHLET_BOUNDARY:
	    case NEUMANN_BOUNDARY:
	    case PASSIVE_BOUNDARY:
	    case REFLECTION_BOUNDARY:
	    case SUBDOMAIN_BOUNDARY:
	        obstacle_state(intfc,lstate,size_of_state(intfc));
	        break;

	    default:
	        screen("ERROR in g_intfc_initializer(), "
	               "unknown bdry type\n");
	        print_wave_type("typel == EXTERIOR, wave_type = ",
	        	wave_type(hs),"\n",intfc);
	        print_comp_types(intfc);
	        print_hypersurface(hs);
	        print_interface(intfc);
	        clean_up(ERROR);
	    }
	}

	if ((typer == EXTERIOR) || (typer == OBSTACLE))
	{
	    switch (wave_type(hs)) 
	    {
	    case DIRICHLET_BOUNDARY:
	    case NEUMANN_BOUNDARY:
	    case PASSIVE_BOUNDARY:
	    case REFLECTION_BOUNDARY:
	    case SUBDOMAIN_BOUNDARY:
	    	obstacle_state(intfc,rstate,size_of_state(intfc));
	    	break;

	    default:
	    	screen("ERROR in g_intfc_initializer(), "
	    	       "unknown bdry type\n");
	    	print_wave_type("typer == EXTERIOR, wave_type = ",
	    		wave_type(hs),"\n",intfc);
	    	print_comp_types(intfc);
	    	print_hypersurface(hs);
	    	print_interface(intfc);
	    	clean_up(ERROR);
	    }
	}

leave:
	if (debugging("init_states"))
	{
	    print_general_vector("negative side state at ",Coords(p),
				 intfc->dim,"");
	    (void) printf(" on hs %llu\n",hypersurface_number(hs));
	    print_gas_state(lstate);
	    print_general_vector("positive side state at ",Coords(p),
			         intfc->dim,"");
	    (void) printf(" on hs %llu\n",hypersurface_number(hs));
	    print_gas_state(rstate);
	}
	debug_print("init_states","Left g_intfc_initializer()\n");
}		/*end g_intfc_initializer*/


LOCAL	void	print_comp_types(
	INTERFACE	*intfc)
{
	HYPER_SURF	**hs;
	COMPONENT	*comps;
	COMPONENT	comp;
	int		len = max_component(intfc) - min_component(intfc) + 1;
	int		i, num_comps;

	vector(&comps,len,sizeof(COMPONENT));
	for (num_comps = 0, hs = intfc->hss; hs && *hs; hs++)
	{
	    comp = positive_component(*hs);
	    for (i = 0; i < num_comps; i++)
	    	if (comp == comps[i]) break;
	    if (i == num_comps)
	    	comps[num_comps++] = comp;
	    comp = negative_component(*hs);
	    for (i = 0; i < num_comps; i++)
	    	if (comp == comps[i]) break;
	    if (i == num_comps)
	    	comps[num_comps++] = comp;
	}
	for (i = 0; i < num_comps; i++)
	{
	    (void) printf("Type of component %d = ",comps[i]);
	    print_component_type("",comp_type(comps[i]),"\n");
	}
	free(comps);
}		/*end print_comp_types*/

EXPORT	const char *comp_type_name(
	COMP_TYPE_TYPE type)
{
	static char s[512];
	switch (type)
	{
	case EXTERIOR:
	    return "EXTERIOR";
	case OBSTACLE:
	    return "OBSTACLE";
	case AMBIENT:
	    return "AMBIENT";
	case ELLIPTICAL:
	    return "ELLIPTICAL";
	case KH_SINE_PERTURBED:
	    return "KH_SINE_PERTURBED";
	case NWAVE:
	    return "NWAVE";
	case RANDOM_SURFACE_PERTURBED:
	    return "RANDOM_SURFACE_PERTURBED";
	case BUBBLE:
	    return "BUBBLE";
	case PRANDTL_MEYER_WAVE:
	    return "PRANDTL_MEYER_WAVE";
	case UNTRACKED_SHOCK:
	    return "UNTRACKED_SHOCK";
	case RESTART:
	    return "RESTART";
	case TAYLOR_WAVE:
	    return "TAYLOR_WAVE";
	case RT_KH:
	    return "RT_KH";
	case RT_PERTURBED:
	    return "RT_PERTURBED";
	case KH_PERTURBED:
	    return "KH_PERTURBED";
	case RAREFACTION_WAVE_1D:
	    return "RAREFACTION_WAVE_1D";
	case TRANS_LAYER:
	    return "TRANS_LAYER";
	case STRETCHING:
	    return "STRETCHING";
	case ONE_DIMENSIONAL_OVERLAY:
	    return "ONE_DIMENSIONAL_OVERLAY";
	default:
	    (void) sprintf(s,"UNKNOWN COMPONENT TYPE %d",type);
	    return s;
	}
}		/*end comp_type_name*/

LOCAL	void	print_component_type(
	const char *mesg1,
	COMP_TYPE  *comp_type,
	const char *mesg2)
{
	if (comp_type == NULL)
	    return;
	(void) printf("%s%s%s",mesg1,comp_type_name(comp_type->type),mesg2);
}		/*end print_component_type*/

LOCAL   void init_ambient(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	Gas_param	*params;

	params = init_eos_params(init,ip,"",YES);

	prompt_for_ambient_state(comp_type(FIRST_DYNAMIC_COMPONENT),params,
			         " interior",ip->root->front,init);
}		/*end init_ambient*/


/*ARGSUSED*/
LOCAL	void	init_tri_grid_test(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	COMPONENT	comp;
	Front		*front = ip->root->front;
	INTERFACE	*intfc = front->interf;

	for (comp = min_component(intfc); comp <= max_component(intfc); comp++)
	{
	    set_obstacle_comp_type(comp_type(comp),front);
	}
}		/*end init_tri_grid_test*/

/*ARGSUSED*/
EXPORT	void	g_prompt_for_ref_state(
	const char *message,
	Locstate   state,
	int	   st_type,
	Gas_param  *params,
	INIT_DATA  *init)
{
	int	   i, dim = params->dim;
	static const char vname[3][3] = {"vx","vy","vz"};

	Init_params(state,params);
	set_type_of_state(state,TGAS_STATE);
#if defined(COMBUSTION_CODE)
	if (params->composition_type == ZND) 
	{
	    screen("Enter the gas state%s\n",message);
	    screen("\t\trho, pr, reaction progress");
	    for (i = 0; i < dim; i++)
		screen(", %s",vname[i]);
	    screen(": ");
	    (void) Scanf("%f %f %f",&Dens(state),&Press(state),
	    	     &React(state));
	    for (i = 0; i < dim; i++)
		(void) Scanf("%f",&Vel(state)[i]);
	    (void) Scanf("\n");
	}
	else
	{
	    screen("Enter the gas state%s\n\t\trho, pr",message);
	    for (i = 0; i < dim; i++)
		screen(", %s",vname[i]);
	    screen(": ");
	    (void) Scanf("%f %f",&Dens(state),&Press(state));
	    for (i = 0; i < dim; i++)
		(void) Scanf("%f",&Vel(state)[i]);
	    (void) Scanf("\n");
	    prompt_for_burning(&Params(state),message);
	}
#else /* defined(COMBUSTION_CODE) */
	screen("Enter the gas state%s\n\t\trho, pr",message);
	for (i = 0; i < dim; i++)
	    screen(", %s",vname[i]);
	screen(": ");
	(void) Scanf("%f %f",&Dens(state),&Press(state));
	for (i = 0; i < dim; i++)
	    (void) Scanf("%f",&Vel(state)[i]);
	(void) Scanf("\n");
#endif /* defined(COMBUSTION_CODE) */

	Dens(state)=density(state); /* set correct density if fuel */
	screen("Density = %"FFMT" for pressure = %"FFMT"\n"
	       ,Dens(state),Press(state));

	if (st_type != state_type(state))
	    set_state(state,st_type,state);
}		/*end g_prompt_for_ref_state*/


EXPORT void prompt_for_ambient_state(
	COMP_TYPE	*comp_type,
	Gas_param	*params,
	const char	*message,
	Front		*front,
	INIT_DATA	*init)
{
 	size_t		sizest = params->sizest;
	Locstate	state,tstate;


	(*params->_alloc_state)(&tstate,sizest);
	prompt_for_ref_state(message,tstate,TGAS_STATE,params,init);

	set_ambient_comp_type(comp_type,front);
	state = Ambient(comp_type);
	set_state(state,GAS_STATE,tstate);

	if (debugging("init")) 
	    verbose_print_state("ambient",state);
	free(tstate);
}		/*end prompt_for_ambient_state*/

EXPORT	void	set_ambient_comp_type(
	COMP_TYPE	*comp_type,
	Front		*front)
{
	if (comp_type->type == AMBIENT)	/*ALREADY DONE*/
	    return;

	if (comp_type->free_comp_type_extra != NULL)
	    (*comp_type->free_comp_type_extra)(comp_type);

	comp_type->type = AMBIENT;
	if ((front != NULL) && (comp_type->extra == NULL))
	{
	    Locstate	state;
	    if (front->sizest != 0)
	    	alloc_state(front->interf,&state,front->sizest);
	    comp_type->extra = (POINTER)state;
	}
	comp_type->_get_state = get_state_ambient;
	comp_type->free_comp_type_extra = free_ambient_comp_type;
}		/*end set_ambient_comp_type*/

LOCAL	void	free_ambient_comp_type(
	COMP_TYPE	*comp_type)
{
	if (comp_type->type != AMBIENT)
	    return;
	if (comp_type->extra != NULL)
	    free(comp_type->extra);
	comp_type->extra = NULL;
}		/*end free_ambient_comp_type*/

LOCAL	void	set_exterior_comp_type(
	COMP_TYPE	*comp_type)
{
	if (comp_type->type == EXTERIOR)
	    return;

	if (comp_type->free_comp_type_extra != NULL)
	    (*comp_type->free_comp_type_extra)(comp_type);

	comp_type->type = EXTERIOR;
	comp_type->extra = NULL;
	comp_type->_get_state = get_state_exterior;
	comp_type->free_comp_type_extra = NULL;
}		/*end set_exterior_comp_type*/

/*ARGSUSED*/
LOCAL	void	get_state_exterior(
	float           *coords,
	Locstate        s,
	COMP_TYPE       *ct,
	HYPER_SURF	*hs,
	INTERFACE	*intfc,
	INIT_DATA	*init,
	int		stype)
{
	g_obstacle_state(s,g_sizest());
        /** 
	screen("ERROR in get_state_exterior(),  EXTERIOR comp found\n");
	print_general_vector("coords = ",coords,intfc->dim,"\n");
	(void) printf("comp = %d\n",ct->comp);
	print_interface(intfc);
	clean_up(ERROR);
        **/
}		/*end get_state_exterior*/

EXPORT	void	set_obstacle_comp_type(
	COMP_TYPE	*comp_type,
	Front		*front)
{
	if (comp_type->type == OBSTACLE)
	    return;

	if (comp_type->free_comp_type_extra != NULL)
	    (*comp_type->free_comp_type_extra)(comp_type);

	comp_type->type = OBSTACLE;
	comp_type->extra = NULL;
	comp_type->_get_state = get_state_obstacle;
	comp_type->free_comp_type_extra = NULL;
	if (front != NULL)
	{
	    COMPONENT	comp = comp_type->comp;
	    exclude_comp(comp,front->interf);
	    (void)SetConstantFlowRegion(comp,return_obst_state(),front->interf);
	}
}		/*end set_obstacle_comp_type*/

/*ARGSUSED*/
LOCAL	void	get_state_obstacle(
	float           *coords,
	Locstate        s,
	COMP_TYPE       *ct,
	HYPER_SURF	*hs,
	INTERFACE	*intfc,
	INIT_DATA	*init,
	int             stype)
{
	g_obstacle_state(s,g_sizest());
}		/*end get_state_obstacle*/


#if defined(ROTATIONAL_SYMMETRY)
/* 
*			prompt_for_cylindrical_pencil():
*
*	In cylindrical geometry, a "pencil" near the axis is used.
*	To prevent curves from being constructed in the pencil,
*	RL_eff is set to the outer edge of the pencil.
*	TODO: 
*	A better solution would be to create curves in the whole region
*	and have make_vertical_axis_curve() delete all curves in the 
*	pencil.
*/

LOCAL	void	prompt_for_cylindrical_pencil(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	RECT_GRID *gr = ip->root->front->rect_grid;
	float     sfactor;
	char	  s[Gets_BUF_SIZE];

	RL_eff(init) = gr->GL[0];
	if (coord_system() != CYLINDRICAL)
	    return;

	if (RL_eff(init) != 0.0) /*User requests positive lower boundary*/
	    return;

        screen("\nRequest Neumann Wall to cut off zero radius  [y,n(dflt)]: ");
        (void) Gets(s);
        if(s[0] != 'y' && s[0] != 'Y')
            return;

        sfactor = 1.002;/*TOLERANCE*/
        screen("Enter a nonzero, positive, non-integer "
               "amount for the number\n");
        screen("of grid-blocks the wall is away "
               "from the center (default=%g): ",sfactor);

        (void) Gets(s);
        if (s[0] != '\0')
        {
	    float rtol = 0.000001;/*TOLERANCE*/
            (void) sscan_float(s, &sfactor);
            if(sfactor <= 0.0)
            {
                screen("ERROR in prompt_for_cylindrical_pencil(), "
                       "The factor chosen %g is not positive\n",sfactor);
                clean_up(ERROR);
            }
            else if (sfactor-floor(sfactor) < rtol)/*TOLERANCE*/
            {
                (void) printf("WARNING in prompt_for_cylindrical_pencil(), "
                              "The factor chosen %g is almost integer\n",
                              sfactor);
	        sfactor += rtol;
            }
        }

        RL_eff(init) = sfactor*cell_width(0,0,gr);
	RL_eff(init) = pos_radius(RL_eff(init),gr);

        if (RL_eff(init) < gr->GL[0])
	    RL_eff(init) = gr->GL[0];
}		/*end prompt_for_cylindrical_pencil*/
#endif /* defined(ROTATIONAL_SYMMETRY) */

#if defined(ONED)


LOCAL   void init_oned_density_step(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	DENSITY_STEP_WALL_DATA   *wdata;
	Grid                     *grid = ip->root->grid;
	Front                    *front = ip->root->front;
	Printplot                *prt = ip->prt;
	INTERFACE                *intfc = front->interf;
	RECT_GRID                *gr = front->rect_grid;
	COMP_TYPE                *ctype[3];
	POINT                    *pc, *ps;
	Locstate                 initial_state[3];
	Locstate                 state[7];
	Locstate                 left, right;
	bool                     isforward;
	char                     buf[Gets_BUF_SIZE];
	float                    pml, pmr, uml, umr, ml, mr;
	float                    c, s;
	float                    L, U, sL, sU;
	float                    W[9];
	float                    DX, dx;
	float                    center[3], r[3];
	float                    sgn;
	float                    dWall;
	int                      rside;
	int                      n[3];
	int                      N;
	int                      i;
	int                      dim = gr->dim;
	RIEMANN_SOLVER_WAVE_TYPE rswtype[9];
	RIEMANN_SOLVER_WAVE_TYPE l_wave, r_wave;
	int                      w_type;
	size_t                   sizest = front->sizest;
	static float             nor[] = {1.0, 0.0, 0.0};

	if (dim != 1)
	{
	    screen("ERROR in init_oned_density_step(), "
	           "dimension %d not supported\n",dim);
	    clean_up(ERROR);
	}

	ctype[0] = comp_type(FIRST_DYNAMIC_COMPONENT);
	ctype[1] = comp_type(FIRST_DYNAMIC_COMPONENT+1);
	ctype[2] = comp_type(FIRST_DYNAMIC_COMPONENT+2);
	for (i = 0; i < 3; ++i)
	{
	    set_ambient_comp_type(ctype[i],front);
	    initial_state[i] = Ambient(ctype[i]);
	}

	(void) prompt_for_eos_params(init,ip,YES,"");
	ctype[0]->params = prompt_for_eos_params(init,ip,YES,
	                                         "of the target");
	isforward = NO;
	screen("Enter propagation direction of the incident shock "
	       "(dflt = %s): ",(isforward)?"left to right":"right to left");
	(void) Gets(buf);
	if ((buf[0] == 'R') || (buf[0] == 'r'))
	    isforward = NO;
	else if ((buf[0] == 'L') || (buf[0] == 'l'))
	    isforward = YES;
	if (isforward)
	{
	    sgn = 1.0;
	    rside = 1;
	}
	else
	{
	    sgn = -1.0;
	    rside = 0;
	}
	screen("Enter the density and pressure of the target: ");
	Init_params(initial_state[0],ctype[0]->params);
	set_type_of_state(initial_state[0],TGAS_STATE);
	(void) Scanf("%f %f\n",
	             &Dens(initial_state[0]),&Press(initial_state[0]));
	zero_state_velocity(initial_state[0],dim);

	ctype[1]->params = prompt_for_eos_params(init,ip,YES,
	                                         "ahead of the incident shock");
	prompt_for_behind_contact_state(initial_state[0],initial_state[1],
	                                ctype[1]->params,NO,nor,GAS_STATE,init);

	ctype[2]->params = ctype[1]->params;
	prompt_for_behind_shock_state(initial_state[1],initial_state[2],
				      NO,nor,GAS_STATE,isforward,init);
	W[0] = 0.0;
	W[1] = sgn*mass_flux(pressure(initial_state[2]),initial_state[1])/
	           Dens(initial_state[1]);

	L = gr->L[0];
	U = gr->U[0];
	screen("Enter the initial contact position c (%g < c < %g): ",L,U);
	(void) Scanf("%f\n",&c);
	if ((c <= L) || (U <= c))
	{
	    screen("ERROR in init_oned_density_step(), "
	           "initial contact position %g out of range\n",c);
	    clean_up(ERROR);
	}
	if (isforward)
	{
	    sL = L;
	    sU = c;
	    dWall = c - U;
	}
	else
	{
	    sL = c;
	    sU = U;
	    dWall = c - L;
	}
	screen("Enter the initial shock position s (%g < s < %g): ",sL,sU);
	(void) Scanf("%f\n",&s);
	if ((s <= sL) || (sU <= s))
	{
	    screen("ERROR in init_oned_density_step(), "
	           "initial shock position %g out of range\n",s);
	    clean_up(ERROR);
	}

	rect_boundary_type(intfc,0,rside) = REFLECTION_BOUNDARY;
	screen("Use Neumman (N) or reflecting (R) "
	       "boundaries on the %s (dflt = %s): ",(isforward)?"right":"left",
	       (rect_boundary_type(intfc,0,1) == REFLECTION_BOUNDARY) ?
	       "R" : "N");
	(void) Gets(buf);
	if ((buf[0] == 'R') || (buf[0] == 'r'))
	    rect_boundary_type(intfc,0,rside) = REFLECTION_BOUNDARY;
	else if ((buf[0] == 'N') || (buf[0] == 'n'))
	    rect_boundary_type(intfc,0,rside) = NEUMANN_BOUNDARY;

	pc = (isforward) ? make_point(&c,ctype[1]->comp,ctype[0]->comp) :
	                   make_point(&c,ctype[0]->comp,ctype[1]->comp);
	wave_type(pc) = CONTACT;
	if (ctype[0]->params == ctype[1]->params)
	{
	    screen("Type y to track the contact: ");
	    (void) Gets(buf);
	    if ((buf[0] != 'y') && (buf[0] != 'Y'))
	        untracked_hyper_surf(pc) = YES;
	}
	ps = (isforward) ? make_point(&s,ctype[2]->comp,ctype[1]->comp) :
	                   make_point(&s,ctype[1]->comp,ctype[2]->comp);
	wave_type(ps) = (isforward) ? FORWARD_SHOCK_WAVE : BACKWARD_SHOCK_WAVE;
	screen("Type y to track the incident shock: ");
	(void) Gets(buf);
	if ((buf[0] != 'y') && (buf[0] != 'Y'))
	    untracked_hyper_surf(ps) = YES;

	DX = 3.0*gr->h[0];
	N = 5;
	screen("Enter an averaging length for the "
	       "pressure near the wall (dflt = %g): ",DX);
	(void) Gets(buf);
	if (buf[0] != '\0')
	{
	    if (sscan_float(buf,&DX) != 1)
	    {
	        screen("ERROR in init_oned_density_step(), improper input "
	               "of pressure averaging length\n");
	        clean_up(ERROR);
	    }
	}
	screen("Enter the number of pressure samples per averaging length "
	       "(dflt = %d): ",N);
	(void) Gets(buf);
	if (buf[0] != '\0')
	{
	    if (sscanf(buf,"%d",&N) != 1)
	    {
	        screen("ERROR in init_oned_density_step(), improper input "
	               "of number of pressure samples per averaging length\n");
	        clean_up(ERROR);
	    }
	}
	for (i = 0; i < 7; ++i)
	    alloc_state(intfc,state+i,max(sizest,sizeof(VGas)));
	wdata = alloc_DENSITY_STEP_WALL_DATA(2,N,front);
	wdata->isforward = isforward;
	set_state(state[0],VGAS_STATE,initial_state[0]);
	set_state(state[1],VGAS_STATE,initial_state[1]);
	set_state(state[2],VGAS_STATE,initial_state[2]);
	set_state(wdata->exact_wstate[0],VGAS_STATE,state[0]);
	wdata->Plast = pressure(wdata->exact_wstate[0]);
	wdata->c = c;
	wdata->s = s;
	wdata->t[0] = grid->time;
	wdata->DX = DX;
	dx = DX/N;
	if (isforward)
	{
	    for (i = 0; i < N; ++i)
	        wdata->px[i] = U - (i+0.5)*dx;
	}
	else
	{
	    for (i = 0; i < N; ++i)
	        wdata->px[i] = L + (i+0.5)*dx;
	}

	alloc_state(intfc,&left,max(sizest,sizeof(VGas)));
	alloc_state(intfc,&right,max(sizest,sizeof(VGas)));

	/* Incident shock collides with the contact */
	if (isforward)
	{
	    if (!find_mid_state(state[2],state[0],0.0,&pml,&pmr,&uml,
	                        &umr,&ml,&mr,rswtype+4,rswtype+2))
	    {
	        screen("ERROR in init_oned_density_step(), find_mid_state() "
	                      "did not converge\n");
	        clean_up(ERROR);
	    }
	    state_behind_sound_wave(state[0],state[3],NULL,W+2,0.0,mr,umr,
	                            pmr,VGAS_STATE,FORWARD_SHOCK_WAVE,
				    rswtype[2],RIGHT_FAMILY);
	    w_type = (rswtype[4] == SHOCK) ? BACKWARD_SHOCK_WAVE :
	                                     BACKWARD_SOUND_WAVE_TE;
	    state_behind_sound_wave(state[2],state[4],NULL,W+4,0.0,ml,uml,
	                            pml,VGAS_STATE,w_type,rswtype[4],
				    LEFT_FAMILY);
	}
	else
	{
	    set_state_for_find_mid_state(state[2],state[2]);
	    if (!find_mid_state(state[0],state[2],0.0,&pml,&pmr,&uml,&umr,
	                        &ml,&mr,rswtype+2,rswtype+4))
	    {
	        screen("ERROR in init_oned_density_step(), find_mid_state() "
	                      "did not converge\n");
	        clean_up(ERROR);
	    }
	    state_behind_sound_wave(state[0],state[3],NULL,W+2,0.0,ml,uml,
	                            pml,VGAS_STATE,BACKWARD_SHOCK_WAVE,
				    rswtype[2],LEFT_FAMILY);
	    w_type = (rswtype[4] == SHOCK) ? FORWARD_SHOCK_WAVE :
	                                     FORWARD_SOUND_WAVE_TE;
	    state_behind_sound_wave(state[2],state[4],NULL,W+4,0.0,mr,umr,
	                            pmr,VGAS_STATE,w_type,rswtype[4],
				    RIGHT_FAMILY);
	}
	if (rswtype[4] == RAREFACTION)
	    W[4] = vel(0,state[2]) - sgn*sound_speed(state[2]);
	wdata->t[1] = wdata->t[0] + (c - s)/W[1] - dWall/W[2];
	wdata->exact_incomingWs[0] = W[2];
	W[3] = 0.5*(uml+umr);
	wdata->wave_tol = ((pressure(state[3])-pressure(state[0]))/
	                        (pressure(state[3])+pressure(state[0])))*
			   fabs(W[2])/
			   (sound_speed(state[3])+sound_speed(state[0]));
	screen("Enter an error tolerance to identify the pressure wave "
	        "profile tails (dflt = %g): ",wdata->wave_tol);
	    (void) Gets(buf);
	if (buf[0] != '\0')
	{
	    if (sscan_float(buf,&wdata->wave_tol) != 1)
	    {
	        screen("ERROR in init_oned_density_step(), improper input "
	               "of pressure rise tolerance\n");
	        clean_up(ERROR);
	    }
	}

	r[0] = 0.5*DX;         r[1] = 0.5;      r[2] = 0.5;
	center[0] = (isforward) ? center[0] = U - r[0] : L + r[0];
	center[1] = 0.0; center[2] = 0.0;
	n[0] = max((N-1)/2,0); n[1] = 0;        n[2] = 0;
	(void) init_probe(center,r,n,&wdata->wave_tol,init,ip);


	/* Transmitted shock reaches the wall */
	if (isforward)
	{
	    set_state_for_find_mid_state(right,state[3]);
	    Vel(right)[0] *= -1.0;
	    if (!find_mid_state(state[3],right,0.0,&pml,&pmr,&uml,&umr,&ml,&mr,
	                        rswtype+5,&r_wave))
	    {
	        screen("ERROR in init_oned_density_step(), find_mid_state() "
	                      "did not converge\n");
	        clean_up(ERROR);
	    }
	    state_behind_sound_wave(state[3],wdata->exact_wstate[1],NULL,
	                            W+5,0.0,ml,uml,pml,VGAS_STATE,
				    BACKWARD_SHOCK_WAVE,rswtype[5],LEFT_FAMILY);
	    wdata->exact_outgoingWs[0] = W[5];

	    /* Reflected shock from wall collides with contact */
	    set_state_for_find_mid_state(left,state[4]);
	    set_state_for_find_mid_state(right,wdata->exact_wstate[1]);
	    if (!find_mid_state(left,right,0.0,&pml,&pmr,&uml,&umr,&ml,&mr,
	                        rswtype+8,rswtype+6))
	    {
	        screen("ERROR in init_oned_density_step(), find_mid_state() "
	                      "did not converge\n");
	        clean_up(ERROR);
	    }
	    w_type = (rswtype[6]==SHOCK) ? FORWARD_SHOCK_WAVE :
	                                   FORWARD_SOUND_WAVE_LE;
	    state_behind_sound_wave(right,state[5],NULL,W+6,0.0,mr,umr,
	                            pmr,VGAS_STATE,w_type,rswtype[6],
				    RIGHT_FAMILY);
	    state_behind_sound_wave(left,state[6],NULL,W+8,0.0,ml,uml,
	                            pml,VGAS_STATE,w_type,rswtype[8],
				    LEFT_FAMILY);
	}
	else
	{
	    set_state_for_find_mid_state(left,state[3]);
	    Vel(left)[0] *= -1.0;
	    if (!find_mid_state(left,state[3],0.0,&pml,&pmr,&uml,&umr,&ml,&mr,
	                        &l_wave,rswtype+5))
	    {
	        screen("ERROR in init_oned_density_step(), find_mid_state() "
	                      "did not converge\n");
	        clean_up(ERROR);
	    }
	    state_behind_sound_wave(state[3],wdata->exact_wstate[1],NULL,
	                            W+5,0.0,mr,umr,pmr,VGAS_STATE,
				    FORWARD_SHOCK_WAVE,rswtype[5],RIGHT_FAMILY);
	    wdata->exact_outgoingWs[0] = W[5];

	    /* Reflected shock from wall collides with contact */
	    set_state_for_find_mid_state(left,wdata->exact_wstate[1]);
	    set_state_for_find_mid_state(right,state[4]);
	    if (!find_mid_state(left,right,0.0,&pml,&pmr,&uml,&umr,&ml,&mr,
	                           rswtype+6,rswtype+8))
	    {
	        screen("ERROR in init_oned_density_step(), find_mid_state() "
	                      "did not converge\n");
	        clean_up(ERROR);
	    }
	    w_type = (rswtype[6]==SHOCK) ? BACKWARD_SHOCK_WAVE :
	                                   BACKWARD_SOUND_WAVE_LE;
	    state_behind_sound_wave(left,state[5],NULL,W+6,0.0,ml,uml,
	                            pml,VGAS_STATE,w_type,rswtype[6],
				    LEFT_FAMILY);
	    state_behind_sound_wave(right,state[6],NULL,W+8,0.0,mr,umr,
	                            pmr,VGAS_STATE,w_type,rswtype[8],
				    RIGHT_FAMILY);
	}
	W[7] = 0.5*(uml+umr);
	wdata->t[2] = wdata->t[1] +
	             dWall*(1.0-W[3]/W[2])*(1.0-W[5]/W[6])/(W[5]-W[3]);

	/*stop_time(grid) = wdata->t[2];*/

	init_output_data(init,&wdata->odata,grid,prt,"the wall data",YES,NO,NO);
	add_user_output_function(print_density_step_wall_data,
				 &wdata->odata,prt);

	(void) printf("\nIncident shock collides with contact at time %g\n",
	              (c-s)/W[1]);
	(void) printf("Transmitted wave = %s, velocity = %g\n",
	              rsoln_wave_name(rswtype[2]),W[2]);
	verbose_print_state("transmitted mid state",state[3]);
	(void) printf("Contact velocity = %g\n",W[3]);
	(void) printf("Reflected wave = %s, velocity = %g\n",
	              rsoln_wave_name(rswtype[4]),W[4]);
	verbose_print_state("reflected mid state",state[4]);

	(void) printf("First wall collision at time %g\n",wdata->t[1]);
	(void) printf("Reflected wave at wall = %s, velocity = %g\n",
	              rsoln_wave_name(rswtype[5]),W[5]);
	verbose_print_state("state on wall after collision",
	                    wdata->exact_wstate[1]);

	(void) printf("Second shock contact interaction at time %g\n",
	              wdata->t[1]+dWall*(1.0-W[3]/W[2])/(W[5]-W[3]));
	(void) printf("Reflected wave = %s, velocity = %g\n",
	              rsoln_wave_name(rswtype[6]),W[6]);
	verbose_print_state("reflected mid state",state[5]);
	(void) printf("Contact velocity = %g\n",W[7]);
	(void) printf("Transmitted wave = %s, velocity = %g\n",
	              rsoln_wave_name(rswtype[8]),W[8]);
	verbose_print_state("transmitted mid state",state[6]);
	(void) printf("Second wall collision at time %g\n",wdata->t[2]);

	(void) printf("Initial States\n");
	verbose_print_state("Target state",state[0]);
	verbose_print_state("Ahead shock state",state[1]);
	verbose_print_state("Behind shock state",state[2]);
	(void) printf("Incident shock velocity = %g\n",W[1]);
	(void) printf("Incident shock Mach number = %g\n",
	              fabs(W[1])/sound_speed(state[1]));

	verbose_print_state("State behind transmitted shock",state[3]);

	(void) printf("Wall states\n");
	for (i = 0; i < wdata->num_exact_wall_states; ++i)
	{
	    (void) sprintf(buf,"Wall state for %g < t < %g",
	                   wdata->t[i],wdata->t[i+1]);
	    verbose_print_state(buf,wdata->exact_wstate[i]);
	    (void) printf("\n");
	}
	free_these(2,left,right);
	for (i = 0; i < 7; ++i)
	    free(state[i]);
}		/*end init_oned_density_step*/

LOCAL	DENSITY_STEP_WALL_DATA *alloc_DENSITY_STEP_WALL_DATA(
	int    num_exact_wall_states,
	int    num_samples,
	Front *front)
{
	DENSITY_STEP_WALL_DATA   *wdata;
	size_t   sizest = front->sizest;
	INTERFACE *intfc = front->interf;
	int i, n, N, M;

	scalar(&wdata,sizeof(DENSITY_STEP_WALL_DATA));
	n = wdata->num_exact_wall_states = num_exact_wall_states;
	vector(&wdata->exact_wstate,n,sizeof(Locstate));
	vector(&wdata->t,n+1,FLOAT);
	for (i = 0; i < n; ++i)
	    alloc_state(intfc,wdata->exact_wstate+i,max(sizest,sizeof(VGas)));
	vector(&wdata->exact_incomingWs,n,FLOAT);
	vector(&wdata->exact_outgoingWs,n,FLOAT);
	N = wdata->num_samples = num_samples;
	vector(&wdata->pstate,N,sizeof(Locstate));
	vector(&wdata->pcomp,N,sizeof(COMPONENT));
	vector(&wdata->px,N,FLOAT);
	for (i = 0; i < N; ++i)
	    alloc_state(intfc,wdata->pstate+i,sizest);
	wdata->inside_wave = NO;
	M = wdata->NumAllocWallEvents = 10;
	vector(&wdata->wave_head_time,M,FLOAT);
	vector(&wdata->wave_tail_time,M,FLOAT);
	vector(&wdata->wave_mid_time,M,FLOAT);
	vector(&wdata->wave_head,M,sizeof(Locstate));
	vector(&wdata->wave_tail,M,sizeof(Locstate));
	for (i = 0; i < M; ++i)
	{
	    alloc_state(intfc,wdata->wave_head+i,max(sizest,sizeof(VGas)));
	    alloc_state(intfc,wdata->wave_tail+i,max(sizest,sizeof(VGas)));
	    wdata->wave_head_time[i] = HUGE_VAL;
	    wdata->wave_mid_time[i] = HUGE_VAL;
	    wdata->wave_tail_time[i] = HUGE_VAL;
	}
	wdata->num_events = 0;
	return wdata;
}		/*end alloc_DENSITY_STEP_WALL_DATA*/

LOCAL	void expand_num_wall_events(
	DENSITY_STEP_WALL_DATA *wdata,
	Front                  *front)
{
	INTERFACE *intfc = front->interf;
	size_t    sizest = front->sizest;
	int       i, M;
	float *pht = wdata->wave_head_time;
	float *ptt = wdata->wave_tail_time;
	float *mtt = wdata->wave_mid_time;
	Locstate *pwh = wdata->wave_head;
	Locstate *pwt = wdata->wave_tail;
	M = 2*wdata->NumAllocWallEvents;
	vector(&wdata->wave_head_time,M,FLOAT);
	vector(&wdata->wave_tail_time,M,FLOAT);
	vector(&wdata->wave_mid_time,M,FLOAT);
	vector(&wdata->wave_head,M,sizeof(Locstate));
	vector(&wdata->wave_tail,M,sizeof(Locstate));
	for (i = wdata->NumAllocWallEvents; i < M; ++i)
	{
	    alloc_state(intfc,wdata->wave_head+i,max(sizest,sizeof(VGas)));
	    alloc_state(intfc,wdata->wave_tail+i,max(sizest,sizeof(VGas)));
	    wdata->wave_head_time[i] = HUGE_VAL;
	    wdata->wave_mid_time[i] = HUGE_VAL;
	    wdata->wave_tail_time[i] = HUGE_VAL;
	}
	for (i = 0; i < wdata->NumAllocWallEvents; ++i)
	{
	    wdata->wave_head_time[i] = pht[i];
	    wdata->wave_tail_time[i] = ptt[i];
	    wdata->wave_mid_time[i] = mtt[i];
	    wdata->wave_head[i] = pwh[i];
	    wdata->wave_tail[i] = pwt[i];
	}
	wdata->NumAllocWallEvents = M;
}		/*end expand_num_wall_events*/

/*ARGSUSED*/
LOCAL	void	print_density_step_wall_data(
	Grid        *grid,
	Wave        *wave,
	Front       *front,
	Printplot   *prt,
	OUTPUT_DATA *data,
	bool        about_to_stop)
{
	DENSITY_STEP_WALL_DATA *wdata = (DENSITY_STEP_WALL_DATA*)data;
	INTERFACE              *intfc = front->interf;
	FILE                   *file;
	float                  time;
	float                  Pe, Pn, dP, adP, rdP, ardP;
	float                  rhon;
	float                  Plast;
	int                    i, n;

	n = wdata->num_exact_wall_states;
	if (Output_file(data) == NULL)
	{
	    file = Output_file(data) = fopen(Output_filename(data),"w");
	    print_machine_parameters(file);
	    (void) foutput(file);
	    (void) fprintf(file,"%-18s %-18s %-18s\n",
	                   "Time","Pressure","EXACT-SOLUTION");
	    for (i = 0; i < n; ++i)
	    {
	        time = wdata->t[i];
	        Pe = pressure(wdata->exact_wstate[i]);
	        if (Output_in_binary(data))
	        {
	            (void) fprintf(file,"\f%c",4);
		    (void) fwrite((const void *)&time,FLOAT,1,file);
		    (void) fwrite((const void *)&Pe,FLOAT,1,file);
	        }
	        else
	            (void) fprintf(file,"%-"FFMT" %-"FFMT"\n",time,Pe);
	        time = wdata->t[i+1];
	        if (Output_in_binary(data))
	        {
	            (void) fprintf(file,"\f%c",4);
		    (void) fwrite((const void *)&time,FLOAT,1,file);
		    (void) fwrite((const void *)&Pe,FLOAT,1,file);
	        }
	        else
	            (void) fprintf(file,"%-"FFMT" %-"FFMT"\n",time,Pe);
	    }
	    (void) fprintf(file,"\n");

	    (void) foutput(file);
	    (void) fprintf(file,"%-18s %-18s %-18s %-18s %-18s "
	                        "%-18s %-18s %-18s\n",
	                   "Time","P-numerical","P-exact","P-error","aP-error",
			   "%rel-P-error","%arel-P-error",
			   "NUMERICAL-SOLUTION");
	}
	else
	    file = Output_file(data);

	time = grid->time;
	Plast = wdata->Plast;
	Pn = 0.0;
	rhon = 0.0;
	for (i = 0; i < wdata->num_samples; ++i)
	{
	    wdata->pcomp[i] = component(wdata->px+i,intfc);
	    hyp_solution(wdata->px+i,wdata->pcomp[i],NULL,UNKNOWN_SIDE,
		         front,wave,wdata->pstate[i],NULL);
	    Pn += pressure(wdata->pstate[i]);
	    rhon += Dens(wdata->pstate[i]);
	}
	Pn /= wdata->num_samples;
	rhon /= wdata->num_samples;
	dP = 2.0*fabs(Pn - Plast)/(Pn + Plast);
	if (wdata->num_events >= wdata->NumAllocWallEvents)
	    expand_num_wall_events(wdata,front);
	if (wdata->inside_wave == NO)
	{
	    if (dP > wdata->wave_tol)
	    {
	        int n = wdata->num_events;
	        wdata->inside_wave = YES;
	        wdata->wave_head_time[n] = wdata->wave_mid_time[n] =
	            wdata->wave_tail_time[n] = time;
	        set_type_of_state(wdata->wave_head[n],TGAS_STATE);
	        Dens(wdata->wave_head[n]) = rhon;
	        Press(wdata->wave_head[n]) = Pn;
	        zero_state_velocity(wdata->wave_head[n],front->rect_grid->dim);
	        Set_params(wdata->wave_head[n],wdata->pstate[0]);
	        set_state(wdata->wave_head[n],VGAS_STATE,wdata->wave_head[n]);
	        set_state(wdata->wave_tail[n],VGAS_STATE,wdata->wave_head[n]);
	    }
	}
	else
	{
	    if (dP < wdata->wave_tol)
	    {
	        int n = wdata->num_events;
	        wdata->wave_tail_time[n] = time;
		wdata->wave_mid_time[n] =
		    0.5*(wdata->wave_head_time[n]+wdata->wave_tail_time[n]);
	        set_type_of_state(wdata->wave_tail[n],TGAS_STATE);
	        Dens(wdata->wave_tail[n]) = rhon;
	        Press(wdata->wave_tail[n]) = Pn;
	        zero_state_velocity(wdata->wave_tail[n],front->rect_grid->dim);
	        Set_params(wdata->wave_tail[n],wdata->pstate[0]);
	        set_state(wdata->wave_tail[n],VGAS_STATE,wdata->wave_tail[n]);
		++wdata->num_events;
	        wdata->inside_wave = NO;
	    }
	}
	wdata->Plast = Pn;

	if (time < wdata->t[n])
	{
	    for (i = 0; i < n; ++i)
	        if (wdata->t[i] <= time && time <= wdata->t[i+1])
		    break;
	    Pe = pressure(wdata->exact_wstate[i]);
	    dP = Pn-Pe;
	    rdP = (Pn-Pe)/Pe;
	    adP = fabs(dP);
	    ardP = fabs(rdP);
	    if (Output_in_binary(data))
	    {
	        float prdP = 100.0*rdP, pardP = 100.0*ardP;
	        (void) fprintf(file,"\f%c",4);
		(void) fwrite((const void *)&time,FLOAT,1,file);
		(void) fwrite((const void *)&Pn,FLOAT,1,file);
		(void) fwrite((const void *)&Pe,FLOAT,1,file);
		(void) fwrite((const void *)&dP,FLOAT,1,file);
		(void) fwrite((const void *)&adP,FLOAT,1,file);
		(void) fwrite((const void *)&prdP,FLOAT,1,file);
		(void) fwrite((const void *)&pardP,FLOAT,1,file);
	    }
	    else
	        (void) fprintf(file,"%-"FFMT" %-"FFMT" %-"FFMT" %-"FFMT" "
		               "%-"FFMT" %-"FFMT" %-"FFMT"\n",
	                       time,Pn,Pe,dP,adP,100.0*rdP,100.0*ardP);
	}
	else if (wdata->num_events > 0)
	{
	    static bool first = YES;
	    if (first)
	    {

	        float dat;
	        float Wexact;
	        float Wmid, Ws, We, Werr, dW;
		float tail_time, head_time, mid_time;

	        first = NO;
	        Wexact = wdata->exact_incomingWs[0];
	        (void) fprintf(file,"\n");
	        (void) fprintf(file,"Exact arrival time at wall "
	                            "of transmitted shock = %g\n",wdata->t[1]);
	        (void) fprintf(file,"Exact transmitted wave speed = %g\n",
	                       Wexact);

		tail_time = wdata->wave_tail_time[0];
		mid_time = wdata->wave_mid_time[0];
		head_time = wdata->wave_head_time[0];
	        (void) fprintf(file,"\nEstimates for inital "
		                    "transmitted shock\n");
	        (void) fprintf(file,"start of arrival = %g\n",head_time);
	        (void) fprintf(file,"middle of arrival = %g\n",mid_time);
	        (void) fprintf(file,"end of arrival = %g\n",tail_time);
	        dat = mid_time - wdata->t[1];
	        (void) fprintf(file,"Error in middle of arrival = %g\n",dat);
	        (void) fprintf(file,"Absolute relative error in "
	                            "middle of arrival = %g%%\n",
	                            100.0*fabs(dat)/wdata->t[1]);
	        (void) fprintf(file,"Width of arrival interval = %g\n",
	                       tail_time-head_time);
	        (void) fprintf(file,"Relative width of "
		                    "arrival interval = %g%%\n",
	                       100.0*(tail_time - head_time)/wdata->t[1]);

	        Ws = -(wdata->c - front->rect_grid->L[0])/
	              (head_time - wdata->t[0]);
	        (void) fprintf(file,"Estimated transmitted wave "
	                       "head velocity = %g\n",Ws);
	        Werr = Ws - Wexact;
	        (void) fprintf(file,"Error in estimated transmitted wave "
	                       "head velocity = %g\n",Werr);
	        (void) fprintf(file,"Relative error in "
	                            "transmitted wave head velocity = %g%%\n",
	                            100.0*fabs(Werr/Wexact));
    
	        Wmid = -(wdata->c-front->rect_grid->L[0])/
	                (mid_time - wdata->t[0]);
	        (void) fprintf(file,"Estimated transmitted wave midpoint "
	                       "velocity = %g\n",Wmid);
	        Werr = Wmid - Wexact;
	        (void) fprintf(file,"Error in estimated transmitted wave "
	                       "velocity = %g\n",Werr);
	        (void) fprintf(file,"Absolute relative error in "
	                            "transmitted wave velocity = %g%%\n",
	                            100.0*fabs(Werr/Wexact));
	                       
	        We = -(wdata->c-front->rect_grid->L[0])/
	              (tail_time - wdata->t[0]);
	        (void) fprintf(file,"Estimated transmitted wave "
	                       "tail velocity = %g\n",We);
	        Werr = We - Wexact;
	        (void) fprintf(file,"Error in estimated transmitted wave "
	                       "tail velocity = %g\n",Werr);
	        (void) fprintf(file,"Relative error in "
	                            "transmitted wave tail velocity = %g%%\n",
	                       100.0*fabs(Werr/Wexact));
	        dW = 100.0*0.5*fabs((We-Ws)/Wmid);
	        (void) fprintf(file,"Transmitted wave velocity = %g +- %g%%\n",
	                       Wmid,dW);
	    }
	}
	if (about_to_stop)
	{
	    char name[256];
	    (void) fprintf(file,"\nDetected Wall Events\n");
	    for (i = 0; i < wdata->num_events; ++i)
	    {
	        (void) fprintf(file,"Time of wave %d-%s head = %g\n",i+1,
		              ordinal_suffix(i+1),
		              wdata->wave_head_time[i]);
	        (void) fprintf(file,"Time of %d-%s wave middle = %g\n",i+1,
		              ordinal_suffix(i+1),
		              wdata->wave_mid_time[i]);
	        (void) fprintf(file,"Time of wave %d-%s tail = %g\n",i+1,
		              ordinal_suffix(i+1),
		              wdata->wave_tail_time[i]);
		(void) fprintf(file,"pressure jump across wave = %g\n",
		               pressure(wdata->wave_tail[i])-
			       pressure(wdata->wave_head[i]));
		(void) sprintf(name,"State at wave head");
		verbose_fprint_state(file,name,wdata->wave_head[i]);
		(void) sprintf(name,"State at wave tail");
		verbose_fprint_state(file,name,wdata->wave_tail[i]);
	    }
	}
}		/*end print_density_step_wall_data*/
#endif /*defined(ONED)*/

/*
 * storage for time dependent pressure input
 * time_dep_pres = { warming-up time,
 *                   peak time,
 *                   cooling-down time,
 *                   base pressure,
 *                   peak pressure }
 *
 *              set_time_dep_pres();
 *      Set warming-up, peak, cooling-down times, base and peak pressures
 *      into the array time_dep_pres
 */
LOCAL 	void    set_time_dep_pres(
	BOUNDARY_STATE *bstate,
	float   tr,
	float   tp,
	float   ts,
	float   pr_a,
	float   pr_p)
{
	FD_DATA *fd_data;
	stat_scalar(&bstate->_boundary_state_data,sizeof(FD_DATA));
	fd_data = (FD_DATA*)bstate->_boundary_state_data;
        fd_data->tr = tr;
        fd_data->tp = tp;
        fd_data->ts = ts;
        fd_data->pr_a = pr_a;
        fd_data->pr_p = pr_p;
}               /* end set_time_dep_pres */

EXPORT  void      g_init_parabolic(
        Front     *front)
{
        INTERFACE *intfc = front->interf;
	Gas_param **prms_list; /* = gas_params_list(intfc);*/
	int       nprms;
	int       i, nz, *idx=NULL;
        char      s[121];
	char      fmt[1024],fmt2[1024];

	debug_print("init","Entered g_init_parabolic()\n");

	g_set_HR_times();

	debug_print("init","Left g_init_parabolic()\n");
}		/*end g_init_parabolic*/

LOCAL void prompt_for_adherence_coeff(HYPER_SURF *hs)
{
        char s[64];
        float adherence_coeff;
        no_slip(hs) = YES;
        adherence_coeff = 0.0;
        screen("Enter the adherence coefficient\n");
        screen("\tadherence_coeff = 0.0 for fully slippery\n");
        screen("\tadherence_coeff = 1.0 for fully non-slippery\n");
        screen("\tdefault = %g (non-slippery)\n",adherence_coeff);
        screen("Enter the value of adherence coefficient: ");
        (void) Gets(s);
        sscan_float(s,&adherence_coeff);
        if (adherence_coeff < 0.0 || adherence_coeff > 1.0)
        {
            printf("ERROR in prompt_for_adherence_coeff,"
               " invalid input of adherence coefficient, setting it to 0, full slip\n");
            adherence_coeff = 0.0;
        }
        if(adherence_coeff == 0.0)
            no_slip(hs) = NO;
        adherence_coeff(hs) = adherence_coeff;
}


LOCAL float ff_Shu_Osher(
        float *crds,
        int   n,
        float t)
{
        float den, p = 1.0, u[2] = {0.0, 0.0};
        float tmp;

        if(crds[1] < -4.0)
        {
            den = 3.857143;
            p = 10.333333;
            u[1] = 2.629369;
        }
        else
        {
            den = 1.0 + 0.2*sin(5.0*crds[1]);
            p = 1.0;
        }


        switch(n)
        {
        case 0:
            tmp = den; 
        break;
        case 1:
            tmp = den*u[0];
        break;
        case 2:
            tmp = den*u[1];
        break;
        case 3:
            tmp = p/0.4 + 0.5*den*(sqr(u[0]) + sqr(u[1]));
        break;
        }
        return tmp;
}


LOCAL void twod_riemann_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           gam = 1.4, P;
        float           bar[3];
        int             i, j, k, indx, dim = 2;
        POINT   *p[3];
        float   *pcrds[3], dbcent[MAXD];
        float   emid0[3], emid1[3], emid2[3], *pemid[3], conu[4];
        double   *cent = fg_centroid(tri);
        double   rhs[4][MAX_N_COEF], mulrhs[4][MAX_N_COEF], dens[7];
        float   area = fg_area(tri), v = 0.0, t = 0.0;
        double      **mass_inv, **Lmass_matrix;
        int     debug_flag = NO;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
          w3 = 0.175615257433204,  w4 = 0.077113760890257;
        float crds[13][2];
        third = 1.0/3.0;

        for(indx = 1; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = 0.0;
            dg_Mom(state)[0][indx] = 0.0;
            dg_Mom(state)[1][indx] = 0.0;
            dg_Energy(state)[indx] = 0.0;
        }

        for(i = 0; i < dim; i++)
            dbcent[i] = cent[i];  

        Dens(state) = dg_Dens(state)[0] = ff_2d_riemann(dbcent, 0);
        Mom(state)[0] = dg_Mom(state)[0][0] = ff_2d_riemann(dbcent, 1);
        Mom(state)[1] = dg_Mom(state)[1][0] = ff_2d_riemann(dbcent, 2);
        Energy(state) = dg_Energy(state)[0] = ff_2d_riemann(dbcent, 3);
}


LOCAL float ff_2d_riemann(
        float *crds,
        int   n)
{
        float den, p = 1.0, u[2] = {0.0, 0.0};
        float tmp;

        if(crds[0] >= 0.5 && crds[1] >= 0.5)
        {
            den = 1.1;
            u[0] = 0.0;
            u[1] = 0.0;
            p = 1.1;
        } 
        else if(crds[0] > 0.0 && crds[0] < 0.5 && crds[1] >= 0.5)
        {
            den = 0.5065;
            u[0] = 0.8939;
            u[1] = 0.0;
            p = 0.35;
        }
        else if(crds[0] > 0.0 && crds[0] < 0.5 && crds[1] >= 0.0 && crds[1] < 0.5)
        {
            den = 1.1;
            u[0] = 0.8939;
            u[1] = 0.8939;
            p = 1.1;
        }
        else
        {
            den = 0.5065;
            u[0] = 0.0;
            u[1] = 0.8939;
            p = 0.35;
        }

        switch(n)
        {
        case 0:
            tmp = den;
        break;
        case 1:
            tmp = den*u[0];
        break;
        case 2:
            tmp = den*u[1];
        break;
        case 3:
            tmp = p/0.4 + 0.5*den*(sqr(u[0]) + sqr(u[1]));
        break;
        }
        return tmp;
}


// Mach = 1.1 relative to the right state.
// left u should be u = 1.1*sqrt(gamma=1.4)
LOCAL float ff_shock_vort(
        float *centroid,
        float *crds,
        int   n)
{
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u, v, T = 1.0, dT, nT, Gam = 0.4, S;
        float           den = 1.0, P = 1.0, gam = 1.4, tmp, c;

        if(centroid[1] > 0.5)
        {
            den = 2.4*sqr(1.1)/(0.4*sqr(1.1)+2.0);
            P = 2.0*1.4*(sqr(1.1)-1.0)/2.4+1.0;
            v = -2.0*(sqr(1.1)-1.0)/(2.4*1.1)*sqrt(1.4)+1.1*sqrt(1.4);
            u = 0.0;
        }
        else
        {
            u = 0.0; v = 1.1*sqrt(1.4);
            cent[0] = 0.5;
            cent[1] = 0.25;
            rc = 0.05;

            v2[0] = crds[0]-cent[0];
            v2[1] = crds[1]-cent[1];
            r = sqrt(sqr(v2[0]) + sqr(v2[1]));
            tau = r/rc;

            norv2[0] = v2[0]/r;
            norv2[1] = v2[1]/r;
            r_angle = angle(norv2[0],norv2[1]);
            costheta = cos(r_angle);
            sintheta = sin(r_angle);
            u += ep*tau*exp(alpha*(1-sqr(tau)))*sintheta;
            v += -ep*tau*exp(alpha*(1-sqr(tau)))*costheta;
            dT = -Gam*sqr(ep)*exp(2.0*alpha*(1.0-sqr(tau)))/(4.0*alpha*(Gam+1.0));
            T += dT;
            S = log(P/(pow(den, gam)));
            den = pow(exp(S)/T, 1.0/(1.0-gam));
            P = T*den;
        }

        switch(n)
        {
        case 0:
            tmp = den;
        break;
        case 1:
            tmp = den*u;
        break;
        case 2:
            tmp = den*v;
        break;
        case 3:
            tmp = P/0.4 + 0.5*den*(sqr(u) + sqr(v));
        break;
        }
        return tmp;
}

/// The initial condition is from J.C.P. 2010. 229, 7893-7910. 
/// Test case 5.2
/// To get the Lobatto Quadrature's Abscissas and Weight, ref:
/// Abscissas and Weight Coefficients for Lobatto Quadrature.
///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
LOCAL void accurate_magnetic_vortex_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[3], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[8][MAX_N_COEF], mulrhs[8][MAX_N_COEF], dens[8], tmp;
        float           area = fg_area(tri), t = 0.0;
        double          **mass_inv, **Lmass_matrix;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[3][8];
        double       *lcent = fg_centroid(tri);
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        static double **tmp_mass_inv = NULL;
        int           Lobatto_N = 7;
        long double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {-0.666667, -1.08333};

        /// TMP
        if(   (fabs(lcent[0]-tmpcent0[0])<1.0e-5  && 
               fabs(lcent[1]-tmpcent0[1])<1.0e-5) 
            ||
              (fabs(lcent[0]-tmpcent1[0])<1.0e-5  && 
               fabs(lcent[1]-tmpcent1[1])<1.0e-5)  
          )
        {
            debug_flag = YES;
            printf("Tri[%d] entered accurate_magnetic_vortex_initializer()\n", tri->id);
            print_tri_crds(tri);
        }
        /// END TMP

        if(Lobatto_N == 4)
        {
            q[0] = -1.0; q[1] = -sqrt(5.0)/5.0;
            q[2] =  sqrt(5.0)/5.0; q[3] = 1.0;
            qw[0] =  1.0/6.0; qw[1] = 5.0/6.0;
            qw[2] =  5.0/6.0; qw[3] = 1.0/6.0;
        }
        else if(Lobatto_N == 5)
        {
            q[0] = -1.0; q[1] = -sqrt(21.0)/7.0;
            q[2] = 0.0;  q[3] =  sqrt(21.0)/7.0; q[4] = 1.0;
            qw[0] =  0.1; qw[1] = 49.0/90.0; qw[2] = 32.0/45.0;
            qw[3] =  49.0/90.0; qw[4] = 0.1;
        }
        else if(Lobatto_N == 6)
        {
            q[0] = -1.0; q[1] = -sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));
            q[2] = -sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));  q[3] =  sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0))); 
            q[4] =  sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));     q[5] = 1.0;
            qw[0] =  1.0/15.0; qw[1] = 1.0/30.0*(14.0-sqrt(7.0)); 
            qw[2] = 1.0/30.0*(14.0+sqrt(7.0)); qw[3] =  1.0/30.0*(14.0+sqrt(7.0)); 
            qw[4] = 1.0/30.0*(14.0-sqrt(7.0));       qw[5] = 1.0/15.0;
        }
        else if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987; 
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905; 
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905; 
        }
        else 
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: accurate_magnetic_vortex_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }


        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;
        Lmass_matrix = tri->Lmass_matrix;
        if(tri->mass_inv != NULL)
            mass_inv = tri->mass_inv;
        else
        {
            mass_inv = tmp_mass_inv;
            inverse_matrix(Lmass_matrix,MAX_N_COEF,mass_inv);
        }

        for(i = 0; i < 8; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
            cent[i] = dbcent[i] = fg_centroid(tri)[i];

        /***
        tri_quadrature_13_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = w1*ff_accurate_magnetic_vort(cent, crds[0],j)*vh_val(crds[0],cent,k) +
                      w2*(ff_accurate_magnetic_vort(cent, crds[1],j)*vh_val(crds[1],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[2],j)*vh_val(crds[2],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[3],j)*vh_val(crds[3],cent,k)) +
                      w3*(ff_accurate_magnetic_vort(cent, crds[4],j)*vh_val(crds[4],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[5],j)*vh_val(crds[5],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[6],j)*vh_val(crds[6],cent,k)) +
                      w4*(ff_accurate_magnetic_vort(cent, crds[7],j)*vh_val(crds[7],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[8],j)*vh_val(crds[8],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[9],j)*vh_val(crds[9],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[10],j)*vh_val(crds[10],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[11],j)*vh_val(crds[11],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[12],j)*vh_val(crds[12],cent,k));
            }
        }
        ***/
        /// for 16 pts case
        tri_quadrature_16_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = 0.0;
                for(i = 0; i < 16; i++)
                {
                    rhs[j][k] += w[i]*ff_accurate_magnetic_vort(dbcent, crds[i],j,YES)*vh_val(crds[i],lcent,k);
                }
            }
        }
        ///


        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);

        if(N_EQN == 4)
        {
            for(indx = 0; indx < MAX_N_COEF; indx++)
            {
                dg_Dens(state)[indx] = mulrhs[0][indx];
                dg_Mom(state)[0][indx] = mulrhs[1][indx];
                dg_Mom(state)[1][indx] = mulrhs[2][indx];
                dg_Energy(state)[indx] = mulrhs[3][indx];
            }
        }
        else 
        {
            for(indx = 0; indx < MAX_N_COEF; indx++)
            {
                dg_Dens(state)[indx] = mulrhs[0][indx];
                dg_Mom(state)[0][indx] = mulrhs[1][indx];
                dg_Mom(state)[1][indx] = mulrhs[2][indx];
                dg_Energy(state)[indx] = mulrhs[3][indx];
                // dg_Mom(state)[2][indx] = mulrhs[4][indx];
                dg_Mom(state)[2][indx] = 0.0;
                dg_B(state)[0][indx] =  mulrhs[5][indx];
                dg_B(state)[1][indx] =  mulrhs[6][indx];
                dg_B(state)[2][indx] =  0.0;
            }
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Mom(state)[2] = 0.0;
        Energy(state) = 0.0;
        Mag(state)[0] = 0.0;
        Mag(state)[1] = 0.0;
        Mag(state)[2] = 0.0;

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Lmass_matrix[0][indx];
            
            Mag(state)[0] += dg_B(state)[0][indx]*Lmass_matrix[0][indx];
            Mag(state)[1] += dg_B(state)[1][indx]*Lmass_matrix[0][indx];
        }
        Dens(state) /= Lmass_matrix[0][0];
        Mom(state)[0] /= Lmass_matrix[0][0];
        Mom(state)[1] /= Lmass_matrix[0][0];
        Energy(state) /= Lmass_matrix[0][0];

        Mag(state)[0] /= Lmass_matrix[0][0];
        Mag(state)[1] /= Lmass_matrix[0][0];

        /// We also need to init edge B
        if(N_EQN == 8)
        {
            for(side=0; side < 3; side++)
            {
                fg_side_B(tri)[side] = 0.0;
                // for(j = 0; j < N_EQN; j++)
                //     v_conU[side][j] = ff_accurate_magnetic_vort(dbcent, pcrds[side],j);
            }
            /**
            for(side=0; side < 3; side++)
            {
                v_conU[side][1] /= v_conU[side][0];
                v_conU[side][2] /= v_conU[side][0];
                v_conU[side][4] /= v_conU[side][0];
            }
            **/
 
            for(side=0; side < 3; side++)
            {
                for(i = 0; i < 2; i++)
                    v2[i] = fg_side_vector(tri)[side][i];
                nor[0] = v2[1];
                nor[1] = -v2[0];

                for(k = 0; k < Lobatto_N; k++)
                {
                    for(i = 0; i < 2; i++)
                        qcrds[i] = (pcrds[(side+1)%3][i] + pcrds[side][i])/2.0 +
                               (pcrds[(side+1)%3][i] - pcrds[side][i])/2.0*q[k];
                    for(j = 5; j < N_EQN; j++)
                        v_conU[side][j] = ff_accurate_magnetic_vort(dbcent, qcrds,j,YES);

                    Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];
                    fg_side_B(tri)[side] += Bn[k]*qw[k];
                }
                // fg_side_B(tri)[side] *= fg_length_side(tri)[side];
            }
            tmp = fg_side_B(tri)[0]*fg_length_side(tri)[0] +
                  fg_side_B(tri)[1]*fg_length_side(tri)[1] +
                  fg_side_B(tri)[2]*fg_length_side(tri)[2];

            if(debug_flag == YES || fabs(tmp) > 1.0e-10)
            {
                printf("tri[%d] in accurate_magnetic_vortex_initializer(), divg B = %14.13g, Bn = [%13.12g, %13.12g, %13.12g]\n",
                     tri->id, tmp,
                     fg_side_B(tri)[0], fg_side_B(tri)[1], fg_side_B(tri)[2]);
                for(j = 0; j < N_EQN; j++)
                    v_conU[0][j] = ff_accurate_magnetic_vort(dbcent, dbcent,j, YES);
                printf("velocity at cent[%g %g], B at cent[%g %g]\n",  v_conU[0][1], v_conU[0][2], v_conU[0][5], v_conU[0][6]);

                g_verbose_print_state(state);

                for(side =0; side < 3; side++)
                {
                    pcrds[0] = Coords(Point_of_tri(tri)[side]);
                    pcrds[1] = Coords(Point_of_tri(tri)[ (side +1)%3 ]);

                    dy = pcrds[1][1] - pcrds[0][1];
                    dx = pcrds[1][0] - pcrds[0][0];
                    len = fg_length_side(tri)[side];
    
                    sum = dy*dg_B(state)[0][0] + (0.5*dy*dx + dy*pcrds[0][0])*dg_B(state)[0][1] + (0.5*sqr(dy) + dy*pcrds[0][1])*dg_B(state)[0][2] +
                          -dx*dg_B(state)[1][0] - (0.5*sqr(dx) + dx*pcrds[0][0])*dg_B(state)[1][1] - (0.5*dx*dy + dx*pcrds[0][1])*dg_B(state)[1][2];
                    sum /= len;
                    printf("Side[%d], cell average recoved Bn = %13.12g\n", side, sum);
                    printf("Bx[%g, %g, %g]\n", dg_B(state)[0][0], dg_B(state)[0][1], dg_B(state)[0][2]);
                    printf("By[%g, %g, %g]\n", dg_B(state)[1][0], dg_B(state)[1][1], dg_B(state)[1][2]);
                }
                // clean_up(0);
            }
        }

}

//// Initialize for locally divergence-free basis
LOCAL void accurate_magnetic_vortex_initializer_4(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3]; 
        float           *pcrds[3], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[10][MAX_N_COEF], mulrhs[10][MAX_N_COEF], dens[10], tmp;
        float           area = fg_area(tri), t = 0.0, sqrt_area; 
        double          **Bmass_inv, **Bmass_matrix;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[3][8];
        double       *lcent = fg_centroid(tri);
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        static double **tmp_mass_inv = NULL;
        int           Lobatto_N = 7;
        long double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        third = 1.0/3.0;
        Bmass_matrix = tri->Bmass_matrix;
        if(tri->Bmass_inv != NULL)
            Bmass_inv = tri->Bmass_inv;

        sqrt_area = sqrt(area);

        for(i = 0; i < 8; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
            cent[i] = dbcent[i] = fg_centroid(tri)[i];

        tri_quadrature_16_pts(pcrds[0], pcrds[1], pcrds[2], crds);

        // Integration of DG // Huijing
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = 0.0;
                for(i = 0; i < 16; i++)
                {
                    rhs[j][k] += w[i]*ff_accurate_magnetic_vort(dbcent, crds[i],j,YES)
                                 *vh_val_loc_div_free_basis(crds[i],cent,sqrt_area,k);
                }
            }
        }

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(Bmass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);//mass_inv*rhs[i]

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }
        if(N_EQN == 8)
        {
            for(indx = 0; indx < MAX_N_COEF; indx++)
            {
                dg_Mom(state)[2][indx] = mulrhs[4][indx];
                dg_B(state)[0][indx] =  mulrhs[5][indx];
                dg_B(state)[1][indx] =  mulrhs[6][indx];
                dg_B(state)[2][indx] =  mulrhs[7][indx];
            }
        }

        if(MAX_N_COEF == 3 || MAX_N_COEF == 6 || MAX_N_COEF == 10)
        {
            /// printf("EXIT in accurate_magnetic_vortex_initializer_4()\n");
            /// printf("dg_B(state)[0][1] = %14.13g, dg_B(state)[1][2] = %14.13g\n", 
            ///    dg_B(state)[0][1], dg_B(state)[1][2]);
            if(dg_B(state)[0][1] >= 0.0)
            {
                dg_B(state)[0][1] = 0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            else
            {
                dg_B(state)[0][1] = -0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            dg_B(state)[1][2] = -dg_B(state)[0][1];
            /// printf("--------dg_B(state)[0][1] = %14.13g, dg_B(state)[1][2] = %14.13g\n", 
            ///    dg_B(state)[0][1], dg_B(state)[1][2]);
        } 

        if(MAX_N_COEF == 6)
        {
            if(dg_B(state)[0][3] >= 0.0)
                dg_B(state)[0][3] = 0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));
            else
                dg_B(state)[0][3] = -0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));

            dg_B(state)[1][4] = -2.0*dg_B(state)[0][3];

            if(dg_B(state)[0][4] >= 0.0)
                dg_B(state)[0][4] = 0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));
            else
                dg_B(state)[0][4] = -0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));

            dg_B(state)[1][5] = -0.5*dg_B(state)[0][4];
        }

        if(MAX_N_COEF == 10)
        {
            if(dg_B(state)[0][6] >= 0.0)
                dg_B(state)[0][6] = 0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            else
                dg_B(state)[0][6] = -0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            dg_B(state)[1][7] = -3.0*dg_B(state)[0][6];            

            if(dg_B(state)[0][7] >= 0.0)
                dg_B(state)[0][7] = 0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            else
                dg_B(state)[0][7] = -0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            dg_B(state)[1][8] = -dg_B(state)[0][7];            

            if(dg_B(state)[0][8] >= 0.0)
                dg_B(state)[0][8] = 0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            else
                dg_B(state)[0][8] = -0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            dg_B(state)[1][9] = -dg_B(state)[0][8]/3.0;
        }

        if(MAX_N_COEF != 1 && MAX_N_COEF != 3 && MAX_N_COEF != 6 && MAX_N_COEF != 10)
        {
            printf("EXIT in accurate_magnetic_vortex_initializer_4()\n");
            printf("initialize with divgerence free for MAX_N_COEF %d\n", MAX_N_COEF);
            clean_up(0);
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        Mom(state)[2] = 0.0;
        Mag(state)[0] = 0.0;
        Mag(state)[1] = 0.0;
        Mag(state)[2] = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Bmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Bmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[2] += dg_Mom(state)[2][indx]*Bmass_matrix[0][indx];
            Mag(state)[0] += dg_B(state)[0][indx]*Bmass_matrix[0][indx];
            Mag(state)[1] += dg_B(state)[1][indx]*Bmass_matrix[0][indx];
            Mag(state)[2] += dg_B(state)[2][indx]*Bmass_matrix[0][indx];
        }
        Dens(state) /= Bmass_matrix[0][0];
        Mom(state)[0] /= Bmass_matrix[0][0];
        Mom(state)[1] /= Bmass_matrix[0][0];
        Energy(state) /= Bmass_matrix[0][0];
        Mom(state)[2] /= Bmass_matrix[0][0];
        Mag(state)[0] /= Bmass_matrix[0][0];
        Mag(state)[1] /= Bmass_matrix[0][0];
        Mag(state)[2] /= Bmass_matrix[0][0];
}

// NEW for DG // Huijing
/// The initial condition is from J.C.P. 2010. 229, 7893-7910. 
/// Test case 5.2
/// To get the Lobatto Quadrature's Abscissas and Weight, ref:
/// Abscissas and Weight Coefficients for Lobatto Quadrature.
///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
///  Only init cell average. The energy does not have |B|^2.
LOCAL void accurate_magnetic_vortex_initializer_3(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[3], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[10][MAX_N_COEF], mulrhs[10][MAX_N_COEF], dens[10], tmp;
        float           area = fg_area(tri), t = 0.0;
        double          **mass_inv, **Lmass_matrix;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[3][8];
        double       *lcent = fg_centroid(tri);
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        static double **tmp_mass_inv = NULL;
        int           Lobatto_N = 7;
        long double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        // Lobatto quadrature
        if(Lobatto_N == 4)
        {
            q[0] = -1.0; q[1] = -sqrt(5.0)/5.0;
            q[2] =  sqrt(5.0)/5.0; q[3] = 1.0;
            qw[0] =  1.0/6.0; qw[1] = 5.0/6.0;
            qw[2] =  5.0/6.0; qw[3] = 1.0/6.0;

            // Gaussian
            // q[0] = -0.86113631159405257522; q[1] = -0.33998104358485626480;
            // q[2] =  0.33998104358485626480; q[3] = 0.86113631159405257522;
            // qw[0] =  0.34785484513745385737; qw[1] = 0.65214515486254614263;
            // qw[2] =  0.65214515486254614263; qw[3] = 0.34785484513745385737;
        }
        else if(Lobatto_N == 5)
        {
            q[0] = -1.0; q[1] = -sqrt(21.0)/7.0;
            q[2] = 0.0;  q[3] =  sqrt(21.0)/7.0; q[4] = 1.0;
            qw[0] =  0.1; qw[1] = 49.0/90.0; qw[2] = 32.0/45.0;
            qw[3] =  49.0/90.0; qw[4] = 0.1;
        }
        else if(Lobatto_N == 6)
        {
            q[0] = -1.0; q[1] = -sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));
            q[2] = -sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));  q[3] =  sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0))); 
            q[4] =  sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));     q[5] = 1.0;
            qw[0] =  1.0/15.0; qw[1] = 1.0/30.0*(14.0-sqrt(7.0)); 
            qw[2] = 1.0/30.0*(14.0+sqrt(7.0)); qw[3] =  1.0/30.0*(14.0+sqrt(7.0)); 
            qw[4] = 1.0/30.0*(14.0-sqrt(7.0));       qw[5] = 1.0/15.0;
        }
        else if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987; 
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905; 
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905; 
        }
        else 
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: accurate_magnetic_vortex_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }

        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;
        Lmass_matrix = tri->Lmass_matrix;
        if(tri->mass_inv != NULL)
            mass_inv = tri->mass_inv;
        else
        {
            if(MAX_N_COEF != 1)
            {
                mass_inv = tmp_mass_inv;
                inverse_matrix(Lmass_matrix,MAX_N_COEF,mass_inv);
            }
        }

        for(i = 0; i < 8; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
            cent[i] = dbcent[i] = fg_centroid(tri)[i];

        /****
        printf("\n\ntri[%d]:\n",tri->id);
        for(k=0;k<Lobatto_N;k++)
            printf("k=%d q=%Lf qw=%Lf\n",k,q[k],qw[k]);
        printf("pcrds=(%g %g)(%g %g)(%g %g)\n",pcrds[0][0],pcrds[0][1],pcrds[1][0],pcrds[1][1],pcrds[2][0],pcrds[2][1]);
        printf("%g %g %g %g %g %g\n",pcrds[0][0],pcrds[0][1],pcrds[1][0],pcrds[1][1],pcrds[2][0],pcrds[2][1]);
        printf("center=(%g %g)\n",cent[0],cent[1]);
        printf("mass=\n");
        for(i = 0; i < MAX_N_COEF; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
            {
                printf("%g ",Lmass_matrix[i][j]);
            }
            printf("\n");
        }
        printf("mass_inv=\n");
        for(i = 0; i < MAX_N_COEF; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
            {
                printf("%g ",mass_inv[i][j]);
            }
            printf("\n");
        }
        ****/


        /***
        tri_quadrature_13_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = w1*ff_accurate_magnetic_vort(cent, crds[0],j)*vh_val(crds[0],cent,k) +
                      w2*(ff_accurate_magnetic_vort(cent, crds[1],j)*vh_val(crds[1],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[2],j)*vh_val(crds[2],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[3],j)*vh_val(crds[3],cent,k)) +
                      w3*(ff_accurate_magnetic_vort(cent, crds[4],j)*vh_val(crds[4],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[5],j)*vh_val(crds[5],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[6],j)*vh_val(crds[6],cent,k)) +
                      w4*(ff_accurate_magnetic_vort(cent, crds[7],j)*vh_val(crds[7],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[8],j)*vh_val(crds[8],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[9],j)*vh_val(crds[9],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[10],j)*vh_val(crds[10],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[11],j)*vh_val(crds[11],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[12],j)*vh_val(crds[12],cent,k));
            }
        }
        ***/
        /// for 16 pts case
        tri_quadrature_16_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        // cords returns 16 points for 2D triangular integration using Gaussian quadrature
        /****
        printf("16points:\n");
        for(i = 0; i < 16; i++)
            printf("%g %g\n",crds[i][0],crds[i][1]);
        ****/

        // Integration of DG // Huijing
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = 0.0;
                for(i = 0; i < 16; i++)
                {
                    // rhs[j][k] += w[i]*ff_accurate_magnetic_vort(dbcent, crds[i],j,NO)*vh_val(crds[i],cent,k);
                    rhs[j][k] += w[i]*ff_accurate_magnetic_vort(dbcent, crds[i],j,YES)*vh_val(crds[i],cent,k);
                }
            }
        }

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);//mass_inv*rhs[i]

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }
        if(N_EQN == 8)
        {
            for(indx = 0; indx < MAX_N_COEF; indx++)
            {
                dg_Mom(state)[2][indx] = mulrhs[4][indx];
                dg_B(state)[0][indx] =  mulrhs[5][indx];
                dg_B(state)[1][indx] =  mulrhs[6][indx];
                dg_B(state)[2][indx] =  mulrhs[7][indx];
            }
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        Mom(state)[2] = 0.0;
        Mag(state)[0] = 0.0;
        Mag(state)[1] = 0.0;
        Mag(state)[2] = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Lmass_matrix[0][indx];
            Mom(state)[2] += dg_Mom(state)[2][indx]*Lmass_matrix[0][indx];
            Mag(state)[0] += dg_B(state)[0][indx]*Lmass_matrix[0][indx];
            Mag(state)[1] += dg_B(state)[1][indx]*Lmass_matrix[0][indx];
            Mag(state)[2] += dg_B(state)[2][indx]*Lmass_matrix[0][indx];
        }
        Dens(state) /= Lmass_matrix[0][0];
        Mom(state)[0] /= Lmass_matrix[0][0];
        Mom(state)[1] /= Lmass_matrix[0][0];
        Energy(state) /= Lmass_matrix[0][0];
        Mom(state)[2] /= Lmass_matrix[0][0];
        Mag(state)[0] /= Lmass_matrix[0][0];
        Mag(state)[1] /= Lmass_matrix[0][0];
        Mag(state)[2] /= Lmass_matrix[0][0];

        /// We also need to init edge B
        if(N_EQN == 8)
        {
            for(side = 0; side < 3; side++)
            {
                fg_side_B(tri)[side] = 0.0;
                // for(j = 0; j < N_EQN; j++)
                //     v_conU[side][j] = ff_accurate_magnetic_vort(dbcent, pcrds[side],j);
                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    fg_side_dgB(tri)[side][indx] = 0.0;//TMP huijing
                }
            }
 
            for(side = 0; side < 3; side++)
            {
                for(i = 0; i < 2; i++)
                    v2[i] = fg_side_vector(tri)[side][i];
                nor[0] = v2[1];
                nor[1] = -v2[0];

                for(k = 0; k < Lobatto_N; k++)//use Lobatto points(include vertices)
                //for(k = 0; k < Gauss_N; k++)
                {
                    // printf("k=%d q=%Lf qw=%Lf\n",k,q[k],qw[k]);
                    for(i = 0; i < 2; i++)
                        qcrds[i] = (pcrds[(side+1)%3][i] + pcrds[side][i])/2.0 +
                               (pcrds[(side+1)%3][i] - pcrds[side][i])/2.0*q[k];
                    for(j = 5; j < N_EQN; j++)
                        v_conU[side][j] = ff_accurate_magnetic_vort(dbcent, qcrds,j,YES);

                    Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];
                    fg_side_B(tri)[side] += Bn[k]*qw[k];
                    for(indx = 0; indx < N_COEF_EDGE; indx++)
                    {
                        //printf("q[%d]=%Lf vh_val_ver2=%g\n",indx,q[k],vh_val_ver2(q[k],indx));
                        fg_side_dgB(tri)[side][indx] += qw[k]*Bn[k]*vh_val_ver2(q[k],indx);//rhs
                    }
                }
                // fg_side_B(tri)[side] *= fg_length_side(tri)[side]/2.0;
                fg_side_B(tri)[side] *= 0.5;

                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    fg_side_dgB(tri)[side][indx] = fg_side_dgB(tri)[side][indx]*(2.0*indx+1.0)/2.0;
                }
            }
            tmp = fg_side_B(tri)[0]*fg_length_side(tri)[0] +
                  fg_side_B(tri)[1]*fg_length_side(tri)[1] +
                  fg_side_B(tri)[2]*fg_length_side(tri)[2];

            if(debug_flag == YES || fabs(tmp) > 1.0e-10)
            {
                printf("tri[%d] in accurate_magnetic_vortex_initializer_3(), divB = %14.13g, Bn = [%13.12g, %13.12g, %13.12g]\n",
                     tri->id, tmp,
                     fg_side_B(tri)[0], fg_side_B(tri)[1], fg_side_B(tri)[2]);
                /***
                for(j = 0; j < N_EQN; j++)
                    v_conU[0][j] = ff_accurate_magnetic_vort(dbcent, dbcent,j, YES);
                printf("velocity at cent[%g %g], B at cent[%g %g]\n",  v_conU[0][1], v_conU[0][2], v_conU[0][5], v_conU[0][6]);

                g_verbose_print_state(state);

                for(side =0; side < 3; side++)
                {
                    pcrds[0] = Coords(Point_of_tri(tri)[side]);
                    pcrds[1] = Coords(Point_of_tri(tri)[ (side +1)%3 ]);

                    dy = pcrds[1][1] - pcrds[0][1];
                    dx = pcrds[1][0] - pcrds[0][0];
                    len = fg_length_side(tri)[side];
    
                    sum = dy*dg_B(state)[0][0] + (0.5*dy*dx + dy*pcrds[0][0])*dg_B(state)[0][1] + (0.5*sqr(dy) + dy*pcrds[0][1])*dg_B(state)[0][2] +
                          -dx*dg_B(state)[1][0] - (0.5*sqr(dx) + dx*pcrds[0][0])*dg_B(state)[1][1] - (0.5*dx*dy + dx*pcrds[0][1])*dg_B(state)[1][2];
                    sum /= len;
                    printf("Side[%d], cell average recoved Bn = %13.12g\n", side, sum);
                    printf("Bx[%g, %g, %g]\n", dg_B(state)[0][0], dg_B(state)[0][1], dg_B(state)[0][2]);
                    printf("By[%g, %g, %g]\n", dg_B(state)[1][0], dg_B(state)[1][1], dg_B(state)[1][2]);
                }
                // clean_up(0);
                ***/
            }
        }

}

//basis for DG on edge
LOCAL   double vh_val_ver2(
        double x,
        int   indx)
{
        double ans;
        switch(indx)
        {
        case 0:
            ans = 1.0;
        break;
        case 1:
            ans = x;
        break;
        case 2:
            ans = 0.5*(3.0*sqr(x) - 1.0);
        break;
        case 3:
            ans = 0.5*(5.0*cub(x) - 3.0*x);
        break;
        case 4:
            ans = 0.125*(35.0*sqr(sqr(x)) - 30.0*sqr(x) + 3.0);
        break;
        case 5:
            ans = 0.125*(63.0*cub(sqr(x)) - 70.0*cub(x) + 15.0*x);
        break;
        default:
            printf("ERROR vh_val_ver2, implement 2D degree %d\n", indx);
            clean_up(ERROR);
        }
        return ans;
}




/// The initial condition is from J.C.P. 2010. 229, 7893-7910. 
/// Test case 5.2
/// To get the Lobatto Quadrature's Abscissas and Weight, ref:
/// Abscissas and Weight Coefficients for Lobatto Quadrature.
///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
///  Only init cell average. The energy does not have |B|^2.
LOCAL void accurate_magnetic_vortex_initializer_2(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[3], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[8][MAX_N_COEF], mulrhs[8][MAX_N_COEF], dens[8], tmp;
        float           area = fg_area(tri), t = 0.0;
        double          **mass_inv, **Lmass_matrix;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[3][8];
        double       *lcent = fg_centroid(tri);
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        static double **tmp_mass_inv = NULL;
        int           Lobatto_N = 7;
        long double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        /// TMP
        /**
        if(   (fabs(lcent[0]-tmpcent0[0])<1.0e-5  && 
               fabs(lcent[1]-tmpcent0[1])<1.0e-5) 
            ||
              (fabs(lcent[0]-tmpcent1[0])<1.0e-5  && 
               fabs(lcent[1]-tmpcent1[1])<1.0e-5)  
          )
        **/
        /***
        if(    
              (fabs(lcent[0]-tmpcent1[0])<1.0e-4  && 
               fabs(lcent[1]-tmpcent1[1])<1.0e-4)
          )
        {
            debug_flag = YES;
            printf("Tri[%d] entered accurate_magnetic_vortex_initializer_2()\n", tri->id);
            print_tri_crds(tri);
        }
        ***/
        /// END TMP

        if(Lobatto_N == 4)
        {
            q[0] = -1.0; q[1] = -sqrt(5.0)/5.0;
            q[2] =  sqrt(5.0)/5.0; q[3] = 1.0;
            qw[0] =  1.0/6.0; qw[1] = 5.0/6.0;
            qw[2] =  5.0/6.0; qw[3] = 1.0/6.0;

            // Gaussian
            // q[0] = -0.86113631159405257522; q[1] = -0.33998104358485626480;
            // q[2] =  0.33998104358485626480; q[3] = 0.86113631159405257522;
            // qw[0] =  0.34785484513745385737; qw[1] = 0.65214515486254614263;
            // qw[2] =  0.65214515486254614263; qw[3] = 0.34785484513745385737;
        }
        else if(Lobatto_N == 5)
        {
            q[0] = -1.0; q[1] = -sqrt(21.0)/7.0;
            q[2] = 0.0;  q[3] =  sqrt(21.0)/7.0; q[4] = 1.0;
            qw[0] =  0.1; qw[1] = 49.0/90.0; qw[2] = 32.0/45.0;
            qw[3] =  49.0/90.0; qw[4] = 0.1;
        }
        else if(Lobatto_N == 6)
        {
            q[0] = -1.0; q[1] = -sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));
            q[2] = -sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));  q[3] =  sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0))); 
            q[4] =  sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));     q[5] = 1.0;
            qw[0] =  1.0/15.0; qw[1] = 1.0/30.0*(14.0-sqrt(7.0)); 
            qw[2] = 1.0/30.0*(14.0+sqrt(7.0)); qw[3] =  1.0/30.0*(14.0+sqrt(7.0)); 
            qw[4] = 1.0/30.0*(14.0-sqrt(7.0));       qw[5] = 1.0/15.0;
        }
        else if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987; 
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905; 
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905; 
        }
        else 
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: accurate_magnetic_vortex_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }


        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;
        Lmass_matrix = tri->Lmass_matrix;
        if(tri->mass_inv != NULL)
            mass_inv = tri->mass_inv;
        else
        {
            if(MAX_N_COEF != 1)
            {
                mass_inv = tmp_mass_inv;
                inverse_matrix(Lmass_matrix,MAX_N_COEF,mass_inv);
            }
        }

        for(i = 0; i < 8; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
            cent[i] = dbcent[i] = fg_centroid(tri)[i];

        /***
        tri_quadrature_13_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = w1*ff_accurate_magnetic_vort(cent, crds[0],j)*vh_val(crds[0],cent,k) +
                      w2*(ff_accurate_magnetic_vort(cent, crds[1],j)*vh_val(crds[1],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[2],j)*vh_val(crds[2],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[3],j)*vh_val(crds[3],cent,k)) +
                      w3*(ff_accurate_magnetic_vort(cent, crds[4],j)*vh_val(crds[4],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[5],j)*vh_val(crds[5],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[6],j)*vh_val(crds[6],cent,k)) +
                      w4*(ff_accurate_magnetic_vort(cent, crds[7],j)*vh_val(crds[7],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[8],j)*vh_val(crds[8],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[9],j)*vh_val(crds[9],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[10],j)*vh_val(crds[10],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[11],j)*vh_val(crds[11],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[12],j)*vh_val(crds[12],cent,k));
            }
        }
        ***/
        /// for 16 pts case
        tri_quadrature_16_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            rhs[j][0] = 0.0;
            for(i = 0; i < 16; i++)
            {
                rhs[j][0] += w[i]*ff_accurate_magnetic_vort(dbcent, crds[i],j,NO);//Integration
            }
        }
        ///

        /**
        for(j = 0; j < N_EQN; j++)
        {
            rhs[j][0] *= area;
        }
        **/

        if(N_EQN == 4)
        {
            for(indx = 1; indx < MAX_N_COEF; indx++)
            {
                dg_Dens(state)[indx] = 0.0;
                dg_Mom(state)[0][indx] = 0.0;
                dg_Mom(state)[1][indx] = 0.0;
                dg_Energy(state)[indx] = 0.0;
            }
        }
        else 
        {
            for(indx = 1; indx < MAX_N_COEF; indx++)
            {
                dg_Dens(state)[indx] = 0.0;
                dg_Mom(state)[0][indx] = 0.0;
                dg_Mom(state)[1][indx] = 0.0;
                dg_Energy(state)[indx] = 0.0;
                // dg_Mom(state)[2][indx] = mulrhs[4][indx];
                dg_Mom(state)[2][indx] = 0.0;
                dg_B(state)[0][indx] =  0.0;
                dg_B(state)[1][indx] =  0.0;
                dg_B(state)[2][indx] =  0.0;
            }
        }

        // Compute average soln
        Dens(state) = dg_Dens(state)[0] = rhs[0][0];
        Mom(state)[0] = dg_Mom(state)[0][0] = rhs[1][0];
        Mom(state)[1] = dg_Mom(state)[1][0] = rhs[2][0];
        Energy(state) = dg_Energy(state)[0] = rhs[3][0]; // energy without |B|
        Mom(state)[2] = dg_Mom(state)[2][0] = 0.0;
        Mag(state)[0] = dg_B(state)[0][0] = rhs[5][0];
        Mag(state)[1] = dg_B(state)[1][0] = rhs[6][0];
        Mag(state)[2] = dg_B(state)[2][0] = 0.0;

        /// We also need to init edge B
        if(N_EQN == 8)
        {
            for(side=0; side < 3; side++)
            {
                fg_side_B(tri)[side] = 0.0;
                // for(j = 0; j < N_EQN; j++)
                //     v_conU[side][j] = ff_accurate_magnetic_vort(dbcent, pcrds[side],j);
                for(indx=0; indx < MAX_N_COEF; indx++)
                    fg_side_dgB(tri)[side][indx] = 0.0;//TMP huijing
            }
            /**
            for(side=0; side < 3; side++)
            {
                v_conU[side][1] /= v_conU[side][0];
                v_conU[side][2] /= v_conU[side][0];
                v_conU[side][4] /= v_conU[side][0];
            }
            **/
 
            for(side=0; side < 3; side++)
            {
                for(i = 0; i < 2; i++)
                    v2[i] = fg_side_vector(tri)[side][i];
                nor[0] = v2[1];
                nor[1] = -v2[0];

                for(k = 0; k < Lobatto_N; k++)
                {
                    for(i = 0; i < 2; i++)
                        qcrds[i] = (pcrds[(side+1)%3][i] + pcrds[side][i])/2.0 +
                               (pcrds[(side+1)%3][i] - pcrds[side][i])/2.0*q[k];
                    for(j = 5; j < N_EQN; j++)
                        v_conU[side][j] = ff_accurate_magnetic_vort(dbcent, qcrds,j,NO);

                    Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];
                    fg_side_B(tri)[side] += Bn[k]*qw[k];
                }
                // fg_side_B(tri)[side] *= fg_length_side(tri)[side]/2.0;
                fg_side_B(tri)[side] *= 0.5;
                fg_side_dgB(tri)[side][0] = fg_side_B(tri)[side];//TMP huijing
            }
            tmp = fg_side_B(tri)[0]*fg_length_side(tri)[0] +
                  fg_side_B(tri)[1]*fg_length_side(tri)[1] +
                  fg_side_B(tri)[2]*fg_length_side(tri)[2];

            if(debug_flag == YES || fabs(tmp) > 1.0e-10)
            {
                printf("tri[%d] in accurate_magnetic_vortex_initializer_2(), divg B = %14.13g, Bn = [%13.12g, %13.12g, %13.12g]\n",
                     tri->id, tmp,
                     fg_side_B(tri)[0], fg_side_B(tri)[1], fg_side_B(tri)[2]);
                /***
                for(j = 0; j < N_EQN; j++)
                    v_conU[0][j] = ff_accurate_magnetic_vort(dbcent, dbcent,j, YES);
                printf("velocity at cent[%g %g], B at cent[%g %g]\n",  v_conU[0][1], v_conU[0][2], v_conU[0][5], v_conU[0][6]);

                g_verbose_print_state(state);

                for(side =0; side < 3; side++)
                {
                    pcrds[0] = Coords(Point_of_tri(tri)[side]);
                    pcrds[1] = Coords(Point_of_tri(tri)[ (side +1)%3 ]);

                    dy = pcrds[1][1] - pcrds[0][1];
                    dx = pcrds[1][0] - pcrds[0][0];
                    len = fg_length_side(tri)[side];
    
                    sum = dy*dg_B(state)[0][0] + (0.5*dy*dx + dy*pcrds[0][0])*dg_B(state)[0][1] + (0.5*sqr(dy) + dy*pcrds[0][1])*dg_B(state)[0][2] +
                          -dx*dg_B(state)[1][0] - (0.5*sqr(dx) + dx*pcrds[0][0])*dg_B(state)[1][1] - (0.5*dx*dy + dx*pcrds[0][1])*dg_B(state)[1][2];
                    sum /= len;
                    printf("Side[%d], cell average recoved Bn = %13.12g\n", side, sum);
                    printf("Bx[%g, %g, %g]\n", dg_B(state)[0][0], dg_B(state)[0][1], dg_B(state)[0][2]);
                    printf("By[%g, %g, %g]\n", dg_B(state)[1][0], dg_B(state)[1][1], dg_B(state)[1][2]);
                }
                // clean_up(0);
                ***/
            }
        }

}

/// To get the Lobatto Quadrature's Abscissas and Weight, ref:
/// Abscissas and Weight Coefficients for Lobatto Quadrature.
///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
///  Only init cell average. The energy does not have |B|^2.
LOCAL void accurate_magnetic_rotor_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[3], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[8][MAX_N_COEF], mulrhs[8][MAX_N_COEF], dens[8], tmp;
        float           area = fg_area(tri), t = 0.0, sqrt_area;
        double          **Bmass_inv, **Bmass_matrix;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[3][8];
        double       *lcent = fg_centroid(tri);
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        static double **tmp_mass_inv = NULL;
        int           Lobatto_N = 7;
        long double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        if(Lobatto_N == 4)
        {
            q[0] = -1.0; q[1] = -sqrt(5.0)/5.0;
            q[2] =  sqrt(5.0)/5.0; q[3] = 1.0;
            qw[0] =  1.0/6.0; qw[1] = 5.0/6.0;
            qw[2] =  5.0/6.0; qw[3] = 1.0/6.0;
        }
        else if(Lobatto_N == 5)
        {
            q[0] = -1.0; q[1] = -sqrt(21.0)/7.0;
            q[2] = 0.0;  q[3] =  sqrt(21.0)/7.0; q[4] = 1.0;
            qw[0] =  0.1; qw[1] = 49.0/90.0; qw[2] = 32.0/45.0;
            qw[3] =  49.0/90.0; qw[4] = 0.1;
        }
        else if(Lobatto_N == 6)
        {
            q[0] = -1.0; q[1] = -sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));
            q[2] = -sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));  q[3] =  sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0))); 
            q[4] =  sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));     q[5] = 1.0;
            qw[0] =  1.0/15.0; qw[1] = 1.0/30.0*(14.0-sqrt(7.0)); 
            qw[2] = 1.0/30.0*(14.0+sqrt(7.0)); qw[3] =  1.0/30.0*(14.0+sqrt(7.0)); 
            qw[4] = 1.0/30.0*(14.0-sqrt(7.0));       qw[5] = 1.0/15.0;
        }
        else if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987; 
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905; 
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905; 
        }
        else 
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: accurate_magnetic_rotor_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }


        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;
        Bmass_matrix = tri->Bmass_matrix;
        sqrt_area = sqrt(area);

        if(tri->Bmass_inv != NULL)
            Bmass_inv = tri->Bmass_inv;
        else
        {
            if(MAX_N_COEF != 1)
            {
                Bmass_inv = tmp_mass_inv;
                inverse_matrix(Bmass_matrix,MAX_N_COEF,Bmass_inv);
            }
        }

        for(i = 0; i < 8; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
            cent[i] = dbcent[i] = fg_centroid(tri)[i];

        /***
        tri_quadrature_13_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = w1*ff_accurate_magnetic_vort(cent, crds[0],j)*vh_val(crds[0],cent,k) +
                      w2*(ff_accurate_magnetic_vort(cent, crds[1],j)*vh_val(crds[1],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[2],j)*vh_val(crds[2],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[3],j)*vh_val(crds[3],cent,k)) +
                      w3*(ff_accurate_magnetic_vort(cent, crds[4],j)*vh_val(crds[4],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[5],j)*vh_val(crds[5],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[6],j)*vh_val(crds[6],cent,k)) +
                      w4*(ff_accurate_magnetic_vort(cent, crds[7],j)*vh_val(crds[7],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[8],j)*vh_val(crds[8],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[9],j)*vh_val(crds[9],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[10],j)*vh_val(crds[10],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[11],j)*vh_val(crds[11],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[12],j)*vh_val(crds[12],cent,k));
            }
        }
        ***/
        /// for 16 pts case
        tri_quadrature_16_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = 0.0;
                for(i = 0; i < 16; i++)
                {
                    // rhs[j][0] += w[i]*ff_accurate_magnetic_rotor(dbcent, crds[i],j,NO);
                    // rhs[j][k] += w[i]*ff_accurate_magnetic_rotor(dbcent, crds[i],j,YES)*vh_val(crds[i],cent,k);
                    rhs[j][k] += w[i]*ff_accurate_magnetic_rotor(dbcent, crds[i],j,YES)*
                                   vh_val_loc_div_free_basis(crds[i],cent,sqrt_area,k);
                }
            }
        }
        ///

        for(j = 0; j < N_EQN; j++)
        {    
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            } 
        } 

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(Bmass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);//mass_inv*rhs[i]
    
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }
        if(N_EQN == 8)
        {
            for(indx = 0; indx < MAX_N_COEF; indx++)
            {
                dg_Mom(state)[2][indx] = mulrhs[4][indx];
                dg_B(state)[0][indx] =  mulrhs[5][indx];
                dg_B(state)[1][indx] =  mulrhs[6][indx];
                dg_B(state)[2][indx] =  mulrhs[7][indx];
            }
        }

        if(MAX_N_COEF == 3 || MAX_N_COEF == 6 || MAX_N_COEF == 10)
        {
            /// printf("EXIT in accurate_magnetic_vortex_initializer_4()\n");
            /// printf("dg_B(state)[0][1] = %14.13g, dg_B(state)[1][2] = %14.13g\n", 
            ///    dg_B(state)[0][1], dg_B(state)[1][2]);
            if(dg_B(state)[0][1] >= 0.0)
            {
                dg_B(state)[0][1] = 0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            else
            {
                dg_B(state)[0][1] = -0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            dg_B(state)[1][2] = -dg_B(state)[0][1];
            /// printf("--------dg_B(state)[0][1] = %14.13g, dg_B(state)[1][2] = %14.13g\n", 
            ///    dg_B(state)[0][1], dg_B(state)[1][2]);
        }

        if(MAX_N_COEF == 6)
        {
            if(dg_B(state)[0][3] >= 0.0)
                dg_B(state)[0][3] = 0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));
            else
                dg_B(state)[0][3] = -0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));

            dg_B(state)[1][4] = -2.0*dg_B(state)[0][3];

            if(dg_B(state)[0][4] >= 0.0)
                dg_B(state)[0][4] = 0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));
            else
                dg_B(state)[0][4] = -0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));

            dg_B(state)[1][5] = -0.5*dg_B(state)[0][4];
        }

        if(MAX_N_COEF == 10)
        {
            if(dg_B(state)[0][6] >= 0.0)
                dg_B(state)[0][6] = 0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            else
                dg_B(state)[0][6] = -0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            dg_B(state)[1][7] = -3.0*dg_B(state)[0][6];

            if(dg_B(state)[0][7] >= 0.0)
                dg_B(state)[0][7] = 0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            else
                dg_B(state)[0][7] = -0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            dg_B(state)[1][8] = -dg_B(state)[0][7];

            if(dg_B(state)[0][8] >= 0.0)
                dg_B(state)[0][8] = 0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            else
                dg_B(state)[0][8] = -0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            dg_B(state)[1][9] = -dg_B(state)[0][8]/3.0;
        }

        if(MAX_N_COEF != 1 && MAX_N_COEF != 3 && MAX_N_COEF != 6 && MAX_N_COEF != 10)
        {
            printf("EXIT in accurate_magnetic_rotor_initializer()\n");
            printf("initialize with divgerence free for MAX_N_COEF %d\n", MAX_N_COEF);
            clean_up(0);
        }


        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        Mom(state)[2] = 0.0;
        Mag(state)[0] = 0.0;
        Mag(state)[1] = 0.0;
        Mag(state)[2] = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Bmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Bmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[2] += dg_Mom(state)[2][indx]*Bmass_matrix[0][indx];
            Mag(state)[0] += dg_B(state)[0][indx]*Bmass_matrix[0][indx];
            Mag(state)[1] += dg_B(state)[1][indx]*Bmass_matrix[0][indx];
            Mag(state)[2] += dg_B(state)[2][indx]*Bmass_matrix[0][indx];
        }
        Dens(state) /= Bmass_matrix[0][0];
        Mom(state)[0] /= Bmass_matrix[0][0];
        Mom(state)[1] /= Bmass_matrix[0][0];
        Energy(state) /= Bmass_matrix[0][0];
        Mom(state)[2] /= Bmass_matrix[0][0];
        Mag(state)[0] /= Bmass_matrix[0][0];
        Mag(state)[1] /= Bmass_matrix[0][0];
        Mag(state)[2] /= Bmass_matrix[0][0];

        /****
        if(tri->id==2604 || tri->id==2722 ||tri->id==2721 ||tri->id==2723 || dg_Dens(state)[0] < 0 || Dens(state) < 0)
        {
            printf("tri[%d] cent(%g %g) \n",tri->id,cent[0],cent[1]);
            printf("dg_Dens=(%g %g %g) Dens(state)=%g\n",dg_Dens(state)[0],dg_Dens(state)[1],dg_Dens(state)[2],Dens(state));
        }
        ****/

        return; 
        ////  For the central-overlapping grid scheme, edge Bn is initialized on grid edges of dual grid.

        /// We also need to init edge B
        if(N_EQN == 8)
        {
            for(side = 0; side < 3; side++)
            {
                fg_side_B(tri)[side] = 0.0;
                // for(j = 0; j < N_EQN; j++)
                //     v_conU[side][j] = ff_accurate_magnetic_vort(dbcent, pcrds[side],j);
                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    fg_side_dgB(tri)[side][indx] = 0.0;//TMP huijing
                }
            }
            for(side = 0; side < 3; side++)
            {
                for(i = 0; i < 2; i++)
                    v2[i] = fg_side_vector(tri)[side][i];
                nor[0] = v2[1];
                nor[1] = -v2[0];

                for(k = 0; k < Lobatto_N; k++)//use Lobatto points(include vertices)
                //for(k = 0; k < Gauss_N; k++)
                {
                    // printf("k=%d q=%Lf qw=%Lf\n",k,q[k],qw[k]);
                    for(i = 0; i < 2; i++)
                        qcrds[i] = (pcrds[(side+1)%3][i] + pcrds[side][i])/2.0 +
                               (pcrds[(side+1)%3][i] - pcrds[side][i])/2.0*q[k];
                    for(j = 5; j < N_EQN; j++)
                        v_conU[side][j] = ff_accurate_magnetic_rotor(dbcent, qcrds,j,YES);

                    Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];
                    fg_side_B(tri)[side] += Bn[k]*qw[k];
                    for(indx = 0; indx < N_COEF_EDGE; indx++)
                    {
                        // printf("q[%d]=%Lf vh_val_ver2=%g\n",indx,q[k],vh_val_ver2(q[k],indx));
                        fg_side_dgB(tri)[side][indx] += qw[k]*Bn[k]*vh_val_ver2(q[k],indx);//rhs
                    }
                }
                // fg_side_B(tri)[side] *= fg_length_side(tri)[side]/2.0;
                fg_side_B(tri)[side] *= 0.5;

                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    fg_side_dgB(tri)[side][indx] = fg_side_dgB(tri)[side][indx]*(2.0*indx+1.0)/2.0;
                }
            }
            tmp = fg_side_B(tri)[0]*fg_length_side(tri)[0] +
                  fg_side_B(tri)[1]*fg_length_side(tri)[1] +
                  fg_side_B(tri)[2]*fg_length_side(tri)[2];

            if(debug_flag == YES || fabs(tmp) > 1.0e-12)
            {
                printf("tri[%d] in accurate_magnetic_rotor_initializer(),"
                   " divg B = %14.13g, Bn = [%13.12g, %13.12g, %13.12g]\n", tri->id, tmp,
                     fg_side_B(tri)[0], fg_side_B(tri)[1], fg_side_B(tri)[2]);
                /***
                for(j = 0; j < N_EQN; j++)
                    v_conU[0][j] = ff_accurate_magnetic_vort(dbcent, dbcent,j, YES);
                printf("velocity at cent[%g %g], B at cent[%g %g]\n",  v_conU[0][1], 
                              v_conU[0][2], v_conU[0][5], v_conU[0][6]);

                g_verbose_print_state(state);

                for(side =0; side < 3; side++)
                {
                    pcrds[0] = Coords(Point_of_tri(tri)[side]);
                    pcrds[1] = Coords(Point_of_tri(tri)[ (side +1)%3 ]);

                    dy = pcrds[1][1] - pcrds[0][1];
                    dx = pcrds[1][0] - pcrds[0][0];
                    len = fg_length_side(tri)[side];
    
                    sum = dy*dg_B(state)[0][0] + (0.5*dy*dx + dy*pcrds[0][0])*dg_B(state)[0][1] + (0.5*sqr(dy) + dy*pcrds[0][1])*dg_B(state)[0][2] +
                          -dx*dg_B(state)[1][0] - (0.5*sqr(dx) + dx*pcrds[0][0])*dg_B(state)[1][1] - (0.5*dx*dy + dx*pcrds[0][1])*dg_B(state)[1][2];
                    sum /= len;
                    printf("Side[%d], cell average recoved Bn = %13.12g\n", side, sum);
                    printf("Bx[%g, %g, %g]\n", dg_B(state)[0][0], dg_B(state)[0][1], dg_B(state)[0][2]);
                    printf("By[%g, %g, %g]\n", dg_B(state)[1][0], dg_B(state)[1][1], dg_B(state)[1][2]);
                }
                // clean_up(0);
                ***/
            }
        } /// if(N_EQN == 8)

}

LOCAL void accurate_Orszag_T_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[3], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[8][MAX_N_COEF], mulrhs[8][MAX_N_COEF], dens[8], tmp;
        float           area = fg_area(tri), t = 0.0, sqrt_area;
        double          **Bmass_inv, **Bmass_matrix;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[3][8];
        double       *lcent = fg_centroid(tri);
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        static double **tmp_mass_inv = NULL;
        int           Lobatto_N = 7;
        long double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        if(Lobatto_N == 4)
        {
            q[0] = -1.0; q[1] = -sqrt(5.0)/5.0;
            q[2] =  sqrt(5.0)/5.0; q[3] = 1.0;
            qw[0] =  1.0/6.0; qw[1] = 5.0/6.0;
            qw[2] =  5.0/6.0; qw[3] = 1.0/6.0;
        }
        else if(Lobatto_N == 5)
        {
            q[0] = -1.0; q[1] = -sqrt(21.0)/7.0;
            q[2] = 0.0;  q[3] =  sqrt(21.0)/7.0; q[4] = 1.0;
            qw[0] =  0.1; qw[1] = 49.0/90.0; qw[2] = 32.0/45.0;
            qw[3] =  49.0/90.0; qw[4] = 0.1;
        }
        else if(Lobatto_N == 6)
        {
            q[0] = -1.0; q[1] = -sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));
            q[2] = -sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));  q[3] =  sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0))); 
            q[4] =  sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));     q[5] = 1.0;
            qw[0] =  1.0/15.0; qw[1] = 1.0/30.0*(14.0-sqrt(7.0)); 
            qw[2] = 1.0/30.0*(14.0+sqrt(7.0)); qw[3] =  1.0/30.0*(14.0+sqrt(7.0)); 
            qw[4] = 1.0/30.0*(14.0-sqrt(7.0));       qw[5] = 1.0/15.0;
        }
        else if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987; 
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905; 
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905; 
        }
        else 
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: accurate_Orszag_T_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }


        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;
        Bmass_matrix = tri->Bmass_matrix;
        sqrt_area = sqrt(area);

        if(tri->Bmass_inv != NULL)
            Bmass_inv = tri->Bmass_inv;
        else
        {
            if(MAX_N_COEF != 1)
            {
                Bmass_inv = tmp_mass_inv;
                inverse_matrix(Bmass_matrix,MAX_N_COEF,Bmass_inv);
            }
        }

        for(i = 0; i < 8; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
            cent[i] = dbcent[i] = fg_centroid(tri)[i];

        /***
        tri_quadrature_13_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = w1*ff_accurate_magnetic_vort(cent, crds[0],j)*vh_val(crds[0],cent,k) +
                      w2*(ff_accurate_magnetic_vort(cent, crds[1],j)*vh_val(crds[1],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[2],j)*vh_val(crds[2],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[3],j)*vh_val(crds[3],cent,k)) +
                      w3*(ff_accurate_magnetic_vort(cent, crds[4],j)*vh_val(crds[4],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[5],j)*vh_val(crds[5],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[6],j)*vh_val(crds[6],cent,k)) +
                      w4*(ff_accurate_magnetic_vort(cent, crds[7],j)*vh_val(crds[7],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[8],j)*vh_val(crds[8],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[9],j)*vh_val(crds[9],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[10],j)*vh_val(crds[10],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[11],j)*vh_val(crds[11],cent,k) +
                          ff_accurate_magnetic_vort(cent, crds[12],j)*vh_val(crds[12],cent,k));
            }
        }
        ***/
        /// for 16 pts case
        tri_quadrature_16_pts(pcrds[0], pcrds[1], pcrds[2], crds);

        // Integration of DG // Huijing
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = 0.0;
                for(i = 0; i < 16; i++)
                {
                    // rhs[j][k] += w[i]*ff_accurate_Orszag_T(dbcent, crds[i],j,YES)*vh_val(crds[i],cent,k);
                    rhs[j][k] += w[i]*ff_accurate_Orszag_T(dbcent, crds[i],j,YES)*
                                 vh_val_loc_div_free_basis(crds[i],cent,sqrt_area,k);
                }
            }
        }

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(Bmass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);//mass_inv*rhs[i]

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }
        if(N_EQN == 8)
        {
            for(indx = 0; indx < MAX_N_COEF; indx++)
            {
                dg_Mom(state)[2][indx] = mulrhs[4][indx];
                dg_B(state)[0][indx] =  mulrhs[5][indx];
                dg_B(state)[1][indx] =  mulrhs[6][indx];
                dg_B(state)[2][indx] =  mulrhs[7][indx];
            }
        }

        if(MAX_N_COEF == 3 || MAX_N_COEF == 6 || MAX_N_COEF == 10)
        {
            if(dg_B(state)[0][1] >= 0.0)
            {
                dg_B(state)[0][1] = 0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            else
            {
                dg_B(state)[0][1] = -0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            dg_B(state)[1][2] = -dg_B(state)[0][1];
        }

        if(MAX_N_COEF == 6)
        {
            if(dg_B(state)[0][3] >= 0.0)
                dg_B(state)[0][3] = 0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));
            else
                dg_B(state)[0][3] = -0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));

            dg_B(state)[1][4] = -2.0*dg_B(state)[0][3];

            if(dg_B(state)[0][4] >= 0.0)
                dg_B(state)[0][4] = 0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));
            else
                dg_B(state)[0][4] = -0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));

            dg_B(state)[1][5] = -0.5*dg_B(state)[0][4];
        }

        if(MAX_N_COEF == 10)
        {
            if(dg_B(state)[0][6] >= 0.0)
                dg_B(state)[0][6] = 0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            else
                dg_B(state)[0][6] = -0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            dg_B(state)[1][7] = -3.0*dg_B(state)[0][6];

            if(dg_B(state)[0][7] >= 0.0)
                dg_B(state)[0][7] = 0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            else
                dg_B(state)[0][7] = -0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            dg_B(state)[1][8] = -dg_B(state)[0][7];

            if(dg_B(state)[0][8] >= 0.0)
                dg_B(state)[0][8] = 0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            else
                dg_B(state)[0][8] = -0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            dg_B(state)[1][9] = -dg_B(state)[0][8]/3.0;
        }

        if(MAX_N_COEF != 1 && MAX_N_COEF != 3 && MAX_N_COEF != 6 && MAX_N_COEF != 10)
        {
            printf("EXIT in accurate_Orszag_T_initializer()\n");
            printf("initialize with divgerence free for MAX_N_COEF %d\n", MAX_N_COEF);
            clean_up(0);
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        Mom(state)[2] = 0.0;
        Mag(state)[0] = 0.0;
        Mag(state)[1] = 0.0;
        Mag(state)[2] = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Bmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Bmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[2] += dg_Mom(state)[2][indx]*Bmass_matrix[0][indx];
            Mag(state)[0] += dg_B(state)[0][indx]*Bmass_matrix[0][indx];
            Mag(state)[1] += dg_B(state)[1][indx]*Bmass_matrix[0][indx];
            Mag(state)[2] += dg_B(state)[2][indx]*Bmass_matrix[0][indx];
        }
        Dens(state) /= Bmass_matrix[0][0];
        Mom(state)[0] /= Bmass_matrix[0][0];
        Mom(state)[1] /= Bmass_matrix[0][0];
        Energy(state) /= Bmass_matrix[0][0];
        Mom(state)[2] /= Bmass_matrix[0][0];
        Mag(state)[0] /= Bmass_matrix[0][0];
        Mag(state)[1] /= Bmass_matrix[0][0];
        Mag(state)[2] /= Bmass_matrix[0][0];

        return;
        ////  For the central-overlapping grid scheme, edge Bn is initialized on grid edges of dual grid.

        /// We also need to init edge B
        if(N_EQN == 8)
        {
            for(side = 0; side < 3; side++)
            {
                fg_side_B(tri)[side] = 0.0;
                // for(j = 0; j < N_EQN; j++)
                //     v_conU[side][j] = ff_accurate_magnetic_vort(dbcent, pcrds[side],j);
                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    fg_side_dgB(tri)[side][indx] = 0.0;//TMP huijing
                }
            }

            for(side = 0; side < 3; side++)
            {
                for(i = 0; i < 2; i++)
                    v2[i] = fg_side_vector(tri)[side][i];
                nor[0] = v2[1];
                nor[1] = -v2[0];

                for(k = 0; k < Lobatto_N; k++)//use Lobatto points(include vertices)
                //for(k = 0; k < Gauss_N; k++)
                {
                    // printf("k=%d q=%Lf qw=%Lf\n",k,q[k],qw[k]);
                    for(i = 0; i < 2; i++)
                        qcrds[i] = (pcrds[(side+1)%3][i] + pcrds[side][i])/2.0 +
                               (pcrds[(side+1)%3][i] - pcrds[side][i])/2.0*q[k];
                    for(j = 5; j < N_EQN; j++)
                        v_conU[side][j] = ff_accurate_Orszag_T(dbcent, qcrds,j,YES);

                    Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];
                    fg_side_B(tri)[side] += Bn[k]*qw[k];
                    for(indx = 0; indx < N_COEF_EDGE; indx++)
                    {
                        //printf("q[%d]=%Lf vh_val_ver2=%g\n",indx,q[k],vh_val_ver2(q[k],indx));
                        fg_side_dgB(tri)[side][indx] += qw[k]*Bn[k]*vh_val_ver2(q[k],indx);//rhs
                    }
                }
                // fg_side_B(tri)[side] *= fg_length_side(tri)[side]/2.0;
                fg_side_B(tri)[side] *= 0.5;

                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    fg_side_dgB(tri)[side][indx] = fg_side_dgB(tri)[side][indx]*(2.0*indx+1.0)/2.0;
                }
            }
            tmp = fg_side_B(tri)[0]*fg_length_side(tri)[0] +
                  fg_side_B(tri)[1]*fg_length_side(tri)[1] +
                  fg_side_B(tri)[2]*fg_length_side(tri)[2];

            if(debug_flag == YES || fabs(tmp) > 1.0e-12)
            {
                printf("tri[%d] in accurate_Orszag_T_initializer(), divg B = %14.13g, Bn = [%13.12g, %13.12g, %13.12g]\n",
                     tri->id, tmp,
                     fg_side_B(tri)[0], fg_side_B(tri)[1], fg_side_B(tri)[2]);
            }
        }

}

/// To get the Lobatto Quadrature's Abscissas and Weight, ref:
/// Abscissas and Weight Coefficients for Lobatto Quadrature.
///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
///  Only init cell average. The energy does not have |B|^2.
LOCAL void accurate_magnetic_blast_MHD_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[3], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[8][MAX_N_COEF], mulrhs[8][MAX_N_COEF], dens[8], tmp;
        float           area = fg_area(tri), t = 0.0, sqrt_area;
        double          **Bmass_inv, **Bmass_matrix;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[3][8];
        double       *lcent = fg_centroid(tri);
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        static double **tmp_mass_inv = NULL;
        int           Lobatto_N = 7;
        long double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        if(Lobatto_N == 4)
        {
            q[0] = -1.0; q[1] = -sqrt(5.0)/5.0;
            q[2] =  sqrt(5.0)/5.0; q[3] = 1.0;
            qw[0] =  1.0/6.0; qw[1] = 5.0/6.0;
            qw[2] =  5.0/6.0; qw[3] = 1.0/6.0;
        }
        else if(Lobatto_N == 5)
        {
            q[0] = -1.0; q[1] = -sqrt(21.0)/7.0;
            q[2] = 0.0;  q[3] =  sqrt(21.0)/7.0; q[4] = 1.0;
            qw[0] =  0.1; qw[1] = 49.0/90.0; qw[2] = 32.0/45.0;
            qw[3] =  49.0/90.0; qw[4] = 0.1;
        }
        else if(Lobatto_N == 6)
        {
            q[0] = -1.0; q[1] = -sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));
            q[2] = -sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));  q[3] =  sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0))); 
            q[4] =  sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));     q[5] = 1.0;
            qw[0] =  1.0/15.0; qw[1] = 1.0/30.0*(14.0-sqrt(7.0)); 
            qw[2] = 1.0/30.0*(14.0+sqrt(7.0)); qw[3] =  1.0/30.0*(14.0+sqrt(7.0)); 
            qw[4] = 1.0/30.0*(14.0-sqrt(7.0));       qw[5] = 1.0/15.0;
        }
        else if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987; 
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905; 
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905; 
        }
        else 
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: accurate_magnetic_blast_MHD_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }


        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;
        Bmass_matrix = tri->Bmass_matrix;
        sqrt_area = sqrt(area);

        if(tri->Bmass_inv != NULL)
            Bmass_inv = tri->Bmass_inv;
        else
        {
            Bmass_inv = tmp_mass_inv;
            inverse_matrix(Bmass_matrix,MAX_N_COEF,Bmass_inv);
        }

        for(i = 0; i < 8; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
            cent[i] = dbcent[i] = fg_centroid(tri)[i];

        /// for 16 pts case
        /***
        tri_quadrature_16_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            rhs[j][0] = 0.0;
            for(i = 0; i < 16; i++)
            {
                rhs[j][0] += w[i]*ff_accurate_magnetic_blast(dbcent, crds[i],j,YES);
            }
        }
        ***/ 

        //For FV
        /****
        for(j = 0; j < N_EQN; j++)
        {
            rhs[j][0] = ff_accurate_magnetic_blast(dbcent, dbcent,j,YES);
        }
        ****/

        //// Simply initialize piece-wise const states.
        tri_quadrature_16_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = 0.0;
                for(i = 0; i < 16; i++)
                {
                    // rhs[j][k] += w[i]*ff_accurate_magnetic_blast(dbcent, crds[i],j,YES)*
                    //               vh_val_loc_div_free_basis(crds[i],cent,sqrt_area,k);
                    rhs[j][k] += w[i]*ff_accurate_magnetic_blast(dbcent, dbcent,j,YES)*
                                  vh_val_loc_div_free_basis(crds[i],cent,sqrt_area,k);
                }
            }
        }

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(Bmass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);//mass_inv*rhs[i]

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }
        if(N_EQN == 8)
        {
            for(indx = 0; indx < MAX_N_COEF; indx++)
            {
                dg_Mom(state)[2][indx] = mulrhs[4][indx];
                dg_B(state)[0][indx] =  mulrhs[5][indx];
                dg_B(state)[1][indx] =  mulrhs[6][indx];
                dg_B(state)[2][indx] =  mulrhs[7][indx];
            }
        }

        if(MAX_N_COEF == 3 || MAX_N_COEF == 6 || MAX_N_COEF == 10)
        {
            /// printf("EXIT in accurate_magnetic_vortex_initializer_4()\n");
            /// printf("dg_B(state)[0][1] = %14.13g, dg_B(state)[1][2] = %14.13g\n", 
            ///    dg_B(state)[0][1], dg_B(state)[1][2]);
            if(dg_B(state)[0][1] >= 0.0)
            {
                dg_B(state)[0][1] = 0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            else
            {
                dg_B(state)[0][1] = -0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            dg_B(state)[1][2] = -dg_B(state)[0][1];
            /// printf("--------dg_B(state)[0][1] = %14.13g, dg_B(state)[1][2] = %14.13g\n", 
            ///    dg_B(state)[0][1], dg_B(state)[1][2]);
        }

        if(MAX_N_COEF == 6)
        {
            if(dg_B(state)[0][3] >= 0.0)
                dg_B(state)[0][3] = 0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));
            else
                dg_B(state)[0][3] = -0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));

            dg_B(state)[1][4] = -2.0*dg_B(state)[0][3];

            if(dg_B(state)[0][4] >= 0.0)
                dg_B(state)[0][4] = 0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));
            else
                dg_B(state)[0][4] = -0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));

            dg_B(state)[1][5] = -0.5*dg_B(state)[0][4];
        }

        if(MAX_N_COEF == 10)
        {
            if(dg_B(state)[0][6] >= 0.0)
                dg_B(state)[0][6] = 0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            else
                dg_B(state)[0][6] = -0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            dg_B(state)[1][7] = -3.0*dg_B(state)[0][6];

            if(dg_B(state)[0][7] >= 0.0)
                dg_B(state)[0][7] = 0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            else
                dg_B(state)[0][7] = -0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            dg_B(state)[1][8] = -dg_B(state)[0][7];

            if(dg_B(state)[0][8] >= 0.0)
                dg_B(state)[0][8] = 0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            else
                dg_B(state)[0][8] = -0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            dg_B(state)[1][9] = -dg_B(state)[0][8]/3.0;
        }

        if(MAX_N_COEF != 1 && MAX_N_COEF != 3 && MAX_N_COEF != 6 && MAX_N_COEF != 10)
        {
            printf("EXIT in accurate_magnetic_blast_MHD_initializer()\n");
            printf("initialize with divgerence free for MAX_N_COEF %d\n", MAX_N_COEF);
            clean_up(0);
        }


        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        Mom(state)[2] = 0.0;
        Mag(state)[0] = 0.0;
        Mag(state)[1] = 0.0;
        Mag(state)[2] = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Bmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Bmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[2] += dg_Mom(state)[2][indx]*Bmass_matrix[0][indx];
            Mag(state)[0] += dg_B(state)[0][indx]*Bmass_matrix[0][indx];
            Mag(state)[1] += dg_B(state)[1][indx]*Bmass_matrix[0][indx];
            Mag(state)[2] += dg_B(state)[2][indx]*Bmass_matrix[0][indx];
        }
        Dens(state) /= Bmass_matrix[0][0];
        Mom(state)[0] /= Bmass_matrix[0][0];
        Mom(state)[1] /= Bmass_matrix[0][0];
        Energy(state) /= Bmass_matrix[0][0];
        Mom(state)[2] /= Bmass_matrix[0][0];
        Mag(state)[0] /= Bmass_matrix[0][0];
        Mag(state)[1] /= Bmass_matrix[0][0];
        Mag(state)[2] /= Bmass_matrix[0][0];

        return;

        /// We also need to init edge B
        if(N_EQN == 8)
        {
            for(side = 0; side < 3; side++)
            {
                fg_side_B(tri)[side] = 0.0;
                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    fg_side_dgB(tri)[side][indx] = 0.0;//TMP huijing
                }
            }
            for(side = 0; side < 3; side++)
            {
                for(i = 0; i < 2; i++)
                    v2[i] = fg_side_vector(tri)[side][i];
                nor[0] = v2[1];
                nor[1] = -v2[0];

                for(k = 0; k < Lobatto_N; k++)//use Lobatto points(include vertices)
                //for(k = 0; k < Gauss_N; k++)
                {
                    // printf("k=%d q=%Lf qw=%Lf\n",k,q[k],qw[k]);
                    for(i = 0; i < 2; i++)
                        qcrds[i] = (pcrds[(side+1)%3][i] + pcrds[side][i])/2.0 +
                               (pcrds[(side+1)%3][i] - pcrds[side][i])/2.0*q[k];
                    for(j = 5; j < N_EQN; j++)
                        v_conU[side][j] = ff_accurate_magnetic_blast(dbcent, qcrds,j,YES);

                    Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];
                    fg_side_B(tri)[side] += Bn[k]*qw[k];
                    for(indx = 0; indx < N_COEF_EDGE; indx++)
                    {
                        // printf("q[%d]=%Lf vh_val_ver2=%g\n",indx,q[k],vh_val_ver2(q[k],indx));
                        fg_side_dgB(tri)[side][indx] += qw[k]*Bn[k]*vh_val_ver2(q[k],indx);//rhs
                    }
                }

                // fg_side_B(tri)[side] *= fg_length_side(tri)[side]/2.0;
                fg_side_B(tri)[side] *= 0.5;

                for(indx = 0; indx < N_COEF_EDGE; indx++)
                {
                    fg_side_dgB(tri)[side][indx] = fg_side_dgB(tri)[side][indx]*(2.0*indx+1.0)/2.0;
                }
            }
            tmp = fg_side_B(tri)[0]*fg_length_side(tri)[0] +
                  fg_side_B(tri)[1]*fg_length_side(tri)[1] +
                  fg_side_B(tri)[2]*fg_length_side(tri)[2];

            if(debug_flag == YES || fabs(tmp) > 1.0e-10)
            {
                printf("tri[%d] in accurate_magnetic_blast_initializer(), divg B = %14.13g, Bn = [%13.12g, %13.12g, %13.12g]\n",
                     tri->id, tmp,
                     fg_side_B(tri)[0], fg_side_B(tri)[1], fg_side_B(tri)[2]);
                /***
                for(j = 0; j < N_EQN; j++)
                    v_conU[0][j] = ff_accurate_magnetic_vort(dbcent, dbcent,j, YES);
                printf("velocity at cent[%g %g], B at cent[%g %g]\n",  v_conU[0][1], v_conU[0][2], v_conU[0][5], v_conU[0][6]);

                g_verbose_print_state(state);

                for(side =0; side < 3; side++)
                {
                    pcrds[0] = Coords(Point_of_tri(tri)[side]);
                    pcrds[1] = Coords(Point_of_tri(tri)[ (side +1)%3 ]);

                    dy = pcrds[1][1] - pcrds[0][1];
                    dx = pcrds[1][0] - pcrds[0][0];
                    len = fg_length_side(tri)[side];
    
                    sum = dy*dg_B(state)[0][0] + (0.5*dy*dx + dy*pcrds[0][0])*dg_B(state)[0][1] + (0.5*sqr(dy) + dy*pcrds[0][1])*dg_B(state)[0][2] +
                          -dx*dg_B(state)[1][0] - (0.5*sqr(dx) + dx*pcrds[0][0])*dg_B(state)[1][1] - (0.5*dx*dy + dx*pcrds[0][1])*dg_B(state)[1][2];
                    sum /= len;
                    printf("Side[%d], cell average recoved Bn = %13.12g\n", side, sum);
                    printf("Bx[%g, %g, %g]\n", dg_B(state)[0][0], dg_B(state)[0][1], dg_B(state)[0][2]);
                    printf("By[%g, %g, %g]\n", dg_B(state)[1][0], dg_B(state)[1][1], dg_B(state)[1][2]);
                }
                // clean_up(0);
                ***/
            }
        }
}

/// To get the Lobatto Quadrature's Abscissas and Weight, ref:
/// Abscissas and Weight Coefficients for Lobatto Quadrature.
///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
///  Only init cell average. The energy does not have |B|^2.
LOCAL void accurate_magnetic_decay_alfven_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[3], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[8][MAX_N_COEF], mulrhs[8][MAX_N_COEF], dens[8], tmp;
        float           area = fg_area(tri), t = 0.0, sqrt_area;
        double          **Bmass_inv, **Bmass_matrix;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[3][8];
        double       *lcent = fg_centroid(tri);
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        static double **tmp_mass_inv = NULL;
        int           Lobatto_N = 7;
        long double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};

        if(Lobatto_N == 4)
        {
            q[0] = -1.0; q[1] = -sqrt(5.0)/5.0;
            q[2] =  sqrt(5.0)/5.0; q[3] = 1.0;
            qw[0] =  1.0/6.0; qw[1] = 5.0/6.0;
            qw[2] =  5.0/6.0; qw[3] = 1.0/6.0;
        }
        else if(Lobatto_N == 5)
        {
            q[0] = -1.0; q[1] = -sqrt(21.0)/7.0;
            q[2] = 0.0;  q[3] =  sqrt(21.0)/7.0; q[4] = 1.0;
            qw[0] =  0.1; qw[1] = 49.0/90.0; qw[2] = 32.0/45.0;
            qw[3] =  49.0/90.0; qw[4] = 0.1;
        }
        else if(Lobatto_N == 6)
        {
            q[0] = -1.0; q[1] = -sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));
            q[2] = -sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));  q[3] =  sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0))); 
            q[4] =  sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));     q[5] = 1.0;
            qw[0] =  1.0/15.0; qw[1] = 1.0/30.0*(14.0-sqrt(7.0)); 
            qw[2] = 1.0/30.0*(14.0+sqrt(7.0)); qw[3] =  1.0/30.0*(14.0+sqrt(7.0)); 
            qw[4] = 1.0/30.0*(14.0-sqrt(7.0));       qw[5] = 1.0/15.0;
        }
        else if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987; 
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905; 
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905; 
        }
        else 
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: accurate_magnetic_rotor_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }


        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;
        Bmass_matrix = tri->Bmass_matrix;
        if(tri->Bmass_inv != NULL)
            Bmass_inv = tri->Bmass_inv;

        sqrt_area = sqrt(area);

        for(i = 0; i < 8; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
            cent[i] = dbcent[i] = fg_centroid(tri)[i];

        /// for 16 pts case
        tri_quadrature_16_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = 0.0;
                for(i = 0; i < 16; i++)
                {
                    rhs[j][k] += w[i]*ff_accurate_magnetic_decay_alfven(dbcent, crds[i],j,YES)
                                 *vh_val_loc_div_free_basis(crds[i],cent,sqrt_area,k);
                }
            }
        }
        ///

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(Bmass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);//mass_inv*rhs[i]

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }
        if(N_EQN == 8)
        {
            for(indx = 0; indx < MAX_N_COEF; indx++)
            {
                dg_Mom(state)[2][indx] = mulrhs[4][indx];
                dg_B(state)[0][indx] =  mulrhs[5][indx];
                dg_B(state)[1][indx] =  mulrhs[6][indx];
                dg_B(state)[2][indx] =  mulrhs[7][indx];
            }
        }

        if(MAX_N_COEF == 3 || MAX_N_COEF == 6 || MAX_N_COEF == 10)
        {
            if(dg_B(state)[0][1] >= 0.0)
            {
                dg_B(state)[0][1] = 0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            else
            {
                dg_B(state)[0][1] = -0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            dg_B(state)[1][2] = -dg_B(state)[0][1];
        }

        if(MAX_N_COEF == 6)
        {
            if(dg_B(state)[0][3] >= 0.0)
                dg_B(state)[0][3] = 0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));
            else
                dg_B(state)[0][3] = -0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));

            dg_B(state)[1][4] = -2.0*dg_B(state)[0][3];

            if(dg_B(state)[0][4] >= 0.0)
                dg_B(state)[0][4] = 0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));
            else
                dg_B(state)[0][4] = -0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));

            dg_B(state)[1][5] = -0.5*dg_B(state)[0][4];
        }

        if(MAX_N_COEF == 10)
        {
            if(dg_B(state)[0][6] >= 0.0)
                dg_B(state)[0][6] = 0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            else
                dg_B(state)[0][6] = -0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            dg_B(state)[1][7] = -3.0*dg_B(state)[0][6];

            if(dg_B(state)[0][7] >= 0.0)
                dg_B(state)[0][7] = 0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            else
                dg_B(state)[0][7] = -0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            dg_B(state)[1][8] = -dg_B(state)[0][7];

            if(dg_B(state)[0][8] >= 0.0)
                dg_B(state)[0][8] = 0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            else
                dg_B(state)[0][8] = -0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            dg_B(state)[1][9] = -dg_B(state)[0][8]/3.0;
        }

        if(MAX_N_COEF != 1 && MAX_N_COEF != 3 && MAX_N_COEF != 6 && MAX_N_COEF != 10)
        {
            printf("EXIT in accurate_magnetic_vortex_initializer_4()\n");
            printf("initialize with divgerence free for MAX_N_COEF %d\n", MAX_N_COEF);
            clean_up(0);
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        Mom(state)[2] = 0.0;
        Mag(state)[0] = 0.0;
        Mag(state)[1] = 0.0;
        Mag(state)[2] = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Bmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Bmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[2] += dg_Mom(state)[2][indx]*Bmass_matrix[0][indx];
            Mag(state)[0] += dg_B(state)[0][indx]*Bmass_matrix[0][indx];
            Mag(state)[1] += dg_B(state)[1][indx]*Bmass_matrix[0][indx];
            Mag(state)[2] += dg_B(state)[2][indx]*Bmass_matrix[0][indx];
        }
        Dens(state) /= Bmass_matrix[0][0];
        Mom(state)[0] /= Bmass_matrix[0][0];
        Mom(state)[1] /= Bmass_matrix[0][0];
        Energy(state) /= Bmass_matrix[0][0];
        Mom(state)[2] /= Bmass_matrix[0][0];
        Mag(state)[0] /= Bmass_matrix[0][0];
        Mag(state)[1] /= Bmass_matrix[0][0];
        Mag(state)[2] /= Bmass_matrix[0][0];

        /// We also need to init edge B
        /***
        if(N_EQN == 8)
        {
            for(side=0; side < 3; side++)
            {
                fg_side_B(tri)[side] = 0.0;
            }
 
            for(side=0; side < 3; side++)
            {
                for(i = 0; i < 2; i++)
                    v2[i] = fg_side_vector(tri)[side][i];
                nor[0] = v2[1];
                nor[1] = -v2[0];

                for(k = 0; k < Lobatto_N; k++)
                {
                    for(i = 0; i < 2; i++)
                        qcrds[i] = (pcrds[(side+1)%3][i] + pcrds[side][i])/2.0 +
                               (pcrds[(side+1)%3][i] - pcrds[side][i])/2.0*q[k];
                    for(j = 5; j < N_EQN; j++)
                        v_conU[side][j] = ff_accurate_magnetic_decay_alfven(dbcent, qcrds,j,YES);

                    Bn[k] = nor[0]*v_conU[side][5] + nor[1]*v_conU[side][6];
                    fg_side_B(tri)[side] += Bn[k]*qw[k];
                }
                // fg_side_B(tri)[side] *= fg_length_side(tri)[side]/2.0;
                fg_side_B(tri)[side] *= 0.5;
            }
            tmp = fg_side_B(tri)[0]*fg_length_side(tri)[0] +
                  fg_side_B(tri)[1]*fg_length_side(tri)[1] +
                  fg_side_B(tri)[2]*fg_length_side(tri)[2];

            if(debug_flag == YES || fabs(tmp) > 1.0e-10)
            {
                printf("tri[%d] in accurate_magnetic_decay_alfven_initializer(), divg B = %14.13g, Bn = [%13.12g, %13.12g, %13.12g]\n",
                     tri->id, tmp,
                     fg_side_B(tri)[0], fg_side_B(tri)[1], fg_side_B(tri)[2]);
            }
        }
        ***/
}

LOCAL void field_loop_advection_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle;
        POINT           *p[3];
        float           *pcrds[3], dbcent[MAXD], qcrds[3];
        // float           Gam, S, gam = 1.4;
        double          rhs[8][MAX_N_COEF], mulrhs[8][MAX_N_COEF], dens[8], tmp, tmp2;
        float           area = fg_area(tri), t = 0.0, sqrt_area;
        double          **Bmass_inv, **Bmass_matrix;
        int             debug_flag = NO, i, j, k, indx, side;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
              w3 = 0.175615257433204, w4 = 0.077113760890257;
        double       crds[16][2], v_conU[3][8];
        double       *lcent = fg_centroid(tri);
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        static double **tmp_mass_inv = NULL;
        int           Lobatto_N = 7;
        long double   q[9], qw[9], Bn[9];
        double        tmpcent0[2] = {-3.33333, -3.66667}, dx, dy, sum, len;
        double        tmpcent1[2] = {0.666667, -1.6666667};
        double        dir[2][2] = {{1.0, 0.0}, {0.0, 1.0}}, Az[3];

        if(Lobatto_N == 4)
        {
            q[0] = -1.0; q[1] = -sqrt(5.0)/5.0;
            q[2] =  sqrt(5.0)/5.0; q[3] = 1.0;
            qw[0] =  1.0/6.0; qw[1] = 5.0/6.0;
            qw[2] =  5.0/6.0; qw[3] = 1.0/6.0;
        }
        else if(Lobatto_N == 5)
        {
            q[0] = -1.0; q[1] = -sqrt(21.0)/7.0;
            q[2] = 0.0;  q[3] =  sqrt(21.0)/7.0; q[4] = 1.0;
            qw[0] =  0.1; qw[1] = 49.0/90.0; qw[2] = 32.0/45.0;
            qw[3] =  49.0/90.0; qw[4] = 0.1;
        }
        else if(Lobatto_N == 6)
        {
            q[0] = -1.0; q[1] = -sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));
            q[2] = -sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0)));  q[3] =  sqrt(1.0/21.0*(7.0-2.0*sqrt(7.0))); 
            q[4] =  sqrt(1.0/21.0*(7.0+2.0*sqrt(7.0)));     q[5] = 1.0;
            qw[0] =  1.0/15.0; qw[1] = 1.0/30.0*(14.0-sqrt(7.0)); 
            qw[2] = 1.0/30.0*(14.0+sqrt(7.0)); qw[3] =  1.0/30.0*(14.0+sqrt(7.0)); 
            qw[4] = 1.0/30.0*(14.0-sqrt(7.0));       qw[5] = 1.0/15.0;
        }
        else if(Lobatto_N == 7)
        {
            q[0] = -1.0; q[1] = -0.83022389627856692987; 
            q[2] = -0.46884879347071421380; q[3]  = 0.0;
            q[4] = 0.46884879347071421380; q[5] = 0.83022389627856692987;
            q[6] = 1.0;

            qw[0] =  0.04761904761904761905; 
            qw[1] =  0.27682604736156594801;
            qw[2] =  0.43174538120986262342;
            qw[3] =  0.48761904761904761905;
            qw[4] =  0.43174538120986262342;
            qw[5] =  0.27682604736156594801;
            qw[6] =  0.04761904761904761905; 
        }
        else 
        {
            /// To get the Lobatto Quadrature's Abscissas and Weight, ref:
            /// Abscissas and Weight Coefficients for Lobatto Quadrature.
            ///  H.H.Michels. 1963, Mathematics of Computation, Vol. 17, No. 83 (Jul., 1963), pp. 237-244.
            printf("ERROR: field_loop_advection_initializer(), do not have Lobatto quadrature for %d\n",
                     Lobatto_N);
            clean_up(ERROR);
        }


        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;
        Bmass_matrix = tri->Bmass_matrix;
        if(tri->Bmass_inv != NULL)
            Bmass_inv = tri->Bmass_inv;

        sqrt_area = sqrt(area);

        for(i = 0; i < 8; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
            cent[i] = dbcent[i] = fg_centroid(tri)[i];

        /// for 16 pts case
        tri_quadrature_16_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = 0.0;
                for(i = 0; i < 16; i++)
                {
                    rhs[j][k] += w[i]*ff_accurate_field_loop_advection(dbcent, crds[i],j,YES)
                                 *vh_val_loc_div_free_basis(crds[i],cent,sqrt_area,k);
                }
            }
        }
        ///

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(Bmass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);//mass_inv*rhs[i]

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }
 
        // THE Magnetic field will be initialized using edge values again.
        if(N_EQN == 8)
        {
            for(indx = 0; indx < MAX_N_COEF; indx++)
            {
                dg_Mom(state)[2][indx] = mulrhs[4][indx];
                dg_B(state)[0][indx] =  mulrhs[5][indx];
                dg_B(state)[1][indx] =  mulrhs[6][indx];
                dg_B(state)[2][indx] =  mulrhs[7][indx];
            }
        }

        if(MAX_N_COEF == 3 || MAX_N_COEF == 6 || MAX_N_COEF == 10)
        {
            if(dg_B(state)[0][1] >= 0.0)
            {
                dg_B(state)[0][1] = 0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            else
            {
                dg_B(state)[0][1] = -0.5*(fabs(dg_B(state)[0][1]) + fabs(dg_B(state)[1][2]));
            }
            dg_B(state)[1][2] = -dg_B(state)[0][1];
        }

        if(MAX_N_COEF == 6)
        {
            if(dg_B(state)[0][3] >= 0.0)
                dg_B(state)[0][3] = 0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));
            else
                dg_B(state)[0][3] = -0.5*(fabs(dg_B(state)[0][3]) + 0.5*fabs(dg_B(state)[1][4]));

            dg_B(state)[1][4] = -2.0*dg_B(state)[0][3];

            if(dg_B(state)[0][4] >= 0.0)
                dg_B(state)[0][4] = 0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));
            else
                dg_B(state)[0][4] = -0.5*(fabs(dg_B(state)[0][4]) + 2.0*fabs(dg_B(state)[1][5]));

            dg_B(state)[1][5] = -0.5*dg_B(state)[0][4];
        }

        if(MAX_N_COEF == 10)
        {
            if(dg_B(state)[0][6] >= 0.0)
                dg_B(state)[0][6] = 0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            else
                dg_B(state)[0][6] = -0.5*(fabs(dg_B(state)[0][6]) + fabs(dg_B(state)[1][7])/3.0);
            dg_B(state)[1][7] = -3.0*dg_B(state)[0][6];

            if(dg_B(state)[0][7] >= 0.0)
                dg_B(state)[0][7] = 0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            else
                dg_B(state)[0][7] = -0.5*(fabs(dg_B(state)[0][7]) + fabs(dg_B(state)[1][8]));
            dg_B(state)[1][8] = -dg_B(state)[0][7];

            if(dg_B(state)[0][8] >= 0.0)
                dg_B(state)[0][8] = 0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            else
                dg_B(state)[0][8] = -0.5*(fabs(dg_B(state)[0][8]) + 3.0*fabs(dg_B(state)[1][9]));
            dg_B(state)[1][9] = -dg_B(state)[0][8]/3.0;
        }

        if(MAX_N_COEF != 1 && MAX_N_COEF != 3 && MAX_N_COEF != 6 && MAX_N_COEF != 10)
        {
            printf("EXIT in ff_accurate_field_loop_advection()\n");
            printf("initialize with divgerence free for MAX_N_COEF %d\n", MAX_N_COEF);
            clean_up(0);
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        Mom(state)[2] = 0.0;
        Mag(state)[0] = 0.0;
        Mag(state)[1] = 0.0;
        Mag(state)[2] = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Bmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Bmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Bmass_matrix[0][indx];
            Mom(state)[2] += dg_Mom(state)[2][indx]*Bmass_matrix[0][indx];
            Mag(state)[0] += dg_B(state)[0][indx]*Bmass_matrix[0][indx];
            Mag(state)[1] += dg_B(state)[1][indx]*Bmass_matrix[0][indx];
            Mag(state)[2] += dg_B(state)[2][indx]*Bmass_matrix[0][indx];
        }
        Dens(state) /= Bmass_matrix[0][0];
        Mom(state)[0] /= Bmass_matrix[0][0];
        Mom(state)[1] /= Bmass_matrix[0][0];
        Energy(state) /= Bmass_matrix[0][0];
        Mom(state)[2] /= Bmass_matrix[0][0];
        Mag(state)[0] /= Bmass_matrix[0][0];
        Mag(state)[1] /= Bmass_matrix[0][0];
        Mag(state)[2] /= Bmass_matrix[0][0];

        // Use magnetic potential to init. B on tri edges. 
        for(side = 0; side < 3; side++)
            Az[side] = Az_field_loop_adv(pcrds[side]);

        for(side = 0; side < 3; side++)
        {
            for(indx = 1; indx < N_COEF_EDGE; indx++)
                fg_side_dgB(tri)[side][indx] = 0.0;
        }

        sum = 0.0;
        for(side = 0; side < 3; side++)
        {
            len = fg_length_side(tri)[side];         
            fg_side_dgB(tri)[side][0] = (Az[(side+1)%3] - Az[side])/len; 
            fg_side_B(tri)[side] = fg_side_dgB(tri)[side][0];
            sum += fg_side_dgB(tri)[side][0]*len;
        }

        if(fabs(sum) > 1.0e-12)
        {
            printf("ERROR: field_loop_advection_initializer() not div-free %e\n", sum);
            clean_up(ERROR); 
        }

        /***
        sum = 0.0;
        // first do vertical or horizontal edges 
        for(side = 0; side < 3; side++)
        {
            for(i = 0; i < 2; i++)
                v2[i] = fg_side_vector(tri)[side][i];
            nor[0] = v2[1];
            nor[1] = -v2[0];
            tmp  = fabs(nor[0]*dir[0][0] + nor[1]*dir[0][1]);
            tmp2 = fabs(nor[0]*dir[1][0] + nor[1]*dir[1][1]);
            len = fg_length_side(tri)[side];         

            // NOTE: the direction is consistent with edge normal of tri.
            if(tmp > 0.5 && fabs(nor[0]) > 0.5) // vertical edge of tri.
            {
                fg_side_dgB(tri)[side][0] = (Az[(side+1)%3] - Az[side])/len; 
                fg_side_B(tri)[side] = fg_side_dgB(tri)[side][0];
                sum += fg_side_dgB(tri)[side][0]*len;
            }
            else if(tmp2 > 0.5 && fabs(nor[1]) > 0.5) // horizontal edge of tri.
            {
                fg_side_dgB(tri)[side][0] = (Az[(side+1)%3] - Az[side])/len; 
                fg_side_B(tri)[side] = fg_side_dgB(tri)[side][0];
                sum += fg_side_dgB(tri)[side][0]*len;
            }
        }

        // now deal with the 3rd edge by using divergence-free condition.
        for(side = 0; side < 3; side++)
        {
            for(i = 0; i < 2; i++)
                v2[i] = fg_side_vector(tri)[side][i];
            nor[0] = v2[1];
            nor[1] = -v2[0];
            tmp  = fabs(nor[0]*dir[0][0] + nor[1]*dir[0][1]);
            tmp2 = fabs(nor[0]*dir[1][0] + nor[1]*dir[1][1]);

            if(tmp > 0.5 && fabs(nor[0]) > 0.5) // vertical edge of tri.
                continue;
            if(tmp2 > 0.5 && fabs(nor[1]) > 0.5) // horizontal edge of tri.
                continue;

            len = fg_length_side(tri)[side];         
            fg_side_dgB(tri)[side][0] = -sum/len;
            fg_side_B(tri)[side] = fg_side_dgB(tri)[side][0];
        }
        ****/
}


LOCAL void add_accurate_vortex_initializer(
        float           *coords,
        COMPONENT       comp,
        Locstate        state,
        INTERFACE       *intfc,
        INIT_DATA       *init,
        TRI             *tri)
{
        RECT_GRID       *gr = computational_grid(intfc);
        float           cent[3], rc, r, alpha = 0.204, ep = 0.3, tau;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u = 0.0, v = 0.0, T = 0.0, dT, nT;
        POINT           *p[3];
        float           *pcrds[3], dbcent[MAXD];
        float           Gam, S, gam = 1.4;
        double   rhs[4][MAX_N_COEF], mulrhs[4][MAX_N_COEF], dens[7];
        float         area = fg_area(tri), t = 0.0;
        double      **mass_inv, **Lmass_matrix;
        int     debug_flag = NO, i, j, k, indx;
        float a = 0.065130102902216, b = 0.869739794195568;
        float c = 0.312865496004875, d = 0.638444188569809;
        float e = 0.048690315425316, f = 0.260345966079038;
        float g = 0.479308067841923, third;
        float w1 =-0.149570044467670, w2 = 0.053347235608839,
          w3 = 0.175615257433204,  w4 = 0.077113760890257;
        float crds[16][2];
        double     *lcent = fg_centroid(tri);
        static float w[16] ={0.144315607677787,0.095091634267285,0.095091634267285,0.095091634267285,
                             0.103217370534718, 0.103217370534718,0.103217370534718,
                             0.032458497623198,0.032458497623198,0.032458497623198,
                             0.027230314174435,0.027230314174435,0.027230314174435,
                             0.027230314174435,0.027230314174435,0.027230314174435};
        static double **tmp_mass_inv = NULL;

        if(NULL == tmp_mass_inv)
            matrix(&tmp_mass_inv, MAX_N_COEF, MAX_N_COEF, sizeof(double));

        third = 1.0/3.0;
        Lmass_matrix = tri->Lmass_matrix;
        if(tri->mass_inv != NULL)
            mass_inv = tri->mass_inv;
        else
        {
            mass_inv = tmp_mass_inv;
            inverse_matrix(Lmass_matrix,MAX_N_COEF,mass_inv);
        }

        for(i = 0; i < 4; i++)
        {
            for(j = 0; j < MAX_N_COEF; j++)
                rhs[i][j] = 0.0;
        }

        for(i = 0; i < 3; i++)
        {
            p[i] = Point_of_tri(tri)[i];
            pcrds[i] = Coords(p[i]);
        }

        for(i = 0; i < 2; i++)
            dbcent[i] = fg_centroid(tri)[i];

        // tri_quadrature_13_pts(pcrds[0], pcrds[1], pcrds[2], crds);
        tri_quadrature_16_pts(pcrds[0], pcrds[1], pcrds[2], crds);

        // for 13 pts case
        /*****
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = w1*ff_accurate_vort(dbcent, crds[0],j)*vh_val(crds[0],lcent,k) +
                      w2*(ff_accurate_vort(dbcent, crds[1],j)*vh_val(crds[1],lcent,k) +
                          ff_accurate_vort(dbcent, crds[2],j)*vh_val(crds[2],lcent,k) +
                          ff_accurate_vort(dbcent, crds[3],j)*vh_val(crds[3],lcent,k)) +
                      w3*(ff_accurate_vort(dbcent, crds[4],j)*vh_val(crds[4],lcent,k) +
                          ff_accurate_vort(dbcent, crds[5],j)*vh_val(crds[5],lcent,k) +
                          ff_accurate_vort(dbcent, crds[6],j)*vh_val(crds[6],lcent,k)) +
                      w4*(ff_accurate_vort(dbcent, crds[7],j)*vh_val(crds[7],lcent,k) +
                          ff_accurate_vort(dbcent, crds[8],j)*vh_val(crds[8],lcent,k) +
                          ff_accurate_vort(dbcent, crds[9],j)*vh_val(crds[9],lcent,k) +
                          ff_accurate_vort(dbcent, crds[10],j)*vh_val(crds[10],lcent,k) +
                          ff_accurate_vort(dbcent, crds[11],j)*vh_val(crds[11],lcent,k) +
                          ff_accurate_vort(dbcent, crds[12],j)*vh_val(crds[12],lcent,k));
            }
        }
        ********/
        /// for 16 pts case
        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] = 0.0;
                for(i = 0; i < 16; i++)
                {
                    rhs[j][k] += w[i]*ff_accurate_vort(dbcent, crds[i],j)*vh_val(crds[i],lcent,k);
                }
            }
        }
        ///

        for(j = 0; j < N_EQN; j++)
        {
            for(k = 0; k < MAX_N_COEF; k++)
            {
                rhs[j][k] *= area;
            }
        }

        for(i = 0; i < N_EQN; i++)
            matrix_vec_mult(mass_inv, rhs[i], MAX_N_COEF, MAX_N_COEF, mulrhs[i]);

        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            dg_Dens(state)[indx] = mulrhs[0][indx];
            dg_Mom(state)[0][indx] = mulrhs[1][indx];
            dg_Mom(state)[1][indx] = mulrhs[2][indx];
            dg_Energy(state)[indx] = mulrhs[3][indx];
        }

        // Compute average soln
        Dens(state) = 0.0;
        Mom(state)[0] = 0.0;
        Mom(state)[1] = 0.0;
        Energy(state) = 0.0;
        for(indx = 0; indx < MAX_N_COEF; indx++)
        {
            Dens(state) += dg_Dens(state)[indx]*Lmass_matrix[0][indx];
            Mom(state)[0] += dg_Mom(state)[0][indx]*Lmass_matrix[0][indx];
            Mom(state)[1] += dg_Mom(state)[1][indx]*Lmass_matrix[0][indx];
            Energy(state) += dg_Energy(state)[indx]*Lmass_matrix[0][indx];
        }
        Dens(state) /= Lmass_matrix[0][0];
        Mom(state)[0] /= Lmass_matrix[0][0];
        Mom(state)[1] /= Lmass_matrix[0][0];
        Energy(state) /= Lmass_matrix[0][0];
}

/***
/// J.C.P. 2010, 229, 7893-7910.
/// domain [-5, 5] times [-5, 5].
EXPORT float ff_accurate_magnetic_vort(
        float *centroid,
        float *crds,
        int   n)
{
        float           cent[3], rc, r, ep = 1.0;
        float           sintheta = 0.0, costheta = 0.0, nor[2] = {1.0, 0.0}, v2[2];
        float           norv2[2], r_angle, u, v, w, dP;
        float           den = 1.0, P = 1.0, tmp, c;
        float           y_bar, x_bar, kappa, mu;
        float           Bx, By, Bz, gamma, Pgas; 

        u = 1.0; v = 1.0; w = 0.0;
        Bx = By = Bz = 0.0;
        gamma = 5.0/3.0;
        kappa = mu = 1.0;

        cent[0] = 0.0;
        cent[1] = 0.0;

        x_bar = -crds[1];
        y_bar = crds[0];

        r = (sqr(crds[0]) + sqr(crds[1]));

        // norv2[0] = v2[0]/r;
        // norv2[1] = v2[1]/r;
        u += kappa/(2.0*PI) * exp(0.5*(1-r)) * (x_bar);
        v += kappa/(2.0*PI) * exp(0.5*(1-r)) * (y_bar);

        dP = (sqr(kappa)*(1-r) - sqr(mu))/(8.0*sqr(PI)) * exp(1-r);  
        P += dP;

        Bx += mu/(2.0*PI) * exp(0.5*(1-r)) * (x_bar);
        By += mu/(2.0*PI) * exp(0.5*(1-r)) * (y_bar);

        Pgas = P - 0.5*( sqr(Bx) + sqr(By) );

        switch(n)
        {
        case 0:
            tmp = den;
        break;
        case 1:
            tmp = den*u;
        break;
        case 2:
            tmp = den*v;
        break;
        case 3:
             tmp = 0.0;
        break;
        case 4:  /// total energy
            tmp = Pgas/(gamma-1) + 0.5*den*(sqr(u) + sqr(v)) + 0.5*( sqr(Bx) + sqr(By) );
        break;
        case 5:
            tmp = Bx;
        break;
        case 6:
            tmp = By;
        break;
        case 7:
            tmp = 0.0;
        break;
        }

        return tmp;
}
***/

LOCAL   double  Az_field_loop_adv(double *crds)
{
	double  rc;
        double  R = 0.3, A0 = 0.001;
        
        rc = sqrt(sqr(crds[0]) + sqr(crds[1]));
       
        if(rc > R)
            return 0.0;
        else
        {
            return (A0*(R-rc));
        }
}
