/*
*				ghyp.h:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*/

#if !defined(_GHYP_H)
#define _GHYP_H

#include <gdecs/gdecs.h>

		/* eigenvalue and left/right eigenvectors */

typedef struct {
	int	negn;
	float	**lambda; /* eigenvalues  */
	float	**sgnl;   /* 1 for left moving characteric 0 otherwise */
	float	**sgnr;   /* 1 for right moving characteric 0 otherwise */
	float	**sgnm;   /* algegbraic sign of average characteristic speed */
	float	***l;     /* left eigenvectors */
	float	***r;     /* right engenvectors */

	struct {
	    bool lambda;
	    bool sgnl;   /* 1 for left moving characteric 0 otherwise */
	    bool sgnr;   /* 1 for right moving characteric 0 otherwise */
	    bool sgnm;   /* algegbraic sign of average characteristic speed */
	    bool l;     /* left eigenvectors */
	    bool r;     /* right engenvectors */
	    bool vegn;
	} alloc;
} Vec_Eigen;

typedef struct {
	float	**g;		/* artificial viscosity coefficients */
	float	*cs_ave;	/* Average sound speed squared */
	float	*c_ave;		/* Average sound speed */
	float	*vn_ave;	/* Average normal velocity */
	float	**b;		/* Wave speed differences times
				 * viscosity parameter */
	float	*visc;		/* Vector of viscosity parameters */
	float	**uconM;	/* Conservative form of mid state */
	float	*mdlambda;	/* max(lambda[i] - lambda[i+1]) */
	bool	use_lapidus;	/* use Lapidus artificial viscosity */
	bool	use_linear;	/* use linear artificial viscosity */
	bool	use_upwind;	/* use upwind weighted artificial viscosity */

	struct {
	    bool g;
	    bool cs_ave;
	    bool c_ave;
	    bool vn_ave;
	    bool b;
	    bool visc;
	    bool uconM;
	    bool mdlambda;
	    bool avisc;
	} alloc;
} Vec_Avisc;

typedef struct {
	float *chi;
	struct {
	    bool chi;
	    bool msf;
	} alloc;
} Vec_MSF;

/* CG_PARAMS contains different 'fudge' factors and parameters */

struct _CG_PARAMS {
	int pv_iterations;    /* Number of times to iterate in
	                       * Pressure-Velocity plane */
	float min_p_jump;
	float min_v_jump;    /* velocity convergence criteria */
	float sw_tol;
	float min_mass_flux;
	float sqr_min_mass_flux;
};
typedef struct _CG_PARAMS CG_PARAMS;

struct _MUSCL_FLUX {
	float	**F;	/* Volume flux vector F[k][j] is the volume flux of */
	                /* the kth state variable (0 = density, 1 = total   */
			/* energy density, 2 = sweep direction momemtum,    */
			/* etc,  see the definition of ucon above) across   */
			/* the cell boundary between cells j-1 and cell j   */
	float   **H;    /* Line flux vectors, see the documentation above   */
};
typedef struct _MUSCL_FLUX MUSCL_FLUX;


/* User defined options for MUSCL method */
struct _Muscl_Opts {
	int	kmax;	/* Number of independent state variables */
	int     worksp_len;
	int     nfloats;
	bool    monotone_reconstruction;
	bool    link_reconstructions;

	/* Load data for MUSCL solvers */
	struct _Vec_Muscl* (*_load_state_data)(struct _Muscl_Opts*,int,int*,
                                               Front*,Wave*,Stencil*,
                                               Tan_stencil*,Vec_Gas*,Vec_Src*,
					       int,int,int,float,float);

	/* compute eigenvectors for df/du */
	void (*_compute_eigens)(int,int,struct _Vec_Muscl*);

	/* compute artificial viscosity */
	void (*_compute_art_visc_coefs)(int,int,struct _Vec_Muscl*);

	/* State reconstruction function */
	void (*_reconstructor)(int,int,struct _Vec_Muscl*);

	/* Half-step propagation */
	void (*_half_step)(int,int,float,float,struct _Vec_Muscl*);
	void (*_strong_wave_half_step)(int,int,float,float,struct _Vec_Muscl*);

	/* Riemann solver */
	void (*_rsolver)(int,int,float**,Vec_Gas*,float**,Vec_Gas*,
                         float**,Vec_Gas*,MUSCL_FLUX*,struct _Vec_Muscl*);

	void (*_rmidstate)(int,int,Vec_Gas*,Vec_Gas*,
	                   float*,float*,struct _Vec_Muscl*);

	/* Method of characteristics */
	void (*_characteristic_solve)(int,int,struct _Vec_Muscl*);

	/* Compute flux vectors */
	void (*_flux_vectors)(int,int,float**,float*,MUSCL_FLUX*,
                              struct _Vec_Muscl*);

	/* addition of artificial viscosity before half step */
	void (*_add_art_visc1)(int,int,float**,struct _Vec_Muscl*);

	/* addition of artificial viscosity after half step */
	void (*_add_art_visc2)(int,int,struct _Vec_Muscl*);

	/* Transforms src terms from noncons. to cons. form */
	void (*_cons_src)(int,int,int,int*,Tan_stencil*,struct _Vec_Muscl*);

	void (*_print_internal_energy)(const char*,float**,
				       struct _Vec_Muscl*,int,int);

	/* Stencil operators */

	void (*_npt_solver)(float,float,Locstate,const float*,
	                    int,int*,int*,Stencil*);
	void (*_one_side_npt_tang_solver)(float,float,Tan_stencil*,Locstate,
					  Front*);
	int               _npts_tan_sten;
	void (*_npt_tang_solver)(float,float,Tan_stencil*,
	                         Locstate,Locstate,Front*);

	/* Storage allocators */

	void (*_alloc_phys_vecs)(Wave*,int);
	void (*_free_phys_vecs)(Wave*);

	/* State Loaders */

	TIME_DYNAMICS (*_load_state_vectors)(int,int*,Vec_Gas*,int,int,
				             Wave*,Wave*,int*,int);
	void (*_assign_wave_state_vectors)(int,int*,Wave*,Wave*,
		                           Vec_Gas*,int,int,int*,int);

	CG_PARAMS	_Cg_params;

	struct _MUSCL_PromptType_Reconstructor        *Sintrp;
	struct _MUSCL_PromptType_Rsolver              *Rsolver;
	struct _MUSCL_PromptType_characteristic_solve *Moc;
};
typedef struct _Muscl_Opts Muscl_Opts;


/*
*    The Vec_Muscl structure assumes a system of conservation laws of
*    the form
*
*	U_t + d(A*F)/dV(x) + dH/dx = S
*
*    For gas dynamics the vector U is the vector of conservative
*    quantities U = (rho, E, m) where rho is the mass density,
*    E is the total energy (kinetic + internal) density,  and m is the
*    momentum density vector.
*
*    A(x) is the cross sectional area of the section at x.  For example
*    in rectangular geometry A(x) = 1,  while A(x) = x for cylindrical
*    geometry and A(x) = x^2 for spherical geometry.  V is the volume
*    dV = A dx.   S is the body source term.
*    
*    Depending on the solver the definitions of F, H, and S may vary.
*    Note that all of the conservation laws below are equivalent to
*    the one above.
*
*	U_t + d(F+H)/dx = S - F*(dA/dx)/A
*
*	U_t + d(A*(F+H))/dV = S + H*dA/dV = S + H*(dA/dx)/A
*
*    For the oned_MUSCL solver we use
*
*	F = (m, m*m/rho + P, m*E + m*P/rho), H = 0,
*       S = ( -m*(dA/dx)/A,
*             rho*g - (m*m/rho + P)*(dA/dx)/A,
*             <m,g> - (m*E + m*P/rho)*(dA/dx)/A ).
*    
*    For the oned_PLM solver we use
*
*	F = ((m, m*m/rho, m*E + m*P/rho), H = (0,P,0), S = (0,rho*g,<m,g>)
*/

struct _Vec_Muscl_Index {	/* Indices of states stored in u             */
	int  density;
	int  energy;
	int  internal_energy_density;
	int  v[3];
	int  pressure;
	int  sound_speed;
	int  GAM;
	int  FD;
	int  grav;
};
typedef struct _Vec_Muscl_Index Vec_Muscl_Index;

enum _SWEEP_TYPE {
	UNSET_SWEEP_TYPE = 0,
	REGULAR_INTERIOR_SWEEP,
	IRREGULAR_INTERIOR_SWEEP,
	TANGENTIAL_SWEEP
};
typedef enum _SWEEP_TYPE SWEEP_TYPE;

struct _Vec_Muscl {
	Vec_Gas	*vst;		/* Initial data in vec gas form */

#if defined(CONSERVATIVE_ALG) && defined(TWOD)
        Vec_Gas *Tvst[2];      /* Initial data in the line above and below vst */
                               /* For the unsplit scheme  */
#endif /* if defined(CONSERVATIVE_ALG) && defined(TWOD) */

	float   **CellEdge;	/* Coordinates of cell edges    */
	float   **coords;       /* Coordinates of cell centers  */

	float	**u;		/* Initial state data, unconservative form */
	             /* u[index.density] = rho = density                    */
		     /* u[index.energy] = e = specific internal energy      */
		     /* u[index.v[0]] = sweep direction velocity component  */
		     /* u[index.v[i]], 0 < i < dim, velocity components     */
		     /*                ...,    orthogonal                   */
		     /* For PLM the additional values are set               */
		     /* u[index.pressure] = pressure                        */
		     /* u[index.sound_speed] = sound speed                  */
		     /* u[index.GAM] = Gruneisen exponent                   */
		     /* u[index.FD] = Fundamental derivative                */
		     /* u[index.grav] = sweep direction gravity component   */

	int             nvar_u; /* Number of variables stored in the u array */

	Vec_Muscl_Index index;

#if defined(CONSERVATIVE_ALG) && defined(TWOD)
        float   **pucon;        /* The storage is allocated, to save the 
                                 * previous godunov step state.  
                                 */
#endif /* if defined(CONSERVATIVE_ALG) && defined(TWOD) */
	float	**ucon;		/* Initial state data, conservative form   */
	                        /* ucon[0] = rho = density                 */
				/* ucon[1] = total energy density =        */
				/*           rho*(0.5*v*v + e)             */
				/* ucon[2] = sweep direction momentum      */
				/*           density                       */
				/* ucon[3],    momentum density components */
				/* ...,        orthogonal                  */
				/* ucon[1+dim] to sweep direction          */

				/* Note: the arrays stored in src and sourc */
				/* can be in either conservative or         */
				/* non-conservative form depending on the   */
				/* the section of the algorithm             */
	Vec_Src	*src;		/* Source terms */
	float	**source;	/* Array form of src */

	Vec_Eigen	Vegn;	/* Eigenvector matricies for initial data */

		/* State reconstruction data */
	float	**central;	/* Central differences */
	float	**forward;	/* Forward differences */
	float	**backward;	/* Backward differences */
	float   **dulim;        /* Limited forward and backward differences */
	float   **duf;		/* Van Leer differences */
	float   **sgn;          /* Signs of central differences */
	float	**du;	  	/* limited state slope */

	float	**q;	  	/* state in eigen coordinate */
	float	**dq;	        /* Slopes from linear reconstruction,   */
	                        /* in eigenvalue coordinates            */

		/* Half step data structure */
	Vec_Gas	VLst;	 /* Vec Gas form of left half step state      */
	                 /* or u+c family characteristic trace back   */
	float	**uL;	 /* Half step evolved left state              */
	float	*pL;	 /* Half step evolved left state pressure     */
	float   *gL;     /* Gravity value at left position            */

	Vec_Gas	VMst;	 /* Vec Gas form of mid state from Riemann solution */
	                 /* or contact family characteristic trace back     */
	float	**uM;	 /* Mid state from Riemann solution                 */
	float	*pM;	 /* pressure at mid state                           */
	float   *gM;     /* Gravity value at mid position                   */

	Vec_Gas	VRst;	 /* Vec Gas form of right half step state           */
	                 /* or u-c family characteristic trace back         */
	float	**uR;	 /* Half step evolved right state                   */
	float	*pR;	 /* Half step evolved right state pressure          */
	float   *gR;     /* Gravity value at left position            */

	float	**right_wvs;  /* Contributions from right moving waves */
	float	**left_wvs;   /* Contributions from left moving waves */
	float	**source_sum; /* Source term contributions from 1/2 step */

	MUSCL_FLUX Flux;

	float   **uM0;  /* Cell edge state at start of time step */
	Vec_Gas	VM0st;	/* Vec Gas form of right half step state */
	float	*pM0;	/* pressure at mid state */
	float   *gM0;	/* gravity at mid state */

	float   **uM1;       /* Cell edge state at end of time step   */
	Vec_Gas	VM1st;	/* Vec Gas form of right half step state */
	float	*pM1;	/* pressure at mid state */
	float   *gM1;	/* gravity at mid state */

	MUSCL_FLUX Flux0; /* Fluxes at begining of time step       */
	MUSCL_FLUX Flux1; /* Fluxes at begining of time step       */

	float	**awv;	/* Eigen coordinates of jump at cell boundaries */

	/* Grid description */
	float    *A;      /* A[j] = area of cell face between cells j-1     */
	                  /*        and cell j                              */
	float    *dV;     /* Volume of cell j                               */
	float    alpha;   /* Geometry factor */

	Muscl_Opts Opts;

	Vec_Avisc	*avisc; /* artificial viscosity paramters */
	Vec_MSF		*msf; /* Muscl slope flattening parameters */
	bool            monotone_reconstruction;
	bool            link_reconstructions;

	int		offset;	/* Starting index of Vec_Gas arrarys */
	int		vsize;	/* Size of arrays in Vec_Gas structures */
	int		max_vsize;	/* Allocated length of arrays */
	bool		is_src;	/* YES if source terms are present */
	int		dim;	/* Spatial dimension of flow */
	size_t		sizest;	/* Size of Locstate */
	int		sten_rad;/* Radius of finite difference stencil */
	int		idir;	/* Sweep direction */
	const float* const *Q;	/* rotation vector */
	float		dt;	/* time step */
	float		dn;	/* spatial step */
	float		*grav;	/* array of cell centered projected gravities */

#if defined(CONSERVATIVE_ALG) && defined(TWOD)
	float		**gravity; /* array of cell centered gravities. */
                                        /* For unsplit multiD. solver */
#endif /* if defined(CONSERVATIVE_ALG) && defined(TWOD) */

	Front		*front;
	Wave		*wave;

	Stencil         *sten;
	Tan_stencil     *tsten;

	float		***worksp;

        float   *J;     /* Jacobi */
        float   *J_half;

	struct {
	    bool CellEdge;
	    bool u;
	    bool ucon;
#if defined(CONSERVATIVE_ALG) && defined(TWOD)
            bool pucon;        
#endif /* if defined(CONSERVATIVE_ALG) && defined(TWOD) */
	    bool source;
	    bool central;
	    bool forward;
	    bool backward;
	    bool dulim;
	    bool duf;
	    bool sgn;
	    bool du;
	    bool q;
	    bool dq;
	    bool uL;
	    bool pL;
	    bool uM;
	    bool pM;
	    bool uR;
	    bool pR;
	    bool right_wvs;
	    bool left_wvs;
	    bool source_sum;
	    struct {
	        bool F;
	        bool H;
	    } Flux;
	    bool uM0;
	    bool uM1;
	    struct {
	        bool F;
	        bool H;
	    } Flux0;
	    struct {
	        bool F;
	        bool H;
	    } Flux1;
	    bool awv;
	    bool grav;
#if defined(CONSERVATIVE_ALG) && defined(TWOD)
	    bool gravity;
#endif /* if defined(CONSERVATIVE_ALG) && defined(TWOD) */
	    bool worksp;
	    bool A;
	    bool dV;
	    bool vmuscl;

            bool J;
            bool J_half;
	} alloc;

	SWEEP_TYPE _MusclSweepType;
};
typedef struct _Vec_Muscl Vec_Muscl;

#define MusclSweepType(vmuscl) vmuscl->_MusclSweepType

struct _MUSCL_PromptType_Reconstructor {
	const char    *prompt, *select;
	void (*reconstructor)(int,int,Vec_Muscl*);
	void (*half_step)(int,int,float,float,struct _Vec_Muscl*);
	void (*strong_wave_half_step)(int,int,float,float,struct _Vec_Muscl*);
};
typedef struct _MUSCL_PromptType_Reconstructor MUSCL_PromptType_Reconstructor;

struct _MUSCL_PromptType_Rsolver {
	const char *prompt, *select;
	void (*rsolver)(int,int,float**,Vec_Gas*,float**,Vec_Gas*,
                        float**,Vec_Gas*,MUSCL_FLUX*,Vec_Muscl*);
	void (*rmidstate)(int,int,Vec_Gas*,Vec_Gas*,
	                  float*,float*,Vec_Muscl*);
};
typedef struct _MUSCL_PromptType_Rsolver MUSCL_PromptType_Rsolver;

struct _MUSCL_PromptType_characteristic_solve {
	const char    *prompt, *select;
	void (*characteristic_solve)(int,int,struct _Vec_Muscl*);
};
typedef struct _MUSCL_PromptType_characteristic_solve MUSCL_PromptType_characteristic_solve;

/* Access macros for Muscl_Opts functions */
	
#define compute_eigens(start,end,vmuscl)				\
    if ((vmuscl)->Opts._compute_eigens)					\
        (*(vmuscl)->Opts._compute_eigens)(start,end,vmuscl)

#define compute_art_visc_coefs(start,end,vmuscl)			\
    if ((vmuscl)->Opts._compute_art_visc_coefs)				\
        (*(vmuscl)->Opts._compute_art_visc_coefs)(start,end,vmuscl)

#define reconstructor(start,end,vmuscl)					\
    (*(vmuscl)->Opts._reconstructor)(start,end,vmuscl)

#define half_step(start,end,dt,dn,vmuscl)				\
    (*(vmuscl)->Opts._half_step)(start,end,dt,dn,vmuscl)

#define rsolver(start,end,uL,vlst,uR,vrst,uM,vmst,flux,vmuscl)		\
    (*(vmuscl)->Opts._rsolver)(start,end,uL,vlst,uR,vrst,uM,vmst,flux,vmuscl)

#define rmidstate(start,end,vlst,vrst,pm,vm,vmuscl)			\
    (*(vmuscl)->Opts._rmidstate)(start,end,vlst,vrst,pm,vm,vmuscl)

#define characteristic_solve(start,end,vmuscl)				\
    (*(vmuscl)->Opts._characteristic_solve)(start,end,vmuscl)

#define flux_vectors(start,end,uM,p,Flux,vmuscl)			\
    (*(vmuscl)->Opts._flux_vectors)(start,end,uM,p,Flux,vmuscl)

#define add_art_visc1(start,end,uM,vmuscl)				\
    if ((vmuscl)->Opts._add_art_visc1)					\
        (*(vmuscl)->Opts._add_art_visc1)(start,end,uM,vmuscl)

#define add_art_visc2(start,end,vmuscl)					\
    if ((vmuscl)->Opts._add_art_visc2)					\
        (*(vmuscl)->Opts._add_art_visc2)(start,end,vmuscl)

#define cons_src(start,end,swp_num,iperm,tsten,vmuscl)		\
    (*(vmuscl)->Opts._cons_src)(start,end,swp_num,iperm,tsten,vmuscl)

#define print_internal_energy(mesg,ucon,vmuscl,start,end)		\
    (*(vmuscl)->Opts._print_internal_energy)(mesg,ucon,vmuscl,start,end)

#define	Cg_params(mopts)	((mopts)->_Cg_params)

///////////////// gDG.c
// #define NEW_LIMIT

enum {RK_STEP = 2}; // Should be 3rd or 4th order TVD RK 

struct _Mid_soln{
        // int        id;
        Locstate   *st; // st[RK_STEP]. st[0] = reconstructed states
        byte       *worksp_st_store;
        double     edge_Bn[3][RK_STEP];  /// [tri side][]
        double     edge_dgBn[3][N_COEF_EDGE][RK_STEP];  /// [tri side][]
};
typedef struct _Mid_soln Mid_soln;

struct _Dual_cell_Mid_soln{
        double     edge_dgBn[RK_STEP][MAX_N_POLY_SIDE][N_COEF_EDGE];  /// [step][side][moment]
        double     Nodal_B[RK_STEP][2][MAX_N_POLY_SIDE];
        double     Bubble_B[RK_STEP][2][3]; 
        double     sub_reg_dgB[RK_STEP][MAX_N_POLY_SIDE][2][MAX_N_COEF]; //[indx of each sub-region][x- or y- component][moments]
};
typedef struct _Dual_cell_Mid_soln Dual_cell_Mid_soln;


struct _Dual_cell_buffer_soln{
        int        id;
        int        n_sides;
        double     cent[MAXD];
        double     polyg_crds[MAX_N_POLY_SIDE][3];
        double     edge_dgBn[RK_STEP][MAX_N_POLY_SIDE][N_COEF_EDGE];  /// [step][side][moment]
        double     sub_reg_dgB[RK_STEP][MAX_N_POLY_SIDE][2][MAX_N_COEF]; //01-28-2015
};
typedef struct _Dual_cell_buffer_soln Dual_cell_Buf_soln;

/** OLD before 02122014
struct _buffer_soln{
        // int        id;
        float      cent[MAXD];
        Locstate   st; // st[RK_STEP]. st[0] = reconstructed states
};
typedef struct _buffer_soln Buf_soln;
**/

struct _buffer_soln{
        int        id;
        float      cent[MAXD];
        double     tri_crds[3][3];
        double     area;
        double     edge_Bn[3][RK_STEP];  /// [tri side][], edge-length averaged B normal at RK stage
        double     edge_dgBn[3][N_COEF_EDGE][RK_STEP];  /// [tri side][],  dg B normal at RK stage
        Locstate   st; // st[RK_STEP]. st[0] = reconstructed states
};
typedef struct _buffer_soln Buf_soln;

// Use to save reconstructed polynomial on CV //[cv_indx][COEF]
struct _Limiting_store{
        float           cv_dg_rho[N_PART][MAX_N_COEF];
        float           cv_dg_e[N_PART][MAX_N_COEF];
        //float           cv_dg_m[2][N_PART][MAX_N_COEF];
        float           cv_dg_m[3][N_PART][MAX_N_COEF];//huijing
        float           cv_dg_B[3][N_PART][MAX_N_COEF];//huijing
};      
typedef struct _Limiting_store Limiting_store;

struct _Tri_HR_sten{
        int      HR_sten_set_3rd, HR_sten_set_2nd, HR_sten_set_1st, HR_sten_set_1st_pt;
        double   ***Axx, ***Axy, ***Ayy, *c_num_xx, *c_num_xy, *c_num_yy;
        double   ***Ax, ***Ay, *c_num_x, *c_num_y;
        double   ***A, *c_num;
        double   ***A_pt, *c_num_pt;
        ////// MHD part  /////
        int      Bsten_set, WENO_cent_set, WENO_side_set[3], WENO_rev_set[3]; 
        double   **Bsten[3], **MB_A, **MB_B, **Bsten_rev[3], **Bsten_rev_MB_B[3]; // stencil for reconstructing zone center B field from zone edge.
        double   **MB_one_A[3], **MB_one_B[3]; // stencil for reconstructing zone center B field from zone edge.
        ///// WENO stencil
        double   **WENO_Acent, **WENO_Bcent, **WENO_Aside[3], **WENO_Arev[3]; // WENO_Aside[3] for three edges of tri
                                                                              // WENO_Arev[3]  for three reverse side of tri
        int      B_2nd_recons_sten_set;
        double   **B_2nd_recons_sten;                                         // stencil for the 2nd reconstructing zone 
                                                                              // center B field from zone edge. 
                                                                              // variation on edge is available.
        int      P2_B_2nd_recons_sten_set;
        double   **P2_B_2nd_recons_sten_A, **P2_B_2nd_recons_sten_B;          // constrained least square is used for P2 case
};
typedef struct _Tri_HR_sten Tri_HR_sten;

struct _Tri_mass_1st_rows{
    double **mass_1st_rows[30];
    double  **mass_1st_rows_ppcell[30]; /// important, this 30 is wired with other code, for the partial partial cell on 1st degree HR
    double  **constrainA;
    double  **inverse_A;
};
typedef struct _Tri_mass_1st_rows Tri_mass_1st_rows;


struct _Dual_cell_sten{
        ////// MHD part  /////
        int      Bsten_set;
        double   **MB_A, **MB_B; // stencil for reconstructing zone center B field from zone edge.
};
typedef struct _Dual_cell_sten Dual_cell_sten;

struct _DUAL_CELL_PAIR
{
    POLYGON  *dual_cells[2];
    int      sh_edges[2]; /// index of shared edges for these two cells respectively           
    TRI      *tri[2];        /// On the common edge of tri[0] and tri[1], pressure is negative
                             /// tri[0] and tri[1] are covered by dual_cells[0]
};
typedef struct _DUAL_CELL_PAIR DUAL_CELL_PAIR;


#include <ghyp/ghypprotos.h>

#endif /* !defined(_GHYP_H) */
