/*
*				curv_gmuscl.c
*
*	MUSCL solver for curvilinear coordinates
*/

#include <ghyp/ghyp.h>
#include <gdecs/vecdecs.h>

	/* LOCAL Function Declarations */
LOCAL	Vec_Muscl *curv_load_muscl_state_data(Muscl_Opts*,int,int*,Front*,
						Wave*,Stencil*,Tan_stencil*,
						Vec_Gas*,Vec_Src*,int,int,
						int,float,float);
LOCAL	Vec_Muscl *curv_muscl_alloc_wk_space(Muscl_Opts*,int,int,Vec_Muscl*);
LOCAL	bool	curv_muscl_load_source_vectors(int,int*,Tan_stencil*,float*,
                                     	Vec_Gas*,Vec_Src*,int,int,RECT_GRID*);
LOCAL	float	*curv_set_vmuscl_vec_gas_storage(float**,Vec_Gas*,Vec_Muscl*);
LOCAL	void	left_multiply_state_by_matrix(float**,float***,float**,
						int,int,int);
LOCAL	void	g_curv_cons_src(int,int,int,int*,Tan_stencil*,struct _Vec_Muscl*);
   
LOCAL	float	f_Jacobi_x_i(float,float,float); /* det|dq/dx| */	
LOCAL	float	f_Jacobi_x_c(float,float,float); /* det|dq/dx| */	
LOCAL	float	f_Jacobi_x_s(float,float,float); /* det|dq/dx| */	

EXPORT void curv_oned_MUSCL(
	int		swp_num,
	int		*iperm,
	int		*icoords,
	Wave		*wave,
	Wave            *newwave,
	Front		*fr,
	Front           *newfr,
	Stencil         *sten,
	Tan_stencil	*tsten,
	int		offset,
	int		vsize,
	Vec_Gas		*vst,
	Vec_Src		*src,
	float		dt,
	float		dn,
	int		dim)
{
	Vec_Muscl	*vmuscl;
	float		dtdni = dt/dn;
	float		**ucon, **F, **source;
	float		*uconk, *Fk, *sourcek;
	int		start, end;
	int		j, k, kmax;
	int		sten_rad;
	float		*J,*J_half;
	Muscl_Opts	*MOpts;
	float		temp_J,temp_J2;
	float		*pre,*den,*mom_n,*velo_n,*eng,*Fluxk,*cur_srck,*pM;
	float 		**Flux, **cur_src;
	float 		vn, pres_new, pr, preM, preJM, radl, radr;
	Locstate	st,*state;
	float		sum_dens;
	RECT_GRID	*r_grid;

	MOpts = muscl_options();

	vmuscl = curv_load_muscl_state_data(MOpts,swp_num,iperm,fr,wave,sten,tsten,
	        		 vst,src,offset,vsize,dim,dt,dn);

	sten_rad = vmuscl->sten_rad;
	
	/* compute the eigenvalues, eigenvectors and max wavespeed*/
	compute_eigens(0,vsize,vmuscl);

	start = sten_rad-1;
	end = vsize-start;
	/* compute the coefficients for Lapidus and slope limiting viscosity */
	compute_art_visc_coefs(start,end,vmuscl);


	/* compute the linear reconstruction of the state variables */
	reconstructor(start,end,vmuscl);


	/* Evolve for half time step.  The returned data uL and uR
	 * are these evolved states on the two sides of each mesh edge. 
	 * They will be put in the Riemann solver. */
	start = sten_rad;
	end = vsize - sten_rad + 1;
	
	half_step(start,end,dt,dn,vmuscl);

	/* Solve the Riemann problem at each mesh edge with the states 
	 * uL and uR on the two sides.  The returned data uM is the solution
	 * state at the middle. */

	rsolver(start,end,vmuscl->uL,&vmuscl->VLst,vmuscl->uR,&vmuscl->VRst,
	                  vmuscl->uM,&vmuscl->VMst,&vmuscl->Flux,vmuscl);

	if (vmuscl->avisc)
	    add_art_visc2(start,end,vmuscl);

	/* Set conservative source terms */
	sten_rad = vmuscl->sten_rad;
	start = sten_rad;
	end = vsize - sten_rad;
	g_curv_cons_src(start,end,swp_num,iperm,tsten,vmuscl);

	/* compute the cell average of the approximate solution for the
	   next time step */
	kmax	= dim+2;
	ucon	= vmuscl->ucon;
	F	= vmuscl->Flux.F;
	J	= vmuscl->J + offset;
	J_half	= vmuscl->J_half +offset;
	source	= vmuscl->source;

	matrix(&cur_src,kmax,vsize,FLOAT);
	pre   = vst->p + offset;
	pM    = vmuscl->pM;
	den   = ucon[0];
	mom_n = ucon[2];

	
        for (j = start; j < end; ++j)
        {
            cur_src[0][j] = 0.0;
            cur_src[1][j] = 0.0;
	    for (k = 0; k < dim; ++k)
	        cur_src[k+2][j] = 0.0;
	}     
	if (tsten == NULL)
	{
	    /* Rect sweep */
	    
	    r_grid = wave->rect_grid; 
	    for (j = start; j < end; ++j)
	    {
	        if (r_grid->Remap.remap == CYLINDRICAL_REMAP)
	        {		 
		    if (iperm[swp_num]==0) 
			cur_src[2][j] = 0.5*(pM[j]+pM[j+1]);
                }
		if (r_grid->Remap.remap == SPHERICAL_REMAP) 
		{
		    radl = sqrt(J[j]);
		    cur_src[2][j] = 2.0*radl*pre[j];
		}
	    }
	}
	else
	{
	    /* Tangential sweep */
	    const float* const *Q = vmuscl->Q;
	    
	    r_grid = fr->rect_grid; 
	    for (j = start; j < end; ++j)
	    {
	        if (r_grid->Remap.remap == CYLINDRICAL_REMAP)
	            cur_src[2][j] = 0.5*(pM[j]+pM[j+1])*Q[0][0];
		if (r_grid->Remap.remap == SPHERICAL_REMAP) 
		{
		    radl = sqrt(J[j]);
		    cur_src[2][j] = 2.0*radl*pre[j]*Q[0][0];
		}
	    }
	}


	if (debugging("curv_MUSCL_2d") && tsten != NULL)
	{
	    printf("Before advanced: \n");
	    for (j = start; j < end; ++j)
	    {
	        printf("Dens[%d]=%f, Engy[%d]=%f, Mom0[%d]=%f, Mom1[%d]=%f \n",
				j,ucon[0][j],j,ucon[1][j],j,ucon[2][j],j,ucon[3][j]);
	    }
	}

	if (source != NULL)
	{
	    for (k = 0; k < kmax; ++k)
	    {
	    	uconk = ucon[k];
		cur_srck = cur_src[k];
	    	Fk = F[k];
	    	sourcek = source[k];
	    	for (j = start; j < end; ++j)
	        {
	            uconk[j] += dtdni*(J_half[j]*Fk[j] - J_half[j+1]*Fk[j+1])/J[j] 
				+ dt*cur_srck[j]/J[j] + dt*sourcek[j];
		}
	    }
	}
	else
	{
	    for (k = 0; k < kmax; ++k)
	    {
	    	uconk = ucon[k];
	    	Fk = F[k];
	    	for (j = start; j < end; ++j)
		{
		    uconk[j] += dtdni*(J_half[j]*Fk[j] - J_half[j+1]*Fk[j+1])/J[j]; 
		}
	    }
	}

	if (debugging("curv_MUSCL_2d") && tsten != NULL)
	{
	    printf("After advanced: \n");
	    for (j = start; j < end; ++j)
	    {
	        printf("Dens[%d]=%f, Engy[%d]=%f, Mom0[%d]=%f, Mom1[%d]=%f \n",
				j,ucon[0][j],j,ucon[1][j],j,ucon[2][j],j,ucon[3][j]);
	    }
	}
	
	free(cur_src);

} /*end curv_oned_MUSCL*/


EXPORT void g_compute_Jacobi(
	int		swp_num,
	int		*iperm,
	int		start,
	int		end,
        Stencil         *sten,
	Tan_stencil	*tsten,
	Vec_Muscl	*vmuscl,
	Wave		*wv,
	Front		*fr,
	Vec_Gas		*vst,
	int		offset)
{
	int		j;
	float		*J = vmuscl->J + offset;
	float		*J_half = vmuscl->J_half + offset;
	float		**coords = vst->coords + offset;
	float		ncoord;
	RECT_GRID	*r_grid;
	float		radi, radi_r, radi_l, dr;
	float		(*Jacobi_func)(float,float,float);

	Jacobi_func = f_Jacobi_x_i;
	
	if (tsten == NULL)
	{
	    /* Rect sweep */
	    r_grid = wv->rect_grid;
	
	    if (r_grid->Remap.remap == SPHERICAL_REMAP)
                Jacobi_func = f_Jacobi_x_s; 
	    if (r_grid->Remap.remap == CYLINDRICAL_REMAP) 
                Jacobi_func = f_Jacobi_x_c;
    	    
	    dr = r_grid->h[0];
            if(sten == NULL)
            {
                for (j = start; j < end; ++j)
                    J[j] = Jacobi_func(coords[j][0], 0.0, 0.0);
            }
            else
            {
                float  J0;
                int    endpt; 

                endpt = (sten->npts - sten->npts%2)/2; 
                J0 = Coords(sten->p[0])[0];
                if(iperm[swp_num] == 0)
                {
                    for(j = start; j < end; ++j)
                        J[j] = Jacobi_func(J0+(j-endpt)*dr, 0.0, 0.0); 
                }
                else
                {
                    for(j = start; j < end; ++j)
                        J[j] = Jacobi_func(J0, 0.0, 0.0);
                }
            }

            if(iperm[swp_num] == 0)
            {
                for (j = start; j < end; ++j)
                {
                    J_half[j] = J[j] - dr/2.0;
                    if(fabs(J_half[j]) < 1.0e-6)
                        J_half[j] = (J[j] < 0.0) ? -0.005*dr : 0.005*dr;
                }
            }
            else
            {
                for (j = start; j < end; ++j)
                    J_half[j] = J[j];
            }

            if(debugging("g_compute_Jacobi"))
            {
                printf("g_compute_Jacobi: idir[%d]\n", iperm[swp_num]); 
                for (j = start; j < end; ++j)
                    printf("Jacobi[%d] = %g\n", j, J[j]); 
                for (j = start; j < end; ++j)
                    printf("J_half_Jacobi[%d] = %g\n", j, J_half[j]); 
            }
	}
	else
	{
	    /* Tangential sweep */
            POINT **pt;
		
	    pt = tsten->p;           
	    r_grid = fr->rect_grid;

	    if (r_grid->Remap.remap == SPHERICAL_REMAP) Jacobi_func = f_Jacobi_x_s; 
	    if (r_grid->Remap.remap == CYLINDRICAL_REMAP) Jacobi_func = f_Jacobi_x_c;
	    
	    radi_l = pos_radius(Coords(pt[start-2])[0],r_grid);
	    radi_r = pos_radius(Coords(pt[start+1-2])[0],r_grid);
	    dr = radi_r - radi_l;
	    if (fabs(radi_l) < 1.0e-6)
	    {
		radi_l = dr*dr*dr*dr; //SMALL_NUMBER
	    }
	    if (fabs(radi_r) < 1.0e-6)
	    {
		radi_r = dr*dr*dr*dr; //SMALL_NUMBER
	    }
	    radi_l = 2.0*radi_l - radi_r;
	    for (j = start; j < end; ++j)
	    {
		radi_r = pos_radius(Coords(pt[j-2])[0],r_grid);
	        if (fabs(radi_r) < 1.0e-6)
	        {
		    radi_r = dr*dr*dr*dr; //SMALL_NUMBER
	        }
	        J[j] = Jacobi_func(radi_r, 0.0, 0.0);	     
		radi = 0.5*(radi_l + radi_r);
		J_half[j] = Jacobi_func(radi, 0.0, 0.0);	     
		radi_l = radi_r;
	    }

	}

} /*end g_compute_Jacobi*/


LOCAL	Vec_Muscl *curv_load_muscl_state_data(
	Muscl_Opts	*mopts,
	int		swp_num,
	int		*iperm,
	Front		*fr,
	Wave		*wave,
	Stencil         *sten,
	Tan_stencil	*tsten,
	Vec_Gas		*vst,
	Vec_Src		*src,
	int		offset,
	int		vsize,
	int		dim,
	float		dt,
	float		dn)
{
	static Vec_Muscl *vmuscl = NULL;
	float		 **u, **ucon, **source;
	const float* const *Q = vst->Q;
	bool		is_src;
	int		i, j;
	float		*J, *J_half;
	
	if (vmuscl == NULL || vsize > vmuscl->max_vsize || dim > vmuscl->dim)
	{
	    vmuscl = g_muscl_free_wk_space(vmuscl);
	    vmuscl = curv_muscl_alloc_wk_space(mopts,vsize,dim,vmuscl);
	}
	vmuscl->sizest = fr->sizest;
	vmuscl->idir = (iperm != NULL) ? iperm[swp_num] : dim;
	vmuscl->Q = Q;
	vmuscl->dt = dt;
	vmuscl->dn = dn;
	vmuscl->front = fr;
	vmuscl->wave = wave;
	if (wave != NULL)
	    vmuscl->sten_rad = wave->npts_sten/2;
	else if (tsten != NULL)
	    vmuscl->sten_rad = tsten->npts/2;
	else
	{
	    vmuscl->sten_rad = 2;
	    screen("ERROR in g_load_muscl_state_data(), can't determine "
	           "stencil size\n");
	    clean_up(ERROR);
	}
	vmuscl->vst = vst;
	vmuscl->src = src;
	vmuscl->offset = offset;	vmuscl->vsize = vsize;

	g_load_VGas_state_vectors(offset,vsize,vst,dim);
	
	clear_Vec_Gas_set_flags(&vmuscl->VMst);
	clear_Vec_Gas_set_flags(&vmuscl->VLst);
	clear_Vec_Gas_set_flags(&vmuscl->VRst);
	vmuscl->VMst.Q = vmuscl->VLst.Q = vmuscl->VRst.Q = Q;

	g_compute_Jacobi(swp_num,iperm,0,vsize,sten,tsten,vmuscl,wave,fr,vst,offset);

	/* Set params jump points for vec gas states */
	copy_vec_state_params(&vmuscl->VMst,vst,offset,vsize);
	copy_vec_state_params(&vmuscl->VRst,vst,offset,vsize);
	copy_vec_state_params(&vmuscl->VLst,vst,offset-1,vsize);

	/*put the state variables and source terms in a form more
	  easily usable by the subroutines*/
	
	u = vmuscl->u;
	ucon = vmuscl->ucon;
	u[vmuscl->index.density] = ucon[0] = vst->rho + offset;
	u[vmuscl->index.energy] = vst->e + offset;
	ucon[1] = vst->en_den + offset;
	for (i = 0; i < dim; ++i)    
	{
	    u[vmuscl->index.v[i]] = vst->v[i] + offset;
	    ucon[i+2] = vst->m[i] + offset;
	}
	if ((src != NULL) && ((source = vmuscl->source) != NULL))
	{
	    source[0] = src->mass + offset;
	    source[1] = src->energy + offset;
	    for (i=0; i < dim; ++i)	 
	    	source[i+2] = src->mom[i] + offset;
	}

	if (vmuscl->grav != NULL)
	{
	    float time = fr->time + 0.5*dt;
	    float **coords = vst->coords + offset;
	    float *g = vmuscl->grav + offset;
	    int   j;

	    if (tsten == NULL)
	    {
	        for (j = 0; j < vsize; ++j)
	            g[j] = gravity(coords[j],time)[iperm[swp_num]];
	    }
	    else
	    {
	    	for (j = 0; j < vsize; ++j)
	    	    g[j] = scalar_product(Q[0],gravity(coords[j],time),dim);
	    }
	}

	is_src = curv_muscl_load_source_vectors(swp_num,iperm,tsten,vmuscl->grav,
			                     vst,src,offset,vsize,
					     fr->rect_grid);

	vmuscl->is_src = is_src;

#if defined(DEBUG_MUSCL)
	if (debugging("load_Vgas")) 
	{
	    (void) printf("All quantities before MUSCL computation:\n");

	    g_printout_vec_data("The state variables",
				u[vmuscl->index.density],
				u[vmuscl->index.energy],
				u+vmuscl->index.v[0],
				dim,0,vsize,"muncons");
	    g_printout_vec_data("The conserved variables",
				ucon[0],ucon[1],ucon+2,dim,0,vsize,"cons");
	    if (is_src)
	    {
	    	g_printout_vec_data("The source terms",
	    			    source[0],source[1],source+2,dim,1,
	    			    vsize-1,"src");
	    }
 
	    (void) printf("Miscelaneous quantities:\n");
	    (void) printf("%-4s %-14s %-14s %-14s\n","n","pressure",
	    	          "sound speed","Grun. coef");
	    for (i = offset; i < vsize; ++i) 
	        (void) printf("%-4d %-14g %-14g %-14g\n",
			      i,vst->p[i],vst->c[i],vst->GAM[i]);
	    (void) printf("\n");
	}
#endif /* defined(DEBUG_MUSCL) */
	return vmuscl;
}		/*end curv_load_muscl_state_data*/


LOCAL bool curv_muscl_load_source_vectors(
	int		   swp_num,
	int		   *iperm,
	Tan_stencil	   *tsten,
	float		   *grav,
	Vec_Gas		   *vst,
	Vec_Src		   *src,
	int		   offset,
	int		   vs,
	RECT_GRID	   *gr)
{
	int		   i, j;
	int		   dim = gr->dim;
	const float* const *Q = vst->Q;
	float		   *rho_src;
	float		   *e_src;
	float		   *v_src[SMAXD];
	float		   *v_src0, *v_srci;
	static	bool	   first = YES;
#if defined(ROTATIONAL_SYMMETRY)
	float		*pr;
	float		*v0, *rho;
	static	float	alpha;
#endif /* defined(ROTATIONAL_SYMMETRY) */

	if (src == NULL)
	    return NO;

	rho_src = src->mass + offset;
	e_src = src->energy + offset;

	for (i = 0; i < dim; ++i)
	    v_src[i] = src->mom[i] + offset;
	if (first == YES) 
	{
	    first = NO;
#if defined(ROTATIONAL_SYMMETRY)
	    alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */
	}

	for (j = 0; j < vs; ++j)
	{
	    rho_src[j] = 0.0;
	    e_src[j] = 0.0;
	}
	for (i = 0; i < dim; ++i)
	{
	    v_srci = v_src[i];
	    for (j = 0; j < vs; ++j)
		v_srci[j] = 0.0;
	}
	if (grav != NULL)
	{
	    float	*g = grav + offset;
	    v_src0 = v_src[0];
	    for (j = 0; j < vs; ++j)
	    	v_src0[j] += g[j];
	}

	if (tsten == NULL)
	{
	    /* Rect sweep */

#if defined(ROTATIONAL_SYMMETRY)
	    if ((alpha > 0.0) && (iperm[swp_num]==0))
	    {
		/* Include cylindrical source terms */
 
		float *radii = src->radii + offset;
		float rmin = src->rmin;
		float *m;

		m = vst->m[0] + offset;
		pr = vst->p + offset;
		rho = vst->rho + offset;
		v0 = vst->v[0] + offset;
		for (j = 0; j < vs; ++j)
		{
		    if (fabs(radii[j]) > fabs(rmin))
		    {
		        rho_src[j] -= alpha*m[j]/radii[j];
		        e_src[j] -= alpha*pr[j]*v0[j]/(radii[j]*rho[j]);
		    }
		}
	    }
#endif /* defined(ROTATIONAL_SYMMETRY) */
	}
	else
	{
	    /* Tangential sweep */

#if defined(ROTATIONAL_SYMMETRY)
	    /* For cylindrical coordinates the mass equation for the
	    *  tangential sweep becomes
	    *
	    *	drho/dt + d(rho*vtan)/dr = - rho*vtan/r * dr/dtan
	    *
	    *  where dr/dtan is the change in radius with respect to
	    *  changes in the tangential coordinate.  If we let
	    *  (r, z) = tT + nN, i.e. represent r and z by a combination
	    *  of vectors T & N tangent and normal to the curve, dr/dtan
	    *  becomes d(tT[0])/dt = T[0] ( = Q[0][0] )
	    */

	    if (alpha > 0.0)
	    {
	        float rmin, rad;
		float *mom0;
	        POINT **pt;

	        mom0 = vst->m[0];
	        pt = tsten->p;
	        pr = vst->p;
	        rho = vst->rho;
	        v0 = vst->v[0];

		rmin = src->rmin;
	        for (j = 0; j < vs; ++j)
	        {
	            rad = pos_radius(Coords(pt[j-2])[0],gr);
		    if (fabs(rad) > fabs(rmin))
		    {
	                rho_src[j] -= alpha*mom0[j]*Q[0][0]/rad;
	                e_src[j] -= alpha*Q[0][0]*v0[j]*pr[j]/(rho[j]*rad);
		    }
	        }
	    }
#endif /* defined(ROTATIONAL_SYMMETRY) */
	}

#if defined(DEBUG_MUSCL)
	if (debugging("vsrc"))
	{
	    g_printout_vec_data("The source vectors from "
	        	        "g_muscl_load_source_vectors:",
	        	        src->mass,src->energy,src->mom,dim,
	        	        offset,vs,"src");
	}
#endif /* defined(DEBUG_MUSCL) */
	return YES;
}		/*end g_muscl_load_source_vectors*/


LOCAL	Vec_Muscl *curv_muscl_alloc_wk_space(
	Muscl_Opts *mopts,
	int	   vsize,
	int	   dim,
	Vec_Muscl  *vmuscl)
{
	float		 ***worksp;
	AVISC		 Avisc;
	Vec_Eigen	 *vegn;
	int		 nfloats = mopts->nfloats;
	int		 negn = 3;
	int		 worksp_len = mopts->worksp_len;
	int              i, nvar_u;

	vmuscl = alloc_Vec_Muscl(vmuscl);
	vegn = &vmuscl->Vegn;

	vmuscl->Opts = *mopts;
	vmuscl->dim = dim;

	nvar_u = 0;
	vmuscl->index.density = nvar_u++;
	vmuscl->index.energy = nvar_u++;
	for (i = 0; i < dim; ++i)
	    vmuscl->index.v[i] = nvar_u++;
	vmuscl->nvar_u = nvar_u;

	zero_scalar(vegn,sizeof(Vec_Eigen));
	vegn->negn = negn;
	set_no_alloc(vegn,vegn);
	MATRIX(vegn,lambda,negn,vsize,FLOAT);
	MATRIX(vegn,sgnl,negn,vsize,FLOAT);
	MATRIX(vegn,sgnr,negn,vsize,FLOAT);
	MATRIX(vegn,sgnm,negn,vsize,FLOAT);
	TRI_ARRAY(vegn,l,negn,negn,vsize,FLOAT);
	TRI_ARRAY(vegn,r,negn,negn,vsize,FLOAT);

	if (is_gravity() == YES)
	    VECTOR(vmuscl,grav,vsize,FLOAT);

	/* Jacobi is alloced here */
	VECTOR(vmuscl,J,vsize,FLOAT);
	VECTOR(vmuscl,J_half,vsize,FLOAT);

	TRI_ARRAY(vmuscl,worksp,worksp_len,nfloats,vsize,FLOAT);
	worksp = vmuscl->worksp;

	VECTOR(vmuscl,u,nfloats,sizeof(float*));
	VECTOR(vmuscl,ucon,nfloats,sizeof(float*));
	if (source_terms_exist() == YES)
	    VECTOR(vmuscl,source,nfloats,sizeof(float*));

	use_artificial_dissipation(&Avisc);
	if (use_lapidus_artificial_viscosity(Avisc))
	{
	    scalar(&vmuscl->avisc,sizeof(Vec_Avisc));
	    set_alloc(vmuscl->avisc,avisc);
	    MATRIX(vmuscl->avisc,g,3,vsize,FLOAT);
	    ASSIGN_ARRAY_POINTER(vmuscl->avisc,cs_ave,worksp[0][0]);
	    ASSIGN_ARRAY_POINTER(vmuscl->avisc,c_ave,worksp[0][1]);
	    ASSIGN_ARRAY_POINTER(vmuscl->avisc,vn_ave,worksp[0][2]);
	    ASSIGN_ARRAY_POINTER(vmuscl->avisc,b,worksp[1]);
	    ASSIGN_ARRAY_POINTER(vmuscl->avisc,visc,worksp[2][0]);
	    ASSIGN_ARRAY_POINTER(vmuscl->avisc,uconM,worksp[1]);
	    set_no_alloc(vmuscl->avisc,mdlambda);
	}
	else if (use_upwind_artificial_viscosity(Avisc) ||
	         use_linear_artificial_viscosity(Avisc))
	{
	    scalar(&vmuscl->avisc,sizeof(Vec_Avisc));
	    set_alloc(vmuscl->avisc,avisc);
	    MATRIX(vmuscl->avisc,g,1,vsize,sizeof(float*));
	    set_no_alloc(vmuscl->avisc,cs_ave);
	    set_no_alloc(vmuscl->avisc,c_ave);
	    set_no_alloc(vmuscl->avisc,vn_ave);
	    set_no_alloc(vmuscl->avisc,b);
	    set_no_alloc(vmuscl->avisc,visc);
	    set_no_alloc(vmuscl->avisc,uconM);
	    ASSIGN_ARRAY_POINTER(vmuscl->avisc,mdlambda,worksp[0][0]);
	}

	if (vmuscl->avisc != NULL)
	{
	    vmuscl->avisc->use_lapidus=use_lapidus_artificial_viscosity(Avisc);
	    vmuscl->avisc->use_linear = use_linear_artificial_viscosity(Avisc);
	    vmuscl->avisc->use_upwind = use_upwind_artificial_viscosity(Avisc);
	}

	if (use_muscl_slope_flattening(Avisc))
	{
	    scalar(&vmuscl->msf,sizeof(Vec_MSF));
	    set_alloc(vmuscl->msf,msf);
	    VECTOR(vmuscl->msf,chi,vsize,FLOAT);
	}
	else
	    vmuscl->msf = NULL;
	vmuscl->monotone_reconstruction = mopts->monotone_reconstruction;
	vmuscl->link_reconstructions = mopts->link_reconstructions;

	vmuscl->max_vsize = vsize;

	        /* Set Linear reconstruction data structure */
	ASSIGN_ARRAY_POINTER(vmuscl,backward,worksp[0]);
	ASSIGN_ARRAY_POINTER(vmuscl,central,worksp[1]);
	ASSIGN_ARRAY_POINTER(vmuscl,forward,worksp[2]);
	ASSIGN_ARRAY_POINTER(vmuscl,du,worksp[2]);
	ASSIGN_ARRAY_POINTER(vmuscl,q,worksp[3]);

	        /* Set half step calculation data */

	ASSIGN_ARRAY_POINTER(vmuscl,uL,worksp[0]);
	ASSIGN_ARRAY_POINTER(vmuscl,uR,worksp[1]);    
	ASSIGN_ARRAY_POINTER(vmuscl,uM,worksp[2]);
	ASSIGN_ARRAY_POINTER(vmuscl,right_wvs,worksp[2]);
	ASSIGN_ARRAY_POINTER(vmuscl,left_wvs,worksp[2]);
	ASSIGN_ARRAY_POINTER(vmuscl,Flux.F,worksp[3]);
	ASSIGN_ARRAY_POINTER(vmuscl,source_sum,worksp[3]);
	ASSIGN_ARRAY_POINTER(vmuscl,awv,worksp[3]);
	ASSIGN_ARRAY_POINTER(vmuscl,dq,worksp[4]);

	        /* Set Riemann solver date */
	vmuscl->pL = curv_set_vmuscl_vec_gas_storage(vmuscl->uL,&vmuscl->VLst,
	        	                          vmuscl);
	vmuscl->pR = curv_set_vmuscl_vec_gas_storage(vmuscl->uR,&vmuscl->VRst,
	        	                          vmuscl);
	vmuscl->pM = curv_set_vmuscl_vec_gas_storage(vmuscl->uM,&vmuscl->VMst,
	        	                          vmuscl);
	set_no_alloc(vmuscl,A);
	set_no_alloc(vmuscl,dV);
	return vmuscl;
}		/*end curv_muscl_alloc_wk_space*/


LOCAL	float *curv_set_vmuscl_vec_gas_storage(
	float	  **u,
	Vec_Gas	  *vst,
	Vec_Muscl *vmuscl)
{
	int	  dim = vmuscl->dim;
	int	  vsize = vmuscl->max_vsize;
	int i;

	zero_scalar(vst,sizeof(Vec_Gas));
	ASSIGN_ARRAY_POINTER(vst,rho,u[vmuscl->index.density]);
	ASSIGN_ARRAY_POINTER(vst,e,u[vmuscl->index.energy]);
	VECTOR(vst,v,dim,sizeof(float*));
	for (i = 0; i < dim; ++i)
	    vst->v[i] = u[vmuscl->index.v[i]];
	VECTOR(vst,prms_jmp,vsize+1,INT);
	VECTOR(vst,p,vsize,FLOAT);
	VECTOR(vst,re,vsize,FLOAT);
	VECTOR(vst,GAM,vsize,FLOAT);
	VECTOR(vst,FD,vsize,FLOAT);
	VECTOR(vst,c2,vsize,FLOAT);
	VECTOR(vst,c,vsize,FLOAT);
	return vst->p;
}		/*end curv_set_vmuscl_vec_gas_storage*/


LOCAL	void	left_multiply_state_by_matrix(
	float		**q,
	float		***l,
	float		**u,
	int		dim,
	int		start,
	int		end)
{
	float		*u0, *u1, *u2, *uk;
	float		*lk0, *lk1, *lk2;
	float		*qk;
	int		j, k;
	int		kmax = 2+dim;

	u0 = u[0]; u1 = u[1]; u2 = u[2];
	for (k = 0; k < 3; ++k)
	{
	    lk0 = l[k][0]; lk1 = l[k][1]; lk2 = l[k][2];
	    qk = q[k];
	    for (j = start; j < end; ++j)
	    	qk[j] = lk0[j]*u0[j] + lk1[j]*u1[j] + lk2[j]*u2[j];
	}
	for (k = 3; k < kmax; ++k)
	{
	    qk = q[k]; uk = u[k];
	    for (j = start; j < end; ++j)
	    	qk[j] = uk[j];
	}
}		/*end left_multiply_state_by_matrix*/


LOCAL void g_curv_cons_src(
	int		start,
	int		end,
	int		swp_num,
	int		*iperm,
	Tan_stencil	*tsten,
	Vec_Muscl	*vmuscl)
{
	float		*grav = vmuscl->grav;
	const float* const *Q = vmuscl->Q;
	int		i, j,dim;
	float		*rho_src, *E_src, *m_src[MAXD], *m_srci;
	float		**uM, **source;
	float		*rho;
	float		*v0;
	static bool	first = YES;

	if (((source = vmuscl->source) == NULL) || (vmuscl->is_src == NO)) 
	    return;

	dim = vmuscl->dim;
	uM = vmuscl->uM;
	rho = uM[0];
	v0 = uM[2];
	rho_src = source[0];
	E_src = source[1];
	for (i = 0; i < dim; ++i)
	    m_src[i] = source[i+2];

	if (first == YES)
	{
	    first = NO;
	}
	for (j = start; j <= end; ++j)
	{
	    rho_src[j] = 0.0;
	    E_src[j] = 0.0;
	}
	for (i = 0; i < dim; ++i)
	{
	    m_srci = m_src[i];
	    for (j = start; j <= end; ++j)
	    	m_srci[j] = 0.0;
	}
	if (tsten == NULL)
	{
	    /* Rect sweep */

	    if (grav != NULL)
	    {
	    	m_srci = m_src[0];
	    	for (j = start; j < end; ++j)
	    	{
	    	    m_srci[j] += 0.5*(rho[j] + rho[j+1])*grav[j];
	    	    E_src[j] += 0.5*(rho[j]*v0[j] + rho[j+1]*v0[j+1])*grav[j];
	    	}
	    }
	}
	else
	{
	    /* Tangential sweep */

	    if (grav != NULL)
	    {
	    	m_srci = m_src[0];
	    	for (j = start; j < end; ++j)
	        {
	            m_srci[j] += 0.5*(rho[j] + rho[j+1])*grav[j];
	            E_src[j] += 0.5*(rho[j]*v0[j] + rho[j+1]*v0[j+1])*grav[j];
	        }
	    }
	}
#if defined(DEBUG_MUSCL)
	if (debugging("csrc"))
	{
	    g_printout_vec_data("Here are the conservative source vectors",
	    	                source[0],source[1],source+2,
	    	                vmuscl->dim,start,end,"src");
	}
#endif /* defined(DEBUG_MUSCL) */
}		/*end g_curv_cons_src*/


LOCAL	float f_Jacobi_x_i(
	float	q1,
	float	q2,
	float	q3)
{
	return 1.0;
}

LOCAL	float f_Jacobi_x_c(
	float	q1,
	float	q2,
	float	q3)
{
	return q1;
}

LOCAL	float f_Jacobi_x_s(
	float	q1,
	float	q2,
	float	q3)
{
	return (q1*q1*cos(q3));
}
