/*                     
*				fsub.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains the miscellaneous subroutines
*
*		angle_from_c1_c2_at_common_node()
*		measure_front()
*
*		assign_front_interface()
*
*		f_max_front_time_step()
*
*		robust_quad_roots_in_interval()
*/


#include <front/fdecs.h>		/* includes int.h, table.h */


	/* LOCAL Function Declarations */
LOCAL	Front*	f_copy_front(Front*);
#if defined(USE_OVERTURE)
LOCAL   Front*  f_deep_copy_front(Front*);
LOCAL   void    f_deep_free_front(Front*);
#endif /* if defined(USE_OVERTURE) */

LOCAL	void	f_copy_MaxFrontSpeed(MAX_FRONT_SPEED*,MAX_FRONT_SPEED*,Front*);
LOCAL	void	f_destroy_MaxFrontSpeed(Front*);
LOCAL	void	f_free_front(Front*);
LOCAL	void	f_include_max_front_speed_info(MAX_FRONT_SPEED*,Front*);

#if defined(TWOD)
LOCAL	float	compute_curvature2d(float*,float*,int);
LOCAL	void	delete_passive_curves(INTERFACE*);
#endif /* defined(TWOD) */

#if defined(THREED) || defined(TWOD)
LOCAL	bool	f_reconstruct_front_at_grid_crossing(Front*,Front*,int);
#endif /* defined(THREED) || defined(TWOD)*/

/*
*			f_principal_tangent():
*
*	A simple function for computing a particular tangent vector to
*	a hypersurface.  In two space dimensions this will simply rotate
*	the normal vector by 90 degrees.  In 3D it flips (with a change of sign)
*	the largest and smallest components of the normal vector.
*/

/*ARGSUSED*/
EXPORT	void	f_principal_tangent(
	POINT			*p,
	HYPER_SURF_ELEMENT	*hse,
	HYPER_SURF		*hs,
	float			*nor,
	float			*vdir)
{
	int	i, imax, imin;
	int	dim = hs->interface->dim;
	float	len;

	for (i = 0; i < dim; ++i)
	    vdir[i] = 0.0;
	if (dim == 1)
	    return;

	imax = 0;	imin = dim-1;
	for (i = 1; i < dim; ++i)
	{
	    if (nor[i] > nor[imax])
	    	imax = i;
	    if (nor[i] < nor[imin])
	    	imin = i;
	}
	vdir[imax] = -nor[imin];
	vdir[imin] =  nor[imax];
	len = mag_vector(vdir,dim);
	for (i = 0; i < dim; ++i)
	    vdir[i] /= len;
}		/*end f_principal_tangent*/


/*
*                       f_deep_copy_front():
*
*       Basic default function for copying a front structure.
*       Allocates storage for the new front and copies the
*       argument into the new structure.
*/
#if defined(USE_OVERTURE)
LOCAL   Front *f_deep_copy_front(
        Front           *fr)
{
        Front           *newfr;

        scalar(&newfr,sizeof(Front));
        copy_into_front(newfr,fr);
        scalar(&(newfr->rect_grid), sizeof(RECT_GRID));
        *(newfr->rect_grid) = *(fr->rect_grid);
        scalar(&(newfr->pd_flag),sizeof(Patch_bdry_flag));
        return newfr;
}               /*end f_deep_copy_front*/
#endif /* if defined(USE_OVERTURE)  */


/*
*			f_copy_front():
*
*	Basic default function for copying a front structure.
*	Allocates storage for the new front and copies the
*	argument into the new structure.
*/

LOCAL	Front *f_copy_front(
	Front		*fr)
{
	Front		*newfr;

	scalar(&newfr,sizeof(Front));
	copy_into_front(newfr,fr);
	return newfr;
}		/*end f_copy_front*/

/*
*			f_copy_into_front():
*
*	Copies fr into newfr.  Assumes newfr is already allocated.
*/

EXPORT	void f_copy_into_front(
	Front		*newfr,
	Front		*fr)
{
	*newfr = *fr;
}		/*end f_copy_into_front*/

/*
*			f_free_front():
*
*	Basic front destructor.  Deletes fr->interf and then frees the
*	corresponding front.  Should only be used on fronts that were
*	created by f_copy_front.
*/

LOCAL	void	f_free_front(
	Front		*fr)
{
	if (fr == NULL)
	    return;
	if (fr->interf != NULL)
	    (void) delete_interface(fr->interf);
	free(fr);
}		/*end f_free_front*/


/*
*                       f_deep_free_front():
*
*       Basic front destructor.  Deletes fr->interf and then frees the
*       corresponding front.  Should only be used on fronts that were
*       created by f_copy_front.
*/
#if defined(USE_OVERTURE)
LOCAL   void    f_deep_free_front(
        Front           *fr)
{
        if (fr == NULL)
            return;

        if (fr->interf != NULL)
            (void) delete_interface(fr->interf);
        free(fr->rect_grid);
        free(fr->pd_flag);
        free(fr);
}               /*end f_deep_free_front*/
#endif /* if defined(USE_OVERTURE)  */

/*
*			f_set_default_front_parameters():
*
*	Sets default values for most fields in the front structure.  Fields
*	not initialized in this function are indicated by comments below.  As
*	of this writing, the order below reflects the declaration of the Front
*	structure in fdecs.h as much as possible.
*	Note: this function assumes that fr->rect_grid->dim and fr->sizest
*	have already been set.
*/

EXPORT	void	f_set_default_front_parameters(
	INIT_DATA	*init,
	Front		*fr)
{
	int		 dim = fr->rect_grid->dim;
	F_USER_INTERFACE *fuh = f_user_hook(dim);

	/*
	 * Remark: set_advance_front uses the value of Tracking_algorithm(fr)
	 * to select the appropriate advance_front function (in 3D)
	 * so the default value of this flag must be set prior to calling
	 * set_advance_front.
	 */
	Tracking_algorithm(fr) = (init != NULL) ?
	    tracking_algorithm(init) : NO_DYNAMIC_TRACKING;
	set_advance_front(init,fr);

	fr->_free_front =				f_free_front;
	fr->_copy_front =				f_copy_front;
	fr->_copy_into_front =				f_copy_into_front;

	fr->_print_Front_structure =			f_print_Front_structure;
	fr->_fprint_front =				f_fprint_front;
	fr->_read_print_front =				f_read_print_front;

#if defined(USE_OVERTURE)
        fr->_deep_free_front =                          f_deep_free_front;
        fr->_deep_copy_front =                          f_deep_copy_front;
#endif /* if defined(USE_OVERTURE) */

	/*	fr->sizest (assumed already set) */
	/*	fr->_state_interpolator */
	/*	fr->_tri_state_interpolator */
	fr->transform_state =				NULL;

	fr->_is_correspondence_possible =		NULL; 

	/*	fr->Redist */
	Node_redistribute_function(fr) =		NULL;

	fr->max_front_time_step =			f_max_front_time_step;
	/*	fr->_MaxFrontSpeed  (see below) */
	/*	fr->Tstep */

	/*	fr->dt */
	/*	fr->dt_frac */
	/*	fr->time */
	/*	fr->step */

	fr->hyperbolic =				FULL_STATES;
	/*	fr->npts_tan_sten */
	fr->init_topology_of_new_interface =		NULL;
	f_wave_capture(fr) =				NULL;
	fr->_init_propagate =				NULL;
	fr->curve_propagate =				NULL; 
	fr->node_propagate =				NULL; 
	/*	fr->_point_propagate */
	fr->bond_propagate =				NULL;
	fr->snd_node_propagate =			NULL; 
	fr->tan_curve_propagate =			NULL; 
	/*	fr->_npt_tang_solver */
	/*	fr->_one_side_npt_tang_solver */
	fr->impose_bc =					NULL;

#if defined(ONED)
	fr->_untrack_point = NULL;
#endif /* defined(ONED) */
	fr->_untrack_curve = NULL;
#if defined(THREED)
	fr->_untrack_surface = NULL;
#endif /* defined(THREED) */

	switch (dim)
	{
#if defined(ONED)
	case 1:
	    fr->fr_bdry_untangle =		NULL;
	    fr->_check_delete_redundant_node =	NULL;
	    fr->_replace_unphys_loop = 		NULL;
	    fr->_untrack_point =		f_untrack_point;
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    if (init != NULL)
	        set_tangent_operator(tangent_method(init),dim);
	    fr->_check_delete_redundant_node =	NULL;
	    fr->fr_bdry_untangle =		f_boundary_untangle;
	    fr->_replace_unphys_loop = 		f_replace_unphys_loop;
	    Rect_boundary_redistribute_function(fr) = rect_bdry_redist2d;
	    fr->_untrack_curve =		f_untrack_curve;
	    fr->_reconstruct_front_at_grid_crossing =
	        	f_reconstruct_front_at_grid_crossing;
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    if (init != NULL)
	        set_normal3d_method(normal3d_method(init),dim);
	    if (debugging("no_tan_prop"))
	        fr->_tan_point_propagate =		NULL;
	    else
	        fr->_tan_point_propagate =		f_tan_point_propagate;
	    fr->_check_delete_redundant_node =	NULL;
	    fr->fr_bdry_untangle =		NULL;
	    fr->_replace_unphys_loop = 		NULL;
	    fr->_reconstruct_front_at_grid_crossing =
	        	f_reconstruct_front_at_grid_crossing;
	    fr->_principal_tangent =		f_principal_tangent;
	    fr->_untrack_surface =		f_untrack_surface;
	    break;
#endif /* defined(THREED) */
	}

	fr->fr_vec_bdry_untangle =			NULL; 
	fr->untangle_front =				NULL; 
	fr->B_node_bifurcation =			NULL; 
	fr->twodrproblem =				NULL; 
	fr->identify_physical_node =			NULL; 
	fr->init_2drproblem =				NULL; 
	fr->phys_split_bdry_cross =			NULL; 
	fr->phys_set_node_types =			NULL; 
	fr->parallel_refl_vec_wave =			NULL;

	fr->tan_curve_propagate =			NULL; 
	fr->_fgraph_front_states =			NULL; 
	fr->_fgraph_curve_states =			NULL; 
	fr->_fprint_header_for_graph_curve_states =	NULL; 
	fr->_find_i_to_prop_dir =			NULL;

	fr->neumann_bdry_state =			NULL;
	fr->_find_i_to_prop_dir = 			NULL;
	fr->is_nzn_bdry =				NULL;

	fr->_alloc_state = fuh->_alloc_state;
	fr->_clear_state = fuh->_clear_state;
	fr->_obstacle_state = fuh->_obstacle_state;

	MaxFrontSpeed(fr) = (init != NULL) ? InitialMaxFrontSpeed(init) : NULL;

	/*      fr->nfloats */
	fr->print_state =				NULL;
	fr->_fgraph_front_states =			NULL;
	fr->_fprint_header_for_graph_curve_states = 	NULL;
	fr->_fgraph_curve_states =			NULL;
	fr->mass_consv_diagn_driver =			NULL;

	fr->head_fsr =					AddToFsrList(NULL);

	/*	fr->FDIVIDER */
	/*	fr->interf */

}		/*end f_set_default_front_parameters*/

EXPORT	MAX_FRONT_SPEED	*f_alloc_MaxFrontSpeed(
	MAX_FRONT_SPEED	*mxsp,
	INTERFACE	*intfc,
	size_t		sizest)
{
	static MAX_FRONT_SPEED_OPERATORS DefaultOperators = {
	    f_set_max_front_speed,
	    f_include_max_front_speed_info,
	    f_initialize_max_front_speed,
	    f_fprint_max_front_speed_info,
	    f_read_print_max_front_speed_info,
	    f_copy_MaxFrontSpeed,
	    f_destroy_MaxFrontSpeed
	};
	int	i, j;
	byte    **buf;

	if (mxsp != NULL)
	    return mxsp;

	scalar(&mxsp,sizeof(MAX_FRONT_SPEED));

	mxsp->_sizest = sizest;
	matrix(&buf,MAXD+1,1,mxsp->_sizest);
	mxsp->_mxspst = (Locstate*)buf;
	matrix(&mxsp->_coords,MAXD+1,MAXD,FLOAT);
	for (i = 0; i <= MAXD; ++i)
	{
	    clear_state(intfc,mxsp->_mxspst[i],mxsp->_sizest);
	    for (j = 0; j < MAXD; ++j)
	    	mxsp->_coords[i][j] = HUGE_VAL;
	}
	mxsp->operators = DefaultOperators;

	return mxsp;
}		/*end f_alloc_MaxFrontSpeed*/

LOCAL	void	f_copy_MaxFrontSpeed(
	MAX_FRONT_SPEED	*nmxsp,
	MAX_FRONT_SPEED	*omxsp,
	Front		*fr)
{
	int	i, j, dim = fr->rect_grid->dim;
	
	nmxsp = alloc_MaxFrontSpeed(nmxsp,fr->interf,omxsp->_sizest);
	nmxsp->operators = omxsp->operators;
	nmxsp->_sizest = omxsp->_sizest;
	for (i = 0; i < dim+1; ++i)
	{
	    nmxsp->_spfr[i] = omxsp->_spfr[i];
	    if (nmxsp->_sizest != 0)
	        assign(nmxsp->_mxspst[i],omxsp->_mxspst[i],nmxsp->_sizest);
	    for (j = 0; j < dim; ++j)
	    	nmxsp->_coords[i][j] = omxsp->_coords[i][j];
	}
}		/*end f_copy_MaxFrontSpeed*/

LOCAL	void	f_destroy_MaxFrontSpeed(
	Front	*fr)
{
	free(MaxFrontSpeedState(fr));
	free(MaxFrontSpeedCoords(fr));
	free(MaxFrontSpeed(fr));
	MaxFrontSpeed(fr) = NULL;
}		/*end f_destroy_MaxFrontSpeed*/

/*
*
*			assign_interface_and_free_front():
*
*	Copies fr->interf into newfr->interf and then frees fr.
*/

EXPORT	void	assign_interface_and_free_front(
	Front		*newfr,
	Front		*fr)
{
	assign_front_interface(newfr,fr);
	fr->interf = NULL;
	// fr->mesh = NULL;
	free_front(fr);
}		/*end assign_interface_and_free_front*/

/*
*			assign_front_interface():
*
*/

EXPORT void assign_front_interface(
	Front		*left,
	Front		*right)
{
	if (left != NULL)
        {
	    (void) delete_interface(left->interf);
	    // (void) delete_interface(left->mesh);
        }

	left->interf = right->interf;
	// left->mesh = right->mesh;
}		/*end assign_front_interface*/


/*
*			f_max_front_time_step():
*
*	Sets max_dt to the maximum time step allowed by the
*	Courant-Friedrichs-Levy condition for the advancing front.
*/

EXPORT	float	f_max_front_time_step(
	Front		*fr,
	float		*coords)
{
	float		max_dt = HUGE_VAL;
	float		dt[MAXD+1];
	float		*spfr = Spfr(fr);
	float		*h = fr->rect_grid->h;
	int		i, j, dim = fr->interf->dim;

	for (i = 0; i < dim; ++i)
	{
	    if (spfr[i] > 0.0)
	    {
	        dt[i] = h[i]/spfr[i];
	        if (max_dt > dt[i])
	        {
	            max_dt = dt[i];
	            for (j = 0; j < dim; ++j)
	                coords[j] = MaxFrontSpeedCoords(fr)[i][j];
	        }
	    }
	    else
	        dt[i] = HUGE_VAL;
	}
	if (spfr[dim] > 0.0)
	{
	    dt[dim] = 1.0 / spfr[dim];
	    if (max_dt > dt[dim])
	    {
	        max_dt = dt[dim];
	        for (j = 0; j < dim; ++j)
	    	    coords[j] = MaxFrontSpeedCoords(fr)[dim][j];
	    }
	}
	else
	    dt[dim] = HUGE_VAL;
	if (debugging("time_step"))
	{
	    (void) printf("In f_max_front_time_step()\n");
	    for (i = 0; i < dim; ++i)
	    {
	        (void) printf("front: spfr[%d] %g dt[%d]  %g dx[%d] %g\n",
	                      i,spfr[i],i,dt[i],i,h[i]);
	        (void) printf("front - max_dt = %g\n",max_dt);
	    }
	    (void) printf("front: spfr[%d] %g dt[%d]  %g\n",
	                  i,spfr[i],i,dt[i]);
	    (void) printf("front - max_dt = %g\n",max_dt);
	    print_general_vector("coords = ",coords,dim,"\n");
	}
	return max_dt;
}		/*end f_max_front_time_step*/



EXPORT	void set_default_tan_stencil(
	Tan_stencil	*stencil)
{
	int		i;

	for (i = 0; i < stencil->npts; ++i)
	{
	    stencil->hsstore[i] = NULL;
	    stencil->hsestore[i] = NULL;
	    stencil->tstore[i] = ERROR_FLOAT;
	}		
	stencil->curvature = 0.0;
}			/*end set_default_tan_stencil*/

/*
*			alloc_tan_stencil():
*
*/

EXPORT	Tan_stencil *alloc_tan_stencil(
	Front		*fr,
	int		nrad)
{
	byte		*store;
	int		npts = 2*nrad + 1;
	Tan_stencil	*sten;
	int		i;
	size_t		size;
	size_t		hs_offset, hse_offset;
	size_t		p_offset, t_offset, ls_offset, rs_offset;

	size = sizeof(Tan_stencil);
	hs_offset = (size%sizeof(HYPER_SURF*) != 0) ?
	            (size/sizeof(HYPER_SURF*) + 1)*sizeof(HYPER_SURF*) : size;
	size = hs_offset + npts*sizeof(HYPER_SURF*);
	hse_offset = (size%sizeof(HYPER_SURF_ELEMENT*) != 0) ?
	    (size/sizeof(HYPER_SURF_ELEMENT*) + 1)*sizeof(HYPER_SURF_ELEMENT*) :
	    size;
	size = hse_offset + npts*sizeof(HYPER_SURF_ELEMENT*);
	p_offset = (size%sizeof(POINT*) != 0) ?
	    (size/sizeof(POINT*) + 1)*sizeof(POINT*) : size;
	size = p_offset + npts*sizeof(POINT*);
	t_offset = (size%sizeof(float) != 0) ?
	    (size/sizeof(float) + 1)*sizeof(float) : size;
	size = t_offset + npts*sizeof(float);
	ls_offset = (size%sizeof(Locstate) != 0) ?
	    (size/sizeof(Locstate) + 1)*sizeof(Locstate) : size;
	size = ls_offset + 2*npts*sizeof(Locstate);
	rs_offset = ls_offset + npts*sizeof(Locstate);

	scalar(&store,size);
	sten = (Tan_stencil*)store;
	sten->hsstore = (HYPER_SURF**)(store + hs_offset);
	sten->hsestore = (HYPER_SURF_ELEMENT**)(store + hse_offset);
	sten->pstore = (POINT**)(store + p_offset);
	sten->tstore = (float*)(store + t_offset);
	sten->leftststore = (Locstate*)(store + ls_offset);
	sten->rightststore = (Locstate*)(store + rs_offset);

	for (i = 0; i < npts; ++i)
	    sten->pstore[i] = Static_point(fr->interf);
	sten->npts = npts;
	sten->hs = sten->hsstore + nrad;
	sten->hse = sten->hsestore + nrad;
	sten->p = sten->pstore + nrad;
	sten->t = sten->tstore + nrad;
	sten->leftst = sten->leftststore + nrad;
	sten->rightst = sten->rightststore + nrad;
	for (i = 0; i < npts; ++i)
	{
	    sten->leftststore[i] = left_state(sten->pstore[i]);
	    sten->rightststore[i] = right_state(sten->pstore[i]);
	}
	return sten;
}			/*end alloc_tan_stencil*/

EXPORT	void measure_front(
	Front		*front)
{
	switch (front->rect_grid->dim)
	{
#if defined(ONED)
	case 1: /*TODO implement measure front*/
	    Front_length(front) = 0;
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    {
	        BOND  *b;
	        CURVE *c;

	        /* Compute Length of Front */

	        Front_length(front) = 0.0;
	        (void) next_bond(front->interf,NULL,NULL);
	        while (next_bond(front->interf,&b,&c)) 
	        {
	    	    if (is_bdry(c))
			continue;
	    	    Front_length(front) += bond_length(b);
	        }
	    }
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3: /*TODO implement measure front*/
	    Front_length(front) = ERROR_FLOAT;
	    break;
#endif /* defined(THREED) */
	}
}		/*end measure_front*/


EXPORT	int syncronize_time_step_status(
	int	 status,
	PP_GRID* pp_grid)
{
	long		max_status, min_status;

	if (pp_grid->nn == 1)
	    return status;

	min_status = status;
	pp_global_lmin(&min_status,1L);

	if (min_status == ERROR_IN_STEP)
	    return ERROR_IN_STEP;

	max_status = status;
	pp_global_lmax(&max_status,1L);

	if (max_status == MODIFY_TIME_STEP)
	    return MODIFY_TIME_STEP;

	return status;
}		/*end syncronize_time_step_status*/

EXPORT	void	f_set_max_front_speed(
	int		i,
	float		spd,
	Locstate	state,
	float		*coords,
	Front		*fr)
{
	if (fabs(spd) > Spfr(fr)[i])
	{
	    int	j, dim = fr->rect_grid->dim;
	    Spfr(fr)[i] = fabs(spd);
	    if (coords != NULL)
	    {
	    	for (j = 0; j < dim; ++j)
	    	    MaxFrontSpeedCoords(fr)[i][j] = coords[j];
	    }
	    if (state != NULL)
	    	assign(MaxFrontSpeedState(fr)[i],state,fr->sizest);
	}
}		/*end f_set_max_front_speed*/

LOCAL	void	f_include_max_front_speed_info(
	MAX_FRONT_SPEED	*mxsp,
	Front		*fr)
{
	int i, dim = fr->rect_grid->dim;

	debug_print("time_step","Entered f_include_max_front_speed_info()\n");
	for (i = 0; i <= dim; ++i)
	{
	    if (fabs(mxsp->_spfr[i]) > Spfr(fr)[i])
	    {
	    	int	j;

	    	Spfr(fr)[i] = fabs(mxsp->_spfr[i]);
	    	for (j = 0; j < dim; ++j)
	    	    MaxFrontSpeedCoords(fr)[i][j] = mxsp->_coords[i][j];
	    	assign(MaxFrontSpeedState(fr)[i],mxsp->_mxspst[i],fr->sizest);
	    }
	}
	debug_print("time_step","Left f_include_max_front_speed_info()\n");
}		/*end f_include_max_front_speed_info*/

EXPORT	void	f_initialize_max_front_speed(
	Front	*fr)
{
	MAX_FRONT_SPEED	*mxsp = MaxFrontSpeed(fr);
	int	i, j;

	debug_print("time_step","Entered f_initialize_max_front_speed()\n");
	for (i = 0; i <= MAXD; ++i)
	    mxsp->_spfr[i] = 0.0;
	for (i = 0; i <= MAXD; ++i)
	{
	    clear_state(fr->interf,mxsp->_mxspst[i],mxsp->_sizest);
	    for (j = 0; j < MAXD; ++j)
	    	mxsp->_coords[i][j] = HUGE_VAL;
	}
	if (debugging("time_step"))
	{
	    (void) printf("Spfr(fr) initialized to zero, ");
	    print_general_vector("Spfr(fr) = ",Spfr(fr),
	                         fr->rect_grid->dim+1,"\n");
	}
	debug_print("time_step","Left f_initialize_max_front_speed()\n");
}		/*end f_initialize_max_front_speed*/


/*
*			f_average_points():
*
*	Extension of i_average_points() that updates the states at the
*	averaged points.
*/

EXPORT	POINT *f_average_points(
	bool               newpoint,
	POINT		   *p1,
	HYPER_SURF_ELEMENT *hse1,
	HYPER_SURF	   *hs1,
	POINT		   *p2,
	HYPER_SURF_ELEMENT *hse2,
	HYPER_SURF	   *hs2)
{
	INTERFACE *intfc = hs1->interface;
	Locstate  sl1, sl2, sr1, sr2;
	Locstate  sl, sr;
	POINT     *pmid;
	float	  crds1[3];
	float	  crds2[3];
	size_t	  sizest = size_of_state(intfc);
	int	  i, dim = hs1->interface->dim;

	for (i = 0; i < dim; ++i)
	{
	    crds1[i] = Coords(p1)[i];
	    crds2[i] = Coords(p2)[i];
	}
	pmid = i_average_points(newpoint,p1,hse1,hs1,p2,hse2,hs2);

	if (sizest != 0)
	{
	  if (pmid != NULL)
	  {
	    slsr(p1,hse1,hs1,&sl1,&sr1);
	    slsr(p2,hse2,hs2,&sl2,&sr2);
	    sl = left_state(pmid);
	    sr = right_state(pmid);
	    bi_interpolate_intfc_states(intfc,0.5,0.5,crds1,sl1,crds2,sl2,sl);
	    bi_interpolate_intfc_states(intfc,0.5,0.5,crds1,sr1,crds2,sr2,sr);
	  }
	  else
	  {
	    static Locstate stmp;
	    if (stmp == NULL)
		alloc_state(intfc,&stmp,sizest);
	    slsr(p1,hse1,hs1,&sl1,&sr1);
	    slsr(p2,hse2,hs2,&sl2,&sr2);
	    bi_interpolate_intfc_states(intfc,0.5,0.5,crds1,sl1,crds2,sl2,stmp);
	    assign(sl1,stmp,sizest);
	    assign(sl2,stmp,sizest);
	    bi_interpolate_intfc_states(intfc,0.5,0.5,crds1,sr1,crds2,sr2,stmp);
	    assign(sr1,stmp,sizest);
	    assign(sr2,stmp,sizest);
#if defined(TWOD) || defined(THREED)
	    if ((hs1 == hs2) && (hse1 == hse2))
	    {
	      /* Update all instances of states at the two points */
#if defined(TWOD)
	      if (dim == 2)
	      {
		BOND  *b = Bond_of_hse(hse1); /* recall hse1 == hse2 */
		CURVE *c = Curve_of_hs(hs1);  /* recall hs1 == hs2   */
		if ( ( (b->prev == NULL) && (b->next == NULL) ) &&
		     ( ((p1 == b->start) && (p2 == b->end)) ||
		       ((p1 == b->end)   && (p2 == b->start)) ) )
		{
		  /* Do other curves share p1 and p2 as node positions? */
		  NODE  *ns, *ne;
		  CURVE **cc;
		  ns = c->start;
		  ne = c->end;
		  for (cc = intfc->curves; cc && *cc; ++cc)
		  {
		    if (*cc == c)
		      continue;
		    if (((*cc)->start == ns) && ((*cc)->end == ne))
		    {
	              slsr(p1,Hyper_surf_element((*cc)->first),
			      Hyper_surf(*cc),&sl1,&sr1);
	              slsr(p2,Hyper_surf_element((*cc)->last),
			      Hyper_surf(*cc),&sl2,&sr2);
	              bi_interpolate_intfc_states(intfc,0.5,0.5,crds1,sl1,
				                  crds2,sl2,stmp);
	              assign(sl1,stmp,sizest);
	              assign(sl2,stmp,sizest);
	              bi_interpolate_intfc_states(intfc,0.5,0.5,crds1,sr1,
				                  crds2,sr2,stmp);
	              assign(sr1,stmp,sizest);
	              assign(sr2,stmp,sizest);
		    }
		    if (((*cc)->end == ns) && ((*cc)->start == ne))
		    {
	              slsr(p1,Hyper_surf_element((*cc)->last),
			      Hyper_surf(*cc),&sl1,&sr1);
	              slsr(p2,Hyper_surf_element((*cc)->first),
			      Hyper_surf(*cc),&sl2,&sr2);
	              bi_interpolate_intfc_states(intfc,0.5,0.5,crds1,sl1,
				                  crds2,sl2,stmp);
	              assign(sl1,stmp,sizest);
	              assign(sl2,stmp,sizest);
	              bi_interpolate_intfc_states(intfc,0.5,0.5,crds1,sr1,
				                  crds2,sr2,stmp);
	              assign(sr1,stmp,sizest);
	              assign(sr2,stmp,sizest);
		    }
		  }
		}
	      }
#endif /* defined(TWOD) */
#if defined(THREED)
	      if ((dim == 3) && Boundary_point(p1) && Boundary_point(p2))
	      {
		TRI     *tri = Tri_of_hse(hse1); /* recall hse1 == hse2 */
		SURFACE *s = Surface_of_hs(hs1); /* recall hs1 == hs2   */
		int     v1, v2, side;

		v1 = Vertex_of_point(tri,p1);
		v2 = Vertex_of_point(tri,p2);
		if ((v1 == ERROR) || (v2 == ERROR))
		{
		  screen("ERROR in f_average_points(), invalid vertex\n");
		  clean_up(ERROR);
		}
		side = (v2 == Next_m3(v1)) ? v1 : v2;
		if (is_side_bdry(tri,side))
		{
		  BOND *b = Bond_on_side(tri,side);
		  BOND_TRI **btris;
		  for (btris = Btris(b); btris && *btris; ++btris)
		  {
		    if ((*btris)->surface != s) /*s already done*/
		    {
	              slsr(p1,Hyper_surf_element((*btris)->tri),
			      Hyper_surf((*btris)->surface),&sl1,&sr1);
	              slsr(p2,Hyper_surf_element((*btris)->tri),
			      Hyper_surf((*btris)->surface),&sl2,&sr2);
	              bi_interpolate_intfc_states(intfc,0.5,0.5,crds1,sl1,
				                  crds2,sl2,stmp);
	              assign(sl1,stmp,sizest);
	              assign(sl2,stmp,sizest);
	              bi_interpolate_intfc_states(intfc,0.5,0.5,crds1,sr1,
				                  crds2,sr2,stmp);
	              assign(sr1,stmp,sizest);
	              assign(sr2,stmp,sizest);
		    }
		  }
		}
	      }
#endif /* defined(THREED) */
	    }
#endif /* defined(TWOD) || defined(THREED) */
	  }
	}
	return pmid;
}		/*end f_average_points*/

EXPORT	void	delete_passive_boundaries(
	INTERFACE	*intfc)
{

	DEBUG_ENTER(delete_passive_boundaries)
	switch (intfc->dim)
	{
#if defined(ONED)
	case 1:
	    {
	        POINT **p;
	        for (p = intfc->points; p && *p; ++p)
	        {
		    if (is_passive_boundary(Hyper_surf(*p)))
	            {
		        delete_point(*p);
		        if (intfc->points == NULL)
		            break;
		        p = intfc->points-1;
	            }
	        }
	    }
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    delete_passive_curves(intfc);
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:/*TODO implement delete_passive_boundaries*/
	    break;
#endif /* defined(THREED) */
	}
	DEBUG_LEAVE(delete_passive_boundaries)
}		/*end delete_passive_boundaries*/

#if defined(TWOD)

LOCAL	void	delete_passive_curves(
	INTERFACE *intfc)
{
	CURVE	**delete_curves, **c;
	NODE	**delete_nodes, **join_nodes, **n;
	bool    sav_intrp;

	DEBUG_ENTER(delete_passive_curves)
	sav_intrp = interpolate_intfc_states(intfc);
	interpolate_intfc_states(intfc) = YES;
	delete_curves = NULL;
	for (c = intfc->curves; c && *c; ++c)
	{
	    if (is_passive_boundary(Hyper_surf(*c)))
	    {
	    	if (!add_to_pointers(*c,&delete_curves))
		{
		    screen("ERROR in delete_passive_curves(), "
			   "add_to_pointers() failed\n");
		    clean_up(ERROR);
		}
	    }
	}
	for (c = delete_curves; c && *c; ++c)
	    (void) delete_curve(*c);

	join_nodes = NULL;
	for (n = intfc->nodes; n && *n; ++n)
	{
	    if ((size_of_pointers((*n)->in_curves) == 1) &&
	        (size_of_pointers((*n)->out_curves) == 1) &&
		(wave_type((*n)->in_curves[0]) ==
		    wave_type((*n)->out_curves[0])) &&
		(is_bdry((*n)->in_curves[0]) == is_bdry((*n)->out_curves[0])) &&
		(positive_component((*n)->in_curves[0]) ==
		    positive_component((*n)->out_curves[0])) &&
		(negative_component((*n)->in_curves[0]) ==
		    negative_component((*n)->out_curves[0])) )
	    {
		bool join;
	        CURVE *cin, *cout;
		int   idir, iside;
		RECT_GRID *gr;

		join = YES;

	    	/* Check for corner nodes */
		gr = computational_grid(intfc);
		cin = (*n)->in_curves[0];
		cout = (*n)->out_curves[0];
		if (is_bdry(*n) && is_bdry(cin) && is_bdry(cout) &&
		    (rect_bdry_side_for_curve(&idir,&iside,cin,gr)
		     != rect_bdry_side_for_curve(&idir,&iside,cout,gr)))
		     join = NO;

		if (wave_type(cin) == DIRICHLET_BOUNDARY)
		{
		    HYPER_SURF *hsin, *hsout;

		    /*Dont join boundaries with different boundary states */
		    hsin = Hyper_surf(cin);
		    hsout = Hyper_surf(cout);

		    if (boundary_state_function(hsin) !=
		                    boundary_state_function(hsout))
		    	join = NO;
		    if (boundary_state(hsin) != boundary_state(hsout))
		    {
		        if ((boundary_state(hsin) == NULL) ||
			    (boundary_state(hsout) == NULL))
			    join = NO;
			else
			{
			    size_t sizest = size_of_state(intfc);
			    if (memcmp(boundary_state(hsin),
			               boundary_state(hsout),sizest) != 0)
			        join = NO;
			}
		    }
		}

		if (join)
		{
	            if (!add_to_pointers(*n,&join_nodes))
		    {
		        screen("ERROR in delete_passive_curves(), "
			       "add_to_pointers() failed\n");
		        clean_up(ERROR);
		    }
	        }
	    }
	}
	for (n = join_nodes; n && *n; ++n)
	{
	    join_curves((*n)->in_curves[0],(*n)->out_curves[0],
	                negative_component((*n)->in_curves[0]),
	                positive_component((*n)->in_curves[0]),
			NULL);
	}

	delete_nodes = NULL;
	for (n = intfc->nodes; n && *n; ++n)
	{
	    if (((*n)->in_curves == NULL) && ((*n)->out_curves == NULL))
	    {
	    	if (!add_to_pointers(*n,&delete_nodes))
		{
		    screen("ERROR in delete_passive_curves(), "
			   "add_to_pointers() failed\n");
		    clean_up(ERROR);
		}
	    }
	}
	for (n = delete_nodes; n && *n; ++n)
	    (void) delete_node(*n);
	interpolate_intfc_states(intfc) = sav_intrp;

	DEBUG_LEAVE(delete_passive_curves)
}		/*end delete_passive_curves*/

/*
*		angle_from_c1_c2_at_common_node():
*
*	Finds the angle between the curves c1 and c2 at their common node.
*	The angle is normalized to be in -PI to PI and is positive if
*	counter clockwise.
*/

EXPORT	float	angle_from_c1_to_c2_at_common_node(
	CURVE		*c1,
	ORIENTATION	c1_orient,
	CURVE		*c2,
	ORIENTATION	c2_orient,
	Front		*fr)
{
	float		a1[MAXD], a2[MAXD], ang;

	find_tangent_to_curve(Node_of(c1,c1_orient)->posn,
	        Bond_at_node(c1,c1_orient),c1,c1_orient,a1,fr);
	find_tangent_to_curve(Node_of(c2,c2_orient)->posn,
	        Bond_at_node(c2,c2_orient),c2,c2_orient,a2,fr);
	ang = normalized_angle(angle(a2[0],a2[1]) - angle(a1[0],a1[1]));
	if (ang < PI)
	    return ang;
	else 
	    return ang - 2*PI;
}		/*end angle_from_c1_to_c2_at_common_node*/

LOCAL	float compute_curvature2d(
	float		*vp,
	float		*vn,
	int		dim)
{
	float vprod[MAXD];
	float denominator;
	float mvp, mvn, mdpn;
	float kappa;

	if(debugging("curvature"))
	{
	    print_general_vector("vp = ",vp,dim,"\n");
	    print_general_vector("vn = ",vn,dim,"\n");
	}


	mvp = mag_vector(vp,dim); mvn = mag_vector(vn,dim);
	mdpn = distance_between_positions(vp,vn,dim);
	(void) vector_product(vp,vn,vprod,dim);
	denominator = mvp*mvn*mdpn;
	if (denominator == 0.0)
	{
	    if (vprod[0] > 0.0)
	        kappa = HUGE_VAL;
	    else if (vprod[0] < 0.0)
	        kappa = -HUGE_VAL;
	    else
	    {
	        (void) printf("WARNING in compute_curvature2d(), "
		              "undefined curvature\n");
		kappa = 0.0;
	    }
	}
	else
	    kappa = vprod[0] / denominator;
	return kappa;
}		/*end compute_curvature2d*/

/*ARGSUSED*/
EXPORT	float	f_mean_curvature_at_point2d(
	POINT		*p,
	HYPER_SURF_ELEMENT	*hse,
	HYPER_SURF		*hs,
	Front		*front)
{
	float		curvature = 0.0;
	CURVE		*c = Curve_of_hs(hs);
	BOND		*b = Bond_of_hse(hse);
	int		i, dim = c->interface->dim;
	float		vn[MAXD], vp[MAXD];
	float		space;
	static BOND	*bn = NULL, *bp = NULL;

	if (bn == NULL)
	{
	    scalar(&bn,sizeof(BOND));
	    bn->start = Static_point(front->interf);
	    bn->end = Static_point(front->interf);
	    scalar(&bp,sizeof(BOND));
	    bp->start = Static_point(front->interf);
	    bp->end = Static_point(front->interf);
	}
	debug_print("curvature","Entered f_mean_curvature_at_point2d()\n");
	if (debugging("curvature"))
	{
	    (void) printf("Point, Bond and Curve\n");
	    print_general_vector("p = ",Coords(p),dim,"\n");
	    /*print_bond(b);
	    print_curve(c);*/
	}

	if (b == NULL || c == NULL)
	    return 0.0;

	space = (wave_type(c) < FIRST_VECTOR_PHYSICS_WAVE_TYPE) ?
	        	Front_spacing(front,GENERAL_WAVE) : 
	        	Front_spacing(front,VECTOR_WAVE);

	if (p == b->start && b->prev == NULL)
	{
	    if (b->next == NULL)
	    {
	        if (debugging("curvature"))
	    	    (void) printf("curvature is FLAT\n");
	        debug_print("curvature","Left f_mean_curvature_at_point2d()\n");
	    	return 0.0;
	    }
	    bond_secant_to_curve(b->end,b,c,POSITIVE_ORIENTATION,bn,
	        	         front,space);
	    for (i = 0; i < dim; ++i)
	    {
	    	vp[i] = Coords(b->start)[i] - Coords(b->end)[i];
	    	vn[i] = 0.5*(Coords(bn->start)[i]+Coords(bn->end)[i])
	    			- Coords(b->end)[i];
	    }
	    curvature = compute_curvature2d(vp,vn,dim);
	    debug_print("curvature","Left f_mean_curvature_at_point2d()\n");
	    return curvature;
	}

	if (p == b->end && b->next == NULL)
	{
	    if (b->prev == NULL)
	    {
	        debug_print("curvature","Left f_mean_curvature_at_point2d()\n");
	        return 0.0;
	    }
	    bond_secant_to_curve(b->start,b,c,NEGATIVE_ORIENTATION,bp,
	    	                 front,space);
	    for (i = 0; i < dim; ++i)
	    {
	    	vp[i] = Coords(b->end)[i] - Coords(b->start)[i];
	    	vn[i] = 0.5*(Coords(bp->start)[i]+Coords(bp->end)[i])
	    			- Coords(b->start)[i];
	    }
	    curvature = compute_curvature2d(vp,vn,dim);
	    debug_print("curvature","Left f_mean_curvature_at_point2d()\n");
	    return curvature;
	}

	bond_secant_to_curve(p,b,c,NEGATIVE_ORIENTATION,bp,front,space);
	bond_secant_to_curve(p,b,c,POSITIVE_ORIENTATION,bn,front,space);
	for (i = 0; i < dim; ++i)
	{
	    vp[i] = Coords(bp->start)[i] - Coords(p)[i];
	    vn[i] = Coords(bn->end)[i]   - Coords(p)[i];
	}

	curvature = compute_curvature2d(vp,vn,dim);
	debug_print("curvature","Left f_mean_curvature_at_point2d()\n");
	return curvature;
}		/*end f_mean_curvature_at_point2d*/
#endif /* defined(TWOD) */

#if defined(THREED)

/*
*			f_mean_curvature_at_point3d():
*
*	The mean curvature at a point on a surface is given by the
*	trace of curvature tensor defined by the symmetric bilinear form
*
*			K(w1,w2) = <dn/dw1, w2>,
*
*	where w1 and w2 are elements of the tangent plane to the surface
*	at the point p,  n is the normal vector field to the surface, and
*	dn/dw1 is the directional derivative of the normal field n in the
*	direction w1.
*
*	The value of the mean curvature at an interior point of S can be
*	computed by the integral:
*
*	                        /
*		tr(K) = (1/PI)  \  <Kw,w> dl(w).
*			        /
*			      |w| = 1
*			     w in T (S)
*			           p
*
*	where l(w) is the arc length along the unit disk in the tangent
*	plane at p of S.
*
*	Expression for the mean curvature at boundary points of S can also
*	be derived.  However such expressions will involve integrals of
*	components of Kw in directions other than w.
*/

/*ARGSUSED*/
EXPORT	float	f_mean_curvature_at_point3d(
	POINT		   *p,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	   *hs,
	Front		   *fr)
{
	float	curvature = 0.0;
	TRI	*t;
	TRI	**tris;
	float	nor[3];
	float	dtheta;
	int	n, i, nt;
	static	float	**we = NULL;
	static	float	*km = NULL, *ke = NULL;
	static	TRI	**tri_list = NULL;
	static	int	n_alloc = 0;

	if (Boundary_point(p))
	{
	    /*TODO: computation of curvature for bdry points */
	    return 0.0;
	}
	normal(p,hse,hs,nor,fr);

	nt = tri_list_computed_by_normal(p,Tri_of_hse(hse),&tris,hs->interface);

	if (we == NULL)
	{
	    n_alloc = max(24,nt);
	    vector(&ke,n_alloc+1,FLOAT);
	    vector(&km,n_alloc+1,FLOAT);
	    matrix(&we,n_alloc+1,3,FLOAT);
	    vector(&tri_list,n_alloc,sizeof(TRI*));
	}
	if (n_alloc < nt)
	{
	    free_these(4,ke,km,we,tri_list);
	    n_alloc = max(n_alloc+24,nt);
	    vector(&ke,n_alloc+1,FLOAT);
	    vector(&km,n_alloc+1,FLOAT);
	    matrix(&we,n_alloc,3,FLOAT);
	    vector(&tri_list,n_alloc,sizeof(TRI*));
	}

	for (n = 0; n < nt; ++n)
	{
	    tri_list[n] = tris[n];
	}
	tris = tri_list;

	for (n = 0; n < nt; ++n)
	{
	    POINT       *pn, *p0, *p1, *p2;
	    float       wm[3], dndwm[3], dndwe[3];
	    float       nm[3], ne[3];
	    float       tmp_nm, tmp_wm, tmp_we;
	    float       tmp_dndwm, tmp_dndwe;
	    const float *tnor;
	    int         v;

	    t = tris[n];

	    /* Get the normal vector at the center of the triangle */
	    tnor = Tri_normal(t);
	    tmp_nm = Mag3d(tnor);

	    /* Compute displacement vectors from point to the adjacent    */
	    /* edge of the triangle (we) and to the triangle center (wm). */
	    v = Vertex_of_point(t,p);
	    pn = Point_of_tri(t)[Next_m3(v)];
	    p0 = Point_of_tri(t)[0];
	    p1 = Point_of_tri(t)[1];
	    p2 = Point_of_tri(t)[2];
	    for (i = 0; i < 3; ++i)
	    {
	    	wm[i] = (Coords(p0)[i] + Coords(p1)[i] + Coords(p2)[i])/3.0 -
			 Coords(p)[i];
	    	we[n][i] = Coords(pn)[i] - Coords(p)[i];

		/* The vector nm is the unit normal vector to the tri t */
	    	nm[i] = tnor[i]/tmp_nm;
	    }

	    /* Compute the directional derivative of n in the directions */
	    /* we and wm                                                 */
	    normal(p,Hyper_surf_element(t),hs,ne,fr);
	    tmp_wm = mag_vector(wm,3);
	    tmp_we = mag_vector(we[n],3);
	    for (i = 0; i < 3; ++i)
	    {
	    	wm[i] /= tmp_wm;
	    	we[n][i] /= tmp_we;
	    	dndwm[i] = (nm[i] - nor[i])/tmp_wm;
	    	dndwe[i] = (ne[i] - nor[i])/tmp_we;
	    }

	    /* Project wm, we, dndwm, and dndwe on the plane normal to nor */
	    tmp_wm = scalar_product(wm,nor,3);
	    tmp_we = scalar_product(we[n],nor,3);
	    tmp_dndwm = scalar_product(dndwm,nor,3);
	    tmp_dndwe = scalar_product(dndwe,nor,3);
	    for (i = 0; i < 3; ++i)
	    {
	    	wm[i] -= tmp_wm*nor[i];
	    	we[n][i] -= tmp_we*nor[i];
	    	dndwm[i] -= tmp_dndwm*nor[i];
	    	dndwe[i] -= tmp_dndwe*nor[i];
	    }

	    /* Store the estimates for the curvature tensor forms */
	    /* K(we,we) and K(wm,wm)                              */
	    ke[n] = scalar_product(dndwe,we[n],3);
	    km[n] = scalar_product(dndwm,wm,3);
	}

	/* Make the lists ke and we circular */
	ke[nt] = ke[0];
	for (i = 0; i < 3; ++i)
	    we[nt][i] = we[0][i];

	/* Use Simpson's rule to compute the integral 
	 *
	 *                      /
	 *	                \  <Kw,w> dl(w).
	 *		        /
	 *		      |w| = 1
	 *		     w in T (S)
	 *	                   p
	 */
	for (n = 0; n < nt; ++n)
	{
	    dtheta = acos(scalar_product(we[n],we[n+1],3));
	    curvature += (ke[n] + 4.0*km[n] + ke[n+1])*dtheta/6.0;
	}
	// curvature /= PI;
	curvature /= 2.0*PI;

	return curvature;
}		/*end f_mean_curvature_at_point3d*/

#endif /* defined(THREED) */

/*ARGSUSED*/
#if defined(THREED) || defined(TWOD)
LOCAL	bool	f_reconstruct_front_at_grid_crossing(
	Front*	fr,
        Front*  oldfr,
        int     which_grid)
{
	return YES;
}		/*end f_reconstruct_front_at_grid_crossing*/
#endif /* if defined(THREED) || defined(TWOD) */

