/*
*				fredist3d.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	This file contains the high-level routine for the re-distribution 
*	of triangles on given surfaces according to given criteria,
*
*			redistribute_surface()
*
*	and several elementary routines for splitting and joining
*	triangles.  Consider two adjacent triangles with a common side.
*	The two elementary operations, which are regarded as inverses of
*	each other, are to either split the common side by insertion of
*	a new point (and thus to split the triangles) or to shrink the
*	common side, and reduce it to a single point (and thus to reduce the
*	two triangles to a pair of edges).  A further elementary operation is
*	to flip the diagonal of a pair of adjacent triangles, ie of a
*	elementary quadralateral.
*
*	Also in this file are the three D additions to insert and delete_
*	point_in_bond.	With these three D additions, these two D routines
*	applied to curves of a three D interface, will modify the triangles
*	of the surfaces bounded by the curves and bonds, so that a valid
*	interface is produced.	These subroutines must not be called 
*	independently.
*
*	Each has the functionality as stated above.
*/

#if defined(THREED)

#define DEBUG_STRING	"redist3d"
#include <front/fdecs.h>		/* includes int.h, table.h */

enum _SPQ_FLAG {
	SHORTEST = 0,
	LONGEST	 = 1
};
typedef enum _SPQ_FLAG SPQ_FLAG;

enum _TRI_STATUS {
	SMALL = 1,
	LARGE,
	BAD_ANGLE,
	GOOD
};
typedef enum _TRI_STATUS TRI_STATUS;

	/* LOCAL Function Declarations */
LOCAL	POINTER_Q  *alloc_and_add_to_queue(TRI*,SURFACE*,POINTER_Q*,int);
LOCAL	POINTER_Q  *dequeue(TRI*,POINTER_Q*);
LOCAL	TRI_STATUS tri_status(TRI*,RECT_GRID*);
LOCAL	bool	   delete_min_side_of_tri(TRI*,int,SURFACE*,POINTER_Q**,Front*);
LOCAL	bool	   flip_max_side_of_tri(TRI*,int,Front*,POINTER_Q**);
LOCAL	bool	   is_tri_in_queue(TRI*,POINTER_Q*);
LOCAL	bool	   redistribute_surface(SURFACE*,Front*);
LOCAL 	bool 	   is_critical_side(TRI*,int);
LOCAL   bool       two_points_share_side(POINT*,TRI*,POINT*,INTERFACE*);
LOCAL	float	   angle_weight_of_tri(float*,float*,float*,RECT_GRID*);
LOCAL	float	   scaled_tri_area(float*,float*,float*,RECT_GRID*);
LOCAL	int	   find_scaled_extrem_edge(TRI*,RECT_GRID*,SPQ_FLAG);
LOCAL	void	   exchange_queues(POINTER_Q*,POINTER_Q*);
LOCAL	void	   redistribute_curve3d(CURVE*,Front*);
LOCAL	void	   sort_pointer_queue(POINTER_Q*,INTERFACE*,SPQ_FLAG);
LOCAL 	void 	   print_tri_surf_queue(POINTER_Q*);  

#if defined(DEBUG_STRING)
#define	DEBUG_FRONT(mesg,fr)	 debug_front(DEBUG_STRING,mesg,fr);
#else /* defined(DEBUG_STRING) */
#define	DEBUG_FRONT(mesg,fr)
#endif /* defined(DEBUG_STRING) */

/*
*			redistribute3d():
*
*	Returns one of the following values
*
*	BAD_REDISTRIBUTION  	
*		if Surface_redistribute() fails.
*	      	if intersections() fails after Surface_redistribute() succeeds. 
*		if restart_init is set and the restart interface is tangled.
*						
*	UNABLE_TO_UNTANGLE	
*		if redistributed intfc is tangled and scalar_unravel fails.
*		if Surface_redistribute() fails after scalar_unravel succeeds.
*		if intersections() fails after Surface_redistribute() succeeds.
*		if new tangle found after old tangle was successfully removed.
*
*	GOOD_REDISTRIBUTION	
*		otherwise.
*
*	If the do_redist flag is NO, the redistribution step
*	is omitted.
*/


EXPORT int redistribute3d(
	Front		*fr,
	bool		do_redist,
	bool		restart_init)
{
	CROSS		*cross;
	INTERFACE	*intfc;
	bool		status;
	bool		force_redistribute = NO;

	DEBUG_ENTER(redistribute3d)
	start_clock("redistribute");

	if (debugging("gvrdst3d"))
	{
	    char s[120];

	    (void) sprintf(s,"before-redist-ts%d",fr->step);
	    gview_plot_interface(s,fr->interf);
	}

	intfc = fr->interf;
	printf("Before Surface_redistribute()\n");
	if (consistent_interface(fr->interf))
	    printf("Interface is consistent\n");
	if (do_redist)
	{
	    status = (Interface_redistributed(fr) == YES) ?
			FUNCTION_SUCCEEDED :
			Surface_redistribute(fr,&force_redistribute);
	    intfc = fr->interf;
	    if (status == FUNCTION_FAILED)
	    {
		stop_clock("redistribute");
		(void) printf("WARNING in redistribute3d(), "
		              "Surface_redistribute() failed\n");
		print_interface(intfc);
		DEBUG_LEAVE(redistribute3d)
		return BAD_REDISTRIBUTION;
	    }
	}
	printf("After Surface_redistribute()\n");
	if (consistent_interface(fr->interf))
	    printf("Interface is consistent\n");
	if (Tracking_algorithm(fr) == LOCALLY_GRID_BASED_TRACKING)
	{
	    /*TMP*/
	    printf("Check interface consistency before repair\n");
	    consistent_interface(fr->interf);
	    printf("Interface is consistent!\n");
	    status = repair_front_at_grid_crossing(fr);
	    /*
	    if (status != FUNCTION_SUCCEEDED)
	    {
	    	status = reconstruct_front_at_grid_crossing(fr);
	    }
	    */
	    if (status != FUNCTION_SUCCEEDED)
	    {
	    	(void) printf("ERROR: repair_front_at_grid_crossing() failed\n");
		clean_up(ERROR);

	    }
	    start_clock("scatter_front");
	    status = scatter_front(fr);
	    if (!status)
	    {
	        (void) printf("WARNING in surface_redistribute(), "
	    	          "scatter_front() failed\n");
	    }
	    stop_clock("scatter_front");
	    stop_clock("redistribute");
	    DEBUG_LEAVE(redistribute3d)
	    return GOOD_REDISTRIBUTION;
	}

	if (debugging("gvrdst3d"))
	{
	    char s[120];

	    (void) sprintf(s,"before-intersect-ts%d",fr->step);
	    gview_plot_interface(s,fr->interf);
	}
	DEBUG_FRONT("before intersections check",fr)

		/* Check for Intersections in Front */

	  /* intersections does one of  the following:
	     1) returns NO if topology construction fails;
	     2) crashes if add_to_pointers() fails;
	     3) returns YES otherwise.
	  */

	if (pp_min_status(intersections(intfc,&cross,YES)) == FUNCTION_FAILED)
	{
	    stop_clock("redistribute");
	    (void) printf("WARNING in redistribute3d(), "
	                  "intersections() failed\n");
	    print_interface(intfc);
	    DEBUG_LEAVE(redistribute3d)
	    return BAD_REDISTRIBUTION;
	}

	if (debugging("gvrdst3d"))
	{
	    char s[120];

	    (void) sprintf(s,"before-untangle-ts%d",fr->step);
	    gview_plot_interface(s,fr->interf);
	}

	if (interface_is_tangled(cross))
	{
	    static const int Max_nattemps = 3;
	    int              nattemps;
	    (void) print_number_of_tangles("",intfc,cross);
	    start_clock("untangle");
	    if (restart_init) 
	    {
		stop_clock("untangle");
		stop_clock("redistribute");
		(void) printf("WARNING in redistribute(), "
		              "Restart interface tangled, cannot continue\n");
		DEBUG_LEAVE(redistribute3d)
		return BAD_REDISTRIBUTION;
	    }

	    nattemps = 0;
	    while (cross) 
	    {
		++nattemps;
		if (!scalar_unravel_3d(fr,&cross))
		{
		    stop_clock("untangle");
		    stop_clock("redistribute");
		    (void) printf("WARNING in redistribute3d(), "
		                  "scalar_unravel_3d() failed\n");
		    DEBUG_LEAVE(redistribute3d)
		    return UNABLE_TO_UNTANGLE;
		}
		force_redistribute = YES;
		if (!Surface_redistribute(fr,&force_redistribute))
		{
		    stop_clock("untangle");
		    stop_clock("redistribute");
		    (void) printf("WARNING in redistribute3d(), after "
		                  "untangling Surface_redistribute() failed\n");
		    DEBUG_LEAVE(redistribute3d)
		    return UNABLE_TO_UNTANGLE;
		}
		intfc = fr->interf;
		if (!pp_min_status(intersections(intfc,&cross,YES)))
		{
		    (void) printf("WARNING in redistribute3d(), "
		                  "After untangle, intersections() failed\n");
		    DEBUG_LEAVE(redistribute3d)
		    return UNABLE_TO_UNTANGLE;
		}
		if (interface_is_tangled(cross))
		{
		    if (nattemps>=Max_nattemps)
		    {
		        (void) printf("WARNING in redistribute3d(), "
				      "After untangle, intersections() finds "
				      "new tangle, too many attemps\n");
		        DEBUG_LEAVE(redistribute3d)
		        return UNABLE_TO_UNTANGLE;
		    }
		    else
		    {
		        (void) printf("WARNING in redistribute3d(), "
				      "After untangle, intersections() finds "
				      "new tangle, will attempt to untangle\n");
		    }
		}
		
	    }
	    stop_clock("untangle");
	}

	if (debugging("gvrdst3d"))
	{
	    char s[120];

	    (void) sprintf(s,"after-untangle-ts%d",fr->step);
	    gview_plot_interface(s,fr->interf);
	}

	stop_clock("redistribute");
	DEBUG_LEAVE(redistribute3d)
	return GOOD_REDISTRIBUTION;
}		/*end redistribute3d*/

/*
* 		surface_redistribute()
*
* 	The choice of name surface_redistribute() clashes in an obvious 
* 	way with the name redistribute_surface().  
*/

EXPORT bool surface_redistribute(
	Front		*fr,
	bool		*force_redist)
{
	SURFACE		**s;
	CURVE		**c;
	bool		status, min_status;
	bool		redist_non_vec_cur, redist_vec_cur;

	DEBUG_ENTER(surface_redistribute)
	if (DEBUG)
	{
	    (void) printf("Unredistributed interface\n");
	    print_interface(fr->interf);
	    (void) printf("checking consistency of input interface "
			      "into surface_redistribute\n");
	    if (!consistent_interface(fr->interf))
	    {
		screen("ERROR in surface_redistribute(), "
		       "inconsistent redistributed interface\n");
		clean_up(ERROR);
	    }
	    else
		(void) printf("redistributed interface is consistent\n");
	}

	status = YES;

		/* Check on redistribution conditions */

	if (*force_redist)
	{
	    redist_non_vec_cur = redist_vec_cur = YES;
	    *force_redist = NO;
	}
	else if (Redistribution_count(fr) < 0)
	{
	    redist_non_vec_cur = redist_vec_cur = NO;
	}
	else
	{
	    redist_vec_cur = redist_needed(fr,VECTOR_WAVE);
	    redist_non_vec_cur = redist_needed(fr,GENERAL_WAVE);
	}
	++Redistribution_count(fr);

		/* Redistribute vector surfaces */
	set_size_of_intfc_state(size_of_state(fr->interf));
	set_copy_intfc_states(YES);

	set_current_interface(fr->interf);

	if (redist_vec_cur == YES)
	{
	    for (s = fr->interf->surfaces; *s ; ++s) 
	    {
	    	if ((!omit_redistribution(*s)) &&
		    (wave_type(*s) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE))
	    	    if (!redistribute_surface(*s,fr))
			status = NO;
	    }
	}
	if (redist_non_vec_cur == YES)
	{
	    for (c = fr->interf->curves; c && *c; ++c)
	    {
	    	if (hsbdry_type(*c) != SUBDOMAIN_HSBDRY)
	    	    redistribute_curve3d(*c,fr);
	    }

	    for (s = fr->interf->surfaces; *s ; ++s) 
	    {
	    	if ((!omit_redistribution(*s)) &&
	    	    (wave_type(*s) >= FIRST_PHYSICS_WAVE_TYPE) &&
	    	    (wave_type(*s) < FIRST_VECTOR_PHYSICS_WAVE_TYPE))
	    	    if (!redistribute_surface(*s,fr))
			status = NO;
	    }
	}

	min_status = pp_min_status(status);
	if (min_status == NO)
	{
	    (void) printf("WARNING in surface_redistribute(), "
		          "redistribute_surface(), failed\n");
	}
	start_clock("scatter_front");
	status = scatter_front(fr);
	if (!status)
	{
	    (void) printf("WARNING in surface_redistribute(), "
	    	          "scatter_front() failed\n");
	}
	stop_clock("scatter_front");
	set_current_interface(fr->interf);

	Interface_redistributed(fr) =
	    ((redist_vec_cur==YES) || (redist_non_vec_cur==YES)) ? YES : NO;

	if (DEBUG)
	{
	    if (redist_vec_cur || redist_non_vec_cur)
	    {
	    	(void) printf("Redistributed interface\n");
	    	print_interface(fr->interf);
	        (void) printf("checking consistency of redistributed "
			      "interface\n");
	        if (!consistent_interface(fr->interf))
	        {
		    screen("ERROR in surface_redistribute(), "
		           "inconsistent redistributed interface\n");
		    clean_up(ERROR);
	        }
	        else
		    (void) printf("interface is consistent\n");
	    }
	    else
	    	(void) printf("Interface left unredistributed\n");
	}
	DEBUG_LEAVE(surface_redistribute)
	return status;
}		/*end surface_redistribute*/

LOCAL	const int	Num_pqs_in_block = 1000;

struct _TRI_SURF {
	TRI	*tri;
	SURFACE *surf;
	CURVE   *c01, *c12, *c20;
	float   sqr_norm;
	int     side;
};
typedef struct _TRI_SURF	TRI_SURF; 

#define Tri_surf(p)			((TRI_SURF *) (p)->pointer)
#define PQ_for_tri(tri)			((POINTER_Q *) Tri_workspace(tri))
#define tri_surface_from_queue(tri)	(Tri_surf(PQ_for_tri(tri)))
#define Bond_of_q(pq)		        ((BOND *)(pq)->pointer)
#define Tri_of_q(pq)		        (Tri_surf(pq)->tri)


LOCAL	float	max_sqr_area;	/* maximum triangle area	*/
LOCAL	float	min_sqr_area;	/* minimum triangle area	*/
LOCAL	float	max_sqr_length; /* maximun triangle side length */
LOCAL	float	aspect_tol2;	/* square of aspect ratio tolerance */


LOCAL POINTER_Q *dequeue(
	TRI       *tri,
	POINTER_Q *pq)
{
	if (PQ_for_tri(tri))
	{
	    if (head_of_pointer_queue(PQ_for_tri(tri))
				    == head_of_pointer_queue(pq))
	    {
	        pq = delete_from_pointer_queue(PQ_for_tri(tri));
	        Tri_workspace(tri) = NULL;
	    }
	}
	return pq;
}		/*end dequeue*/

LOCAL POINTER_Q *alloc_and_add_to_queue(
	TRI       *t,
	SURFACE   *s,
	POINTER_Q *pq,
	int       nside)
{
	TRI_SURF    *ts;
	const float *nor;
	pq = add_to_pointer_queue(NULL,pq);
	Tri_workspace(t) = (POINTER) pq;
	ts = tri_surface_from_queue(t);
	if (ts == NULL)
	{
	    screen("ERROR in alloc_and_add_to_queue(), "
	           "tri_surface_from_queue() returns NULL\n");
	    clean_up(ERROR);
	}
	ts->tri = t;
	nor = Tri_normal(t);
	ts->sqr_norm = Dot3d(nor,nor);
	ts->surf = s;
	ts->c01 = ts->c12 = ts->c20 = NULL;
	ts->side = nside;
	return pq;
}		/*end alloc_and_add_to_queue*/



LOCAL  bool redistribute_surface(
	SURFACE		*s,
	Front		*fr)
{
	INTERFACE *intfc = s->interface;
	POINT	  *midp;
	POINTER_Q *insert_queue = NULL;
	POINTER_Q *delete_queue = NULL;
	POINTER_Q *flip_queue = NULL;
	RECT_GRID *gr = fr->rect_grid;
	TRI	  *tri = NULL;
	bool      status;
	int	  n, nside;
	int       nf, nl, ns, nf_last, nl_last, ns_last;
	int	  dim = intfc->dim;
	int	  wc = (wave_type(s)< FIRST_VECTOR_PHYSICS_WAVE_TYPE) ?
				GENERAL_WAVE : VECTOR_WAVE;
	const int Ncyles = 2;

	DEBUG_ENTER(redistribute_surface)

			/* initialize queue */
	set_pointer_queue_opts(PQ_BLOCK_SIZE,Num_pqs_in_block,PQ_ALLOC_TYPE,
		               "vmalloc",PQ_ALLOC_SIZE_FOR_POINTERS,
			       sizeof(TRI_SURF),0);

	max_sqr_area = Max_tri_sqr_area(fr,wc);
	min_sqr_area = Min_tri_sqr_area(fr,wc);
	max_sqr_length = Max_scaled_tri_side_sqr_length(fr);
	aspect_tol2 = sqr(Aspect_ratio_tolerance(fr,wc));
	/*TMP*/
	printf("Entering redistribute_surface()\n");
	printf("Number of nodes = %d\n",
		size_of_pointers((POINTER *)fr->interf->nodes));
	if (consistent_interface(fr->interf))
	{
	    printf("Before operation: interface is consistent!\n");
	}
	else
	    clean_up(ERROR);

	nf_last = nl_last = ns_last = INT_MIN;
	for (n = 0; n < Ncyles; ++n)
	{
	    status = YES;
	    flip_queue = insert_queue = delete_queue = NULL;
	    nside = 0;
	    nf = nl = ns = 0;
	    for (tri=first_tri(s); !at_end_of_tri_list(tri,s); tri=tri->next)
	    {
		Tri_index(tri) = 0;
	        switch (tri_status(tri,gr))
	        {
	        case BAD_ANGLE:	
		    ++nf;
	            flip_queue = alloc_and_add_to_queue(tri,s,flip_queue,nside);
	            break;
	        case LARGE:
		    ++nl;
	            insert_queue = alloc_and_add_to_queue(tri,s,
							  insert_queue,nside);
	            break;
	        case SMALL:
		    ++ns;
	            delete_queue = alloc_and_add_to_queue(tri,s,
							  delete_queue,nside);
	            break;
	        case GOOD:
	        default:
	            break;
	        }
	    }
	    if ((flip_queue==NULL) && (insert_queue==NULL) &&
		(delete_queue==NULL))
		break;

	    if ((nf == nf_last) && (nl == nl_last) && (ns == ns_last))
		break;
	    nf_last = nf; nl_last = nl; ns_last = ns;

	    if (DEBUG)
	    {
	        if (flip_queue)
	            (void) printf("%d tri%s with BAD_ANGLE%s\n",nf,
				  (nf>1)?"s":"",(nf>1)?"s":"");	
	        if (insert_queue)
	            (void) printf("%d LARGE large tri%s\n",nl,
				  (nf>1)?"s":"");	
	        if (delete_queue)
	            (void) printf("%d SMALL tri%s\n",ns,(nf>1)?"s":"");	
	    }
	    if (debugging("redist_queues"))
	    {
	        (void) printf("list of flip_queue:\n");
	        print_tri_surf_queue(flip_queue);
	        (void) printf("list of insert_queue:\n");
	        print_tri_surf_queue(insert_queue);
	        (void) printf("list of delete_queue:\n");
	        print_tri_surf_queue(delete_queue);
	    }

	    /*main loops for modifying the triangles in the queues. */

	    sort_pointer_queue(flip_queue,intfc,SHORTEST);
	    while (flip_queue)
	    {
	        TRI	 *oppt = NULL;
	        flip_queue = head_of_pointer_queue(flip_queue);

	        tri = Tri_of_q(flip_queue);
	        if (tri == NULL)
		    break;

	        nside = find_scaled_extrem_edge(tri,gr,LONGEST);
	        if (is_side_bdry(tri,nside))
	        {
	            flip_queue = dequeue(tri,flip_queue);
	            continue;
	        }

	        oppt = Tri_on_side(tri,nside);
	        if (oppt == NULL)
	        {
	            flip_queue = dequeue(tri,flip_queue);
	            continue;
	        }

	        flip_queue = dequeue(tri,flip_queue);

	        if (flip_max_side_of_tri(tri,nside,fr,&flip_queue))
		{
	            if (is_tri_in_queue(oppt,flip_queue))
	                flip_queue = dequeue(oppt,flip_queue);

	            if (is_tri_in_queue(oppt,insert_queue))
	                insert_queue = dequeue(oppt,insert_queue);

	            if (is_tri_in_queue(oppt,delete_queue))
	                delete_queue = dequeue(oppt,delete_queue);
		}
	 
	        flip_queue = tail_of_pointer_queue(flip_queue);
	    }/*flip_queue loop*/

	    sort_pointer_queue(insert_queue,intfc,LONGEST);
	    while (insert_queue)
	    {
	        TRI	 *oppt = NULL;
	        int	 i;
	        float coords[3];

	        insert_queue = head_of_pointer_queue(insert_queue);
	        tri = Tri_of_q(insert_queue);
		/*TMP*/
		/*
		printf("Tri Coords:\n");
		print_tri_coords(tri);
		printf("Neighbors:\n");
		if (!is_side_bdry(tri,0))
		    print_tri_coords(Tri_on_side(tri,0));
		if (!is_side_bdry(tri,1))
		    print_tri_coords(Tri_on_side(tri,1));
		if (!is_side_bdry(tri,2))
		    print_tri_coords(Tri_on_side(tri,2));
		*/
	        if (tri == NULL)
		    break;

	        nside = find_scaled_extrem_edge(tri,gr,LONGEST);
	        if (is_side_bdry(tri,nside))
	        {
	            insert_queue = dequeue(tri,insert_queue);
	            continue;
	        }

	        /* find and make tri side mid point */
	        for (i = 0; i < dim; ++i)
	            coords[i] = 0.5*(Coords(Point_of_tri(tri)[nside])[i] +
		                Coords(Point_of_tri(tri)[Next_m3(nside)])[i]);
	        midp = Point(coords);
		/*TMP*/
		if (the_point(midp))
		    printf("midp is the point\n");

	        oppt = Tri_on_side(tri,nside);
	        if (oppt == NULL)
	        {
	            insert_queue = dequeue(tri,insert_queue);
	            continue;
	        }

	        if (is_tri_in_queue(oppt,insert_queue))
		    insert_queue = dequeue(oppt,insert_queue);

	        if (is_tri_in_queue(oppt,delete_queue))
		    delete_queue = dequeue(oppt,delete_queue);

	        if (!insert_point_in_tri_side(midp,nside,tri,s))
		    status = NO;
	        insert_queue = dequeue(tri,insert_queue);
	        insert_queue = tail_of_pointer_queue(insert_queue);

	    } /*insert_queue loop*/

	    sort_pointer_queue(delete_queue,intfc,SHORTEST);
	    while (delete_queue)
	    {
	        BOND	 *b1,*b2;

	        delete_queue = head_of_pointer_queue(delete_queue);
	        tri = Tri_of_q(delete_queue);

	        nside = find_scaled_extrem_edge(tri,gr,SHORTEST);
	        if (is_side_bdry(tri,nside))
	        {
	            TRI	 *oppt;
	            int mside,lside;
	            b1 = Bond_on_side(tri,nside);
	            lside = find_scaled_extrem_edge(tri,gr,LONGEST);

	            mside = Next_m3(nside);
	            if (lside == mside)
		        nside = Prev_m3(nside);
	            else
		        nside = mside;

	            /*nside has been replaced by non_extrem side*/

	            if (is_side_bdry(tri,nside))
	            {
	                if (is_side_bdry(tri,lside))
	                {
		            delete_queue = dequeue(tri,delete_queue);
		            continue;
	                }
	                b2 = Bond_on_side(tri,nside);
	                if ((b1 != b2->next) && (b1 != b2->prev))
	                {
		            delete_queue = dequeue(tri,delete_queue);
		            continue;
	                }
	                oppt = Tri_on_side(tri,lside);
	                if (oppt == NULL)
	                {
		            delete_queue = dequeue(tri,delete_queue);
		            continue;
	                }

	                delete_queue = dequeue(tri,delete_queue);
			/*TMP*/
			printf("flipping in delete_queue\n");
	                if (flip_max_side_of_tri(tri,lside,fr,&delete_queue) &&
			    is_tri_in_queue(oppt,delete_queue))
	                    delete_queue = dequeue(oppt,delete_queue);

	                continue;
	            }
	        }

		/*TMP
		printf("In delete_queue()\n");
		if (the_point(Point_of_tri(tri)[0]) ||
		    the_point(Point_of_tri(tri)[1]) ||
		    the_point(Point_of_tri(tri)[2]))
		{
		    print_tri_coords(tri);
		}
		*/
	        if (!delete_min_side_of_tri(tri,nside,s,&delete_queue,fr))
		    status = NO;
		/*TMP
		if (consistent_interface(fr->interf))
		{
	    	    printf("After operation: interface is consistent!\n");
		}
		else
	    	    clean_up(ERROR);
	    	*/
	    }/*delete_queue loop */
	    if ((flip_queue==NULL) && (insert_queue==NULL) &&
		(delete_queue==NULL))
		break;
	}
	if (n == Ncyles)
        {
            status = NO;
            (void) printf("WARNING in surface_redistribute(), "
                          "bad surface elements remain after %d "
            		  "iterations\n",Ncyles);
        }
	/*TMP*/
	printf("Leaving redistribute_surface()\n");
	if (consistent_interface(fr->interf))
	    printf("After operation: interface is consistent!\n");
	else
	    clean_up(ERROR);
	DEBUG_LEAVE(redistribute_surface)
	return status;
}		/*end redistribute_surface*/

LOCAL  int find_scaled_extrem_edge(
	TRI		*tri,
	RECT_GRID	*grid,
	SPQ_FLAG	to_find)
{
	const float* const *s;
	float	h0 = grid->h[0], h1 = grid->h[1], h2 = grid->h[2];
	float	s00, s01, s02;
	float	s10, s11, s12;
	float	s20, s21, s22;
	float	len0, len1, len2;

	s = side_vector(tri);
	s00 = s[0][0]/h0; s01 = s[0][1]/h1; s02 = s[0][2]/h2;
	s10 = s[1][0]/h0; s11 = s[1][1]/h1; s12 = s[1][2]/h2;
	s20 = s[2][0]/h0; s21 = s[2][1]/h1; s22 = s[2][2]/h2;
	len0 = QDot3d(s0,s0); len1 = QDot3d(s1,s1); len2 = QDot3d(s2,s2);

	switch (to_find)
	{
	case LONGEST:
	    return (len0<len1) ? ((len1<len2) ? 2:1) : ((len0<len2) ? 2:0);
	case SHORTEST:
	    return (len0>len1) ? ((len1>len2) ? 2:1) : ((len0>len2) ? 2:0);
	default:
	    return -1;
	}
}		/*end find_scaled_extrem_edge*/


/*
*				tri_status():
*
*	Determines whether redistribution of a triangle is needed by
*	comparing the triangles normalized area with two tolerances.
*	In addition, this routine also checks the squared edge lengths and
*	determines their aspect ratios.
*	This routine has the following return values:
*
*	if (norm_area > max_sqr_area) return LARGE;
*	if (norm_area < min_sqr_area) return SMALL;
*	if (aspect ratio < aspect_tol) return BAD_ANGLE;
*	return GOOD;
*
*	The aspect ratio of a triangle is defined as A/(l0^2+l1^2+l2^2)
*	where A is the area of the triangle in the grid scaled metric
*	and the li are the lengths of the sides of the triangle in
*	grid scaled metric.
*/

LOCAL	TRI_STATUS tri_status(
	TRI *tri,
	RECT_GRID *gr)
{
	float	*p0 = Coords(Point_of_tri(tri)[0]);
	float	*p1 = Coords(Point_of_tri(tri)[1]);
	float	*p2 = Coords(Point_of_tri(tri)[2]);
	float	s00, s01, s02;
	float	s10, s11, s12;
	float	s20, s21, s22;
	float	N0, N1, N2;
	float	h0 = gr->h[0], h1 = gr->h[1], h2 = gr->h[2];
	float	sqr_area;
	float	a2,len[3],tri_area;
	const float *nor = Tri_normal(tri);

	sqr_area = 0.25*Dot3d(nor,nor); 
	if (sqr_area <= min_sqr_area)
	    return SMALL;

	s00 = (p1[0]-p0[0])/h0; s01 = (p1[1]-p0[1])/h1; s02 = (p1[2]-p0[2])/h2;
	s10 = (p2[0]-p1[0])/h0; s11 = (p2[1]-p1[1])/h1; s12 = (p2[2]-p1[2])/h2;
	s20 = (p0[0]-p2[0])/h0; s21 = (p0[1]-p2[1])/h1; s22 = (p0[2]-p2[2])/h2;
	QCross3d(s0,s2,N);

	tri_area = 0.5*sqrt(QDot3d(N,N));

			/* Check aspect ratio	*/
	len[0] = QDot3d(s0,s0);
	len[1] = QDot3d(s1,s1);
	len[2] = QDot3d(s2,s2);

	a2 = len[0]+len[1]+len[2];
	a2 = tri_area/sqr(a2);
	if (a2 < aspect_tol2)
	    return BAD_ANGLE;

	if ((len[0] > max_sqr_length) || (len[1] > max_sqr_length) ||
	    (len[2] > max_sqr_length))
	    return LARGE;

	if (sqr_area >= max_sqr_area)
	    return LARGE;
	return GOOD;
}		/*end tri_status*/

LOCAL bool delete_min_side_of_tri(
	TRI	  *tri,
	int	  side,
	SURFACE	  *s,
	POINTER_Q **pq,
	Front     *fr)
{
	BOND_TRI   **nb_bond_tri;
	HYPER_SURF *hs = Hyper_surf(s);
	POINT      *p[2], *pmid;
	POINT      *ps, *pe;
	POINT      **v, **start, **end;
	RECT_GRID  *gr = fr->rect_grid;
	INTERFACE  *intfc = s->interface;
	TRI        **tris[2], **tmp_tris, **otris;
	TRI        **nb_tri;
	TRI        **oldtris, **newtris;
	TRI        *t, *nbt, *nbtri;
	const float *tnor;
	float      max_sqr_norm, nor[3];
	int        nv, nt, ntris[2], num_newtris;
	int        i, j, k, is, ie;
	int        i0, i1, nside;
	const char *warn = "WARNING in delete_min_side_of_tri(),";
	
	DEBUG_ENTER(delete_min_side_of_tri)

	if (tri == NULL)
	{
	    DEBUG_LEAVE(delete_min_side_of_tri)
	    return YES;
	}

	if (DEBUG || the_tri(tri))
	{
	    POINT *q;
	    TRI   **t;
	    int   n;
	    (void) printf("Deleting side %d of tri ",side);
	    print_tri(tri,intfc);
	    nbtri = Tri_on_side(tri,side);
	    for (nside = 0; nside < 3; ++nside)
	       if (Tri_on_side(nbtri,nside) == tri)
		   break;
	    (void) printf("Deleting side %d of nbtri ",nside);
	    print_tri(nbtri,intfc);
	    (void) printf("Tri lists\n");
	    q = Point_of_tri(tri)[side];
	    n = set_tri_list_around_point(q,tri,&t,intfc);
	    (void) printf("%d triangles about point %llu (%g %g %g)\n",
			  n,point_number(q),
			  Coords(q)[0],Coords(q)[1],Coords(q)[2]);
	    for (i = 0; i < n; i++)
		/*TMP
		print_tri(t[i],intfc);
		*/ print_tri_coords(t[i]);
	    q = Point_of_tri(tri)[Next_m3(side)];
	    n = set_tri_list_around_point(q,tri,&t,intfc);
	    (void) printf("%d triangles about point %llu (%g %g %g)\n",
			  n,point_number(q),
			  Coords(q)[0],Coords(q)[1],Coords(q)[2]);
	    for (i = 0; i < n; i++)
		/*TMP
		print_tri(t[i],intfc);
		*/ print_tri_coords(t[i]);

	    if (debugging("consista"))
	    {
	        (void) printf("checking consistency of interface into "
			      "delete_min_side_of_tri\n");
	        if (!consistent_interface(s->interface))
	        {
	    	    screen("ERROR in delete_min_side_of_tri(), "
	    	           "inconsistent input interface\n");
	            clean_up(ERROR);
	        }
	        else
		    (void) printf("interface is consistent\n");
	    }
	}

	/* Delete side of two adjacent tris */

	if (is_side_bdry(tri,side))/*the impossible case*/
	{
	    (void) printf("%s tri is side bdry\n",warn);
	    DEBUG_LEAVE(delete_min_side_of_tri)
	    return NO;
	}
	if (is_critical_side(tri,side))
	{
	    nbtri = Tri_on_side(tri,side);
	    *pq = dequeue(tri,*pq);
	    *pq = dequeue(nbtri,*pq);
	    DEBUG_LEAVE(delete_min_side_of_tri)
	    return YES;
	}
	p[0] = Point_of_tri(tri)[side];
	p[1] = Point_of_tri(tri)[Next_m3(side)];
	if (Boundary_point(p[0]) && !Boundary_point(p[1]))
	{
	    p[0] = p[1];
	    p[1] = Point_of_tri(tri)[side];
	}

	if (Boundary_point(p[0]) && Boundary_point(p[1]))
	{
	    bool status;
	    int  nside = find_scaled_extrem_edge(tri,gr,LONGEST);

	    if (is_side_bdry(tri,nside))
	    {
		nbtri = Tri_on_side(tri,side);
	    	*pq = dequeue(tri,*pq);
	    	*pq = dequeue(nbtri,*pq);
	    	DEBUG_LEAVE(delete_min_side_of_tri)
		return YES;
	    }
	    status = flip_max_side_of_tri(tri,nside,fr,pq);
	    if (status != YES)
	    {
		nbtri = Tri_on_side(tri,nside);
	    	*pq = dequeue(tri,*pq);
	    	*pq = dequeue(nbtri,*pq);
	    }
	    DEBUG_LEAVE(delete_min_side_of_tri)
	    return YES;
	}
	else if (Boundary_point(p[1]))
	{
	    BOND_TRI *bt;
	    bool link_tris;
	    nbtri = Tri_on_side(tri,side);
	    for (nside = 0; nside < 3; ++nside)
	       if (Tri_on_side(nbtri,nside) == tri)
		   break;
	    ntris[0] = set_tri_list_around_point(p[0],tri,tris,intfc);
	    i0 = Vertex_of_point(tri,p[0]);
	    i1 = Vertex_of_point(nbtri,p[0]);
	    for (i = 0; i < ntris[0]; ++i)
	    {
		t = tris[0][i];
		k = Vertex_of_point(t,p[0]);
		Point_of_tri(t)[k] = p[1];
		if ((t != tri) && (t != nbtri))
		    set_normal_of_tri(t);
	    }
	    if (i0 == side)
	    {
		if (!is_side_bdry(tri,Next_m3(side)))
		{
		    link_tris = YES;
		    nbt = Tri_on_side(tri,Next_m3(side));
		}
		else
		{
		    link_tris = NO;
		    bt = Bond_tri_on_side(tri,Next_m3(side));
		}
		t = Tri_on_side(tri,Prev_m3(side));
	    }
	    else
	    {
		if (!is_side_bdry(tri,Prev_m3(side)))
		{
		    link_tris = YES;
		    nbt = Tri_on_side(tri,Prev_m3(side));
		}
		else
		{
		    link_tris = NO;
		    bt = Bond_tri_on_side(tri,Prev_m3(side));
		}
		t = Tri_on_side(tri,Next_m3(side));
	    }
	    if (link_tris)
	    	link_neighbor_tris(t,nbt);
	    else
	    	(void) link_tri_to_bond(bt,t,s,bt->bond,bt->curve);
	    remove_tri_from_surface(tri,s,YES);
	    if (i1 == nside)
	    {
		if (!is_side_bdry(nbtri,Next_m3(nside)))
		{
		    link_tris = YES;
		    nbt = Tri_on_side(nbtri,Next_m3(nside));
		}
		else
		{
		    link_tris = NO;
		    bt = Bond_tri_on_side(nbtri,Next_m3(nside));
		}
		t = Tri_on_side(nbtri,Prev_m3(nside));
	    }
	    else
	    {
		if (!is_side_bdry(nbtri,Prev_m3(nside)))
		{
		    link_tris = YES;
		    nbt = Tri_on_side(nbtri,Prev_m3(nside));
		}
		else
		{
		    link_tris = NO;
		    bt = Bond_tri_on_side(nbtri,Prev_m3(nside));
		}
		t = Tri_on_side(nbtri,Next_m3(nside));
	    }
	    if (link_tris)
	    	link_neighbor_tris(t,nbt);
	    else
	    	(void) link_tri_to_bond(bt,t,s,bt->bond,bt->curve);
	    remove_tri_from_surface(nbtri,s,YES);
	    *pq = dequeue(tri,*pq);
	    *pq = dequeue(nbtri,*pq);
	    DEBUG_LEAVE(delete_min_side_of_tri)
	    return YES;
	}

	otris = NULL;
	start = NULL;
	end = NULL;
	tnor = Tri_normal(tri);
	max_sqr_norm = Dot3d(tnor,tnor);
	normal(p[0],Hyper_surf_element(tri),hs,nor,fr);
	for (k = 0; k < 2; ++k)
	{
	    ntris[k] = set_tri_list_around_point(p[k],tri,&tmp_tris,intfc);
	    tris[k] = (TRI**) store(ntris[k]*sizeof(TRI*));
	    for (i = 0; i < ntris[k]; ++i)
		tris[k][i] = tmp_tris[i];
	    for (i = 0; i < ntris[k]; ++i)
	    {
		tnor = Tri_normal(tris[k][i]);
		if (Dot3d(tnor,tnor) > max_sqr_norm)
		    normal(p[k],Hyper_surf_element(tris[k][i]),hs,nor,fr);
	        *pq = dequeue(tris[k][i],*pq);
	        j = Vertex_of_point(tris[k][i],p[k]);
	        ps = Point_of_tri(tris[k][i])[Next_m3(j)];
	        pe = Point_of_tri(tris[k][i])[Prev_m3(j)];
	        if ((ps!=p[0]) && (ps!=p[1]) && (pe!=p[0]) && (pe!=p[1]))
	        {
		    if (index_of_pointer_in_array(ps,start,&is) &&
		        index_of_pointer_in_array(pe,end,&ie) &&
			(is == ie))
		    {
		        if (!delete_from_ordered_pointers(ps,&start) ||
			    !delete_from_ordered_pointers(pe,&end) ||
			    !delete_from_ordered_pointers(otris[is],&otris))
		        {
	                    (void) printf("%s can't remove duplicate side, "
					  "is = %d ie = %d\n",warn,is,ie);
			    DEBUG_LEAVE(delete_min_side_of_tri)
	                    return NO;
		        }
		    }
		    else if (index_of_pointer_in_array(ps,end,&ie) &&
		             index_of_pointer_in_array(pe,start,&is) &&
			     (is == ie))
		    {
		        if (!delete_from_ordered_pointers(ps,&end) ||
			    !delete_from_ordered_pointers(pe,&start) ||
			    !delete_from_ordered_pointers(otris[is],&otris))
		        {
	                    (void) printf("%s can't remove duplicate side, "
					  "is = %d ie = %d\n",warn,is,ie);
			    DEBUG_LEAVE(delete_min_side_of_tri)
	                    return NO;
		        }
		    }
		    else
		    {
		        if (!add_to_ordered_pointers(ps,NULL,&start) ||
			    !add_to_ordered_pointers(pe,NULL,&end) ||
	                    !add_to_pointers(tris[k][i],&otris))
	                {
	                    (void) printf("%s can't side\n",warn);
			    DEBUG_LEAVE(delete_min_side_of_tri)
	                    return NO;
	                }
		    }
	        }
	    }
	}
	nv = (int) size_of_pointers(start);
	v = NULL;
	oldtris = NULL;
	if (!add_to_pointers(start[0],&v) ||
	    !add_to_pointers(end[0],&v) ||
	    !add_to_pointers(otris[0],&oldtris))
	{
	    (void) printf("%s can't add vertices\n",warn);
	    DEBUG_LEAVE(delete_min_side_of_tri)
	    return NO;
	}
	for (i = 1; i < nv-1; ++i)
	{
	    if (!index_of_pointer_in_array(v[i],start,&j))
	    {
	        (void) printf("%s vertex list is not a loop\n",warn);
		DEBUG_LEAVE(delete_min_side_of_tri)
	        return NO;
	    }
	    if (!add_to_pointers(end[j],&v) ||
	        !add_to_pointers(otris[j],&oldtris))
	    {
	        (void) printf("%s can't add vertices\n",warn);
		DEBUG_LEAVE(delete_min_side_of_tri)
	        return NO;
	    }
	}
	if (!index_of_pointer_in_array(v[i],start,&j) || (end[j] != v[0]))
	{
	    (void) printf("%s vertex list is not a loop\n",warn);
	    DEBUG_LEAVE(delete_min_side_of_tri)
	    return NO;
	}
	if (!add_to_pointers(otris[j],&oldtris))
	{
	    (void) printf("%s can't add vertices\n",warn);
	    DEBUG_LEAVE(delete_min_side_of_tri)
	    return NO;
	}
	nb_bond_tri = (BOND_TRI**)store(nv*sizeof(BOND_TRI*));
	nb_tri = (TRI**)store(nv*sizeof(TRI*));
	for (i = 0; i < nv; ++i)
	{
	    j = Vertex_of_point(oldtris[i],v[i]);
	    if (is_side_bdry(oldtris[i],j))
		nb_bond_tri[i] = Bond_tri_on_side(oldtris[i],j);
	    else
		nb_tri[i] = Tri_on_side(oldtris[i],j);
	}
	nt = nv;
	for (k = 0; k < 2; ++k)
	{
	    for (i = 0; i < ntris[k]; ++i)
	    {
		if (!pointer_is_in_array(tris[k][i],oldtris))
		{
	            if (!add_to_pointers(tris[k][i],&oldtris))
	            {
	                (void) printf("%s can't add vertices\n",warn);
		        DEBUG_LEAVE(delete_min_side_of_tri)
	                return NO;
	            }
		    ++nt;
		}
	    }
	}
	if (DEBUG)
	{
	    (void) printf("Bounding polygon has %d verticies\n",nv);
	    for (i = 0; i < nv; ++i)
		(void) printf("v[%d] %llu = %g %g %g\n",i,
			      point_number(v[i]),
			      Coords(v[i])[0],Coords(v[i])[1],Coords(v[i])[2]);
	    for (i = 0; i < nv; ++i)
	    {
	        j = Vertex_of_point(oldtris[i],v[i]);
		(void) printf("side[%d] = %d\n",i,j);
	    }
	    for (i = 0; i < nv; ++i)
	    {
		(void) printf("oldtris[%d] - ",i);
		print_tri(oldtris[i],intfc);
	    }
	    (void) printf("Neighbors on boundary\n");
	    for (i = 0; i < nv; ++i)
	    {
		if (nb_bond_tri[i])
		    (void) printf("Bounding bond[%d] %llu\n",i,
				  bond_number(nb_bond_tri[i]->bond,fr->interf));
		if (nb_tri[i])
		    (void) printf("Bounding tri[%d] %llu\n",i,
				  tri_number(nb_tri[i],fr->interf));
	    }
	    for (i = 0; i < nv; ++i)
	    {
		if (nb_bond_tri[i])
		{
		    (void) printf("Bounding bond[%d] ",i);
		    print_bond(nb_bond_tri[i]->bond);
		}
		if (nb_tri[i])
		{
		    (void) printf("Bounding tri[%d] ",i);
		    print_tri(nb_tri[i],intfc);
		}
	    }
	    if (nt > nv)
	    {
	        (void) printf("%d Additional triangles to be deleted\n",nt-nv);
		for (i = nv; i < nt; ++i)
		{
		    (void) printf("oldtris[%d] - ",i);
		    print_tri(oldtris[i],intfc);
		}
	    }
	}
	pmid = average_points(YES,p[0],Hyper_surf_element(tri),Hyper_surf(s),
				  p[1],Hyper_surf_element(tri),Hyper_surf(s));
	/*
	if (!retriangulate_polygon(v,nv,&pmid,1,nor,oldtris,nt,s,nb_bond_tri,
				   nb_tri,&newtris,&num_newtris))
	{
	    (void) printf("%s can't retriangulate "
			  "possible fold at edge\n",warn);
	    */
	    nbtri = Tri_on_side(tri,side);
	    for (nside = 0; nside < 3; ++nside)
	       if (Tri_on_side(nbtri,nside) == tri)
		   break;
	    for (i = 0; i < 2; ++i)
	    {
	        for (j = 0; j < ntris[i]; ++j)
	        {
		    t = tris[i][j];
		    k = Vertex_of_point(t,p[i]);
		    Point_of_tri(t)[k] = pmid;
		    if ((t != tri) && (t != nbtri))
		        set_normal_of_tri(t);
	        }
	    }
	    nbt = Tri_on_side(tri,Next_m3(side));
	    t = Tri_on_side(tri,Prev_m3(side));
	    for (i = 0; i < 3; ++i)
	    {
	       if (Tri_on_side(t,i) == tri)
	           Tri_on_side(t,i) = nbt;
	       if (Tri_on_side(nbt,i) == tri)
	           Tri_on_side(nbt,i) = t;
	    }
	    nbt = Tri_on_side(nbtri,Next_m3(nside));
	    t = Tri_on_side(nbtri,Prev_m3(nside));
	    for (i = 0; i < 3; ++i)
	    {
	       if (Tri_on_side(t,i) == nbtri)
	           Tri_on_side(t,i) = nbt;
	       if (Tri_on_side(nbt,i) == nbtri)
	           Tri_on_side(nbt,i) = t;
	    }
	    remove_tri_from_surface(tri,s,YES);
	    remove_tri_from_surface(nbtri,s,YES);
	    *pq = dequeue(tri,*pq);
	    *pq = dequeue(nbtri,*pq);
	/*
	}
	*/
	if (DEBUG && debugging("consista"))
	{
	    (void) printf("checking consistency of interface after "
			  "delete_min_side_of_tri\n");
	    if (!consistent_interface(s->interface))
	    {
		screen("ERROR in delete_min_side_of_tri(), "
		       "inconsistent interface produced\n");
	        clean_up(ERROR);
	    }
	    else
		(void) printf("interface is consistent\n");
	}
	DEBUG_LEAVE(delete_min_side_of_tri)
	return YES;
}		/*end delete_min_side_of_tri*/

/*
*			angle_weight_of_tri():
*
*	Returns 2*(1 + cos(theta)) where theta is the maximum internal
*	vertex of the triangle when mapped into the scaled metric space
*	x[i] -> x[i]/h[i].
*	
*	Note the formulas are based on the law of cosines which states
*	that
*	
*	sqr(si) + sqr(sj) - sqr(sk) = 2*si*sj*cos(theta_k)
*
*	where si, sj, and sk are the lengths of the three sides of the
*	triangle and theta_k is the interior angle of the triangle at the
*	vertex opposite to side k.
*/

LOCAL float angle_weight_of_tri(
	float     *p0,
	float     *p1,
	float     *p2,
	RECT_GRID *grid)
{
	float s00, s01, s02;
	float s10, s11, s12;
	float s20, s21, s22;
	float l0,l1,l2,angle;
	float t0,t1,t2;
	float h0 = grid->h[0], h1 = grid->h[1], h2 = grid->h[2];

	s00 = (p1[0]-p0[0])/h0; s01 = (p1[1]-p0[1])/h1; s02 = (p1[2]-p0[2])/h2;
	s10 = (p2[0]-p1[0])/h0; s11 = (p2[1]-p1[1])/h1; s12 = (p2[2]-p1[2])/h2;
	s20 = (p0[0]-p2[0])/h0; s21 = (p0[1]-p2[1])/h1; s22 = (p0[1]-p2[1])/h2;
	l0 = QDot3d(s0,s0); l1 = QDot3d(s1,s1); l2 = QDot3d(s2,s2);

	if (l0 == 0 || l1 == 0 || l2 == 0)
	    return 0;
	t0 = (l0+l1-l2)/sqrt(l0*l1);
	t1 = (l1+l2-l0)/sqrt(l1*l2);
	t2 = (l0+l2-l1)/sqrt(l0*l2);
	angle = 2.0 + (t0 < t1) ? min(t0,t2) : min(t1,t2);

	return angle;
}		/*end angle_weight_of_tri*/


LOCAL bool flip_max_side_of_tri(
	 TRI		*tri,
	 int		side,
	 Front          *fr,
	 POINTER_Q	**pq)
{
	BOND_TRI     *bt;
	TRI          *otri;
	TRI_NEIGHBOR Nbr[3], ONbr[3];
	POINT        *pt1, *pt2;
	RECT_GRID    *gr = fr->rect_grid;
	float        *p, *op, *p1, *p2;
	float        f1,f2,l1,l2;
	int          i, nside, pside;
	int          oside, noside, poside;
	bool         bdry[3], obdry[3];

	DEBUG_ENTER(flip_max_side_of_tri)

	if (is_side_bdry(tri,side))
	{
	    DEBUG_LEAVE(flip_max_side_of_tri)
	    return NO;
	}

	otri = Tri_on_side(tri,side);
	nside = Next_m3(side);
	pside = Prev_m3(side);
	p = Coords(Point_of_tri(tri)[side]);
	op = Coords(Point_of_tri(tri)[nside]);
	pt1 = Point_of_tri(tri)[pside];
	p1 = Coords(pt1);

	for (oside = 0; oside < 3; ++oside)
	    if (Tri_on_side(otri,oside) == tri)
		break;

	noside = Next_m3(oside);
	poside = Prev_m3(oside);
	pt2 = Point_of_tri(otri)[poside];
	p2 = Coords(pt2);

	if (two_points_share_side(pt1,tri,pt2,fr->interf))
	{
	    screen("WARNING: Cannot do flipping, side already existed\n");
	    *pq = dequeue(tri,*pq);
	    *pq = dequeue(Tri_on_side(tri,side),*pq);
	    DEBUG_LEAVE(flip_max_side_of_tri)
	    return NO;
	}

	l1 = angle_weight_of_tri(p,op,p1,gr);
	l2 = angle_weight_of_tri(p,p2,op,gr);

	if (DEBUG)
	{
	    (void) printf("flipping common side of tri %llu and otri %llu\n",
			  tri_number(tri,fr->interf),
			  tri_number(otri,fr->interf));
	    (void) printf("side = %d, oside = %d\n",side,oside);
	    (void) printf("tri - ");
	    print_tri(tri,fr->interf);
	    (void) printf("otri - ");
	    print_tri(otri,fr->interf);
	    (void) printf("angle weight of tri = %g\n",l1);
	    (void) printf("angle weight of otri = %g\n",l2);
	}

	if (l2 < l1)/* we may have bad otri*/
	{
	    if (find_scaled_extrem_edge(otri,gr,LONGEST) == oside)
	    	f1 = l2;
	    else
	    {
		*pq = dequeue(tri,*pq);
		*pq = dequeue(Tri_on_side(tri,side),*pq);
		DEBUG_LEAVE(flip_max_side_of_tri)
		return NO;
	    }
	}
	else
	    f1 = l1;

	l1 = angle_weight_of_tri(p,p2,p1,gr);
	l2 = angle_weight_of_tri(op,p1,p2,gr);

	if (DEBUG)
	{
	    (void) printf("angle weight of flipped tri = %g\n",l1);
	    (void) printf("angle weight of flipped otri = %g\n",l2);
	}

	f2 = min(l1,l2);
	if (f2 <= f1)
	{
	    /* The flipped tris with an max angle bigger than the
	     * original max angle. In this case we need to compare
	     * the area of the tris.  If the min_area of the
	     * original two will be smaller than that of the flipped
	     * two, do flipping.  The area business is good for
	     * these tris in delete_queue since in some case we
	     * couldn't delete_side. We do flipping instead.
	     */

	    float atri, otri, antri, aontri, amin, namin;

	    atri = scaled_tri_area(p,op,p1,gr);
	    otri = scaled_tri_area(p,p2,op,gr);
	    amin = min(atri,otri);
	    antri = scaled_tri_area(p,p2,p1,gr);
	    aontri = scaled_tri_area(op,p1,p2,gr);
	    namin = min(antri,aontri);
	    if (amin >= namin)
	    {
		*pq = dequeue(tri,*pq);
		*pq = dequeue(Tri_on_side(tri,side),*pq);
		DEBUG_LEAVE(flip_max_side_of_tri)
	        return NO;
	    }
	}
	for (i = 0; i < 3; ++i)
	{
	    bdry[i] = is_side_bdry(tri,i) ? YES : NO;
	    Nbr[i] = Tri_neighbor(tri)[i];
	    obdry[i] = is_side_bdry(otri,i) ? YES : NO;
	    ONbr[i] = Tri_neighbor(otri)[i];
	}

	Point_of_tri(tri)[nside] = pt2;
	Tri_on_side(tri,nside) = otri;
	set_side_bdry(Boundary_tri(tri),nside,0);
	if (obdry[noside])
	{
	    bt = ONbr[noside].btri;
	    (void) link_tri_to_bond(bt,tri,bt->surface,bt->bond,bt->curve);
	}
	else
	{
	    set_side_bdry(Boundary_tri(tri),side,0);
	    Tri_on_side(tri,side) = ONbr[noside].tri;
	    for (i = 0; i < 3; ++i)
	    {
		if (Tri_on_side(ONbr[noside].tri,i) == otri)
		{
		    Tri_on_side(ONbr[noside].tri,i) = tri;
		    break;
		}
	    }
	}

	Point_of_tri(otri)[noside] = pt1;
	Tri_on_side(otri,noside) = tri;
	set_side_bdry(Boundary_tri(otri),noside,0);
	if (bdry[nside])
	{
	    bt = Nbr[nside].btri;
	    (void) link_tri_to_bond(bt,otri,bt->surface,bt->bond,bt->curve);
	}
	else
	{
	    set_side_bdry(Boundary_tri(otri),oside,0);
	    Tri_on_side(otri,oside) = Nbr[nside].tri;
	    for (i = 0; i < 3; ++i)
	    {
		if (Tri_on_side(Nbr[nside].tri,i) == tri)
		{
		    Tri_on_side(Nbr[nside].tri,i) = otri;
		    break;
		}
	    }
	}
	set_normal_of_tri(tri);
	set_normal_of_tri(otri);
	*pq = dequeue(tri,*pq);
	*pq = dequeue(otri,*pq);
	DEBUG_LEAVE(flip_max_side_of_tri)
	return YES;
}		/*end flip_max_side_of_tri*/


LOCAL bool is_tri_in_queue(
	TRI		*tri,
	POINTER_Q		*pq)
{
	POINTER_Q		*tri_q;

	for (tri_q = head_of_pointer_queue(pq); tri_q; tri_q = tri_q->next)
	{
	    if (Tri_of_q(tri_q) == tri)
		return YES;
	}
	return NO;
}		/*end is_tri_in_queue*/

LOCAL bool is_critical_side(
	TRI *tri,
	int side)
{
	TRI *nbtri = Tri_on_side(tri,side);
	TRI *nbp,*nbn;
	int i,nside;

	for (nside = 0; nside < 3; ++nside)
	    if (Tri_on_side(nbtri,nside) == tri) break;

	if (!is_side_bdry(tri,Prev_m3(side)) &&
	    !is_side_bdry(tri,Next_m3(side)))
	{
	    nbp = Tri_on_side(tri,Prev_m3(side));
	    nbn = Tri_on_side(tri,Next_m3(side));
	    for (i = 0; i < 3; ++i)
		if (Tri_on_side(nbp,i) == nbn) return YES;
	}

	if (!is_side_bdry(nbtri,Prev_m3(nside)) &&
	    !is_side_bdry(nbtri,Next_m3(nside)))
	{
	    nbp = Tri_on_side(nbtri,Prev_m3(nside));
	    nbn = Tri_on_side(nbtri,Next_m3(nside));
	    for (i = 0; i < 3; ++i)
		if (Tri_on_side(nbp,i) == nbn) return YES;
	}

	return NO;
}	/* end is_critical_side */

LOCAL	void	exchange_queues(
	POINTER_Q *pq1,
	POINTER_Q *pq2)
{
	TRI_SURF *ts1, *ts2, T;

	ts1 = Tri_surf(pq1);
	ts2 = Tri_surf(pq2);
	T = *ts1;
	*ts1 = *ts2;
	*ts2 = T;
	Tri_workspace(ts1->tri) = (POINTER) pq1;
	Tri_workspace(ts2->tri) = (POINTER) pq2;
}		/*end exchange_queues*/

/*ARGSUSED*/
LOCAL void sort_pointer_queue(
	POINTER_Q	*pq,
	INTERFACE	*intfc,
	SPQ_FLAG	flag)
{
	POINTER_Q	*pq1,*pq2;

	if (pq == NULL)
	    return;

	pq1 = head_of_pointer_queue(pq);

	while (pq1 != tail_of_pointer_queue(pq))
	{
	    pq2 = pq1->next;
	    if (flag == SHORTEST)
	    {
	        if (Tri_surf(pq1)->sqr_norm > Tri_surf(pq2)->sqr_norm)
	        {
	    	    exchange_queues(pq1,pq2);
	        }
	        while (pq2 != tail_of_pointer_queue(pq))
	        {
	    	    pq2 = pq2->next;
	    	    if (Tri_surf(pq1)->sqr_norm > Tri_surf(pq2)->sqr_norm)
	    	    {
	    	        exchange_queues(pq1,pq2);
	    	    }
	        }
	    }
	    else if (flag == LONGEST)
	    {
	        if (Tri_surf(pq1)->sqr_norm < Tri_surf(pq2)->sqr_norm)
	        {
	    	    exchange_queues(pq1,pq2);
	        }
	        while (pq2 != tail_of_pointer_queue(pq))
	        {
	    	    pq2 = pq2->next;
	    	    if (Tri_surf(pq1)->sqr_norm < Tri_surf(pq2)->sqr_norm)
	    	    {
	    	        exchange_queues(pq1,pq2);
	    	    }
	        }
	    }
	    pq1 = pq1->next;
	}
}		/*end sort_pointer_queue*/


LOCAL void redistribute_curve3d(
	CURVE		*c,
	Front		*fr)
{

	BOND		*b;
	static bool	first = YES;
	static float	max_b_length, min_b_length;


	if (first == YES)
	{
	    first = NO;
	    max_b_length = Max_bond_len(fr,GENERAL_WAVE);
	    min_b_length = Min_bond_len(fr,GENERAL_WAVE);
	}
       
	if (debugging("b_length"))
	{
	    (void) printf("max_b_length = %g, min_b_length = %g\n",
	                  max_b_length,min_b_length);
	    detail_of_curve(c);
	}
	
	for (b = c->first; b != NULL; b = b->next)
	{
	    if (b->length >= max_b_length)
	    {
	        POINT *pm = Point(NULL);
		int   i;
	        for (i = 0; i < 3; ++i)
		    Coords(pm)[i] = 0.5*(Coords(b->start)[i]+Coords(b->end)[i]);
	        (void) insert_point_in_bond(pm,b,c);
		b = b->next;
	    }
	    else if (hsbdry_type(c) < FIRST_PHYSICS_HSBDRY_TYPE)
	    {
	        if (b->length <= min_b_length)
		{
		    BOND *bp, *bn;
		    if ((bp = b->prev))
		    {
			if (!delete_start_of_bond(b,c))
		        {
			    screen("ERROR in redistribute_curve3d(), "
			           "delete_start_of_bond() failed\n");
			    clean_up(ERROR);
		        }
			b = bp;
		    }
		    else if ((bn = b->next))
		    {
			if (!delete_end_of_bond(b,c))
		        {
			    screen("ERROR in redistribute_curve3d(), "
			           "delete_end_of_bond() failed\n");
			    clean_up(ERROR);
		        }
			b = bn;
		    }
		}
	    }
	}
	if (debugging("redist_curve"))
	{
	    detail_of_curve(c);
	    summarize_interface("redist_curve3d","exit",c->interface,
				XY_PLANE,"redist_curve3d","exit");
	}
}		/*end redistribute_curve3d */

LOCAL float scaled_tri_area(
	float     *p0,
	float     *p1,
	float     *p2,
	RECT_GRID *gr)
{
	float		s00, s01, s02;
	float		s20, s21, s22;
	float		N0, N1, N2;
	float		h0 = gr->h[0], h1 = gr->h[1], h2 = gr->h[2];
	float		sqr_area;


	s00 = (p1[0]-p0[0])/h0; s01 = (p1[1]-p0[1])/h1; s02 = (p1[2]-p0[2])/h2;
	s20 = (p0[0]-p2[0])/h0; s21 = (p0[1]-p2[1])/h1; s22 = (p0[2]-p2[2])/h2;
	QCross3d(s0,s2,N);

	sqr_area = 0.25*QDot3d(N,N);

	return	sqr_area;
}		/*end scaled_tri_area*/

LOCAL 	void	print_tri_surf_queue(
	POINTER_Q *p_q)
{
  	POINTER_Q 	*q;
	TRI_SURF 	*t_surf;
	TRI 		*tri;
	int		cnt = 0;
  
	if (p_q == NULL) 
	{
	    (void) printf("NULL POINTER_Q\n");
	    return;
        }

	q = head_of_pointer_queue(p_q);
	while (q != tail_of_pointer_queue(p_q))
        {
	    t_surf = Tri_surf(q);
	    tri = t_surf->tri;
	    (void) printf("%3d ( %g %g %g ) ( %g %g %g ) "
			  "( %g %g %g )  %p  side = %d\n", ++cnt,
			  Coords(Point_of_tri(tri)[0])[0],
			  Coords(Point_of_tri(tri)[0])[1],
			  Coords(Point_of_tri(tri)[0])[2],
			  Coords(Point_of_tri(tri)[1])[0],
			  Coords(Point_of_tri(tri)[1])[1],
			  Coords(Point_of_tri(tri)[1])[2],
			  Coords(Point_of_tri(tri)[2])[0],
			  Coords(Point_of_tri(tri)[2])[1],
			  Coords(Point_of_tri(tri)[2])[2],tri,t_surf->side);
	    q = q->next;
	}
}		/*end print_tri_surf_queue*/


LOCAL bool two_points_share_side(
	POINT	  *p1,
	TRI	  *tri1,
	POINT     *p2,
	INTERFACE *intfc)
{
	TRI       **tris;
	int	  i,j,num_tris;

        num_tris = set_tri_list_around_point(p1,tri1,&tris,intfc);
	for (i = 0; i < num_tris; ++i)
	{
	    for (j = 0; j < 3; ++j)
	    {
	    	if (Point_of_tri(tris[i])[j] == p2)
		    return YES;
	    }
	}
	return NO;

}	/*end two_points_share_side*/		
#endif /* defined(THREED) */
