/*
*				fredist2d.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*/

#if defined(TWOD)

#define DEBUG_STRING    "redistribute"

#include <front/flocaldecs.h>         /* includes int.h, table.h */

	/* LOCAL Function Declarations */
LOCAL	bool  attach_sink_nodes(Front*);
LOCAL	bool  check_for_intersections(INTERFACE*,CROSS**,const bool);
LOCAL	bool  vector_wave_interaction(CROSS*);
LOCAL	bool  too_many_tangled_points(CROSS*,float*,int);
LOCAL	float redistribution_spacing(BOND*,CURVE*,float,Front*);
LOCAL	void  prepare_to_leave_redistribute2d(INTERFACE**);
LOCAL	void  replace_curve_seg_by_bond(CURVE*,BOND*,BOND*);
LOCAL	void  set_force_tangle(bool);

#if defined(DEBUG_STRING)
#define debug_bond_cross(cross,fr)					\
	if (cross != NULL && debugging("bond_cross"))			\
		print_crossing_elements(cross,fr->interf);
#define	DEBUG_FRONT(mesg,fr)	 debug_front(DEBUG_STRING,mesg,fr);
#else /* defined(DEBUG_STRING) */
#define debug_bond_cross(cross,fr)
#define	DEBUG_FRONT(mesg,fr)
#endif /* defined(DEBUG_STRING) */

#if defined(USE_OVERTURE) 
 #if defined(SMOOTH_INTERFACE_FOR_SCATTER)
LOCAL   bool  attach_nodes_on_bdry(Front*,NODE**,int*,int*); 
LOCAL   void  rm_attached_nodes_on_bdry(Front*,NODE**,int*,int*); 
LOCAL   void  attach_nodes_on_cut_line(INTERFACE*,float,int,int,int,NODE**,int*,int*);
 #endif /* if defined(SMOOTH_INTERFACE_FOR_SCATTER) */
LOCAL   int   no_untangle_redistribute2d(Front*,bool,bool);
#endif /* if defined(USE_OVERTURE) */ 

#define COMPLICATED_CROSS_STRUCTURE

LOCAL	bool _force_tangle = NO;

EXPORT	bool	force_tangle(void)
{
    return _force_tangle;
}		/*end force_tangle*/

LOCAL	void	set_force_tangle(
	bool	y_or_n)
{
	_force_tangle = y_or_n;
}		/*end set_force_tangle*/

/*
*			redistribute2d():
*
*	The redistribution leaves the endpoints of CURVES fixed, arrang-
*	ing that the distance along the front between points is approx.
*	equal to the constant spacing scaled by a factor depending on the
*	local interface curvature. The meaning of 'approximately'
*	is determined by the value of the paramater type_redistribute.
*	As described at the top of this file, three different redistribution
*	algorithms are available, depending on the this value of paramater.
*
*	After a redistribution, the routine intersections() determines
*	whether the new front is self-intersecting.  If so, a physics
*	dependent routine untangle_front() will be called to deal with
*	the intersections, and redistribute should be called again.
*
*	Returns one of the following values
*
*		GOOD_REDISTRIBUTION
*		UNABLE_TO_UNTANGLE
*		BAD_REDISTRIBUTION
*		MODIFY_TIME_STEP_REDISTRIBUTE
*
*	If the do_redist flag is NO, the redistribution step
*	is omitted, leaving only the attachment of sinks and the
*	untangle step.
*
*	Redistribute() is designed to interact with the function
*	advance_front(). If do_redist is YES (the usual case)
*	and the untangle routines fail, then advance_front() will
*	be restarted with the do_redist flag equals NO.
*	The reason for this behavior is that it is possible for
*	the redistribution algorithm to produce unphysical tangles
*	that were not present in the original interface.  If on
*	this second attempt untangle again fails, them the error
*	condition BAD_REDISTRIBUTION is returned.
*/

EXPORT int redistribute2d(
	Front		*fr,
	bool		do_redist,
	bool		restart_init)
{
	CROSS		*cross;
	INTERFACE	*intfc = fr->interf;
	INTERFACE	*save_intfc[2];
	O_NODE		*onode_list;
	float		*h = fr->rect_grid->h;
	float		hx = h[0], hy = h[1];
	float		dt, *dt_frac, min_dt_frac;
	int		status;
	bool		do_scatter = NO;
	int		small_loops_deleted;
	int		redist_status;
	bool		force_redistribute;
	bool		delete_small_loops_before_untangle;
	bool		istatus;
	bool		any_redist;
	int		num_untangles = 0;
	int		flag = NORMAL_ATTEMPT_TO_UNTANGLE;
	int		dim = fr->rect_grid->dim;
	static const int       MAX_NUM_UNTANGLES = 3;/*TOLERANCE*/
	static bool	first = YES;
	static float	dt_max, dt_min;
	DEBUG_ENTER(redistribute2d)

#if defined(USE_OVERTURE)
        if(fr->patch_level == 0 && fr->NumberOfLevels != 1)
            return no_untangle_redistribute2d(fr, do_redist, restart_init);
#endif /* if defined(USE_OVERTURE) */

	if (first)
	{
	    first = NO;
	    dt_max = HUGE_VAL;
	    dt_min = 0.0;
	}

	if (fr->interf->curves == NULL)
	{
	    DEBUG_LEAVE(redistribute2d)
	    return GOOD_REDISTRIBUTION;
	}

	if (DEBUG)
	{
	    if (Frequency_of_redistribution(fr,GENERAL_WAVE) > 0)
	    {
	        (void) printf("Count redistribute(%d) %% "
	                      "general curve frequency redistribute(%d) = %d\n",
	    		      Redistribution_count(fr),
			      Frequency_of_redistribution(fr,GENERAL_WAVE),
			      Redistribution_count(fr) % 
	    		      Frequency_of_redistribution(fr,GENERAL_WAVE));
	    }
	    if (Frequency_of_redistribution(fr,VECTOR_WAVE) > 0)
	    {
	        (void) printf("Count redistribute(%d) %% "
	                      "vector curve frequency redistribute(%d) = %d\n",
	    		      Redistribution_count(fr),
			      Frequency_of_redistribution(fr,VECTOR_WAVE),
	    		      Redistribution_count(fr) %
	    		      Frequency_of_redistribution(fr,VECTOR_WAVE));
	    }
	    DEBUG_FRONT("at start of redistribute2d()",fr)
	    (void) printf("Interface before redistribute2d\n");
	    print_interface(intfc);
	}

	save_intfc[0] = make_save_intfc(intfc);
	save_intfc[1] = NULL;

	print_storage("before redistribute","REDIST_storage");
	start_clock("redistribute");

#if defined(USE_OVERTURE)
        if (!attach_sink_nodes(fr))
        {
            (void) printf("WARNING - in redistribute2d(), "
                          "attach_sink_nodes() failed for front[%d]\n",
                           fr->patch_number);
            (void) reset_interface_of_front(fr,save_intfc);
            prepare_to_leave_redistribute2d(save_intfc);
            DEBUG_LEAVE(redistribute2d)
            return BAD_REDISTRIBUTION;
        }
#else  /* if defined(USE_OVERTURE) */
        if (!pp_min_status(attach_sink_nodes(fr)))
        {
            (void) printf("WARNING - in redistribute2d(), "
                          "attach_sink_nodes() failed\n");
            (void) reset_interface_of_front(fr,save_intfc);
            prepare_to_leave_redistribute2d(save_intfc);
            DEBUG_LEAVE(redistribute2d)
            return BAD_REDISTRIBUTION;
        }
#endif /* if defined(USE_OVERTURE) */

	DEBUG_FRONT("after attach_sink_nodes",fr)

	    /* Redistribute Interface */

	Interface_redistributed(fr) = NO;
	small_loops_deleted = NO;
	force_redistribute = NO;
	delete_small_loops_before_untangle = NO;

redistribute_interface:

	if (do_redist)
	{
	    bool status;

	    status = (intfc->curves == NULL) ? YES :
	    		(Interface_redistributed(fr)) ? YES :
	    		Curve_redistribute(fr,&force_redistribute);

	    status = closed_curve_node_redistribute(intfc,status);

#if defined(USE_OVERTURE)
            if (!status)
            {
                (void) printf("WARNING in redistribute2d(), "
                              "redistribution failed\n");
                (void) reset_interface_of_front(fr,save_intfc);
                prepare_to_leave_redistribute2d(save_intfc);
                DEBUG_LEAVE(redistribute2d)
                return BAD_REDISTRIBUTION;
            }
#else /* if defined(USE_OVERTURE) */
            if (!pp_min_status(status))
            {
                (void) printf("WARNING in redistribute2d(), "
                              "redistribution failed\n");
                (void) reset_interface_of_front(fr,save_intfc);
                prepare_to_leave_redistribute2d(save_intfc);
                DEBUG_LEAVE(redistribute2d)
                return BAD_REDISTRIBUTION;
            }
#endif /* if defined(USE_OVERTURE) */
	}

#if defined(USE_OVERTURE)
        any_redist = Interface_redistributed(fr);
#else /* if defined(USE_OVERTURE) */
        any_redist = pp_max_status(Interface_redistributed(fr));
#endif /* if defined(USE_OVERTURE) */

	if (any_redist)
	{
	    /* redistribute or untangle may produce 2 and 3 bond loops */
	    delete_small_loops(fr);
	    if (!restart_init)
	    {
	    	print_storage("before copy/delete intfc",
	    		      "REDIST_storage");
	    	start_clock("copy/delete intfc");
	    	set_size_of_intfc_state(size_of_state(fr->interf));
	    	set_copy_intfc_states(YES); /* Ensure states copied */
	    	fr->interf = copy_interface(intfc);
	    	(void) delete_interface(intfc);
	    	intfc = fr->interf;
	    	set_copy_intfc_states(YES);
	    	stop_clock("copy/delete intfc");
	    	print_storage("after copy/delete intfc","REDIST_storage");
	    }
	    do_scatter = YES;
	}

	DEBUG_FRONT("before intersections check",fr)

	    	/* Check for Intersections in Front */
	if (DEBUG) (void) printf("Checking for intersections\n");
	istatus = check_for_intersections(intfc,&cross,YES);
	if (!istatus)
	{
	    if (debugging("pionfail"))
	    	print_interface(intfc);
	    (void) reset_interface_of_front(fr,save_intfc);
	    prepare_to_leave_redistribute2d(save_intfc);
	    DEBUG_LEAVE(redistribute2d)
	    return BAD_REDISTRIBUTION;
	}

	if (DEBUG) (void) printf("Intersections check completed\n");

	if (interface_is_tangled(cross))
	{
	    static const char *redist_mesg =   "redistributed hyp ";
	    static const char *unredist_mesg = "unredistributed hyp ";
	    int	              num_tangles;

	    ++num_untangles;
	    if (DEBUG)
	    {
	    	(void) printf("Entering untangle block, "
	    	              "attempt number %d.\n",num_untangles);
	    }

	    num_tangles = print_number_of_tangles(
	    	(Interface_redistributed(fr)==YES)?redist_mesg:unredist_mesg,
		fr->interf,cross);

	    if (num_untangles > MAX_NUM_UNTANGLES)
	    {
	    	(void) printf("WARNING in redistribute2d(), too many attempts "
			      "(%d)  to untangle without success.\n",
			      num_untangles);
	    	if (DEBUG && (cross != NULL))
	    	    print_intersections(cross,intfc);
	    	(void) reset_interface_of_front(fr,save_intfc);
	    	prepare_to_leave_redistribute2d(save_intfc);
	    	DEBUG_LEAVE(redistribute2d)
	    	return BAD_REDISTRIBUTION;
	    }

	    if ((save_intfc[0] != NULL) && any_redist &&
	        vector_wave_interaction(cross))
	    {
	    	CROSS	   *ur_cross = NULL;
		CROSS      *cr;
		HYPER_SURF *hs;
	    	int	   unredist_num_tangles;

	    	/* Check to see if redistribute created vector
	    	 * tangles.  We do not allow this, as the
	    	 * untangle code may not be able to handle the
	    	 * resulting (possibly unphysical) configuration. */

	    	if (DEBUG)
	    	    (void) printf("Second intersections check\n");
	    	istatus = check_for_intersections(save_intfc[0],&ur_cross,YES);
	    	if (!istatus)
	    	{
	    	    (void) printf("Second intersections check "
	    	                  "detects bad status.\n");
	    	    if (debugging("pionfail"))
	    	    {
	    	        (void) printf("Unredistributed Interface:\n\n");
	    	        print_interface(save_intfc[0]);
	    	        (void) printf("Redistributed Interface:\n\n");
	    	        print_interface(intfc);
	    	    }
	    	    (void) reset_interface_of_front(fr,save_intfc);
	    	    prepare_to_leave_redistribute2d(save_intfc);
	    	    DEBUG_LEAVE(redistribute2d)
	    	    return BAD_REDISTRIBUTION;
	    	}
	    	if (DEBUG)
	    	    (void) printf("Second intersections check completed\n");
	        if (!interface_is_tangled(ur_cross))
		{
		    /*
		     * The unredistributed interface is untangled
		     * Omit redistribution on this time step
		     */
	    	    (void) reset_interface_of_front(fr,save_intfc);
		    intfc = fr->interf;
	            save_intfc[0] = make_save_intfc(intfc);
	            any_redist = NO;
		    cross = NULL;
		}
		else
		{
		    bool repeat_redistribute = NO;
	    	    unredist_num_tangles = print_number_of_tangles(
	    			unredist_mesg,save_intfc[0],ur_cross);
	    	    if (unredist_num_tangles != num_tangles)
	    	    {
			repeat_redistribute = YES;
	    		(void) printf("WARNING in redistribute2d(), "
	    		              "redistribute causes vector tangle.\n");
		        for (cr = cross; cr != NULL; cr = cr->next)
		        {
			    if (wave_type(cr->c1) >=
				FIRST_VECTOR_PHYSICS_WAVE_TYPE)
			    {
				hs = Hyper_surf(cr->c1);
			        do_not_redistribute(hs) = YES;
				hs = correspond_hyper_surf(hs);
				if (hs != NULL)
			            do_not_redistribute(hs) = YES;
				else
				    repeat_redistribute = NO;
			    }
			    if (wave_type(cr->c2) >=
				FIRST_VECTOR_PHYSICS_WAVE_TYPE)
			    {
				hs = Hyper_surf(cr->c2);
			        do_not_redistribute(cr->c2) = YES;
				hs = correspond_hyper_surf(hs);
				if (hs != NULL)
			            do_not_redistribute(hs) = YES;
				else
				    repeat_redistribute = NO;
			    }
		        }
		    }
		
#if defined(USE_OVERTURE)
                    if (repeat_redistribute)
                    {
                        (void) reset_interface_of_front(fr,save_intfc);
                        prepare_to_leave_redistribute2d(save_intfc);
                        status = redistribute2d(fr,do_redist,restart_init);
                        printf("Patch front[%d] repeat_redistribute\n",
                                   fr->patch_number);
                        DEBUG_LEAVE(redistribute2d)
                        return status;
                    }
#else /* if defined(USE_OVERTURE) */
                    if (pp_max_status(repeat_redistribute))
                    {
                        (void) reset_interface_of_front(fr,save_intfc);
                        prepare_to_leave_redistribute2d(save_intfc);
                        status = redistribute2d(fr,do_redist,restart_init);
                        DEBUG_LEAVE(redistribute2d)
                        return status;
                    }
#endif /* if defined(USE_OVERTURE) */
	    	}
	    }
	}
	if (interface_is_tangled(cross))
	{
	    if (restart_init) 
	    {
	    	(void) printf("WARNING in redistribute2d(), "
	    	              "Restart interface tangled, cannot continue\n");
	    	(void) reset_interface_of_front(fr,save_intfc);
	    	prepare_to_leave_redistribute2d(save_intfc);
	    	DEBUG_LEAVE(redistribute2d)
	    	return BAD_REDISTRIBUTION;
	    }

	    set_force_tangle(NO);

#if defined(COMPLICATED_CROSS_STRUCTURE)
	    if (too_many_tangled_points(cross,h,dim))
	    {
	    	(void) printf("WARNING in redistribute2d(), "
	    	              "Too many tangled points on interface\n");
	    	redist_status = (do_redist &&
	    			(Interface_redistributed(fr))) ?
	    		 UNABLE_TO_UNTANGLE : BAD_REDISTRIBUTION;
	    	(void) reset_interface_of_front(fr,save_intfc);
	    	prepare_to_leave_redistribute2d(save_intfc);
	    	DEBUG_LEAVE(redistribute2d)
	    	return redist_status;
	    }
#endif /* defined(COMPLICATED_CROSS_STRUCTURE) */

	    if (DEBUG)
	    {
	    	(void) printf("Checking to delete small ");
	    	(void) printf("loops before untangle\n");
	    }

#if defined(USE_OVERTURE)
            if (delete_small_loops_before_untangle)
            {
                eliminate_small_loops(intfc,hx,hy,&cross);
                small_loops_deleted = YES;
            }
#else /* if defined(USE_OVERTURE) */
            if (pp_max_status(delete_small_loops_before_untangle))
            {
                eliminate_small_loops(intfc,hx,hy,&cross);
                small_loops_deleted = YES;
            }
#endif /* if defined(USE_OVERTURE) */

	    /* Make backup copy of interface in case untangle fails */

	    save_intfc[1] = make_save_intfc(intfc);


	    	/* Boundary untangle */

	    if (fr->fr_bdry_untangle)
	    {
	        if (DEBUG)
	    	    (void) printf("Calling boundary untangle\n");
	        status = (cross == NULL) ? CURVES_UNTANGLED :
	    			           (*fr->fr_bdry_untangle)(fr,&cross,
	    		                                           NULL,NULL,
								   flag);
#if !defined(USE_OVERTURE)
	        status = synchronize_untangle_status(status);
#endif /* if !defined(USE_OVERTURE) */

	        switch (status)
	        {
	        case CURVES_UNTANGLED:
	    	    break;
	        case MODIFY_TIME_STEP_TO_UNTANGLE:
	    	    (void) printf("WARNING in redistributed2d, "
	    	                  "bdry_untangle returns \n"
	    	                  "\t\tMODIFY_TIME_STEP_TO_UNTANGLE\n");
	    	    (void) reset_interface_of_front(fr,save_intfc);
	    	    prepare_to_leave_redistribute2d(save_intfc);
	    	    DEBUG_LEAVE(redistribute2d)
	    	    return MODIFY_TIME_STEP_REDISTRIBUTE;
	        case MODIFY_TIME_STEP_TO_UNTANGLE_BUT_FORCE_TANGLE:
	    	    (void) printf("WARNING in redistributed2d, "
	    	                  "bdry_untangle returns \n"
	    	                  "\t\tMODIFY_TIME_STEP_TO_"
	    	                  "UNTANGLE_BUT_FORCE_TANGLE\n");
	    	    dt_frac = fr->dt_frac;
	    	    set_force_tangle(YES);
	    	    dt_max = dt = fr->dt;
	    	    min_dt_frac = dt_min/dt;
	    	    if (*dt_frac < min_dt_frac)
	    		*dt_frac = 0.5*(min_dt_frac + 1.0);
	    	    (void) reset_interface_of_front(fr,save_intfc);
	    	    prepare_to_leave_redistribute2d(save_intfc);
	    	    DEBUG_LEAVE(redistribute2d)
	    	    return MODIFY_TIME_STEP_REDISTRIBUTE;
	        case ERROR_IN_UNTANGLE:
	        default:
	    	    if (do_redist && 
	    	        (Curve_redistribution_function(fr) != NULL) && 
	    	        (!Interface_redistributed(fr)))
	    	    {
	    	        (void) printf("WARNING in redistribute2d(), "
	    	                      "unable to untangle front and"
	    	                      " boundary on first attempt\n"
	    	                      "Redistributing and trying again\n");
	    	        force_redistribute = YES;
	    	        intfc = reset_interface_of_front(fr,save_intfc+1);
	    	        goto redistribute_interface;
	    	    }
	    	    else if (!small_loops_deleted) 
	    	    {
	    	        (void) printf("WARNING in redistribute2d(), "
	    	                      "unable to untangle front and "
	    	                      "boundary\n"
	    	                      "Deleting small loops and "
	    	                      "trying again\n");
	    	        intfc = reset_interface_of_front(fr,save_intfc+1);
	    	        delete_small_loops_before_untangle = YES;
	    	        goto redistribute_interface;
	    	    }
	    	    (void) printf("WARNING in redistribute2d(), "
	    	                  "unable to untangle front and bdry\n");
	    	    (void) reset_interface_of_front(fr,save_intfc);
	    	    redist_status = (do_redist &&
	    			Interface_redistributed(fr)) ?
	    		UNABLE_TO_UNTANGLE : BAD_REDISTRIBUTION;
	    	    prepare_to_leave_redistribute2d(save_intfc);
	    	    DEBUG_LEAVE(redistribute2d)
	    	    return redist_status;
	        }
	        DEBUG_FRONT("after boundary untangle:",fr)
	    }

	    if (debugging("intersect"))
	    {
	        (void) printf("After fr_bdry_untangle.\n");
	        if (cross)
	        {
	           (void) printf("Printing interior intersections only:\n");
	           print_intersections(cross,intfc);
	        }
	        else
	           (void) printf("No interior intersections\n");
	    }
	    debug_bond_cross(cross,fr)

	/* Interior untangle */

	    if (interface_is_tangled(cross))
	    {
	        if (DEBUG)
	    	    (void) printf("Calling untangle front\n");
	        status = (cross == NULL) ? CURVES_UNTANGLED :
	    		(fr->untangle_front) ?
	                        (*fr->untangle_front)(fr,&cross,flag) :
	                        ERROR_IN_UNTANGLE;

#if !defined(USE_OVERTURE)
	        status = synchronize_untangle_status(status);
#endif /* if !defined(USE_OVERTURE) */

	        switch (status)
	        {
	        case CURVES_UNTANGLED:
	            break;
	        case MODIFY_TIME_STEP_TO_UNTANGLE:
	    	    (void) printf("WARNING in redistributed2d, "
	    	                  "untangle returns "
	    	                  "MODIFY_TIME_STEP_TO_UNTANGLE\n");
	    	    (void) reset_interface_of_front(fr,save_intfc);
	            prepare_to_leave_redistribute2d(save_intfc);
	    	    DEBUG_LEAVE(redistribute2d)
	            return MODIFY_TIME_STEP_REDISTRIBUTE;
	        case MODIFY_TIME_STEP_TO_UNTANGLE_BUT_FORCE_TANGLE:
	    	    (void) printf("WARNING in redistributed2d, "
	    	                  "untangle returns "
	    	                  "MODIFY_TIME_STEP_TO_UNTANGLE_");
	    	    (void) printf("BUT_FORCE_TANGLE\n");
	    	    set_force_tangle(YES);
	    	    dt_frac = fr->dt_frac;
	    	    dt_max = dt = fr->dt;
	    	    min_dt_frac = dt_min/dt;
	    	    if (*dt_frac < min_dt_frac)
	    		*dt_frac = 0.5*(min_dt_frac + 1.0);
	    	    (void) reset_interface_of_front(fr,save_intfc);
	    	    prepare_to_leave_redistribute2d(save_intfc);
	    	    DEBUG_LEAVE(redistribute2d)
	    	    return MODIFY_TIME_STEP_REDISTRIBUTE;
	        case ERROR_IN_UNTANGLE:
	    	    if (do_redist && 
	    	        (Curve_redistribution_function(fr) != NULL) && 
	    	        (!Interface_redistributed(fr)))
	            {
	                (void) printf("WARNING in redistribute2d(), "
	                              "unable to untangle interior"
	                              " crosses on first attempt\n");
	                (void) printf("Redistributing and trying again\n");
	                force_redistribute = YES;
	                flag = DIFFICULT_ATTEMPT_TO_UNTANGLE;
	    	        intfc = reset_interface_of_front(fr,save_intfc+1);
	                goto redistribute_interface;
	            }
	            else if (!small_loops_deleted) 
	            {
	                (void) printf("WARNING in redistribute2d(), "
	                              "unable to untangle interior crosses\n");
	    	        (void) printf("Deleting small loops and ");
	                (void) printf("trying again\n");
	                flag = LAST_ATTEMPT_TO_UNTANGLE;
	    	        intfc = reset_interface_of_front(fr,save_intfc+1);
	                delete_small_loops_before_untangle = YES;
	                goto redistribute_interface;
	            }
	            (void) printf("WARNING in redistribute2d(), "
	                          "unable to unravel the interface\n");
	    	    (void) reset_interface_of_front(fr,save_intfc);
	    	    redist_status = (do_redist &&
	    			Interface_redistributed(fr)) ?
	    		UNABLE_TO_UNTANGLE : BAD_REDISTRIBUTION;
	    	    prepare_to_leave_redistribute2d(save_intfc);
	    	    DEBUG_LEAVE(redistribute2d)
	    	    return redist_status;
	        }
	    }
	    else
	    {
	    	(void) delete_interface(save_intfc[1]);
	    	save_intfc[1] = NULL;
	    }
	    dt_min = 0.0;	dt_max = HUGE_VAL;

	    /* Untangle may produce zero length bonds */

	    if (DEBUG)
	    	(void) printf("Deleting very short bonds\n");
	    intfc_delete_very_short_bonds(fr);

#if defined(USE_OVERTURE)
            if (!intfc_delete_fold_back_bonds(fr))
            {
                (void) printf("WARNING in redistribute2d(), "
                     "unable to delete fold back bonds for front[%d]\n",fr->patch_number);
                (void) reset_interface_of_front(fr,save_intfc);
                prepare_to_leave_redistribute2d(save_intfc);
                DEBUG_LEAVE(redistribute2d)
                return BAD_REDISTRIBUTION;
            }
#else /* if defined(USE_OVERTURE) */
	    if (!pp_min_status(intfc_delete_fold_back_bonds(fr)))
	    {
	    	(void) printf("WARNING in redistribute2d(), "
	    	              "unable to delete fold back bonds\n");
	    	(void) reset_interface_of_front(fr,save_intfc);
	    	prepare_to_leave_redistribute2d(save_intfc);
	    	DEBUG_LEAVE(redistribute2d)
	    	return BAD_REDISTRIBUTION;
	    }
#endif /* if defined(USE_OVERTURE) */

	    if (DEBUG)
	    	(void) printf("Calling scatter front\n");

#if !defined(USE_OVERTURE)
	    if (!scatter_front(fr))
	    {
	    	(void) printf("WARNING in redistributed2d, "
	    	              "scatter_front() failed, "
	    	              "MODIFY_TIME_STEP_TO_UNTANGLE\n");
	    	(void) reset_interface_of_front(fr,save_intfc);
	    	prepare_to_leave_redistribute2d(save_intfc);
	    	dt_frac = fr->dt_frac;
	    	*dt_frac = Max_time_step_modification_factor(fr);
	    	DEBUG_LEAVE(redistribute2d)
	    	return MODIFY_TIME_STEP_REDISTRIBUTE;
	    }
#endif /* defined(USE_OVERTURE) */
	    do_scatter = NO;
	    DEBUG_FRONT("after interior untangle:",fr)
	    goto redistribute_interface;
	}
	else if (force_tangle() && fr->dt > 0.0)
	{
	    dt_frac = fr->dt_frac;
	    dt = fr->dt;
	    dt_min = dt;
	    min_dt_frac = 0.5*(dt_max + dt)/dt;
	    *dt_frac = max(*dt_frac,min_dt_frac);
	    (void) reset_interface_of_front(fr,save_intfc);
	    prepare_to_leave_redistribute2d(save_intfc);
	    DEBUG_LEAVE(redistribute2d)
	    return MODIFY_TIME_STEP_REDISTRIBUTE;
	}

#if defined(USE_OVERTURE)
        if (!correct_for_exterior_curves(fr))
        {
            (void) printf("WARNING in redistribute2d(), "
                          "unable to correct for exterior curves\n");
            (void) reset_interface_of_front(fr,save_intfc);
            prepare_to_leave_redistribute2d(save_intfc);
            DEBUG_LEAVE(redistribute2d)
            return BAD_REDISTRIBUTION;
        }
#else /* if defined(USE_OVERTURE) */
	if (!pp_min_status(correct_for_exterior_curves(fr)))
	{
	    (void) printf("WARNING in redistribute2d(), "
	                  "unable to correct for exterior curves\n");
	    (void) reset_interface_of_front(fr,save_intfc);
	    prepare_to_leave_redistribute2d(save_intfc);
	    DEBUG_LEAVE(redistribute2d)
	    return BAD_REDISTRIBUTION;
	}
#endif /* if defined(USE_OVERTURE) */

	DEBUG_FRONT("after correction for exterior curves",fr)

#if defined(USE_OVERTURE)
        if (do_scatter)
#else /* if defined(USE_OVERTURE) */
        if (pp_max_status(do_scatter))
#endif /* if defined(USE_OVERTURE) */
	{
#if !defined(USE_OVERTURE)
	    if (!scatter_front(fr))
	    {
	    	(void) printf("WARNING in redistributed2d, "
	    	              "scatter_front() failed, "
	    	              "\t\tMODIFY_TIME_STEP_TO_UNTANGLE\n");
	    	(void) reset_interface_of_front(fr,save_intfc);
	    	prepare_to_leave_redistribute2d(save_intfc);
	    	dt_frac = fr->dt_frac;
	    	*dt_frac = Max_time_step_modification_factor(fr);
	    	DEBUG_LEAVE(redistribute2d)
	    	return MODIFY_TIME_STEP_REDISTRIBUTE;
	    }
#else /* if !defined(USE_OVERTURE) */
            if(clip_patch_front(fr,NO) == FUNCTION_FAILED)
            {
                printf("WARNING redistributed2d(),"
                   " clip_patch_front() for frs[%s] level[%d]failed \n",
                   fr->patch_number, fr->patch_level);
                (void) reset_interface_of_front(fr,save_intfc);
                prepare_to_leave_redistribute2d(save_intfc);
                dt_frac = fr->dt_frac;
                *dt_frac = Max_time_step_modification_factor(fr);
                DEBUG_LEAVE(redistribute2d)
                return MODIFY_TIME_STEP_REDISTRIBUTE;
            }
#endif /* if !defined(USE_OVERTURE) */
	}
	else
	{
	    long	num_inconsistent_nodes;

	    /* scatter_front() includes a component check, so this
	     * is not needed in the if block */

	    num_inconsistent_nodes = check_comps_at_nodes(fr->interf,
	    					          &onode_list);
#if !defined(USE_OVERTURE)
	    pp_global_lmax(&num_inconsistent_nodes,1L);
#endif /* if !defined(USE_OVERTURE) */

	    if (num_inconsistent_nodes != 0)
	    {
	        /* TODO: insert function to handle inconsistent comps
	         * For now this will result in a repeat of this step
	         * with dt halved. */

                printf("do scatter = %d\n", do_scatter); 
	        (void) printf("WARNING in redistribute2d(), "
	                      "Inconsistent comps found at %ld nodes\n",
	    		      num_inconsistent_nodes);
	        if (debugging("inconsistent"))
	        {
	    	    print_onode_list(&onode_list);
	    	    print_interface(fr->interf);
	        }
	        (void) reset_interface_of_front(fr,save_intfc);
	        prepare_to_leave_redistribute2d(save_intfc);
	        DEBUG_LEAVE(redistribute2d)
	        return BAD_REDISTRIBUTION;
	    }
	}

	DEBUG_FRONT("after redistribute:",fr)
	prepare_to_leave_redistribute2d(save_intfc);
	DEBUG_LEAVE(redistribute2d)
	return GOOD_REDISTRIBUTION;
}		/*end redistribute2d*/

LOCAL	bool check_for_intersections(
	INTERFACE *intfc,
	CROSS     **cross,
	const bool   bdry)
{
	CROSS *cr;
	bool istatus;

	istatus = intersections(intfc,cross,bdry);
#if defined(USE_OVERTURE)
	if (!istatus)
	{
	    (void) printf("WARNING in check_for_intersections(), "
	                  "intersections() failed");
	    if (pp_numnodes() > 1)
		(void) printf(" on node %d\n",pp_mynode());
	    else
		(void) printf("\n");
            return FUNCTION_FAILED;
	}
#else /* if defined(USE_OVERTURE) */
	if (!istatus)
	{
	    (void) printf("WARNING in check_for_intersections(), "
	                  "intersections() failed");
	    if (pp_numnodes() > 1)
		(void) printf(" on node %d\n",pp_mynode());
	    else
		(void) printf("\n");
	}
	if (!pp_min_status(istatus))
	{
	    if (istatus)
	    {
	        (void) printf("WARNING in redistribute2d(), "
	                      "intersections() failed on remote node\n");
	    }
	    return FUNCTION_FAILED;
	}
#endif /* if defined(USE_OVERTURE) */

	if (*cross != NULL)
	{
	    for (cr = *cross; cr; cr = cr->next)
	    {
	        if (is_passive_boundary(cr->c1) || is_passive_boundary(cr->c2))
	        {
		    if (*cross == cr)
		        *cross = cr->next;
	    	    delete_from_cross_list(cr);
	        }
	    }
	}
	return FUNCTION_SUCCEEDED;
}		/*end check_for_intersections*/

LIB_LOCAL INTERFACE *reset_interface_of_front(
	Front		*fr,
	INTERFACE	**save_intfc)
{
	if (save_intfc == NULL || *save_intfc == NULL)
	    return fr->interf;
	(void) delete_interface(fr->interf);
	fr->interf = *save_intfc;
	set_current_interface(fr->interf);
	*save_intfc = NULL;
	return fr->interf;
}		/*end reset_interface_of_front*/

LOCAL	void prepare_to_leave_redistribute2d(
	INTERFACE	**save_intfc)
{
	if (save_intfc[0] != NULL)
	    (void) delete_interface(save_intfc[0]);
	if (save_intfc[1] != NULL)
	    (void) delete_interface(save_intfc[1]);
	stop_clock("redistribute");
	print_storage("after redistribute","REDIST_storage");
}		/*end prepare_to_leave_redistribute2d*/


/*
*			make_save_intfc():
*
*	Creates a backup copy of intfc and leaves the current interface 
*	unaltered.
*/

LIB_LOCAL INTERFACE *make_save_intfc(
	INTERFACE	*intfc)
{
	INTERFACE	*sav_intfc, *cur_intfc;
	bool		sav_interp;
	
	print_storage("before make save intfc","REDIST_storage");
	start_clock("make save intfc");
	cur_intfc = current_interface();
	sav_interp = interpolate_intfc_states(intfc);
	set_add_to_correspond_list(YES);
	set_size_of_intfc_state(size_of_state(intfc));
	set_copy_intfc_states(YES); /* Ensure states copied */
	if ((sav_intfc = copy_interface(intfc)) == NULL) 
	{
	    set_current_interface(cur_intfc);
	    return NULL;
	}
	interpolate_intfc_states(sav_intfc) = sav_interp;
	set_current_interface(cur_intfc);
	stop_clock("make save intfc");
	print_storage("after make save intfc","REDIST_storage");
	return sav_intfc;
}		/*end make_save_intfc*/

/*
*			expansion_redistribute():
*
*	Defines a redistribution of an interface which never drops
*	points.   New points are added to keep the maximum front
*	spacing below the quantity space.
*/

EXPORT bool expansion_redistribute(
	Front		*fr,
	bool		*force_redistribute)
{
	INTERFACE	*interf = fr->interf;
	CURVE		**c;

	DEBUG_ENTER(expansion_redistribute)

	for (c = interf->curves; *c; ++c) 
	{
	    if (omit_redistribution(*c))
		continue;
	    if (!expand_redist_cur(fr,*c))
	    {
	    	Interface_redistributed(fr) = NO;
	    	DEBUG_LEAVE(expansion_redistribute)
	    	return NO;
	    }
	}

	Interface_redistributed(fr) = YES;
	*force_redistribute = NO;
	DEBUG_LEAVE(expansion_redistribute)
	return YES;
}		/*end expansion_redistribute*/


/*
*			expand_redist_cur():
*
*	This routine continually bisects the bonds in the curve c until
*	they have (scaled) length less than
*	spacing = max(space, 2*MIN_SC_SEP(fr->interf)),
*	where space is equal to the front spacing factor divided by
*	(1 + kbar*hypot(hx,hy)) and kbar is the average of the curvature
*	at the start and end of the bond.
*	(If there are no roundoff errors,  2^N - 1 (for some N)
*	equally spaced points are added to each old bond.)
*	The resulting new bonds will have (scaled) lengths >= 0.5*spacing.
*
*	Bonds are deleted only when the scaled bond length is <
*	MIN_SC_SEP(fr->interf).
*
*	After expansion, with one exception, all bonds on the curve will 
*	have (scaled) lengths between MIN_SC_SEP(fr->interf) and "spacing".
*	The exception is that a very short curve (total scaled length
*	< MIN_SC_SEP(fr->interf)) will end up as a single very short bond.
*	(A WARNING is written in this case.)
*
*/

/*ARGSUSED*/
EXPORT bool expand_redist_cur(
	Front		*fr,
	CURVE		*c)
{
	BOND		*b, *bprev;
	RECT_GRID	*gr = fr->rect_grid;
	float		space, spacing;
	float		*h = gr->h;
	float		distance;	/* scaled bond lengths on curve */
	float		coords[MAXD];	/* coordinates of points added */
	int		i, dim = gr->dim;
	int		 pts_added;	/* number of points added to a bond */

	DEBUG_ENTER(expand_redist_cur)
	if (DEBUG)
	{
	    (void) printf("Before adding points in expand_redist_cur()\n");
	    print_curve(c);
	}

	    /* skip passive boundary curves */

	if (wave_type(c) == PASSIVE_BOUNDARY)
	    goto Exit;

	space = Front_spacing(fr,
	    	    (wave_type(c) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE) ?
	    		VECTOR_WAVE : GENERAL_WAVE);

	    /* guarantee that no bonds shorter than MIN_SC_SEP(fr->interf)
	    *  will be added
	    */

	if (space < 2.0*MIN_SC_SEP(fr->interf)) 
	{
	    (void) printf("WARNING in expand_redist_cur(), "
	                  "desired spacing (%g) too small, "
	                  "using 2.0*MIN_SC_SEP(fr->interf)\n",space);
	    space = 2.0*MIN_SC_SEP(fr->interf);
	}

	    /*  loop over bonds in the curve */ 

	b = c->first;
	bprev = b->prev;
	(void) redistribution_spacing(NULL,NULL,0.0,NULL);
	while (b != NULL) 
	{
	    distance = scaled_bond_length(b,h,dim);
	    if (DEBUG) 
	    {
	    	print_bond(b);
	    	(void) printf("distance %g, space %g\n",distance,space);
	    }

	    	/* delete very short bonds */
	
	    if (distance < MIN_SC_SEP(fr->interf)) 
	    {
	    	if (c->num_points == 2) 
	    	{
	    	    (void) printf("WARNING in expand_redist_cur(), "
	    	                  "very short curve\n");
	    	    print_curve(c);
	    	    goto Exit;
	    	}
	    	if (b == c->last) 
	    	{
	    	    b = b->prev;
	    	    bprev = b->prev;
	    	}
	    	if (DEBUG)
	    	    (void) printf("deleting point (x,y) = (%g,%g)\n",
	    			  Coords(b->end)[0],Coords(b->end)[1]);
	    	(void) delete_start_of_bond(b->next,c);
	    	continue;
	    }

	    	/* Divide long bond into n equally spaced bonds */

	    spacing = redistribution_spacing(b,c,space,fr);
	    if (distance > spacing)
	    {
	    	float t;

	    	pts_added = (int) (distance/spacing);
	    	while (pts_added > 0)
	    	{
	    	    t = 1.0/((float)(pts_added-- + 1));
	    	    for (i = 0; i < dim; ++i)
	    		coords[i] = (1.0 - t)*Coords(b->start)[i]
	    				 + t *Coords(b->end)[i];
	    	    if (insert_point_in_bond(Point(coords),b,c) !=
			FUNCTION_SUCCEEDED)
	    	    {
	    	        screen("ERROR in expand_redist_cur(), "
	    	               "insert_point_in_bond failed\n");
	    	        clean_up(ERROR);
	    	    }
	    	    b = b->next;
	    	}
	    }
	    bprev = b;
	    b = b->next;
	}

	if (bprev != c->last) 
	{
	    (void) printf("WARNING in expand_redist_cur(), "
	                  "loop over bonds incomplete\n");
	    DEBUG_LEAVE(expand_redist_cur)
	    return NO;
	}

Exit:
	if (DEBUG)
	{
	    (void) printf("After adding points in expand_redist_cur()\n");
	    print_curve(c);
	}
	DEBUG_LEAVE(expand_redist_cur)
	return YES;
}		/*end expand_redist_cur*/

LOCAL	float	redistribution_spacing(
	BOND	*b,
	CURVE	*c,
	float	space,
	Front	*fr)
{
	static	float	kstart;
	static	float	kend = -1.0;
	float	kappa;
	float	hx, hy;

	if (b == NULL || c == NULL || fr == NULL)
	{
	    kend = -1.0;
	    return -1.0;
	}
	hx = fr->rect_grid->h[0];
	hy = fr->rect_grid->h[1];
	kstart = (kend >= 0.0) ? kend :
		fabs(mean_curvature_at_point(b->start,
				     Hyper_surf_element(b),Hyper_surf(c),fr));

	kend = fabs(mean_curvature_at_point(b->end,
				     Hyper_surf_element(b),Hyper_surf(c),fr));

	kappa = 0.5*(kstart+kend)*hypot(hx,hy);
	return space/(1.0 + kappa);
}		/*end redistribution_spacing*/


EXPORT bool full_redistribute(
	Front		*fr,
	bool		*force_redist)
{
	CURVE	**c;
	bool	status = YES;
	bool	force;
	int	redist_nodes;
#if defined(USE_OVERTURE)
 #if defined(SMOOTH_INTERFACE_FOR_SCATTER)
	NODE    **nodes, **tmpn; 
        int     alloc_num, use_alloc = 0; 
        int     n_in, n_out, n_total;
        bool    sav_interp;
        CURVE   *newc;
 #endif /* if defined(SMOOTH_INTERFACE_FOR_SCATTER) */
#endif /* if defined(USE_OVERTURE) */ 

	DEBUG_ENTER(full_redistribute)
	if (DEBUG)
	{
	    (void) printf("Unredistributed interface\n");
	    (void) printf("*force_redist = %s\n",y_or_n(*force_redist));
	    print_interface(fr->interf);
	}

	    /* Check on redistribution conditions */

	force = *force_redist;
	*force_redist = NO; /*set force_redist flag back to NO*/
	if (force)
	    redist_nodes = YES;
	else if (Redistribution_count(fr) < 0)
	    redist_nodes = NO;
	else
	    redist_nodes = redist_needed(fr,GENERAL_NODE);

	    /* Redistribute rect boundary curves */

	if (Use_rect_boundary_redistribution(fr))
	    rect_boundary_redistribute(fr->interf,fr->rect_grid,fr->step,fr);

#if defined(USE_OVERTURE)
 #if defined(SMOOTH_INTERFACE_FOR_SCATTER)
        if(fr->rect_grid->dim == 2)
            alloc_num = max((int)((fr->rect_grid->gmax[0]+fr->rect_grid->gmax[1])/2), 5);
        else
        {
            printf("ERROR use attach_nodes_on_bdry\n");
            printf("NO implimentation for %D\n", fr->rect_grid->dim);
            clean_up(ERROR);
        }
        vector(&nodes, alloc_num, sizeof(NODE*)); 
        attach_nodes_on_bdry(fr, nodes, &alloc_num, &use_alloc); 
 #endif /* if defined(SMOOTH_INTERFACE_FOR_SCATTER) */
#endif /* if defined(USE_OVERTURE) */

	    /* Redistribute vector curves */

	for (c = fr->interf->curves; *c ; ++c) 
	{
	    if (perform_redistribution(*c,fr,force))
	    {
		switch (redistribution_direction(*c))
		{
		case FORWARD_REDISTRIBUTION:
	            status = Forward_curve_redistribute(fr,*c,status);
		    if (status)
		    {
		        redistributed(*c) = YES;
			redistribution_direction(*c) = BACKWARD_REDISTRIBUTION;
		    }
		    break;
		case BACKWARD_REDISTRIBUTION:
		    status = Backward_curve_redistribute(fr,*c,status);
		    if (status)
		    {
		        redistributed(*c) = YES;
			redistribution_direction(*c) = FORWARD_REDISTRIBUTION;
		    }
		    break;
		}
	    }
	}
	Redistribution_count(fr)++;

#if defined(USE_OVERTURE)
 #if defined(SMOOTH_INTERFACE_FOR_SCATTER)
        rm_attached_nodes_on_bdry(fr, nodes, &alloc_num, &use_alloc); 
        free(nodes);  
 #endif /* if defined(SMOOTH_INTERFACE_FOR_SCATTER) */
#endif /* if defined(USE_OVERTURE) */

		/* Redistribute non-vector curves */

	if (redist_nodes)
	    status = Node_redistribute(fr,status);

	for (c = fr->interf->curves; *c ; ++c) 
	    if (redistributed(*c))
	        Interface_redistributed(fr) = YES;

	if (debugging("redistribute"))
	{
	    if (Interface_redistributed(fr))
	    {
	    	(void) printf("Redistributed interface\n");
	    	print_interface(fr->interf);
	    }
	    else
	    	(void) printf("Interface left unredistributed\n");
	}
	if (debugging("ck_b_len"))
	{
	    BOND *bb;
	    CURVE *cc;
	    float len;

	    (void) printf("Checking bond lengths after redistribution\n");
	    (void) next_bond(fr->interf,NULL,NULL);
	    while (next_bond(fr->interf,&bb,&cc))
	    {
	    	print_bond(bb);
	    	len = separation(bb->start,bb->end,fr->rect_grid->dim);
	    	(void) printf("Computed length = %g, "
	    	              "Computed length - bond_length = %g\n\n",
		              len,len - bond_length(bb));
	    }
	}
	DEBUG_LEAVE(full_redistribute)
	return status;
}		/*end full_redistribute*/

EXPORT bool full_inc_redist_cur(
	Front		*fr,
	CURVE		*c,
	bool		status)
{
	float		space, spacing;
	float		cos_big_angle;
	RECT_GRID	*gr = fr->rect_grid;
	BOND		*b;
	BOND		*next_delete;	/* bond whose starting vertex is to be
					 * deleted next. This variable allows
					 * insertions to be done before 
					 * deletions, ie out of the order set 
					 * by order of points along curve */
	float		*h = gr->h;
	float		distance;   /* Distance between pts of fr */
	float		offset;	    /* Next newfr pt at offset from fr pt */
	float		slopex;	    /* Unit vector, joining pair of fr pts */
	float		slopey;
	float		cos_angle;  /* sin, cos of exterior angle */
	float		sin_angle;
	float		scaled_length;  /* scaling for mismatched h */
	float		p[MAXD];	/* Loop starts from p */
	float		coords[MAXD];
	int		i, dim = gr->dim;

	DEBUG_ENTER(full_inc_redist_cur)

		/* Conditionally skip boundary curves */
	if (wave_type(c) == PASSIVE_BOUNDARY)
	{
	    DEBUG_LEAVE(full_inc_redist_cur)
	    return status;
	}

	if (wave_type(c) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE)
	{
	    space = Front_spacing(fr,VECTOR_WAVE);
	    cos_big_angle = Cosine_big_angle(fr,VECTOR_WAVE);
	}
	else
	{
	    space = Front_spacing(fr,GENERAL_WAVE);
	    cos_big_angle = Cosine_big_angle(fr,GENERAL_WAVE);
	}

	    /* Skip short non-boundary curves with few points */

	if (!is_bdry_like_curve(c) && c->num_points <= 5) 
	{
	    scaled_length = 0.;
	    for (b = c->first; b; b = b->next)
	    	scaled_length += scaled_bond_length(b,h,dim);
	    if (scaled_length < 1.5 * space * (c->num_points -1))
	    {
	    	DEBUG_LEAVE(full_inc_redist_cur)
	    	return status;
	    }
	}

	    /* Set initial parameters for curve */

	offset = 0.5*space;
	for (i = 0; i < dim; ++i)
	    p[i] = Coords(c->first->start)[i];
	(void) redistribution_spacing(NULL,NULL,0.0,NULL);
	b = c->first;
	next_delete = NULL;

	    /* Process the next bond */

Loop:

	if (b == NULL) 
	{
	    status = NO;
	    (void) printf("WARNING in full_inc_redist_cur(), NULL BOND\n");
	    DEBUG_LEAVE(full_inc_redist_cur)
	    return status;
	}

	distance = _scaled_separation(Coords(b->end),p,h,dim);
	spacing = redistribution_spacing(b,c,space,fr);


	    /* Delete small bonds */

	if (b != c->last) 
	{
	    float	next_scaled_length;

	    if (distance < 0.1 * spacing)
	    {
	    	if (next_delete != NULL)
	    	    (void) delete_start_of_bond(next_delete,c);
		next_delete = b->next;
		b = b->next;
		goto Loop;
	    }
	    next_scaled_length = scaled_bond_length(b->next,h,dim);
	    if (next_scaled_length < 0.1 * spacing)
	    {
	    	if (next_delete != NULL)
	    	    (void) delete_start_of_bond(next_delete,c);
		if (next_delete != b->next)
		    (void) delete_start_of_bond(b->next,c);
		if (next_delete == b)
		    b = b->prev;
		next_delete = NULL;
		goto Loop;
	    }
	    big_angle(b,c,b->next,c,&cos_angle,&sin_angle,gr);
	}

	    /* Main loop for inserting points in a single bond */

	slopex = (Coords(b->end)[0] - p[0]) / distance;
	slopey = (Coords(b->end)[1] - p[1]) / distance;

	for ( ; offset <= distance; offset += spacing,b = b->next) 
	{
	    coords[0] = p[0] + slopex*offset;
	    coords[1] = p[1] + slopey*offset;
	    if (insert_point_in_bond(Point(coords),b,c) !=
		FUNCTION_SUCCEEDED)
	    {
		status = NO;
	    	screen("ERROR in full_inc_redist_cur(), "
	    	       "insert_point_in_bond failed\n");
	    	clean_up(ERROR);
		return status;
	    }
	}

	for (i = 0; i < dim; ++i)
	    p[i] = Coords(b->end)[i];
	offset -= distance;
	

		/* End of Curve */
	if (b == c->last) 
	{
	    scaled_length = scaled_bond_length(b,h,dim);

	    	/* Remove Short or Long Segment */

	    if (scaled_length <= 0.1 * spacing)
	    {
	    	(void) delete_start_of_bond(b,c);
	    	if (next_delete != NULL && next_delete != b)
	    	    (void) delete_start_of_bond(next_delete,c);
	    }
	    else if (next_delete != NULL)
	    	(void) delete_start_of_bond(next_delete,c);

	    b = c->last;
	    scaled_length = scaled_bond_length(b,h,dim);
	    if (scaled_length >= 0.5 * spacing)
	    {
	    	coords[0] = 0.5 * (Coords(b->start)[0] + Coords(b->end)[0]);
		coords[1] = 0.5 * (Coords(b->start)[1] + Coords(b->end)[1]);
		if (insert_point_in_bond(Point(coords),b,c) !=
		    FUNCTION_SUCCEEDED)
		{
		    status = NO;
		    screen("ERROR in full_inc_redist_cur(), "
		           "insert_point_in_bond failed\n");
		    clean_up(ERROR);
		    return status;
		}
	    }
	    DEBUG_LEAVE(full_inc_redist_cur)
	    return status;
	}

	/* Delete original vertex at end of b provided ... */

	scaled_length = scaled_bond_length(b,h,dim);

	if (
	    /* Initial segment of redist. b->next will be small */

	    (offset <= 0.1 * spacing)

	    /* b, ie final segment of redist. b, is small */

	    || (scaled_length <= 0.1 * spacing)

	    /* Exterior angle at vertex at end of b is small */

	    || (cos_angle >= cos_big_angle)
	) 
	{
	    if (next_delete != NULL)
	    	(void) delete_start_of_bond(next_delete,c);
	    next_delete = b->next;
	    b = b->next;
	}

	/* Else increment b; truncate if sharp bend at vertex */

	else 
	{

	    offset = scaled_length;

	    if (cos_angle <= -0.3) 
	    {
	    	coords[0] = 0.9*Coords(b->end)[0] + 0.1*Coords(b->start)[0];
		coords[1] = 0.9*Coords(b->end)[1] + 0.1*Coords(b->start)[1];
		if (insert_point_in_bond(Point(coords),b,c) !=
		    FUNCTION_SUCCEEDED)
		{
		    status = NO;
		    screen("ERROR in full_inc_redist_cur(), "
		           "insert_point_in_bond failed\n");
		    clean_up(ERROR);
		    return status;
		}
		b = b->next;
		(void) delete_start_of_bond(b->next,c);
		coords[0] = 0.1*Coords(b->end)[0] + 0.9*p[0];
		coords[1] = 0.1*Coords(b->end)[1] + 0.9*p[1];
		if (insert_point_in_bond(Point(coords),b,c) !=
		    FUNCTION_SUCCEEDED)
		{
		    status = NO;
		    screen("ERROR in full_inc_redist_cur(), "
		           "insert_point_in_bond failed\n");
		    clean_up(ERROR);
		    return status;
		}
	    }
	    b = b->next;
	}

	goto Loop;
}		/*end full_inc_redist_cur*/


EXPORT bool full_dec_redist_cur(
	Front		*fr,
	CURVE		*c,
	bool		status)
{
	invert_curve(c);
	status = full_inc_redist_cur(fr,c,status);
	invert_curve(c);
	return status;
}		/*end full_dec_redist_cur*/

EXPORT	bool backward_equi_curve_redistribute(
	Front		*fr,
	CURVE		*c,
	bool		status)
{
	invert_curve(c);
	status = equi_curve_redistribute(fr,c,status);
	invert_curve(c);
	return status;
}		/*end backward_equi_curve_redistribute*/

EXPORT	bool equi_curve_redistribute(
	Front		*fr,
	CURVE		*c,
	bool		status)
{
	BOND		*b;
	float		fr_space;
	RECT_GRID	*rgr = fr->rect_grid;
	float		*h = rgr->h;
	float		c_len;
	int		dim = rgr->dim;
	int		nbds;

	DEBUG_ENTER(equi_curve_redistribute)
		/* Conditionally skip boundary curves */

	if (wave_type(c) == PASSIVE_BOUNDARY && (c->num_points > 2))
	{
	    DEBUG_LEAVE(equi_curve_redistribute)
	    return status;
	}
	if (is_subdomain_boundary(Hyper_surf(c)))
	{
	    DEBUG_LEAVE(equi_curve_redistribute)
	    return YES;
	}

	fr_space = Front_spacing(fr,
			 (wave_type(c) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE) ?
				VECTOR_WAVE : GENERAL_WAVE);

	if (DEBUG)
	{
	    (void) printf("curve - ");
	    print_curve(c);
	}


	    /* Compute curve_length */

	c_len = 0.0;
	for (b = c->first;  b;  b = b->next)
	{
	    c_len += scaled_bond_length(b,h,dim);
	}

	    /* For short, non-boundary curves with few points - each */
	    /* such curve is equi-distributed, however the number   */
	    /* of points currently on the curve remains unchanged   */

	nbds = c->num_points - 1;
	if (DEBUG)
	    (void) printf("nbds %d\n",nbds);
	if (wave_type(c) >= FIRST_PHYSICS_WAVE_TYPE && 
			is_short_curve(c,POSITIVE_ORIENTATION,rgr,1.5))
	{
	    (void) expand_redist_cur(fr,c);
	    DEBUG_LEAVE(equi_curve_redistribute)
	    return status;
	}
	else if ((!is_bdry_like_curve(c)) && (nbds <= 4))
	{
	    if (c_len < 1.5 * fr_space * nbds)
	    {
		if (DEBUG)
		    (void) printf("c_len %g < tol %g",c_len,1.5*fr_space*nbds);
		if (nbds > 1)
		{
		    if (DEBUG)
			(void) printf("nbds > 1\n");
		    equi_redist_curve_seg(c,c->first,c->last,nbds,
					  c_len,fr_space,rgr);
		    if (DEBUG)
		    {
		    	(void) printf("After equi_redist_curve_seg:\n");
		    	(void) printf("curve - ");
			print_curve(c);
		    }
		}
		DEBUG_LEAVE(equi_curve_redistribute)
		return status;
	    }
	}

	    /* Pass 1, delete very short bonds */

	curve_delete_very_short_bonds(c);
	if (DEBUG)
	{
	    (void) printf("After delete_very_short_bonds:\ncurve - ");
	    print_curve(c);
	}

	    /* Can't redistribute single bond, short_curves */

	if ((c->first == c->last) && (c_len < fr_space))
	{
	    if (DEBUG)
		(void) printf("single_bond, short curve\n");
	    DEBUG_LEAVE(equi_curve_redistribute)
	    return status;
	}

	equi_redist_curve_seg(c,c->first,c->last,-1,c_len,fr_space,rgr);
	if (DEBUG)
	{
	    (void) printf("After equi_redist_curve_seg:\ncurve - ");
	    print_curve(c);
	}
	DEBUG_LEAVE(equi_curve_redistribute)
	return status;
}		/*end equi_curve_redistribute*/


/*
*			replace_curve_seg_by_bond():
*
*	Replaces the curve segment from the start of bond bs to the end
*	of bond be by a single bond. The resultant single bond has the
*	same pointer value as bs.
*/

LOCAL   void replace_curve_seg_by_bond(
	CURVE		*c,
	BOND		*bs,
	BOND		*be)
{
	BOND		*bnext;

	if ((bs == NULL) || (be == NULL))
	    return;
	if (bs == be)
	    return;
	bnext = be->next;
	while (bs->next != bnext)
		(void) delete_start_of_bond(bs->next,c);
}		/*end replace_curve_seg_by_bond*/


/*
*			attach_sink_nodes():
*
*	Attaches sink nodes to interface when interface gets sufficiently
*	close. Note this cannot be done in point_propagate routines as
*	it modifies the interface.
*/

LOCAL	bool attach_sink_nodes(
	Front		*fr)
{
	CURVE		**cc;
	INTERFACE	*intfc = fr->interf;
	NODE		**nn, *m, *n, **sink_node_list;
	BOND		*b;
	float		min_sc_sep = MIN_SC_SEP(fr->interf);
	float		*h = computational_grid(intfc)->h;
	int		dim = intfc->dim;
	int		flag;
	int		i, j;

	flag = 0;

	for (nn = intfc->nodes, i = 0;  *nn;  ++nn)
	{
	    if (node_type(*nn) == SINK_NODE) ++i;
	}
	if (i == 0)
	    return YES;

	vector(&sink_node_list,i,sizeof(NODE *));

	for (nn = intfc->nodes, i = 0;  *nn;  ++nn)
	{
	    if (node_type(*nn) == SINK_NODE)
	    {
	    	sink_node_list[i] = *nn;
	    	++i;
	    }
	}

tmprst:
	for (cc = intfc->curves;  cc && *cc;  ++cc)
	{
	    for (b = (*cc)->first;  b != (*cc)->last;  b = b->next)
	    {
		for (j = 0;  j < i;  ++j)
		{
		    m = sink_node_list[j];
		    if (scaled_separation(m->posn,b->end,h,dim) <= min_sc_sep)
		    {
		       (void) attach_curve_to_node(*cc,b->end,b,m);
		       node_type(m) = SINK_NODE;
		       flag = 1;
		       goto tmprst;
		    }
		}
	    }

			/* Check for breakthrough at closed nodes */

	    if (!is_closed_node((*cc)->start)) continue;
	    for (j = 0;  j < i;  ++j)
	    {
		m = sink_node_list[j];
		n = (*cc)->start;
		if (scaled_separation(m->posn,n->posn,h,dim) <= min_sc_sep)
		{
		    merge_and_delete_nodes(m,n);
		    node_type(m) = SINK_NODE;
		    flag = 1;
		    goto tmprst;
		}
	    }
	}
			/* eliminate redundant CLOSED_NODES */
	if(flag == 1)
	{
	    intfc_delete_very_short_bonds(fr);
	    if (!join_curves_at_closed_nodes(intfc))
	    {
	    	(void) printf("WARNING in attach_sink_nodes(), "
	    	              "can't join closed curves\n");
	    	return NO;
	    }
	}
	free(sink_node_list);
	return YES;
}		/*end attach_sink_nodes*/

EXPORT int join_curves_at_closed_nodes(
	INTERFACE	*intfc)
{
	NODE		**nn;
	CURVE		*c1, *c2;
	bool		sav_interp;

	sav_interp = interpolate_intfc_states(intfc);
	interpolate_intfc_states(intfc) = YES;
	for(nn = intfc->nodes; *nn; ++nn)
	{
	    if((node_type(*nn) == CLOSED_NODE) &&
	    	 ((*nn)->in_curves[0] != (*nn)->out_curves[0]))
	    {
	    	c1 = (*nn)->in_curves[0];
	    	c2 = (*nn)->out_curves[0];
	    	if (join_curves(c1,c2,negative_component(c1),
	    		        positive_component(c1),NULL) == NULL)
	    	{
		    interpolate_intfc_states(intfc) = sav_interp;
	    	    return NO;
	    	}
	    	(void) delete_node(*nn);
	    } 
	}
	interpolate_intfc_states(intfc) = sav_interp;
	return YES;
}		/*end join_curves_at_closed_nodes*/

LOCAL	bool	vector_wave_interaction(
	CROSS		*cross)
{
	bool		status = NO;
	CROSS		*cr;

	for (cr = cross; cr != NULL; cr = cr->next)
	{
	    if (is_scalar_vector_cross(cr) || is_vector_vector_cross(cr))
	    {
	        status = YES;
	        break;
	    }
	}
#if defined(USE_OVERTURE)
        return status;
#else /* if defined(USE_OVERTURE) */
        return pp_max_status(status);
#endif /* if defined(USE_OVERTURE) */
}		/*end vector_wave_interaction*/


#if defined(COMPLICATED_CROSS_STRUCTURE)



LOCAL	bool too_many_tangled_points(
	CROSS		*cross,
	float		*h,
	int		dim)
{
	CROSS	  *cr, *cr1;
	int	  ncc, mncc;
	static const int MAX_TANGLED_POINTS = 2;
	static const float MIN_CROSS_SEP = 3.0; /*TOLERANCE*/
	bool	  status;

	mncc = 0;

	for (cr = cross; cr; cr = cr->next)
	{
	    ncc = 0;
	    for (cr1 = cr->next; cr1; cr1 = cr1->next)
	    {
	        if (((cr1->c1 == cr->c1) || (cr1->c1 == cr->c2) ||
	    	     (cr1->c2 == cr->c1) || (cr1->c2 == cr->c2))
						&&
		    (scaled_separation(cr->p,cr1->p,h,dim) < MIN_CROSS_SEP))
		    ++ncc;
	    }
	    if (ncc > mncc) mncc = ncc;
	}
	if (mncc > MAX_TANGLED_POINTS)
	{
	    (void) printf("Too many (%d > %d) tangled points in a small area\n",
			  mncc,MAX_TANGLED_POINTS);
	    if (DEBUG) print_cross_list(cross);
	    status = YES;
	}
	status = NO;

#if defined(USE_OVERTURE)
        return status;
#else /* if defined(USE_OVERTURE) */
	return pp_max_status(status);
#endif /* if defined(USE_OVERTURE) */
}		/*end too_many_tangled_points*/
#endif /* defined(COMPLICATED_CROSS_STRUCTURE) */


#if defined(USE_OVERTURE)
 #if defined(SMOOTH_INTERFACE_FOR_SCATTER)

LOCAL void rm_attached_nodes_on_bdry(
	Front   *fr,
        NODE    **nodes,
        int     *alloc_num,
        int     *use_alloc) 
{
        NODE    *tmpn, **n;
        int     n_in, n_out, n_total;
        bool    sav_interp;
        int     i, j, merged = 0; 
        CURVE   *newc;
        BOND    *b; 

        if(current_interface() != fr->interf)
        {
            printf("ERROR in rm_attached_nodes_on_bdry\n");
            printf("current interface not equal fr->interf\n");
            clean_up(ERROR);
        }
        /* Drop possible dup. */
        for(i = 0; i < *use_alloc; i++)
        {
            for(j = 0; j < *use_alloc; j++)
            {
                if(i == j) continue;  
                if(nodes[i] == nodes[j] && nodes[j] != NULL) 
                    nodes[j] = NULL;  
            }
        }

        for(i = 0; i < *use_alloc; i++)
        {
            if((tmpn = nodes[i]) == NULL) 
                continue; 
            n_total = num_curves_at_node(tmpn, &n_in, &n_out);
            if (node_type(tmpn) == ERROR && n_total == 2 &&
                n_in == 1 && n_out == 1 &&
                ((tmpn)->in_curves[0] != (tmpn)->out_curves[0]))
            {
                merged++; 
                sav_interp = interpolate_intfc_states(fr->interf);
                interpolate_intfc_states(fr->interf) = YES;
                newc = join_curves((tmpn)->in_curves[0], (tmpn)->out_curves[0],
                            negative_component((tmpn)->in_curves[0]),
                            positive_component((tmpn)->in_curves[0]),NULL);
                if(newc == NULL)
                {
                    printf("ERROR, in full_redistribute\n");
                    printf("join_curves failed\n");
                    clean_up(ERROR);
                }
                (void) delete_node(tmpn);
                interpolate_intfc_states(fr->interf) = sav_interp;
                continue; 
            }
            if (node_type(tmpn) == ERROR && n_total == 1 &&
                n_in == 1 && n_out == 1 &&
                ((tmpn)->in_curves[0] == (tmpn)->out_curves[0]))
            {
                merged++;  
                node_type(tmpn) = CLOSED_NODE;
                clear_node_flags(tmpn);
                b = random_bond_on_curve(tmpn->in_curves[0]); 
                if (!move_closed_loop_node(tmpn->in_curves[0], b))
                {
                    printf("ERROR, in full_redistribute\n");
                    printf("ERROR, rm_attached_nodes_on_bdry(), random move node failed\n");
                    print_node(tmpn);
                    print_interface(fr->interf);
                    clean_up(ERROR); 
                }
            }
        }

        if(merged < *use_alloc)
        {
            int stop_run = NO; 
            for (n = fr->interf->nodes; n && *n; ++n)
            {
                if(node_type(*n) == ERROR)
                {
                    stop_run = YES; 
                    n_total = num_curves_at_node(*n, &n_in, &n_out);
                    printf("ERROR in rm_attached_nodes_on_bdry\n");
                    printf("The inserted nodes are not all deleted\n");
                    printf("The remaining node n_total = %d, n_in = %d, n_out = %d\n",
                        n_total, n_in, n_out);
                    printf("in_curves[0] = %d, out_curves[0] = %d\n", 
                        (*n)->in_curves[0], (*n)->out_curves[0]); 
                    print_node(*n);
                    print_node_flags(*n); 
                }
            }
            if(stop_run)
            {
                printf("In rm_attached_nodes_on_bdry, NODES: \n");  
                printf("merged = %d, use_alloc (alloced nodes) = %d\n", 
                     merged, *use_alloc);  
                printf("STOP in rm_attached_nodes_on_bdry\n");  
                printf("ERROR in fr[%d], level[%d]\n", fr->patch_number, fr->patch_level);  
                print_interface(fr->interf);
                clean_up(ERROR);
            }
        }
}

LOCAL bool attach_nodes_on_bdry(
	Front   *fr,
        NODE    **nodes,
        int     *alloc_num,
        int     *use_alloc) 
{
        RECT_GRID   *rgr = fr->rect_grid;
        int         dim = rgr->dim, dir;
        int         *lbuf = rgr->lbuf;
        int         *ubuf = rgr->ubuf;
        INTERFACE   *intfc = fr->interf;
        INTERFACE   *cur_intfc;
        int         use_side; 

        DEBUG_ENTER(attach_nodes_on_bdry)

        cur_intfc = current_interface();
        set_current_interface(intfc);

        for (dir = 0; dir < dim; dir++)
        {
            if(lbuf[dir] > 0)
            {
                if(rect_boundary_type(intfc,dir,0) == REFLECTION_BOUNDARY)
                    use_side  = YES;
                else
                    use_side = NO;
                attach_nodes_on_cut_line(intfc, rgr->L[dir], dir, 1, 
                   use_side, nodes, alloc_num, use_alloc);
            }
            if(ubuf[dir] > 0)
            {
                if(rect_boundary_type(intfc,dir,1) == REFLECTION_BOUNDARY)
                    use_side = YES;
                else
                    use_side = NO;
                attach_nodes_on_cut_line(intfc, rgr->U[dir], dir, 0, 
                   use_side, nodes, alloc_num, use_alloc);
            }
        }
        set_current_interface(cur_intfc);

        DEBUG_LEAVE(attach_nodes_on_bdry)
        return YES;  
}

/* attach_nodes_on_cut_line() is a simplified 
 * version of cut_interface(). It inserts NODE
 * points to the intersection. There are no other interface 
 * topology changes. 
 */
LOCAL void attach_nodes_on_cut_line(
	INTERFACE   *intfc,
        float       cut,
        int         dir,
        int         side,
        int         use_side,   /* if we use side param. to force symmtry*/
        NODE        **nodes_list,
        int         *alloc_num,
        int         *use_alloc)
{
        CROSS           Cr, *cr;
        RECT_GRID       *rgr = computational_grid(intfc);
        float           lbdry, ubdry; /* lower boundary, upper boundary of the cut line*/
        int             dim = intfc->dim;
        BOND            *b, *b1, *b2;
        CURVE           *c, **cc, **curves;
        POINT           *newp, *p;
        static bool     *adj_bond_cross = NULL; 
        static int      cr_alloc_len = 0;
        float           *h = rgr->h;
        float           min_sc_sep = MIN_SC_SEP(intfc);/*TOLERANCE*/
        const float     eps = 100.0*MACH_EPS;/*TOLERANCE*/
        int             num_deletes = 0;  
        int             i, num_cr = 0, cr_index = 0;
        float           crds[MAXD], newcrds[MAXD];
        bool            sav_intrp;
        static POINT    *cut_p = NULL;
        int             dead_loop = 0;
        int             want_debug = NO;
        float           grid_tol, len, min_h;
        float           start_sep, end_sep, clip_sep; 

        DEBUG_ENTER(attach_nodes_on_cut_line)

        if(intfc != current_interface())
        {
            printf("ERROR in attach_nodes_on_cut_line\n");
            printf("interface not current_interface\n");
            clean_up(ERROR);
        }

        min_h = rgr->h[0];
        for (i = 1; i < dim; ++i)
            if (rgr->h[i] < min_h)
                min_h = rgr->h[i];
        grid_tol = (min_sc_sep + min_sc_sep * 0.01)* min_h;

        min_sc_sep = 100.0*MACH_EPS;   
        if(dir == 0)
        {
            lbdry = rgr->L[1]-rgr->h[1];
            ubdry = rgr->U[1]+rgr->h[1];
        }
        else
        {
            lbdry = rgr->L[0]-rgr->h[0];
            ubdry = rgr->U[0]+rgr->h[0];
        }

        if (cut_p == NULL)
            cut_p = Static_point(intfc);

        /* Identify bonds crossing cut line */
        Cr.next = Cr.prev = NULL;
        cr = &Cr;
        for (cc = intfc->curves; cc && *cc; ++cc)
        {
            if(wave_type(*cc) < FIRST_PHYSICS_WAVE_TYPE)
                continue;
            for (b = (*cc)->first; b != NULL; b = b->next)
            {
                p = NULL;
                if(use_side == NO)
                {
                    /* make sure no double counting */
                    if((Coords(b->start)[dir] == cut) && (Coords(b->end)[dir] != cut))
                        p = b->start;   
                    else if ((Coords(b->start)[dir] > cut) && (Coords(b->end)[dir] < cut))
                        p = b->start;
                    else if ((Coords(b->end)[dir] > cut) && (Coords(b->start)[dir] < cut))
                        p = b->end;
                    /* old
                    if ((Coords(b->start)[dir] >= cut) && (Coords(b->end)[dir] < cut))
                        p = b->start;
                    else if ((Coords(b->end)[dir] >= cut) && (Coords(b->start)[dir] < cut))
                        p = b->end;
                    */
                }
                else
                    p = bond_crosses_cut_line(b,dir,cut,side);
                if (p == NULL)
                    continue;

                point_on_cut_line(intfc,cut_p,b,cut,dir);

                if(Coords(cut_p)[(dir+1)%dim] < lbdry ||
                   Coords(cut_p)[(dir+1)%dim] > ubdry)
                {
                    if(want_debug)
                    {
                        printf("Cut point [%17.15f, %17.15f]"
                            " outside the none-buffered rect_grid, omit \n",
                            Coords(cut_p)[0], Coords(cut_p)[1]);
                        print_bond(b);
                    }
                    continue;
                }

                /* Is it a good idea to delete loops on cut line at here? */
                if (cr && cr->prev &&
                    b->prev && b->prev->prev && b->prev->prev->prev &&
                    ((cr->b1 == b->prev) || (cr->b1 == b->prev->prev)) &&
                    (cr->prev->b1 == b->prev->prev->prev) &&
                    cross_bonds(b,b->prev->prev->prev,cut_p))
                {
                    num_deletes = 3;
                }
                else if (cr && cr->prev &&
                         b->prev && b->prev->prev &&
                         (cr->b1 == b->prev) &&
                         (cr->prev->b1 == b->prev->prev) &&
                         cross_bonds(b,b->prev->prev,cut_p))
                {
                    num_deletes = 2;
                }
                else if(cr && cr->prev &&
                        b->prev && b->prev->prev &&
                        (cr->b1 == b->prev) &&
                        (cr->prev->b1 == b->prev->prev) &&
                        (! cross_bonds(b,b->prev->prev,cut_p))) /* NOTE: ! cross_bonds() */
                {
                    /* Zigged interface */
                    num_deletes = 1;
                }

                if (num_deletes != 0)
                {
                    BOND        *newb;

                    if(num_deletes == 1)
                    {
                        newb = b->prev->prev;
                        cr = cr->prev;
                        cr->next = NULL;
                        (void) delete_end_of_bond(newb->next,*cc);
                        b = newb;
                    }
                    else
                    {
                        Coords(cut_p)[dir] = cut;
                        newp = Point(Coords(cut_p));
                        sav_intrp = interpolate_intfc_states(intfc);
                        interpolate_intfc_states(intfc) = YES;
                        (void) insert_point_in_bond(newp,b,*cc);
                        interpolate_intfc_states(intfc) = sav_intrp;
                        cr = cr->prev;
                        cr->next = NULL;
                        newb = b->next;
                        for (i = 0; i < num_deletes; ++i)
                            (void) delete_start_of_bond(newb->prev,*cc);
                        b = newb->prev;
                    }

                    num_deletes = 0;
                    if ((p = bond_crosses_cut_line(b,dir,cut,side)) == NULL)
                    {
                        if(cr->b1 == b)
                        {
                            cr = cr->prev;
                            if(cr == NULL)
                            {
                                Cr.next = Cr.prev = NULL;
                                cr = &Cr;
                            }
                            else
                                cr->next = NULL;
                        }
                        continue;
                    }
                }

                cr->next = (CROSS *)store(sizeof(CROSS));
                ++num_cr;
                cr->next->prev = cr;
                cr = cr->next;
                cr->c1 = *cc;
                cr->b1 = b;
                cr->p = p;
            }
        }
        if (Cr.next != NULL)
            Cr.next->prev = NULL;

        /* Check the cross list for adjacent bonds.  Closed loops are checked
         * so that the first/last bonds are considered adjacent.  Note that
         * this check must be done here as the bonds stored in the crosses will
         * no longer be accurate with respect to adjacency once splitting
         * occurs.   (adj_bond_cross[cr_index] == YES) if the i'th cross is
         * an adjacent bond cross. */

        if (num_cr > cr_alloc_len)
        {
            cr_alloc_len = 2*num_cr;
            if (adj_bond_cross != NULL)
                free(adj_bond_cross);
            vector(&adj_bond_cross,cr_alloc_len,sizeof(bool));
        }

        for (cr = Cr.next, cr_index = 0; cr != NULL; cr = cr->next, ++cr_index)
        {
            CROSS       *tmp_cr;

            adj_bond_cross[cr_index] = NO;
            for (tmp_cr = Cr.next; tmp_cr != NULL; tmp_cr = tmp_cr->next)
            {
                if (tmp_cr == cr)
                    continue;

                if ((tmp_cr->b1 == cr->b1->prev) ||
                    (tmp_cr->b1 == cr->b1->next) ||
                    (   (tmp_cr->c1 == cr->c1) &&
                        (node_type(cr->c1->start) == CLOSED_NODE) &&
                        (
                            ((cr->b1 == cr->c1->first) &&
                                (tmp_cr->b1 == cr->c1->last)) ||
                            ((cr->b1 == cr->c1->last)  &&
                                (tmp_cr->b1 == cr->c1->first))
                        )
                    )
                )
                {
                    adj_bond_cross[cr_index] = YES;
                    break;
                }
            }
        }

        /* Now insert nodes by  splitting 
         * curves at appropriate endpoint of crossing bonds.
         */
        for (cr = Cr.next, cr_index = 0; cr != NULL; cr = cr->next, ++cr_index)
        {
            bool        endpt;

            b = cr->b1;
            c = cr->c1;
            p = cr->p;

            if (want_debug)
            {
                (void) printf("full cross list at top of split loop\n");
                print_cross_list(Cr.next);
            }

            endpt = (p == b->end) ? YES : NO;

            point_on_cut_line(intfc,cut_p,b,cut,dir); 
            /*  Without insert new point, 
            clip_sep = sqrt(sqr_separation(p,cut_p,dim));
            */
            start_sep = sqrt(sqr_separation(b->start,cut_p,dim));
            end_sep  = sqrt(sqr_separation(b->end,cut_p,dim));

            /* what about do not insert point but just split at one bond point ????? */
            /* Try to use the bond point which is closest to the cut line */
            if(start_sep < end_sep)
            {
                if (endpt)
                {
                    if (b->prev != NULL)
                    {
                        cr->b1 = b = b->prev;
                        cr->p = p = (endpt) ? b->end : b->start;
                    }
                    else
                        endpt = NO;
                }
            }
            else
            {
                if (!endpt)
                {
                    if (b->next != NULL)
                    {
                        cr->b1 = b = b->next;
                        cr->p = p = (endpt) ? b->end : b->start;
                    }
                    else
                        endpt = YES;
                }
            }

            /*
            if (start_sep < min_sc_sep)
            {
                if (endpt)
                {
                    if (b->prev != NULL)
                    {
                        cr->b1 = b = b->prev;
                        cr->p = p = (endpt) ? b->end : b->start;
                    }
                    else
                        endpt = NO;
                }
            }
            else if (end_sep < min_sc_sep)
            {
                if (!endpt)
                {
                    if (b->next != NULL)
                    {
                        cr->b1 = b = b->next;
                        cr->p = p = (endpt) ? b->end : b->start;
                    }
                    else
                        endpt = YES;
                }
            }
            else if(clip_sep >= eps)
            {

                newp = Point(Coords(cut_p));
                sav_intrp = interpolate_intfc_states(intfc);
                interpolate_intfc_states(intfc) = YES;
                (void) insert_point_in_bond(newp,b,c);
                interpolate_intfc_states(intfc) = sav_intrp;
                rcl_after_insert_point(cr,newp,b);
                if (!endpt)
                    cr->b1 = b = b->next;
                cr->p = p = newp;
            }
            */


            if (node_type(c->start) == CLOSED_NODE)
            {
                if (endpt)
                    (void) move_closed_loop_node(c,b->next);
                else
                    (void) move_closed_loop_node(c,b);
                /* c->start */
                node_type(c->start) = ERROR;
                if (adj_bond_cross[cr_index])
                    set_adj_bond_cross_node(c->start);
                if(*use_alloc+1 == *alloc_num) 
                {
                    NODE **tmp_list; 
                    vector(&tmp_list, *alloc_num + 10, sizeof(NODE*)); 
                    for(int jj = 0; jj < *alloc_num; jj++)
                        tmp_list[jj] = nodes_list[jj]; 
                    free(nodes_list);
                    nodes_list = tmp_list; 
                    *alloc_num += 10; 
                }
                nodes_list[*use_alloc] = c->start; 
                (*use_alloc)++; 
                nodes_list[*use_alloc] = NULL; 
            }
            else if ((!endpt) && (b->prev == NULL))
            {
                /* For an elliptic interface, it is possible to shift
                 * a point onto a subdomain boundary (at U and L, NOT
                 * VU and VL).  This point then becomes interior after
                 * the communication, so it needs to be processed, and
                 * the curves joined back into one.  This is the
                 * reason for resetting SUBDOMAIN_NODE below.
                 */
                /* c->start */
                if (node_type(c->start) == SUBDOMAIN_NODE)
                    node_type(c->start) = ERROR;
                if (adj_bond_cross[cr_index])
                    set_adj_bond_cross_node(c->start);
                if(*use_alloc+1 == *alloc_num) 
                {
                    NODE **tmp_list; 
                    vector(&tmp_list, *alloc_num + 10, sizeof(NODE*)); 
                    for(int jj = 0; jj < *alloc_num; jj++)
                        tmp_list[jj] = nodes_list[jj]; 
                    free(nodes_list);
                    nodes_list = tmp_list; 
                    *alloc_num += 10; 
                }
                nodes_list[*use_alloc] = c->start; 
                (*use_alloc)++; 
                nodes_list[*use_alloc] = NULL; 
            }
            else if (endpt && (b->next == NULL))
            {
                /* See comment for above block. */
                /* c->end */
                if (node_type(c->end) == SUBDOMAIN_NODE)
                    node_type(c->end) = ERROR;
                if (adj_bond_cross[cr_index])
                    set_adj_bond_cross_node(c->end);
                if(*use_alloc+1 == *alloc_num) 
                {
                    NODE **tmp_list; 
                    vector(&tmp_list, *alloc_num + 10, sizeof(NODE*)); 
                    for(int jj = 0; jj < *alloc_num; jj++)
                        tmp_list[jj] = nodes_list[jj]; 
                    free(nodes_list);
                    nodes_list = tmp_list; 
                    *alloc_num += 10; 
                }
                nodes_list[*use_alloc] = c->end; 
                (*use_alloc)++; 
                nodes_list[*use_alloc] = NULL; 
            }
            else
            {
                bool    sav_scss;

                sav_scss = interpolate_states_at_split_curve_node();
                set_interpolate_states_at_split_curve_node(NO);
                curves = split_curve(p,b,c,negative_component(c),
                                     positive_component(c),
                                     negative_component(c),
                                     positive_component(c));
                set_interpolate_states_at_split_curve_node(sav_scss);

                if ((curves == NULL) && (!is_adj_bond_node(c->start)))
                {
                    /* The curve crosses twice, and has already been
                     * split for the first cross. */

                    break;
                }
                if (endpt)
                {
                    /* curves[0]->end */
                    node_type(curves[0]->end) = ERROR;
                    if (adj_bond_cross[cr_index])
                        set_adj_bond_cross_node(curves[0]->end);
                    if(*use_alloc+1 == *alloc_num) 
                    {
                        NODE **tmp_list; 
                        vector(&tmp_list, *alloc_num + 10, sizeof(NODE*)); 
                        for(int jj = 0; jj < *alloc_num; jj++)
                            tmp_list[jj] = nodes_list[jj]; 
                        free(nodes_list);
                        nodes_list = tmp_list; 
                        *alloc_num += 10; 
                    }
                    nodes_list[*use_alloc] = curves[0]->end; 
                    (*use_alloc)++; 
                    nodes_list[*use_alloc] = NULL; 
                } 
                else
                {
                    /* curves[1]->start */
                    node_type(curves[1]->start) = ERROR;
                    if (adj_bond_cross[cr_index])
                        set_adj_bond_cross_node(curves[1]->start);
                    if(*use_alloc+1 == *alloc_num) 
                    {
                        NODE **tmp_list; 
                        vector(&tmp_list, *alloc_num + 10, sizeof(NODE*)); 
                        for(int jj = 0; jj < *alloc_num; jj++)
                            tmp_list[jj] = nodes_list[jj]; 
                        free(nodes_list);
                        nodes_list = tmp_list; 
                        *alloc_num += 10; 
                    }
                    nodes_list[*use_alloc] = curves[1]->start; 
                    (*use_alloc)++; 
                    nodes_list[*use_alloc] = NULL; 
                }
                /* Reset pointer lists after split_curve */
                rcl_after_split(cr,p,b,c,curves);
            }
        }

        DEBUG_LEAVE(attach_nodes_on_cut_line)
} 
 #endif /* if defined(SMOOTH_INTERFACE_FOR_SCATTER) */ 

/* no_untangle_redistribute2d() 
 * is only used for the amr base patch interface redistribution.
 * The base patch interface contains tracked interior curves 
 * and phyiscal boundaries (which may not be covered by the fine patches)
 * For the tracked interior curves, since fine patches always have more 
 * accurate ones, these base grid interior curves are discarded eventually.
 * For the phyiscal boundaries which are not covered by the fine patches,
 * we still need to keep them, the purpose of no_untangle_redistribute2d()
 * is to redistribute these kinds of interface, though it does not check
 * the curve types. We have already removed base grid interior curves
 * before we send them into no_untangle_redistribute2d()
 */
LOCAL int no_untangle_redistribute2d(
        Front           *fr,
        bool            do_redist,
        bool            restart_init)
{
        INTERFACE       *intfc = fr->interf;
        INTERFACE       *save_intfc[2];
        float           *h = fr->rect_grid->h;
        float           hx = h[0], hy = h[1];
        float           dt, *dt_frac, min_dt_frac;
        int             redist_status;
        bool            force_redistribute;
        bool            any_redist;
        int             dim = fr->rect_grid->dim;
        static bool     first = YES;
        static float    dt_max, dt_min;

        DEBUG_ENTER(no_untangle_redistribute2d)

        if (first)
        {
            first = NO;
            dt_max = HUGE_VAL;
            dt_min = 0.0;
        }

        if (fr->interf->curves == NULL)
        {
            DEBUG_LEAVE(no_untangle_redistribute2d)
            return GOOD_REDISTRIBUTION;
        }

        if (DEBUG)
        {
            if (Frequency_of_redistribution(fr,GENERAL_WAVE) > 0)
            {
                (void) printf("Count redistribute(%d) %% "
                              "general curve frequency redistribute(%d) = %d\n",
                              Redistribution_count(fr),
                              Frequency_of_redistribution(fr,GENERAL_WAVE),
                              Redistribution_count(fr) %
                              Frequency_of_redistribution(fr,GENERAL_WAVE));
            }
            if (Frequency_of_redistribution(fr,VECTOR_WAVE) > 0)
            {
                (void) printf("Count redistribute(%d) %% "
                              "vector curve frequency redistribute(%d) = %d\n",
                              Redistribution_count(fr),
                              Frequency_of_redistribution(fr,VECTOR_WAVE),
                              Redistribution_count(fr) %
                              Frequency_of_redistribution(fr,VECTOR_WAVE));
            }
            DEBUG_FRONT("at start of no_untangle_redistribute2d()",fr)
            (void) printf("Interface before redistribute2d\n");
            print_interface(intfc);
        }

        save_intfc[0] = make_save_intfc(intfc);
        save_intfc[1] = NULL;

        print_storage("before redistribute","REDIST_storage");
        start_clock("redistribute");

#if defined(USE_OVERTURE)
        /* Should not worry about do sink node
         *
        if (!attach_sink_nodes(fr))
        {
            (void) printf("WARNING - in no_untangle_redistribute2d(), "
                          "attach_sink_nodes() failed for front[%d]\n",
                           fr->patch_number);
            (void) reset_interface_of_front(fr,save_intfc);
            prepare_to_leave_redistribute2d(save_intfc);
            DEBUG_LEAVE(no_untangle_redistribute2d)
            return BAD_REDISTRIBUTION;
        }
        */
#else  /* if defined(USE_OVERTURE) */
        if (!pp_min_status(attach_sink_nodes(fr)))
        {
            (void) printf("WARNING - in no_untangle_redistribute2d(), "
                          "attach_sink_nodes() failed\n");
            (void) reset_interface_of_front(fr,save_intfc);
            prepare_to_leave_redistribute2d(save_intfc);
            DEBUG_LEAVE(no_untangle_redistribute2d)
            return BAD_REDISTRIBUTION;
        }
#endif /* if defined(USE_OVERTURE) */

        DEBUG_FRONT("after attach_sink_nodes",fr)

            /* Redistribute Interface */

        Interface_redistributed(fr) = NO;
        force_redistribute = NO;

        if (do_redist)
        {
            bool status;

            status = (intfc->curves == NULL) ? YES :
                        (Interface_redistributed(fr)) ? YES :
                        Curve_redistribute(fr,&force_redistribute);

            status = closed_curve_node_redistribute(intfc,status);

#if defined(USE_OVERTURE)
            if (!status)
            {
                (void) printf("WARNING in no_untangle_redistribute2d(), "
                              "redistribution failed\n");
                (void) reset_interface_of_front(fr,save_intfc);
                prepare_to_leave_redistribute2d(save_intfc);
                DEBUG_LEAVE(no_untangle_redistribute2d)
                return BAD_REDISTRIBUTION;
            }
#else /* if defined(USE_OVERTURE) */
            if (!pp_min_status(status))
            {
                (void) printf("WARNING in no_untangle_redistribute2d(), "
                              "redistribution failed\n");
                (void) reset_interface_of_front(fr,save_intfc);
                prepare_to_leave_redistribute2d(save_intfc);
                DEBUG_LEAVE(no_untangle_redistribute2d)
                return BAD_REDISTRIBUTION;
            }
#endif /* if defined(USE_OVERTURE) */
        }

#if defined(USE_OVERTURE)
        any_redist = Interface_redistributed(fr);
#else /* if defined(USE_OVERTURE) */
        any_redist = pp_max_status(Interface_redistributed(fr));
#endif /* if defined(USE_OVERTURE) */

        if (any_redist)
        {
            /* redistribute or untangle may produce 2 and 3 bond loops */

            delete_small_loops(fr);
            if (!restart_init)
            {
                print_storage("before copy/delete intfc",
                              "REDIST_storage");
                start_clock("copy/delete intfc");
                set_size_of_intfc_state(size_of_state(fr->interf));
                set_copy_intfc_states(YES); /* Ensure states copied */
                fr->interf = copy_interface(intfc);
                (void) delete_interface(intfc);
                intfc = fr->interf;
                set_copy_intfc_states(YES);
                stop_clock("copy/delete intfc");
                print_storage("after copy/delete intfc","REDIST_storage");
            }
            /* This flag should be check outside this function call
            do_scatter = YES;
            */
        }

        DEBUG_FRONT("after no_untangle_redistribute:",fr)
        prepare_to_leave_redistribute2d(save_intfc);
        DEBUG_LEAVE(no_untangle_redistribute2d)
        return GOOD_REDISTRIBUTION;
}
#endif /* if defined(USE_OVERTURE) */ 

#endif /* defined(TWOD) */
