/*
*				freconst2d.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*/

#if defined(TWOD)
#include <front/flocaldecs.h>

#if defined(SMOOTH_INTERFACE_FOR_SCATTER)

struct _Cross_w_node{
        int        used;        /* indication of the usage of the crossing bond */
        CROSS      cr;
        int        ic[MAXD];    /* Closest grid nodes to crossing point */
        float      cut_p[MAXD]; /* Crossing point coords */
        POINT      *p;          /* Closer bond point to the crossing point */
        int        dir;         /* Cut direction */
        float      cut;         /* Cut line coords */
};
typedef struct _Cross_w_node Cross_w_node;
LOCAL int    alloc_cr_w_n, used_cr_w_n;
LOCAL Cross_w_node  *cr_w_n = NULL;

	/* LOCAL Function Declarations */
/*
LOCAL   void    adjust_dist_of_grid_node_w_crx(INTERFACE*,float,int,int,int);
LOCAL   void    set_adjust_crxing_on_boundary_segment(INTERFACE*,float,int,int);
LOCAL   void    print_adjust_crxing_on_boundary_segment(INTERFACE*);
LOCAL   void    remove_tangle_figure_on_boundary(INTERFACE*,float,int,int);
*/
LOCAL   void    adjust_dist_of_grid_node_w_crx_ver2(INTERFACE*,float,int,int,int,int);
LOCAL   void    in_out_box_bond(CURVE*,BOND*,int*,RECT_GRID*,BOND**,BOND**);
// IMPORT   bool    cross_segments(float,float,float,float,float,float,
//                      float,float,float*);
LOCAL   bool    point_on_cut_seg(float*,BOND*,int,float,float,float,int);  
LOCAL   void    reconstruct_bond_in_box(INTERFACE*,CURVE*,BOND*,int*,RECT_GRID*,BOND**,BOND**);
LOCAL   bool    bond_parallel_cut_line(BOND*,int,float);
LOCAL   void    shift_parallel_cut_line_bond(INTERFACE*,CURVE*,BOND*,int);

LOCAL int DEBUG_CUT = NO;

/* Function adjust_front_boundary_crxing_spacing_1 and
 * adjust_front_boundary_crxing_spacing() are responsible ONLY for
 * adjusting front boundary crxings.
 * The cases handled in this function:
 * 1) a bond point on domain corner (Could be
 * interior grid point also, for amr purpose).
 * 2) A bond passes through domain corner
 * (Could be interior grid point also, for amr purpose).
 * The un-tangle is not handled here except
 * for the tangle case on the boundary as in
 * cut_interface notes.
 *
 */

LIB_LOCAL void adjust_front_boundary_crxing_spacing_1(
        Front   *fr)
{
        RECT_GRID   *rgr = computational_grid(fr->interf);
        int         dim = rgr->dim, dir;
        int         *lbuf = rgr->lbuf;
        int         *ubuf = rgr->ubuf;
#if defined(USE_OVERTURE)
        int         *pplbuf = rgr->pplbuf;
        int         *ppubuf = rgr->ppubuf;
#endif /* if defined(USE_OVERTURE) */
        INTERFACE   *intfc = fr->interf;
        INTERFACE   *cur_intfc;
        int         use_side = NO;
        float       cut;
        int         cutic;

        cur_intfc = current_interface();
        set_current_interface(intfc);

        for (dir = 0; dir < dim; dir++)
        {
            if(lbuf[dir] > 0)
            {
                if(rect_boundary_type(intfc,dir,0) == REFLECTION_BOUNDARY)
                    use_side  = YES;
                else
                    use_side = NO;
                adjust_dist_of_grid_node_w_crx_ver2(intfc, rgr->L[dir],
                         0, dir, use_side, 1);
            }
            if(ubuf[dir] > 0)
            {
                if(rect_boundary_type(intfc,dir,1) == REFLECTION_BOUNDARY)
                    use_side = YES;
                else
                    use_side = NO;
                adjust_dist_of_grid_node_w_crx_ver2(intfc, rgr->U[dir],
                         rgr->gmax[dir], dir, use_side, 0);
            }
        }
}

LIB_LOCAL void adjust_front_boundary_crxing_spacing(
        Front   *fr)
{
        /*
        RECT_GRID   *rgr = fr->rect_grid;
        */
        RECT_GRID   *rgr = computational_grid(fr->interf);
        int         dim = rgr->dim, dir;
        int         *lbuf = rgr->lbuf;
        int         *ubuf = rgr->ubuf;
#if defined(USE_OVERTURE)
        int         *pplbuf = rgr->pplbuf;
        int         *ppubuf = rgr->ppubuf;
#endif /* if defined(USE_OVERTURE) */
        INTERFACE   *intfc = fr->interf;
        INTERFACE   *cur_intfc;
        int         use_side = NO;
        float       cut;
        int         cutic; 

        cur_intfc = current_interface();
        set_current_interface(intfc);

        /*
        for (dir = 0; dir < dim; dir++)
        {
            if(lbuf[dir] > 0)
                remove_tangle_figure_on_boundary(intfc, rgr->L[dir], dir, 1);
            if(ubuf[dir] > 0)
                remove_tangle_figure_on_boundary(intfc, rgr->U[dir], dir, 0);
        }
        */

        for (dir = 0; dir < dim; dir++)
        {
            if(lbuf[dir] > 0)
            {
                if(rect_boundary_type(intfc,dir,0) == REFLECTION_BOUNDARY)
                    use_side  = YES;
                else
                    use_side = NO;
                /*
                 *  adjust_dist_of_grid_node_w_crx(intfc, rgr->L[dir], dir, use_side, 1);
                 */
                adjust_dist_of_grid_node_w_crx_ver2(intfc, rgr->L[dir],
                         0, dir, use_side, 1);
            }
            if(ubuf[dir] > 0)
            {
                if(rect_boundary_type(intfc,dir,1) == REFLECTION_BOUNDARY)
                    use_side = YES;
                else
                    use_side = NO;
                /*
                 *  adjust_dist_of_grid_node_w_crx(intfc, rgr->U[dir], dir, use_side, 0);
                 */
                adjust_dist_of_grid_node_w_crx_ver2(intfc, rgr->U[dir],
                         rgr->gmax[dir], dir, use_side, 0);
            }
        }


        /* Adjust interface which will be cut for the parallel comm. */
        for (dir = 0; dir < dim; dir++)
        {
            use_side = NO;
#if defined(USE_OVERTURE)
            if(pplbuf[dir] > 0)
#else
            if(lbuf[dir] > 0)
#endif /* if defined(USE_OVERTURE) */
            {
#if defined(USE_OVERTURE)
                cut = rgr->GL[dir] + rgr->pph[dir]*rgr->pplbuf[dir];
#else  /* if defined(USE_OVERTURE) */ 
                cut = rgr->L[dir] + rgr->h[dir]*rgr->lbuf[dir];
#endif /* if defined(USE_OVERTURE) */
                if(cut > (rgr->L[dir] + 10.0*MACH_EPS) && cut < (rgr->U[dir] - 10.0*MACH_EPS))
                {
                    cutic = cell_index(cut, dir, rgr); 
                    if(fabs(cell_edge(cutic, dir, rgr) - cut) > 10.0*MACH_EPS)
                    {
                        if(fabs(cell_edge(cutic+1, dir, rgr) - cut) < 10.0*MACH_EPS)
                            cutic += 1; 
                        else 
                            cutic -= 1; 
                        if(fabs(cell_edge(cutic, dir, rgr) - cut) > 10.0*MACH_EPS)
                        {
                            printf("ERROR in adjust_front_boundary_crxing_spacing\n");
                            printf("pp cut line does not match\n");
                            printf("cut = %10.8f, cell_edge = %10.8f, cutic = %d, dir[%d]\n",
                               cut, cell_edge(cutic, dir, rgr), cutic, dir); 
                            clean_up(ERROR); 
                        } 
                    }
                    adjust_dist_of_grid_node_w_crx_ver2(intfc, cut, cutic, dir, use_side, 1);
                }
            }
#if defined(USE_OVERTURE)
            if(ppubuf[dir] > 0)
#else 
            if(ubuf[dir] > 0)
#endif /* if defined(USE_OVERTURE) */
            {
#if defined(USE_OVERTURE)
                cut = rgr->GU[dir] - rgr->pph[dir]*rgr->ppubuf[dir];
#else  /* if defined(USE_OVERTURE) */ 
                cut = rgr->U[dir] - rgr->h[dir]*rgr->ubuf[dir];
#endif /* if defined(USE_OVERTURE) */
                if(cut > (rgr->L[dir] + 10.0*MACH_EPS) && cut < (rgr->U[dir] - 10.0*MACH_EPS))
                {
                    cutic = cell_index(cut, dir, rgr); 
                    if(fabs(cell_edge(cutic, dir, rgr) - cut) > 10.0*MACH_EPS)
                    {
                        if(fabs(cell_edge(cutic+1, dir, rgr) - cut) < 10.0*MACH_EPS)
                            cutic += 1; 
                        else 
                            cutic -= 1; 
                        if(fabs(cell_edge(cutic, dir, rgr) - cut) > 10.0*MACH_EPS)
                        {
                            printf("ERROR in adjust_front_boundary_crxing_spacing\n");
                            printf("pp cut line does not match\n");
                            printf("cut = %10.8f, cell_edge = %10.8f, cutic = %d, dir[%d]\n",
                               cut, cell_edge(cutic, dir, rgr), cutic, dir); 
                            clean_up(ERROR); 
                        } 
                    }
                    adjust_dist_of_grid_node_w_crx_ver2(intfc, cut, cutic, dir, use_side, 0);
                }
            }
        }

        /*
        alloc_cr_w_n = used_cr_w_n = 0;
        for(dir = 0; dir < dim; dir++)
            alloc_cr_w_n = max(rgr->gmax[dir], alloc_cr_w_n);
        vector(&cr_w_n, alloc_cr_w_n, sizeof(Cross_w_node));
        printf("After adjust_close_pt_w_grid_node_w_crx\n");
        for (dir = 0; dir < dim; dir++)
        {
            if(lbuf[dir] > 0)
                set_adjust_crxing_on_boundary_segment(intfc, rgr->L[dir], dir, 1);
            if(ubuf[dir] > 0)
                set_adjust_crxing_on_boundary_segment(intfc, rgr->U[dir], dir, 0);
        }
        for (dir = 0; dir < dim; dir++)
        {
            if(pplbuf[dir] > 0)
            {
                cut = rgr->GL[dir] + rgr->pph[dir]*rgr->pplbuf[dir];
                if(cut > (rgr->L[dir] + 10.0*MACH_EPS) && cut < (rgr->U[dir] - 10.0*MACH_EPS))
                    set_adjust_crxing_on_boundary_segment(intfc, cut, dir, 1);
            }
            if(ppubuf[dir] > 0)
            {
                cut = rgr->GU[dir] - rgr->pph[dir]*rgr->ppubuf[dir];
                if(cut > (rgr->L[dir] + 10.0*MACH_EPS) && cut < (rgr->U[dir] - 10.0*MACH_EPS))
                    set_adjust_crxing_on_boundary_segment(intfc, cut, dir, 0);
            }
        }
        print_adjust_crxing_on_boundary_segment(intfc);
        free(cr_w_n);
        */

        set_current_interface(cur_intfc);
}

LOCAL void adjust_dist_of_grid_node_w_crx_ver2(
        INTERFACE   *intfc,
        float       cut,            /* coordinate of cut line */
        int         cutic,          /* icoords of cut line in compute grid */
        int         dir,            /* direction of cut line normal */
        int         use_side,       /* if we use side param. to force symmtry*/
        int         side)
{
        RECT_GRID       *rgr = computational_grid(intfc);
        float           lbdry, ubdry; /* lower boundary, upper boundary of the cut line*/
        int             bmin, bmax;
        int             dim = intfc->dim;
        BOND            *b, *bin, *bout;
        CURVE           *c, **cc;
        POINT           *newp, *p, *p2;
        float           *h = rgr->h;
        float           min_sc_sep = MIN_SC_SEP(intfc);/*TOLERANCE*/
        const float     eps = 100.0*MACH_EPS;/*TOLERANCE*/
        int             i, j;
        int             indx, tmpi, ic[MAXD], tmpic[MAXD];
        float           segcut, seglbdry, segubdry;
        float           sep_dist, sep_dist2, sep_dist3;
        float           crds[MAXD], newcrds[MAXD];
        bool            sav_intrp;
        static POINT    *cut_p = NULL, *cut_p2 = NULL;
        float           cut_crds[MAXD];
        int             dead_loop = 0;
        int             want_debug = NO;
        float           grid_tol, len, min_h;

        DEBUG_ENTER(adjust_dist_of_grid_node_w_crx_ver2)

        if(intfc != current_interface())
        {
            printf("ERROR in adjust_dist_of_grid_node_w_crx\n");
            printf("interface not current_interface\n");
            clean_up(ERROR);
        }

        min_h = rgr->h[0];
        for (i = 1; i < dim; ++i)
            if (rgr->h[i] < min_h)
                min_h = rgr->h[i];
        grid_tol = (min_sc_sep + min_sc_sep * 0.01)* min_h;

        if(dir == 0)
        {
            lbdry = rgr->L[1]-rgr->h[1];
            ubdry = rgr->U[1]+rgr->h[1];
            bmin = 0; bmax = rgr->gmax[1];
        }
        else
        {
            lbdry = rgr->L[0]-rgr->h[0];
            ubdry = rgr->U[0]+rgr->h[0];
            bmin = 0; bmax = rgr->gmax[0];
        }
        if (cut_p == NULL)
        {
            cut_p = Static_point(intfc);
            cut_p2 = Static_point(intfc);
        }


        if(want_debug)
            printf("CUT = %17.14f, dir[%d], lbdry[%g], ubdry[%g]\n",
                 cut, dir, lbdry, ubdry);

        sav_intrp = interpolate_intfc_states(intfc);
        interpolate_intfc_states(intfc) = YES;

        for (cc = intfc->curves; cc && *cc; ++cc)
        {
            if(wave_type(*cc) < FIRST_PHYSICS_WAVE_TYPE)
                continue;
            for (b = (*cc)->first; b != NULL; b = b->next)
            {
test_this_bond_again:
                if(bond_parallel_cut_line(b, dir, cut))
                {
                    /* Is it safe not to check b->prev again?
                     */
                    shift_parallel_cut_line_bond(intfc,*cc, b, dir); 
                }
                p = bond_crosses_cut_segment(b,dir,side,cut,lbdry,ubdry,cut_crds);
                if( p != NULL)
                {
                    for(i = 0;i < dim; i++)
                        Coords(cut_p)[i] = cut_crds[i];
                }
                /* Test if bond crossing other direction a set of grid line segments */
                for(j = bmin; j <= bmax; j++)
                {
                    sep_dist = sep_dist2 = HUGE_VAL;
                    tmpic[dir] = cutic;
                    tmpic[(dir+1)%dim] = j;
                    segcut = cell_edge(tmpic[(dir+1)%dim], (dir+1)%dim, rgr);
                    seglbdry = cell_edge(tmpic[dir], dir, rgr) - h[dir];
                    segubdry = cell_edge(tmpic[dir], dir, rgr) + h[dir];
                    for(i = 0;i < dim; i++)
                        crds[i] = cell_edge(tmpic[i], i, rgr);
                    /* The side is meaningless here. */
                    p2 = bond_crosses_cut_segment(b, (dir+1)%dim, 0,
                            segcut, seglbdry, segubdry,cut_crds);
                    if(p2 != NULL)
                    {
                        for(i = 0;i < dim; i++)
                            Coords(cut_p2)[i] = cut_crds[i];
                    }
                    if(p != NULL)
                        sep_dist = _scaled_separation(crds, Coords(cut_p), h, dim);
                    if(p2 != NULL)
                        sep_dist2 = _scaled_separation(crds, Coords(cut_p2), h, dim);
                    if(sep_dist <= 2.0*min_sc_sep || sep_dist2 <= 2.0*min_sc_sep)
                    {
                        /* Bond is close to grid node, we should apply
                         * local grid-based reconstruction.
                         */
                        if(dead_loop == 20)
                        {
                            /* Purely for the debugging purpose */
                            printf("ERROR adjust_dist_of_grid_node_w_crx_ver2\n");
                            printf("Dead loop formed\n");
                            printf("close to ic[%d, %d] [%10.8f, %10.8f]\n",
                               tmpic[0], tmpic[1], crds[0], crds[1]);
                            print_bond(b);
                            print_curve(*cc);
                            clean_up(ERROR);
                        }

                        in_out_box_bond(*cc, b, tmpic, rgr, &bin, &bout);
                        if(bin == NULL || bout == NULL)
                            continue;
                        reconstruct_bond_in_box(intfc, *cc, b, tmpic, rgr, &bin, &bout);
                        b = bout;
                        dead_loop++;
                        goto test_this_bond_again;
                    }
                }
            }
        }

        DEBUG_LEAVE(adjust_dist_of_grid_node_w_crx_ver2)
}


/* The box is centered at ic[], with edge length h[]. */
/* The side effect is considered when determining whether
 * bond intersects with box edge.
 * Exception:
 * For a bond, if start point overlaps edge, end point inside (or overlaping edge)
 * it is bin.
 * For a bond, if end point overlaps edge, start point inside (or overlaping edge)
 * it is bout.
 *    -----------------
 *    |               |
 *    S ------------> E
 *    |               |
 *    -----------------
 * This bond is bin and bout.
 * If node point of a closed curve is inside this box, this node
 * point is moved to bin->start. 
 */
LOCAL void in_out_box_bond(
        CURVE       *c,
        BOND        *bond,
        int         *ic,
        RECT_GRID   *gr,
        BOND        **bin,
        BOND        **bout)
{
        float       *h = gr->h;
        int         i, dir, dim = gr->dim;
        float       crds[MAXD], cut_crds[MAXD];
        BOND        *b;
        POINT       *p;
        float       segcut, seglbdry, segubdry;
        int         loop;
        bool        found = NO;
        bool        n_in_between; 

        *bin = *bout = NULL;
        for(i = 0;i < dim; i++)
            crds[i] = cell_edge(ic[i], i, gr);

        /* Find the entering box bond */
        loop = 0;
        for(b = bond;  b != NULL; )
        {
            if(b == bond) loop++;
            for(dir = 0; dir < dim; dir++)
            {
                segcut = crds[dir] - h[dir]/2.0;
                seglbdry = crds[(dir+1)%dim] - h[(dir+1)%dim]/2.0;
                segubdry = crds[(dir+1)%dim] + h[(dir+1)%dim]/2.0;
                /*
                printf("dir[%d] segcut[%15.12f] LU[%15.12f, %15.12f]\n",
                       dir, segcut, seglbdry, segubdry);
                */
                /* L side */
                p = bond_crosses_cut_segment(b, dir, 1,
                            segcut, seglbdry, segubdry, cut_crds);
                if(p == b->start)
                    goto find_bin;
                /* U side */
                segcut = crds[dir] + h[dir]/2.0;
                p = bond_crosses_cut_segment(b, dir, 0,
                            segcut, seglbdry, segubdry, cut_crds);
                if(p == b->start)
                    goto find_bin;
            }
            if(loop == 2)
            {
                printf("ERROR in in_out_box_bond\n");
                printf("When find bin, curve is looped, there is no bond entering box\n");
                printf("ic[%d, %d] [%15.12f, %15.12f]\n",
                    ic[0], ic[1], crds[0], crds[1]);
                print_bond(bond);
                print_curve(c);
                clean_up(ERROR);
            }
            b = b->prev;     
            if(b == NULL && is_closed_curve(c))
                b = c->last; 
        } 
find_bin:
        *bin = b;

        /* Find the leaving box bond */
        loop = 0;
        for(b = bond;  b != NULL; )
        {
            if(b == bond) loop++;
            for(dir = 0; dir < dim; dir++)
            {
                segcut = crds[dir] - h[dir]/2.0;
                seglbdry = crds[(dir+1)%dim] - h[(dir+1)%dim]/2.0;
                segubdry = crds[(dir+1)%dim] + h[(dir+1)%dim]/2.0;
                /* L side */
                p = bond_crosses_cut_segment(b, dir, 1,
                            segcut, seglbdry, segubdry, cut_crds);
                if(p == b->end)
                    goto find_bout;
                /* U side */
                segcut = crds[dir] + h[dir]/2.0;
                p = bond_crosses_cut_segment(b, dir, 0,
                            segcut, seglbdry, segubdry, cut_crds);
                if(p == b->end)
                    goto find_bout;
            }
            if(loop == 2)
            {
                printf("ERROR in in_out_box_bond\n");
                printf("When find bout, curve is looped, there is no bond leaving box\n");
                printf("ic[%d, %d] [%15.12f, %15.12f]\n",
                    ic[0], ic[1], crds[0], crds[1]);
                print_bond(bond);
                print_curve(c);
                clean_up(ERROR);
            }
            b = b->next;
            if(b == NULL && is_closed_curve(c))
                b = c->first; 
        }
find_bout:
        *bout = b;

        if(*bin == NULL || *bout == NULL)
            return;
        /*
        if(*bin == NULL || *bout == NULL)
        {
            printf("ERROR in in_out_box_bond\n");
            printf("There is no bond enter or leaving box\n");
            print_bond(*bin);
            print_bond(*bout);
            printf("ic[%d, %d] [%15.12f, %15.12f]\n",
                ic[0], ic[1], crds[0], crds[1]);
            print_bond(bond);
            print_curve(c);
            clean_up(ERROR);
        }
        */

        loop = 0; 
        n_in_between = NO; 
        for(b = *bin; b != NULL; )
        {
            if(b == *bout)
            {
                if(is_closed_curve(c) && 
                   (*bout)->start == c->start->posn)
                    n_in_between = YES; 
                found = YES;
                break;
            }
            b = b->next; 
            if(b == NULL && is_closed_curve(c))
            {
                loop++; 
                b = c->first;
                n_in_between = YES;  
            }
            if(loop == 2)
            {
                printf("ERROR in in_out_box_bond\n");
                printf("When found bin and bout, curve is looped\n");
                printf("At the last test of bin & bout connection\n"); 
                printf("ic[%d, %d] [%15.12f, %15.12f]\n",
                    ic[0], ic[1], crds[0], crds[1]);
                print_bond(bond);
                print_curve(c);
                clean_up(ERROR);
            }
        }
        if(n_in_between)
            move_closed_loop_node(c, (*bin)); 
 
        if(found == NO)
        {
            printf("ERROR: in_out_box_bond\n");
            printf("Does not find bout when search starts from bin\n");
            print_bond(*bin); print_bond(*bout);
            print_curve(c);
            clean_up(ERROR);
        }
}


LOCAL void reconstruct_bond_in_box(
        INTERFACE   *intfc,
        CURVE       *c,
        BOND        *bond,
        int         *ic,
        RECT_GRID   *gr,
        BOND        **bin,
        BOND        **bout)
{
        float       *h = gr->h;
        int         i, dir, dim = gr->dim;
        float       min_sc_sep = MIN_SC_SEP(intfc);/*TOLERANCE*/
        float       crds[MAXD], in_crds[MAXD], out_crds[MAXD];
        float       crx_crds[2][MAXD];
        float       newcrds[MAXD];
        int         crx_dir[2];
        BOND        *b, *b2, *bindel, *boutdel;
        POINT       *newp, *p, *p2;
        float       segcut, seglbdry, segubdry;
        float       sep_dist, sep_dist2, len, len2;
        bool        sav_intrp, found = NO;
        bool        want_debug = NO;
        int         num_deletes = 0, num_crx = 0;
        bool        connected = NO; 

        sav_intrp = interpolate_intfc_states(intfc);
        interpolate_intfc_states(intfc) = YES;

        for(i = 0;i < dim; i++)
            crds[i] = cell_edge(ic[i], i, gr);

        for(b = *bin; b != NULL; b = b->next)
        {
            if(b == *bout)
            {
                connected = YES; 
                break; 
            }
        }
        if(connected == NO)
        {
            printf("ERROR reconstruct_bond_in_box\n");
            printf("bin and bout are not connected\n");
            clean_up(ERROR);  
        }

        if(want_debug)
        {
            printf("Entered reconstruct_bond_in_box\n"); 
            printf("Box ic[%d, %d] center[%10.8f, %10.8f]\n", 
                    ic[0], ic[1], crds[0], crds[1]); 
            printf("CUT line is L[%10.8f, %10.8f], U[%10.8f, %10.8f]\n",
                 crds[0] - h[0]/2.0, crds[1] - h[1]/2.0,
                 crds[0] + h[0]/2.0, crds[1] + h[1]/2.0); 
            for(b = (*bin); b != (*bout)->next; b = b->next)
                print_bond(b);
            printf("End of print BOND inside the BOX\n");
        }
       
        /* Find the entering box bond cxing*/
        for(dir = 0; dir < dim; dir++)
        {
            segcut = crds[dir] - h[dir]/2.0;
            seglbdry = crds[(dir+1)%dim] - h[(dir+1)%dim]/2.0;
            segubdry = crds[(dir+1)%dim] + h[(dir+1)%dim]/2.0;

            p = bond_crosses_cut_segment(*bin, dir, 1,
                        segcut, seglbdry, segubdry, in_crds);
            if(p == (*bin)->start)
            {
                if(want_debug)
                {
                    /* 
                    printf("cut dir[%d] cut line[%10.8f, %10.8f, %10.8f]\n",
                       dir, segcut, seglbdry, segubdry);
                    */
                    printf("Dir[%d] bond[%10.8f, %10.8f, %10.8f, %10.8f], cutP[%10.8f, %10.8f]\n",
                      dir, Coords((*bin)->start)[0], Coords((*bin)->start)[1],
                      Coords((*bin)->end)[0], Coords((*bin)->end)[1],
                      in_crds[0], in_crds[1]);
                }
                break;
            }
            segcut = crds[dir] + h[dir]/2.0;
            p = bond_crosses_cut_segment(*bin, dir, 0,
                        segcut, seglbdry, segubdry, in_crds);
            if(p == (*bin)->start)
            {
                if(want_debug)
                {
                    /* 
                    printf("cut dir[%d] cut line[%10.8f, %10.8f, %10.8f]\n",
                       dir, segcut, seglbdry, segubdry);
                    */
                    printf("Dir[%d] bond[%10.8f, %10.8f, %10.8f, %10.8f], cutP[%10.8f, %10.8f]\n",
                      dir, Coords((*bin)->start)[0], Coords((*bin)->start)[1],
                      Coords((*bin)->end)[0], Coords((*bin)->end)[1],
                      in_crds[0], in_crds[1]);
                }
                break;
            }
        }

        /* Find the leaving box bond crxing */
        for(dir = 0; dir < dim; dir++)
        {
            segcut = crds[dir] - h[dir]/2.0;
            seglbdry = crds[(dir+1)%dim] - h[(dir+1)%dim]/2.0;
            segubdry = crds[(dir+1)%dim] + h[(dir+1)%dim]/2.0;
            p2 = bond_crosses_cut_segment(*bout, dir, 1,
                        segcut, seglbdry, segubdry, out_crds);
            if(p2 == (*bout)->end)
            {
                if(want_debug)
                {
                    /* 
                    printf("cut dir[%d] cut line[%10.8f, %10.8f, %10.8f]\n",
                       dir, segcut, seglbdry, segubdry);
                    */
                    printf("Dir[%d] bond[%10.8f, %10.8f, %10.8f, %10.8f], cutP[%10.8f, %10.8f]\n",
                      dir, Coords((*bout)->start)[0], Coords((*bout)->start)[1],
                      Coords((*bout)->end)[0], Coords((*bout)->end)[1],
                      out_crds[0], out_crds[1]);
                }
                break;
            }
            segcut = crds[dir] + h[dir]/2.0;
            p2 = bond_crosses_cut_segment(*bout, dir, 0,
                        segcut, seglbdry, segubdry, out_crds);
            if(p2 == (*bout)->end)
            {
                if(want_debug)
                {
                    /* 
                    printf("cut dir[%d] cut line[%10.8f, %10.8f, %10.8f]\n",
                       dir, segcut, seglbdry, segubdry);
                    */
                    printf("Dir[%d] bond[%10.8f, %10.8f, %10.8f, %10.8f], cutP[%10.8f, %10.8f]\n",
                      dir, Coords((*bout)->start)[0], Coords((*bout)->start)[1],
                      Coords((*bout)->end)[0], Coords((*bout)->end)[1],
                      out_crds[0], out_crds[1]);
                }
                break;
            }
        }
        if(p == p2)
        {
            printf("ERROR in reconstruct_bond_in_box\n");
            printf("p == p2\n");
            clean_up(ERROR);  
        }

        /* Insert points at scaled_separation(insert_p, in_cut_p, h) = 0.25 */
        sep_dist = _scaled_separation(in_crds, Coords((*bin)->end), h, dim);
        if(sep_dist > 0.25)
        {
            for(i = 0; i < dim; i++)
                newcrds[i] = 0.25/sep_dist*(Coords((*bin)->end)[i]-in_crds[i])
                           + in_crds[i];
            newp = Point(newcrds);
            if((*bin) == (*bout))
            {
                insert_point_in_bond(newp, (*bin), c);
                (*bout) = (*bin)->next;
            }
            else
                insert_point_in_bond(newp, (*bin), c);
            set_bond_length((*bin), dim);
            set_bond_length((*bin)->next, dim);
            bindel = (*bin)->next;
        }
        else
        {
            float tmp_sep, sep_dist3; 
            /* 
            printf("ERROR in reconstruct_bond_in_box\n");
            printf("dist |in_crds - bin->end| is less than 0.25, sep_dist = %8.6f\n", 
                sep_dist);
            printf("cutP[%10.8f, %10.8f]\n", in_crds[0], in_crds[1]);
            printf("need to find point on bin->next bond\n");
            */
            tmp_sep = sep_dist; 
            for(b = (*bin)->next; b != (*bout)->next; b = b->next)
            {
                tmp_sep += scaled_separation(b->start, b->end, h, dim);
                if(tmp_sep > 0.25)
                    break;
            }
            if(tmp_sep < 0.25)
            {
                printf("ERROR in reconstruct_bond_in_box\n");
                printf("dist |in_crds - bin| is less than 0.25, tmp_sep = %8.6f\n",
                   tmp_sep);
                for(b = (*bin); b != (*bout)->next; b = b->next)
                {
                    print_bond(b);
                    sep_dist = _scaled_separation(Coords(b->start), 
                             Coords(b->end), h, dim);
                    printf("bond sep_idst = %8.6f\n", sep_dist); 
                }
                clean_up(ERROR); 
            }
            sep_dist3 = scaled_separation(b->start, b->end, h, dim);   
            sep_dist2 = 0.25 - (tmp_sep - sep_dist3);   
            for(i = 0; i < dim; i++)
                newcrds[i] = sep_dist2/sep_dist3*(Coords(b->end)[i]-Coords(b->start)[i])
                           + Coords(b->start)[i];
            /* 
            printf("sep_dist3 = %8.6f, sep_dist2 = %8.6f\n", sep_dist3, sep_dist2); 
            printf("newcrds[%10.8f, %10.8f]\n", newcrds[0], newcrds[1]); 
            */
            if(sep_dist2 < min_sc_sep)
            {
                for(i = 0; i < dim; i++)
                    Coords(b->start)[i] = newcrds[i];  
                bindel = b; 
                set_bond_length(b, dim);
                set_bond_length(b->prev, dim);
            }
            else
            {
                newp = Point(newcrds);
                if(b == (*bout))
                {
                    insert_point_in_bond(newp, b, c);
                    (*bout) = b->next;
                }
                else
                    insert_point_in_bond(newp, b, c);
                bindel = b->next;
            }
            /* 
            for(b = (*bin); b != (*bout)->next; b = b->next)
            {
                print_bond(b);
                sep_dist = _scaled_separation(Coords(b->start), Coords(b->end), h, dim);
                printf("bond sep_idst = %8.6f\n", sep_dist); 
            }
            printf("End of print BOND inside the BOX\n");
            printf("Bond bindel\n"); 
            print_bond(bindel); 
            clean_up(ERROR);
            */
        }

        if((*bin) == (*bout))
        {
            printf("ERROR in reconstruct_bond_in_box\n");
            printf("bin should not equal bout\n");
            clean_up(ERROR);
        }
        /* Insert points at scaled_separation(insert_p, out_cut_p, h) = 0.25 */
        sep_dist = _scaled_separation(out_crds, Coords((*bout)->start), h, dim);
        if(sep_dist > 0.25)
        {
            for(i = 0; i < dim; i++)
                newcrds[i] = 0.25/sep_dist*(Coords((*bout)->start)[i] - out_crds[i])
                           + out_crds[i];
            newp = Point(newcrds);
            insert_point_in_bond(newp, (*bout), c);
            (*bout) = (*bout)->next;
            boutdel = (*bout)->prev;
        }
        else
        {
            float tmp_sep, sep_dist3;
            tmp_sep = sep_dist;
            /* 
            printf("ERROR in reconstruct_bond_in_box\n");
            printf("dist |out_crds - bout->start| is less than 0.25, sep_dist = %8.6f\n", sep_dist);
            printf("cutP[%10.8f, %10.8f]\n", out_crds[0], out_crds[1]);
            printf("need to find point on bout->prev bond\n");
            */
            for(b = (*bout)->prev; b != bindel->prev; b = b->prev)
            {
                tmp_sep += scaled_separation(b->start, b->end, h, dim);
                if(tmp_sep > 0.25)
                    break;
            }
            if(tmp_sep < 0.25)
            {
                printf("ERROR in reconstruct_bond_in_box\n");
                printf("dist |out_crds - bout| is less than 0.25, tmp_sep = %8.6f\n",
                   tmp_sep);
                clean_up(ERROR);
            }

            sep_dist3 = scaled_separation(b->start, b->end, h, dim);
            sep_dist2 = 0.25 - (tmp_sep - sep_dist3);
            for(i = 0; i < dim; i++)
                newcrds[i] = sep_dist2/sep_dist3*(Coords(b->start)[i] - Coords(b->end)[i])
                           + Coords(b->end)[i];
            /* 
            printf("sep_dist3 = %8.6f, sep_dist2 = %8.6f\n", sep_dist3, sep_dist2);
            printf("newcrds[%10.8f, %10.8f]\n", newcrds[0], newcrds[1]);
            */ 
            if(sep_dist2 < min_sc_sep)
            {
                for(i = 0; i < dim; i++)
                    Coords(b->end)[i] = newcrds[i];
                boutdel = b;
                set_bond_length(b, dim);
                set_bond_length(b->next, dim);
            }
            else
            {
                newp = Point(newcrds);
                insert_point_in_bond(newp, b, c);
                boutdel = b;
            }
            /* 
            for(b = (*bin); b != (*bout)->next; b = b->next)
            {
                print_bond(b);
                sep_dist = _scaled_separation(Coords(b->start), Coords(b->end), h, dim);
                printf("bond sep_idst = %8.6f\n", sep_dist);
            }
            printf("ERROR in reconstruct_bond_in_box\n");
            printf("dist |out_crds - bout->start| is less than 0.25\n");
            printf("need to find point on bout->prev bond\n");
            print_bond(boutdel); 
            clean_up(ERROR);
            */
        }

        /* Replace the bonds in between with one single bond */
        for(b = bindel; b != NULL; b = b->next)
        {
            if(b == boutdel)
            {
                found = YES;
                break;
            }
            num_deletes++;
        }
        if(found == NO)
        {
            printf("ERROR: reconstruct_bond_in_box\n");
            printf("Does not find boutdel when search starts from bindel\n");
            print_bond((*bin)); print_bond((*bout));
            print_curve(c);
            clean_up(ERROR);
        }

        b2 = boutdel->next;
        for(i = 0; i < num_deletes; i++)
            delete_start_of_bond(b2->prev, c);
        if(bindel->next != b2)
        {
            printf("ERROR: reconstruct_bond_in_box\n");
            printf("After delete points in between\n");
            printf("bindel->next != boutdel->next\n");
            print_bond(bindel); print_bond(b2);
            print_curve(c);
            clean_up(ERROR);
        }

        if(want_debug)
        {
            printf("After delete points between bindel & doutdel\n");
            for(b = (*bin); b != (*bout)->next; b = b->next)
                print_bond(b);
            printf("End of print BOND inside the BOX\n");
        }

        /* bindel should be the only bond we need to test again */
        sep_dist = scaled_separation(bindel->start, bindel->end, h, dim);
        if(sep_dist < min_sc_sep)
        {
            for(i = 0; i < dim; i++)
                newcrds[i] = (Coords(bindel->start)[i] + Coords(bindel->end)[i])/2.0;
            for(i = 0; i < dim; i++)
                Coords(bindel->end)[i] = newcrds[i];
            set_bond_length(bindel->next, dim);
            delete_start_of_bond(bindel, c);
        }
        else
        {
            for(dir = 0; dir < dim; dir++)
            {
                segcut = crds[dir];
                seglbdry = crds[(dir+1)%dim] - h[(dir+1)%dim]/2.0;
                segubdry = crds[(dir+1)%dim] + h[(dir+1)%dim]/2.0;
                p = bond_crosses_cut_segment(bindel, dir, 0,
                        segcut, seglbdry, segubdry, out_crds);
                if(p)
                {
                    for(i = 0; i < dim; i++)
                        crx_crds[num_crx][i] = out_crds[i];
                    crx_dir[num_crx] = dir;
                    num_crx++;
                }
            }
            switch(num_crx)
            {
            case 1:
                sep_dist = _scaled_separation(crx_crds[0], crds, h, dim);
                if(sep_dist <= 2.0*min_sc_sep)
                {
                    if(crx_dir[0] == 0)
                    {
                        if(crx_crds[0][1] > crds[1])
                            crx_crds[0][1] = crds[1] + 2.5*min_sc_sep*h[1];
                        else
                            crx_crds[0][1] = crds[1] - 2.5*min_sc_sep*h[1];
                    }
                    else
                    {
                        if(crx_crds[0][0] > crds[0])
                            crx_crds[0][0] = crds[0] + 2.5*min_sc_sep*h[0];
                        else
                            crx_crds[0][0] = crds[0] - 2.5*min_sc_sep*h[0];
                    }
                    newp = Point(crx_crds[0]);
                    insert_point_in_bond(newp, bindel, c);
                }
            break;
            case 2:
                sep_dist = _scaled_separation(crx_crds[0], crds, h, dim);
                sep_dist2 = _scaled_separation(crx_crds[1], crds, h, dim);
                if(sep_dist <= 2.0*min_sc_sep && sep_dist2 > 2.0*min_sc_sep)
                {
                    if(crx_dir[0] == 0)
                    {
                        if(crx_crds[0][1] > crds[1])
                            crx_crds[0][1] = crds[1] + 2.5*min_sc_sep*h[1];
                        else
                            crx_crds[0][1] = crds[1] - 2.5*min_sc_sep*h[1];
                    }
                    else
                    {
                        if(crx_crds[0][0] > crds[0])
                            crx_crds[0][0] = crds[0] + 2.5*min_sc_sep*h[0];
                        else
                            crx_crds[0][0] = crds[0] - 2.5*min_sc_sep*h[0];
                    }
                    newp = Point(crx_crds[0]);
                    insert_point_in_bond(newp, bindel, c);
                }
                else if(sep_dist > 2.0*min_sc_sep && sep_dist2 <= 2.0*min_sc_sep)
                {
                    if(crx_dir[1] == 0)
                    {
                        if(crx_crds[1][1] > crds[1])
                            crx_crds[1][1] = crds[1] + 2.5*min_sc_sep*h[1];
                        else
                            crx_crds[1][1] = crds[1] - 2.5*min_sc_sep*h[1];
                    }
                    else
                    {
                        if(crx_crds[1][0] > crds[0])
                            crx_crds[1][0] = crds[0] + 2.5*min_sc_sep*h[0];
                        else
                            crx_crds[1][0] = crds[0] - 2.5*min_sc_sep*h[0];
                    }
                    newp = Point(crx_crds[1]);
                    insert_point_in_bond(newp, bindel, c);
                }
                else if(sep_dist <= 2.0*min_sc_sep && sep_dist2 <= 2.0*min_sc_sep)
                {
                    len = _scaled_separation(crx_crds[0], Coords(bindel->start), h, dim);
                    len2 = _scaled_separation(crx_crds[1], Coords(bindel->start), h, dim);
                    if(crx_dir[0] == 0)
                    {
                        if(crx_crds[0][1] > crds[1])
                            crx_crds[0][1] = crds[1] + 2.5*min_sc_sep*h[1];
                        else
                            crx_crds[0][1] = crds[1] - 2.5*min_sc_sep*h[1];
                    }
                    else
                    {
                        if(crx_crds[0][0] > crds[0])
                            crx_crds[0][0] = crds[0] + 2.5*min_sc_sep*h[0];
                        else
                            crx_crds[0][0] = crds[0] - 2.5*min_sc_sep*h[0];
                    }
                    if(crx_dir[1] == 0)
                    {
                        if(crx_crds[1][1] > crds[1])
                            crx_crds[1][1] = crds[1] + 2.5*min_sc_sep*h[1];
                        else
                            crx_crds[1][1] = crds[1] - 2.5*min_sc_sep*h[1];
                    }
                    else
                    {
                        if(crx_crds[1][0] > crds[0])
                            crx_crds[1][0] = crds[0] + 2.5*min_sc_sep*h[0];
                        else
                            crx_crds[1][0] = crds[0] - 2.5*min_sc_sep*h[0];
                    }
                    if(len < len2)
                    {
                        newp = Point(crx_crds[0]);
                        insert_point_in_bond(newp, bindel, c);
                        newp = Point(crx_crds[1]);
                        insert_point_in_bond(newp, bindel->next, c);
                    }
                    else
                    {
                        newp = Point(crx_crds[1]);
                        insert_point_in_bond(newp, bindel, c);
                        newp = Point(crx_crds[0]);
                        insert_point_in_bond(newp, bindel->next, c);
                    }
                }
            break;
            }
        }

        if(want_debug)
        {
            printf("LEAVING reconstruct_bond_in_box\n"); 
            printf("BOND inside the BOX LU[%10.8f, %10.8f] [%10.8f, %10.8f]\n",
                 crds[0] - h[0]/2.0, crds[1] - h[1]/2.0,
                 crds[0] + h[0]/2.0, crds[1] + h[1]/2.0);
            for(b = (*bin); b != (*bout)->next; b = b->next)
                print_bond(b);
            printf("End of print BOND inside the BOX\n");
        }

        interpolate_intfc_states(intfc) = sav_intrp;
}

/* Due to the adjusting front on the boundary spacing,
 * strictly bond parallel cut line case might happen.
 */
LOCAL bool bond_parallel_cut_line(
        BOND    *b,
        int     dir,
        float   cut)
{
    if ((Coords(b->start)[dir] == cut) && (Coords(b->end)[dir] == cut))
        return YES;
    else
        return NO; 
}
/* shift bond which is parallel to the cut line. 
 * X direction, shift toward right.
 * Y direction, shift toward up. 
 * This function can not be applied to the rect. boundary curves, 
 * since these boundary curves are indeed parallel to cut line.
 */
LOCAL void shift_parallel_cut_line_bond( 
        INTERFACE   *intfc, 
        CURVE       *c, 
        BOND        *b,
        int         dir)
{
        RECT_GRID       *rgr = computational_grid(intfc);
        float           *h = rgr->h;
        int             dim = rgr->dim; 
        /*
        float           min_sc_sep = MIN_SC_SEP(intfc);        
        */
        float           eps = 1000.0*MACH_EPS;/*TOLERANCE*/
    
        /* make it machine distinguishable, but do not big enough to alter
         *  other bonds position. 
         */
        Coords(b->start)[dir] += eps; 
        Coords(b->end)[dir] += eps; 

        set_bond_length(b, dim); 
        set_bond_length(b->prev, dim); 
        set_bond_length(b->next, dim); 

        if(is_closed_curve(c) && (b == c->first || b == c->last))
        {
            if(b == c->first)
                set_bond_length(c->last, dim); 
            else
                set_bond_length(c->first, dim); 
        }
}

/*
*                       bond_crosses_cut_segment():
*
*       Determines if the given bond crosses the cut segments of a given
*       rectangle.  Returning a null pointer indicates that the
*       bond does not cross the rectangle boundary.
*       Otherwise, a pointer to the point contained inside the rectangle
*       is returned. If one end of the
*       bond lies on the cut line, it is said to cross only if the other
*       end lies on the side to be saved. This function is implimented
*       by using same logic as used in bond_crosses_cut_line().
*/
LIB_LOCAL   POINT   *bond_crosses_cut_segment(
        BOND    *b,
        int     dir,
        int     side,
        float   cut,
        float   L,
        float   U,
        float   *crs_crd)
{
        /* heavy version
        float   x1, y1, x2, y2;
        if(dir == 0)
        {
            x1 = cut; y1 = L;
            x2 = cut; y2 = U;
        }
        else
        {
             x1 = L; y1 = cut;
             x2 = U; y2 = cut;
        }
        */
  
        if (side == 0)
        {
            if (Coords(b->start)[dir] >= cut &&
                Coords(b->end)[dir] < cut)
            {
                if(point_on_cut_seg(crs_crd, b, 2, cut, L, U, dir)) 
                    return b->start;

                /* heavy version 
                if(cross_segments(Coords(b->start)[0],
                     Coords(b->start)[1],
                     Coords(b->end)[0],
                     Coords(b->end)[1],
                     x1, y1, x2, y2,
                     crs_crd))
                {
                    crs_crd[dir] = cut;
                    return b->start;
                }
                */
            }
            if (Coords(b->end)[dir] >= cut &&
                Coords(b->start)[dir] < cut)
            {
                if(point_on_cut_seg(crs_crd, b, 2, cut, L, U, dir)) 
                    return b->end;
                /* heavy version 
                if(cross_segments(Coords(b->start)[0],
                     Coords(b->start)[1],
                     Coords(b->end)[0],
                     Coords(b->end)[1],
                     x1, y1, x2, y2,
                     crs_crd))
                {
                    crs_crd[dir] = cut;
                    return b->end;
                }
                */
            }
        }
        else
        {
            if (Coords(b->start)[dir] <= cut &&
                Coords(b->end)[dir] > cut)
            {
                if(point_on_cut_seg(crs_crd, b, 2, cut, L, U, dir)) 
                    return b->start;
                /* heavy version 
                if(cross_segments(Coords(b->start)[0],
                     Coords(b->start)[1],
                     Coords(b->end)[0],
                     Coords(b->end)[1],
                     x1, y1, x2, y2,
                     crs_crd))
                {
                    crs_crd[dir] = cut;
                    return b->start;
                }
                */
            }
            if (Coords(b->end)[dir] <= cut &&
                Coords(b->start)[dir] > cut)
            {
                if(point_on_cut_seg(crs_crd, b, 2, cut, L, U, dir)) 
                    return b->end;
                /* heavy version
                if(cross_segments(Coords(b->start)[0],
                     Coords(b->start)[1],
                     Coords(b->end)[0],
                     Coords(b->end)[1],
                     x1, y1, x2, y2,
                     crs_crd))
                {
                    crs_crd[dir] = cut;
                    return b->end;
                }
                */
            }
        }
        crs_crd[0] = crs_crd[1] = -HUGE_VAL;
        return NULL;
}               /*end bond_crosses_cut_segment*/

LOCAL bool point_on_cut_seg(
        float           *cut_crds, 
        BOND            *b,     /* bond being clipped */
        int             dim,    /* For 2d Only  */
        float           cut,    /* coordinate of cut line */
        float           L,      /* Lower end of cut line */
        float           U,      /* Upper end of cut line */
        int             dir)    /* direction of cut line normal */
{
        float *crds_s = Coords(b->start);
        float *crds_e = Coords(b->end);
        float ps = crds_s[dir];
        float pe = crds_e[dir];
        float t;
        int   i;

        if (pe == ps)
        {
            for (i = 0; i < dim; ++i)
                cut_crds[i] = 0.5*(crds_s[i] + crds_e[i]);
        }
        else
        {
            t = (cut - ps)/(pe - ps);
            if (t < 0.0)
                t = 0.0;
            if (t > 1.0)
                t = 1.0;
            for (i = 0; i < dim; ++i)
                cut_crds[i] = (1.0 - t)*crds_s[i] + t*crds_e[i];
        }
        cut_crds[dir] = cut;   /* this value must be exact */

        if(cut_crds[(dir+1)%dim] <= U &&
           cut_crds[(dir+1)%dim] >= L)
            return YES;
        else
            return NO; 
}

      /* (x1,y1): start point of bond1,
       * (x2,y2): end point of bond1;
       * (x2,y2): start point of bond2,
       * (x3,y3): end point of bond2.
       * crs_crd: if cross, the crossing point coords.
       */
/**
// Moved to front/fcrosscur.c
LOCAL bool cross_segments(
        float       x1,
        float       y1,
        float       x2,
        float       y2,
        float       x3,
        float       y3,
        float       x4,
        float       y4,
        float       *crs_crd)
{
        double          nor_dist_t,nor_dist_s;
        double          sinth;          // sin of angle between bonds 
        double          xcross,ycross;  // coord of intersection
                                        //    of lines 12 34 
        double          x00,y00;        // beginning of long bond 
        double          x0,y0;          // beginning of short bond after
                                        //   coord translation 
        double          x,y;            // end of long bond after
                                        //   coord translation 
        double          dx,dy;          // short bond end - start 
        double          t;              // fractional distance on short bond 
        double          s;              // fractional distance on long bond 
        double          len12;          // length b1 * length b2 
        double          parallel = PARALLEL(current_interface());
        int             i;
        float           b1_len, b2_len;

        b1_len = sqrt(sqr(x1-x2) + sqr(y1-y2));
        b2_len = sqrt(sqr(x3-x4) + sqr(y3-y4));

        if (b1_len > b2_len)
        {
            x00 = x1;           y00 = y1;
            x0 = x3 - x1;       y0 = y3 - y1;
            x  = x2 - x1;       y  = y2 - y1;
            dx = x4 - x3;       dy = y4 - y3;
        }
        else
        {
            x00 = x3;           y00 = y3;
            x0 = x1 - x3;       y0 = y1 - y3;
            x  = x4 - x3;       y  = y4 - y3;
            dx = x2 - x1;       dy = y2 - y1;
        }
        sinth = dx*y - dy*x;
        nor_dist_t = x0*y - y0*x;
        nor_dist_s = dx*y0 - dy*x0;
        len12 = b1_len * b2_len;

        if (fabs(sinth) <= parallel * len12)
        {
            // Case of parallel lines 
            if (fabs(nor_dist_t) <= parallel * len12)
            {
                // Lines coincide 
                if (Between(x0,0.0,x) && Between(y0,0.0,y))
                {
                    // Cross at x0,y0 
                    crs_crd[0] = (float)(x0 + x00);
                    crs_crd[1] = (float)(y0 + y00);
                    return YES;
                }
                if (Between(x0+dx,0.0,x) && Between(y0+dy,0.0,y))
                {
                    // Cross at x0+dx,y0+dy 
                    crs_crd[0] = (float)(x0 + dx + x00);
                    crs_crd[1] = (float)(y0 + dy + y00);
                    return YES;
                }
                return NO; // No cross; line segments don't overlap 
            }
            return NO; // No cross; lines distinct although parallel 
        }

                // Now lines are not parallel 

        t = - nor_dist_t / sinth;
        s = nor_dist_s / sinth;
        if (t < 0.0 || t > 1.0 || s < 0.0 || s > 1.0)
            return NO;
        xcross = 0.5*(x0 + t*dx + s*x);
        ycross = 0.5*(y0 + t*dy + s*y);
        crs_crd[0] = (float)(xcross + x00);
        crs_crd[1] = (float)(ycross + y00);
        return YES;
}
***/

/*
LOCAL void remove_tangle_figure_on_boundary(
        INTERFACE   *intfc,
        float       cut,            // coordinate of cut line 
        int         dir,            // direction of cut line normal 
        int         side)           // side to be retained 
{
        CROSS           Cr, *cr;
        BOND            *b;
        CURVE           *c, **cc;
        CURVE           **curves;
        POINT           *newp;
        POINT           *p;
        int             num_deletes = 0;
        int             i;
        int             num_cr = 0;
        bool            sav_intrp;
        static POINT    *cut_p = NULL;

        if (cut_p == NULL)
            cut_p = Static_point(intfc);

        // Identify bonds crossing cut line 

        Cr.next = Cr.prev = NULL;
        cr = &Cr;
        for (cc = intfc->curves; cc && *cc; ++cc)
        {
            if(wave_type(*cc) < FIRST_PHYSICS_WAVE_TYPE)
                continue;
            for (b = (*cc)->first; b != NULL; b = b->next)
            {
                if ((p = bond_crosses_cut_line(b,dir,cut,side)) == NULL)
                        continue;

                // See note at top of function. 
                if (cr && cr->prev &&
                    b->prev && b->prev->prev && b->prev->prev->prev &&
                    ((cr->b1 == b->prev) || (cr->b1 == b->prev->prev)) &&
                    (cr->prev->b1 == b->prev->prev->prev) &&
                    cross_bonds(b,b->prev->prev->prev,cut_p))
                {
                    num_deletes = 3;
                }
                else if (cr && cr->prev &&
                         b->prev && b->prev->prev &&
                         (cr->b1 == b->prev) &&
                         (cr->prev->b1 == b->prev->prev) &&
                         cross_bonds(b,b->prev->prev,cut_p))
                {
                    num_deletes = 2;
                }
                if (num_deletes != 0)
                {
                    BOND        *newb;

                    Coords(cut_p)[dir] = cut;
                    newp = Point(Coords(cut_p));
                    sav_intrp = interpolate_intfc_states(intfc);
                    interpolate_intfc_states(intfc) = YES;
                    (void) insert_point_in_bond(newp,b,*cc);
                    interpolate_intfc_states(intfc) = sav_intrp;
                    cr = cr->prev;
                    cr->next = NULL;
                    newb = b->next;
                    for (i = 0; i < num_deletes; ++i)
                        (void) delete_start_of_bond(newb->prev,*cc);
                    b = newb->prev;
                    num_deletes = 0;
                    if ((p = bond_crosses_cut_line(b,dir,cut,side)) == NULL)
                        continue;
                }
                cr->next = (CROSS *)store(sizeof(CROSS));
                ++num_cr;
                cr->next->prev = cr;
                cr = cr->next;
                cr->c1 = *cc;
                cr->b1 = b;
                cr->p = p;
            }
        }
}
*/

/*
 * Adjust bond point passes through grid node (crossing point)
 * in the tolerance sense.
 */

/*
LOCAL void adjust_dist_of_grid_node_w_crx(
        INTERFACE   *intfc,
        float       cut,            // coordinate of cut line 
        int         dir,            // direction of cut line normal 
        int         use_side,       // if we use side param. to force symmtry
        int         side)
{
        RECT_GRID       *rgr = computational_grid(intfc);
        float           lbdry, ubdry; // lower boundary, upper boundary of the cut line
        int             dim = intfc->dim;
        BOND            *b, *b1, *b2;
        CURVE           *c, **cc;
        POINT           *newp, *p, *p1, *p2, *p3;
        float           *h = rgr->h;
        float           min_sc_sep = MIN_SC_SEP(intfc);
        const float     eps = 100.0*MACH_EPS;
        int             i;
        int             indx, tmpi, ic[MAXD], tmpic[MAXD];
        float           sep_dist, sep_dist2, sep_dist3;
        float           crds[MAXD], newcrds[MAXD];
        bool            sav_intrp;
        static POINT    *cut_p = NULL;
        int             dead_loop = 0;
        int             want_debug = NO;
        float           grid_tol, len, min_h;

        DEBUG_ENTER(adjust_dist_of_grid_node_w_crx)

        if(intfc != current_interface())
        {
            printf("ERROR in adjust_dist_of_grid_node_w_crx\n");
            printf("interface not current_interface\n");
            clean_up(ERROR);
        }

        min_h = rgr->h[0];
        for (i = 1; i < dim; ++i)
            if (rgr->h[i] < min_h)
                min_h = rgr->h[i];
        grid_tol = (min_sc_sep + min_sc_sep * 0.01)* min_h;
        //
        //  tolerance min_sc_sep*h is greater than grid_tolerance(),
        // no crxing point and grid node distance should be in this
        //  range. Because of set_subdomain_bounardy() and scatter_front.
        //
        if(dir == 0)
        {
            lbdry = rgr->L[1]-rgr->h[1];
            ubdry = rgr->U[1]+rgr->h[1];
        }
        else
        {
            lbdry = rgr->L[0]-rgr->h[0];
            ubdry = rgr->U[0]+rgr->h[0];
        }
        if (cut_p == NULL)
            cut_p = Static_point(intfc);

        if(want_debug)
            printf("CUT = %17.14f, dir[%d], lbdry[%g], ubdry[%g]\n",
                 cut, dir, lbdry, ubdry);

        sav_intrp = interpolate_intfc_states(intfc);
        interpolate_intfc_states(intfc) = YES;
        for (cc = intfc->curves; cc && *cc; ++cc)
        {
            if(wave_type(*cc) < FIRST_PHYSICS_WAVE_TYPE)
                continue;

            for (b = (*cc)->first; b != NULL; b = b->next)
            {
do_this_new_bond:
                p = NULL;
                if(use_side == NO)
                {
                    // We should not be biased by the side 
                    // Amr boundary, periodic boundary 
                    if ((Coords(b->start)[dir] >= cut) && (Coords(b->end)[dir] < cut))
                        p = b->start;
                    else if ((Coords(b->end)[dir] >= cut) && (Coords(b->start)[dir] < cut))
                        p = b->end;
                }
                else
                    p = bond_crosses_cut_line(b,dir,cut,side);
                if (p == NULL)
                    continue;

                point_on_cut_line(intfc,cut_p,b,cut,dir);

                // If the cut_p is outside the rect_grid (None-buffered),
                // we do not make any correction on this crxing point.
                //
                if(Coords(cut_p)[(dir+1)%dim] < lbdry ||
                   Coords(cut_p)[(dir+1)%dim] > ubdry)
                {
                    if(want_debug)
                    {
                        printf("Cut point [%17.15f, %17.15f]"
                            " outside the none-buffered rect_grid, omit \n",
                            Coords(cut_p)[0], Coords(cut_p)[1]);
                        print_bond(b);
                    }
                    continue;
                }

                ic[dir] = tmpic[dir] = cell_index(Coords(cut_p)[dir],dir,rgr);
                indx = ic[(dir+1)%dim] = tmpic[(dir+1)%dim] =
                     cell_index(Coords(cut_p)[(dir+1)%dim], (dir+1)%dim, rgr);
                sep_dist = HUGE_VAL;
                for(i = -1; i <= 1; i++)
                {
                    float tmpsep_dist;
                    tmpic[(dir+1)%dim] = indx + i;
                    tmpsep_dist = (sqr(Coords(cut_p)[0] - cell_edge(tmpic[0], 0, rgr)) +
                        sqr(Coords(cut_p)[1] - cell_edge(tmpic[1], 1, rgr)));
                    if(tmpsep_dist < sep_dist)
                    {
                        sep_dist = tmpsep_dist;
                        ic[(dir+1)%dim] = tmpic[(dir+1)%dim];
                    }
                }

                for(i = 0;i < dim; i++)
                    crds[i] = cell_edge(ic[i], i, rgr);

                // If distance of grid node and cut_p is greater than
                // the tolerance, we do not make any change.
                //
                sep_dist = _scaled_separation(crds, Coords(cut_p), h, dim);
                if(sep_dist > min_sc_sep)
                    continue;

                if(want_debug)
                {
                    printf("Grid Node Crds[%g, %g] ic[%d, %d]\n",
                        crds[0], crds[1], ic[0], ic[1]);
                    print_bond(b);
                    printf("Cut point [%17.15f, %17.15f], grid-cut_p dist = %17.15f\n",
                        Coords(cut_p)[0], Coords(cut_p)[1],
                        sqrt(sqr(crds[0] - Coords(cut_p)[0])+sqr(crds[1] - Coords(cut_p)[1])));
                }
                if(dead_loop++ > 40)
                {
                    printf("ERROR in adjust_dist_of_grid_node_w_crx\n");
                    printf("Dead loop formed\n");
                    printf("Grid Node Crds[%g, %g] ic[%d, %d]\n",
                        crds[0], crds[1], ic[0], ic[1]);
                    printf("Cut point [%17.15f, %17.15f], grid-cut_p dist = %17.15f\n",
                        Coords(cut_p)[0], Coords(cut_p)[1],
                        sqrt(sqr(crds[0] - Coords(cut_p)[0])+sqr(crds[1] - Coords(cut_p)[1])));
                    printf("The bond when dead is:\n");
                    print_bond(b);
                    printf("This bond is on curve\n");
                    print_curve(*cc);
                    clean_up(ERROR);
                }
                // If crxing is close to grid node, bond is short,
                //  delete bond end point. The bond length should be
                //  longer than 2*sqrt(2)*(min_sc_sep*h[0]).
                //
                p1 = b->start; p2 = b->end;
                sep_dist = scaled_separation(p1, p2, h, dim);
                if(sep_dist <= sqrt(8.0)*min_sc_sep)
                {
                    if(want_debug)
                        printf("Entered crxing is close to grid node, bond is short ");
                    if(p2 != (*cc)->last->end)
                    {
                        if(b != (*cc)->first)
                        {
                            b = b->prev;
                            if(want_debug)
                            {
                                printf("Delete bond end point:\n");
                                print_bond(b->next);
                            }
                            delete_end_of_bond(b->next,*cc);
                            goto do_this_new_bond;
                        }
                        else
                        {
                            // This curve start point coords close to cut line.
                            // Suggest to delete start point. (No closed curve ???)
                            //
                            printf("ERROR in adjust_dist_of_grid_node_w_crx\n");
                            printf("un-processed cases\n");
                            printf("Want to delete curve start point\n");
                            print_bond(b);
                            print_curve(*cc);
                            clean_up(ERROR);
                        }
                    }
                    else
                    {
                        // This is an closed curve, need to be handled 
                        // Or this curve end point coords close to cut line.
                        //
                        printf("ERROR in adjust_dist_of_grid_node_w_crx\n");
                        printf("un-processed cases\n");
                        print_curve(*cc);
                        clean_up(ERROR);
                    }
                }
                // If crxing is close to grid  node, bond points are p and p2.
                // p is closer to crossing PT than p2.
                // Crossing PT, grid node pt and p
                // are all very close, in a circle, centered at grid node pt,
                // radius grid_tol. We trim this bond length by half,
                // starting with point p.
                //
                if((sqr(Coords(cut_p)[0] - Coords(b->start)[0]) +
                    sqr(Coords(cut_p)[1] - Coords(b->start)[1])) <
                   (sqr(Coords(cut_p)[0] - Coords(b->end)[0]) +
                    sqr(Coords(cut_p)[1] - Coords(b->end)[1]))
                  )
                    p = b->start;
                else
                    p = b->end;
                sep_dist = _scaled_separation(crds, Coords(cut_p), h, dim);
                sep_dist2 = _scaled_separation(crds, Coords(p), h, dim);

                if(sep_dist < min_sc_sep && sep_dist2 < min_sc_sep)
                {
                    len = bond_length(b);
                    for(i = 0; i < dim; i++)
                        Coords(p)[i] = newcrds[i] = (Coords(b->start)[i] + Coords(b->end)[i])/2.0;
                    set_bond_length(b, dim);
                    if(p == b->start && b->prev != NULL)
                        set_bond_length(b->prev, dim);
                    if(p == b->end && b->next != NULL)
                        set_bond_length(b->next, dim);
                    if(want_debug && p == b->start)
                    {
                        printf("Entered crxing closer to b->start\n");
                        printf("New b->start pos[%17.14f, %17.14f]\n",
                                Coords(b->start)[0], Coords(b->start)[1]);
                    }
                    if(want_debug && p == b->end)
                    {
                        printf("Entered crxing closer to b->end\n");
                        printf("New b->end pos[%17.14f, %17.14f]\n",
                                Coords(b->end)[0], Coords(b->end)[1]);
                    }
                    if(b != (*cc)->first)
                    {
                        p1 = b->start; p2 = b->end;
                        sep_dist = scaled_separation(p1, p2, h, dim);
                        b = b->prev;
                        if(sep_dist <= sqrt(8.0)*min_sc_sep)
                        {
                            // The trimmed bond is short, delete b->end 
                            if(want_debug)
                            {
                                printf("Delete trimmed bond end point:\n");
                                print_bond(b->next);
                            }
                            delete_end_of_bond(b->next,*cc);
                        }
                        goto do_this_new_bond;
                    }
                    else
                    {
                        // If this is the case, consider to shift b point along the cut line 
                        printf("ERROR: n adjust_dist_of_grid_node_w_crx\n");
                        printf("Crxing very close to the curve startor end pt, un-processed cases\n");
                        printf("Grid Node Crds[%g, %g] ic[%d, %d]\n",
                            crds[0], crds[1], ic[0], ic[1]);
                        printf("Cut point [%17.15f, %17.15f], grid-cut_p dist = %17.15f\n",
                            Coords(cut_p)[0], Coords(cut_p)[1],
                            sqrt(sqr(crds[0] - Coords(cut_p)[0])+sqr(crds[1] - Coords(cut_p)[1])));
                        print_bond(b);
                        print_curve(*cc);
                        clean_up(ERROR);
                    }
                }
                // Now, the crossing pt is in the middle of the bond 
                // We insert a new point (with pos. of crxing pt) to
                // the bond, and shift this point away from the grid node.
                //
                if(want_debug)
                    printf("Crxing is close to grid node, and in the middle of Bond\n");
                for(i = 0; i < dim; i++)
                {
                    newcrds[i] = Coords(cut_p)[i];
                    if(i == (dir+1)%dim)
                    {
                        if(crds[i] > Coords(cut_p)[i] + eps)
                            newcrds[i] -= grid_tol;
                        else if(crds[i] < Coords(cut_p)[i] + eps)
                            newcrds[i] += grid_tol;
                        else
                        {
                            // The grid nodes and crxing are too close. We treat them as
                            // they overlap.
                            //
                            printf("Crxing overlaps grid node\n");
                            if(p == b->start)
                            {
                                if(Coords(b->end)[i] > crds[i])
                                    newcrds[i] += grid_tol;
                                else
                                    newcrds[i] -= grid_tol;
                            }
                            else if(p == b->end)
                            {
                                if(Coords(b->start)[i] > crds[i])
                                    newcrds[i] += grid_tol;
                                else
                                    newcrds[i] -= grid_tol;
                            }
                            else
                            {
                                printf("ERROR in adjust_dist_of_grid_node_w_crx\n");
                                printf("Crxing in the middle of bond, un-processed cases\n");
                                clean_up(ERROR);
                            }
                        }
                    }
                }
                if(want_debug)
                {
                    printf("Insert pt[%17.15f, %17.15f] to bond\n",
                             newcrds[0], newcrds[1]);
                    print_bond(b);
                }
                if(b != (*cc)->first)
                {
                    b1 = b->prev;
                    newp = Point(newcrds);
                    insert_point_in_bond(newp, b, *cc);
                    if(want_debug)
                    {
                        printf("After insert point b->prev, b, b->next are:\n");
                        print_bond(b->prev); print_bond(b);
                        print_bond(b->next);
                    }
                    b = b1;
                    continue;
                }
                else
                {
                    newp = Point(newcrds);
                    insert_point_in_bond(newp, b, *cc);
                    if(want_debug)
                    {
                        printf("Crxing is on the first bond of the curve\n");
                        printf("After insert point b->prev, b, b->next are:\n");
                        print_bond(b->prev); print_bond(b);
                        print_bond(b->next);
                    }
                    goto do_this_new_bond;
                }
            }
        }

        interpolate_intfc_states(intfc) = sav_intrp;

        DEBUG_LEAVE(adjust_dist_of_grid_node_w_crx)
}
*/

/* 
LOCAL void set_adjust_crxing_on_boundary_segment(
        INTERFACE   *intfc,
        float       cut,            
        int         dir,
        int         side)  
{
        BOND            *b;
        POINT           *p;
        CURVE           *c, **cc;
        CURVE           **curves;
        float           *h = computational_grid(intfc)->h;
        float           min_sc_sep = MIN_SC_SEP(intfc);
        const float     eps = 100*MACH_EPS;
        int             i, dim = intfc->dim;
        static POINT    *cut_p = NULL;
        int             indx, tmpi, ic[MAXD], tmpic[MAXD];
        int             old_used_cr_w_n = used_cr_w_n;
        float           sep_dist, sep_dist2;
        RECT_GRID       *rgr = computational_grid(intfc);
        float           grid_tol;

        DEBUG_ENTER(set_adjust_crxing_on_boundary_segment)
        grid_tol = grid_tolerance(rgr);
        if (cut_p == NULL)
            cut_p = Static_point(intfc);

        for (cc = intfc->curves; cc && *cc; ++cc)
        {
            if(wave_type(*cc) < FIRST_PHYSICS_WAVE_TYPE)
                continue;

            for (b = (*cc)->first; b != NULL; b = b->next)
            {
                if ((p = bond_crosses_cut_line(b,dir,cut,side)) == NULL)
                        continue;
                cr_w_n[used_cr_w_n].cr.c1 = *cc;
                cr_w_n[used_cr_w_n].cr.b1 = b;
                cr_w_n[used_cr_w_n].cr.p = p;
                cr_w_n[used_cr_w_n].dir = dir;
                cr_w_n[used_cr_w_n].cut = cut;
                cr_w_n[used_cr_w_n].used = NO;
                used_cr_w_n++;
                if(used_cr_w_n == alloc_cr_w_n)
                {
                    Cross_w_node  *new_cr_w_n;
                    alloc_cr_w_n += alloc_cr_w_n;
                    vector(&new_cr_w_n, alloc_cr_w_n, sizeof(Cross_w_node));
                    for(tmpi = 0; tmpi = used_cr_w_n; tmpi++)
                    {
                        new_cr_w_n[tmpi].cr.c1 = cr_w_n[tmpi].cr.c1;
                        new_cr_w_n[tmpi].cr.b1 = cr_w_n[tmpi].cr.b1;
                        new_cr_w_n[tmpi].cr.p = cr_w_n[tmpi].cr.p;
                        free(cr_w_n);
                        cr_w_n = new_cr_w_n;
                    }
                }
            }
        }

        for(tmpi = old_used_cr_w_n; tmpi < used_cr_w_n; tmpi++)
        {
            b = cr_w_n[tmpi].cr.b1;
            c = cr_w_n[tmpi].cr.c1;
            p = cr_w_n[tmpi].cr.p;

            point_on_cut_line(intfc,cut_p,b,cut,dir);

            if((sqr(Coords(cut_p)[0] - Coords(b->start)[0]) +
                sqr(Coords(cut_p)[1] - Coords(b->start)[1])) <
               (sqr(Coords(cut_p)[0] - Coords(b->end)[0]) +
                sqr(Coords(cut_p)[1] - Coords(b->end)[1]))
              )
                cr_w_n[tmpi].p = b->start;
            else
                cr_w_n[tmpi].p = b->end;

            ic[dir] = tmpic[dir] = cell_index(Coords(cut_p)[dir],dir,rgr);
            indx = ic[(dir+1)%dim] = tmpic[(dir+1)%dim] =
                     cell_index(Coords(cut_p)[(dir+1)%dim], (dir+1)%dim, rgr);
            sep_dist = HUGE_VAL;
            for(i = -1; i <= 1; i++)
            {
                float tmpsep_dist;
                tmpic[(dir+1)%dim] = indx + i;
                tmpsep_dist = (sqr(Coords(cut_p)[0] - cell_edge(tmpic[0], 0, rgr)) +
                        sqr(Coords(cut_p)[1] - cell_edge(tmpic[1], 1, rgr)));
                if(tmpsep_dist < sep_dist)
                {
                    sep_dist = tmpsep_dist;
                    ic[(dir+1)%dim] = tmpic[(dir+1)%dim];
                }
            }
            for(i = 0; i < dim; i++)
                cr_w_n[tmpi].ic[i] = ic[i];
            for(i = 0; i < dim; i++)
                cr_w_n[tmpi].cut_p[i] = Coords(cut_p)[i];
        }

        DEBUG_LEAVE(set_adjust_crxing_on_boundary_segment)
}

LOCAL void print_adjust_crxing_on_boundary_segment(
        INTERFACE     *intfc)
{
        BOND            *b;
        POINT           *p;
        CURVE           *c;
        int             i, dim = intfc->dim, dir;
        const float     eps = 100*MACH_EPS; 
        float           min_sc_sep = MIN_SC_SEP(intfc);  
        float           cut_p[MAXD], crds[MAXD];
        float           sep_dist, sep_dist2, cut;
        int             tmpi, ic[MAXD];
        RECT_GRID       *rgr = computational_grid(intfc);
        float           *h = rgr->h;
        float           grid_tol;

        grid_tol = grid_tolerance(rgr);
        for(tmpi = 0; tmpi < used_cr_w_n; tmpi++)
        {
            b = cr_w_n[tmpi].cr.b1;
            c = cr_w_n[tmpi].cr.c1;
            p = cr_w_n[tmpi].cr.p;
            dir = cr_w_n[tmpi].dir;
            cut = cr_w_n[tmpi].cut;
            for(i = 0;i < dim; i++)
            {
                ic[i] = cr_w_n[tmpi].ic[i];
                cut_p[i] = cr_w_n[tmpi].cut_p[i];
                crds[i] = cell_edge(ic[i], i, rgr);

            }
            (void) printf("PRINT CROSSS TO BE PROCESSED :\n");
            print_cross(&(cr_w_n[tmpi].cr));

            printf("The ic[%d, %d]: crds[%17.15f, %17.15f], cut_p[%17.15f, %17.15f]\n",
                    ic[0], ic[1], crds[0], crds[1],cut_p[0], cut_p[1]);
            if( (cut_p[0] == Coords(b->start)[0] &&
                 cut_p[1] == Coords(b->start)[1]) ||
                (cut_p[0] == Coords(b->end)[0] &&
                 cut_p[1] == Coords(b->end)[1])
              )
                printf("CUT POINT pass [bond start or end point], ");
            else
                printf("CUT POINT ");

            if(cut_p[0] == crds[0] && cut_p[1] == crds[1])
                 printf("[grid node], ");

            if(dir == 0)
            {
                 if(cut_p[0] > cut)
                     printf("[Right side], ");
                 else if(cut_p[0] < cut)
                     printf("[Left side], ");
                 else
                     printf("[On vertical line], ");
            }
            if(dir == 1)
            {
                 if(cut_p[1] > cut)
                     printf("[Upper side], ");
                 else if(cut_p[1] < cut)
                     printf("[Lower side], ");
                 else
                     printf("[On horizontal line], ");
            }
            sep_dist = _scaled_separation(crds, cut_p, h, dim);
            if(sep_dist < min_sc_sep)
                printf("[Close to grid node]\n");
            else
                printf("DIST = %17.15f\n", sep_dist);
            sep_dist = distance_between_positions(Coords(b->start), cut_p, dim);
            sep_dist2 = distance_between_positions(Coords(b->end), cut_p, dim);
            if(sep_dist < grid_tol)
                printf("B->START close to CUT_P\n");
            if(sep_dist2 < grid_tol)
                printf("B->END close to CUT_P\n");
        }
}
*/

#endif /* if defined(SMOOTH_INTERFACE_FOR_SCATTER)*/
#endif /* if defined(TWOD) */

