/*
*                               fpatchplot.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*
*       this file includes functions set_patch_front, set_patch_topo_grid,
*                       and set_patch_comput_grid
*/

#define DEBUG_STRING    "fpatchplot"

#include <front/flocaldecs.h>

#if defined(USE_OVERTURE)

#if defined(THREED)
LOCAL   void   geomview_amr_fronts_plot3d(const char*,Front**,int,Wv_on_pc**,int);
LOCAL   void   gview_plot_box3d(FILE*,float*,float*,float*,int);  
LOCAL   void   gview_plot_grid3d(FILE*,RECT_GRID*,float*); 
LOCAL   void   write_interpolated_color(FILE*,SURFACE_COLOR,SURFACE_COLOR,float,float);
LOCAL   void   gview_plot_trisurfaces(FILE*,INTERFACE*,RECT_GRID*,
                      bool,bool,SURFACE_COLOR,SURFACE_COLOR);
#endif /* if defined(THREED) */

#if defined(TWOD)
LOCAL  void    geomview_amr_fronts_plot2d(const char*,Front**,int,Wv_on_pc**,int);
LOCAL   void   gview_plot_box2d(FILE*,float*,float*,float*);  
LOCAL   void   gview_plot_grid2d(FILE*,RECT_GRID*,float*); 
LOCAL   void   gview_plot_curve2d(FILE*,CURVE*,float*);  
#endif /* if defined(TWOD) */

EXPORT void geomview_amr_fronts_plot(
        const char    *dname,
        Front         **frs,
        int           num_patches,
        Wv_on_pc      **redistr_table,
        int           max_n_patch)
{
        int dim = frs[0]->rect_grid->dim;

        switch(dim)
        {
        case 2:
#if defined(TWOD)
            geomview_amr_fronts_plot2d(dname, frs, 
                   num_patches, redistr_table, max_n_patch);
#endif /* if defined(TWOD) */
        break;
        case 3:
#if defined(THREED)
            geomview_amr_fronts_plot3d(dname, frs, 
                   num_patches, redistr_table, max_n_patch);
#endif /* if defined(THREED) */
        break;
        }
}

#if defined(THREED)
LOCAL void geomview_amr_fronts_plot3d(
        const char    *dname,
        Front         **frs,
        int           num_patches,
        Wv_on_pc      **redistr_table,
        int           max_n_patch)
{
        FILE              *fp;
        int               i; 
        char              fmt[256];
        static const char *indent = "    ";
        static char       *fname = NULL, *ppfname = NULL;
        static size_t     fname_len = 0, ppfname_len = 0;
        INTERFACE         *intfc = frs[0]->interf; 
        INTERFACE         *tmpintfc;  
        CURVE             **c;
        INTERFACE         *sav_intfc;
        bool              sav_copy; 
        static float      **clrmap = NULL; 
        float             ccolor[4] = {0.0, 0.0, 0.0, 1.0};
        int               myid, numnodes;
        const char        *nstep;
        char              outname[256],outdir[256];

        myid = pp_mynode(); numnodes = pp_numnodes();
        sprintf(outdir,"%s/%s",dname,"geomintfc");
        ppfname = set_ppfname(ppfname,"intfc",&ppfname_len);
        nstep = right_flush(frs[0]->step,7);
        sprintf(outname,"%s.ts%s",ppfname,nstep);

        if (create_directory(dname,YES) == FUNCTION_FAILED)
        {
            (void) printf("WARNING in geomview_intfc_plot2d(), directory "
                          "%s doesn't exist and can't be created\n",dname);
            return;
        }
        if (create_directory(outdir,YES) == FUNCTION_FAILED)
        {
            (void) printf("WARNING in geomview_intfc_plot2d(), directory "
                         "%s doesn't exist and can't be created\n",outdir);
            return;
        }
        if(clrmap == NULL)
        {
            vector(&clrmap,6,sizeof(float*));
            for(i = 0; i < 6; i++)
                vector(&clrmap[i],4,sizeof(float));
            set_default_clr_map(clrmap, 6);
        }

        fname = get_list_file_name(fname,outdir,outname,&fname_len);
        if ((fp = fopen(fname,"w")) == NULL)
        {
            (void) printf("WARNING in gview_plot_intfc3d(), "
                           "can't open %s\n",fname);
            return;
        }

        /* beginning of writting Vect to file */
        fprintf(fp,"{ LIST \n");
        gview_plot_trisurfaces(fp, intfc, frs[0]->rect_grid, YES, NO, pRED, pRED);
        gview_plot_trisurfaces(fp, intfc, frs[0]->rect_grid, YES, YES, pBLUE, pGREEN);

        for(int i = 0; i < num_patches; i++)
        {
           int use_clr, width;
           if(NULL != redistr_table && pp_numnodes() != 1)
               use_clr = redistr_table[myid][i].pc_id % 6;
           else
               use_clr = i%6;
           /* 
           gview_plot_grid3d(fp, frs[i]->rect_grid, clrmap[use_clr]);
           */
           if(frs[i]->patch_level == 0)
               width = 1;
           else
               width = frs[i]->patch_level*2;
           gview_plot_box3d(fp, frs[i]->rect_grid->L, 
               frs[i]->rect_grid->U, clrmap[use_clr], width);
        }

        /* end of LIST OBJ */
        fprintf(fp,"}\n");
        fclose(fp);
}

LOCAL void gview_plot_box3d(
	FILE      *fp,
        float     *L,
        float     *U,
        float     *color,
        int       width)
{
        fprintf(fp,"{appearance{linewidth %d}\n", width);
        fprintf(fp," VECT\n");
        fprintf(fp,"%1d %1d %1d\n", 6, 18, 6);
        fprintf(fp,"%2d %2d %2d %2d %2d %2d\n", 5, 5, 2, 2, 2, 2);
        fprintf(fp,"%2d %2d %2d %2d %2d %2d\n", 1, 1, 1, 1, 1, 1);
        (void) fprintf(fp,"%f %f %f \n",L[0],L[1],L[2]);
        (void) fprintf(fp,"%f %f %f \n",U[0],L[1],L[2]);
        (void) fprintf(fp,"%f %f %f \n",U[0],U[1],L[2]);
        (void) fprintf(fp,"%f %f %f \n",L[0],U[1],L[2]);
        (void) fprintf(fp,"%f %f %f \n",L[0],L[1],L[2]);

        (void) fprintf(fp,"%f %f %f \n",L[0],L[1],U[2]);
        (void) fprintf(fp,"%f %f %f \n",U[0],L[1],U[2]);
        (void) fprintf(fp,"%f %f %f \n",U[0],U[1],U[2]);
        (void) fprintf(fp,"%f %f %f \n",L[0],U[1],U[2]);
        (void) fprintf(fp,"%f %f %f \n",L[0],L[1],U[2]);

        (void) fprintf(fp,"%f %f %f \n",L[0],L[1],L[2]);
        (void) fprintf(fp,"%f %f %f \n",L[0],L[1],U[2]);

        (void) fprintf(fp,"%f %f %f \n",U[0],L[1],L[2]);
        (void) fprintf(fp,"%f %f %f \n",U[0],L[1],U[2]);

        (void) fprintf(fp,"%f %f %f \n",U[0],U[1],L[2]);
        (void) fprintf(fp,"%f %f %f \n",U[0],U[1],U[2]);

        (void) fprintf(fp,"%f %f %f \n",L[0],U[1],L[2]);
        (void) fprintf(fp,"%f %f %f \n",L[0],U[1],U[2]);

        fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
        fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
        fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
        fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
        fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
        fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
        fprintf(fp,"}\n");
}

LOCAL void gview_plot_grid3d(
	FILE      *fp,
        RECT_GRID *gr, 
        float     *color)
{
        int           i, j;
        int           imin[MAXD], imax[MAXD];
        int           *gmax = gr->gmax;
        int           n_x = 0, n_y = 0, n_z = 0;
        float         *L = gr->L; 
        float         *U = gr->U;  
        float         crds1[MAXD+1], crds2[MAXD+1];
        float         dh[MAXD]; 

        for (i = 0; i < gr->dim; i++)
        {
            imin[i] = 0;
            imax[i] = gmax[i];
            dh[i] = gr->h[i];  
        }

        n_x = imax[0]-imin[0]+1;
        n_y = imax[1]-imin[1]+1;
        n_z = imax[2]-imin[2]+1;
 
        /* Draw all x-y plane */
        for(j = 0; j <= gmax[2]; j++)
        { 
            crds1[2] = crds2[2] = L[2] + j*dh[2];
            for(i=0; i<n_x; i++)
            {

                crds1[0] = crds2[0] = L[0]+(imin[0]+i)*dh[0];
                crds1[1] = L[1];
                crds2[1] = U[1];

                fprintf(fp,"{ VECT\n");
                fprintf(fp,"%1d %1d %1d\n", 1, 2, 1);
                fprintf(fp,"%1d %1d\n", 2, 1);
 
                fprintf(fp,"%f %f %f \t",crds1[0],crds1[1],crds1[2]);
                fprintf(fp,"%f %f %f \t",crds2[0],crds2[1],crds2[2]);
                fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
                fprintf(fp,"}\n");
            }
 
            for(i=0; i<n_y; i++)
            {
                crds1[0] = L[0];
                crds1[1] = crds2[1] = L[1]+(imin[1]+i)*dh[1];
                crds2[0] = U[0];

                fprintf(fp,"{ VECT\n");
                fprintf(fp,"%1d %1d %1d\n", 1, 2, 1);
                fprintf(fp,"%1d %1d\n", 2, 1);
                fprintf(fp,"%f %f %f \t",crds1[0],crds1[1],crds1[2]);
                fprintf(fp,"%f %f %f \t",crds2[0],crds2[1],crds2[2]);
                fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
                fprintf(fp,"}\n");
            }
        }

        /* Draw all x-z plane */
        for(j = 0; j <= gmax[1]; j++)
        { 
            crds1[1] = crds2[1] = L[1] + j*dh[1];
            for(i=0; i<n_x; i++)
            {
                crds1[0] = crds2[0] = L[0]+(imin[0]+i)*dh[0];
                crds1[2] = L[2];
                crds2[2] = U[2];

                fprintf(fp,"{ VECT\n");
                fprintf(fp,"%1d %1d %1d\n", 1, 2, 1);
                fprintf(fp,"%1d %1d\n", 2, 1);
 
                fprintf(fp,"%f %f %f \t",crds1[0],crds1[1],crds1[2]);
                fprintf(fp,"%f %f %f \t",crds2[0],crds2[1],crds2[2]);
                fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
                fprintf(fp,"}\n");
            }
        }
}  

LOCAL   void    gview_plot_trisurfaces(
        FILE          *file, 
        INTERFACE     *intfc,
        RECT_GRID     *gr,
        bool          clip,
        bool          bdry,
        SURFACE_COLOR color1,
        SURFACE_COLOR color2)
{
        POINT             *p;
        SURFACE           **s;
        TRI               *tri;
        bool              plot_surf;
        float             D, intensity = .5;
        float             L[MAXD],U[MAXD],tol[MAXD];
        float             *crds;
        int               num_surfs, num_tris, i, j, k, l;
        int               npts, ntris;
        static const char *indent = "    ";
        static float      *pts = NULL;
        static int        *verts = NULL;
        static int        alloc_len_verts = 0, alloc_len_pts = 0;

        for (num_tris = 0, s = intfc->surfaces; s && *s; ++s)
        {
            num_tris += (*s)->num_tri;
            for (tri=first_tri(*s); !at_end_of_tri_list(tri,*s); tri=tri->next)
            {
                for (k = 0; k < 3; ++k)
                    Index_of_point(Point_of_tri(tri)[k]) = -1;
            }
        }


        if (alloc_len_pts < 3*intfc->num_points)
        {
            if (pts != NULL)
                free(pts);
            alloc_len_pts = 3*intfc->num_points;
            vector(&pts,alloc_len_pts,FLOAT);
        }
        if (alloc_len_verts < 4*num_tris)
        {
            if (verts != NULL)
                free(verts);
            alloc_len_verts = 4*num_tris;
            vector(&verts,alloc_len_verts,INT);
        }
        for (i = 0; i < 3; i++)
        {
            L[i] = gr->L[i] - 0.5*gr->h[i];
            U[i] = gr->U[i] + 0.5*gr->h[i];
            tol[i] = 0.00001*gr->h[i];
        }

        for (npts=0, ntris=0, num_surfs=0, s = intfc->surfaces; s && *s; ++s)
        {
            if (bdry == YES  &&  !Boundary(*s))
                continue;
            if (bdry == NO  &&  Boundary(*s))
                continue;
            if (clip == YES)
            {
                plot_surf = NO;
                for (tri = first_tri(*s); !at_end_of_tri_list(tri,*s);
                     tri = tri->next)
                {
                    for (k = 0; k < 3; ++k)
                    {
                        crds = Coords(Point_of_tri(tri)[k]);
                        for (l = 0; l < 3; ++l)
                            if ((crds[l] < L[l] - tol[l]) ||
                                (U[l] + tol[l] < crds[l]))
                                break;
                        if (l < 3)
                            break;
                    }
                    if (k == 3)
                    {
                        plot_surf = YES;
                        for (k = 0; k < 3; ++k)
                        {
                            p = Point_of_tri(tri)[k];
                            if (Index_of_point(p) == -1)
                            {
                                crds = Coords(p);
                                for (l = 0; l < 3; ++l)
                                    pts[3*npts+l] = crds[l];
                                Index_of_point(p) = npts++;
                            }
                            verts[4*ntris+k] = Index_of_point(p);
                        }
                        verts[4*ntris+3] = num_surfs;
                        ++ntris;
                    }
                }
                if (plot_surf == YES)
                    ++num_surfs;
            }
            else
            {
                for (tri = first_tri(*s); !at_end_of_tri_list(tri,*s);
                     tri = tri->next)
                {
                    for (k = 0; k < 3; ++k)
                    {
                        p = Point_of_tri(tri)[k];
                        if (Index_of_point(p) == -1)
                        {
                            crds = Coords(p);
                            for (l = 0; l < 3; ++l)
                                pts[3*npts+l] = crds[l];
                            Index_of_point(p) = npts++;
                        }
                        verts[4*ntris+k] = Index_of_point(p);
                    }
                    verts[4*ntris+3] = num_surfs;
                    ++ntris;
                }
                ++num_surfs;
            }
        }
        if (num_surfs == 0)
            return;

        (void) fprintf(file,"%s{\n%s%sOFF\n%s%s%6d %6d %6d\n",indent,
                       indent,indent,indent,indent,npts,ntris,0);
        for (i = 0; i < npts; ++i)
        {
            (void) fprintf(file,"%s%s%-9g %-9g %-9g\n",indent,indent,
                           pts[3*i],pts[3*i+1],pts[3*i+2]);
        }
        D = (num_surfs == 1) ? 1.0 : 1/(num_surfs - 1.0);
        for (j = 0; j < ntris; ++j)
        {
            (void) fprintf(file,"%s%s%-4d %-4d %-4d %-4d ",indent,indent,
                           3,verts[4*j],verts[4*j+1],verts[4*j+2]);
            write_interpolated_color(file,color1,color2,verts[4*j+3]/D,
                                     intensity);
        }
        (void) fprintf(file,"%s}\n",indent);
}

 /* copied from igview.c, should unify */
LOCAL void write_interpolated_color(
        FILE          *file,
        SURFACE_COLOR color1,
        SURFACE_COLOR color2,
        float         d,
        float         intensity)
{
        float color[8][4] =
        {
            { 0.0, 0.0, 0.0, 0.75 }, /* black     */
            { 1.0, 0.0, 0.0, 0.75 }, /* red       */
            { 0.0, 1.0, 0.0, 0.75 }, /* green     */
            { 1.0, 1.0, 0.0, 0.75 }, /* yellow    */
            { 0.0, 0.0, 1.0, 0.75 }, /* blue      */
            { 1.0, 0.0, 1.0, 0.75 }, /* magenta   */
            { 0.0, 1.0, 1.0, 0.75 }, /* cyan      */
            { 1.0, 1.0, 1.0, 0.75 }, /* white     */
        };

        float write_color;
        int i;

        for (i = 0 ; i < 4 ; ++i)
        {
            write_color = ((1.0 - d)*color[color1][i] + d*color[color2][i])*
                          intensity;
            (void) fprintf(file,"%7.5f ",write_color);
        }
        (void) fprintf(file,"\n");
}               /* end write_interpolated_color */

#endif /* if defined(THREED) */

#if defined(TWOD)
LOCAL void geomview_amr_fronts_plot2d(
        const char    *dname,
        Front         **frs,
        int           num_patches,
        Wv_on_pc      **redistr_table,
        int           max_n_patch)
{
        FILE              *fp;
        int               i; 
        char              fmt[256];
        static const char *indent = "    ";
        static char       *fname = NULL, *ppfname = NULL;
        static size_t     fname_len = 0, ppfname_len = 0;
        INTERFACE         *intfc = frs[0]->interf; 
        INTERFACE         *tmpintfc;  
        CURVE             **c;
        INTERFACE         *sav_intfc;
        bool              sav_copy; 
        static float      **clrmap = NULL; 
        float             ccolor[4] = {0.0, 0.0, 0.0, 1.0};
        int               myid, numnodes;
        const char        *nstep;
        char              outname[256],outdir[256];

        myid = pp_mynode(); numnodes = pp_numnodes();
        sprintf(outdir,"%s/%s",dname,"geomintfc"); 
        ppfname = set_ppfname(ppfname,"intfc",&ppfname_len);
        nstep = right_flush(frs[0]->step,7);
        sprintf(outname,"%s.ts%s",ppfname,nstep);

        if (create_directory(dname,YES) == FUNCTION_FAILED)
        {
            (void) printf("WARNING in geomview_intfc_plot2d(), directory "
                          "%s doesn't exist and can't be created\n",dname);
            return;
        }
        if (create_directory(outdir,YES) == FUNCTION_FAILED)
        {
            (void) printf("WARNING in geomview_intfc_plot2d(), directory "
                         "%s doesn't exist and can't be created\n",outdir);
            return;
        }

        if(clrmap == NULL)
        {
            vector(&clrmap,6,sizeof(float*));  
            for(i = 0; i < 6; i++)
                vector(&clrmap[i],4,sizeof(float)); 
            set_default_clr_map(clrmap, 6); 
        }

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_size_of_intfc_state(size_of_state(intfc));
        set_copy_intfc_states(NO);
        tmpintfc = copy_interface(intfc);
  
        clip_to_interior_region(tmpintfc,
            frs[0]->rect_grid->lbuf,frs[0]->rect_grid->ubuf); 
 
        fname = get_list_file_name(fname,outdir,outname,&fname_len); 
        if ((fp = fopen(fname,"w")) == NULL)
        {
            (void) printf("WARNING in gview_plot_intfc2d(), "
                           "can't open %s\n",fname);
            delete_interface(tmpintfc);  
            set_current_interface(sav_intfc);
            set_copy_intfc_states(sav_copy);
            return;
        }

        /* beginning of writting Vect to file */
        fprintf(fp,"{ LIST \n");
        for(c = tmpintfc->curves; c and *c;  c++)
            gview_plot_curve2d(fp,*c,ccolor); 

        for(int i = 0; i < num_patches; i++)
        {
           int use_clr;  
           /* 
           gview_plot_box2d(fp, frs[i]->rect_grid->L,
                frs[i]->rect_grid->U,clrmap[i%3]); 
           */ 
           if(NULL != redistr_table && pp_numnodes() != 1)
               use_clr = redistr_table[myid][i].pc_id % 6;
           else
               use_clr = i % 6;  
           gview_plot_grid2d(fp,frs[i]->rect_grid,clrmap[use_clr]);  
        }  

        /* end of LIST OBJ */
        fprintf(fp,"}\n");
        fclose(fp); 

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);
        delete_interface(tmpintfc);  

}

LOCAL void gview_plot_box2d(
	FILE      *fp,
        float     *L,
        float     *U,
        float     *color)
{
        fprintf(fp,"{ VECT\n");
        fprintf(fp,"%1d %1d %1d\n", 1, 5, 1);
        fprintf(fp,"%1d\n%1d\n", 5,1);
        (void) fprintf(fp,"%f %f %f \t",L[0],L[1],1.0);
        (void) fprintf(fp,"%f %f %f \t",U[0],L[1],1.0);
        (void) fprintf(fp,"%f %f %f \t",U[0],U[1],1.0);
        (void) fprintf(fp,"%f %f %f \t\n",L[0],U[1],1.0);
        (void) fprintf(fp,"%f %f %f \t",L[0],L[1],1.0);
        fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
        fprintf(fp,"}\n");

}  

LOCAL void gview_plot_curve2d(
        FILE      *fp,
        CURVE     *c,
        float     *color)
{
            BOND *b;

            b = c->first;
            fprintf(fp,"{ \n");
            fprintf(fp,"VECT\n");
            fprintf(fp,"%1d %1d %1d\n", 1, c->num_points, 1);
            fprintf(fp,"%1d\n%1d\n", c->num_points,1);

            while (b)
            {
                fprintf(fp,"%f %f %f \t",  Coords(b->start)[0],
                                Coords(b->start)[1] , 1.0);
                if (b == c->last)
                    break;
                b = b->next;
            }
            fprintf(fp,"%f %f %f \t\n",  Coords(b->end)[0],
                  Coords(b->end)[1] , 1.0);
            fprintf(fp,"%f %f %f %f \t\n",color[0],color[1],color[2],color[3]);
            fprintf(fp,"}\n");

}

LOCAL void gview_plot_grid2d(
	FILE      *fp,
        RECT_GRID *gr, 
        float     *color)
{
        int           i, j;
        int           imin[MAXD], imax[MAXD];
        int           *gmax = gr->gmax;
        int           n_x = 0, n_y = 0, num;
        float         *L = gr->L; 
        float         *U = gr->U;  
        float         crds1[MAXD+1], crds2[MAXD+1];
        float         dh[MAXD]; 

        for (i = 0; i < gr->dim; i++)
        {
            imin[i] = 0;
            imax[i] = gmax[i];
            dh[i] = gr->h[i];  
        }

        n_x = imax[0]-imin[0]+1;
        n_y = imax[1]-imin[1]+1;
 
        for(i=0; i<n_x; i++)
        {
            crds1[0] = crds2[0] = L[0]+(imin[0]+i)*dh[0];
            crds1[1] = L[1];
            crds2[1] = U[1];

            fprintf(fp,"{ VECT\n");
            fprintf(fp,"%1d %1d %1d\n", 1, 2, 1);
            fprintf(fp,"%1d %1d\n", 2, 1);
 
            fprintf(fp,"%f %f %f \t",crds1[0],crds1[1],1.0);
            fprintf(fp,"%f %f %f \t",crds2[0],crds2[1],1.0);
            fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
            fprintf(fp,"}\n");
        }
 
        for(i=0; i<n_y; i++)
        {
            crds1[0] = L[0];
            crds1[1] = crds2[1] = L[1]+(imin[1]+i)*dh[1];
            crds2[0] = U[0];

            fprintf(fp,"{ VECT\n");
            fprintf(fp,"%1d %1d %1d\n", 1, 2, 1);
            fprintf(fp,"%1d %1d\n", 2, 1);
            fprintf(fp,"%f %f %f \t",crds1[0],crds1[1],1.0);
            fprintf(fp,"%f %f %f \t",crds2[0],crds2[1],1.0);
            fprintf(fp,"%f %f %f %f \t\n", 
                  color[0], color[1], color[2], color[3]);
            fprintf(fp,"}\n");
        }

}  
#endif /* if defined(TWOD) */

#endif  /* if defined(USE_OVERTURE) */

