/*
*                               fpatch2d.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*
*       this file includes functions set_patch_front, set_patch_topo_grid,
*                       and set_patch_comput_grid
*/

#define DEBUG_STRING    "fpatch2d"

#include <front/flocaldecs.h>

#if defined(TWOD)  
#if defined(USE_OVERTURE)

LOCAL  int       use_delete_short_bonds = YES;  

LOCAL  bool      pseudo_scatter_patch_fronts(Front**,Front**,
                      Front*,int,Wv_on_pc**,int,COMPONENT*,int);
LOCAL  bool 	 set_amr_subdomain_boundary(Front*,COMPONENT);  
LOCAL  void      amr_clip_to_interior_region(INTERFACE*,int*,int*,int); 
LOCAL  int       merge_fronts_ver2(Front*,Front**,int); 
/**** THE new added functions */
LOCAL  void      set_copy_frs_intfc_null_ver2(Front**,Wv_on_pc**,int,int); 
/**** ENd of THE new added functions */
/*
LOCAL  void      old_reinstall_undistribute_fronts_and_misc(Front**,Front**,
                   Wv_on_pc**,int,int,int);
*/

/**** THE new added functions */
LOCAL  int       set_copy_proc_frs_ver3(Front***,Front**,int,Wv_on_pc**,int); 
LOCAL  void      set_copy_frs_intfc_null_ver3(Front**,Wv_on_pc**,int,int); 
LOCAL  bool      distribute_coarse_patch_fronts(Front**,Front**,Front*,int,Wv_on_pc**,int); 
/**** ENd of THE new added functions */


/*     This is the old version of assembly_fine_patch_fronts_to_one. 
LOCAL  int       assembly_fine_patch_fronts_to_one2d(Front**,Front*);
LOCAL  void      remove_dup_nodes_from_assembly_intfc(INTERFACE*); 
LOCAL  void      retrieve_boundary_curves_from_zero(Front*,INTERFACE*);
*/

LIB_LOCAL void delete_intfc_all_curves(
        INTERFACE       *intfc)
{
        INTERFACE       *sav_intfc;
        NODE            **n;
        CURVE           **c;
        CURVE           **delete_curves = NULL;
        NODE            **delete_nodes = NULL;
        bool            sav_copy;

        DEBUG_ENTER(delete_intfc_all_curves)

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();

        /* remove subdomain bdry curves and physical boundary curves */
        for(c = intfc->curves; c && *c; c++)
        {
            if (! add_to_pointers((POINTER)*c,(POINTER**)&delete_curves))
            {
                screen("ERROR in delete_patch_all_curves(), "
                    "add_to_pointers() failed\n");
                clean_up(ERROR);
            }
        }
        for (c = delete_curves; c && *c; c++)
            (void) delete_curve(*c);


        delete_nodes = NULL;
        for (n = intfc->nodes; n && *n; n++)
        {
            if (((*n)->in_curves == NULL) && ((*n)->out_curves == NULL))
            {
                if (! add_to_pointers((POINTER)*n,(POINTER**)&delete_nodes))
                {
                    screen("ERROR in delete_patch_all_curves(), "
                              "add_to_pointers() failed\n");
                    clean_up(ERROR);
                }
            }
        }
        for (n = delete_nodes; n && *n; n++)
            (void) delete_node(*n);


        if(debugging("delete_intfc_all_curves"))
        {
            printf("In delete_remove_patch_all_curves\n");
            printf("Print left tmp interface:::\n");
            print_interface(intfc);
            show_intfc_states(intfc);
        }

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(delete_intfc_all_curves)
}

EXPORT   void delete_patch_all_curves(
        Front           *fr)
{
        delete_intfc_all_curves(fr->interf);         
}

/*  BE CAREFUL: ONLY subdomain curves are removed 
    032803
*/

EXPORT   void remove_patch_all_boundary_curves(
        Front           *fr)
{
        INTERFACE       *sav_intfc;
        INTERFACE       *intfc = fr->interf;
        NODE            **n;
        CURVE           **c;
        CURVE           **delete_curves = NULL;
        NODE            **delete_nodes = NULL;
        bool            sav_copy;

        DEBUG_ENTER(remove_patch_all_boundary_curves)

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();

        /* remove subdomain bdry curves && physical boundary curves */
        delete_subdomain_curves(intfc);

        for(c = intfc->curves; c && *c; c++)
        {
            if(wave_type(*c) < FIRST_PHYSICS_WAVE_TYPE)
            {
                if (not add_to_pointers((POINTER)*c,(POINTER**)&delete_curves))
                {
                    screen("ERROR in remove_patch_all_boundary_curves(), "
                        "add_to_pointers() failed\n");
                    clean_up(ERROR);
                }
            }
        }
        for (c = delete_curves; c && *c; c++)
            (void) delete_curve(*c);


        delete_nodes = NULL;
        for (n = intfc->nodes; n && *n; n++)
        {
            if (((*n)->in_curves == NULL) && ((*n)->out_curves == NULL))
            {
                if (not add_to_pointers((POINTER)*n,(POINTER**)&delete_nodes))
                {
                    screen("ERROR in remove_patch_all_boundary_curves(), "
                              "add_to_pointers() failed\n");
                    clean_up(ERROR);
                }
            }
        }
        for (n = delete_nodes; n && *n; n++)
            (void) delete_node(*n);

        if(debugging("remove_patch_all_boundary_curves"))
        {
            printf("In retrieve_boundary_curves_from_zero\n");
            printf("Print left tmp interface:::\n");
            print_interface(intfc);
            show_intfc_states(intfc);
        }

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(remove_patch_all_boundary_curves)
}

EXPORT void delete_interior_curves(
        INTERFACE       *intfc)
{
        INTERFACE       *sav_intfc;
        NODE            **n;
        CURVE           **c;
        CURVE           **delete_curves = NULL;
        NODE            **delete_nodes = NULL;
        bool            sav_copy;

        DEBUG_ENTER(delete_interior_curves)

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_current_interface(intfc); 

        for(c = intfc->curves; c && *c; c++)
        {
            if(wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE)
            {
                if (not add_to_pointers((POINTER)*c,(POINTER**)&delete_curves))
                {
                    screen("ERROR in delete_interior_curves(), "
                        "add_to_pointers() failed\n");
                    clean_up(ERROR);
                }
            }
        }
        for (c = delete_curves; c && *c; c++)
            (void) delete_curve(*c);


        delete_nodes = NULL;
        for (n = intfc->nodes; n && *n; n++)
        {
            if (((*n)->in_curves == NULL) && ((*n)->out_curves == NULL))
            {
                if (not add_to_pointers((POINTER)*n,(POINTER**)&delete_nodes))
                {
                    screen("ERROR in delete_interior_curves(), "
                              "add_to_pointers() failed\n");
                    clean_up(ERROR);
                }
            }
        }
        for (n = delete_nodes; n && *n; n++)
            (void) delete_node(*n);

        if(debugging("delete_interior_curves"))
        {
            printf("In delete_interior_curves\n");
            printf("Print left tmp interface:::\n");
            print_interface(intfc);
            show_intfc_states(intfc);
        }

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(delete_interior_curves)
}

/* This function is built based on the assumption that
 * the base patch interface is assembled from the fine
 * patch interface. AND the base interface performs the
 * real interface buffer communication. Then the
 * patch interface first copies the rebuilt base interface,
 * and the patch interface only needs to do the cut to
 * fit the interface to the domain.
 */

EXPORT bool f_form_patch_subintfc_via_cut2d(
        Front           *fr)
{
        COMPONENT       i_comp = 0;
        INTERFACE       *intfc = fr->interf;
        INTERFACE       *sav_intfc;
        bool            sav_copy;
        O_NODE          *onode_list;
        RECT_GRID       *gr = fr->rect_grid;
        float           coords[MAXD];
        bool            status = FUNCTION_SUCCEEDED;
        int             i, dir, dim = gr->dim;

        DEBUG_ENTER(f_form_patch_subintfc_via_cut2d)

        /* Find an interior component on this processor's domain.  This
         * is needed to initialize the components on the subdomain
         * boundaries when no physical curves appear on this processor. */
        for (i = 0; i < dim; i++)
            coords[i] = grid_center_coord(i,gr);
        i_comp = long_component(coords,intfc);

        delete_subdomain_curves(intfc);
        delete_passive_boundaries(intfc);

        /* Cut off the patch interface to fit to the boundary*/

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_current_interface(fr->interf);

        for (dir = 0; dir < dim; dir++)
        {
            if (fr->rect_grid->lbuf[dir] > 0)
            {
                cut_interface(intfc,computational_grid(intfc)->VL[dir],
                      dir,1,NO,YES);
            }
            if (fr->rect_grid->ubuf[dir] > 0)
            {
                cut_interface(intfc,computational_grid(intfc)->VU[dir],
                            dir,0,NO,YES);
            }
        }

        if (debugging("f_form_patch_subintfc_via_cut2d")
            && fr->patch_number == 4)
        {
            printf("in f_form_patch_subintfc_via_cut2d()\n");
            printf("patch_[%d] after cut to virtual grid\n",
                    fr->patch_number);
            print_interface(fr->interf);
        }
        /* TODO:  a post-processing loop is needed here to shift the
         * subdomain nodes onto VL.  A problem can occur for periodic
         * boundaries on restart, where some accuracy is lost in VL or VU,
         * so that the subdomains have different sizes on each side of the
         * domain.  This will leave the curves hanging over the edge on the
         * shorter side, and these nodes will not be processed when
         * creating the subdomain boundary.
         * A better solution would be to guarantee the location of the
         * virtual boundaries, perhaps by printing them out as an integer
         * multiple of the mesh spacing instead of an absolute (float)
         * value. */

        status = set_amr_subdomain_boundary(fr,i_comp);
        if (status == FUNCTION_FAILED)
        {
            (void) printf("WARNING in f_form_patch_subintfc_via_cut2d(), "
                          "patch[%d] set_amr_subdomain_boundary() failed\n", 
                          fr->patch_number);
            if (DEBUG)
            {
                (void) printf("Offending interface: \n");
                print_interface(fr->interf);
            }
        }

        /* Zero length bonds can be produced, especially on an elliptic
         * interface.  If this happens AT a node, the component check gets
         * confused because it computes an angle for each  curve at a node
         * using only the node position and the adjacent point.  */

        /* Flag use_delete_short_bonds is set to be NO
         * in assembly_distribute_patch_fronts(),
         * after we glue interface, scatter the glued interface,
         * and clip_patch_front() for the fine grid patches.
         */ 
        if(use_delete_short_bonds == YES)
            intfc_delete_very_short_bonds(fr);
        else
            use_delete_short_bonds = YES;  

        /* The following code is intended to tell whether the scatter
         * succeeded by identifying problems/inconsistencies in the
         * new interface. */

        if (check_for_cut_nodes("f_form_patch_subintfc_via_cut2d",intfc) == YES)
        {
            status = FUNCTION_FAILED;
            (void) printf("WARNING: in f_form_patch_subintfc_via_cut2d");
            (void) printf(" check_for_cut_nodes() detected cut node.\n");
            clean_up(ERROR);  
        }
        else if (check_comps_at_nodes(fr->interf,&onode_list) != 0)
        {
            status = FUNCTION_FAILED;
            (void) printf("WARNING in f_form_patch_subintfc_via_cut2d");
            (void) printf(" check_comps_at_nodes() detected inconsistency\n");
            print_interface(fr->interf);
            if (DEBUG)
            {
                print_onode_list(&onode_list);
                (void) printf("Offending interface\n");
                print_interface(fr->interf);
            }
        }

        /*
        status = pp_min_status(status);
        */  
        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(f_form_patch_subintfc_via_cut2d)
        return status;
}  

EXPORT bool f_form_patch_subintfc_2d(
        Front           *fr,
        COMPONENT       i_comp)
{
        INTERFACE       *intfc = fr->interf;
        INTERFACE       *sav_intfc;
        bool            sav_copy;
        O_NODE          *onode_list;
        bool            status = FUNCTION_SUCCEEDED;

        DEBUG_ENTER(f_form_patch_subintfc_2d)

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_current_interface(fr->interf);

        if (debugging("f_form_patch_subintfc_2d"))
        {
            printf("in f_form_patch_subintfc_2d()\n");
            printf("patch [%d] after cut to virtual grid\n",
                    fr->patch_number);
            print_interface(fr->interf);
        }
        /* TODO:  a post-processing loop is needed here to shift the
         * subdomain nodes onto VL.  A problem can occur for periodic
         * boundaries on restart, where some accuracy is lost in VL or VU,
         * so that the subdomains have different sizes on each side of the
         * domain.  This will leave the curves hanging over the edge on the
         * shorter side, and these nodes will not be processed when
         * creating the subdomain boundary.
         * A better solution would be to guarantee the location of the
         * virtual boundaries, perhaps by printing them out as an integer
         * multiple of the mesh spacing instead of an absolute (float)
         * value. */

        status = set_amr_subdomain_boundary(fr,i_comp);
        if (status == FUNCTION_FAILED)
        {
            (void) printf("WARNING in f_form_patch_subintfc_2d(), "
                          "patch[%d] set_amr_subdomain_boundary() failed\n", 
                          fr->patch_number);
            if (DEBUG)
            {
                (void) printf("Offending interface: \n");
                print_interface(fr->interf);
            }
        }

        /* Zero length bonds can be produced, especially on an elliptic
         * interface.  If this happens AT a node, the component check gets
         * confused because it computes an angle for each  curve at a node
         * using only the node position and the adjacent point.  */
        
        if(use_delete_short_bonds == YES)
            intfc_delete_very_short_bonds(fr);
        else
            use_delete_short_bonds = NO;  

        /* The following code is intended to tell whether the scatter
         * succeeded by identifying problems/inconsistencies in the
         * new interface. */

        if (check_for_cut_nodes("f_form_patch_subintfc_2d",intfc) == YES)
        {
            status = FUNCTION_FAILED;
            (void) printf("\n WARNING in f_form_patch_subintfc_2d");
            (void) printf(" check_for_cut_nodes() detected cut node\n");
            (void) printf(" for patch front[%d] at level[%d]\n", 
                      fr->patch_number, fr->patch_level);
            print_rectangular_grid(fr->rect_grid);  
            print_interface(intfc);  
            clean_up(ERROR);  
        }
        else if (check_comps_at_nodes(fr->interf,&onode_list) != 0)
        {
            status = FUNCTION_FAILED;
            (void) printf("WARNING in f_form_patch_subintfc_via_cut2d");
            (void) printf(" check_comps_at_nodes() detected inconsistency\n");
            print_interface(fr->interf);
            if (DEBUG)
            {
                print_onode_list(&onode_list);
                (void) printf("Offending interface\n");
                print_interface(fr->interf);
            }
        }

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(f_form_patch_subintfc_2d)
        return status;
}  

/*
*                       set_amr_subdomain_boundary():
*
*/

LOCAL   bool set_amr_subdomain_boundary(
        Front           *fr,
        COMPONENT       i_comp)
{
        INTERFACE       *intfc = fr->interf;
        CURVE           **c;
        NODE            **n;
        RECT_GRID       *gr = computational_grid(intfc);
        float           tol[MAXD];
        int             i, j, dim = gr->dim;
        ORIENTATION     orient;
        float           grid_tol;

        DEBUG_ENTER(set_amr_subdomain_boundary)

        /* 
        grid_tol = grid_tolerance(gr)*pow(2.0,-(fr->NumberOfLevels-1.0-fr->patch_level));
        */
        grid_tol = 100.0*MACH_EPS;

        /* This can fail if a tangle occurs across the virtual boundary. */

        if (set_boundary(intfc,gr,i_comp,grid_tol) == NO)
        {
            (void) printf("WARNING in set_amr_subdomain_boundary(), "
                          "set_boundary() failed\n");
            return FUNCTION_FAILED;
        }

        for (i = 0; i < dim; i++)
            tol[i] = MIN_SC_SEP(intfc) * gr->h[i];
        orient = (fr->step % 2) ? POSITIVE_ORIENTATION : NEGATIVE_ORIENTATION;

        /* 042503 closed 
        for (c = intfc->curves; c && *c; c++)
        {
            if (is_bdry(*c) && (wave_type(*c) == ERROR))
            {
                wave_type(*c) = SUBDOMAIN_BOUNDARY;
                rect_bdry_curve_redist(*c,orient,gr,tol);
            }
        }
        for (n = intfc->nodes; n && *n; n++)
        {
            if (is_bdry(*n) && (node_type(*n) == ERROR))
                node_type(*n) = FIXED_NODE;
        }
        */
/* This part of the code is copied fromt set_subdomain_boundary */
/* 042103 */
        for (c = intfc->curves; c && *c; ++c)
        {
            if (is_bdry(*c) && (wave_type(*c) == ERROR))
            {
                rect_bdry_side_for_curve(&i,&j,*c,gr);
                switch(rect_boundary_type(intfc,i,j))
                {
                case PASSIVE_BOUNDARY:
                    wave_type(*c) = PASSIVE_BOUNDARY;
                    break;
                case SUBDOMAIN_BOUNDARY:
                case AMR_SUBDOMAIN_BOUNDARY:
                case REFLECTION_BOUNDARY:
                    wave_type(*c) = SUBDOMAIN_BOUNDARY;
                    break;
                case MIXED_TYPE_BOUNDARY:
                    if (is_excluded_comp(positive_component(*c),intfc) &&
                        is_excluded_comp(negative_component(*c),intfc))
                        wave_type(*c) = PASSIVE_BOUNDARY;
                    break;
                default:
                    screen("ERROR in set_amr_subdomain_boundary(), "
                           "unexpected case for rect boundary type\n");
                    print_rectangular_grid(gr);
                    printf("wave_type = %d, is_bdry = %d front[%d]\n",
                      wave_type(*c),is_bdry(*c),fr->patch_number);
                    printf("boundary type = %d\n", rect_boundary_type(intfc,i,j));
                    print_curve(*c);
                    clean_up(ERROR);
                }
                rect_bdry_curve_redist(*c,orient,gr,tol);
                if (size_of_state(intfc) != 0)
                {
                    BOND *b;
                    size_t sizest = size_of_state(intfc);
                    obstacle_state(intfc,left_start_state(*c),sizest);
                    obstacle_state(intfc,right_start_state(*c),sizest);
                    for (b=(*c)->first; b!=NULL && b!=(*c)->last; b=b->next)
                    {
                        obstacle_state(intfc,left_state(b->end),sizest);
                        obstacle_state(intfc,right_state(b->end),sizest);
                    }
                    obstacle_state(intfc,left_end_state(*c),sizest);
                    obstacle_state(intfc,right_end_state(*c),sizest);
                }
            }
        }
        for (n = intfc->nodes; n && *n; ++n)
        {
            if (is_bdry(*n) && (node_type(*n) == ERROR))
                node_type(*n) = FIXED_NODE;
        }

        DEBUG_LEAVE(set_amr_subdomain_boundary)
        return FUNCTION_SUCCEEDED;
}

/* Before send the fine grid interface back to its orignal
 * processor, it is trimmed into interior. Then 
 * the assembly process and the following 
 * distribution of buffer zone interface back to
 * fine grid interface (pseudo_scatter_patch_fronts())
 * do not need to trim interface into 
 * interior again.
 */
LIB_LOCAL void clip_patch_fronts_to_interior(
        Front      **frs,
        int        num_patches,
        COMPONENT  *i_comp)    /* number of patches computed in the proc */ 
{
        INTERFACE  *sav_intfc;
        bool       sav_copy, save_intrp;
        int        *lbuf,*ubuf;
        int        i, j, dim;  
        float      coords[MAXD];
        RECT_GRID  *gr;  

        sav_intfc = current_interface();

        /*
         * 070903 add. Use this flag, not to alter the interface during
         * the interface assembly and distribution process.
         */
        set_min_sc_sep_val_flag(YES); 

        for(i = 0; i < num_patches; i++)
        {
            if(frs[i]->patch_level == frs[i]->NumberOfLevels-1)
            {
                lbuf = frs[i]->rect_grid->lbuf;
                ubuf = frs[i]->rect_grid->ubuf;
                set_current_interface(frs[i]->interf);
                save_intrp = interpolate_intfc_states(frs[i]->interf);
                interpolate_intfc_states(frs[i]->interf) = NO;
                for(j = 0; j < frs[i]->rect_grid->dim; j++)
                    coords[j] = grid_center_coord(j,frs[i]->rect_grid);
                i_comp[i] = (frs[i]->interf->modified) ?
                    long_component(coords,frs[i]->interf) :
                    component(coords,frs[i]->interf);
                /*
                 * 051503, now fine level grids also contribute to
                 * boundary curve reconstruction.
                 * remove_patch_all_boundary_curves(frs[i]);
                 */
                delete_subdomain_curves(frs[i]->interf);
                delete_passive_boundaries(frs[i]->interf);
                if(debugging("clip_fronts_to_interior"))
                {
                    if(pp_mynode() == 3 && 
                       (frs[i]->patch_number == 2))
                    {
                        set_debug_cut_flag(YES); 
                        printf("Patch[%d] level[%d], BEFORE clip\n", 
                             frs[i]->patch_number, frs[i]->patch_level);
                        print_rectangular_grid(frs[i]->rect_grid); 
                        for( CURVE **c = frs[i]->interf->curves; c && *c; ++c) 
                        {
                            if (wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE)
                                print_curve(*c); 
                        }
                        printf("End of print interior curves\n");  
                    }
                }
                amr_clip_to_interior_region(frs[i]->interf,lbuf,ubuf,YES);
                interpolate_intfc_states(frs[i]->interf) = save_intrp;
                if(debugging("clip_fronts_to_interior"))
                {
                    if(pp_mynode() == 3 && 
                       (frs[i]->patch_number == 2))
                    {
                        set_debug_cut_flag(NO); 
                        printf("Patch[%d] level[%d], AFTER clip\n", 
                             frs[i]->patch_number, frs[i]->patch_level);
                        print_rectangular_grid(frs[i]->rect_grid); 
                        for( CURVE **c = frs[i]->interf->curves; c && *c; ++c) 
                        {
                            if (wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE)
                            {
                                print_curve(*c); 
                                print_node_flags((*c)->start);
                                print_node_flags((*c)->end);
                            }
                        }
                    }
                }
            }
        }
        set_min_sc_sep_val_flag(NO); 
        set_current_interface(sav_intfc);  
}

LIB_LOCAL  int  assembly_coarse_patch_fronts2d(
        Front      **frs,
        int        num_patches,    /* number of patches computed in the proc */ 
        Wv_on_pc   **redistr_table,
        int        max_n_patch)
{
        int        source, numnodes, myid;
        int        dist;
        int        patch_id, i;
        Front      **tmpfrs, *front;
        Front      *glue_fr;
        int        total_patch, levels;
        bool       sav_copy, status;
        INTERFACE  *current_intfc;
        COMPONENT  dummy = -1,  i_comp, *i_comps;
        float      coords[MAXD];
        RECT_GRID  *cgr, *tgr;
        RECT_GRID  *rgr;

        DEBUG_ENTER(assembly_coarse_patch_fronts2d)

        numnodes = pp_numnodes();
        myid = pp_mynode();
        sav_copy = copy_intfc_states();

        /* total_patch: number of total patches before redistribute */
        /* NUMBER of tmpfrs = total_patch */

        total_patch = set_copy_proc_frs_ver3(&tmpfrs,frs,num_patches,
                         redistr_table,max_n_patch);
        levels = tmpfrs[0]->NumberOfLevels;

        set_current_interface(frs[0]->interf);

        reinstall_undistribute_fronts_and_misc(tmpfrs,frs,redistr_table,
                              total_patch,num_patches,max_n_patch);

        vector(&i_comps, total_patch, sizeof(COMPONENT));
        clip_patch_fronts_to_interior(tmpfrs, total_patch, i_comps);

        glue_fr = copy_front(tmpfrs[0]);
        status = assembly_fine_patch_fronts_to_one(tmpfrs, glue_fr);
        status = pp_min_status(status);
        if(status != FUNCTION_SUCCEEDED)
        {
            printf("WARNING: assembly_coarse_patch_fronts()\n");
            printf("assembly_fine_patch_fronts_to_one failed\n");
            for(CURVE **c = glue_fr->interf->curves; c && *c; c++)
            {
                if(wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE)
                    print_curve(*c);
            }
            for(NODE **n = glue_fr->interf->nodes; n && *n; n++)
            {
                print_node(*n);
                print_node_flags(*n);
            }
            set_current_interface(frs[0]->interf);
            free_front(glue_fr);
            set_copy_frs_intfc_null_ver3(tmpfrs,redistr_table,max_n_patch,NO);
            clear_copy_frs(tmpfrs,redistr_table,max_n_patch);
            free(tmpfrs);
            free(i_comps);
            return status;
        }

        set_current_interface(glue_fr->interf);
        /*
         * Use this flag, not to alter the interface during
         * the interface assembly and distribution process.
         */
        set_use_delete_short_bonds_flag();
        set_min_sc_sep_val_flag(YES);

        if(glue_fr->interf->nodes == NULL)
        {
            for (i = 0; i < frs[0]->rect_grid->dim; ++i)
                coords[i] = grid_center_coord(i,frs[0]->rect_grid);
            i_comp = (frs[0]->interf->modified) ?
                long_component(coords,frs[0]->interf) :
                component(coords,frs[0]->interf);
            set_center_comp(YES, i_comp);
        }

        set_amr_intfc_tol(glue_fr->interf,
            pow(2.0,-(glue_fr->NumberOfLevels-1.0-glue_fr->patch_level)));
        status = scatter_front(glue_fr);
        set_min_sc_sep_val_flag(NO);
        set_center_comp(NO, 0);
        set_amr_intfc_tol(glue_fr->interf,
            pow(2.0,(glue_fr->NumberOfLevels-1.0-glue_fr->patch_level)));
        if(status != FUNCTION_SUCCEEDED)
        {
            printf("WARNING: assembly_coarse_patch_fronts()\n");
            printf("scatter glued front interface failed\n");
            set_current_interface(frs[0]->interf);
            free_front(glue_fr);
            set_copy_frs_intfc_null_ver3(tmpfrs,redistr_table,max_n_patch,NO);
            clear_copy_frs(tmpfrs,redistr_table, max_n_patch);
            free(tmpfrs);
            free(i_comps);
            return status;
        }

        /*
         * Base grid interface is not handled here.
         * Coarse level grid interfaces are set here
         * by the calling of clip_patch_front().
         */
        set_current_interface(glue_fr->interf);
        set_copy_frs_intfc_null_ver3(tmpfrs,redistr_table,max_n_patch,NO);

        for(i = 1; i < total_patch; i++)
        {
            if(tmpfrs[i]->patch_level == (levels - 1))
                continue;
            set_patch_front(glue_fr,tmpfrs[i],tmpfrs[i]->rect_grid,i);
        }

        set_min_sc_sep_val_flag(YES);
        status = FUNCTION_SUCCEEDED;
        for(i = 1; i < total_patch; i++)
        {
            if(tmpfrs[i]->patch_level == (levels - 1))
                continue;
            set_amr_intfc_tol(tmpfrs[i]->interf,
                pow(2.0,-(tmpfrs[i]->NumberOfLevels-1.0-tmpfrs[i]->patch_level)));
            if((status = clip_patch_front(tmpfrs[i],NO)) == FUNCTION_FAILED)
            {
                printf("WARNING in assembly_distribute_patch_fronts(),"
                   " clip_patch_front() for tmpfrs[%d] failed \n",i);
                break;
            }
            set_amr_intfc_tol(tmpfrs[i]->interf,
                pow(2.0,(tmpfrs[i]->NumberOfLevels-1.0-tmpfrs[i]->patch_level)));
        }
        set_min_sc_sep_val_flag(NO);
        status = pp_min_status(status);

        if(status != FUNCTION_SUCCEEDED)
        {
            printf("WARNING: assembly_coarse_patch_fronts()\n");
            printf("clip_patch_front failed\n");
            set_current_interface(frs[0]->interf);
            free_front(glue_fr);
            set_copy_frs_intfc_null_ver3(tmpfrs,redistr_table,max_n_patch,YES);
            clear_copy_frs(tmpfrs,redistr_table, max_n_patch);
            free(tmpfrs); free(i_comps);
            return status;
        }

        set_current_interface(frs[0]->interf);

        distribute_coarse_patch_fronts(frs, tmpfrs, glue_fr,
          num_patches, redistr_table, max_n_patch);
        set_copy_frs_intfc_null_ver3(tmpfrs,redistr_table,max_n_patch,YES);
        clear_copy_frs(tmpfrs,redistr_table, max_n_patch);
        free(tmpfrs);

        set_current_interface(glue_fr->interf);
        delete_interface(frs[0]->interf);
        frs[0]->interf = glue_fr->interf;
        glue_fr->interf = NULL;
        free_front(glue_fr);

        frs[0]->interf->modified = YES;
        rgr = frs[0]->rect_grid;
        cgr = computational_grid(frs[0]->interf);
        copy_rect_grid(cgr,rgr);
        tgr = &topological_grid(frs[0]->interf);
        tgr->Remap.remap = rgr->Remap.remap;
        set_patch_topo_grid(rgr,tgr);
        free(i_comps);

        DEBUG_LEAVE(assembly_coarse_patch_fronts2d)
        return FUNCTION_SUCCEEDED;
}

/* Base patch interface is not handled here */
LOCAL void set_copy_frs_intfc_null_ver3(
        Front      **cpyfrs,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int        del_intfc)
{
        int        myid;
        int        i, j;
        int        total_patch;
        int        is_local;

        myid = pp_mynode();
        total_patch = cpyfrs[0]->totalNumberOfPatches;
        for(i = 0; i < total_patch; i++)
        {
            is_local = NO;
            for(j = 0; j < max_n_patch; j++)
            {
                if(-1 == redistr_table[myid][j].wv_id)
                    continue;
                if(myid == redistr_table[myid][j].pc_id)
                {
                    if(redistr_table[myid][j].wv_id == i)
                        is_local = YES;
                }
            }
            if(YES == is_local)
            {
                if(i == 0)
                    cpyfrs[i]->interf = NULL;
                if(del_intfc == YES)
                {
                    delete_interface(cpyfrs[i]->interf);
                    cpyfrs[i]->interf = NULL;
                }
                else
                {
                    if(cpyfrs[i]->patch_level !=
                       cpyfrs[i]->NumberOfLevels-1)
                        cpyfrs[i]->interf = NULL;
                    else
                    {
                        delete_interface(cpyfrs[i]->interf);
                        cpyfrs[i]->interf = NULL;
                    }
                }
            }
            else
            {
                delete_interface(cpyfrs[i]->interf);
                cpyfrs[i]->interf = NULL;
            }
        }
}

LOCAL bool distribute_coarse_patch_fronts(
        Front      **frs,
        Front      **tmpfrs,
        Front      *glue_fr,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch)
{
        int        source, numnodes, myid;
        int        i, j, patch_id, levels;
        Front      *front;
        bool       sav_copy;
        bool       status = FUNCTION_SUCCEEDED;
        INTERFACE  *sav_intfc;
        int        item[2]; 

        DEBUG_ENTER(distribute_coarse_patch_fronts)

        numnodes = pp_numnodes();
        myid = pp_mynode();
        levels = tmpfrs[0]->NumberOfLevels;

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();

        /* Delete intfc of NON-LOCAL COARSE LEVEL patch if 
         * it will receive one. 
         * For the LOCAL COARSE LEVEL grid fronts,
         * delete the OLD LOCAL front interface,
         * set the grid front interface points to the copy
         * front("tmpfrs") interface.
         */
        for(i = 1; i < num_patches; i++)
        {
            front = frs[i]; 
            if(front->patch_level == levels-1)
                continue; 
            if(NO == loc_front_on_redistr_table(front,redistr_table,
                   max_n_patch,item))
            {
                printf("ERROR in distribute_coarse_patch_fronts\n");  
                printf("Front is not on the redistribute table\n");
                clean_up(ERROR); 
            }
            if(myid != item[0])
                delete_patch_all_curves(front);
            if(myid == item[0])
            {
                patch_id = item[1]; 
                delete_interface(front->interf);
                front->interf = tmpfrs[patch_id]->interf;
                tmpfrs[patch_id]->interf = NULL;
            }
        } 

        /*
         * For the COARSE LEVEL non-local grid fronts, 
         * the completed interfaces are also sent by the MPI.
         */
        patch_front_distribute_to_nodes(tmpfrs, frs, redistr_table,
            num_patches, max_n_patch, 1);

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(distribute_coarse_patch_fronts)
        return status;
}


/*  set_copy_proc_frs_ver3().
 * 1. Collect fronts (by making a copy) which are originally
 * on this proc, for the local fine patch, the interface is 
 * copied. 2. Make storages for the fronts which are 
 * transfer back to this proc.
 */
LOCAL int set_copy_proc_frs_ver3(
        Front      ***tmpfrs,
        Front      **frs,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch)
{
        int        source, numnodes, myid;
        int        patch_id, i;
        Front      *basefront = NULL;
        Front      *front;
        int        total_patch;
        INTERFACE  *sav_intfc;
        bool       sav_copy;

        numnodes = pp_numnodes();
        myid = pp_mynode();
        sav_copy = copy_intfc_states();
        sav_intfc = current_interface();

        for(i = 0; i < max_n_patch; i++)
        {
            if(-1 == redistr_table[myid][i].wv_id) continue;
            if(myid == redistr_table[myid][i].pc_id &&
               redistr_table[myid][i].wv_id == 0)
            {
                basefront = redistr_table[myid][i].front;
                total_patch = basefront->totalNumberOfPatches;
                break;
            }
        }
        if((basefront == NULL) ||
           (basefront->patch_number != 0 && basefront->patch_level != 0))
        {
            printf("ERROR set_copy_proc_frs_ver3() ");
            printf("did not find out the basefront\n");
            clean_up(ERROR);
        }
        vector(tmpfrs,total_patch,sizeof(Front*));
        for(i = 0; i < total_patch; i++) (*tmpfrs)[i] = NULL;

        /* Save local grids which are not transfered to the other procs */
        /* redistr_table[myid][i].front = frs[i], they are set to be equal */
        for(i = 0; i < max_n_patch; i++)
        {
            int found_same;
            if(-1 == redistr_table[myid][i].wv_id) continue;
            if(myid == redistr_table[myid][i].pc_id)
            {
                patch_id = redistr_table[myid][i].wv_id;
                (*tmpfrs)[patch_id] = copy_front(redistr_table[myid][i].front);
                if((*tmpfrs)[patch_id]->patch_level == 
                   (*tmpfrs)[patch_id]->NumberOfLevels-1)
                {
                    set_size_of_intfc_state(
                      size_of_state(redistr_table[myid][i].front->interf));
                    set_copy_intfc_states(YES);
                    (*tmpfrs)[patch_id]->interf = copy_interface(
                             redistr_table[myid][i].front->interf);
                    if((*tmpfrs)[patch_id]->interf == NULL)
                    {
                        printf("ERROR: set_copy_proc_frs_ver3()\n");
                        printf("Copy interface failed\n"); 
                        clean_up(ERROR); 
                    }
                }

                found_same = NO;
                for(int tmpjj = 0; tmpjj < num_patches; tmpjj++)
                {
                    if(frs[tmpjj] == redistr_table[myid][i].front)
                    {
                        found_same = YES;
                        break;
                    }
                }
                if(found_same == NO || 
                   patch_id != redistr_table[myid][i].front->patch_number)
                {
                    printf("ERROR: set_copy_proc_frs_ver3()\n");
                    printf("Fronts NOT in redistr_table[%d][%d] and frs\n",
                          myid, i);
                    printf("patch_id [%d] from redistr_table\n", patch_id);
                    clean_up(ERROR);
                }
            }
        }

        /* make storage for recving patches from other procs */
        for(i = 0; i < total_patch; i++)
        {
            if((*tmpfrs)[i] != NULL)
                continue;
            (*tmpfrs)[i] = deep_copy_front(basefront);
            set_size_of_intfc_state(size_of_state(basefront->interf));
            set_copy_intfc_states(YES);
            (*tmpfrs)[i]->interf = copy_interface(basefront->interf);
            delete_patch_all_curves((*tmpfrs)[i]);
        }

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        return  total_patch;
}


LIB_LOCAL  int  assembly_distribute_patch_fronts2d(
        Front      **frs,
        int        num_patches,    /* number of patches computed in the proc */ 
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int        all_level)
{
        int        source, numnodes, myid;
        int        dist;  
        int        patch_id, i;
        Front      **tmpfrs, *front; 
        Front      *glue_fr; 
        int        total_patch, levels;
        bool       sav_copy, status; 
        INTERFACE  *current_intfc; 
        COMPONENT  dummy = -1,  i_comp, *i_comps; 
        float      coords[MAXD];
        RECT_GRID  *cgr, *tgr;
        RECT_GRID  *rgr; 

        DEBUG_ENTER(assembly_distribute_patch_fronts2d)  

        numnodes = pp_numnodes();
        myid = pp_mynode(); 
        sav_copy = copy_intfc_states();
    
#if defined(SMOOTH_INTERFACE_FOR_SCATTER)
        /* This routine is responsible for simplifing 
         * interface across the domain boundary. 
         * So as to make the followed scattering and 
         * assembly interface routine more robust.
         */
        for(i = 0; i < num_patches; i++)
        {
            if(frs[i]->patch_level == frs[i]->NumberOfLevels-1)
                adjust_front_boundary_crxing_spacing(frs[i]);
        }
#endif /* if defined(SMOOTH_INTERFACE_FOR_SCATTER) */

        vector(&i_comps, num_patches, sizeof(COMPONENT));
        clip_patch_fronts_to_interior(frs, num_patches, i_comps);

        /* total_patch: number of total patches before redistribute */
        /* NUMBER of tmpfrs = total_patch */  
        total_patch = set_copy_proc_frs_ver2(&tmpfrs,frs,num_patches,
                         redistr_table,max_n_patch); 
        levels = tmpfrs[0]->NumberOfLevels; 

        set_current_interface(frs[0]->interf); 

        reinstall_undistribute_fronts_and_misc(tmpfrs,frs,redistr_table,
                              total_patch,num_patches,max_n_patch);
        // old_reinstall_undistribute_fronts_and_misc(tmpfrs,frs,redistr_table,
        //                      total_patch,num_patches,max_n_patch);

        /* 070403, add set_amr_intfc_tol() in
         * assembly_fine_patch_fronts_to_one() to the
         * glue->interf to get the consistence of
         * the tolerance on the interface operation.
         * The glued interface based on the coarse grid should
         * also have fine grid interface tolerance.
         */ 
        glue_fr = copy_front(tmpfrs[0]); 
        status = assembly_fine_patch_fronts_to_one(tmpfrs, glue_fr); 
        status = pp_min_status(status); 
        if(status != FUNCTION_SUCCEEDED)
        {
            printf("WARNING: assembly_distribute_patch_fronts()\n");
            printf("assembly_fine_patch_fronts_to_one failed\n");
            for(CURVE **c = glue_fr->interf->curves; c && *c; c++)
            {
                if(wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE)
                    print_curve(*c);
            }
            for(NODE **n = glue_fr->interf->nodes; n && *n; n++)
            {
                print_node(*n);   
                print_node_flags(*n);
            }
            set_current_interface(frs[0]->interf);
            free_front(glue_fr);
            set_copy_frs_intfc_null_ver2(tmpfrs,redistr_table,max_n_patch,NO);
            clear_copy_frs(tmpfrs,redistr_table, max_n_patch);
            free(tmpfrs);
            free(i_comps);
            return status;
        }
 
        set_current_interface(glue_fr->interf); 
        /*
         * 070903 add. Use this flag, not to alter the interface during
         * the interface assembly and distribution process.
         */
        set_use_delete_short_bonds_flag();  
        set_min_sc_sep_val_flag(YES);    
 
        if(glue_fr->interf->nodes == NULL)
        {
            for (i = 0; i < frs[0]->rect_grid->dim; ++i)
                coords[i] = grid_center_coord(i,frs[0]->rect_grid);
            i_comp = (frs[0]->interf->modified) ?
                long_component(coords,frs[0]->interf) : 
                component(coords,frs[0]->interf);
            set_center_comp(YES, i_comp); 
        }

        set_amr_intfc_tol(glue_fr->interf,
            pow(2.0,-(glue_fr->NumberOfLevels-1.0-glue_fr->patch_level)));
        status = scatter_front(glue_fr); 
        set_min_sc_sep_val_flag(NO); 
        set_center_comp(NO, 0); 
        set_amr_intfc_tol(glue_fr->interf,
            pow(2.0,(glue_fr->NumberOfLevels-1.0-glue_fr->patch_level)));
        if(status != FUNCTION_SUCCEEDED)  
        {
            printf("WARNING: assembly_distribute_patch_fronts()\n");
            printf("scatter glued front interface failed\n"); 
            set_current_interface(frs[0]->interf); 
            free_front(glue_fr); 
            set_copy_frs_intfc_null_ver2(tmpfrs,redistr_table,max_n_patch,NO);
            clear_copy_frs(tmpfrs,redistr_table, max_n_patch);
            free(tmpfrs);
            free(i_comps); 
            return status; 
        }
        
        /*
         * Set fine grids buffer zone interface on tmpfrs.
         * The buffer zone interface is going to stitch
         * to the interior interface originally contained
         * in the fine grids. This procedure is an
         * analogy to the scatter_front().
         * Base grid interface is not handled here.
         * Other level grid interfaces are also set here
         * by the calling of clip_patch_front().
         */
        set_current_interface(glue_fr->interf); 
        set_copy_frs_intfc_null_ver2(tmpfrs,redistr_table,max_n_patch,NO);
        for(i = 1; i < total_patch; i++)
        {
            if(all_level != YES)
            {
                if(tmpfrs[i]->patch_level != (levels - 1))
                    continue;
            }
            set_patch_front(glue_fr,tmpfrs[i],tmpfrs[i]->rect_grid,i);
        }

        set_min_sc_sep_val_flag(YES);
        status = FUNCTION_SUCCEEDED; 
        for(i = 1; i < total_patch; i++)
        {
            RECT_GRID   *tmpgr;
            if(all_level != YES)
            {
                if(tmpfrs[i]->patch_level != (levels - 1))
                    continue;
            }
            if(tmpfrs[i]->patch_level == (levels - 1)) 
                use_delete_short_bonds = NO;  
          
            set_amr_intfc_tol(tmpfrs[i]->interf,
                pow(2.0,-(tmpfrs[i]->NumberOfLevels-1.0-tmpfrs[i]->patch_level)));

            if((status = clip_patch_front(tmpfrs[i],NO)) == FUNCTION_FAILED)
            {
                printf("WARNING in assembly_distribute_patch_fronts(),"
                   " clip_patch_front() for tmpfrs[%d] failed \n",i);
                break; 
            }
            set_amr_intfc_tol(tmpfrs[i]->interf,
                pow(2.0,(tmpfrs[i]->NumberOfLevels-1.0-tmpfrs[i]->patch_level)));

            if(tmpfrs[i]->patch_level != (levels - 1))
                continue;
            tmpgr = computational_grid(tmpfrs[i]->interf);
            set_current_interface(tmpfrs[i]->interf);
            delete_subdomain_curves(tmpfrs[i]->interf);
            delete_passive_boundaries(tmpfrs[i]->interf);
            clip_interface_with_rect(tmpfrs[i]->interf,tmpgr->L,tmpgr->U,YES);
        }
        set_min_sc_sep_val_flag(NO);
        status = pp_min_status(status);
        if(status != FUNCTION_SUCCEEDED)
        {
            printf("WARNING: assembly_distribute_patch_fronts()\n");
            printf("clip_patch_front failed\n");
            set_current_interface(frs[0]->interf);
            free_front(glue_fr);
            set_copy_frs_intfc_null_ver2(tmpfrs,redistr_table,max_n_patch,YES);
            clear_copy_frs(tmpfrs,redistr_table, max_n_patch);
            free(tmpfrs); free(i_comps);
            return status;
        }

        set_current_interface(frs[0]->interf); 

        pseudo_scatter_patch_fronts(frs, tmpfrs, glue_fr,
          num_patches, redistr_table, max_n_patch, i_comps, all_level);
        set_copy_frs_intfc_null_ver2(tmpfrs,redistr_table,max_n_patch,YES);
        clear_copy_frs(tmpfrs,redistr_table, max_n_patch);
        free(tmpfrs);

        if(all_level == YES)
        {
            set_current_interface(glue_fr->interf);
            delete_interface(frs[0]->interf);
            frs[0]->interf = glue_fr->interf;
            glue_fr->interf = NULL;

            frs[0]->interf->modified = YES;
            rgr = frs[0]->rect_grid;
            cgr = computational_grid(frs[0]->interf);
            copy_rect_grid(cgr,rgr);
            tgr = &topological_grid(frs[0]->interf);
            tgr->Remap.remap = rgr->Remap.remap;
            set_patch_topo_grid(rgr,tgr);
        }
        else
        {
            INTERFACE *save_intfc[2]; 

            set_current_interface(frs[0]->interf);
            set_amr_intfc_tol(frs[0]->interf,pow(2.0,1.0-frs[0]->NumberOfLevels));
            set_use_delete_short_bonds_flag();  
            
            save_intfc[0] = make_save_intfc(frs[0]->interf); 
            save_intfc[1] = NULL; 
            if(! scatter_front(frs[0]))
            {
                printf("WARNING: assembly_distribute_patch_fronts(), "
                     "scatter_front() failed for the base front\n");
                reset_interface_of_front(frs[0],save_intfc);
            }
            else
                delete_interface(save_intfc[0]); 
            set_amr_intfc_tol(frs[0]->interf,pow(2.0,frs[0]->NumberOfLevels-1.0));
        } 

        for(i = 0; i < num_patches; i++)
        {
            set_amr_intfc_tol(frs[i]->interf,
                pow(2.0,-(frs[i]->NumberOfLevels-1.0-frs[i]->patch_level)));
            if(frs[i]->patch_level == (levels - 1) || 
               frs[i]->patch_level == 0)
                intfc_delete_very_short_bonds(frs[i]);
            set_amr_intfc_tol(frs[i]->interf,
                pow(2.0,frs[i]->NumberOfLevels-1.0-frs[i]->patch_level));
        }  
        free_front(glue_fr);
        set_copy_intfc_states(sav_copy);
        free(i_comps);  
       
        DEBUG_LEAVE(assembly_distribute_patch_fronts2d)  
        return FUNCTION_SUCCEEDED; 
}

/* set_copy_frs_intfc_null_ver2()
 * If the front is copied from local front,
 * the associated interface pointer is set the NULL.
 * If the front is copied from non-local front(by MPI),
 * the associated interface is deleted.
 */
LOCAL  void set_copy_frs_intfc_null_ver2(
        Front      **cpyfrs,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int        del_locintfc)
{
        int        myid;
        int        i, j;
        int        total_patch;
        int        is_local;

        myid = pp_mynode();
        total_patch = cpyfrs[0]->totalNumberOfPatches;
        for(i = 0; i < total_patch; i++)
        {
            if(i == 0)
            {
                cpyfrs[i]->interf = NULL;
                continue;  
            }
            is_local = NO;
            for(j = 0; j < max_n_patch; j++)
            {
                if(-1 == redistr_table[myid][j].wv_id)
                    continue;
                if(myid == redistr_table[myid][j].pc_id)
                {
                    if(redistr_table[myid][j].wv_id == i)
                    { 
                        is_local = YES;
                        break; 
                    }
                }
            }
            if(YES == is_local)
            {
                if(del_locintfc == YES)
                {
                    delete_interface(cpyfrs[i]->interf);
                    cpyfrs[i]->interf = NULL;
                }
                else
                {
                    cpyfrs[i]->interf = NULL;
                }
            }
            else
            {
                delete_interface(cpyfrs[i]->interf);
                cpyfrs[i]->interf = NULL;
            }
        }
}


EXPORT  int   assembly_fine_patch_fronts_to_one2d_ver2(
        Front    **oldfrs,
        Front    *newfr)
{
        int             i, j, levels, num_fine_frs;
        int             total_patch;
        int             dim, dir, side;
        INTERFACE       *sav_intfc, *bdy_intfc;
        bool            sav_copy;
        Front           **frs, *front = NULL;
        Front           *basefront = NULL;   
        int             status; 

        DEBUG_ENTER(assembly_fine_patch_fronts_to_one_ver2)

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();

        levels = oldfrs[0]->NumberOfLevels;
        total_patch = oldfrs[0]->totalNumberOfPatches;
        num_fine_frs = 0; j = 0; 
        for (i = 0; i < total_patch; i++)
        {
            if( oldfrs[i]->patch_level == levels-1)
                num_fine_frs++;
            if( oldfrs[i]->patch_level == 0 && 
                oldfrs[i]->patch_number == 0)
                basefront = oldfrs[i];  
        }

        set_current_interface(basefront->interf);
        set_size_of_intfc_state(size_of_state(oldfrs[0]->interf));
        set_copy_intfc_states(YES);
        if (( newfr->interf = copy_interface(basefront->interf)) == NULL)
        {
            screen("ERROR assembly_fine_patch_fronts_to_one2d_ver2()",
                 "copy_interface() failed\n");
            clean_up(ERROR);
        }

        /* 070403, add set_amr_intfc_tol()
         * to get the consistence of the tolerance
         * on the interface operation */
        set_amr_intfc_tol(newfr->interf,
            pow(2.0,-(newfr->NumberOfLevels-1.0-newfr->patch_level)));

        if(0 == num_fine_frs) 
        {
            set_current_interface(newfr->interf);
            set_copy_intfc_states(sav_copy);
            set_amr_intfc_tol(newfr->interf,
                pow(2.0,(newfr->NumberOfLevels-1.0-newfr->patch_level)));
            DEBUG_LEAVE(assembly_fine_patch_fronts_to_one2d_ver2)
            return FUNCTION_SUCCEEDED;   
        }  

        vector(&frs,num_fine_frs,sizeof(Front*));

        for (i = 0; i < total_patch; i++)
        {
            if (oldfrs[i]->patch_level == levels-1)
            {
                frs[j] = copy_front(oldfrs[i]);
                set_size_of_intfc_state(size_of_state(oldfrs[i]->interf));
                set_copy_intfc_states(YES);
                if (( frs[j]->interf = copy_interface(oldfrs[i]->interf)) == NULL)
                {
                    screen("ERROR assembly_fine_patch_fronts_to_one2d_ver2()",
                         "copy_interface() failed\n");
                    clean_up(ERROR);
                }
                j++;
            }
        }
        if(j!=num_fine_frs)
        {
            printf("Error: assembly_fine_patch_fronts_to_one2d_ver2()\n");
            printf("did not find finest level front, exit j = %d\n", j);
            clean_up(ERROR);
        }
        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        status = merge_fronts_ver2(newfr,frs,num_fine_frs);
        set_current_interface(newfr->interf); 
        set_amr_intfc_tol(newfr->interf,
            pow(2.0,(newfr->NumberOfLevels-1.0-newfr->patch_level)));

        for(i = 0 ; i < num_fine_frs; i++)
            free_front(frs[i]);
        free_these(1,frs);

        if(debugging("assembly_fine_patch_fronts_to_one"))
        {
            printf("In assembly_patch_fronts_to_one2d_ver2()\n");
            printf("Print the assembled interface with states\n");
            print_interface(newfr->interf);
            show_intfc_states(newfr->interf);
        }

        DEBUG_LEAVE(assembly_fine_patch_fronts_to_one2d_ver2)
        return status;
}

/* OLD version, now is not used. But 
 * contains retrieve_boundary_curves_from_zero() 
LOCAL  int    assembly_fine_patch_fronts_to_one2d(
        Front    **oldfrs,
        Front    *newfr)
{
        int             i, j, levels, num_fine_frs;
        int             total_patch;
        int             dim, dir, side;
        INTERFACE       *sav_intfc, *bdy_intfc;
        bool            sav_copy;
        Front           **frs, *front = NULL;
        Front           *basefront = NULL;   

        DEBUG_ENTER(assembly_fine_patch_fronts_to_one)

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();

        levels = oldfrs[0]->NumberOfLevels;
        total_patch = oldfrs[0]->totalNumberOfPatches;
        num_fine_frs = 0; j = 0;

        for (i = 0; i < total_patch; i++)
        {
            if (oldfrs[i]->patch_level == levels-1)
                num_fine_frs++;
            if(oldfrs[i]->patch_level == 0 && 
               oldfrs[i]->patch_number == 0)
                basefront = oldfrs[i];  
        }

        if(0 != num_fine_frs)
            vector(&frs,num_fine_frs,sizeof(Front*));

        for (i = 0; i < total_patch; i++)
        {
            if (oldfrs[i]->patch_level == levels-1)
            {
                frs[j] = copy_front(oldfrs[i]);
                set_size_of_intfc_state(size_of_state(oldfrs[i]->interf));
                set_copy_intfc_states(YES);
                if (( frs[j]->interf = copy_interface(oldfrs[i]->interf)) == NULL)
                {
                    screen("ERROR in assembly_fine_patch_fronts_to_one()",
                         "copy_interface() failed\n");
                    clean_up(ERROR);
                }
                j++;
            }
        }

        if(j!=num_fine_frs)
        {
            printf("Error: in assembly_fine_patch_fronts_to_one()\n");
            printf("did not find finest level front, exit j = %d\n", j);
            clean_up(ERROR);
        }
      
        if(0 == j) 
        {
            set_size_of_intfc_state(size_of_state(oldfrs[0]->interf));
            set_copy_intfc_states(YES);
            if (( newfr->interf = copy_interface(oldfrs[0]->interf)) == NULL)
            {
                screen("ERROR in assembly_fine_patch_fronts_to_one()",
                     "copy_interface() failed\n");
                clean_up(ERROR);
            }
            set_current_interface(sav_intfc);
            set_copy_intfc_states(sav_copy);
            DEBUG_LEAVE(assembly_fine_patch_fronts_to_one)
            return FUNCTION_SUCCEEDED;   
        }  

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        merge_fronts(newfr,frs,num_fine_frs);
        
        for(i = 0 ; i < num_fine_frs; i++)
            free_front(frs[i]);
        free_these(1,frs);

        front = oldfrs[0];
        dim = front->rect_grid->dim;
        for(dir = 0; dir < dim; dir++)
        {
            for(side = 0; side < 2; side ++)
            {
                rect_boundary_type(newfr->interf, dir, side) =
                    rect_boundary_type(front->interf, dir, side);
            }
        }


        if (debugging("assembly_fine_patch_fronts_to_one"))
        {
            printf("Patch interface after merging:\n");
            print_interface(newfr->interf);
        }

        // assume only zero level patch contains boundary curves 
        retrieve_boundary_curves_from_zero(newfr, oldfrs[0]->interf);

        newfr->rect_grid = oldfrs[0]->rect_grid;
        Computational_grid(newfr->interf) = Computational_grid(oldfrs[0]->interf);
        topological_grid(newfr->interf) = topological_grid(oldfrs[0]->interf); 

        if(debugging("assembly_fine_patch_fronts_to_one"))
        {
            printf("In assembly_patch_fronts_to_one()\n");
            printf("Print the assembled interface with boundary\n");
            print_interface(newfr->interf);
            show_intfc_states(newfr->interf);
            screen("EXIT in assembly_fine_patch_fronts_to_one\n");
            exit(0);
        }

        DEBUG_LEAVE(assembly_fine_patch_fronts_to_one)
        return FUNCTION_SUCCEEDED; 
}
*/

LOCAL  void amr_clip_to_interior_region(
        INTERFACE       *intfc,
        int             *lbuf,
        int             *ubuf,
        int             all_force)
{
        RECT_GRID       *gr = computational_grid(intfc);
        int             dir, dim = gr->dim;
        bool            force_clip;
        NODE            **n;

        DEBUG_ENTER(amr_clip_to_interior_region)

        for (dir = 0; dir < dim; dir++)
        {
            if (lbuf[dir] > 0)
            {
                if(YES == all_force)
                {
                    force_clip = YES;
                }
                else
                {
                    if(rect_boundary_type(intfc,dir,0) == REFLECTION_BOUNDARY ||
                       rect_boundary_type(intfc,dir,0) == AMR_SUBDOMAIN_BOUNDARY)
                        force_clip = YES;
                    else
                        force_clip = NO;
                }
                /* See fscat2d.c for the comment. */ 
                set_cut_none_local_flag(YES);  
                cut_interface(intfc,gr->L[dir],dir,1,YES,force_clip);
                set_cut_none_local_flag(NO);  
            }
            if (ubuf[dir] > 0)
            {
                if(YES == all_force)
                {
                    force_clip = YES;
                }
                else
                {
                    if(rect_boundary_type(intfc,dir,1) == REFLECTION_BOUNDARY ||
                       rect_boundary_type(intfc,dir,1) == AMR_SUBDOMAIN_BOUNDARY)
                        force_clip = YES;
                    else
                        force_clip = NO;
                } 
                cut_interface(intfc,gr->U[dir],dir,0,YES,force_clip);
            }
        }
        /* 
        *  To assure that if two nodes from twodifferent patches matches, 
        *  one should set to be LOCAL, the other should not set 
        *  to be LOCAL. This is required by the match_overlaps.
        *  For nodes overlap with patch computational rect. boundry, 
        *  we now set node on lower side X_CUT and Y_CUT to be NONE_LOCAL,
        *  node on upper side X_CUT and Y_CUT to be LOCAL. 
        *  See set_cut_none_local_flag().  
        */  
        DEBUG_LEAVE(amr_clip_to_interior_region)
}

LOCAL  int merge_fronts_ver2(
        Front *newfront,
        Front **frs,         /* Fronts all come from finest level. */  
        int num_frs)
{
        int             i, j, dir, dim = frs[0]->rect_grid->dim;
        int             *lbuf,*ubuf;
        int             patch_num;
        INTERFACE       *sav_intfc;
        CURVE           **c;
        RECT_GRID       *gr = computational_grid(newfront->interf); 
        int             n_nodes;
        Locstate        *left_s_states, *left_e_states;
        Locstate        *right_s_states, *right_e_states; 
        float           dist; 
        bool            sav_copy;
        bool            save_intrp;
        bool            status;

        debug_print("repatch","Entering merge_fronts()\n");

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_current_interface(newfront->interf);  

        /* remove subdomain bdry curves and physical curves */
        delete_subdomain_curves(newfront->interf);
        delete_passive_boundaries(newfront->interf);
        delete_interior_curves(newfront->interf);  

        /* 
         * 070903 add. Use this flag, not to alter the interface during 
         * the interface assembly and distribution process.
         */
        set_min_sc_sep_val_flag(YES); 
        for (i = 0; i < num_frs; i++)
        {  
            RECT_GRID   *tmpgr; 
            tmpgr = computational_grid(frs[i]->interf); 
            clip_interface_with_rect(newfront->interf,tmpgr->L,tmpgr->U,YES); 
        }  
        set_min_sc_sep_val_flag(NO); 

        if (debugging("merge_fronts"))
        {
            /* 
            print_interface(newfront->interf);
            geomview_intfc_plot2d("jet_patch0_intfc",
                      newfront->interf, newfront->rect_grid); 
            printf("EXIT Base interface after clip\n");
            exit(0); 
            */  
        }
 
        for (i = 0; i < num_frs; i++)
        {
            /* 
             * 051503: Only the nodes of physical curves (contact curve, etc.)
             * which overlap with the base grid computational domain boundary
             * are cleaned cut_node_flags. Because no two physical curves
             * will be merged on these nodes at this moment.
             */  
            for(c = frs[i]->interf->curves; c && *c; c++)
            {
                if(wave_type(*c) < FIRST_PHYSICS_WAVE_TYPE) 
                    continue;  
                for(dir = 0; dir < dim; dir++)
                {
                    /* This is important. Some nodes pass both
                     * rect. patch corner and subdomain bondary will be
                     * set with wrong cut-node (X_CUT, Y_CUT) flags. 
                     * These wrong flags should be cleaned up.
                     */
                    if((Coords((*c)->start->posn)[dir] <= gr->L[dir] || 
                        Coords((*c)->start->posn)[dir] >= gr->U[dir]) &&
                        (!is_adj_bond_node((*c)->start)) ) 
                        clear_node_flags((*c)->start);
                    if((Coords((*c)->end->posn)[dir] <= gr->L[dir] || 
                        Coords((*c)->end->posn)[dir] >= gr->U[dir]) &&
                        (!is_adj_bond_node((*c)->end)) )
                        clear_node_flags((*c)->end);
                } 
            } 
        }

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        save_intrp = interpolate_intfc_states(newfront->interf);

        set_current_interface(newfront->interf);
        interpolate_intfc_states(newfront->interf) = NO;
        for (i = 0; i < num_frs; i++)
            copy_interface_into(frs[i]->interf,newfront->interf);
        interpolate_intfc_states(newfront->interf) = save_intrp;

        if (debugging("merge_fronts"))
        {
            printf("BEFORE MERGE interface in merge_fronts_ver2\n");
            for(c = newfront->interf->curves; c && *c; c++)
            {
                if(wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE) 
                    print_curve(*c);   
            }
            for(NODE **n = newfront->interf->nodes; n && *n; n++)
            {
                /* 
                printf("NODE[%25.23f,%25.23f]\n", Coords((*n)->posn)[0],
                   Coords((*n)->posn)[1]);
                */
                print_node(*n);  
                print_node_flags(*n);
            }
        }

        for(i = 0; i < dim; i++)
        {
            status = merge_interface(newfront,i);
            if (status == FUNCTION_FAILED)
            {
                (void) printf("ERROR in merge_fronts(), "
                      "merge_interface() failed at dir = %d\n", i);
                clean_up(ERROR);
            }
        }

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        if (debugging("merge_fronts"))
        {
            printf("AFTER MERGE interface in merge_fronts_ver2\n");
            for(c = newfront->interf->curves; c && *c; c++)
            {
                if(wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE) 
                    print_curve(*c);   
            }
            for(NODE **n = newfront->interf->nodes; n && *n; n++)
            {
                /* 
                printf("NODE[%25.23f,%25.23f]\n", Coords((*n)->posn)[0],
                   Coords((*n)->posn)[1]);
                */
                print_node(*n);  
                print_node_flags(*n);
            }
            printf("End of merge_fronts_ver2\n"); 
        }

        if (check_for_cut_nodes("at end of merge_fronts_ver2()",newfront->interf) == YES)
        {
            (void) printf("WARNING in  merge_fronts_ver2()");
            (void) printf(" check_for_cut_nodes() detected cut node\n");
            return FUNCTION_FAILED; 
            /* clean_up(ERROR); */
        }

        debug_print("repatch","Leaving merge_fronts_ver2()\n");
        return FUNCTION_SUCCEEDED;  
} 

/* If the base interface boundary nodes and the assembled
*  new interface have same nodes, remove the duplicated 
*  nodes from interface after copied the base interface into
*  the assembled interface. 
*/ 
/* THIS function is not used anymore
LOCAL    void remove_dup_nodes_from_assembly_intfc(
        INTERFACE       *intfc)
{
        int             dim, i;  
        NODE            **n, **nextn;
        CURVE           **c;    
        BOND            *b;   
        NODE            **delete_nodes = NULL;
        float           dist; 
        INTERFACE       *sav_intfc; 


        DEBUG_ENTER(remove_dup_nodes_from_assembly_intfc) 

        sav_intfc = current_interface();
        set_current_interface(intfc); 
        dim = intfc->dim;  
loop_again:  
        for (n = intfc->nodes; n && *n; n++)
        {
            for(nextn = n+1; nextn && *nextn; nextn++)
            {
                dist = 0.0; 
                for(i = 0; i < dim; i++)
                {
                    dist += fabs(Coords((*n)->posn)[i]-Coords((*nextn)->posn)[i]); 
                } 
                if( dist <= MACH_EPS*1000) 
                {
                    for(c = (*nextn)->in_curves; c && *c; c++)
                         change_node_of_curve(*c,NEGATIVE_ORIENTATION,*n);  
                    for(c = (*nextn)->out_curves; c && *c; c++)
                         change_node_of_curve(*c,POSITIVE_ORIENTATION,*n); 
                    delete_node(*nextn);
                    goto loop_again;  
                }  
            }  
        } 
        set_current_interface(sav_intfc);  

        DEBUG_LEAVE(remove_dup_nodes_from_assembly_intfc) 
}  
*/

/* THIS function is not used in the current assembly_dist()
LOCAL   void retrieve_boundary_curves_from_zero(
        Front           *newfr,
        INTERFACE       *intfc)
{
        INTERFACE       *tmp_intfc, *sav_intfc, *send_intfc;
        NODE            **n;
        CURVE           **c;
        CURVE           **delete_curves = NULL;
        NODE            **delete_nodes = NULL;
        bool            sav_copy;
        bool            save_intrp;


        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_size_of_intfc_state(size_of_state(intfc));
        set_copy_intfc_states(YES);

        tmp_intfc = copy_interface(intfc);
        if (tmp_intfc == NULL)
        {
            (void) printf("ERROR in retrieve_boundary_curves_from_zero()\n");
            (void) printf("copy_interface() returns NULL\n");
            clean_up(ERROR);
        }

        // remove subdomain bdry curves and interior physical curves 
        delete_subdomain_curves(tmp_intfc);
        for(c = tmp_intfc->curves; c && *c; c++)
        {
            if(wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE)
            {
                if (not add_to_pointers((POINTER)*c,(POINTER**)&delete_curves))
                {
                    screen("ERROR in retrieve_boundary_curves_from_zero(), "
                        "add_to_pointers() failed\n");
                    clean_up(ERROR);
                }
            }
        }
        for (c = delete_curves; c && *c; c++)
            (void) delete_curve(*c);

        delete_nodes = NULL;
        for (n = tmp_intfc->nodes; n && *n; n++)
        {
            if (((*n)->in_curves == NULL) && ((*n)->out_curves == NULL))
            {
                if (not add_to_pointers((POINTER)*n,(POINTER**)&delete_nodes))
                {
                    screen("ERROR in retrieve_boundary_curves_from_zero(), "
                              "add_to_pointers() failed\n");
                    clean_up(ERROR);
                }
            }
        }
        for (n = delete_nodes; n && *n; n++)
            (void) delete_node(*n);

        if(debugging("retrieve_boundary_curves_from_zero"))
        {
            printf("In retrieve_boundary_curves_from_zero\n");
            printf("Print retrieved tmp interface:::\n");
            print_interface(tmp_intfc);
            show_intfc_states(tmp_intfc);
        }

        set_size_of_intfc_state(size_of_state(intfc));
        set_current_interface(newfr->interf);
        save_intrp = interpolate_intfc_states(newfr->interf);
        interpolate_intfc_states(newfr->interf) = NO;
        copy_interface_into(tmp_intfc, newfr->interf);
        delete_interface(tmp_intfc); 

        interpolate_intfc_states(newfr->interf) = save_intrp;
        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        remove_dup_nodes_from_assembly_intfc(newfr->interf); 

        if(debugging("retrieve_boundary_curves_from_zero"))
        {
            printf("after copy tmp_intfc into newfront, print new intfc\n");
            print_interface(newfr->interf);
            show_intfc_states(newfr->interf);
        }
        DEBUG_LEAVE(retrieve_boundary_curves_from_zero)
}  
*/

/*
LOCAL void old_reinstall_undistribute_fronts_and_misc(
        Front      **tmpfrs,
        Front      **frs,
        Wv_on_pc   **redistr_table,
        int        total_patch,    // number of patches originally genereated 
        int        num_patches,    // number of patches compute in this proc 
        int        max_n_patch)
{
        int        source, numnodes, myid;
        int        dist;
        int        patch_id, i;
        Front      *front;
        bool       sav_copy, status;
        COMPONENT  dummy = -1,  i_comp, *i_comps;
        float      coords[MAXD];
        RECT_GRID  *cgr, *tgr;
        RECT_GRID  *rgr;


        numnodes = pp_numnodes();
        myid = pp_mynode();
        sav_copy = copy_intfc_states();

        for(source = 0; source < numnodes; source++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                patch_id = -100;
                if(-1 == redistr_table[source][i].wv_id) continue;
                if(source == redistr_table[source][i].pc_id) continue;

                if(myid == redistr_table[source][i].pc_id)
                {
                    patch_id = redistr_table[source][i].wv_id;
                    front = redistr_table[source][i].front;
                    // printf("redist[%d][%d] back to original: front[%p]->interf:%p\n",
                    //   source, i, front, front->interf);
                    // if(exists_interface(front->interf) != YES)
                    //     printf("WARNING: Invalid Interface send in "
                    //       " assembly_distribute_patch_fronts, "
                    //           "Probably already deleted\n");

                    pp_send(0,(POINTER)(&patch_id),sizeof(int),source);
                    send_front_misc(front,&dummy,source);

                    // printf("patch[%d] front %p: Proc [%d] Send to proc[%d]\n",
                    //         redistr_table[source][i].wv_id, front, myid, source);
                }
                if(source == myid)
                {
                    pp_recv(0,redistr_table[source][i].pc_id,
                            (POINTER)(&patch_id),sizeof(int));
                    recv_front_misc(tmpfrs[patch_id],
                        &dummy, redistr_table[source][i].pc_id);
                    front = tmpfrs[patch_id];
                    front->interf->modified = YES;
                    rgr = front->rect_grid;
                    cgr = computational_grid(front->interf);
                    copy_rect_grid(cgr,rgr);
                    tgr = &topological_grid(front->interf);
                    tgr->Remap.remap = rgr->Remap.remap;
                    set_patch_topo_grid(rgr,tgr);
                    // printf("redist[%d][%d] received at original: ",source,i);
                    // printf("patch[%d]: Proc [%d] recv from proc[%d]\n",
                    //       patch_id, source, redistr_table[source][i].pc_id);
                }
                pp_gsync();
            }
        }
}
*/

LOCAL bool pseudo_scatter_patch_fronts(
        Front      **frs,
        Front      **tmpfrs,
        Front      *glue_fr,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        COMPONENT  *i_comp,
        int        all_level)
{
        int        source, numnodes, myid;
        int        dist, patch_id;
        int        i, j, dim = frs[0]->rect_grid->dim;
        Front      *front;
        int        total_patch, levels;
        bool       sav_copy, status;
        bool       save_intrp;
        INTERFACE  *sav_intfc;
        RECT_GRID  *cgr, *tgr;
        RECT_GRID  *rgr;
        int        item[2]; 

        DEBUG_ENTER(pseudo_scatter_patch_fronts)

        numnodes = pp_numnodes();
        myid = pp_mynode();
        total_patch = tmpfrs[0]->totalNumberOfPatches;
        levels = tmpfrs[0]->NumberOfLevels;

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();

        /* Delete intfc of NON-LOCAL COARSE LEVEL patch if 
         * it will receive one. For the LOCAL COARSE LEVEL 
         * grid fronts, delete the OLD LOCAL front interface,
         * set the grid front interface points to the copy
         * front("tmpfrs") interface.
         */
        if(all_level == YES)
        {
            for(i = 1; i < num_patches; i++)
            {
                front = frs[i]; 
                if(front->patch_level == levels-1)
                    continue;
                if(NO == loc_front_on_redistr_table(front,
                      redistr_table,max_n_patch,item))
                {
                    printf("ERROR in pseudo_scatter_patch_fronts\n");
                    printf("Front is not on the redistribute table\n");
                    clean_up(ERROR);
                }
                if(myid != item[0])
                    delete_patch_all_curves(front);
                if(myid == item[0])
                {
                    patch_id = item[1];
                    delete_interface(front->interf);
                    front->interf = tmpfrs[patch_id]->interf;
                    tmpfrs[patch_id]->interf = NULL;
                }
            }
        }

        /*
         * For the NON-LOCAL FINE grid fronts which are transfered
         * to the other processors, MPI-send the buffer zone
         * interface into them. For the OTHER LEVEL NON-LOCAL grid
         * fronts, the completed interfaces are also sent by the MPI.
         */
        if(all_level == YES)
            patch_front_distribute_to_nodes(tmpfrs, frs, redistr_table,
                num_patches, max_n_patch, 0);
        else
            patch_front_distribute_to_nodes(tmpfrs, frs, redistr_table,
                num_patches, max_n_patch, 2);

        /*
         * For the local FINE grid fronts which are not transfered,
         * copy the buffer zone interface into them.
         */
        for(i = 0; i < max_n_patch; i++)
        {
            int found_same;
            if(-1 == redistr_table[myid][i].wv_id) continue;
            if(myid != redistr_table[myid][i].pc_id) continue;
            if(redistr_table[myid][i].front->patch_level != (levels - 1))
                continue;

            patch_id = redistr_table[myid][i].wv_id;
            front = redistr_table[myid][i].front;

            found_same = NO;
            for(int tmpjj = 0; tmpjj < num_patches; tmpjj++)
            {
                if(frs[tmpjj] == front)
                {
                    found_same = YES;
                    break;
                }
            }
            if(found_same == NO)
            {
                printf("ERROR: pseudo_scatter_patch_fronts()\n");
                printf("front from redistribute table and frs"
                       " are not consistent\n");
                printf("Fronts NOT in redistr_table[%d][%d] and frs\n",
                      myid, i);
                printf("patch_id [%d] from redistr_table\n", patch_id);
                clean_up(ERROR);
            }

            save_intrp = interpolate_intfc_states(front->interf);
            set_current_interface(front->interf);
            interpolate_intfc_states(front->interf) = NO;
            copy_interface_into(tmpfrs[patch_id]->interf,front->interf);
            interpolate_intfc_states(front->interf) = save_intrp;

            delete_interface(tmpfrs[patch_id]->interf);
            tmpfrs[patch_id]->interf = NULL;

            for(j = 0; j < dim; j++)
            {
                status = merge_interface(front,j);
                if (status == FUNCTION_FAILED)
                {
                    printf("ERROR in pseudo_scatter_patch_fronts(), "
                      "local merge_interface() failed at dir = %d\n", j);
                    print_interface(front->interf);
                    clean_up(ERROR);
                }
            }
        }

        /*
         * For the NON-LOCAL FINE grid fronts, interface needs to be merged.
         * For the other level non-local grid fronts, set rect_grids.
         */
        for(source = 0; source < numnodes; source++)
        {
            for(i = 1; i < max_n_patch; i++)
            {
                if(-1 == redistr_table[source][i].wv_id) continue;
                if(source == redistr_table[source][i].pc_id) continue;
                if(myid == source) continue;
                if(myid != redistr_table[source][i].pc_id) continue;

                if(redistr_table[source][i].wv_level != (levels - 1))
                {
                    front = front_on_redistr_table(redistr_table,
                         max_n_patch, myid, redistr_table[source][i].wv_id, source);
                    rgr = front->rect_grid;
                    cgr = computational_grid(front->interf);
                    copy_rect_grid(cgr,rgr);
                    tgr = &topological_grid(front->interf);
                    tgr->Remap.remap = rgr->Remap.remap;
                    set_patch_topo_grid(rgr,tgr);
                    continue;
                }
                front = front_on_redistr_table(redistr_table,
                      max_n_patch, myid, redistr_table[source][i].wv_id, source);
                set_current_interface(front->interf);

                for(j = 0; j < dim; j++)
                {
                    status = merge_interface(front,j);
                    if (status == FUNCTION_FAILED)
                    {
                        printf("ERROR in pseudo_scatter_patch_fronts(), "
                          "non-local merge_interface() failed at dir = %d\n", j);
                        clean_up(ERROR);
                    }
                }
            }
        }

        for(i = 1; i < num_patches; i++)
        {
            if (exists_interface(frs[i]->interf) != YES)
            {
                printf("ERROR in pseudo_scatter_patch_fronts\n");
                printf("Patch[%d] interface is not in exist lists\n", i);
                printf("Patch intfc = %p\n", frs[i]->interf);
                clean_up(ERROR);
            }
        }

        /* Complete the interfaces with boundary and physics info. link.
         * Only the FINE grid interface needs to do so. Other level
         * interfaces have been taken care of.
         */
        for( i = 1; i < num_patches; i++)
        {
            if(frs[i]->patch_level != (levels-1))
                continue;
            use_delete_short_bonds = NO;
            (*f_user_interface(frs[i]->interf)._form_patch_subintfc)(frs[i],i_comp[i]);
        }

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);
        DEBUG_LEAVE(pseudo_scatter_patch_fronts)
        return FUNCTION_SUCCEEDED;
}


#endif  /* if defined(USE_OVERTURE) */
#endif  /* if defined(TWOD) */

