/*
*                               famrscat2d.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*
*       this file includes functions set_patch_front, set_patch_topo_grid,
*                       and set_patch_comput_grid
*/

#define DEBUG_STRING    "famrscat2d"

#include <front/flocaldecs.h>

#if defined(TWOD)
#if defined(USE_OVERTURE)

struct _Transbox_set{
      int     item[2]; /* This set of points is from table item */
      int     n_box;
      Trans_box *Tbox;
};
typedef struct _Transbox_set Transbox_set;

struct _Trans_intfc{
      int     item[2];
      int     to_item[2];
      int     from_item[2];
      INTERFACE  *intfc; 
      
      struct  Table  *table;
      INTERFACE  *irecv_intfc; 
#if defined(__MPI__)
      int         ipp_tag;
      MPI_Request ipp_reqs;
      MPI_Status  ipp_stat;
#endif /* if defined(__MPI__) */
}; 
typedef struct _Trans_intfc Trans_intfc; 

LOCAL int       set_send_scatter_fronts(Front**,int,Overparam*,Wv_on_pc**,int,Transbox_set**);
LOCAL int       set_recv_scatter_fronts(Front**,int,Overparam*,Wv_on_pc**,int,Transbox_set**);
LOCAL int       get_trans_front_blk(Wv_on_pc,int,int,int*,int*); 
LOCAL void      set_grid_interior_domain(Wv_on_pc,int,int*,int*); 
LOCAL Transbox_set *front_buf_scattered_by_patches(int,int,int*,Wv_on_pc**,int,
                           PP_GRID*,Overparam*); 
LOCAL Trans_box    *add_info_to_box(int*,int*,int*,int*,int*,int,Trans_box*,int*,int*);
LOCAL Transbox_set *scatter_ffront_buf_in_domain(int*,int,int,Wv_on_pc**,int,PP_GRID*,Overparam*); 
LOCAL int       base_scatter_fine_front_buf_in_domain(Transbox_set**,int*,int,
                     Wv_on_pc**,int,Overparam*,PP_GRID*);
LOCAL int       finefr_buf_supplied_by_coarse(Transbox_set*); 
LOCAL Trans_intfc **frs_make_send_intfc(Front**,int,Wv_on_pc**,int,Overparam*,int,Transbox_set*); 
LOCAL int       **count_send_fintfc_blk(int*,int,Transbox_set*,int*); 
LOCAL void      set_send_fblk_intfc(Trans_intfc*,Front*,int,Transbox_set*,
                   Wv_on_pc**,int); 
LOCAL void      set_c_send_fblk_intfc(Trans_intfc*,Front*,int,Transbox_set*,
                   Wv_on_pc**,int,Overparam*); 
LOCAL void      rect_crds(RECT_GRID*,int*,int*,float*,float*); 
LOCAL INTERFACE *make_rect_cut_intfc(Front*,int*,int*,Wv_on_pc**,int*,int*); 
LOCAL void      cut_intfc_into_trans(INTERFACE*,Trans_intfc*); 
LOCAL void      make_coarse_rect_cut_intfc(Transbox_set*,Trans_intfc*,Front*,Wv_on_pc**,
                      int,Overparam*); 
LOCAL Trans_intfc **frs_make_recv_intfc_storage(Front**,int,Wv_on_pc**,int,int,Transbox_set*);
LOCAL int       **count_recv_fintfc_blk(int*,int,Transbox_set*,int*); 
LOCAL void      frs_recv_intfc(Front**,int,Wv_on_pc**,int,int,Transbox_set*,int,
                     Transbox_set*,Trans_intfc**,Trans_intfc**);
LOCAL void      frs_recv_intfc_chunks(Front**,int,Wv_on_pc**,int,int,Transbox_set*,int,
                     Transbox_set*,Trans_intfc**,Trans_intfc**);
LOCAL void      frs_recv_intfc_tables(Front**,int,Wv_on_pc**,int,int,Transbox_set*,int,
                     Transbox_set*,Trans_intfc**,Trans_intfc**);
LOCAL Trans_intfc  *find_loc_match_snd_intfc(Trans_intfc,Trans_intfc**,Transbox_set*,int,
                        Front**,int,Wv_on_pc**,int); 
LOCAL void      free_send_trans_intfc_storage(Front**,int,Wv_on_pc**,int,int,Transbox_set*,
                   Trans_intfc**); 
LOCAL int       merge_fintfc_interior_with_buff(Front**,int,Wv_on_pc**,int,int,Transbox_set*,
                   Trans_intfc**); 
LOCAL int       make_intfc_reflection_buf(Front*,int*); 
LOCAL INTERFACE   *set_send_intfc_w_node_flag(INTERFACE*,int,int,int*,int*);
LOCAL void        clip_patch_fronts_to_interior_ver2(Front**,int,COMPONENT*); 
LOCAL void        amr_clip_to_interior_region_ver2(INTERFACE*,int*,int*,int); 
LOCAL INTERFACE   *set_send_intfc_ver2(INTERFACE*,int,int,int*,int*);

EXPORT int loc_front_on_redistr_table(
        Front        *front,
        Wv_on_pc     **redistr_table,
        int          max_n_patch,
        int          *item)
{
        int          numnodes;
        int          i, src;

        numnodes = pp_numnodes();

        if(front == NULL)
            return NO;
        for(src = 0; src < numnodes; src++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                if(redistr_table[src][i].wv_id == -1 ||
                   redistr_table[src][i].wv_id != front->patch_number ||
                   redistr_table[src][i].wv_level != front->patch_level)
                    continue;
                if(redistr_table[src][i].front == front)
                {
                    item[0] = src;
                    item[1] = i;
                    return YES;
                }
            }
        }
        return NO;
}



LIB_LOCAL int    scatter_patch_fronts2d(
        Front      **frs,
        int        num_patches,    /* number of patches computed in the proc */
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        Overparam  *overparam,
        int        all_level)
{
        int        i, j;
        bool       sav_copy;
        Transbox_set  *gettrans = NULL;
        int        get_blk_n;
        Transbox_set  *sndtrans = NULL;
        int        snd_blk_n;
        Trans_intfc   **gettransintfc, **sndtransintfc;
        INTERFACE     *save_intfc[2];
        int           status;

        DEBUG_ENTER(scatter_patch_fronts2d)

#if defined(SMOOTH_INTERFACE_FOR_SCATTER)
        /* This routine is responsible for simplifing
         * interface across the domain boundary.
         * So as to make the followed scattering and
         * assembly interface routine more robust.
         */
        start_clock("amr_fr_scat_adjust"); 
        for(i = 0; i < num_patches; i++)
        {
            if(frs[i]->patch_level == frs[i]->NumberOfLevels-1)
                adjust_front_boundary_crxing_spacing_1(frs[i]);
        }
        stop_clock("amr_fr_scat_adjust"); 
#endif /* if defined(SMOOTH_INTERFACE_FOR_SCATTER) */

        get_blk_n = set_recv_scatter_fronts(frs,num_patches,overparam,
                          redistr_table,max_n_patch,&gettrans);

        /* 
        printf("\n\n++++++++++++++++++++++++++++++++\n"); 
        for(i = 0; i < get_blk_n; i++)
        {
            printf("Item[%d][%d] L[%d %d] U[%d %d] recv buf front from :\n", 
             gettrans[i].item[0], gettrans[i].item[1],
             redistr_table[gettrans[i].item[0]][gettrans[i].item[1]].base[0],
             redistr_table[gettrans[i].item[0]][gettrans[i].item[1]].base[1],
             redistr_table[gettrans[i].item[0]][gettrans[i].item[1]].bound[0],
             redistr_table[gettrans[i].item[0]][gettrans[i].item[1]].bound[1]);

            for(j = 0; j < gettrans[i].n_box; j++)
            {
                printf("FROM item[%d][%d] L[%d %d] U[%d %d]\n",
                   gettrans[i].Tbox[j].from_item[0], 
                   gettrans[i].Tbox[j].from_item[1], 
                   gettrans[i].Tbox[j].base[0], gettrans[i].Tbox[j].base[1], 
                   gettrans[i].Tbox[j].bound[0], gettrans[i].Tbox[j].bound[1]);
            }
        }
        */

        snd_blk_n = set_send_scatter_fronts(frs,num_patches,overparam,
                          redistr_table,max_n_patch,&sndtrans);

        /*
        printf("\n\n++++++++++++++++++++++++++++++++\n"); 
        for(i = 0; i < snd_blk_n; i++)
        {
            printf("Item[%d][%d] send front buf to :\n",
             sndtrans[i].item[0], sndtrans[i].item[1]);
            if(sndtrans[i].item[1] == 0)
            {
                printf("Made on coarse\n"); 
                for(j = 0; j < sndtrans[i].n_box; j++)
                {
                    if(sndtrans[i].Tbox[j].from_item[1] == 0)
                    {
                        printf("[%d]'s trans TO item[%d][%d] L[%d %d] U[%d %d]\n",
                           j, sndtrans[i].Tbox[j].to_item[0], 
                           sndtrans[i].Tbox[j].to_item[1], 
                           sndtrans[i].Tbox[j].base[0], sndtrans[i].Tbox[j].base[1], 
                           sndtrans[i].Tbox[j].bound[0], sndtrans[i].Tbox[j].bound[1]);
                    }
                    else
                    {
                        printf("[%d]'s trans TO item[%d][%d] L[%d %d] U[%d %d] a hole by [%d][%d]\n",
                           j, sndtrans[i].Tbox[j].to_item[0], 
                           sndtrans[i].Tbox[j].to_item[1], 
                           sndtrans[i].Tbox[j].base[0], sndtrans[i].Tbox[j].base[1], 
                           sndtrans[i].Tbox[j].bound[0], sndtrans[i].Tbox[j].bound[1],
                           sndtrans[i].Tbox[j].from_item[0],
                           sndtrans[i].Tbox[j].from_item[1]);
                    }
                }
                printf("\n"); 
            }
            else
            {
                for(j = 0; j < sndtrans[i].n_box; j++)
                {
                    printf("TO item[%d][%d] L[%d %d] U[%d %d]\n",
                       sndtrans[i].Tbox[j].to_item[0], 
                       sndtrans[i].Tbox[j].to_item[1], 
                       sndtrans[i].Tbox[j].base[0], sndtrans[i].Tbox[j].base[1], 
                       sndtrans[i].Tbox[j].bound[0], sndtrans[i].Tbox[j].bound[1]);
                }
            }     
        }
        */

        start_clock("amr_fr_scat_mk_recv"); 
        gettransintfc = frs_make_recv_intfc_storage(frs,num_patches,redistr_table,
             max_n_patch,get_blk_n,gettrans);
        stop_clock("amr_fr_scat_mk_recv"); 

        start_clock("amr_fr_scat_0fr"); 
        set_current_interface(frs[0]->interf);
        set_amr_intfc_tol(frs[0]->interf,pow(2.0,1.0-frs[0]->NumberOfLevels));
        set_use_delete_short_bonds_flag();

        save_intfc[0] = make_save_intfc(frs[0]->interf);
        save_intfc[1] = NULL;
        if(! scatter_front(frs[0]))
        {
            printf("WARNING: scatter_patch_fronts2d(), "
                 "scatter_front() failed for the base front\n");
            reset_interface_of_front(frs[0],save_intfc);
        }
        else
            delete_interface(save_intfc[0]);
        set_amr_intfc_tol(frs[0]->interf,pow(2.0,frs[0]->NumberOfLevels-1.0));
        stop_clock("amr_fr_scat_0fr"); 

        start_clock("amr_fr_scat_mk_send"); 
        sndtransintfc = frs_make_send_intfc(frs,num_patches,redistr_table,
             max_n_patch,overparam,snd_blk_n,sndtrans);  
        stop_clock("amr_fr_scat_mk_send"); 

        frs_recv_intfc(frs,num_patches,redistr_table,max_n_patch,
             get_blk_n,gettrans,snd_blk_n,sndtrans,sndtransintfc,gettransintfc); 

        free_send_trans_intfc_storage(frs,num_patches,redistr_table,max_n_patch,
                    snd_blk_n,sndtrans,sndtransintfc); 

        start_clock("amr_fr_scat_merg"); 
        status = merge_fintfc_interior_with_buff(frs,num_patches,redistr_table,max_n_patch,
                    get_blk_n,gettrans,gettransintfc); 
        stop_clock("amr_fr_scat_merg"); 

        for(i = 0; i < get_blk_n; i++)
            free(gettrans[i].Tbox); 
        if(gettrans != NULL)    
            free(gettrans); 
        for(i = 0; i < snd_blk_n; i++)
            free(sndtrans[i].Tbox); 
        if(sndtrans != NULL)    
            free(sndtrans); 
 
        DEBUG_LEAVE(scatter_patch_fronts2d)
        return status; 
}

LOCAL int merge_fintfc_interior_with_buff(
        Front      **frs,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int         get_blk_n,
        Transbox_set  *gettrans,
        Trans_intfc   **gettransintfc)
{
        COMPONENT    *i_comps; 
        int          grid, j, level, **do_item; 
        int          item[2], n_get_frs, dim; 
        Front        *front; 
        bool         sav_copy, save_intrp;
        INTERFACE    *sav_intfc;
        int          status = FUNCTION_SUCCEEDED; 

        DEBUG_ENTER(merge_fintfc_interior_with_buff) 

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        dim = frs[0]->rect_grid->dim; 

        vector(&i_comps, num_patches, sizeof(COMPONENT));
        clip_patch_fronts_to_interior_ver2(frs, num_patches, i_comps);
 
        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                do_item = count_recv_fintfc_blk(item,get_blk_n,gettrans,&n_get_frs);

                save_intrp = interpolate_intfc_states(front->interf);
                set_current_interface(front->interf);
                interpolate_intfc_states(front->interf) = NO;

                for(j = 0; j < n_get_frs; j++)
                {
                    if(gettransintfc[grid][j].irecv_intfc == NULL)
                        continue; 
                    copy_interface_into(gettransintfc[grid][j].irecv_intfc,front->interf);
                } 

                interpolate_intfc_states(front->interf) = save_intrp;

                status = make_intfc_reflection_buf(front,item); 
                if (status == FUNCTION_FAILED)
                {
                    printf("WARNING: merge_fintfc_interior_with_buff\n");  
                    printf("(item[%d][%d]) front[%d]_level[%d]" 
                           " make_intfc_reflection_buf failed\n",
                         item[0],item[1],front->patch_number, front->patch_level); 
                    break;   
                }
                for(j = 0; j < dim; j++)
                {
                    status = merge_interface(front,j);
                    if (status == FUNCTION_FAILED)
                    {
                        printf("ERROR in merge_fintfc_interior_with_buff(), "
                          "local merge_interface() failed at dir = %d\n", j);
                        print_interface(front->interf);
                        clean_up(ERROR);
                    }
                }
                status = (*f_user_interface(front->interf)._form_patch_subintfc)(
                               front,i_comps[grid]);
                if (status == FUNCTION_FAILED)
                {
                    printf("WARNING: merge_fintfc_interior_with_buff\n");  
                    printf("(item[%d][%d]) front[%d]_level[%d] form subintfc failed\n",
                         item[0],item[1],front->patch_number, front->patch_level); 
                    break; 
                }
            }
        } 

        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                do_item = count_recv_fintfc_blk(item,get_blk_n,gettrans,&n_get_frs);
                for(j = 0; j < n_get_frs; j++)
                {
                    delete_interface(gettransintfc[grid][j].irecv_intfc); 
                }
                if(n_get_frs != 0)
                    free(gettransintfc[grid]);
            }
        }
        free(gettransintfc); 
        free(i_comps); 

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(merge_fintfc_interior_with_buff) 
        return status; 
}

/* Before the fine grid interface gets its buffer zone interface,
 * it is trimmed into interior. Then this interior part of 
 * the interface merged with the obtained buffer zone interface. 
 * The difference between this function call and clip_patch_fronts_to_interior()
 * is in the clip node type set. In this function, the cut-types of nodes
 * on the REFLECTING boundary sides are all LOCAL. 
 */
LOCAL void clip_patch_fronts_to_interior_ver2(
        Front      **frs,
        int        num_patches,
        COMPONENT  *i_comp)    /* number of patches computed in the proc */
{
        INTERFACE  *sav_intfc;
        bool       sav_copy, save_intrp;
        int        *lbuf,*ubuf;
        int        i, j, dim;
        float      coords[MAXD];
        RECT_GRID  *gr;

        sav_intfc = current_interface();
        set_min_sc_sep_val_flag(YES);

        for(i = 0; i < num_patches; i++)
        {
            if(frs[i]->patch_level == frs[i]->NumberOfLevels-1)
            {
                lbuf = frs[i]->rect_grid->lbuf;
                ubuf = frs[i]->rect_grid->ubuf;
                set_current_interface(frs[i]->interf);
                save_intrp = interpolate_intfc_states(frs[i]->interf);
                interpolate_intfc_states(frs[i]->interf) = NO;
                for(j = 0; j < frs[i]->rect_grid->dim; j++)
                    coords[j] = grid_center_coord(j,frs[i]->rect_grid);
                i_comp[i] = (frs[i]->interf->modified) ?
                    long_component(coords,frs[i]->interf) :
                    component(coords,frs[i]->interf);

                delete_subdomain_curves(frs[i]->interf);
                delete_passive_boundaries(frs[i]->interf);

                if(debugging("clip_fronts_to_interior"))
                {
                }

                amr_clip_to_interior_region_ver2(frs[i]->interf,lbuf,ubuf,YES);
                interpolate_intfc_states(frs[i]->interf) = save_intrp;

                /*
                if(frs[i]->patch_number == 2 && frs[i]->patch_level == 2)
                {
                    printf("AFTER clip front 2, level 2 into interior\n");
                    for(NODE **n = frs[i]->interf->nodes; n && *n; ++n)
                    {
                        print_node(*n);
                        print_node_flags(*n);
                    }
                }
                */
            }
        }
        set_min_sc_sep_val_flag(NO);
        set_current_interface(sav_intfc);
}


LOCAL  void amr_clip_to_interior_region_ver2(
        INTERFACE       *intfc,
        int             *lbuf,
        int             *ubuf,
        int             all_force)
{
        RECT_GRID       *gr = computational_grid(intfc);
        int             dir, dim = gr->dim;
        bool            force_clip;
        NODE            **n;

        DEBUG_ENTER(amr_clip_to_interior_region_ver2)

        for (dir = 0; dir < dim; dir++)
        {
            if (lbuf[dir] > 0)
            {
                if(YES == all_force)
                {
                    force_clip = YES;
                }
                else
                {
                    if(rect_boundary_type(intfc,dir,0) == REFLECTION_BOUNDARY ||
                       rect_boundary_type(intfc,dir,0) == AMR_SUBDOMAIN_BOUNDARY)
                        force_clip = YES;
                    else
                        force_clip = NO;
                }
                // if(rect_boundary_type(intfc,dir,0) != REFLECTION_BOUNDARY)
                //     set_cut_none_local_flag(YES);
                cut_interface(intfc,gr->L[dir],dir,1,YES,force_clip);
                // if(rect_boundary_type(intfc,dir,0) != REFLECTION_BOUNDARY)
                //     set_cut_none_local_flag(NO);
            }
            if (ubuf[dir] > 0)
            {
                if(YES == all_force)
                {
                    force_clip = YES;
                }
                else
                {
                    if(rect_boundary_type(intfc,dir,1) == REFLECTION_BOUNDARY ||
                       rect_boundary_type(intfc,dir,1) == AMR_SUBDOMAIN_BOUNDARY)
                        force_clip = YES;
                    else
                        force_clip = NO;
                }
                cut_interface(intfc,gr->U[dir],dir,0,YES,force_clip);
            }
        }

        DEBUG_LEAVE(amr_clip_to_interior_region_ver2)
}







LOCAL int make_intfc_reflection_buf(
        Front      *front,
        int        *item)
{
        INTERFACE       *intfc = front->interf;
        INTERFACE       *sav_intfc;
        RECT_GRID       *gr = front->rect_grid;
        bool            sav_copy, save_intrp, status = FUNCTION_SUCCEEDED;
        int             i, j, dir, dim = gr->dim, side;
        INTERFACE       *recv_intfc;
        int             *buf;
        static float nors[] = {  1.0,  0.0,  0.0,
                                 0.0,  1.0,  0.0,
                                 0.0,  0.0,  1.0,
                                -1.0,  0.0,  0.0,
                                 0.0, -1.0,  0.0,
                                 0.0,  0.0, -1.0};
        float        *nor, p[MAXD];
        int          me[MAXD], myid, need_merge = NO;  
        PP_GRID      *pp_grid = front->pp_grid;
        int          *G = pp_grid->gmax;

        DEBUG_ENTER(make_intfc_reflection_buf)

        myid = item[0];
        find_Cartesian_coordinates(myid, pp_grid, me);

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_current_interface(intfc);
        set_copy_intfc_states(YES); 
        save_intrp = interpolate_intfc_states(intfc);
        interpolate_intfc_states(intfc) = NO;

        for(dir = 0; dir < dim; dir++)
        {
            for(side = 0; side < 2; side++)
            {
                if(rect_boundary_type(intfc,dir,side) == REFLECTION_BOUNDARY)
                {
                    // need_merge = YES; 
                    buf = (side == 0) ? gr->lbuf : gr->ubuf;
                    nor = nors + 3*dir + 9*side;
                    recv_intfc = set_send_intfc_ver2(intfc,dir,side,me,G); 
                    p[dir] = (side) ? gr->U[dir] : gr->L[dir];
                    for (j = 1; j < dim; ++j)
                    {
                        int    k = (j+dir)%dim;
                        p[k] = 0.5*(gr->U[k] + gr->L[k]);
                    }
                    reflect_interface(recv_intfc,p,nor);
                    copy_interface_into(recv_intfc,intfc);
                    delete_interface(recv_intfc); 
                }
            } 

            /*
            if(need_merge == YES)
                status = merge_interface(front,dir);
            if (!status)
            {
                (void) printf("WARNING in make_intfc_reflection_buf(), "
                          "merge_interface() failed\n");
            }
            need_merge = NO; 
            */
        }

        interpolate_intfc_states(intfc) = save_intrp;
        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(make_intfc_reflection_buf)
        return status;  
}


LOCAL void free_send_trans_intfc_storage(
        Front      **frs,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int         snd_blk_n,
        Transbox_set  *sndtrans,
        Trans_intfc   **sndtransintfc)
{
        int          j, grid, level;
        Front        *front;
        int          myid, item[2];
        int          to_it[2], send_dst_pc;
        int          **do_item, n_snd_frs;
 
        myid = pp_mynode(); 
        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1 ||
               level == 0)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                do_item = count_send_fintfc_blk(item,snd_blk_n,sndtrans,&n_snd_frs);
                for(j = 0; j < n_snd_frs; j++)
                {
                    to_it[0] = sndtransintfc[grid][j].to_item[0];
                    to_it[1] = sndtransintfc[grid][j].to_item[1];
                    send_dst_pc = redistr_table[to_it[0]][to_it[1]].pc_id;
                    if(myid != send_dst_pc)
                        delete_interface(sndtransintfc[grid][j].intfc); 
                }
                if(n_snd_frs != 0)
                    free(sndtransintfc[grid]);  
            }
        }
        free(sndtransintfc); 
}

LOCAL void frs_recv_intfc(
        Front      **frs,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int         get_blk_n,
        Transbox_set  *gettrans,
        int         snd_blk_n,
        Transbox_set  *sndtrans,
        Trans_intfc   **sndtransintfc,
        Trans_intfc   **gettransintfc)
{
        DEBUG_ENTER(frs_recv_intfc)

        start_clock("amr_fr_scat_gettbl"); 
        frs_recv_intfc_tables(frs,num_patches,redistr_table,max_n_patch,get_blk_n,
           gettrans,snd_blk_n,sndtrans,sndtransintfc,gettransintfc); 
        stop_clock("amr_fr_scat_gettbl"); 

        start_clock("amr_fr_scat_getchunk"); 
        frs_recv_intfc_chunks(frs,num_patches,redistr_table,max_n_patch,get_blk_n,
           gettrans,snd_blk_n,sndtrans,sndtransintfc,gettransintfc); 
        stop_clock("amr_fr_scat_getchunk"); 
        
        DEBUG_LEAVE(frs_recv_intfc)
}

LOCAL void frs_recv_intfc_tables(
        Front      **frs,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int         get_blk_n,
        Transbox_set  *gettrans,
        int         snd_blk_n,
        Transbox_set  *sndtrans,
        Trans_intfc   **sndtransintfc,
        Trans_intfc   **gettransintfc)
{
        Trans_intfc  **ppgettrans, **ppsndtrans;
        int          pp_get, pp_snd, pp_getalloc, pp_sndalloc; 
        int          numnodes, myid;
        int          j, k, kk, grid, level; 
        Front        *front;
        int          item[2], from_it[2], recv_src_pc;
        int          to_it[2], send_dst_pc; 
        int          **do_item, n_get_frs, n_snd_frs; 
#if defined(__MPI__)
        MPI_Request *ipp_reqs;
        MPI_Status  *ipp_stat;
        int         *irecv_indices, *irecv_flags;
        int         ndone, not_all_recved = YES;
#endif /* if defined(__MPI__) */

        DEBUG_ENTER(frs_recv_intfc_tables)

        numnodes = pp_numnodes();
        myid = pp_mynode();

        pp_getalloc = 20;
        pp_get = 0; 
        vector(&ppgettrans,pp_getalloc,sizeof(Trans_intfc*));
  
        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                do_item = count_recv_fintfc_blk(item,get_blk_n,gettrans,&n_get_frs);
                for(j = 0; j < n_get_frs; j++)
                {
                    from_it[0] = gettransintfc[grid][j].from_item[0];
                    from_it[1] = gettransintfc[grid][j].from_item[1];
                    recv_src_pc = redistr_table[from_it[0]][from_it[1]].pc_id;
                    if(recv_src_pc != myid)
                    {
#if defined(__MPI__)
                        scalar(&gettransintfc[grid][j].table,sizeof(struct Table));
                        gettransintfc[grid][j].ipp_tag = numnodes*max_n_patch*
                          trans_item_offset(redistr_table,item[0],item[1],max_n_patch)+
                          trans_item_offset(redistr_table,from_it[0],from_it[1],max_n_patch);

                        ppgettrans[pp_get] = &gettransintfc[grid][j]; 
                        pp_get++; 
                        if(pp_get == pp_getalloc)
                        {
                            Trans_intfc  **tmptrans;
                            pp_getalloc += 20;  
                            vector(&tmptrans,pp_getalloc,sizeof(Trans_intfc*));
                            for(k = 0; k < pp_get; k++)
                                tmptrans[k] = ppgettrans[k]; 
                            free(ppgettrans);
                            ppgettrans = tmptrans;  
                        }

                        pp_irecv(gettransintfc[grid][j].ipp_tag, recv_src_pc,
                          (POINTER)(gettransintfc[grid][j].table),sizeof(struct Table),
                          &(gettransintfc[grid][j].ipp_reqs));

                        /*
                        printf("[%d][%d] PPrecv itable from item[%d][%d] level[%d] tag[%d] \n",
                           item[0],item[1],from_it[0], from_it[1],
                           redistr_table[from_it[0]][from_it[1]].wv_level, 
                           gettransintfc[grid][j].ipp_tag);
                        */
#endif /* if defined(__MPI__) */
                    }
                    else
                    {
                        /*
                        int tmptag; 
                        tmptag = numnodes*max_n_patch*
                          trans_item_offset(redistr_table,item[0],item[1],max_n_patch)+
                          trans_item_offset(redistr_table,from_it[0],from_it[1],max_n_patch);
                        printf("[%d][%d] CPrecv itable from item[%d][%d] level[%d] tag[%d] \n",
                           item[0],item[1],from_it[0], from_it[1],
                           redistr_table[from_it[0]][from_it[1]].wv_level, tmptag);
                        */
                    }
                }
            }
        }
         
        pp_sndalloc = 20;
        pp_snd = 0; 
        vector(&ppsndtrans,pp_sndalloc,sizeof(Trans_intfc*));

        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1 || 
               level == 0)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                do_item = count_send_fintfc_blk(item,snd_blk_n,sndtrans,&n_snd_frs);
                for(j = 0; j < n_snd_frs; j++)
                {
                    to_it[0] = sndtransintfc[grid][j].to_item[0];
                    to_it[1] = sndtransintfc[grid][j].to_item[1];
                    send_dst_pc = redistr_table[to_it[0]][to_it[1]].pc_id;
                    if(myid != send_dst_pc)
                    {
#if defined(__MPI__)
                        sndtransintfc[grid][j].ipp_tag = numnodes*max_n_patch*
                          trans_item_offset(redistr_table,to_it[0],to_it[1],max_n_patch) +
                          trans_item_offset(redistr_table,item[0],item[1],max_n_patch) ;

                        ppsndtrans[pp_snd] = &sndtransintfc[grid][j];
                        pp_snd++;
                        if(pp_snd == pp_sndalloc)
                        {
                            Trans_intfc  **tmptrans;
                            pp_sndalloc += 20;
                            vector(&tmptrans,pp_sndalloc,sizeof(Trans_intfc*));
                            for(k = 0; k < pp_snd; k++)
                                tmptrans[k] = ppsndtrans[k];
                            free(ppsndtrans);
                            ppsndtrans = tmptrans;
                        }

                        if(sndtransintfc[grid][j].intfc->nodes == NULL)
                            sndtransintfc[grid][j].intfc->table->rect_grid.dim = 0; 

                        /**
                        pp_isend(sndtransintfc[grid][j].ipp_tag,
                          (POINTER)sndtransintfc[grid][j].intfc->table,
                          sizeof(struct Table),send_dst_pc,
                          &(sndtransintfc[grid][j].ipp_reqs));
                        **/

                        isend_intfc_table(sndtransintfc[grid][j].intfc,send_dst_pc,
                             sndtransintfc[grid][j].ipp_tag,
                             &(sndtransintfc[grid][j].ipp_reqs));
                        /*
                        printf("[%d][%d] PPsend itable to item[%d][%d] level[%d] slevel tag[%d]\n",
                           item[0],item[1],to_it[0], to_it[1],
                           redistr_table[to_it[0]][to_it[1]].wv_level,
                           sndtransintfc[grid][j].ipp_tag);
                        */
#endif /* if defined(__MPI__) */
                    }
                    else
                    {
                        /*
                        int tmptag; 
                        tmptag = numnodes*max_n_patch*
                          trans_item_offset(redistr_table,to_it[0],to_it[1],max_n_patch) +
                          trans_item_offset(redistr_table,item[0],item[1],max_n_patch) ;
                        printf("[%d][%d] CPsend itable to item[%d][%d] level[%d] slevel tag[%d]\n",
                           item[0],item[1],to_it[0], to_it[1],
                           redistr_table[to_it[0]][to_it[1]].wv_level,tmptag);
                        */
                    } 
                }
            } 
        }

        /* perform receive interface tables */

#if defined(__MPI__)
        if(pp_get != 0)
        {
            vector(&ipp_reqs, pp_get, sizeof(MPI_Request));
            vector(&ipp_stat, pp_get, sizeof(MPI_Status));
            vector(&irecv_indices, pp_get, sizeof(int));
            vector(&irecv_flags, pp_get, sizeof(int));
            for(k = 0; k < pp_get; k++)
            {
                assign(&ipp_reqs[k], &(ppgettrans[k]->ipp_reqs), sizeof(MPI_Request));
                irecv_flags[k] = NO;
            }
            while(not_all_recved == YES)
            {
                MPI_Waitsome(pp_get, ipp_reqs, &ndone, irecv_indices, ipp_stat);
                for(k = 0; k < ndone; k++)
                {
                    kk = irecv_indices[k];
                    if(irecv_flags[kk] != YES)
                        irecv_flags[kk] = YES;
                }
                not_all_recved = NO;
                for(k = 0; k < pp_get; k++)
                {
                    if(irecv_flags[k] == NO)
                    {
                        not_all_recved = YES;
                        break;
                    }
                }
            }
            free_these(4,ipp_reqs,ipp_stat,irecv_indices,irecv_flags);
        }
        if(pp_snd != 0)
        {
            for(k = 0; k < pp_snd; k++)   
            {
                MPI_Wait(&(ppsndtrans[k]->ipp_reqs), &(ppsndtrans[k]->ipp_stat));
                if(ppsndtrans[k]->intfc->table->rect_grid.dim == 0)
                    ppsndtrans[k]->intfc->table->rect_grid.dim = frs[0]->rect_grid->dim;
            }
        }
#endif /* if defined(__MPI__) */

        free_these(2,ppsndtrans,ppgettrans);  

        DEBUG_LEAVE(frs_recv_intfc_tables)
}


LOCAL void frs_recv_intfc_chunks(
        Front      **frs,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int         get_blk_n,
        Transbox_set  *gettrans,
        int         snd_blk_n,
        Transbox_set  *sndtrans,
        Trans_intfc   **sndtransintfc,
        Trans_intfc   **gettransintfc)
{
        Trans_intfc  **ppgettrans, **ppsndtrans, *cpsndtrans;
        int          pp_get, pp_snd, pp_getalloc, pp_sndalloc;
        int          numnodes, myid;
        int          j, k, kk, grid, level;
        Front        *front;
        int          item[2], from_it[2], recv_src_pc;
        int          to_it[2], send_dst_pc;
        int          **do_item, n_get_frs, n_snd_frs;

        int         isend_tags, irecv_tags;
        int         *isend_chunks, max_n_chunks;
#if defined(__MPI__)
        MPI_Request **isend_reqs, **irecv_reqs;
        MPI_Status  **isend_stat, **irecv_stat;
#endif /* if defined(__MPI__) */
        POINTER     **out_ncaddr, **out_ocaddr, **top_addr;
        INTERFACE   *intfc;

        DEBUG_ENTER(frs_recv_intfc_chunks)

        numnodes = pp_numnodes();
        myid = pp_mynode();

        pp_getalloc = 20;
        pp_get = 0;
        vector(&ppgettrans,pp_getalloc,sizeof(Trans_intfc*));
        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                do_item = count_recv_fintfc_blk(item,get_blk_n,gettrans,&n_get_frs);
                for(j = 0; j < n_get_frs; j++)
                {
                    from_it[0] = gettransintfc[grid][j].from_item[0];
                    from_it[1] = gettransintfc[grid][j].from_item[1];
                    recv_src_pc = redistr_table[from_it[0]][from_it[1]].pc_id;
                    if(recv_src_pc != myid)
                    {
#if defined(__MPI__)
                        ppgettrans[pp_get] = &gettransintfc[grid][j];
                        pp_get++;
                        if(pp_get == pp_getalloc)
                        {
                            Trans_intfc  **tmptrans;
                            pp_getalloc += 20;
                            vector(&tmptrans,pp_getalloc,sizeof(Trans_intfc*));
                            for(k = 0; k < pp_get; k++)
                                tmptrans[k] = ppgettrans[k];
                            free(ppgettrans);
                            ppgettrans = tmptrans;
                        }
#endif /* if defined(__MPI__) */
                    }
                }
            }
        }

        max_n_chunks = 0; 
        pp_sndalloc = 20;
        pp_snd = 0;
        vector(&ppsndtrans,pp_sndalloc,sizeof(Trans_intfc*));
        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1 ||
               level == 0)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                do_item = count_send_fintfc_blk(item,snd_blk_n,sndtrans,&n_snd_frs);
                for(j = 0; j < n_snd_frs; j++)
                {
                    to_it[0] = sndtransintfc[grid][j].to_item[0];
                    to_it[1] = sndtransintfc[grid][j].to_item[1];
                    send_dst_pc = redistr_table[to_it[0]][to_it[1]].pc_id;
                    if(myid != send_dst_pc)
                    {
#if defined(__MPI__)
                        ppsndtrans[pp_snd] = &sndtransintfc[grid][j];
                        if(ppsndtrans[pp_snd]->intfc->table->num_chunks > max_n_chunks)
                            max_n_chunks = ppsndtrans[pp_snd]->intfc->table->num_chunks;                        
                        pp_snd++;
                        if(pp_snd == pp_sndalloc)
                        {
                            Trans_intfc  **tmptrans;
                            pp_sndalloc += 20;
                            vector(&tmptrans,pp_sndalloc,sizeof(Trans_intfc*));
                            for(k = 0; k < pp_snd; k++)
                                tmptrans[k] = ppsndtrans[k];
                            free(ppsndtrans);
                            ppsndtrans = tmptrans;
                        }
#endif /* if defined(__MPI__) */
                    }
                }
            }
        }

        pp_global_imax(&max_n_chunks, 1);
        max_n_chunks += 4;
        
#if defined(__MPI__)
        if(pp_get != 0)
        {
            vector(&irecv_reqs, pp_get, sizeof(MPI_Request*));
            vector(&irecv_stat, pp_get, sizeof(MPI_Status*));
            vector(&out_ncaddr, pp_get, sizeof(POINTER*));
            vector(&out_ocaddr, pp_get, sizeof(POINTER*));
            for(k = 0; k < pp_get; k++)
            {
                if(ppgettrans[k]->table->rect_grid.dim == 0)
                {
                    ppgettrans[k]->irecv_intfc = NULL;
                    continue;   
                }
                item[0] = ppgettrans[k]->item[0]; 
                item[1] = ppgettrans[k]->item[1]; 
                from_it[0] = ppgettrans[k]->from_item[0]; 
                from_it[1] = ppgettrans[k]->from_item[1]; 
                recv_src_pc = redistr_table[from_it[0]][from_it[1]].pc_id; 
                
                irecv_tags = max_n_chunks*(numnodes*max_n_patch*
                      trans_item_offset(redistr_table,item[0],item[1],max_n_patch) + 
                      trans_item_offset(redistr_table,from_it[0],from_it[1],max_n_patch)); 
		// irecv_tags = max_n_chunks*ppgettrans[k]->ipp_tag; 
	
                ppgettrans[k]->irecv_intfc = irecv_intfc_chunks(ppgettrans[k]->table,
                     recv_src_pc,irecv_tags,&irecv_reqs[k],&out_ncaddr[k],&out_ocaddr[k]);
                vector(&irecv_stat[k], ppgettrans[k]->table->num_chunks+1, sizeof(MPI_Status)); 

                /*
                printf("[%d][%d] PPrecv ichunks(n=%d) from item[%d][%d] level[%d] tag[%d]\n",
                    item[0], item[1], ppgettrans[k]->table->num_chunks+1, from_it[0], from_it[1],
                    redistr_table[from_it[0]][from_it[1]].wv_level, irecv_tags);
                */
            }
        }

        if(pp_snd != 0)
        {
            vector(&isend_chunks, pp_snd, sizeof(int));
            vector(&isend_reqs, pp_snd, sizeof(MPI_Request*));
            vector(&isend_stat, pp_snd, sizeof(MPI_Status*));
            vector(&top_addr, pp_snd, sizeof(POINTER*));

            for(k = 0; k < pp_snd; k++)
            {
                if(ppsndtrans[k]->intfc->nodes == NULL)
                    continue; 

                to_it[0] = ppsndtrans[k]->to_item[0]; 
                to_it[1] = ppsndtrans[k]->to_item[1]; 
                item[0] = ppsndtrans[k]->item[0];
                item[1] = ppsndtrans[k]->item[1];
                send_dst_pc = redistr_table[to_it[0]][to_it[1]].pc_id; 
                isend_tags = max_n_chunks*(numnodes*max_n_patch*
                      trans_item_offset(redistr_table,to_it[0],to_it[1],max_n_patch) +
                      trans_item_offset(redistr_table,item[0],item[1],max_n_patch));
                // isend_tags = max_n_chunks*ppsndtrans[k]->ipp_tag;
 		
                isend_intfc_chunks(ppsndtrans[k]->intfc, send_dst_pc, isend_tags,
                                 &isend_reqs[k], &top_addr[k]);
                isend_chunks[k] = ppsndtrans[k]->intfc->table->num_chunks+1;
                vector(&isend_stat[k], isend_chunks[k], sizeof(MPI_Status));

                /*
                printf("[%d][%d] PPsend ichunks(n = %d) to item[%d][%d] level[%d] slevel tag[%d]\n",
                   item[0],item[1],ppsndtrans[k]->intfc->table->num_chunks+1,
                   to_it[0], to_it[1],redistr_table[to_it[0]][to_it[1]].wv_level, isend_tags);
                */
            } 
        }
#endif /* if defined(__MPI__) */
 
        /* THE LOCAL RECEIVE front buffer interface */
        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                do_item = count_recv_fintfc_blk(item,get_blk_n,gettrans,&n_get_frs);
                for(j = 0; j < n_get_frs; j++)
                {
                    from_it[0] = gettransintfc[grid][j].from_item[0];
                    from_it[1] = gettransintfc[grid][j].from_item[1];
                    recv_src_pc = redistr_table[from_it[0]][from_it[1]].pc_id;
                    if(recv_src_pc == myid)
                    {
                        cpsndtrans = find_loc_match_snd_intfc(gettransintfc[grid][j],
                           sndtransintfc,sndtrans,snd_blk_n,
                             frs,num_patches,redistr_table,max_n_patch);
                        gettransintfc[grid][j].irecv_intfc = cpsndtrans->intfc;
                        cpsndtrans->intfc = NULL; 
                    }
                }
            }
        }

#if defined(__MPI__)
        if(pp_get != 0)
        {
            for(k = 0; k < pp_get; k++)
            {
                if(ppgettrans[k]->table->rect_grid.dim == 0)
                {
                    free(ppgettrans[k]->table); 
                    ppgettrans[k]->table = NULL; 
                    continue; 
                }
                MPI_Waitall(ppgettrans[k]->table->num_chunks+1, irecv_reqs[k], irecv_stat[k]);
                ireconstruct_interface_pointers(ppgettrans[k]->irecv_intfc,ppgettrans[k]->table,
                          out_ocaddr[k],out_ncaddr[k]);
                free_these(2, irecv_reqs[k], irecv_stat[k]);
            }
            free_these(4,irecv_reqs,irecv_stat,out_ncaddr,out_ocaddr);
        }

        if(pp_snd != 0)
        {
            for(k = 0; k < pp_snd; k++)
            {
                if(ppsndtrans[k]->intfc->nodes == NULL)
                    continue;
                MPI_Waitall(isend_chunks[k], isend_reqs[k], isend_stat[k]);
                free_these(3, isend_reqs[k], isend_stat[k], top_addr[k]);
            }
            free_these(4, isend_chunks, isend_reqs, isend_stat, top_addr);
        }
#endif /* if defined(__MPI__) */

        free_these(2,ppsndtrans,ppgettrans);  

        DEBUG_LEAVE(frs_recv_intfc_chunks)
}


LOCAL Trans_intfc *find_loc_match_snd_intfc(
        Trans_intfc  gettransintfc,
        Trans_intfc  **sndtransintfc,
        Transbox_set  *sndtrans,
        int           snd_blk_n,
        Front         **frs,
        int           num_patches,
        Wv_on_pc      **redistr_table,
        int           max_n_patch)
{
        int           n_snd_frs, j, grid, level, **do_item;
        int           me[2], megetfrom[2], item[2], to_it[2];
        int           send_dst_pc;  
        Front         *front;

        megetfrom[0] = gettransintfc.from_item[0];
        megetfrom[1] = gettransintfc.from_item[1];
        me[0] = gettransintfc.item[0];
        me[1] = gettransintfc.item[1];

        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1 ||
               level == 0)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                do_item = count_send_fintfc_blk(item,snd_blk_n,sndtrans,&n_snd_frs);
                for(j = 0; j < n_snd_frs; j++)
                {
                    to_it[0] = sndtransintfc[grid][j].to_item[0];
                    to_it[1] = sndtransintfc[grid][j].to_item[1];
                    send_dst_pc = redistr_table[to_it[0]][to_it[1]].pc_id;
                    if(item[0] == megetfrom[0] && item[1] == megetfrom[1] &&
                       to_it[0] == me[0] && to_it[1] == me[1] &&
                       send_dst_pc == redistr_table[me[0]][me[1]].pc_id)
                    {
                        return &sndtransintfc[grid][j]; 
                    }  
                }
            }
        }
 
        printf("ERROR  find_loc_match_snd_intfc\n");
        printf("NOT found match\n");
        clean_up(ERROR); 
}


LOCAL Trans_intfc **frs_make_recv_intfc_storage(
        Front      **frs,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int         get_blk_n,
        Transbox_set  *gettrans)
{
        int        i, j;
        int        grid, level;
        Front      *front;
        int        item[2];
        int        n_get_frs, **do_item;
        Trans_intfc   **transintfc;
        INTERFACE  *sav_intfc;
        bool       sav_copy;

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();

        set_size_of_intfc_state(size_of_state(frs[0]->interf));
        set_copy_intfc_states(YES);

        vector(&transintfc, num_patches, sizeof(Trans_intfc*));
        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                do_item = count_recv_fintfc_blk(item,get_blk_n,gettrans,&n_get_frs); 
                if(n_get_frs != 0)
                {
                    vector(&transintfc[grid],n_get_frs,sizeof(Trans_intfc));
                    for(j = 0; j < n_get_frs; j++)
                    {
                        transintfc[grid][j].item[0] = item[0];
                        transintfc[grid][j].item[1] = item[1];
                        transintfc[grid][j].from_item[0] = do_item[j][0];
                        transintfc[grid][j].from_item[1] = do_item[j][1];
                    } 
                }
            }
        } 

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);
        return transintfc; 
}

LOCAL int **count_recv_fintfc_blk(
        int         *item,
        int         get_blk_n,
        Transbox_set  *gettrans,
        int         *num_frs)
{
        int          i, j, k, l;
        static int   nin = 10, **do_item;
        static int   first = YES;

        *num_frs = 0;
        if(YES == first)
        {
            matrix(&do_item, nin, 2, sizeof(int));
            first = NO;
        }

        k = 0;
        for(i = 0; i < get_blk_n; i++)
        {
            if(gettrans[i].item[0] == item[0] &&
               gettrans[i].item[1] == item[1])
            {
                for(j = 0; j < gettrans[i].n_box; j++)
                {
                    if(k == 0)
                    {
                        do_item[k][0] = gettrans[i].Tbox[j].from_item[0];
                        do_item[k][1] = gettrans[i].Tbox[j].from_item[1];
                        k++;
                    }
                    else
                    {
                        int in_array = NO;
                        for(l = 0; l < k; l++)
                        {
                            if(gettrans[i].Tbox[j].from_item[0] == do_item[l][0] &&
                               gettrans[i].Tbox[j].from_item[1] == do_item[l][1])
                            {
                                in_array = YES;
                                break;
                            }
                        }
                        if(in_array == NO)
                        {
                            do_item[k][0] = gettrans[i].Tbox[j].from_item[0];
                            do_item[k][1] = gettrans[i].Tbox[j].from_item[1];
                            k++;
                            if(k == nin)
                            {
                                int **tmp_in_item, mm;
                                nin += 10;
                                matrix(&tmp_in_item, nin, 2, sizeof(int));
                                for(mm = 0; mm < k; mm++)
                                {
                                    tmp_in_item[mm][0] = do_item[mm][0];
                                    tmp_in_item[mm][1] = do_item[mm][1];
                                }
                                free(do_item);
                                do_item = tmp_in_item;
                            }
                        }
                    }
                }
            }
        }
        *num_frs = k;
        return do_item;
}


LOCAL Trans_intfc **frs_make_send_intfc(
        Front      **frs,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        Overparam  *overparam,
        int         snd_blk_n,
        Transbox_set  *sndtrans)
{
        int        i, j;
        int        grid, level;
        Front      *front;
        int        item[2], dim = frs[0]->rect_grid->dim; 
        int        n_snd_frs, **do_item;
        Trans_intfc   **transintfc;  

        vector(&transintfc, num_patches, sizeof(Trans_intfc*));
        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item); 

                do_item = count_send_fintfc_blk(item,snd_blk_n,sndtrans,&n_snd_frs);

                /*
                printf("Patch[%d][%d] sends buffer fronts to %d fronts\n",
                    item[0], item[1], n_snd_frs);
                */

                if(n_snd_frs != 0)
                {
                    vector(&transintfc[grid],n_snd_frs,sizeof(Trans_intfc));
                    for(j = 0; j < n_snd_frs; j++)
                    {
                        transintfc[grid][j].item[0] = item[0]; 
                        transintfc[grid][j].item[1] = item[1]; 
                        transintfc[grid][j].to_item[0] = do_item[j][0]; 
                        transintfc[grid][j].to_item[1] = do_item[j][1]; 
                        transintfc[grid][j].intfc = NULL; 
                        set_send_fblk_intfc(&transintfc[grid][j],front,
                              snd_blk_n,sndtrans,redistr_table,max_n_patch); 
                        /*
                        if(item[0] == 1 && item[1] == 2 && do_item[j][0] == 0 && do_item[j][1] == 4)
                        {
                            printf("Patch[%d][%d] sends buffer fronts [%d] to [%d][%d]\n",
                                item[0], item[1], transintfc[grid][j].intfc, 
                                do_item[j][0], do_item[j][1]);
                            show_intfc_states(transintfc[grid][j].intfc); 
                        }
                        */ 
                    }
                }
            }  
            else if(level == 0)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item); 
                do_item = count_send_fintfc_blk(item,snd_blk_n,sndtrans,&n_snd_frs);

                /*
                if(item[0] == 1 && item[1] == 0)
                {
                    printf("-----------------------------\n"); 
                    printf("Patch[%d][%d] sends buffer fronts to %d fronts\n",
                        item[0], item[1], n_snd_frs);
                } 
                */

                if(n_snd_frs != 0)
                {
                    vector(&transintfc[grid],n_snd_frs,sizeof(Trans_intfc));
                    for(j = 0; j < n_snd_frs; j++)
                    {
                        transintfc[grid][j].item[0] = item[0]; 
                        transintfc[grid][j].item[1] = item[1]; 
                        transintfc[grid][j].to_item[0] = do_item[j][0]; 
                        transintfc[grid][j].to_item[1] = do_item[j][1]; 
                        transintfc[grid][j].intfc = NULL; 
                        // printf("--: to front [%d][%d]\n", do_item[j][0], do_item[j][1]); 

                        set_c_send_fblk_intfc(&transintfc[grid][j],front,
                              snd_blk_n,sndtrans,redistr_table,max_n_patch,overparam);
                    }
                }
            }
        }
        
        return transintfc;  
}

LOCAL void set_c_send_fblk_intfc(
        Trans_intfc *transintfc,
        Front       *front,
        int         snd_blk_n,
        Transbox_set  *sndtrans,
        Wv_on_pc     **redistr_table,
        int          max_n_patch,
        Overparam    *overparam)
{
        int         i, item[2], to_item[2];

        item[0] = transintfc->item[0];
        item[1] = transintfc->item[1];
        to_item[0] = transintfc->to_item[0]; 
        to_item[1] = transintfc->to_item[1]; 

        for(i = 0; i < snd_blk_n; i++)
        {
            if(sndtrans[i].item[0] == item[0] &&
               sndtrans[i].item[1] == item[1] &&
               sndtrans[i].Tbox[0].to_item[0] == to_item[0] &&
               sndtrans[i].Tbox[0].to_item[1] == to_item[1])
            {
                // printf("set_c_send_fblk_intfc[%d] \n", i); 
                make_coarse_rect_cut_intfc(&sndtrans[i],transintfc,
                 front,redistr_table,max_n_patch,overparam); 
            }
        }
}

LOCAL void make_coarse_rect_cut_intfc(
        Transbox_set  *sndtrans,
        Trans_intfc *transintfc,
        Front       *front,
        Wv_on_pc     **redistr_table,
        int          max_n_patch,
        Overparam    *overparam)
{
        int         i, j, k;
        int         item[2];
        int         L[MAXD], U[MAXD];
        int         gL[MAXD], gU[MAXD];
        float       fL[MAXD], fU[MAXD]; 
        int         toit[2];
        int         dim = front->rect_grid->dim;
        RECT_GRID   *rect = front->rect_grid, refrect;
        int         refine, level, found_base;
        int         refoff_set[MAXD], refbase[MAXD]; 
        int         refgmin[MAXD], refgmax[MAXD]; 
        INTERFACE   *current_intfc, *intfc;
        NODE        **n; 
        bool        save_interior, sav_copy, status, save_intrp;
        int         force_clip = YES; 

        // int        debug_flag = NO;

        item[0] = transintfc->item[0];
        item[1] = transintfc->item[1];
        toit[0] = transintfc->to_item[0];
        toit[1] = transintfc->to_item[1];

        /*
        // if(item[0] == 1 && item[1] == 0)
        {
            printf("Item[%d][%d] make buf interface for item[%d][%d]\n",
               item[0], item[1], toit[0], toit[1]);
            printf("to item[%d][%d] base[%d %d] bound[%d %d] nbox = %d\n",
                       toit[0], toit[1],
                       redistr_table[toit[0]][toit[1]].base[0],
                       redistr_table[toit[0]][toit[1]].base[1],
                       redistr_table[toit[0]][toit[1]].bound[0],
                       redistr_table[toit[0]][toit[1]].bound[1],
                       sndtrans->n_box);
            for(j = 0; j < sndtrans->n_box; j++)
            {
                if(sndtrans->Tbox[j].from_item[1] == 0)
                {
                    printf("[%d]'s trans TO item[%d][%d] L[%d %d] U[%d %d]\n",
                       j, sndtrans->Tbox[j].to_item[0],
                       sndtrans->Tbox[j].to_item[1],
                       sndtrans->Tbox[j].base[0], sndtrans->Tbox[j].base[1],
                       sndtrans->Tbox[j].bound[0], sndtrans->Tbox[j].bound[1]);
                }
                else
                {
                    printf("[%d]'s trans TO item[%d][%d] L[%d %d] U[%d %d] a hole by [%d][%d]\n",
                       j, sndtrans->Tbox[j].to_item[0],
                       sndtrans->Tbox[j].to_item[1],
                       sndtrans->Tbox[j].base[0], sndtrans->Tbox[j].base[1],
                       sndtrans->Tbox[j].bound[0], sndtrans->Tbox[j].bound[1],
                       sndtrans->Tbox[j].from_item[0],
                       sndtrans->Tbox[j].from_item[1]);
                }
            }
        }
        */

        level = redistr_table[toit[0]][toit[1]].wv_level;
        refine = 1;
        for(i = 0; i < level; i++)
            refine *= overparam->refinementRatio;

        copy_rect_grid(&refrect,rect);
        for(i = 0; i < dim; i++)
        {
            refrect.lbuf[i] *= refine;
            refrect.ubuf[i] *= refine;
            refrect.gmax[i] *= refine;
            refrect.h[i] /= refine;
            refoff_set[i] = redistr_table[item[0]][item[1]].off_set[i]*refine;
            refbase[i] = redistr_table[item[0]][item[1]].base[i]*refine;
            refgmin[i] = -refrect.lbuf[i]; 
            refgmax[i] = refrect.ubuf[i] + refrect.gmax[i]; 
        }

        current_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_size_of_intfc_state(size_of_state(front->interf));
        set_copy_intfc_states(YES);

        if((intfc = copy_interface(front->interf)) == NULL)
        {
            printf("ERROR make_rect_cut_intfc, copy intfc failed\n");
            clean_up(ERROR);
        }

        save_intrp = interpolate_intfc_states(intfc);
        interpolate_intfc_states(intfc) = NO;

        set_min_sc_sep_val_flag(YES);
        set_amr_intfc_tol(intfc,
            pow(2.0,-(front->NumberOfLevels-1.0-front->patch_level)));
        delete_subdomain_curves(intfc);
        delete_passive_boundaries(intfc);

        found_base = 0; 
        for(j = 0; j < sndtrans->n_box; j++)
        {
            if(sndtrans->Tbox[j].from_item[1] == 0)
            {
                for(k = 0; k < dim; k++)
                {
                    L[k] = sndtrans->Tbox[j].base[k] +
                           redistr_table[toit[0]][toit[1]].base[k] +
                           redistr_table[toit[0]][toit[1]].off_set[k] -
                          (refbase[k] + refoff_set[k]);
                    U[k] = sndtrans->Tbox[j].bound[k] +
                           redistr_table[toit[0]][toit[1]].base[k] +
                           redistr_table[toit[0]][toit[1]].off_set[k] -
                          (refbase[k] + refoff_set[k]);
                }
                found_base++;   
            }
        }
        if(found_base != 1)
        {
            printf("ERROR make_coarse_rect_cut_intfc\n");
            printf("Base num not 1\n"); 
            clean_up(ERROR); 
        }
        for(k = 0; k < dim; k++)
        {
            if(L[k] < refgmin[k] || U[k] > refgmax[k])
            {
                printf("ERROR make_coarse_rect_cut_intfc\n");
                printf("L or U outside range L[%d %d] U[%d %d]\n",
                  L[0], L[1], U[0], U[1]);
                printf("rect_grid L[%d %d] U[%d %d]\n",
                  refgmin[0], refgmin[1], refgmax[0], refgmax[1]);
                clean_up(ERROR);
            }  
        }

        rect_crds(&refrect,L,U,fL,fU);
        for(k = 0; k < dim; k++)
        {
            if(L[k] != refgmin[k])
            {
                gL[k] = L[k] + refbase[k] + refoff_set[k] - 
                       (redistr_table[toit[0]][toit[1]].base[k] +
                        redistr_table[toit[0]][toit[1]].off_set[k]);
                if(gL[k] == -redistr_table[toit[0]][toit[1]].lbuf[k])
                    save_interior = NO;
                else
                    save_interior = YES;

                /*
                if(debug_flag == YES)
                {
                    printf("item[1][0] cut at dir[%d] Lposition[%g], save_interior %d gL = %d\n",
                        k, fL[k], save_interior, gL[k]);
                }
                */

               if(rect_boundary_type(intfc,k,0) != REFLECTION_BOUNDARY ||
                   L[k] != 0)
                    set_cut_none_local_flag(YES);

                cut_interface(intfc,fL[k],k,1,save_interior,force_clip);
                set_cut_none_local_flag(NO);
            }
            if(U[k] != refgmax[k])
            {
                gU[k] = U[k] + refbase[k] + refoff_set[k] -
                       (redistr_table[toit[0]][toit[1]].base[k] +
                        redistr_table[toit[0]][toit[1]].off_set[k]);
                if(gU[k] == redistr_table[toit[0]][toit[1]].ubuf[k] +
                     (redistr_table[toit[0]][toit[1]].bound[k] -
                      redistr_table[toit[0]][toit[1]].base[k]))
                    save_interior = NO;
                else
                {
                    save_interior = YES;
                    if(gU[k] == 0)
                        set_cut_none_local_flag(YES);
                }

                /*
                if(debug_flag == YES)
                {
                    printf("item[1][0] cut at dir[%d] Uposition[%g],"
                        " save_interior %d gU = %d\n",
                        k, fU[k], save_interior, gU[k]);
                }
                */

                cut_interface(intfc,fU[k],k,0,save_interior,force_clip);
                set_cut_none_local_flag(NO);
            }
        }

        for(j = 0; j < sndtrans->n_box; j++)
        {
            if(sndtrans->Tbox[j].from_item[1] != 0)
            {
                for(k = 0; k < dim; k++)
                {
                    L[k] = sndtrans->Tbox[j].base[k] +
                           redistr_table[toit[0]][toit[1]].base[k] +
                           redistr_table[toit[0]][toit[1]].off_set[k] -
                          (refbase[k] + refoff_set[k]);
                    U[k] = sndtrans->Tbox[j].bound[k] +
                           redistr_table[toit[0]][toit[1]].base[k] +
                           redistr_table[toit[0]][toit[1]].off_set[k] -
                          (refbase[k] + refoff_set[k]);
                         
                }
                // printf("CUT box L[%d %d] U[%d %d]\n",L[0], L[1], U[0], U[1]); 
                rect_crds(&refrect,L,U,fL,fU);
                clip_interface_with_rect(intfc,fL,fU,YES);
            }
        }
        set_amr_intfc_tol(intfc,
            pow(2.0,(front->NumberOfLevels-1.0-front->patch_level)));
        
        set_current_interface(current_intfc);
        set_copy_intfc_states(sav_copy);

        cut_intfc_into_trans(intfc,transintfc);
        interpolate_intfc_states(intfc) = save_intrp;
}


LOCAL void set_send_fblk_intfc(
        Trans_intfc *transintfc,
        Front       *front,
        int         snd_blk_n,
        Transbox_set  *sndtrans,
        Wv_on_pc     **redistr_table,
        int          max_n_patch)
{
        int         i, j, k;
        int         item[2]; 
        int         L[MAXD], U[MAXD]; 
        int         gmin[MAXD], gmax[MAXD];
        int         toit[2];
        int         dim = front->rect_grid->dim;
        RECT_GRID   *rect = front->rect_grid;  
        INTERFACE   *intfc; 

        item[0] = transintfc->item[0]; 
        item[1] = transintfc->item[1]; 
        toit[0] = transintfc->to_item[0];
        toit[1] = transintfc->to_item[1];

        for(i = 0; i < dim; i++)
        {
            gmin[i] = 0;
            gmax[i] = front->rect_grid->gmax[i]; 
        }

        for(i = 0; i < snd_blk_n; i++)
        {
            if(sndtrans[i].item[0] == item[0] &&
               sndtrans[i].item[1] == item[1])
            {
                for(j = 0; j < sndtrans[i].n_box; j++)
                {
                    if(sndtrans[i].Tbox[j].to_item[0] == toit[0] &&
                       sndtrans[i].Tbox[j].to_item[1] == toit[1])
                    {
                        for(k = 0; k < dim; k++)
                        {
                            L[k] = sndtrans[i].Tbox[j].base[k] + 
                                    redistr_table[toit[0]][toit[1]].base[k] + 
                                    redistr_table[toit[0]][toit[1]].off_set[k] - 
                                   (redistr_table[item[0]][item[1]].base[k] +
                                    redistr_table[item[0]][item[1]].off_set[k]);  
                            U[k] = sndtrans[i].Tbox[j].bound[k] + 
                                    redistr_table[toit[0]][toit[1]].base[k] + 
                                    redistr_table[toit[0]][toit[1]].off_set[k] - 
                                   (redistr_table[item[0]][item[1]].base[k] +
                                    redistr_table[item[0]][item[1]].off_set[k]);  
                        }
                        for(k = 0; k < dim; k++)
                        {
                            if(L[k] < gmin[k] || U[k] > gmax[k])
                            {
                                printf("ERROR in set_send_fblk_intfc\n");
                                printf("L or U outside range L[%d %d] U[%d %d]\n",
                                  L[0], L[1], U[0], U[1]);
                                printf("rect_grid L[%d %d] U[%d %d]\n",
                                   0, 0, gmax[0], gmax[1]); 
                                clean_up(ERROR); 
                            }
                        } 
                        /*
                        if(item[0] == 1 && item[1] == 2)
                        {
                            printf("Item[%d][%d] send to" 
                                " item[%d][%d] L[%d %d] U[%d %d] \n",
                                item[0], item[1], toit[0], toit[1], L[0],L[1],U[0],U[1]); 
                            printf("item[%d][%d] base[%d %d] bound[%d %d]\n", 
                                    item[0], item[1],
                                    redistr_table[item[0]][item[1]].base[0], 
                                    redistr_table[item[0]][item[1]].base[1], 
                                    redistr_table[item[0]][item[1]].bound[0], 
                                    redistr_table[item[0]][item[1]].bound[1]);
                            printf("to item[%d][%d] base[%d %d] bound[%d %d]\n", 
                                    toit[0], toit[1], 
                                    redistr_table[toit[0]][toit[1]].base[0], 
                                    redistr_table[toit[0]][toit[1]].base[1], 
                                    redistr_table[toit[0]][toit[1]].bound[0], 
                                    redistr_table[toit[0]][toit[1]].bound[1]);
                        }
                        */
                        intfc = make_rect_cut_intfc(front,L,U,redistr_table,toit,item); 

                        /*
                        if(item[0] == 1 && item[1] == 2 && toit[0] == 0 && toit[1] == 4)
                        {
                            printf("The cut interface [%d] state\n",intfc);
                            show_intfc_states(intfc); 
                        }
                        */
                        cut_intfc_into_trans(intfc,transintfc);  
                    }
                }
            }
        } 
}

LOCAL void cut_intfc_into_trans(
        INTERFACE   *intfc, 
        Trans_intfc *transintfc)
{
        INTERFACE   *sav_intfc;
        bool        sav_copy, save_intrp;  

        sav_intfc = current_interface();
	sav_copy = copy_intfc_states();
	set_copy_intfc_states(YES);

        if(intfc == sav_intfc)
        {
            printf("ERROR cut_intfc_into_trans\n");
            printf("current interface = intfc\n");
            clean_up(ERROR); 
        }
        if(transintfc->intfc == NULL)
        {
            set_size_of_intfc_state(size_of_state(intfc));
            if((transintfc->intfc = copy_interface(intfc)) == NULL)
            {
                printf("ERROR cut_intfc_into_trans\n");
                printf("copy interface returned NULL\n");
                clean_up(ERROR); 
            }
            delete_interface(intfc);  
	    set_copy_intfc_states(sav_copy);
            set_current_interface(sav_intfc); 
            return;  
        }

        set_size_of_intfc_state(size_of_state(transintfc->intfc));
        save_intrp = interpolate_intfc_states(transintfc->intfc);
        interpolate_intfc_states(transintfc->intfc) = NO; 

        copy_interface_into(intfc,transintfc->intfc);

        if(size_of_state(transintfc->intfc) == 0)
        {
            printf("ERROR cut_intfc_into_trans\n");
            printf("size_of_state(transintfc->intfc) = %d\n", 
                     size_of_state(transintfc->intfc)); 
            clean_up(ERROR); 
        }
      
        delete_interface(intfc); 
        set_current_interface(sav_intfc); 
        interpolate_intfc_states(transintfc->intfc) = save_intrp; 
	set_copy_intfc_states(sav_copy);
}


LOCAL INTERFACE *make_rect_cut_intfc(
        Front      *front,
        int        *L,
        int        *U,
        Wv_on_pc   **redistr_table, 
        int        *toit,
        int        *item)
{
        RECT_GRID   *rect = front->rect_grid;  
        float       fL[MAXD], fU[MAXD]; 
        INTERFACE   *current_intfc, *intfc;
        bool       sav_copy, status, save_intrp; 
        int        dir, dim = rect->dim; 
        int        gmin[MAXD], gmax[MAXD]; 
        int        force_clip = YES; 
        NODE       **n; 
        bool       save_interior;
        int        gL[MAXD], gU[MAXD]; 

        /*
        int        debug_flag = NO; 
        if(item[0] == 1 && item[1] == 3)
        {
            printf("Item[1][3] make buf interface for item[%d][%d]\n", 
               toit[0], toit[1]); 
            debug_flag = YES;
        } 
        */

        current_intfc = current_interface(); 
        sav_copy = copy_intfc_states(); 

        set_size_of_intfc_state(size_of_state(front->interf));
        set_copy_intfc_states(YES);
        if((intfc = copy_interface(front->interf)) == NULL)
        {
            printf("ERROR make_rect_cut_intfc, copy intfc failed\n");
            clean_up(ERROR); 
        }  

        save_intrp = interpolate_intfc_states(intfc); 
        interpolate_intfc_states(intfc) = NO;

        rect_crds(rect,L,U,fL,fU); 
        for(dir = 0; dir < dim; dir++)
        {
            gmin[dir] = -rect->lbuf[dir]; 
            gmax[dir] = rect->gmax[dir] + rect->ubuf[dir]; 
        }

        set_min_sc_sep_val_flag(YES);
        delete_subdomain_curves(intfc);
        delete_passive_boundaries(intfc);

        for(dir = 0; dir < dim; dir++)
        {
            if(L[dir] != gmin[dir])
            {
                if(rect_boundary_type(intfc,dir,0) != REFLECTION_BOUNDARY ||
                   L[dir] != 0)
                    set_cut_none_local_flag(YES);
 
                gL[dir] = L[dir]+
                        (redistr_table[item[0]][item[1]].base[dir] +
                         redistr_table[item[0]][item[1]].off_set[dir]) -
                        (redistr_table[toit[0]][toit[1]].base[dir] +
                         redistr_table[toit[0]][toit[1]].off_set[dir]);                
                if(gL[dir] == -redistr_table[toit[0]][toit[1]].lbuf[dir])
                    save_interior = NO;
                else
                    save_interior = YES;

                /*
                if(debug_flag == YES)
                {
                    printf("item[1][3] cut at dir[%d] Lposition[%g],"
                        " save_interior %d gL = %d\n",
                        dir, fL[dir], save_interior, gL[dir]); 
                }
                */

                cut_interface(intfc,fL[dir],dir,1,save_interior,force_clip);
                set_cut_none_local_flag(NO);
            } 
            if(U[dir] != gmax[dir])
            {
                gU[dir] = U[dir]+
                        (redistr_table[item[0]][item[1]].base[dir] +
                         redistr_table[item[0]][item[1]].off_set[dir]) -
                        (redistr_table[toit[0]][toit[1]].base[dir] +
                         redistr_table[toit[0]][toit[1]].off_set[dir]);                
                if(gU[dir] == redistr_table[toit[0]][toit[1]].ubuf[dir] +
                     (redistr_table[toit[0]][toit[1]].bound[dir] - 
                      redistr_table[toit[0]][toit[1]].base[dir]))
                    save_interior = NO;
                else
                {
                    save_interior = YES;
                    if(gU[dir] == 0)
                        set_cut_none_local_flag(YES);
                }

                /*
                if(debug_flag == YES)
                {
                    printf("item[1][3] cut at dir[%d] Uposition[%g],"
                        " save_interior %d gU = %d\n",
                        dir, fU[dir], save_interior, gU[dir]); 
                }
                */

                cut_interface(intfc,fU[dir],dir,0,save_interior,force_clip);
                set_cut_none_local_flag(NO);
            }
        }

        interpolate_intfc_states(intfc) = save_intrp;

        set_min_sc_sep_val_flag(NO);
        set_current_interface(current_intfc);
        set_copy_intfc_states(sav_copy);

        /* One more pass over the nodes is now needed.  We are checking
         * nodes that became CUT_NODE's without a split.  They should be
         * given node type ERROR if the following tests are satisfied.  The
         * reason for this loop is that the last test may not be satisfied
         * until after delete_curves_outside_of_cut_line(). */
        for (n = intfc->nodes; n && *n; ++n)
        {
            if ((node_type(*n) != ERROR) && is_cut_node(*n) &&
                (node_type(*n) < FIRST_PHYSICS_NODE_TYPE) &&
                (num_curves_at_node(*n,NULL,NULL) <= 1))
                node_type(*n) = ERROR;
        }

        return intfc; 
}

LOCAL void rect_crds(
        RECT_GRID  *rect,
        int        *iL,
        int        *iU,
        float      *fL,
        float      *fU)
{
        int      i, dim = rect->dim;
        float    *L = rect->L, *h = rect->h;

        for(i = 0; i < dim; i++)
        {
            fL[i] = L[i] + h[i]*iL[i];
            fU[i] = L[i] + h[i]*iU[i];
        }
}

LOCAL int **count_send_fintfc_blk(
        int         *item,
        int         snd_blk_n,
        Transbox_set  *sndtrans,
        int         *num_frs)
{
        int          i, j, k, l;
        static int   nin = 10, **do_item;
        static int   first = YES;

        *num_frs = 0;
        if(YES == first)
        {
            matrix(&do_item, nin, 2, sizeof(int));
            first = NO;
        }

        k = 0;
        for(i = 0; i < snd_blk_n; i++)
        {
            if(sndtrans[i].item[0] == item[0] &&
               sndtrans[i].item[1] == item[1])
            {
                for(j = 0; j < sndtrans[i].n_box; j++)
                {
                    if(k == 0)
                    {
                        do_item[k][0] = sndtrans[i].Tbox[j].to_item[0];
                        do_item[k][1] = sndtrans[i].Tbox[j].to_item[1];
                        k++;
                    }
                    else
                    {
                        int in_array = NO;
                        for(l = 0; l < k; l++)
                        {
                            if(sndtrans[i].Tbox[j].to_item[0] == do_item[l][0] &&
                               sndtrans[i].Tbox[j].to_item[1] == do_item[l][1])
                            {
                                in_array = YES;
                                break;
                            }
                        }
                        if(in_array == NO)
                        {
                            do_item[k][0] = sndtrans[i].Tbox[j].to_item[0];
                            do_item[k][1] = sndtrans[i].Tbox[j].to_item[1];
                            k++;
                            if(k == nin)
                            {
                                int **tmp_in_item, mm;
                                nin += 10;
                                matrix(&tmp_in_item, nin, 2, sizeof(int));
                                for(mm = 0; mm < k; mm++)
                                {
                                    tmp_in_item[mm][0] = do_item[mm][0];
                                    tmp_in_item[mm][1] = do_item[mm][1];
                                }
                                free(do_item);
                                do_item = tmp_in_item;
                            }
                        }
                    } 
                }
            }
        }
        *num_frs = k;
        return do_item;
}
        


LOCAL int  set_send_scatter_fronts(
        Front      **frs,
        int        num_patches,
        Overparam  *overparam,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        Transbox_set  **sndtrans)
{
        PP_GRID      *pp_grid = frs[0]->pp_grid;
        int          dim = frs[0]->rect_grid->dim;
        int          item[3], level;
        int          i, grid;
        Transbox_set *tmptrans, *trans = NULL, *ctrans = NULL;
        int          num = 0, alloc = 0, ctransnum;
        Front        *front;

        alloc = 10;
        vector(sndtrans, alloc, sizeof(Transbox_set));

        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            level = front->patch_level;
            if(level == front->NumberOfLevels-1)
            {
                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                trans = scatter_ffront_buf_in_domain(item,level,dim,
                    redistr_table,max_n_patch,pp_grid,overparam);
                if(trans != NULL)
                {
                    assign(&(*sndtrans)[num],trans,sizeof(Transbox_set));
                    num++;
                    if(num == alloc)
                    {
                        alloc += 10;
                        vector(&tmptrans, alloc, sizeof(Transbox_set));
                        assign(tmptrans, *sndtrans, sizeof(Transbox_set)*num);
                        free(*sndtrans);
                        *sndtrans = tmptrans;
                    }
                    free(trans);
                }
            }
            else if(level == 0)
            {
                /* For the base grid, it checks whether it need to send buff to fine */
                /* For the current situation, this sending buffer will contain physical 
                   boundary curves only. Confine this search to the subdomain*/

                loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
                ctransnum = base_scatter_fine_front_buf_in_domain(&ctrans,item,
                             dim,redistr_table,max_n_patch,overparam,pp_grid);
                if(ctransnum != 0)
                {
                    for(i = 0; i < ctransnum; i++)
                    {
                        assign(&(*sndtrans)[num],&ctrans[i],sizeof(Transbox_set));
                        num++;
                        if(num == alloc)
                        {
                            alloc += 10;
                            vector(&tmptrans, alloc, sizeof(Transbox_set));
                            assign(tmptrans, *sndtrans, sizeof(Transbox_set)*num);
                            free(*sndtrans);
                            *sndtrans = tmptrans;
                        }
                    }
                    free(ctrans); 
                }
            }
        }

        if(num == 0)
        {
            free(*sndtrans);
            *sndtrans = NULL;
        }
        return num;
}

LOCAL Transbox_set *scatter_ffront_buf_in_domain(
        int       *item,
        int       level,
        int       dim,
        Wv_on_pc  **redistr_table,
        int       max_n_patch,
        PP_GRID   *pp_grid,
        Overparam    *overparam)
{
        int       L[MAXD], U[MAXD], gL[MAXD], gU[MAXD];
        int       iL[MAXD], iU[MAXD];
        int       bL[MAXD], bU[MAXD], gbL[MAXD], gbU[MAXD], it[2];
        int       *me, him[MAXD];
        int       i, src, grid, side, numnodes;
        Transbox_set  *sndtrans = NULL;
        Trans_box *Tbox = NULL;  
        int       k, allocbox = 0, nbox = 0;

        numnodes = pp_numnodes();
        me = redistr_table[item[0]][item[1]].pc_ic;

        set_grid_interior_domain(redistr_table[item[0]][item[1]],dim,L,U);
        for(i = 0; i < dim; i++)
        {
            gL[i] = L[i] + redistr_table[item[0]][item[1]].base[i] +
                    redistr_table[item[0]][item[1]].off_set[i];
            gU[i] = U[i] + redistr_table[item[0]][item[1]].base[i] +
                    redistr_table[item[0]][item[1]].off_set[i];
        }

        for(src = 0; src < numnodes; src++)
        {
            if(YES == pp_node_in_neighbor(src,pp_grid,him,me) ||
               (redistr_table[src][0].pc_ic[0] == me[0] &&
                redistr_table[src][0].pc_ic[1] == me[1]))
            {
                for(grid = 0; grid < max_n_patch; grid++)
                {
                    if(src == item[0] && grid == item[1]) continue; 
                    if(redistr_table[src][grid].wv_id == -1) continue;
                    if(redistr_table[src][grid].wv_level != level) continue;
                    it[0] = src;
                    it[1] = grid;
                    for(k = 0; k < 8; k++)
                    {
                        if(YES != get_trans_front_blk(redistr_table[it[0]][it[1]],dim,k,bL,bU))
                            continue;
                        for(i = 0; i < dim; i++)
                        {
                            gbL[i] = bL[i] + redistr_table[it[0]][it[1]].base[i] +
                                    redistr_table[it[0]][it[1]].off_set[i];
                            gbU[i] = bU[i] + redistr_table[it[0]][it[1]].base[i] +
                                    redistr_table[it[0]][it[1]].off_set[i];
                        }
                        if(YES == box_intersect(gL,gU,gbL,gbU,dim,iL,iU))
                        {
                            if(sndtrans == NULL)
                            {
                                scalar(&sndtrans,sizeof(Transbox_set));
                                sndtrans->item[0] = item[0];
                                sndtrans->item[1] = item[1];
                            }
                            for(i = 0; i < dim; i++)
                            {
                                iL[i] -= (redistr_table[it[0]][it[1]].base[i] +
                                          redistr_table[it[0]][it[1]].off_set[i]);
                                iU[i] -= (redistr_table[it[0]][it[1]].base[i] +
                                          redistr_table[it[0]][it[1]].off_set[i]);
                            }
                            Tbox = add_info_to_box(iL,iU,item,NULL,it,dim,Tbox,&nbox,&allocbox);
                            /*
                            if(item[0] == 1 && item[1] == 2)
                            {
                                printf("item[%d][%d] set [%d][%d] post[%d]buffer L[%d %d] U[%d %d]\n", 
                                       item[0], item[1], it[0], it[1], k, iL[0], iL[1], iU[0], iU[1]); 
                            }
                            */
                        }
                    }
                }
            }
        }

        if(sndtrans != NULL)
        {
            sndtrans->Tbox = Tbox;
            sndtrans->n_box = nbox; 
        }
 
        return sndtrans;  
}

LOCAL int base_scatter_fine_front_buf_in_domain(
        Transbox_set **sndtrans,
        int          *item,
        int          dim,
        Wv_on_pc  **redistr_table,
        int       max_n_patch,
        Overparam *overparam,
        PP_GRID   *pp_grid)
{
        int       it[2];
        int       i, j, grid;
        int       k, levels;
        Transbox_set *tmptrans, *trans = NULL;
        Trans_box  *Tbox = NULL;
        int       allocbox = 0, nbox = 0;
        int       all_covered = YES;
        int       alloc = 0, num = 0; 

        DEBUG_ENTER(base_scatter_fine_front_buf_in_domain)

        alloc = 10;
        vector(sndtrans, alloc, sizeof(Transbox_set));

        levels = overparam->numberOfRefinementLevels;

        for(grid = 0; grid < max_n_patch; grid++)
        {
            if(redistr_table[item[0]][grid].wv_id == -1) continue;
            if(redistr_table[item[0]][grid].wv_level != levels-1) continue;
            it[0] = item[0];
            it[1] = grid;
            
            for(k = 0; k < 8; k++)
            {
                trans = front_buf_scattered_by_patches(k,dim,it,redistr_table,
                             max_n_patch,pp_grid,overparam);      
                if(trans != NULL)
                {
                    /* Check whether it is all covered by the fine */
                    if(YES == finefr_buf_supplied_by_coarse(trans)) 
                    {
                        trans->item[0] = item[0];  
                        trans->item[1] = item[1];  
                        for(i = 0; i < trans->n_box; i++)
                        {
                            trans->Tbox[i].to_item[0] = trans->Tbox[i].on_item[0]; 
                            trans->Tbox[i].to_item[1] = trans->Tbox[i].on_item[1]; 
                            // trans->Tbox[i].on_item[0] = item[0]; 
                            // trans->Tbox[i].on_item[1] = item[1]; 
                        }
                        assign(&(*sndtrans)[num],trans,sizeof(Transbox_set));
                        num++;
                        if(num == alloc)
                        {
                            alloc += 10;
                            vector(&tmptrans, alloc, sizeof(Transbox_set));
                            assign(tmptrans, *sndtrans, sizeof(Transbox_set)*num);
                            free(*sndtrans);
                            *sndtrans = tmptrans;
                        }
                    }
                    else
                    {
                         free(trans->Tbox); 
                    }
                    free(trans);
                }
            }   
        }

        if(num == 0)
        {
            free(*sndtrans);
            *sndtrans = NULL;  
        }

        DEBUG_LEAVE(base_scatter_fine_front_buf_in_domain)
        return num; 
}
        

LOCAL int finefr_buf_supplied_by_coarse(
        Transbox_set  *trans)
{
        int   i; 
        for(i = 0; i < trans->n_box; i++)
        {
             if(trans->Tbox[i].from_item[1] == 0)
                 return YES; 
        }
        return NO; 
}


LOCAL int  set_recv_scatter_fronts(
        Front      **frs,
        int        num_patches,
        Overparam  *overparam,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        Transbox_set  **gettrans)
{
        int          item[3];
        int          i, j, k, grid;
        int          L[MAXD], U[MAXD];
        int          alloc = 0, num = 0;
        Transbox_set *tmptrans, *trans = NULL;
        Front        *front;
        int          dim = frs[0]->rect_grid->dim; 
        PP_GRID      *pp_grid = frs[0]->pp_grid;

        alloc = 10;
        vector(gettrans, alloc, sizeof(Transbox_set));

        for(grid = 0; grid < num_patches; grid++)
        {
            front = frs[grid];
            if(front->patch_level != front->NumberOfLevels-1) 
                continue; 
            loc_front_on_redistr_table(front,redistr_table,max_n_patch,item);
            for(k = 0; k < 8; k++)
            {
                trans = front_buf_scattered_by_patches(k,dim,item,redistr_table,
                             max_n_patch,pp_grid,overparam);
                if(trans != NULL)
                {
                    assign(&(*gettrans)[num],trans,sizeof(Transbox_set));
                    num++;
                    if(num == alloc)
                    {
                        alloc += 10;
                        vector(&tmptrans, alloc, sizeof(Transbox_set));
                        assign(tmptrans, *gettrans, sizeof(Transbox_set)*num);
                        free(*gettrans);
                        *gettrans = tmptrans;
                    }
                    free(trans); 
                }
            }
        }

        if(num == 0)
        {
            free(*gettrans); 
            *gettrans = NULL; 
        }

        return num; 
}

LOCAL Transbox_set *front_buf_scattered_by_patches(
        int       pos,
        int       dim, 
        int       *item,
        Wv_on_pc  **redistr_table,
        int       max_n_patch,
        PP_GRID   *pp_grid,
        Overparam    *overparam)
{
        int       bL[MAXD], bU[MAXD], it[2];
        int       L[MAXD], U[MAXD], gL[MAXD], gU[MAXD];
        int       iL[MAXD], iU[MAXD];
        int       *me, him[MAXD],level;
        int       i, src, grid, numnodes;
        Transbox_set  *gettrans = NULL;
        Trans_box *Tbox = NULL;
        int       allocbox = 0, nbox = 0;

        numnodes = pp_numnodes();
        me = redistr_table[item[0]][item[1]].pc_ic;
        // src_id = blk2d_neighbor_id(him,me,pos,pp_grid);
        level = redistr_table[item[0]][item[1]].wv_level;

        if(YES == get_trans_front_blk(redistr_table[item[0]][item[1]],dim,pos,bL,bU))
        {
            for(src = 0; src < numnodes; src++)
            {
                /*
                if((redistr_table[src][0].pc_ic[0] == me[0] &&
                    redistr_table[src][0].pc_ic[1] == me[1]) ||
                   (redistr_table[src][0].pc_ic[0] == him[0] &&
                    redistr_table[src][0].pc_ic[1] == him[1]))
                */
                if((redistr_table[src][0].pc_ic[0] == me[0] &&
                    redistr_table[src][0].pc_ic[1] == me[1]) ||
                   YES == pp_node_in_neighbor(src,pp_grid,him,me))
                {
                    for(grid = 0; grid < max_n_patch; grid++)
                    {
                        if(redistr_table[src][grid].wv_id == -1) continue;
                        if(redistr_table[src][grid].wv_level != level) continue;
                        it[0] = src;
                        it[1] = grid;
                        set_grid_interior_domain(redistr_table[it[0]][it[1]],dim,L,U);
                        for(i = 0; i < dim; i++)
                        {
                            gL[i] = L[i] + redistr_table[it[0]][it[1]].base[i] +
                                      redistr_table[it[0]][it[1]].off_set[i] -
                                     (redistr_table[item[0]][item[1]].base[i] +
                                      redistr_table[item[0]][item[1]].off_set[i]);
                            gU[i] = U[i] + redistr_table[it[0]][it[1]].base[i] +
                                      redistr_table[it[0]][it[1]].off_set[i] -
                                     (redistr_table[item[0]][item[1]].base[i] +
                                      redistr_table[item[0]][item[1]].off_set[i]);
                        }
                        if(YES == box_intersect(bL,bU,gL,gU,dim,iL,iU))
                        {
                            if(gettrans == NULL)
                            {
                                scalar(&gettrans,sizeof(Transbox_set));
                                gettrans->item[0] = item[0];
                                gettrans->item[1] = item[1];
                            }
                            Tbox = add_info_to_box(iL,iU,item,it,NULL,dim,Tbox,&nbox,&allocbox);
                        }
                    }
                }
            }
        }
        else 
        {
            return NULL; 
        }
 
        if(gettrans != NULL)
        {
            gettrans->Tbox = Tbox;
            gettrans->n_box = nbox; 
        }

        if(Tbox != NULL)
        {
            if(YES == box_in_boxes(bL,bU,Tbox,nbox,dim))
                return gettrans;
        }

        it[0] = item[0];
        it[1] = 0; 
        Tbox = add_info_to_box(bL,bU,item,it,NULL,dim,Tbox,&nbox,&allocbox); 
       
        if(gettrans == NULL)
        {
            scalar(&gettrans,sizeof(Transbox_set));
            gettrans->item[0] = item[0];
            gettrans->item[1] = item[1];
        }

        gettrans->Tbox = Tbox;
        gettrans->n_box = nbox; 
        return gettrans; 
}

LOCAL Trans_box *add_info_to_box(
        int          *L,
        int          *U,
        int          *on_item,
        int          *from_item,
        int          *to_item,
        int          dim,
        Trans_box    *Tbox,
        int          *nbox,
        int          *alloc)
{
        Tbox = add_to_box(L,U,dim,Tbox,nbox,alloc);
        if(on_item != NULL)
        {
            Tbox[*nbox-1].on_item[0] = on_item[0]; 
            Tbox[*nbox-1].on_item[1] = on_item[1]; 
        }
        if(from_item != NULL)
        {
            Tbox[*nbox-1].from_item[0] = from_item[0]; 
            Tbox[*nbox-1].from_item[1] = from_item[1]; 
        }
        if(to_item != NULL)
        {
            Tbox[*nbox-1].to_item[0] = to_item[0]; 
            Tbox[*nbox-1].to_item[1] = to_item[1]; 
        }
        return Tbox; 
}

LOCAL void set_grid_interior_domain(
        Wv_on_pc   redistr_table,
        int        dim,
        int        *L,
        int        *U)
{
        int        i;

        for(i = 0; i < dim; i++)
        {
            L[i] = 0;
            U[i] = redistr_table.bound[i] - redistr_table.base[i];
        }
}


/*
      6  |  5  |  4
     ----------------
      7  |     |  3
     ----------------
      0  |  1  |  2
NOTE: If the blk(0,...,7) are on the reflection boundary side,
      these blks are not considered as buffer blks.
*/
LOCAL int get_trans_front_blk(
        Wv_on_pc   redistr_table,
        int        dim,
        int        pos,
        int        *L,
        int        *U)
{
        int        i;

        if(dim == 2)
        {
            switch(pos)
            {
            case 0:
                if((redistr_table.rect_bdry_type[0][0] == AMR_SUBDOMAIN_BOUNDARY ||
                    redistr_table.rect_bdry_type[0][0] == SUBDOMAIN_BOUNDARY) &&
                   (redistr_table.rect_bdry_type[1][0] == AMR_SUBDOMAIN_BOUNDARY ||
                    redistr_table.rect_bdry_type[1][0] == SUBDOMAIN_BOUNDARY))
                {
                    for(i = 0; i < dim; i++)
                    {
                        L[i] = -redistr_table.lbuf[i];
                        U[i] = 0;
                    }
                    return YES;
                }
            break;
            case 1:
                if(redistr_table.rect_bdry_type[1][0] == AMR_SUBDOMAIN_BOUNDARY ||
                   redistr_table.rect_bdry_type[1][0] == SUBDOMAIN_BOUNDARY)
                {
                    L[0] = 0;
                    L[1] = -redistr_table.lbuf[1];
                    U[0] = redistr_table.bound[0] - redistr_table.base[0];
                    U[1] = 0;
                    return YES;
                }
            break;
            case 2:
                if((redistr_table.rect_bdry_type[0][1] == AMR_SUBDOMAIN_BOUNDARY ||
                    redistr_table.rect_bdry_type[0][1] == SUBDOMAIN_BOUNDARY) &&
                   (redistr_table.rect_bdry_type[1][0] == AMR_SUBDOMAIN_BOUNDARY ||
                    redistr_table.rect_bdry_type[1][0] == SUBDOMAIN_BOUNDARY))
                {
                    L[0] = redistr_table.bound[0] - redistr_table.base[0];
                    L[1] = -redistr_table.lbuf[1];
                    U[0] = redistr_table.bound[0] - redistr_table.base[0]
                            + redistr_table.ubuf[0];
                    U[1] = 0;
                    return YES;
                }
            break;
            case 3:
                if(redistr_table.rect_bdry_type[0][1] == AMR_SUBDOMAIN_BOUNDARY ||
                   redistr_table.rect_bdry_type[0][1] == SUBDOMAIN_BOUNDARY)
                {
                    L[0] = redistr_table.bound[0] - redistr_table.base[0];
                    L[1] = 0;
                    U[0] = redistr_table.bound[0] - redistr_table.base[0]
                           + redistr_table.ubuf[0];
                    U[1] = redistr_table.bound[1] - redistr_table.base[1];
                    return YES;
                }
            break;
            case 4:
                if((redistr_table.rect_bdry_type[0][1] == AMR_SUBDOMAIN_BOUNDARY ||
                    redistr_table.rect_bdry_type[0][1] == SUBDOMAIN_BOUNDARY) &&
                   (redistr_table.rect_bdry_type[1][1] == AMR_SUBDOMAIN_BOUNDARY ||
                    redistr_table.rect_bdry_type[1][1] == SUBDOMAIN_BOUNDARY))
                {
                    for(i = 0; i < dim; i++)
                    {
                        L[i] = redistr_table.bound[i] - redistr_table.base[i];
                        U[i] = redistr_table.bound[i] - redistr_table.base[i]
                               + redistr_table.ubuf[i];
                    }
                    return YES;
                }
            break;
            case 5:
                if(redistr_table.rect_bdry_type[1][1] == AMR_SUBDOMAIN_BOUNDARY ||
                   redistr_table.rect_bdry_type[1][1] == SUBDOMAIN_BOUNDARY)
                {
                    L[0] = 0;
                    L[1] = redistr_table.bound[1] - redistr_table.base[1];
                    U[0] = redistr_table.bound[0] - redistr_table.base[0];
                    U[1] = redistr_table.bound[1] - redistr_table.base[1]
                           + redistr_table.ubuf[1];
                    return YES;
                }
            break;
            case 6:
                if((redistr_table.rect_bdry_type[0][0] == AMR_SUBDOMAIN_BOUNDARY ||
                    redistr_table.rect_bdry_type[0][0] == SUBDOMAIN_BOUNDARY) &&
                   (redistr_table.rect_bdry_type[1][1] == AMR_SUBDOMAIN_BOUNDARY ||
                    redistr_table.rect_bdry_type[1][1] == SUBDOMAIN_BOUNDARY))
                {
                    L[0] = -redistr_table.lbuf[0];
                    U[0] = 0;
                    L[1] = redistr_table.bound[1] - redistr_table.base[1];
                    U[1] = redistr_table.bound[1] - redistr_table.base[1]
                           + redistr_table.ubuf[1];
                    return YES;
                }
            break;
            case 7:
                if(redistr_table.rect_bdry_type[0][0] == AMR_SUBDOMAIN_BOUNDARY ||
                   redistr_table.rect_bdry_type[0][0] == SUBDOMAIN_BOUNDARY)
                {
                    L[0] = -redistr_table.lbuf[0];
                    U[0] = 0;
                    L[1] = 0;
                    U[1] = redistr_table.bound[1] - redistr_table.base[1];
                    return YES;
                }
            break;
            default:
                printf("ERROR get_trans_front_blk Unknown position\n");
                clean_up(ERROR);
            }
            return NO;
        }
        else
        {
            printf("ERROR get_trans_front_blk\n");
            clean_up(ERROR);
        }
        return NO;
}


/* blk2d_neighbor_id can not be used for amr patches for which
 * the size does not fully cover the base patch. For exsample,
 *           |     _________
 *           |     | patch |
 *  ---------|---------------  subdomain 
 *           |   |0|
 *      a    |   |_|   b
 *       subdomain
 *  If we use blk2d_neighbor_id() for blk "0" as in the fig.,
 *  It will return "a" as neighbor instead of "b". 
 *  pp_node_in_neighbor() will give all possible neighbors according
 *  to the subdomain (base patch) adjacency. 
 */

EXPORT int blk2d_neighbor_id(
        int             *him,
        int             *me,
        int             pos,
        PP_GRID         *pp_grid)
{
        int             *G = pp_grid->gmax;
        int             i, dir, dim = pp_grid->Global_grid.dim;

        for (i = 0; i < dim; i++)
            him[i] = me[i];

        if(pos == 0)
        {
            for(i = 0; i < dim; i++)
                him[i]--;
        }
        else if(pos == 1)
        {
            him[1]--;
        }
        else if(pos == 2)
        {
            him[0]++;
            him[1]--;
        }
        else if(pos == 3)
        {
            him[0]++;
        }
        else if(pos == 4)
        {
            for(i = 0; i < dim; i++)
                him[i]++;
        }
        else if(pos == 5)
        {
            him[1]++;
        }
        else if(pos == 6)
        {
            him[0]--;
            him[1]++;
        }
        else if(pos == 7)
        {
            him[0]--;
        }
        for(dir = 0; dir < dim; dir++)
        {
            if (him[dir] < 0)
                him[dir] = G[dir] - 1;
            if (him[dir] >= G[dir])
                him[dir] = 0;
        }
        return domain_id(him,G,dim);
}

/* Check whether nodes are in 8 neighbors of a node "me" */
EXPORT int pp_node_in_neighbor(
        int             id,
        PP_GRID         *pp_grid,
        int             *icrds,
        int             *me)
{
        int     dim =  pp_grid->Global_grid.dim;
        int     d, G;

        for (d = 0; d < dim; d++)
        {
            G = pp_grid->gmax[d];
            icrds[d] = id % G;
            id = (id - icrds[d])/G;
        }
        if(me[0] == icrds[0]+1 &&
           me[1] == icrds[1]+1)
            return YES;
        if(me[0] == icrds[0] &&
           me[1] == icrds[1]+1)
            return YES;
        if(me[0] == icrds[0]-1 &&
           me[1] == icrds[1]+1)
            return YES;
        if(me[0] == icrds[0]-1 &&
           me[1] == icrds[1])
            return YES;
        if(me[0] == icrds[0]-1 &&
           me[1] == icrds[1]-1)
            return YES;
        if(me[0] == icrds[0] &&
           me[1] == icrds[1]-1)
            return YES;
        if(me[0] == icrds[0]+1 &&
           me[1] == icrds[1]-1)
            return YES;
        if(me[0] == icrds[0]+1 &&
           me[1] == icrds[1])
            return YES;

        return NO;
}

/*
*                               set_send_intfc_ver2():
*
*       Clips off an appropriate piece of intfc for communication to
*       adjacent processors.  For periodic boundaries, the interface
*       is translated to the opposite side of the global domain.
*       This function is called only by make_intfc_reflection_buf()
*       because of the treatment on the cut_node settings.
*       What it does is the following:
                   reflecting line
        -------------       VU[1]
*                   |
*       ------------|       U[1]
               /    |
*             /     |       interior
            1 \     |
               \    |
        --------X---|       L[1]
               2/   |
*       -------/-----       VL[1]

        For the curve 1 existing between L[1]-U[1] which is going to joining with
        curve 2 at X, curve 1 sets it's node at position X with LOCAL flag.
        Curve 2 sets it's X node with NON_LOCAL flag.
*       TODO: might need to fix periodic boundary also.
*/

LOCAL INTERFACE *set_send_intfc_ver2(
        INTERFACE       *intfc,
        int             dir,
        int             side,
        int             *me,
        int             *G)
{
        INTERFACE       *tmp_intfc, *sav_intfc, *send_intfc;
        RECT_GRID       *gr = computational_grid(intfc);
        NODE            **n;
        BOND            *b;
        CURVE           **c;
        float           cut;
        bool            sav_copy;
        int             orth_dir, num_total, found;
        float           orth_bdry[MAXD];

        DEBUG_ENTER(set_send_intfc_ver2)
        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_size_of_intfc_state(size_of_state(intfc));
        set_copy_intfc_states(YES);
        tmp_intfc = copy_interface(intfc);
        if (tmp_intfc == NULL)
        {
            screen("ERROR in set_send_intfc_ver2(), copy_interface() failed\n");
            clean_up(ERROR);
        }
        cut = (side == 0) ? gr->L[dir] + (gr->L[dir] - gr->VL[dir]) :
                            gr->U[dir] + (gr->U[dir] - gr->VU[dir]);
        cut_interface(tmp_intfc,cut,dir,side,NO,YES);
                                                                                                     
        if ((rect_boundary_type(intfc,dir,side) != REFLECTION_BOUNDARY)
                                        &&
            ((me[dir]==0 && side==0) || (me[dir]==(G[dir]-1) && side==1)))
        {
            float T = gr->GU[dir] - gr->GL[dir];

            if (side == 1)
                T = -T;
            for (n = tmp_intfc->nodes; n && *n; ++n)
                Coords((*n)->posn)[dir] += T;
            for (c = tmp_intfc->curves; c && *c; ++c)
                for (b = (*c)->first; b != (*c)->last; b = b->next)
                    Coords(b->end)[dir] += T;
            gr = computational_grid(tmp_intfc);
            gr->L[dir] += T;
            gr->U[dir] += T;
            set_rect_grid(gr->L,gr->U,gr->GL,gr->GU,gr->lbuf,gr->ubuf,
                          gr->gmax,gr->dim,&gr->Remap,gr);
            gr = &topological_grid(tmp_intfc);
            gr->L[dir] += T;
            gr->U[dir] += T;
            set_rect_grid(gr->L,gr->U,gr->GL,gr->GU,gr->lbuf,gr->ubuf,
                          gr->gmax,gr->dim,&gr->Remap,gr);
        }

        orth_dir = (dir+1)%gr->dim;

        if(rect_boundary_type(intfc,dir,side) == REFLECTION_BOUNDARY)
        {
            orth_bdry[0] = gr->L[orth_dir];
            orth_bdry[1] = gr->U[orth_dir];
            for (n = tmp_intfc->nodes; n && *n; ++n)
            {
                num_total = 0;
                for (c = (*n)->in_curves;  c && *c; ++c)
                    ++num_total;
                for (c = (*n)->out_curves; c && *c; ++c)
                    ++num_total;
                found = NO;
                if (orth_dir == 1 && (!is_x_cut_node(*n)) && is_cut_node(*n) &&
                    Coords((*n)->posn)[dir] < cut && side == 0 &&
                    fabs(Coords((*n)->posn)[orth_dir] - orth_bdry[0]) < 10.0*MACH_EPS &&
                    num_total == 1)
                {
                     for (c = (*n)->in_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->last->start)[orth_dir] >= orth_bdry[0])
                         {
                             found = YES;
                             break;;
                         }
                     }
                     for (c = (*n)->out_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->first->end)[orth_dir] >= orth_bdry[0])
                         {
                             found = YES;
                             break;
                         }
                     }
                }
                if (orth_dir == 1 && (!is_x_cut_node(*n)) && is_cut_node(*n) &&
                    Coords((*n)->posn)[dir] < cut && side == 0 &&
                    fabs(Coords((*n)->posn)[orth_dir] - orth_bdry[1]) < 10.0*MACH_EPS &&
                    num_total == 1)
                {
                     for (c = (*n)->in_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->last->start)[orth_dir] <= orth_bdry[1])
                         {
                             found = YES;
                             break;;
                         }
                     }
                     for (c = (*n)->out_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->first->end)[orth_dir] <= orth_bdry[1])
                         {
                             found = YES;
                             break;;
                         }
                     }
                }
                if (orth_dir == 0 && is_cut_node(*n) && is_x_cut_node(*n) &&
                    Coords((*n)->posn)[dir] < cut && side == 0 &&
                    fabs(Coords((*n)->posn)[orth_dir] - orth_bdry[0]) < 10.0*MACH_EPS &&
                    num_total == 1)
                {
                     for (c = (*n)->in_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->last->start)[orth_dir] >= orth_bdry[0])
                         {
                             found = YES;
                             break;;
                         }
                     }
                     for (c = (*n)->out_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->first->end)[orth_dir] >= orth_bdry[0])
                         {
                             found = YES;
                             break;;
                         }
                     }
                }
                if (orth_dir == 0 && is_cut_node(*n) && is_x_cut_node(*n) &&
                    Coords((*n)->posn)[dir] < cut && side == 0 &&
                    fabs(Coords((*n)->posn)[orth_dir] - orth_bdry[1]) < 10.0*MACH_EPS &&
                    num_total == 1)
                {
                     for (c = (*n)->in_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->last->start)[orth_dir] <= orth_bdry[1])
                         {
                             found = YES;
                             break;;
                         }
                     }
                     for (c = (*n)->out_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->first->end)[orth_dir] <= orth_bdry[1])
                         {
                             found = YES;
                             break;;
                         }
                     }
                }

                if (orth_dir == 1 && (!is_x_cut_node(*n)) && is_cut_node(*n) &&
                    Coords((*n)->posn)[dir] > cut && side == 1 &&
                    fabs(Coords((*n)->posn)[orth_dir] - orth_bdry[0]) < 10.0*MACH_EPS &&
                    num_total == 1)
                {
                     for (c = (*n)->in_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->last->start)[orth_dir] >= orth_bdry[0])
                         {
                             found = YES;
                             break;;
                         }
                     }
                     for (c = (*n)->out_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->first->end)[orth_dir] >= orth_bdry[0])
                         {
                             found = YES;
                             break;;
                         }
                     }
                }
                if (orth_dir == 1 && (!is_x_cut_node(*n)) && is_cut_node(*n) &&
                    Coords((*n)->posn)[dir] > cut && side == 1 &&
                    fabs(Coords((*n)->posn)[orth_dir] - orth_bdry[1]) < 10.0*MACH_EPS &&
                    num_total == 1)
                {
                     for (c = (*n)->in_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->last->start)[orth_dir] <= orth_bdry[1])
                         {
                             found = YES;
                             break;;
                         }
                     }
                     for (c = (*n)->out_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->first->end)[orth_dir] <= orth_bdry[1])
                         {
                             found = YES;
                             break;;
                         }
                     }
                }
                if (orth_dir == 0 && is_cut_node(*n) && is_x_cut_node(*n) &&
                    Coords((*n)->posn)[dir] > cut && side == 1 &&
                    fabs(Coords((*n)->posn)[orth_dir] - orth_bdry[0]) < 10.0*MACH_EPS &&
                    num_total == 1)
                {
                     for (c = (*n)->in_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->last->start)[orth_dir] >= orth_bdry[0])
                         {
                             found = YES;
                             break;;
                         }
                     }
                     for (c = (*n)->out_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->first->end)[orth_dir] >= orth_bdry[0])
                         {
                             found = YES;
                             break;;
                         }
                     }
                }
                if (orth_dir == 0 && is_cut_node(*n) && is_x_cut_node(*n) &&
                    Coords((*n)->posn)[dir] > cut && side == 1 &&
                    fabs(Coords((*n)->posn)[orth_dir] - orth_bdry[1]) < 10.0*MACH_EPS &&
                    num_total == 1)
                {
                     for (c = (*n)->in_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->last->start)[orth_dir] <= orth_bdry[1])
                         {
                             found = YES;
                             break;;
                         }
                     }
                     for (c = (*n)->out_curves;  c && *c; ++c)
                     {
                         if(Coords((*c)->first->end)[orth_dir] <= orth_bdry[1])
                         {
                             found = YES;
                             break;;
                         }
                     }
                }
                if(found == YES)
                    continue;
                set_imported_node(*n);
            }
        }
        else
        {
            for (n = tmp_intfc->nodes; n && *n; ++n)
            {
                if (is_cut_node(*n))
                    set_imported_node(*n);
            }
        }

        set_size_of_intfc_state(size_of_state(intfc));
        send_intfc = copy_interface(tmp_intfc);
        (void) delete_interface(tmp_intfc);
                                                                                                     
        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);
                                                                                                     
#if defined(CONSERVATIVE_ALG)
        if(intfc->dim == 2)
            clr_send_intfc_pt_indx(send_intfc);
#endif /* if defined(CONSERVATIVE_ALG) */
                                                                                                     
        DEBUG_LEAVE(set_send_intfc_ver2)
        return send_intfc;
}               /*end set_send_intfc_ver2*/

#endif /* if defined(USE_OVERTURE) */
#endif /* if defined(TWOD) */
