/*
*				dprint.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains statistics/printout driver routines.
*/


#include <driver/ddecs.h>
#if defined(float)
#undef float
#define float double
#endif /* defined(float) */

enum	_TIME_STEP_SET_BY { TIME_STEP_NOT_SET,
			    TIME_STEP_SET_BY_FRONT,
			    TIME_STEP_SET_BY_WAVE,
			    TIME_STEP_SET_BY_PREVIOUS,
			    TIME_STEP_SET_BY_USER_LIMIT,
			    TIME_STEP_SET_BY_PRINTING
};
typedef enum _TIME_STEP_SET_BY TIME_STEP_SET_BY;

	/* LOCAL Function Declarations */
LOCAL	bool	print_fronts_or_states(Grid*,Printplot*);
LOCAL	const char *TimeStepSetByString(TIME_STEP_SET_BY);
LOCAL	float	check_for_ts_restriction(OUTPUT_DATA*,float,float);
LOCAL	float	comm_time_step(float,float*,int,TIME_STEP_SET_BY*);
LOCAL	void	ensure_printout(OUTPUT_DATA*,Grid*);
LOCAL	void	print_TIME_DATA_stamp(FILE*,CHART*);
LOCAL	void	print_Time_Data_stamp(FILE*,Grid*);
LOCAL	void	print_front_and_wave(FILE*,Grid*,Wave*,Front*,Printplot*);
LOCAL	void	print_next_dt(Grid*,Wave*,Front*,Printplot*,int,FILE*);
LOCAL	void	print_solution(FILE*,Grid*,Wave*,Front*,Printplot*);
LOCAL	void	user_output(Grid*,Wave*,Front*,Printplot*,bool);
LOCAL	void	wall_time_dump(CHART*,Printplot*,bool,int);
LOCAL   void    print_repart_solution(CHART*,Printplot*,Grid*,Wave*,Front*);

#if defined(ONED)
LOCAL	void	print_1d_solution_line(FILE*,float*,COMPONENT,Locstate,
				       Front*,Wave*,Printplot*,char*);
#endif /* defined(ONED) */

LOCAL	void	plot_prostar_data(Wave*,Front*,PROSTAR_plot_data*);
#if defined(USE_HDF)
LOCAL	int32	dfnt_size_float(void);
LOCAL	long	fill_hdf_values1d(Wave*,Front*,HDF_plot_data*,float*,float*,
				  float*,float*,int*);
LOCAL	long	fill_hdf_values2d(Wave*,Front*,HDF_plot_data*,float*,float*,
				  float*,float*,int*);
LOCAL	long	fill_hdf_values3d(Wave*,Front*,HDF_plot_data*,float*,float*,
				  float*,float*,int*);
LOCAL	void	plot_hdf_data(Wave*,Front*,HDF_plot_data*);
LOCAL	void	pp_collect_hdf_data(long,HDF_plot_data*);
LOCAL	void	print_int32_vector(const char*,int32*,int32,const char*);
LOCAL	void	print_raster_data(INTERFACE*,HDF_frame_data*,
                                  int*,HDF_plot_data*);
LOCAL	void	print_sds_data(Front*,float*,float*,
			       HDF_frame_data*,HDF_plot_data*);
LOCAL	void	set_HDF_plotting_range(Front*,HDF_plot_data*,
				     int*,float*,float*);
LOCAL	void	set_PROSTAR_plotting_range(Front*,PROSTAR_plot_data*,
				     int*,int*,float*,float*);
LOCAL	bool	write_sds_slab(bool,int32,int32,int32*,int32*,int32*,
			       float**,const char**,const char*,int32,
			       comp_coder_t,comp_info*,VOIDP);
#endif /* defined(USE_HDF) */

#if defined(USE_OVERTURE)
LOCAL   float   find_single_proc_time_step(Grid*,Wave*,Front*,float*,TIME_STEP_SET_BY*);
#endif /* if defined(USE_OVERTURE) */

/* Jul 11 2002 : Myoung-Nyoun : Add : print only fuels using component*/
#if defined(TWOD)
LOCAL   void print_density_for_only_one_component(
        Grid		*grid,
	Printplot	*prt)
{
        OUTPUT_SOLN	*os = (prt->output_soln)[0];
	OUTPUT_VALUE	*(*solution)(OUTPUT_SOLN*,float*,int*) = os->solution;
	OUTPUT_VALUE	*value;
	RECT_GRID	*gr = grid->rect_grid;
	INTERFACE       *intfc= os->intfc;
	char            nfname[512];
	FILE            *nfile;
	COMPONENT       comp;
	int             xmin=0;
	int             ymin=0;
	int             xmax;
	int             ymax;
	int             ix,iy,icoords[MAXD],i,dim;
	float		coords[MAXD];
	      
	xmax=gr->gmax[0];
	ymax=gr->gmax[1];
	  
	(void) set_output_file_name(pp_mynode(),nfname,prt->outfile,
                                grid->step,0);
	(void) sprintf(nfname,"%s-%s",nfname,os->name);
	if ((nfile = fopen(nfname,"w")) == NULL)
	{
	  screen("ERROR in print_density_for_only_one_component(), "
		 "can't open output file %s\n",nfname);
	  clean_up(ERROR);
	}
	
	(void) fprintf(nfile,"\n#ONLY FUEL %s Using COMPONENT DATA\n",os->name);
	for (iy = ymax - 1; iy >= ymin; iy--)
	{
	  icoords[1] = iy;
	  coords[1] = cell_edge(iy,1,gr);
	  for (ix = xmin; ix < xmax; ix++)
	  {
	    icoords[0] = ix;
	    coords[0] = cell_edge(ix,0,gr);
	    comp=component(coords,intfc);
	    if(is_exterior_comp(comp,intfc))
	      comp=nearest_interior_comp(YES,NO_COMP,coords,intfc);
	    if( comp==2 )
	    {
	      value = (*solution)(os,coords,icoords);
	      (void) fprintf(nfile,"%- 15.8g%s",value->uval.fval," ");
	    }
	    else
	      (void) fprintf(nfile,"0 ");
	  }
	  (void) fprintf(nfile,"\n");
	}
	(void) fprintf(nfile,"\n#End ONLY FUEL %s Using COMPONENT DATA\n",os->name);
	(void) Fclose(nfile);
}    /* end print_density_for_only_one_component() */
#endif /* defined(TWOD) */  
/* Jul 11 2002 : Myoung-Nyoun : Add : End */




/*
*			d_print_initial_data():
*/


EXPORT void d_print_initial_data(
	FILE		*file,
	CHART		*chart,
	Printplot	*prt)
{
	Grid	     *grid = chart->grid;
	RECT_GRID    *gr = grid->rect_grid;
	static const char *FORMAT = 
	    "\n      stop_time = %-10g               stop_step = %-10d\n";

	(void) fprintf(file,"\n\n\n");
	(void) foutput(file);
	(void) fprintf(file,"\t\t\tINITIAL DATA:\n\n\n");
	fprint_rectangular_grid(file,gr);
	(void) fprintf(file,FORMAT,stop_time(grid),stop_step(grid));
	(void) fprintf(file,"\n\t\tComputational ");
	switch (gr->Remap.remap)
	{
	case IDENTITY_REMAP:
	    switch (gr->dim)
	    {
	    case 1:
	        (void) fprintf(file,"Length");
	        break;
	    case 2:
	        (void) fprintf(file,"Area");
	        break;
	    case 3:
	        (void) fprintf(file,"Volume");
	        break;
	    }
	    break;
	case CYLINDRICAL_REMAP:
	    switch (gr->dim)
	    {
	    case 1:
	        (void) fprintf(file,"Area");
	        break;
	    case 2:
	        (void) fprintf(file,"Volume");
	        break;
	    case 3:
	    default:
	        screen("ERROR in d_print_initial_data(), "
		       "3D CYLINDRICAL_REMAP not supported\n");
	        clean_up(ERROR);
	        break;
	    }
	    break;
	case SPHERICAL_REMAP:
	    switch (gr->dim)
	    {
	    case 1:
	        (void) fprintf(file,"Volume");
	        break;
	    case 2:
	        screen("ERROR in d_print_initial_data(), "
		       "2D SPHERICAL_REMAP not supported\n");
	        clean_up(ERROR);
	        break;
	    case 3:
	    default:
	        screen("ERROR in d_print_initial_data(), "
		       "3D SPHERICAL_REMAP not supported\n");
	        clean_up(ERROR);
	        break;
	    }
	    break;
	case INVALID_REMAP:
	default:
	    screen("ERROR in d_print_initial_data(), invalid remap %d\n",
		   gr->Remap.remap);
	    clean_up(ERROR);
	    break;
	}
	(void) fprintf(file," = %g\n",gr->Remap.area);
	(void) fprintf(file,"\n\t\tRemap Geometry:  %s\n",
		       gr->Remap.remap_name);

	if (debugging("init_printout"))
	{
	    (void) printf("\t\tBEFORE TIME LOOP:\n");
	    if (prt->printout != NULL)
	    {
	    	int step_sav;

	    	step_sav = grid->step;
	    	if (step_sav > 0)
		    grid->step = step_sav-1;

	    	(*prt->printout)(chart,prt,NO,0);

	    	if (step_sav > 0) grid->step = step_sav;
	    }
	}
}		/*end d_print_initial_data*/


/*
*				d_printout():
*
*	Prints results for the purpose of plotting and data analysis.
*/

/*ARGSUSED*/
EXPORT void d_printout(
	CHART		*chart,
	Printplot 	*prt,
	bool		about_to_stop,
	int		error)
{
	Grid		*grid = chart->grid;
	Wave		*wave = chart->wave;
	Front		*front = chart->front;
	FILE		*file = stdout;
	char		fname[1024];
	OUTPUT_DATA	**mod = prt->main_output_data;
	bool		extra_print = NO;
	bool		print_time = NO;
	bool		prt_fr_or_sts;
	int		i;

	wall_time_dump(chart,prt,about_to_stop,error);

	interactive_printing_control(prt,grid,&extra_print,&about_to_stop);

	if (extra_print == YES)
	{
	    print_time = YES;
	    prt_fr_or_sts = YES;
	}
	else if (about_to_stop == YES)
	{
	    print_time = YES;
	    prt_fr_or_sts = YES;
	    for (i = 0; i < NUM_OUTPUT_FORMATS; ++i)
	    {
	        if (output_format_on(i,prt))
		    ensure_printout(prt->main_output_data[i],grid);
	    }
	}
	else
	{
	    prt_fr_or_sts = print_fronts_or_states(grid,prt);
	    for (i = 0; i < NUM_OUTPUT_FORMATS; ++i)
	    {
	    	if (is_ts_for_output_format(i,grid,prt))
	    	{
	    	    print_time = YES;
		    break;
	    	}
	    }
	}

        if (!(about_to_stop == YES && grid->repart_at_end_of_run == YES))
        {
	    if (print_time == NO)
	    {
	        print_Time_Data_stamp(file,grid);
	        print_extreme_values(file,chart,prt);
    	    }
    	    else
	    {
	        if (prt->outfile != NULL)
	        {
	            /* print time stamp to master output file */
	            print_Time_Data_stamp(file,grid);
	            print_extreme_values(file,chart,prt);

	            if (prt_fr_or_sts == YES)
		    {
	                set_output_file_name(pp_mynode(),fname,
                                prt->outfile,grid->step,error);
	                if ((file = fopen(fname,"w")) == NULL)
	                {
	                    screen("ERROR in d_printout(), "
	                           "can't open output file %s\n",fname);
	                    clean_up(ERROR);
	                }
	                if (debugging("nobuf"))
	                    setbuf(file,NULL);
	                record_print_version(file);
	                print_title(file,prt->title);
	                if (error != 0)
	                    (void) printf("\n\nCLEAN_UP printout, error = %d\n\n",
		                          error);
	                if (prt->print_initial_data)
	                    (*prt->print_initial_data)(file,chart,prt);
		    }
		    else
		        file = NULL;
	        }

	        if (file != NULL)
	        {
	    		/* Print Time Information */

	            print_TIME_DATA_stamp(file,chart);
	            print_extreme_values(file,chart,prt);
	        }

	    		/* Print Interface and States */

	        print_solution(file,grid,wave,front,prt);

	        // if (prt->plot_ellip_vars != NULL)
	        //     (*prt->plot_ellip_vars)(grid);
	    }
		/* Compute Statistics */

	    if (debugging("wall_time"))
	        (void) printf("\n\t\tCurrent Wall Time:	%s\n",date_string());

			/* More specific statistics/diagnostics */

	    // user_output(grid,wave,front,prt,about_to_stop);

	    (void) printf("\n\n\n");

	    if (print_time == YES)
	    {
	        for (i = 0; i < NUM_OUTPUT_FORMATS; ++i)
	        {
		    if (is_ts_for_output_format(i,grid,prt))
		    {
		        if (real_time_output(Output_mode(mod[i])))
		            Output_next_print_time(mod[i]) +=
			    Output_time_freq(mod[i]);
		        else
		            Output_next_print_step(mod[i]) +=
			    Output_step_freq(mod[i]);
		    }
	        }

	        if (file != NULL)
	        {
		    print_next_dt(grid,wave,front,prt,error,file);

	            if (file != stdout)
	            {
		        trace_foutput(file);
		        (void) Fclose(file);
		        if (prt->compress == YES)
		        {
		            char s[256];

		            (void) sprintf(s,"gzip %s &",fname);
			    (void) system(s);
		        }
	            }
	        }
	    }

	    if (about_to_stop == NO)
	    {
	        IMPORT bool  suppress_prompts;
	        bool         sav_suppress_prompts;
	        INIT_DATA    *init = grid->init;

	        sav_suppress_prompts = suppress_prompts;
	        suppress_prompts = (interactive_prompting(init) == YES) ? 
                                         NO : YES;

			/*  Call await() at Pause Time */

	        if (real_time_output(pause_mode(grid)) &&
	            (grid->time >= pause_time(grid)))
	        {
	            screen("PAUSE - pause time pause_time(grid) = %g reached\n",
	    	           pause_time(grid));
	            await(grid,front,wave,prt,NO);
	        }
	        else if (mesh_time_output(pause_mode(grid)) &&
	    	     (grid->step == pause_step(grid)))
	        {
	            screen("PAUSE - pause step pause_step(grid) = %d reached\n",
	    	           pause_step(grid));
	            await(grid,front,wave,prt,NO);
	        }
	        suppress_prompts = sav_suppress_prompts;
            }
	}
        else
            print_repart_solution(chart,prt,grid,wave,front);
}		/*end d_printout*/

LOCAL	void	print_next_dt(
	Grid	  *grid,
	Wave	  *wave,
	Front	  *front,
	Printplot *prt,
	int	  error,
	FILE	  *file)
{
	OUTPUT_DATA	**mod = prt->main_output_data;
	float		next_dt;
	// next_dt = (error == 0) ? find_time_step(grid,wave,front,prt,NO) :
	//			 grid->dt;
	next_dt = (error == 0) ? grid->next_dt :
				 grid->dt;
	(void) foutput(file);
	(void) fprintf(file,"\tnext_dt:  ");
	if (output_format_on(RECT_STATES,prt) &&
	    (Output_in_binary(mod[RECT_STATES]) == YES))
	{
	    /*
	     * We assume that binary printing of states implies restart
	     * is the main concern, thus a binary next_dt.
	     */

	    (void) fprintf(file,"\f%c",2);
	    (void) fwrite((const void *) &grid->time,sizeof(float),1,file);
	    (void) fwrite((const void *) &next_dt,sizeof(float),1,file);
	}
	else
	    (void) fprintf(file,"%-"FFMT"\n",next_dt);
}		/*end print_next_dt*/

LOCAL	void	ensure_printout(
	OUTPUT_DATA	*data,
	Grid		*grid)
{
	Output_mode(data) = MESH_TIME;
	Output_start_step(data) = 0;
	Output_next_print_step(data) = grid->step;
}		/*end ensure_printout*/

EXPORT	void	set_output_file_name(
        int             my_node,
	char		*fname,
	const char	*bname,
	const int	step,
	const int	error)
{
        char tmp[1028];
	(void) strcpy(fname,bname);
	if (error != 0)
	    (void) strcat(fname,".CLEAN_UP");
	if (step >= 0)
        {
            /***
	    (void) sprintf(fname,"%s.ts%s",fname,
			   right_flush(step,TSTEP_FIELD_WIDTH));
            ***/
            (void) sprintf(tmp,"%s.ts%s",fname,
                       right_flush(step,TSTEP_FIELD_WIDTH));
            (void) sprintf(fname, "%s", tmp);
        }
	if (pp_numnodes() > 1)
        {
            /**
	    (void) sprintf(fname,"%s-nd%s",fname,
				 right_flush(my_node,PP_NODE_FIELD_WIDTH));
            **/
            (void) sprintf(tmp,"%s-nd%s",fname,
                           right_flush(my_node,PP_NODE_FIELD_WIDTH));
            (void) sprintf(fname, "%s", tmp);
        }
}		/*end set_output_file_name*/

EXPORT	void	adjoin_node_number(
	char	*s)
{
	int myid = pp_mynode();
	int nn = pp_numnodes();
	int nd;
	for (nd = 0; nn != 0; nn /=10, ++nd);
	(void) sprintf(s,"%s-nd%s",s,right_flush(myid,nd));
}		/*end adjoin_node_number*/

LOCAL	void	wall_time_dump(
	CHART		*chart,
	Printplot 	*prt,
	bool		about_to_stop,
	int		error)
{
	PRINT_OPTIONS	*pto = wall_time_dump_options(prt);
	OUTPUT_DATA	*prt_save_output_data[NUM_OUTPUT_FORMATS];
	Grid		*grid;
	Wave		*wave;
	Front		*front;
	float		current_time, elapsed_time;
	int		i;
	FILE		*file;
	char		fname[1024];
	bool		sav_binary;
	int		do_wall_time_print;
	static OUTPUT_DATA **wall_dump_output_data = NULL;

	debug_print("walltime","Entered wall_time_dump()\n");

	if ((error != 0) || (about_to_stop == YES) || (pto == NULL))
	{
	    debug_print("walltime","Left wall_time_dump()\n");
	    return;
	}

	if (print_wall_time_interval(pto) < 0.0)
	{
	    debug_print("walltime","Left wall_time_dump()\n");
	    return;
	}

	current_time = real_time();
	elapsed_time = current_time - last_wall_time_dump(pto);
	do_wall_time_print =
	    (elapsed_time < print_wall_time_interval(pto)) ? 0 : 1;
	pp_global_imax(&do_wall_time_print,1);
	if (do_wall_time_print == 0)
	{
	    debug_print("walltime","Left wall_time_dump()\n");
	    return;
	}

	last_wall_time_dump(pto) = current_time;

	(void) sprintf(fname,"%s%d",
		       print_filename(pto),wall_time_dump_number(pto)++);
	wall_time_dump_number(pto) = wall_time_dump_number(pto)%2;
	if (debugging("walltime"))
	{
	    (void) printf("wall_time_dump_number = %d\n",
			  wall_time_dump_number(pto));
	}
	set_output_file_name(pp_mynode(),fname,fname,-1,0);

	grid = chart->grid;
	wave = chart->wave;
	front = chart->front;
	if (debugging("walltime"))
	    (void) printf("Opening wall time dump file %s\n",fname);

	if ((file = fopen(fname,"w")) == NULL)
	{
	    screen("ERROR in wall_time_dump(), can't open %s\n",fname);
	    clean_up(ERROR);
	}

	if (wall_dump_output_data == NULL)
	{
	    PRINT_OPTIONS Pto;

	    vector(&wall_dump_output_data,
		   NUM_OUTPUT_FORMATS,sizeof(OUTPUT_DATA*));

	    set_defaults_for_print_options(&Pto,grid->init);
	    Prt_mode(Pto) = MESH_TIME;
	    Print_step_interval(Pto) = 1;
	    Print_start_step(Pto) = 0;
	    Print_in_binary(Pto) = print_in_binary(pto);

	    scalar(&wall_dump_output_data[PRT_FRONTS],sizeof(OUTPUT_DATA));
	    set_output_data(&Pto,wall_dump_output_data[PRT_FRONTS],grid,
			    prt,NO,NO);
	    scalar(&wall_dump_output_data[RECT_STATES],sizeof(OUTPUT_DATA));
	    set_output_data(&Pto,wall_dump_output_data[RECT_STATES],grid,
			    prt,NO,NO);
	    wall_dump_output_data[TRI_STATES] = NULL;
	    wall_dump_output_data[HDF_STATES] = NULL;
	    wall_dump_output_data[SDS_STATES] = NULL;
	}

	for (i = 0; i < NUM_OUTPUT_FORMATS; ++i)
	{
	    prt_save_output_data[i] = prt->main_output_data[i];
	    prt->main_output_data[i] = wall_dump_output_data[i];
	}
	ensure_printout(prt->main_output_data[PRT_FRONTS],grid);
	ensure_printout(prt->main_output_data[RECT_STATES],grid);

	sav_binary = is_binary_output();
	set_binary_output(print_in_binary(pto));

	record_print_version(file);
	print_title(file,prt->title);
	if (prt->print_initial_data)
	    (*prt->print_initial_data)(file,chart,prt);
	print_TIME_DATA_stamp(file,chart);

	print_solution(file,grid,wave,front,prt);

	print_next_dt(grid,wave,front,prt,error,file);
	trace_foutput(file);
	(void) Fclose(file);

	set_binary_output(sav_binary);
	for (i = 0; i < NUM_OUTPUT_FORMATS; ++i)
	    prt->main_output_data[i] = prt_save_output_data[i];
	debug_print("walltime","Left wall_time_dump()\n");

}		/*end wall_time_dump*/

EXPORT void interactive_printing_control(
	Printplot	*prt,
	Grid		*grid,
	bool		*extra_print,
	bool		*about_to_stop)
{
	FILE		*fp;
	OUTPUT_DATA	**mod = prt->main_output_data;
	int		i;
	int		step;
	static	char	filename[NUM_OUTPUT_FORMATS][25];
	static	bool	first = YES;

	if (first == YES)
	{
	    first = NO;
	    for (i = 0; i < NUM_OUTPUT_FORMATS; ++i)
	        (void) sprintf(filename[i],"%s.ts",output_format_name(i));
	}

	for (i = 0; i < NUM_OUTPUT_FORMATS; ++i)
	{
	    if ((mod[i] != NULL) && (fp = fopen(filename[i],"r")) != NULL)
	    {
		if (fscanf(fp,"%d",&step) != EOF)
		{
		    do
		    {
		        if ((grid->step == step) && 
			(is_print_time(grid->time,Print_control(mod[i])) == NO)
						&&
			(is_print_step(grid->step,Print_control(mod[1])) == NO))
			    *extra_print = YES;
		    }
		    while (fscanf(fp,"%d",&step) != EOF);
	        }
		else
		{
		    *about_to_stop = YES;
		    stop_step(grid) = grid->step;
		    (void) printf("\nRun being stopped from outside\n");
	        }
		(void) fclose(fp);
	    }
	}
}		/*end interactive_printing_control*/


/*
*				user_output():
*
*	Provides a method of recording data at time step intervals.
*	Each output function is loaded in an array in the Printplot.
*	This function steps through the array, executing each function
*	in turn.  Each function has its own data structure of
*	information stored as an array of generic pointers in the
*	Printplot.  The correspondence is one-to-one, so the ith
*	function corresponds to the ith data pointer.  An OUTPUT_DATA
*	structure with suitable values should be the first element of
*	any data structure intended to be used in prt->user_output_data.
*
*	To add a new function to the list, the function
*	add_user_output_function() is provided.
*/

/* ARGSUSED */
LOCAL void user_output(
	Grid		*grid,
	Wave		*wave,
	Front		*front,
	Printplot	*prt,
	bool		about_to_stop)
{
	int	    i;
	void	    (**f)(Grid*,Wave*,Front*,Printplot*,OUTPUT_DATA*,bool);
	OUTPUT_DATA *data;

	if (prt->user_output_funcs == NULL)
	    return;

	debug_print("user_output","Entered user_output()\n");
	for (i = 0, f = prt->user_output_funcs; *f != NULL; ++f, ++i)
	{
	    data = prt->user_output_data[i];
	    if ((is_print_time(grid->time,Print_control(data)) == YES) ||
	        (is_print_step(grid->step,Print_control(data)) == YES) ||
		about_to_stop)
	    {
	        (*(*f))(grid,wave,front,prt,data,about_to_stop);
	        if (real_time_output(Output_mode(data)))
		    Output_next_print_time(data) += Output_time_freq(data);
	        else
		    Output_next_print_step(data) += Output_step_freq(data);
	    }
        }
	debug_print("user_output","Left user_output()\n");
}		/*end user_output*/


LOCAL	void print_Time_Data_stamp(
	FILE		*file,
	Grid		*grid)
{
	if (file == NULL)
	    return;
	// (void) fprintf(file,"\n\n\n\n");
	(void) fprintf(file,"\n\n");
	(void) foutput(file);
	(void) fprintf(file,"\tTime Data:  t = %-"FFMT" j = %-10d dt = %-"FFMT"\n",
		       grid->time,grid->step,grid->dt);
}		/*end print_Time_Data_stamp*/

LOCAL	void print_TIME_DATA_stamp(
	FILE		*file,
	CHART		*chart)
{
	Grid		*grid = chart->grid;

	if (file == NULL)
	    return;
	// (void) fprintf(file,"\n\n\n\n");
	(void) fprintf(file,"\n\n");
	(void) foutput(file);
	(void) fprintf(file,"\tTIME DATA:  t = %-"FFMT" j = %-10d dt = %-"FFMT"\n",
		       grid->time,grid->step,grid->dt);
	(void) foutput(file);
}		/*end print_TIME_DATA_stamp*/

/*
*			print_solution():
*
*	Calls print_front_and_wave() for the front and wave
*	associated with each chart that is below chart_of_front(front)
*	in the tree structure.
*/

LOCAL	void print_solution(
	FILE		*file,
	Grid		*grid,
	Wave		*wave,
	Front		*front,
	Printplot	*prt)
{
	LEVEL		*lev;
	CHART		*root, *chart;

		/* print information for root */
        // TMP
        if(front->step != 0)
        {
            // printf("step %d, file name %s\n", front->step, prt->outfile);
            (*front->_show_state_on_tri)("visual",front->step,front);
            (*front->_rect_state_on_tri)("rect_visual",front->step,front);
        }
        return;
}		/*end print_solution*/

/*
*       Set up new zoom_grid and print solution with new parallel partition
*/
LOCAL void print_repart_solution(
        CHART           *chart,
        Printplot       *prt,
        Grid            *grid,
        Wave            *wave,
        Front           *front)
{
        FILE            *file;
        Front           *tempfront;
        RECT_GRID       dual_grid;
        RECT_GRID       *zoom_grid = &(grid->new_pp_grid->Zoom_grid);
        char            fname[1024];
        float           L[MAXD], U[MAXD], *GL, *GU;
        int             i,j,k;
        int             ratio[MAXD];
        int             dim = front->rect_grid->dim;
        int             icrds_old[MAXD],icrds_new[MAXD];
        int             gmax[MAXD],lbuf[MAXD], ubuf[MAXD];

        GL = grid->pp_grid->Global_grid.L;
        GU = grid->pp_grid->Global_grid.U;

        find_Cartesian_coordinates(pp_mynode(),grid->pp_grid,icrds_old);
        for (i = 0; i < dim; ++i)
            ratio[i] = (grid->new_pp_grid->gmax[i])/(grid->pp_grid->gmax[i]);
        for (i = 0; i < grid->new_pp_grid->nn; ++i)
        {
            find_Cartesian_coordinates(i,grid->new_pp_grid,icrds_new);

            k = 1;
            for (j = 0; j < dim; ++j)
            {
                k = k && ((ratio[j]*icrds_old[j] <= icrds_new[j])
                            &&(icrds_new[j] <= ratio[j]*icrds_old[j] +
                               ratio[j]-1));
            }
            if (k)
            {

               /*  set the zoom_grid of new_pp_grid here */
                for (j = 0; j < dim; ++j)
                {
                    L[j] = grid->new_pp_grid->dom[j][icrds_new[j]];
                    U[j] = grid->new_pp_grid->dom[j][icrds_new[j] + 1];
                    gmax[j] = irint((U[j] - L[j])/(front->rect_grid->h[j]));

                    switch (dim) /* TODO Unify 2 and 3 D */
                    {
                    case 1:
                    case 2:
                        lbuf[j] = (icrds_new[j] > 0) ?
                                     grid->new_pp_grid->buf[j] : 0;
                        ubuf[j] = (icrds_new[j] <(grid->new_pp_grid->gmax[j]-1))
                                   ? grid->new_pp_grid->buf[j]:0;
                        break;
                    case 3:
                        lbuf[j] = (icrds_new[j] > 0)
                                ? grid->new_pp_grid->buf[i] :
                                front->rect_grid->lbuf[j];

                        ubuf[j] = (icrds_new[j] <(grid->new_pp_grid->gmax[j]-1))
                                ? grid->new_pp_grid->buf[i] :
                                front->rect_grid->ubuf[j];
                        break;
                    }
                }
                set_rect_grid(L,U,GL,GU,lbuf,ubuf,gmax,dim,
                    &front->rect_grid->Remap,&grid->new_pp_grid->Zoom_grid);

                   /* set up filename and output front and wave*/
                set_output_file_name(i,fname,prt->outfile,grid->step,0);
                if ((file = fopen(fname,"w")) == NULL)
                {
                    screen("ERROR: unable to open file '%s'\n",fname);
                    clean_up(ERROR);
                }

                record_print_version(file);
                print_title(file,prt->title);
                if (prt->print_initial_data)
                    (*prt->print_initial_data)(file,chart,prt);
                print_TIME_DATA_stamp(file,chart);
                print_extreme_values(file,chart,prt);

                      /*clip the front with the new zoom_grid*/

                tempfront = copy_front(front);
                set_copy_intfc_states(YES);
                set_size_of_intfc_state(size_of_state(front->interf));
                tempfront->interf = copy_interface(front->interf);
                for (j = 0; j < dim; ++j)
                {
                    if (icrds_new[j] > 0)
                        rect_boundary_type(tempfront->interf,j,0)
                                = SUBDOMAIN_BOUNDARY;
                    if (icrds_new[j] <(grid->new_pp_grid->gmax[j]-1))
                        rect_boundary_type(tempfront->interf,j,1)
                                = SUBDOMAIN_BOUNDARY;
                }
                        /*set topological grid here*/
                set_dual_grid(&dual_grid,zoom_grid);
                for (j = 0; j < dim; ++j)
                    gmax[j] = dual_grid.gmax[j]+dual_grid.lbuf[j]+
                                       dual_grid.ubuf[j];
                set_rect_grid(dual_grid.VL,dual_grid.VU,dual_grid.GL,
                      dual_grid.GU,NOBUF,NOBUF,gmax,dim,&zoom_grid->Remap,
                                     &topological_grid(tempfront->interf));
                set_computational_grid(tempfront->interf,zoom_grid);

                           /*clip the interface to subdomain*/
                clip_front_for_output(tempfront,zoom_grid);
                           /*print repart front and wave*/

                        /*set relative coordinates of new nodes */
                for(j = 0; j < dim; ++j)
                    icrds_new[j] = icrds_new[j] - icrds_old[j]*ratio[j];

                for (j = 0; j < prt->n_rect_state_vars; ++j)
                {
                    prt->output_soln[j]->repart_at_end = YES;
                    prt->output_soln[j]->icrds_new = icrds_new;
                }

                        /*print repart front and wave*/
                print_front_and_wave(file,grid,wave,tempfront,prt);
                print_next_dt(grid,wave,tempfront,prt,0,file);
                trace_foutput(file);
                (void)Fclose(file);
                free_front(tempfront);
           }
        }
}               /*end print_repart_solution*/

/*
*			print_front_and_wave():
*
*	Provides printout of front->interf and of all state variables
*	associated with wave.
*/

LOCAL void print_front_and_wave(
	FILE		*file,
	Grid		*grid,
	Wave		*wave,
	Front		*front,
	Printplot	*prt)
{
	int		var;
	int		dim = front->rect_grid->dim;
	POINTER		extra[2];
	OUTPUT_DATA	**mod = prt->main_output_data;
	bool		sav_binary = is_binary_output();

			/* Initialize */

	if (prt->initialize_for_printout != NULL)
	    (*prt->initialize_for_printout)(grid,wave,front,prt);

	extra[0] = (POINTER) front;
	extra[1] = (POINTER) wave;
	for (var = 0; var < prt->n_rect_state_vars; ++var) 
	{
	    prt->output_soln[var]->intfc = front->interf;
	    prt->output_soln[var]->extra = (POINTER) extra;
	}

	if ((print_fronts_or_states(grid,prt) == YES) && (file != NULL))
	{
	    /* Print Front.  We assume that printing or plotting
	     * states is more useful WITH the fronts. */

	    if      (output_format_on(PRT_FRONTS,prt))
	    	set_binary_output(Output_in_binary(mod[PRT_FRONTS]));
	    else if (output_format_on(RECT_STATES,prt))
	    	set_binary_output(Output_in_binary(mod[RECT_STATES]));
	    else 
	    	set_binary_output(Output_in_binary(mod[TRI_STATES]));

	    (void) fprintf(file,"\n\n\n\n");
	    (void) foutput(file);
	    (void) fprintf(file,"\t\t\tFRONT DATA:\n");
	    fprint_front(front,file);
	    (void) fprintf(file,"\n\n\n\n");
	    (void) foutput(file);
	    (void) fprintf(file,"\t\t\tEND OF FRONT DATA:\n");
	}

	if ((is_ts_for_output_format(RECT_STATES,grid,prt) ||
	     is_ts_for_output_format(TRI_STATES,grid,prt)) && (file != NULL))
	{
	    (void) fprintf(file,"\n\n\n\n");
	    (void) foutput(file);
	    (void) fprintf(file,"\t\t\tSTATE DATA:\n");

	    	/* Printout Components of Locstates in Wave */

	    if (is_ts_for_output_format(RECT_STATES,grid,prt))
	    {
	    	set_binary_output(Output_in_binary(mod[RECT_STATES]));
	    	print_states(file,prt,dim);
	    }

	    	/* Plot Components of Locstates in Wave */

	    if (is_ts_for_output_format(TRI_STATES,grid,prt))
	    {
	    	set_binary_output(Output_in_binary(mod[TRI_STATES]));
	    	plot_states(file,front,wave,prt);
	    }

	    (void) foutput(file);
	    (void) fprintf(file,"\t\t\tWAVE SPEEDS:\n");
	    print_max_wave_speed_info(file,wave);

	    if (debugging("pp_solution"))
	    	(*wave->show_wave_states)(wave);

	    (void) fprintf(file,"\n\n\n\n");
	    (void) foutput(file);
	    (void) fprintf(file,"\t\t\tEND OF STATE DATA\n");
	}
	set_binary_output(sav_binary);

#if defined(USE_HDF)
/* needed for VTK */
        if (is_ts_for_output_format(VTK_STATES,grid,prt))
        {
            for (var = 0; var < prt->n_VTK_vars; ++var)
                print_vtk_states(wave,front,prt->vtk_data+var);
        }
/* end needed for VTK */
	if (is_ts_for_output_format(HDF_STATES,grid,prt))
	{
	    for (var = 0; var < prt->n_HDF_vars; ++var)
	    	plot_hdf_data(wave,front,prt->HDF_data+var);
	}
	if (is_ts_for_output_format(SDS_STATES,grid,prt))
	{
	    for (var = 0; var < prt->n_SDS_vars; ++var)
	    	plot_hdf_data(wave,front,prt->SDS_data+var);
	}
#endif /* defined(USE_HDF) */
	if (is_ts_for_output_format(PROSTAR_STATES,grid,prt))
	{
	    for (var = 0; var < prt->n_PROSTAR_vars; var++)
	    	plot_prostar_data(wave,front,prt->PROSTAR_data+var);
	}
}		/*end print_front_and_wave*/

EXPORT	OUTPUT_DATA	**d_alloc_output_datas(
	int num_datas)
{
	OUTPUT_DATA	**datas;

	matrix(&datas,num_datas,1,sizeof(OUTPUT_DATA));
	return datas;
}		/*end alloc_output_datas*/


LOCAL	bool	print_fronts_or_states(
	Grid		*grid,
	Printplot	*prt)
{
	return (is_ts_for_output_format(PRT_FRONTS,grid,prt) ||
	    	is_ts_for_output_format(RECT_STATES,grid,prt) ||
	    	is_ts_for_output_format(TRI_STATES,grid,prt)) ?
		YES : NO;
}		/*end print_fronts_or_states*/


#if defined(ONED)
/*
*			plot_states1d():
*
*	Prints state variables using a graphs format.
*/

/*ARGSUSED*/
EXPORT  void plot_states1d(
	FILE		*file,
	Front		*front,
	Wave		*wave,
	Printplot	*prt)
{
	INTERFACE	*intfc = front->interf;
	CRXING		*crx;
	TRI_GRID	*grid = wave_tri_soln(wave)->tri_grid;
	RECT_GRID	*gr = &grid->rect_grid;
	COMPONENT	*comps = grid->components;
	Locstate	state;
	byte		*storage;
	float		coords[MAXD];
	size_t		sizest = front->sizest;
	size_t		len;
	size_t		max_len;
	int             precision;
	int		var;
	size_t		num_var = prt->n_tri_vars;
	int		ix, xmax;
	int		i, nc, *list;
	char		fmt[80];

	if (num_var == 0)
	    return;

	max_len = 26;
	precision = 20;
	for (var = 0;  var < num_var;  ++var)
	{
	    len = strlen(prt->tri_plot_name[var]);
	    if (len > max_len)
	    	max_len = len;
	}
	(void) sprintf(fmt," %%-%ds",(int)max_len);
	(void) foutput(file);
	(void) fprintf(file,fmt,"POSITION");
	for (var = 0;  var < num_var;  ++var)
	    (void) fprintf(file,fmt,prt->tri_plot_name[var]);
	(void) fprintf(file,"\n");

	xmax = gr->gmax[0];
	storage = grid->rect_state_storage;
	(void) sprintf(fmt," %%-%d.%dg",(int)max_len,precision);
	for (ix = 0; ix < xmax; ++ix)
	{
	    if (!is_excluded_comp(comps[ix],intfc))
	    {
	    	state = (Locstate)(storage + ix*sizest);
	    	coords[0] = cell_edge(ix,0,gr);
	    	print_1d_solution_line(file,coords,comps[ix],state,
				       front,wave,prt,fmt);
	    }
	    nc = grid->seg_crx_count[ix];
	    list = grid->seg_crx_lists[ix];
	    for (i = 0; i < nc; ++i)
	    {
	    	crx = &(grid->crx_store[list[i]]);
	    	coords[0] = Coords(crx->pt)[0];
	    	print_1d_solution_line(file,coords,negative_component(crx->pt),
				       left_state(crx->pt),front,wave,prt,fmt);
		print_1d_solution_line(file,coords,positive_component(crx->pt),
				       right_state(crx->pt),front,wave,prt,fmt);
	    }
	}
}		/*end plot_states1d*/

LOCAL	void	print_1d_solution_line(
	FILE		*file,
	float		*coords,
	COMPONENT	comp,
	Locstate	state,
	Front		*front,
	Wave		*wave,
	Printplot	*prt,
	char		*fmt)
{
	int	var;
	size_t	num_var = prt->n_tri_vars;
	static	float	*fvals = NULL;
	static	size_t	fval_len = 0;

	if (fvals == NULL)
	{
	    fval_len = num_var+1;
	    vector(&fvals,fval_len,FLOAT);
	}
	else if (fval_len < (num_var+1))
	{
	    free(fvals);
	    fval_len = num_var+1;
	    vector(&fvals,fval_len,FLOAT);
	}
	fvals[0] = coords[0];
	for (var = 0; var < num_var; ++var)
	{
	    fvals[var+1] = (*prt->tri_plot_function[var])(coords,front,
							  wave,comp,state);
	}

	if (is_binary_output() == YES)
	{
	    (void) fprintf(file,"\f%c",(char)(num_var+1));
	    (void) fwrite((const void *) fvals,FLOAT,num_var+1,file);
	}
	else
	{
	    for (var = 0; var <= num_var; ++var)
	    	(void) fprintf(file,fmt,fvals[var]);
	    (void) fprintf(file,"\n");
	}
}		/*end print_1d_solution_line*/
#endif /* defined(ONED) */

#if defined(TWOD)

/*
*			plot_states2d():
*
*	Calls print_tri_soln() to generate input data for the
*	program "tri", which will make discontinuous contour plots.
*/

EXPORT  void plot_states2d(
	FILE		*file,
	Front		*front,
	Wave		*wave,
	Printplot	*prt)
{
	float	   *gl = wave->rect_grid->GL;
	float	   *gu = wave->rect_grid->GU;
	int	   num_sources = wave->num_point_sources;
	int	   i, j, var, dim = wave->rect_grid->dim;
	static const char *CL[3] = {"CXL", "CYL", "CZL"};
	static const char *CU[3] = {"CXU", "CYU", "CZU"};

	if (prt->n_tri_vars == 0)
	    return;

	(void) foutput(file);
	(void) fprintf(file,"TRI_SOLN:");

	for (var = 0;  var < prt->n_tri_vars;  ++var)
	{
	    (void) fprintf(file,"%s%s"," ",prt->tri_plot_name[var]);
	}
	(void) fprintf(file,"\n");

	(void) fprintf(file,"#Point Source Data\n#%d\n",num_sources);
	for (i = 0;  i < num_sources;  ++i)
	{
	    (void) fprintf(file,"#%d ",wave->source_type[i]);
	    for (j = 0; j < dim; ++j)
	    	(void) fprintf(file,"%g ",wave->srcpt[j][i]);
	    (void) fprintf(file,"%g\n",wave->strength[i]);
	}

	(void) fprintf(file,"#Clipping Boundary\t");
	for (i = 0; i < dim; ++i)
	    (void) fprintf(file,"%s = %g %s = %g ",CL[i],gl[i],CU[i],gu[i]);
	(void) fprintf(file,"\n");

	print_tri_soln(file,front,wave,wave_tri_soln(wave),
		       prt->n_tri_vars,prt->tri_plot_function);

	(void) foutput(file);
	(void) fprintf(file,"END OF TRI_SOLN\n");

}		/*end plot_states2d*/
#endif /* defined(TWOD) */

#if defined(THREED)
/*ARGSUSED*/
EXPORT  void plot_states3d(
	FILE		*file,
	Front		*front,
	Wave		*wave,
	Printplot	*prt)
{
	if (prt->n_tri_vars == 0)
	    return;

	screen("ERROR in plot_states3d(), function not implemented\n");
	clean_up(ERROR);
}		/*end plot_states3d*/
#endif /* defined(THREED) */


LOCAL void plot_prostar_data(
	Wave		  *wave,
	Front		  *front,
	PROSTAR_plot_data *prostar_data)
{

	PROSTAR_PRINT_OPTIONS *opts = &PROSTAR_print_opts(prostar_data);
	char                  *v_file_name = prostar_data->v_file_name;
	char                  *c_file_name = prostar_data->c_file_name;
	char                  *p_file_name = prostar_data->p_file_name;
	char                  *base_file_name = prostar_data->base_file_name;
	FILE                  *vfile, *cfile, *pfile;
	RECT_GRID	      *gr = front->rect_grid;
	int		      nvars = prostar_num_vars(opts);
	int		      i, j, k, dim = prostar_data->dim;
	bool                  vrt_cel_created = prostar_data->vrt_cel_created;
	int		      var, cell_count;
	int		      *pixels, *vertices;
	int                   width, length, height;
	float		      L[3], U[3];
	float		      coords[3];
	float		      *step=prostar_data->step;
	COMPONENT             comp;
	static Locstate	      state = NULL;
	float                 intfc_jump[1];
	float                 phys_data[6], temp[6];
	int		      stp = front->step;
	const char	      *vert_suffix, *cell_suffix, *phys_suffix;

	vert_suffix = ".vrt";
	cell_suffix = ".cel";
	phys_suffix = ".usr";

	if (debugging("PROSTAR"))
	    printf("Entering plot_prostar_data()\n");

	if (state == NULL)
	    alloc_state(front->interf,&state,front->sizest);

	pixels = prostar_pixels(opts);
	vertices = prostar_vertices(opts);

	for(i = 0; i < 3; i++)
	{
	    vertices[i] = pixels[i] + 1;
	    L[i] = prostar_L0(opts)[i];
	    L[i] = max(L[i],front->rect_grid->GL[i]);
	    U[i] = prostar_U0(opts)[i];	
	    U[i] = min(U[i],front->rect_grid->GU[i]);	
	}	

	for (i = 0; i < dim; i++)
	{
	  if (pixels[i] == 0)
	  {
	      if (debugging("PROSTAR"))
		  (void) printf("pixels[%d] = 0\n",i);
	      break;
	  }
	}

	width = pixels[0];
	length = pixels[1];
	height = pixels[2];

	for(i = 0; i < 3; i++)
	{ 
	    prostar_data->step[i] = prostar_len(opts)[i]/pixels[i];
	    step[i] = prostar_len(opts)[i]/pixels[i];	   	
	}
   	
	(void) sprintf(prostar_data->p_file_name,
		       "%s-ts%s%s",prostar_data->base_file_name,
		       right_flush(stp,TSTEP_FIELD_WIDTH),phys_suffix);

	if (vrt_cel_created == 0)
	{
	    (void) sprintf(prostar_data->v_file_name,
			   "%s%s", prostar_data->base_file_name,
			   vert_suffix);
	    (void) sprintf(prostar_data->c_file_name,
			   "%s%s",prostar_data->base_file_name,
			   cell_suffix);
	    
	    if ((vfile = fopen(v_file_name,"w")) == NULL)
	    {
		(void) screen("WARNING in plot_prostar_data(), "
			      "can't open %s\n",v_file_name);
		return;
	    }
	    if ((cfile = fopen(c_file_name,"w")) == NULL)
	    {
		(void) screen("WARNING in plot_prostar_data(), "
			      "can't open %s\n",c_file_name);
		return;
            }
           
	    
	    /* set up for vertices */
	    for (i = 0; i < 3; i++)
	    {
		vector(prostar_data->vert_scale+i+1,
		       prostar_vertices(opts)[i],FLOAT);
	    }
	    for (i = 0; i <= width; i++)
	        prostar_data->vert_scale[1][i] = L[0] + (i)*step[0];
	    for (j = 0; j <= length; j++)
	        prostar_data->vert_scale[2][j] = L[1] + (j)*step[1];
	    for (k = 0; k <= height; k++)
	        prostar_data->vert_scale[3][k] = L[2] + (k)*step[2];
	    
	    cell_count = 1;
	    for (k = 0; k <= height; k++)
	    {
		coords[2] = prostar_data->vert_scale[3][k];
		
		for (j = 0; j <= length; j++)
		{
		    coords[1] = prostar_data->vert_scale[2][j];
		    
		    for (i = 0; i <= width; i++)
	            {
			coords[0] = prostar_data->vert_scale[1][i];
			
			/* print vertex number */
			(void) fprintf(vfile,"%9d      ",
				       k*(width+1)*(length+1)+j*(width+1)+i+1);
			
			/* print coords of vertex */
			if (dim == 3)
		       	    (void) fprintf(vfile,"%16.9lf%16.9lf%16.9lf\n",
					 coords[0], coords[1], 
					 coords[2]);
			/* if dim = 2 need dummy coords for z coordinate */
			if (dim == 2)
			{
			    if (k == 0) 
			        (void) fprintf(vfile,"%16.9lf%16.9lf%16.9lf\n",
					     coords[0], coords[1], 
					     coords[2]);
			    if (k == 1) 
			        /* height will be default=.001 for dim=2*/
			        (void) fprintf(vfile,"%16.9lf%16.9lf%16.9lf\n",
					       coords[0], coords[1], 
					       coords[2]+.001);
			}
			/* print cell data */
			if ((i != width) && (j != length) && ( k != height))
			{
			    /* print cell number */
			    (void) fprintf(cfile,"%9d      ",cell_count++);
			    /* print the numbers of its vertices */
			    (void) fprintf(cfile,
					   "%9d%9d%9d%9d%9d%9d%9d%9d%9d%4d\n",
					   k*(width+1)*(length+1)+j*(width+1)+i+1,
					   k*(width+1)*(length+1)+j*(width+1)+i+2,
					   k*(width+1)*(length+1)+(j+1)*(width+1)
					   +i+2,
					   k*(width+1)*(length+1)+(j+1)*(width+1)
					   +i+1,
					   (k+1)*(width+1)*(length+1)+j*(width+1)
					   +i+1,
					   (k+1)*(width+1)*(length+1)+j*(width+1)
					   +i+2,
					   (k+1)*(width+1)*(length+1)+(j+1)*
					   (width+1)+i+2,
					   (k+1)*(width+1)*(length+1)+(j+1)*
					   (width+1)+i+1,1,1);
			}
		    }
		}
	    }
	    fclose(vfile);
	    fclose(cfile);
	    prostar_data->vrt_cel_created = 1;
	}
	
	if ((pfile = fopen(p_file_name,"w")) == NULL)
	{
	    (void) screen("WARNING in plot_prostar_data(), "
			  "can't open %s\n",p_file_name);
	    return;
	}

	/* set up for  physics, need coords of cell centers */
	for (i = 0; i < width; i++)
	    prostar_data->scale[1][i] = U[0] - (i+.5)*step[0];
	for (j = 0; j < length; j++)
	    prostar_data->scale[2][j] = L[1] + (j+.5)*step[1];
	for (k = 0; k < height; k++)
	    prostar_data->scale[3][k] = L[2] + (k+.5)*step[2];
	
	/* default data */
	for (var = 0; var < 6; var++)
	{
	    phys_data[var] = 0.0; /*default vales */
	    temp[var] = 0.0; 
	}
	cell_count = 1;
	for (k = 0; k < height; k++)
	{
	    coords[2] = prostar_data->scale[3][k];
 
	    for (j = 0; j < length; j++)
            {
		coords[1] = prostar_data->scale[2][j];

		for (i = 0; i < width; i++)
		{
		    coords[0] = prostar_data->scale[1][i];

		    /* find component at coords, needed for hyp_solution */
		    comp = component(coords,front->interf);
		    if (is_exterior_comp(comp,front->interf))
		    {
			comp = nearest_interior_comp(YES,NO_COMP,coords,
						      front->interf);
		    }

		    /* find state data at coords */
		    hyp_solution(coords,comp,NULL,UNKNOWN_SIDE,
				 front,wave,state,NULL);

		    if ((dim == 2)&&(nvars >5))
		        nvars = 5;
		    if ((dim == 3)&&(nvars > 6))
		        nvars = 6;

		    for (var = 0; var < nvars; var++)
		    {
			phys_data[var] =
			  (PROSTAR_frame_plot_function(prostar_data->frame_data[var])
			    (coords,front,wave,comp,state));
		    }
		    if (dim == 2)
		    {
		        for (var = 2; var < nvars; var++)
			{
			    temp[var] = phys_data[var];
			}
			phys_data[2] = 0.0;
		        for (var = 3; var < nvars+1; var++)
			{
			    phys_data[var] = temp[var-1];
			}

		    }
		    (void) fprintf(pfile,"%9d      ",cell_count++);
		    (void) fprintf(pfile,
				   "%16.9lf%16.9lf%16.9lf%16.9lf%16.9lf%16.9lf\n",
				   phys_data[0], phys_data[1], phys_data[2],
				   phys_data[3], phys_data[4], phys_data[5]);
		}
	    }
	}
	fclose(pfile);
	(void) output();

	if (debugging("PROSTAR"))
	    printf("Leaving plot_prostar_data()\n");
}	/* end plot_prostar_data */

LOCAL	void	set_PROSTAR_plotting_range(
	Front		   *front,
	PROSTAR_plot_data  *prostar_data,
	int		   *pixels,
	int                *vertices,
	float		   *L,
	float		   *U)
{
	PROSTAR_PRINT_OPTIONS	*opts = &PROSTAR_print_opts(prostar_data);
	int	i, dim = prostar_data->dim;

	debug_print("PROSTAR","Entered set_PROSTAR_plotting_range().\n");
	for (i = 0; i < dim; i++)
	{
	    pixels[i] = prostar_pixels(opts)[i];
	    vertices[i] = prostar_vertices(opts)[i];
	    vertices[i] = pixels[i] + 1;

	    L[i] = prostar_L0(opts)[i];
	    L[i] = max(L[i],front->rect_grid->GL[i]);
	    U[i] = prostar_U0(opts)[i];	
	    U[i] = min(U[i],front->rect_grid->GU[i]);
	}
	debug_print("PROSTAR","Left set_PROSTAR_plotting_range().\n");
}		/*end set_PROSTAR_plotting_range*/

#if defined(USE_HDF)
LOCAL void plot_hdf_data(
	Wave		*wave,
	Front		*front,
	HDF_plot_data	*hdf_data)
{
	COMPONENT         *comp;
	HDF_PRINT_OPTIONS *opts = &HDF_print_opts(hdf_data);
	RECT_GRID	  *gr = front->rect_grid;
	const char	  *type_name;
	int		  nvars = hdf_num_vars(opts);
	int		  i, dim = hdf_data->dim;
	int		  no_pixels;
	long		  count;
	int		  var;
	int		  pixels[3];
	float		  L[3], U[3];
	float		  l[3], u[3];
	static	long (*fill_hdf_values[4])(Wave*,Front*,HDF_plot_data*,
					   float*,float*,float*,float*,int*) =
					{ NULL,
					  fill_hdf_values1d,
					  fill_hdf_values2d,
					  fill_hdf_values3d};

	debug_print("HDF","Entered plot_hdf_data().\n");

	set_HDF_plotting_range(front,hdf_data,pixels,L,U);

	no_pixels = 1;
	for (i = 0; i < dim; ++i)
	{
	    if (pixels[i] == 0)
	    {
	        if (debugging("HDF"))
		    (void) printf("pixels[%d] = 0\n",i);
		no_pixels = 0;
	    	break;
	    }
	}
	pp_global_imin(&no_pixels,1L);
	if (no_pixels == 0)
	{
	    if (debugging("HDF"))
	    {
		(void) printf("zero size image detected\n");
	    }
	    debug_print("HDF","Left plot_hdf_data().\n");
	    return;
	}

	for (i = 0; i < dim; ++i)
	{
	    l[i] = gr->L[i] /*old had: - 0.5*gr->lbuf[i]*gr->h[i]*/;
	    u[i] = gr->U[i] /*old had: + 0.5*gr->ubuf[i]*gr->h[i]*/;
	}

	count = (*fill_hdf_values[dim])(wave,front,hdf_data,l,u,L,U,pixels);

	comp = hdf_data->comp;
	for (var = 0; var < nvars; ++var)
	{
	    float	min_val, max_val;
	    float	*values = hdf_data->frame_data[var].values;
	    long	k;

	    min_val = HUGE_VAL, max_val = -HUGE_VAL;
	    for (k = 0; k < count; ++k)
	    {
	        if (!is_excluded_comp(comp[k],front->interf))
		{
		    min_val = min(min_val,values[k]);
		    max_val = max(max_val,values[k]);
	        }
	    }
	    hdf_data->frame_data[var].current_time_min = min_val;
	    hdf_data->frame_data[var].current_time_max = max_val;
	    if (hdf_data->frame_data[var].cumulative_time_min > min_val)
		hdf_data->frame_data[var].cumulative_time_min = min_val;
	    if (hdf_data->frame_data[var].cumulative_time_max < max_val)
		hdf_data->frame_data[var].cumulative_time_max = max_val;
	}

	if (is_io_node(pp_mynode()))
	{
	    if (hdf_data_type(opts) == HDF_RASTER)
	    {
	        for (var = 0; var < nvars; ++var)
	            print_raster_data(front->interf,hdf_data->frame_data+var,
		                      pixels,hdf_data);
	    }
	    if (hdf_data_type(opts) == HDF_SDS)
	    {
	        for (var = 0; var < nvars; ++var)
		    print_sds_data(front,L,U,
				   hdf_data->frame_data+var,hdf_data);
	    }
	}
	pp_gsync();

	type_name = (hdf_data_type(opts) == HDF_RASTER) ? "HDF" : "SDS";
	(void) printf("\n\n\n");
	(void) output();
	(void) printf("\t\t\t%s GENERATED STATISTICS:\n\n",type_name);
	(void) printf("Min/max values up to time %g\n\n",front->time);
	(void) printf("%-20s %-20s %-20s %-20s %-20s\n",
		      "STATE_VARIABLE","CURRENT_MIN","CURRENT_MAX",
		      "CUMULATIVE_MIN","CUMULATIVE_MAX");
	for (var = 0; var < nvars; ++var)
	{
	    (void) printf("%-20s %-20g %-20g %-20g %-20g\n",
			  HDF_frame_plot_name(hdf_data->frame_data[var]),
			  hdf_data->frame_data[var].current_time_min,
			  hdf_data->frame_data[var].current_time_max,
			  hdf_data->frame_data[var].cumulative_time_min,
			  hdf_data->frame_data[var].cumulative_time_max);
	}
	(void) printf("\n\n");
	(void) output();
	(void) printf("\t\t\tEND %s GENERATED STATISTICS:\n\n",type_name);
	debug_print("HDF","Left plot_hdf_data().\n");
}		/*end plot_hdf_data*/ 

/*ARGSUSED*/
LOCAL	long	fill_hdf_values1d(
	Wave		*wave,
	Front		*front,
	HDF_plot_data	*hdf_data,
	float		*l,
	float		*u,
	float		*L,
	float		*U,
	int		*pixels)
{
	COMPONENT	*comp;
	float		coords[3];
	float		*step = hdf_data->step;
	int		nvars = HDF_num_vars(HDF_print_opts(hdf_data));
	int		width = pixels[0];
	int		nn = pp_numnodes();
	int		var, i;
	long		count, indx;
	static Locstate	state = NULL;

	debug_print("HDF","Entered fill_hdf_values1d().\n");

	if (state == NULL)
	    alloc_state(front->interf,&state,front->sizest);

	if (nn > 1)
	{
	    comp = hdf_data->comp;
	    for (count = 0, i = 0; i < width; ++i, ++count, ++comp)
	    {
		*comp = NO_COMP;
	        for (var = 0; var < nvars; ++var)
	            hdf_data->frame_data[var].values[count] = -HUGE_VAL;
	    }
	}
	else
	    count = width;
	comp = hdf_data->comp;

	/*Fill scale array*/

	for (i = 0; i < width; i++)
	    hdf_data->scale[1][i] = L[0] + (i+.5)*step[0];

	for (indx = 0, i = 0; i < width; ++i, ++indx, ++comp)
	{
	    coords[0] = hdf_data->scale[1][i];
	    if ((nn > 1) && ((coords[0] < l[0]) || (coords[0] > u[0])))
		continue;
	    *comp = component(coords,front->interf);
	    if (is_excluded_comp(comp[indx],front->interf))
		obstacle_state(front->interf,state,front->sizest);
	    else
	    {
	        if (is_exterior_comp(*comp,front->interf))
		    *comp = nearest_interior_comp(YES,NO_COMP,
		                                  coords,front->interf);
	        hyp_solution(coords,*comp,NULL,UNKNOWN_SIDE,
		             front,wave,state,NULL);
	    }

	    for (var = 0; var < nvars; ++var)
	    {
	        hdf_data->frame_data[var].values[indx] =
	            (*HDF_frame_plot_filter(hdf_data->frame_data[var]))(
			(*HDF_frame_plot_function(hdf_data->frame_data[var]))(
	    			                  coords,front,wave,
						  *comp,state));
	    }
	}
	pp_collect_hdf_data(count,hdf_data);
	debug_print("HDF","Left fill_hdf_values1d().\n");
	return count;
}		/*end fill_hdf_values1d*/

LOCAL	long	fill_hdf_values2d(
	Wave		*wave,
	Front		*front,
	HDF_plot_data	*hdf_data,
	float		*l,
	float		*u,
	float		*L,
	float		*U,
	int		*pixels)
{
	COMPONENT	*comp;
	float		coords[3];
	float		*step = hdf_data->step;
	int		nvars = HDF_num_vars(HDF_print_opts(hdf_data));
	int		width = pixels[0];
	int		height = pixels[1];
	int		nn = pp_numnodes();
	int		var, i, j;
	long		count,indx;
	int             p = 0, q = 0;
	float           intfc_jump[1];
	float           sum[2][nvars];
	static Locstate	state = NULL;

	debug_print("HDF","Entered fill_hdf_values2d().\n");

	for (i = 0; i < nvars; i++)
	{
	    sum[0][i] = 0.0;
	    sum[1][i] = 0.0;
	}
	intfc_jump[0] = 0.0;

	if (state == NULL)
	    alloc_state(front->interf,&state,front->sizest);

	if (nn > 1)
	{
	    comp = hdf_data->comp;
	    for (count = 0, j = 0; j < height; ++j)
	    for (i = 0; i < width; ++i, ++count, ++comp)
	    {
		*comp = NO_COMP;
	        for (var = 0; var < nvars; ++var)
	            hdf_data->frame_data[var].values[count] = -HUGE_VAL;
	    }
	}
	else
	    count = width*height;

	for (i = 0; i < width; i++)
	    hdf_data->scale[1][i] = U[0] - (i+.5)*step[0];
	for (j = 0; j < height; j++)
	    hdf_data->scale[2][j] = L[1] + (j+.5)*step[1];

	for (j = 0, comp = hdf_data->comp; j < height; ++j)
	{
	    coords[1] = hdf_data->scale[2][j];
	    if ((nn > 1) && ((coords[1] < l[1]) || (coords[1] > u[1])))
		continue;
	    for (i = 0; i < width; ++i)
	    {
	        coords[0] = hdf_data->scale[1][i];
	        if ((nn > 1) && ((coords[0] < l[0]) || (coords[0] > u[0])))
		    continue;
		indx = i + j*width;
	        comp[indx] = component(coords,front->interf);
		if (is_excluded_comp(comp[indx],front->interf))
		    obstacle_state(front->interf,state,front->sizest);
		else
		{
		    if (is_exterior_comp(comp[indx],front->interf))
		        comp[indx] = nearest_interior_comp(YES,NO_COMP,coords,
						           front->interf);
	            hyp_solution(coords,comp[indx],NULL,UNKNOWN_SIDE,
			         front,wave,state,NULL);
		}

	        for (var = 0; var < nvars; ++var)
	        {
	            hdf_data->frame_data[var].values[indx] =
	              (*HDF_frame_plot_filter(hdf_data->frame_data[var]))(
		        (*HDF_frame_plot_function(hdf_data->frame_data[var]))(
	    				          coords,front,wave,
						  comp[indx],state));
		    sum[0][var] = sum[0][var] +  
		      hdf_data->frame_data[var].values[indx] - 
		      (*intfc_jump);
		    sum[1][var] = sum[1][var] + (*intfc_jump);
		    if (var == 0) /* so we don't count more than once */
		    {
			if (*intfc_jump == 0.0)
			  p++;		
			else if (*intfc_jump != 0.0)
			  q++;
		    }
	        }
	    }
	}
	/*	printf("\ntotal number of pixels is %d.\n",p+q);
	printf("number of interior pixels is %d = %lf percent of pixels.\n",
	       p,(float)p/(p+q));
	printf("number of interface pixels is %d = %lf percent of pixels.\n\n",
	       q,(float)q/(p+q));
	for (var = 0; var < nvars; var++)
	{
	    pp_global_sum(sum[0],var);
	    pp_global_sum(sum[1],var);
	    printf("Total var%d = %lf.\n", var, sum[0][var] + sum[1][var]);
	    printf("Interior integral for var%d = %lf, = %lf  of total\n", 
		   var,sum[0][var],sum[0][var]/(sum[0][var]+sum[1][var]));
	    printf("Interface integral for var%d = %lf, = %lf  of total\n\n", 
		   var,sum[1][var],sum[1][var]/(sum[0][var]+sum[1][var]));
	    
		   }*/
	pp_collect_hdf_data(count,hdf_data);
	debug_print("HDF","Left fill_hdf_values2d().\n");
	return count;
}		/*end fill_hdf_values2d*/

LOCAL	long	fill_hdf_values3d(
	Wave		*wave,
	Front		*front,
	HDF_plot_data	*hdf_data,
	float		*l,
	float		*u,
	float		*L,
	float		*U,
	int		*pixels)
{
	COMPONENT	*comp;
	float		coords[3];
	float		*step = hdf_data->step;
	int		nvars = HDF_num_vars(HDF_print_opts(hdf_data));
	float           sum[2][nvars], lsum[2][nvars];
	int		width = pixels[0];
	int		length = pixels[1];
	int		height = pixels[2];
	int		nn = pp_numnodes();
	int             mn = pp_mynode();
	int		var, i, j, k;
	int             p = 0, q = 0;
	float           intfc_jump[1];
	long		count, indx;
	static Locstate	state = NULL;

	debug_print("HDF","Entered fill_hdf_values3d().\n");

	for (i = 0; i < nvars; i++)
        {
	    sum[0][i] = 0.0;
	    sum[1][i] = 0.0;
	}
	intfc_jump[0] = 0.0;

	if (state == NULL)
	    alloc_state(front->interf,&state,front->sizest);

	if (nn > 1)
	{
	    comp = hdf_data->comp;
	    for (count = 0, k = 0; k < height; ++k)
	    for (           j = 0; j < length; ++j)
	    for (           i = 0; i < width;  ++i, ++count, ++comp)
	    {
		*comp = NO_COMP;
	        for (var = 0; var < nvars; ++var)
	            hdf_data->frame_data[var].values[count] = -HUGE_VAL;
	    }
	}
	else
	    count = width*length*height;

	for (i = 0; i < width; i++)
	    hdf_data->scale[1][i] = U[0] - (i+.5)*step[0];
	for (j = 0; j < length; j++)
	    hdf_data->scale[2][j] = L[1] + (j+.5)*step[1];
	for (k = 0; k < height; k++)
	    hdf_data->scale[3][k] = L[2] + (k+.5)*step[2];
	

	for (k = 0, comp = hdf_data->comp; k < height; ++k)
	{
	    coords[2] = hdf_data->scale[3][k];
	    if ((nn > 1) && ((coords[2] < l[2]) || (coords[2] > u[2])))
		continue;
	    for (j = 0; j < length; ++j)
	    {
	        coords[1] = hdf_data->scale[2][j];
	        if ((nn > 1) && ((coords[1] < l[1]) || (coords[1] > u[1])))
		    continue;
	        for (i = 0; i < width; ++i)
	        {
	            coords[0] = hdf_data->scale[1][i];
	            if ((nn > 1) && ((coords[0] < l[0]) || (coords[0] > u[0])))
		        continue;
		    indx = i + j*width + k*width*length;
	            comp[indx] = component(coords,front->interf);
		    if (is_excluded_comp(comp[indx],front->interf))
		        obstacle_state(front->interf,state,front->sizest);
		    else
		    {
		        if (is_exterior_comp(comp[indx],front->interf))
		            comp[indx] = nearest_interior_comp(YES,NO_COMP,
			                                       coords,
							       front->interf);
	                hyp_solution(coords,comp[indx],NULL,UNKNOWN_SIDE,
				     front,wave,state,NULL);
		    }

	            for (var = 0; var < nvars; ++var)
	            {
	              hdf_data->frame_data[var].values[indx] =
	                (*HDF_frame_plot_filter(hdf_data->frame_data[var]))(
			  (*HDF_frame_plot_function(hdf_data->frame_data[var]))(
	    				            coords,front,wave,
						    comp[indx],state));
			sum[0][var] = sum[0][var] +  
			  hdf_data->frame_data[var].values[indx] - 
			  (*intfc_jump);
			sum[1][var] = sum[1][var] + (*intfc_jump);
			if (*intfc_jump == 0.0)
			    p++;		
			else if (*intfc_jump != 0.0)
			    q++;
	            }
	        }
	    }
	}
	
	printf("total number of pixels is %d.\n",p+q);
	printf("number of interior pixels is %d = %lf percent of pixels.\n",
	       p,(float)p/(p+q));
	printf("number of interface pixels is %d = %lf percent of pixels .\n",
	       q,(float)q/(p+q));
	for (var = 0; var < nvars; var++)
	{
	    pp_global_sum(sum[0],var);
	    pp_global_sum(sum[1],var);
	    printf("Total var%d = %lf.\n", var, sum[0][var] + sum[1][var]);
	    printf("Interior total for var%d = %lf, = %lf of total.\n",
		   var, sum[0][var], sum[0][var]/(sum[0][var] + sum[1][var]) );
	    printf("Interface total for var%d = %lf, = %lf of total.\n\n",
		   var, sum[1][var], sum[1][var]/(sum[0][var] + sum[1][var]) );
	}
	
	pp_collect_hdf_data(count,hdf_data);
	debug_print("HDF","Left fill_hdf_values3d().\n");
	return count;
}		/*end fill_hdf_values3d*/

LOCAL	void	pp_collect_hdf_data(
	long		count,
	HDF_plot_data	*hdf_data)
{
	int nvars = HDF_num_vars(HDF_print_opts(hdf_data));
	int var;

	if (pp_numnodes() == 1)
	    return;

	if (debugging("HDF"))
	    (void) printf("PP %d has count = %ld\n",pp_mynode(),count);

	pp_global_imax(hdf_data->comp,hdf_data->num_values);
	for (var = 0; var < nvars; ++var)
	    pp_global_max(hdf_data->frame_data[var].values,
			  hdf_data->num_values);

}		/*end pp_collect_hdf_data*/

LOCAL	void	set_HDF_plotting_range(
	Front		*front,
	HDF_plot_data	*hdf_data,
	int		*pixels,
	float		*L,
	float		*U)
{
	HDF_PRINT_OPTIONS	*opts = &HDF_print_opts(hdf_data);
	int	i, dim = hdf_data->dim;

	debug_print("HDF","Entered set_HDF_plotting_range().\n");
	for (i = 0; i < dim; ++i)
	{
	    pixels[i] = hdf_pixels(opts)[i];
	    L[i] = hdf_L0(opts)[i] + front->time*hdf_V(opts)[i];	
	    L[i] = max(L[i],front->rect_grid->GL[i]);
	    U[i] = hdf_U0(opts)[i] + front->time*hdf_V(opts)[i];	
	    U[i] = min(U[i],front->rect_grid->GU[i]);
	    if (hdf_data_type(opts) == HDF_RASTER)
		pixels[i] = ((U[i] - L[i]) > 0.0) ? 
			    irint(pixels[i]*(U[i] - L[i])/hdf_len(opts)[i]) :
			    0;
	}
	debug_print("HDF","Left set_HDF_plotting_range().\n");
}		/*end set_HDF_plotting_range*/

LOCAL	void	print_raster_data(
	INTERFACE       *intfc,
	HDF_frame_data	*frame_data,
	int		*pixels,
	HDF_plot_data	*hdf_data)
{
	COMPONENT         *comp = hdf_data->comp;
	HDF_PRINT_OPTIONS *opts = &HDF_print_opts(hdf_data);
	float	          max_val, min_val;
	float	          *f_val;
	float	          vrng;
	float	          fmax_color;
	uint32		  comp_code = ras_compression_type(opts);
	int	          i, j, k;
	int	          width = pixels[0];
	int	          height = pixels[1];
	uint8	          line_color = frame_data->line_color;
	uint8	          barrier_color = frame_data->line_color-1;
	uint8	          num_table_colors = frame_data->num_table_colors;
	uint8	          max_color = frame_data->num_colors-1;
	uint8	          *r_val = hdf_data->raster_data;;

	fmax_color = max_color;
	f_val = frame_data->values;
	if (hdf_frame_dflt_scale(frame_data) == YES)
	{
	    hdf_frame_scale_min(frame_data) = min_val =
		frame_data->current_time_min;
	    hdf_frame_scale_max(frame_data) = max_val =
		frame_data->current_time_max;
	}
	else
	{
	    min_val = hdf_frame_scale_min(frame_data);
	    max_val = hdf_frame_scale_max(frame_data);
	}
	f_val = frame_data->values;
	vrng = max_val-min_val;
	for (i = 0; i < width; ++i)
	    r_val[i] = r_val[i+height-1] = line_color;
	for (j = 1; j < height-1; ++j)
	    r_val[width*j] = r_val[width*(j+1)-1] = line_color;
	for (j = 1; j < height-1; ++j)
	for (i = 1; i < width-1; ++i)
	{
	    k = j*width+i;
	    /* the following commented code will print the 
	       interfaces in black */
	    /*if ( (comp[k] != comp[k-1]) or (comp[k] != comp[k-width]) or
	      (comp[k] != comp[k-width-1]) )
	      r_val[k] = line_color;
	      else */
	    if (f_val[k] == 0.0)
	        r_val[k] == line_color;
	    else if (f_val[k] < min_val)
		r_val[k] = num_table_colors;
	    else if (f_val[k] > max_val)
		r_val[k] = num_table_colors + max_color;
	    else
	        r_val[k] = num_table_colors + 
	    	    (uint8)irint(fmax_color*(f_val[k]-min_val)/vrng);
	}

	(void) DFR8setpalette(frame_data->palette);
	if ((frame_data->first == YES) && (frame_data->append == NO))
	    (void) DFR8putimage(frame_data->file_name,r_val,width,height,
				comp_code);
	else
	    (void) DFR8addimage(frame_data->file_name,r_val,width,height,
				comp_code);
	frame_data->first = NO;
}		/*end print_raster_data*/

/*
*			print_sds_data():
*
*	Prints a FronTier HDF scientific data set for a single scalar variable.
*
*	File Format:
*
*	The computation domain occupies the dim dimensional rectangle with lower and upper
*	corners L and U respectively.  The ploting window occupies the region defined by
*	the dim dimensional rectangle with lower and upper corners wL_i and wU_i respectively for
*	i = 1,..,N where N is the number of frames printed in the file.
*
*	Set Name                      rank  dims                start   edges     data
*       1   "Plot Name"	              1     (strlen(name)+1)    (0)     (dims[0]) Character string
*
*                                                                                 -                 -
*	2   "Computational Domain"    2     (2 dim)             (0 0)   (dims)    | L[0]...L[dim-1] |
*                                                                                 | U[0]...I[dim-1] |
*						                                  -                 -
*
*                                                                                 -                       -
*	3   "Current Plotting Window" 3     (N 2 dim)           (0 0 0) (dims)    | wL_0[0]...wL_0[dim-1] |
*                                                                                 | wU_0[0]...wU_0[dim-1] |
*                                                                                 -                       -
*								         		  ...
*                                                                                 -                               -
*                                                                                 | wL_{N-1}[0]...wL_{N-1}[dim-1] |
*                                                                                 | wU_{N-1}[0]...wU_{N-1}[dim-1] |
*                                                                                 -                               -
*
*	4   "Time"                    1     (N)                 (0)     (dims)    ( T_0,...T_{N-1} )
*
*                                                                                 -                                                 -
*	5  "Max/Min Values"           2     (N 4)               (0 0)   (dims)    | min_0 max_0 cum_min_0 cum_max_0                 |
*                                                                                 |               ...                               |
*	                                                                          | min_{N-1} max_{N-1} cum_min_{N-1} cum_max_{N-1} |
*                                                                                 -                                                 -
*       Note: min_i and max_i refer to the minimum and maximum of the scalar field being printed over the data in the 
*       corresponding frame (ie the max and min over each frames data).  The data in cum_min and cum_max refer the
*	cummulative min and max over the history of the run, that is the cummulative min and max over a series of time steps.
*
*	6   "COMPONENTS"              dim+1 (N l[dim-1]...l[0]) (0 0 0) (dims)    Each frame contains a component number array
*                                                                                 of the current state of the computation, sampled
*                                                                                 onto a rectangular lattice.  This lattice is
*                                                                                 stored as an up to 3 dimensional matrix.  The
*									          X (or R in cylindrical or spherical geometry) values
*                                                                                 are stored in increasing order as consecutive data
*                                                                                 along a row.  X or R coordinates correspond to the
*                                                                                 last dimension in dims.  Y (in 2D cylindrical Z)
*                                                                                 are stored in decreasing (in direction) order along
*                                                                                 the second to last dimension in dims.  Thus for
*                                                                                 a fixed frame number and Z coordinate slice, X and Y
*                                                                                 are indexed in the geometrically natural order so
*                                                                                 that the image of the matrix with X varing along
*                                                                                 the columns and Y along the rows corresponds to
*                                                                                 the natural coordinate system with the highest Y
*                                                                                 value occuring along the first (ie top) row.  In 3D
*                                                                                 Z coordinates are added as consecutive planes 
*                                                                                 perpendicular to the Z-axis with the Z coordinate
*                                                                                 decreasing with plane index.
*
*	7   the name stored as the    dim+1 (N l[dim-1]...l[0]) (0 0 0) (dims)    This field contains the values of the scalar
*           value of the "Plot Name"                                              field being printed.  Its layout is the same as
*           field                                                                 described for the component field above.
*/

LOCAL	void	print_sds_data(
	Front		*front,
	float		*L,
	float		*U,
	HDF_frame_data	*frame_data,
	HDF_plot_data	*hdf_data)
{
	COMPONENT	  *comp = hdf_data->comp;
	HDF_PRINT_OPTIONS *opts = &HDF_print_opts(hdf_data);
	bool		  newfile, status;
	comp_info         *c_info = &hdf_compression_info(opts);
	char		  file_name[1024];
	float		  mm[4];
	float		  domain[6];
	int32		  sd_id;
	int32		  rank;
	int32		  dims[4], start[4], edges[4];
	comp_coder_t	  comp_code = sds_compression_type(opts);
	int		  dim = hdf_data->dim;
	int		  *pixels = HDF_pixels(HDF_print_opts(hdf_data));
	int		  i;
	int		  step = front->step;
	int32		  access_mode;
	const char	  **Dnm = front->rect_grid->Remap.Dnm;
	const char	  *dir_names[4];
	static const char *win_dim_names[] = {
						"Window Print Step",
						"Window Lower Bound",
						"Window Upper Bound"
					   };
	static const char *time_dim_name[] = {"Time"};

	static int32	  size_float = -1;

	debug_print("HDF","Entered print_sds_data().\n");

	newfile = YES;
	access_mode = (newfile == YES) ? DFACC_CREATE : DFACC_RDWR;

	if (debugging("HDF"))
	{
	    (void) printf("In print_sds_data(), access_mode = %s\n",
			  (access_mode==DFACC_CREATE)?"DFACC_CREATE":
			  (access_mode==DFACC_RDWR)?"DFACC_RDWR":"UNKNOWN");
	}

	if (size_float == -1)
	    size_float = dfnt_size_float();

	(void) sprintf(file_name,"%s-ts%s.hdf",frame_data->file_name,
		       right_flush(step,TSTEP_FIELD_WIDTH));
	sd_id = SDstart(file_name,access_mode);

	if (newfile == YES)
	{
	    int32	sds_id, dim_id;

	    /*Write data name*/
	    rank = 1;
	    dims[0] = (int32)strlen(hdf_frame_plot_name(frame_data))+1;
	    start[0] = 0;
	    edges[0] = dims[0];
	    sds_id = SDcreate(sd_id,"Plot Name",DFNT_CHAR,rank,dims);
	    if (SDwritedata(sds_id,start,NULL,edges,
			    (VOIDP)hdf_frame_plot_name(frame_data)) != SUCCEED)
	    {
		(void) printf("WARNING in print_sds_data(), SDwritedata failed "                              
                              "writing plot_name for new file\n"
			      "plot name = %s\n",
			      hdf_frame_plot_name(frame_data));
	    }
	    if (SDendaccess(sds_id) != SUCCEED)
	    {
	        (void) printf("WARNING in print_sds_data(), "
			      "SDendaccess failed after writing "
			      "plot_name for new file\n"
			      "plot name = %s\n",
			      hdf_frame_plot_name(frame_data));
	    }

	    /*Write computational domain*/
	    rank = 2;
	    dims[0] = 2;
	    dims[1] = dim;
	    start[0] = 0;
	    start[1] = 0;
	    edges[0] = 2;
	    edges[1] = dim;
	    for (i = 0; i < dim; ++i)
	    {
	    	domain[i] = front->rect_grid->L[i];
	    	domain[i+dim] = front->rect_grid->U[i];
	    }
	    sds_id = SDcreate(sd_id,"Computational Domain",
			      size_float,rank,dims);
	    dim_id = SDgetdimid(sds_id,0);
	    if (SDsetdimname(dim_id,"Domain Lower Bounday") != SUCCEED)
	    {
	        (void) printf("WARNING in print_sds_data(), SDsetdimname "
			      "failed for Domain Lower Bounday\n"
			      "plot name = %s\n",
			      hdf_frame_plot_name(frame_data));
	    }
	    dim_id = SDgetdimid(sds_id,1);
	    if (SDsetdimname(dim_id,"Domain Upper Bounday") != SUCCEED)
	    {
	        (void) printf("WARNING in print_sds_data(), SDsetdimname "
			      "failed for Domain Upper Bounday\n"
			      "plot name = %s\n",
			      hdf_frame_plot_name(frame_data));
	    }
	    if (SDwritedata(sds_id,start,NULL,edges,(VOIDP)domain) != SUCCEED)
	    {
	        (void) printf("WARNING in print_sds_data(), SDwritedata failed "
		              "writing edges for the Computational Domain\n"
			      "plot name = %s\n",
			      hdf_frame_plot_name(frame_data));
	    }
	    if (SDendaccess(sds_id) != SUCCEED)
	    {
	        (void) printf("WARNING in print_sds_data(), SDendaccess failed "
			      "for the Computational Domain\n"
			      "plot name = %s\n",
			      hdf_frame_plot_name(frame_data));
	    }
	}

	/*Write current plotting window*/
	rank = 3;
	dims[0] = SD_UNLIMITED;
	dims[1] = 2;
	dims[2] = dim;
	start[1] = 0;
	start[2] = 0;
	edges[1] = 2;
	edges[2] = dim;
	for (i = 0; i < dim; ++i)
	{
	    domain[i] = L[i];
	    domain[i+dim] = U[i];
	}
	if (write_sds_slab(newfile,sd_id,rank,start,edges,dims,NULL,
			   win_dim_names,"Current Plotting Window",
			   size_float,comp_code,
			   c_info,(VOIDP)domain) != FUNCTION_SUCCEEDED)
	{
	    (void) printf("WARNING in print_sds_data(), write_sds_slab failed "
			  "for the Current Plotting Window\n"
			  "plot name = %s\n",hdf_frame_plot_name(frame_data));
	}

	/*Record current time*/
	rank = 1;
	dims[0] = SD_UNLIMITED;
	if (write_sds_slab(newfile,sd_id,rank,start,edges,dims,NULL,
			   time_dim_name,"Time",size_float,comp_code,
			   c_info,(VOIDP)&front->time) != FUNCTION_SUCCEEDED)
	{
	    (void) printf("WARNING in print_sds_data(), write_sds_slab failed "
			  "for Time\n"
			  "plot name = %s\n",hdf_frame_plot_name(frame_data));
	}

	/* Write min/max values */
	rank = 2;
	dims[0] = SD_UNLIMITED;
	dims[1] = 4;
	start[1] = 0;
	edges[1] = 4;
	mm[0] = frame_data->current_time_min;
	mm[1] = frame_data->current_time_max;
	mm[2] = frame_data->cumulative_time_min;
	mm[3] = frame_data->cumulative_time_max;

	if (write_sds_slab(newfile,sd_id,rank,start,edges,dims,NULL,NULL,
		           "Max/Min Values",size_float,comp_code,
			   c_info,(VOIDP)mm) != FUNCTION_SUCCEEDED)
	{
	    (void) printf("WARNING in print_sds_data(), write_sds_slab failed "
			  "for Max/Min Values\n"
			  "plot name = %s\n",hdf_frame_plot_name(frame_data));
	}

	/* Set rank and dims for values and comps arrays */
	rank = dim + 1;
	dims[0] = SD_UNLIMITED;
	for (i = 1; i <= dim; ++i)
	{
	    dims[i] = pixels[dim-i];
	    start[i] = 0;
	    edges[i] = dims[i];
	}
	dir_names[0] = "Time Slice";
	for (i = 1; i <= dim; ++i)
	    dir_names[i] = Dnm[dim-i];

	if (debugging("HDF"))
	{
	    (void) printf("Dimensions of sds data\n");
	    print_int32_vector("dims = ",dims+1,dim,"\n");
	    print_int32_vector("start = ",start+1,dim,"\n");
	    print_int32_vector("edges = ",edges+1,dim,"\n");
	}

	if (sizeof(COMPONENT) != sizeof(int32))
	{
	    static int32 *i32comp;
	    if (!i32comp)
	        vector(&i32comp,hdf_data->num_values,sizeof(int32));
	    for (i = 0; i < hdf_data->num_values; ++i)
	        i32comp[i] = comp[i];
	    status = write_sds_slab(newfile,sd_id,rank,start,edges,dims,
	                            hdf_data->scale,dir_names,"COMPONENTS",
				    DFNT_INT32,comp_code,c_info,(VOIDP)i32comp);
	}
	else
	{
	    status = write_sds_slab(newfile,sd_id,rank,start,edges,dims,
	                            hdf_data->scale,dir_names,"COMPONENTS",
				    DFNT_INT32,comp_code,c_info,(VOIDP)comp);
	}
	if (!status)
	{
	    (void) printf("WARNING in print_sds_data(), write_sds_slab failed "
			  "for COMPONENTS\n"
			  "plot name = %s\n",hdf_frame_plot_name(frame_data));
	}

	/*Write values array*/
	if (write_sds_slab(newfile,sd_id,rank,start,edges,dims,hdf_data->scale,
		           dir_names,hdf_frame_plot_name(frame_data),size_float,
		           comp_code,c_info,
			   (VOIDP)frame_data->values) != FUNCTION_SUCCEEDED)
	{
	    (void) printf("WARNING in print_sds_data(), write_sds_slab failed "
			  "for the values array\n"
			  "plot name = %s\n",hdf_frame_plot_name(frame_data));
	}

	if (SDend(sd_id) != SUCCEED)
	    (void) printf("WARNING in print_sds_data(), SDend failed\n");

	frame_data->first = NO;
	debug_print("HDF","Left print_sds_data().\n");
}		/*end print_sds_data*/


LOCAL	bool	write_sds_slab(
	bool	     newfile,
	int32	     sd_id,
	int32	     rank,
	int32	     *start,
	int32	     *edges,
	int32	     *dims,
	float	     **scale,
	const char   **dim_names,
	const char   *name,
	int32	     size_data,
	comp_coder_t comp_code,
	comp_info    *c_info,
	VOIDP	     values)
{
	bool status = FUNCTION_SUCCEEDED;
	int32	sds_id, sds_idx, dim_id;
	int32	num_type, num_attrs;
	intn	i;

	debug_print("HDF","Entered write_sds_slab().\n");

	if (newfile == YES)
	{
	    start[0] = 0;
	    edges[0] = 1;
	    sds_id = SDcreate(sd_id,name,size_data,rank,dims);
	    if (dim_names != NULL)
	    {
	        for (i = 0; i < rank; ++i)
	        {
	    	    dim_id = SDgetdimid(sds_id,i);
	    	    if (SDsetdimname(dim_id,dim_names[i]) != SUCCEED)
	    	    {
	                (void) printf("WARNING in write_sds_slab(), "
			              "SDsetdimname failed, name = %s\n",name);
			status = FUNCTION_FAILED;
	    	    }
		    if (debugging("HDF"))
		    {
			(void) printf("Dimension name %d for %s set to %s\n",
				      i,name,dim_names[i]);
		    }
	        }
	    }
	}
	else
	{
	    sds_idx = SDnametoindex(sd_id,name);
	    sds_id = SDselect(sd_id,sds_idx);
	    dim_id = SDgetdimid(sds_id,0);
	    if (SDdiminfo(dim_id,NULL,start,&num_type,&num_attrs) != SUCCEED)
	    {
		(void) printf("WARNING in write_sds_slab(), "
			      "SDdiminfo failed, name = %s\n",name);
		status = FUNCTION_FAILED;
	    }
	    ++start[0];
	    edges[0] = 1;
	}
	if (scale != NULL)
	{
	    static int32 size_float = -1;

	    if (size_float == -1)
		size_float = dfnt_size_float();

	    if (debugging("HDF"))
		(void) printf("Setting dimscale for %s\n",name);
	    for (i = 0; i < rank; ++i)
	    {
		if (scale[i] != NULL)
		{
	    	    dim_id = SDgetdimid(sds_id,i);
                    /*  size of dims[i] is not consistent with size of scale[i]
                     *  see init_HDF_plots() and print_sds_data.
                     * if (SDsetdimscale(dim_id,dims[i],size_float,scale[i])
                     *                     != SUCCEED)
                     */
	    	    if (SDsetdimscale(dim_id,dims[i],size_float,scale[rank-i])
								   != SUCCEED)
	    	    {
	                (void) printf("WARNING in write_sds_slab(), "
			              "SDsetdimscale failed, name = %s\n",name);
			(void) printf("dim_id = %d, dims[%d] = %d, "
				      "size_data = %d, scale[%d] = %p\n",
				      dim_id,i,dims[i],size_data,
				      i,(POINTER)scale[i]);
			status = FUNCTION_FAILED;
	    	    }
		}
	    }
	}
	switch (comp_code)
	{
	case COMP_CODE_RLE:
	    if (debugging("HDF"))
		(void) printf("Setting RLE compression\n");
	    if (SDsetcompress(sds_id,comp_code,c_info) != SUCCEED)
	    {
		(void) printf("WARNING in write_sds_slab(), "
		              "RLE compression failed\n");
	    }
	    break;
	case COMP_CODE_DEFLATE:
	    if (debugging("HDF"))
		(void) printf("Setting GZIP compression\n");
	    if (SDsetcompress(sds_id,comp_code,c_info) != SUCCEED)
	    {
		(void) printf("WARNING in write_sds_slab(), "
		              "DEFLATE compression failed\n");
	    }
	    break;
	case COMP_CODE_SKPHUFF:
	    if (debugging("HDF"))
		(void) printf("Setting SKPHUFF compression\n");
	    c_info->skphuff.skp_size = size_data;
	    if (SDsetcompress(sds_id,comp_code,c_info) != SUCCEED)
	    {
		(void) printf("WARNING in write_sds_slab(), "
		              "SKPHUFF compression failed\n");
	    }
	    break;
	case COMP_CODE_NONE:
	default:
	    break;
	}
	if (debugging("HDF"))
	    (void) printf("Calling SDwritedata()\n");
	if (SDwritedata(sds_id,start,NULL,edges,values) != SUCCEED)
	{
	    (void) printf("WARNING in write_sds_slab(), SDwritedata failed "
		   "writing edges, name = %s\n",name);
	    status = FUNCTION_FAILED;
	}
	if (SDendaccess(sds_id) != SUCCEED)
	{
	    (void) printf("WARNING in write_sds_slab(), "
			  "SDendaccess failed, name = %s\n",name);
	    status = FUNCTION_FAILED;
	}
	debug_print("HDF","Left write_sds_slab().\n");
	return status;
}		/*end write_sds_slab*/

LOCAL	int32	dfnt_size_float(void)
{
	static int32 size_float = -1;

	if (size_float == -1)
	{
	    if (sizeof(float)==sizeof(float64))
	        size_float = DFNT_FLOAT64;
	    else if (sizeof(float)==sizeof(float32))
	        size_float = DFNT_FLOAT32;
	    else
	    {
	        screen("ERROR in print_sds_data(), "
	               "can't determine float size\n");
	        clean_up(ERROR);
	    }
	}
	return size_float;
}		/*end dfnt_size_float*/

LOCAL	void print_int32_vector(
	const char *mesg,
	int32	   *v,
	int32	   dim,
	const char *end)
{
	int32		i;

	if (mesg != NULL) (void) printf("%s",mesg);
	(void) printf("(");
	for (i = 0; i < dim; ++i)
	    (void) printf("%d%s",v[i],(i==(dim-1)) ? ")" : ", ");
	(void) printf("%s",end);
}		/*end print_int32_vector*/

#endif /* defined(USE_HDF) */

/*
*			find_time_step():
*
*	Computes the new time-step, as large as allowed by the
*	CFL condition and parabolic restrictions. 
*/

EXPORT	float find_time_step(
	Grid		*grid,
	Wave		*wave,
	Front		*front,
	Printplot	*prt,
	bool		print_set_by)
{
	TIME_STEP_SET_BY TimeStepSetBy;
	float		 max_dt, CFL;
	float		 newdt;
	float	         coords[MAXD], fcrds[MAXD], wcrds[MAXD];
	int	         i, dim = front->rect_grid->dim;

	CFL = Time_step_factor(front);
	if (dim > 1)
	    CFL *= min(Front_spacing(front,GENERAL_WAVE),
		       Front_spacing(front,VECTOR_WAVE));

	newdt = HUGE_VAL;

		/* Maximum Interior Wave Speeds (previous time) */

	TimeStepSetBy = TIME_STEP_NOT_SET;
        
	if (wave->max_hyp_time_step)
	{
	    max_dt = (*wave->max_hyp_time_step)(wave,wcrds);
#if !defined(_AIX)
	    if (!finite(max_dt) || !finite(-max_dt))
		(void) printf("WARNING in find_time_step(), "
			      "max_hyp_time_step returns infinity\n");
	    else
#endif /* !defined(_AIX) */
	    {
	        max_dt *= CFL;
	        if (max_dt < newdt)
	        {
	    	    newdt = max_dt;
	    	    TimeStepSetBy = TIME_STEP_SET_BY_WAVE;
	    	    for (i = 0; i < dim; ++i)
	    	        coords[i] = wcrds[i];
	        }
	    }
	}

	    /* Maximum Front Speeds (previous time) */

	if (front->max_front_time_step)
	{
	    max_dt = (*front->max_front_time_step)(front,fcrds);
#if !defined(_AIX)
	    if ((!finite(max_dt) || !finite(-max_dt)) &&
	        (front->interf->num_points>0))
		(void) printf("WARNING in find_time_step(), "
			      "max_front_time_step returns infinity\n");
	    else
#endif /* !defined(_AIX) */
	    {
	        max_dt *= CFL;
	        if (max_dt < newdt)
	        {
	    	    newdt = max_dt;
	    	    TimeStepSetBy = TIME_STEP_SET_BY_FRONT;
	    	    for (i = 0; i < dim; ++i)
	    	        coords[i] = fcrds[i];
	        }
	    }
	}
        /* This becomes annoying when the time step was */
        /* forced to be very small to fit the print interval */
        /***
	if ((grid->dt != 0.0) && (grid->step > 0))
	{
	    if (newdt > Max_time_step_modification_factor(front)*grid->dt)
	    {
	        newdt = Max_time_step_modification_factor(front)*grid->dt;
	        TimeStepSetBy = TIME_STEP_SET_BY_PREVIOUS;
	        for (i = 0; i < dim; ++i)
	            coords[i] = HUGE_VAL;
	    }
	}
        ***/

        if (debugging("limit_time_step"))
	{
	    if (newdt > dt_lim(grid))
	    {
	    	newdt = dt_lim(grid);
	    	TimeStepSetBy = TIME_STEP_SET_BY_USER_LIMIT;
	    	for (i = 0; i < dim; ++i)
	    	    coords[i] = HUGE_VAL;
	    }
	}

	newdt = comm_time_step(newdt,coords,dim,&TimeStepSetBy);

	if (!stop_run(grid,front,wave))
	{
	    float tmpdt;
	    tmpdt = nonphysics_timestep_reductions(grid,wave,front,prt,newdt);
	    if (tmpdt != newdt)
	    {
	    	newdt = tmpdt;
	    	TimeStepSetBy = TIME_STEP_SET_BY_PRINTING;
	    }
	}

	if (print_set_by == YES)
	{
	    (void) printf("\nTime step set by ");
	    switch (TimeStepSetBy)
	    {
	    case TIME_STEP_SET_BY_WAVE:
		print_general_vector("interior update at ",coords,dim,", ");
		print_general_vector("Maxsp(wave) = ",Maxsp(wave),dim,"\n");
		break;
	    case TIME_STEP_SET_BY_FRONT:
		print_general_vector("front update at ",coords,dim,", ");
		print_general_vector("Spfr(front) = ",Spfr(front),dim+1,"\n");
		break;
	    case TIME_STEP_SET_BY_PREVIOUS:
		(void) printf("%g times previous dt\n",
			Max_time_step_modification_factor(front));
		break;
	    case TIME_STEP_SET_BY_USER_LIMIT:
		(void) printf("user defined limit %g\n",dt_lim(grid));
		break;
	    case TIME_STEP_SET_BY_PRINTING:
		(void) printf("printing restriction\n");
		break;
	    case TIME_STEP_NOT_SET:
	    default:
		screen("ERROR in find_time_step(), time step not set\n");
		clean_up(ERROR);
	    }
	}
	return newdt;
}		/*end find_time_step*/

LOCAL const int NUM_LOOK_AHEAD = 3;/*TOLERANCE*/

LOCAL float check_for_ts_restriction(
	OUTPUT_DATA	*odata,
	float		time,
	float		dt)
{
	int		 i;

	if ((odata != NULL) && exact_time_output(Output_mode(odata)))
	{
	    for (i = 1; i <= NUM_LOOK_AHEAD; ++i)
	    {
	        if (is_print_time(time+dt*i,Print_control(odata)) == YES)
		{
	            return (Output_next_print_time(odata) - time) / i;
	        }
	    }
	}
	return dt; /* no restriction found */
}		/*end check_for_ts_restriction*/


/*
*			nonphysics_timestep_reductions():
*
*	Reduces the timestep based on non-physical considerations, allowing
*	the computation to reach (exactly) a particular real time.  The
*	restrictions currently include global print times, pause times, stop
*	times, and user print times.
*
*	This routine looks ahead NUM_LOOK_AHEAD steps to check for target
*	times.  This allows the step sizes up to the target to be adjusted
*	to be approximately equal, avoiding very tiny time steps.
*/

/*ARGSUSED*/
EXPORT float nonphysics_timestep_reductions(
	Grid		*grid,
	Wave		*wave,
	Front		*front,
	Printplot	*prt,
	float		dt)
{
	float		newdt = dt;
	int		i;

	/* If newdt takes us past next printing time, then reduce. */
	for (i = 0; i < NUM_OUTPUT_FORMATS; ++i)
	{
	    newdt = check_for_ts_restriction(prt->main_output_data[i],
					     grid->time,newdt);
	}

	/* If newdt takes us past pause time, then reduce (maybe again). */
	if (exact_time_output(pause_mode(grid)))
	{
	    for (i = 1; i <= NUM_LOOK_AHEAD; ++i)
	    {
		if ((grid->time + newdt*i) >= pause_time(grid))
		{
		    newdt = (pause_time(grid) - grid->time)/i;
		    break;
	        }
	    }
	}

	if (exact_time_output(stop_time_mode(grid)))
	{
	    /* If newdt takes us past stop time, then reduce (maybe again). */
	    for (i = 1; i <= NUM_LOOK_AHEAD; ++i)
	    {
	        if ((grid->time + newdt*i) >= stop_time(grid))
	        {
	    	    newdt = (stop_time(grid) - grid->time)/i;
	    	    break;
	        }
            }
	}

	/* Check user printing routines also. */
	if (prt->user_output_funcs != NULL)
	{
	    int		i;
	    void	(**f)(Grid*,Wave*,Front*,Printplot*,
			      OUTPUT_DATA*,bool);

	    for (i = 0, f = prt->user_output_funcs; *f != NULL; ++f, ++i)
	    {
	        newdt = check_for_ts_restriction(prt->user_output_data[i],
						 grid->time,newdt);
	    }
        }
	if (grid->user_dt_control)
	    (*grid->user_dt_control)(grid,&newdt);

	return newdt;
}		/*end nonphysics_timestep_reductions*/


/*
*			is_print_time():
*
*	Returns YES if time >= next printing time and NO otherwise.
*/

EXPORT	bool is_print_time(
	float		time,
	PRINT_CONTROL	*prtctrl)
{
	PRINT_OPTIONS	*pto = &print_options(prtctrl);
	float npt;

	if (!real_time_output(prt_mode(pto)))
	    return NO;
	if (time < (print_start_time(pto) - MACH_EPS)) /*TOLERANCE*/
	    return NO;

	npt = next_print_time(prtctrl);
	npt *= (npt < 0) ?	(1.0 + print_time_tolerance(prtctrl)) :
				(1.0 - print_time_tolerance(prtctrl));
	return (time >= npt) ? YES : NO;
}		/*end is_print_time*/


/*
*			is_print_step():
*
*	Returns YES if step == next printing step and NO otherwise.
*/

EXPORT bool is_print_step(
	int		step,
	PRINT_CONTROL	*prtctrl)
{
	PRINT_OPTIONS	*pto = &print_options(prtctrl);
	if (mesh_time_output(prt_mode(pto)) &&
	    (step >= print_start_step(pto)) &&
	    (step == next_print_step(prtctrl)))
	    return YES;
	else
	    return NO;
}		/*end is_print_step*/


/*
*				d_await():
*
*	Handles pauses in execution due to exceeding specified pause time.
*
*	Provides indefinite delay before resuming execution; if desired
*	the program may be killed.  It is possible to modify various paramaters
*	governing debugging and printout.
*/

/*ARGSUSED*/
EXPORT void d_await(
	Grid		*grid,
	Front		*front,
	Wave		*wave,
	Printplot	*prt,
	bool		got_intfc_from_file)
{
	IMPORT bool  suppress_prompts;
	bool		sav_suppress_prompts;
	INIT_DATA	*init = grid->init;
	char		s[Gets_BUF_SIZE];

	sav_suppress_prompts = suppress_prompts;
	suppress_prompts = (interactive_prompting(init) == YES) ? NO : YES;

	screen("\n\nRequest continuation or termination of run "
	       "(cont(dflt),term): ");
	(void) Gets(s);
	if ((s[0] == 't') || (s[0] == 'T'))
	{
	    suppress_prompts = sav_suppress_prompts;
	    clean_up(0);
	}

	screen("\nCurrent Time = %g, current step = %d.\n",
	       grid->time,grid->step);
	screen("Specify the new pause time mode, "
	       "exact%s, constant%s, or mesh%s: ",
	       (pause_mode(grid) == EXACT_TIME) ? " (dflt)" : "",
	       (pause_mode(grid) == CONSTANT_TIME) ? " (dflt)" : "",
	       (pause_mode(grid) == MESH_TIME) ? " (dflt)" : "");
	(void) Gets(s);

	if (s[0] == 'm' || s[0] == 'M')
	    pause_mode(grid) = MESH_TIME;
	else if (s[0] == 'c' || s[0] == 'M')
	    pause_mode(grid) = CONSTANT_TIME;
	else if (s[0] == 'e' || s[0] == 'E')
	    pause_mode(grid) = EXACT_TIME;

	if (mesh_time_output(pause_mode(grid)))
	{
	    pause_step(grid) = INT_MAX;
	    screen("Enter the new Pause Time Step (dflt = %d): ",
		   pause_step(grid));
	    (void) Gets(s);
	    if (s[0] != '\0')
	    	(void) sscanf(s,"%d \n",&pause_step(grid));
	}
	else
	{
	    pause_time(grid) = HUGE_VAL;
	    screen("Enter the new Pause Time (dflt = %g): ",
		   pause_time(grid));
	    (void) Gets(s);
	    if (s[0] != '\0')
	    	(void) sscan_float(s,&pause_time(grid));
	}

	screen("Type 'm' to modify the debugging: ");
	(void) Gets(s);
	if ((s[0] == 'm') || (s[0] == 'M'))
	{
	    screen("Type 'init', 'debug', or 'nodebug' to initialize, "
	           "continue, or terminate\n\tthe debug mode: ");
	    (void) Gets(s);
	    if      ((s[0] == 'i') || (s[0] == 'I'))
	    	dbparams(prt->init) = init_debug(PROMPT_FOR_DEBUG);
	    else if ((s[0] == 'd') || (s[0] == 'D'))
	    	dbparams(prt->init) = init_debug(SOME);
	    else if ((s[0] == 'n') || (s[0] == 'N'))
	    	dbparams(prt->init) = init_debug(NONE);
	}

	if (debugging("vm_1") || debugging("vmalloc_1"))
	    vmalloc_debug(1);
	if (debugging("vm_2") || debugging("vmalloc_2"))
	    vmalloc_debug(2);
	if (debugging("vm_3") || debugging("vmalloc_3"))
	    vmalloc_debug(3);
	if (debugging("vm_4") || debugging("vmalloc_4"))
	    vmalloc_debug(4);
	if (debugging("vm_5") || debugging("vmalloc_5"))
	    vmalloc_debug(5);

	screen("Type 'm' to modify the printout: ");
	(void) Gets(s);
	if ((s[0] == 'm') || (s[0] == 'M'))
	{
	    init_printplot(prt->init,grid,front,prt);
	    grid->dt = nonphysics_timestep_reductions(grid,wave,
						      front,prt,grid->dt);
	}
	if (Init_redistribution_function(front) != NULL)
	{
	    screen("Type 'm' to modify the redistribution parameters: ");
	    (void) Gets(s);
	    if ((s[0] == 'm') || (s[0] == 'M'))
	    {
	        INTERFACE	*sav_intfc = i_intfc(init);

	        i_intfc(init) = front->interf;
		copy_redistribution_values(init,front);
	        prompt_for_redistribute(init);
	        Init_redistribution(init,front);
	        i_intfc(init) = sav_intfc;
	    }
	}
	suppress_prompts = sav_suppress_prompts;
}		/*end d_await*/

/*
*			give_peep():
*
*	Provides a peep at the solutions.
*/

/* ARGSUSED */
EXPORT void give_peep(
	Grid		*grid,
	Wave		*wave,
	Front		*front,
	Printplot	*prt,
	OUTPUT_DATA	*odata,
	bool		about_to_stop)
{
	(void) fprintf(stderr,"\n\n\nCurrent Time: %.4g   Time Step: %d\n",
				grid->time,grid->step);
	show_COMP(Output_file(odata),front->interf);
}		/*end give_peep*/


EXPORT	void d_print_stopping_criteria(Grid *grid)
{
	(void) printf("\t\t\tStopping Criteria\n");
	(void) printf("\tstop_time %g\tstop_step %d",
		      stop_time(grid),stop_step(grid));
}		/*end d_print_stopping_criteria*/

EXPORT	void	d_print_D_Front_structure(Front* fr)
{
	(void) printf("\n\n\n\t\tD_Front %p structure\n",(POINTER)fr);
	h_print_H_Front_structure(fr);
	(void) printf("amr info: chart_of_front(fr) = %p\n",
		      (POINTER)chart_of_front(fr));
	(void) printf("\n\n\n\t\tEnd D_Front %p structure\n",(POINTER)fr);
}		/*end d_print_D_Front_structure*/

LOCAL float comm_time_step(
	float			dt,
	float			*coords,
	int			dim,
	TIME_STEP_SET_BY	*TimeStepSetBy)
{
	int		numnodes = pp_numnodes();
	int		i, imin;
	static struct	_COMM_TIME_STEP {
					  float	           dt;
		                          float	           coords[MAXD];
		                          TIME_STEP_SET_BY TimeStepSetBy;
	                                } Cts;
	static	struct   _COMM_TIME_STEP *cts = NULL;

	if (numnodes == 1)
	    return dt;

	debug_print("time_step","Entered comm_time_step()\n");

	if (cts == NULL)
	    vector(&cts,numnodes,sizeof(struct _COMM_TIME_STEP));

	Cts.dt = dt;
	Cts.TimeStepSetBy = *TimeStepSetBy;
	for (i = 0; i < dim; ++i)
	    Cts.coords[i] = coords[i];

	pp_all_gather((POINTER)&Cts,sizeof(struct _COMM_TIME_STEP),
		      (POINTER)cts,sizeof(struct _COMM_TIME_STEP));

	if (debugging("time_step"))
	{
	    char buf[20];
	    for (i = 0; i < numnodes; ++i)
	    {
		(void) printf("cts[%d].dt = %g\n",i,cts[i].dt);
		(void) sprintf(buf,"cts[%d].coords = ",i);
		print_general_vector(buf,cts[i].coords,dim,"\n");
		(void) printf("cts[%d].TimeStepSetBy = %s\n",i,
			      TimeStepSetByString(cts[i].TimeStepSetBy));
	    }
	}

	for (imin = 0, i = 1; i < numnodes; ++i)
	    if (cts[imin].dt > cts[i].dt)
		imin = i;

	dt = cts[imin].dt;
	*TimeStepSetBy = cts[imin].TimeStepSetBy;
	for (i = 0; i < dim; ++i)
	    coords[i] = cts[imin].coords[i];

	debug_print("time_step","Left comm_time_step()\n");
	return dt;
}		/*end comm_time_step*/

LOCAL	const char *TimeStepSetByString(
	TIME_STEP_SET_BY TimeStepSetBy)
{
	switch (TimeStepSetBy)
	{
	case TIME_STEP_NOT_SET:
	    return "TIME_STEP_NOT_SET";
	case TIME_STEP_SET_BY_FRONT:
	    return "TIME_STEP_SET_BY_FRONT";
	case TIME_STEP_SET_BY_WAVE:
	    return "TIME_STEP_SET_BY_WAVE";
	case TIME_STEP_SET_BY_PREVIOUS:
	    return "TIME_STEP_SET_BY_PREVIOUS";
	case TIME_STEP_SET_BY_USER_LIMIT:
	    return "TIME_STEP_SET_BY_USER_LIMIT";
	case TIME_STEP_SET_BY_PRINTING:
	    return "TIME_STEP_SET_BY_PRINTING";
	default:
	    return NULL;
	}
}		/*end TimeStepSetByString*/

#if defined(USE_OVERTURE)

EXPORT float find_amr_time_step(
        CHART        **rts,
        int          num_patches,
        Printplot    *prt,
        bool         print_set_by)
{
        int          i;
        float        dt = HUGE_VAL, tmpdt;
        float        coords[MAXD], crds[MAXD];
        float        newdt;
        int          dim = rts[0]->front->rect_grid->dim;

        TIME_STEP_SET_BY TimeStepSetBy, tmpTimeStepSetBy;

        TimeStepSetBy = TIME_STEP_NOT_SET;

        for(i = 0; i < num_patches; i++)
        {
            if(rts[i]->wave->patch_level ==
                 rts[0]->overparam->numberOfRefinementLevels-1 ||
                 rts[i]->wave->patch_level == 0)
            {
                tmpdt = find_single_proc_time_step(rts[i]->grid,
                      rts[i]->wave,rts[i]->front,crds, &tmpTimeStepSetBy);
                if(tmpdt < dt)
                {
                    dt = tmpdt;
                    assign(coords,crds,sizeof(float)*MAXD);
                    TimeStepSetBy = tmpTimeStepSetBy;
                }
            }
        }

        newdt = comm_time_step(dt,coords,dim,&TimeStepSetBy);

        /* lack nonphysics_timestep_reductions() here */

        if (print_set_by == YES)
        {
            (void) printf("\nTime step set by ");
            switch (TimeStepSetBy)
            {
            case TIME_STEP_SET_BY_WAVE:
                print_general_vector("interior update at ",coords,dim,", ");
                print_general_vector("Maxsp(wave) = ",Maxsp(rts[0]->wave),dim,"\n");
                break;
            case TIME_STEP_SET_BY_FRONT:
                print_general_vector("front update at ",coords,dim,", ");
                print_general_vector("Spfr(front) = ",Spfr(rts[0]->front),dim+1,"\n");
                break;
            case TIME_STEP_SET_BY_PREVIOUS:
                (void) printf("%g times previous dt\n",
                        Max_time_step_modification_factor(rts[0]->front));
                break;
            case TIME_STEP_SET_BY_USER_LIMIT:
                (void) printf("user defined limit %g\n",dt_lim(rts[0]->grid));
                break;
            case TIME_STEP_SET_BY_PRINTING:
                (void) printf("printing restriction\n");
                break;
            case TIME_STEP_NOT_SET:
            default:
                screen("ERROR in find_time_step(), time step not set\n");
                clean_up(ERROR);
            }
        }
        return newdt;
}

LOCAL float find_single_proc_time_step(
        Grid             *grid,
        Wave             *wave,
        Front            *front,
        float            *coords,
        TIME_STEP_SET_BY *TimeStepSetBy)
{
        float            max_dt, CFL;
        float            newdt;
        float            fcrds[MAXD], wcrds[MAXD];
        int              i, dim = front->rect_grid->dim;

        CFL = Time_step_factor(front);
        if (dim > 1)
            CFL *= min(Front_spacing(front,GENERAL_WAVE),
                       Front_spacing(front,VECTOR_WAVE));

        newdt = HUGE_VAL;
                /* Maximum Interior Wave Speeds (previous time) */

        if (wave->max_hyp_time_step)
        {
            max_dt = (*wave->max_hyp_time_step)(wave,wcrds);
#if !defined(_AIX)
            if (!finite(max_dt) || !finite(-max_dt))
            {
#if defined(USE_OVERTURE)
                (void) printf("WARNING in find_time_step(), "
                        "max_hyp_time_step returns infinity\n");
                (void) printf("From front[%d], level[%d]\n",
                        front->patch_number, front->patch_level);
#else  /* if defined(USE_OVERTURE) */
                (void) printf("WARNING in find_time_step(), "
                        "max_hyp_time_step returns infinity\n");
#endif /* if defined(USE_OVERTURE) */
            }
            else
#endif /* !defined(_AIX) */
            {
                max_dt *= CFL;
                if (max_dt < newdt)
                {
                    newdt = max_dt;
                    *TimeStepSetBy = TIME_STEP_SET_BY_WAVE;
                    for (i = 0; i < dim; ++i)
                        coords[i] = wcrds[i];
                }
            }
        }

            /* Maximum Front Speeds (previous time) */

        if (front->max_front_time_step)
        {
            max_dt = (*front->max_front_time_step)(front,fcrds);
#if !defined(_AIX)
            if ((!finite(max_dt) || !finite(-max_dt)) &&
                (front->interf->num_points>0))
            {
#if defined(USE_OVERTURE)
                if(front->rect_grid->dim == 2)
                {
                    CURVE **c;
                    int   interior = NO;
                    for(c = front->interf->curves; c && *c; c++)
                    {
                        if(wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE)
                        {
                            interior = YES; break;
                        }
                    }
                    if(YES == interior)
                    {
                        (void) printf("WARNING in find_time_step(), "
                              "max_front_time_step returns infinity\n");
                        (void) printf("From front[%d], level[%d]\n",
                               front->patch_number, front->patch_level);
                    }
                }
#else  /* if defined(USE_OVERTURE) */
                (void) printf("WARNING in find_time_step(), "
                              "max_front_time_step returns infinity\n");
#endif /* if defined(USE_OVERTURE) */
            }
            else
#endif /* !defined(_AIX) */
            {
                max_dt *= CFL;
                if (max_dt < newdt)
                {
                    newdt = max_dt;
                    *TimeStepSetBy = TIME_STEP_SET_BY_FRONT;
                    for (i = 0; i < dim; ++i)
                        coords[i] = fcrds[i];
                }
            }
        }
       
        if ((grid->dt != 0.0) && (grid->step > 0))
        {
            if (newdt > Max_time_step_modification_factor(front)*grid->dt)
            {
                newdt = Max_time_step_modification_factor(front)*grid->dt;
                *TimeStepSetBy = TIME_STEP_SET_BY_PREVIOUS;
                for (i = 0; i < dim; ++i)
                    coords[i] = HUGE_VAL;
            }
        }

        if (debugging("limit_time_step"))
        {
            if (newdt > dt_lim(grid))
            {
                newdt = dt_lim(grid);
                *TimeStepSetBy = TIME_STEP_SET_BY_USER_LIMIT;
                for (i = 0; i < dim; ++i)
                    coords[i] = HUGE_VAL;
            }
        }

        return newdt;
}
#endif /* defined(USE_OVERTURE) */

EXPORT	float find_time_step_on_tris(
	Front		*front)
{
	float		 max_dt, CFL;
	float		 newdt, min_len = HUGE_VAL;
	float	         coords[MAXD] = {0.0, 0.0}, fcrds[MAXD], wcrds[MAXD];
	int	         i, dim = front->rect_grid->dim;
        TRI              *tri, *on_tri;
        SURFACE          **surf = front->mesh->surfaces;
        Locstate         st;
        // TMP
        float            peri = 0.0, area, max_speed;
	TIME_STEP_SET_BY TimeStepSetBy;

        // TMP
        CFL = Time_step_factor(front);

	newdt = HUGE_VAL;
        for(; surf && *surf; surf++)
        {
            for (tri = first_tri(*surf);
                    !at_end_of_tri_list(tri,*surf);
                    tri = tri->next)
            {
                if(tri->BC_type == SUBDOMAIN)
                    continue;
                max_dt = (*front->_time_step_on_tri)(front, tri);
                // TMP
                if(max_dt < newdt)
                    on_tri = tri;
                // END TMP
                newdt = min(newdt, max_dt);
            }
        }

	TimeStepSetBy = TIME_STEP_SET_BY_WAVE;
	newdt = comm_time_step(newdt,coords,dim,&TimeStepSetBy);

        // TMP
        /** 
        area = fg_area(on_tri);
        for(i = 0; i < 3; i++)
        {
            peri += fg_length_side(on_tri)[i];
        }

        // dt = CFL*area/(max_speed*peri);
        max_speed = CFL*area/(newdt*peri);
        printf("find_time_step_on_tris() Triangle(%d), max_speed = %g, area = %g, peri = %g\n",
                           on_tri->id, max_speed, area, peri);
        printf("newdt = %22.15g\n", newdt);
        print_general_vector("Tri_pt", Coords(Point_of_tri(on_tri)[0]), dim, "\n");
        print_general_vector("Tri_pt", Coords(Point_of_tri(on_tri)[1]), dim, "\n");
        print_general_vector("Tri_pt", Coords(Point_of_tri(on_tri)[2]), dim, "\n");
        (*front->print_state)(on_tri->st);
        **/
        // END TMP
	return newdt;
}		/*end find_time_step*/

