/*
*                               tri_csg_opt.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*/


#include <tri/trilocaldecs.h>

#if defined(TWOD) && defined(CONSERVATIVE_ALG)

/* Local Globals */
LOCAL TRI_GRID     *local_ntg,  *local_new_ntg, *local_old_ntg;
float              dt_step;
unsigned long      sno = 0, vno = 0, fno = 0;
                   /* # of solids, vertice, and faces */
unsigned nodesize[] =
{
        sizeof(CSG_Solid), sizeof(CSG_Face),
        sizeof(CSG_Loop), sizeof(CSG_HalfEdge),
        sizeof(CSG_Edge), sizeof(CSG_Vertex), 0
};

/* End of Local Globals */  

LOCAL CSG_Node     *newCSG_Node(int,CSG_Node*);
LOCAL void         CSG_addlist(int,CSG_Node*,CSG_Node*);
LOCAL CSG_Solid    *mvfs(unsigned long,unsigned long,CSG_POINT*);
LOCAL int          mev(CSG_Solid*,unsigned long,unsigned long,unsigned long,CSG_POINT*);
LOCAL CSG_Face     *fface(CSG_Solid*,unsigned long);
LOCAL CSG_HalfEdge *fhe1(CSG_Face*,unsigned long);
LOCAL int          mef(CSG_Solid*,unsigned long,unsigned long,
                      unsigned long,unsigned long);
LOCAL CSG_Face     *lmef(CSG_HalfEdge*,CSG_HalfEdge*,unsigned long);
LOCAL CSG_HalfEdge *addhe(CSG_Edge*,CSG_Vertex*,CSG_HalfEdge*,int);
LOCAL void         lmev(CSG_HalfEdge*,CSG_HalfEdge*,unsigned long,CSG_POINT*);
LOCAL void         lmekr(CSG_HalfEdge*,CSG_HalfEdge*);
LOCAL void         merge(CSG_Solid*,CSG_Solid**);
LOCAL void         lkfmrh(CSG_Face*,CSG_Face*);  
LOCAL void         loopglue(CSG_Face*); 
LOCAL void         lkev(CSG_HalfEdge*,CSG_HalfEdge*);  


LOCAL CSG_HalfEdge  *comm_edge_of_faces(CSG_Face*,CSG_Face*);
LOCAL void         lkef(CSG_HalfEdge*,CSG_HalfEdge*);  
LOCAL void         CSG_del(int,CSG_Node*,CSG_Node*); 
LOCAL void         CSG_dellist(int,CSG_Node*,CSG_Node*);
LOCAL CSG_HalfEdge *delhe(CSG_HalfEdge*);  

LOCAL void         glue(CSG_Solid*,CSG_Solid**,CSG_Face*,CSG_Face*); 
LOCAL int          match(CSG_HalfEdge*,CSG_HalfEdge*);  

// LOCAL float        triangle_area_3d(float*,float*,float*);
// LOCAL float        enorm0_3d(float*,float*);
LOCAL int          point_in_polygon_2d(float**,int,float*);

LOCAL float        findArea2D(int,float*,float*);


LIB_LOCAL void set_CSG_local_global(
          TRI_GRID      *ntg,
          TRI_GRID      *old_ntg,
          float         dt)
{
        /* set local global */

        local_ntg = local_new_ntg = ntg;
        local_old_ntg = old_ntg;
        dt_step = dt;

        /* Index for VOL */
        sno = 0, vno = 0, fno = 0;

}

LOCAL  CSG_Node *newCSG_Node(
        int            what,
        CSG_Node       *where)
{
        CSG_Node       *node;

        scalar(&node, nodesize[what]);

        switch(what)
        {
            case  CSG_SOLID:
                CSG_addlist(CSG_SOLID, node, NULL);
                node->s.sfaces = NULL;
                node->s.sedges = NULL;
                node->s.sverts = NULL;
            break;
            case  CSG_FACE:
                CSG_addlist(CSG_FACE, node, where);
                node->f.floops = NULL;
                node->f.flout = NULL;
                node->f.i_stat = NULL;
                // node->f.o_stat = NULL;
                node->f.smid = NULL;
            break;
            case  CSG_LOOP:
                CSG_addlist(CSG_LOOP, node, where);
            break;
            case  CSG_EDGE:
                CSG_addlist(CSG_EDGE, node, where);
            break;
            case  CSG_VERTEX:
                CSG_addlist(CSG_VERTEX, node, where);
                node->v.vedge = NULL;
            break;
            default:
            break;
        }
        return node;
}

/* adds the node pointed to by "which" into the linked list
   of nodes owned by "where".
   Solid is a special case, solids are simply linked to the list
   of all solids. In this case, "where" has no effect.
   "Which" is given in terms of SOLID, FACE, LOOP, EDGE or VERTEX.
*/

LOCAL void CSG_addlist(
        int             what,
        CSG_Node        *which,
        CSG_Node        *where)
{
        switch(what)
        {
            case  CSG_SOLID:
                /*
                which->s.nexts = firsts;
                which->s.prevs = NULL;
                if(firsts)
                    firsts->prevs = (CSG_Solid *)which;
                firsts = (CSG_Solid *)which;
                */
                which->s.prevs = NULL;
                which->s.nexts = NULL;
            break;
            case  CSG_FACE:
                which->f.nextf = where->s.sfaces;
                which->f.prevf = NULL;
                if(where->s.sfaces)
                    where->s.sfaces->prevf = (CSG_Face *)which;
                where->s.sfaces = (CSG_Face *)which;
                which->f.fsolid = (CSG_Solid *)where;
            break;
            case  CSG_LOOP:
                which->l.nextl = where->f.floops;
                which->l.prevl = NULL;
                if(where->f.floops)
                    where->f.floops->prevl = (CSG_Loop *)which;
                where->f.floops = (CSG_Loop *)which;
                which->l.lface = (CSG_Face *)where;
            break;
            case  CSG_EDGE:
                which->e.nexte = where->s.sedges;
                which->e.preve = NULL;
                if(where->s.sedges)
                    where->s.sedges->preve = (CSG_Edge *)which;
                where->s.sedges = (CSG_Edge *)which;
            break;
            case CSG_VERTEX:
                which->v.nextv = where->s.sverts;
                which->v.prevv = NULL;
                if(where->s.sverts)
                    where->s.sverts->prevv = (CSG_Vertex *)which;
                where->s.sverts = (CSG_Vertex *)which;
            break;
        }
}

LOCAL CSG_Face *fface(
         CSG_Solid          *s,
         unsigned long      fn)
{
        CSG_Face            *f;

        f = s->sfaces;
        while(f)
        {
            if(f->faceno == fn)
                return(f);
            f = f->nextf;
        }
        return((CSG_Face *) NULL);
}


LOCAL   CSG_Solid  *mvfs(
        unsigned long    fn,   /* face number  */
        unsigned long    vn,   /* vertex number  */
        CSG_POINT        *v1)
{
        CSG_Solid           *newsolid;
        CSG_Face            *newface;
        CSG_Vertex          *newvertex;
        CSG_HalfEdge        *newhe;
        CSG_Loop            *newloop;

        newsolid = (CSG_Solid *) newCSG_Node(CSG_SOLID, NULL);
        newface = (CSG_Face *) newCSG_Node(CSG_FACE, (CSG_Node*)newsolid);
        newloop = (CSG_Loop *) newCSG_Node(CSG_LOOP, (CSG_Node*)newface);
        newvertex = (CSG_Vertex *) newCSG_Node(CSG_VERTEX, (CSG_Node*)newsolid);
        newhe = (CSG_HalfEdge *) newCSG_Node(CSG_HALFEDGE, NULL);

        newface->faceno = fn;
        newface->flout = newloop;
        newloop->ledg = newhe;
        newhe->wloop = newloop;
        newhe->nxt = newhe->prv = newhe;
        newhe->vtx = newvertex;
        newhe->edg = NULL;
        newvertex->vertexno = vn;
        newvertex->pt = v1;

        newsolid->unset = YES; 
        return(newsolid);
}


LOCAL int mev(
         CSG_Solid          *s,
         unsigned long      fn,   /* face number  */
         unsigned long      vn1,  /* vertex number  */
         unsigned long      vn2,  /* vertex number  */
         CSG_POINT          *v2)
{
        CSG_Face            *oldface;
        CSG_HalfEdge        *he;

        if((oldface = fface(s, fn)) == NULL)
        {
             printf("ERROR: mev(): face %d not found\n", fn);
             return(ERROR);
        }
        if((he = fhe1(oldface, vn1)) == NULL)
        {
             printf("ERROR: mev(): vertex %d not found in face %d\n",
                          vn1, fn);
             return(ERROR);
        }

        lmev(he, he, vn2, v2);

        return(FUNCTION_SUCCEEDED);
}

LOCAL   CSG_HalfEdge *fhe1(
        CSG_Face           *f,
        unsigned long      vn)
{
        CSG_Loop           *l;
        CSG_HalfEdge       *h;

        l = f->floops;
        while(l)
        {
            h = l->ledg;
            do
            {
                if(h->vtx->vertexno == vn)
                    return(h);
            }
            while((h = h->nxt) != l->ledg);
            l = l->nextl;
        }
        return((CSG_HalfEdge *) NULL);
}

LOCAL int mef(
         CSG_Solid          *s,
         unsigned long      vert1,
         unsigned long      vert2,
         unsigned long      fac1,
         unsigned long      fac2)
{
        CSG_Face            *oldface, *newface;
        register CSG_HalfEdge    *he1, *he2;

        if(!(oldface = fface(s, fac1)))
        {
            printf("ERROR mef(): face %d not found\n", fac1);
            return(ERROR);
        }

    /* get vert1 */
        if(!(he1 = fhe1(oldface, vert1)))
        {
            printf("ERROR: mef(): vertex %d not found\n", vert1);
            return(ERROR);
        }


    /* get the next occurance of vert2 */
        he2 = he1;
        if(he2->vtx->vertexno != vert2)
        do
        {
            he2 = he2->nxt;
            if(he2 == he1)
            {
                printf("ERROR(): mef(): vertex %d not found\n", vert2);
                return(ERROR);
            }
        }
        while(he2->vtx->vertexno != vert2);

        newface = lmef(he1, he2, fac2);
        return(FUNCTION_SUCCEEDED);
}


LOCAL CSG_Face *lmef(
         CSG_HalfEdge       *he1,
         CSG_HalfEdge       *he2,
         unsigned long      nf)
{
        CSG_Face            *newface;
        CSG_Loop            *newloop;
        CSG_Edge            *newedge;
        CSG_HalfEdge        *he, *nhe1, *nhe2, *temp;

        newface = (CSG_Face *)newCSG_Node(CSG_FACE,
                            (CSG_Node *)(he1->wloop->lface->fsolid));
        newloop = (CSG_Loop *)newCSG_Node(CSG_LOOP, (CSG_Node *)newface);
        newedge = (CSG_Edge *)newCSG_Node(CSG_EDGE,
                            (CSG_Node *)(he1->wloop->lface->fsolid));
        newface->faceno = nf;
        newface->flout = newloop;

        he = he1;
        while(he != he2)
        {
            he->wloop = newloop;
            he = he->nxt;
        }

        nhe1 = addhe(newedge, he2->vtx, he1, MINUS);
        nhe2 = addhe(newedge, he1->vtx, he2, PLUS);

        nhe1->prv->nxt = nhe2;
        nhe2->prv->nxt = nhe1;
        temp = nhe1->prv;
        nhe1->prv = nhe2->prv;
        nhe2->prv = temp;

        newloop->ledg = nhe1;
        he2->wloop->ledg = nhe2;

        return(newface);
}


LOCAL   CSG_HalfEdge  *addhe(
        CSG_Edge            *e,
        CSG_Vertex          *v,
        CSG_HalfEdge        *where,
        int                 sign)
{
        CSG_HalfEdge        *he;

        if(where->edg == NULL)
        {
            he = where;
        }
        else
        {
            he = (CSG_HalfEdge *) newCSG_Node(CSG_HALFEDGE, NULL);
            where->prv->nxt = he;
            he->prv = where->prv;
            where->prv = he;
            he->nxt = where;
        }

        he->edg = e;
        he->vtx = v;
        he->wloop = where->wloop;

        if(sign == PLUS)
            e->he1 = he;
        else
            e->he2 = he;
        return(he);
}

LOCAL void lmev(
         CSG_HalfEdge       *he1,
         CSG_HalfEdge       *he2,
         unsigned long      vn,
         CSG_POINT          *v)
{
        CSG_HalfEdge        *he;
        CSG_Vertex          *newvertex;
        CSG_Edge            *newedge;
        CSG_Solid           *s;

#ifdef TEST
s = he1->wloop->lface->fsolid;
#endif

        newedge = (CSG_Edge *) newCSG_Node(CSG_EDGE,
                               (CSG_Node *)he1->wloop->lface->fsolid);
        newvertex = (CSG_Vertex *) newCSG_Node(CSG_VERTEX,
                               (CSG_Node *)he1->wloop->lface->fsolid);
        newvertex->pt = v;
        newvertex->vertexno = vn;
        he = he1;
        while(he != he2)
        {
            he->vtx = newvertex;
            he = mate(he)->nxt;
        }

        addhe(newedge, he2->vtx, he1, MINUS);
        addhe(newedge, newvertex, he2, PLUS);

        newvertex->vedge = he2->prv;
        he2->vtx->vedge = he2;

}

LOCAL void lmekr(
         CSG_HalfEdge       *he1,
         CSG_HalfEdge       *he2)
{
         CSG_Edge           *n_ed;
         register CSG_HalfEdge  *n_he1, *n_he2;
         CSG_Face           *f;
         CSG_Loop           *l1, *l2;

         l1 = he1->wloop;
         l2 = he2->wloop;
         f = l1->lface;

         n_he1 = l2->ledg;
         do
         {
             n_he1->wloop = l1;
         }while((n_he1 = n_he1->nxt) != l2->ledg);

         n_ed = (CSG_Edge *) newCSG_Node(CSG_EDGE,
                (CSG_Node *)f->fsolid);
         n_he1 = addhe(n_ed, he1->vtx, he1, PLUS);
         n_he2 = addhe(n_ed, he2->vtx, he2, MINUS);


         n_he1->nxt = he2;
         n_he2->nxt = he1;
         he2->prv = n_he1;
         he1->prv = n_he2;

         if(f->flout == l2) f->flout = l1;
         CSG_del(CSG_LOOP, (CSG_Node*)l2, (CSG_Node*)l2->lface);
}

LOCAL void lkev(
        CSG_HalfEdge        *he1,
        CSG_HalfEdge        *he2)
{
        CSG_HalfEdge     *he;
        CSG_HalfEdge     *rhe1, *rhe2;
        CSG_Loop         *l1, *l2;
        CSG_Vertex       *v1, *v2;
        CSG_Edge         *e1;

        he = he2->nxt;
        while(he != he1)
        {
           he->vtx = he2->vtx;
           he = mate(he)->nxt;
        }

        l1 = he1->wloop;
        l2 = he2->wloop;
        v1 = he1->vtx;
        v2 = he2->vtx;
        rhe2 = he2->nxt;
        e1 = he1->edg;
        l1->ledg = delhe(he1);
        l2->ledg = delhe(he2);
        v2->vedge = rhe2;

        if(!v2->vedge->edg)
           v2->vedge = (CSG_HalfEdge *)NULL;
        CSG_del(CSG_EDGE, (CSG_Node *)e1, (CSG_Node *)l1->lface->fsolid);
        CSG_del(CSG_VERTEX, (CSG_Node *)v1, (CSG_Node *)l1->lface->fsolid);
}

LOCAL void merge(
        CSG_Solid       *s1,
        CSG_Solid       **s2)
{
        CSG_Face        *f;
        CSG_Edge        *e;
        CSG_Vertex      *v;

        while((f = (*s2)->sfaces))
        {
            CSG_dellist(CSG_FACE, (CSG_Node*)f, (CSG_Node*)(*s2));
            CSG_addlist(CSG_FACE, (CSG_Node*)f, (CSG_Node*)s1);
        }
        while((e = (*s2)->sedges))
        {
            CSG_dellist(CSG_EDGE, (CSG_Node*)e, (CSG_Node*)(*s2));
            CSG_addlist(CSG_EDGE, (CSG_Node*)e, (CSG_Node*)s1);
        }
        while((v = (*s2)->sverts))
        {
            CSG_dellist(CSG_VERTEX, (CSG_Node*)v, (CSG_Node*)(*s2));
            CSG_addlist(CSG_VERTEX, (CSG_Node*)v, (CSG_Node*)s1);
        }
        CSG_del(CSG_SOLID, (CSG_Node*)(*s2), NULL);
        *s2 = NULL;
}


LOCAL void lkfmrh(
         CSG_Face           *f1,
         CSG_Face           *f2)
{
         CSG_Loop           *l;
         while(l = f2->floops)
         {
             CSG_dellist(CSG_LOOP, (CSG_Node*)l, (CSG_Node*)f2);
             CSG_addlist(CSG_LOOP, (CSG_Node*)l, (CSG_Node*)f1);
         }
         CSG_del(CSG_FACE, (CSG_Node*)f2, (CSG_Node*)f1->fsolid);
}


LOCAL void loopglue(
         CSG_Face           *fac)
{
         CSG_HalfEdge       *h1, *h2, *h1next;
         h1 = fac->floops->ledg;
         h2 = fac->floops->nextl->ledg;
         while(!match(h1, h2)) h2 = h2->nxt;
         lmekr(h1, h2);

         lkev(h1->prv, h2->prv);

         while(h1->nxt != h2)
         {
             h1next = h1->nxt;
             /************************/
             /* lmef() is problematic here */
             lmef(h1->nxt, h1->prv, -1);
             lkev(h1->nxt, mate(h1->nxt));
             lkef(mate(h1), h1);
             h1 = h1next;
         }
         lkef(mate(h1), h1);
}


LIB_LOCAL CSG_Solid *create_corner_vol2(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if(mev(s, fno, vno, vno+1, v2) != ERROR)
          ++vno;
        else
        {
            printf("ERROR in create_corner_vol2 mev\n");
            clean_up(ERROR);  
        }
        if(mev(s, fno, vno, vno+1, v3) != ERROR)
          ++vno;
        else
        {
            printf("ERROR in create_corner_vol2 mev\n");
            clean_up(ERROR);  
        }

        /* make the first face */
        if( mef(s, vno-2, vno, fno, 1+fno) != ERROR )
            ++fno;
        else
        {
            printf("ERROR in create_corner_vol2 mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;
        lmev(scan, scan, ++vno, v4);    /* 4th vertex  */
        lmef(scan->prv, scan->nxt->nxt, ++fno);
        lmef(scan->prv, scan->nxt, ++fno);
        /*
        solidls(s, 2);
        */
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *create_complement_vol_for_corner_vol2(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v2) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_complement_vol_for_corner_vol2 mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v3) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_complement_vol_for_corner_vol2 mev\n");
            clean_up(ERROR);  
        }

        /* make the first face */
        if( mef(s, vno-2, vno, fno, 1+fno) != ERROR)   ++fno;
        else
        {
            printf("ERROR create_complement_vol_for_corner_vol2 mef\n");
            clean_up(ERROR);  
        }
        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;


        lmev(scan, scan, ++vno, v4);    /* 4th vertex  */
        lmev(scan->nxt, scan->nxt, ++vno, v8);
        lmev(first->nxt->nxt, first->nxt->nxt, ++vno, v5);
        /* Face3: first side face  V(5, 4, 3, 2, 8) */
        lmef(scan->nxt->nxt, scan->prv->prv,++fno);

        scan = first->prv;
        lmev(scan->nxt, scan->nxt, ++vno, v10);
        lmev(scan->prv, scan->prv, ++vno, v9);
        lmef(scan->prv->prv, first->prv, ++fno);  /* Face V(10, 9, 8, 2, 1) */

        scan = first->prv->prv;
        lmev(scan, scan, ++vno, v6);
        lmef(first->nxt->nxt->nxt, scan->prv, ++fno); /* Face V(6, 5, 8, 9) */

        lmev(scan->nxt, scan->nxt, ++vno, v7);
        lmef(scan->nxt->nxt, scan->prv, ++fno); /* Face V(7, 6, 9, 10) */
        lmef(first->nxt->nxt,  first->prv->prv,  ++fno); /* Face V(7, 10, 1, 3, 4) */

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *create_3rd_vol_for_2pt_isolate1(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10,
        CSG_POINT        *v11,
        CSG_POINT        *v12)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if(mev(s, fno, vno, vno+1, v3) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_3rd_vol_for_2pt_isolate1 mev\n");
            clean_up(ERROR);  
        }
        if(mev(s, fno, vno, vno+1, v2) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_3rd_vol_for_2pt_isolate1 mev\n");
            clean_up(ERROR);  
        }

        /* make the first face */
        if(mef(s, vno-2, vno, fno, 1+fno) != ERROR )  ++fno;
        else
        {
            printf("ERROR create_3rd_vol_for_2pt_isolate1 mef\n");
            clean_up(ERROR);  
        }
 
        fac = fface(s, fno-1);
        l = fac->floops;
        scan = l->ledg;

        lmev(scan->nxt, scan->nxt, ++vno, v10);
        lmev(scan->prv, scan->prv, ++vno, v7);
        lmev(scan->prv->prv, scan->prv->prv, ++vno, v8);
        /* face <7,8,10,2,3> */
        lmef(scan->nxt->nxt, scan->prv->prv->prv, ++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt, ++vno, v11);
        lmev(scan->prv->prv->prv,scan->prv->prv->prv, ++vno, v9);
        /* face <8,9,11,10> */
        lmef(scan->nxt->nxt->nxt,scan->prv->prv->prv->prv,++fno);

        lmev(scan->nxt->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt,++vno,v5);
        lmev(scan->prv->prv, scan->prv->prv, ++vno, v4);
        /* face <7,4,5,9,8> */
        lmef(scan->nxt->nxt->nxt->nxt->nxt,scan->prv->prv->prv,++fno);

        lmev(scan->nxt->nxt->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++vno,v6);
        /* face <4,6,5> */
        lmef(scan->nxt->nxt->nxt->nxt->nxt->nxt,scan->prv->prv->prv,++fno);

        lmev(scan->nxt->nxt->nxt->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt->nxt,++vno,v12);
        /* face <9,5,6,12,11> */
        lmef(scan->nxt->nxt->nxt,scan->prv->prv->prv->prv->prv,++fno);
        /* face <7,3,4> */
        lmef(scan->prv,scan->prv->prv->prv,++fno);
        /* face <3,1,6,4> */
        lmef(scan,scan->prv->prv->prv, ++fno);
        /* face <1,2,10,11,12> */
        lmef(scan,scan->nxt->nxt->nxt->nxt,++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            s->nexts = blk_crx->s;
            blk_crx->s->prevs = s;
            blk_crx->s = s;
        }
        return s; 
}


LIB_LOCAL CSG_Solid *create_3rd_vol_for_2pt_isolate2(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10,
        CSG_POINT        *v11,
        CSG_POINT        *v12)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if(mev(s, fno, vno, vno+1, v3) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_3rd_vol_for_2pt_isolate2, mev\n"); 
            clean_up(ERROR); 
        }
        if(mev(s, fno, vno, vno+1, v2) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_3rd_vol_for_2pt_isolate2, mev\n"); 
            clean_up(ERROR); 
        }

        /* make the first face */
        if(mef(s, vno-2, vno, fno, 1+fno) != ERROR)   ++fno;
        else
        {
            printf("ERROR create_3rd_vol_for_2pt_isolate2, mef\n"); 
            clean_up(ERROR); 
        }
        fac = fface(s, fno-1);
        l = fac->floops;
        scan = l->ledg;

        lmev(scan->nxt, scan->nxt, ++vno, v10);
        lmev(scan->prv, scan->prv, ++vno, v9);
        lmev(scan->prv->prv,scan->prv->prv, ++vno, v5);
        lmev(scan->nxt->nxt,scan->nxt->nxt, ++vno, v6);
        /* face <9,5,3> */
        lmef(scan->prv->prv->prv,scan->prv->prv->prv->prv->prv,++fno);
        /* face <6,1,2> */
        lmef(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);
        /* face <5,6,2,3,9> */
        lmef(scan->nxt->nxt->nxt,scan->prv->prv->prv,++fno);

        lmev(scan->prv->prv->prv,scan->prv->prv->prv,++vno,v4);
        /* face <5,4,6> */
        lmef(scan->prv->prv->prv->prv,scan->nxt->nxt->nxt,++fno);

        lmev(scan->prv->prv->prv->prv,scan->prv->prv->prv->prv,++vno,v7);
        lmev(scan->prv->prv,scan->prv->prv,++vno, v8);
        /* face <5,9,8,7,4> */
        lmef(scan->prv->prv->prv,scan->nxt->nxt->nxt->nxt->nxt,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno,v11);
        /* face <7,8,12,11> */
        lmef(scan->nxt->nxt->nxt,scan->prv->prv->prv->prv,++fno);

        lmev(scan,scan,++vno,v12);
        /* face <9,3,1,12,8> */
        lmef(scan->prv,scan->nxt->nxt->nxt->nxt->nxt,++fno);
        /* face <10,11,12,1,2> */
        lmef(scan->prv,scan->nxt->nxt->nxt,++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            s->nexts = blk_crx->s;
            blk_crx->s->prevs = s;
            blk_crx->s = s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *create_3rd_vol_for_2pt_isolate4(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10,
        CSG_POINT        *v11,
        CSG_POINT        *v12)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if(mev(s, fno, vno, vno+1, v3) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_3rd_vol_for_2pt_isolate4 mev\n");
            clean_up(ERROR); 
        }
        if(mev(s, fno, vno, vno+1, v2) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_3rd_vol_for_2pt_isolate4 mev\n");
            clean_up(ERROR); 
        }

        /* make the first face */
        if(mef(s, vno-2, vno, fno, 1+fno) != ERROR)  ++fno;
        else
        {
            printf("ERROR create_3rd_vol_for_2pt_isolate4 mef\n");
            clean_up(ERROR); 
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        scan = l->ledg;


        lmev(scan->prv, scan->prv, ++vno, v10);
        lmev(scan, scan, ++vno, v7);
        lmev(scan->prv,scan->prv, ++vno, v9);
        /* face <10,3,1,7,9> */
        lmef(scan->prv->prv,scan->nxt->nxt->nxt,++fno);

        lmev(scan->nxt->nxt->nxt, scan->nxt->nxt->nxt, ++vno, v11);
        lmev(scan->nxt, scan->nxt, ++vno, v12);
        /* face<11,12,2,3,10> */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt->nxt->nxt,++fno);

        lmev(scan->nxt->nxt->nxt, scan->nxt->nxt->nxt, ++vno, v6);
        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v8);
        lmev(scan->nxt->nxt->nxt, scan->nxt->nxt->nxt, ++vno, v5);
        /* face<6,5,8,12,11> */
        lmef(scan->nxt->nxt->nxt->nxt,
             scan->nxt->nxt->nxt->nxt->nxt->nxt->nxt->nxt,++fno);

        lmev(scan->prv->prv, scan->prv->prv, ++vno, v4);
        /* face<11,10,9,4,6> */
        lmef(scan->prv->prv->prv, scan->nxt->nxt->nxt->nxt->nxt, ++fno);
        /* face<6,4,5> */
        lmef(scan->nxt->nxt->nxt->nxt, scan->nxt->nxt->nxt->nxt->nxt->nxt, ++fno);
        /* face < 12, 8, 7, 1, 2> */
        lmef(scan->prv, scan->nxt->nxt->nxt, ++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            s->nexts = blk_crx->s;
            blk_crx->s->prevs = s;
            blk_crx->s = s;
        }
        return s; 
}

/**************************************************/
/*    creat_vol_for_edge_case                     */
/*    vertice order is defined as follows:        */
/*    1, 2, 3, 4 are assumed on x-o-y plane,      */
/*    two flux plane: V(5, 3, 2) and V(6, 5, 2)   */
/*    1-->6,   4-->5                              */
/*    1 is at orgion,  2 is (0, dy, 0)            */
/*    The arbitrary input order for vertice is not tested */


LIB_LOCAL CSG_Solid *creat_vol_for_edge_case(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;

        if(mev(s, fno, vno, vno+1, v2) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_vol_for_edge_case mev\n");
            clean_up(ERROR);  
        }
        if(mev(s, fno, vno, vno+1, v3) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_vol_for_edge_case mev\n");
            clean_up(ERROR);  
        }
        if(mev(s, fno, vno, vno+1, v4) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_vol_for_edge_case mev\n");
            clean_up(ERROR);  
        }

        /* make the first face */
        if(mef(s, vno-3, vno, fno, 1+fno) != ERROR)  ++fno;
        else
        {
            printf("ERROR creat_vol_for_edge_case mef\n");
            clean_up(ERROR);  
        }


        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v5);  /* from 4 --> 5, 5th vertex  */
        lmev(first, first, ++vno, v6); /* from 1 --> 6, 6th vertex  */
        lmef(scan->prv, scan->nxt, ++fno);  /* 3: first side face  V(5, 4, 3) */

        scan = mate(scan->nxt)->nxt;
        lmef(scan->prv, scan->nxt, ++fno);  /* 4: Time-space Interface  V(5, 3, 2) */

        scan = mate(scan->nxt)->nxt;
        lmef(scan, scan->nxt->nxt, ++fno);  /* 5: side face  on y-z plane  */

        /* side face on x-z plane, V(5 -> 6 -> 1 -> 4) */
        scan = mate(scan->nxt)->nxt;
        lmef(scan->prv, scan->nxt->nxt, ++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_complement_vol_for_edge_case(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if(mev(s, fno, vno, vno+1, v2) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_edge_case, mev\n");
            clean_up(ERROR);  
        }
        if(mev(s, fno, vno, vno+1, v3) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_edge_case, mev\n");
            clean_up(ERROR);  
        }
        if(mev(s, fno, vno, vno+1, v4) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_edge_case, mev\n");
            clean_up(ERROR);  
        }

    /* make the first face */
        if(mef(s, vno-3, vno, fno, 1+fno) != ERROR)  ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_edge_case, mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v5);    /* 5th vertex  */
        lmev(first->nxt->nxt, first->nxt->nxt, ++vno, v6);   /* 6th vertex  */
        lmev(scan->nxt, scan->nxt, ++vno, v7);    /* 7th vertex  */

     /* Face3: first side face  V(7, 6, 5, 4, 3) */
        lmef(scan->prv->prv, scan->nxt->nxt, ++fno);

        scan = mate(scan->nxt)->nxt;
        lmev(scan->nxt, scan->nxt, ++vno, v8);    /* 8th vertex  */
     /* Face 4: side face  V(7, 3, 2, 8)  */
        lmef(scan->prv, scan->nxt->nxt, ++fno);

        scan = mate(scan->nxt)->nxt;
        lmev(scan->nxt, scan->nxt, ++vno, v9);    /* 9th vertex  */
        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v10);   /* 10th vertex  */
     /* Face 5: side face  V(8, 2, 1, 9, 10) */
        lmef(scan->prv, scan->nxt->nxt->nxt, ++fno);
        scan = mate(scan->nxt->nxt);
     /* Face 6: side face  V(6, 7, 8, 10)  */
        lmef(scan->prv->prv->prv, scan, ++fno);
     /* Face 7: side face  V(6, 10, 9 ,5)  */
        lmef(scan->prv->prv, scan->nxt, ++fno);

     /* construct 2 triangular flux faces */
        scan = mate(scan->nxt);
        lmef(scan, scan->nxt->nxt, ++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_vol_for_edge_case_angl2(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;

        if(mev(s, fno, vno, vno+1, v2) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_vol_for_edge_case_angl2, mev\n");
            clean_up(ERROR); 
        }
        if(mev(s, fno, vno, vno+1, v3) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_vol_for_edge_case_angl2, mev\n");
            clean_up(ERROR); 
        }
        if(mev(s, fno, vno, vno+1, v4) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_vol_for_edge_case_angl2, mev\n");
            clean_up(ERROR); 
        }
    /* make the first face */
        if(mef(s, vno-3, vno, fno, 1+fno)  != ERROR)   ++fno;
        else
        {
            printf("ERROR creat_vol_for_edge_case_angl2, mef\n");
            clean_up(ERROR); 
        }
 
        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v5);  /* from 4 --> 5, 5th vertex  */
        lmev(first, first, ++vno, v6); /* from 1 --> 6, 6th vertex  */
         /* face  V(1,4,5,6) */
        lmef(scan->prv, scan->nxt->nxt->nxt->nxt, ++fno);

        /* face  V(1,6,2) */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt, ++fno);

        /* face V(2,6,3) */
        lmef(scan->nxt, scan->nxt->nxt->nxt, ++fno);

        /* face, V(6,5,3) */
        lmef(scan->nxt, scan->nxt->nxt->nxt, ++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_complement_vol_for_edge_case_angl2(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if(mev(s, fno, vno, vno+1, v2) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_edge_case_angl2, mev\n");
            clean_up(ERROR);  
        }
        if(mev(s, fno, vno, vno+1, v3) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_edge_case_angl2, mev\n");
            clean_up(ERROR);  
        }
        if(mev(s, fno, vno, vno+1, v4) != ERROR)  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_edge_case_angl2, mev\n");
            clean_up(ERROR);  
        }

        /* make the first face */
        if(mef(s, vno-3, vno, fno, 1+fno)  != ERROR)   ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_edge_case_angl2, mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;
        lmev(scan, scan, ++vno, v5);
        lmev(first,first,++vno, v9);

        /* face <4,9,1) */
        lmef(scan->prv->prv,scan->prv->prv->prv->prv,++fno);
        /* face <4,5,9> */
        lmef(scan->prv,scan->prv->prv->prv,++fno);
        lmev(scan->prv,scan->prv,++vno,v6);
        lmev(scan->nxt,scan->nxt,++vno,v7);
        scan = scan->nxt->nxt->nxt;
        /* face<6,5,4,3,7> */
        lmef(scan->prv,scan->prv->prv->prv->prv->prv,++fno);

        lmev(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt,++vno,v10);
        lmev(scan->nxt,scan->nxt,++vno,v8);
        /* face <8,2,1,9,10> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt->nxt,++fno);
        /* face <6,10,9,5> */
        lmef(scan->prv->prv,scan->prv->prv->prv->prv->prv,++fno);
        /* face <7,3,2,8> */
        lmef(scan->prv,scan->nxt->nxt,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid  *create_float_vol_3pt(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10,
        CSG_POINT        *v11,
        CSG_POINT        *v12)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if(mev(s, fno, vno, vno+1, v4) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_float_vol_3pt, mev\n"); 
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v12) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_float_vol_3pt, mev\n"); 
            clean_up(ERROR);  
        }
        if(mev(s, fno, vno, vno+1, v7) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_float_vol_3pt, mev\n"); 
            clean_up(ERROR);  
        }
        if(mev(s, fno, vno, vno+1, v5) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_float_vol_3pt, mev\n"); 
            clean_up(ERROR);  
        }
        if(mev(s, fno, vno, vno+1, v8) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_float_vol_3pt, mev\n"); 
            clean_up(ERROR);  
        }

        /* make the first face */
        if(mef(s, vno-5, vno, fno, 1+fno)  != ERROR)   ++fno;
        else
        {
            printf("ERROR create_float_vol_3pt, mef\n"); 
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        scan = l->ledg;

        lmev(scan, scan, ++vno, v2);
        /* face <1,2,4> */
        lmef(scan->prv, scan->prv->prv->prv, ++fno);

        lmev(scan->prv,scan->prv, ++vno, v3);
        /* face <4,2,3> */
        lmef(scan->prv->prv, scan->prv->prv->prv->prv, ++fno);

        lmev(scan->prv->prv, scan->prv->prv, ++vno, v11);
        /* face <11,12,4,3> */
        lmef(scan->prv->prv->prv, scan->nxt->nxt->nxt->nxt, ++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt, ++vno, v6);
        /* face <6,5,7> */
        lmef(scan->nxt->nxt->nxt, scan->nxt->nxt->nxt->nxt->nxt, ++fno);

        lmev(scan->nxt->nxt->nxt, scan->nxt->nxt->nxt, ++vno, v10);
        /* face <10, 6, 7, 12, 11> */
        lmef(scan->nxt->nxt->nxt->nxt,scan->prv->prv->prv, ++fno);

        lmev(scan->nxt, scan->nxt, ++vno, v9);
        /* face <9,8,5,6,10> */
        lmef(scan->nxt->nxt, scan->prv->prv->prv->prv, ++fno);
        /* face <1,8,9,2> */
        lmef(scan->nxt->nxt, scan->prv, ++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            s->nexts = blk_crx->s;
            blk_crx->s->prevs = s;
            blk_crx->s = s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid  *create_center_vol_degenerate_3pt(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v8) != ERROR)  ++vno;
        else
        {
            printf("ERROR create_center_vol_degenerate_3pt, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v4) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_center_vol_degenerate_3pt, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v7) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_center_vol_degenerate_3pt, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v10) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_center_vol_degenerate_3pt, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v5) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_center_vol_degenerate_3pt, mev\n");
            clean_up(ERROR);  
        }

        /* make the first face */
        if( mef(s, vno-5, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR create_center_vol_degenerate_3pt, mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        scan = l->ledg;

        lmev(scan->prv, scan->prv, ++vno, v9);
        lmev(scan, scan, ++vno, v2);
        /* face <1,2,9,8> */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);

        lmev(scan->prv->prv->prv->prv, scan->prv->prv->prv->prv, ++vno, v3);
        /* face <3,4,8,9> */
        lmef(scan->prv->prv, scan->prv->prv->prv->prv->prv, ++fno);
        /* face <3,7,4> */
        lmef(scan->prv->prv->prv, scan->prv->prv->prv->prv->prv, ++fno);


        lmev(scan->nxt->nxt->nxt, scan->nxt->nxt->nxt, ++vno, v6);
        /* face <3,6,7> */
        lmef(scan->nxt->nxt->nxt->nxt, scan->prv->prv->prv, ++fno);
        /* face <2,3,9> */
        lmef(scan->prv, scan->prv->prv->prv, ++fno);
        /* face <2,6,3> */
        lmef(scan->prv, scan->prv->prv->prv, ++fno);
        /* face <6,10,7> */
        lmef(scan->prv->prv, scan->nxt->nxt, ++fno);
        /* face <6,5,10> */
        lmef(scan->nxt, scan->nxt->nxt->nxt, ++fno);
        /* face <2,1,5> */
        lmef(scan->prv, scan->nxt, ++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            s->nexts = blk_crx->s;
            blk_crx->s->prevs = s;
            blk_crx->s = s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid  *create_complement_center_vol_degenerate_3pt(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10,
        CSG_POINT        *v11,
        CSG_POINT        *v12)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v2);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v9) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_complement_center_vol_degenerate_3pt, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v10) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_complement_center_vol_degenerate_3pt, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v11) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_complement_center_vol_degenerate_3pt, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v3) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_complement_center_vol_degenerate_3pt, mev\n");
            clean_up(ERROR);  
        }

        /* make the first face */
        if( mef(s, vno-4, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR create_complement_center_vol_degenerate_3pt, mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        scan = l->ledg;

        lmev(scan->prv, scan->prv, ++vno, v8);
        lmev(scan, scan, ++vno, v1);
        /* face <1,8,9,2> */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);

        lmev(scan->prv, scan->prv, ++vno, v5);
        /* face <1,5,8> */
        lmef(scan->prv->prv, scan->prv->prv->prv->prv, ++fno);
        /* face <2,5,1> */
        lmef(scan->prv->prv, scan, ++fno);

        lmev(scan->prv, scan->prv, ++vno, v6);
        /* face <9,8,5,6,10> */
        lmef(scan->prv->prv, scan->nxt->nxt->nxt, ++fno);
        /* face <2,6,5> */
        lmef(scan->prv->prv, scan, ++fno);
        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v12);
        lmev(scan->nxt, scan->nxt, ++vno, v4);
        /* face <3,11,,12,4> */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt->nxt, ++fno);

        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v7);
        /* face <4,12,7> */
        lmef(scan->nxt->nxt->nxt, scan->nxt->nxt->nxt->nxt->nxt, ++fno);
        /* face <11,10,6,7,12> */
        lmef(scan->nxt->nxt->nxt, scan->nxt->nxt->nxt->nxt->nxt->nxt->nxt, ++fno);
        /* face <3,4,7> */
        lmef(scan->nxt, scan->nxt->nxt->nxt, ++fno);
        /* face <3,7,6> */
        lmef(scan->nxt, scan->nxt->nxt->nxt, ++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            s->nexts = blk_crx->s;
            blk_crx->s->prevs = s;
            blk_crx->s = s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_vol_for_ceining_case(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v8);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v6) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v7) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case mev\n");
            clean_up(ERROR);  
        }

    /* make the first face */
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;
/*
        print_CSG_point(first->vtx->pt);
        print_CSG_point(scan->vtx->pt);
*/
        lmev(scan, scan, ++vno, v1);  /* from 7 --> 1   */
        lmev(first, first, ++vno, v2);  /* from 8 --> 2  */
        lmef(first->prv, scan->prv, ++fno);  /* 3: first side face V(1,2,8,7) */

        scan = scan->nxt;   /* scan is now using vtx6 as base */
        lmev(scan, scan, ++vno, v5);  /* from 6 --> 5  */
        lmef(scan->prv->prv->prv->prv, scan->prv, ++fno);  /* 4: face V(1,7,6,5) */

        lmev(scan, scan, ++vno, v4);  /* from 6 --> 4 */
        lmef(scan->prv->prv->prv, scan->prv, ++fno);  /* 5: face V(4,5,6) */

        scan = scan->nxt;    /* scan is now using vtx8 as base */
        lmef(scan, scan->prv->prv, ++fno);  /* 6: face V(4,6,8) */

        lmev(scan, scan, ++vno, v3);  /* from 8 --> 3 */
        lmef(scan->prv, scan->prv->prv->prv, ++fno);  /* 7: face V(3,4,8) */
        lmef(scan->nxt, scan->prv, ++fno);  /* 8: face V(2,3,8) */
/*
        solidls(s, 2);
*/
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}


LIB_LOCAL CSG_Solid *creat_complement_vol_for_floor_case(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v2) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v3) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v4) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case mev\n");
            clean_up(ERROR);  
        }
    /* make the first face */
        if( mef(s, vno-3, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->prv;
/*
        print_CSG_point(first->vtx->pt);
        print_CSG_point(scan->vtx->pt);
*/
        lmev(scan, scan, ++vno, v9);  /* from 2 -->9 */
        lmev(first, first, ++vno, v8); /* from 1 --> 8 */
        lmef(first->prv, scan->prv, ++fno);  /* 3: first side face  V(1,8,9,2) */

        scan = first->nxt;
        lmev(scan->nxt, scan->nxt, ++vno, v10);  /* from 3 -->10 */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt->nxt, ++fno);  /* face V(2,9,10,3) */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt, ++fno);  /* face V(8,10,9) */

        lmev(scan, scan, ++vno, v7);  /* from 4 --> 7 */
        scan = scan->nxt->nxt;
        lmev(scan, scan, ++vno, v6);  /* from 10 -->6 */
        lmef(scan->prv, scan->prv->prv->prv->prv->prv, ++fno);  /* face V(4,3,10,6,7) */

        lmev(scan, scan, ++vno, v5);  /* from 10 -->5 */
        lmef(scan->prv, scan->prv->prv->prv, ++fno);  /* face V(5,6,10) */
        lmef(scan->prv, scan->prv->prv->prv, ++fno);  /* face V(5,7,6) */
        lmef(scan->nxt, scan->prv, ++fno);  /* face V(8,5,10) */
/*
        solidls(s, 2);
*/
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_vol_for_ceining_case2(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v8);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v3) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case2 mev\n");
            clean_up(ERROR); 
        }

        if( mev(s, fno, vno, vno+1, v7) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case2 mev\n");
            clean_up(ERROR); 
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case2 mef\n");
            clean_up(ERROR); 
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first;

        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v6);
        /* face <3, 6, 7> */
        lmef(scan->nxt, scan->nxt->nxt->nxt, ++fno);

//        scan = first;
        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v4);
        /* face <3,4,6> */
        lmef(scan->nxt->nxt->nxt,scan->prv, ++fno);

        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v5);
        /* face <6,4,5> */
        lmef(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);

//        scan = scan->nxt;
        lmev(scan,scan,++vno, v2);
        /* face <8,2,3> */
        lmef(scan->prv->prv->prv,scan->prv,++fno);

        lmev(scan->nxt,scan->nxt,++vno, v1);
        /* face <7,6,5,1> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);
        /* face <8,7,1,2> */
        lmef(scan->prv,scan->nxt->nxt,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_complement_vol_for_floor_case2(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v10);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v9) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case2 mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v6) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case2 mev\n");
            clean_up(ERROR);  
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case2 mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan->nxt, scan->nxt, ++vno, v8);
        /* face <6,9,8> */
        lmef(scan,scan->nxt->nxt,++fno);

        scan = first;
        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v5);
        /* face <6,8,5> */
        lmef(scan->nxt->nxt->nxt,scan->nxt,++fno);

        lmev(scan->prv->prv,scan->prv->prv,++vno, v1);
        lmev(scan->prv,scan->prv,++vno, v2);
        /*  face<8,9,2,1> */
        lmef(scan->prv->prv,scan->prv->prv->prv->prv->prv, ++fno);

        lmev(scan->nxt, scan->nxt, ++vno, v7);
        /* face < 6,5,7> */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt, ++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v4);
        /* face <4,7,5,8,1> */
        lmef(scan->nxt->nxt->nxt, scan->prv->prv->prv, ++fno);

        lmev(scan,scan,++vno, v3);
        /* face <3,2,9,10> */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);
        /* face <3,10,6,7,4> */
        lmef(scan->prv,scan->nxt->nxt->nxt,++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_vol_for_ceining_case3(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v3);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v4) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case3 mev\n"); 
            clean_up(ERROR); 
        }
        if( mev(s, fno, vno, vno+1, v8) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case3 mev\n"); 
            clean_up(ERROR); 
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case3 mef\n"); 
            clean_up(ERROR); 
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v7);
        /* face <7,8,4> */
        lmef(scan->prv, scan->nxt, ++fno);

        scan = first;
        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v6);
        /* face <7,4,6> */
        lmef(scan->nxt->nxt->nxt,scan->prv,++fno);

        scan = scan->nxt;
        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v5);
        /* face <6,4,5> */
        lmef(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);

        lmev(scan,scan,++vno, v2);
        /* face <8,2,3> */
        lmef(scan->prv->prv->prv,scan->prv,++fno);

        lmev(scan->nxt,scan->nxt,++vno, v1);
        /* face <7,6,5,1> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);
        /* face <8,7,1,2> */
        lmef(scan->prv,scan->nxt->nxt,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_complement_vol_for_floor_case3(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v6);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v10) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case3 mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v5) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case3 mev\n");
            clean_up(ERROR);  
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case3 mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan->nxt, scan->nxt, ++vno, v9);
        /* face <5,10,9> */
        lmef(scan,scan->nxt->nxt,++fno);
        scan = first;
        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v8);
        /* face <5,9,8> */
        lmef(scan->nxt->nxt->nxt,scan->nxt,++fno);

        scan = scan->prv;
        lmev(scan->prv->prv,scan->prv->prv,++vno, v1);
        lmev(scan->prv,scan->prv,++vno, v2);
        /* face<<9,8,2,1> */
        lmef(scan->prv->prv,scan->prv->prv->prv->prv->prv, ++fno);

        lmev(scan->nxt, scan->nxt, ++vno, v7);
        /* face < 6,5,7> */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt, ++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v4);
        /* face <4,7,5,8,1> */
        lmef(scan->nxt->nxt->nxt, scan->prv->prv->prv, ++fno);

        lmev(scan,scan,++vno, v3);
        /* face <3,2,9,10> */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);
        /* face <3,10,6,7,4> */
        lmef(scan->prv,scan->nxt->nxt->nxt,++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_vol_for_ceining_case4(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v7);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v8) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case4 mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v6) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case4 mev\n");
            clean_up(ERROR);  
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case4 mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v3);
        /* face <6,8,3> */
        lmef(scan->nxt,scan->prv, ++fno);

        scan = first;
        lmev(scan->nxt,scan->nxt,++vno, v4);
        /* face <6,3,4> */
        lmef(scan->nxt->nxt, scan->prv->prv,++fno);

        scan = scan->prv;
        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v5);
        /* face <6,4,5> */
        lmef(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);

        lmev(scan,scan,++vno, v2);
        /* face <8,2,3> */
        lmef(scan->prv->prv->prv,scan->prv,++fno);

        lmev(scan->nxt,scan->nxt,++vno, v1);
        /* face <7,6,5,1> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);
        /* face <8,7,1,2> */
        lmef(scan->prv,scan->nxt->nxt,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_complement_vol_for_floor_case4(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v5);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v6) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case4 mev\n");
            clean_up(ERROR); 
        }
        if( mev(s, fno, vno, vno+1, v8) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case4 mev\n");
            clean_up(ERROR); 
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case4 mef\n");
            clean_up(ERROR); 
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan->nxt, scan->nxt, ++vno, v10);
        /* face <6,10,8> */
        lmef(scan,scan->nxt->nxt,++fno);
        scan = first;
        lmev(scan->nxt,scan->nxt,++vno, v9);
        /* face <10,9,8> */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt, ++fno);

        scan = scan->prv->prv;
        lmev(scan->prv->prv,scan->prv->prv,++vno, v1);
        lmev(scan->prv,scan->prv,++vno, v2);
        /* face<9,8,2,1> */
        lmef(scan->prv->prv,scan->prv->prv->prv->prv->prv, ++fno);

        lmev(scan->nxt, scan->nxt, ++vno, v7);
        /* face < 6,5,7> */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt, ++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v4);
        /* face <4,7,5,8,1> */
        lmef(scan->nxt->nxt->nxt, scan->prv->prv->prv, ++fno);

        lmev(scan,scan,++vno, v3);
        /* face <3,2,9,10> */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);
        /* face <3,10,6,7,4> */
        lmef(scan->prv,scan->nxt->nxt->nxt,++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_vol_for_ceining_case5(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v8);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v3) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case5 mev\n");
            clean_up(ERROR); 
        }
        if( mev(s, fno, vno, vno+1, v7) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case5 mev\n");
            clean_up(ERROR); 
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR creat_vol_for_ceining_case5 mef\n");
            clean_up(ERROR); 
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v4);
        /* face <3, 4, 7> */
        lmef(scan->prv, scan->nxt, ++fno);

        scan = first;
        lmev(scan->nxt,scan->nxt,++vno, v6);
        /* face <7,4,6> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt, ++fno);

        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v5);
        /* face <6,4,5> */
        lmef(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);

        lmev(scan,scan,++vno, v2);
        /* face <8,2,3> */
        lmef(scan->prv->prv->prv,scan->prv,++fno);

        lmev(scan->nxt,scan->nxt,++vno, v1);
        /* face <7,6,5,1> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);
        /* face <8,7,1,2> */
        lmef(scan->prv,scan->nxt->nxt,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_complement_vol_for_floor_case5(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v10);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v9) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case5 mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v6) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case5 mev\n");
            clean_up(ERROR);  
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_floor_case5 mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v5);
        /* face <6,9,5> */
        lmef(scan->prv,scan->nxt,++fno);

        scan = first;
        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v8);
        /* face <9,8,5> */
        lmef(scan->nxt->nxt->nxt,scan->prv,++fno);

        lmev(scan->prv->prv,scan->prv->prv,++vno, v1);
        lmev(scan->prv,scan->prv,++vno, v2);
        /*  face<9,8,2,1> */
        lmef(scan->prv->prv,scan->prv->prv->prv->prv->prv, ++fno);

        lmev(scan->nxt, scan->nxt, ++vno, v7);
        /* face < 6,5,7> */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt, ++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v4);
        /* face <4,7,5,8,1> */
        lmef(scan->nxt->nxt->nxt, scan->prv->prv->prv, ++fno);

        lmev(scan,scan,++vno, v3);
        /* face <3,2,9,10> */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);
        /* face <3,10,6,7,4> */
        lmef(scan->prv,scan->nxt->nxt->nxt,++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_complement_vol_for_ceining_case(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;

        if( mev(s, fno, vno, vno+1, v2) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v3) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v4) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case, mev\n");
            clean_up(ERROR);  
        }
        
        /* make the first face */
        if( mef(s, vno-3, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case, mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v9);  /* from 4 -->9 */
        lmev(first, first, ++vno, v8); /* from 1 --> 8 */
        lmef(first->prv, scan->prv, ++fno);  /* 3: first side face  V(1,4,9,8) */


        scan = scan->nxt;
        lmev(scan, scan, ++vno, v10);  /* from 3 -->10 */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);  /* face V(9,4,3,10) */
        lmef(scan->prv, scan->prv->prv->prv, ++fno);  /* face V(8,9,10) */

        lmev(scan->prv, scan->prv, ++vno, v6);  /* from 10 -->6 */
        scan = scan->nxt;
        lmev(scan, scan, ++vno, v7);  /* from 2 --> 7 */
        lmef(scan->prv, scan->prv->prv->prv->prv->prv, ++fno);  /* face V(7,6,10,3,2) */

        scan = scan->nxt;
        lmev(scan->nxt, scan->nxt, ++vno, v5);  /* from 8 -->5 */
        lmef(scan->nxt->nxt->nxt, scan->nxt->nxt->nxt->nxt->nxt, ++fno);  /* face V(8,10,6) */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt, ++fno);  /* face V(5,8,6) */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt, ++fno);  /* face V(8,5,7,2,1) */
/*
        solidls(s, 2);
*/
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_vol_for_ceining_ceining_case2(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v8);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v3) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_ceining_case2, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v7) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_ceining_case2, mev\n");
            clean_up(ERROR);  
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_vol_for_ceining_ceining_case2, mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v6);
        /* face <3, 6, 7> */
        lmef(scan->prv, scan->nxt, ++fno);

        scan = first;
        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v4);
        /* face <3,4,6> */
        lmef(scan->nxt->nxt->nxt,scan->prv, ++fno);

        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v5);
        /* face <6,4,5> */
        lmef(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);

        lmev(scan,scan,++vno, v2);
        /* face <8,2,3> */
        lmef(scan->prv->prv->prv,scan->prv,++fno);

        lmev(scan->nxt,scan->nxt,++vno, v1);
        /* face <7,6,5,1> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);
        /* face <8,7,1,2> */
        lmef(scan->prv,scan->nxt->nxt,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_complement_vol_for_ceining_case2(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v6);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v5) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case2, mev\n"); 
            clean_up(ERROR); 
        }
        if( mev(s, fno, vno, vno+1, v10) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case2, mev\n"); 
            clean_up(ERROR); 
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case2, mef\n"); 
            clean_up(ERROR); 
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan->nxt, scan->nxt, ++vno, v9);
        /* face <5,9,10> */
        lmef(scan->nxt->nxt,scan,++fno);
        scan = first;
        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v8);
        /* face <8,9,5> */
        lmef(scan->nxt->nxt->nxt,scan->prv,++fno);

        lmev(scan->prv->prv,scan->prv->prv,++vno, v1);
        lmev(scan->prv,scan->prv,++vno, v7);
        /* face < 5,6,7> */
        lmef(scan,scan->prv->prv, ++fno);


        lmev(scan->prv,scan->prv,++vno,v2);
        /* face <1,8,5,7,2> */
        lmef(scan->prv->prv,scan->nxt->nxt->nxt->nxt,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v4);
        /* face <1,4,9,8> */
        lmef(scan->nxt->nxt->nxt,scan->prv->prv->prv, ++fno);

        lmev(scan->nxt,scan->nxt,++vno, v3);
        /* face <9,4,3,10> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt, ++fno);
        /* face <7,6,10,3,2> */
        lmef(scan->nxt->nxt,scan->prv->prv,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_complement_vol_for_ceining_case3(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v10);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v6) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case3, mev\n"); 
            clean_up(ERROR); 
        }
        
        if( mev(s, fno, vno, vno+1, v9) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case3, mev\n"); 
            clean_up(ERROR); 
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case3, mef\n"); 
            clean_up(ERROR); 
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan->nxt, scan->nxt, ++vno, v8);
        /* face <9,6,8> */
        lmef(scan,scan->nxt->nxt,++fno);
        scan = first;
        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v5);
        /* face <5,8,6> */
        lmef(scan->nxt->nxt->nxt,scan->prv,++fno);

        scan = scan->prv;
        lmev(scan->prv->prv,scan->prv->prv,++vno, v1);
        lmev(scan->prv,scan->prv,++vno, v7);
        /* face < 5,6,7> */
        lmef(scan,scan->prv->prv, ++fno);

        lmev(scan->prv,scan->prv,++vno,v2);
        /* face <1,8,5,7,2> */
        lmef(scan->prv->prv,scan->nxt->nxt->nxt->nxt,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v4);
        /* face <1,4,9,8> */
        lmef(scan->nxt->nxt->nxt,scan->prv->prv->prv, ++fno);

        lmev(scan->nxt,scan->nxt,++vno, v3);
        /* face <9,4,3,10> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt, ++fno);
        /* face <7,6,10,3,2> */
        lmef(scan->nxt->nxt,scan->prv->prv,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_vol_for_ceining_ceining_case4(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v7);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v8) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_ceining_case4, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v6) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ceining_ceining_case4, mev\n");
            clean_up(ERROR);  
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_vol_for_ceining_ceining_case4, mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v3);
        /* face <8,3,5> */
        lmef(scan->prv, scan->nxt, ++fno);

        scan = first;
        lmev(scan->nxt,scan->nxt,++vno, v4);
        /* face <3,4,6> */
        lmef(scan->nxt->nxt->nxt->nxt,scan->nxt->nxt, ++fno);

        scan = scan->prv;
        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v5);
        /* face <6,4,5> */
        lmef(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);

        lmev(scan,scan,++vno, v2);
        /* face <8,2,3> */
        lmef(scan->prv->prv->prv,scan->prv,++fno);
        lmev(scan->nxt,scan->nxt,++vno, v1);
        /* face <7,6,5,1> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);
        /* face <8,7,1,2> */
        lmef(scan->prv,scan->nxt->nxt,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_complement_vol_for_ceining_case4(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v6);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v5) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case4, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v10) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case4, mev\n");
            clean_up(ERROR);  
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case4, mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;
        lmev(scan, scan, ++vno, v8);
        /* face <10,5,8> */
        lmef(scan->prv,scan->nxt,++fno);
        scan = first;
        lmev(scan->nxt,scan->nxt,++vno, v9);
        /* face <9,10,8> */
        lmef(scan->nxt->nxt,scan->prv->prv,++fno);

        lmev(scan->prv->prv,scan->prv->prv,++vno, v1);
        lmev(scan->prv,scan->prv,++vno, v7);
        /* face < 5,6,7> */
        lmef(scan,scan->prv->prv, ++fno);

        lmev(scan->prv,scan->prv,++vno,v2);
        /* face <1,8,5,7,2> */
        lmef(scan->prv->prv,scan->nxt->nxt->nxt->nxt,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v4);
        /* face <1,4,9,8> */
        lmef(scan->nxt->nxt->nxt,scan->prv->prv->prv, ++fno);

        lmev(scan->nxt,scan->nxt,++vno, v3);
        /* face <9,4,3,10> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt, ++fno);
        /* face <7,6,10,3,2> */
        lmef(scan->nxt->nxt,scan->prv->prv,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *creat_vol_for_ceining_ceining_case5(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v8);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v3) != ERROR )  ++vno;
        else
        {
            printf("ERROR  creat_vol_for_ceining_ceining_case5. mev\n"); 
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v7) != ERROR )  ++vno;
        else
        {
            printf("ERROR  creat_vol_for_ceining_ceining_case5. mev\n"); 
            clean_up(ERROR);  
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR  creat_vol_for_ceining_ceining_case5. mef\n"); 
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v4);
        /* face <3, 4, 7> */
        lmef(scan->prv, scan->nxt, ++fno);

        scan = first;
        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v6);
        /* face <4,6,7> */
        lmef(scan->nxt->nxt->nxt,scan->nxt, ++fno);

        lmev(scan->nxt->nxt, scan->nxt->nxt, ++vno, v5);
        /* face <6,4,5> */
        lmef(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);

        lmev(scan,scan,++vno, v2);
        /* face <8,2,3> */
        lmef(scan->prv->prv->prv,scan->prv,++fno);

        lmev(scan->nxt,scan->nxt,++vno, v1);
        /* face <7,6,5,1> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);
        /* face <8,7,1,2> */
        lmef(scan->prv,scan->nxt->nxt,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}


LIB_LOCAL CSG_Solid *creat_complement_vol_for_ceining_case5(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v6);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v9) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case5, mev\n");
            clean_up(ERROR); 
        }
        if( mev(s, fno, vno, vno+1, v10) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case5, mev\n");
            clean_up(ERROR); 
        }
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_ceining_case5, mef\n");
            clean_up(ERROR); 
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;

        lmev(scan->nxt, scan->nxt, ++vno, v5);
        /* face <5,9,6> */
        lmef(scan->nxt->nxt,scan->prv,++fno);

        lmev(scan->nxt,scan->nxt,++vno, v8);
        /* face <5,8,9> */
        lmef(scan->nxt->nxt,scan->prv->prv,++fno);

        scan = scan->prv;
        lmev(scan->prv->prv,scan->prv->prv,++vno, v1);
        lmev(scan->prv,scan->prv,++vno, v7);
        /* face < 5,6,7> */
        lmef(scan,scan->prv->prv, ++fno);

        lmev(scan->prv,scan->prv,++vno,v2);
        /* face <1,8,5,7,2> */
        lmef(scan->prv->prv,scan->nxt->nxt->nxt->nxt,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v4);
        /* face <1,4,9,8> */
        lmef(scan->nxt->nxt->nxt,scan->prv->prv->prv, ++fno);

        lmev(scan->nxt,scan->nxt,++vno, v3);
        /* face <9,4,3,10> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt, ++fno);
        /* face <7,6,10,3,2> */
        lmef(scan->nxt->nxt,scan->prv->prv,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

/**************************************************/
/*    creat_vol_for_plane_case                    */
/*    vertice order is defined as follows:        */
/*    1, 2, 3, 4 are assumed on x-o-z plane,      */
/*    two flux plane: V(5, 7, 6) and V(7, 8, 6)   */
/*    1-->6,   2-->8, 3-->7, 4-->5                */
/*    1 is at orgion,  2 is (dx, 0, 0)            */
/*    The arbitrary input order for vertice is not tested */

LIB_LOCAL CSG_Solid *creat_vol_for_plane_case(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;
        CSG_Vertex       *v;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v2) != ERROR )  ++vno;
        else
        { 
            printf("ERROR creat_vol_for_plane_case, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v3) != ERROR )  ++vno;
        else
        { 
            printf("ERROR creat_vol_for_plane_case, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v4) != ERROR )  ++vno;
        else
        { 
            printf("ERROR creat_vol_for_plane_case, mev\n");
            clean_up(ERROR);  
        }
        /* make the first face */
        if( mef(s, vno-3, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        { 
            printf("ERROR creat_vol_for_plane_case, mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;

        first = l->ledg;
        scan = first->nxt;
        v = scan->vtx;
        lmev(scan, scan, ++vno, v5);  /* from 4 --> 5, 5th vertex  */
        lmev(first, first, ++vno, v6); /* from 1 --> 6, 6th vertex  */
    /* face3: first side face  V(1, 4, 5, 6)  */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);

    /*  vtx 3 is for scan vtx; */
        scan = scan->nxt;
    /* from 3 --> 7, 7th vertex */
        lmev(scan, scan, ++vno, v7);
    /* face4: V(4, 3, 7, 5) */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);

    /*  vtx 2 is for scan vtx; */
        scan = scan->nxt;
    /* from 2 --> 8, 8th vertex */
        lmev(scan, scan, ++vno, v8);
    /* face5: V(3, 2, 8, 7) */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);

    /* vtx 1 is for scan vtx; */
        scan = scan->nxt;
    /* face6: V(1, 6, 8, 2)  */
        lmef(scan->nxt, scan->prv->prv, ++fno);

    /*  split flux face  */
        scan = mate(scan->nxt)->nxt;
    /*  face7: V(5, 7, 6) or V(7, 8, 6)  */
        lmef(scan, scan->prv->prv, ++fno);
/*
        solidls(s, 2);
*/
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid  *creat_vol_for_corner_star_hexagon(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;
        CSG_Vertex       *v;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v2) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_corner_star_hexagon, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v3) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_corner_star_hexagon, mev\n");
            clean_up(ERROR);  
        }

        /* make first face <1,2,3> */
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR creat_vol_for_corner_star_hexagon, mef\n");
            clean_up(ERROR);  
        }
        fac = fface(s, fno-1);
        l = fac->floops;

        first = l->ledg;
        scan = first->nxt;

        lmev(scan, scan, ++vno, v4);
        /* make face <1,3,4> */
        lmef(first,scan->prv,++fno);

        scan = scan->prv;
        lmev(scan,scan,++vno,v5);
        lmev(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt,++vno,v6);
        /* make face <1,4,6> */
        lmef(scan->prv->prv,scan->prv->prv->prv->prv,++fno);
        /* make face <6,4,5> */
        lmef(scan->prv,scan->prv->prv->prv,++fno);

        lmev(scan,scan,++vno,v7);
        scan = scan->nxt;
        /* make face <4,3,7> */
        lmef(scan,scan->prv->prv,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno,v10);
        /* make face <1, 10, 2> */
        lmef(scan->nxt,scan->nxt->nxt->nxt,++fno);

        scan = scan->nxt->nxt;   /* end pt is 10 */
        lmev(scan,scan,++vno,v9);
        /* make face<7,3,2,10,9> */
        lmef(scan->prv,scan->prv->prv->prv->prv->prv,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno,v8);
        /* make face<5,8,6> */
        lmef(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);
        /* make face<8,9,10,1,6> */
        lmef(scan->prv,scan->nxt->nxt->nxt,++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid  *creat_complement_vol_for_corner_star_hexagon(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;
        CSG_Vertex       *v;

        s = mvfs(++fno, ++vno, v3);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v2) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_corner_star_hexagon, mev\n");
            clean_up(ERROR); 
        }
        if( mev(s, fno, vno, vno+1, v1) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_corner_star_hexagon, mev\n");
            clean_up(ERROR); 
        }

        /* make first face <1,2,3> */
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_corner_star_hexagon, mef\n");
            clean_up(ERROR); 
        }

        fac = fface(s, fno-1);
        l = fac->floops;

        first = l->ledg;
        scan = first->prv;

        lmev(scan->nxt, scan->nxt, ++vno, v4);
        /* make face <1,3,4> */
        lmef(scan->nxt->nxt,scan->prv,++fno);

        lmev(scan->prv,scan->prv,++vno, v6);
        /* make face <4,1,6> */
        lmef(scan->nxt->nxt,scan->prv->prv,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v5);
        /* make face <5,4,6> */
        lmef(scan->nxt->nxt->nxt,scan->prv->prv,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno, v10);
        /* make face<10,4,5> */
        lmef(scan->nxt->nxt->nxt,scan->prv->prv->prv,++fno);

        lmev(scan->prv,scan->prv,++vno, v7);
        /* make face<6,1,7> */
        lmef(scan->prv->prv,scan->prv->prv->prv->prv,++fno);

        lmev(scan->prv->prv,scan->prv->prv,++vno, v8);
        /* make face<7,8,10,5,6> */
        lmef(scan->prv->prv->prv,scan->nxt->nxt->nxt,++fno);

        lmev(scan,scan,++vno, v9);
        /* make face<8,7,1,2,9> */
        lmef(scan->prv,scan->nxt->nxt->nxt->nxt,++fno);
        /* make face<3,9,2> */
        lmef(scan->nxt,scan->prv,++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid  *creat_vol_for_ij_twist1(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;
        CSG_Vertex       *v;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v2) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ij_twist1, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v5) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ij_twist1, mev\n");
            clean_up(ERROR);  
        }

        /* make first face <1,2,5> */
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR creat_vol_for_ij_twist1, mef\n");
            clean_up(ERROR);  
        }
        fac = fface(s, fno-1);
        l = fac->floops;

        first = l->ledg;
        scan = first->nxt;
        lmev(scan, scan, ++vno, v6);
        /* make face <1,5,6> */
        lmef(first,scan->prv,++fno);

        lmev(scan,scan,++vno,v4);
        /* make face <6,5,4> */
        lmef(scan->prv,scan->prv->prv->prv,++fno);

        lmev(scan->prv,scan->prv,++vno,v3);
        /* make face <3,6,4> */
        lmef(scan->prv->prv,scan->nxt->nxt->nxt,++fno);

        lmev(scan->prv,scan->prv,++vno,v10);
        /* face <10,3,4> */
        lmef(scan->prv->prv,scan->prv->prv->prv->prv,++fno);

        lmev(scan->prv->prv,scan->prv->prv,++vno,v9);
        lmev(scan,scan,++vno,v7);
        /* face <10,4,5,7,9> */
        lmef(scan->prv,scan->prv->prv->prv->prv->prv,++fno);

        lmev(scan->nxt,scan->nxt,++vno,v8);
        /* face <1,8,2> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt,++fno);

        /* face <10,9,6,3> */
        lmef(scan->prv->prv,scan->prv->prv->prv->prv->prv,++fno);
        /* face<2,8,7,5> */
        lmef(scan->prv,scan->nxt->nxt,++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid  *creat_complement_vol_for_ij_twist1(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;
        CSG_Vertex       *v;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v5) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ij_twist1, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v2) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ij_twist1, mev\n");
            clean_up(ERROR);  
        }

        /* make first face <1,5,2> */
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_ij_twist1, mev\n");
            clean_up(ERROR);  
        }
        fac = fface(s, fno-1);
        l = fac->floops;

        first = l->ledg;
        scan = first->nxt;
        lmev(first, first, ++vno, v6);
        /* make face <1,6,5> */
        lmef(first->prv,scan->nxt,++fno);

        lmev(scan->nxt,scan->nxt,++vno,v4);
        /* make face <4,5,6> */
        lmef(scan->nxt->nxt,scan->prv->prv,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno,v3);
        /* make face <3,4,6> */
        lmef(scan->prv->prv,scan->nxt->nxt->nxt,++fno);

        lmev(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt,++vno,v9);
        lmev(scan->prv->prv,scan->prv->prv,++vno,v10);
        /* face <3,6,10,9> */
        lmef(scan->prv->prv->prv,scan->nxt->nxt->nxt->nxt,++fno);
        /* face <6,1,10> */
        lmef(scan->prv,scan->prv->prv->prv,++fno);

        lmev(scan->nxt,scan->nxt,++vno,v7);
        /* face <4,7,5> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt,++fno);

        lmev(scan,scan,++vno,v8);
        /* face <9,10,2,8> */
        lmef(scan->prv,scan->prv->prv->prv->prv->prv,++fno);
        /* face <7,8,2,5> */
        lmef(scan->prv,scan->nxt->nxt,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid  *creat_vol_for_ji_twist1(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;
        CSG_Vertex       *v;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v2) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ji_twist1, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v3) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_vol_for_ji_twist1, mev\n");
            clean_up(ERROR);  
        }

        /* make first face <1,2,3> */
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR creat_vol_for_ji_twist1, mef\n");
            clean_up(ERROR);  
        }
        fac = fface(s, fno-1);
        l = fac->floops;

        first = l->ledg;
        scan = first->nxt;
        lmev(scan, scan, ++vno, v6);
        /* face <6,1,3> */
        lmef(first,scan->prv,++fno);

        lmev(scan,scan,++vno,v4);
        /* make face <6,3,4> */
        lmef(scan->prv,scan->prv->prv->prv,++fno);

        lmev(scan->prv,scan->prv,++vno,v5);
        /* face <6,4,5> */
        lmef(scan->prv->prv,scan->nxt->nxt->nxt,++fno);

        lmev(scan,scan,++vno,v10);
        /* face <10,4,3> */
        lmef(scan->prv,scan->prv->prv->prv,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno,v7);
        /* face <6,7,1> */
        lmef(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt,++fno);

        lmev(scan->prv->prv->prv,scan->prv->prv->prv,++vno,v8);
        /* face <8,7,6,5> */
        lmef(scan->prv->prv->prv->prv,scan->nxt->nxt->nxt,++fno);

        lmev(scan->nxt,scan->nxt,++vno,v9);
        /* face <8,9,2,1,7> */
        lmef(scan->nxt->nxt,scan->nxt->nxt->nxt->nxt->nxt->nxt,++fno);
        /* face <10,3,2,9> */
        lmef(scan->prv,scan->nxt->nxt,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid  *creat_complement_vol_for_ji_twist1(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;
        CSG_Vertex       *v;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v3) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ji_twist1, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v2) != ERROR )  ++vno;
        else
        {
            printf("ERROR creat_complement_vol_for_ji_twist1, mev\n");
            clean_up(ERROR);  
        }

        /* make first face <1,3,2> */
        if( mef(s, vno-2, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR creat_complement_vol_for_ji_twist1, mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;

        first = l->ledg;
        scan = first->nxt;
        lmev(first, first, ++vno, v6);
        /* make face <1,6,3> */
        lmef(first->prv,scan->nxt,++fno);

        lmev(scan->nxt,scan->nxt,++vno,v4);
        /* make face <4,3,6> */
        lmef(scan->nxt->nxt,scan->prv->prv,++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt,++vno,v5);
        /* make face <4,6,5> */
        lmef(scan->prv->prv,scan->nxt->nxt->nxt,++fno);

        lmev(scan->nxt->nxt->nxt,scan->nxt->nxt->nxt,++vno,v7);
        lmev(scan->prv->prv,scan->prv->prv,++vno,v8);

        /* face <5,6,8,7> */
        lmef(scan->prv->prv->prv,scan->nxt->nxt->nxt->nxt,++fno);
        /* face <7,4,5> */
        lmef(scan->nxt->nxt,scan->prv->prv->prv->prv,++fno);

        lmev(scan->prv,scan->prv,++vno,v10);
        /* face <1,2,10> */
        lmef(scan,scan->prv->prv,++fno);

        lmev(scan->prv,scan->prv,++vno,v9);
        /* face <8,6,1,10,9> */
        lmef(scan->prv->prv,scan->prv->prv->prv->prv->prv->prv,++fno);
        /* face <9,10,2,3> */
        lmef(scan->nxt,scan->prv->prv,++fno);
        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            blk_crx->s->nexts = s;
            s->prevs = blk_crx->s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *create_vol_for_plan_float_vol(
        CSG_BLK_CRX      *blk_crx,
        CSG_POINT        *v1,
        CSG_POINT        *v2,
        CSG_POINT        *v3,
        CSG_POINT        *v4,
        CSG_POINT        *v5,
        CSG_POINT        *v6,
        CSG_POINT        *v7,
        CSG_POINT        *v8,
        CSG_POINT        *v9,
        CSG_POINT        *v10,
        CSG_POINT        *v11,
        CSG_POINT        *v12)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;

        s = mvfs(++fno, ++vno, v1);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, v4) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_vol_for_plan_float_vol, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v11) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_vol_for_plan_float_vol, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v8) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_vol_for_plan_float_vol, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v5) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_vol_for_plan_float_vol, mev\n");
            clean_up(ERROR);  
        }
        if( mev(s, fno, vno, vno+1, v9) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_vol_for_plan_float_vol, mev\n");
            clean_up(ERROR);  
        }

        /* make the first face */
        if( mef(s, vno-5, vno, fno, 1+fno)  != ERROR )  ++fno;
        else
        {
            printf("ERROR create_vol_for_plan_float_vol, mef\n");
            clean_up(ERROR);  
        }

        fac = fface(s, fno-1);
        l = fac->floops;
        scan = l->ledg;

        lmev(scan, scan, ++vno, v2);
        /* face <1,2,4> */
        lmef(scan->prv, scan->prv->prv->prv, ++fno);

        lmev(scan->prv,scan->prv, ++vno, v3);
        /* face <4,2,3> */
        lmef(scan->prv->prv, scan->prv->prv->prv->prv, ++fno);

        lmev(scan->prv->prv, scan->prv->prv, ++vno, v12);
        /* face <12,11,4,3> */
        lmef(scan->prv->prv->prv, scan->nxt->nxt->nxt->nxt, ++fno);

        lmev(scan->nxt->nxt->nxt, scan->nxt->nxt->nxt, ++vno, v7);
        /* face <7,8,11,12> */
        lmef(scan->nxt->nxt->nxt->nxt, scan->nxt->nxt->nxt->nxt->nxt->nxt->nxt, ++fno);

        lmev(scan->nxt->nxt,scan->nxt->nxt, ++vno, v6);
        /* face <7,5,8> */
        lmef(scan->nxt->nxt->nxt->nxt->nxt->nxt, scan->nxt->nxt->nxt->nxt, ++fno);
        /* face <7,6,5> */
        lmef(scan->nxt->nxt->nxt, scan->nxt->nxt->nxt->nxt->nxt, ++fno);

        lmev(scan->nxt, scan->nxt, ++vno, v10);
        /* face <6,10, 9, 5> */
        lmef(scan->nxt->nxt, scan->nxt->nxt->nxt->nxt->nxt, ++fno);
        /* face <1,9,10,2> */
        lmef(scan->nxt->nxt, scan->prv, ++fno);

        if(blk_crx->s == NULL)
        {
            blk_crx->s = s;
        }
        else
        {
            s->nexts = blk_crx->s;
            blk_crx->s->prevs = s;
            blk_crx->s = s;
        }
        return s; 
}

LIB_LOCAL CSG_Solid *create_box_vol(
        CSG_BLK_CRX     *blk)
{
        CSG_Solid        *s;
        CSG_HalfEdge     *first, *scan;
        CSG_Loop         *l;
        CSG_Face         *fac;
        CSG_POINT        ***iv = blk->icrds;

        if(blk->s != NULL)
        {
            printf("ERROR: in create_box_vol(), vol already constructed \n");
            clean_up(ERROR);
        }

        s = mvfs(++fno, ++vno, &iv[0][0][0]);
        s->solidno = ++sno;
        if( mev(s, fno, vno, vno+1, &iv[0][1][0]) == ERROR )
        {
            solidls(s,2);
            printf("ERROR: in creat_vol_blk(), mev failed\n");
            printf("vno = %d,\n", vno);
            clean_up(ERROR);
        }
          ++vno;
        if( mev(s, fno, vno, vno+1, &iv[1][1][0]) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_box_vol, mev\n");
            clean_up(ERROR);
        }

        if( mev(s, fno, vno, vno+1, &iv[1][0][0]) != ERROR )  ++vno;
        else
        {
            printf("ERROR create_box_vol, mev\n");
            clean_up(ERROR);
        }

        /* make the first face */
        if( mef(s, vno-3, vno, fno, 1+fno)  != ERROR )   ++fno;
        else
        {
            printf("ERROR create_box_vol, mef\n");
            clean_up(ERROR);
        }
        fac = fface(s, fno-1);
        l = fac->floops;
        first = l->ledg;
        scan = first->nxt;
        lmev(scan, scan, ++vno, &iv[1][0][1]);  /* from 4 --> 5, 5th vertex  */
        lmev(first, first, ++vno, &iv[0][0][1]); /* from 1 --> 6, 6th vertex  */
        /* face3: first side face  V(1, 4, 5, 6)  */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);
        /*  vtx 3 is for scan vtx; */
        scan = scan->nxt;
        /* from 3 --> 7, 7th vertex */
        lmev(scan, scan, ++vno, &iv[1][1][1]);
        /* face4: V(4, 3, 7, 5) */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);


        /*  vtx 2 is for scan vtx; */
        scan = scan->nxt;
        /* from 2 --> 8, 8th vertex */
        lmev(scan, scan, ++vno, &iv[0][1][1]);
        /* face5: V(3, 2, 8, 7) */
        lmef(scan->prv, scan->prv->prv->prv->prv, ++fno);
        /* vtx 1 is for scan vtx; */
        scan = scan->nxt;
        /* face6: V(1, 6, 8, 2)  */
        lmef(scan->nxt, scan->prv->prv, ++fno);
        /*
        solidls(s, 2);
        */
        blk->s = s;
        return s; 
}


LOCAL void glue(
        CSG_Solid       *master,
        CSG_Solid       **slave,
        CSG_Face        *fmas,
        CSG_Face        *fsal)
{
        if(CSG_face_match(fmas,fsal) != YES)
        {
            printf("ERROR glue, face not match\n");
            solidls(master,2);
            solidls((*slave),2);
            clean_up(ERROR);  
        }
        
        master->vol += (*slave)->vol; 
        merge(master, slave);
        lkfmrh(fmas, fsal);
        loopglue(fmas);
}

LOCAL int match(
        CSG_HalfEdge        *h1,
        CSG_HalfEdge        *h2)
{
        CSG_POINT           *v1, *v2;
        v1 = h1->vtx->pt;
        v2 = h2->vtx->pt;
        if(v1->type == v2->type)
        {
            if(v1->type == IS_ICRDS)
            {
                if( 0 == memcmp(v1->csg_pt.icrds,
                 v2->csg_pt.icrds,sizeof(int)*3))
                return 1;
            }
            else
            {
                if(v1->csg_pt.crx == v2->csg_pt.crx)
                return 1;
            }
        }
        return 0;
}


LIB_LOCAL void solid_glue(
        CSG_Solid       *master,
        CSG_Solid       **slave)
{
        CSG_Face        *match_f[10][2];
        CSG_Face        *mf, *sf;
        int              n = 0;
        CSG_HalfEdge    *comm_mhe, *comm_she;

        for(mf = master->sfaces; mf != NULL; mf = mf->nextf)
        {
            for(sf = (*slave)->sfaces; sf != NULL; sf = sf->nextf)
            {
                if(CSG_face_match(mf,sf))
                {
                    match_f[n][0] = mf;
                    match_f[n++][1] = sf;
                    break;
                }
            }
        }

        if(master->comp != (*slave)->comp)
        {
            printf("ERROR, in solid_glue(), different COMP solid merging\n");
            printf("master, slave COMP <%d, %d> \n",master->comp,(*slave)->comp);
            solidls(master,2);
            solidls((*slave),2);
            clean_up(ERROR);
        }

        if(n > 10)
        {
            printf("ERROR, in solid_glue(), n > 10\n");
            solidls(master,2);
            solidls((*slave),2);
            clean_up(ERROR);
        }
        switch(n)
        {
        case 0:
            printf("ERROR In solid_glue(), num of match_face is zero\n");
            solidls(master,2);
            solidls(*slave,2);
            clean_up(ERROR);
        break;
        case 1:
            if(CSG_face_match(match_f[0][0], match_f[0][1]) != YES)
            {
                printf("ERROR In solid_glue(), case 1, num of match_face is zero\n");
                solidls(master,2);
                solidls(*slave,2);
                clean_up(ERROR);
            }
            glue(master,slave,match_f[0][0], match_f[0][1]);
            return;
        break;
        case 2:
            comm_mhe =  comm_edge_of_faces(match_f[0][0],match_f[1][0]);
            comm_she =  comm_edge_of_faces(match_f[0][1],match_f[1][1]);
            if(comm_mhe == NULL || comm_she == NULL)
            {
                printf("ERROR solid_glue() comm_mhe NULL || comm_she NULL\n");
                solidls(master,2);
                solidls(*slave,2);
                clean_up(ERROR);  
            }
            lkef(comm_mhe, mate(comm_mhe));
            lkef(comm_she, mate(comm_she));
            if(CSG_face_match(match_f[0][0], match_f[0][1]) != YES)
            {
                printf("ERROR solid_glue(), match_f[0][0] != match_f[0][1]\n");
                solidls(master,2);
                solidls(*slave,2);
                clean_up(ERROR);  
            }
            glue(master,slave,match_f[0][0], match_f[0][1]);
            return;
        break;
        default:
            printf("In solid_glue(), num of match_face greater than 2\n");
            printf("PROCESSING\n");
            clean_up(ERROR);
        break;
        }
        printf("In solid_glue(), num of match_face greater than 10\n");
        printf("PROCESSING\n");
        clean_up(ERROR);
}

LOCAL void lkef(
        CSG_HalfEdge        *h1,
        CSG_HalfEdge        *h2)
{
        CSG_Face            *f1, *f2;
        CSG_Loop            *l, *l1, *l2;
        register CSG_HalfEdge *he;

        CSG_Vertex          *v2, *v1;
        CSG_HalfEdge        *rh1, *rh2;
        CSG_Edge            *e2;

        l1 = h1->wloop;
        f1 = l1->lface;
        l2 = h2->wloop;
        f2 = l2->lface;

        while(l = f2->floops)
        {
            CSG_dellist(CSG_LOOP, (CSG_Node *)l, (CSG_Node *)f2);
            CSG_addlist(CSG_LOOP, (CSG_Node *)l, (CSG_Node *)f1);
        }
        he = l2->ledg;
        do
        {
            he->wloop = l1;
        }while((he = he->nxt) != l2->ledg);

        h1->prv->nxt = h2;
        h2->prv->nxt = h1;
        he = h2->prv;
        h2->prv = h1->prv;
        h1->prv = he;

        v1 = h1->vtx; v2 = h2->vtx;
        rh1 = h1->nxt;  rh2 = h2->nxt;
        e2 = h2->edg;

        delhe(h2);
        delhe(h1);

        v2->vedge = rh1;
        if(!v2->vedge->edg)
            v2->vedge = (CSG_HalfEdge *)NULL;
        v1->vedge = rh2;
        if(!v1->vedge->edg)
            v1->vedge = (CSG_HalfEdge *)NULL;
        l1->ledg = rh1;

        CSG_del(CSG_FACE, (CSG_Node *)f2, (CSG_Node *)f1->fsolid);
        CSG_del(CSG_LOOP, (CSG_Node *)l2, (CSG_Node *)f1);
        CSG_del(CSG_EDGE, (CSG_Node *)e2, (CSG_Node *)f1->fsolid);
}


LOCAL CSG_HalfEdge *delhe(
        CSG_HalfEdge        *he)
{
         CSG_HalfEdge     *rhe;

         if(he->edg == NULL)
         {
             CSG_del(CSG_HALFEDGE, (CSG_Node*)he, NULL);
             return(NULL);
         }
         else if(he->nxt == he)
         {
             he->edg = NULL;
             return(he);
         }
         else
         {
             he->prv->nxt = he->nxt;
             he->nxt->prv = he->prv;
             rhe = he->prv;
             CSG_del(CSG_HALFEDGE, (CSG_Node*)he, NULL);
             return rhe;
         }
}

LOCAL void CSG_del(
        int             what,
        CSG_Node        *which,
        CSG_Node        *where)
{
        switch(what)
        {
        case CSG_SOLID:
/*  NEW Add */
            CSG_dellist(CSG_SOLID, which, NULL);
/* end NEW ADD */
        break;
        case CSG_FACE:
            CSG_dellist(CSG_FACE, which, where);
        break;
        case CSG_LOOP:
            CSG_dellist(CSG_LOOP, which, where);
        break;
        case CSG_EDGE:
            CSG_dellist(CSG_EDGE, which, where);
        break;
        case CSG_HALFEDGE:
        break;
        case CSG_VERTEX:
            CSG_dellist(CSG_VERTEX, which, where);
        break;
        }
        free(which);
}


LOCAL void CSG_dellist(
        int             what,
        CSG_Node        *which,
        CSG_Node        *where)
{
        switch(what)
        {
/* New Add */
        case CSG_SOLID:
            if(which->s.prevs)
                which->s.prevs->nexts = which->s.nexts;
            if(which->s.nexts)
                which->s.nexts->prevs = which->s.prevs;
        break;
/* end New add */
        case CSG_FACE:
            if(which->f.prevf)
                which->f.prevf->nextf = which->f.nextf;
            if(which->f.nextf)
                which->f.nextf->prevf = which->f.prevf;
            if((CSG_Face *) which == where->s.sfaces)
                where->s.sfaces = which->f.nextf;
        break;
        case CSG_LOOP:
            if(which->l.prevl)
                which->l.prevl->nextl = which->l.nextl;
            if(which->l.nextl)
                which->l.nextl->prevl = which->l.prevl;
            if((CSG_Loop *) which == where->f.floops)
                where->f.floops = which->l.nextl;
        break;
        case CSG_EDGE:
            if(which->e.preve)
                which->e.preve->nexte = which->e.nexte;
            if(which->e.nexte)
                which->e.nexte->preve = which->e.preve;
            if((CSG_Edge *) which == where->s.sedges)
                where->s.sedges = which->e.nexte;
        break;
        case CSG_VERTEX:
            if(which->v.prevv)
                which->v.prevv->nextv = which->v.nextv;
            if(which->v.nextv)
                which->v.nextv->prevv = which->v.prevv;
            if((CSG_Vertex *) which == where->s.sverts)
                where->s.sverts = which->v.nextv;
        break;
        }
}



/* f1 and f2 must be in the same solid,
   and assume they are simple connected only
   through ONE EDGE
*/
LOCAL CSG_HalfEdge *comm_edge_of_faces(
        CSG_Face       *f1,
        CSG_Face       *f2)
{
        CSG_HalfEdge    *h1, *h2;
        CSG_HalfEdge    *comm_he[3];
        int             n = 0;

        if(f1 == f2)
        {
            printf("ERROR comm_edge_of_faces, f1 = f2\n");
            clean_up(ERROR); 
        }
        h1 = f1->floops->ledg;
        do
        {
            h2 = f2->floops->ledg;
            do
            {
                if(mate(h1) == h2)
                {
                    comm_he[n++] = h1;
                    /*
                    return h1;
                    */
                }
            }while((h2=h2->nxt) != f2->floops->ledg);
        }while((h1=h1->nxt) != f1->floops->ledg);

        if(n > 1)
        {
            printf("ERROR comm_edge_of_faces, n = %d > 1\n", n);
            clean_up(ERROR);  
        }
        if(n == 1)
            return comm_he[0];
        return NULL;
}


LIB_LOCAL int CSG_face_match(
        CSG_Face      *f1,
        CSG_Face      *f2)
{
        CSG_HalfEdge   *h1, *h2, *h2_start;
        CSG_Vertex     *v1, *v2;
        int            found = NO;

        if(f1 == NULL || f2 == NULL)
        {
            printf("ERROR In CSG_face_match()\n");
            printf("f1 or f2 is NULL\n");
            clean_up(ERROR);
        }
        if(f1 == f2)
        {
            printf("ERROR In CSG_face_match()\n");
            printf("f1 = f2\n");
            clean_up(ERROR);
        }
        if(CSG_loop_vtx_num(f1->floops) != CSG_loop_vtx_num(f2->floops))
            return NO;


        h1 = f1->floops->ledg; v1 = h1->vtx;
        h2 = f2->floops->ledg;
        do
        {
            v2 = h2->vtx;
            if(CSG_Vertex_equal(v1,v2))
            {
                found = YES;
                break;
            }
        }while((h2 = h2->nxt) != f2->floops->ledg);

        if(found == NO)
            return NO;

        if(CSG_Vertex_equal(v1,h2->vtx) != YES)
        {
            printf("ERROR in CSG_face_match()\n");
            printf("v1 not equal h2->vtx\n");
            clean_up(ERROR);
        }

        found = YES;
        h2_start = h2;
        do
        {
            v1 = h1->vtx;
            v2 = h2->vtx;
            if(! CSG_Vertex_equal(v1,v2))
            {
                found = NO;
                break;
            }
        } while((h1=h1->nxt) != f1->floops->ledg &&
                (h2=h2->nxt) != h2_start);
        if( found == YES)
        {
           if(fabs(f1->area - f2->area) > 0.0000000001)
           {
               printf("ERROR in CSG_face_match()\n");
               printf("area of two face are diff\n");
               clean_up(ERROR);
           }
           return YES;
        }

        found = YES;
        h1 = f1->floops->ledg;
        h2 = h2_start;
        do
        {
            v1 = h1->vtx;
            v2 = h2->vtx;
            if(! CSG_Vertex_equal(v1,v2))
            {
                found = NO;
                break;
            }
        } while((h1=h1->nxt) != f1->floops->ledg &&
                (h2=h2->prv) != h2_start);
        if(found == NO)
           return NO;


        if( fabs(f1->area - f2->area) > 0.0000000001)
        {
             printf("ERROR in CSG_face_match()\n");
             printf("area of two face are diff\n");
             clean_up(ERROR);
        }

        return YES;
}


LIB_LOCAL int CSG_loop_vtx_num(
         CSG_Loop       *l)
{
         CSG_HalfEdge   *he;
         int            n = 0;
         he = l->ledg;
         do
         {
             n++;
         }while((he = he->nxt) != l->ledg);
         return n;
}


LIB_LOCAL int CSG_Vertex_equal(
        CSG_Vertex    *v1,
        CSG_Vertex    *v2)
{
       return(CSG_Point_equal(v1->pt,v2->pt));
}


LIB_LOCAL int CSG_Point_equal(
        CSG_POINT    *v1,
        CSG_POINT    *v2)
{
       if(v1->type == v2->type)
       {
            if(v1->type == IS_ICRDS)
            {
                if(0 == memcmp(v1->csg_pt.icrds,
                           v2->csg_pt.icrds, sizeof(int)*3))
                    return YES;
            }
            else
            {
                if(v1->type != IS_BBI_P)
                {
                    printf("ERROR in CSG_Point_equal\n");
                    printf("v1 type != IS_BBI_P\n");
                    clean_up(ERROR);
                }
                if(v1->csg_pt.crx ==
                   v2->csg_pt.crx)
                    return YES;
            }
        }

        return NO;
}



LIB_LOCAL void free_CSG_vol(
        CSG_Solid       **vol)
{
        CSG_Face        *f;
        CSG_Edge        *e;
        CSG_Vertex      *v;
        CSG_Loop        *l;

        e = (*vol)->sedges;
        while(e)
        {
            (*vol)->sedges = e->nexte;
            free(e->he1);
            free(e->he2);
            free(e);
            e = (*vol)->sedges;
        }

        v = (*vol)->sverts;
        while(v)
        {
            (*vol)->sverts = v->nextv;
            free(v);
            v = (*vol)->sverts;
        }
        f = (*vol)->sfaces;
        while(f)
        {
            (*vol)->sfaces = f->nextf;

            l = f->floops;
            while(l)
            {
                f->floops = l->nextl;
                free(l);
                l = f->floops;
            }
            free(f);
            f = (*vol)->sfaces;
        }

        free(*vol);
        *vol = NULL;
}

/*  The AXIS setting in faceeq is different from the
 *  settings from FT code, the faceeq
 *  is consistent with CG. An equiv. function is:
 *   findNormal3D()
 */
LIB_LOCAL int  faceeq(
        CSG_Loop        *l,
        float           *eq,
        TRI_GRID        *ntg,
        float           dt)
{
        CSG_HalfEdge    *he;
        float           a, b, c, norm;
        float           xi, yi, zi, xj, yj, zj, xc, yc, zc;
        CSG_POINT       *pt;
        int             length;

        a = b = c = xc = yc = zc = 0.0;
        length = 0;
        he = l->ledg;
        do
        {
        /* printf("vertex %d, vertex next %d \n",
               he->vtx->vertexno,  he->nxt->vtx->vertexno);
        */
            pt = he->vtx->pt;
            if(pt->type == IS_ICRDS)
            {
                /*
                printf(" no %d icrds <%d, %d, %d>\n",
                he->vtx->vertexno, pt->csg_pt.icrds[0],
                pt->csg_pt.icrds[1], pt->csg_pt.icrds[2]);
                */
                zi = Coords(Regular_comp_grid_node(pt->csg_pt.icrds, ntg))[0];
                xi = Coords(Regular_comp_grid_node(pt->csg_pt.icrds, ntg))[1];
                if(pt->csg_pt.icrds[2] == 1)
                    yi = dt;
                else
                    yi = 0;
            }
            else
            {
                zi = Coords(pt->csg_pt.crx->pt)[0];
                xi = Coords(pt->csg_pt.crx->pt)[1];
                yi = Coords(pt->csg_pt.crx->pt)[2];
            }

            /*  printf("<%g, %g, %g>\n", xi, yi, zi);  */
            pt = he->nxt->vtx->pt;
            if(pt->type == IS_ICRDS)
            {
                /*
                printf(" no %d icrds <%d, %d, %d>\n",
                he->nxt->vtx->vertexno, pt->csg_pt.icrds[0],
                pt->csg_pt.icrds[1], pt->csg_pt.icrds[2]);
                */
                zj = Coords(Regular_comp_grid_node(pt->csg_pt.icrds, ntg))[0];
                xj = Coords(Regular_comp_grid_node(pt->csg_pt.icrds, ntg))[1];
                if(pt->csg_pt.icrds[2] == 1)
                    yj = dt;
                else
                    yj = 0;
            }
            else
            {
                zj = Coords(pt->csg_pt.crx->pt)[0];
                xj = Coords(pt->csg_pt.crx->pt)[1];
                yj = Coords(pt->csg_pt.crx->pt)[2];
            }
            /*  printf("<%g, %g, %g>\n", xj, yj, zj);  */

            a += (yi - yj) * (zi + zj);
            b += (zi - zj) * (xi + xj);
            c += (xi - xj) * (yi + yj);
            xc += xi;
            yc += yi;
            zc += zi;
            length++;
        } while((he = he->nxt) != l->ledg);
        if((norm = sqrt(a*a + b*b + c*c)) != 0.0)
        {   /* correcting AXIS setting */
            eq[1] = a / norm;
            eq[2] = b / norm;
            eq[0] = c / norm;
            eq[3] = (eq[0]*yc + eq[1]*zc + eq[2]*xc) / (-length);

            if(sqrt(eq[0]*eq[0] + eq[1]*eq[1] + eq[2]*eq[2]) > 1.0+TOL*TOL)
            {
                printf("ERROR: faceeq: Twisted face %d\n", l->lface->faceno);
                printf("eq<%g, %g, %g>\n", eq[0], eq[1], eq[2]);
                printf("sqrt<eq[0]*eq[0] + eq[1]*eq[1] + eq[2]*eq[2]> = %g\n",
                        sqrt(eq[0]*eq[0] + eq[1]*eq[1] + eq[2]*eq[2]));
                solidls(l->lface->fsolid, 2);
                return ERROR;
            }

            return FUNCTION_SUCCEEDED;
        }
        printf("ERROR: faceeq(): null face %d\n", l->lface->faceno);
        facels(l->lface, 2);
        solidls(l->lface->fsolid, 2);
        printf("dt = %f. \n", dt);

        return ERROR;
}

/* return the signed area of a 2D polygon */
LOCAL float findArea2D(
    int      n, 
    float    *x, 
    float    *y)             // 2D polygon

{
    float sum = 0.0;
    float *xptr = x+1, *ylow = y, *yhigh = y+2;
    int   i; 

    /* guarantee the first two vertices are also at array end */
    x[n] = x[0];    y[n] = y[0];
    x[n+1] = x[1];  y[n+1] = y[1];

    for(i=1; i <= n; i++)
        sum += (*xptr++) * ( (*yhigh++) - (*ylow++) );

    return (sum / 2.0);
}

/*
 * From :  Daniel Sunday
 *      "Fast Polygon Area and Newell Normal Computation"
 *      journal of graphics tools, 7(2):9-13, 2002
 */
/* output the approximate unit normal of a 3D nearly planar polygon
 * return the area of the polygon
 */
LIB_LOCAL float findNormal3D(
        CSG_Loop     *l,
        float        *eq)
{
        CSG_HalfEdge    *he;
        static int      saved_nv = 0;
        static float    *x = NULL, *y, *z; // 3D polygon
        int             i, nv = 0;
        float           nwx, nwy, nwz, nlen; 
        float           pt[3]; 

        he = l->ledg;
        do
        {
            nv++;
        }while((he = he->nxt) != l->ledg);

        if(x == NULL || saved_nv < nv)
        {
            saved_nv = nv;
            if(x != NULL) free_these(3, x, y, z);
            vector(&x, nv+2, sizeof(float));
            vector(&y, nv+2, sizeof(float));
            vector(&z, nv+2, sizeof(float));
        }

        he = l->ledg; i = 0; 
        do
        {
            CSG_point_Coord(he->vtx->pt,pt);
            x[i] = pt[0]; 
            y[i] = pt[1]; 
            z[i] = pt[2]; 
            i++; 
        }while((he = he->nxt) != l->ledg);


        // get the Newell normal

        nwx = findArea2D(nv, y, z);
        nwy = findArea2D(nv, z, x);
        nwz = findArea2D(nv, x, y);

        // get length of the Newell normal

        nlen = sqrt( nwx*nwx + nwy*nwy + nwz*nwz );
        // compute the unit normal

        eq[0] = nwx / nlen;
        eq[1] = nwy / nlen;
        eq[2] = nwz / nlen;

        return nlen;    // area of polygon = length of Newell normal
}

EXPORT void facels(
         CSG_Face          *f,
         int               sw)
{
        CSG_Loop        *l;
        CSG_Vertex      *v;
        CSG_Edge        *e;
        CSG_HalfEdge    *he;
        register int    i;
        int             breakline;
        CSG_POINT       *pt;
        float           vpt[3];
        int             icrds[MAXD+1];

        breakline = 8;
        if(sw > 2) breakline = 4;
        printf("Face no. %4d:    ", f->faceno);
        l = f->floops;
        while(l)
        {
            he = l->ledg;
            i = 0;
            do
            {
                v = he->vtx;
                e = he->edg;

                pt = v->pt;
                CSG_point_Coord(pt, vpt);

                if(sw > 2 && e)
                {
                    if(he == e->he1)
                        printf("%4d +(%4d,%4d)-> ",
                            v->vertexno,
                            e->he1->vtx->vertexno,
                            e->he2->vtx->vertexno);
                    else
                        printf("%4d -(%4d,%4d)-> ",
                            v->vertexno,
                            e->he1->vtx->vertexno,
                            e->he2->vtx->vertexno);
                }
                else
                {
                    /*
                    if(pt->type == IS_ICRDS)
                    {
                        printf("%4d (crds <%f, %f, %f>)-> ",
                          v->vertexno, vpt[0], vpt[1], vpt[2]);
                    }
                    else
                    {
                        printf("%4d (crds <%17.15f, %17.15f, %17.15f>)-> ",
                          v->vertexno, vpt[0], vpt[1], vpt[2]);
                    }
                    */
                    printf("%4d -> ", v->vertexno);

                }
                if((++i)%breakline == 0)
                    printf("\n        ");
            }while((he = he->nxt) != l->ledg);
            if(l == f->flout)
                printf("<outer>\n");
            else
                printf("<inner>\n");
            l = l->nextl;
            if(l) printf("        ");
        }

/*  Havenot implemented yet.
    printf("equation: ");
    vecprint(f->feq);
*/
}

EXPORT void solidls(
         CSG_Solid          *s,
         int                sw)
{
        CSG_Face        *f;
        CSG_Loop        *l;
        CSG_Vertex      *v;
        CSG_Edge        *e;
        CSG_HalfEdge    *he;
        int             icrds[MAXD+1];
        CSG_POINT       *pt;
        float           vpt[3];
        int             comp;

        if(s == NULL)
        {
            printf("No Solid\n");
            return;
        }
        printf("\nSolid: %3d, COMP<%d>\n", s->solidno, s->comp);
        printf("Faces:\n");
        f = s->sfaces;
        while(f)
        {
            facels(f, sw);
            f = f->nextf;
        }

        if(sw>1)
        {
            printf("vertices:");
            v = s->sverts;
            while(v)
            {
                pt = v->pt;
                CSG_point_Coord(pt, vpt);
                printf("\n");
                if(pt->type == IS_ICRDS)
                {
                    icrds[0] = pt->csg_pt.icrds[0];
                    icrds[1] = pt->csg_pt.icrds[1];
                    icrds[2] = pt->csg_pt.icrds[2];
                    if(icrds[2] == 1)
                        comp = Regular_comp_grid_comp(icrds,local_new_ntg);
                    else
                        comp = Regular_comp_grid_comp(icrds,local_old_ntg);
                    /* printf("%d, (%d) COMP<%d>", v->vertexno, v->pt, comp);  */
                    printf("%d, COMP<%d> ", v->vertexno, comp);
                    printf("icrds <%3d, %3d, %3d>", icrds[0], icrds[1], icrds[2]);
                    printf(" crds <%g, %g, %g>", vpt[0], vpt[1], vpt[2]);
                }
                else
                {
                    /*  printf("%d, (%d)", v->vertexno, pt->csg_pt.crx->pt);  */
                    printf("%d, ", v->vertexno, pt->csg_pt.crx->pt);
                    printf("crds <%g, %g, %g>", vpt[0], vpt[1], vpt[2]); // used to be %11.9f
                }
            /* print coords here */

                if(sw>2)
                {
                    he = v->vedge;
                    if(he)
                    {
                        do
                        {
                            e = he->edg;
                            if(he == e->he1)
                                printf(" +");
                            else
                                printf(" -");
                            printf("<%d,%d>",e->he1->vtx->vertexno,
                                   e->he2->vtx->vertexno);
                        }while((he = mate(he)->nxt) != v->vedge);
                    }
                }
                v = v->nextv;
            }
            printf("\n");
            printf("Edges: <v1, v2, f1, f2>");
            f = s->sfaces;
            while(f)
            {
                l = f->floops;
                while(l)
                {
                    he = l->ledg;
                    do
                    {
                        if(e = he->edg)
                        {
                            if(he == e->he1)
                            {
                                printf("\n");
                                printf("<%d, %d, %d, %d>",
                                    e->he1->vtx->vertexno,
                                    e->he2->vtx->vertexno,
                                    e->he1->wloop->lface->faceno,
                                    e->he2->wloop->lface->faceno);
                            }
                        }
                    }while((he = he->nxt) != l->ledg);
                    l = l->nextl;
                }
                f = f->nextf;
            }
        }
        printf("\n");
}


EXPORT void print_solid(
         CSG_BLK_CRX   *blk_crx)
{
         CSG_Solid     *s;
         int           i = 0;

         if(blk_crx == NULL)
         {
             printf("NULL CSG_BLK_CRX sent in print_solid()\n");
             return;
         }
         s = blk_crx->s;
         if(s == NULL)
         {
             printf("\nNO CSG Solid in CSG_BLK_CRX<%3d %3d> %d\n",
             blk_crx->icrds[0][0][0].csg_pt.icrds[0],
             blk_crx->icrds[0][0][0].csg_pt.icrds[1], blk_crx);
             return;
         }
         else
         {
             while(s)
             {
                 s = s->nexts;
                 i++;
             }
             printf("\nIN CSG_BLK_CRX<%3d, %3d> %d, there are %d solids\n",
             blk_crx->icrds[0][0][0].csg_pt.icrds[0],
             blk_crx->icrds[0][0][0].csg_pt.icrds[1], blk_crx, i);
             s = blk_crx->s;
             while(s)
             {
                 solidls(s, 2);
                 s = s->nexts;
             }
         }
         printf("End of print CSG_Solid in CSG_BLK_CRX %d\n", blk_crx);
}


EXPORT void print_CSG_point(
         CSG_POINT     *v)
{
         float         p[3];
         int           icrds[MAXD+1];

         CSG_point_Coord(v, p);
         if(v->type == IS_ICRDS)
         {
              icrds[0] = v->csg_pt.icrds[0];
              icrds[1] = v->csg_pt.icrds[1];
              icrds[2] = v->csg_pt.icrds[2];
              printf("icrds <%3d, %3d, %3d>", icrds[0], icrds[1], icrds[2]);
              printf(" crds <%f, %f, %f>\n", p[0], p[1], p[2]);
         }
         else
              printf("crds <%f, %f, %f>\n", p[0], p[1], p[2]);
}



EXPORT void CSG_point_Coord(
        CSG_POINT       *v,
        float           *p)
{
        if(v->type == IS_ICRDS)
        {
            p[0] = Coords(Regular_comp_grid_node(
                     v->csg_pt.icrds, local_ntg))[0];
            p[1] = Coords(Regular_comp_grid_node(
                     v->csg_pt.icrds, local_ntg))[1];
            if(v->csg_pt.icrds[2] == 1)
                p[2] = dt_step;
            else
                p[2] = 0;
        }
        else if(v->type == IS_BBI_P)
        {
            p[0] = Coords(v->csg_pt.crx->pt)[0];
            p[1] = Coords(v->csg_pt.crx->pt)[1];
            p[2] = Coords(v->csg_pt.crx->pt)[2];
        }
        else
        {
            printf("ERROR: IN CSG_point_Coord(), NO valid point %d", v);
            clean_up(ERROR);
        }
}

LIB_LOCAL float larea(
        CSG_Loop        *l,
        float           *normal,
        TRI_GRID        *ntg,
        float           dt)
{
        CSG_HalfEdge    *he;
        CSG_Vertex      *v1, *v2, *v3;
        float           p1[3], p2[3], p3[3], norm[3];
        float           aa[3], bb[3], cc[3], dd[3];
        float           area;

        dd[0] = dd[1] = dd[2] = 0.0;
        he = l->ledg;
        v1 = he->vtx;
        he = he->nxt;
        do
        {
            v2 = he->vtx;
            v3 = he->nxt->vtx;
            if(v1->pt->type == IS_ICRDS)
            {
                p1[2] = Coords(Regular_comp_grid_node(v1->pt->csg_pt.icrds, ntg))[0];
                p1[0] = Coords(Regular_comp_grid_node(v1->pt->csg_pt.icrds, ntg))[1];
                if(v1->pt->csg_pt.icrds[2] == 1)
                    p1[1] = dt;
                else
                    p1[1] = 0;
            }
            else
            {
                p1[2] = Coords(v1->pt->csg_pt.crx->pt)[0];
                p1[0] = Coords(v1->pt->csg_pt.crx->pt)[1];
                p1[1] = Coords(v1->pt->csg_pt.crx->pt)[2];
            }
            if(v2->pt->type == IS_ICRDS)
            {
                p2[2] = Coords(Regular_comp_grid_node(v2->pt->csg_pt.icrds, ntg))[0];
                p2[0] = Coords(Regular_comp_grid_node(v2->pt->csg_pt.icrds, ntg))[1];
                if(v2->pt->csg_pt.icrds[2] == 1)
                    p2[1] = dt;
                else
                    p2[1] = 0;
            }
            else
            {
                p2[2] = Coords(v2->pt->csg_pt.crx->pt)[0];
                p2[0] = Coords(v2->pt->csg_pt.crx->pt)[1];
                p2[1] = Coords(v2->pt->csg_pt.crx->pt)[2];
            }
            if(v3->pt->type == IS_ICRDS)
            {
                p3[2] = Coords(Regular_comp_grid_node(v3->pt->csg_pt.icrds, ntg))[0];
                p3[0] = Coords(Regular_comp_grid_node(v3->pt->csg_pt.icrds, ntg))[1];
                if(v3->pt->csg_pt.icrds[2] == 1)
                    p3[1] = dt;
                else
                    p3[1] = 0;
            }
            else
            {
                p3[2] = Coords(v3->pt->csg_pt.crx->pt)[0];
                p3[0] = Coords(v3->pt->csg_pt.crx->pt)[1];
                p3[1] = Coords(v3->pt->csg_pt.crx->pt)[2];
            }
            aa[0] = p2[0] - p1[0];
            aa[1] = p2[1] - p1[1];
            aa[2] = p2[2] - p1[2];
            bb[0] = p3[0] - p1[0];
            bb[1] = p3[1] - p1[1];
            bb[2] = p3[2] - p1[2];
            vector_product(aa, bb, cc, 3);
            dd[0] += cc[0];
            dd[1] += cc[1];
            dd[2] += cc[2];
            /*
            printf(" cc <%g, %g, %g>\n",
                  cc[0], cc[1], cc[2]);
            */
        }while((he = he->nxt) != l->ledg);
        norm[0] = normal[1];
        norm[1] = normal[2];
        norm[2] = normal[0];
        /*
        printf(" dd <%g, %g, %g>, norm <%g, %g, %g>\n",
             dd[0], dd[1], dd[2], norm[0], norm[1], norm[2]);
        */
        area = 0.5*scalar_product(norm, dd, 3);

        /*
        if(area < TOL*TOL*dt)
        {
            print_csg_blk_crx(l->lface->fsolid->blk);
            solidls(l->lface->fsolid, 2);
            printf("ERROR: In larea(), face no<%d> too small, area = %g\n",
            l->lface->faceno, area);
            clean_up(ERROR);
        }
        */
        return (area);
}

LIB_LOCAL void polygon_centroid_3d(
        CSG_Loop        *l)
{
        CSG_HalfEdge    *he;
        static float    **pt = NULL;
        static int      saved_nv = 0; 

        float           cx, cy, cz, areat, area; 
        int             i, nv = 0;  

        he = l->ledg;
        do
        {
            nv++;   
        }while((he = he->nxt) != l->ledg);
          
        if(pt == NULL || saved_nv < nv)
        {
            saved_nv = nv; 
            if(pt != NULL) free(pt); 
            matrix(&pt, nv, 3, sizeof(float));
        }
 
        he = l->ledg; i = 0; 
        do
        {
            CSG_point_Coord(he->vtx->pt,pt[i]); 
            i++; 
        }while((he = he->nxt) != l->ledg);

        cx = cy = cz = area = 0.0;
        for(i = 0; i < nv-2; i++)
        {
            areat = triangle_area_3d(pt[i], pt[i+1], pt[nv-1]);
            area += areat;    
            cx += areat * (pt[i][0] + pt[i+1][0] + pt[nv-1][0]) / 3.0;
            cy += areat * (pt[i][1] + pt[i+1][1] + pt[nv-1][1]) / 3.0;
            cz += areat * (pt[i][2] + pt[i+1][2] + pt[nv-1][2]) / 3.0;
        }
 
        cx = cx / area;
        cy = cy / area;
        cz = cz / area;

        l->lface->centroid[0] = cx;
        l->lface->centroid[1] = cy;
        l->lface->centroid[2] = cz;
}

/***
LOCAL float triangle_area_3d(
       float   *p1,
       float   *p2,
       float   *p3)
{
       float a;
       float alpha;
       float area;
       float b;
       float base;
       float c;
       float dot;
       float height;

       dot =
        (p2[0] - p1[0]) * (p3[0] - p1[0]) +
        (p2[1] - p1[1]) * (p3[1] - p1[1]) +
        (p2[2] - p1[2]) * (p3[2] - p1[2]);

       base = enorm0_3d(p1, p2);
       if (base == 0.0)
       {
           height = 0.0;
       }
       else
       {
           alpha = dot / ( base * base );
           a = p3[0] - p1[0] - alpha * (p2[0] - p1[0]);
           b = p3[1] - p1[1] - alpha * (p2[1] - p1[1]);
           c = p3[2] - p1[2] - alpha * (p2[2] - p1[2]);

           height = sqrt ( a * a + b * b + c * c );
       }
       area = 0.5 * base * height;
       return area;
}
***/

/**
LOCAL float enorm0_3d(
       float   *p0,
       float   *p1)
{
       float value;

       value = sqrt(
         sqr(p1[0] - p0[0])  +
         sqr(p1[1] - p0[1])  +
         sqr(p1[2] - p0[2]));

       return value;
}
*/

EXPORT int point_in_face2d(
        CSG_Loop        *l,
        float           *point)
{
        CSG_HalfEdge    *he;
        static float    **pt = NULL;
        static int      saved_nv = 0;

        float           cx, cy, cz, areat, area;
        int             i, nv = 0;

        he = l->ledg;
        do
        {
            nv++;
        }while((he = he->nxt) != l->ledg);

        if(pt == NULL || saved_nv < nv)
        {
            saved_nv = nv;
            if(pt != NULL) free(pt);
            matrix(&pt, nv, 3, sizeof(float));
        }

        he = l->ledg; i = 0;
        do
        {
            CSG_point_Coord(he->vtx->pt,pt[i]);
            i++;
        }while((he = he->nxt) != l->ledg);

        return point_in_polygon_2d(pt, nv, point);  
}

LOCAL int  point_in_polygon_2d(
        float      **pt,
        int        N,
        float      *point)
{
        int     counter = 0;
        int     i;
        float   xinters;
        float   *p1, *p2;

        p1 = pt[0];
        for (i=1;i<=N;i++)
        {
            p2 = pt[i % N];
            if (point[1] > min(p1[1],p2[1]))
            {
                if (point[1] <= max(p1[1],p2[1])) 
                {
                    if (point[0] <= max(p1[0],p2[0]))
                    {
                        if (p1[1] != p2[1])
                        {
                            xinters = (point[1]-p1[1])*(p2[0]-p1[0])/(p2[1]-p1[1])+p1[0];
                            if (p1[0] == p2[0] || point[0] <= xinters)
                                counter++;
                        }
                    }
                }
            }
            p1 = p2;
        }

        if (counter % 2 == 0)
            return NO;
        else
            return YES;         
} 


LIB_LOCAL float edge_len(
        CSG_HalfEdge    *he)
{
        CSG_POINT       *p1, *p2;
        float           pt1[3], pt2[3];

        p1 = he->vtx->pt;
        p2 = he->nxt->vtx->pt;

        CSG_point_Coord(p1, pt1);
        CSG_point_Coord(p2, pt2);

        return sqrt(sqr(pt2[0]-pt1[0])+
                sqr(pt2[1]-pt1[1])+sqr(pt2[2]-pt1[2]));
}

#endif /* if defined(TWOD) && defined(CONSERVATIVE_ALG) */
