/*
*				tri2d.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Two dimensional specific triangulation functions.
*	A triangulation of the computational region is constructed
*	from a rectangular grid and an interface.  The corners of
*	the triangles and quadrangles in the triangulation are either
*	intersections between grid lines of the rectangular grid or
*	intersections of grid lines with the interface.
*
*	TODO:
*		1. Rig up a mechanism for triangulating only selected
*			components.
*		2. Speed up area computations.
*/



#if defined(TWOD)
#include <tri/trilocaldecs.h>


	/* LOCAL Function Declarations */
LOCAL	COMPONENT	set_comp2d(COMPONENT,COMPONENT,COMPONENT,COMPONENT,
			             COMPONENT);
LOCAL	int	add_crx_to_list2d(TRI_GRID*,int*,int,CROSSING_DIRECTION,
				  POINT*,CURVE*,int*);
LOCAL	int	find_index_of_node(NODE*,INTERFACE*);
LOCAL	int	grid_crossings_on_segment(TRI_GRID*,float,float,float,float,
					  float,float,CURVE*,int*,int*,
					  RECT_GRID*,int);
LOCAL	int	inconsistent_cross2d(int,int,int,int,COMPONENT,COMPONENT,
				     COMPONENT,COMPONENT,COMPONENT,int,int,int,
				     int,int*,int*,int*,int*,TRI_GRID*,
				     INTERFACE*);
LOCAL	int	shift_from_cell_edge(float*,int,RECT_GRID*,float);
LOCAL	void	add_blk_node_to_list(BLK_EL1*,NODE*,TG_PT*);
LOCAL	void	count_crossings(TRI_GRID*,float,float,float,float,float,float,
				CURVE*,int*,RECT_GRID*,int);
#if defined(CONSERVATIVE_ALG)
LOCAL	int	add_comp_crx_to_list2d(TRI_GRID*,int*,int,CROSSING_DIRECTION,
				  POINT*,CURVE*,int*);
LOCAL   int     shift_from_cell_edge_on_comp(float*,int,RECT_GRID*,float);  
LOCAL   void    preserv_indx_on_insertion(CURVE*,BOND*);
#endif /* if defined(CONSERVATIVE_ALG) */
#if defined(DEBUG_TRI_GRID)
LOCAL	void	debug_comp_for_node(int,int,int,COMPONENT*,COMPONENT,COMPONENT,
				    COMPONENT,COMPONENT,int,int*,int,int*,int,
				    int*,int,int*,TRI_GRID*);
#endif /* defined(DEBUG_TRI_GRID) */
/*
*			insert_grid_crossings2d():
*
*	Inserts points in grid_intfc wherever a grid line intersects
*	the interface.  Stores these grid crossings in arrays indexed
*	by the appropriate grid line segment.
*
*	The algorithm for inserting grid crossings is taken from
*	merge_with_points_on_mesh_lines() in psgrid.c.  In this
*	algorithm grid_crossings_on_segment() is called for each
*	bond on the interface.  grid_crossings_on_segment() loads
*	the new_intfc_points[] array with the points to be inserted.
*	Then the insert_point_in_bond() is called to insert these
*	new points in grid_intfc.  The state associated with a new
*	point is expected to be calculated by USER_insert_point_in_bond,
*	presumably by interpolation along the bond.
*/


/*
*	LOCAL variables shared by insert_grid_crossings2d(),
*	grid_crossings_on_segment(), and related routines below.
*/

LOCAL	int	ix1, iy1, ix2, iy2;
LOCAL	float	max_lenx, max_leny;
LOCAL	float	*xx_grid = NULL, *yy_grid = NULL;
LOCAL	int	n_new_intfc_points = 0;
LOCAL	POINT	*new_intfc_points[50];/*POTENTIAL BUG, static array size*/
LOCAL	int	xmax, ymax;
LOCAL	float	XL, YL, XU, YU, hx, hy;



/*
*	Remark about tolerences in insert_grid_crossings2d().
*
*		The shifting of points on the interface to positions
*	at least some minimum distance from certain grid lines
*	has the potential of producing new tangles in the
*	hyp trigrid interface.  This problem can be expected
*	to occur when a node occupies a position that corresponds
*	to a corner of the dual lattice.  Since human nature
*	tends to think in terms of whole numbers,  it is
*	quite likely that stationary nodes such as sources and
*	sinks will be put in such locations.  In particular
*	runs that are being compared with finite difference
*	methods can be expected to have this property since
*	the wells will be placed at the centers of the computational
*	lattice and hence at the corners of the dual lattice.
*
*		An example occurs when a sink node with two curves
*	lies at a corner.  The node is shifted away from the
*	corner for the purposes of the component and triangulation
*	computation.  This shift causes the resulting grid crossing
*	of one of these curves to move out of the neighborhood of
*	of the nearest orthogonal grid line,  while the other curve 
*	crosses the same grid line as the first but lies inside this 
*	neighborhood.  This second point is then shifted by the tolerence
*	TOL*[hx or hy] to move it away from the corner of the grid
*	block,  but this shift can it past the other
*	crossing producing a new tangle.
*
*		Ideally,  the solution to this problem is to have
*	no (zero) tolerences,  but this will likely cause floating
*	problems and difficulties in ordering the grid crossings
*	around the grid blocks,  since it is possible for the
*	crossings to occur at corners (for example a 45 degree
*	Neumann boundary on a square mesh).   This problem can
*	be reduced by lowering the tolerences and requiring
*	that nodes be further shifted into the grid blocks than
*	regular grid crossings.  Other resolutions to this problem
*	will require complicated logic in order to detect and eliminate
*	(or never produce) these tangles.
*/

LOCAL	int	find_index_of_node(
	NODE		*node,
	INTERFACE	*intfc)
{
	NODE		**n;
	int		n_index = 0;
	
	for ( n = intfc->nodes; n && (*n != node); ++n)
	    ++n_index;

	return n_index;
}		/*end find_index_of_node*/

/*ARGSUSED*/
/* Note: the arg. intfc can be the interface other than
 * ntg->grid_intfc (flag which_grid = ON_DUAL_GRID); 
 * We can also use front->interf (flag which_grid = ON_COMP_GRID).
 * When calling insert_grid_crossings2d(), make sure
 * the tri storage is properly set.
 */
LIB_LOCAL	int insert_grid_crossings2d(
	TRI_GRID	*ntg,
	int		**ic_of_node,
        INTERFACE       *intfc, 
        RECT_GRID       *rgr,
        int             which_grid)
{
	register CURVE	**c;
	register BOND	*b;
        /*
	INTERFACE	*grid_intfc = ntg->grid_intfc;
	RECT_GRID	*rgr = &ntg->rect_grid;
        */
	NODE		**node;
	int		n_crx;
	int		crx_num;
	int		i, n_index;
	int		First_cross_on_curve;
	int		status;

        if(which_grid == ON_DUAL_GRID)
	    n_crx = ntg->n_crx;
#if defined(CONSERVATIVE_ALG)
        else
           n_crx = ntg->n_c_crx;
#endif /* if defined(CONSERVATIVE_ALG) */

			/* record crossings */
	crx_num = -1;
	for (c = intfc->curves; c && *c;  ++c)
	{
	    First_cross_on_curve = YES;
	    b = (*c)->first;
	    n_index = find_index_of_node((*c)->start,intfc);
	    ix2 = ic_of_node[n_index][0];
	    iy2 = ic_of_node[n_index][1];
	    while (b != NULL)
	    {
	    	n_new_intfc_points = 0;
	    	ix1 = ix2;		iy1 = iy2;
	    	if (b != (*c)->last)
	    	{
	    	    ix2 = cell_index(Coords(b->end)[0],0,rgr);
	    	    iy2 = cell_index(Coords(b->end)[1],1,rgr);
	    	}
	       	else
	    	{
	    	    n_index = find_index_of_node((*c)->end,intfc);
	    	    ix2 = ic_of_node[n_index][0];
	    	    iy2 = ic_of_node[n_index][1];
	    	}

#if defined(DEBUG_TRI_GRID)
	    	if (debugging("grd_crx"))
	    	{
	    	    (void) printf("Searching for grid crossings on bond %llu\n",
	    			  bond_number(b,intfc));
	    	    print_bond(b);
	    	}
#endif /* defined(DEBUG_TRI_GRID) */

	    	status = grid_crossings_on_segment(ntg,
				Coords(b->start)[0],Coords(b->start)[1],
				Coords(b->end)[0],Coords(b->end)[1],
				fabs(Coords(b->end)[0] - Coords(b->start)[0]),
				fabs(Coords(b->end)[1] - Coords(b->start)[1]),
				*c,&crx_num,&First_cross_on_curve,rgr,which_grid);

	    	if (status != GOOD_STEP) 
	    	{
	    	    return status;
	    	}

#if defined(CONSERVATIVE_ALG)
                if(which_grid == ON_COMP_GRID)
                {
                    if((! is_passive_boundary(Hyper_surf(*c))) &&
                       (! is_subdomain_boundary(Hyper_surf(*c))) )
                    {
                        preserv_indx_on_insertion(*c, b);
                    }
                }
#endif /* if defined(CONSERVATIVE_ALG)  */

	    	for (i = 0;  i < n_new_intfc_points;  ++i)
	    	{
	    	    if (insert_point_in_bond(new_intfc_points[i],b,*c) !=
			FUNCTION_SUCCEEDED)
	            {
	                screen("ERROR in insert_grid_crossings2d(), "
		               "insert_point_in_bond() failed\n");
	                clean_up(ERROR);
	            }
	    	    b = b->next;
	    	}
	    	b = b->next;
	    }
	}

	if (crx_num > 0)
        {
            if(which_grid == ON_DUAL_GRID)
	        ntg->crx_store[crx_num].end = YES;
#if defined(CONSERVATIVE_ALG)
            else
	        ntg->c_crx_store[crx_num].end = YES;
#endif /* if defined(CONSERVATIVE_ALG) */
        }

		/* identify mesh squares having required interior nodes */

        if(which_grid == ON_DUAL_GRID)
        { 
	    i = (xmax+1)*(ymax+1) + n_crx;
	    ntg->n_nodes = 0;
	    for (node = intfc->nodes; node && *node;  ++node)
    	    {
	        BLK_EL1 *blk_el1;

	        n_index = find_index_of_node(*node,intfc);
	        if ((((*node)->in_curves != NULL) || ((*node)->out_curves != NULL))
		    		 &&
	            ((Coords((*node)->posn)[0] > XL) &&
	             (Coords((*node)->posn)[0] < XU) &&
	             (Coords((*node)->posn)[1] > YL) &&
	             (Coords((*node)->posn)[1] < YU)))
	        {
	            blk_el1 = &Blk_el1(ic_of_node[n_index],ntg);
	            ++blk_el1->interior_node;
	            Coords(ntg->node_points+i)[0] = Coords((*node)->posn)[0];
	            Coords(ntg->node_points+i)[1] = Coords((*node)->posn)[1];
	            add_blk_node_to_list(blk_el1,*node,&(ntg->node_points[i]));
	            ++i;
	            ++(ntg->n_nodes);
	        }
	    }
        }

	if (crx_num != n_crx-1)
	{
            (void) printf("WARNING on grid %s\n",
                          which_grid == ON_DUAL_GRID ? "ON_DUAL_GRID":"ON_COMP_GRID"); 
	    (void) printf("WARNING in insert_grid_crossings2d(), "
		          "inconsistent cross count, crx_num %d, n_crx %d\n",
			  crx_num,n_crx);
	    return MODIFY_TIME_STEP;
	}

#if defined(DEBUG_TRI_GRID)
	if (debugging("grd_crx") && which_grid == ON_DUAL_GRID)
	{
	    (void) printf("After insert_grid_crossings2d()\n");
	    print_seg_crx_count(ntg,n_crx);
	    print_crx_store(ntg,n_crx);
	    print_seg_crx_lists_store(ntg,n_crx);
	    print_seg_crx_lists(ntg);
	    print_blk_els0(ntg);
	    print_blk_els1(ntg);
	}
#endif /* defined(DEBUG_TRI_GRID) */

	return GOOD_STEP;
}		/*end insert_grid_crossings2d*/

LIB_LOCAL int count_intfc_crossings_of_grid_lattice2d(
	TRI_GRID	*ntg,
	INTERFACE	*intfc,
	RECT_GRID	*rgr,
	int		**ic_of_node,
        int             which_grid)
{
	register CURVE	**c;
	register BOND	*b;
	float		tolx, toly;
	int		n_index;
	int		n_crx = 0;

	max_lenx = ONEMTOL*hx;		max_leny = ONEMTOL*hy;
	xx_grid = rgr->edges[0];	yy_grid = rgr->edges[1];
	tolx = TOL * hx;		toly = TOL * hy;

			/* count intfc crossings of dual lattice */

	for (c = intfc->curves; c && *c;  ++c)
	{
	    b = (*c)->first;
	    n_index = find_index_of_node((*c)->start,intfc);
	    ix2 = ic_of_node[n_index][0];
	    iy2 = ic_of_node[n_index][1];
	    while (b != NULL)
	    {
	    	ix1 = ix2;		iy1 = iy2;

	    	if (b != (*c)->last)
	    	{
#if defined(CONSERVATIVE_ALG)
                    if(which_grid == ON_COMP_GRID)
                    {
	    	        ix2 = shift_from_cell_edge_on_comp(Coords(b->end),0,rgr,tolx);
		        iy2 = shift_from_cell_edge_on_comp(Coords(b->end),1,rgr,toly);
                    }
                    else
#endif /* if defined(CONSERVATIVE_ALG) */
                    {
	    	        ix2 = shift_from_cell_edge(Coords(b->end),0,rgr,tolx);
		        iy2 = shift_from_cell_edge(Coords(b->end),1,rgr,toly);
                    }
		}
		else
		{
		    n_index = find_index_of_node((*c)->end,intfc);
		    ix2 = ic_of_node[n_index][0];
		    iy2 = ic_of_node[n_index][1];
		}

		count_crossings(ntg,Coords(b->start)[0],Coords(b->start)[1],
				Coords(b->end)[0],Coords(b->end)[1],
				fabs(Coords(b->end)[0] - Coords(b->start)[0]),
				fabs(Coords(b->end)[1] - Coords(b->start)[1]),
				*c,&n_crx,rgr,which_grid);
		b = b->next;
	    }
	}
	return n_crx;
}		/*end count_intfc_crossings_of_grid_lattice2d*/


#define north_cross_on_grid(iy,ymax) (0 <= (iy) && (iy) <  (ymax))
#define south_cross_on_grid(iy,ymax) (0 <  (iy) && (iy) <= (ymax))
#define east_cross_on_grid(ix,xmax)  (0 <= (ix) && (ix) <  (xmax))
#define west_cross_on_grid(ix,xmax)  (0 <  (ix) && (ix) <= (xmax))

/*
*	Counts total grid crossings found. For use in initializing storage.
*/

LOCAL	void count_crossings(
	TRI_GRID	*ntg,
	float		x1,
	float		y1,
	float		x2,
	float		y2,
	float		lenx,
	float		leny,
	CURVE		*curve,
	int		*n_crx,
	RECT_GRID	*rgr,
        int             which_grid)
{
	register int	ix, iy;

	float		p3[MAXD], x_h, y_h, x_v, y_v;
	float		tolx, toly, wtolx, wtoly;
	int		msk, signf, signx;
 
        int             *seg_crx_count; 
#if defined(DEBUG_TRI_GRID)
	int		deb_grd_crx = NO;
#endif /* defined(DEBUG_TRI_GRID) */

#if defined(DEBUG_TRI_GRID)
	if (debugging("count_crossings"))
	{
	    deb_grd_crx = YES;
	    (void) printf("\ncount_crossings() for bond (piece) ");
	    (void) printf("%g %g -> %g %g\n",x1,y1,x2,y2);
	}
#endif /* defined(DEBUG_TRI_GRID) */


        if(which_grid == ON_DUAL_GRID)
            seg_crx_count = ntg->seg_crx_count;
#if defined(CONSERVATIVE_ALG)
        else
            seg_crx_count = ntg->c_seg_crx_count;
#endif /* if defined(CONSERVATIVE_ALG) */

		/* if bond is too long, recursively cut off */
		/* hunks with lenx > max_lenx or leny > max_leny */

	tolx = TOL * hx;	toly = TOL * hy;
	if ((lenx > max_lenx) || (leny > max_leny))
	{
	    int	save_ix2, save_iy2;
	    float	s0, s1;

	    s0 = 1.0;
	    if (lenx > max_lenx)
		s0 = max_lenx/lenx;
	    if (leny > max_leny)
		s0 = min(s0,max_leny/leny);
	    s1 = 1.0 - s0;

	    p3[0] = x2 - s0 * (x2-x1);	p3[1] = y2 - s0 * (y2-y1);
	    save_ix2 = ix2;			save_iy2 = iy2;

#if defined(CONSERVATIVE_ALG)
            if(which_grid == ON_COMP_GRID)
            {
                ix2 = shift_from_cell_edge_on_comp(p3,0,rgr,tolx);
                iy2 = shift_from_cell_edge_on_comp(p3,1,rgr,toly);
            }
            else
#endif /* if defined(CONSERVATIVE_ALG) */
            { 
	        ix2 = shift_from_cell_edge(p3,0,rgr,tolx);
	        iy2 = shift_from_cell_edge(p3,1,rgr,toly);
            }

	    count_crossings(ntg,x1,y1,p3[0],p3[1],s1*lenx,s1*leny,
			    curve,n_crx,rgr,which_grid);
	    ix1 = ix2; 		iy1 = iy2;
	    ix2 = save_ix2; 	iy2 = save_iy2;
	    x1  = p3[0];		y1  = p3[1];
	}


	if ((iy1 != iy2) && (ix1 != ix2))    /* crossings of both vertical */
	{				      /* and horizontal grid bonds */

	    iy = max(iy1,iy2);	ix = max(ix1,ix2);
	    y_h = yy_grid[iy];
	    x_h = x1 + (x2 - x1) * (y_h - y1) / (y2 - y1);
	    x_v = xx_grid[ix];
	    y_v = y1 + (y2 - y1) * (x_v - x1) / (x2 - x1);

#if defined(DEBUG_TRI_GRID)
	    if (deb_grd_crx)
	    {
	    	(void) printf("bond seg crosses horiz at %g %g ",x_h,y_h);
	    	(void) printf("and vert at %g %g\n",x_v,y_v);
	    }
#endif /* defined(DEBUG_TRI_GRID) */

	    	/* test if too close to grid crossings */

	    wtolx = x_h - x_v;		wtoly = y_v - y_h;

	    signf = ((ix1-ix2)*(iy1-iy2) > 0) ? 1 : -1;
	    signx = 0;
#if defined(DEBUG_TRI_GRID)
	    if (deb_grd_crx)
	    	(void) printf("wtolx %g wtoly %g signf %d signx %d\n",
			      wtolx,wtoly,signf,signx);
#endif /* defined(DEBUG_TRI_GRID) */
	    if (fabs(wtolx) < tolx)
	    {
	    	if (wtolx >= 0)
	    	{
	    	    x_h += tolx;
	    	    signx =  1;
	    	}
	    	else
	    	{
	    	    x_h -= tolx;
	    	    signx = -1;
	    	}
	    }
	    if (fabs(wtoly) < toly)
	    {
	    	if (signx != 0)
	    	    y_v -= signx * signf * toly;
		else
		{
		    if (wtoly >= 0)
			y_v += toly;
		    else
			y_v -= toly;
		}	
	    }
#if defined(DEBUG_TRI_GRID)
	    if (deb_grd_crx)
	    {
	    	(void) printf("after pot'l shift horz %g %g  vert %g %g\n",
			      x_h,y_h,x_v,y_v);
	    }
#endif /* defined(DEBUG_TRI_GRID) */

	    msk = 2*iy*xmax + iy + ix;
	    if (x_h <= x_v)
	    {
	    	if (west_cross_on_grid(ix,xmax))
	    	{
	    	    ++(*n_crx);
	    	    ++seg_crx_count[msk-1];
	    	}
	    	if ((ix2 - ix1)*(iy2 - iy1) > 0)
	    	{
	    	    if (north_cross_on_grid(iy,ymax))
	    	    {
	    	    	++(*n_crx);
	    	    	++seg_crx_count[msk+xmax];
	    	    }
	    	}
	    	else
	    	{
	    	    if (south_cross_on_grid(iy,ymax))
	    	    {
	    	    	++(*n_crx);
	    	    	++seg_crx_count[msk-xmax-1];
	    	    }
	    	}
	    }
	    else
	    {
	    	if (east_cross_on_grid(ix,xmax))
	    	{
	    	    ++(*n_crx);
	    	    ++seg_crx_count[msk];
	    	}
	    	if ((ix2 - ix1)*(iy2 - iy1) > 0)
	    	{
	    	    if (south_cross_on_grid(iy,ymax))
	    	    {
	    	    	++(*n_crx);
	    	    	++seg_crx_count[msk-xmax-1];
	    	    }
	    	}
	    	else
	    	{
	    	    if (north_cross_on_grid(iy,ymax))
	    	    {
	    	    	++(*n_crx);
	    	    	++seg_crx_count[msk+xmax];
	    	    }
	    	}
	    }
	}
	else if (iy1 != iy2 && east_cross_on_grid(ix1,xmax))
	{
		/* crossing of horizontal grid bond */

	    ++(*n_crx);
	    iy = max(iy1,iy2);
	    msk = 2*iy*xmax + iy + ix1;
	    ++seg_crx_count[msk];
	}
	else if (ix1 != ix2 && north_cross_on_grid(iy1,ymax))
	{
	    /* crossing of vertical grid bond */

	    ++(*n_crx);
	    ix = max(ix1,ix2);
	    msk = 2*iy1*xmax+iy1+ix+xmax;
	    ++seg_crx_count[msk];
	}
}		/*end count_crossings*/



/*
*	Recursively inserts the grid crossings for each bond of the intfc
*/

LOCAL	int grid_crossings_on_segment(
	TRI_GRID	*ntg,
	float		x1,
	float		y1,
	float		x2,
	float		y2,
	float		lenx,
	float		leny,
	CURVE		*curve,
	int		*crx_num,
	int		*fcoc,
	RECT_GRID	*rgr,
        int             which_grid)
{
	register int	ix, iy;
	register POINT	*p_h, *p_v;

	float	           p3[MAXD], x_h, y_h, x_v, y_v;
	float	           coords_h[MAXD], coords_v[MAXD];
	float	           tolx, toly, wtolx, wtoly;
	bool            weight;
	CROSSING_DIRECTION dir_v, dir_h;
	int 	           signf, signx, msk, status;
#if defined(DEBUG_TRI_GRID)
	int		deb_grd_crx = NO;
#endif /* defined(DEBUG_TRI_GRID) */

#if defined(DEBUG_TRI_GRID)
	if (debugging("grd_crx")) deb_grd_crx = YES;
#endif /* defined(DEBUG_TRI_GRID) */

		/* if bond is too long, recursively cut off */
		/* hunks with lenx > max_lenx or leny > max_leny */

#if defined(DEBUG_TRI_GRID)
	if (debugging("grd_crx"))
	{
	    (void) printf("\ngrid_crossings_on_segment() ");
	    (void) printf("for bond (piece) %g %g -> %g %g\n",x1,y1,x2,y2);
	    (void) printf("(ix1, iy1) = (%d, %d), (ix2, iy2) = (%d, %d)\n",
			ix1,iy1,ix2,iy2);
	}
#endif /* defined(DEBUG_TRI_GRID) */

	tolx = TOL * hx;	toly = TOL * hy;
	if ((lenx > max_lenx) || (leny > max_leny))
	{
	    int	save_ix2, save_iy2;
	    float	s0, s1;

#if defined(DEBUG_TRI_GRID)
	    if (debugging("grd_crx"))
	    {
	    	(void) printf("bond (piece) %g %g -> %g %g, ",x1,y1,x2,y2);
	    	(void) printf("too long, subdividing\n");
	    }
#endif /* defined(DEBUG_TRI_GRID) */

	    s0 = 1.0;
	    if (lenx > max_lenx)
		s0 = max_lenx/lenx;
	    if (leny > max_leny)
		s0 = min(s0,max_leny/leny);
	    s1 = 1.0 - s0;

	    p3[0] = x2 - s0 * (x2-x1);	p3[1] = y2 - s0 * (y2-y1);
	    save_ix2 = ix2;			save_iy2 = iy2;

#if defined(CONSERVATIVE_ALG)
            if(which_grid == ON_COMP_GRID)
            {
	        ix2 = shift_from_cell_edge_on_comp(p3,0,rgr,tolx);
	        iy2 = shift_from_cell_edge_on_comp(p3,1,rgr,toly);
            }
            else
#endif /* if defined(CONSERVATIVE_ALG) */
            {
	        ix2 = shift_from_cell_edge(p3,0,rgr,tolx);
	        iy2 = shift_from_cell_edge(p3,1,rgr,toly);
            }

	    status = grid_crossings_on_segment(ntg,x1,y1,p3[0],p3[1],s1*lenx,
					       s1*leny,curve,crx_num,fcoc,rgr,which_grid);

	    if (status != GOOD_STEP)
		return status;

	    ix1 = ix2;		iy1 = iy2;
	    ix2 = save_ix2;	iy2 = save_iy2;
	    x1  = p3[0];	y1  = p3[1];
	}

#if defined(DEBUG_TRI_GRID)
	if (debugging("grd_crx"))
	{
	    (void) printf("Finding crossings on bond (piece) %g %g -> %g %g\n",
								x1,y1,x2,y2);
	    (void) printf("(ix1, iy1) = (%d, %d), (ix2, iy2) = (%d, %d)\n",
			  ix1,iy1,ix2,iy2);
	}
#endif /* defined(DEBUG_TRI_GRID) */


	if ((iy1 != iy2) && (ix1 != ix2))    /* crossings of both vertical */
	{				       /* and horizontal grid bonds */
	    iy = max(iy1,iy2);
	    y_h = yy_grid[iy];
	    x_h = x1 + (x2 - x1) * (y_h - y1) / (y2 - y1);

	    dir_h = (iy1 < iy2) ? BELOW_TO_ABOVE : ABOVE_TO_BELOW;

	    ix = max(ix1,ix2);
	    x_v = xx_grid[ix];
	    y_v = y1 + (y2 - y1) * (x_v - x1) / (x2 - x1);

	    dir_v = (ix1 < ix2) ? LEFT_TO_RIGHT : RIGHT_TO_LEFT;

#if defined(DEBUG_TRI_GRID)
	    if (deb_grd_crx)
	    {
	    	(void) printf("bond seg crosses horiz at %g %g ",x_h,y_h);
	    	(void) printf("and vert at %g %g\n",x_v,y_v);
	    	(void) printf("Grid node (%d, %d)\n",ix,iy);
	    }
#endif /* defined(DEBUG_TRI_GRID) */


	    	/* test if too close to grid crossings */

	    wtolx = x_h - x_v;		wtoly = y_v - y_h;

	    signf = ((ix1-ix2)*(iy1-iy2) > 0) ? 1 : -1;
	    signx =  0;

	    if (fabs(wtolx) < tolx)
	    {
	    	if (wtolx >= 0)
	    	{
	    	    x_h += tolx;
	    	    signx =  1;
	    	}
	    	else
	    	{
	    	    x_h -= tolx;
	    	    signx = -1;
	    	}
	    }
	    if (fabs(wtoly) < toly)
	    {
	    	if (signx != 0)
	    	    y_v -= signx * signf * toly;
	    	else
	    	{
	    	    if (wtoly >= 0)
			y_v += toly;
	    	    else
			y_v -= toly;
	    	}	
	    }
#if defined(DEBUG_TRI_GRID)
	    if (deb_grd_crx)
	    {
	    	(void) printf("wtolx %g wtoly %g signf %d signx %d\n",
			      wtolx,wtoly,signf,signx);
	    	(void) printf("after pot'l shift horz %g %g  vert %g %g\n",
			      x_h,y_h,x_v,y_v);
	    }
#endif /* defined(DEBUG_TRI_GRID) */

		/* NOTE: weight is a bool    	   */
		/* NOTE: for the rest of this else{} loop  */
		/*	 x_h,y_h and x_v,y_v must be equiv */
		/*	 expressions to p_h and p_v	   */

	    weight = ((sqr(x_h-x1)+sqr(y_h-y1)) > (sqr(x_v-x1)+sqr(y_v-y1))) ?
		     YES : NO;
	    coords_h[0] = x_h;	coords_h[1] = y_h;
	    coords_v[0] = x_v;	coords_v[1] = y_v;

	    msk = 2*iy*xmax + iy + ix;
	    if (x_h <= x_v)
	    {
	        if ((ix2 - ix1)*(iy2 - iy1) > 0)
	        {
	            if (weight == YES)
	            {
	    	        if (north_cross_on_grid(iy,ymax))
	    	        {
	    		    p_v = Point(coords_v);
	    		    new_intfc_points[n_new_intfc_points++] = p_v;
                            if(which_grid == ON_DUAL_GRID)
	    		        status = add_crx_to_list2d(ntg,crx_num,msk+xmax,
						       dir_v,p_v,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
	    		        status = add_comp_crx_to_list2d(ntg,crx_num,msk+xmax,
						       dir_v,p_v,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
	    		    if (status != GOOD_STEP)
				return status;
	    	        }
#if defined(DEBUG_TRI_GRID)
	    	        else if (deb_grd_crx)
	    	        {
	    		    (void) printf("Cross (%g, %g), not on grid\n",
	    			          x_v,y_v);
	    	        }
#endif /* defined(DEBUG_TRI_GRID) */

	    	        if (west_cross_on_grid(ix,xmax))
	    	        {
	    		    p_h = Point(coords_h);
	    		    new_intfc_points[n_new_intfc_points++] = p_h;
                            if(which_grid == ON_DUAL_GRID)
	    		        status = add_crx_to_list2d(ntg,crx_num,msk-1,dir_h,
						       p_h,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
	    		        status = add_comp_crx_to_list2d(ntg,crx_num,msk-1,dir_h,
						       p_h,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
	    		    if (status != GOOD_STEP)
				return status;
	    	        }
#if defined(DEBUG_TRI_GRID)
	    	        else if (deb_grd_crx)
	    	        {
	    		    (void) printf("Cross (%g, %g), not on grid\n",
				          x_h,y_h);
			}
#endif /* defined(DEBUG_TRI_GRID) */
		    }
		    else
		    {
		        if (west_cross_on_grid(ix,xmax))
		        {
		    	    p_h = Point(coords_h);
		    	    new_intfc_points[n_new_intfc_points++] = p_h;
                            if(which_grid == ON_DUAL_GRID)
		    	        status = add_crx_to_list2d(ntg,crx_num,msk-1,dir_h,
						       p_h,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
		    	        status = add_comp_crx_to_list2d(ntg,crx_num,msk-1,dir_h,
						       p_h,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
		            if (status != GOOD_STEP)
				return status;
		        }
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_h,y_h);
		        }
#endif /* defined(DEBUG_TRI_GRID) */

		        if (north_cross_on_grid(iy,ymax))
			{
			    p_v = Point(coords_v);
			    new_intfc_points[n_new_intfc_points++] = p_v;
                            if(which_grid == ON_DUAL_GRID)
			        status = add_crx_to_list2d(ntg,crx_num,msk+xmax,
						       dir_v,p_v,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
			        status = add_comp_crx_to_list2d(ntg,crx_num,msk+xmax,
						       dir_v,p_v,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
			    if (status != GOOD_STEP) return status;
			}
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_v,y_v);
			}
#endif /* defined(DEBUG_TRI_GRID) */
		    }
		}
		else
	        {
		    if (weight == YES)
		    {
		        if (south_cross_on_grid(iy,ymax))
		        {
		    	    p_v = Point(coords_v);
		    	    new_intfc_points[n_new_intfc_points++] = p_v;
                            if(which_grid == ON_DUAL_GRID)
		    	        status = add_crx_to_list2d(ntg,crx_num,msk-xmax-1,
						       dir_v,p_v,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
		    	        status = add_comp_crx_to_list2d(ntg,crx_num,msk-xmax-1,
						       dir_v,p_v,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
		    	    if (status != GOOD_STEP)
				return status;
		        }
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_v,y_v);
			}
#endif /* defined(DEBUG_TRI_GRID) */

			if (west_cross_on_grid(ix,xmax))
			{
			    p_h = Point(coords_h);
			    new_intfc_points[n_new_intfc_points++] = p_h;
                            if(which_grid == ON_DUAL_GRID)
			        status = add_crx_to_list2d(ntg,crx_num,msk-1,dir_h,
						       p_h,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
			        status = add_comp_crx_to_list2d(ntg,crx_num,msk-1,dir_h,
						       p_h,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
			    if (status != GOOD_STEP) return status;
			}
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_h,y_h);
			}
#endif /* defined(DEBUG_TRI_GRID) */
		    }
		    else
		    {
		        if (west_cross_on_grid(ix,xmax))
		        {
			    p_h = Point(coords_h);
			    new_intfc_points[n_new_intfc_points++] = p_h;
                            if(which_grid == ON_DUAL_GRID)
			        status = add_crx_to_list2d(ntg,crx_num,msk-1,dir_h,
						       p_h,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
			        status = add_comp_crx_to_list2d(ntg,crx_num,msk-1,dir_h,
						       p_h,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
			    if (status != GOOD_STEP)
				return status;
			}
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_h,y_h);
			}
#endif /* defined(DEBUG_TRI_GRID) */

			if (south_cross_on_grid(iy,ymax))
			{
			    p_v = Point(coords_v);
			    new_intfc_points[n_new_intfc_points++] = p_v;
                            if(which_grid == ON_DUAL_GRID)
			        status = add_crx_to_list2d(ntg,crx_num,msk-xmax-1,
						       dir_v,p_v,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
			        status = add_comp_crx_to_list2d(ntg,crx_num,msk-xmax-1,
						       dir_v,p_v,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
			    if (status != GOOD_STEP)
				return status;
			}
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_v,y_v);
			}
#endif /* defined(DEBUG_TRI_GRID) */
		    }
		}
	    }
	    else
	    {
	        if ((ix2 - ix1)*(iy2 - iy1) > 0)
	        {
	    	    if (weight == YES)
	    	    {
	    	        if (south_cross_on_grid(iy,ymax))
	    	        {
	    		    p_v = Point(coords_v);
	    		    new_intfc_points[n_new_intfc_points++] = p_v;
                            if(which_grid == ON_DUAL_GRID)
	    		        status = add_crx_to_list2d(ntg,crx_num,msk-xmax-1,
						       dir_v,p_v,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
	    		        status = add_comp_crx_to_list2d(ntg,crx_num,msk-xmax-1,
						       dir_v,p_v,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
	    		    if (status != GOOD_STEP)
				return status;
	    	        }
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
	    	        {
	    		    (void) printf("Cross (%g, %g), not on grid\n",
	    		                  x_v,y_v);
	    	        }
#endif /* defined(DEBUG_TRI_GRID) */

	    	        if (east_cross_on_grid(ix,xmax))
	    	        {
	    		    p_h = Point(coords_h);
	    		    new_intfc_points[n_new_intfc_points++] = p_h;
                            if(which_grid == ON_DUAL_GRID)
	    		        status = add_crx_to_list2d(ntg,crx_num,msk,dir_h,
						       p_h,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
	    		        status = add_comp_crx_to_list2d(ntg,crx_num,msk,dir_h,
						       p_h,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
			    if (status != GOOD_STEP)
				return status;
			}
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_h,y_h);
			}
#endif /* defined(DEBUG_TRI_GRID) */
		    }
		    else
		    {
		        if (east_cross_on_grid(ix,xmax))
		        {
			    p_h = Point(coords_h);
			    new_intfc_points[n_new_intfc_points++] = p_h;
                            if(which_grid == ON_DUAL_GRID)
			        status = add_crx_to_list2d(ntg,crx_num,msk,dir_h,
						       p_h,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
			        status = add_comp_crx_to_list2d(ntg,crx_num,msk,dir_h,
						       p_h,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
			    if (status != GOOD_STEP)
				return status;
			}
#if defined(DEBUG_TRI_GRID)
		        else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_h,y_h);
			}
#endif /* defined(DEBUG_TRI_GRID) */

			if (south_cross_on_grid(iy,ymax))
			{
			    p_v = Point(coords_v);
			    new_intfc_points[n_new_intfc_points++] = p_v;
                            if(which_grid == ON_DUAL_GRID)
			        status = add_crx_to_list2d(ntg,crx_num,msk-xmax-1,
						       dir_v,p_v,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
			        status = add_comp_crx_to_list2d(ntg,crx_num,msk-xmax-1,
						       dir_v,p_v,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
			    if (status != GOOD_STEP)
				return status;
			}
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_v,y_v);
			}
#endif /* defined(DEBUG_TRI_GRID) */
		    }
		}
		else
		{
		    if (weight == YES)
		    {
		        if (north_cross_on_grid(iy,ymax))
		        {
			    p_v = Point(coords_v);
			    new_intfc_points[n_new_intfc_points++] = p_v;
                            if(which_grid == ON_DUAL_GRID)
			        status = add_crx_to_list2d(ntg,crx_num,msk+xmax,
						       dir_v,p_v,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
			        status = add_comp_crx_to_list2d(ntg,crx_num,msk+xmax,
						       dir_v,p_v,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
			    if (status != GOOD_STEP)
				return status;
		        }
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_v,y_v);
			}
#endif /* defined(DEBUG_TRI_GRID) */

			if (east_cross_on_grid(ix,xmax))
			{
			    p_h = Point(coords_h);
			    new_intfc_points[n_new_intfc_points++] = p_h;
                            if(which_grid == ON_DUAL_GRID)
			        status = add_crx_to_list2d(ntg,crx_num,msk,dir_h,
						       p_h,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
			        status = add_comp_crx_to_list2d(ntg,crx_num,msk,dir_h,
						       p_h,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
			    if (status != GOOD_STEP)
				return status;
			}
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_h,y_h);
			}
#endif /* defined(DEBUG_TRI_GRID) */
		    }
		    else
		    {
		        if (east_cross_on_grid(ix,xmax))
		        {
		    	    p_h = Point(coords_h);
			    new_intfc_points[n_new_intfc_points++] = p_h;
                            if(which_grid == ON_DUAL_GRID)
			        status = add_crx_to_list2d(ntg,crx_num,msk,dir_h,
						       p_h,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
			        status = add_comp_crx_to_list2d(ntg,crx_num,msk,dir_h,
						       p_h,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
			    if (status != GOOD_STEP)
				return status;
			}
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_h,y_h);
			}
#endif /* defined(DEBUG_TRI_GRID) */

			if (north_cross_on_grid(iy,ymax))
			{
			    p_v = Point(coords_v);
			    new_intfc_points[n_new_intfc_points++] = p_v;
                            if(which_grid == ON_DUAL_GRID)
			        status = add_crx_to_list2d(ntg,crx_num,msk+xmax,
						       dir_v,p_v,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                            else
			        status = add_comp_crx_to_list2d(ntg,crx_num,msk+xmax,
						       dir_v,p_v,curve,fcoc);

#endif /* if defined(CONSERVATIVE_ALG) */
			    if (status != GOOD_STEP)
				return status;
			}
#if defined(DEBUG_TRI_GRID)
			else if (deb_grd_crx)
			{
			    (void) printf("Cross (%g, %g), not on grid\n",
				          x_v,y_v);
			}
#endif /* defined(DEBUG_TRI_GRID) */
		    }
	        }
	    }
	}
	else if (iy1 != iy2) 
	{
	    /* crossing of horizontal grid bond */

	    iy = max(iy1,iy2);
	    y_h = yy_grid[iy];	x_v = xx_grid[ix1];
	    x_h = x1 + (x2 - x1) * (y_h - y1) / (y2 - y1);

#if defined(DEBUG_TRI_GRID)
	    if (deb_grd_crx)
	    {
	    	(void) printf("bond seg crosses horiz at %g %g\n",x_h,y_h);
	    	(void) printf("Grid node (%d, %d)\n",ix1,iy);
	    }
#endif /* defined(DEBUG_TRI_GRID) */

	    wtolx = x_h - x_v;
	    if (fabs(wtolx) < tolx)
	    {
	    	if (wtolx > 0.0)
		    x_h += tolx;
	    	else
		    x_h -= tolx;
	    }

	    if (east_cross_on_grid(ix1,xmax))
	    {
	    	dir_h = (iy1 < iy2) ? BELOW_TO_ABOVE : ABOVE_TO_BELOW;
	    	coords_h[0] = x_h;
		coords_h[1] = y_h;
	    	p_h = Point(coords_h);
	    	new_intfc_points[n_new_intfc_points++] = p_h;
                if(which_grid == ON_DUAL_GRID)
	    	    status = add_crx_to_list2d(ntg,crx_num,2*iy*xmax+iy+ix1,dir_h,
					   p_h,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                else
	    	    status = add_comp_crx_to_list2d(ntg,crx_num,2*iy*xmax+iy+ix1,dir_h,
					   p_h,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */

	    	if (status != GOOD_STEP)
		    return status;
	    }
#if defined(DEBUG_TRI_GRID)
	    else if (deb_grd_crx)
	    {
	    	(void) printf("Cross (%g, %g), not on grid\n",x_h,y_h);
	    }
#endif /* defined(DEBUG_TRI_GRID) */
	}
	else if (ix1 != ix2)
	{
	    /* crossing of vertical bond grid */

	    ix = max(ix1,ix2);
	    x_v = xx_grid[ix];	y_h = yy_grid[iy1];
	    y_v = y1 + (y2 - y1) * (x_v - x1) / (x2 - x1);

#if defined(DEBUG_TRI_GRID)
	    if (deb_grd_crx)
	    {
	    	(void) printf("bond seg crosses vert at %g %g\n",x_v,y_v);
	    	(void) printf("Grid node (%d, %d)\n",ix,iy1);
	    }
#endif /* defined(DEBUG_TRI_GRID) */

	    wtoly = y_v - y_h;
	    if (fabs(wtoly) < toly)
	    {
	    	if (wtoly < 0.0)
		    y_v -= toly;
	    	else
		    y_v += toly;
	    }

	    if (north_cross_on_grid(iy1,ymax))
	    {
	    	dir_v = (ix1 < ix2) ? LEFT_TO_RIGHT : RIGHT_TO_LEFT;

	    	coords_v[0] = x_v;	coords_v[1] = y_v;
	    	p_v = Point(coords_v);
	    	new_intfc_points[n_new_intfc_points++] = p_v;
                if(which_grid == ON_DUAL_GRID)
	    	    status = add_crx_to_list2d(ntg,crx_num,2*iy1*xmax+iy1+ix+xmax,
					   dir_v,p_v,curve,fcoc);
#if defined(CONSERVATIVE_ALG)
                else
	    	    status = add_comp_crx_to_list2d(ntg,crx_num,2*iy1*xmax+iy1+ix+xmax,
					   dir_v,p_v,curve,fcoc);
#endif /* if defined(CONSERVATIVE_ALG) */
	    	if (status != GOOD_STEP)
		    return status;
	    }
#if defined(DEBUG_TRI_GRID)
	    else if (deb_grd_crx)
	    {
	    	(void) printf("Cross (%g, %g), not on grid\n",x_v,y_v);
	    }
#endif /* defined(DEBUG_TRI_GRID) */
	}
	return GOOD_STEP;
}		/*end grid_crossings_on_segment*/



LOCAL	int add_crx_to_list2d(
	TRI_GRID	   *ntg,
	int		   *n_crx,
	int		   msk,
	CROSSING_DIRECTION dir,
	POINT		   *p,
	CURVE		   *cur,
	int		   *First_cross_on_curve)
{
	register CRXING *cross;
	CRXING		*crx_store = ntg->crx_store;

	int		nx, i, j, k, hold, input;
	int		*start;

	++(*n_crx);

	cross = &(crx_store[*n_crx]);
	cross->hs = Hyper_surf(cur);
	cross->pt = p;
	cross->crossing_direction = dir;
	cross->crx_num = *n_crx;

	if (First_cross_on_curve != NULL)
	{
	    j = (ymax+1)*(xmax+1)+(*n_crx);
#if defined(DEBUG_TRI_GRID)
	    if (j >= ntg->n_node_points)
	    {
	    	(void) printf("WARNING in add_crx_to_list2d(), "
	    	              "too many node points\n"
	    	              "j %d, ntg->n_node_points %d\n",j,
	    	       ntg->n_node_points);
	    	return MODIFY_TIME_STEP;
	    }
#endif /* defined(DEBUG_TRI_GRID) */
	    Coords(ntg->node_points+j)[0] = Coords(p)[0];
	    Coords(ntg->node_points+j)[1] = Coords(p)[1];
	    cross->nd = &ntg->node_points[j];
	}

	if ((First_cross_on_curve != NULL) && (*First_cross_on_curve == YES))
	{
	    cross->end = YES;
	    if (*n_crx > 0)
	    {
	    	crx_store[*n_crx - 1].end = YES;
	    }
	    *First_cross_on_curve = NO;
	}
	else
	    cross->end = NO;

#if defined(DEBUG_TRI_GRID)
	if (debugging("grd_crx"))
	{
	    (void) printf("\nn_crx %d, msk %d\n",*n_crx,msk);
	    print_crxings(cross,NO);
	}
#endif /* defined(DEBUG_TRI_GRID) */

	nx = ntg->seg_crx_count[msk];
	if (nx == 0)
	{
	    (void) printf("WARNING in add_crx_to_list2d(), nx = 0");
	    return MODIFY_TIME_STEP;
	}

	start = ntg->seg_crx_lists[msk];

#if defined(DEBUG_TRI_GRID)
	if (debugging("grd_crx"))
	    (void) printf("nx %d start %p *start %d\n",
	    	          nx,(POINTER)start,*start);
#endif /* defined(DEBUG_TRI_GRID) */

	switch (dir)
	{
	case BELOW_TO_ABOVE:
	case ABOVE_TO_BELOW:
	    for (j = 0;  j < nx;  ++j)
	    {
	    	if (*(start+j) == -1) break;
	    }

	    if (j == 0)
	    	*start = *n_crx;
	    else
	    {
	    	for (i = 0;  i < j;  ++i)
	    	{
	    	    if (Coords((crx_store[*(start+i)]).pt)[0] >
						 Coords(cross->pt)[0])
			break;
		}

		if (i < j)
		{
		    input = *n_crx;
		    for (k = i;  k < j;  ++k)
		    {
		    	hold = *(start+k);
		    	*(start+k) = input;
		    	input = hold;
		    }
		    *(start+j) = input;
		}
		else
		    *(start+j) = *n_crx;
	    }
	    break;
	case LEFT_TO_RIGHT:
	case RIGHT_TO_LEFT:
	    for (j = 0;  j < nx;  ++j)
	    {
	    	if (*(start+j) == -1) break;
	    }

	    if (j == 0)
	    	*start = *n_crx;
	    else
	    {
	    	for (i = 0;  i < j;  ++i)
	    	{
	    	    if (Coords((crx_store[*(start+i)]).pt)[1] >
	    				 Coords(cross->pt)[1])
	    		    break;
	    	}

	    	if (i < j)
	    	{
	    	    input = *n_crx;
	    	    for (k = i;  k < j;  ++k)
	    	    {
	    	    	hold = *(start+k);
	    	    	*(start+k) = input;
	    	    	input = hold;
	    	    }
	    	    *(start+j) = input;
	    	}
	    	else
	    	    *(start+j) = *n_crx;
	    }
	    break;
	}
	return GOOD_STEP;
}		/*end add_crx_to_list2d*/

#if defined(CONSERVATIVE_ALG)

LOCAL	int add_comp_crx_to_list2d(
	TRI_GRID	   *ntg,
	int		   *n_crx,
	int		   msk,
	CROSSING_DIRECTION dir,
	POINT		   *p,
	CURVE		   *cur,
	int		   *First_cross_on_curve)
{
	register CRXING *cross;
	CRXING		*crx_store = ntg->c_crx_store;

	int		nx, i, j, k, hold, input;
	int		*start;

	++(*n_crx);

	cross = &(crx_store[*n_crx]);
	cross->hs = Hyper_surf(cur);
	cross->pt = p;
	cross->crossing_direction = dir;
	cross->crx_num = *n_crx;
        cross->lcomp = NO_COMP;
        cross->ucomp = NO_COMP;
        p->crx = YES; 

	if (First_cross_on_curve != NULL)
	{
	    j = (ymax+1)*(xmax+1)+(*n_crx);
#if defined(DEBUG_TRI_GRID)
	    if (j >= ntg->n_c_node_points)
	    {
	    	(void) printf("WARNING in add_comp_crx_to_list2d(), "
	    	              "too many node points\n"
	    	              "j %d, ntg->n_c_node_points %d\n",j,
	    	       ntg->n_c_node_points);
	    	return MODIFY_TIME_STEP;
	    }
#endif /* defined(DEBUG_TRI_GRID) */
	    Coords(ntg->c_node_points+j)[0] = Coords(p)[0];
	    Coords(ntg->c_node_points+j)[1] = Coords(p)[1];
	    cross->nd = &ntg->c_node_points[j];
	}

	if ((First_cross_on_curve != NULL) && (*First_cross_on_curve == YES))
	{
	    cross->end = YES;
	    if (*n_crx > 0)
	    {
	    	crx_store[*n_crx - 1].end = YES;
	    }
	    *First_cross_on_curve = NO;
	}
	else
	    cross->end = NO;

#if defined(DEBUG_TRI_GRID)
	if (debugging("grd_crx"))
	{
	    (void) printf("\nn_crx %d, msk %d\n",*n_crx,msk);
	    print_crxings(cross,NO);
	}
#endif /* defined(DEBUG_TRI_GRID) */

	nx = ntg->c_seg_crx_count[msk];
	if (nx == 0)
	{
	    (void) printf("WARNING in add_crx_to_list2d(), nx = 0");
	    return MODIFY_TIME_STEP;
	}

	start = ntg->c_seg_crx_lists[msk];

#if defined(DEBUG_TRI_GRID)
	if (debugging("grd_crx"))
	    (void) printf("nx %d start %p *start %d\n",
	    	          nx,(POINTER)start,*start);
#endif /* defined(DEBUG_TRI_GRID) */

	switch (dir)
	{
	case BELOW_TO_ABOVE:
	case ABOVE_TO_BELOW:
	    for (j = 0;  j < nx;  ++j)
	    {
	    	if (*(start+j) == -1) break;
	    }

	    if (j == 0)
	    	*start = *n_crx;
	    else
	    {
	    	for (i = 0;  i < j;  ++i)
	    	{
	    	    if (Coords((crx_store[*(start+i)]).pt)[0] >
						 Coords(cross->pt)[0])
			break;
		}

		if (i < j)
		{
		    input = *n_crx;
		    for (k = i;  k < j;  ++k)
		    {
		    	hold = *(start+k);
		    	*(start+k) = input;
		    	input = hold;
		    }
		    *(start+j) = input;
		}
		else
		    *(start+j) = *n_crx;
	    }
	    break;
	case LEFT_TO_RIGHT:
	case RIGHT_TO_LEFT:
	    for (j = 0;  j < nx;  ++j)
	    {
	    	if (*(start+j) == -1) break;
	    }

	    if (j == 0)
	    	*start = *n_crx;
	    else
	    {
	    	for (i = 0;  i < j;  ++i)
	    	{
	    	    if (Coords((crx_store[*(start+i)]).pt)[1] >
	    				 Coords(cross->pt)[1])
	    		    break;
	    	}

	    	if (i < j)
	    	{
	    	    input = *n_crx;
	    	    for (k = i;  k < j;  ++k)
	    	    {
	    	    	hold = *(start+k);
	    	    	*(start+k) = input;
	    	    	input = hold;
	    	    }
	    	    *(start+j) = input;
	    	}
	    	else
	    	    *(start+j) = *n_crx;
	    }
	    break;
	}
	return GOOD_STEP;
}		/*end add_comp_crx_to_list2d*/
#endif /* if defined(CONSERVATIVE_ALG) */


LOCAL	void add_blk_node_to_list(
	BLK_EL1		*blk_el1,
	NODE		*node,
	TG_PT		*nd_pt)
{
	BLK_NODE	*blk_node, *blkn;

	scalar(&blk_node,sizeof(BLK_NODE));
	blk_node->next = NULL;
	blk_node->n    = node;
	blk_node->nd   = nd_pt;
	if (blk_el1->blk_node == NULL)
	{
		blk_node->prev = NULL;
		blk_el1->blk_node = blk_node;
	}
	else
	{
		for (blkn = blk_el1->blk_node;  blkn->next != NULL;
							blkn = blkn->next)
			;
		blk_node->prev = blkn;
		blkn->next = blk_node;
	}
}		/*end add_blk_node_to_list*/

#if defined(DEBUG_TRI_GRID)
#define	DEBUG_COMP_FOR_NODE(ix,iy,indx,comp,cS,cE,cN,cW,ncs,s_list,nce,e_list,ncn,n_list,ncw,w_list,ntg)	\
	if (debugging("tmp_debug"))					\
		debug_comp_for_node(ix,iy,indx,comp,cS,cE,cN,cW,	\
			ncs,s_list,nce,e_list,ncn,n_list,ncw,w_list,ntg);
#else /* defined(DEBUG_TRI_GRID) */
#define	DEBUG_COMP_FOR_NODE(ix,iy,indx,comp,cS,cE,cN,cW,ncs,s_list,nce,e_list,ncn,n_list,ncw,w_list,ntg) 
#endif /* defined(DEBUG_TRI_GRID) */

/*
*			set_components2d():
*
*	Sets the component list associated with each grid node
*	of tri_grid->rect_grid.
*/

EXPORT	int set_components2d(
	TRI_GRID	*ntg,
	INTERFACE	*intfc,
        int             on_grid)
{
	COMPONENT	cS, cE, cN, cW;
	COMPONENT	*comp;
	COMPONENT	ext_comp = exterior_component(ntg->grid_intfc);
	CRXING	 	*cr;
        /*
	CRXING		*crx_store = ntg->crx_store;
	RECT_GRID	*gr = &ntg->rect_grid;
	int		xmax = gr->gmax[0], ymax = gr->gmax[1];
	int		xmaxx = xmax+1;
        */
	CRXING		*crx_store;
	RECT_GRID	*gr;
	float		coords[MAXD];
	int		xmax, ymax, xmaxx;
	int		ix, iy, ix0;
	int		k_s, k_w, k_n, k_e, ncs, ncw, ncn, nce;
	int		*s_list, *e_list, *n_list, *w_list;
        /*
	int		*seg_crx_count = ntg->seg_crx_count;
	int		**seg_crx_lists = ntg->seg_crx_lists;
        */
	int		*seg_crx_count;
	int		**seg_crx_lists;

	debug_print("set_components","Entered set_components2d()\n");
#if defined(DEBUG_TRI_GRID)
	if (debugging("set_comp_intfc"))
	{
	    (void) printf("\tinput interface into set_components2d()\n");
	    print_interface(intfc);
	    (void) printf("\tgrid interface into set_components2d()\n");
	    print_interface(ntg->grid_intfc);
	}
#endif /* defined(DEBUG_TRI_GRID) */

        if(on_grid == ON_DUAL_GRID)
        {
            comp = ntg->components;
            gr   = &ntg->rect_grid;
            crx_store = ntg->crx_store;
            seg_crx_count = ntg->seg_crx_count;
            seg_crx_lists = ntg->seg_crx_lists;
        }
#if defined(CONSERVATIVE_ALG) 
        else
        {
            comp = ntg->c_components;
            gr   = &ntg->aug_comp_grid;
            crx_store = ntg->c_crx_store;
            seg_crx_count = ntg->c_seg_crx_count;
            seg_crx_lists = ntg->c_seg_crx_lists;
        }
#endif /* if defined(CONSERVATIVE_ALG) */
        xmax = gr->gmax[0];
        ymax = gr->gmax[1];
        xmaxx = xmax+1;

#if defined(USE_OVERTURE)
        {
            if(intfc->curves == NULL)
            {
                int n_reg_nodes, i;
                n_reg_nodes = xmaxx;
                n_reg_nodes *= (ymax + 1);
                if(ntg->center_comp == NO_COMP ||
                   ntg->center_comp == 0)
                {
                    printf("ERROR in set_components2d()\n");
                    printf("WHen USE_OVERTURE, patch_component[%d] of patch[%d]"
                         " is not properly set\n",ntg->center_comp,ntg->patch_number);
                    clean_up(ERROR);
                }
                for(i = 0; i < n_reg_nodes; i++)
                {
                    ntg->components[i] = ntg->center_comp;
                }
                debug_print("set_components",
                   "Leaving set_components2d(), status = GOOD_STEP\n");
                return GOOD_STEP;
            }
        }
#endif /* if defined(USE_OVERTURE) */

	if (intfc->curves == NULL) /* Tracking turned off */
	    return set_untracked_components(ntg,intfc);

        /****
	// comp = ntg->components;
        {
            // for the test purpose 
	    coords[0] = grid_center_coord(0,gr);
	    coords[1] = gr->L[1];
	    cN = component(coords,intfc);
        }
        ****/

	/* Set components on lower boundary */

	/*
	*  Try to find at least one crossing near
	*  the lower boundary to get started
	*/

	for (ix0 = 0; ix0 <= xmax; ++ix0)
	{
	    k_n = ix0 + xmax;
	    ncn    = seg_crx_count[k_n];

	    if (ix0 < xmax)
	    {
	    	k_e = ix0;
	    	nce    = seg_crx_count[k_e];
	    }
	    else
	    {
	    	k_e = -1;	nce = 0;
	    }
	    if (ncn > 0 || nce > 0)
		break;
	}

	if (ix0 > xmax)
	{
	    /*
	     *  There are no crossings near the lower boundary
	     *  get nearby component from original interface
	     */

	    coords[0] = grid_center_coord(0,gr);
	    coords[1] = gr->L[1];
	    cN = component(coords,intfc);
	    for (ix = 0; ix <= xmax; ++ix)
		comp[ix] = cN;
	}
	else
	{
	    /* Set components up to starting point found above */

	    cS = cE = cN = cW = ext_comp;
	    k_s = -1;	ncs = 0;	s_list = NULL;
	    if (ncn > 0)
	    {
	    	n_list = seg_crx_lists[k_n];
	    	cr = crx_store + *n_list;
	    	cN = (cr->crossing_direction == LEFT_TO_RIGHT) ?
				positive_component(cr->hs) : 
				negative_component(cr->hs);
	    }

	    if (nce > 0)
	    {
	    	e_list = seg_crx_lists[k_e];
	    	cr = crx_store + *e_list;
	    	cE = (cr->crossing_direction == BELOW_TO_ABOVE) ?
				negative_component(cr->hs) :
				positive_component(cr->hs);
	    }

	    comp[0] = set_comp2d(cS,cE,cN,cW,ext_comp);
	    if (comp[0] == NO_COMP)
	        return inconsistent_cross2d(ix0,0,xmax,ymax,cS,cE,cN,cW,
				            ext_comp,0,nce,ncn,0,NULL,
					    e_list,n_list,NULL,ntg,intfc);

	    for (ix = 1; ix <= ix0; ++ix)
		comp[ix] = comp[0];

	    /* Set remaining components on lower boundary */

	    k_s = -1;	ncs = 0;	s_list = NULL;
	    for (ix = ix0+1; ix <= xmax; ++ix)
	    {
	    	k_w    = ix - 1;
	    	ncw    = seg_crx_count[k_w];
	    	w_list = seg_crx_lists[k_w];

	    	k_n    = ix + xmax;
	    	ncn    = seg_crx_count[k_n];
	    	n_list = seg_crx_lists[k_n];

	    	if (ix == xmax)
	    	{
	    	    k_e = -1;	nce = 0;	e_list = NULL;
	    	}
	    	else
	    	{
	    	    k_e    = ix;
	    	    nce    = seg_crx_count[k_e];
	    	    e_list = seg_crx_lists[k_e];
	    	}

		if (ncw)
		{
		    cr = crx_store + *(w_list+ncw-1);
		    cW = (cr->crossing_direction == BELOW_TO_ABOVE) ?
				positive_component(cr->hs) :
				negative_component(cr->hs);
		}
		else
		    cW = comp[ix - 1];

		if (ncn)
		{
		    cr = crx_store + *n_list;
		    cN = (cr->crossing_direction == LEFT_TO_RIGHT) ?
				positive_component(cr->hs) :
				negative_component(cr->hs);
		}
		else
		    cN = cW;

		if (nce)
		{
		    cr = crx_store + *e_list;
		    cE = (cr->crossing_direction == BELOW_TO_ABOVE) ?
				negative_component(cr->hs) :
				positive_component(cr->hs);
		}
		else
		    cE = cW;

		comp[ix] = set_comp2d(cS,cE,cN,cW,ext_comp);
		if (comp[ix] == NO_COMP)
		    return inconsistent_cross2d(ix,0,xmax,ymax,cS,cE,cN,cW,
					        ext_comp,ncs,nce,ncn,ncw,s_list,
					        e_list,n_list,w_list,ntg,intfc);
		DEBUG_COMP_FOR_NODE(ix,0,ix,comp,cS,cE,cN,cW,ncs,s_list,nce,
				    e_list,ncn,n_list,ncw,w_list,ntg);
	    }
	}

	/* Set components on upper boundary */

	for (ix0 = 0; ix0 <= xmax; ++ix0)
	{
	    k_s = 2*ymax*xmax + ymax + ix0 - xmax - 1;
	    ncs = seg_crx_count[k_s];

	    if (ix0 < xmax)
	    {
	    	k_e = k_s + xmax + 1;
	    	nce    = seg_crx_count[k_e];
	    }
	    else
	    {
	    	k_e = -1;	nce = 0;
	    }
	    if (ncs > 0 || nce > 0)
		break;
	}

	if (ix0 > xmax)
	{
	    /*
	     *  There are no crossings near the upper boundary
	     *  get nearby component from original interface
	     */

	    coords[0] = grid_center_coord(0,gr);
	    coords[1] = gr->U[1];
	    cS = component(coords,intfc);
	    for (ix = 0; ix <= xmax; ++ix)
	    {
	    	comp[ymax*xmaxx + ix] = cS;
	    	DEBUG_COMP_FOR_NODE(ix,ymax,ymax*xmaxx+ix,comp,cS,cE,cN,cW,ncs,
				    s_list,nce,e_list,0,n_list,0,w_list,ntg);
	    }
	}
	else
	{
	    /* Set components up to starting point found above */

	    cS = cE = cN = cW = ext_comp;
	    k_n = -1;	ncn = 0;	n_list = 0;
	    if (ncs > 0)
	    {
	    	s_list = seg_crx_lists[k_s];
	    	cr = crx_store + *(s_list+ncs-1);
	    	cS = (cr->crossing_direction == LEFT_TO_RIGHT) ?
	    		negative_component(cr->hs) :
	    		positive_component(cr->hs);
	    }

	    if (nce > 0)
	    {
	    	e_list = seg_crx_lists[k_e];
	    	cr = crx_store + *e_list;
	    	cE = (cr->crossing_direction == BELOW_TO_ABOVE) ?
	    		negative_component(cr->hs) :
	    		positive_component(cr->hs);
	    }

	    comp[ymax*xmaxx] = set_comp2d(cS,cE,cN,cW,ext_comp);
	    if (comp[ymax*xmaxx] == NO_COMP)
	        return inconsistent_cross2d(ix0,ymax,xmax,ymax,cS,cE,cN,cW,
					    ext_comp,0,nce,ncn,0,NULL,e_list,
					    n_list,NULL,ntg,intfc);

	    for (ix = 1; ix <= ix0; ++ix)
	    	comp[ymax*xmaxx+ix] = comp[ymax*xmaxx];

	    /* Set remaining components on upper boundary */

	    k_n = -1;	ncn = 0;	n_list = NULL;
	    for (ix = 1+ix0; ix <= xmax; ++ix)
	    {
	    	k_w    = 2*ymax*xmax + ymax + ix - 1;
	    	ncw    = seg_crx_count[k_w];
	    	w_list = seg_crx_lists[k_w];
	
	    	k_s    = k_w - xmax;
	    	ncs    = seg_crx_count[k_s];
	    	s_list = seg_crx_lists[k_s];

	    	if (ix == xmax)
	    	{
	    	    k_e = -1;	nce = 0;	e_list = NULL;
	    	}
	    	else
	    	{
	    	    k_e    = k_w + 1;
	    	    nce    = seg_crx_count[k_e];
	    	    e_list = seg_crx_lists[k_e];
	    	}

		if (ncw)
		{
		    cr = crx_store + *(w_list+ncw-1);
		    cW = (cr->crossing_direction == BELOW_TO_ABOVE) ?
				positive_component(cr->hs) :
				negative_component(cr->hs);
		}
		else
		    cW = comp[ymax*xmaxx + ix - 1];

		if (ncs)
		{
		    cr = crx_store + *(s_list+ncs-1);
		    cS = (cr->crossing_direction == LEFT_TO_RIGHT) ?
				negative_component(cr->hs) :
				positive_component(cr->hs);
		}
		else
		    cS = cW;

		if (nce)
		{
		    cr = crx_store + *e_list;
		    cE = (cr->crossing_direction == BELOW_TO_ABOVE) ?
				negative_component(cr->hs) :
				positive_component(cr->hs);
		}
		else
		    cE = cW;

		comp[ymax*xmaxx+ix] = set_comp2d(cS,cE,cN,cW,ext_comp);
		if (comp[ymax*xmaxx+ix] == NO_COMP)
		    return inconsistent_cross2d(ix,ymax,xmax,ymax,cS,cE,cN,cW,
					        ext_comp,ncs,nce,ncn,ncw,s_list,
					        e_list,n_list,w_list,ntg,intfc);
		DEBUG_COMP_FOR_NODE(ix,ymax,ymax*xmaxx+ix,comp,cS,cE,cN,cW,ncs,
				    s_list,nce,e_list,ncn,n_list,
				    ncw,w_list,ntg);
	    }
	}


	/* Set components on left boundary */

	cW = ext_comp;
	k_w = -1;	ncw = 0;	w_list = NULL;
	for (iy = 1; iy < ymax; ++iy)
	{
	    k_e    = 2*iy*xmax + iy;
	    nce    = seg_crx_count[k_e];
	    e_list = seg_crx_lists[k_e];

	    k_s    = k_e - xmax - 1;
	    ncs    = seg_crx_count[k_s];
	    s_list = seg_crx_lists[k_s];

	    k_n    = k_e + xmax;
	    ncn    = seg_crx_count[k_n];
	    n_list = seg_crx_lists[k_n];


	    if (ncs)
	    {
	    	cr = crx_store + *(s_list+ncs-1);
	    	cS = (cr->crossing_direction == LEFT_TO_RIGHT) ?
	    		negative_component(cr->hs) :
	    		positive_component(cr->hs);
	    }
	    else
	    	cS = comp[(iy-1)*xmaxx];

	    if (nce)
	    {
	    	cr = crx_store + *e_list;
	    	cE = (cr->crossing_direction == BELOW_TO_ABOVE) ?
	    		negative_component(cr->hs) :
	    		positive_component(cr->hs);
	    }
	    else
	    	cE = cS;

	    if (ncn)
	    {
	    	cr = crx_store + *n_list;
	    	cN = (cr->crossing_direction == LEFT_TO_RIGHT) ?
	    		positive_component(cr->hs) :
	    		negative_component(cr->hs);
	    }
	    else
	    	cN = cS;

	    comp[xmaxx*iy] = set_comp2d(cS,cE,cN,cW,ext_comp);
	    if (comp[xmaxx*iy] == NO_COMP)
	    	return inconsistent_cross2d(0,iy,xmax,ymax,cS,cE,cN,cW,ext_comp,
					    ncs,nce,ncn,ncw,s_list,e_list,
					    n_list,w_list,ntg,intfc);
	    DEBUG_COMP_FOR_NODE(0,iy,xmaxx*iy,comp,cS,cE,cN,cW,ncs,s_list,nce,
				e_list,ncn,n_list,ncw,w_list,ntg);
	}


	/* Set components on right boundary */

	cE = ext_comp;
	k_e = -1;	nce = 0;	e_list = NULL;
	for (iy = 1; iy < ymax; ++iy)
	{
	    k_s    = 2*iy*xmax + iy - 1;
	    ncs    = seg_crx_count[k_s];
	    s_list = seg_crx_lists[k_s];

	    k_w    = k_s + xmax;
	    ncw    = seg_crx_count[k_w];
	    w_list = seg_crx_lists[k_w];

	    k_n    = k_w + 1 + xmax;
	    ncn    = seg_crx_count[k_n];
	    n_list = seg_crx_lists[k_n];

	    if (ncs)
	    {
	    	cr = crx_store+ *(s_list+ncs-1);
	    	cS = (cr->crossing_direction == LEFT_TO_RIGHT) ?
	    		negative_component(cr->hs) :
	    		positive_component(cr->hs);
	    }
	    else
	    	cS = comp[(iy-1)*xmaxx + xmax];

	    if (ncw)
	    {
	    	cr = crx_store + *(w_list+ncw-1);
	    	cW = (cr->crossing_direction == BELOW_TO_ABOVE) ?
	    		positive_component(cr->hs) :
	    		negative_component(cr->hs);
	    }
	    else
	    	cW = cS;

	    if (ncn)
	    {
	    	cr = crx_store + *n_list;
	    	cN = (cr->crossing_direction == LEFT_TO_RIGHT) ?
	    		positive_component(cr->hs) :
	    		negative_component(cr->hs);
	    }
	    else
	    	cN = cS;

	    comp[iy*xmaxx+xmax] = set_comp2d(cS,cE,cN,cW,ext_comp);
	    if (comp[iy*xmaxx+xmax] == NO_COMP)
	    	return inconsistent_cross2d(xmax,iy,xmax,ymax,cS,cE,cN,cW,
					    ext_comp,ncs,nce,ncn,ncw,s_list,
					    e_list,n_list,w_list,ntg,intfc);
	    DEBUG_COMP_FOR_NODE(xmax,iy,iy*xmaxx+xmax,comp,cS,cE,cN,cW,ncs,
				s_list,nce,e_list,ncn,n_list,ncw,w_list,ntg);
	}


        /*   components inside computational region   */

	for (iy = 1;  iy < ymax;  ++iy)
	{
	    for (ix = 1;  ix < xmax;  ++ix)
	    {
	    	k_e    = 2*iy*xmax + iy + ix;
	    	nce    = seg_crx_count[k_e];
	    	e_list = seg_crx_lists[k_e];

	    	k_w    = k_e - 1;
	    	ncw    = seg_crx_count[k_w];
	    	w_list = seg_crx_lists[k_w];

	    	k_s    = k_w - xmax;
	    	ncs    = seg_crx_count[k_s];
	    	s_list = seg_crx_lists[k_s];

	    	k_n    = k_e + xmax;
	    	ncn    = seg_crx_count[k_n];
	    	n_list = seg_crx_lists[k_n];

	    	cS = cE = cN = cW = ext_comp;
	    	if (ncw)
	    	{
	    	    cr = crx_store + *(w_list+ncw-1);
	    	    cW = (cr->crossing_direction == BELOW_TO_ABOVE) ?
	    			positive_component(cr->hs) :
	    			negative_component(cr->hs);
	    	}
	    	else
	    	    cW = comp[iy*xmaxx+ix-1];

	    	if (ncs)
	    	{
	    	    cr = crx_store + *(s_list+ncs-1);
	    	    cS = (cr->crossing_direction == LEFT_TO_RIGHT) ?
	    			negative_component(cr->hs) :
	    			positive_component(cr->hs);
	    	}
	    	else
	    	    cS = comp[(iy-1)*xmaxx+ix];

	    	if (nce)
	    	{
	    	    cr = crx_store + *e_list;
	    	    cE = (cr->crossing_direction == BELOW_TO_ABOVE) ?
	    			negative_component(cr->hs) :
	    			positive_component(cr->hs);
	    	}

	    	if (ncn)
	    	{
	    	    cr = crx_store + *n_list;
	    	    cN = (cr->crossing_direction == LEFT_TO_RIGHT) ?
	    			positive_component(cr->hs) :
	    			negative_component(cr->hs);
	    	}

	    	comp[iy*xmaxx+ix] = set_comp2d(cS,cE,cN,cW,ext_comp);
	    	if (comp[iy*xmaxx+ix] == NO_COMP)
	    	    return inconsistent_cross2d(ix,iy,xmax,ymax,cS,cE,cN,cW,
					        ext_comp,ncs,nce,ncn,ncw,s_list,
					        e_list,n_list,w_list,ntg,intfc);
	    	DEBUG_COMP_FOR_NODE(ix,iy,iy*xmaxx+ix,comp,cS,cE,cN,cW,ncs,
				    s_list,nce,e_list,ncn,n_list,
				    ncw,w_list,ntg);
	    }

	}

	debug_print("set_components",
		"Leaving set_components2d(), status = GOOD_STEP\n");
	return GOOD_STEP;
}		/*end set_components2d*/

#if defined(DEBUG_TRI_GRID)
LOCAL	void debug_comp_for_node(
	int		ix,
	int		iy,
	int		indx,
	COMPONENT	*comp,
	COMPONENT	cS,
	COMPONENT	cE,
	COMPONENT	cN,
	COMPONENT	cW,
	int		ncs,
	int		*s_list,
	int		nce,
	int		*e_list,
	int		ncn,
	int		*n_list,
	int		ncw,
	int		*w_list,
	TRI_GRID	*ntg)
{
	(void) printf("component for node[%d] (%d %d) = %d\n",
		      indx,ix,iy,comp[indx]);
	(void) printf("cS %d cE %d cN %d cW %d\n",cS,cE,cN,cW);
	print_side_crx_list(ntg,'S',ncs,s_list);
	print_side_crx_list(ntg,'E',nce,e_list);
	print_side_crx_list(ntg,'N',ncn,n_list);
	print_side_crx_list(ntg,'W',ncw,w_list);
}		/*end debug_comp_for_node*/
#endif /* defined(DEBUG_TRI_GRID) */


LOCAL	int	shift_from_cell_edge(
	float		*p,
	int		idir,
	RECT_GRID	*rgr,
	float		tol)
{
	int		ic;
	float		sfrac;

	ic = cell_index(p[idir],idir,rgr);
	sfrac = (p[idir] - rgr->edges[idir][ic]) / cell_width(ic,idir,rgr);
	if (sfrac < TOL)     		p[idir] += tol;
	else if (sfrac > ONEMTOL)	p[idir] -= tol;
	return ic;
}		/*end shift_from_cell_edge*/

LOCAL	int inconsistent_cross2d(
	int		ix,
	int		iy,
	int		xmax,
	int		ymax,
	COMPONENT	cS,
	COMPONENT	cE,
	COMPONENT	cN,
	COMPONENT	cW,
	COMPONENT	ext_comp,
	int		ncs,
	int		nce,
	int		ncn,
	int		ncw,
	int		*s_list,
	int		*e_list,
	int		*n_list,
	int		*w_list,
	TRI_GRID	*ntg,
	INTERFACE	*intfc)
{
	(void) printf("WARNING in inconsistent_cross2d(), "
	              "inconsistent crossings, reducing time step\n");
	if (debugging("set_components"))
	{
	    RECT_GRID	*gr = &ntg->rect_grid;

	    (void) printf("node: (%d %d), posn (%g, %g)\n",
			  ix,iy,cell_edge(ix,0,gr),cell_edge(iy,1,gr));
	    (void) printf("xmax %d ymax %d\n",xmax,ymax);
	    (void) printf("cS %d cE %d cN %d cW %d\n",cS,cE,cN,cW);
	    (void) printf("ext_comp %d\n",ext_comp);

	    print_side_crx_list(ntg,'S',ncs,s_list);
	    print_side_crx_list(ntg,'E',nce,e_list);
	    print_side_crx_list(ntg,'N',ncn,n_list);
	    print_side_crx_list(ntg,'W',ncw,w_list);

	    (void) printf("\tinput interface\n");
	    print_interface(intfc);
	    plot_interface(intfc,"Input_interface",NULL,NULL,"Input interface");
	    (void) printf("\tgrid interface\n");
	    print_interface(ntg->grid_intfc);
	    plot_interface(ntg->grid_intfc,"Grid_interface",NULL,NULL,
			   "Grid interface");
	}
	debug_print("set_components",
	      "Left set_components(), status = MODIFY_TIME_STEP\n");
	return MODIFY_TIME_STEP;
}		/*end inconsistent_cross2d*/




LOCAL	COMPONENT set_comp2d(
	COMPONENT	cS,
	COMPONENT	cE,
	COMPONENT	cN,
	COMPONENT	cW,
	COMPONENT	ext_comp)
{
	register COMPONENT c;

	c = ext_comp;
	if (cS != ext_comp)
	    c = cS;
	if (cE != ext_comp)
	{
	    if (c == ext_comp)
		c = cE;
	    else if (c != cE)
		return NO_COMP;
	}
	if (cN != ext_comp)
	{
	    if (c == ext_comp)
		c = cN;
	    else if (c != cN)
		return NO_COMP;
	}
	if (cW != ext_comp)
	{
	    if (c == ext_comp)
		c = cW;
	    else if (c != cW)
		return NO_COMP;
	}

	return c;
}		/*end set_comp2d*/


LIB_LOCAL	void set_crx_structure_storage2d(
	TRI_GRID	*ntg,
	int		**ic_of_node)
{
	RECT_GRID	*r_gr = &ntg->rect_grid;
	INTERFACE	*intfc = ntg->grid_intfc;
	BLK_EL0		*blk_el0;
	BLK_EL1		*blk_el1;
	NODE		**node;
	int		n_crx, ii, n_segs;
	int		ix, iy;
	int		n_intfc_nodes;
	static	BLK_EL0 CLEAR_BLK_EL0;
	static	BLK_EL1 CLEAR_BLK_EL1;

	xmax = r_gr->gmax[0];	ymax = r_gr->gmax[1];
	XL   = r_gr->L[0];	YL   = r_gr->L[1];
	XU   = r_gr->U[0];	YU   = r_gr->U[1];
	hx   = r_gr->h[0];	hy   = r_gr->h[1];

	n_segs = xmax*(ymax+1) + ymax*(xmax+1);
	VECTOR(ntg,seg_crx_count,n_segs,INT);
	for (ii = 0;  ii < n_segs;  ++ii) ntg->seg_crx_count[ii] = 0;

	n_crx = count_intfc_crossings_of_grid_lattice2d(ntg,ntg->grid_intfc,
			r_gr, ic_of_node, ON_DUAL_GRID);

	ntg->n_crx = n_crx;
	init_seg_crx_lists(ntg,n_crx,n_segs,ON_DUAL_GRID);

	n_intfc_nodes = 0;
	for (node = intfc->nodes; node && *node;  ++node)
		++n_intfc_nodes;

	if (interpolation_method(ntg) != ELEMENT_ON_THE_FLY)
	{
	    int nlin;
	    nlin = ntg->guessed_num_lin =
		2*((int)(ntg->num_lin_guess_factor*n_crx));
	    VECTOR(ntg,lin_els,nlin,sizeof(LINEAR_ELEMENT));
	    VECTOR(ntg,bilin_els,xmax*ymax,sizeof(BILINEAR_ELEMENT));
	    ntg->n_fr_pts = intfc->num_points;
	}
	else
	    ntg->n_fr_pts = intfc->num_points + n_crx;
	ntg->n_node_points += n_crx+n_intfc_nodes;
	alloc_node_points(ntg,ntg->n_node_points);
	VECTOR(ntg,front_points,ntg->n_fr_pts,sizeof(TG_PT));

	ntg->n_tg_pts = 0;

#if defined(DEBUG_TRI_GRID)
	if (debugging("TRI_storage"))
	{
		int n_reg_nodes = (xmax+1)*(ymax+1);
		(void) printf("Storage set for %d triangles\n",
			      ntg->guessed_num_lin);
		(void) printf("gmax %d %d\n",xmax,ymax);
		(void) printf("n_intfc_nodes %d n_reg_nodes %d n_crx %d\n",
					n_intfc_nodes,n_reg_nodes,n_crx);
	}
#endif /* defined(DEBUG_TRI_GRID) */

	alloc_blk_els0(ntg,ymax*xmax);
	alloc_blk_els1(ntg,ymax*xmax);
	blk_el0 = ntg->blk_els0;	blk_el1 = ntg->blk_els1;
	for (iy = 0; iy < ymax; ++iy)
	{
	    for (ix = 0; ix < xmax; ++ix)
	    {
	    	*blk_el0++ = CLEAR_BLK_EL0;
	    	*blk_el1++ = CLEAR_BLK_EL1;
	    }
	}
}		/*end set_crx_structure_storage2d*/

#if defined(CONSERVATIVE_ALG)
LIB_LOCAL   void set_crx_indx_local_global(
        TRI_GRID    *ntg,
        int         which_grid)
{
        RECT_GRID   *expanded_gr;

        if(which_grid == ON_COMP_GRID)
            expanded_gr = &(ntg->aug_comp_grid);
        else
            expanded_gr = &(ntg->rect_grid);

        xmax = expanded_gr->gmax[0];   ymax = expanded_gr->gmax[1];
        XL   = expanded_gr->L[0];      YL   = expanded_gr->L[1];
        XU   = expanded_gr->U[0];      YU   = expanded_gr->U[1];
        hx   = expanded_gr->h[0];      hy   = expanded_gr->h[1];
}

LIB_LOCAL   void set_crx_storage_for_reconstruction_on_grid2d(
        TRI_GRID    *ntg,
        INTERFACE   *intfc,
        RECT_GRID   *expanded_gr,
        int         **ic_of_node,
        int         which_grid)
{
        int         *gmax = expanded_gr->gmax;
        int         dim = expanded_gr->dim;
        int         n_segs,n_crx,i,n_reg_nodes, blks;
        int         n_intfc_nodes = 0;
        NODE        **node; 

        DEBUG_ENTER(set_crx_storage_for_reconstruction_on_grid2d)
       
        /* Set local global variables here */
        xmax = expanded_gr->gmax[0];   ymax = expanded_gr->gmax[1];
        XL   = expanded_gr->L[0];      YL   = expanded_gr->L[1];
        XU   = expanded_gr->U[0];      YU   = expanded_gr->U[1];
        hx   = expanded_gr->h[0];      hy   = expanded_gr->h[1];
 
        for (node = intfc->nodes; node && *node;  ++node)
                ++n_intfc_nodes;
        n_segs = 0;
        n_reg_nodes = 1;
        for (i = 0; i < 2; i++)
        {
            n_segs += gmax[i]*(gmax[(i+1)%2] + 1);
            n_reg_nodes *= gmax[i] + 1;
        }
        if(which_grid == ON_DUAL_GRID)
        {
            ntg->n_node_points = ntg->n_reg_nodes = n_reg_nodes;
            ntg->n_segs = n_segs;
            VECTOR(ntg,seg_crx_count,n_segs,INT);
            for (i = 0; i < n_segs; ++i)
                ntg->seg_crx_count[i] = 0;

            n_crx = count_intfc_crossings_of_grid_lattice2d(ntg,
                      intfc, expanded_gr, ic_of_node, which_grid);
            ntg->n_crx = n_crx;
            ntg->n_node_points += n_crx+n_intfc_nodes; 
            alloc_node_points(ntg,ntg->n_node_points);  
            init_seg_crx_lists(ntg, n_crx, n_segs, which_grid);
            alloc_components_array(ntg,n_reg_nodes);
        }
        else
        {
            ntg->n_c_node_points = ntg->n_c_reg_nodes = n_reg_nodes;
            ntg->n_c_segs = n_segs;
            VECTOR(ntg,c_seg_crx_count,n_segs,INT);
            for (i = 0; i < n_segs; i++)
                ntg->c_seg_crx_count[i] = 0;
            ntg->n_c_crx = n_crx = count_intfc_crossings_of_grid_lattice2d(ntg,
                          intfc, expanded_gr, ic_of_node, which_grid);
            ntg->n_c_node_points += n_crx+n_intfc_nodes; 

            VECTOR(ntg, c_node_points, ntg->n_c_node_points, sizeof(TG_PT)); 
            ntg->c_cg_npts = ntg->c_node_points + ntg->c_node_offset;
            init_seg_crx_lists(ntg,n_crx, n_segs, which_grid);
            VECTOR(ntg,c_components,ntg->n_c_reg_nodes,sizeof(COMPONENT));
            ntg->c_cg_comps = ntg->c_components + ntg->c_node_offset;

            VECTOR(ntg,blk_type,xmax*ymax,sizeof(int));
            blks = xmax*ymax;
            for (i = 0; i < blks; i++)
                ntg->blk_type[i] = UNSET;

            {
                int                 j = 0; 
                register float      y;
                register float      *xx_grid = expanded_gr->edges[0];
                register float      *yy_grid = expanded_gr->edges[1];
                int                 xmax1 = expanded_gr->gmax[0],
                                    ymax1 = expanded_gr->gmax[1];
                int                 ix,iy;

                for (iy = 0;  iy <= ymax1;  iy++)
                {
                    y = yy_grid[iy];
                    for (ix = 0;  ix <= xmax1;  ix++)
                    {
                        Coords(ntg->c_node_points+j)[0] = xx_grid[ix];
                        Coords(ntg->c_node_points+j)[1] = y;
                        j++;
                    }
                }
            }
        }
        DEBUG_LEAVE(set_crx_storage_for_reconstruction_on_grid2d)
}


/* NOTE: the rgr used here should be
 * tri_grid->aug_comp_grid, The 
 * aug_comp_grid->G[L,U] is set to be the
 * the buffered computational grid subdomain.
 */ 
LOCAL int shift_from_cell_edge_on_comp(
        float           *p,
        int             idir,
        RECT_GRID       *rgr,
        float           tol)
{
        int             ic;
        float           sfrac;
        float           *L = rgr->GL;
        float           *U = rgr->GU;

        sfrac = fabs(p[idir] - L[idir]) / cell_width(ic,idir,rgr);
        if(sfrac < TOL) /* close to L bdry */
        {
            p[idir] -= tol;
            ic = cell_index(p[idir],idir,rgr);
            return ic; 
        }

        sfrac = fabs(p[idir] - U[idir]) / cell_width(ic,idir,rgr);
        if(sfrac < TOL) /* close to U bdry */
        {
            p[idir] += tol;
            ic = cell_index(p[idir],idir,rgr);
            return ic; 
        }

        ic = cell_index(p[idir],idir,rgr);
        sfrac = (p[idir] - rgr->edges[idir][ic]) / cell_width(ic,idir,rgr);
        if (sfrac < TOL)                p[idir] += tol;
        else if (sfrac > ONEMTOL)       p[idir] -= tol;
        return ic;
}

LOCAL void preserv_indx_on_insertion(
        CURVE   *c,
        BOND    *b)
{
        int     i; 
 
        for (i = 0;  i < n_new_intfc_points;  ++i)
        {
            if(b == (c)->first)
            {
                new_intfc_points[i]->indx = (c)->sindx;
                new_intfc_points[i]->curv_id = b->start->curv_id;
            }
            else if(b == (c)->last)
            {
                new_intfc_points[i]->indx = (c)->eindx;
                new_intfc_points[i]->curv_id = b->end->curv_id;
            }
            else
            {
                new_intfc_points[i]->indx = b->start->indx;
                new_intfc_points[i]->curv_id = b->start->curv_id;
            }
            /*
            if(b != (c)->first)
                new_intfc_points[i]->indx = b->start->indx;
            else
                new_intfc_points[i]->indx = b->end->indx;
            */
        }
}
#endif /* if defined(CONSERVATIVE_ALG) */

/* This function needs local global variables */
/* Viewing from grid NODE (ip), the grid line segment index*/
LIB_LOCAL int  seg_index2d(
        int             *ip,
        int             dir)
{
        int             nix,niy;
        int             k;

        nix = ip[0]; niy = ip[1];
        switch (dir)
        {
        case SOUTH:
            k  = 2*niy*xmax + niy - 1 + nix - xmax;
        break;
        case WEST:
            k  = 2*niy*xmax + niy + nix-1;
        break;
        case NORTH:
            k  = 2*niy*xmax + niy + xmax + nix;
        break;
        case EAST:
            k  = 2*niy*xmax + niy + nix;
        break;
        }
        return k;
}

#endif /* defined(TWOD) */
