/*
*				int.h:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*			INTERFACE Structures:
*/

#if !defined(_INT_H)
#define _INT_H

#include <intfc/geom.h>

typedef int COMPONENT;		/* Labels Connected Domains */

#if !defined(MAX_N_COEF)
enum {MAX_N_COEF = 6}; // 15 correspond to 4th degree polynomial (2D)
                       // 10 correspond to cubic polynomial (2D)
                       // 6 correspond to quadratic polynomial (2D)
                       // 3: linear polynomial (2D)
                       // 1: const. polynomial
enum {N_EQN = 4};      // 4 correspond to  (2D) Euler
                       // 1: const. 2D Burger's
#endif /* !defined(MAX_N_COEF) */


 /* Point in the Plane: */
struct _POINT
{
	int _boundary;		/* Mandatory first element,  see notes on
				 * boundary macros below */

	float	_coords[MAXD];

	struct {
	    unsigned int _boundary : 1;
	    unsigned int _user0 : 1;
	    unsigned int _user1 : 1;
	    unsigned int _user2 : 1;
	    unsigned int _user3 : 1;
	    unsigned int _user4 : 1;
	    unsigned int _user5 : 1;
	    unsigned int _user6 : 1;
	    unsigned int _user7 : 1;
	    unsigned int _user8 : 1;
	    unsigned int _user9 : 1;
	} _point_flags;

#if defined(ONED)
	struct _INTERFACE *interface;
	struct _POINT *obj;	/* refers back to point, see boundary macro */
	struct _HYPER_SURF *hs;
#endif /* defined(ONED) */

	union {
	    bool      _sorted;
	    POINTER	 _opaque_pointer;
	    struct _NODE *_node;
	    int          _index;
	} private_data;
#if defined(THREED)
	float		_nor[3];
#endif /* defined(THREED) */

#if defined(CONSERVATIVE_ALG)
        bool         crx;   /* The flag indicates whether the 
                               point is on the grid line */
        int          indx;  /* The point index on the curve 
                               specified by curv_id */
        int          curv_id; /* The curve this point belongs to */
#endif /* if defined(CONSERVATIVE_ALG) */
};
typedef struct _POINT POINT;

 /* Macros of accessing private fields of points */
#define Private_data(p)	   (p)->private_data
#define Index_of_point(p)  Private_data(p)._index
#define node_at_point(p)   Private_data(p)._node
#define sorted(p)          Private_data(p)._sorted
#define opaque_pointer(p)  Private_data(p)._opaque_pointer


 /* An Elementary Bond on a Curve: */

#if defined(THREED)
struct _BOND_TRI
{
	struct _TRI     *tri;
  	struct _SURFACE *surface;
	struct _CURVE   *curve;
	struct _BOND    *bond;
	ORIENTATION     orient;
};
typedef struct _BOND_TRI BOND_TRI;
#endif /* defined(THREED) */

struct _BOND
{
	struct _BOND *next;
	struct _BOND *prev;
	POINT *start;
	POINT *end;
	float length;

#if defined(THREED)
	struct _BOND_TRI **_btris;	/* Bounding triangles in angle order */
#endif /* defined(THREED) */
};
typedef struct _BOND BOND;


 /* A Curve in the Plane: */
struct _CURVE
{

	int _boundary;		/* Mandatory first element,  see notes on
				 *  boundary macros below */
	struct _CURVE *obj;	/* refers back to curve, see boundary macro */
	struct _HYPER_SURF *hs;
	struct _HYPER_SURF_BDRY *hsb;

	/* Topological Part: */
	struct _INTERFACE *interface;
	struct _NODE *start;
	struct _NODE *end;

#if defined(THREED)
	struct _SURFACE **pos_surfaces;
	struct _SURFACE **neg_surfaces;
#endif /* defined(THREED) */

	/* Representation Specific Part: */
	BOND *first;
	BOND *last;
	int  num_points;
#if defined(CONSERVATIVE_ALG)
        int  curv_id;
        int  pp_node; 
        int  sindx, eindx;  /* To avoid mutiple curves collision
                             * which will make point index unconsistent
                             * the start or end pt. 
                             */
#endif /* if defined(CONSERVATIVE_ALG) */
};
typedef struct _CURVE CURVE;


 /* Oriented curves */

struct _O_CURVE
{
	struct _CURVE *curve;
	ORIENTATION orient;
	/* REMOVE THE TWO LINES BELOW */
	struct _O_CURVE *prev;
	struct _O_CURVE *next;
};
typedef struct _O_CURVE O_CURVE;

 /* REMOVE THIS DATA TYPE FROM THE CODE */
struct _O_CURVE_FAMILY
{
	struct _O_CURVE *first;
	struct _O_CURVE *last;
};
typedef struct _O_CURVE_FAMILY O_CURVE_FAMILY;


 /* A Node of an Interface: */
struct _NODE
{
	int _boundary;		/* Mandatory first element,  see notes on
				 *  boundary macros below */
	struct _NODE *obj;	/* refers back to node, see boundary macro */
	struct _HYPER_SURF_BDRY *hsb;

	struct _INTERFACE *interface;
	POINT *posn;

	struct _CURVE **in_curves;	/* Pointer to Set of In Curves */
	struct _CURVE **out_curves;	/* Pointer to Set of Out Curves */
};
typedef struct _NODE NODE;


#if defined(THREED)
 /* Allowable neighbor objects to tris */

union _TRI_NEIGHBOR
{
	struct _TRI      *tri;
	struct _BOND_TRI *btri;
	POINTER          ptr;
	int              index;
};
typedef union  _TRI_NEIGHBOR TRI_NEIGHBOR;

enum _TRI_STORAGE_TYPE {
	MIN_TRI_STORAGE   = 0x001,
	TRI_PLUS_NORMAL   = 0x010,
	FULL_TRI_GEOMETRY = 0x100
};
typedef enum _TRI_STORAGE_TYPE TRI_STORAGE_TYPE;

enum _TRI_PHY_BC_TYPE {
        IN_FLOW   = 0x001,
        OUT_FLOW  = 0x010,
        NEUMANN   = 0x100,
        CONST_P   = 0x002,   // constant post shock state, for double mach reflection
	SUBDOMAIN = 0x004    // for tris in the subdomain
};
typedef enum _TRI_PHY_BC_TYPE TRI_PHY_BC_TYPE;

struct _TRI
{
	POINT *__pts[3];
	/* object adj to edge ij */
	TRI_NEIGHBOR    neighbor[3];
	struct _SURFACE	*surf; 	 /* surface in which the triangle lies */
	struct _TRI     *prev;
	struct _TRI     *next;
	int             boundary; /* tri bonds on interface curves? */
        POINTER         st;       // state on tri
        int             BC_type;  // physical boundary type, now only associate with one
                                  // edge, possibly need to associate with edge directly.
        int             id;       // indx for computing
        double     **Lmass_matrix;
        double     **mass_inv;
        int             redo_limiting; // limit_1st[4];
	union
	{
	    int		   _index;
	    POINTER 	   _workspace;
	    struct _C_BOND **_tri_cross_list;
	    int		   _icoords[3];
	    bool           _projection_computed;
            bool           _modified;
            /* The following is used in ellip for 3d tetrahedral mesh */
            struct
            {
                    char c0;
                    char c1;
                    char c2;
                    char c3;
            } c;
	} private_data;
};
typedef struct _TRI TRI;

/* Moved from intfc/isub.c */
struct _TRI_Plus_normal {
        TRI Tri;
        float _nor[3];            // normal to the tri plane
        float _sqr_norm;          // sqr(n1) + sqr(n2) +sqr(n3)
};
typedef struct _TRI_Plus_normal TRI_Plus_normal;

struct _TRI_FullGeometry {
        TRI_Plus_normal TriPN;
        float _length_side[3];    // length of sides
        float _sv_store[3][3];    // storage for _side_vector
        float *_side_vector[3];   // side vectors v[i] = p[i+1]-p[i]
        double  centroid[3];
        float  area;
        float  _edge_flux_store[3][5][4]; //side:point:flux_vector
        int    _ef_flag[3];                // edge_flux_set flag
        int    _e_type[3];                 // edge type flag
        float  diam;              // incircle diam of the tri = 2.0*2.0*area/(sum of side length)
};
typedef struct _TRI_FullGeometry TRI_FullGeometry;

#define Tri_normal_vector(_tri_)  (((TRI_Plus_normal*)(_tri_))->_nor)
#define sqr_normal_vector(_tri_)  (((TRI_Plus_normal*)(_tri_))->_sqr_norm)
#define fg_sv_store(_tri_)        (((TRI_FullGeometry*)(_tri_))->_sv_store)
#define fg_side_vector(_tri_)     (((TRI_FullGeometry*)(_tri_))->_side_vector)
#define fg_length_side(_tri_)     (((TRI_FullGeometry*)(_tri_))->_length_side)
#define fg_centroid(_tri_)        (((TRI_FullGeometry*)(_tri_))->centroid)
#define fg_area(_tri_)            (((TRI_FullGeometry*)(_tri_))->area)
#define fg_diam(_tri_)            (((TRI_FullGeometry*)(_tri_))->diam)

#define fg_side_flx(_tri_)      (((TRI_FullGeometry*)(_tri_))->_edge_flux_store)
#define fg_sf_flag(_tri_)       (((TRI_FullGeometry*)(_tri_))->_ef_flag)
#define fg_e_type(_tri_)       (((TRI_FullGeometry*)(_tri_))->_e_type)

/* End of Moved from intfc/isub.c */ 

typedef struct {struct _TRI tri; struct _SURFACE *s;} TRI_LIST_HEAD;

struct _SURFACE
{
	int _boundary;		/* Mandatory first element,  see notes on
				 *  boundary macros below */
	struct _SURFACE *obj;	/* refers back to surface, see boundary macro */
	struct _HYPER_SURF *hs;

	/* Topological Part */

	struct _INTERFACE *interface;

	/*
	 * SURFACE BOUNDARIES.  The boundary of a surface consists of
	 * a set of curves gathered into two classes,  pos_curves and
	 * neg_curves.  A given curve is defined as positive with respect
	 * the surface if its orientation agrees with that of the adjoining
	 * triangles that connect the curve to the surface, negative otherwise.
	 * Triangles are assumed to be oriented in a counter clockwise manner
	 * where observed from the side into which the normal vector points.
	 * Thus if the vertices of a triangle are p[0], p[1], p[2], then the
	 * normal has direction given by the vector cross product
	 * (p[1] - p[0])X(p[2] - p[0]).
	 * This means that a curve is positive with respect to a surface if
	 * each bond on that curve statisfies bond->start = p[i] and
	 * bond->end = p[(i+1)%3] for the triangle connected to that bond.
	 * Similarlly the curve is negative if bond->start = p[i] and
	 * bond->end = p[(i-1)%3].  It is assumed that all triangles on
	 * a surface are oriented consistently.
	 */

	struct _CURVE **pos_curves;	/* pos orient curves in bdry */
	struct _CURVE **neg_curves;	/* neg orient curves in bdry */

	struct _C_CURVE **c_curves;	/* c_curves on surface */

	TRI_LIST_HEAD _First_tri;
	struct _TRI  _Last_tri;

	/* Representation Specific Part */

	int num_tri;
};
typedef struct _SURFACE SURFACE;

struct _O_SURFACE
{
	struct _SURFACE *surface;
	ORIENTATION	orient;
};
typedef struct _O_SURFACE O_SURFACE;

struct _WEDGE{
	SURFACE	*s;
	TRI	**tris;
	int	n_tris;

};  
typedef struct _WEDGE WEDGE;

struct _NEIGHBORHOOD{
	struct  _POINT	*posn;
	struct  _WEDGE	*wedge;
	int		n_wedges;
	int		n_max_wedges;
};
typedef struct _NEIGHBORHOOD NEIGHBORHOOD;

#endif /* defined(THREED) */

	/* Hypersurface Data structures */

struct _HYPER_SURF
{
	union {
		int	*bptr;
#if defined(ONED)
		POINT	*p;
#endif /* defined(ONED) */
#if defined(TWOD)
		CURVE	*c;
#endif /* defined(TWOD) */
#if defined(THREED)
		SURFACE	*s;
#endif /* defined(THREED) */
	} obj;

	struct _INTERFACE *interface;
	struct _HYPER_SURF *hs;
	/*
	 * NOTE on components.  The positive side of a hypersurface is by
	 * definition the side into which the hypersurface normal points.
	 *
	 * For points in 1D this is the right side of the point and the
	 * negative side is the left side.
	 *
	 * For curves in 2D the positive side is the right side of curve
	 * and the negative side is the left side of the curve when
	 * viewed in a direction aligned with the curve tangent (ie from the
	 * start to end of a bond).
	 *
	 * For surfaces in 3D the positive side is defined as the side
	 * into which the normal points. The normal vector on a triangle
	 * has direction given by the vector cross product (p1 - p0)X(p2 - p0)
	 * so standing on point p0 and looking into the normal direction we
	 * are looking into the local positive side of surface.  The negative
	 * side is of curve the side opposite to the positive side.
	 */

	COMPONENT pos_comp, neg_comp;
	int pp_index;  /* Identifies families of connected hypersurfaces */
};
typedef struct _HYPER_SURF HYPER_SURF;


union _HYPER_SURF_ELEMENT {
	byte	bt;
#if defined(TWOD)
	BOND	b;
#endif /* defined(TWOD) */
#if defined(THREED)
	TRI	tri;
#endif /* defined(THREED) */
};
typedef union _HYPER_SURF_ELEMENT HYPER_SURF_ELEMENT;

struct _HYPER_SURF_BDRY
{
	union {
		int	*bptr;
#if defined(TWOD)
		NODE	*n;
#endif /* defined(TWOD) */
#if defined(THREED)
		CURVE	*c;
#endif /* defined(THREED) */
	} obj;
	struct _INTERFACE *interface;
	struct _HYPER_SURF_BDRY *hsb;
};
typedef struct _HYPER_SURF_BDRY HYPER_SURF_BDRY;

#if defined(TWOD) && defined(CONSERVATIVE_ALG)
struct _PT_INDX{
       float    coords[MAXD];
       int      indx;
       int      curv_id;
};
typedef struct _PT_INDX  PT_Indx; 

struct _G_INDX_CURVE{
        int      curv_id;
        int      wave_type;       /* INT. indx of the curve */
        int      pp_node;
        int      num;  
        PT_Indx  *Pt_indx; 
        struct   _G_INDX_CURVE *prev;
        struct   _G_INDX_CURVE *next;  
}; 
typedef struct _G_INDX_CURVE G_Indx_Curve;
         
struct _TIME_SPACE_INTFC2D{
       BOND_TRI    *btris; 
       int         num_tris;       /* Total number of tris */ 
       int         num_new_tris;   /* Number of tris with 
                                          NEW BOND as one of the edges  */
       int         num_old_tris;   /* Number of tris with 
                                          OLD BOND as one of the edges  */
};
typedef struct _TIME_SPACE_INTFC2D TIME_SPACE_INTFC2D; 
#endif /* if defined(TWOD) && defined(CONSERVATIVE_ALG) */

struct _COMP_LIST {
	int ncomps, max_ncomps;
	COMPONENT *comps;
	bool (*_is_comp_in_list)(COMPONENT,struct _COMP_LIST*,
				    struct _INTERFACE*);
	void (*_add_comp_to_list)(COMPONENT,struct _COMP_LIST*,
				  struct _INTERFACE*);
};
typedef struct _COMP_LIST COMP_LIST;

 /* Interface Structure: */
struct _INTERFACE
{
	struct _HYPER_SURF	**hss;
	struct _HYPER_SURF_BDRY **hses;

#if defined(ONED)
	struct _POINT	**points; /* Pointer to Set of Points */
#endif /* defined(ONED) */
	struct _NODE	**nodes;	/* Pointer to Set of Nodes */
	struct _CURVE	**curves;	/* Pointer to Set of Curves */
#if defined(THREED)
	struct _SURFACE **surfaces;	/* Pointer to Set of Surfaces */
	struct _C_CURVE **c_curves;	/* c_curves on interface */
#endif /* defined(THREED) */

	int		dim;		/* Dimension of Imbedding Space */
	int		num_points;	/* Total from curves */

	/* Internal Variables: */
	struct Table	*table;	/* Pointer to Interface Table */
	bool		modified;	/* Interface Recently Modified */
	bool            _interface_reconstructed;
	int		rect_bdry_type[MAXD][2];
	POINTER		e_comps;

#if defined(CONSERVATIVE_ALG)
        G_Indx_Curve    *g_indx_curves; 
#endif /* if defined(CONSERVATIVE_ALG) */
};
typedef struct _INTERFACE INTERFACE;

#define interface_reconstructed(intfc) ((intfc)->_interface_reconstructed)

 /* Node with curves angle ordered */
 /* documentation needed for structure elements */
struct _O_NODE
{
	struct _O_NODE	*prev, *next;
	struct _NODE	*node;
	struct _NODE	**nopp;
	struct _CURVE	**nc;
	POINT		**pt;
	float		*ang;
	ORIENTATION	*orient;
	int		num_c;
};
typedef struct _O_NODE O_NODE;

/* Pointer linkage used in hash table */

struct _P_LINK {
	POINTER pl, pr;
};
typedef struct _P_LINK P_LINK;

/*
*		Block based interface reconstruction structure:
*/
#if defined(THREED) || defined(TWOD)

struct _BBI_POINT {             /* Block based interface point */
#if defined(THREED)
        SURFACE         *s;
#endif /* defined(THREED) */
        CURVE           *c;
        POINT           *p;
#if defined(TWOD)
        HYPER_SURF      *hs;
        COMPONENT       lcomp, ucomp;   /* add for 2d bond construction */
#endif /* defined(TWOD) */
};
typedef struct _BBI_POINT BBI_POINT;

#endif /* defined(THREED) || defined(TWOD)  */

#if defined(THREED)

struct _BLK_INFO {
        SURFACE         **surfs;
        CURVE           **curves;
        TRI             **cur_tris;
        int             num_surfs;
        int             num_curves;
};
typedef struct _BLK_INFO BLK_INFO;

enum _BLK_TYPE {
        UNKNOWN_BLOCK = -1,
        BDRY_BLOCK = 0,
        COMP1_BLOCK = 1,
        COMP2_BLOCK = 2,
        COMP3_BLOCK = 3
};
typedef enum _BLK_TYPE BLK_TYPE;

struct _BLK_CRX {
        COMPONENT       ***comp;
        int             ***ix;
        int             ***iy;
        int             ***iz;
        int             num_comps;
        int             nv[8];
        COMPONENT       comps[8];
        BBI_POINT       ****crx;
        BBI_POINT       ***curve_crx;
        BBI_POINT       *node_crx;
        BBI_POINT       *crx_store;
        BLK_INFO        *blk_info;
        BLK_TYPE        blk_type;
};
typedef struct _BLK_CRX BLK_CRX;

struct _BLK_TRI {
        TRI             *first[3];
        BOND            *bonds[3];
        CURVE           *curves[3];
        SURFACE         *surfs[3];
        int             num_null_sides[3];
        int             num_tris[3];
        int             num_curves;
        int             num_surfaces;
        BLK_INFO        *blk_info;
};
typedef struct _BLK_TRI BLK_TRI;

#endif /* defined(THREED) */

#if defined(TWOD)

struct _BLK_BOND {
        BOND            **cur_bonds;
        CURVE           **curves;
        int             num_curves;
        BOND            **first;
        int             *num_bonds;
        /*
        int             num_null_sides;
        */
};
typedef struct _BLK_BOND BLK_BOND;

/* Join two different curves bond */
struct _JC_BLK_BOND{
        CURVE      *c0;
        CURVE      *c1;
        BOND       *jc_bond;     /* The bond made to join c0 and c1 */
        struct _JC_BLK_BOND *prev;
        struct _JC_BLK_BOND *next;
};
typedef struct _JC_BLK_BOND JC_BLK_BOND;
struct _BLK_CRX2 {
        int             blkic[MAXD]; /* blk icoords */
        COMPONENT       ***comp;
        int             ***ix;
        int             ***iy;
        int             ***iz;
        COMPONENT       comps[4];   /* save different comp index of this blk corner */
        int             num_comps; 
        int             num_curves; /* Number of curves inside this blk */
        int             num_waves;  /* The number of different waves inside blk*/
                                    /* Different curves may belong to the same wave */
        COMPONENT       pos_comp;
        COMPONENT       neg_comp;
        BBI_POINT       ****crx;    /* For the record, current max. # of crx on a grid line
                                     * is limited to 5 */
        BBI_POINT       *crx_store;
        int             ***n_crx;   /* Number of CRX on each grid line, multi crxs are possible */
        BLK_TYPE        blk_type;   
        int             bdry;       /* The flags whether the boundary curves appear in blk */
        int             m_crx;      /* If there are mixed types of crxings(tracked, untracked), 
                                     * m_crx = YES */
        void        (*assign_2dblk_type)(struct _BLK_CRX2*,int,BBI_POINT  *crxs[]);
        int         (*trk_wv_val)(CURVE*); 
};
typedef struct _BLK_CRX2 BLK_CRX2;

#endif /* if defined(TWOD) */

/*
*				CROSS Structure:
*
*	This structure describes the an intersection point on an INTERFACE.
*	It is a highly redundant object.   The NODE entry is a NODE
*	describing the crossing in terms of position and sets of in and
*	out curves.  The other entries consist of the POINT p of
*	intersection, the intersecting BONDS b1,b2 and the intersecting
*	CURVES c1,c2.
*
*	The start and end verticies and sides identify the sides cut
*	by the intersection line.  If this line crosses a vertex, that
*	vertex index is recorded in the start or end field corresponding
*	to the start or end of the bond.  Otherwise these fields are set
*	to ERROR.  Similarly the start and end fields identify the sides
*	of the triangles cut by the intersection line.  If the start or
*	end occurs at a vertex,  the corresponding side field is ERROR.
*/

#if defined(THREED)

struct _C_SURF_FLAG
{
	bool _on_bdry;        /* true if the crossing point is on the
				  * boundary (edge or vertex) of the
				  * correspond triangle */
	bool _edge_vertex;    /* only used when _on_bdry is true,  true if
				  * the crossing point is not a triangle
				  * vertex */
	int     _tri_side_index; /* only used when both _on_bdry iand
			          * _edge_vertex are true,  gives the side of
			          * triangle upon which the crossing point
				  * lies */
};
typedef struct _C_SURF_FLAG C_SURF_FLAG;

#define cs_tri_side_index(f) (f)._tri_side_index
#define cs_on_bdry(f) (f)._on_bdry
#define cs_edge_vertex(f) (f)._edge_vertex

struct _C_SURF           /* records detailed crossing information */
{
    	struct	_TRI	*t,          /* pair of intersecting triangles */
			*prev_t,     /* prev, next intscting tris */
			*next_t;
	C_SURF_FLAG 	_flag_start,  /* new stuff, records detailed crossing */
			_flag_end;    /* information */
};
typedef struct _C_SURF C_SURF;

#define	cs_flag_start(cs) (cs)._flag_start
#define	cs_flag_end(cs)	  (cs)._flag_end

struct _C_BOND   /* cross bond */
{
	struct	_C_BOND	*prev;	/* fields present in a normal BOND */
	struct	_C_BOND	*next;
	struct	_POINT	*start;
	struct	_POINT	*end;
	struct	_BOND	*bond;
	struct	_C_SURF	s[2];  /* 2 structures, one for each surface */
	struct _C_CURVE	*c_curve;
};
typedef struct _C_BOND   C_BOND;

#define	cb_flag_start(cb,i)           cs_flag_start((cb)->s[i])
#define	cb_start_tri_side_index(cb,i) cs_tri_side_index(cb_flag_start(cb,i))
#define is_start_on_bdry(cb,i)        cs_on_bdry(cb_flag_start(cb,i))
#define is_start_to_edge(cb,i)        cs_edge_vertex(cb_flag_start(cb,i))
#define is_start_to_vertex(cb,i)      (!is_start_to_edge(cb,i))

#define	cb_flag_end(cb,i)           cs_flag_end((cb)->s[i])
#define	cb_end_tri_side_index(cb,i) cs_tri_side_index(cb_flag_end(cb,i))
#define is_end_on_bdry(cb,i)        cs_on_bdry(cb_flag_start(cb,i))
#define is_end_to_edge(cb,i)        cs_edge_vertex(cb_flag_end(cb,i))
#define is_end_to_vertex(cb,i)      (!is_end_to_edge(cb,i))

struct _C_CURVE    /* 3d Cross (intersection) Curve */
{
	/* Topological Part: */
	INTERFACE *interface;
	NODE	  *start;
	NODE	  *end;

	SURFACE	  *s[2];
	int	  boundary;
	/* Representation Specific Part: */
	C_BOND	  *first;
	C_BOND	  *last;
	int	  num_points;
	CURVE	  *curve;
};
typedef struct _C_CURVE C_CURVE;
#endif /* defined(THREED) */

struct _Cross
{
	struct _Cross *prev;
	struct _Cross *next;

#if defined(ONED)
	POINT	**pt;	/* Points that have crossed */
	int     npt;
#endif /* defined(ONED) */

#if defined(TWOD)
	POINT *p;		/* Intersection POINT */
	BOND *b1, *b2;		/* BONDS that Intersect */
	CURVE *c1, *c2;		/* CURVES that Intersect */
#endif /* defined(TWOD) */

#if defined(THREED)
	C_CURVE *c_curve;
#endif /* defined(THREED) */
};
typedef struct _Cross CROSS;

struct _INDEX {
	int I1, I2, I3;  
	int tempspace;   
	POINTER  tri_workspace;
};		         
typedef struct _INDEX INDEX;

#define	Is_outside(icoord,G,dir)					\
	((icoord)[(dir)] < 0 || (icoord)[(dir)] >= (G)[(dir)])

 /* Macros for accessing hyper surface data */

#define	hyper_surf_list(intfc) (intfc)->hss
#define Hyper_surf(ptr)	       ((ptr)->hs)
#define Hyper_surf_index(ptr)  Hyper_surf(ptr)->pp_index

#if defined(ONED)
#define Point_of_hs(hs)	       (hs)->obj.p
#else /* defined(ONED) */
#define Point_of_hs(hs)	       ((POINT*) NULL)
#endif /* defined(ONED) */

#if defined(TWOD)
#define Curve_of_hs(hs)	       (hs)->obj.c
#else /* defined(TWOD) */
#define Curve_of_hs(hs)	       ((CURVE*) NULL)
#endif /* defined(TWOD) */

#if defined(THREED)
#define Surface_of_hs(hs)      (hs)->obj.s
#else /* defined(THREED) */
#define Surface_of_hs(hs)      ((SURFACE*) NULL)
#endif /* defined(THREED) */



 /* Macros for accessing hyper surface element data */
#define Hyper_surf_element(ptr)	((HYPER_SURF_ELEMENT *) (ptr))
#define Bond_of_hse(hse)	((BOND *) hse)
#if defined(THREED)
#define Tri_of_hse(hse)		((TRI *) hse)
#endif /* defined(THREED) */

/* Macros for accessing hyper surface boundary data */
#define	hyper_surf_bdry_list(intfc)	(intfc)->hses
#define Hyper_surf_bdry(ptr)   		((ptr)->hsb)

#if defined(TWOD)
#define Node_of_hsb(hsb)       		(hsb)->obj.n
#else /* defined(TWOD) */
#define Node_of_hsb(hsb)       		((NODE*)NULL)
#endif /* defined(TWOD) */

#if defined(THREED)
#define Curve_of_hsb(hsb)      		(hsb)->obj.c
#else /* defined(THREED) */
#define Curve_of_hsb(hsb)      		((CURVE*)NULL)
#endif /* defined(THREED) */


/*
*		Node, Curve and Surface Boundary Flags:
*
*	These flags use 6 bits, 2 per coordinate direction.
*	The boundary flags are stored in bits, the first 2 for the x 
*	coordinate, the next 2 for the y and the third 2 for z. A two bit
*	flag is thus 0 = INSIDE, 1 = BOTTOM, 2 = TOP, 3 unassigned. Thus
*	the upper x,y,z corner has the binary flag: 101010 = (octal) 022,
*	and the upper z face has the binary flag: 100000 = (octal) 040.
*
*	In summary,  each flag is a binary number of the form:
*	flag = (zzyyxx),  where zz, yy, or xx = 00, 01, 10, or 11 (binary).
*
*	IMPORTANT NOTE:	The following macro assumes that the boundary field
*	is the first element of any of the possible objects to which
*	hs or hsb can point.  The current legal objects are
*
*		SURFACE, CURVE, NODE (2d only), and POINT (1d only)
*
*	The arrangement of data allows the boundary information to
*	be extracted from these structures without an explicit knowledge
*	of the spatial dimension.  This in turn provides an increase
*	in run time efficiency.
*
*	Note:  In many cases, the specific boundary is not specified.  In
*	fact, as of this writing (9/23/96), only parts of the 3d code use the
*	mapping described above.  The rest of the time, we only care whether
*	the object is on the boundary or not.  A set of macros is provided
*	below using BDRY_MASK to serve this purpose.  They set or check all
*	six of the lower order bits.  This is necessary because the higher
*	order bits are being used in certain parts of the code.  This will
*	require some care.  Some of the bit field declarations may be local
*	to facilitate development, and due to the fact that they are used
*	only locally. These fields may need to be preserved globally, however,
*	which is the reason for the BDRY_MASK macros below.  One cannot
*	simply set theboundary field to YES or NO, as this will not preserve
*	the higher order bits.	
*/

enum {
	INSIDE 	  = 000,
	BOTTOM	  = 001,
	TOP	  = 002,
	BDRY_MASK = 077
};

#define Boundary(ptr)		(ptr)->_boundary
#define Boundary_hs(hs)		(*(hs )->obj.bptr)
#define Boundary_hsb(hsb)	(*(hsb)->obj.bptr)

#define set_is_bdry(ptr)	(Boundary(ptr)     |=  BDRY_MASK)
#define set_not_bdry(ptr)	(Boundary(ptr)     &= ~BDRY_MASK)
#define is_bdry(ptr)		(Boundary(ptr)     &   BDRY_MASK)
#define is_bdry_hs(ptr)		(Boundary_hs(ptr)  &   BDRY_MASK)
#define is_bdry_hsb(ptr)	(Boundary_hsb(ptr) &   BDRY_MASK)

	/* Macros for evaluation of boundary fields */
#define xbdry_side(f) 	((f)        & 03)
#define ybdry_side(f)	((f >> 2)   & 03)
#define zbdry_side(f)	((f >> 4)   & 03)
#define bdry_side(f,i)	((f >> 2*i) & 03)

	/* Macros for setting of boundary fields */
#define set_xbdry_side(f,v)						\
	( (f) = ( ((f) & ~03         ) | (((v) & 03)       ) ) )
#define set_ybdry_side(f,v)						\
	( (f) = ( ((f) & ~014        ) | (((v) & 03) << 2  ) ) )
#define set_zbdry_side(f,v)						\
	( (f) = ( ((f) & ~060        ) | (((v) & 03) << 4  ) ) )
#define set_bdry_side(f,d,v)						\
	( (f) = ( ((f) & ~(03 << 2*d)) | (((v) & 03) << 2*d) ) )



 /* Macros for accessing component information */

#define positive_component(ptr)	(Hyper_surf(ptr)->pos_comp)
#define negative_component(ptr)	(Hyper_surf(ptr)->neg_comp)

#define	comp_is_on_hyper_surf(hs,comp)					\
		(((comp) == NO_COMP) || 				\
		(negative_component(hs) == (comp)) || 			\
		(positive_component(hs) == (comp)))

#define	comps_are_on_hyper_surf(hs,l_comp,r_comp)			\
	((((l_comp)==NO_COMP) || (negative_component(hs)==(l_comp)))	\
	&&								\
	(((r_comp)==NO_COMP) || (positive_component(hs)==(r_comp))))


 /* Sides of Boundary Square -- for nearest_boundary() */

enum _BDRY_SIDE {
    NOT_A_BDRY  = -1,
    LEFT_BDRY	=  0,
    RIGHT_BDRY	=  1,
    LOWER_BDRY	=  2,
    UPPER_BDRY	=  3,
    ZMIN_BDRY	=  4,
    ZMAX_BDRY	=  5
};
typedef enum _BDRY_SIDE BDRY_SIDE;

 /* Values for rectangular boundary type */

#define rect_boundary_type(intfc,i,j)	((intfc)->rect_bdry_type[i][j])

enum {
	UNKNOWN_BOUNDARY_TYPE = -3,
	SUBDOMAIN_BOUNDARY    =  1,
	PERIODIC_BOUNDARY     =  1,
#if defined(USE_OVERTURE)
        AMR_SUBDOMAIN_BOUNDARY,
#endif /* defined(USE_OVERTURE) */
	REFLECTION_BOUNDARY,
	MIXED_TYPE_BOUNDARY,
	FIRST_USER_BOUNDARY_TYPE
#if defined(USE_OVERTURE)
#endif /* defined(USE_OVERTURE) */
};

#if defined(USE_OVERTURE)
#define buffered_boundary_type(b_type)                                  \
        (((b_type) == SUBDOMAIN_BOUNDARY) || ((b_type) == REFLECTION_BOUNDARY) \
         || ((b_type) == AMR_SUBDOMAIN_BOUNDARY))
#else /* if defined(USE_OVERTURE) */
#define buffered_boundary_type(b_type)                                  \
        (((b_type) == SUBDOMAIN_BOUNDARY) || ((b_type) == REFLECTION_BOUNDARY))
#endif /* if defined(USE_OVERTURE) */

#define	is_pp_node(node)						\
	(is_subdomain_node(node) || is_virtual_fixed_node(node))

enum {
	NO_COMP          = -1,
	ONFRONT          = -2,
	NEW_COMP         = -3,
	ON_RECT_BOUNDARY = -4,
	UNUSED_COMP      = -5,
        ERROR_COMP       = -6
};

enum _USE_BOUNDARIES {
	NO_BOUNDARIES         = 0,
	INCLUDE_BOUNDARIES    = 1,
	NO_SUBDOMAIN          = 2
};
typedef enum _USE_BOUNDARIES USE_BOUNDARIES;

#define	skip_boundary_hs(hs,bdry)					\
	(((bdry == NO_BOUNDARIES) && is_bdry_hs(hs)) ||			\
	 ((bdry == NO_SUBDOMAIN) && is_subdomain_boundary(hs)))


#define	min3(xx,yy,zz) 	(min((xx),min((yy),(zz))))
#define	max3(xx,yy,zz) 	(max((xx),max((yy),(zz))))
#define	min4(ww,xx,yy,zz) (min((ww),min3((xx),(yy),(zz))))
#define	max4(ww,xx,yy,zz) (max((ww),max3((xx),(yy),(zz))))
#define same_sign(c1,c2)                                                \
        ((((c1) > 0.0 && (c2) > 0.0) || ((c1) < 0.0 && (c2) < 0.0)) ? \
        YES : NO)

#define within_interval(x1,x2,x)                                \
        (((x1) <= (x) && (x) <= (x2)) || ((x1) >= (x) && (x) >= (x2)) \
        ? YES : NO)

enum {
	MAXNCORNERS = (1<<MAXD)	/* MAX no. corners */
};
#define	Ncorners(dim)		(1<<dim)	/* no. corners of hyper cube */

#define Dot3d(A,B)							\
	((A)[0]*(B)[0] + (A)[1]*(B)[1] + (A)[2]*(B)[2])

#define	Mag3d(A) sqrt(Dot3d(A,A))

#define QDot3d(A,B)							\
	(A ## 0*B ## 0 + A ## 1*B ## 1 + A ## 2*B ## 2)

#define Cross3d(B,C,ans)						\
	{								\
		(ans)[0] = ((B)[1])*((C)[2]) - ((B)[2])*((C)[1]);	\
		(ans)[1] = ((B)[2])*((C)[0]) - ((B)[0])*((C)[2]);	\
		(ans)[2] = ((B)[0])*((C)[1]) - ((B)[1])*((C)[0]);	\
	}

#define QCross3d(B,C,ans)						\
		ans ## 0 = B ## 1*C ## 2 - B ## 2*C ## 1;		\
		ans ## 1 = B ## 2*C ## 0 - B ## 0*C ## 2;		\
		ans ## 2 = B ## 0*C ## 1 - B ## 1*C ## 0

#define Det3d(a,b,c) ( (a)[0]*(b)[1]*(c)[2] + (a)[1]*(b)[2]*(c)[0] + (a)[2]*(b)[0]*(c)[1] - (a)[0]*(b)[2]*(c)[1] - (a)[1]*(b)[0]*(c)[2] - (a)[2]*(b)[1]*(c)[0] )

#define QDet3d(a,b,c) ( a ## 0*b ##1*c ## 2 + a ## 1*b ## 2*c ## 0 + a ## 2*b ## 0*c ## 1 - a ## 0*b ## 2*c ## 1 - a ## 1*b ## 0*c ## 2 - a ## 2*b ## 1*c ## 0 )

#define difference(B,C,ans,dim)                                         \
        {                                                               \
            int i;                                                      \
            for (i = 0; i < (dim); ++i)                                 \
            {                                                           \
                (ans)[i] = (B)[i] - (C)[i];                             \
            }                                                           \
        }

 /* Position of a Point as an array */

#define Coords(p)	((p)->_coords)
#define COORDS(P)	((P)._coords)

/* Stored normal vector at point */
#if defined(THREED)
#define	normal_at_point(p) ((p)->_nor)
#endif /* defined(THREED) */

 /* Length of a Bond */

#define  bond_length(b)  ((b)->length)

 /* Separation between two POINTS: */

#define scaled_separation(p,q,h,dim)					\
	_scaled_separation(Coords(p),Coords(q),h,dim)

#if defined(ONED) && !defined(TWOD) && !defined(THREED)
#define separation(p,q,dim)	fabs(Coords(q)[0] - Coords(p)[0])
#endif /* defined(ONED) && !defined(TWOD) && !defined(THREED) */

#if defined(TWOD) && !defined(ONED) && !defined(THREED)
#define  separation(p,q,dim)						\
		hypot(Coords(q)[0] - Coords(p)[0],Coords(q)[1] - Coords(p)[1])
#endif /* defined(TWOD) && !defined(ONED) && !defined(THREED) */

#if defined(THREED) && !defined(ONED) && !defined(TWOD)
#define  separation(p,q,dim)  sqrt(sqr(Coords(q)[0] - Coords(p)[0])	\
	+ sqr(Coords(q)[1] - Coords(p)[1]) + sqr(Coords(q)[2] - Coords(p)[2]))
#endif /* defined(THREED) && !defined(ONED) && !defined(TWOD) */

#if (defined(ONED) && defined(TWOD)) || (defined(ONED) && defined(THREED)) || (defined(TWOD) && defined(THREED))
IMPORT float separation(POINT *p, POINT *q, int dim);

#endif /* (defined(ONED) && defined(TWOD)) || (defined(ONED) && defined(THREED)) || (defined(TWOD) && defined(THREED)) */


 /* Is a CURVE closed ? */

#define	 is_closed_curve(c)	 ((c)->start == (c)->end)


 /* Is a NODE the joining point of a simple closed curve? */

#define  is_node_of_closed_curve_only(node) 			\
	 (    ((node)->in_curves != NULL)			\
	   && ((node)->out_curves != NULL)			\
	   && (node)->in_curves[0]				\
	   && ((node)->in_curves[0] == (node)->out_curves[0])	\
	   && !(node)->in_curves[1]				\
	   && !(node)->out_curves[1]    )

 /* The bond on a given curve that attaches to a node */

#define Bond_at_node(curve,orient)					\
	((orient) == POSITIVE_ORIENTATION ? (curve)->first : (curve)->last)

#define Bond_at_node_of_o_curve(oc)					\
	((oc)->orient == POSITIVE_ORIENTATION ? (oc)->curve->first : 	\
					       (oc)->curve->last)

#define Bond_at_opp_node_of_o_curve(oc)					\
	((oc)->orient == NEGATIVE_ORIENTATION ? (oc)->curve->first : 	\
						(oc)->curve->last)


 /* Returns the bond following a bond, as defined by the orientation */

#define Following_bond(bond,orient)					\
	((orient) == POSITIVE_ORIENTATION ? (bond)->next : (bond)->prev)


 /* Returns the point associated with a bond and an orientation */

#define Point_of_bond(b,orient)						\
	(((orient) == POSITIVE_ORIENTATION) ? (b)->start : (b)->end)


 /* Returns the first point out along the curve from the node */

#define Point_adjacent_to_node(curve,orient)				\
	((orient) == POSITIVE_ORIENTATION				\
		? (curve)->first->end : (curve)->last->start)


 /* Returns the node associated with a curve and an orientation */

#define Node_of(curve,orient)						\
	((orient) == POSITIVE_ORIENTATION ? (curve)->start : (curve)->end)

#define Node_of_o_curve(oc)						\
	(((oc)->orient == POSITIVE_ORIENTATION) ? (oc)->curve->start : 	\
						 (oc)->curve->end)

#define Opp_node_of_o_curve(oc)						\
	(((oc)->orient == NEGATIVE_ORIENTATION) ? (oc)->curve->start : 	\
						  (oc)->curve->end)

 /* Point flags fields for points */
#define Point_flags(p)		((p)->_point_flags)
#define	Boundary_point(p)	(Point_flags(p)._boundary)

#if defined(THREED)
#define Btris(b)		(b)->_btris
#define Point_of_tri(_tri_)	(_tri_)->__pts

/*	Macros controling tri list on surfaces */
#define surface_for_head_of_tri_list(htl) ((TRI_LIST_HEAD *) htl)->s
#define head_of_tri_list(s) (&(s)->_First_tri.tri)
#define tail_of_tri_list(s) (&(s)->_Last_tri)
#define first_tri(s)	(s)->_First_tri.tri.next
#define last_tri(s)	(s)->_Last_tri.prev
#define at_end_of_tri_list(tri,s)	((tri) == tail_of_tri_list(s))
#define at_start_of_tri_list(tri,s)	((tri) == head_of_tri_list(s))
#define no_tris_on_surface(s)		(at_end_of_tri_list(first_tri(s),s))



/*
*			TRI boundary flags:
*
*	The flag Boundary_tri(tri) flag specifies which sides of the triangle
*	lie on curves of the interface, according to the following
*	truth table.
*
*		   0  1  2  3  4  5  6  7
*	side 01:   N  Y  N  Y  N  Y  N  Y
*	side 12:   N  N  Y  Y  N  N  Y  Y
*	side 20:   N  N  N  N  Y  Y  Y  Y
*
*/

#define Boundary_tri(tri)	(tri)->boundary
#define	Bin_side(side) 	(1<<(side))

enum {
	BIN_SIDE01 = 0x1,
	BIN_SIDE12 = 0x2,
	BIN_SIDE20 = 0x4
};

#define is_side01_a_bond(_tri_)		(Boundary_tri(_tri_) & BIN_SIDE01)
#define is_side12_a_bond(_tri_)		(Boundary_tri(_tri_) & BIN_SIDE12)
#define is_side20_a_bond(_tri_)		(Boundary_tri(_tri_) & BIN_SIDE20)
#define is_side_bdry(_tri_,side)	(Boundary_tri(_tri_) & Bin_side(side))

#define set_01_bdry(bdry,yes_no)					\
	(bdry) = (((bdry) & ~BIN_SIDE01) | (yes_no))
#define set_12_bdry(bdry,yes_no)					\
	(bdry) = (((bdry) & ~BIN_SIDE12) | ((yes_no) << 1))
#define set_20_bdry(bdry,yes_no)					\
	(bdry) = (((bdry) & ~BIN_SIDE20) | ((yes_no) << 2))
#define set_side_bdry(bdry,side,yes_no)					\
	(bdry) = ( ((bdry) & ~ Bin_side(side)) | ((yes_no) << (side)) )

#define vertex_on_bond(_tri_,i)						\
	( Boundary_tri(_tri_) & (Bin_side(i) & Bin_side(Prev_m3(i))) )

 /* Macros for accessing tri data fields */

enum {
	GLOBAL_INDICES = 0,
	LOCAL_INDICES  = 1
};

#define	Surface_of_tri(_tri_)		((_tri_)->surf)
#define Tri_workspace(_tri_)		((_tri_)->private_data._workspace)
#define Tri_index(_tri_)		((_tri_)->private_data._index)
#define Tri_cross_list(_tri_)		((_tri_)->private_data._tri_cross_list)
#define	Tri_icoords(_tri_)		((_tri_)->private_data._icoords)
#define Tri_modified(_tri_)             ((_tri_)->private_data._modified)
#define	Tri_projection_computed(_tri_)					\
    ((_tri_)->private_data._projection_computed)

#define	Tri_neighbor(_tri_)	     (_tri_)->neighbor
#define	Tri_on_side01(_tri_)	     (Tri_neighbor(_tri_)[0].tri)
#define	Tri_on_side12(_tri_)	     (Tri_neighbor(_tri_)[1].tri)
#define	Tri_on_side20(_tri_)	     (Tri_neighbor(_tri_)[2].tri)
#define	Tri_on_side(_tri_,side)	     ((Tri_neighbor(_tri_) + (side))->tri)

#define	Bond_tri_on_side01(_tri_)    (Tri_neighbor(_tri_)[0].btri)
#define	Bond_tri_on_side12(_tri_)    (Tri_neighbor(_tri_)[1].btri)
#define	Bond_tri_on_side20(_tri_)    (Tri_neighbor(_tri_)[2].btri)
#define	Bond_tri_on_side(_tri_,side) ((Tri_neighbor(_tri_) + (side))->btri)

#define	Bond_on_side01(_tri_)	     (Bond_tri_on_side01(_tri_)->bond)
#define	Bond_on_side12(_tri_)	     (Bond_tri_on_side12(_tri_)->bond)
#define	Bond_on_side20(_tri_)	     (Bond_tri_on_side20(_tri_)->bond)
#define	Bond_on_side(_tri_,side)     (Bond_tri_on_side(_tri_,side)->bond)

#define	Neighbor_on_side01(_tri_)    (Tri_neighbor(_tri_)[0].ptr)
#define	Neighbor_on_side12(_tri_)    (Tri_neighbor(_tri_)[1].ptr)
#define	Neighbor_on_side20(_tri_)    (Tri_neighbor(_tri_)[2].ptr)
#define	Neighbor_on_side(_tri_,side) ((Tri_neighbor(_tri_) + (side))->ptr)

#define	Index_on_side01(_tri_)	     (Tri_neighbor(_tri_)[0].index)
#define	Index_on_side12(_tri_)	     (Tri_neighbor(_tri_)[1].index)
#define	Index_on_side20(_tri_)	     (Tri_neighbor(_tri_)[2].index)
#define	Index_on_side(_tri_,side)    ((Tri_neighbor(_tri_) + (side))->index)

#define Point_on_tri(_tri_,p)						\
	(((p)==Point_of_tri(_tri_)[0]) ||                               \
	 ((p)==Point_of_tri(_tri_)[1]) || 				\
	 ((p)==Point_of_tri(_tri_)[2]))

#define Vertex_of_point(_tri_,p)					\
	( (((p)==Point_of_tri(_tri_)[0])) ? 0 :				\
	  (((p)==Point_of_tri(_tri_)[1])) ? 1 :				\
	  (((p)==Point_of_tri(_tri_)[2])) ? 2 : ERROR)

#define Prev_side_at_vertex(_tri_,p)					\
	(Point_on_tri(_tri_,p) ? Prev_m3(Vertex_of_point(_tri_,p)) 	\
	: ERROR)

#define Next_side_at_vertex(_tri_,p)					\
	(Point_on_tri(_tri_,p) ? Vertex_of_point(_tri_,p) : ERROR)

#define Following_tri_at_vertex(_tri_,p,dir)				\
	( ((dir) == COUNTER_CLOCK) ?	Prev_tri_at_vertex(_tri_,p)	\
	: ((dir) == CLOCKWISE)	   ?	Next_tri_at_vertex(_tri_,p)	\
	:				NULL				)

#define Previous_side_at_vertex(_tri_,p,dir)				\
	( ((dir) == COUNTER_CLOCK) ?	Prev_side_at_vertex(_tri_,p)	\
	: ((dir) == CLOCKWISE)	   ?	Next_side_at_vertex(_tri_,p)	\
	:				ERROR				)

#define Previous_tri_at_vertex(_tri_,p,dir)				\
	( ((dir) == CLOCKWISE)	   ?	Prev_tri_at_vertex(_tri_,p)	\
	: ((dir) == COUNTER_CLOCK) ?	Next_tri_at_vertex(_tri_,p)	\
	:				NULL				)

#define Following_side_at_vertex(_tri_,p,dir)				\
	( ((dir) == CLOCKWISE)	   ?	Prev_side_at_vertex(_tri_,p)	\
	: ((dir) == COUNTER_CLOCK) ?	Next_side_at_vertex(_tri_,p)	\
	:				ERROR				)

#endif /* defined(THREED) */

#define Next_corner(n,nc)       (((n) + 1) % (nc))

#define Prev_corner(n,nc)       (((n) + (nc) - 1) % (nc))

#include <intfc/array.h>

#if defined(THREED)
#define Num_pos_curves_of_surface(surface)				\
		size_of_pointers((POINTER *)(surface)->pos_curves)

#define Num_neg_curves_of_surface(surface)				\
		size_of_pointers((POINTER *)(surface)->neg_curves)

#define Num_surfaces_bding_curve(curve)					\
		(Num_pos_surfaces_of_curve(curve) +			\
			Num_neg_surfaces_of_curve(curve))

#define Num_pos_surfaces_of_curve(curve)				\
		size_of_pointers((POINTER *)(curve)->pos_surfaces)

#define Num_neg_surfaces_of_curve(curve)				\
		size_of_pointers((POINTER *)(curve)->neg_surfaces)

#define Num_surfaces(intfc) size_of_pointers((POINTER *)(intfc)->surfaces)
#endif /* defined(THREED) */

#define Num_nodes(intfc) size_of_pointers((POINTER *)(intfc)->nodes)

#define Num_curves(intfc) size_of_pointers((POINTER *)(intfc)->curves)

#define Num_in_curves(node) size_of_pointers((POINTER *)(node)->in_curves)

#define Num_out_curves(node) size_of_pointers((POINTER *)(node)->out_curves)

#define new_address(intfc,p,ocad,ncad,nchks)				\
		_new_address(intfc,(POINTER) (p),(ocad),(ncad),(nchks))

enum {
	TABLE_ID = USER_MIN_MESG_ID + 1,
	CHUNK_ADDR_ID,
	CHUNK0_ID,
	FIRST_USER_MESSAGE_ID = USER_MIN_MESG_ID + 10000
};

#define chunk_id(i)		(CHUNK0_ID + (i))

		/* Structure Defining a Parallel Processing Grid: */

typedef struct {
	float   *dom[MAXD];     /* corner position of domain */
	int     gmax[MAXD];     /* # of subdomains in each dir */
	int	buf[MAXD];	/* shaded subdomain extension. every mesh
				block has the same buffer extension.	*/
	int	nn;		/* total number of nodes 		*/	
	RECT_GRID Global_grid;	/* Rect_grid of total region		*/
	RECT_GRID Zoom_grid;	/* Rect_grid for subdomain		*/
} PP_GRID; 

/*
*	Initialization structure for interface library
*/

struct _I_INIT_DATA {
	INIT_DATA	U_init_data;
	INTERFACE	*_intfc;
	int		_subdomains[3];		/*COMMAND LINE*/
	int		_buffer_zones[3];	/*COMMAND LINE*/
	bool		_pp_grid_set;		/*COMMAND LINE*/
};
typedef struct _I_INIT_DATA I_INIT_DATA;
#define	i_init_data(init)	((I_INIT_DATA*)(init))
#define i_intfc(init)		i_init_data(init)->_intfc
#define Top_grid(init)      	topological_grid(i_intfc(init))
#define	subdomains(init)	i_init_data(init)->_subdomains
#define	buffer_zones(init)	i_init_data(init)->_buffer_zones
#define	pp_grid_set(init)	i_init_data(init)->_pp_grid_set

#if defined(THREED)
enum _COORDINATE_PLANE {
    YZ_PLANE = 0,
    XZ_PLANE = 1,
    XY_PLANE = 2
};
typedef enum _COORDINATE_PLANE COORDINATE_PLANE;

enum _SURFACE_COLOR {
	pBLACK   = 0,
	pRED	 = 1,
	pBLUE	 = 2,
	pMAGENTA = 3,
	pGREEN	 = 4,
	pYELLOW	 = 5,
	pCYAN	 = 6,
	pWHITE	 = 7
};
typedef enum _SURFACE_COLOR SURFACE_COLOR;
#endif /* defined(THREED) */

#include <intfc/int_amr.h>

#include <intfc/table.h>
#include <intfc/userint.h>
#include <intfc/iprotos.h>

#endif /* !defined(_INT_H) */
