/*
*				imksurf.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Containing function of rebuilding interface within a mesh block.
*
*/

#if defined(THREED)

#define DEBUG_STRING "i_make_surf"

#include <intfc/int.h>

struct  _EG_CRX {
        BBI_POINT ****x_crx;
        BBI_POINT ****y_crx;
        BBI_POINT ****z_crx;
        BBI_POINT ****x_curve_crx;
        BBI_POINT ****y_curve_crx;
        BBI_POINT ****z_curve_crx;
        BBI_POINT ****node_crx;
        BBI_POINT *crx_store;
        COMPONENT ***comp;
        CURVE     **curves;
        int       num_curves;
};
typedef struct _EG_CRX EG_CRX;


LOCAL   void 	assign_blk_crx(BLK_CRX*,int,int,int,const EG_CRX*,bool);
LOCAL	void    alloc_grid_crx_mem(EG_CRX*,int*,int,bool);
LOCAL	void    free_grid_crx_mem(EG_CRX*,bool);
LOCAL	void 	make_grid_surfaces(BLK_CRX*,const EG_CRX*,int*,bool);
LOCAL	void 	reset_domain_comp(COMPONENT***,RECT_GRID);
LOCAL	void 	assign_positive_comp(float (*func)(POINTER,float*),POINTER,
			COMPONENT***,RECT_GRID,COMPONENT);
LOCAL	void 	assign_negative_comp(float (*func)(POINTER,float*),POINTER,
			COMPONENT***,RECT_GRID,COMPONENT);
LOCAL	void 	assign_two_comp_domain(float (*func)(POINTER,float*),POINTER,
			COMPONENT***,RECT_GRID,COMPONENT,COMPONENT);
LOCAL	void 	assign_intersection_comp(float (*func_1)(POINTER,float*),
			POINTER,float (*func_2)(POINTER,float*),POINTER,
			COMPONENT***,RECT_GRID,COMPONENT,SIDE,SIDE);
LOCAL 	void 	set_grid_for_surface_construction(RECT_GRID*,RECT_GRID*);
LOCAL	void 	install_bdry_objects(INTERFACE*,NODE****,CURVE****,
			SURFACE***,RECT_GRID*,RECT_GRID*);
LOCAL   bool 	onfront_block(int,int,int,const EG_CRX*);
LOCAL	bool 	grid_line_crx_in_dir(float (*func)(POINTER,float*),
				POINTER,float*,float*,float*,int);
LOCAL	bool 	is_positive_bdry_curve(int,int,int,NODE*,NODE*);
LOCAL	int     count_crx_through_comp(int*,COMPONENT***);
LOCAL   int     count_bdry_coner_crx(int*,COMPONENT***);
LOCAL   int 	install_grid_crx(float (*func)(POINTER,float*),POINTER,
			EG_CRX*,RECT_GRID);
LOCAL   int     install_bdry_corner_crx(EG_CRX*,RECT_GRID,float*,float*,
			NODE****,CURVE****,SURFACE***);
LOCAL	int 	count_side_comp(COMPONENT,COMPONENT,COMPONENT,COMPONENT);

EXPORT	bool make_bdry_surfaces(
	INTERFACE *intfc,
	RECT_GRID *rgr)
{
	int 		i, j, k, l, num_crx,*gmax;
	int             ns, nc;
	RECT_GRID   	dual_gr;
	NODE		****corners;
	CURVE		****edges;
	SURFACE		***faces;
	EG_CRX		Eg_crx;
	BDRY_BOX_PARAMS bp;
	COMPONENT	compin,compout;
	BLK_INFO	blk_info;
	static BLK_CRX  *blk_crx;

	DEBUG_ENTER(make_bdry_surfaces)
 
        /*
	gview_plot_interface("imp3d0",intfc);
        */
	if (blk_crx == NULL)
	    blk_crx = alloc_blk_crx(NO);
	set_grid_for_surface_construction(&dual_gr,rgr);
	gmax = dual_gr.gmax;

	tri_array(&corners,2,2,2,sizeof(NODE*));
	tri_array(&edges,3,2,2,sizeof(CURVE*));
	matrix(&faces,3,2,sizeof(CURVE*));

	install_bdry_objects(intfc,corners,edges,faces,rgr,&dual_gr);

	tri_array(&Eg_crx.comp,gmax[0]+1,gmax[1]+1,gmax[2]+1,sizeof(COMPONENT));
	reset_domain_comp(Eg_crx.comp,dual_gr);

	bp.L = rgr->VL; 	bp.U = rgr->VU;
	for (i = 0; i < 3; ++i)
	{
	    if (rgr->lbuf[i] != 0) bp.L[i] -= rgr->h[i];
	    if (rgr->ubuf[i] != 0) bp.U[i] += rgr->h[i];
	}
	compin = NO_COMP;
	compout = exterior_component(intfc);
	assign_two_comp_domain(bdry_box_func,(POINTER)&bp,Eg_crx.comp,
				dual_gr,compout,compin);
	num_crx = count_crx_through_comp(gmax,Eg_crx.comp);
	num_crx += count_bdry_coner_crx(gmax,Eg_crx.comp);

	alloc_grid_crx_mem(&Eg_crx,gmax,num_crx,YES);

	Eg_crx.num_curves = 0;
	for (i = 0; i < 3; ++i)
	    for (j = 0; j < 2; ++j)
	    	for (k = 0; k < 2; ++k)
		    if (edges[i][j][k] != NULL)
		    	++Eg_crx.num_curves;
	vector(&Eg_crx.curves,Eg_crx.num_curves,sizeof(CURVE*));
	for (l = 0, i = 0; i < 3; ++i)
	    for (j = 0; j < 2; ++j)
	    	for (k = 0; k < 2; ++k)
		    if (edges[i][j][k] != NULL)
		    	Eg_crx.curves[l++] = edges[i][j][k];

	num_crx = install_bdry_corner_crx(&Eg_crx,dual_gr,bp.L,bp.U,
			corners,edges,faces);

	blk_info.num_surfs = blk_info.num_curves = 0;
	for (i = 0; i < 3; ++i)
	{
	    for (j = 0; j < 2; ++j)
	    {
	    	for (k = 0; k < 2; ++k)
		{
		    if (edges[i][j][k] != NULL)
		    	++blk_info.num_curves;
		}
		if (faces[i][j] != NULL)
		    ++blk_info.num_surfs;
	    }
	}
	vector(&blk_info.surfs,blk_info.num_surfs,sizeof(SURFACE*));
	vector(&blk_info.curves,blk_info.num_curves,sizeof(CURVE*));
	vector(&blk_info.cur_tris,blk_info.num_surfs,sizeof(TRI*));

	for (ns = 0, nc = 0, i = 0; i < 3; ++i)
	{
	    for (j = 0; j < 2; ++j)
	    {
	    	for (k = 0; k < 2; ++k)
		{
		    if (edges[i][j][k] != NULL)
		    	blk_info.curves[nc++] = edges[i][j][k];
		}
		if (faces[i][j] != NULL)
		    blk_info.surfs[ns++] = faces[i][j];
	    }
	}

	for (i = 0; i < blk_info.num_surfs; ++i)
	    first_tri(blk_info.surfs[i]) = last_tri(blk_info.surfs[i]) = NULL;

	blk_crx->comps[0] = compout;
	blk_crx->comps[1] = compin;
	blk_crx->blk_info = &blk_info;

	make_grid_surfaces(blk_crx,&Eg_crx,gmax,YES);

	for (i = 0; i < blk_info.num_surfs; ++i)
        {
	    last_tri(blk_info.surfs[i]) = blk_info.cur_tris[i];
	    last_tri(blk_info.surfs[i])->next = 
	    		tail_of_tri_list(blk_info.surfs[i]);
	    first_tri(blk_info.surfs[i])->prev = 
	    		head_of_tri_list(blk_info.surfs[i]);
	}
	reset_intfc_num_points(intfc);

	free_grid_crx_mem(&Eg_crx,YES);
	free_these(4,Eg_crx.comp,corners,edges,faces);
	free_these(3,blk_info.surfs,blk_info.curves,blk_info.cur_tris);

	DEBUG_LEAVE(install_faces)
	return YES;
}	/* end make_bdry_surfaces */


/*******************************************************************
*       This function make a surface described by the function     *
*       func = 0. The negative side of the surface has neg_comp    *
*       and the positive side of the surface has pos_comp.         *
*******************************************************************/

EXPORT bool make_comp2_surface(
	RECT_GRID   *rgr,
	COMPONENT   neg_comp,
	COMPONENT   pos_comp,
	float       (*func)(POINTER,float*),
        POINTER     func_params,
	SURFACE     **s)
{
	int		i,num_crx, *gmax;
	RECT_GRID	dual_gr;
	SURFACE		*surf;
	EG_CRX		Eg_crx;
	BLK_INFO	blk_info;
	static BLK_CRX  *blk_crx;

	if (blk_crx == NULL)
	    blk_crx = alloc_blk_crx(NO);

	zero_scalar(&Eg_crx,sizeof(EG_CRX));
	set_grid_for_surface_construction(&dual_gr,rgr);
	gmax = dual_gr.gmax;
	tri_array(&Eg_crx.comp,gmax[0]+1,gmax[1]+1,gmax[2]+1,
			sizeof(COMPONENT));

	reset_domain_comp(Eg_crx.comp,dual_gr);
	assign_two_comp_domain(func,func_params,Eg_crx.comp,
				dual_gr,neg_comp,pos_comp);
	num_crx = count_crx_through_comp(gmax,Eg_crx.comp);

	alloc_grid_crx_mem(&Eg_crx,gmax,num_crx,NO);

	num_crx = install_grid_crx(func,func_params,&Eg_crx,dual_gr);

	surf = make_surface(neg_comp,pos_comp,NULL,NULL);

	first_tri(surf) = last_tri(surf) = NULL;

	blk_info.num_surfs = 1;
	vector(&blk_info.surfs,1,sizeof(SURFACE*));
	vector(&blk_info.cur_tris,1,sizeof(TRI*));

	blk_info.cur_tris[0] = NULL;
	blk_info.surfs[0] = surf;
	surf->num_tri = 0;

	blk_crx->comps[0] = pos_comp;
	blk_crx->comps[1] = neg_comp;
	blk_crx->blk_info = &blk_info;

	for (i = 0; i < num_crx; ++i)
	    Eg_crx.crx_store[i].s = surf;

	make_grid_surfaces(blk_crx,&Eg_crx,gmax,NO);

	last_tri(surf) = blk_info.cur_tris[0];
	last_tri(surf)->next = tail_of_tri_list(surf);
	first_tri(surf)->prev = head_of_tri_list(surf);
	reset_intfc_num_points(surf->interface);

	free_grid_crx_mem(&Eg_crx,NO);
	free_these(3,Eg_crx.comp,blk_info.surfs,blk_info.cur_tris);
	*s = surf;
	return YES;
}	/* end make_comp2_surface */

/*******************************************************************
*       This function make a surface described by the function     *
*       func = 0. The negative side of the surface has neg_comp    *
*       and the positive side of the surface has pos_comp.         *
*******************************************************************/

EXPORT bool make_comp3_surfaces(
	RECT_GRID   *rgr,
	COMPONENT   comp0,
	COMPONENT   comp1,
	COMPONENT   comp2,
	float       (*func1)(POINTER,float*),
        POINTER     func1_params,
	float       (*func2)(POINTER,float*),
        POINTER     func2_params,
	SURFACE     ***s,
	CURVE	    **c)
{
	int		i,num_crx, *gmax;
	RECT_GRID	dual_gr;
	SURFACE		*surf;
	EG_CRX		Eg_crx;
	BLK_INFO	blk_info;
	static BLK_CRX  *blk_crx;

	if (blk_crx == NULL)
	    blk_crx = alloc_blk_crx(NO);

	zero_scalar(&Eg_crx,sizeof(EG_CRX));
	set_grid_for_surface_construction(&dual_gr,rgr);
	gmax = dual_gr.gmax;
	tri_array(&Eg_crx.comp,gmax[0]+1,gmax[1]+1,gmax[2]+1,
			sizeof(COMPONENT));

	reset_domain_comp(Eg_crx.comp,dual_gr);
	print_rectangular_grid(&dual_gr);
	assign_negative_comp(func1,func1_params,Eg_crx.comp,
				dual_gr,comp0);
	assign_intersection_comp(func1,func1_params,func2,func2_params,
				Eg_crx.comp,dual_gr,comp1,
				POSITIVE_SIDE,NEGATIVE_SIDE);
	assign_positive_comp(func2,func2_params,Eg_crx.comp,
				dual_gr,comp2);
	num_crx = count_crx_through_comp(gmax,Eg_crx.comp);
	printf("num_crx = %d\n",num_crx);

	alloc_grid_crx_mem(&Eg_crx,gmax,num_crx,YES);

	free_grid_crx_mem(&Eg_crx,YES);
	return YES;
}	/* end make_comp3_surfaces */

LOCAL	void install_bdry_objects(
	INTERFACE *intfc,
	NODE ****corners,
	CURVE ****edges,
	SURFACE ***faces,
	RECT_GRID *gr,
	RECT_GRID *dual_gr)
{
	int i,j,k,ii,jj;
	float coords[3];
	float corner_coords[3][2];
	NODE *nodes[2][2],*ns,*ne;
	COMPONENT ext_comp = exterior_component(intfc);


	for (i = 0; i < 3; ++i)
	{
	    for (j = 0; j < 2; ++j)
	    {
	    	for (k = 0; k < 2; ++k)
		{
		    if (i != 2) 
		    	corners[i][j][k] = NULL;
		    edges[i][j][k] = NULL;
		}
		faces[i][j] = NULL;
	    }
	}
	for (i = 0; i < 3; ++i)
	{
	    if (!buffered_boundary_type(intfc->rect_bdry_type[i][0]))
		corner_coords[i][0] = gr->L[i];
	    else
	    	corner_coords[i][0] = dual_gr->L[i];
	    if (!buffered_boundary_type(intfc->rect_bdry_type[i][1]))
		corner_coords[i][1] = gr->U[i];
	    else
	    	corner_coords[i][1] = dual_gr->U[i];
	}
	for (i = 0; i < 3; ++i)
	{
	    for (j = 0; j < 2; ++j)
	    {
	        if ((!buffered_boundary_type(intfc->rect_bdry_type[i][j])) &&
                    (intfc->rect_bdry_type[i][j] != MIXED_TYPE_BOUNDARY))
	        {
		    coords[i] = corner_coords[i][j];
		    for (ii = 0; ii < 2; ++ii)
		    {
		        for (jj = 0; jj < 2; ++jj)
		        {
		    	    coords[(i+1)%3] = corner_coords[(i+1)%3][ii];
		    	    coords[(i+2)%3] = corner_coords[(i+2)%3][jj];
			    switch (i)
			    {
			    case 0:
				if (corners[j][ii][jj] == NULL)
				    corners[j][ii][jj] = 
				    	make_node(Point(coords));
			    	nodes[ii][jj] = corners[j][ii][jj];
				break;
			    case 1:
				if (corners[jj][j][ii] == NULL)
				    corners[jj][j][ii]
					= make_node(Point(coords));
			    	nodes[ii][jj] = corners[jj][j][ii]; 
				break;
			    case 2:
				if (corners[ii][jj][j] == NULL)
				    corners[ii][jj][j]
					= make_node(Point(coords));
			    	nodes[ii][jj] = corners[ii][jj][j]; 
			    }
			    if (j == 0)
			    	set_bdry_side(Boundary(nodes[ii][jj]),
				    		i,BOTTOM);
			    else
			    	set_bdry_side(Boundary(nodes[ii][jj]),
				    		i,TOP);
		        }
		    }
		    faces[i][j] = make_surface(ext_comp,NO_COMP,
		    		(CURVE**)NULL,(CURVE**)NULL);

		    user_install_faces(faces[i][j],2*i+j);
		    set_is_bdry(faces[i][j]);
		    if (j == 0)
		    	set_bdry_side(Boundary(faces[i][j]),i,BOTTOM);
		    else
		    	set_bdry_side(Boundary(faces[i][j]),i,TOP);

		    if (edges[(i+1)%3][0][j] == NULL)
		    {
			ns = nodes[0][0]; ne = nodes[1][0];
		        edges[(i+1)%3][0][j] = make_curve(NO_COMP,
					NO_COMP,ns,ne);
			if (j == 0)
			    set_bdry_side(Boundary(edges[(i+1)%3][0][j]),
			    		i,BOTTOM);
			else
			    set_bdry_side(Boundary(edges[(i+1)%3][0][j]),
			    		i,TOP);
			set_bdry_side(Boundary(edges[(i+1)%3][0][j]),
			    		(i+2)%3,BOTTOM);
		    	assign_curve_boundary_flag(edges[(i+1)%3][0][j]);
		    }
		    else
		    {
		        ns = edges[(i+1)%3][0][j]->start;
		        ne = edges[(i+1)%3][0][j]->end;
		    }
		    if (is_positive_bdry_curve(i,j,0,ns,ne))
		    {
			add_to_pointers((POINTER)edges[(i+1)%3][0][j],
				(POINTER**)&faces[i][j]->pos_curves);
			add_to_pointers((POINTER)faces[i][j],(POINTER**)
				&edges[(i+1)%3][0][j]->pos_surfaces);
		    }
		    else
		    {
			add_to_pointers((POINTER)edges[(i+1)%3][0][j],
				(POINTER**)&faces[i][j]->neg_curves);
			add_to_pointers((POINTER)faces[i][j],(POINTER**)
				&edges[(i+1)%3][0][j]->neg_surfaces);
		    }

		    if (edges[(i+2)%3][j][1] == NULL)
		    {
			ns = nodes[1][0]; ne = nodes[1][1];
		    	edges[(i+2)%3][j][1] = make_curve(NO_COMP,
					NO_COMP,ns,ne);
			if (j == 0)
			    set_bdry_side(Boundary(edges[(i+2)%3][j][1]),
			    		i,BOTTOM);
			else
			    set_bdry_side(Boundary(edges[(i+2)%3][j][1]),
			    		i,TOP);
			set_bdry_side(Boundary(edges[(i+2)%3][j][1]),
			    		(i+1)%3,TOP);
		    	assign_curve_boundary_flag(edges[(i+2)%3][j][1]);
		    }
		    else
		    {
		    	ns = edges[(i+2)%3][j][1]->start;
		    	ne = edges[(i+2)%3][j][1]->end;
		    }
		    if (is_positive_bdry_curve(i,j,1,ns,ne))
		    {
			add_to_pointers((POINTER)edges[(i+2)%3][j][1],
				(POINTER**)&faces[i][j]->pos_curves);
			add_to_pointers((POINTER)faces[i][j],(POINTER**)
				&edges[(i+2)%3][j][1]->pos_surfaces);
		    }
		    else
		    {
			add_to_pointers((POINTER)edges[(i+2)%3][j][1],
				(POINTER**)&faces[i][j]->neg_curves);
			add_to_pointers((POINTER)faces[i][j],(POINTER**)
				&edges[(i+2)%3][j][1]->neg_surfaces);
		    }

		    if (edges[(i+1)%3][1][j] == NULL)
		    {
			ns = nodes[1][1]; ne = nodes[0][1];
		        edges[(i+1)%3][1][j] = make_curve(NO_COMP,
					NO_COMP,ns,ne);
			if (j == 0)
			    set_bdry_side(Boundary(edges[(i+1)%3][1][j]),
			    		i,BOTTOM);
			else
			    set_bdry_side(Boundary(edges[(i+1)%3][1][j]),
			    		i,TOP);
			set_bdry_side(Boundary(edges[(i+1)%3][1][j]),
			    		(i+2)%3,TOP);
		    	assign_curve_boundary_flag(edges[(i+1)%3][1][j]);
		    }
		    else
		    {
		    	ns = edges[(i+1)%3][1][j]->start;
		    	ne = edges[(i+1)%3][1][j]->end;
		    }
		    if (is_positive_bdry_curve(i,j,2,ns,ne))
		    {
			add_to_pointers((POINTER)edges[(i+1)%3][1][j],
				(POINTER**)&faces[i][j]->pos_curves);
			add_to_pointers((POINTER)faces[i][j],(POINTER**)
				&edges[(i+1)%3][1][j]->pos_surfaces);
		    }
		    else
		    {
			add_to_pointers((POINTER)edges[(i+1)%3][1][j],
				(POINTER**)&faces[i][j]->neg_curves);
			add_to_pointers((POINTER)faces[i][j],(POINTER**)
				&edges[(i+1)%3][1][j]->neg_surfaces);
		    }

		    if (edges[(i+2)%3][j][0] == NULL)
		    {
			ns = nodes[0][1]; ne = nodes[0][0];
		    	edges[(i+2)%3][j][0] = make_curve(NO_COMP,
					NO_COMP,ns,ne);
			if (j == 0)
			    set_bdry_side(Boundary(edges[(i+2)%3][j][0]),
			    		i,BOTTOM);
			else
			    set_bdry_side(Boundary(edges[(i+2)%3][j][0]),
			    		i,TOP);
			set_bdry_side(Boundary(edges[(i+2)%3][j][0]),
			    		(i+1)%3,BOTTOM);
		    	assign_curve_boundary_flag(edges[(i+2)%3][j][0]);
		    }
		    else
		    {
		    	ns = edges[(i+2)%3][j][0]->start;
		    	ne = edges[(i+2)%3][j][0]->end;
		    }
		    if (is_positive_bdry_curve(i,j,3,ns,ne))
		    {
			add_to_pointers((POINTER)edges[(i+2)%3][j][0],
				(POINTER**)&faces[i][j]->pos_curves);
			add_to_pointers((POINTER)faces[i][j],(POINTER**)
				&edges[(i+2)%3][j][0]->pos_surfaces);
		    }
		    else
		    {
			add_to_pointers((POINTER)edges[(i+2)%3][j][0],
				(POINTER**)&faces[i][j]->neg_curves);
			add_to_pointers((POINTER)faces[i][j],(POINTER**)
				&edges[(i+2)%3][j][0]->neg_surfaces);
		    }
	        }
	    }
	}
}	/* end install_bdry_objects */

LOCAL	bool is_positive_bdry_curve(
	int dir,
	int nb,
	int n,
	NODE *ns,
	NODE *ne)
{
	float ns_coord,ne_coord;

	if (n == 0 || n == 2)
	{
	    ns_coord = Coords(ns->posn)[(dir+1)%3];
	    ne_coord = Coords(ne->posn)[(dir+1)%3];
	}
	else
	{
	    ns_coord = Coords(ns->posn)[(dir+2)%3];
	    ne_coord = Coords(ne->posn)[(dir+2)%3];
	}
	if (nb == 0)
	{
	    if (n == 0 || n == 1)
	    {
	    	if (ns_coord < ne_coord)
		    return YES;
	    	else
		    return NO;
	    }
	    else
	    {
	    	if (ns_coord < ne_coord)
		    return NO;
	    	else
		    return YES;
	    }
	}
	else
	{
	    if (n == 0 || n == 1)
	    {
	    	if (ns_coord < ne_coord)
		    return NO;
	    	else
		    return YES;
	    }
	    else
	    {
	    	if (ns_coord < ne_coord)
		    return YES;
	    	else
		    return NO;
	    }
	}
}	/* end is_positive_bdry_curve */

LOCAL	int install_grid_crx(
	float (*func)(POINTER,float*),
        POINTER func_params,
	EG_CRX *eg_crx,
	RECT_GRID grid)
{
	float coords1[3];
	float coords2[3];
	float crds_crx[3];
	float *L = grid.L;
	float *h = grid.h;
	int *gmax = grid.gmax;
	int i,j,k,n_crx;
	BBI_POINT ****x_crx = eg_crx->x_crx;
	BBI_POINT ****y_crx = eg_crx->y_crx;
	BBI_POINT ****z_crx = eg_crx->z_crx;
	BBI_POINT *crx_store = eg_crx->crx_store;
	COMPONENT ***comp = eg_crx->comp;

	n_crx = 0;

	/* install x-crossings */

	for (j = 0; j <= gmax[1]; ++j)
	{
	    coords1[1] = coords2[1] = L[1] + j*h[1];
	    for (k = 0; k <= gmax[2]; ++k)
	    {
		coords1[2] = coords2[2] = L[2] + k*h[2];
		for (i = 0; i < gmax[0]; ++i)
		{
		    x_crx[i][j][k] = NULL;
		    if (comp[i][j][k] != comp[i+1][j][k])
		    {
		        coords1[0] = L[0] + i*h[0];
		        coords2[0] = L[0] + (i+1)*h[0];
			if (! grid_line_crx_in_dir(func,func_params,
				coords1,coords2,crds_crx,0))
			{
			    screen("ERROR: in install_grid_crx(), no x-crxing!");
			    clean_up(ERROR);
			}
			if (crds_crx[0] - coords1[0] < 0.004*h[0])
			    crds_crx[0] = coords1[0] + 0.004*h[0];
			if (coords2[0] - crds_crx[0] < 0.004*h[0])
			    crds_crx[0] = coords2[0] - 0.004*h[0];
			crx_store[n_crx].p = Point(crds_crx);
			x_crx[i][j][k] = &crx_store[n_crx++];
		    }
		}
	    }
	}

	/* install y-crossings */

	for (i = 0; i <= gmax[0]; ++i)
	{
	    coords1[0] = coords2[0] = L[0] + i*h[0];
	    for (k = 0; k <= gmax[2]; ++k)
	    {
		coords1[2] = coords2[2] = L[2] + k*h[2];
		for (j = 0; j < gmax[1]; ++j)
		{
		    y_crx[i][j][k] = NULL;
		    if (comp[i][j][k] != comp[i][j+1][k])
		    {
		        coords1[1] = L[1] + j*h[1];
		        coords2[1] = L[1] + (j+1)*h[1];
			if (! grid_line_crx_in_dir(func,func_params,
				coords1,coords2,crds_crx,1))
			{
			    screen("ERROR: in install_grid_crx(), no y-crxing!");
			    clean_up(ERROR);
			}
			if (crds_crx[1] - coords1[1] < 0.004*h[1])
			    crds_crx[1] = coords1[1] + 0.004*h[1];
			if (coords2[1] - crds_crx[1] < 0.004*h[1])
			    crds_crx[1] = coords2[1] - 0.004*h[1];
			crx_store[n_crx].p = Point(crds_crx);
			y_crx[i][j][k] = &crx_store[n_crx++];
		    }
		}
	    }
	}

	/* install z-crossings */

	for (i = 0; i <= gmax[0]; ++i)
	{
	    coords1[0] = coords2[0] = L[0] + i*h[0];
	    for (j = 0; j <= gmax[1]; ++j)
	    {
		coords1[1] = coords2[1] = L[1] + j*h[1];
		for (k = 0; k < gmax[2]; ++k)
		{
		    z_crx[i][j][k] = NULL;
		    if (comp[i][j][k] != comp[i][j][k+1])
		    {
		        coords1[2] = L[2] + k*h[2];
		        coords2[2] = L[2] + (k+1)*h[2];
			if (! grid_line_crx_in_dir(func,func_params,
				coords1,coords2,crds_crx,2))
			{
			    screen("ERROR: in install_grid_crx(), no z-crxing!");
			    clean_up(ERROR);
			}
			if (crds_crx[2] - coords1[2] < 0.004*h[2])
			    crds_crx[2] = coords1[2] + 0.004*h[2];
			if (coords2[2] - crds_crx[2] < 0.004*h[2])
			    crds_crx[2] = coords2[2] - 0.004*h[2];
			crx_store[n_crx].p = Point(crds_crx);
			z_crx[i][j][k] = &crx_store[n_crx++];
		    }
		}
	    }
	}
	return n_crx;
}	/* end install_grid_crx */

LOCAL void assign_blk_crx(
	BLK_CRX      *blk_crx,
	int          i,
	int          j,
	int          k,
	const EG_CRX *eg_crx,
	bool         include_curve_crx)
{
	int ic,num_curve_crx,ii,jj,kk;
	BBI_POINT ****x_crx = eg_crx->x_crx;
	BBI_POINT ****y_crx = eg_crx->y_crx;
	BBI_POINT ****z_crx = eg_crx->z_crx;
	BBI_POINT ****x_curve_crx = eg_crx->x_curve_crx;
	BBI_POINT ****y_curve_crx = eg_crx->y_curve_crx;
	BBI_POINT ****z_curve_crx = eg_crx->z_curve_crx;
	BBI_POINT ****node_crx = eg_crx->node_crx;
	COMPONENT c,***comp = eg_crx->comp;

	blk_crx->num_comps = 0;
	for (ii = 0; ii < 2; ++ii)
	{
	    for (jj = 0; jj < 2; ++jj)
	    {
		for (kk = 0; kk < 2; ++kk)
		{
		    c = comp[i+ii][j+jj][k+kk];
		    blk_crx->ix[ii][jj][kk] = ii;
		    blk_crx->iy[ii][jj][kk] = jj;
		    blk_crx->iz[ii][jj][kk] = kk;
		    for (ic = 0; ic < blk_crx->num_comps; ++ic)
		    {
		    	if (c == blk_crx->comps[ic])
			{
			    ++blk_crx->nv[ic];
			    break;
			}
		    }
		    if (ic == blk_crx->num_comps)
		    {
		    	blk_crx->comps[ic] = c;
			blk_crx->nv[ic] = 1;
			++blk_crx->num_comps;
		    }
		    blk_crx->comp[ii][jj][kk] = c;
		}
		blk_crx->crx[0][ii][jj] = x_crx[i][j+ii][k+jj];
		blk_crx->crx[1][ii][jj] = y_crx[i+jj][j][k+ii];
		blk_crx->crx[2][ii][jj] = z_crx[i+ii][j+jj][k];
	    }
	}
	for (ii = 0; ii < blk_crx->num_comps-1; ++ii)
	{
	    for (jj = 1; jj < blk_crx->num_comps; ++jj)
	    {
	    	if (blk_crx->nv[ii] > blk_crx->nv[jj])
		{
		    int nv_tmp;
		    COMPONENT c_tmp;
		    nv_tmp = blk_crx->nv[ii];
		    blk_crx->nv[ii] = blk_crx->nv[jj];
		    blk_crx->nv[jj] = nv_tmp;
		    c_tmp = blk_crx->comps[ii];
		    blk_crx->comps[ii] = blk_crx->comps[jj];
		    blk_crx->comps[jj] = c_tmp;
		}
	    }
	}
	if (! include_curve_crx)
	{
	    blk_crx->blk_type = COMP2_BLOCK;
	    return;
	}
	num_curve_crx = 0;
	for (ii = 0; ii < 2; ++ii)
	{
	    blk_crx->curve_crx[0][ii] = x_curve_crx[i+ii][j][k];
	    if (x_curve_crx[i+ii][j][k] != NULL)
		++num_curve_crx;
	    blk_crx->curve_crx[1][ii] = y_curve_crx[i][j+ii][k];
	    if (y_curve_crx[i][j+ii][k] != NULL)
		++num_curve_crx;
	    blk_crx->curve_crx[2][ii] = z_curve_crx[i][j][k+ii];
	    if (z_curve_crx[i][j][k+ii] != NULL)
		++num_curve_crx;
	}
	blk_crx->node_crx = node_crx[i][j][k];
	if (blk_crx->num_comps == 2)
	{
	    if (num_curve_crx == 0)
	    	blk_crx->blk_type = COMP2_BLOCK;
	    else if (num_curve_crx != 0)
	    	blk_crx->blk_type = BDRY_BLOCK;
	}
	else if (blk_crx->num_comps == 3)
	    blk_crx->blk_type = COMP3_BLOCK;
}	/* end assign_blk_crx */

LOCAL	bool onfront_block(
	int          i,
	int          j,
	int          k,
	const EG_CRX *eg_crx)
{
	int ii,jj;

	for (ii = 0; ii < 2; ++ii)
	{
	    for (jj = 0; jj < 2; ++jj)
	    {
		if (eg_crx->x_crx[i][j+ii][k+jj] != NULL)
		    return YES;
		if (eg_crx->y_crx[i+jj][j][k+ii] != NULL)
		    return YES;
		if (eg_crx->z_crx[i+ii][j+jj][k] != NULL)
		    return YES;
	    }
	}
	return NO;
}	/* end onfront_block */

LOCAL int count_bdry_coner_crx(
	int *gmax,
	COMPONENT ***comp)
{
	int i,num_crx;

	num_crx = 0;
	for (i = 0; i <= gmax[0]; ++i)
	{
	    if (comp[i][1][1] != comp[i][0][1] &&
	    	comp[i][1][1] != comp[i][1][0])
		++num_crx;
	    if (comp[i][gmax[1]-1][1] != comp[i][gmax[1]][1] &&
	    	comp[i][gmax[1]-1][1] != comp[i][gmax[1]-1][0])
		++num_crx;
	    if (comp[i][gmax[1]-1][gmax[2]-1] != comp[i][gmax[1]][gmax[2]-1] &&
	    	comp[i][gmax[1]-1][gmax[2]-1] != comp[i][gmax[1]-1][gmax[2]])
		++num_crx;
	    if (comp[i][1][gmax[2]-1] != comp[i][0][gmax[2]-1] &&
	    	comp[i][1][gmax[2]-1] != comp[i][1][gmax[2]])
		++num_crx;
	}
	for (i = 0; i <= gmax[1]; ++i)
	{
	    if (comp[1][i][1] != comp[0][i][1] &&
	    	comp[1][i][1] != comp[1][i][0])
		++num_crx;
	    if (comp[gmax[0]-1][i][1] != comp[gmax[0]][i][1] &&
	    	comp[gmax[0]-1][i][1] != comp[gmax[0]-1][i][0])
		++num_crx;
	    if (comp[gmax[0]-1][i][gmax[2]-1] != comp[gmax[0]][i][gmax[2]-1] &&
	    	comp[gmax[0]-1][i][gmax[2]-1] != comp[gmax[0]-1][i][gmax[2]])
		++num_crx;
	    if (comp[1][i][gmax[2]-1] != comp[0][i][gmax[2]-1] &&
	    	comp[1][i][gmax[2]-1] != comp[1][i][gmax[2]])
		++num_crx;
	}
	for (i = 0; i <= gmax[2]; ++i)
	{
	    if (comp[1][1][i] != comp[0][1][i] &&
	    	comp[1][1][i] != comp[1][0][i])
		++num_crx;
	    if (comp[gmax[0]-1][1][i] != comp[gmax[0]][1][i] &&
	    	comp[gmax[0]-1][1][i] != comp[gmax[0]-1][0][i])
		++num_crx;
	    if (comp[gmax[0]-1][gmax[1]-1][i] != comp[gmax[0]][gmax[1]-1][i] &&
	    	comp[gmax[0]-1][gmax[1]-1][i] != comp[gmax[0]-1][gmax[1]][i])
		++num_crx;
	    if (comp[1][gmax[1]-1][i] != comp[0][gmax[1]-1][i] &&
	    	comp[1][gmax[1]-1][i] != comp[1][gmax[1]][i])
		++num_crx;
	}
	if (comp[1][1][1] != comp[0][1][1] &&
	    comp[1][1][1] != comp[1][0][1] &&
	    comp[1][1][1] != comp[1][1][0]) 
	    ++num_crx;
	if (comp[gmax[0]-1][1][1] != comp[gmax[0]][1][1] &&
	    comp[gmax[0]-1][1][1] != comp[gmax[0]-1][0][1] &&
	    comp[gmax[0]-1][1][1] != comp[gmax[0]-1][1][0]) 
	    ++num_crx;
	if (comp[1][gmax[1]-1][1] != comp[0][gmax[1]-1][1] &&
	    comp[1][gmax[1]-1][1] != comp[1][gmax[1]][1] &&
	    comp[1][gmax[1]-1][1] != comp[1][gmax[1]-1][0]) 
	    ++num_crx;
	if (comp[1][1][gmax[2]-1] != comp[0][1][gmax[2]-1] &&
	    comp[1][1][gmax[2]-1] != comp[1][0][gmax[2]-1] &&
	    comp[1][1][gmax[2]-1] != comp[1][1][gmax[2]]) 
	    ++num_crx;
	if (comp[gmax[0]-1][gmax[1]-1][1] != comp[gmax[0]][gmax[1]-1][1] &&
	    comp[gmax[0]-1][gmax[1]-1][1] != comp[gmax[0]-1][gmax[1]][1] &&
	    comp[gmax[0]-1][gmax[1]-1][1] != comp[gmax[0]-1][gmax[1]-1][0]) 
	    ++num_crx;
	if (comp[1][gmax[1]-1][gmax[2]-1] != comp[0][gmax[1]-1][gmax[2]-1] &&
	    comp[1][gmax[1]-1][gmax[2]-1] != comp[1][gmax[1]][gmax[2]-1] &&
	    comp[1][gmax[1]-1][gmax[2]-1] != comp[1][gmax[1]-1][gmax[2]]) 
	    ++num_crx;
	if (comp[gmax[0]-1][1][gmax[2]-1] != comp[gmax[0]][1][gmax[2]-1] &&
	    comp[gmax[0]-1][1][gmax[2]-1] != comp[gmax[0]-1][0][gmax[2]-1] &&
	    comp[gmax[0]-1][1][gmax[2]-1] != comp[gmax[0]-1][1][gmax[2]]) 
	    ++num_crx;
	if (comp[gmax[0]-1][gmax[1]-1][gmax[2]-1] != 
	    comp[gmax[0]][gmax[1]-1][gmax[2]-1] &&
	    comp[gmax[0]-1][gmax[1]-1][gmax[2]-1] != 
	    comp[gmax[0]-1][gmax[1]][gmax[2]-1] &&
	    comp[gmax[0]-1][gmax[1]-1][gmax[2]-1] != 
	    comp[gmax[0]-1][gmax[1]-1][gmax[2]]) 
	    ++num_crx;

	return num_crx;
}	/* end count_bdry_coner_crx */


LOCAL int install_bdry_corner_crx(
	EG_CRX    *eg_crx,
	RECT_GRID dual_gr,
	float     *L,
	float     *U,
	NODE      ****corners,
	CURVE     ****edges,
	SURFACE   ***faces)
{
	int i,j,k;
	BBI_POINT ****x_crx = eg_crx->x_crx;
	BBI_POINT ****y_crx = eg_crx->y_crx;
	BBI_POINT ****z_crx = eg_crx->z_crx;
	BBI_POINT ****x_curve_crx = eg_crx->x_curve_crx;
	BBI_POINT ****y_curve_crx = eg_crx->y_curve_crx;
	BBI_POINT ****z_curve_crx = eg_crx->z_curve_crx;
	BBI_POINT ****node_crx = eg_crx->node_crx;
	BBI_POINT *crx_store = eg_crx->crx_store;
	COMPONENT ***comp = eg_crx->comp;
	float coords[3];
	float *DL = dual_gr.L;
	float *h  = dual_gr.h;
	int *gmax = dual_gr.gmax;
	int m,n,num_crx = 0;

	/* install x-crossings */

	for (j = 0; j <= gmax[1]; ++j)
	{
	    coords[1] = DL[1] + j*h[1];
	    for (k = 0; k <= gmax[2]; ++k)
	    {
		coords[2] = DL[2] + k*h[2];
		if (comp[0][j][k] != comp[1][j][k])
		{
		    coords[0] = L[0];
		    crx_store[num_crx].p = Point(coords);
		    crx_store[num_crx].s = faces[0][0];
		    x_crx[0][j][k] = &crx_store[num_crx];
		    if (j == 0 || j == gmax[1])
		    {
			m = (j == 0) ? 0 : gmax[1]-1;
			n = (j == 0) ? 0 : 1;
		    	z_curve_crx[0][m][k] = x_crx[0][j][k];
		    	z_curve_crx[0][m][k]->c = 
				(z_curve_crx[0][m][k]->c == NULL) ? 
				edges[2][0][n] : NULL;
		    }
		    if (k == 0 || k == gmax[2])
		    {
			m = (k == 0) ? 0 : gmax[2]-1;
			n = (k == 0) ? 0 : 1;
		    	y_curve_crx[0][j][m] = x_crx[0][j][k];
		    	y_curve_crx[0][j][m]->c = 
				(y_curve_crx[0][j][m]->c == NULL) ?
				edges[1][n][0] : NULL;
		    }
		    if ((j == 0 || j == gmax[1]) &&
		    	(k == 0 || k == gmax[2]))
		    {
		    	m = (j == 0) ? 0 : 1;
		    	n = (k == 0) ? 0 : 1;
			crx_store[num_crx].p = corners[0][m][n]->posn;
		    }
		    ++num_crx;
		}
		if (comp[gmax[0]-1][j][k] != comp[gmax[0]][j][k])
		{
		    coords[0] = U[0];
		    crx_store[num_crx].p = Point(coords);
		    crx_store[num_crx].s = faces[0][1];
		    x_crx[gmax[0]-1][j][k] = &crx_store[num_crx];
		    if (j == 0 || j == gmax[1])
		    {
			m = (j == 0) ? 0 : gmax[1]-1;
			n = (j == 0) ? 0 : 1;
		    	z_curve_crx[gmax[0]-1][m][k] = x_crx[gmax[0]-1][j][k];
		    	z_curve_crx[gmax[0]-1][m][k]->c = 
				(z_curve_crx[gmax[0]-1][m][k]->c == NULL) ?
				edges[2][1][n] : NULL;
		    }
		    if (k == 0 || k == gmax[2])
		    {
			m = (k == 0) ? 0 : gmax[2]-1;
			n = (k == 0) ? 0 : 1;
		    	y_curve_crx[gmax[0]-1][j][m] = x_crx[gmax[0]-1][j][k];
		    	y_curve_crx[gmax[0]-1][j][m]->c = 
				(y_curve_crx[gmax[0]-1][j][m]->c == NULL) ?
				edges[1][n][1] : NULL;
		    }
		    if ((j == 0 || j == gmax[1]) &&
		    	(k == 0 || k == gmax[2]))
		    {
		    	m = (j == 0) ? 0 : 1;
		    	n = (k == 0) ? 0 : 1;
			crx_store[num_crx].p = corners[1][m][n]->posn;
		    }
		    ++num_crx;
		}
	    }
	}

	/* install y-crossings */

	for (i = 0; i <= gmax[0]; ++i)
	{
	    coords[0] = DL[0] + i*h[0];
	    for (k = 0; k <= gmax[2]; ++k)
	    {
		coords[2] = DL[2] + k*h[2];
		if (comp[i][0][k] != comp[i][1][k])
		{
		    coords[1] = L[1];
		    crx_store[num_crx].p = Point(coords);
		    crx_store[num_crx].s = faces[1][0];
		    y_crx[i][0][k] = &crx_store[num_crx];
		    if (i == 0 || i == gmax[0])
		    {
			m = (i == 0) ? 0 : gmax[0]-1;
			n = (i == 0) ? 0 : 1;
		    	z_curve_crx[m][0][k] = y_crx[i][0][k];
		    	z_curve_crx[m][0][k]->c = 
				(z_curve_crx[m][0][k]->c == NULL) ?
				edges[2][n][0] : NULL;
		    }
		    if (k == 0 || k == gmax[2])
		    {
			m = (k == 0) ? 0 : gmax[2]-1;
			n = (k == 0) ? 0 : 1;
		    	x_curve_crx[i][0][m] = y_crx[i][0][k];
		    	x_curve_crx[i][0][m]->c = 
				(x_curve_crx[i][0][m]->c == NULL) ?
				edges[0][0][n] : NULL;
		    }
		    if ((i == 0 || i == gmax[0]) &&
		    	(k == 0 || k == gmax[2]))
		    {
		    	m = (i == 0) ? 0 : 1;
		    	n = (k == 0) ? 0 : 1;
			crx_store[num_crx].p = corners[m][0][n]->posn;
		    }
		    ++num_crx;
		}
		if (comp[i][gmax[1]-1][k] != comp[i][gmax[1]][k])
		{
		    coords[1] = U[1];
		    crx_store[num_crx].p = Point(coords);
		    crx_store[num_crx].s = faces[1][1];
		    y_crx[i][gmax[1]-1][k] = &crx_store[num_crx];
		    if (i == 0 || i == gmax[0])
		    {
			m = (i == 0) ? 0 : gmax[0]-1;
			n = (i == 0) ? 0 : 1;
		    	z_curve_crx[m][gmax[1]-1][k] = y_crx[i][gmax[1]-1][k];
		    	z_curve_crx[m][gmax[1]-1][k]->c = 
				(z_curve_crx[m][gmax[1]-1][k]->c == NULL) ?
				edges[2][n][1] : NULL;
		    }
		    if (k == 0 || k == gmax[2])
		    {
			m = (k == 0) ? 0 : gmax[2]-1;
			n = (k == 0) ? 0 : 1;
		    	x_curve_crx[i][gmax[1]-1][m] = y_crx[i][gmax[1]-1][k];
		    	x_curve_crx[i][gmax[1]-1][m]->c = 
				(x_curve_crx[i][gmax[1]-1][m]->c == NULL) ?
				edges[0][1][n] : NULL;
		    }
		    if ((i == 0 || i == gmax[0]) &&
		    	(k == 0 || k == gmax[2]))
		    {
		    	m = (i == 0) ? 0 : 1;
		    	n = (k == 0) ? 0 : 1;
			crx_store[num_crx].p = corners[m][1][n]->posn;
		    }
		    ++num_crx;
		}
	    }
	}

	/* install z-crossings */

	for (i = 0; i <= gmax[0]; ++i)
	{
	    coords[0] = DL[0] + i*h[0];
	    for (j = 0; j <= gmax[1]; ++j)
	    {
		coords[1] = DL[1] + j*h[1];
		if (comp[i][j][0] != comp[i][j][1])
		{
		    coords[2] = L[2];
		    crx_store[num_crx].p = Point(coords);
		    crx_store[num_crx].s = faces[2][0];
		    z_crx[i][j][0] = &crx_store[num_crx];
		    if (i == 0 || i == gmax[0])
		    {
		    	m = (i == 0) ? 0 : gmax[0]-1;
		    	n = (i == 0) ? 0 : 1;
			y_curve_crx[m][j][0] = z_crx[i][j][0];
			y_curve_crx[m][j][0]->c = 
				(y_curve_crx[m][j][0]->c == NULL) ? 
				edges[1][0][n] : NULL;
		    }
		    if (j == 0 || j == gmax[1])
		    {
		    	m = (j == 0) ? 0 : gmax[1]-1;
		    	n = (j == 0) ? 0 : 1;
			x_curve_crx[i][m][0] = z_crx[i][j][0];
			x_curve_crx[i][m][0]->c = 
				(x_curve_crx[i][m][0]->c == NULL) ?
				edges[0][n][0] : NULL;
		    }
		    if ((i == 0 || i == gmax[0]) &&
		    	(j == 0 || j == gmax[1]))
		    {
		    	m = (i == 0) ? 0 : 1;
		    	n = (j == 0) ? 0 : 1;
			crx_store[num_crx].p = corners[m][n][0]->posn;
		    }
		    ++num_crx;
		}
		if (comp[i][j][gmax[2]-1] != comp[i][j][gmax[2]])
		{
		    coords[2] = U[2];
		    crx_store[num_crx].p = Point(coords);
		    crx_store[num_crx].s = faces[2][1];
		    z_crx[i][j][gmax[2]-1] = &crx_store[num_crx];
		    if (i == 0 || i == gmax[0])
		    {
		    	m = (i == 0) ? 0 : gmax[0]-1;
		    	n = (i == 0) ? 0 : 1;
			y_curve_crx[m][j][gmax[2]-1] = z_crx[i][j][gmax[2]-1];
			y_curve_crx[m][j][gmax[2]-1]->c = 
				(y_curve_crx[m][j][gmax[2]-1]->c == NULL) ?
				edges[1][1][n] : NULL;
		    }
		    if (j == 0 || j == gmax[1])
		    {
		    	m = (j == 0) ? 0 : gmax[1]-1;
		    	n = (j == 0) ? 0 : 1;
			x_curve_crx[i][m][gmax[2]-1] = z_crx[i][j][gmax[2]-1];
			x_curve_crx[i][m][gmax[2]-1]->c = 
				(x_curve_crx[i][m][gmax[2]-1]->c == NULL) ?
				edges[0][n][1] : NULL;
		    }
		    if ((i == 0 || i == gmax[0]) &&
		    	(j == 0 || j == gmax[1]))
		    {
		    	m = (i == 0) ? 0 : 1;
		    	n = (j == 0) ? 0 : 1;
			crx_store[num_crx].p = corners[m][n][1]->posn;
		    }
		    ++num_crx;
		}
	    }
	}

	/* install curves crxings */

	for (i = 0; i <= gmax[0]; ++i)
	{
	    coords[0] = DL[0] + i*h[0];
	    if (comp[i][1][1] != comp[i][0][1] &&
	    	comp[i][1][1] != comp[i][1][0])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[0][0][0]->posn;
		else if (i == gmax[0])
		    crx_store[num_crx].p = corners[1][0][0]->posn;
		else
		{
	    	    coords[1] = L[1]; coords[2] = L[2];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[0][0][0];
		x_curve_crx[i][0][0] = &crx_store[num_crx++];
	    }
	    if (comp[i][gmax[1]-1][1] != comp[i][gmax[1]][1] &&
	    	comp[i][gmax[1]-1][1] != comp[i][gmax[1]-1][0])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[0][1][0]->posn;
		else if (i == gmax[0])
		    crx_store[num_crx].p = corners[1][1][0]->posn;
		else
		{
	    	    coords[1] = U[1]; coords[2] = L[2];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[0][1][0];
		x_curve_crx[i][gmax[1]-1][0] = &crx_store[num_crx++];
	    }
	    if (comp[i][1][gmax[2]-1] != comp[i][0][gmax[2]-1] &&
	    	comp[i][1][gmax[2]-1] != comp[i][1][gmax[2]])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[0][0][1]->posn;
		else if (i == gmax[0])
		    crx_store[num_crx].p = corners[1][0][1]->posn;
		else
		{
	    	    coords[1] = L[1]; coords[2] = U[2];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[0][0][1];
		x_curve_crx[i][0][gmax[2]-1] = &crx_store[num_crx++];
	    }
	    if (comp[i][gmax[1]-1][gmax[2]-1] != comp[i][gmax[1]][gmax[2]-1] &&
	    	comp[i][gmax[1]-1][gmax[2]-1] != comp[i][gmax[1]-1][gmax[2]])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[0][1][1]->posn;
		else if (i == gmax[0])
		    crx_store[num_crx].p = corners[1][1][1]->posn;
		else
		{
	    	    coords[1] = U[1]; coords[2] = U[2];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[0][1][1];
		x_curve_crx[i][gmax[1]-1][gmax[2]-1] = &crx_store[num_crx++];
	    }
	}
	for (i = 0; i <= gmax[1]; ++i)
	{
	    coords[1] = DL[1] + i*h[1];
	    if (comp[1][i][1] != comp[0][i][1] &&
	    	comp[1][i][1] != comp[1][i][0])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[0][0][0]->posn;
		else if (i == gmax[1])
		    crx_store[num_crx].p = corners[0][1][0]->posn;
		else
		{
	    	    coords[0] = L[0]; coords[2] = L[2];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[1][0][0];
		y_curve_crx[0][i][0] = &crx_store[num_crx++];
	    }
	    if (comp[gmax[0]-1][i][1] != comp[gmax[0]][i][1] &&
	    	comp[gmax[0]-1][i][1] != comp[gmax[0]-1][i][0])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[1][0][0]->posn;
		else if (i == gmax[1])
		    crx_store[num_crx].p = corners[1][1][0]->posn;
		else
		{
	    	    coords[0] = U[0]; coords[2] = L[2];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[1][0][1];
		y_curve_crx[gmax[0]-1][i][0] = &crx_store[num_crx++];
	    }
	    if (comp[1][i][gmax[2]-1] != comp[0][i][gmax[2]-1] &&
	    	comp[1][i][gmax[2]-1] != comp[1][i][gmax[2]])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[0][0][1]->posn;
		else if (i == gmax[1])
		    crx_store[num_crx].p = corners[0][1][1]->posn;
		else
		{
	    	    coords[0] = L[0]; coords[2] = U[2];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[1][1][0];
		y_curve_crx[0][i][gmax[2]-1] = &crx_store[num_crx++];
	    }
	    if (comp[gmax[0]-1][i][gmax[2]-1] != comp[gmax[0]][i][gmax[2]-1] &&
	    	comp[gmax[0]-1][i][gmax[2]-1] != comp[gmax[0]-1][i][gmax[2]])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[1][0][1]->posn;
		else if (i == gmax[1])
		    crx_store[num_crx].p = corners[1][1][1]->posn;
		else
		{
	    	    coords[0] = U[0]; coords[2] = U[2];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[1][1][1];
		y_curve_crx[gmax[0]-1][i][gmax[2]-1] = &crx_store[num_crx++];
	    }
	}
	for (i = 0; i <= gmax[2]; ++i)
	{
	    coords[2] = DL[2] + i*h[2];
	    if (comp[1][1][i] != comp[0][1][i] &&
	    	comp[1][1][i] != comp[1][0][i])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[0][0][0]->posn;
		else if (i == gmax[2])
		    crx_store[num_crx].p = corners[0][0][1]->posn;
		else
		{
	    	    coords[0] = L[0]; coords[1] = L[1];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[2][0][0];
		z_curve_crx[0][0][i] = &crx_store[num_crx++];
	    }
	    if (comp[gmax[0]-1][1][i] != comp[gmax[0]][1][i] &&
	    	comp[gmax[0]-1][1][i] != comp[gmax[0]-1][0][i])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[1][0][0]->posn;
		else if (i == gmax[2])
		    crx_store[num_crx].p = corners[1][0][1]->posn;
		else
		{
	    	    coords[0] = U[0]; coords[1] = L[1];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[2][1][0];
		z_curve_crx[gmax[0]-1][0][i] = &crx_store[num_crx++];
	    }
	    if (comp[1][gmax[1]-1][i] != comp[0][gmax[1]-1][i] &&
	    	comp[1][gmax[1]-1][i] != comp[1][gmax[1]][i])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[0][1][0]->posn;
		else if (i == gmax[2])
		    crx_store[num_crx].p = corners[0][1][1]->posn;
		else
		{
	    	    coords[0] = L[0]; coords[1] = U[1];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[2][0][1];
		z_curve_crx[0][gmax[1]-1][i] = &crx_store[num_crx++];
	    }
	    if (comp[gmax[0]-1][gmax[1]-1][i] != comp[gmax[0]][gmax[1]-1][i] &&
	    	comp[gmax[0]-1][gmax[1]-1][i] != comp[gmax[0]-1][gmax[1]][i])
	    {
		if (i == 0)
		    crx_store[num_crx].p = corners[1][1][0]->posn;
		else if (i == gmax[2])
		    crx_store[num_crx].p = corners[1][1][1]->posn;
		else
		{
	    	    coords[0] = U[0]; coords[1] = U[1];
		    crx_store[num_crx].p = Point(coords);
		}
		crx_store[num_crx].c = edges[2][1][1];
		z_curve_crx[gmax[0]-1][gmax[1]-1][i] = &crx_store[num_crx++];
	    }
	}

	/* install node crxings */

	if (comp[1][1][1] != comp[0][1][1] &&
	    comp[1][1][1] != comp[1][0][1] &&
	    comp[1][1][1] != comp[1][1][0]) 
	{
	    crx_store[num_crx].p = corners[0][0][0]->posn;
	    node_crx[0][0][0] = &crx_store[num_crx++];
	}
	if (comp[gmax[0]-1][1][1] != comp[gmax[0]][1][1] &&
	    comp[gmax[0]-1][1][1] != comp[gmax[0]-1][0][1] &&
	    comp[gmax[0]-1][1][1] != comp[gmax[0]-1][1][0]) 
	{
	    crx_store[num_crx].p = corners[1][0][0]->posn;
	    node_crx[gmax[0]-1][0][0] = &crx_store[num_crx++];
	}
	if (comp[1][gmax[1]-1][1] != comp[0][gmax[1]-1][1] &&
	    comp[1][gmax[1]-1][1] != comp[1][gmax[1]][1] &&
	    comp[1][gmax[1]-1][1] != comp[1][gmax[1]-1][0]) 
	{
	    crx_store[num_crx].p = corners[0][1][0]->posn;
	    node_crx[0][gmax[1]-1][0] = &crx_store[num_crx++];
	}
	if (comp[1][1][gmax[2]-1] != comp[0][1][gmax[2]-1] &&
	    comp[1][1][gmax[2]-1] != comp[1][0][gmax[2]-1] &&
	    comp[1][1][gmax[2]-1] != comp[1][1][gmax[2]]) 
	{
	    crx_store[num_crx].p = corners[0][0][1]->posn;
	    node_crx[0][0][gmax[2]-1] = &crx_store[num_crx++];
	}
	if (comp[gmax[0]-1][gmax[1]-1][1] != comp[gmax[0]][gmax[1]-1][1] &&
	    comp[gmax[0]-1][gmax[1]-1][1] != comp[gmax[0]-1][gmax[1]][1] &&
	    comp[gmax[0]-1][gmax[1]-1][1] != comp[gmax[0]-1][gmax[1]-1][0]) 
	{
	    crx_store[num_crx].p = corners[1][1][0]->posn;
	    node_crx[gmax[0]-1][gmax[1]-1][0] = &crx_store[num_crx++];
	}
	if (comp[1][gmax[1]-1][gmax[2]-1] != comp[0][gmax[1]-1][gmax[2]-1] &&
	    comp[1][gmax[1]-1][gmax[2]-1] != comp[1][gmax[1]][gmax[2]-1] &&
	    comp[1][gmax[1]-1][gmax[2]-1] != comp[1][gmax[1]-1][gmax[2]]) 
	{
	    crx_store[num_crx].p = corners[0][1][1]->posn;
	    node_crx[0][gmax[1]-1][gmax[2]-1] = &crx_store[num_crx++];
	}
	if (comp[gmax[0]-1][1][gmax[2]-1] != comp[gmax[0]][1][gmax[2]-1] &&
	    comp[gmax[0]-1][1][gmax[2]-1] != comp[gmax[0]-1][0][gmax[2]-1] &&
	    comp[gmax[0]-1][1][gmax[2]-1] != comp[gmax[0]-1][1][gmax[2]]) 
	{
	    crx_store[num_crx].p = corners[1][0][1]->posn;
	    node_crx[gmax[0]-1][0][gmax[2]-1] = &crx_store[num_crx++];
	}
	if (comp[gmax[0]-1][gmax[1]-1][gmax[2]-1] != 
	    comp[gmax[0]][gmax[1]-1][gmax[2]-1] &&
	    comp[gmax[0]-1][gmax[1]-1][gmax[2]-1] != 
	    comp[gmax[0]-1][gmax[1]][gmax[2]-1] &&
	    comp[gmax[0]-1][gmax[1]-1][gmax[2]-1] != 
	    comp[gmax[0]-1][gmax[1]-1][gmax[2]]) 
	{
	    crx_store[num_crx].p = corners[1][1][1]->posn;
	    node_crx[gmax[0]-1][gmax[1]-1][gmax[2]-1] = &crx_store[num_crx++];
	}

	return num_crx;
}	/* end install_bdry_corner_crx */


LOCAL	int count_crx_through_comp(
	int *gmax,
	COMPONENT ***comp)
{
	int i,j,k,num_crx,num_comp;

	num_crx = 0;
	for (i = 0; i <= gmax[0]; ++i)
	{
	    for (j = 0; j <= gmax[1]; ++j)
	    {
	    	for (k = 0; k <= gmax[2]; ++k)
		{
		    if (i != gmax[0])
		    {
		    	if (comp[i][j][k] != comp[i+1][j][k])
			    ++num_crx;
		    	if (j != gmax[1])
			{
			    num_comp = count_side_comp(comp[i][j][k],
			    	     comp[i+1][j][k],comp[i][j+1][k],
				     comp[i+1][j+1][k]);
			    if (num_comp == 3)
			    	++num_crx;
			}
		    }
		    if (j != gmax[1])
		    {
		    	if (comp[i][j][k] != comp[i][j+1][k])
			    ++num_crx;
		    	if (k != gmax[2])
			{
			    num_comp = count_side_comp(comp[i][j][k],
			    	     comp[i][j+1][k],comp[i][j][k+1],
				     comp[i][j+1][k+1]);
			    if (num_comp == 3)
			    	++num_crx;
			}
		    }
		    if (k != gmax[2])
		    {
		    	if (comp[i][j][k] != comp[i][j][k+1])
			    ++num_crx;
		    	if (i != gmax[0])
			{
			    num_comp = count_side_comp(comp[i][j][k],
			    	     comp[i][j][k+1],comp[i+1][j][k],
				     comp[i+1][j][k+1]);
			    if (num_comp == 3)
			    	++num_crx;
			}
		    }
		}
	    }
	}
	return num_crx;
}	/* end count_crx_through_comp */

LOCAL	int count_side_comp(
	COMPONENT c1,
	COMPONENT c2,
	COMPONENT c3,
	COMPONENT c4)
{
	int nc = 1;
	if (c1 != c2)
	{
	    ++nc;
	    if (c3 != c1 && c3 != c2)
	    {
	    	++nc;
		if (c4 != c1 && c4 != c2 && c4 != c3)
	    	    ++nc;
	    }
	    else if (c4 != c1 && c4 != c2)
	    	++nc;
	}
	else if (c2 != c3)
	{
	    ++nc;
	    if (c4 != c2 && c4 != c3)
                ++nc;
	}
	return nc;
}	/* end count_side_comp */

LOCAL	void alloc_grid_crx_mem(
	EG_CRX *eg_crx,
	int *gmax,
	int num_crx,
	bool include_curve_crx)
{
	tri_array(&eg_crx->x_crx,gmax[0],gmax[1]+1,gmax[2]+1,
				sizeof(BBI_POINT*));
        tri_array(&eg_crx->y_crx,gmax[0]+1,gmax[1],gmax[2]+1,
				sizeof(BBI_POINT*));
        tri_array(&eg_crx->z_crx,gmax[0]+1,gmax[1]+1,gmax[2],
				sizeof(BBI_POINT*));
	if (include_curve_crx)
	{
	    tri_array(&eg_crx->x_curve_crx,gmax[0]+1,gmax[1],gmax[2],
				sizeof(BBI_POINT*));
            tri_array(&eg_crx->y_curve_crx,gmax[0],gmax[1]+1,gmax[2],
				sizeof(BBI_POINT*));
            tri_array(&eg_crx->z_curve_crx,gmax[0],gmax[1],gmax[2]+1,
				sizeof(BBI_POINT*));
            tri_array(&eg_crx->node_crx,gmax[0],gmax[1],gmax[2],
				sizeof(BBI_POINT*));
	}
        vector(&eg_crx->crx_store,num_crx,sizeof(BBI_POINT));
}	/* end alloc_grid_crx_mem */

LOCAL  void free_grid_crx_mem(
	EG_CRX *eg_crx,
	bool   include_curve_crx)
{
	free_these(4,eg_crx->x_crx,eg_crx->y_crx,eg_crx->z_crx,
		   eg_crx->crx_store);
	if (include_curve_crx)
        {
	    free_these(5,eg_crx->x_curve_crx,eg_crx->y_curve_crx,
	    		eg_crx->z_curve_crx,eg_crx->node_crx,
			eg_crx->curves);
	}
}	/* end free_grid_crx_mem */


/**********************************************************************
*	This function reset components of the domain to NO_COMP       *
**********************************************************************/

LOCAL	void reset_domain_comp(
	COMPONENT ***comp,
	RECT_GRID gr)
{
	int i,j,k;
	int *gmax = gr.gmax;

	for (i = 0; i <= gmax[0]; ++i)
	{
	    for (j = 0; j <= gmax[1]; ++j)
	    {
	    	for (k = 0; k <= gmax[2]; ++k)
		{
	    	    comp[i][j][k] = NO_COMP;
		}
	    }
	}
}	/* end reset_domain_comp */


/**********************************************************************
*	This function set pos_comp to the positive side of the        *
*	surface func = 0 (func > 0) domain.                           *
**********************************************************************/

LOCAL	void assign_positive_comp(
	float (*func)(POINTER,float*),
	POINTER func_params,
	COMPONENT ***comp,
	RECT_GRID gr,
	COMPONENT pos_comp)
{
	int i,j,k;
	int *gmax = gr.gmax;
	float *L = gr.L;
	float *h = gr.h;
	float coords[3];

	for (i = 0; i <= gmax[0]; ++i)
	{
	    coords[0] = L[0] + i*h[0];
	    for (j = 0; j <= gmax[1]; ++j)
	    {
	    	coords[1] = L[1] + j*h[1];
	    	for (k = 0; k <= gmax[2]; ++k)
		{
	    	    coords[2] = L[2] + k*h[2];
		    if ((*func)(func_params,coords) > 0 &&
		    	comp[i][j][k] == NO_COMP)
		    	comp[i][j][k] = pos_comp;
		}
	    }
	}
}	/* end assign_positive_comp */

/**********************************************************************
*	This function set neg_comp to the negative side of the        *
*	surface func = 0 (func < 0) domain.                           *
**********************************************************************/

LOCAL	void assign_negative_comp(
	float (*func)(POINTER,float*),
	POINTER func_params,
	COMPONENT ***comp,
	RECT_GRID gr,
	COMPONENT neg_comp)
{
	int i,j,k;
	int *gmax = gr.gmax;
	float *L = gr.L;
	float *h = gr.h;
	float coords[3];

	for (i = 0; i <= gmax[0]; ++i)
	{
	    coords[0] = L[0] + i*h[0];
	    for (j = 0; j <= gmax[1]; ++j)
	    {
	    	coords[1] = L[1] + j*h[1];
	    	for (k = 0; k <= gmax[2]; ++k)
		{
	    	    coords[2] = L[2] + k*h[2];
		    if ((*func)(func_params,coords) <= 0 &&
		    	comp[i][j][k] == NO_COMP)
		    	comp[i][j][k] = neg_comp;
		}
	    }
	}
}	/* end assign_negative_comp */

/**********************************************************************
*	This function set cross_comp to the subdomain bounded by      *
*	two surfaces func_1 = 0 and func_2 = 0. The side of each      *
*	surface is given by the input values of side1 and side2.      *
**********************************************************************/

LOCAL	void assign_intersection_comp(
	float (*func_1)(POINTER,float*),
	POINTER func_1_params,
	float (*func_2)(POINTER,float*),
	POINTER func_2_params,
	COMPONENT ***comp,
	RECT_GRID gr,
	COMPONENT cross_comp,
	SIDE side1,
	SIDE side2)
{
	int i,j,k;
	int *gmax = gr.gmax;
	float *L = gr.L;
	float *h = gr.h;
	float coords[3];
	float y1,y2;

	for (i = 0; i <= gmax[0]; ++i)
	{
	    coords[0] = L[0] + i*h[0];
	    for (j = 0; j <= gmax[1]; ++j)
	    {
	    	coords[1] = L[1] + j*h[1];
	    	for (k = 0; k <= gmax[2]; ++k)
		{
	    	    coords[2] = L[2] + k*h[2];
		    y1 = (*func_1)(func_1_params,coords);
		    y2 = (*func_2)(func_2_params,coords);
		    if (comp[i][j][k] != NO_COMP) continue;
		    if (side1 == POSITIVE_SIDE && y1 > 0)
		    {
		    	if (side2 == POSITIVE_SIDE && y2 > 0)
			    comp[i][j][k] = cross_comp;
		    	else if (side2 == NEGATIVE_SIDE && y2 <= 0)
			    comp[i][j][k] = cross_comp;
		    }
		    else if (side1 == NEGATIVE_SIDE && y1 < 0)
		    {
		    	if (side2 == POSITIVE_SIDE && y2 > 0)
			    comp[i][j][k] = cross_comp;
		    	else if (side2 == NEGATIVE_SIDE && y2 <= 0)
			    comp[i][j][k] = cross_comp;
		    }
		}
	    }
	}
}	/* end assign_intersection_comp */

/**********************************************************************
*	This function sets components for a two component domain      *
*	described by the rectangular grid gr, the side func < 0       *
*	will be assigned neg_comp while the side func > 0 will be     *
*	assigned pos_comp.   					      *
**********************************************************************/

LOCAL	void assign_two_comp_domain(
	float (*func)(POINTER,float*),
	POINTER func_params,
	COMPONENT ***comp,
	RECT_GRID gr,
	COMPONENT neg_comp,
	COMPONENT pos_comp)
{
	int i,j,k;
	int *gmax = gr.gmax;
	float *L = gr.L;
	float *h = gr.h;
	float coords[3];

	for (i = 0; i <= gmax[0]; ++i)
	{
	    coords[0] = L[0] + i*h[0];
	    for (j = 0; j <= gmax[1]; ++j)
	    {
	    	coords[1] = L[1] + j*h[1];
	    	for (k = 0; k <= gmax[2]; ++k)
		{
	    	    coords[2] = L[2] + k*h[2];
		    if (comp[i][j][k] != NO_COMP) continue;
		    if ((*func)(func_params,coords) > 0)
		    	comp[i][j][k] = pos_comp;
		    else 
		    	comp[i][j][k] = neg_comp;
		}
	    }
	}
}	/* end assign_two_comp_domain */


EXPORT	float ellipsoid_func(
	POINTER func_params,
	float *coords)
{
	ELLIP_PARAMS *params;
	const float *cen,*rad;
	float arg;

	params = (ELLIP_PARAMS *)func_params;
	cen = params->cen;
	rad = params->rad;

	arg = 1.0 -
                sqr(coords[0] - cen[0])/sqr(rad[0]) -
                sqr(coords[1] - cen[1])/sqr(rad[1]) -
                sqr(coords[2] - cen[2])/sqr(rad[2]);

	return -arg;
}	/* end ellipsoid_func */

EXPORT	float bdry_box_func(
	POINTER func_params,
	float *coords)
{
	BDRY_BOX_PARAMS *params;
	float *L,*U;
	int i;

	params = (BDRY_BOX_PARAMS *)func_params;
	L = params->L;
	U = params->U;
	for (i = 0; i < 3; ++i)
	{
	    if (coords[i] < L[i] || coords[i] > U[i])
	    	return -1.0;
	}
	return 1.0;
}	/* end bdry_box_func */

EXPORT	float plane_func(
	POINTER func_params,
	float *coords)
{
	PLANE_PARAMS *params;
	const float *N,*P;
	float d;

	params = (PLANE_PARAMS*)func_params;
	N = params->N;	P = params->P;
	d = N[0]*(coords[0] - P[0]) + N[1]*(coords[1] - P[1])
			+ N[2]*(coords[2] - P[2]);
	return d;
}	/* end plane_func */

LOCAL 	void set_grid_for_surface_construction(
	RECT_GRID *dual_gr,
	RECT_GRID *rgr)
{
	int i,gmax[3];
	float L[3],U[3];

	set_dual_grid(dual_gr,rgr);
	for (i = 0; i < 3; ++i)
	{
	    L[i] = dual_gr->VL[i];
	    U[i] = dual_gr->VU[i];
	    gmax[i] = dual_gr->gmax[i];
	    if (dual_gr->lbuf[i] != 0)
	    {
	    	L[i] -= dual_gr->h[i];
		gmax[i] += dual_gr->lbuf[i] + 1;
	    }
	    if (dual_gr->ubuf[i] != 0)
	    {
	    	U[i] += dual_gr->h[i];
		gmax[i] += dual_gr->ubuf[i] + 1;
	    }
	}
	set_rect_grid(L,U,dual_gr->GL,dual_gr->GU,NULL,NULL,gmax,
			dual_gr->dim,&rgr->Remap,dual_gr);
}	/* end set_grid_for_surface_construction */

LOCAL	bool grid_line_crx_in_dir(
	float (*func)(POINTER,float*),
	POINTER func_params,
	float *crds1,
	float *crds2,
	float *crx_crds,
	int dir)
{
	float f1,f2,fm,a1,a2;
	int i,num_iter = 8;
        float coords1[MAXD],coords2[MAXD];

        for (i = 0; i < 3; ++i)
        {
            coords1[i] = crds1[i];
            coords2[i] = crds2[i];
        }

	f1 = (*func)(func_params,coords1);
	f2 = (*func)(func_params,coords2);

	crx_crds[(dir+1)%3] = coords1[(dir+1)%3];
	crx_crds[(dir+2)%3] = coords1[(dir+2)%3];
	if (f1 == 0.0)
	{
	    crx_crds[dir] = coords1[dir];
	    return YES;
	}
	else if (f2 == 0.0)
	{
	    crx_crds[dir] = coords2[dir];
	    return YES;
	}

	if ((f1 > 0.0 && f2 > 0.0) ||
	    (f1 < 0.0 && f2 < 0.0))
	    return NO;

	for (i = 0; i < num_iter; ++i)
	{
	    crx_crds[dir] = 0.5*(coords1[dir] + coords2[dir]);
	    fm = (*func)(func_params,crx_crds);
	    if (fm == 0.0)
	    {
	    	return YES;
	    }
	    else if ((f1 > 0.0 && fm > 0.0) ||
	    	     (f1 < 0.0 && fm < 0.0))
	    {
	    	coords1[dir] = crx_crds[dir];
		f1 = fm;
	    }
	    else
	    {
	    	coords2[dir] = crx_crds[dir];
		f2 = fm;
	    }
	}
	a1 = (fm - f1)/(f2 - f1);
	a2 = (f2 - fm)/(f2 - f1);
	crx_crds[dir] = a1*coords2[dir] + a2*coords1[dir];
	return YES;
}	/* end grid_line_crx_in_dir */

LOCAL	bool face_crx_in_dir(
	float (*func1)(POINTER,float*),
	POINTER func1_params,
	float (*func2)(POINTER,float*),
	POINTER func2_params,
	float *L,
	float *U,
	float *crx_crds,
	int dir)
{
	float coords[3],coords1[3],coords2[3];
	bool lower_set,upper_set;
	int i,k,num_iter = 8;
	float f1,f2;

	lower_set = upper_set = NO;
	for (i = 0; i < 3; i++)
	    coords[i] = L[i];
	f1 = (*func1)(func1_params,coords);
	f2 = (*func2)(func2_params,coords);
	if (f1 < 0 && f2 < 0)
	{
	    lower_set = YES;
	    for (i = 0; i < 3; i++)
	    	coords1[i] = coords[i];
	}
	else if (f1 >= 0 && f2 >= 0)
	{
	    upper_set = YES;
	    for (i = 0; i < 3; i++)
	    	coords2[i] = coords[i];
	}
	coords[(dir+1)%3] = U[(dir+1)%3];
	if (f1 < 0 && f2 < 0)
	{
	    lower_set = YES;
	    for (i = 0; i < 3; i++)
	    	coords1[i] = coords[i];
	}
	else if (f1 >= 0 && f2 >= 0)
	{
	    upper_set = YES;
	    for (i = 0; i < 3; i++)
	    	coords2[i] = coords[i];
	}
	coords[(dir+2)%3] = U[(dir+2)%3];
	if (f1 < 0 && f2 < 0)
	{
	    lower_set = YES;
	    for (i = 0; i < 3; i++)
	    	coords1[i] = coords[i];
	}
	else if (f1 >= 0 && f2 >= 0)
	{
	    upper_set = YES;
	    for (i = 0; i < 3; i++)
	    	coords2[i] = coords[i];
	}
	coords[(dir+1)%3] = L[(dir+1)%3];
	if (f1 < 0 && f2 < 0)
	{
	    lower_set = YES;
	    for (i = 0; i < 3; i++)
	    	coords1[i] = coords[i];
	}
	else if (f1 >= 0 && f2 >= 0)
	{
	    upper_set = YES;
	    for (i = 0; i < 3; i++)
	    	coords2[i] = coords[i];
	}
	if (!(lower_set && upper_set))
	    return NO;
	for (i = 0; i < num_iter; ++i)
	{
	    for (k = 0; k < 3; ++k)
	    	coords[k] = coords1[k];
	    coords[(dir+1)%3] = 0.5*(coords1[(dir+1)%3] + 
	    		coords2[(dir+1)%3]);
	    f1 = (*func1)(func1_params,coords);
	    f2 = (*func2)(func2_params,coords);
	    if (f1 < 0 && f2 < 0)
	    	coords1[(dir+1)%3] = coords[(dir+1)%3];
	    else if ( f1 >= 0 && f2 >= 0)
	    	coords2[(dir+1)%3] = coords[(dir+1)%3];

	    for (k = 0; k < 3; ++k)
	    	coords[k] = coords1[k];
	    coords[(dir+2)%3] = 0.5*(coords1[(dir+2)%3] + 
	    		coords2[(dir+1)%3]);
	    f1 = (*func1)(func1_params,coords);
	    f2 = (*func2)(func2_params,coords);
	    if (f1 < 0 && f2 < 0)
	    	coords1[(dir+1)%3] = coords[(dir+1)%3];
	    else if ( f1 >= 0 && f2 >= 0)
	    	coords2[(dir+1)%3] = coords[(dir+1)%3];
	}

}	/* end face_crx_in_dir */

LOCAL	void make_grid_surfaces(
	BLK_CRX      *blk_crx,
	const EG_CRX *eg_crx,
	int          *gmax,
	bool         include_curve_crx)
{
	int i,j,k,num_blk;
	BLK_TRI *bm,****blk_mem,*blk_mem_store;

	num_blk = 0;
	for (i = 0; i < gmax[0]; ++i)
	{
	    for (j = 0; j < gmax[1]; ++j)
	    {
		for (k = 0; k < gmax[2]; ++k)
		{
		    if (onfront_block(i,j,k,eg_crx))
			++num_blk;
		}
	    }
	}
	tri_array(&blk_mem,gmax[0],gmax[1],gmax[2],sizeof(BLK_TRI*));
        vector(&blk_mem_store,num_blk,sizeof(BLK_TRI));

	num_blk = 0;
	for (i = 0; i < gmax[0]; ++i)
	{
	    for (j = 0; j < gmax[1]; ++j)
	    {
		for (k = 0; k < gmax[2]; ++k)
		{
		    if (onfront_block(i,j,k,eg_crx))
		    {
			bm = blk_mem[i][j][k] = &blk_mem_store[num_blk++];
			bm->blk_info = blk_crx->blk_info;
			assign_blk_crx(blk_crx,i,j,k,eg_crx,include_curve_crx);
			switch (blk_crx->blk_type)
			{
			case COMP2_BLOCK:
			    construct_comp2_blk(blk_crx,bm);
			    break;
			case COMP3_BLOCK:
			    break;
			case BDRY_BLOCK:
			    construct_bdry_blk(blk_crx,bm);
			    break;
			default:
			    screen("UNKNOWN BLOCK: code needed!\n");
			    clean_up(ERROR);
			}
			if (i != 0)
			    stitch_adj_blk(blk_mem[i-1][j][k],bm);
			if (j != 0)
			    stitch_adj_blk(blk_mem[i][j-1][k],bm);
			if (k != 0)
			    stitch_adj_blk(blk_mem[i][j][k-1],bm);
		    }
		}
	    }
	}
	for (i = 0; i < gmax[0]; ++i)
	{
	    for (j = 0; j < gmax[1]; ++j)
	    {
		for (k = 0; k < gmax[2]; ++k)
		{
		    if ((bm = blk_mem[i][j][k]) != NULL)
		    {
			if (i != gmax[0]-1)
			    remove_null_pair(bm,blk_mem[i+1][j][k],0);
			if (j != gmax[1]-1)
			    remove_null_pair(bm,blk_mem[i][j+1][k],1);
			if (k != gmax[2]-1)
			    remove_null_pair(bm,blk_mem[i][j][k+1],2);
		    }
		}
	    }
	}
	for (i = 0; i < eg_crx->num_curves; ++i)
	    reorder_curve_link_list(eg_crx->curves[i]);

	free_these(2,blk_mem,blk_mem_store);
}	/* end make_grid_surfaces */

#endif /* defined(THREED) */
