/*
*				hsoln2.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains a vectorized row-by-row (column-by-column) driver for a
*	finite difference stencil solution of the hyperbolic equations
*	within a single component of a tracked interface problem.
*	Point sources in the region are found and passed appropriately.
*/


#include <hyp/hlocaldecs.h>

#if defined(CONSERVATIVE_ALG)

LOCAL  int hyp_tri_grid_driver2(TRI_GRID*,Front*,Wave*,RECT_GRID*);

LIB_LOCAL int init_hyp_solution_function2(
          TRI_GRID      *tri_grid,
          Wave          *wave,
          Front         *front)
{
        RECT_GRID  Dual_grid, *comp_grid = wave->rect_grid;
        int        status = ERROR_IN_STEP;

        clear_wave_pointers(wave);
        if (wave->sizest == 0)
            return GOOD_STEP;
        /*
        *    hsoln.c locates interior states at crossings
        *    of grid lines; thus it should not be given
        *    wave->rect_grid but rather its dual.
        */

        set_dual_grid(&Dual_grid,comp_grid);

        scalar(&wave_tri_soln(wave),sizeof(TRI_SOLN));
        if (wave_tri_soln(wave) == NULL)
        {
            (void) printf("WARNING in init_hyp_solution_function(), "
                          "can't allocate tri_soln\n");
            return ERROR_IN_STEP;
        }
#if defined(USE_OVERTURE)
        wave_tri_soln(wave)->cg_over = wave->cg_over;
        wave_tri_soln(wave)->patch_number = wave->patch_number;
        wave_tri_soln(wave)->use_overture_state = wave->use_overture_state;
        wave_tri_soln(wave)->overture_init_step = wave->overture_init_step;
        wave_tri_soln(wave)->cg_over_function = wave->cg_over_function;
        wave_tri_soln(wave)->patch_level = wave->patch_level;
        wave_tri_soln(wave)->NumberOfLevels = wave->NumberOfLevels;
        if (wave->patch_number > 0)
        {
            wave_tri_soln(wave)->center_comp = wave->center_comp;
        }
#endif /* if defined(USE_OVERTURE) */

        wave_tri_soln(wave)->Tri_grid_hooks = wave->Tri_grid_hooks;

        wave_tri_soln(wave)->new_tri_grid = NULL;
        status = hyp_tri_grid_driver2(tri_grid,front,wave,&Dual_grid);

        if (status != GOOD_STEP)
        {
            (void) printf("WARNING in init_hyp_solution_function2(), "
                          "hyp_tri_grid_driver2() failed\n");
            free_wave_pointers(wave);
            return status;
        }
#if defined(USE_OVERTURE)
        wave->cg_over_function =
                wave_tri_soln(wave)->tri_grid->cg_over_function;
        wave_tri_soln(wave)->cg_over_function =
                wave_tri_soln(wave)->tri_grid->cg_over_function;
#endif /* if defined(USE_OVERTURE) */

        wave->hyp_soln = h_hyp_solution;
        wave->hyp_grad_soln = h_hyp_grad_solution;

        return GOOD_STEP;
}

LOCAL int hyp_tri_grid_driver2(
        TRI_GRID  *tri_grid,
        Front     *front,
        Wave      *wave,
        RECT_GRID *rect_grid)
{
        int                status;
        TRI_SOLN           *soln = wave_tri_soln(wave);
        INTERFACE          *intfc = front->interf;
        size_t             sizest = front->sizest;
        INTERPOLATORS      *interpolator = &wave->interpolator;
        EL_INTEGRALS       *el_integral = &wave->el_integral;
        UNSPLIT            *unsplit = &wave->unsplit;

        debug_print("tri_grid","Entered hyp_tri_grid_driver2()\n");

        if(tri_grid == NULL)
            return ERROR_IN_STEP; 
 
#if defined(DEBUG_TRI_GRID)
        if (debugging("tri_grid"))
        {
            (void) printf("Interface input to tri grid %llu\n",
                      interface_number(intfc));
            print_interface(intfc);
            (void) printf("Trigrid rectangular grid\n");
            print_rectangular_grid(rect_grid);
            (void) printf("\n");
        }
#endif /* defined(DEBUG_TRI_GRID) */
        tri_grid->tri_grid_hooks = soln->Tri_grid_hooks;
        tri_grid->num_lin_guess_factor = 1.5;        /*TOLERANCE*/


#if defined(USE_OVERTURE)
        tri_grid->cg_over = soln->cg_over;
        tri_grid->cg_over_function = soln->cg_over_function;
        tri_grid->patch_number = soln->patch_number;
        tri_grid->use_overture_state = soln->use_overture_state;
        tri_grid->overture_init_step = soln->overture_init_step;
        tri_grid->center_comp = soln->center_comp;
        tri_grid->patch_level = soln->patch_level;
        tri_grid->NumberOfLevels = soln->NumberOfLevels;
#endif /* if defined(USE_OVERTURE) */

        status = construct_tri_grid(tri_grid,rect_grid,front);

        if (status != GOOD_STEP)
        {
            (void) printf("WARNING in hyp_tri_grid_driver(), "
                          "construct_tri_grid() failed\n");
        }

        set_tri_soln_struct(soln,intfc,tri_grid,sizest,interpolator,
                            el_integral,unsplit);

        debug_print("tri_grid","Left hyp_tri_grid_driver2()\n");
        return status;
}  /* end of hyp_tri_grid_driver2 */ 


#endif /* if defined(CONSERVATIVE_ALG) */

