/*
*				hdriver.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains drivers for the hyperbolic library.
*/

#include <hyp/hlocaldecs.h>

	/* LOCAL Function Declarations */
LOCAL	int	hyp_split_driver(float,float*,Wave*,Front*,
				 void(*)(int,int*,float,float,
					 Wave*,Wave*,Front*,Front*,COMPONENT));
#if defined(CONSERVATIVE_ALG)
LOCAL   int     hyp_parab_mix_unsplit_consv_driver(float,float*,Wave*,Front*,
                                 void(*)(int,int*,float,float,
                                         Wave*,Wave*,Front*,Front*,COMPONENT),
                                 int(*)(float,float*,Wave*,Front*));

LOCAL	int	hyp_unsplit_consv_driver(float,float*,Wave*,Front*,
				 void(*)(int,int*,float,float,
					 Wave*,Wave*,Front*,Front*,COMPONENT));

LOCAL   void    clear_tri_blk_type(Wave*);
LOCAL   int     hyp_DG_driver(float,float*,Wave*,Front*);
LOCAL   INTERFACE *next_mesh = NULL;
#endif /* if defined(CONSERVATIVE_ALG) */

EXPORT int hyp_vector_driver(
	float		dt,
	float		*dt_frac,
	Wave		*wave,
	Front		*front)
{
	int		status;
	DEBUG_ENTER(hyp_vector_driver)

	debug_print("hyp","Entered hyp_vector_driver()\n");

	set_pt_source_interior_vectors(wave);

	front->interf->e_comps = NULL;

        // status = hyp_split_driver(dt,dt_frac,wave,front,hyp_reg_vec);
        status = hyp_DG_driver(dt,dt_frac,wave,front);

	front->interf->e_comps = NULL;

	free_pt_source_interior_vectors(wave);

	debug_print("hyp","Left hyp_vector_driver()\n");

	DEBUG_LEAVE(hyp_vector_driver)
	return status;
}		/*end hyp_vector_driver*/

EXPORT int hyp_scalar_driver(
	float		dt,
	float		*dt_frac,
	Wave		*wave,
	Front		*front)
{
	int		status;
	DEBUG_ENTER(hyp_scalar_driver)

	debug_print("hyp","Entered hyp_scalar_driver()\n");

	set_pt_source_interior_vectors(wave);

	front->interf->e_comps = NULL;

	status = hyp_split_driver(dt,dt_frac,wave,front,hyp_npt);

	front->interf->e_comps = NULL;

	free_pt_source_interior_vectors(wave);

	debug_print("hyp","Left hyp_scalar_driver()\n");

	DEBUG_LEAVE(hyp_scalar_driver)

	return status;
}		/*end hyp_scalar_driver*/

/*
*			hyp_split_driver():
*/

LOCAL	int hyp_split_driver(
	float		dt,
	float		*dt_frac,
	Wave		*wave,
	Front		*front,
	void		(*sweep)(int,int*,float,float,
				 Wave*,Wave*,Front*,Front*,COMPONENT))
{
	Front		*newfront;
	Front		*infront[3], *outfront[3];
	INTERFACE	*current_intfc, *tmp_intfc;
	Wave		*wk_wv1 = NULL, *wk_wv2 = NULL;
	Wave		*tmpwave = NULL;
	Wave		*inwave[3],  *outwave[3];
	float		dh[MAXD];
	int		step = front->step;
	int		status;
	int		i, k, dim = front->interf->dim;
	int		*iperm;	/* permutation of {0,...,dim-1} */
	COMPONENT	max_comp;
	static char	warn[] = "WARNING in hyp_split_driver()";
	static char	err[] = "ERROR in hyp_split_driver()";
	DEBUG_ENTER(hyp_split_driver)

	debug_print("hyp","Entered hyp_split_driver()\n");
	set_hyp_npt_globals(wave);

		/* Advance Front */

	start_clock("advance_front");
	status = advance_front(dt,dt_frac,front,&newfront,(POINTER) wave); 
	stop_clock("advance_front");

	if (debugging("hyp"))
	{
	    (void) printf("front->interf\n");
	    print_interface(front->interf);
	    (void) printf("newfront->interf\n");
	    print_interface(newfront->interf);
	}

	if (status != GOOD_STEP)
	{
	    (void) printf("%s advance_front() failed, "
	                  "dt_frac = %g\n",warn,*dt_frac);
	    print_time_step_status("time step status = ",status,"\n");
	    DEBUG_LEAVE(hyp_split_driver)
	    return status;
	}
	initialize_max_wave_speed(wave);

			/* hyperbolic solver */
	if( debugging("hyp_states") )
	{
	    (void) printf("States before calling hyp_solver:\n\n");
	    (void) printf("Old front\n");
	    graph_front_states(front);
	    (void) printf("New front\n");
	    graph_front_states(newfront);
	    (void) printf("wave %p\n",(POINTER)wave);
	    (*wave->show_wave_states)(wave);
	}

	iperm = set_iperm(step,dim);
	for (i = 0; i < dim; ++i)
	{
	    dh[i] = wave->rect_grid->h[iperm[i]];
	}
	if (debugging("x_sweep_only"))
	{
	    for (i = 0; i < dim; ++i)
	    {
	    	iperm[i] = i;
	    	dh[i]	 = wave->rect_grid->h[i];
	    }
	    dim = 1;
	}


		/* Initialize Intermediate Storage for States */

	wk_wv1 = copy_wave(wave);
	clear_wave_pointers(wk_wv1);
	switch (dim)
	{
	case 1:
	    inwave[0] = wave;	outwave[0] = wk_wv1;
	    infront[0] = front;	outfront[0] = newfront;
	    break;
	case 2:
	    if (wave->min_storage == YES)
	    {
	    	inwave[0] = wave;	outwave[0] = wk_wv1;
	    	inwave[1] = outwave[0];	outwave[1] = wave;
	    }
	    else
	    {
	    	wk_wv2 = copy_wave(wave);
	    	clear_wave_pointers(wk_wv2);
	    	inwave[0] = wave;	outwave[0] = wk_wv1;
	    	inwave[1] = outwave[0];	outwave[1] = wk_wv2;
	    }
	    infront[0] = front;	outfront[0] = newfront;
	    infront[1] = newfront;	outfront[1] = newfront;
	    tmpwave = wk_wv1;
	    break;
	case 3:
	    if (wave->min_storage == YES)
	    {
	    	inwave[0] = wave;	outwave[0] = wk_wv1;
	    	inwave[1] = outwave[0];	outwave[1] = wave;
	    	inwave[2] = outwave[1];	outwave[2] = wk_wv1;
	    }
	    else
	    {
	    	wk_wv2 = copy_wave(wave);
	    	clear_wave_pointers(wk_wv2);
	    	inwave[0] = wave;	outwave[0] = wk_wv2;
	    	inwave[1] = outwave[0];	outwave[1] = wk_wv1;
	    	inwave[2] = outwave[1];	outwave[2] = wk_wv2;
	    	tmpwave = wk_wv1;
	    }
	    infront[0] = front;	   outfront[0] = newfront;
	    infront[1] = newfront; outfront[1] = newfront;
	    infront[2] = newfront; outfront[2] = newfront;
	    break;
	}

	start_clock("init_hyp_solution");
	assign_wave_parameters(outwave[0],wave);
        outwave[0]->old_wave = wave;
#if defined(CONSERVATIVE_ALG)
        if(wave_tri_soln(wave)->new_tri_grid != NULL)
        {
            status = init_hyp_solution_function2(
                          wave_tri_soln(wave)->new_tri_grid,
                          outwave[0],newfront);
            wave_tri_soln(wave)->new_tri_grid = NULL;
        }
        else
#endif /* if defined(CONSERVATIVE_ALG) */
        {
	    status = init_hyp_solution_function(outwave[0],newfront);
        }
	status = syncronize_time_step_status(status,front->pp_grid);
	if (status != GOOD_STEP) 
	{
	    *dt_frac = min(*dt_frac,Min_time_step_modification_factor(front));
	    free_front(newfront);
	    free_wave_pointers(outwave[0]);
	    if (wk_wv1 != NULL)
	    	free_wave(wk_wv1);
	    if (wk_wv2 != NULL)
	    	free_wave(wk_wv2);
	    (void) printf("%s, init_hyp_solution_function() failed\n",warn);
	    DEBUG_LEAVE(hyp_split_driver)
	    return status;
	}
	if (dim > 1 && (wave->min_storage != YES))
	{
	    assign_wave_parameters(outwave[1],outwave[0]);
	    if( !copy_hyp_solution_function(outwave[0],outwave[1]) )
	    {
	    	screen("%s, copy_hyp_solution_function() failed\n",err);
	    	free_front(newfront);
	    	free_wave_pointers(outwave[0]);
	    	if (wk_wv1 != NULL)
	    	    free_wave(wk_wv1);
		if (wk_wv2 != NULL)
		    free_wave(wk_wv2);
		DEBUG_LEAVE(hyp_split_driver)
		return ERROR_IN_STEP;
	    }
	}
	stop_clock("init_hyp_solution");

	start_clock("hyp_solver");


	/*
	*	Temporary storage allocated by store() in directional sweeps 
	*	should be allocated from the tmp_intfc table, since
	*	this storage is freed when tmp_intfc is deleted at
	*	the two sweeps.
	*/

	current_intfc = current_interface();
	tmp_intfc = make_interface(wave->rect_grid->dim);
		/* Call sweep functions in cyclic order */
	max_comp = max_component(front->interf);
	for (k = 0; k < dim; ++k)
	{
	    (*sweep)(k,iperm,dh[k],dt,inwave[k],outwave[k],
		     infront[k],outfront[k],max_comp);

	    if( debugging("hyp_states") )
	    {
	    	(void) printf("After %d%s sweep: outwave[%d] %p\n",
			      k,ordinal_suffix(k),k,(POINTER)outwave[k]);
		if (wave->show_tri_soln)
		    (*wave->show_tri_soln)(outfront[k],outwave[k]);
		(*wave->show_wave_states)(outwave[k]);
	    }
	    
	    /* parallel part for interior states */

            if (!scatter_states(outwave[k],outfront[k],iperm,k))
            {
                screen("%s, scatter_states() failed\n",err);
                clean_up(ERROR);
            }

	    if( debugging("special_plot") && wave->plot_hyp_soln)
	    	(*wave->plot_hyp_soln)(outfront[k],outwave[k],step);
	    if (k == dim-1) break;
	    if (k == 0 && (wave->min_storage == YES))
	    {
	    	free_wave_pointers(outwave[1]);
	    	if (! copy_hyp_solution_function(outwave[0],outwave[1]))
	    	{
	    	    free_front(newfront);
	    	    if (wk_wv1 != NULL)
	    		free_wave(wk_wv1);
	    	    if (wk_wv2 != NULL)
	    		free_wave(wk_wv2);
	    	    screen("%s, copy_hyp_solution_function() failed\n",err);
		    DEBUG_LEAVE(hyp_split_driver)
		    return ERROR_IN_STEP;
		}
	    }
	}

	if (dim == 1)
	    free_wave_pointers(wave);
	set_current_interface(current_intfc);
	(void) delete_interface(tmp_intfc);

		/* Copy updated front, wave */
	if ((dim % 2) || (wave->min_storage == NO))
	    assign_copy_wave_pointers(wave,outwave[dim-1]);

	/* Free temporary storage, update front */

	if (tmpwave != NULL)
	    free_copy_wave_pointers(tmpwave);

	assign_interface_and_free_front(front,newfront);

	if (wk_wv1 != NULL)
	    free_wave(wk_wv1);
	if (wk_wv2 != NULL)
	    free_wave(wk_wv2);

	stop_clock("hyp_solver");

	if( debugging("hyp_states") )
	{
	    int i;
	    for (i = 0; i < dim; ++i)
	    	(void) printf("sweep %d Maxsp(wave)[%d] %g\n",i,
			      i,Maxsp(wave)[i]);
	    (void) printf("Wave %p after calling hyp_solver:\n",(POINTER)wave);
	    if (wave->show_tri_soln)
		(*wave->show_tri_soln)(front,wave);
	    (*wave->show_wave_states)(wave);
	}

	debug_print("hyp","Left hyp_split_driver()\n");
	DEBUG_LEAVE(hyp_split_driver)
	return status;
}		/*end hyp_split_driver*/


#if defined(CONSERVATIVE_ALG)
EXPORT  int hyp_parab_mix_driver(
	float		dt,
	float		*dt_frac,
	Wave		*wave,
	Front		*front,
        int            (*parab_driver)(float,float*,Wave*,Front*))
{
        int             status;
        debug_print("hyp","Entered hyp_parab_mix_driver()\n");

        set_pt_source_interior_vectors(wave);

        front->interf->e_comps = NULL;

        status = hyp_parab_mix_unsplit_consv_driver(dt,
                  dt_frac,wave,front,hyp_reg_vec,parab_driver);

        front->interf->e_comps = NULL;

        free_pt_source_interior_vectors(wave);

        debug_print("hyp","Left hyp_parab_mix_driver()\n");
        return status;
}

LOCAL   int hyp_parab_mix_unsplit_consv_driver(
	float		dt,
	float		*dt_frac,
	Wave		*wave,
	Front		*front,
	void		(*hyp_sweep)(int,int*,float,float,
				 Wave*,Wave*,Front*,Front*,COMPONENT),
        int            (*parab_driver)(float,float*,Wave*,Front*))
{
	Front		*newfront;
	Front		*infront[3], *outfront[3];
	INTERFACE	*current_intfc, *tmp_intfc;
	Wave		*wk_wv1 = NULL;
	Wave		*tmpwave = NULL;
	Wave		*inwave[3],  *outwave[3];
	float		dh[MAXD];
	int		step = front->step;
	int		status;
	int		i, k, dim = front->interf->dim;
	int		*iperm;	/* permutation of {0,...,dim-1} */
	COMPONENT	max_comp;
	static char	warn[] = "WARNING in hyp_parab_mix_unsplit_consv_driver()";
	static char	err[] = "ERROR in hyp_parab_mix_unsplit_consv_driver()";

        static int      first_consv_step = YES;  

        DEBUG_ENTER(hyp_parab_mix_unsplit_consv_driver)

	set_hyp_npt_globals(wave);

		/* Advance Front */

        reset_pt_index_on_intfc(front->interf);

	start_clock("advance_front");
	status = advance_front(dt,dt_frac,front,&newfront,(POINTER)wave); 
	stop_clock("advance_front");

	if (debugging("hyp"))
	{
	    (void) printf("front->interf\n");
	    print_interface(front->interf);
	    (void) printf("newfront->interf\n");
	    print_interface(newfront->interf);
	}

	if (status != GOOD_STEP)
	{
	    (void) printf("%s advance_front() failed, "
	                  "dt_frac = %g\n",warn,*dt_frac);
	    print_time_step_status("time step status = ",status,"\n");
	    DEBUG_LEAVE(hyp_parab_mix_unsplit_consv_driver)
	    return status;
	}
	initialize_max_wave_speed(wave);

			/* hyperbolic solver */
	if( debugging("hyp_states") )
	{
	    (void) printf("States before calling hyp_solver:\n\n");
	    (void) printf("Old front\n");
	    graph_front_states(front);
	    (void) printf("New front\n");
	    graph_front_states(newfront);
	    (void) printf("wave %p\n",(POINTER)wave);
	    (*wave->show_wave_states)(wave);
	}

	iperm = set_iperm(step,dim);
	for (i = 0; i < dim; ++i)
	    dh[i] = wave->rect_grid->h[i];

		/* Initialize Intermediate Storage for States */

	wk_wv1 = copy_wave(wave);
	clear_wave_pointers(wk_wv1);
	switch (dim)
	{
	case 1:
	    inwave[0] = wave;	outwave[0] = wk_wv1;
	    infront[0] = front;	outfront[0] = newfront;
	    break;
	case 2:
	    inwave[0] = wave; outwave[0] = wk_wv1;
	    infront[0] = front;	outfront[0] = newfront;
	    tmpwave = wk_wv1;
	    break;
	}

	start_clock("init_hyp_solution");
	assign_wave_parameters(outwave[0],wave);
        if(wave_tri_soln(wave)->new_tri_grid != NULL)
        {
	    status = init_hyp_solution_function2(
                          wave_tri_soln(wave)->new_tri_grid,
                          outwave[0],newfront);
            wave_tri_soln(wave)->new_tri_grid = NULL; 
        }
        else
        {
            /**** FOR Grid-Free and Unsplit FD Solver ****/ 
            init_hyp_solution_function(outwave[0],newfront); 
        }

	status = syncronize_time_step_status(status,front->pp_grid);
	if (status != GOOD_STEP) 
	{
	    *dt_frac = min(*dt_frac,Min_time_step_modification_factor(front));
	    free_front(newfront);
	    free_wave_pointers(outwave[0]);
	    if (wk_wv1 != NULL)
	    	free_wave(wk_wv1);
	    (void) printf("%s, init_hyp_solution_function2() failed\n",warn);
	    DEBUG_LEAVE(hyp_parab_mix_unsplit_consv_driver)
	    return status;
	}
	stop_clock("init_hyp_solution");
        
        /**** CONSTRUCT SPCACE-TIME INTERFACE and CONTROL VOLUME ****/
        /** THE FRACTIONAL VOLUME STATE STORAGE IS ALSO ALLOCATED HERE **/
        if(interface_reconstructed(front->interf) == YES &&
           interface_reconstructed(newfront->interf) == YES)
        {
            status = construct_hex_intfc(front,wave,newfront,outwave[0],dt); 
            if(first_consv_step == YES)
            {
                first_consv_step = NO; 
                // This is a tmp function, the state should not be set this way. 
                set_vol_frac_states(wave,front); 
                // For the debug purpose
                scatter_frac_cell_states(wave, front);
            }
        }
  
	/*
	*	Temporary storage allocated by store() in directional sweeps 
	*	should be allocated from the tmp_intfc table, since
	*	this storage is freed when tmp_intfc is deleted at
	*	the two sweeps.
        */
	current_intfc = current_interface();
	tmp_intfc = make_interface(wave->rect_grid->dim);

	start_clock("hyp_unsplit_reg_vec");

        hyp_unsplit_reg_vec(iperm,dh,dt,wave,outwave[0],
                     front,newfront,max_component(front->interf)); 
        wave_tri_soln(outwave[0])->tri_grid->Volume.blk = 
            wave_tri_soln(wave)->tri_grid->Volume.blk;
        //g_wave_conservation_check 
        if(debugging("check_mass_conservation"))
            (*wave->_wave_conservation_check)(outwave[0],front,dt);
        wave_tri_soln(outwave[0])->tri_grid->Volume.blk = NULL; 
        
	stop_clock("hyp_unsplit_reg_vec");

        if(parab_driver != NULL)
        {
            wave_tri_soln(outwave[0])->tri_grid->Volume.blk = 
            wave_tri_soln(wave)->tri_grid->Volume.blk;
            status = (*parab_driver)(dt,dt_frac,outwave[0],newfront); 
	    if (status != GOOD_STEP) 
	    {
	        *dt_frac = min(*dt_frac,Min_time_step_modification_factor(front));
	        free_front(newfront);
	        free_wave_pointers(outwave[0]);
	        if (wk_wv1 != NULL)
	        	free_wave(wk_wv1);
	        (void) printf("%s, parab_driver() failed\n",err);
                clean_up(ERROR); 
	        DEBUG_LEAVE(hyp_parab_mix_unsplit_consv_driver)
	        return status;
	    }
            wave_tri_soln(outwave[0])->tri_grid->Volume.blk = NULL; 
        }

        if(wave_tri_soln(wave)->tri_grid->Volume.blk != NULL)
            free_control_volume_storage(wave_tri_soln(wave)->tri_grid);

        if(wave_tri_soln(wave)->tri_grid->alloc.c_seg_crx_count == 1)
        {
            free_comp_tri_storage(wave_tri_soln(wave)->tri_grid);
            free_comp_crx_lists(wave_tri_soln(wave)->tri_grid);
        }

        /*************************************************************/
        /* NEED to remove blk flags (F_NO_VOL) in tri             ****/
        /*************************************************************/ 
        clear_tri_blk_type(outwave[0]); 

        free_wave_pointers(wave);
        if (! copy_hyp_solution_function(outwave[0],wave))
        {
            free_front(newfront);
            if (wk_wv1 != NULL)
                free_wave(wk_wv1);
            screen("%s, copy_hyp_solution_function() failed\n",err);
            DEBUG_LEAVE(hyp_parab_mix_unsplit_consv_driver)
            return ERROR_IN_STEP;
        }
        if(! copy_hyp_soln_state(outwave[0],wave))
        {
            screen("%s, copy_hyp_soln_state() failed\n",err);
            DEBUG_LEAVE(hyp_parab_mix_unsplit_consv_driver)
            return ERROR_IN_STEP;
        }
        
        if( debugging("hyp_states") )
        {
    	    (void) printf("After Unsplit Sweep\n");

            for (i = 0; i < dim; ++i)
	       	(void) printf("sweep %d Maxsp(wave)[%d] %g\n",i,
			      i,Maxsp(wave)[i]);
	    (void) printf("Wave %p after calling hyp_solver:\n",(POINTER)wave);
            if (wave->show_tri_soln)
	        (*wave->show_tri_soln)(newfront,wave);
	    (*wave->show_wave_states)(wave);
        }
	    
        if( debugging("special_plot") && wave->plot_hyp_soln)
            (*wave->plot_hyp_soln)(newfront,wave,step);

        set_current_interface(current_intfc);
        (void) delete_interface(tmp_intfc);

        /* Free temporary storage (outwave[0]), update front */

        if (tmpwave != NULL)
            free_copy_wave_pointers(tmpwave);

        assign_interface_and_free_front(front,newfront);

        if (wk_wv1 != NULL)
            free_wave(wk_wv1);

	DEBUG_LEAVE(hyp_parab_mix_unsplit_consv_driver)
	return status;
}
 
LOCAL void clear_tri_blk_type(
	Wave		*wave)
{
        int   i, xmax, ymax, blks;
        RECT_GRID  *gr;
        gr = &(wave_tri_soln(wave)->tri_grid->aug_comp_grid);
        xmax = gr->gmax[0];   ymax = gr->gmax[1];
        blks = xmax*ymax; 
        if(wave_tri_soln(wave)->tri_grid->alloc.blk_type == 1)
        {
            /*
            free(wave_tri_soln(wave)->tri_grid->blk_type);    
            VECTOR(wave_tri_soln(wave)->tri_grid,
                  blk_type,xmax*ymax,sizeof(int));
            */
            for(i = 0; i < blks; i++)
            {
                if(is_complex_blk(wave_tri_soln(wave)->tri_grid->blk_type[i]))
                    wave_tri_soln(wave)->tri_grid->blk_type[i] &= ~F_NO_VOL;   
            }
        }
}

LOCAL	int hyp_unsplit_consv_driver(
	float		dt,
	float		*dt_frac,
	Wave		*wave,
	Front		*front,
	void		(*sweep)(int,int*,float,float,
				 Wave*,Wave*,Front*,Front*,COMPONENT))
{
	Front		*newfront;
	Front		*infront[3], *outfront[3];
	INTERFACE	*current_intfc, *tmp_intfc;
	Wave		*wk_wv1 = NULL, *wk_wv2 = NULL;
	Wave		*tmpwave = NULL;
	Wave		*inwave[3],  *outwave[3];
	float		dh[MAXD];
	int		step = front->step;
	int		status;
	int		i, k, dim = front->interf->dim;
	int		*iperm;	/* permutation of {0,...,dim-1} */
	COMPONENT	max_comp;
	static char	warn[] = "WARNING in hyp_split_driver()";
	static char	err[] = "ERROR in hyp_split_driver()";

        static int      first_consv_step = YES;  

	set_hyp_npt_globals(wave);

		/* Advance Front */

        reset_pt_index_on_intfc(front->interf);

	start_clock("advance_front");
	status = advance_front(dt,dt_frac,front,&newfront,(POINTER) wave); 
	stop_clock("advance_front");

	if (debugging("hyp"))
	{
	    (void) printf("front->interf\n");
	    print_interface(front->interf);
	    (void) printf("newfront->interf\n");
	    print_interface(newfront->interf);
	}

	if (status != GOOD_STEP)
	{
	    (void) printf("%s advance_front() failed, "
	                  "dt_frac = %g\n",warn,*dt_frac);
	    print_time_step_status("time step status = ",status,"\n");
	    DEBUG_LEAVE(hyp_split_driver)
	    return status;
	}
	initialize_max_wave_speed(wave);

			/* hyperbolic solver */
	if( debugging("hyp_states") )
	{
	    (void) printf("States before calling hyp_solver:\n\n");
	    (void) printf("Old front\n");
	    graph_front_states(front);
	    (void) printf("New front\n");
	    graph_front_states(newfront);
	    (void) printf("wave %p\n",(POINTER)wave);
	    (*wave->show_wave_states)(wave);
	}

	iperm = set_iperm(step,dim);
	for (i = 0; i < dim; ++i)
	    dh[i] = wave->rect_grid->h[i];

		/* Initialize Intermediate Storage for States */

	wk_wv1 = copy_wave(wave);
	clear_wave_pointers(wk_wv1);
	switch (dim)
	{
	case 1:
	    inwave[0] = wave;	outwave[0] = wk_wv1;
	    infront[0] = front;	outfront[0] = newfront;
	    break;
	case 2:
	    inwave[0] = wave; outwave[0] = wk_wv1;
	    // inwave[1] = outwave[0]; outwave[1] = wave;

	    infront[0] = front;	outfront[0] = newfront;
	    // infront[1] = newfront; outfront[1] = newfront;
	    tmpwave = wk_wv1;
	    break;
	}

	start_clock("init_hyp_solution");
	assign_wave_parameters(outwave[0],wave);
        if(wave_tri_soln(wave)->new_tri_grid != NULL)
        {
	    status = init_hyp_solution_function2(
                          wave_tri_soln(wave)->new_tri_grid,
                          outwave[0],newfront);
            wave_tri_soln(wave)->new_tri_grid = NULL; 
        }
        else
        {
            printf("ERROR in hyp_unsplit_consv_driver\n");
            printf("Old wave's new_tri_grid is null\n");
            printf("New grid based interface is not constructed\n");
            clean_up(ERROR); 
        }
	status = syncronize_time_step_status(status,front->pp_grid);
	if (status != GOOD_STEP) 
	{
	    *dt_frac = min(*dt_frac,Min_time_step_modification_factor(front));
	    free_front(newfront);
	    free_wave_pointers(outwave[0]);
	    if (wk_wv1 != NULL)
	    	free_wave(wk_wv1);
	    if (wk_wv2 != NULL)
	    	free_wave(wk_wv2);
	    (void) printf("%s, init_hyp_solution_function2() failed\n",warn);
	    DEBUG_LEAVE(hyp_unsplit_consv_driver)
	    return status;
	}
        
        /* CONSTRUCT SPCACE-TIME INTERFACE and CONTROL VOLUME */
        /* THE FRACTIONAL VOLUME STATE STORAGE IS ALSO ALLOCATED HERE */
        if(interface_reconstructed(front->interf) == YES &&
           interface_reconstructed(newfront->interf) == YES)
        {
            // printf("step = %d, construct hex interface\n", front->step);  
            status = construct_hex_intfc(front,wave,newfront,outwave[0],dt); 
            if(first_consv_step == YES)
            {
                first_consv_step = NO; 
                // This is a tmp function, the state should not be set this way. 
                set_vol_frac_states(wave,front); 
            }
        }
  
	stop_clock("init_hyp_solution");

	start_clock("hyp_unsplit_consv_driver");

	/*
	*	Temporary storage allocated by store() in directional sweeps 
	*	should be allocated from the tmp_intfc table, since
	*	this storage is freed when tmp_intfc is deleted at
	*	the two sweeps.
        */
	current_intfc = current_interface();
	tmp_intfc = make_interface(wave->rect_grid->dim);

        hyp_unsplit_reg_vec(iperm,dh,dt,wave,outwave[0],
                     front,newfront,max_component(front->interf)); 

        if(wave_tri_soln(wave)->tri_grid->alloc.c_seg_crx_count == 1)
        {
            free_comp_tri_storage(wave_tri_soln(wave)->tri_grid);
            free_comp_crx_lists(wave_tri_soln(wave)->tri_grid);
        }
        /*
        else
            printf("Step[%d], tri_grid comp storage not freed\n", front->step);
        */

        free_wave_pointers(wave);
        if (! copy_hyp_solution_function(outwave[0],wave))
        {
            free_front(newfront);
            if (wk_wv1 != NULL)
                free_wave(wk_wv1);
            if (wk_wv2 != NULL)
                free_wave(wk_wv2);
            screen("%s, copy_hyp_solution_function() failed\n",err);
            DEBUG_LEAVE(hyp_unsplit_consv_driver)
            return ERROR_IN_STEP;
        }
        if(! copy_hyp_soln_state(outwave[0],wave))
        {
            screen("%s, copy_hyp_soln_state() failed\n",err);
            DEBUG_LEAVE(hyp_unsplit_consv_driver)
            return ERROR_IN_STEP;
        }
        /* If has parabolic step, need to save volume */
        
        if( debugging("hyp_states") )
        {
    	    (void) printf("After Unsplit Sweep\n");

	    for (i = 0; i < dim; ++i)
	    	(void) printf("sweep %d Maxsp(wave)[%d] %g\n",i,
			      i,Maxsp(wave)[i]);
	    (void) printf("Wave %p after calling hyp_solver:\n",(POINTER)wave);
	    if (wave->show_tri_soln)
	        (*wave->show_tri_soln)(newfront,wave);
	    (*wave->show_wave_states)(wave);
        }
	    
        if( debugging("special_plot") && wave->plot_hyp_soln)
    	    (*wave->plot_hyp_soln)(newfront,wave,step);

        /*
        if(wave_tri_soln(wave)->tri_grid->alloc.c_seg_crx_count == 1) 
        {
            // printf("Step[%d], DOES scatter_frac_cell_states\n", front->step);
            scatter_frac_cell_states(wave, newfront);
        }
        else
            printf("Step[%d], does not scatter_frac_cell_states\n", front->step);
        */

	set_current_interface(current_intfc);
	(void) delete_interface(tmp_intfc);

	/* Free temporary storage, update front */

	if (tmpwave != NULL)
	    free_copy_wave_pointers(tmpwave);

	assign_interface_and_free_front(front,newfront);

	if (wk_wv1 != NULL)
	    free_wave(wk_wv1);
	if (wk_wv2 != NULL)
	    free_wave(wk_wv2);

	stop_clock("hyp_unsplit_consv_driver");

	debug_print("hyp","Left hyp_unsplit_consv_driver()\n");
	DEBUG_LEAVE(hyp_unsplit_consv_driver)
	return status;
}		/*end hyp_unsplit_consv_driver*/


LOCAL   int     hyp_DG_driver(
        float           dt,
        float           *dt_frac,
        Wave            *wave,
        Front           *front)
{
        Front           *newfront;
        Front           *infront[3], *outfront[3];
        INTERFACE       *current_intfc, *tmp_intfc;
        Wave            *wk_wv1 = NULL, *wk_wv2 = NULL;
        Wave            *tmpwave = NULL;
	Wave            *inwave[3],  *outwave[3] = {NULL, NULL, NULL};
        float           dh[MAXD], *cent;
        int             step = front->step;
        int             status = GOOD_STEP;
        int             i, k, dim = front->interf->dim;
        int             *iperm; /* permutation of {0,...,dim-1} */
        COMPONENT       max_comp;
        static char     warn[] = "WARNING in hyp_DG_driver()";
        static char     err[] = "ERROR in hyp_DG_driver()";
        bool            sv_copy_st;
        TRI             *tri;
        SURFACE         **surf;


        DEBUG_ENTER(hyp_DG_driver)

        start_clock("advance_front");

        // printf("Enter hyp_DG_driver()\n");
        /**
        for(surf = front->mesh->surfaces; surf && *surf; surf++)
        {
            for (tri = first_tri(*surf);
                 !at_end_of_tri_list(tri,*surf); tri = tri->next)
            {
                // cent = fg_centroid(tri);
                // if(tri->id == 304)
                {
                    printf("IN Entering of hyp_DG_driver, TRI(%d) on front\n",
                        tri->id);
                    print_general_vector("Tri_pt", Coords(Point_of_tri(tri)[0]), dim, "\n");
                    print_general_vector("Tri_pt", Coords(Point_of_tri(tri)[1]), dim, "\n");
                    print_general_vector("Tri_pt", Coords(Point_of_tri(tri)[2]), dim, "\n");
                    (*front->print_state)(tri->st);
                }
            }
        }
        **/

        newfront = copy_front(front);
        set_size_of_intfc_state(size_of_state(front->interf));
        sv_copy_st = copy_intfc_states();
        current_intfc = current_interface();
        set_copy_intfc_states(YES);
        newfront->interf = copy_interface(front->interf);
        set_copy_intfc_states(sv_copy_st);
        set_current_interface(current_intfc);
        
        if(next_mesh == NULL)
            copy_mesh(front,newfront);
        else
            newfront->mesh = next_mesh;

        initialize_max_wave_speed(wave);

        iperm = set_iperm(step,dim);
        for (i = 0; i < dim; ++i)
            dh[i] = wave->rect_grid->h[i];

                /* Initialize Intermediate Storage for States */

        /**
        wk_wv1 = copy_wave(wave);
        clear_wave_pointers(wk_wv1);
        switch (dim)
        {
        case 1:
            inwave[0] = wave;   outwave[0] = wk_wv1;
            infront[0] = front; outfront[0] = newfront;
            break;
        case 2:
            inwave[0] = wave; outwave[0] = wk_wv1;
            // inwave[1] = outwave[0]; outwave[1] = wave;

            infront[0] = front; outfront[0] = newfront;
            // infront[1] = newfront; outfront[1] = newfront;
            tmpwave = wk_wv1;
            break;
        }

        start_clock("init_hyp_solution");
        assign_wave_parameters(outwave[0],wave);

        init_hyp_solution_function(outwave[0],newfront);

        stop_clock("init_hyp_solution");
        **/
 
        // gDG_tri_vec
        (*(wave->_DG_solver))(iperm,dh,dt,wave,outwave[0],
                        front,newfront,max_component(front->interf));

        set_current_interface(newfront->interf);

        // OLD
        /**
        // assign_interface_and_free_front(front,newfront);
        (void) delete_interface(front->interf);

        for(surf = front->mesh->surfaces; surf && *surf; surf++)
        {
            for (tri = first_tri(*surf);
                 !at_end_of_tri_list(tri,*surf); tri = tri->next)
            {
                free(tri->st);
            }
        }
        (void) delete_interface(front->mesh); 

        front->interf = newfront->interf; 
        front->mesh = newfront->mesh;
        newfront->interf = newfront->mesh = NULL;
        free_front(newfront);
        **/

        // NEW
        next_mesh = front->mesh;

        delete_interface(front->interf);
        front->mesh = NULL;
        front->interf = newfront->interf;
        front->mesh = newfront->mesh;
        newfront->interf = newfront->mesh = NULL;
        free_front(newfront);
        // END NEW
        

        /**
        if(front->step == 7)
        {
            printf("EXIT hyp_DG_driver, step %g\n", front->step);
            long_alloc_view(stdout);
            clean_up(0);
        }
        **/

        // printf("Exit hyp_DG_driver()\n");

        DEBUG_LEAVE(hyp_DG_driver)
        return status;
}

#endif /* if defined(CONSERVATIVE_ALG) */

