/*
*
*				s2phase.h:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*
*/

#if !defined(_S2PHASE_H)
#define _S2PHASE_H

#include <geos/geosdecs.h>

struct _Spline {
    float *x; /* Independent variable array, monotone nondecreasing */
    float *y;   /* Dependent variable array */
    int    m;    /* Number of data points */

    int    k;    /* Degree of spline polynomial */
    int    n;    /* Number of knots in the spline approximation */
    float *t;   /* Knot array */
    float *c;   /* B-spline coefficients */
    int   nest; /* Allocated length of the arrays t and c */

    float     atol;
    float     rtol;
    float     terr;
};
typedef struct _Spline Spline;


	/* Structures for accessing equation of state information */

struct _S2PHASE_EOS {
  EOS    eos;
  float	 a_sat_l;     /* sound speed squared of saturated liquid */ 
  float	 a_sat_v;     /* sound speed squared of saturated vapor */ 
  float	 p_sat_l;     /* pressure of saturated liquid */
  float  p_inf;       /* pressure P_infinity for liquid */ 
  float	 p_sat_v;     /* pressure of saturated vapor */
  float  p_vl;        /* mixed domain coefficient */
  float	 rho_sat_l;   /* density of saturated liquid */
  float	 rho_sat_v;   /* density of saturated vapor */
  float  e_sat_l;     /* specific internal energy of saturated liquid */
  float  e_sat_v;     /* specific internal energy of saturated vapor */
  float	 delta_e;     /* vapor - liquid energy jump (related to the heat of vaporization)*/
  float  e_inf;       /* energy E_infinity for liquid */ 
  float  S_0;         /* entropy (remains constant) */
  float  R_l;         /* gas constant of saturated liquid */
  float  R_v;         /* gas constant of saturated vapor */
  float	 t_sat_v;     /* temperature of saturated vapor */
  float	 t_sat_l;     /* temperature of saturated liquid */
  float  cv_l;        /* C_V of saturated liquid */
  float  gamma_l;     /* adiabatic exponent of liquid */
  float  gamma_v;     /* adiabatic exponent of vapor */
  float  eta_l;       /* eos coef. for liquid */
  float  eta_v;       /* eos coef. for vapor */
  
  /* The followings are from "Fluid Mechanics" by Landau and Lifshitz, page 45
   *
   * sigma_{ij} = -p delta_{ij} + 
   *             eta (e_{ij}+e_{ji})+(zeta-2/3*eta) (div v) delta_{ij}
   * e_{ij} = \partial v_i/\partial x_j
   * eta = dynamic viscosity = ordinary viscosity = shear viscosity
   * zeta = bulk viscosity = second viscosity = expansion viscosity = 
   *        dilatational viscosity
   *
   * Comments from "Physics of Shock Waves and High-Temperature 
   *        Hydrodynamic Phenomena"
   * by Zeldovich and Raizer, vol 1, page 73
   *
   * Sometimes, (zeta-2/3*eta) = lambda is called as second viscosity,
   * so lambda=-2/3*eta means zeta=0
   */

  float  shear_visc_l;    /* dynamic viscosity for liquid */
  float  shear_visc_v;    /* dynamic viscosity for vapor */
  float  bulk_visc_l;    /* bulk viscosity for liquid */
  float  bulk_visc_v;    /* bulk viscosity for vapor */

  Spline *int_c_drho_over_rho_mix_spline;
};
typedef struct _S2PHASE_EOS S2PHASE_EOS;

#define	S2PHASE_Eos(state)	((S2PHASE_EOS *)Params(state)->eos)

	/* Macros */

#define A_sat_l(state)		(S2PHASE_Eos(state)->a_sat_l)
#define A_sat_v(state)		(S2PHASE_Eos(state)->a_sat_v)
#define P_sat_l(state)		(S2PHASE_Eos(state)->p_sat_l)
#define P_inf(state)		(S2PHASE_Eos(state)->p_inf)
#define P_sat_v(state)		(S2PHASE_Eos(state)->p_sat_v)
#define P_vl(state)		(S2PHASE_Eos(state)->p_vl)
#define Rho_sat_l(state)	(S2PHASE_Eos(state)->rho_sat_l)
#define Rho_sat_v(state)	(S2PHASE_Eos(state)->rho_sat_v)
#define Delta_e(state)	        (S2PHASE_Eos(state)->delta_e)
#define E_inf(state)	        (S2PHASE_Eos(state)->e_inf)
#define E_sat_l(state)	        (S2PHASE_Eos(state)->e_sat_l)
#define E_sat_v(state)	        (S2PHASE_Eos(state)->e_sat_v)
#define S_0(state)	        (S2PHASE_Eos(state)->S_0)
#define R_l(state)            	(S2PHASE_Eos(state)->R_l)
#define R_v(state)            	(S2PHASE_Eos(state)->R_v)
#define Gamma_l(state)	        (S2PHASE_Eos(state)->gamma_l)
#define Gamma_v(state)	        (S2PHASE_Eos(state)->gamma_v)
#define Eta_l(state)	        (S2PHASE_Eos(state)->eta_l)
#define Eta_v(state)	        (S2PHASE_Eos(state)->eta_v)
#define T_sat_l(state)	        (S2PHASE_Eos(state)->t_sat_l)
#define T_sat_v(state)	        (S2PHASE_Eos(state)->t_sat_v)

#define Shear_visc_l(state)	        (S2PHASE_Eos(state)->shear_visc_l)
#define Shear_visc_v(state)	        (S2PHASE_Eos(state)->shear_visc_v)
#define Bulk_visc_l(state)	        (S2PHASE_Eos(state)->bulk_visc_l)
#define Bulk_visc_v(state)	        (S2PHASE_Eos(state)->bulk_visc_v)
#define cv_v(state)           		(S2PHASE_Eos(state)->cv_v)

#define Int_c_drho_over_rho_mix_spline(state)				\
	(S2PHASE_Eos(state)->int_c_drho_over_rho_mix_spline)

#endif /* !defined(_S2PHASE_H) */
