/*
*				fuserintfc.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*
*	User definable hooks to the interface libary.
*/




#include <front/fdecs.h>


	/* LOCAL Function Declarations */
LOCAL	Locstate f_read_print_state_data(INIT_DATA*,const IO_TYPE*,
                                         Locstate,INTERFACE*);
LOCAL	float	f_cross_tolerance(INTERFACE*);
LOCAL	void	f_fprint_intfc_state(FILE*,Locstate,INTERFACE*);
LOCAL	void	f_fprint_state_data(FILE*,Locstate,INTERFACE*);
LOCAL	void	f_reflect_state(Locstate,INTERFACE*,float*,float*,float*);
LOCAL	void	f_tangent(POINT*,BOND*,CURVE*,float*,Front*);
LOCAL	void	linear_state_interpolator(float,float,float*,Locstate,float*,
					  Locstate,RECT_GRID*,Locstate);
#if defined(TWOD) || defined(THREED)
LOCAL	bool	linear_tri_state_interpolator(float,float,float,float*,
					      Locstate,float*,Locstate,
					      float*,Locstate,RECT_GRID*,
					      Locstate);
#endif /* defined(TWOD) || defined(THREED) */
#if defined(ONED)
LOCAL	void	f_fshow_intfc_states1d(FILE*,INTERFACE*);
LOCAL	void	normal1d(POINT*,HYPER_SURF_ELEMENT*,HYPER_SURF*,float*,Front*);
LOCAL	void	slsr1d(POINT*,HYPER_SURF_ELEMENT*,HYPER_SURF*,
		       Locstate*,Locstate*);
LOCAL	void	state_at_point(COMPONENT,float*,HYPER_SURF_ELEMENT*,
			       HYPER_SURF*,Locstate);
#endif /* defined(ONED) */
#if defined(TWOD)
LOCAL   float   lagrangian_n_pt(int,float,float*,float*);
LOCAL	void	f_fshow_intfc_states2d(FILE*,INTERFACE*);
LOCAL	void	f_lagrangian_tangent(POINT*,BOND*,CURVE*,float*,Front*);
LOCAL	void	f_spline_tangent(POINT*,BOND*,CURVE*,float*,Front*);
LOCAL	void	normal2d(POINT*,HYPER_SURF_ELEMENT*,HYPER_SURF*,float*,Front*);
LOCAL	void	slsr2d(POINT*,HYPER_SURF_ELEMENT*,HYPER_SURF*,
		       Locstate*,Locstate*);
LOCAL	void	state_along_bond(COMPONENT,float*,HYPER_SURF_ELEMENT*,
				 HYPER_SURF*,Locstate);
#endif /* defined(TWOD) */
#if defined(THREED)
LOCAL	void	f_area_weighted_normal3d(POINT*,HYPER_SURF_ELEMENT*,HYPER_SURF*,
	                                 float*,Front*);
LOCAL	void	f_plane_fit_normal3d(POINT*,HYPER_SURF_ELEMENT*,HYPER_SURF*,
	                             float*,Front*);
LOCAL	bool	f_set_boundary3d(INTERFACE*,RECT_GRID*,COMPONENT,float);
LOCAL	void	f_fshow_intfc_states3d(FILE*,INTERFACE*);
LOCAL	void	slsr3d(POINT*,HYPER_SURF_ELEMENT*,HYPER_SURF*,
		       Locstate*,Locstate*);
LOCAL	void	state_in_tri(COMPONENT,float*,HYPER_SURF_ELEMENT*,
			     HYPER_SURF*,Locstate);
#endif /* defined(THREED) */


EXPORT	F_USER_INTERFACE *f_user_hook(
	int		dim)
{
	static F_USER_INTERFACE Fuser_hooks[3];
	static bool first = YES;

	if (first == YES)
	{
	    int i;
	    static F_INTERFACE_TOLERANCES Itol;

	    first = NO;

	    /* Set default values for F_INTERFACE_TOLERANCES */
	    Itol._DtReductionFac      = 0.8;

	    /* Set default values for Fuser_hooks fields*/

	    /* fields valid for all dimensions */

	    for (i = 0; i < 3; ++i)
	    {
	        zero_scalar(&Fuser_hooks[i]._computational_grid,
			    sizeof(RECT_GRID));
	        Fuser_hooks[i]._bstates = NULL;
	        Fuser_hooks[i]._num_bstates = 0;
	        Fuser_hooks[i]._first_node = 0;
	        Fuser_hooks[i]._last_node = 0;
	        Fuser_hooks[i]._sizest = 0;
	        Fuser_hooks[i]._interpolate_intfc_states = NO;
	        Fuser_hooks[i]._mono_comp_curves = NO;
	        Fuser_hooks[i]._fprint_wave_type = f_fprint_wave_type;
	        Fuser_hooks[i]._wave_type_as_string = f_wave_type_as_string;
	        Fuser_hooks[i]._read_wave_type_from_string =
		    f_read_wave_type_from_string;
	        Fuser_hooks[i]._bi_interpolate_intfc_states =
		    linear_state_interpolator;
	        Fuser_hooks[i]._fprint_state_data = f_fprint_state_data;
	        Fuser_hooks[i]._read_print_state_data = f_read_print_state_data;
	        Fuser_hooks[i]._nearest_intfc_state = f_nearest_intfc_state;
	        Fuser_hooks[i]._reflect_state = f_reflect_state;
	        Fuser_hooks[i]._fprint_intfc_state = f_fprint_intfc_state;
	        Fuser_hooks[i]._alloc_state = f_alloc_state;
	        Fuser_hooks[i]._alloc_intfc_state = f_alloc_intfc_state;
	        Fuser_hooks[i]._clear_state = f_clear_state;
	        Fuser_hooks[i]._obstacle_state = f_clear_state;
	        Fuser_hooks[i]._FInterfaceTolerances = Itol;
	        Fuser_hooks[i]._default_perform_redistribution_function =
		    f_perform_redistribution;
	        Fuser_hooks[i]._merge_hs_flags = f_merge_hs_flags;
	        Fuser_hooks[i]._interface_tangent_function._tangent = f_tangent;
	        Fuser_hooks[i]._interface_tangent_function._tangent_name =
		    strdup("f_tangent");
	        Fuser_hooks[i]._set_tangent_function = f_set_tangent_function;
	        Fuser_hooks[i]._set_normal_function = f_set_normal_function;
	        Fuser_hooks[i]._alloc_MaxFrontSpeed = f_alloc_MaxFrontSpeed;
	    }

#if defined(TWOD) || defined(THREED)
	    /* fields valid for both 2D and 3D */
	    for (i = 1; i < 3; ++i)
	    {
	        Fuser_hooks[i]._fprint_hsbdry_type = f_fprint_hsbdry_type;
	        Fuser_hooks[i]._read_hsbdry_type_from_string =
		    f_read_hsbdry_type_from_string;
	        Fuser_hooks[i]._tri_interpolate_intfc_states =
		    linear_tri_state_interpolator;
	        Fuser_hooks[i]._read_print_boundary_state_data =
		    f_read_print_boundary_state_data;
	    }
#endif /* defined(TWOD) || defined(THREED) */

	    /* Dimension specific fields */
#if defined(ONED)
	    Fuser_hooks[0]._fprint_hsbdry_type = NULL;
	    Fuser_hooks[0]._read_hsbdry_type_from_string = NULL;
	    Fuser_hooks[0]._slsr = slsr1d;
	    Fuser_hooks[0]._tri_interpolate_intfc_states = NULL;
	    Fuser_hooks[0]._state_along_hypersurface_element = state_at_point;
	    Fuser_hooks[0]._form_subintfc_via_communication =
		f_intfc_communication1d;
	    Fuser_hooks[0]._fshow_intfc_states = f_fshow_intfc_states1d;

	    Fuser_hooks[0]._mean_curvature_at_point = NULL;
	    Fuser_hooks[0]._interface_normal_function._normal = normal1d;
	    Fuser_hooks[0]._interface_normal_function._normal_name =
	        strdup("normal1d");
#endif /* defined(ONED) */

#if defined(TWOD)
	    Fuser_hooks[1]._slsr = slsr2d;
	    Fuser_hooks[1]._state_along_hypersurface_element = state_along_bond;
	    Fuser_hooks[1]._form_subintfc_via_communication =
		f_intfc_communication2d;

	    Fuser_hooks[1]._fshow_intfc_states = f_fshow_intfc_states2d;
	    Fuser_hooks[1]._mean_curvature_at_point =
		f_mean_curvature_at_point2d;
	    Fuser_hooks[1]._interface_normal_function._normal = normal2d;
	    Fuser_hooks[1]._interface_normal_function._normal_name =
	        strdup("normal2d");
#endif /* defined(TWOD) */

#if defined(THREED)
	    Fuser_hooks[2]._slsr = slsr3d;
	    Fuser_hooks[2]._state_along_hypersurface_element = state_in_tri;
	    Fuser_hooks[2]._form_subintfc_via_communication =
		f_intfc_communication3d;
	    Fuser_hooks[2]._fshow_intfc_states = f_fshow_intfc_states3d;
	    Fuser_hooks[2]._mean_curvature_at_point =
		f_mean_curvature_at_point3d;
	    Fuser_hooks[2]._interface_normal_function._normal =
	        f_area_weighted_normal3d;
	    Fuser_hooks[2]._interface_normal_function._normal_name =
	        strdup("f_area_weighted_normal3d");
#endif /* defined(THREED) */
	}
	if (dim < 1 || dim > 3)
	{
	    screen("ERROR in f_user_hook(), invalid dim %d\n",dim);
	    clean_up(ERROR);
	    return NULL;
	}
	else
	    return Fuser_hooks + dim - 1;
}		/*end f_user_hook*/

EXPORT	void	f_preserve_user_hooks(
	int                     dim,
	PRESERVE_USER_HOOKS	flag)
{
	F_USER_INTERFACE        *fuh;
	static F_USER_INTERFACE Sav_fuh;
	static F_USER_INTERFACE *sav_fuh = NULL;

	i_preserve_user_hooks(dim,flag);
	switch (flag)
	{
	case SAVE_HOOKS:
	    if (sav_fuh != NULL)
	    {
		screen("ERROR in f_preserve_user_hooks(), "
		       "attempt to save without prior restore\n");
		clean_up(ERROR);
	    }
	    fuh = f_user_hook(dim);
	    sav_fuh = &Sav_fuh;
	    *sav_fuh = *fuh;
	    break;
	case RESTORE_HOOKS:
	    if (sav_fuh == NULL)
	    {
		screen("ERROR in f_preserve_user_hooks(), "
		       "attempt to restore without prior save\n");
		clean_up(ERROR);
	    }
	    fuh = f_user_hook(dim);
	    *fuh = *sav_fuh;
	    sav_fuh = NULL;
	    break;
	}
}		/*end f_preserve_user_hooks*/

EXPORT	void f_set_interface_hooks(
	int		dim)
{
	I_USER_INTERFACE *iuh = i_user_hook(dim);
	F_USER_INTERFACE *fuh = f_user_hook(dim);
	int	         i;

	/* Front extended structure sizes */
	iuh->size_interface = sizeof(F_INTERFACE);
	iuh->size_point = sizeof(F_POINT);
	iuh->size_curve = sizeof(F_CURVE);
	iuh->size_node = sizeof(F_NODE);
	iuh->size_hyper_surf = sizeof(F_HYPER_SURF);
	iuh->size_hyper_surf_bdry = sizeof(F_HYPER_SURF_BDRY);
	switch (dim)
	{
#if defined(ONED)
	case 1:
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    iuh->size_bond_tri = sizeof(F_BOND_TRI);
	    break;
#endif /* defined(THREED) */
	}

	/* Front extended function pointers */
	iuh->_read_boundary_type_from_string = f_read_wave_type_from_string;
	iuh->_fprint_boundary_type = f_fprint_wave_type;
	iuh->_user_make_interface = f_user_make_interface;
	iuh->_copy_interface = f_copy_interface;
	iuh->_user_read_print_interface = f_user_read_print_interface;
	iuh->_user_fprint_interface = f_user_fprint_interface;
	iuh->_delete_interface = f_delete_interface;
	iuh->_user_fprint_intfc_rect_grids = f_user_fprint_intfc_rect_grids;
	iuh->_user_read_print_intfc_rect_grids =
	    f_user_read_print_intfc_rect_grids;
	iuh->_Point = f_Point;
	iuh->_Static_point = f_Static_point;
	iuh->_average_points = f_average_points;
	iuh->_copy_point = f_copy_point;
	iuh->_reconstruct_interface_pointers = f_reconstruct_interface_pointers;
	iuh->_fset_hyper_surf_color = f_fset_hyper_surf_color;
	iuh->_zoom_interface = f_zoom_interface;
	iuh->_reflect_point = f_reflect_point;
	iuh->_make_hypersurface = f_make_hypersurface;
	iuh->_user_copy_hyper_surf = f_user_copy_hyper_surf;
	iuh->_make_hypersurface_boundary = f_make_hypersurface_boundary;
#if defined(TWOD) || defined(THREED)
	iuh->_make_node = f_make_node;
	iuh->_copy_node = f_copy_node;
	iuh->_delete_node = f_delete_node;
	iuh->_user_fprint_node = f_user_fprint_node;
	iuh->_user_read_node = f_user_read_node;
	iuh->_user_read_print_node = f_user_read_print_node;
	iuh->_make_curve = f_make_curve;
	iuh->_copy_curve = f_copy_curve;
	iuh->_delete_curve = f_delete_curve;
	iuh->_user_fprint_curve = f_user_fprint_curve;
	iuh->_user_read_curve = f_user_read_curve;
	iuh->_user_read_print_curve = f_user_read_print_curve;
	iuh->_user_split_curve = f_user_split_curve;
	iuh->_user_join_curves = f_user_join_curves;
	iuh->_insert_point_in_bond = f_insert_point_in_bond;
	iuh->_delete_start_of_bond = f_delete_start_of_bond;
	iuh->_delete_end_of_bond = f_delete_end_of_bond;
	iuh->_reconstruct_point_pointers = f_reconstruct_point_pointers;
	iuh->_reconstruct_node_pointers = f_reconstruct_node_pointers;
	iuh->_reconstruct_bond_pointers = f_reconstruct_bond_pointers;
	iuh->_reconstruct_curve_pointers = f_reconstruct_curve_pointers;
	iuh->_invert_curve = f_invert_curve;
	iuh->_reverse_curve = f_reverse_curve;
	iuh->_is_subdomain_boundary = f_is_subdomain_boundary;
	iuh->_cross_tolerance = f_cross_tolerance;
#endif /* defined(TWOD) || defined(THREED) */
	switch (dim)
	{
#if defined(ONED)
	case 1:
	    iuh->_make_point = f_make_point;
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    iuh->_reflect_node = f_reflect_node2d;
	    iuh->_reflect_curve = f_reflect_curve2d;
	    iuh->_attach_curve_to_node = f_attach_curve_to_node;
	    iuh->_move_closed_loop_node = f_move_closed_loop_node;
	    iuh->_is_subdomain_node = f_is_subdomain_node;
	    iuh->_is_virtual_fixed_node = f_is_virtual_fixed_node;
            // NEW
	    iuh->_copy_surface = f_copy_surface;
	    iuh->_make_surface = f_make_surface;
            // END NEW
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    iuh->_reflect_surface = f_reflect_surface;
	    iuh->_CBond = f_CBond;
	    iuh->_insert_point_in_tri = f_insert_point_in_tri;
	    iuh->_insert_point_in_tri_side = f_insert_point_in_tri_side;
	    iuh->_link_tri_to_bond = f_link_tri_to_bond;
	    iuh->_reverse_bond = f_reverse_bond;
	    iuh->_reorder_curve_link_list = f_reorder_curve_link_list;
	    iuh->_join_surfaces = f_join_surfaces;
	    iuh->_make_surface = f_make_surface;
	    iuh->_copy_surface = f_copy_surface;
	    iuh->_delete_surface = f_delete_surface;
	    iuh->_user_fprint_surface = f_user_fprint_surface;
	    iuh->_user_read_surface = f_user_read_surface;
	    iuh->_user_read_print_surface = f_user_read_print_surface;
	    iuh->_user_install_faces = f_user_install_faces;
	    iuh->_assign_curve_boundary_flag = f_assign_curve_boundary_flag;
	    iuh->_set_boundary = f_set_boundary3d;
	    iuh->_gview_plot_interface = f_gview_plot_interface;
	    iuh->_consistent_interface = f_consistent_interface;
	    iuh->_sort_bond_tris = f_sort_bond_tris;
	    break;
#endif /* defined(THREED) */
	}

	/* Allocated boundary state array */
	fuh->_num_bstates = 6;
	vector(&fuh->_bstates,fuh->_num_bstates+1,sizeof(BOUNDARY_STATE*));
	++fuh->_bstates;
	for (i = -1; i < fuh->_num_bstates; ++i)
	    fuh->_bstates[i] = NULL;
}		/*end f_set_interface_hooks*/


/*
*			f_reflect_state():
*
*	This is a no operation function to be used as a front based
*	default for the operation reflect_state().  It is appropriate
*	for any physics whose states have no vector quantities.
*/

/*ARGSUSED*/
LOCAL	void	f_reflect_state(
	Locstate	state,	/* state being reflected */
	INTERFACE	*intfc,	/* interface of point */
	float		*pt,	/* position of state being reflected */
	float		*p,	/* point on reflection plane */
	float		*n)	/* normal to plane */
{
}		/*end f_reflect_state*/

/*
*			f_fprint_intfc_state():
*
*	Default function for simple Locstate printing.  Assumes that
*	Locstate represents an array of floating point numbers.
*/

/*ARGSUSED*/
LOCAL	void	f_fprint_intfc_state(
	FILE		*file,   /* output file */
	Locstate	state,	/* state being printed */
	INTERFACE	*intfc)	/* interface of state */
{
	float		*st = (float*)state;
	size_t		i, nfloats = size_of_state(intfc)/sizeof(float);

	(void) fprintf(file,"State %p =",(POINTER)state);
	for (i = 0; i < nfloats; ++i)
		(void) fprintf(file," %-14g",st[i]);
}		/*end f_fprint_intfc_states*/


/*
*		linear_state_interpolator():
*
*	Default version of the state interpolator. Uses component-wise
*	linear interpolation on the Locstate's treated as arrays of
*	floats.
*/

/*ARGSUSED*/
LOCAL void linear_state_interpolator(
	float		alpha,
	float		beta,
	float		*crds1,
	Locstate	st1,
	float		*crds2,
	Locstate	st2,
	RECT_GRID	*gr,
	Locstate	answer)
{
	float	  *s1 = (float*)st1, *s2 = (float*)st2;
	float	  *ans = (float*)answer;
	size_t	  i, nfloats;
	INTERFACE *cur_intfc;

	cur_intfc = current_interface();
	nfloats = size_of_state(cur_intfc)/sizeof(float);

	for (i = 0; i < nfloats; ++i)
		ans[i] = alpha * s1[i] +  beta * s2[i];
}		/*end linear_state_interpolator*/




LOCAL	void	f_fprint_state_data(
	FILE		*file,
	Locstate	state,
	INTERFACE	*intfc)
{
	float	*st = (float*)state;
	size_t	i, nfloats;
	size_t	sizest = size_of_state(intfc);

	(void) fprintf(file,"State information for state %p -",(POINTER)state);
	if (is_binary_output() == YES)
	{
	    (void) fprintf(file,"\f%c",1);
	    (void) fwrite((const void *) state,sizest,1,file);
	}
	else
	{
	    nfloats = sizest/sizeof(float);
	    for (i = 0; i < nfloats; ++i)
	    	(void) fprintf(file," %-"FFMT,st[i]);
	}
	(void) fprintf(file,"\n");
}		/*end f_fprint_state_data*/

/*ARGSUSED*/
LOCAL	Locstate f_read_print_state_data(
	INIT_DATA     *init,
	const IO_TYPE *io_type,
	Locstate      state,
	INTERFACE     *intfc)
{
	FILE	*file = io_type->file;
	float	*x = (float *) state;
	int	c;
	size_t	i, nfloats;
	size_t	sizest = size_of_state(intfc);

	if (state == NULL)
	    state = (Locstate) store(sizest);
	(void) fgetstring(file,"State information for state");
	(void) fscanf(file,"%*d%*s");
	nfloats = sizest/sizeof(float);
	if ((c = getc(file)) == '\f')
	{
	    (void) getc(file);
	    (void) read_binary_real_array(x,nfloats,io_type);
	}
	else
	{
	    (void) ungetc(c,file);
	    for (i = 0; i < nfloats; ++i)
	    {
	    	(void) fscan_float(file,x+i);
	    }
	}
	return state;
}		/*end f_read_print_state_data*/


/*
*			f_tangent():
*
*	Finds the unit tangent vector to the curve c at the point p
*	on bond b.  The vector t points in the direction of c,
*	from start towards end.  The algorithm is to compute an effective
*	secant vector joining the opposite endpoints of the bonds meeting
*	at the point p.
*/

/*ARGSUSED*/
LOCAL	void	f_tangent(
	POINT		*p,
	BOND		*b,
	CURVE		*c,
	float		*t,
	Front		*front)
{
	int		dim = front->interf->dim;
#if defined(TWOD) || defined(THREED)
	POINT		*p1, *p2;
	float		length;
	int		i;
	static	BOND  	*bdir1 = NULL, *bdir2 = NULL;
	/*NOTE: the following variables are newly added */
	BOND		*b1,*b2;	
	float		dp1,dp2,l1,l2,t1[MAXD],t2[MAXD];
#endif /* defined(TWOD) || defined(THREED) */

	if (dim == 1)
	{
	    t[0] = 1.0;
	    return;
	} 

#if defined(TWOD) || defined(THREED)
	if (bdir2 == NULL)
	{
	    scalar(&bdir1,sizeof(BOND));
	    bdir1->start = Static_point(front->interf);
	    bdir1->end = Static_point(front->interf);
	    scalar(&bdir2,sizeof(BOND));
	    bdir2->start = Static_point(front->interf);
	    bdir2->end = Static_point(front->interf);
	}
	if (is_bdry(c)) /*Rectangular boundary*/
	{
	    for (i = 0; i < dim; ++i)
	        t[i] = Coords(c->end->posn)[i] - Coords(c->start->posn)[i];
	    length = mag_vector(t,dim);
	    for (i = 0; i < dim; ++i)
	        t[i] /= length;
	    return;
	}
	if (p == c->end->posn)             /* End Node */
	{
	    if (is_closed_curve(c))
	    {
	        bond_tangent_to_curve(p,c->last,c,NEGATIVE_ORIENTATION,
				      bdir1,front);
	        p1 = bdir1->start;
		b1 = c->last;
	        bond_tangent_to_curve(p,c->first,c,POSITIVE_ORIENTATION,
				      bdir2,front);
	        p2 = bdir2->end;
		b2 = c->first;
	    }
	    else
	    {
	        find_tangent_to_curve(p,c->last,c,NEGATIVE_ORIENTATION,
	                              t,front);
	        for (i = 0; i < dim; ++i)
	            t[i] = -t[i];
#if defined(TWOD)
	        if ((dim == 2)
		    &&
		    (wave_type(c) >= FIRST_PHYSICS_WAVE_TYPE)
	            &&
		    (is_bdry_like_node(c->end) || is_fixed_node(c->end))
	            &&
		    (front->impose_bc != NULL))
	        {
	            (*front->impose_bc)(p,b,c,t,front,YES,YES);
	        }
#endif /* defined(TWOD) */
	        return;
	    }
	}
	else if (p == c->start->posn)             /* Start Node */
	{
	    if (is_closed_curve(c))
	    {
	        bond_tangent_to_curve(p,c->last,c,NEGATIVE_ORIENTATION,
	                              bdir1,front);
	        p1 = bdir1->start;
		b1 = c->last;
	        bond_tangent_to_curve(p,c->first,c,POSITIVE_ORIENTATION,
	                              bdir2,front);
	        p2 = bdir2->end;
		b2 = c->first;
	    }
	    else
	    {
	        find_tangent_to_curve(p,c->first,c,POSITIVE_ORIENTATION,
	        		      t,front);
#if defined(TWOD)
	        if ((dim == 2)
		    &&
		    (wave_type(c) >= FIRST_PHYSICS_WAVE_TYPE)
		    &&
		    (is_bdry_like_node(c->start) || is_fixed_node(c->start))
		    &&
		    (front->impose_bc != NULL))
	        {
	            (*front->impose_bc)(p,b,c,t,front,YES,YES);
	        }
#endif /* defined(TWOD) */
	        return;
	    }
	}
	else if (p == b->end)
	{
	    bond_tangent_to_curve(p,b,c,NEGATIVE_ORIENTATION,bdir1,front);
	    p1 = bdir1->start;
	    b1 = b;
	    bond_tangent_to_curve(p,b->next,c,POSITIVE_ORIENTATION,bdir2,front);
	    p2 = bdir2->end;
	    b2 = b->next;
	}
	else if (p == b->start)
	{
	    bond_tangent_to_curve(p,b->prev,c,NEGATIVE_ORIENTATION,bdir1,front);
	    p1 = bdir1->start;
	    b1 = b->prev;
	    bond_tangent_to_curve(p,b,c,POSITIVE_ORIENTATION,bdir2,front);
	    p2 = bdir2->end;
	    b2 = b;
	}
	else
	{
	    float t0[3], t1[3];
	    float dp[3], db[3];
	    float alpha;

	    /*
	     * Calling routine has faked a point. Define the tangent field
	     * along the bond so that the tangent varies continuously from the
	     * bond start to bond end.
	     */

	    tangent(b->start,b,c,t0,front);
	    tangent(b->end,b,c,t1,front);
	    p1 = b->start;
	    p2 = b->end;
	    for (i = 0; i < dim; ++i)
	    {
	        db[i] = Coords(p2)[i] - Coords(p1)[i];
	        dp[i] = Coords(p)[i] - Coords(p1)[i];
	    }
	    length = mag_vector(db,dim);
	    alpha = scalar_product(dp,db,dim)/sqr(length);
	    alpha = max(0.0,alpha);
	    alpha = min(1.0,alpha);
	    for (i = 0; i < dim; ++i)
	        t[i] = (1.0 - alpha)*t0[i] + alpha*t1[i];
	    length = mag_vector(t,dim);
	    for (i = 0; i < dim; ++i)
	        t[i] /= length;
	    return;
	}

	for (i = 0; i < dim; ++i)
	    t[i] = Coords(p2)[i] - Coords(p1)[i];
	/*NOTE: subject to critique */
	dp1 = (Coords(p)[0] - Coords(b1->start)[0])*t[0] +
	      (Coords(p)[1] - Coords(b1->start)[1])*t[1];
	dp2 = (Coords(b2->end)[0] - Coords(p)[0])*t[0] +
	      (Coords(b2->end)[1] - Coords(p)[1])*t[1];
	if (dp1 < 0.0 || dp2 < 0.0)
	{
	    for (i = 0; i < dim; i++)
	    {
		t1[i] = Coords(b2->end)[i] - Coords(p)[i];
		t2[i] = Coords(p)[i] - Coords(b1->start)[i];
	    }
	    l1 = mag_vector(t1,dim);
	    l2 = mag_vector(t2,dim);
	    for (i = 0; i < dim; i++)
		    t[i] = t1[i]/l1 + t2[i]/l2;
	}
	/*NOTE: end of newly added */
	length = mag_vector(t,dim);
	/*On very small, closed loops it can happen that p1 and p2 are
	          at the same location.*/
	if (length == 0.0)
	{
	    p1 = b->start;          p2 = b->end;
	    for (i = 0; i < dim; ++i)
	        t[i] = Coords(p2)[i] - Coords(p1)[i];
	    length = mag_vector(t,dim);
	}
	for (i = 0; i < dim; ++i)
	    t[i] /= length;
#endif /* defined(TWOD) || defined(THREED) */
}		/*end f_tangent*/
	
LOCAL	float	f_cross_tolerance(
	INTERFACE *intfc)
{
	float *h = computational_grid(intfc)->h;
	float hmin;
	hmin = min(h[0],h[1]);
	hmin = min(hmin,h[2]);
	return MIN_SIN_SQR(intfc)*hmin;/*TOLERANCE*/
}		/*end f_cross_tolerance*/

#if defined(TWOD) || defined(THREED)
/*
*		linear_tri_state_interpolator():
*
*	Default version of the tri state interpolator.
*	Uses component-wise linear interpolation on the Locstate's
*	treated as arrays of floats.
*/

/*ARGSUSED*/
LOCAL bool linear_tri_state_interpolator(
	float		alpha,
	float		beta,
	float		gamma,
	float		*crds0,
	Locstate	st0,
	float		*crds1,
	Locstate	st1,
	float		*crds2,
	Locstate	st2,
	RECT_GRID	*gr,
	Locstate	answer)
{
	float	  *s0 = (float*)st0, *s1 = (float*)st1, *s2 = (float*)st2;
	float	  *ans = (float*)answer;
	size_t	  i, nfloats;
	INTERFACE *cur_intfc;

	cur_intfc = current_interface();
	nfloats = size_of_state(cur_intfc)/sizeof(float);

	for (i = 0; i < nfloats; ++i)
		ans[i] = alpha * s0[i] + beta * s1[i] + gamma * s2[i];
	return FUNCTION_SUCCEEDED;
}		/*end linear_tri_state_interpolator*/
#endif /* defined(TWOD) || defined(THREED) */


#if defined(ONED)
/*
*		f_fshow_intfc_states1d():
*
*	Prints states on an interface using print_intfc_state().
*/


/*ARGSUSED*/
LOCAL	void f_fshow_intfc_states1d(
	FILE		*file,
	INTERFACE	*intfc)
{
	POINT	**p;

	(void) fprintf(file,"\t\tSTATES ON THE FRONT\n\n");
	if (size_of_state(intfc) > 0)
	{
		for (p = intfc->points; p && *p;  ++p)
		{
			(void) fprintf(file,"\t\tSTATES AT POINT %g\n\n",
				Coords(*p)[0]);
			(void) fprintf(file,"\t\tl_st ");
			fprint_intfc_state(file,left_state(*p),intfc);
			(void) fprintf(file,"\t\tr_st ");
			fprint_intfc_state(file,right_state(*p),intfc);
		}
		
	}
	else
	{
		(void) fprintf(file,"No states assigned on intfc\n");
		return;
	}
	(void) fprintf(file,"\n\n");
	(void) fprintf(file,"\t\tEND OF STATES ON THE FRONT\n\n");
}		/*end f_fshow_intfc_states1d*/

/*ARGSUSED*/
LOCAL	void	slsr1d(
	POINT		*p,
	HYPER_SURF_ELEMENT* hse,
	HYPER_SURF	*hs,
	Locstate	*sl,
	Locstate	*sr)
{
	*sl = left_state(p);
	*sr = right_state(p);
}		/*end slsr1d*/

/*
*			state_at_point():
*
*	Returns the state matching component at a point on a 1d interface.
*/

/*ARGSUSED*/
LOCAL	void state_at_point(
	COMPONENT	   comp,
	float		   *t,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	   *hs,
	Locstate	   state)
{
	POINT	*p = Point_of_hs(hs);
	size_t	sizest = f_user_interface(hs->interface)._sizest;

	if (comp == negative_component(p))
	{
	    assign(state,left_state(p),sizest);
	}
	else if (comp == positive_component(p))
	{
	    assign(state,right_state(p),sizest);
	}
	else
	{
	    screen("ERROR in state_at_point(), "
	           "comp = %d not on point %d\n",comp,p);
	    (void) printf("Point p\n");
	    print_point(p);
	    (void) printf("Interface of p\n");
	    print_interface(hs->interface);
	    clean_up(ERROR);
	}
}		/*end state_at_point*/

/*
*				normal1d():
*
*	Returns the Normal to the hypersurface hs at the point p on
*	hypersurface element hse of CURVE c.  The normal points from the
*	negative side to the positive side of the hypersurface.  In one
*	space dimensions this means it points from the LEFT side TO the
*	RIGHT side.
*/

/*ARGSUSED*/
EXPORT	void normal1d(
	POINT		*p,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	*hs,
	float		*nor,
	Front		*front)
{
	nor[0] = 1.0;
}		/*end normal1d*/
#endif /* defined(ONED) */


#if defined(TWOD)

EXPORT	void	set_tangent_operator(
	TANGENT_METHOD tan_meth,
	int            dim)
{
	F_USER_INTERFACE *fuh = f_user_hook(dim);
	switch (tan_meth)
	{
	case LANGRANGIAN_INTERPOLANT:
	    fuh->_interface_tangent_function._tangent = f_lagrangian_tangent;
	    fuh->_interface_tangent_function._tangent_name =
	        strdup("f_lagrangian_tangent");
	    break;
	case CUBIC_SPLINE:
	    fuh->_interface_tangent_function._tangent = f_spline_tangent;
	    fuh->_interface_tangent_function._tangent_name =
	        strdup("f_spline_tangent");
	    break;
	case TANGENT_METHOD_FROM_RESTART:
	    break;
	case LINEAR_SECANT:
	default:
	    fuh->_interface_tangent_function._tangent = f_tangent;
	    fuh->_interface_tangent_function._tangent_name =
	        strdup("f_tangent");
	    break;
	}
}		/*end set_tangent_operator*/

/*ARGSUSED*/
EXPORT	void	f_set_normal_function(
	const char      *s,
	NORMAL_FUNCTION *nf,
	INTERFACE       *intfc)
{
	if (strcmp(s,"none") == 0)
	{
	    nf->_normal = NULL;
	    nf->_normal_name = NULL;
	}
#if defined(ONED)
	else if (strstr(s,"normal1d"))
	{
	    nf->_normal = normal1d;
	    nf->_normal_name = strdup("normal1d");
	}
#endif /* defined(ONED) */
#if defined(TWOD)
	else if (strstr(s,"normal2d"))
	{
	    nf->_normal = normal2d;
	    nf->_normal_name = strdup("normal2d");
	}
#endif /* defined(TWOD) */
#if defined(THREED)
	else if (strstr(s,"area_weighted_normal3d"))
	{
	    nf->_normal = f_area_weighted_normal3d;
	    nf->_normal_name = strdup("f_area_weighted_normal3d");
	}
	else if (strstr(s,"plane_fit_normal3d"))
	{
	    nf->_normal = f_plane_fit_normal3d;
	    nf->_normal_name = strdup("f_plane_fit_normal3d");
	}
#endif /* defined(THREED) */
	else
	{
	    screen("ERROR in f_set_normal_function(), unknown normal function "
		   "%s\n",s);
	    clean_up(ERROR);
	}
}		/*end f_set_normal_function*/

/*ARGSUSED*/
EXPORT	void	f_set_tangent_function(
	const char       *s,
	TANGENT_FUNCTION *tf,
	INTERFACE        *intfc)
{
	if (strcmp(s,"none") == 0)
	{
	    tf->_tangent = NULL;
	    tf->_tangent_name = NULL;
	}
#if defined(TWOD)
	else if (strcmp(s,"f_lagrangian_tangent") == 0)
	{
	    tf->_tangent = f_lagrangian_tangent;
	    tf->_tangent_name = strdup("f_lagrangian_tangent");
	}
	else if (strcmp(s,"f_spline_tangent") == 0)
	{
	    tf->_tangent = f_spline_tangent;
	    tf->_tangent_name = strdup("f_spline_tangent");
	}
#endif /* defined(TWOD) */
	else if (strcmp(s,"f_tangent") == 0)
	{
	    tf->_tangent = f_tangent;
	    tf->_tangent_name = strdup("f_tangent");
	}
}		/*end f_set_tangent_function*/

/*
*			f_lagrangian_tangent():
*
*	Finds unit the tangent vector to the curve c at the point p
*	on bond b.  The vector t points in the direction of c,
*	from start towards end.  The algorithm is to fit a fourth order
*	lagrangian interpolation polynomial to five points centered at p.
*	The tangent is defined as the unit tangent vector of this interpolant
*	at p.
*/

/*ARGSUSED*/
LOCAL	void	f_lagrangian_tangent(
	POINT		*p,
	BOND		*b,
	CURVE		*c,
	float		*t,
	Front		*front)
{
	int             dim = front->interf->dim;
	BOND		*bonds[4];
	float		length;
	float		s[4];
	int		is,ie;
	int             i, j;
	static	float	**tt = NULL;

	if (tt == NULL)
	{
	    matrix(&tt,MAXD,4,FLOAT);
	}
	if (is_bdry(c)) /*Rectangular boundary*/
	{
	    for (i = 0; i < dim; ++i)
	        t[i] = Coords(c->end->posn)[i] - Coords(c->start->posn)[i];
	    length = mag_vector(t,dim);
	    for (i = 0; i < dim; ++i)
	        t[i] /= length;
	    return;
	}
	is = 0;		ie = 4;
	if (p == b->start)
	{
	    if (b != c->first)
	    {
	        bonds[1] = b->prev;
	        s[1] = -0.5*bond_length(bonds[1]);
	        if (bonds[1] != c->first)
	        {
	            bonds[0] = bonds[1]->prev;
	            s[0] = -bond_length(bonds[1]) - 0.5*bond_length(bonds[0]);
	        }
	        else
	        {
	            bonds[0] = NULL;
	            is = 1;
	        }
	    }
	    else if (is_closed_curve(c))
	    {
	        bonds[1] = c->last;
	        s[1] = -0.5*bond_length(bonds[1]);
	        if (bonds[1] != c->first)
	        {
	            bonds[0] = bonds[1]->prev;
	            s[0] = -bond_length(bonds[1]) - 0.5*bond_length(bonds[0]);
	        }
	        else
	        {
	            bonds[0] = NULL;
	            is = 1;
	        }
	    }
	    else
	    {
	        bonds[0] = bonds[1] = NULL;
	        is = 2;
	    }
	    bonds[2] = b;
	    s[2] = 0.5*bond_length(bonds[2]);
	    if (b != c->last)
	    {
	        bonds[3] = b->next;
	        s[3] = bond_length(bonds[2]) + 0.5*bond_length(bonds[3]);
	    }
	    else if (is_closed_curve(c))
	    {
	        bonds[3] = c->first;
	        s[3] = bond_length(bonds[2]) + 0.5*bond_length(bonds[3]);
	    }
	    else 
	    {
	        bonds[3] = NULL;
	        ie = 3;
	    }
	}
	else if (p == b->end)
	{
	    bonds[1] = b;
	    s[1] = -0.5*bond_length(bonds[1]);
	    if (b != c->first)
	    {
	        bonds[0] = b->prev;
	        s[0] = -bond_length(bonds[1]) - 0.5*bond_length(bonds[0]);
	    }
	    else if (is_closed_curve(c))
	    {
	        bonds[0] = c->last;
	        s[0] = -bond_length(bonds[1]) - 0.5*bond_length(bonds[0]);
	    }
	    else 
	    {
	        bonds[0] = NULL;
	        is = 1;
	    }
	    if (b != c->last)
	    {
	        bonds[2] = b->next;
	    	s[2] = 0.5*bond_length(bonds[2]);
	        if (bonds[2] != c->last)
	        {
	            bonds[3] = bonds[2]->next;
	            s[3] = bond_length(bonds[2]) + 0.5*bond_length(bonds[3]);
	        }
	        else if (is_closed_curve(c))
	        {
	            bonds[3] = c->first;
	            s[3] = bond_length(bonds[2]) + 0.5*bond_length(bonds[3]);
	        }
	        else
	        {
	            bonds[3] = NULL;
	            ie = 3;
	        }
	    }
	    else if (is_closed_curve(c))
	    {
	        bonds[2] = c->first;
	    	s[2] = 0.5*bond_length(bonds[2]);
	        if (bonds[2] != c->last)
	        {
	            bonds[3] = bonds[2]->next;
	            s[3] = bond_length(bonds[2]) + 0.5*bond_length(bonds[3]);
	        }
	        else
	        {
	            bonds[3] = NULL;
	            ie = 3;
	        }
	    }
	    else
	    {
	        bonds[2] = bonds[3] = NULL;
	        ie = 2;
	    }
	}
	for (i = 0; i < 4; ++i)
	{
	    if (bonds[i] == NULL)
	        continue;
	    else
	    {
	        for (j = 0; j < dim; ++j)
	        {
	            tt[j][i] = (Coords(bonds[i]->start)[j] - 
	        	   Coords(bonds[i]->end)[j])/bond_length(bonds[i]);
	        }
	    }
	}
	for (i = 0; i < dim; ++i)
	{
	    t[i] = lagrangian_n_pt(ie-is,0.0,s+is,tt[i]+is);
	}
	length = mag_vector(t,dim);
	for (i = 0; i < dim; ++i)
	    t[i] /= length;
	if (debugging("tangent"))
	{
	    float t1[3];
	    float x, y, v[3];
	    f_tangent(p,b,c,t1,front);
	    x = scalar_product(t,t1,dim);
	    y = vector_product(t,t1,v,dim); 
	    (void) printf("angle between f_tangent and "
	        	  "f_lagrangian_tangent = %g degrees, sin = %g\n",
	        	  degrees(atan2(y,x)),y);
	}
}		/*end f_lagrangian_tangent*/

LOCAL   float lagrangian_n_pt(
        int n,
        float xx,
        float *x,
        float *f)
{
        int i, j;
        float dd, soln;

        soln = 0.0;
        for (i = 0; i < n; ++i)
        {
            dd = 1.0;
            for (j = 0; j < n; ++j)
            {
                if (j == i)
	            continue;
                dd *= (xx - x[j])/(x[i] - x[j]);
            }
            soln += f[i]*dd;
        }
        return -soln;
}       /* end lagrangian_n_pt */

/*
*				f_spline_tangent():
*
*	Finds unit the tangent vector to the curve c at the point p
*	on bond b.  The vector t points in the direction of c,
*	from start towards end.  The algorithm is to fit a cubic spline
*	interpolant through the point and adjacent points on the curve.
*	The end point conditions are that the spline terms are quadratic
*	at the two extreme sections.
*/

/*ARGSUSED*/
LOCAL	void	f_spline_tangent(
	POINT		*point,
	BOND		*bond,
	CURVE		*curve,
	float		*tangent,
	Front		*front)
{
	INTERFACE        *intfc = curve->interface;
	int		 dim = intfc->dim;
	BOND		 *bb;
	static const int nrad = 2;/*TOLERANCE*/
	float		 length;
	int		 n, m, N, nf, nb, i, j;
	int		 info;
	static float	 *c = NULL, *d = NULL, *e = NULL, *b = NULL;
	static float	 *ds = NULL, **q = NULL;

	if (is_bdry(curve)) /*Rectangular boundary*/
	{
	    for (i = 0; i < dim; ++i)
	    {
	        tangent[i] = Coords(curve->end->posn)[i] -
	        	     Coords(curve->start->posn)[i];
	    }
	    length = mag_vector(tangent,dim);
	    for (i = 0; i < dim; ++i)
	        tangent[i] /= length;
	    return;
	}

	/* If point is a node position revert to f_tangent*/
	/* TODO: closed nodes correctly */
	if ((point == curve->start->posn) || (point == curve->end->posn))
	{
	    f_tangent(point,bond,curve,tangent,front);
	    return;
	}
	if ((point != bond->start) && (point != bond->end))
	{
	    f_tangent(point,bond,curve,tangent,front);
	    return;
	}

	if (c == NULL)
	{
	    vector(&c,(2*nrad+1)*3,FLOAT);
	    vector(&d,(2*nrad+1)*3,FLOAT);
	    vector(&e,(2*nrad+1)*3,FLOAT);
	    vector(&b,(2*nrad+1)*3,FLOAT);
	    vector(&ds,2*nrad,FLOAT);
	    matrix(&q,2*nrad+1,3,FLOAT);
	}

	/*Eliminate degenerate case of point adjacent to node*/
	if ((point == curve->first->end) || (point == curve->last->start))
	{
	    if (debugging("tangent"))
	        (void) printf("Computing tangent adjacent to node\n");
	    if (point == curve->first->end)
	    {
	        for (i = 0; i < dim; ++i)
	        {
	            q[0][i] = Coords(curve->first->start)[i];
	            q[1][i] = Coords(curve->first->end)[i];
	            q[2][i] = Coords(curve->first->next->end)[i];
	        }
	    }
	    if (point == curve->last->start)
	    {
	        for (i = 0; i < dim; ++i)
	        {
	            q[0][i] = Coords(curve->last->prev->start)[i];
	            q[1][i] = Coords(curve->last->start)[i];
	            q[2][i] = Coords(curve->last->end)[i];
	        }
	    }
	    ds[0] = distance_between_positions(q[1],q[0],dim);
	    ds[1] = distance_between_positions(q[2],q[1],dim);
	    length = ds[0] + ds[1];
	    ds[0] /= length;
	    ds[1] /= length;
	    for (i = 0; i < dim; ++i)
	        tangent[i] = ds[1]*(q[1][i]-q[0][i])/ds[0] +
	        	     ds[0]*(q[2][i]-q[1][i])/ds[1];
	    length = mag_vector(tangent,dim);
	    for (i = 0; i < dim; ++i)
	        tangent[i] /= length;
	}
	else
	{

	    /*Set up interpolation points*/
	    for (i = 0; i < dim; ++i)
	        q[nrad][i] = Coords(point)[i];
	    if (point == bond->start)
	    {
	        for (nf=0,bb=bond; (nf<nrad) && (bb!=NULL); ++nf,bb=bb->next)
	        {
	            for (i = 0; i < dim; ++i)
	                q[nrad+nf+1][i] = Coords(bb->end)[i];
	        }
	        for (nb=0,bb=bond->prev;(nb<nrad)&&(bb!=NULL);++nb,bb=bb->prev)
	        {
	            for (i = 0; i < dim; ++i)
	                q[nrad-nb-1][i] = Coords(bb->start)[i];
	        }
	    }
	    if (point == bond->end)
	    {
	        for (nf=0,bb=bond->next;(nf<nrad)&&(bb!=NULL);++nf,bb=bb->next)
	        {
	            for (i = 0; i < dim; ++i)
	                q[nrad+nf+1][i] = Coords(bb->end)[i];
	        }
	        for (nb=0, bb=bond; (nb<nrad) && (bb!=NULL); ++nb, bb=bb->prev)
	        {
	            for (i = 0; i < dim; ++i)
	                q[nrad-nb-1][i] = Coords(bb->start)[i];
	        }
	    }
	    n = min(nf,nb);
	    m = 2*n;
	    if (n < nrad) /*Fewer than nrad points before node is encountered*/
	    {
	        /*
	         * Shift the q matrix so that the interpolation points are
	         * indexed from 0
	         */
	        for (i = 0; i <= m; ++i)
	        {
	            for (j = 0; j < dim; ++j)
	                q[i][j] = q[nrad-n+i][j];
	        }
	    }

	    /*Set up tridiagonal system for spline computation*/
	    for (length = 0, i = 0; i < m; ++i)
	    {
	        ds[i] = distance_between_positions(q[i+1],q[i],dim);
	        length += ds[i];
	    }
	    for (i = 0; i < m; ++i)
	        ds[i] /= length;

	    c[0] = 0;
	    d[0] = e[0] = 0.5*ds[0];
	    for (i = 1; i < m; ++i)
	    {
	        c[i] = ds[i];
	        d[i] = 2.0*(ds[i-1]+ds[i]);
	        e[i] = ds[i-1];
	    }
	    c[m] = d[m] = 0.5*ds[m-1];
	    e[m] = 0.0;
	    for (i = 1; i < dim; ++i)
	    {
	        for (j = 0; j <= m; ++j)
	        {
	            c[j+(m+1)*i] = c[j];
	            d[j+(m+1)*i] = d[j];
	            e[j+(m+1)*i] = e[j];
	        }
	    }
	    for (i = 0; i < dim; ++i)
	    {
	        b[(m+1)*i] = q[1][i] - q[0][i];
	        for (j = 1; j < m; ++j)
	        {
	            b[j+(m+1)*i] = 3.0*(ds[j-1]*(q[j+1][i]-q[j  ][i])/ds[j  ] +
	        		        ds[j  ]*(q[j  ][i]-q[j-1][i])/ds[j-1]);
	        }
	        b[m+(m+1)*i] = q[m][i] - q[m-1][i];
	    }
	    N = (m+1)*dim;
	    /*
	    gtsl(&N,c,d,e,b,&info);//Solve tridiagonal spline linear system
	    */
            printf("ERROR: f_spline_tangent(), gtsl disabled\n");
	    clean_up(ERROR);
	    if (info != 0)
	    {
	        void (*sav_tan)(POINT*,BOND*,CURVE*,float*,Front*);
		const char *sav_tan_name;
	        (void) printf("WARNING in f_spline_tangent(), gtsl failed, "
			      "info = %d\n",info);
		sav_tan = interface_tangent(intfc);
		sav_tan_name = interface_tangent_name(intfc);
		interface_tangent(intfc) = f_tangent;
		interface_tangent_name(intfc) = "f_tangent";
	        f_tangent(point,bond,curve,tangent,front);
		interface_tangent(intfc) = sav_tan;
		interface_tangent_name(intfc) = sav_tan_name;
	        return;
	    }
	    for (i = 0; i < dim; ++i)
	        tangent[i] = b[n+(m+1)*i];
	    length = mag_vector(tangent,dim);
	    for (i = 0; i < dim; ++i)
	        tangent[i] /= length;
	}
	if (debugging("tangent"))
	{
	    void (*sav_tan)(POINT*,BOND*,CURVE*,float*,Front*);
	    const char *sav_tan_name;
	    float t1[3];
	    float x, y, v[3];
	    sav_tan = interface_tangent(intfc);
	    sav_tan_name = interface_tangent_name(intfc);
	    interface_tangent(intfc) = f_tangent;
	    interface_tangent_name(intfc) = "f_tangent";
	    f_tangent(point,bond,curve,t1,front);
	    interface_tangent(intfc) = sav_tan;
	    interface_tangent_name(intfc) = sav_tan_name;
	    x = scalar_product(tangent,t1,dim);
	    y = vector_product(tangent,t1,v,dim); 
	    print_general_vector("At ",Coords(point),dim," ");
	    (void) printf("a(f_spline_tangent,f_tangent) = "
	        	  "%g degrees, sin = %g\n",
	        	  degrees(atan2(y,x)),y);
	}
}		/*end f_spline_tangent*/

/*
*		f_fshow_intfc_states2d():
*
*	Prints states on an interface using print_intfc_state().
*/

LOCAL	void f_fshow_intfc_states2d(
	FILE		*file,
	INTERFACE	*intfc)
{
	CURVE		**c;

	(void) fprintf(file,"\t\tSTATES ON THE FRONT\n\n");
	if (size_of_state(intfc) > 0)
	{
	    for (c = intfc->curves; c && *c;  ++c)
	    	fshow_curve_states(file,*c);
	}
	else
	{
	    (void) fprintf(file,"No states assigned on intfc\n");
	    return;
	}
	(void) fprintf(file,"\n\n");
	(void) fprintf(file,"\t\tEND OF STATES ON THE FRONT\n\n");
}		/*end f_fshow_intfc_states2d*/

LOCAL	void state_along_bond(
	COMPONENT	   comp,
	float		   *t,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	   *hs,
	Locstate	   state)
{
	CURVE		*c = Curve_of_hs(hs);
	BOND		*b = Bond_of_hse(hse);
	Locstate	start_state, end_state;

	if (comp == negative_component(c))
	{
	    start_state = (b == c->first) ?
			      left_start_state(c) : left_state(b->start);
	    end_state = (b == c->last) ?
			      left_end_state(c) : left_state(b->end);
	}
	else if (comp == positive_component(c))
	{
	    start_state = (b == c->first) ?
			      right_start_state(c) : right_state(b->start);
	    end_state = (b == c->last) ?
			      right_end_state(c) : right_state(b->end);
	}
	else
	{
	    screen("ERROR in state_along_bond(), comp = %d not on curve %d\n",
		   comp,c);
	    print_curve(c);
	    clean_up(ERROR);
	}

	bi_interpolate_intfc_states(c->interface,1.0-t[0],t[0],Coords(b->start),
		                    start_state,Coords(b->end),end_state,state);
}		/*end state_along_bond*/

LOCAL	void	slsr2d(
	POINT		*p,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	*hs,
	Locstate	*sl,
	Locstate	*sr)
{
	BOND *b = Bond_of_hse(hse);
	CURVE *c = Curve_of_hs(hs);

	if (p == c->start->posn && b == c->first)
	{
	    *sl = left_start_state(c);
	    *sr = right_start_state(c);
	}
	else if (p == c->end->posn && b == c->last)
	{
	    *sl = left_end_state(c);
	    *sr = right_end_state(c);
	}
	else
	{
	    *sl = left_state(p);
	    *sr = right_state(p);
	}
}		/*end slsr2d*/

/*
*				normal2d():
*
*	Returns the Normal to the hypersurface hs at the point p on
*	hypersurface element hse of CURVE c.  The normal points from the
*	negative side to the positive side of the hypersurface.  In two
*	space dimensions this means it points from the LEFT=NEGATIVE_SIDE
*       side TO the RIGHT=POSITIVE_SIDE side.
*
*	Care is taken to handle the hypersurface boundaries correctly.
*	In case the hypersurface is a curve c that is closed
*	and the point in question is an endpoint,
*	then the normal is computed as if the point was an interior
*	point of the CURVE.
*/

LOCAL	void normal2d(
	POINT		   *p,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	   *hs,
	float		   *nor,
	Front		   *front)
{
	float		t[MAXD];

	tangent(p,Bond_of_hse(hse),Curve_of_hs(hs),t,front);
	nor[0] = t[1];
	nor[1] = -t[0];
}		/*end normal2d*/

#endif /* defined(TWOD) */

#if defined(THREED)
/*
*		f_fshow_intfc_states3d():
*
*	Prints states on an interface using print_intfc_state().
*/

/*ARGSUSED*/
LOCAL	void f_fshow_intfc_states3d(
	FILE		*file,
	INTERFACE	*intfc)
{
	SURFACE		**s;

	(void) fprintf(file,"\t\tSTATES ON THE FRONT\n\n");
	if (size_of_state(intfc) > 0)
	{
	    for (s = intfc->surfaces; s && *s;  ++s)
	    	fshow_surface_states(file,*s);
	}
	else
	{
	    (void) fprintf(file,"No states assigned on intfc\n");
	    return;
	}
	(void) fprintf(file,"\n\n");
	(void) fprintf(file,"\t\tEND OF STATES ON THE FRONT\n\n");
}		/*end f_fshow_intfc_states3d*/

LOCAL	void state_in_tri(
	COMPONENT	   comp,
	float		   *t,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	   *hs,
	Locstate	   state)
{
	TRI		*tri = Tri_of_hse(hse);
	Locstate	sl, sr, s[3];
	POINT		*p;
	int		i;

	for (i = 0; i < 3; ++i)
	{
	    p = Point_of_tri(tri)[i];
            slsr(p,hse,hs,&sl,&sr);
            s[i] = (comp == negative_component(hs)) ? sl : sr;
	}

        if (s[0] == NULL || s[1] == NULL || s[2] == NULL)
	{
	    screen("ERROR in state_in_tri(), NULL state found"
	           "s0= %p, s1= %p, s2= %p\n",
		   (POINTER)s[0],(POINTER)s[1],(POINTER)s[2]);
	    print_tri(tri,hs->interface);
	    clean_up(ERROR);
	}

	if (tri_interpolate_intfc_states(hs->interface,t[0],t[1],t[2],
		                         Coords(Point_of_tri(tri)[0]),s[0],
					 Coords(Point_of_tri(tri)[1]),s[1],
		                         Coords(Point_of_tri(tri)[2]),s[2],
					 state) != FUNCTION_SUCCEEDED)
	{
	    screen("ERROR in state_in_tri(), "
		   "tri_interpolate_intfc_states failed\n");
	    (void) printf("comp = %d\n",comp);
	    print_tri(tri,hs->interface);
	    print_tri_states(tri,hs);
	    print_hypersurface(hs);
	    print_interface(hs->interface);
	    gview_plot_interface("ERROR_in_state_in_tri",hs->interface);
	    clean_up(ERROR);
	}
}		/*end state_in_tri*/

LOCAL bool f_set_boundary3d(
	INTERFACE	*intfc,
	RECT_GRID	*gr,
	COMPONENT	default_comp,
	float		eps)
{
	bool		status;
	bool		sav_copy = copy_intfc_states();

	set_copy_intfc_states(NO);
	status = i_set_boundary3d(intfc,gr,default_comp,eps);
	set_copy_intfc_states(sav_copy);
	return status;
}		/*end f_set_boundary3d*/

/*ARGSUSED*/
LOCAL	void	slsr3d(
	POINT		   *p,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	   *hs,
	Locstate	   *sl,
	Locstate	   *sr)
{
	TRI	 *tri = Tri_of_hse(hse);
	TRI      **tris;
	BOND	 *b;
	BOND_TRI *btri;
	int	 vertex, pside, nside;

        if (!Boundary_point(p))
	{ 
	    *sl = left_state(p);
	    *sr = right_state(p);
	    return;
	} 

	if ((vertex = Vertex_of_point(tri,p)) == ERROR)
	{
	    *sl = *sr = NULL;
	    screen("ERROR in slsr3d(), point not on tri\n");
	    (void) printf("p(%llu) = ",point_number(p));
	    print_general_vector("",Coords(p),3,"\n");
	    (void) printf("Boundary(p) = %d\n",Boundary(p));
	    (void) printf("Boundary_point(p) = %d\n",Boundary_point(p));
	    print_tri(tri,hs->interface);
	    if (!is_side01_a_bond(tri))
	    {
		(void) printf("Tri on side 01\n");
		print_tri(Tri_on_side01(tri),hs->interface);
	    }
	    if (!is_side12_a_bond(tri))
	    {
		(void) printf("Tri on side 12\n");
		print_tri(Tri_on_side12(tri),hs->interface);
	    }
	    if (!is_side20_a_bond(tri))
	    {
		(void) printf("Tri on side 20\n");
		print_tri(Tri_on_side20(tri),hs->interface);
	    }
	    print_interface(hs->interface);
	    clean_up(ERROR);
	}
	nside = vertex;
	pside = Prev_m3(vertex);

	if (is_side_bdry(tri,nside))
	    btri = Bond_tri_on_side(tri,nside);
	else if (is_side_bdry(tri,pside))
	    btri = Bond_tri_on_side(tri,pside);
	else
	{
	    int ntris = set_tri_list_around_point(p,tri,&tris,hs->interface);
	    vertex = Vertex_of_point(tris[0],p);
	    nside = vertex;
	    pside = Prev_m3(vertex);
	    if (is_side_bdry(tris[0],nside))
	        btri = Bond_tri_on_side(tris[0],nside);
	    else if (is_side_bdry(tris[0],pside))
	        btri = Bond_tri_on_side(tris[0],pside);
	    else
	    {
		int i;
	        *sl = *sr = NULL;
	        screen("ERROR in slsr3d(), couldn't find adjacent tri "
		       "with bond\n");
	        (void) printf("p(%llu) = ",point_number(p));
	        print_general_vector("",Coords(p),3,"\n");
	        (void) printf("Boundary(p) = %d\n",Boundary(p));
	        (void) printf("Boundary_point(p) = %d\n",Boundary_point(p));
	        (void) printf("Input triangle\n");
	        print_tri(Tri_of_hse(hse),hs->interface);
	        (void) printf("Boundary triangle\n");
	        (void) printf("Current tri\n");
	        print_tri(tri,hs->interface);
	        if (!is_side01_a_bond(tri))
	        {
	            (void) printf("Tri on side 01\n");
	            print_tri(Tri_on_side01(tri),hs->interface);
	        }
	        if (!is_side12_a_bond(tri))
	        {
		    (void) printf("Tri on side 12\n");
		    print_tri(Tri_on_side12(tri),hs->interface);
	        }
	        if (!is_side20_a_bond(tri))
	        {
		    (void) printf("Tri on side 20\n");
		    print_tri(Tri_on_side20(tri),hs->interface);
	        }
	        tri = Tri_of_hse(hse);
	        (void) printf("Input tri\n");
	        if (!is_side01_a_bond(tri))
	        {
		    (void) printf("Tri on side 01\n");
		    print_tri(Tri_on_side01(tri),hs->interface);
	        }
	        if (!is_side12_a_bond(tri))
	        {
		    (void) printf("Tri on side 12\n");
		    print_tri(Tri_on_side12(tri),hs->interface);
	        }
	        if (!is_side20_a_bond(tri))
	        {
		    (void) printf("Tri on side 20\n");
		    print_tri(Tri_on_side20(tri),hs->interface);
	        }
		(void) printf("adjacent tris\n");
		for (i = 0; i < ntris; ++i)
		{
		    (void) printf("tris[%d] - ",i);
		    print_tri(tris[i],hs->interface);
		}
	        clean_up(ERROR);
	    }
	    tri = tris[0];
	}

	if (btri == NULL)
	{
	    *sl = *sr = NULL;
	    screen("ERROR in slsr3d(), can't find btri\n");
	    (void) printf("p(%llu) = ",point_number(p));
	    print_general_vector("",Coords(p),3,"\n");
	    (void) printf("Boundary(p) = %d\n",Boundary(p));
	    (void) printf("Boundary_point(p) = %d\n",Boundary_point(p));
	    (void) printf("Input triangle\n");
	    print_tri(Tri_of_hse(hse),hs->interface);
	    (void) printf("Boundary triangle\n");
	    (void) printf("Current tri\n");
	    print_tri(tri,hs->interface);
	    if (!is_side01_a_bond(tri))
	    {
	        (void) printf("Tri on side 01\n");
	        print_tri(Tri_on_side01(tri),hs->interface);
	    }
	    if (!is_side12_a_bond(tri))
	    {
		(void) printf("Tri on side 12\n");
		print_tri(Tri_on_side12(tri),hs->interface);
	    }
	    if (!is_side20_a_bond(tri))
	    {
		(void) printf("Tri on side 20\n");
		print_tri(Tri_on_side20(tri),hs->interface);
	    }
	    tri = Tri_of_hse(hse);
	    (void) printf("Input tri\n");
	    if (!is_side01_a_bond(tri))
	    {
		(void) printf("Tri on side 01\n");
		print_tri(Tri_on_side01(tri),hs->interface);
	    }
	    if (!is_side12_a_bond(tri))
	    {
		(void) printf("Tri on side 12\n");
		print_tri(Tri_on_side12(tri),hs->interface);
	    }
	    if (!is_side20_a_bond(tri))
	    {
		(void) printf("Tri on side 20\n");
		print_tri(Tri_on_side20(tri),hs->interface);
	    }
	    clean_up(ERROR);
	}
	b = btri->bond;
	if (p == b->start)
	{
	    *sl = left_start_btri_state(btri);
	    *sr = right_start_btri_state(btri);
	}
	else if (p == b->end)
	{
	    *sl = left_end_btri_state(btri);
	    *sr = right_end_btri_state(btri);
	}
	else
	{
	    *sl = *sr = NULL;
	    screen("ERROR in slsr3d(), point not on boundary tri\n");
	    (void) printf("p(%llu) = ",point_number(p));
	    print_general_vector("",Coords(p),3,"\n");
	    (void) printf("Boundary(p) = %d\n",Boundary(p));
	    (void) printf("Boundary_point(p) = %d\n",Boundary_point(p));
	    (void) printf("bond=%llu\n",bond_number(b,hs->interface));
	    (void) printf("b->start(%llu)= ",point_number(b->start));
	    print_general_vector("",Coords(b->start),3,"\n");
	    (void) printf("b->end(%llu)= ",point_number(b->end));
	    print_general_vector("",Coords(b->end),3,"\n");
	    (void) printf("Input triangle\n");
	    print_tri(Tri_of_hse(hse),hs->interface);
	    (void) printf("Boundary triangle\n");
	    (void) printf("Current tri\n");
	    print_tri(tri,hs->interface);
	    if (!is_side01_a_bond(tri))
	    {
	        (void) printf("Tri on side 01\n");
	        print_tri(Tri_on_side01(tri),hs->interface);
	    }
	    if (!is_side12_a_bond(tri))
	    {
		(void) printf("Tri on side 12\n");
		print_tri(Tri_on_side12(tri),hs->interface);
	    }
	    if (!is_side20_a_bond(tri))
	    {
		(void) printf("Tri on side 20\n");
		print_tri(Tri_on_side20(tri),hs->interface);
	    }
	    tri = Tri_of_hse(hse);
	    (void) printf("Input tri\n");
	    if (!is_side01_a_bond(tri))
	    {
		(void) printf("Tri on side 01\n");
		print_tri(Tri_on_side01(tri),hs->interface);
	    }
	    if (!is_side12_a_bond(tri))
	    {
		(void) printf("Tri on side 12\n");
		print_tri(Tri_on_side12(tri),hs->interface);
	    }
	    if (!is_side20_a_bond(tri))
	    {
		(void) printf("Tri on side 20\n");
		print_tri(Tri_on_side20(tri),hs->interface);
	    }
	    clean_up(ERROR);
	} 
}		/*end slsr3d*/


EXPORT	void	set_normal3d_method(
	NORMAL3D_METHOD nor3d_meth,
	int             dim)
{
	F_USER_INTERFACE *fuh = f_user_hook(dim);
	switch (nor3d_meth)
	{
	case PLANE_FIT_NORMAL:
	    fuh->_interface_normal_function._normal = f_plane_fit_normal3d;
	    fuh->_interface_normal_function._normal_name = 
	        strdup("f_plane_fit_normal3d");
	    break;
	case NORMAL_METHOD_FROM_RESTART:
	    break;
	case AREA_WEIGHTED_NORMAL:
	default:
	    fuh->_interface_normal_function._normal = f_area_weighted_normal3d;
	    fuh->_interface_normal_function._normal_name =
	        strdup("f_area_weighted_normal3d");
	    break;
	}
}		/*end set_normal3d_method*/

/*ARGSUSED*/
LOCAL  void f_plane_fit_normal3d(
	POINT		   *p,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	   *hs,
	float		   *nor,
	Front		   *front)
{
	plane_fit_normal3d(p,hse,hs,nor);
}		/*end f_plane_fit_normal3d*/

/*ARGSUSED*/
LOCAL  void f_area_weighted_normal3d(
	POINT		   *p,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	   *hs,
	float		   *nor,
	Front		   *front)
{
	area_weighted_normal3d(p,hse,hs,nor);
}		/*end f_area_weighted_normal3d*/
#endif /* defined(THREED) */

