/*
*                               foverture_adv.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*
*       this file includes functions set_patch_front, set_patch_topo_grid,
*                       and set_patch_comput_grid
*/

#define DEBUG_STRING    "foverture_adv"

#include <front/flocaldecs.h>

#if defined(USE_OVERTURE)

#if defined(TWOD)

/*
*			normal_advance_front2d():
*
*	Advances the Front structure front by copying its interface
*	to the interface for newfront and calling
*		
*		(*front->node_propagate)() for each node
*		(*front->curve_propagate)() for each curve
*		point_propagate() for (certain) points on each curve
*		(*front->bond_propagate)() for (certain) bonds on each curve
*		(*front->snd_node_propagate)() for each node
*		(*front->tan_curve_propagate)() for each curve
*	and
*		redistribute()
*
*	in this order to modify the new interface.
*	
*	To define the points and bonds to be propagated in the normal or
*	tangential propagation loop, the flags n_pt_propagated(p)
*	and t_pt_propagated(p) respectively are set to NO.  As each point
*	is processed by the respective propagate step the corresponding
*	flag is set to YES.  To supress a given propagation step,  the
*	the corresponding flag can be turned on prior to propagation.
*
*	Returns 1 if successful, or 0 if copying the interface fails or
*	one of the propagation, or redistribution routines returns 0.
*
*	For documentation of debug_front(), see the documentation for that
*	function in another file (currently fprint.c).
*/

LOCAL   int    second_node_propagate2d(float,float*,Front*,Front**,POINTER);  

LIB_LOCAL int normal_advance_front2d(
	float    dt,
	float    *dt_frac,
	Front    *front,
	Front    **newfront,
	POINTER  wave)
{
	CURVE      *oldc,*newc;
	CURVE	   **c;
	NODE 	   *oldn,*newn;
	RPROBLEM   *rp,*rp1;
	int        node_stat;
        NODE_FLAG  flag;  
	const char *fname = "normal_advance_front2d()";
        int        status = GOOD_STEP;

	debug_print("front","Entered %s(step %d time %g dt %g)\n",fname,
	        	        front->step,front->time,dt);
	debug_front("old_front","into advance front",front);

	*newfront = copy_front(front);
	Interface_redistributed(*newfront) = NO;

        if(front->interf->nodes == NULL)
        {
            bool       sav_copy;
            INTERFACE  *sav_intfc;  
            sav_intfc = current_interface();
            sav_copy = copy_intfc_states();
            set_size_of_intfc_state(size_of_state(front->interf));
            set_copy_intfc_states(YES);
            (*newfront)->interf = copy_interface(front->interf);
            set_current_interface(sav_intfc);
            set_copy_intfc_states(sav_copy);
            return return_advance_front(front,newfront,GOOD_STEP,fname);  
        }

	rp = NULL;

        set_to_next_node_only(flag);  

	set_node_doubly_linked_list(front->interf);

	        /* Initialize Newfront */

	start_clock("init_new_front");
	capture_waves(front);
	print_storage("before init_new_front","ADV_storage");
	        /* TODO: Remove this option!!!!! */
	if (front->init_topology_of_new_interface)
	    status = (*front->init_topology_of_new_interface)(front,*newfront);
	else 
	{
	    set_size_of_intfc_state(size_of_state(front->interf));
	    set_copy_intfc_states(NO);
	    set_add_to_correspond_list(YES);
            /* 
            If USE_OVERTURE, can not syncronize_time_step at here 
	    (*newfront)->interf = pp_copy_interface(front->interf);
            */ 
	    (*newfront)->interf = copy_interface(front->interf);
	    reset_hs_flags_on_intfc((*newfront)->interf);
	    status = ((*newfront)->interf != NULL) ? GOOD_STEP : ERROR_IN_STEP;
	    set_copy_intfc_states(YES);
	}
        if (status != GOOD_STEP)
        {
            (void) printf("ERROR in normal_advance_front2d(), "
                          "unable to copy interface\n");
            print_storage("after init_new_front","ADV_storage");
            clean_up(ERROR);
        }
	print_storage("after init_new_front","ADV_storage");
	stop_clock("init_new_front");

	        /* Set Default Propagation Limits */

	set_propagation_limits(front,*newfront);

	        /* Propagate the Curves */

	if (front->curve_propagate != NULL) 
	{
	    start_clock("curve_propagate");
	    if (debugging("front"))
	    	(void) printf("Loop over Curves\n");
	    for (c = front->interf->curves; c && *c; c++)
	    {
	        oldc = *c;
	        if (((newc = correspond_curve(oldc)) != NULL) &&
	    	     (correspond_curve(newc) != NULL))
	        {
	    	    if (debugging("propagate"))
	                (void) printf("\t\tpropagating curve %lu\n",
		                      curve_number(oldc));
		    curve_propagate(front,wave,oldc,newc,dt);
		}
	    }
	    debug_front("cp_front","after curve prop",*newfront);
	    stop_clock("curve_propagate");
	}

		/* Propagate the Nodes */

	if (debugging("front"))
	{
	    print_correspond_hyper_surf_list(front->interf);
	    print_correspond_hyper_surf_list((*newfront)->interf);
	}
	if (front->node_propagate != NULL) 
	{
	    // start_clock("node_propagate");
	    set_corresponds_for_node_prop(front->interf,(*newfront)->interf);
	    oldn = first_node(front->interf);
	    while (oldn != NULL) 
	    {
	        newn = correspond_node(oldn);
	        if (debugging("crx_status"))
	            print_linked_node_list((*newfront)->interf);
	        status = (newn != NULL) ?
	            (*front->node_propagate)(front,wave,oldn,newn,&rp,
	        			     dt,dt_frac,flag,NULL) : GOOD_NODE;

	        if (is_bad_status(status) &&
	          (point_in_buffer(Coords(oldn->posn),front->rect_grid) == YES))
	        {
	            (void) printf("WARNING in normal_advance_front2d(), "
	                          "node_propagation returns ");
                    print_node_status("WARNING in normal_advance_front2d(), "
                                      "node_propagation returns ",status,"\n"); 
                    /* 
	            print_node_status(status);
                    */  
	            (void) printf("Problem occurs in buffer zone - ignoring\n");
	            if (set_node_states_and_continue(oldn,newn,front))
	                status = GOOD_NODE;
	        }

	        switch (status) 
	        {
	        case GOOD_NODE:
	            oldn = adv_node_loop_after_good_prop(oldn,newn,&rp);
	            break;
	        case PSEUDOCROSS_NODE_NODE:
	            debug_print("PSEUDOCROSS","PSEUDOCROSS case\n");
	            oldn = reorder_node_loop(oldn,newn);
	            break;
	        case CROSS_NODE_NODE:
	        case BIFURCATION_NODE:
	            debug_print("CROSS","CROSS case\n");
	            oldn = next_node(oldn);
	            break;
	        case CROSS_PAST_CURVE_NODE:
	            (void) printf("WARNING in normal_advance_front2d(), ");
	            (void) printf("node_propagate failed, ");
                    print_node_status("WARNING in normal_advance_front2d(), "
                                      "node_propagate failed with status ",
                                      status,"\n");
	            print_node(oldn);
	            if (debugging("CROSS_PAST"))
	            {
	                (void) printf("Cross past curve case\n"
	                              "dt_frac = %g\n",*dt_frac);
	                (void) printf("Reducing time step\n");
	            }
	            *dt_frac *= TIME_STEP_REDUCTION_FACTOR(front->interf);
	            free_rp_list(&rp);
	            status = MODIFY_TIME_STEP;
	            goto sync_prop_stat1;
	        case MODIFY_TIME_STEP_NODE:
	            (void) printf("WARNING in normal_advance_front2d(), "
	                          "node_propagate returns "
	                          "MODIFY_TIME_STEP_NODE\n");
	            free_rp_list(&rp);
	            status = MODIFY_TIME_STEP;
	            goto sync_prop_stat1;
	        case REPEAT_TIME_STEP_NODE:
	            (void) printf("WARNING in normal_advance_front2d(), "
	                          "node_propagate returns "
	                          "REPEAT_TIME_STEP_NODE\n");
	            free_rp_list(&rp);
                    status = node_modify_time_step(oldn,front,NULL,
                                                   REPEAT_TIME_STEP);
	            goto sync_prop_stat1;
	        case NO_CROSS_NODE:
	            (void) printf("WARNING in normal_advance_front2d(), ");
	            (void) printf("node_propagate failed, ");
                    print_node_status("WARNING in normal_advance_front2d(), "
                                      "node_propagate failed with status ",
                                      status,"\n"); 
	            print_node(oldn);
	            if (debugging("NO_CROSS"))
	            {
	                (void) printf("No cross case\n");
	                (void) printf("dt_frac = %g\n",*dt_frac);
	                (void) printf("Reducing time step\n");
	            }
	            free_rp_list(&rp);
                    status = node_modify_time_step(oldn,front,dt_frac,
                                                   MODIFY_TIME_STEP);
	            goto sync_prop_stat1;
	        case ERROR_NODE:
	        default:
	            (void) printf("WARNING in normal_advance_front2d(), ");
	            (void) printf("node_propagate failed, ");
                    print_node_status("WARNING in normal_advance_front2d(), "
                                      "node_propagate failed with status ",
                                      status,"\n");
	            print_node(oldn);
	            if (debugging("ERROR_NODE"))
	            {
	                (void) printf("Old interface:\n");
	                print_interface(front->interf);
	                print_correspond_hyper_surf_list(front->interf);
	                (void) printf("New interface:\n");
	                print_interface((*newfront)->interf);
	                print_correspond_hyper_surf_list((*newfront)->interf);
	            }
                    status = node_modify_time_step(oldn,front,dt_frac,
                                                   ERROR_IN_STEP);
                    free_rp_list(&rp);
	            goto sync_prop_stat1;
	        }
	    } /* end of while (oldn != NULL) */

	    set_correspond_hyper_surf_bdrys_to_NULL(front->interf);
	    set_correspond_hyper_surf_bdrys_to_NULL((*newfront)->interf);
	    if (rp && (front->twodrproblem != NULL)) 
	    {
	        for (rp1 = rp; rp1; rp1 = rp1->prev) 
	        {
	            debug_front("2drp_front",
	                "new between node loop and rp loop",*newfront);
	                    
	            status = (*front->twodrproblem)(front,*newfront,wave,&rp1);

	            /* At this point, rp is nothing more than a valid element
	             * of the list which provides a starting point
	             * for deleting the list.  If we delete an element of
	             * the list in front->twodrproblem (presumably due to
	             * merging two RPROBLEM's), then rp may point to freed
	             * storage and will need to be updated.  rp1 should still
	             * be a valid element of the list.
	             */
	            rp = rp1;

                    if (status != GOOD_STEP)
                    {
                        print_time_step_status("WARNING in advance_front2d(), "
                                               "rp failed with status = ",
                                               status,"\n");
                        switch (status)
                        {
                        case GOOD_STEP:
                            break;

                        case REPEAT_TIME_STEP:
                            break;
                        case MODIFY_TIME_STEP:
                            status = rp_modify_time_step(rp1,front,status);
                            if (status == MODIFY_TIME_STEP)
                            {
                                *dt_frac = rp1->dt_frac;
                                if (debugging("2drp"))
                                {
                                    print_rproblem(rp1);
                                    (void) printf("dt_frac %g\n",*dt_frac);
                                    (void) printf("Reducing time step\n");
                                }
                                *dt_frac = limit_dt_frac(*dt_frac,front);
                            }
                            break;

                        case ERROR_IN_STEP:
                        default:
                            print_rproblem(rp1);
                            /* Try reducing the time step */
                            status = rp_modify_time_step(rp1,front,status);
                            if (status == MODIFY_TIME_STEP)
                                *dt_frac *=
                                    TIME_STEP_REDUCTION_FACTOR(front->interf);
                            break;
                        }
                    }
                    if (status != GOOD_STEP)
                        break;
	        }
	        free_rp_list(&rp);
	        debug_front("2drp_front","after 2drp loop",*newfront);
	    }
	    else if (rp) 
	    {
	        for (rp1 = rp; rp1; rp1 = rp1->prev) 
	        {
	            print_rproblem(rp1);
	        }
	        free_rp_list(&rp);
	        (void) printf("WARNING in normal_advance_front2d(), ");
	        (void) printf("CROSS code needed\n");
	        status = ERROR_IN_STEP;
	    }
	}
        /* 061003 closed, since the correspondence is reset. 
         * The second node prop. is done in tangential step now. 
        node_stat = second_node_propagate2d(dt,dt_frac,front,newfront,wave);
        if(GOOD_STEP != node_stat) 
        {
            (void) printf("WARNING in normal_advance_front2d(), "
                      "second node_propagation returns stat= %d", node_stat);
            clean_up(ERROR);  
        }  
        */
sync_prop_stat1:
	return return_advance_front(front,newfront,status,fname);
}		/*end normal_advance_front2d*/


LOCAL int second_node_propagate2d(
	float    dt,
	float    *dt_frac,
	Front    *front,
	Front    **newfront,
	POINTER  wave)
{
	NODE 	   *tempn,*newn;
	INTERFACE  *tempintfc;
	bool	   stat;
	int        status = GOOD_STEP;
	long       intfc_modified;
	const char *fname = "second_node_propagate2d()";

	debug_print("front","Entered %s(step %d time %g dt %g)\n",fname,
	        	        front->step,front->time,dt);

	/* Make Temp Interface for Second/Tangential Propagation */


	if ( front->interf == NULL || front->interf->nodes == NULL )
	    return GOOD_STEP;

	set_node_doubly_linked_list((*newfront)->interf);

	if (front->snd_node_propagate) 
	{
	    start_clock("snd_copy_interface");
	    print_storage("before snd_copy_interface","ADV_storage");
	    tempintfc = (*newfront)->interf;
	    set_size_of_intfc_state(size_of_state(tempintfc));
	    set_add_to_correspond_list(YES);
	    (*newfront)->interf = copy_interface(tempintfc);
	    copy_hypersurface_flags((*newfront)->interf);
	    interpolate_intfc_states((*newfront)->interf) = YES;
	    print_storage("after snd_copy_interface","ADV_storage");
	    stop_clock("snd_copy_interface");
	}
      
	/* Second Propagation for the States Around the Nodes */

	if (front->snd_node_propagate) 
	{
	    start_clock("snd_node_propagate");
	    if (debugging("front"))
	    	(void) printf("Second Loop over Nodes\n");

	    tempn = first_node(tempintfc);
	    newn = first_node((*newfront)->interf);
	    while (newn != NULL)
	    {
	    	(*front->snd_node_propagate)(front,*newfront,wave,
	    				     tempintfc,tempn,newn,dt);
	    	tempn = next_node(tempn);
	    	newn = next_node(newn);
	    }

	    debug_front("snd_front","after snd_node prop",*newfront);
	    stop_clock("snd_node_propagate");
	}

	if (tempintfc)
	    (void) delete_interface(tempintfc);
	print_storage("after delete tempintfc","ADV_storage");

	if (status != GOOD_STEP)
	    return return_advance_front(front,newfront,status,fname);
	return return_advance_front(front,newfront,GOOD_STEP,fname);
}		/*end tangnt_advance_front2d*/

LIB_LOCAL int tangnt_advance_front2d(
	float    dt,
	float    *dt_frac,
	Front    *front,
	Front    **newfront,
	POINTER  wave)
{
	CURVE      *tempc,*newc;
	CURVE	   **c;
	NODE 	   *tempn,*newn;
	INTERFACE  *tempintfc;
	bool	   stat;
        bool       sav_copy;
	int        status = GOOD_STEP;
	long       intfc_modified;
	const char *fname = "tangnt_advance_front2d()";

	debug_print("front","Entered %s(step %d time %g dt %g)\n",fname,
	        	        front->step,front->time,dt);

	if ( front->interf == NULL || front->interf->nodes == NULL )
	    return return_advance_front(front,newfront,GOOD_STEP,fname);

        /* 050703 added, The interface regularization is performed here. */
        /* In advance_front2d(), these functions are called after
           normal_scatter_front, we call them here */ 
        delete_small_loops(*newfront); 

	stat = consistent_propagated_loop_orientations(dt,dt_frac,*newfront,wave);
	if(debugging("tangnt_advance_front2d"))
	    printf("In tangnt_advance_front2d() for patch %d\n", front->patch_number);
	if (stat == NO)
	{
	    (void) printf("WARNING in tangnt_advance_front2d(), "
		          "Inconsistent orientation of propagated loop "
	                  "detected after point and node propagations");
            if (pp_numnodes() > 1)
                (void) printf(" on processor %d\n",pp_mynode());
            else
                (void) printf("\n");
            status = MODIFY_TIME_STEP; 
            goto sync_prop_stat2;  
	}
        delete_exterior_curves(*newfront,front->interf);  
        intfc_delete_very_short_bonds(*newfront);
        stat = intfc_delete_fold_back_bonds(*newfront);
        if(stat == FUNCTION_FAILED)
        {
            (void) printf("WARNING in tangnt_advance_front2d(), "
                          "intfc_delete_fold_back_bonds() loop "
                          "detected error");
            if (pp_numnodes() > 1)
                (void) printf(" on processor %d\n",pp_mynode());
            else
                (void) printf("\n");
            status = MODIFY_TIME_STEP; 
            goto sync_prop_stat2;  
        } 
        delete_small_loops(*newfront); 
        /* 060303 added */ 
        measure_front(*newfront);
         
        /* Make Temp Interface for Second/Tangential Propagation */

        interpolate_intfc_states((*newfront)->interf) = YES;   
	set_node_doubly_linked_list((*newfront)->interf);
	if (front->snd_node_propagate || front->tan_curve_propagate) 
	{
	    start_clock("snd_copy_interface");
	    print_storage("before snd_copy_interface","ADV_storage");
	    tempintfc = (*newfront)->interf;
	    set_size_of_intfc_state(size_of_state(tempintfc));
	    set_add_to_correspond_list(YES);

            /* 060303, added copy_interface flag */
            sav_copy = copy_intfc_states();
            set_copy_intfc_states(YES);

	    (*newfront)->interf = copy_interface(tempintfc);
	    copy_hypersurface_flags((*newfront)->interf);
	    interpolate_intfc_states((*newfront)->interf) = YES;

            /* 060303, added copy_interface flag */
            set_copy_intfc_states(sav_copy);
	    print_storage("after snd_copy_interface","ADV_storage");
	    stop_clock("snd_copy_interface");
	}

        /* Second Propagation for the States Around the Nodes */

        if (front->snd_node_propagate)
        {
            start_clock("snd_node_propagate");
            if (debugging("front"))
                (void) printf("Second Loop over Nodes\n");

            tempn = first_node(tempintfc);
            newn = first_node((*newfront)->interf);
            while (newn != NULL)
            {
                (*front->snd_node_propagate)(front,*newfront,wave,
                                             tempintfc,tempn,newn,dt);
                tempn = next_node(tempn);
                newn = next_node(newn);
            }

            debug_front("snd_front","after snd_node prop",*newfront);
            stop_clock("snd_node_propagate");
        }

	/* Tangential Sweep for States on the Curves */

	if (front->tan_curve_propagate) 
	{
	    start_clock("tan_curve_propagate");
	    if (debugging("front"))
	    	(void) printf("Second Loop over Curves\n");
	    for (c = tempintfc->curves; c && *c; c++)
	    {
	    	tempc = *c;
	    	newc = correspond_curve(tempc);
	    	(*front->tan_curve_propagate)(front,*newfront,
	    				      tempintfc,tempc,newc,dt);
	    }
	    debug_front("tcp_front","after tan_curve_propagate:",*newfront);
	    stop_clock("tan_curve_propagate");
	}

	if (tempintfc)
	    (void) delete_interface(tempintfc);
	print_storage("after delete tempintfc","ADV_storage");

        /* 060303, delete_phys_remn_on_bdry() added */
        /* Provide robustness for untangle algorithms */
        /*   delete remnants of scalar physical   */
        /*  curves sticking to NEUMANN boundaries */
        /* Add to delete_exterior_curves()? */

        if (delete_phys_remn_on_bdry(*newfront) == NO)
        {
            (void) printf("WARNING in tangnt_advance_front2d(), "
                          "delete_phys_remn_on_bdry() detected error\n");
            debug_front("ERROR_front","after error",*newfront);
            *dt_frac = Min_time_step_modification_factor(front);
            status = MODIFY_TIME_STEP;
            goto sync_prop_stat2;
        }
        debug_front("dspr_front",
                    "after 1st delete_phys_remn_on_bdry():",*newfront);

sync_prop_stat2:  
	if (status != GOOD_STEP)
	    return return_advance_front(front,newfront,status,fname);
	return return_advance_front(front,newfront,GOOD_STEP,fname);
}		/*end tangnt_advance_front2d*/


LIB_LOCAL int redist_advance_front2d(
	float    dt,
	float    *dt_frac,
	Front    *front,
	Front    **newfront,
	POINTER  wave)
{
	int        status, do_redist = YES;
	const char *fname = "redist_advance_front2d()";
	INTERFACE  *save_intfc[2];

        set_current_interface((*newfront)->interf); 
        save_intfc[0] = make_save_intfc((*newfront)->interf);
        save_intfc[1] = NULL;

redo_redistribute: 
        switch((status = redistribute(*newfront,do_redist,NO)))
	{
	case	GOOD_REDISTRIBUTION:
	break;
	case	UNABLE_TO_UNTANGLE:
	    (void) printf("WARNING in redist_advance_front2d(), "
	                  "redistribution of front failed\n"
	                  "Restarting redistribute from saving intfc\n");
            set_current_interface(save_intfc[0]);
            if((*newfront)->interf != NULL)
                delete_interface((*newfront)->interf);
            (*newfront)->interf = save_intfc[0];
            save_intfc[0] = NULL;      
            /* 
	    (void) delete_interface((*newfront)->interf);
	    (*newfront)->interf = NULL;
            */
	    do_redist = NO;
            goto redo_redistribute; 
	break;
	case	MODIFY_TIME_STEP_REDISTRIBUTE:
	    (void) printf("WARNING in redist_advance_front2d(), "
	                  "redistribute returns\n"
	                  "\t\tMODIFY_TIME_STEP_REDISTRIBUTE\n");
	break;
	case	BAD_REDISTRIBUTION:
	default:
	    (void) printf("WARNING in redist_advance_front2d(), "
	                  "redistribution of front failed, BAD_REDISTRIBUTION\n");
	    debug_front("ERROR_front","after error",*newfront);
	    *dt_frac = Min_time_step_modification_factor(front);
	break;
	}
        delete_interface(save_intfc[0]);         

	if (status != GOOD_REDISTRIBUTION)
	    return return_advance_front(front,newfront,status,fname);

	Redistribution_count(front) = Redistribution_count(*newfront);
	(*newfront)->step = front->step + 1;
	(*newfront)->time = front->time + dt;
	debug_front("redist_front","after redistribution",*newfront);

	/* Delete non-boundary curves that lie  */
	/* fully on or exterior to the boundary */
	delete_exterior_curves(*newfront,front->interf);
    
        if(delete_phys_remn_on_bdry(*newfront) == NO)
        {
            (void) printf("WARNING in redist_advance_front2d(), "
                          "delete_phys_remn_on_bdry() detected error\n");
            debug_front("ERROR_front","after error",*newfront);
            status = MODIFY_TIME_STEP;
            return return_advance_front(front,newfront,status,fname);
        }  

	delete_small_loops(*newfront);

	measure_front(*newfront);

	test_for_mono_comp_curves((*newfront)->interf);

	return return_advance_front(front,newfront,GOOD_REDISTRIBUTION,fname);
}		/*end redist_advance_front2d*/

#endif /*if defined(TWOD) */

#if defined(THREED)

LIB_LOCAL int normal_advance_front3d(
	float    dt,
	float    *dt_frac,
	Front    *front,
	Front    **newfront,
	POINTER  wave)
{
        static const char *fname = "normal_advance_front3d";
        bool       has_tracked_surfaces;
        int        status;
        INTERFACE       *intfc_old;
        float           V[MAXD];
        
        DEBUG_ENTER(normal_advance_front3d)

        *newfront = copy_front(front);
        Interface_redistributed(*newfront) = NO;
        has_tracked_surfaces = (front->interf->surfaces != NULL) ? YES : NO;
        if (has_tracked_surfaces == NO)
        {
            set_size_of_intfc_state(size_of_state(front->interf));
            set_copy_intfc_states(YES);
            (*newfront)->interf = copy_interface(front->interf);
            status = ((*newfront)->interf != NULL) ? GOOD_STEP : ERROR_IN_STEP;
            return return_advance_front(front,newfront,status,fname);
        }


        start_clock("copy_interface");

                /* Initialize Newfront */

        set_size_of_intfc_state(size_of_state(front->interf));
        set_copy_intfc_states(NO);
        (*newfront)->interf = copy_interface(front->interf);
        if ((*newfront)->interf == NULL)
        {
            status = ERROR_IN_STEP;
            (void) printf("ERROR in advance_3d_front(), "
                          "unable to copy interface\n");
            return return_advance_front(front,newfront,status,fname);
        }
        stop_clock("copy_interface");

             /* Propagate points on surfaces */

        start_clock("propagate");
        set_copy_intfc_states(YES);

        intfc_old = front->interf;

        if (front->_point_propagate != NULL)
        {
            bool tri_list_status;
            start_clock("normal_propagate");
            tri_list_status = (intfc_old->modified &&
                (make_tri_comp_lists(intfc_old) == FUNCTION_FAILED)) ? NO : YES;
            if (tri_list_status == NO)
            {
                stop_clock("normal_propagate");
                screen("ERROR in propagate_3d_front(), "
                       "make_tri_comp_lists() failed\n");
                clean_up(ERROR);
            }

            /* Set Default Propagation Limits */

            set_propagation_limits(front,*newfront);
            start_clock("init_propagate");
            init_propagate(front);
            stop_clock("init_propagate");

            if (front->_point_propagate != NULL)
                propagate_surface_points(front,*newfront,wave,dt,V);

            if (front->curve_propagate != NULL)
                propagate_curve_points(front,*newfront,wave,dt);

            if (front->node_propagate != NULL)
            {
                status = propagate_node_points(front,*newfront,
                                                    wave,dt,dt_frac);
                if (status != GOOD_STEP)
                {
                    DEBUG_LEAVE(propagate_3d_front)
                    return status;
                }
            }
            reset_normal_on_intfc((*newfront)->interf);
            debug_front("np_front","after normal propagation",*newfront);
            stop_clock("normal_propagate");
        }

        /*  mpi_sync inside 
        debug_propagate_3d_front(*newfront);
        */
        interface_reconstructed((*newfront)->interf) = NO;

        DEBUG_LEAVE(normal_advance_front3d)
        return GOOD_STEP;  
}

LIB_LOCAL int tangnt_advance_front3d(
	float    dt,
	float    *dt_frac,
	Front    *front,
	Front    **newfront,
	POINTER  wave)
{
        static const char *fname = "tangnt_advance_front3d";
        bool       has_tracked_surfaces;
        int        status;
        
        DEBUG_ENTER(tangnt_advance_front3d)

        DEBUG_LEAVE(tangnt_advance_front3d)
        return status;  
}

#endif /* if defined(THREED) */

#endif  /* if defined(USE_OVERTURE) */

