/*
*				fdecs.h:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains declarations of variables related to the front.  A front
*	consists of an interface (see int.c for documentation) and
*	and the values of the physical state variables on each side of the
*	interface, together with further variables which specify algorithms for
*	the processing of fronts.
*/

#if !defined(_FDECS_H)
#define _FDECS_H

#include <intfc/int.h>

#define LOCSTATE
typedef POINTER Locstate;

#include <front/frp.h>


		/* Possible values for fr->hyperbolic */

enum {
	NO_STATES   = 0,
	FULL_STATES = 1
};

		/* Possible values returned by node_propagate */

enum {
	GOOD_NODE = 1,
	NO_STORAGE_NODE,
	ERROR_NODE,
	PSEUDOCROSS_NODE_NODE,
	CROSS_NODE_NODE,
	PSEUDOCROSS_CURVE_NODE,
	CROSS_CURVE_NODE,
	BIFURCATION_NODE,
	CROSS_PAST_CURVE_NODE,
	NO_CROSS_NODE,
	MODIFY_TIME_STEP_NODE,
	REPEAT_TIME_STEP_NODE
};

		/* Possible values returned by redistribute() */

enum {
	BAD_REDISTRIBUTION	      = 0x0,
	GOOD_REDISTRIBUTION	      = 0x1,
	UNABLE_TO_UNTANGLE	      = 0x2,
	MODIFY_TIME_STEP_REDISTRIBUTE = 0x4
};

		/* Possible values returned by fr_bdry_untangle */

enum {
	ERROR_IN_UNTANGLE			      = 0,
	CURVES_UNTANGLED			      = 1,
	MODIFY_TIME_STEP_TO_UNTANGLE		      = 2,
	MODIFY_TIME_STEP_TO_UNTANGLE_BUT_FORCE_TANGLE = 3
};

		/* values for loops in untangle */
enum {
	UNPHYSICAL = 0,
	PHYSICAL   = 1
};

		/* Possible values for node propagation flag */

#if !defined(MAX_N_NODE_FLAGS)
#define MAX_N_NODE_FLAGS 64
#endif /* !defined(MAX_N_NODE_FLAGS) */

struct _NODE_FLAG {
	bool _node_flags[MAX_N_NODE_FLAGS];
};
typedef struct _NODE_FLAG NODE_FLAG;

enum _F_NODE_FLAG_INDEX {
	_TO_NEXT_NODE_ONLY_INDEX	      = 0,
	_CONTINUE_PAST_FIXED_NODE_INDEX,
	_CONTINUE_PAST_BIFURCATION_INDEX,
	_POSITION_ONLY_INDEX,
	_PHYS_VIRTUALS_PRESET_INDEX,
	_P_VIRTUALS_SET_FROM_NODE_INDEX,
	_NODE_VELOCITY_PRESET_INDEX,
	_SET_VIRTUALS_BY_ADJACENT_BOND_INDEX,
	_DONT_INSERT_ON_H_EXTEND_INDEX,
	_CONTINUE_PAST_BOUNDARY_INDEX,
	_ORDINARY_CROSS_ONLY_INDEX,
	_SINGLE_EXTEND_TO_CROSS_INDEX,
	_DOUBLE_EXTEND_TO_CROSS_INDEX,
	_FIRST_PHYSICS_NODE_FLAG_INDEX
};

#define clear_node_flag(flag)	zero_scalar(&flag,sizeof(NODE_FLAG))
#define set_to_next_node_only(flag)					\
     (clear_node_flag(flag), to_next_node_only(flag) = YES)
#define set_ordinary_cross_only(flag)					\
     clear_node_flag(flag)
#define to_next_node_only(flag)  					\
    (flag)._node_flags[_TO_NEXT_NODE_ONLY_INDEX]
#define continue_past_fixed_node(flag)					\
    (flag)._node_flags[_CONTINUE_PAST_FIXED_NODE_INDEX]
#define continue_past_bifurcation(flag)					\
    (flag)._node_flags[_CONTINUE_PAST_BIFURCATION_INDEX]
#define position_only(flag)						\
    (flag)._node_flags[_POSITION_ONLY_INDEX]
#define phys_virtuals_preset(flag)					\
    (flag)._node_flags[_PHYS_VIRTUALS_PRESET_INDEX]
#define p_virtuals_set_from_node(flag)					\
    (flag)._node_flags[_P_VIRTUALS_SET_FROM_NODE_INDEX]
#define node_velocity_preset(flag)					\
    (flag)._node_flags[_NODE_VELOCITY_PRESET_INDEX]
#define set_virtuals_by_adjacent_bond(flag)				\
    (flag)._node_flags[_SET_VIRTUALS_BY_ADJACENT_BOND_INDEX]
#define dont_insert_on_h_extend(flag)					\
    (flag)._node_flags[_DONT_INSERT_ON_H_EXTEND_INDEX]
#define continue_past_boundary(flag)					\
    (flag)._node_flags[_CONTINUE_PAST_BOUNDARY_INDEX]
#define single_extend_to_cross(flag)					\
    (flag)._node_flags[_SINGLE_EXTEND_TO_CROSS_INDEX]
#define double_extend_to_cross(flag)					\
    (flag)._node_flags[_DOUBLE_EXTEND_TO_CROSS_INDEX]


	/* Flags returned by check_cross() and is_good_cross() */

enum {
	NO_CROSS	    = 0x0,
	GOOD_CROSS	    = 0x1,
	END_OF_CURVE_CROSS  = 0x2,
	OUT_OF_BOUNDS_CROSS = 0x4,
	FOLD_BACK_CROSS	    = 0x8
};

		/* Possible values for flag in untangle_front */

enum {
	NORMAL_ATTEMPT_TO_UNTANGLE    = 1,
	DIFFICULT_ATTEMPT_TO_UNTANGLE,
	LAST_ATTEMPT_TO_UNTANGLE
};

		/* Integer flags for type of curvature at a point on a curve */

enum {
	PINCHED	= 1, /* Infinite curvature of Front */
	FLAT,	     /* No Curvature of Front */
	CURVED	     /* Finite, Non-zero Curvature */
};


		/* Classes distinguished by redistribute */

enum {
	GENERAL_WAVE = 0,
	VECTOR_WAVE,
	GENERAL_NODE
};

		/* Tracking algorithm */

enum _TRACKING_ALGORITHM {
	NO_DYNAMIC_TRACKING,
	GRID_FREE_TRACKING,
	GRID_BASED_TRACKING,
	MIXED_TRACKING,
	HYBRID_TRACKING,
        LOCALLY_GRID_BASED_TRACKING
};
typedef enum _TRACKING_ALGORITHM TRACKING_ALGORITHM;

		/* Typedefs */

struct _Tan_stencil {
	int                npts;
	POINT	 	   **p,		**pstore;
	HYPER_SURF_ELEMENT **hse,	**hsestore;
	HYPER_SURF	   **hs,	**hsstore;
	Locstate	   *leftst,	*leftststore;
	Locstate	   *rightst,	*rightststore;
	float	 	   *t,		*tstore;
	float		   curvature; /*Mean curvature*/
	COMPONENT          comp;
	HYPER_SURF         *newhs;
	Locstate           *states;
	const float        *dir;
};
typedef struct _Tan_stencil Tan_stencil;

struct _TSTEP_CONTROL {
	float	time_step_factor;
	bool	apply_cfl_at_nodes;
	float	max_sep;
	float	cfl_fudge;
	float	frac_floor;
	float	frac_ceil;
};
typedef struct _TSTEP_CONTROL TSTEP_CONTROL;

struct _AFLIN {
	float a[MAXD][MAXD];
	float b[MAXD];
	float det;
} ;
typedef struct _AFLIN AFLIN;

struct _NNLIST {
	struct _NNLIST	*prev, *next;
	CURVE		*nc[4];
	NODE    	*nn[4], *m;
	COMPONENT 	comp[4];
	float		area[4];
	ORIENTATION	orient[4];
	int     	ni[4];
	int		num_phys;
};
typedef struct _NNLIST NNLIST;

enum _REDISTRIBUTION_MODE {
	NO_REDIST        = 0,
	EXPANSION_REDIST = 1,
	FULL_REDIST      = 2
};
typedef enum _REDISTRIBUTION_MODE REDISTRIBUTION_MODE;

enum _REDISTRIBUTION_VERSION {
	ORDINARY_REDISTRIBUTE  = 1,
	EQUI_BOND_REDISTRIBUTE = 2
};
typedef enum _REDISTRIBUTION_VERSION REDISTRIBUTION_VERSION;

struct _CURVE_REDIST_OPTIONS {
	REDISTRIBUTION_VERSION _full_curve_redist_version; /* Prompted */
	bool _rect_bdry_redist;          /* Prompted */
	float   _cos_big_angle[2];          /* Prompted */
};
typedef struct _CURVE_REDIST_OPTIONS CURVE_REDIST_OPTIONS;

#if defined(THREED)
struct _SURFACE_REDIST_OPTIONS {
	float   _max_tri_area_factor[2];     /* Prompted */
	float   _min_tri_area_factor[2];     /* Prompted */
	float   _min_angle_at_tri_vertex[2]; /* Prompted */
	float   _max_scaled_tri_side_len; /* Prompted */
};
typedef struct _SURFACE_REDIST_OPTIONS SURFACE_REDIST_OPTIONS;
#endif /* defined(THREED) */


struct _REDIST_CONTROL {

	REDISTRIBUTION_MODE _Redistribution_mode;
	CURVE_REDIST_OPTIONS _CurveRedistributionOptions;
#if defined(THREED)
	SURFACE_REDIST_OPTIONS _SurfaceRedistributionOptions;
#endif /* defined(THREED) */

	/* Redistribution initializer */
	void (*_init_redistribute)(INIT_DATA*,struct _Front*);
	void (*_rect_boundary_redistribute)(INTERFACE*,RECT_GRID*,int);

	struct _CURVE_REDISTRIBUTE {
	    struct _CURVE_REDISTRIBUTORS {
	    	/* Redistribute Algorithm for curves*/
	    	bool (*curve_redist_func)(struct _Front*,bool*);

	    	/* Forward orient curve redistribute */
	    	bool (*forward_cur_redist)(struct _Front*,CURVE*,bool);

	    	/* Backward orient curve redistribute */
	    	bool (*backward_cur_redist)(struct _Front*,CURVE*,bool);

	    	/* Node redistribute */
 	    	bool (*node_redistribute)(struct _Front*,bool);
	    } Redistributors;

		/* Clean up function for interface */
	    struct _CURVE_CLEANERS {
	    	void	(*_delete_small_loops)(struct _Front*);
	    	void	(*_delete_fold_back_bonds)(struct _Front*,CURVE*,float,
                                                   int*,int*);
	    	void	(*_delete_exterior_curves)(struct _Front*,INTERFACE*);
	    	bool	(*_delete_phys_remn_on_bdry)(struct _Front*);
	    	bool (*_delete_point_adjacent_to_node)(struct _Front*,
                                                          CURVE*,ORIENTATION);
	    } Cleaners;

			/* Grid based redist. of rect bdry */
	    bool rect_bdry_redist;
	    	/* Big Angles not Truncated in Redis */
	    float cos_big_angle[2];
	    	/* Arc Length along the Front */
	    float length;
	} Curve_redist;

#if defined(THREED)
	struct _SURFACE_REDISTRIBUTE {
	    /* Redistribute Algorithm for surfaces*/
	    bool (*surface_redist_func)(struct _Front*,bool*);
	    	/* maximum length for bond in 3d curve */
	    float max_len[2];
	    	/* minimum length for bond in 3d curve */
	    float min_len[2];
	    	/* square of maximum area allowed for triangle */
	    float max_area2[2];
	    	/* square of minimum area allowed for triangle */
	    float min_area2[2];
	    	/* linear tolerance of aspect ratio     */
	    float max_sqr_tri_len;
	    	/* square of maximum scaled triangle side length */
	    float ar_tol[2];


	} Surface_redist;
#endif /* defined(THREED) */

	int count_redist;		/* Number of calls to Redistribute */
	int freq_redist[3];		/* Frequency for Redistribution */
	int freq_reconstruct;		/* Frequency for reconstruction */
					/* (for hybrid tracking) */
	float spacing[2];		/* Spacing of Points on 2d Front */
	bool intfc_redistributed;	/* Yes if redistribute was performed */
	TRACKING_ALGORITHM tracking_algorithm;	/* method of tracking */
};
typedef struct _REDIST_CONTROL REDIST_CONTROL;

#if defined(__cplusplus)

typedef struct _REDIST_CONTROL::_CURVE_REDISTRIBUTE CURVE_REDISTRIBUTE;
typedef struct _REDIST_CONTROL::_CURVE_REDISTRIBUTE::_CURVE_REDISTRIBUTORS CURVE_REDISTRIBUTORS;
typedef struct _REDIST_CONTROL::_CURVE_REDISTRIBUTE::_CURVE_CLEANERS CURVE_CLEANERS;
#if defined(THREED)
typedef struct _REDIST_CONTROL::_SURFACE_REDISTRIBUTE SURFACE_REDISTRIBUTE;
#endif /* defined(THREED) */

#else /* defined(__cplusplus) */

typedef struct _CURVE_REDISTRIBUTE CURVE_REDISTRIBUTE;
typedef struct _CURVE_REDISTRIBUTORS CURVE_REDISTRIBUTORS;
typedef struct _CURVE_CLEANERS CURVE_CLEANERS;
#if defined(THREED)
typedef struct _SURFACE_REDISTRIBUTE SURFACE_REDISTRIBUTE;
#endif /* defined(THREED) */

#endif /* defined(__cplusplus) */

struct _UNTRACK_FLAG {
	bool start_states_set;
	bool end_states_set;
	int user_untrack_flag;
};
typedef struct _UNTRACK_FLAG UNTRACK_FLAG;


struct _Front {
		/* Grid Specification */
	RECT_GRID *rect_grid;		/* Grid Info */
	PP_GRID* pp_grid;

		/* advancing the front */
	int	(*_advance_front)(float,float*,struct _Front*,
				  struct _Front**,POINTER);

		/* Copy and freeing front */
	void	(*_free_front)(struct _Front*);
	void	(*_copy_into_front)(struct _Front*,struct _Front*);
	struct _Front*	(*_copy_front)(struct _Front*);

		/* printing the front */
	void	(*_print_Front_structure)(struct _Front*);
	void	(*_fprint_front)(struct _Front*,FILE*);
	void	(*_read_print_front)(INIT_DATA*,struct _Front*);

		/* State Variable Specification */
	size_t  sizest;			/* Size of State Variables in Bytes */
	void    (*_state_interpolator)(float,float,float*,Locstate,float*,
				       Locstate,RECT_GRID*,Locstate);
	bool (*_tri_state_interpolator)(float,float,float,float*,Locstate,
					   float*,Locstate,float*,Locstate,
					   RECT_GRID*,Locstate);
	void (*transform_state)(Locstate,AFLIN*); /* Coordinate transforms
						     on states */

		/* Hyper surface corresponder */
	bool (*_is_correspondence_possible)(HYPER_SURF*,HYPER_SURF*,
					       HYPER_SURF_BDRY**,
					       HYPER_SURF_BDRY**);

		/* Redistribution Specification */
	REDIST_CONTROL  Redist;         /* Redistribution control parameters */

		/* Time Step Selection */
	bool (*_last_time_step_modification)(void);
					/* Computes front maximum time step */
	float (*max_front_time_step)(struct _Front*,float*);
	struct	_MAX_FRONT_SPEED	*_MaxFrontSpeed;
	TSTEP_CONTROL	Tstep;

		/* Real and mesh times for front->interf */
	float dt, *dt_frac, time;
	int step;

		/* Advancing the Front */
	int  hyperbolic;
	int  npts_tan_sten;
	int  movingframe;
	int  (*init_topology_of_new_interface)(struct _Front*,struct _Front*);
	struct _F_WAVE_CAPTURE *_f_wave_capture;
	void (*_init_propagate)(struct _Front*);
	void (*curve_propagate)(struct _Front*,POINTER,CURVE*,CURVE*,float);
	int  (*node_propagate)(struct _Front*,POINTER,NODE*,NODE*,RPROBLEM**,
			       float,float*,NODE_FLAG,POINTER);
	void (*_point_propagate)(struct _Front*,POINTER,POINT*,POINT*,
			         HYPER_SURF_ELEMENT*,HYPER_SURF*,float,float*);
	void (*bond_propagate)(struct _Front*,POINTER,BOND*,BOND*,
			       CURVE*,float);
	int  (*snd_node_propagate)(struct _Front*,struct _Front*,POINTER,
				   INTERFACE*,NODE*,NODE*,float);
	void (*tan_curve_propagate)(struct _Front*,struct _Front*,INTERFACE*,
				    CURVE*,CURVE*,float);
	bool (*_tan_point_propagate)(struct _Front*,POINT*,POINT*,
				     HYPER_SURF_ELEMENT*,HYPER_SURF*,float);
	void (*_npt_tang_solver)(float,float,Tan_stencil*,
	                         Locstate,Locstate,struct _Front*);
	void (*_one_side_npt_tang_solver)(float,float,Tan_stencil*,Locstate,
					  struct _Front*);
	void (*impose_bc)(POINT*,BOND*,CURVE*,float*,struct _Front*,
			  bool,bool);

#if defined(TWOD) || defined(THREED)
	bool (*_reconstruct_front_at_grid_crossing)(struct _Front*,struct _Front*,int);
#endif /* if defined(TWOD) || defined(THREED) */

#if defined(THREED)
	bool (*_untrack_surface)(SURFACE*,COMPONENT,struct _Front*);
        bool (*_repair_front_at_grid_crossing)(struct _Front*);
	void (*_principal_tangent)(POINT*,HYPER_SURF_ELEMENT*,HYPER_SURF*,
				   float*,float*);
#endif /* defined(THREED) */

		/* Untangling the Front */
	bool (*_check_delete_redundant_node)(NODE*,CURVE*,CURVE*);
	int  (*fr_bdry_untangle)(struct _Front*,CROSS**,RPROBLEM*,NODE*,int);
	int  (*fr_vec_bdry_untangle)(CURVE*,CURVE*,CURVE**,ORIENTATION,
				     ANGLE_DIRECTION,int,struct _Front*);
	int  (*untangle_front)(struct _Front*,CROSS**,int);
	bool (*_replace_unphys_loop)(NNLIST*,NNLIST**,CURVE**,
				    struct _Front*,int,float,int);
	int  (*B_node_bifurcation)(struct _Front*,POINTER,O_CURVE*,O_CURVE*,
				   O_CURVE*,O_CURVE*,O_CURVE*,O_CURVE*,O_CURVE*,
				   O_CURVE*,O_CURVE*,O_CURVE*,POINT*,Locstate,
				   Locstate,ANGLE_DIRECTION,RPROBLEM**,
				   float,float*,NODE_FLAG);
	int  (*twodrproblem)(struct _Front*,struct _Front*,POINTER,RPROBLEM**);
	bool (*_untrack_curve)(O_CURVE*,O_CURVE*,COMPONENT,float,
			          struct _Front*,POINTER,RPROBLEM*,
				  UNTRACK_FLAG);
#if defined(ONED)
	bool (*_untrack_point)(POINT*,COMPONENT,struct _Front*);
#endif /* defined(ONED) */
	void (*identify_physical_node)(NODE*);
	void (*init_2drproblem)(RPROBLEM*,struct _Front*);
	void (*phys_split_bdry_cross)(CURVE**,CURVE**);
	void (*phys_set_node_types)(struct _Front*);
	void (*parallel_refl_vec_wave)(CURVE*,int,int,CURVE*,int,int,
				       RPROBLEM*,struct _Front*,POINTER);

		/* Identification of boundary states and components */

	int  (*neumann_bdry_state)(float*,COMPONENT,POINT*,HYPER_SURF*,
				   struct _Front*,POINTER,Locstate);
	ANGLE_DIRECTION  (*_find_i_to_prop_dir)(struct _Front*,POINTER,NODE*,
				    	        CURVE*,ORIENTATION,
				    	        float,COMPONENT*,POINT*,float*);
	int  (*is_nzn_bdry)(float,float,COMPONENT,CURVE*);

		/*Locstate allocation and clearing*/

	void (*_alloc_state)(Locstate*,size_t);
	void (*_clear_state)(Locstate,size_t);
	void (*_obstacle_state)(Locstate,size_t);

		/* Output printout functions */

	int nfloats;			/* Number of floats in Locstate */
	void (*print_state)(Locstate);
	void (*_fgraph_front_states)(FILE*,struct _Front*);
	void (*_fprint_header_for_graph_curve_states)(FILE*,struct _Front*,
						      const char*);
	void (*_fgraph_curve_states)(FILE*,CURVE*,struct _Front*,float*);
	void (*mass_consv_diagn_driver)(struct _Front*,POINTER,float);

	struct	_FlowSpecifiedRegion *head_fsr;

	void (*_EnforceFlowSpecifedStates)(struct _Front*);

		/* Marker for Copy Routine */
	int FDIVIDER;

		/* (the rest of the front consists of pointers) */

	INTERFACE *interf;		/* Interface */
        INTERFACE *mesh;                /* 2D DG triangle mesh */
        float    (*_time_step_on_tri)(struct _Front*,TRI*);
        void     (*_show_state_on_tri)(const char*,int,struct _Front*);
        void     (*_rect_state_on_tri)(const char*,int,struct _Front*);
        float    (*_obtain_time_step_on_tri)(struct _Front*);

#if defined(USE_OVERTURE)

        int     (*_normal_advance_front)(float,float*,struct _Front*,
                                  struct _Front**,POINTER);
        int     (*_tangnt_advance_front)(float,float*,struct _Front*,
                                  struct _Front**,POINTER);
        int     (*_redist_advance_front)(float,float*,struct _Front*,
                                  struct _Front**,POINTER);

                /* deep Copy and freeing front, create rect_grid */
        void    (*_deep_free_front)(struct _Front*);
        struct _Front*  (*_deep_copy_front)(struct _Front*);

        int                       use_overture_state;
        POINTER                   cg_over;   /* pointer of CompositeGrid */
        int                       patch_number;
        int                       patch_level;
        int                       NumberOfLevels;
        int                       totalNumberOfPatches;
        /* Neighbor                *neighbor; */
        Patch_bdry_flag           *pd_flag;
        COMPONENT                 center_comp;
        int                       pc_ic[MAXD]; /* proc icrds, patch originally belongs to */
        struct _Front             *newfront;
#endif /* defined(USE_OVERTURE) */
};
typedef struct _Front Front;

#if defined(USE_OVERTURE)
struct _Wv_on_pc{
        int    pc_id;   /* proc id, patch of wv_id, wv_level computed
                           on this  proc. */
        int    pc_ic[MAXD]; /* proc icrds, patch originally belongs to
                             So it might be different from pc_id */
        int    wv_id;   /* patch of wv id */
        int    wv_level; /* for the reduce of communication */
        int    base[MAXD];  /* grid location index on subdomain */
        int    bound[MAXD]; /* grid location index on subdomain */
        int    lbuf[MAXD];  /* grid lower side buffer */
        int    ubuf[MAXD];  /* grid upper side buffer */
        int    off_set[MAXD]; /* location of index[0,0] offset  */
        int    rect_bdry_type[MAXD][2];
        Front  *front;
};
typedef struct _Wv_on_pc Wv_on_pc;
#endif /* if defined(USE_OVERTURE) */

#define	npt_tang_solver(ds,dt,sten,ansl,ansr,fr)			\
	(*(fr)->_npt_tang_solver)(ds,dt,sten,ansl,ansr,fr)

#define	one_side_npt_tang_solver(ds,dt,sten,ans,fr)			\
	(*(fr)->_one_side_npt_tang_solver)(ds,dt,sten,ans,fr)

#define	f_front(front)	((Front*)front)

	/*
	*  Data structure for keeping tracking of the maximum wave speed
	*  on the tracked front.
	*/

struct	_MAX_FRONT_SPEED {
	float		_spfr[MAXD+1];	/* Max front speeds in coord dirs    */
					/* spfr[dim+1] = speed in prop. dirs */
	float		**_coords;	/* Location of maximum front speeds  */
	Locstate	*_mxspst;	/* Copy of state which set the       */
					/* maxium front speed.		     */
	size_t		_sizest;
	struct _MAX_FRONT_SPEED_OPERATORS {
	    void	(*_set)(int,float,Locstate,float*,Front*);
	    void	(*_include)(struct _MAX_FRONT_SPEED*,Front*);
	    void	(*_initialize)(Front*);
	    void	(*_print)(FILE*,Front*);
	    bool	(*_read_print)(INIT_DATA*,const IO_TYPE*,Front*,
				       struct _MAX_FRONT_SPEED*);
	    void	(*_copy)(struct _MAX_FRONT_SPEED*,
				 struct _MAX_FRONT_SPEED*,Front*);
	    void	(*_destroy)(Front*);
	} operators;
};
typedef struct _MAX_FRONT_SPEED MAX_FRONT_SPEED;
#if defined(__cplusplus)
typedef struct _MAX_FRONT_SPEED::_MAX_FRONT_SPEED_OPERATORS MAX_FRONT_SPEED_OPERATORS;
#else /* defined(__cplusplus) */
typedef struct _MAX_FRONT_SPEED_OPERATORS MAX_FRONT_SPEED_OPERATORS;
#endif /* defined(__cplusplus) */

#define	MaxFrontSpeed(fr)	(fr)->_MaxFrontSpeed
#define	Spfr(fr)		MaxFrontSpeed(fr)->_spfr
#define	MaxFrontSpeedState(fr)	MaxFrontSpeed(fr)->_mxspst
#define	MaxFrontSpeedCoords(fr)	MaxFrontSpeed(fr)->_coords

#define	MaxFrontSpeedOperators(fr)	MaxFrontSpeed(fr)->operators
#define	SetMaxFrontSpeed(fr)		MaxFrontSpeedOperators(fr)._set
#define	IncludeMaxFrontSpeedInfo(fr)	MaxFrontSpeedOperators(fr)._include
#define	InitializeMaxFrontSpeed(fr)	MaxFrontSpeedOperators(fr)._initialize
#define	PrintMaxFrontSpeedInfo(fr)	MaxFrontSpeedOperators(fr)._print
#define	ReadPrintMaxFrontSpeedInfo(fr)	MaxFrontSpeedOperators(fr)._read_print
#define	CopyMaxFrontSpeed(fr)		MaxFrontSpeedOperators(fr)._copy
#define	DestroyMaxFrontSpeed(fr)	MaxFrontSpeedOperators(fr)._destroy

#define	set_max_front_speed(i,spd,st,crds,fr)				\
		(*SetMaxFrontSpeed(fr))(i,spd,st,crds,fr)
#define	include_max_front_speedInfo(mxsp,fr)				\
		(*IncludeMaxFrontSpeedInfo(fr))(mxsp,fr)
#define	initialize_max_front_speed(fr)					\
		(*InitializeMaxFrontSpeed(fr))(fr)
#define	print_max_front_speed_info(file,fr)				\
		(*PrintMaxFrontSpeedInfo(fr))(file,fr)
#define	read_print_max_front_speed_info(init,io_type,fr,mfs)		\
		(*ReadPrintMaxFrontSpeedInfo(fr))(init,io_type,fr,mfs)
#define	copy_max_front_speed(nmxsp,omxsp,fr)				\
		(*CopyMaxFrontSpeed(fr))(nmxsp,omxsp,fr)
#define	destroy_max_front_speed(fr)					\
		(*DestroyMaxFrontSpeed(fr))(fr)

struct _F_WAVE_CAPTURE {
	void (*_wave_capture)(Front*);
};
typedef struct _F_WAVE_CAPTURE F_WAVE_CAPTURE;

#define	f_wave_capture(fr)	(fr)->_f_wave_capture
#define capture_waves(fr)						\
	if ((f_wave_capture(fr) != NULL) 	&& 			\
			(f_wave_capture(fr)->_wave_capture != NULL))	\
		(*f_wave_capture(fr)->_wave_capture)(fr)


	/*
	 *  Data structures for specifying a region where the flow
	 *  is specified by a user defined function.
	 */

struct	_FlowSpecifiedRegion {

	COMPONENT	comp;	/*Component number that identifies region*/
	const char	*type;	/*String that identifies type of region*/
	struct _FlowSpecifiedRegion *next, *prev;
	struct _FlowSpecifiedRegion *head, *tail;

	bool (*_ComponentsMatch)(COMPONENT,COMPONENT,INTERFACE*);
	bool (*_SetFlowSpecifiedState)(Locstate,Locstate,float*,COMPONENT,
					  struct _FlowSpecifiedRegion*,Front*);
	void (*_fprint_FlowSpecifiedRegion_data)(FILE*,
					 struct _FlowSpecifiedRegion*,Front*);
	void (*_DestroyFlowSpecifiedRegion)(struct  _FlowSpecifiedRegion*);
};
typedef struct _FlowSpecifiedRegion FlowSpecifiedRegion;

#define	Fsr_list(front)		(front)->head_fsr->next
#define ComponentsMatch(fsr,comp1,comp2,intfc)				\
	(*(fsr)->_ComponentsMatch)(comp1,comp2,intfc)
#define	SetFlowSpecifiedState(nst,ost,coords,ocomp,fsr,fr)		\
	(*(fsr)->_SetFlowSpecifiedState)(nst,ost,coords,ocomp,fsr,fr)
#define	fprint_FlowSpecifiedRegion_data(file,fsr,fr)			\
	(*(fsr)->_fprint_FlowSpecifiedRegion_data)(file,fsr,fr)
#define	DestroyFlowSpecifiedRegion(fsr)					\
	(*(fsr)->_DestroyFlowSpecifiedRegion)(fsr)

struct	_ConstantFlowRegion {
	FlowSpecifiedRegion Fsr;
	Locstate	state;
};
typedef struct _ConstantFlowRegion ConstantFlowRegion;

	/* Macros */

#define	last_time_step_modification(fr)					\
	(((fr)->_last_time_step_modification != NULL) ?			\
		(*(fr)->_last_time_step_modification)() : NO)

#define check_delete_redundant_node(n,c1,c2,fr)                         \
	(((fr)->_check_delete_redundant_node != NULL) ?			\
		(*(fr)->_check_delete_redundant_node)(n,c1,c2) : YES)

#define find_i_to_prop_dir(fr,wave,oldn,oldc,c_orient,dt,ahead_comp,newn,V) \
	(((fr)->_find_i_to_prop_dir != NULL) ?				\
	    (*(fr)->_find_i_to_prop_dir)(fr,wave,oldn,oldc,c_orient,dt, \
					 ahead_comp,newn,V) :           \
	    (f_find_i_to_prop_dir(fr,wave,oldn,oldc,c_orient,dt,        \
				  ahead_comp,newn,V)))

#define advance_front(dt,dt_frac,front,newfront,wave)			\
	(*(front)->_advance_front)(dt,dt_frac,front,newfront,wave)

#define copy_front(fr)	(*(fr)->_copy_front)(fr)

#define copy_into_front(newfr,fr)	(*(fr)->_copy_into_front)(newfr,fr)

#define print_Front_structure(fr)	(*(fr)->_print_Front_structure)(fr)

#define fprint_front(fr,file)		(*(fr)->_fprint_front)(fr,file)

#define print_front(fr)			fprint_front(fr,stdout)

#define read_print_front(init,fr)	(*(fr)->_read_print_front)(init,fr)

#define free_front(fr)	(*(fr)->_free_front)(fr)

#if defined(USE_OVERTURE)
#define deep_copy_front(fr)     (*(fr)->_deep_copy_front)(fr)
#define deep_free_front(fr)     (*(fr)->_deep_free_front)(fr)
#endif /* if defined(USE_OVERTURE) */

#define	init_propagate(fr)						\
	if ((fr)->_init_propagate != NULL)				\
		(*(fr)->_init_propagate)(fr)

#define point_propagate(fr,wave,oldp,newp,oldhse,oldhs,dt,V)		\
	if ((fr)->_point_propagate != NULL)				\
		(*(fr)->_point_propagate)(fr,wave,oldp,newp,		\
					  Hyper_surf_element(oldhse),	\
					  Hyper_surf(oldhs),dt,V)

#define curve_propagate(fr,wave,old_curve,new_curve,dt)			\
    if ((fr)->curve_propagate != NULL)					\
	(*(fr)->curve_propagate)(fr,wave,old_curve,new_curve,dt)

#define tan_point_propagate(fr,tmpp,newp,tmphse,tmphs,dt)   		\
	(((fr)->_tan_point_propagate != NULL) ?				\
	    (*(fr)->_tan_point_propagate)(fr,tmpp,newp,		        \
					  Hyper_surf_element(tmphse),   \
					  Hyper_surf(tmphs),dt) : YES)

#if defined(THREED) || defined(TWOD)
#define	reconstruct_front_at_grid_crossing(fr,oldfr,which_grid)		\
	(*(fr)->_reconstruct_front_at_grid_crossing)(fr,oldfr,which_grid)
#endif /* if defined(THREED) || defined(TWOD) */

#if defined(THREED)
#define principal_tangent(fr,p,hse,hs,nor,t)				\
	(*(fr)->_principal_tangent)(p,hse,hs,nor,t)

#define repair_front_at_grid_crossing(fr)                               \
        (*(fr)->_repair_front_at_grid_crossing)(fr)

#define untrack_surface(s,newcmp,fr)					\
    (((fr)->_untrack_surface != NULL) ?					\
	    (*(fr)->_untrack_surface)(s,newcmp,fr) : NO)
#endif /* defined(THREED) */

#define untrack_curve(oc,ooc,newcmp,dt,fr,wave,rp,flag)			\
    (((fr)->_untrack_curve != NULL) ?	 				\
        (*(fr)->_untrack_curve)(oc,ooc,newcmp,dt,fr,(POINTER)wave,rp,flag) : NO)

#if defined(ONED)
#define	untrack_point(p,comp,fr)					\
    (((fr)->_untrack_point != NULL) ? (*(fr)->_untrack_point)(p,comp,fr) : NO)
#endif /* defined(ONED) */

#define end_of_curve(s,b,c,orient) 					\
	( (!Following_bond(b,orient))	&&			\
	((orient) == POSITIVE_ORIENTATION ? 				\
		(s) >= END_OF_CURVE((c)->interface) :			\
		(s) <= START_OF_CURVE((c)->interface)))

#define	bond_tangent_to_curve(p,b,c,orient,bdir,fr)			\
		bond_secant_to_curve(p,b,c,orient,bdir,fr,0.0)

#define	correspondence_is_possible(hs,c_hs,p_hsb,n_hsb,fr)		\
	(((fr)->_is_correspondence_possible != NULL) ?			\
		(*(fr)->_is_correspondence_possible)(			\
					Hyper_surf(hs),			\
					Hyper_surf(c_hs),		\
					(HYPER_SURF_BDRY **)p_hsb,	\
					(HYPER_SURF_BDRY **)n_hsb	\
		)							\
		:							\
		YES)

#define interpolate_states(fr,alpha,beta,coords0,s0,coords1,s1,ans)	\
	(*(fr)->_state_interpolator)(alpha,beta,coords0,s0,coords1,s1,  \
				     (fr)->rect_grid,ans)

#define tri_interpolate_states(fr,a,b,g,crds0,s0,crds1,s1,crds2,s2,ans)    \
	(*(fr)->_tri_state_interpolator)(a,b,g,crds0,s0,crds1,s1,crds2,s2, \
					 (fr)->rect_grid,ans)

	/* Untangle related macros */
#define replace_unphys_loop(nl,new_node_list,newc,fr,i,min_area,flag)	\
	(*(fr)->_replace_unphys_loop)(nl,new_node_list,newc,fr,i,min_area,flag)

	/* Macros to access front time step control */
#define Time_step_factor(fr)		       ((fr)->Tstep.time_step_factor)
#define Apply_CFL_at_nodes(fr)		       ((fr)->Tstep.apply_cfl_at_nodes)
#define Max_new_node_separation(fr)	       ((fr)->Tstep.max_sep)
#define Time_step_increase_factor(fr)	       ((fr)->Tstep.cfl_fudge)
#define Min_time_step_modification_factor(fr)  ((fr)->Tstep.frac_floor)
#define Max_time_step_modification_factor(fr)  ((fr)->Tstep.frac_ceil)

	/* Macros to access front redistribution control */

#define Redistribution_info(fr)	(fr)->Redist

#define	Curve_redistribution_info(fr)	Redistribution_info(fr).Curve_redist

#define Tracking_algorithm(fr)						\
	(Redistribution_info(fr).tracking_algorithm)

#define Interface_redistributed(fr)					\
	(Redistribution_info(fr).intfc_redistributed)

#define Redistribution_mode(fr)						\
	(Redistribution_info(fr)._Redistribution_mode)

#define CurveRedistributionOptions(fr)					\
	(Redistribution_info(fr)._CurveRedistributionOptions)

#if defined(THREED)
#define SurfaceRedistributionOptions(fr)				\
	(Redistribution_info(fr)._SurfaceRedistributionOptions)
#endif /* defined(THREED) */

#define Init_redistribution_function(fr)				\
	(Redistribution_info(fr)._init_redistribute)
#define Init_redistribution(init,fr)					\
	if (Init_redistribution_function(fr) != NULL)			\
		(*Init_redistribution_function(fr))(init,fr)

#define Curve_redistribution_function(fr)				\
	(Curve_redistribution_info(fr).Redistributors.curve_redist_func)
#define Curve_redistribute(fr,force)					\
	((Curve_redistribution_function(fr) != NULL) ?			\
			(*Curve_redistribution_function(fr))(fr,force) : YES)

#define Forward_curve_redistribute_function(fr)				\
    (Curve_redistribution_info(fr).Redistributors.forward_cur_redist)
#define Forward_curve_redistribute(fr,c,status)				\
    ((Forward_curve_redistribute_function(fr) != NULL) ?		\
         (*Forward_curve_redistribute_function(fr))(fr,c,status) : status)

#define Backward_curve_redistribute_function(fr)			\
    (Curve_redistribution_info(fr).Redistributors.backward_cur_redist)
#define Backward_curve_redistribute(fr,c,status)			\
    ((Backward_curve_redistribute_function(fr) != NULL) ?	\
	 (*Backward_curve_redistribute_function(fr))(fr,c,status) : status)

#define Node_redistribute_function(fr)					\
    (Curve_redistribution_info(fr).Redistributors.node_redistribute)
#define Node_redistribute(fr,status)					\
    ((Node_redistribute_function(fr) != NULL) ?				\
        (*Node_redistribute_function(fr))(fr,status) : status)

#define Rect_boundary_redistribute_function(fr)				\
	(Redistribution_info(fr)._rect_boundary_redistribute)
#define rect_boundary_redistribute(intfc,gr,step,fr)			\
	if (Rect_boundary_redistribute_function(fr) != NULL)		\
	    (*Rect_boundary_redistribute_function(fr))(intfc,gr,step)

#define Delete_small_loops_function(fr)					\
	(Curve_redistribution_info(fr).Cleaners._delete_small_loops)
#define delete_small_loops(fr)						\
	if (Delete_small_loops_function(fr) != NULL)			\
	    (*Delete_small_loops_function(fr))(fr)

#define Delete_fold_back_bonds_function(fr)				\
	(Curve_redistribution_info(fr).Cleaners._delete_fold_back_bonds)
#define delete_fold_back_bonds(fr,c,min_sc_sep,found,zl_c)		\
	if (Delete_fold_back_bonds_function(fr) != NULL) 		\
	 (*Delete_fold_back_bonds_function(fr))(fr,c,min_sc_sep,found,zl_c)

#define Delete_exterior_curves_function(fr)				\
	(Curve_redistribution_info(fr).Cleaners._delete_exterior_curves)
#define delete_exterior_curves(fr,intfc)				\
	if (Delete_exterior_curves_function(fr) != NULL)		\
	    (*Delete_exterior_curves_function(fr))(fr,intfc)

#define Delete_phys_remn_on_bdry_function(fr)				\
	(Curve_redistribution_info(fr).Cleaners._delete_phys_remn_on_bdry)
#define delete_phys_remn_on_bdry(fr)					\
	((Delete_phys_remn_on_bdry_function(fr) != NULL) ?		\
	 (*Delete_phys_remn_on_bdry_function(fr))(fr) : YES)

#define Delete_point_adjacent_to_node_function(fr)			\
	(Curve_redistribution_info(fr).Cleaners._delete_point_adjacent_to_node)
#define delete_point_adjacent_to_node(fr,c,orient)				\
	((Delete_point_adjacent_to_node_function(fr) != NULL) ?		\
		(*Delete_point_adjacent_to_node_function(fr))(fr,c,orient) :\
		FUNCTION_FAILED)


#define Redistribution_count(fr)					\
		(Redistribution_info(fr).count_redist)
#define Frequency_of_redistribution(fr,i) 				\
		(Redistribution_info(fr).freq_redist[i])
#define Frequency_of_reconstruction(fr)					\
		(Redistribution_info(fr).freq_reconstruct)
#define Use_rect_boundary_redistribution(fr) 				\
		(Curve_redistribution_info(fr).rect_bdry_redist)
#define Front_spacing(fr,i) 						\
		(Redistribution_info(fr).spacing[i])
#define Cosine_big_angle(fr,i)						\
		(Curve_redistribution_info(fr).cos_big_angle[i])
#define Front_length(fr)						\
		(Curve_redistribution_info(fr).length)

#define reconstruction_needed(fr)					\
			(Redistribution_count(fr) %			\
			Frequency_of_reconstruction(fr)) ?		\
		NO : YES;

#if defined(THREED)
	/* Redistribution of surfaces macros */

#define	Surface_redistribution_info(fr)	Redistribution_info(fr).Surface_redist

#define Surface_redistribution_function(fr)				\
	(Surface_redistribution_info(fr).surface_redist_func)

#define Surface_redistribute(fr,force)					\
	((Surface_redistribution_function(fr) != NULL) ?		\
			(*Surface_redistribution_function(fr))(fr,force) : YES)

#define Max_bond_len(fr,i)						\
		(Surface_redistribution_info(fr).max_len[i])

#define Min_bond_len(fr,i)						\
		(Surface_redistribution_info(fr).min_len[i])

#define Max_tri_sqr_area(fr,i)						\
		(Surface_redistribution_info(fr).max_area2[i])

#define Min_tri_sqr_area(fr,i)						\
		(Surface_redistribution_info(fr).min_area2[i])

#define	Max_scaled_tri_side_sqr_length(fr)				\
		(Surface_redistribution_info(fr).max_sqr_tri_len)

#define Aspect_ratio_tolerance(fr,i)					\
		(Surface_redistribution_info(fr).ar_tol[i])
#endif /* defined(THREED) */

	/* Macros for untrack_curve control*/

#define set_states_set_at_node_flag(flag,orient,status)			\
	switch (orient)							\
	{								\
	case POSITIVE_ORIENTATION:					\
	    (flag).start_states_set = (status);				\
	    break;							\
	case NEGATIVE_ORIENTATION:					\
	    (flag).end_states_set = (status);				\
	    break;							\
	case ORIENTATION_NOT_SET:					\
	    screen("ERROR in set_states_set_at_node_flag() macro "	\
		   "invalid orientation value\n");			\
	    clean_up(ERROR);						\
	    break;							\
	}

#define states_set_at_node(flag,orient)					 \
	(((orient)==POSITIVE_ORIENTATION) ? (flag).start_states_set :    \
	                                    (flag).end_states_set)

		/* Macros for printing */

#define fgraph_front_states(file,fr)					\
	if ((fr)->_fgraph_front_states != NULL)				\
		(*(fr)->_fgraph_front_states)((file),(fr))

#define graph_front_states(fr)	fgraph_front_states(stdout,fr)

#define fprint_header_for_graph_curve_states(file,fr,title)		\
	if ((fr)->_fprint_header_for_graph_curve_states != NULL)	\
		(*(fr)->_fprint_header_for_graph_curve_states)((file),	\
							       (fr),(title))

#define print_header_for_graph_curve_states(fr,title)			\
		fprint_header_for_graph_curve_states(stdout,(fr),(title))

#define fgraph_curve_states(file,c,fr,arclen)				\
	if ((fr)->_fgraph_curve_states != NULL)				\
	    (*(fr)->_fgraph_curve_states)((file),(c),(fr),(arclen))

#define graph_curve_states(c,fr,arclen)					\
		fgraph_curve_states(stdout,(c),(fr),(arclen))

struct _RESTART_DATA {
	const IO_TYPE   *_io_type;
	INTERFACE	*_intfc;
	RECT_GRID	_comp_grid, _top_grid;
	int		_multi_data;
	bool		_got_intfc_from_file;
	int		_time_step;	/*PROMPTED*/
	float		_time;		/*Read from restart file*/
	float		_dt_last;	/*Read from restart file*/
	float		_next_dt;	/*Read from restart file,
					  can be overridder by prompting*/
	IO_TYPE         _IO_type_store;
};
typedef struct _RESTART_DATA RESTART_DATA;

/*
*	Initialization structure for front library
*/

enum _TANGENT_METHOD {
	LINEAR_SECANT,
	LANGRANGIAN_INTERPOLANT,
	CUBIC_SPLINE,
	TANGENT_METHOD_FROM_RESTART
};
typedef enum _TANGENT_METHOD TANGENT_METHOD;

enum _NORMAL3D_METHOD {
	AREA_WEIGHTED_NORMAL,
	PLANE_FIT_NORMAL,
	NORMAL_METHOD_FROM_RESTART
};
typedef enum _NORMAL3D_METHOD NORMAL3D_METHOD;

struct _F_INIT_DATA {
	I_INIT_DATA	I_init_data;

	RESTART_DATA	_Restart_data;

	size_t		_StateSize;
	int		_NumberFloats;

	TANGENT_METHOD	_tangent_method;
	NORMAL3D_METHOD _normal3d_method;

	int    _redistribution_count;        /* Prompted or from restart file */
	int    _redistribution_frequency[3]; /* Prompted */
	int    _reconstruction_frequency;    /* Prompted */
	float  _front_spacing[2];            /* Prompted */
	TRACKING_ALGORITHM _tracking_algorithm; /* Prompted for 3d and conservative 2d tracking */
	REDISTRIBUTION_MODE _front_redist_mode; /* Prompted */

	TSTEP_CONTROL _Tstep;

	void (*_set_redistribution_defaults)(INIT_DATA*);
	void (*_copy_redistribution_values)(INIT_DATA*,Front*);
	void (*_prompt_for_redistribute)(INIT_DATA*);

	CURVE_REDIST_OPTIONS _curve_redist_options;

#if defined(THREED)
	SURFACE_REDIST_OPTIONS _surface_redist_options;
#endif /* defined(THREED) */

	MAX_FRONT_SPEED	*_MaxFrontSpeed;

	bool _EnforceFlowSpecifedStates;
	bool	(*_read_print_FlowSpecifiedRegion_data)(INIT_DATA*,
	                                                const IO_TYPE*,
				struct _FlowSpecifiedRegion**);
	void    (*_prompt_for_front_options)(INIT_DATA*,Front*);
	void    (*_read_print_front_options)(INIT_DATA*,Front*);
	int  movingframe;
};
typedef struct _F_INIT_DATA F_INIT_DATA;

#define	f_init_data(init)	((F_INIT_DATA*)(init))
#define	Comp_grid(init)		Computational_grid(i_intfc(init))

/* Macros for restart initialization*/
#define	Restart_data(init)		f_init_data(init)->_Restart_data
#define	restart_intfc(init)		Restart_data(init)._intfc
#define	restart_io_type(init)		Restart_data(init)._io_type
#define	Restart_comp_grid(init)		Restart_data(init)._comp_grid
#define	Restart_top_grid(init)		Restart_data(init)._top_grid
#define	restart_multi_data(init)	Restart_data(init)._multi_data
#define	got_intfc_from_file(init)	Restart_data(init)._got_intfc_from_file
#define	restart_time_step(init)		Restart_data(init)._time_step
#define	restart_time(init)		Restart_data(init)._time
#define	restart_dt_last(init)		Restart_data(init)._dt_last
#define	restart_next_dt(init)		Restart_data(init)._next_dt
#define	restart_IO_type_store(init)	Restart_data(init)._IO_type_store

#define	InitialMaxFrontSpeed(init)	f_init_data(init)->_MaxFrontSpeed
#define	initial_read_print_max_front_speed_info(init,front)		\
    (*ReadPrintMaxFrontSpeedInfo(f_init_data(init)))			\
	(init,restart_io_type(init),front,InitialMaxFrontSpeed(init))

#define StateSize(init)		f_init_data(init)->_StateSize
#define NumberFloats(init)	f_init_data(init)->_NumberFloats

#define tangent_method(init)      	f_init_data(init)->_tangent_method
#define movingframe(init)               f_init_data(init)->movingframe
#define normal3d_method(init)      	f_init_data(init)->_normal3d_method

/* Tracking method macros,  3D only */
#define	tracking_algorithm(init)   (f_init_data(init)->_tracking_algorithm)

#define	time_step_control_options(init)	f_init_data(init)->_Tstep

/* Redistribution macros for initialization */
#define redistribution_count(init) (f_init_data(init)->_redistribution_count)
#define redistribution_frequency(init,i)   				\
    (f_init_data(init)->_redistribution_frequency[i])
#define reconstruction_frequency(init)					\
    (f_init_data(init)->_reconstruction_frequency)
#define front_spacing(init,i) (f_init_data(init)->_front_spacing[i])
#define front_redist_mode(init)    f_init_data(init)->_front_redist_mode
#define	curve_redist_options(init) f_init_data(init)->_curve_redist_options
#define full_curve_redist_version(init)					\
    (curve_redist_options(init)._full_curve_redist_version)
#define use_rect_bdry_redistribution(init)				\
    (curve_redist_options(init)._rect_bdry_redist)
#define cosine_big_angle(init,i)					\
    (curve_redist_options(init)._cos_big_angle[i])
#define set_redistribution_defaults(init)   				\
    (*(f_init_data(init)->_set_redistribution_defaults))(init)
#define copy_redistribution_values(init,front)				\
    (*(f_init_data(init)->_copy_redistribution_values))(init,front)
#define prompt_for_redistribute(init)   				\
    (*(f_init_data(init)->_prompt_for_redistribute))(init)
#define surface_redist_options(init)					\
    f_init_data(init)->_surface_redist_options
#define maximum_triangle_area_factor(init,i)				\
    (surface_redist_options(init)._max_tri_area_factor[i])
#define minimum_triangle_area_factor(init,i)				\
    (surface_redist_options(init)._min_tri_area_factor[i])
#define minimum_angle_at_triangle_vertex(init,i)			\
    (surface_redist_options(init)._min_angle_at_tri_vertex[i])
#define maximum_scaled_triangle_side_length(init)			\
    (surface_redist_options(init)._max_scaled_tri_side_len)
#define	read_print_FlowSpecifiedRegion_data(init,iot,pfsr)		\
    (*f_init_data(init)->_read_print_FlowSpecifiedRegion_data)(init,iot,pfsr)
#define	read_print_front_options(init,front)				\
    (*f_init_data(init)->_read_print_front_options)(init,front)
#define	prompt_for_front_options(init,front)				\
    (*f_init_data(init)->_prompt_for_front_options)(init,front)
#define enforce_flow_specified_states(init)				\
    (f_init_data(init)->_EnforceFlowSpecifedStates)

#include <front/fuserint.h>
#include <front/fprotos.h>
#endif /* !defined(_FDECS_H) */
