/*
*				fbdry3.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	This file contains routines to determine, impose or untangle
*	front boundaries and boundary conditions. Specifically:
*			curve_exits_parallel_to_bdry()
*/

#if defined(TWOD)

#include <front/fdecs.h>

	/* LOCAL function prototypes */
LOCAL	bool	correct_exterior_curve(CURVE*,Front*);
LOCAL	void	find_bdry_curves_at_node(NODE*,CURVE**,ORIENTATION*,
					 CURVE**,ORIENTATION*);

/*
*		find_bdry_curves_at_node():
*/

LOCAL	void find_bdry_curves_at_node(
	NODE		*node,
	CURVE		**bc1,
	ORIENTATION	*bc1_orient,
	CURVE		**bc2,
	ORIENTATION	*bc2_orient)
{
	CURVE		**c;

	debug_print("2drp","Entered find_bdry_curves_at_node()\n");
	*bc1 = *bc2 = NULL;
	for (c = node->in_curves; c && *c; c++) 
	{
		if (wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE
		 || wave_type(*c) == PASSIVE_BOUNDARY)
			continue;
		else if (!*bc1) 
		{
			*bc1 = *c;
			*bc1_orient = NEGATIVE_ORIENTATION;
		}
		else if (!*bc2) 
		{
			*bc2 = *c;
			*bc2_orient = NEGATIVE_ORIENTATION;
		}
	}
	for (c = node->out_curves; c && *c; c++) 
	{
		if (wave_type(*c) >= FIRST_PHYSICS_WAVE_TYPE
		 || wave_type(*c) == PASSIVE_BOUNDARY)
			continue;
		else if (!*bc1) 
		{
			*bc1 = *c;
			*bc1_orient = POSITIVE_ORIENTATION;
		}
		else if (!*bc2) 
		{
			*bc2 = *c;
			*bc2_orient = POSITIVE_ORIENTATION;
		}
	}
	debug_print("2drp","Left find_bdry_curves_at_node()\n");
}		/*end find_bdry_curves_at_node*/

/*
*			correct_for_exterior_curves():
*
*	This routine searches for curves of the following form which
*	may become "inverted through the boundary".
*
*	     -------->---------
*	    /		       \
*	---*--------------------*---   ->  ---*----------------------*---  
*					       \                    /
*					        ---------->---------
*
*	This function assumes that fr->interf is untangled.
*/


EXPORT	bool correct_for_exterior_curves(
	Front		*fr)
{
	INTERFACE	*intfc = fr->interf;
	RECT_GRID	*rgr   = fr->rect_grid;

	COMPONENT   extcomp;
	CURVE	    **cc, *c;
	NODE	    *ns, *ne;
	BOND	    *b;
	POINT	    *p1;
	bool	    exterior_points, interior_points;
	int	    i, dim = rgr->dim;
	float	    lo[MAXD], hi[MAXD], tol;
	static const float EXT_TOLFAC = 0.0001;	/* TOLERANCE */

	debug_print("c_ext","Entered correct_for_exterior_curves()\n");

	for (i = 0; i < dim; i++)
	{
	    tol = EXT_TOLFAC * rgr->h[i];
	    lo[i]  = (rgr->lbuf[i] == 0) ? rgr->L[i] - tol : -HUGE_VAL;
	    hi[i]  = (rgr->ubuf[i] == 0) ? rgr->U[i] + tol : HUGE_VAL;
	}


	extcomp = exterior_component(intfc);
	if( debugging("c_ext") )
	{
	    (void) printf("correct_for_exterior_curves(): extcomp %d\n",
	    	      extcomp);
	    print_general_vector("lo = ",lo,dim,"\n");
	    print_general_vector("hi = ",hi,dim,"\n");
	    (void) printf("input interface -\n");
	    print_interface(intfc);
	}

	for( cc = intfc->curves;  cc && *cc;  cc++ )
	{
	    c = *cc;

	    if ((is_bdry_like_curve(c)) ||
		(wave_type(c)<FIRST_PHYSICS_WAVE_TYPE))
		continue;
	    ns = c->start;		ne = c->end;

	    if ((!is_bdry_like_node(ns) ) || (!is_bdry_like_node(ne)))
	    	continue;

	    exterior_points = NO;
	    interior_points = NO;
	    if( c->first == c->last )
	    	exterior_points = YES;
	    else
	    {
	        for( b = c->first;  b != c->last;  b = b->next )
	        {
	            p1 = b->end;
	            if (outside_point(Coords(p1),lo,hi,dim))
	                exterior_points = YES;
		    else
			interior_points = YES;
	        }
	    }

	    if ((exterior_points == NO) || (interior_points == YES))
	        continue;

	    /* Curve is exterior */

	    if( debugging("c_ext") )
	    {
	    	(void) printf("Nodes of exterior curve\n");

	    	(void) printf("ns -\n");	print_node(ns);
	    	(void) printf("ne -\n");	print_node(ne);
		print_curve(c);
	    }

	    if (!correct_exterior_curve(c,fr))
		return NO;

	}
	if( debugging("c_ext") )
	{
	    (void) printf("after correct_for_exterior_curves(), "
	                  "input interface -\n");
	    print_interface(intfc);
	}
	debug_print("c_ext","Left correct_for_exterior_curves()\n");
	return YES;
}		/*end correct_for_exterior_curves*/

LOCAL	bool correct_exterior_curve(
	CURVE		*c,
	Front		*fr)
{
	INTERFACE	*intfc = fr->interf;
	NODE		*ns = c->start, *ne = c->end;
	CURVE		*nsc1, *nsc2, *nec1, *nec2, *bc;
	ORIENTATION	nsc1_or, nsc2_or, nec1_or, nec2_or;
	bool		save_swtch;

	debug_print("c_ext_cur","Entered correct_exterior_curve()\n");

	if (debugging("c_ext_cur"))
	{
	    (void) printf("Correcting for exterior curve %llu\n",
	    	          curve_number(c));
	    print_curve(c);
	}
	find_bdry_curves_at_node(ns,&nsc1,&nsc1_or,&nsc2,&nsc2_or);
	find_bdry_curves_at_node(ne,&nec1,&nec1_or,&nec2,&nec2_or);
	if (debugging("c_ext_cur"))
	{
	    (void) printf("nsc1 = %llu, ",curve_number(nsc1));
	    print_orientation("nsc1_or = ",nsc1_or,"\n");
	    (void) printf("nsc2 = %llu, ",curve_number(nsc2));
	    print_orientation("nsc2_or = ",nsc2_or,"\n");
	    (void) printf("nec1 = %llu, ",curve_number(nec1));
	    print_orientation("nec1_or = ",nec1_or,"\n");
	    (void) printf("nec2 = %llu, ",curve_number(nec2));
	    print_orientation("nec2_or = ",nec2_or,"\n");
	}
	bc = NULL;
	if ((nsc1 == nec1) || (nsc1 == nec2))
	    bc = nsc1; 
	else if ((nsc2 == nec1) || (nsc2 == nec2))
	    bc = nsc2;
	else
	{
	    if (debugging("c_ext_cur"))
	    {
	        (void) printf("Unable to identify boundary curve bc\n"
	                      "Opposite nodes of exterior curve %llu\n",
	    		      curve_number(c));
	        (void) printf("do not share a common boundary curve\n");
	    }
	    debug_print("c_ext_cur","Left correct_exterior_curve()\n");
	    return YES;
	}

	if( bc == NULL )
	{
	    (void) printf("ERROR in correct_for_exterior_curves(), "
	           "Unable to identify boundary curve bc "
	           "at curve %llu\n",curve_number(c));
	    print_interface(intfc);
	    debug_print("c_ext_cur","Left correct_exterior_curve()\n");
	    return NO;
	}

	if (debugging("c_ext_cur"))
	{
	    (void) printf("Boundary curve bc %llu at exterior curve\n",
	    	          curve_number(bc));
	    print_curve(bc);
	}

	save_swtch = interpolate_intfc_states(intfc);
	interpolate_intfc_states(intfc) = YES;
	switch ( wave_type(bc)) 
	{
	case NEUMANN_BOUNDARY:
	    if (debugging("c_ext_cur"))
	    	(void) printf("NEUMANN_BOUNDARY\n");
	    if( wave_type(c) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE )
	    {
	    	(void) printf("WARNING in correct_exterior_curve(), ");
	    	(void) printf("Boundary reflection needed ");
	    	if (debugging("c_ext_cur"))
	    	{
	    	    print_curve(c);
	    	    print_interface(c->interface);
	    	}
	    	return NO;
	    }
	    else
	    {
	    	replace_cphys_by_cbdry(c,bc,fr);
	    	ns = bc->start;		ne = bc->end;
	    	(void) delete_redundant_node(ns,NULL,NULL,fr);
	    	(void) delete_redundant_node(ne,NULL,NULL,fr);
	    }
	    break;
	case DIRICHLET_BOUNDARY:
	    if (debugging("c_ext_cur"))
	    	(void) printf("%s\n","DIRICHLET_BOUNDARY");
	    replace_cphys_by_cbdry(c,bc,fr);
	    ns = bc->start;		ne = bc->end;
	    (void) delete_redundant_node(ns,NULL,NULL,fr);
	    (void) delete_redundant_node(ne,NULL,NULL,fr);
	    break;
	case SUBDOMAIN_BOUNDARY:
	    if (debugging("c_ext_cur"))
	    	(void) printf("%s\n","SUBDOMAIN_BOUNDARY");
	    break;
	}
	interpolate_intfc_states(intfc) = save_swtch;
	debug_print("c_ext_cur","Left correct_exterior_curve()\n");
	return YES;
}		/*end correct_exterior_curve*/


EXPORT	void replace_cphys_by_cbdry(
	CURVE		*cp,
	CURVE		*cb,
	Front		*fr)
{
	BOND	  	*b;
	POINT		*pnew;
	RECT_GRID 	*rgr = fr->rect_grid;
	float		*cbs, t[MAXD], v[MAXD], magt;
	float		l, llast;
	int		flag;
	size_t		sizest = fr->sizest;
	int		i, dim = rgr->dim;
	SIDE		phys_side_bc;
	bool		sav_interp = interpolate_intfc_states(cb->interface);

	debug_print("rcbc","Entered replace_cphys_by_cbdry()\n");
	if (is_subdomain_boundary(Hyper_surf(cb)))
	{
	    debug_print("rcbc","Left replace_cphys_by_cbdry()\n");
	    return;
	}

	cbs = Coords(cb->start->posn);
	for (i = 0; i < dim; i++)
	    t[i] = Coords(cb->end->posn)[i] - cbs[i];
	magt = mag_vector(t,dim);
	interpolate_intfc_states(cb->interface) = NO;

	phys_side_bc = is_excluded_comp(negative_component(cb),cb->interface) ?
				POSITIVE_SIDE : NEGATIVE_SIDE;

	if( debugging("rcbc") )
	{
	    (void) printf("Replacing curve cp %llu by curve cb %llu\n",
	    	          curve_number(cp),curve_number(cb));
	}
	if( cp->start == cb->start )
	    flag = (phys_side_bc == POSITIVE_SIDE) ? 0 : 1;
	else
	    flag = (phys_side_bc == POSITIVE_SIDE) ? 2 : 3;
	if (wave_type(cb) == DIRICHLET_BOUNDARY)
	    flag += 4;

	switch (flag%4)
	{
	case 0:
	    positive_component(cb) = positive_component(cp);
	    assign(right_start_state(cb),right_start_state(cp),sizest);
	    assign(right_end_state(cb),right_end_state(cp),sizest);
	    if (flag/4)
	    {
	        assign(left_start_state(cb),right_start_state(cp),sizest);
	        assign(left_end_state(cb),right_end_state(cp),sizest);
	    }
	    break;
	case 1:
	    negative_component(cb) = negative_component(cp);
	    assign(left_start_state(cb),left_start_state(cp),sizest);
	    assign(left_end_state(cb),left_end_state(cp),sizest);
	    if (flag/4)
	    {
	        assign(right_start_state(cb),left_start_state(cp),sizest);
	        assign(right_end_state(cb),left_end_state(cp),sizest);
	    }
	    break;
	case 2:
	    positive_component(cb) = negative_component(cp);
	    assign(right_start_state(cb),left_end_state(cp),sizest);
	    assign(right_end_state(cb),left_start_state(cp),sizest);
	    if (flag/4)
	    {
	        assign(left_start_state(cb),left_end_state(cp),sizest);
	        assign(left_end_state(cb),left_start_state(cp),sizest);
	    }
	    break;
	case 3:
	    negative_component(cb) = positive_component(cp);
	    assign(left_start_state(cb),right_end_state(cp),sizest);
	    assign(left_end_state(cb),right_start_state(cp),sizest);
	    if (flag/4)
	    {
	        assign(right_start_state(cb),right_end_state(cp),sizest);
	        assign(right_end_state(cb),right_start_state(cp),sizest);
	    }
	    break;
	}

	delete_interior_points_of_curve(fr,cb);

	if (magt > MIN_SCALED_LENGTH(fr->interf))
	{
	    for (i = 0; i < dim; i++)
		t[i] /= magt;
	    llast = 0.0;
	    for( b = cp->first;  b != cp->last;  b = b->next )
	    {
	    	for (i = 0; i < dim; i++)
	    	    v[i] = Coords(b->end)[i] - cbs[i];
	    	l = scalar_product(v,t,dim);
	    	if (l < llast)
	    	{
	    	    delete_interior_points_of_curve(fr,cb);
	    	    break;
	    	}
		if (l >= magt)
		    break;
	    	pnew = Point(NULL);
	    	for (i = 0; i < dim; i++)
	    	    Coords(pnew)[i] = cbs[i] + l*t[i];
	        if( debugging("rcbc") )
		{
		    print_general_vector("v = ",v,dim,"\n");
		    (void) printf("mag(v) = %g\n",mag_vector(v,dim));
		    (void) printf("magt = %g, l = %g\n",magt,l);
		    print_general_vector("inserting pnew at ",
					 Coords(pnew),dim,"\n");
		}
	    	if (insert_point_in_bond(pnew,cb->last,cb)!=FUNCTION_SUCCEEDED)
		{
		    screen("ERROR in replace_cphys_by_cbdry(), "
		           "insert_point_in_bond() failed\n");
		    clean_up(ERROR);
		}
	    	switch (flag%4)
	    	{
	    	case 0:
	    	    assign(right_state(pnew),right_state(b->end),sizest);
		    if (flag/4)
	    	        assign(left_state(pnew),right_state(b->end),sizest);
	    	    break;
	    	case 1:
	    	    assign(left_state(pnew),left_state(b->end),sizest);
		    if (flag/4)
	    	        assign(right_state(pnew),left_state(b->end),sizest);
	    	    break;
	    	case 2:
	    	    assign(right_state(pnew),left_state(b->end),sizest);
		    if (flag/4)
	    	        assign(left_state(pnew),left_state(b->end),sizest);
	    	    break;
	    	case 3:
	    	    assign(left_state(pnew),right_state(b->end),sizest);
		    if (flag/4)
	    	        assign(right_state(pnew),right_state(b->end),sizest);
	    	    break;
	    	}
	    	llast = l;
	    }
	}

	(void) delete_curve(cp);

	if( debugging("rcbc") )
	{
	    (void) printf("Final curve\n");
	    (void) printf("cb -\n"); print_curve(cb);
	    show_curve_states(cb);
	}
	interpolate_intfc_states(cb->interface) = sav_interp;
	debug_print("rcbc","Left replace_cphys_by_cbdry()\n");
}		/*end replace_cphys_by_cbdry*/

/*
*			curve_exits_parallel_to_bdry():
*
*	Performs the interaction which occurs when a curve exits parallel
*	to a boundary curve.  The function currently only supports
*	Dirichlet boundaries.
*/


EXPORT int curve_exits_parallel_to_bdry(
	Front		*front,
	POINTER		wave,
	RPROBLEM	*rp)
{
	RP_NODE		*rpn[2], *rp_node;
	NODE		*ns, *ne;
	O_CURVE		*ocbehind[2];
	CURVE		*null_bc[2];
	CURVE		*cphys, *cbdry, *cbehind[2];
	CURVE		*oldcphys, *oldcbdry, *oldcbehind[2];
	float		t[MAXD], tb[MAXD], cp, V[MAXD];
	float		dt = rp->dt;
	ORIENTATION	cphys_orient, cbdry_orient, cbehind_orient[2];
	SIDE		int_side, bdry_int_side;
	ORIENTATION	null_bc_or[2];
	int		dim = front->rect_grid->dim;
	int		i;
	static	POINT	*pnew = NULL;

	if (pnew == NULL)
	{
	    pnew = Static_point(rp->new_intfc);
	}

	debug_print("parallel","Entered curve_exits_parallel_to_bdry()\n");
	if (debugging("parallel"))
	{
	    (void) printf("Rproblems\n");
	    (void) printf("rp\n");
	    print_rproblem(rp);
	}
	ocbehind[0] = rp->bdry_curves->first;
	if (!rp_node_with_node(&rpn[0],rp,Node_of_o_curve(ocbehind[0])))
	{
	    screen("ERROR in curve_exits_parallel_to_bdry() "
	           "rp_node_with_node() failed\n");
	    clean_up(ERROR);
	}
	if (rpn[0] == NULL)
	{
	    (void) printf("WARNING in curve_exits_parallel_to_bdry(), "
	                  "Unable to find rpn[0]\n");
	    debug_print("parallel","Left curve_exits_parallel_to_bdry()\n");
	    return ERROR_IN_STEP;
	}
	cphys = NULL;
	if ((cphys = find_physical_curve_at_node(rpn[0]->node,&cphys_orient))
								== NULL)
	{
	    (void) printf("WARNING in curve_exits_parallel_to_bdry(), "
	                  "Unable to find physical curve\n");
	    debug_print("parallel","Left curve_exits_parallel_to_bdry()\n");
	    return ERROR_IN_STEP;
	}
	if (!next_boundary(ocbehind[0]->curve,ocbehind[0]->orient,
		              null_bc,null_bc_or)) 
	{
	    (void) printf("WARNING in curve_exits_parallel_to_bdry(), "
	                  "next_boundary() failed\n");
	    debug_print("parallel","Left curve_exits_parallel_to_bdry()\n");
	    return ERROR_IN_STEP;
	}
	if (!next_boundary(null_bc[0],Opposite_orient(null_bc_or[0]),
		&cbdry,&cbdry_orient)) 
	{
	    (void) printf("ERROR in curve_exits_parallel_to_bdry(), "
	                  "next_boundary() failed\n");
	    debug_print("parallel","Left curve_exits_parallel_to_bdry()\n");
	    return ERROR_IN_STEP;
	}
	ocbehind[1] = rp->bdry_curves->last;
	if (!rp_node_with_node(&rpn[1],rp,Node_of_o_curve(ocbehind[1])))
	{
	    (void) printf("WARNING in curve_exits_parallel_to_bdry(), "
	                  "Unable to find rpn[1]\n");
	    debug_print("parallel","Left curve_exits_parallel_to_bdry()\n");
	    return ERROR_IN_STEP;
	}
	if (!next_boundary(ocbehind[1]->curve,ocbehind[1]->orient,
		null_bc+1,null_bc_or+1)) 
	{
	    (void) printf("WARNING in curve_exits_parallel_to_bdry(), "
	                  "next_boundary() failed\n");
	    debug_print("parallel","Left curve_exits_parallel_to_bdry()\n");
	    return ERROR_IN_STEP;
	}


	if (debugging("parallel"))
	{
	    (void) printf("Exiting physical curve, orient = %s\n",
	                  orientation_name(cphys_orient));
	    print_curve(cphys);
	    (void) printf("Boundary curve, orient = %s\n",
	                  orientation_name(cbdry_orient));
	    print_curve(cbdry);
	}
	
	/* Find corresponding curves on old interface */

	if (cphys_orient == POSITIVE_ORIENTATION)
	{
	    ns = rpn[0]->old_node;	ne = rpn[1]->old_node;
	}
	else
	{
	    ns = rpn[1]->old_node;	ne = rpn[0]->old_node;
	}
	oldcphys = find_correspond_curve(cphys,ns,ne,front,rp->old_intfc);

	ns = ne = NULL;
	if (rp_node_with_node(&rp_node,rp,cbdry->start))
	    ns = rp_node->old_node;
	if (rp_node_with_node(&rp_node,rp,cbdry->end))
	    ne = rp_node->old_node;
	oldcbdry = find_correspond_curve(cbdry,ns,ne,front,rp->old_intfc);

	for (i = 0; i < 2; i++)
	{
	    ns = ne = NULL;
	    cbehind[i] = ocbehind[i]->curve;
	    cbehind_orient[i] = ocbehind[i]->orient;
	    if (cbehind_orient[i] == POSITIVE_ORIENTATION)
	    	ns = rpn[i]->old_node;
	    else
	    	ne = rpn[i]->old_node;
	    oldcbehind[i] = find_correspond_curve(cbehind[i],ns,ne,front,
					rp->old_intfc);
	}
	if (oldcphys == NULL || oldcbdry == NULL || oldcbehind[0] == NULL
	 || oldcbehind[1] == NULL)
	{
	    (void) printf("WARNING in curve_exits_parallel_to_bdry(), "
	                  "Unable to find old curves\n");
	    debug_print("parallel","Left curve_exits_parallel_to_bdry()\n");
	    return ERROR_IN_STEP;
	}

	/* Propagate old curves */

	point_propagate(front,wave,oldcphys->start->posn,pnew, 
		oldcphys->first,oldcphys,dt,V);
	assign(left_start_state(cphys),left_state(pnew),front->sizest);
	assign(right_start_state(cphys),right_state(pnew),front->sizest);
	point_propagate(front,wave,oldcphys->end->posn,pnew, 
		oldcphys->last,oldcphys,dt,V);
	assign(left_end_state(cphys),left_state(pnew),front->sizest);
	assign(right_end_state(cphys),right_state(pnew),front->sizest);
	
	
	/* Find direction in which curve exits */

	find_tangent_to_curve(Node_of(oldcphys,cphys_orient)->posn,
		Bond_at_node(oldcphys,cphys_orient),oldcphys,
		cphys_orient,t,front);
	find_tangent_to_curve(Node_of(oldcbehind[0],cbehind_orient[0])->posn,
		Bond_at_node(oldcbehind[0],cbehind_orient[0]),oldcbehind[0],
		cbehind_orient[0],tb,front);
	(void) vector_product(t,tb,&cp,dim);
	if ((cp > 0.0 && cphys_orient == POSITIVE_ORIENTATION) ||
	    (cp < 0.0 && cphys_orient == NEGATIVE_ORIENTATION))
	{
	    int_side = NEGATIVE_SIDE;
	}
	else
	{
	    int_side = POSITIVE_SIDE;
	}
	find_tangent_to_curve(Node_of(oldcbdry,cbdry_orient)->posn,
		Bond_at_node(oldcbdry,cbdry_orient),oldcbdry,
		cbdry_orient,t,front);
	(void) vector_product(t,tb,&cp,dim);
	if ((cp > 0.0 && cbdry_orient == POSITIVE_ORIENTATION) ||
	    (cp < 0.0 && cbdry_orient == NEGATIVE_ORIENTATION))
	{
	    bdry_int_side = NEGATIVE_SIDE;
	}
	else
	{
	    bdry_int_side = POSITIVE_SIDE;
	}

	if (wave_type(cbdry) == NEUMANN_BOUNDARY &&
		wave_type(cphys) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE &&
		front->parallel_refl_vec_wave != NULL)
	{
	    (*front->parallel_refl_vec_wave)(cphys,cphys_orient,
			int_side,cbdry,cbdry_orient,bdry_int_side,
			rp,front,wave);
	}
	else
	{
	    NODE *ntmp;
	    map_phys_cur_states_to_bdry_cur(cphys,cphys_orient,
			int_side,cbdry,cbdry_orient,bdry_int_side,
			YES,YES,rp->new_intfc,front);

	    (void) delete_curve(cphys);
	    (void) delete_curve(null_bc[0]);
	    ntmp = Node_of_o_curve(ocbehind[0]);
	    change_node_of_curve(ocbehind[0]->curve,ocbehind[0]->orient,
			Node_of(cbdry,cbdry_orient));
	    (void) delete_node(ntmp);
	    (void) delete_curve(null_bc[1]);
	    ntmp = Node_of_o_curve(ocbehind[1]);
	    change_node_of_curve(ocbehind[1]->curve,ocbehind[1]->orient,
			Node_of(cbdry,Opposite_orient(cbdry_orient)));
	    (void) delete_node(ntmp);
	}
	debug_print("parallel","Left curve_exits_parallel_to_bdry()\n");
	return GOOD_STEP;
}		/*end curve_exits_parallel_to_bdry*/
#endif /* defined(TWOD) */
