/*
*                               tricrx2d.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*/

#if defined(TWOD) 

#define DEBUG_STRING "crx_intfc"

#include <tri/trilocaldecs.h>

#if defined(CONSERVATIVE_ALG) 

LOCAL void        set_aug_comp_grids(RECT_GRID*,RECT_GRID*);
LOCAL int         **set_node_index_list_on_grid(TRI_GRID*,RECT_GRID*);
LOCAL void        set_interface_topology_on_grid(INTERFACE*,RECT_GRID*);
LOCAL void        set_comp_at_crossings2d1(TRI_GRID*,int);
LOCAL int         point_on_bond2d1(POINT*,BOND*);
LOCAL bool        remove_unphysical_crossings2d(INTERFACE*,TRI_GRID*,int*,
                                              int*,int*,int);
LOCAL bool        track_comp_through_crxings2d(int*,int*,int*,TRI_GRID*,CRX_TYPE,int);
LOCAL void        eliminate_same_crossings2d(TRI_GRID*,int*,int*,int);
LOCAL void        adjust_for_min_spacing2d(CRXING*,float*,int*,float*,int,
                     int,int,int**,CRXING*);
LOCAL int         walk_comp_along_grid_line2d(TRI_GRID*,int*,int*,int*,int*,int,int);
LOCAL void        rm_unphy_crx_along_grid_line2d(TRI_GRID*,int*,int*,int*,int*,
                         GRID_DIRECTION,CRX_TYPE,int);
LOCAL void        rm_select_even_crx(int,int*,int**,CRXING*);
LOCAL void        rm_select_odd_crx(int,int*,int**,CRXING*);
LOCAL void        rm_same_hypsurf_multi_crx(int,int*,int**,CRXING*);
LOCAL bool        check_grid_comp_and_crx2d(TRI_GRID*,int*,int*,int);
LOCAL int         check_comp_at2d(int*,GRID_DIRECTION,TRI_GRID*,int*,int*,int);
/***
LOCAL int         multi_crxings_check(TRI_GRID*,int);
***/
LOCAL void        _assign_2dblk_type(BLK_CRX2*,int,BBI_POINT  *crxs[]);
LOCAL int         _trk_wv_val(CURVE*);
LOCAL bool        reconstruct_crx_intfc2d(TRI_GRID*,int,int*,int*,int);
LOCAL void        remove_intfc_crx_flag(INTERFACE*);


/* NOTE for the aug_comp_grid */
/* The GL, and GU are not used to save 
 * The true global domain. Instead, 
 * It is set to be: aug_comp_grid->G[L,U] = comp_grid->V[L,U]; 
 * This value is used in shift_from_cell_edge_on_comp(). 
 */
LOCAL void set_aug_comp_grids(
        RECT_GRID       *aug_comp_grid,
        RECT_GRID       *comp_grid)
{
        int       gmax[MAXD];
        int       i, dim;        
        float     h[MAXD], VL[MAXD], VU[MAXD];
        float     GL[MAXD], GU[MAXD]; 
        
        dim = comp_grid->dim;
        for (i = 0; i < dim; i++)
        {
            gmax[i] = comp_grid->gmax[i] + comp_grid->lbuf[i] 
                    + comp_grid->ubuf[i] + 2;
            h[i] = comp_grid->h[i];
            VL[i] = comp_grid->VL[i];
            VU[i] = comp_grid->VU[i];
            GL[i] = comp_grid->VL[i];
            GU[i] = comp_grid->VU[i];
            VU[i] += h[i];
            VL[i] -= h[i];
        }
        set_rect_grid(VL, VU, GL, GU, NOBUF, NOBUF, gmax, dim, 
                      &comp_grid->Remap, aug_comp_grid); 
}

/* */
LIB_LOCAL void set_tri_grid_aug_comp_grids(
        TRI_GRID        *grid,
        RECT_GRID       *comp_grid)
{
        int       gmax[MAXD]; 
        int       i, dim = comp_grid->dim;

        set_aug_comp_grids(&grid->aug_comp_grid, comp_grid);
        if (! set_grid_lines(&(grid->aug_comp_grid)))
        {
            screen("ERROR in set_tri_grid_aug_comp_grids(), ");
            screen("set_grid_lines() failed\n");
            clean_up(ERROR);
        }        
        for (i = 0; i < dim; i++)
            gmax[i] = grid->aug_comp_grid.gmax[i];
        switch(dim)
        {
        case 2:
            grid->c_nnx = gmax[0] + 1;
            grid->c_nny = gmax[1] + 1;
            grid->c_node_offset = comp_grid->lbuf[0] + 1 +
                  (gmax[0]+1) * (comp_grid->lbuf[1] + 1);
            grid->c_cell_offset =
                (comp_grid->lbuf[0] + 1) + gmax[0] * (comp_grid->lbuf[1] + 1);
        break; 
        default:
            printf("ERROR: %D case not implemented"
                   " for set_tri_grid_aug_comp_grids\n", dim);
            clean_up(ERROR); 
        }
} 

/* analogy to set_node_index_list, consider to merge these
 * two functions later. This now only works for aug_comp_grid
 */
LOCAL  int **set_node_index_list_on_grid(
        TRI_GRID        *grid,
        RECT_GRID       *expanded_grid)
{
        INTERFACE       *intfc = grid->grid_intfc;
        NODE            **n;
        /*
        RECT_GRID       *expanded_grid = &grid->rect_grid;
        */
        float           s, *posn, *h = expanded_grid->h;
        float           ntol[MAXD];
        float           *L, *U, sfrac; 
        int             **ic_of_node = NULL;
        int             is, i, n_index;
        int             dim = intfc->dim;
        DEBUG_ENTER(set_node_index_list_on_grid)

        /* THIS can only be used for aug_comp_grid */
        L = expanded_grid->GL;
        U = expanded_grid->GU;

        /* count the nodes */
        for (n_index = 0, n = intfc->nodes; n && *n; ++n)
            ++n_index;
        matrix(&ic_of_node,n_index,dim,INT);

        for (i = 0; i < dim; ++i)
        {   /* Used to be NTOL */
            ntol[i] = TOL * h[i];
        }
        for (n_index = 0, n = intfc->nodes;  n && *n;  ++n, ++n_index)
        {
            posn = Coords((*n)->posn);
            for (i = 0; i < dim; ++i)
            {
                sfrac = fabs(posn[i] - L[i]) / cell_width(is,i,expanded_grid);
                if(sfrac < TOL)
                {
                    posn[i] -= ntol[i]; 
                    is = ic_of_node[n_index][i] =
                        cell_index(posn[i],i,expanded_grid);                    
                    continue; 
                }
                sfrac = fabs(posn[i] - U[i]) / cell_width(is,i,expanded_grid);
                if(sfrac < TOL)
                {
                    posn[i] += ntol[i]; 
                    is = ic_of_node[n_index][i] =
                        cell_index(posn[i],i,expanded_grid);                    
                    continue; 
                }

                is = ic_of_node[n_index][i] =
                    cell_index(posn[i],i,expanded_grid);
                s = (posn[i] - expanded_grid->edges[i][is]) /
                    cell_width(is,i,expanded_grid);
                /*
                printf("the floor value = %g, L[%d] = %g\n", 
                      floor((posn[i]-expanded_grid->L[i])/expanded_grid->h[i]),
                       i, expanded_grid->L[i]);
                printf("before set_tol ic[%d] = %d, grid->edge = %g tol = %g\n", 
                      i, is, expanded_grid->edges[i][is], s); 
                */
                if (s < NTOL)
                {
                    posn[i] += ntol[i];
                    if (debugging("grid_tolerance"))
                        (void) printf("TOLERANCE set_node_index_list_on_grid() "
                                      "NTOL = %g\n",NTOL);
                }
                else if (s > ONEMNTOL)
                {
                    posn[i] -= ntol[i];
                    if (debugging("grid_tolerance"))
                        (void) printf("TOLERANCE set_node_index_list_on_grid() "
                                      "ONEMNTOL = %g\n",ONEMNTOL);
                }
            }
        }
        DEBUG_LEAVE(set_node_index_list_on_grid)
        return ic_of_node;
}               /*end set_node_index_list*/

/* analogy to set_dual_interface_topology() 
 * consider to merge them.
 */

LOCAL  void set_interface_topology_on_grid(
        INTERFACE  *intfc,
        RECT_GRID  *expanded_grid)
{
        RECT_GRID *top_grid = &topological_grid(intfc);

        DEBUG_ENTER(set_interface_topology_on_grid)
        if ((intfc->modified) || no_topology_lists(intfc) ||
                (memcmp((const void*)expanded_grid,
                        (const void*)top_grid,sizeof(RECT_GRID)) != 0))
        {
            set_topological_grid(intfc,expanded_grid);
            no_topology_lists(intfc) = NO;

            if (!make_interface_topology_lists(intfc))
            {
                screen("ERROR in set_interface_topology_on_grid(), "
                       "make_interface_topology_lists() failed\n");
                clean_up(ERROR);
            }
        }
        DEBUG_LEAVE(set_interface_topology_on_grid)
}               /*end set_interface_topology_on_grid */



LOCAL   bool remove_unphysical_crossings2d(
        INTERFACE  *oldintfc,
        TRI_GRID   *ngrid,
        int        *n_fr_blk,
        int        *smin,
        int        *smax,
        int        which_grid)
{
        int        *gmax;
        COMPONENT  *comp;
        COMPONENT  **ocoy, *ocoyx, oc;
        COMPONENT  **ncoy, *ncoyx, nc;
        INTERFACE  *new_ntg_intfc = ngrid->grid_intfc;
        int        ixm,iym;
        int        i, ix, iy, n_reg_node, ip[2];
        CURVE      **cc;

        DEBUG_ENTER(remove_unphysical_crossings2d)

        if (oldintfc->modified || oldintfc->table->new_grid ||
            oldintfc->table->on_which_grid != which_grid)
        {
            oldintfc->table->on_which_grid = which_grid;
            if(which_grid == ON_DUAL_GRID)
                set_interface_topology_on_grid(oldintfc,&ngrid->rect_grid);
            else
                set_interface_topology_on_grid(oldintfc,&ngrid->aug_comp_grid);
            oldintfc->table->new_grid = YES;
        }

        if (ngrid->grid_intfc->modified || ngrid->grid_intfc->table->new_grid ||
            ngrid->grid_intfc->table->on_which_grid != which_grid)
        {
            printf("ERROR ngrid not consistent in remove_unphysical_crossings2d\n");
            printf("modified = %d, new_grid = %d, on_which_grid = %d\n",
                ngrid->grid_intfc->modified, 
                ngrid->grid_intfc->table->new_grid,
                ngrid->grid_intfc->table->on_which_grid);
            clean_up(ERROR); 
        }

        if(which_grid == ON_DUAL_GRID)
        {
            gmax = ngrid->rect_grid.gmax;
            comp = ngrid->components;
        }
        else
        {
            gmax = ngrid->aug_comp_grid.gmax;
            comp = ngrid->c_components;
        }
        n_reg_node = (gmax[0]+1)*(gmax[1]+1);
        ocoy = (table_of_interface(oldintfc))->compon2d;
        ncoy = (table_of_interface(new_ntg_intfc))->compon2d;       

        for (i = 0; i < n_reg_node; i++)
            comp[i] = NO_COMP;

        for (iy = smin[1]; iy < smax[1]; iy++)
        {
            for (ocoyx = ocoy[iy], ncoyx = ncoy[iy],
                 ix = smin[0]; ix < smax[0]; ix++)
            {
                oc = ocoyx[ix];
                nc = ncoyx[ix];

                if (oc != ONFRONT && nc != ONFRONT)
                {
                    if(oc != nc)
                    {
                        /* Generally speaking, this case is most likely caused
                         * by the tangling. etc of the new interface ( the old
                         * one is presumely assumed correct) after the
                         * fix of find_compon2d() in set_off_front_comp2d(). 
                         * So new intfc takes the comp. value from old ones.  
                         */
#if defined(DEBUG)
                        /*
                        struct Table  *oldT, *newT; 
                        printf("WARNING in remove_unphysical_crossings2d\n");
                        oldT = table_of_interface(oldintfc); 
                        newT = table_of_interface(new_ntg_intfc); 
                        printf("OFF Front block[%d,%d] of table->infc on grid[%s]\n", 
                            ix, iy, which_grid == ON_DUAL_GRID ? "ON_DUAL_GRID" : "ON_COMP_GRID");
                        printf("DO NOT have the same component, new[%d], old[%d]\n", nc, oc); 
                        printf("newcomp[%d]\n", newT->compon2d[iy][ix]); 
                        printf("oldcomp[%d]\n", oldT->compon2d[iy][ix]); 
                        printf("[iy][ix] = [%d][%d]\n", iy, ix);  
                        printf("x,y[%g %g]\n", 
                          ngrid->aug_comp_grid.L[0]+ngrid->aug_comp_grid.h[0]*ix,
                          ngrid->aug_comp_grid.L[1]+ngrid->aug_comp_grid.h[1]*iy);
                        printf("New interface: %d\n", ngrid->grid_intfc); 
                        for(cc = ngrid->grid_intfc->curves; cc && *cc; cc++)
                            if(wave_type(*cc) >= FIRST_PHYSICS_WAVE_TYPE)
                                print_curve(*cc); 
                        printf("Old interface: %d\n", oldintfc); 
                        for(cc = oldintfc->curves; cc && *cc; cc++)
                            if(wave_type(*cc) >= FIRST_PHYSICS_WAVE_TYPE)
                                print_curve(*cc); 
                        clean_up(ERROR);  
                        */
#endif /* if defined(DEBUG) */
                    }
 
                    ixm = ix + 1;
                    iym = iy + 1;
                    for (ip[1] = iy; ip[1] <= iym; (ip[1])++)
                    {
                        for (ip[0] = ix; ip[0] <= ixm; (ip[0])++)
                        {
                            comp[d_index2d(ip,gmax)] = oc;
                        }
                    }
                }
            }
        }

        if (! track_comp_through_crxings2d(smin,smax,gmax,ngrid,SINGLE,which_grid))
        {
            DEBUG_LEAVE(remove_unphysical_crossings2d)
            return FUNCTION_FAILED;
        }

        *n_fr_blk = 0;
        for (iy = smin[1]; iy < smax[1]; iy++)
        {
            for (ncoyx = ncoy[iy],
                 ix = smin[0]; ix < smax[0]; ix++)
            {
                nc = ncoyx[ix];
                if (nc == ONFRONT) (*n_fr_blk)++;
            }
        }

        DEBUG_LEAVE(remove_unphysical_crossings2d)
        return FUNCTION_SUCCEEDED;
}


LOCAL  bool track_comp_through_crxings2d(
        int             *smin,
        int             *smax,
        int             *gmax,
        TRI_GRID        *ntg,
        CRX_TYPE        crx_type,
        int             which_grid)
{

        int             ix, iy, i, step;
        int             ip[2],ipn[2];
        COMPONENT       *comp;
        COMPONENT       c,cn;
        CURVE           **cc;

        DEBUG_ENTER(track_comp_through_crxings2d)

        if(which_grid == ON_DUAL_GRID)
            comp = ntg->components;
        else
            comp = ntg->c_components;
 
        eliminate_same_crossings2d(ntg,smin,smax,which_grid);

        for (ix = smin[0]; ix <= smax[0]; ix++)
        {
            ip[0] = ipn[0] = ix;
            for (iy = smin[1]; iy <= smax[1]; iy++)
            {
                ip[1] = iy;
                c = comp[d_index2d(ip,gmax)];
                if (c == NO_COMP) continue;

                if (iy != 0)
                {
                    ipn[1] = iy - 1;
                    cn = comp[d_index2d(ipn,gmax)];
                    if (cn == NO_COMP)
                    {
                         step = walk_comp_along_grid_line2d(ntg,smin,smax,
                                        gmax,ip,SOUTH,which_grid);
                    }
                }
                if (iy != smax[1])
                {
                    ipn[1] = iy + 1;
                    cn = comp[d_index2d(ipn,gmax)];
                    if (cn == NO_COMP)
                    {
                         step = walk_comp_along_grid_line2d(ntg,smin,smax,
                                        gmax,ip,NORTH,which_grid);
                         iy +=step;
                    }
                }
            }
        }
        for (iy = smin[1]; iy <= smax[1]; iy++)
        {
            ip[1] = ipn[1] = iy;
            for (ix = smin[0]; ix <= smax[0]; ix++)
            {
                ip[0] = ix;
                c = comp[d_index2d(ip,gmax)];
                if (c == NO_COMP) continue;

                if (ix != 0)
                {
                    ipn[0] = ix - 1;
                    cn = comp[d_index2d(ipn,gmax)];
                    if (cn == NO_COMP)
                    {
                         step = walk_comp_along_grid_line2d(ntg,smin,smax,
                                        gmax,ip,WEST,which_grid);
                    }
                }
                if (ix != smax[0])
                {
                    ipn[0] = ix + 1;
                    cn = comp[d_index2d(ipn,gmax)];
                    if (cn == NO_COMP)
                    {
                         step = walk_comp_along_grid_line2d(ntg,smin,smax,
                                        gmax,ip,EAST,which_grid);
                         ix +=step;
                    }
                }
            }
        }

        /* remove unphysical crossings */
        for (ix = smin[0]; ix <= smax[0]; ix++)
        {
            ip[0] = ipn[0] = ix;
            for (iy = smin[1]; iy <= smax[1]; iy++)
            {
                ip[1] = ipn[1] = iy;
                c = comp[d_index2d(ip,gmax)];
                if (c == NO_COMP) continue;

                if (iy != 0)
                {
                    rm_unphy_crx_along_grid_line2d(ntg,smin,smax,
                                 gmax,ip,SOUTH,crx_type,which_grid);
                }
                if (iy != smax[1])
                {
                    rm_unphy_crx_along_grid_line2d(ntg,smin,smax,
                                 gmax,ip,NORTH,crx_type,which_grid);
                }
            }
        }
        for (iy = smin[1]; iy <= smax[1]; iy++)
        {
            ip[1] = ipn[1] = iy;
            for (ix = smin[0]; ix <= smax[0]; ix++)
            {
                ip[0] = ix;
                c = comp[d_index2d(ip,gmax)];
                if (c == NO_COMP) continue;
                if (ix != 0)
                {
                    rm_unphy_crx_along_grid_line2d(ntg,smin,smax,
                                 gmax,ip,WEST,crx_type,which_grid);
                }
                if (ix != smax[0])
                {
                    rm_unphy_crx_along_grid_line2d(ntg,smin,smax,
                                 gmax,ip,EAST,crx_type,which_grid);
                }
            }
        }

        if (! check_grid_comp_and_crx2d(ntg,smin,smax,which_grid))
        {
            DEBUG_LEAVE(track_comp_through_crxings3d)
            return FUNCTION_FAILED;
        }

#if defined(DEBUG)
        /*
        {
            int dir; 
            ip[0] = 9; ip[1] = 76; dir = EAST;
            i = seg_index2d(ip,dir);
            printf("Before leaving track_comp_through_crxings2d\n");
            printf(" ip[%d,%d] dir[%s], seg_crx_count[%d] = %d\n",
                     ip[0], ip[1], grid_dir(dir), i, ntg->c_seg_crx_count[i]);
        }
        */
#endif /* if defined(DEBUG) */

        /*** NO NEED 
        multi_crxings_check(ntg,which_grid);
        ***/

        DEBUG_LEAVE(track_comp_through_crxings2d)
        return FUNCTION_SUCCEEDED;
}

/**** Support comp_grid only 
LOCAL  int multi_crxings_check(
        TRI_GRID        *ntg,
        int             which_grid)
{
        int             i, j, k, m, l;
        int             imin[MAXD], imax[MAXD];
        int             ic[MAXD], ic_end[MAXD];
        float           coords[MAXD];
        int             dim = ntg->grid_intfc->dim;

        RECT_GRID       *r_grid = &ntg->comp_grid;
        int             *gmax = r_grid->gmax;
        int             nc[4]; // number of CRXING 

        static CRXING   **crx[4] = {NULL, NULL, NULL, NULL};

        if(NULL == crx[0])
        {
             for(i = 0; i < 4; i++)
             {
                  vector(&crx[i],4,sizeof(CRXING *));
             }
        }
        for (i = 0; i < dim; i++)
        {
            imin[i] = 0;
            imax[i] = gmax[i];
            imin[i] -= r_grid->lbuf[i];
            imax[i] += r_grid->ubuf[i];
        }
       for (i=imin[1]; i<=imax[1]; i++)
       {
            for (k=imin[0]; k<=imax[0]; k++)
            {
                  ic[1] = i; ic[0] = k;
                  nc[0] = nc[1] = nc[2] = nc[3] = 0;

                  // if(k != imax[0])
                  //     nc[0] = crossings_in_direction(crx[0],ic,EAST,ntg,ON_COMP_GRID);
                  // if(k != imin[0])
                  //      nc[1] = crossings_in_direction(crx[1],ic,WEST,ntg,ON_COMP_GRID);
                  // if(i != imax[1])
                  //      nc[2] = crossings_in_direction(crx[2],ic,NORTH,ntg,ON_COMP_GRID);
                  // if(i != imin[1])
                  //      nc[3] = crossings_in_direction(crx[3],ic,SOUTH,ntg,ON_COMP_GRID);

                  if(nc[0] >1 || nc[1] >1 || nc[2] >1 || nc[3] >1)
                  {
                       printf("at icrds<%d, %d>, multiple crxings\n",ic[0],ic[1]);
                       printf("nc[0] = %d, nc[1] = %d, nc[2] = %d, nc[3] = %d\n",
                               nc[0], nc[1], nc[2],nc[3]);
                       if(nc[2] == 2)
                       {
                           print_crxings(crx[2][0], NO);
                           print_crxings(crx[2][1], NO);
                       }
                       if(nc[0] > 1 || nc[1] > 1 || nc[2] > 1 || nc[3] > 1)
                       {
                           printf("ERROR at icrds<%d, %d>, multiple crxings\n",ic[0],ic[1]);
                           printf("nc[0] = %d, nc[1] = %d, nc[2] = %d, nc[3] = %d\n",
                                  nc[0], nc[1], nc[2],nc[3]);
                           clean_up(ERROR);
                       }
                   }
              }
        }
        return YES;
}
****/

LOCAL   void    rm_unphy_crx_along_grid_line2d(
        TRI_GRID        *ntg,
        int             *smin,
        int             *smax,
        int             *gmax,
        int             *ip,
        GRID_DIRECTION  dir,
        CRX_TYPE        crx_type,
        int             which_grid)
{
        COMPONENT       *comp;
        int             *seg_crx_count;
        int             **seg_crx_lists;
        CRXING          *crx_store;
        COMPONENT       current_comp;
        CRXING          *crx;
        int             ip1[3],ip2[3];
        int             i,k,nc,list,l;
        float           *L;
        float           *h;
#if defined(DEBUG)
        int             debug_ip = NO; 
#endif /* if defined(DEBUG) */

        DEBUG_ENTER(rm_unphy_crx_along_grid_line2d)

        if(which_grid == ON_DUAL_GRID)
        {
            comp = ntg->components;
            seg_crx_count = ntg->seg_crx_count;
            seg_crx_lists = ntg->seg_crx_lists;
            crx_store = ntg->crx_store;
            L = ntg->rect_grid.L;
            h = ntg->rect_grid.h;
        }
        else
        {
            comp = ntg->c_components;
            seg_crx_count = ntg->c_seg_crx_count;
            seg_crx_lists = ntg->c_seg_crx_lists;
            crx_store = ntg->c_crx_store;
            L = ntg->aug_comp_grid.L;
            h = ntg->aug_comp_grid.h;
        }

        for (i = 0; i < 2; i++) ip1[i] = ip[i];
        current_comp = comp[d_index2d(ip1,gmax)];

#if defined(DEBUG)
        // if(ip[0] == 1 && ip[1] == 44 && dir == SOUTH)
        //    debug_ip = YES; 
        if(debug_ip)
        {
            k = seg_index2d(ip,dir); 
            printf("Entering rm_unphy_crx_along_grid_line2d\n");
            printf(" ip[%d,%d] dir[%s], seg_crx_count[%d] = %d\n", 
                     ip[0], ip[1], grid_dir(dir), k, seg_crx_count[k]);
            printf("ip's current_comp = %d\n", current_comp); 
        }
#endif /* if defined(DEBUG) */

        while (next_ip_in_dir(ip1,dir,ip2,smin,smax))
        {
            k = seg_index2d(ip1,dir);
            nc = seg_crx_count[k];
            if (nc == 0) 
            {
                DEBUG_LEAVE(rm_unphy_crx_along_grid_line2d)
                return;
            }

            if (dir == EAST || dir == NORTH)
            {
                for (i = 0; i < nc; i++)
                {
                    list = seg_crx_lists[k][i];
                    crx = &(crx_store[list]);
                    if (crx->lcomp == current_comp)
                        current_comp = crx->ucomp;
                    else
                    {
                        for (l = i; l < nc-1; l++)
                            seg_crx_lists[k][l] =
                                seg_crx_lists[k][l+1];
                        i--;
                        nc--;
                        seg_crx_count[k]--;
                    }
                }
            }
            else
            {
                for (i = nc-1; i >= 0; i--)
                {
                    list = seg_crx_lists[k][i];
                    crx = &(crx_store[list]);
                    if (crx->ucomp == current_comp)
                        current_comp = crx->lcomp;
                    else
                    {
                        for (l = i; l < nc-1; l++)
                            seg_crx_lists[k][l] =
                                seg_crx_lists[k][l+1];
                        nc--;
                        seg_crx_count[k]--;
                    }
                }
            }

            /* The current code can only handle one tracked curve.
             * The appearing of the multi-component is purely caused
             * by the boundary curves. The following reducing crxing
             * rules should only be applied to the tracked physical wave.
             */
            if (crx_type == SINGLE && nc != 1 &&
                (dir == WEST || dir == SOUTH))
            {
                rm_same_hypsurf_multi_crx(k,seg_crx_count,seg_crx_lists,crx_store);
                nc = seg_crx_count[k];

                if (nc%2 == 0)
                {
                    rm_select_even_crx(k, seg_crx_count, seg_crx_lists, crx_store);
                    /*
                    seg_crx_count[k] = 0;
                    */
                }
                else if(seg_crx_count[k] != 1)
                {
                    /*
                    if(seg_crx_count[k] != 1)
                        find_mid_crx_left(ntg,k,ip1,ip2,dir,which_grid);
                    */
                    rm_select_odd_crx(k, seg_crx_count, seg_crx_lists, crx_store);
                    /*
                    seg_crx_count[k] = 1;
                    */
                }
            }
            if (comp[d_index2d(ip2,gmax)] != NO_COMP)
                break;
            comp[d_index2d(ip2,gmax)] = current_comp;
            for (i = 0; i < 2; i++) ip1[i] = ip2[i];
        }

#if defined(DEBUG)
        if(debug_ip)
        {
            k = seg_index2d(ip,dir); 
            printf("Leaving rm_unphy_crx_along_grid_line2d\n");
            printf(" ip[%d,%d] dir[%s], seg_crx_count[%d] = %d\n", 
                     ip[0], ip[1], grid_dir(dir), k, seg_crx_count[k]);
        }
#endif /* if defined(DEBUG) */

        DEBUG_LEAVE(rm_unphy_crx_along_grid_line2d)
        return; 
}

LOCAL void rm_select_even_crx(
	int        k, 
        int        *seg_crx_count, 
        int        **seg_crx_lists, 
        CRXING     *crx_store)
{
        int        nc = seg_crx_count[k];
        int        list, i, j;
        CRXING     *crx;
        HYPER_SURF *hs[20]; /* Enough ? */
        int        diff, n_trk_w = 0, n_trk_c;
        int        n_un_trk = 0; 

        for(i = 0; i < nc; i++)
        {
            list = seg_crx_lists[k][i];
            crx = &(crx_store[list]);
            if(wave_type(crx->hs) >= FIRST_PHYSICS_WAVE_TYPE)
            {
                hs[0] = crx->hs;
                n_trk_w = 1;
                break;
            }
        }
        if(n_trk_w == 0)
            return;

        n_trk_c = n_trk_w;
        for(i = 0; i < nc; i++)
        {
            list = seg_crx_lists[k][i];
            crx = &(crx_store[list]);
            if(wave_type(crx->hs) < FIRST_PHYSICS_WAVE_TYPE)
            {
                n_un_trk++; 
                continue;
            }
            diff = YES;
            for(j = 0; j < n_trk_c; j++)
            {
                if(crx->hs == hs[j])
                {
                    diff = NO;
                    break;
                }
            }
            if(diff == YES)
            {
                hs[n_trk_c] = crx->hs;
                n_trk_c++;
            }
        }
 
        if(n_un_trk != 0)
        {
            printf("ERROR: in rm_unphy_crx_along_grid_line2d\n");
            printf("seg_crx_count[%d] = %d is even\n",k, seg_crx_count[k]); 
            printf("Contains tracked and untracked crxs, n_un_trk = %d\n",n_un_trk); 
            printf("Need to implement rm_select_even_crx\n");
            for(i = 0; i < nc; i++)
            {
                list = seg_crx_lists[k][i];
                crx = &(crx_store[list]);
                print_crxings(crx,YES); 
            }
            clean_up(ERROR);  
        }

        if(n_trk_c == 1)
            seg_crx_count[k] = 0; 
        else
        {
            // printf("ERROR: in rm_unphy_crx_along_grid_line2d\n");
            // printf("seg_crx_count[%d] = %d is even\n",k, seg_crx_count[k]); 
            // printf("tracked crxings belong to diff curves  n_trk_c[%d]\n",n_trk_c); 
            // printf("Need to implement rm_select_even_crx\n");
            // for(i = 0; i < nc; i++)
            // {
            //     list = seg_crx_lists[k][i];
            //     crx = &(crx_store[list]);
            //     print_crxings(crx,YES); 
            // }
            // clean_up(ERROR);  
            seg_crx_count[k] = 0; 
        }
}

LOCAL void rm_same_hypsurf_multi_crx(
	int        k, 
        int        *seg_crx_count, 
        int        **seg_crx_lists, 
        CRXING     *crx_store)
{
        HYPER_SURF *hs[20]; /* Enough ? */
        int        nc = seg_crx_count[k];
        int        list, i, j, l; 
        CRXING     *crx; 
        int        ncrx = 0, nhs = 0; 
        int        first = YES; 

        for(i = 0; i < nc; i++)
        {
            list = seg_crx_lists[k][i];
            crx = &(crx_store[list]);
            if(wave_type(crx->hs) >= FIRST_PHYSICS_WAVE_TYPE)
            {
                hs[nhs] = crx->hs; 
                nhs++; 
            }
        }

        for(j = 0; j < nhs; j++)
        {

            nc = seg_crx_count[k];
            ncrx = 0; 
            for(i = 0; i < nc; i++)
            {
                list = seg_crx_lists[k][i];
                crx = &(crx_store[list]);
                if(crx->hs == hs[j])
                    ncrx++;
            }
 
            if(ncrx == 0)
                continue; 

            if(ncrx %2 == 0)
            {
                /* Remove them all */
                for(i = 0; i < nc; i++)
                {
                    list = seg_crx_lists[k][i];
                    crx = &(crx_store[list]);
                    if(crx->hs == hs[j])
                    {
                        for (l = i; l < nc-1; l++)
                            seg_crx_lists[k][l] =
                                seg_crx_lists[k][l+1];
                        i--;
                        nc--;
                        seg_crx_count[k]--;
                    }
                }
            }
            else if(ncrx != 1)
            {
                /* Leave the first crx, remove the rest  */
                for(i = 0; i < nc; i++)
                {
                    list = seg_crx_lists[k][i];
                    crx = &(crx_store[list]);
                    if(crx->hs == hs[j])
                    {
                        if(first == YES)
                            first = NO; 
                        else
                        {
                            for (l = i; l < nc-1; l++)
                                seg_crx_lists[k][l] =
                                    seg_crx_lists[k][l+1];
                            i--;
                            nc--;
                            seg_crx_count[k]--;
                        }
                    }
                }
            }
        }
}

LOCAL void rm_select_odd_crx(
	int        k, 
        int        *seg_crx_count, 
        int        **seg_crx_lists, 
        CRXING     *crx_store)
{
        int        nc = seg_crx_count[k];
        int        list, i, j; 
        CRXING     *crx; 
        HYPER_SURF *hs[20]; /* Enough ? */
        int        diff, n_tracked_w = 0, n_tracked_c; 

        for(i = 0; i < nc; i++)
        {
            list = seg_crx_lists[k][i];
            crx = &(crx_store[list]);
            if(wave_type(crx->hs) >= FIRST_PHYSICS_WAVE_TYPE)
            {
                hs[0] = crx->hs;
                n_tracked_w = 1;
                break; 
            }
        }
        if(n_tracked_w == 0) 
            return; 

        n_tracked_c = n_tracked_w; 
        for(i = 0; i < nc; i++)
        {
            list = seg_crx_lists[k][i];
            crx = &(crx_store[list]);
            if(wave_type(crx->hs) < FIRST_PHYSICS_WAVE_TYPE)
                continue; 
            diff = YES; 
            for(j = 0; j < n_tracked_c; j++)
            {
                if(crx->hs == hs[j])
                {
                    diff = NO;
                    break;
                }
            }
            if(diff == YES)
            {
                hs[n_tracked_c] = crx->hs;
                n_tracked_c++;                    
            }
        }

        if(n_tracked_c != 1)
        {
            printf("ERROR: in rm_unphy_crx_along_grid_line2d\n");
            printf("seg_crx_count[%d] = %d is even\n",k, seg_crx_count[k]); 
            printf("Need to implement rm_select_odd_crx, n_tracked curve = %d\n", 
                 n_tracked_c);
            for(i = 0; i < nc; i++)
            {
                list = seg_crx_lists[k][i];
                crx = &(crx_store[list]);
                print_crxings(crx,YES);
            }
            clean_up(ERROR);  
        }
}


LOCAL   int     walk_comp_along_grid_line2d(
        TRI_GRID       *ntg,
        int            *smin,
        int            *smax,
        int            *gmax,
        int            *ip,
        int            dir,
        int            which_grid)
{
        COMPONENT       *comp;
        int             *seg_crx_count;
        int             **seg_crx_lists;
        CRXING          *crx_store;
        COMPONENT       current_comp;
        CRXING          *crx;
        int             ip1[3],ip2[3];
        int             i,k,nc,list,step;
        bool            crx_is_physical = YES;
#if defined(DEBUG)
        int             debug_walk = NO; 
#endif /* if defined(DEBUG) */

        DEBUG_ENTER(walk_comp_along_grid_line2d)

        if(which_grid == ON_DUAL_GRID)
        {
            comp = ntg->components;
            seg_crx_count = ntg->seg_crx_count;
            seg_crx_lists = ntg->seg_crx_lists;
            crx_store = ntg->crx_store;
        }
        else
        {
            comp = ntg->c_components;
            seg_crx_count = ntg->c_seg_crx_count;
            seg_crx_lists = ntg->c_seg_crx_lists;
            crx_store = ntg->c_crx_store;
        }

#if defined(DEBUG)
        /*
        if(ip[0] == 9 && ip[1] == 76 && dir == EAST)
            debug_walk = YES;   
        */
#endif /* if defined(DEBUG) */

        for (i = 0; i < 2; i++) ip1[i] = ip[i];
        current_comp = comp[d_index2d(ip1,gmax)];
        step = 0;
        while (next_ip_in_dir(ip1,dir,ip2,smin,smax))
        {
            if (comp[d_index2d(ip2,gmax)] != NO_COMP)
                break;
            k = seg_index2d(ip1,dir);
            nc = seg_crx_count[k];
            if (dir == EAST || dir == NORTH)
            {
                for (i = 0; i < nc; i++)
                {
                    list = seg_crx_lists[k][i];
                    crx = &(crx_store[list]);
                    if (crx->lcomp == current_comp)
                        current_comp = crx->ucomp;
                    else
                    {
                        crx_is_physical = NO;
                        break;
                    }
                }
            }
            else
            {
                for (i = nc-1; i >= 0; i--)
                {
                    list = seg_crx_lists[k][i];
                    crx = &(crx_store[list]);
                    if (crx->ucomp == current_comp)
                        current_comp = crx->lcomp;
                    else
                    {
                        crx_is_physical = NO;
                        break;
                    }
                }
            }
            if (! crx_is_physical)
                break;
            step++;
            comp[d_index2d(ip2,gmax)] = current_comp;
            for (i = 0; i < 3; i++) ip1[i] = ip2[i];
        }

        DEBUG_LEAVE(walk_comp_along_grid_line2d)
        return step; 
}

LOCAL   void eliminate_same_crossings2d(
        TRI_GRID        *ntg,
        int             *smin,
        int             *smax,
        int             which_grid)
{
        int             ix,iy;
        int             dir[2] = {EAST,NORTH};
        int             ip[2],i,k,l,m,nc,list1,list2;
        CRXING          *crx1,*crx2;
        float           grid_crds;
        float           *L;
        float           *h;
        int             error_comp_cnt = 0, same_crxing_cnt = 0;
        int             *seg_crx_count;
        int             **seg_crx_lists;
        CRXING          *crx_store;
 
        DEBUG_ENTER(eliminate_same_crossings2d)

        if(which_grid == ON_DUAL_GRID)
        {
            L = ntg->rect_grid.L;
            h = ntg->rect_grid.h;
            seg_crx_count = ntg->seg_crx_count;
            seg_crx_lists = ntg->seg_crx_lists;
            crx_store = ntg->crx_store;
        }
        else
        {
            L = ntg->aug_comp_grid.L;
            h = ntg->aug_comp_grid.h;
            seg_crx_count = ntg->c_seg_crx_count;
            seg_crx_lists = ntg->c_seg_crx_lists;
            crx_store = ntg->c_crx_store;
        }


        for (iy = smin[1]; iy <= smax[1]; iy++)
        {
            ip[1] = iy;
            for (ix = smin[0]; ix <= smax[0]; ix++)
            {
                ip[0] = ix;
                for (i = 0; i < 2; i++)
                {
                    if (ix == smax[0] && dir[i] == EAST)
                        continue;
                    if (iy == smax[1] && dir[i] == NORTH)
                        continue;
                    k = seg_index2d(ip,dir[i]);
                    nc = seg_crx_count[k];

                    /*
                    if(ip[0] == 9 && ip[1] == 76 )
                    {
                        printf("In eliminate_same_crossings2d\n");
                        printf("ip[%d,%d] dir[%s], nc[%d], k = %d\n", 
                             ip[0], ip[1],grid_dir(dir[i]), nc, k); 
                        if(nc != 0)
                        {
                            for (l = 0; l < nc; l++)
                            {
                                list1 = seg_crx_lists[k][l];
                                print_crxings(&crx_store[list1], NO);
                            }    
                        }
                    }
                    */
                    if (nc == 0) continue;
                    else
                    {
                        for (l = 0; l < nc; l++)
                        {
                            list1 = seg_crx_lists[k][l];
                            crx1 = &(crx_store[list1]);
                            if (crx1->lcomp == NO_COMP)
                            {
                                for (m = l; m < nc - 1; m++)
                                    seg_crx_lists[k][m] =
                                            seg_crx_lists[k][m+1];
                                l--;
                                nc--;
                                seg_crx_count[k]--;
                                error_comp_cnt++;
                            }
                        }
                        if (nc == 0) continue;
                        list1 = seg_crx_lists[k][0];
                        crx1 = &(crx_store[list1]);
                        adjust_for_min_spacing2d(crx1,L,ip,h,nc,i,k,
                           seg_crx_lists,crx_store);
                    }
                    /*
                    if(ip[0] == 9 && ip[1] == 76 )
                    {
                        printf("In eliminate_same_crossings2d, after reduce\n");
                        printf("ip[%d,%d] dir[%s], nc[%d], k = %d\n", 
                             ip[0], ip[1],grid_dir(dir[i]), nc, k); 
                    }
                    */
                }
            }
        }

        DEBUG_LEAVE(eliminate_same_crossings2d)
}

LOCAL void adjust_for_min_spacing2d(
        CRXING          *crxing,
        float           *L,
        int             *ip,
        float           *h,
        int             n_crx,
        int             dir,
        int             k,
        int             **seg_crx_lists,
        CRXING          *crx_store)
{

        int             i;
        float           crds_start[2], crds_end[2];
        float           mgs = 0.001*h[dir];/*TOLERANCE old = 0.004, 0.008*/
        int             list1;
        CRXING          *crxings;

        for(i = 0; i < 2; i++)
        {
            crds_start[i] = L[i] + ip[i]*h[i];
            crds_end[i] = crds_start[i] + h[i];
        }
        for (i = 0; i < n_crx; i++)
        {
            list1 = seg_crx_lists[k][i];
            crxings = &(crx_store[list1]);
            if(wave_type(crxings->hs) < FIRST_PHYSICS_WAVE_TYPE)
                continue;  
            if(crxings->crossing_direction == BELOW_TO_ABOVE ||
               crxings->crossing_direction == ABOVE_TO_BELOW)
            {
                if(fabs(Coords(crxings->pt)[0] - crds_start[0]) < mgs &&
                   Coords(crxings->pt)[0] > crds_start[0])
                {
                    Coords(crxings->pt)[0] = crds_start[0] + mgs;
                }
                if(fabs(Coords(crxings->pt)[0] - crds_start[0]) < mgs &&
                   Coords(crxings->pt)[0] < crds_start[0])
                {
                    Coords(crxings->pt)[0] = crds_start[0] - mgs;
                }
                if(fabs(crds_end[0] - Coords(crxings->pt)[0]) < mgs &&
                   Coords(crxings->pt)[0] > crds_end[0])
                {
                    Coords(crxings->pt)[0] = crds_end[0] + mgs;
                }
                if(fabs(crds_end[0] - Coords(crxings->pt)[0]) < mgs &&
                   Coords(crxings->pt)[0] < crds_end[0])
                {
                    Coords(crxings->pt)[0] = crds_end[0] - mgs;
                }
            }
            else if(crxings->crossing_direction == RIGHT_TO_LEFT ||
                    crxings->crossing_direction == LEFT_TO_RIGHT)
            {
                if(fabs(Coords(crxings->pt)[1] - crds_start[1]) < mgs &&
                   Coords(crxings->pt)[1] > crds_start[1])
                {
                    Coords(crxings->pt)[1] = crds_start[1] + mgs;
                }
                if(fabs(Coords(crxings->pt)[1] - crds_start[1]) < mgs &&
                   Coords(crxings->pt)[1] < crds_start[1])
                {
                    Coords(crxings->pt)[1] = crds_start[1] - mgs;
                }
                if(fabs(crds_end[1] - Coords(crxings->pt)[1]) < mgs &&
                   Coords(crxings->pt)[1] > crds_end[1])
                {
                    Coords(crxings->pt)[1] = crds_end[1] + mgs;
                }
                if(fabs(crds_end[1] - Coords(crxings->pt)[1]) < mgs &&
                   Coords(crxings->pt)[1] < crds_end[1])
                {
                    Coords(crxings->pt)[1] = crds_end[1] - mgs;
                }
            }
            else
            {
                printf("ERROR in adjust_for_min_spacing2d()\n");
                printf("CRXING %d 's direction is not set\n", crxings);
                printf("%f, %f\n", Coords(crxings->pt)[0],Coords(crxings->pt)[1]);
                print_crxings(crxings, NO);
                print_curve(Curve_of_hs(crxings->hs));
                clean_up(ERROR);
            }
        }
}

LOCAL   bool check_grid_comp_and_crx2d(
        TRI_GRID        *ntg,
        int             *smin,
        int             *smax,
        int             which_grid)
{
        int             ix,iy,iz;
        GRID_DIRECTION  dir[3] = {EAST,NORTH,UPPER};
        int             ip[3],ipn[3],i,l;
        float           coords[MAXD];
        COMPONENT       lcomp,ucomp;
        RECT_GRID       *expanded_grid;
        Locstate        sl,su;
        HYPER_SURF      *hs;
        POINT           *p;
        CRXING          *crx_fill;
        int             num_crx_fill = 0;
        INTERFACE       *ntg_intfc = ntg->grid_intfc;
        COMPONENT       **comp2d;
        COMPONENT       *components;
        int             *seg_crx_count;
        int             **seg_crx_lists;
        CRXING          *crx_store;

        DEBUG_ENTER(check_grid_comp_and_crx2d)

        if(which_grid == ON_DUAL_GRID)
        {
            crx_fill = &ntg->crx_store[ntg->n_crx];
            expanded_grid = &ntg->rect_grid;
            components = ntg->components;
            seg_crx_count = ntg->seg_crx_count;
            seg_crx_lists = ntg->seg_crx_lists;
            crx_store = ntg->crx_store;
        }
        else
        {
            crx_fill = &ntg->c_crx_store[ntg->n_c_crx];
            expanded_grid = &ntg->aug_comp_grid;
            components = ntg->c_components;
            seg_crx_count = ntg->c_seg_crx_count;
            seg_crx_lists = ntg->c_seg_crx_lists;
            crx_store = ntg->c_crx_store;
        }

        comp2d = (table_of_interface(ntg_intfc))->compon2d;
        for (iy = smin[1]; iy <= smax[1]; iy++)
        {
            ip[1] = iy;
            for (ix = smin[0]; ix <= smax[0]; ix++)
            {
                ip[0] = ix;
                for (i = 0; i < 2; i++)
                {
                    if (ix == smax[0] && dir[i] == EAST)
                        continue;
                    if (iy == smax[1] && dir[i] == NORTH)
                        continue;
                    if (! check_comp_at2d(ip,dir[i],ntg,smin,smax,which_grid))
                    {
                        for (l = 0; l < 2; l++)
                        {
                                coords[l] = expanded_grid->L[l] +
                                    ip[l]*expanded_grid->h[l];
                        }
                        coords[i] += 0.5*expanded_grid->h[i];
                        (void) next_ip_in_dir(ip,dir[i],ipn,smin,smax);
                        lcomp = components[d_index2d(
                            ip,expanded_grid->gmax)];
                        ucomp = components[d_index2d(
                            ipn,expanded_grid->gmax)];
                        p = Point(coords);
                        sl = left_state(p);
                        su = right_state(p);
                        ntg->grid_intfc->modified = NO;
                        nearest_intfc_state(coords,lcomp,ntg->grid_intfc,
                                            sl,NULL,&hs);
                        if (negative_component(hs) == lcomp)
                            nearest_intfc_state(coords,ucomp,
                                    ntg->grid_intfc,su,NULL,&hs);
                        else
                        {
                            nearest_intfc_state(coords,ucomp,
                                    ntg->grid_intfc,sl,NULL,&hs);
                            nearest_intfc_state(coords,lcomp,
                                    ntg->grid_intfc,su,NULL,&hs);
                        }
                        crx_fill[num_crx_fill].pt  = p;
                        crx_fill[num_crx_fill].hs = hs;
                        crx_fill[num_crx_fill].lcomp = lcomp;
                        crx_fill[num_crx_fill].ucomp = ucomp;
                        l = seg_index2d(ip,dir[i]);
                        seg_crx_count[l] = 1;
                        if(which_grid == ON_DUAL_GRID)
                        {
                            seg_crx_lists[l] =
                            &(ntg->seg_crx_lists_store[ntg->n_crx
                            + num_crx_fill]);
                            seg_crx_lists[l][0] = ntg->n_crx
                                + num_crx_fill;
                        }
                        else
                        {
                            seg_crx_lists[l] =
                             &(ntg->c_seg_crx_lists_store[ntg->n_c_crx
                              + num_crx_fill]);
/* ?????? Should depending on the grid ????? */
/* ????????????????????????????????????????? */
                            seg_crx_lists[l][0] = ntg->n_c_crx
                                + num_crx_fill;
                        }
                        num_crx_fill++;
                        if (dir[i] == EAST)
                        {
                            comp2d[iy-1][ix] = ONFRONT;
                            if (iy != smax[1])
                                comp2d[iy][ix] = ONFRONT;
                            if (iz != smax[2])
                                comp2d[iy-1][ix] = ONFRONT;
                            if (iy != smax[1] && iz != smax[2])
                                comp2d[iy][ix] = ONFRONT;
                        }
                        else if (dir[i] == NORTH)
                        {
                            comp2d[iy][ix-1] = ONFRONT;
                            if (ix != smax[0])
                                comp2d[iy][ix] = ONFRONT;
                            if (iz != smax[2])
                                comp2d[iy][ix-1] = ONFRONT;
                            if (ix != smax[0] && iz != smax[2])
                                comp2d[iy][ix] = ONFRONT;
                        }
                    }
                    if (num_crx_fill == MAX_CRX_FILL)
                    {
                        (void) printf("WARNING in "
                                      "check_grid_comp_and_crx2d(), "
                                      "too many missing crossings.\n");
                        (void) printf("num_crx_fill = %d\n", num_crx_fill);
                        DEBUG_LEAVE(check_grid_comp_and_crx2d)
                        return NO;
                    }
                }
            }
        }

        if(num_crx_fill != 0)
        {
            if(which_grid == ON_DUAL_GRID)
                ntg->n_crx += num_crx_fill;
            else
                ntg->n_c_crx += num_crx_fill;
            (void) printf("WARNING %d crossings are missing, will fill!\n",
                   num_crx_fill);
        }

        DEBUG_LEAVE(check_grid_comp_and_crx2d)
        return YES;
}

LOCAL   int check_comp_at2d(
        int             *ip,
        GRID_DIRECTION  dir,
        TRI_GRID        *ntg,
        int             *smin,
        int             *smax,
        int             which_grid)
{
        COMPONENT       *comp;
        int             *seg_crx_count;
        int             **seg_crx_lists;
        CRXING          *crx_store;
        RECT_GRID       *exp_grid;
        int             *gmax;
        int             ipn[MAXD+1];
        int             k,nc,list;
        CRXING          *crx;
        float           fp[MAXD];
        CURVE           **c;

        if (next_ip_in_dir(ip,dir,ipn,smin,smax) != YES)
            return YES;

        if(which_grid == ON_DUAL_GRID)
        {
            comp = ntg->components;
            seg_crx_count = ntg->seg_crx_count;
            seg_crx_lists = ntg->seg_crx_lists;
            crx_store = ntg->crx_store;
            gmax = ntg->rect_grid.gmax;
            exp_grid = &ntg->rect_grid;
        }
        else
        {
            comp = ntg->c_components;
            seg_crx_count = ntg->c_seg_crx_count;
            seg_crx_lists = ntg->c_seg_crx_lists;
            crx_store = ntg->c_crx_store;
            gmax = ntg->aug_comp_grid.gmax;
            exp_grid = &ntg->aug_comp_grid;
        }
        k = seg_index2d(ip,dir);
        nc = seg_crx_count[k];
        if (nc == 0)
        {
            if (comp[d_index2d(ip,gmax)] == comp[d_index2d(ipn,gmax)])
                return YES;
            else
            {
                rect_grid_corner(exp_grid,ip,fp);
                /*
                if (debugging("check_comp"))
                */
                {
                    (void) printf("\nWARNING in check_comp_at2d(),nc = %d "
                              "\n WARNING Inconsistent component at "
                              "neighboring grid point\n", nc);
                    (void) printf("seg_crx_count at k = %d, dir[%s]\n",
                               k, grid_dir(dir)); 
                    (void) printf("  ip  = ( %7d %7d ) comp = %2d  ",
                              ip[0],ip[1],
                              comp[d_index2d(ip,gmax)]);
                    (void) printf("fp  = ( %12.10f, %12.10f)\n",
                              fp[0],fp[1]);
                    rect_grid_corner(exp_grid,ipn,fp);
                    (void) printf("  ipn = ( %7d %7d ) comp = %2d  ",
                              ipn[0],ipn[1],
                              comp[d_index2d(ipn,gmax)]);
                    (void) printf("fpn = ( %12.10f, %12.10f)\n",
                              fp[0],fp[1]);
                    (void) printf("  Number of crossings = %d\n\n",nc);
                    printf("dh[0] = %17.14f, dh[1] = %17.14f\n",
                            exp_grid->h[0], exp_grid->h[1]);
                    printf("Print new interface in check_comp_at2d()\n");
                    for (c = ntg->grid_intfc->curves;
                         c && *c; c++)
                    {
                        if(is_bdry(*c))
                             print_curve(*c);
                    }
                    clean_up(ERROR);
                }
                return NO;
            }
        }
        else if (nc == 1)
        {
            list = seg_crx_lists[k][0];
            crx = &(crx_store[list]);
            switch (dir)
            {
            case EAST:
            case NORTH:
                if (crx->lcomp != comp[d_index2d(ip,gmax)])
                {
                    /*
                    if (debugging("check_comp"))
                    */
                    {
                        (void) printf("WARNING in check_comp_at2d(), nc = %d"
                                  "lower component of cross does "
                                  "not agree with grid\n", nc);
                        rect_grid_corner(exp_grid,ip,fp);
                        (void) printf("  ip  = ( %7d %7d ) comp = %2d  ",
                              ip[0],ip[1],
                              comp[d_index2d(ip,gmax)]);
                        (void) printf("fp  = ( %g %g )\n",
                              fp[0],fp[1]);
                        rect_grid_corner(exp_grid,ipn,fp);
                        (void) printf("  ipn = ( %7d %7d ) comp = %2d  ",
                              ipn[0],ipn[1],
                              comp[d_index2d(ipn,gmax)]);
                        (void) printf("fpn = ( %g %g )\n",
                              fp[0],fp[1]);
                        (void) printf("  Number of crossings = %d\n\n",nc);
                    }
                    return NO;
                }
                if (crx->ucomp != comp[d_index2d(ipn,gmax)])
                {
                    /*
                    if (debugging("check_comp"))
                    */
                    {
                        (void) printf("WARNING in check_comp_at()2d, "
                                  "upper component of cross does "
                                  "not agree with grid\n");
                        rect_grid_corner(exp_grid,ip,fp);
                        (void) printf("  ip  = ( %7d %7d ) comp = %2d  ",
                              ip[0],ip[1],
                              comp[d_index2d(ip,gmax)]);
                        (void) printf("fp  = ( %g %g )\n",
                              fp[0],fp[1]);
                        rect_grid_corner(exp_grid,ipn,fp);
                        (void) printf("  ipn = ( %7d %7d ) comp = %2d  ",
                              ipn[0],ipn[1],
                              comp[d_index2d(ipn,gmax)]);
                        (void) printf("fpn = ( %g %g )\n",
                              fp[0],fp[1]);
                        (void) printf("  Number of crossings = %d\n\n",nc);
                        print_crxings(crx,NO); 
                        return NO;
                    }
                }
                if (crx->ucomp == crx->lcomp)
                {
                    /*
                    if (debugging("check_comp"))
                    */
                    {
                        (void) printf("WARNING in check_comp_at()2d, "
                                  "lower component of cross equal "
                                  "to upper component\n");
                        rect_grid_corner(exp_grid,ip,fp);
                        (void) printf("  ip  = ( %7d %7d ) comp = %2d  ",
                              ip[0],ip[1],
                              comp[d_index2d(ip,gmax)]);
                        (void) printf("fp  = ( %g %g )\n",
                              fp[0],fp[1]);
                        rect_grid_corner(exp_grid,ipn,fp);
                        (void) printf("  ipn = ( %7d %7d ) comp = %2d  ",
                              ipn[0],ipn[1],
                              comp[d_index2d(ipn,gmax)]);
                        (void) printf("fpn = ( %g %g )\n",
                              fp[0],fp[1]);
                        (void) printf("  Number of crossings = %d\n\n",nc);
                        return NO;
                    }
                }
                return YES;
            case WEST:
            case SOUTH:
                if (crx->ucomp != comp[d_index2d(ip,gmax)])
                {
                    /*
                    if (debugging("check_comp"))
                    */
                    {
                        (void) printf("WARNING in check_comp_at(), "
                                  "upper component of cross does "
                                  "not agree with grid\n");
                        rect_grid_corner(exp_grid,ip,fp);
                        (void) printf("  ip  = ( %7d %7d ) comp = %2d  ",
                              ip[0],ip[1],
                              comp[d_index2d(ip,gmax)]);
                        (void) printf("fp  = ( %g %g )\n",
                              fp[0],fp[1]);
                        rect_grid_corner(exp_grid,ipn,fp);
                        (void) printf("  ipn = ( %7d %7d ) comp = %2d  ",
                              ipn[0],ipn[1],
                              comp[d_index2d(ipn,gmax)]);
                        (void) printf("fpn = ( %g %g )\n",
                              fp[0],fp[1]);
                        (void) printf("  Number of crossings = %d\n\n",nc);
                        return NO;
                    }
                }
                if (crx->lcomp != comp[d_index2d(ipn,gmax)])
                {
                    /*
                    if (debugging("check_comp"))
                    */
                    {
                        (void) printf("WARNING in check_comp_at(), "
                                  "lower component of cross does "
                                  "not agree with grid\n");
                        rect_grid_corner(exp_grid,ip,fp);
                        (void) printf("  ip  = ( %7d %7d ) comp = %2d  ",
                              ip[0],ip[1],
                              comp[d_index2d(ip,gmax)]);
                        (void) printf("fp  = ( %g %g )\n",
                              fp[0],fp[1]);
                        rect_grid_corner(exp_grid,ipn,fp);
                        (void) printf("  ipn = ( %7d %7d ) comp = %2d  ",
                              ipn[0],ipn[1],
                              comp[d_index2d(ipn,gmax)]);
                        (void) printf("fpn = ( %g %g )\n",
                              fp[0],fp[1]);
                        (void) printf("  Number of crossings = %d\n\n",nc);
                        return NO;
                    }
                }
                if (crx->ucomp == crx->lcomp)
                {
                    /*
                    if (debugging("check_comp"))
                    */
                    {
                        (void) printf("WARNING in check_comp_at2d(), "
                                  "lower component of cross equal "
                                  "to upper component\n");
                        rect_grid_corner(exp_grid,ip,fp);
                        (void) printf("  ip  = ( %7d %7d ) comp = %2d  ",
                              ip[0],ip[1],
                              comp[d_index2d(ip,gmax)]);
                        (void) printf("fp  = ( %g %g )\n",
                              fp[0],fp[1]);
                        rect_grid_corner(exp_grid,ipn,fp);
                        (void) printf("  ipn = ( %7d %7d ) comp = %2d  ",
                              ipn[0],ipn[1],
                              comp[d_index2d(ipn,gmax)]);
                        (void) printf("fpn = ( %g %g )\n",
                              fp[0],fp[1]);
                        (void) printf("  Number of crossings = %d\n\n",nc);
                        return NO;
                    }
                }
                return YES;
            default:
                (void) printf("WARNING in check_comp_at(), "
                          "invalid dir %d\n",dir);
                return NO;
            }
        }
        else
        {
            GRID_PT     gp;
            int         i;
            COMPONENT   next_comp;

            for (i = 0; i < 2; i++)
                gp.ip[i] = ip[i];
            gp.cur_dir = dir;
            gp.comp = comp[d_index2d(ip,gmax)];

            if (edge_comp_walk(&gp,ntg,&next_comp,MULTIPLE,which_grid)
                == UNPHYSICAL_EDGE)
            {
                /* if (debugging("check_comp"))  */
                {
                    (void) printf("WARNING in check_comp_at() UNPHYSICAL_EDGE "
                              "detected.\n");
                    /*
                    print_edge_crxings(&gp,ntg);
                    */
                    printf("ERROR in check_comp_at2d()\n");
                    printf("PROCESSING MULTIPLE CRXINGS\n");
                    clean_up(ERROR);
                }
            }
            return YES;
        }
}

LOCAL   bool reconstruct_crx_intfc2d(
        TRI_GRID     *ntg,
        int          n_fr_blk,
        int          *smin,
        int          *smax,
        int          which_grid)
{
        int             *gmax;
        int             *seg_crx_count;
        int             **seg_crx_lists;
        CRXING          *crx_store;
        COMPONENT       *comp;
        INTERFACE       *ntg_intfc = ntg->grid_intfc;
        BLK_BOND        bm;
        JC_BLK_BOND     jc_blk_bond; 
        COMPONENT       **coy, *coyx;
        int             ix, iy, ixx, iyy;
        int             i, j, k, nbc, ip[3], num_curves;
        CURVE           **c, **curves = NULL; 
        BOND            **cur_bonds, **first;
        int             *num_bonds, alloc_cn = 0;

        static BLK_CRX2 *blk_crx = NULL;
        struct Table    *T;
        COMPONENT       blk_comp[2][2];
        int             blk_ip[2];
        int             l, nc, nn, list;
        CRXING          *crx;

        DEBUG_ENTER(reconstruct_crx_intfc2d)

        if(which_grid == ON_DUAL_GRID)
        {
            gmax = ntg->rect_grid.gmax;
            comp = ntg->components;
            seg_crx_count = ntg->seg_crx_count;
            seg_crx_lists = ntg->seg_crx_lists;
            crx_store = ntg->crx_store;
        }
        else
        {
            gmax = ntg->aug_comp_grid.gmax;
            comp = ntg->c_components;
            seg_crx_count = ntg->c_seg_crx_count;
            seg_crx_lists = ntg->c_seg_crx_lists;
            crx_store = ntg->c_crx_store;
        }

        num_curves = 0;
        for(c = ntg_intfc->curves; c && *c; c++)
            num_curves++;
        if(curves == NULL)
        {
            alloc_cn = num_curves + 20; 
            vector(&curves, alloc_cn, sizeof(CURVE*)); 
            vector(&cur_bonds, alloc_cn, sizeof(BOND*)); 
            vector(&first, alloc_cn, sizeof(BOND*)); 
            vector(&num_bonds, alloc_cn, sizeof(int)); 
        }

        for(i = 0, c = ntg_intfc->curves; c && *c; i++, c++)
        {
            curves[i] = *c;
            cur_bonds[i] = NULL;
            first[i] = NULL; 
            num_bonds[i] = 0; 
        }

        if (blk_crx == NULL)
        {
            vol_alloc_blk_crx2d(&blk_crx);
            blk_crx->assign_2dblk_type = _assign_2dblk_type;
            blk_crx->trk_wv_val = _trk_wv_val;
        }

        bm.curves = curves;
        bm.cur_bonds = cur_bonds;
        bm.first = first;
        bm.num_curves = num_curves;
        bm.num_bonds = num_bonds;

        nbc = 0;
        T = table_of_interface(ntg->grid_intfc);
        if(table_of_interface(ntg_intfc)->on_which_grid != which_grid)
        {
            printf("ERROR: in reconstruct_crx_intfc2d() \n");
            printf("last_which_grid not equals which_grid\n");
            clean_up(ERROR);
        }

        coy = (table_of_interface(ntg_intfc))->compon2d; 
        jc_blk_bond.prev = jc_blk_bond.next = NULL; 
        for (iy = smin[1]; iy < smax[1]; iy++)
        {
            for(coyx = coy[iy], ix = smin[0]; ix < smax[0]; ix++)
            {
                ixx = ix - smin[0];
                iyy = iy - smin[1];

                if (coyx[ix] == ONFRONT)
                {
                    nbc++; 
                    blk_crx->blkic[0] = ix;
                    blk_crx->blkic[1] = iy;
                    for (i = 0; i < 2; i++)
                    {
                        ip[0] = ix + i;
                        for (j = 0; j < 2; j++)
                        {
                            ip[1] = iy + j;
                            blk_crx->comp[i][j][0] =
                                     comp[d_index2d(ip,gmax)];
                            blk_crx->ix[i][j][0] = i; 
                            blk_crx->iy[i][j][0] = j;
                            blk_comp[i][j] = blk_crx->comp[i][j][0];
                            // blk_comp[i][j] = comp[d_index2d(ip,gmax)];
                        }
                    }
                    for (i = 0; i < 2; i++)
                    {
                        blk_ip[0] = ix;
                        blk_ip[1] = iy + i;
                        l = seg_index2d(blk_ip,EAST);
                        nc = seg_crx_count[l];
                        if(nc == 0 && blk_comp[0][i] != blk_comp[1][i])
                        {
                             printf("ERROR in reconstruct_crx_intfc2d() \n");
                             printf("nc = 0\n");
                             clean_up(ERROR); 
                        }
                        if(nc != 0)
                        {
                            for(nn = 0; nn < nc; nn++)
                            {
                                list = seg_crx_lists[l][nn];
                                crx = &crx_store[list];
                                blk_crx->crx[0][i][0][nn].hs = crx->hs;
                                blk_crx->crx[0][i][0][nn].p = crx->pt;
                                blk_crx->crx[0][i][0][nn].lcomp = crx->lcomp;
                                blk_crx->crx[0][i][0][nn].ucomp = crx->ucomp;
                            }
                             blk_crx->n_crx[0][i][0] = nc;
                        }
                        else
                        {
                             blk_crx->crx[0][i][0][0].p = NULL;
                             blk_crx->n_crx[0][i][0] = 0;
                        }

                        blk_ip[0] = ix + i;
                        blk_ip[1] = iy;
                        l = seg_index2d(blk_ip,NORTH);
                        nc = seg_crx_count[l];
                        if(nc == 0 && blk_comp[i][0] != blk_comp[i][1])
                        {
                             printf("ERROR in reconstruct_crx_intfc2d() \n");
                             printf("nc = 0, north\n");
                             clean_up(ERROR); 
                        }
                        if(nc != 0)
                        {
                            for(nn = 0; nn < nc; nn++)
                            {
                                list = seg_crx_lists[l][nn];
                                crx = &crx_store[list];
                                blk_crx->crx[1][0][i][nn].hs = crx->hs;
                                blk_crx->crx[1][0][i][nn].p = crx->pt;
                                blk_crx->crx[1][0][i][nn].lcomp = crx->lcomp;
                                blk_crx->crx[1][0][i][nn].ucomp = crx->ucomp;
                            }
                            blk_crx->n_crx[1][0][i] = nc;
                        }
                        else
                        {
                             blk_crx->crx[1][0][i][0].p = NULL;
                             blk_crx->n_crx[1][0][i] = 0;
                        }
                    }

                    if(! reconstruct_blk_intfc2d(blk_crx, &bm, &jc_blk_bond))
                    {
                        free_these(3, curves, cur_bonds, first);
                        DEBUG_LEAVE(reconstruct_crx_intfc2d)
                        return FUNCTION_FAILED;
                    }

                    /* For the volume construction purpose */
                    /* Maybe can be removed later */
                    if(blk_crx->m_crx == YES || 
                       blk_crx->blk_type == COMP3_BLOCK ||
                       blk_crx->blk_type == COMP1_BLOCK)
                    {
                        // printf("Ic[%d,%d] is set as no volume, nbc = %d\n",
                        //      ix, iy, nbc); 
                        ntg->blk_type[smax[0]*iy+ix] |= F_NO_GBS; 
                    }
                    /*
                    printf("ix, iy, nbc <%d, %d, %d>\n", ix, iy, nbc);
                    print_bond(bm->first);
                    */
                }
                else  /* The OFF_FRONT blk */
                {
                    for (i = 0; i < 2; i++)
                    {
                        ip[0] = ix + i;
                        for (j = 0; j < 2; j++)
                        {
                            ip[1] = iy + j;
                            blk_comp[i][j] = comp[d_index2d(ip,gmax)];
                        }
                    }
                    for (i = 0; i < 2; i++)
                    {
                        if(blk_comp[0][i] != blk_comp[1][i])
                        {
                            printf("ERROR In reconstruct_crx_intfc2d() \n");
                            printf("off front blk component not equal, case1 i = %d\n",
                                   i);
                            printf("%d != %d\n", blk_comp[0][i], blk_comp[1][i]);
                            printf("ip <%d, %d, %d>\n", ix, iy, i);
                            clean_up(ERROR);
                        }
                        if(blk_comp[i][0] != blk_comp[i][1])
                        {
                            printf("ERROR In reconstruct_crx_intfc2d() \n");
                            printf("off front blk component not equal, case2 i = %d\n", i);
                            printf("%d != %d\n", blk_comp[i][0], blk_comp[i][1]);
                            printf("ip <%d, %d, %d>\n", ix, iy, i);
                            clean_up(ERROR);
                        }
                    }
                }
            }
        }

        stitch_curves2d(ntg, &bm, &jc_blk_bond, smin, smax);

        free_these(4, curves, cur_bonds, first, num_bonds); 
        DEBUG_LEAVE(reconstruct_crx_intfc2d)
        return FUNCTION_SUCCEEDED;
}


/*
*               rebuild_intfc_at_crossings2d():
*/

EXPORT TRI_GRID *rebuild_intfc_at_crossings2d(
        RECT_GRID       *dual_grid,
        Front           *fr,
        Front           *oldfr,
        int             which_grid)
{
        TRI_GRID        *new_ntg;
        bool            sav_copy, sav_interp;
        int             i,smin[MAXD],smax[MAXD];
        INTERFACE       *intfc, *oldintfc, *current_intfc; 
        CURVE           **c; 
        BOND            *b; 
        int             dim, **ic_of_node; 
        int             n_fr_blk; 
        size_t          sizest;  

        DEBUG_ENTER(rebuild_intfc_at_crossings2d)

        sizest = fr->sizest; 
        dim = fr->rect_grid->dim; 
        intfc = fr->interf; 

        sav_copy = copy_intfc_states(); 
        sav_interp = interpolate_intfc_states(intfc);         

        if(which_grid ==  ON_COMP_GRID)
        {
            /*** Do not need state, only geometry is needed. ***/
            current_intfc = current_interface(); 
            set_copy_intfc_states(NO);
            if(NULL == (oldintfc = copy_interface(oldfr->interf)))
            {
                printf("ERROR rebuild_intfc_at_crossings2d\n");
                printf("copy interface failed\n");
                clean_up(ERROR); 
            } 
            set_current_interface(current_intfc); 
        }
        else
            oldintfc = oldfr->interf;

        set_copy_intfc_states(YES);
        set_size_of_intfc_state(sizest);

        interpolate_intfc_states(intfc) = YES;         

        scalar(&new_ntg,sizeof(TRI_GRID)); 
        zero_scalar(new_ntg,sizeof(TRI_GRID)); 
        new_ntg->Volume.blk = NULL; 
        new_ntg->grid_intfc = intfc; 
        set_tri_grid_rect_grids(new_ntg,dual_grid,intfc); 
  
        /*** clear the flags ****/
        remove_intfc_crx_flag(intfc); 
 
        if(which_grid == ON_COMP_GRID)
        {
            set_tri_grid_aug_comp_grids(new_ntg, fr->rect_grid);
            ic_of_node = set_node_index_list_on_grid(new_ntg, &new_ntg->aug_comp_grid); 
            new_ntg->grid_intfc->table->on_which_grid = ON_COMP_GRID; 
            if(! new_ntg->grid_intfc->table->fixed_grid)
            {
                printf("ERROR in rebuild_intfc_at_crossings2d\n");
                printf("GRID based reconstruction on COMP_GRID is "
                       "not implemented for the none-fixed_grid case\n");
                clean_up(ERROR);  
            }
            set_crx_storage_for_reconstruction_on_grid2d(new_ntg,new_ntg->grid_intfc,
                        &new_ntg->aug_comp_grid, ic_of_node, which_grid); 
            /* set_crx_storage_for_reconstruction(); */
            insert_grid_crossings2d(new_ntg,ic_of_node, new_ntg->grid_intfc,
                            &new_ntg->aug_comp_grid, which_grid);
            set_interface_topology_on_grid(new_ntg->grid_intfc,
			    &new_ntg->aug_comp_grid); 
            set_comp_at_crossings2d1(new_ntg, which_grid);
        }
        else
        {
            printf("ERROR: implement rebuild_intfc_at_crossings2d\n");
            printf("on the dual_grid\n");
            clean_up(ERROR); 
        }

        if(which_grid == ON_DUAL_GRID)
        {
            /*expanded_dual_grid*/		 
            for (i = 0; i < dim; i++)
            {
                smin[i] = 0;
                smax[i] = new_ntg->rect_grid.gmax[i]; 
            }
        }
        else
        {
            for (i = 0; i < dim; i++)
            {
                smin[i] = 0;
                smax[i] = new_ntg->aug_comp_grid.gmax[i]; /*expanded_comp_grid*/
            }
        }

        if (remove_unphysical_crossings2d(oldintfc,new_ntg,
               &n_fr_blk,smin,smax,which_grid) == FUNCTION_FAILED)
        // if (remove_unphysical_crossings2d(oldfr->interf,new_ntg,
        //        &n_fr_blk,smin,smax,which_grid) == FUNCTION_FAILED)
        {
            free_grid_lines(&new_ntg->aug_comp_grid);
            free(ic_of_node);
            printf("ERROR: IN rebuild_intfc_at_crossings2d(),\n");
            printf("remove_unphysical_crossings2d() failed.\n");
            set_copy_intfc_states(sav_copy);
            interpolate_intfc_states(intfc) = sav_interp;         
            if(which_grid ==  ON_COMP_GRID)
                delete_interface(oldintfc); 
            clean_up(ERROR);
            DEBUG_LEAVE(rebuild_intfc_at_crossings2d)
            return NULL;
        }        
        if(which_grid ==  ON_COMP_GRID)
            delete_interface(oldintfc); 

        if (reconstruct_crx_intfc2d(new_ntg,n_fr_blk,smin,smax,which_grid)
            ==FUNCTION_FAILED)
        {
            new_ntg->grid_intfc = NULL;
            free_grid_lines(&new_ntg->aug_comp_grid);
            free_tri_grid(new_ntg);
            free(new_ntg);
            free(ic_of_node);
            printf("ERROR: IN rebuild_intfc_at_crossings2d(),\n");
            printf("reconstruct_crx_intfc2d() failed.\n");
            set_copy_intfc_states(sav_copy);
            interpolate_intfc_states(intfc) = sav_interp;         
            clean_up(ERROR);
            DEBUG_LEAVE(rebuild_intfc_at_crossings2d)
            return NULL;
        }

        /**** Should replace this function with faster Alg. ****/
        reset_intfc_num_points(new_ntg->grid_intfc); 

        new_ntg->grid_intfc->table->new_grid = YES;

        /*** Need to free grid lines on aug_comp_grid
         *** tri_grid comp_grid storage. ic_of_node
         ***/
        free_grid_lines(&new_ntg->aug_comp_grid); 
        free(ic_of_node);

        set_copy_intfc_states(sav_copy);
        interpolate_intfc_states(intfc) = sav_interp;         

        /* Probably it is a good idea to transfer this 
         * new_ntg structure out to new wave, so there 
         * is no need to build comp_grid crx again. 
         */
        DEBUG_LEAVE(rebuild_intfc_at_crossings2d)
	return new_ntg; 
}

LOCAL void remove_intfc_crx_flag(
        INTERFACE    *intfc)
{
        CURVE        **c;
        BOND         *b; 

        for(c = intfc->curves; c && *c; c++)
        {
            b = (*c)->first;
            b->start->crx = NO; 
            while(b != NULL)
            {
                b->end->crx = NO; 
                b = b->next;
            } 
        }
}

EXPORT void reset_pt_index_on_intfc(
        INTERFACE     *intfc)
{

        CURVE         **cc;
        BOND          *b; 
        int           i, index;
     
        for(i = 0, cc = intfc->curves; cc && *cc; cc++, i++)
        {
            if(is_passive_boundary(Hyper_surf(*cc)) ||
               is_subdomain_boundary(Hyper_surf(*cc)) ) 
                continue; 
            (*cc)->curv_id = i; 

            index = 0; 
            (*cc)->sindx = 0; 
            b = (*cc)->first;

            /*      
            while(b != NULL)
            {
                b->start->indx = index;
                b->start->curv_id = (*cc)->curv_id;
                index++;
                b = b->next;
            }
            if(! is_closed_curve(*cc))
            {
                (*cc)->last->end->indx = index;
                (*cc)->last->end->curv_id = (*cc)->curv_id;
                index++;
            }
            (*cc)->eindx = index - 1;
            */
           
            while(b != NULL)
            {
                /* To avoid several curves end (start)
                 * at one node, the node point does not
                 * mark with index.
                 */
                if(b == (*cc)->first)
                {
                    if(is_closed_curve(*cc))
                    {
                        b->start->indx = index; 
                        b->start->curv_id = (*cc)->curv_id; 
                        index++;  
                    }
                    else
                    {
                        b->start->indx = -100; 
                        b->start->curv_id = -1; 
                    }
                }
                else
                {
                    b->start->indx = index;
                    b->start->curv_id = (*cc)->curv_id;
                    index++;
                } 
                b = b->next;
            }

            if(! is_closed_curve(*cc))
            {
                /*
                (*cc)->last->end->indx = index; 
                (*cc)->last->end->curv_id = (*cc)->curv_id; 
                index++;  
                */
                (*cc)->last->end->indx = -100; 
                (*cc)->last->end->curv_id = -1; 
            }
            (*cc)->eindx = index - 1;  
        }
}

LOCAL int _trk_wv_val(
        CURVE        *c)
{
        if(wave_type(c) >= FIRST_PHYSICS_WAVE_TYPE)
            return YES;
        return NO; 
}

LOCAL void _assign_2dblk_type(
        BLK_CRX2    *blk_crx,
        int         num_crx,
        BBI_POINT   *crxs[])
{
        int  i, j, k;
        COMPONENT  *comp = blk_crx->comps;
        int  num_comp, diff;
        HYPER_SURF *hs[4]  = {NULL, NULL, NULL, NULL};
        int  num_curves, num_waves;

        blk_crx->m_crx = blk_crx->bdry = NO; 
        for (i = 0; i < 2; i++)
        {
            if(blk_crx->n_crx[0][i][0] >= 2)
            {
                blk_crx->m_crx = YES; 
                break;
            }
            if(blk_crx->n_crx[1][0][i] >= 2)
            {
                blk_crx->m_crx = YES;
                break; 
            }
        }

        comp[0] = blk_crx->comp[0][0][0];
        num_comp = 1;
        for(i = 0; i < 2; i++)
        {
            for(j = 0; j < 2; j++)
            {
                diff = YES;
                for(k = 0; k < num_comp; k++)
                {
                    if(blk_crx->comp[i][j][0] == comp[k])
                    {
                        diff = NO;
                        break;
                    }
                }
                if(diff == YES)
                {
                    comp[num_comp] = blk_crx->comp[i][j][0];
                    num_comp++;
                }
            }
        }
        blk_crx->num_comps = num_comp;
        switch(num_comp)
        {
        case 1:
            if(blk_crx->m_crx != YES)
            {
                printf("ERROR in assign_2dblk_type\n");
                printf("num_comps inside blk = %d\n", num_comp);
                printf("multi crx = %d\n", blk_crx->m_crx); 
                clean_up(ERROR);
            }
            else
            blk_crx->blk_type = COMP1_BLOCK;
        case 2:
            blk_crx->blk_type = COMP2_BLOCK;
        break;
        case 3:
            blk_crx->blk_type = COMP3_BLOCK;
        break;
        default:
            printf("ERROR in assign_2dblk_type\n");
            printf("num_comps inside blk = %d\n", num_comp);
            printf("multi crx = %d\n", blk_crx->m_crx); 
            clean_up(ERROR);
        }

        hs[0] = crxs[0]->hs;
        num_curves = 1;
        for (i = 0; i < num_crx; i++)
        {
            diff = YES;
            for(j = 0; j < num_curves; j++)
            {
                if (crxs[i]->hs == hs[j])
                {
                    diff = NO;
                    break;
                }
            }
            if(diff == YES)
            {
                hs[num_curves] = crxs[i]->hs;
                num_curves++;
            }
        }
        blk_crx->num_curves = num_curves;

        hs[0] = crxs[0]->hs;
        num_waves = 1;
        for (i = 0; i < num_crx; i++)
        {
            diff = YES;
            if(wave_type(crxs[i]->hs) < FIRST_PHYSICS_WAVE_TYPE)
                blk_crx->bdry = YES;     
            for(j = 0; j < num_waves; j++)
            {
                if (wave_type(crxs[i]->hs) == wave_type(hs[j]))
                {
                    diff = NO;
                    break;
                }
            }
            if(diff == YES)
            {
                hs[num_waves] = crxs[i]->hs;
                num_waves++;
            }
        }
        blk_crx->num_waves = num_waves;
}

LOCAL   void set_comp_at_crossings2d1(
        TRI_GRID        *ntg,
        int             which_grid)
{
        int             minx, miny, maxx, maxy;
        int             i, j, ix, iy;
        struct Table    *T;
        BOND            ****bonds, **byx;
        CURVE           ****curves, **cyx;
        COMPONENT       **coy, *coyx, c;
        int             num_of_bonds;
        int             smin[MAXD], smax[MAXD], ip[MAXD], ipn[MAXD];
        int             *seg_crx_count;
        int             **seg_crx_lists;
        CRXING          *crx_store, *crx;
        int             m,k,nc,list;

        DEBUG_ENTER(set_comp_at_crossings2d1)

        if(which_grid == ON_DUAL_GRID)
        {
            for (i = 0; i < 3; i++)
            {
                smin[i] = 0;
                smax[i] = ntg->rect_grid.gmax[i]; /*expanded_dual_grid*/
            }
            seg_crx_count = ntg->seg_crx_count;
            seg_crx_lists = ntg->seg_crx_lists;
            crx_store = ntg->crx_store;
        }
        else
        {
            for (i = 0; i < 3; i++)
            {
                smin[i] = 0;
                smax[i] = ntg->aug_comp_grid.gmax[i]; /*expanded_comp_grid*/
            }
            seg_crx_count = ntg->c_seg_crx_count;
            seg_crx_lists = ntg->c_seg_crx_lists;
            crx_store = ntg->c_crx_store;
        }

        T = table_of_interface(ntg->grid_intfc);
        if(T->on_which_grid != which_grid)
        {
            (void) printf("ERROR in set_comp_at_crossings2d1(), "
                          "should make_bond_comp_lists() on grid[%d]\n", which_grid);
            clean_up(ERROR);
        }

        coy = T->compon2d;
        for (iy = smin[1]; iy < smax[1]; iy++)
        {
            ip[1] = iy;
            for (coyx = coy[iy], ix = smin[0]; ix < smax[0]; ix++)
            {
                ip[0] = ix; c = coyx[ix];
                if(c == ONFRONT)
                {
                    byx = T->bonds[iy][ix];
                    cyx = T->curves[iy][ix];
                    num_of_bonds = T->num_of_bonds[iy][ix];

                    /* first EAST  */
                    ipn[0] = ip[0]+1;  ipn[1] = ip[1];
                    k = seg_index2d(ip,EAST);
                    nc = seg_crx_count[k];
                    for (i = 0; i < nc; i++)
                    {
                        list = seg_crx_lists[k][i];
                        crx = &(crx_store[list]);
                        for(j = 0; j <num_of_bonds; j++)
                        {
                            if(point_on_bond2d1(crx->pt, byx[j]))
                            {
                                if(crx->crossing_direction ==  BELOW_TO_ABOVE)
                                {
                                    crx->ucomp =  positive_component(cyx[j]);
                                    crx->lcomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if( crx->crossing_direction == ABOVE_TO_BELOW)
                                {
                                    crx->lcomp =  positive_component(cyx[j]);
                                    crx->ucomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if(crx->crossing_direction == LEFT_TO_RIGHT)
                                {
                                    crx->lcomp =  positive_component(cyx[j]);
                                    crx->ucomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if(crx->crossing_direction == RIGHT_TO_LEFT)
                                {
                                    crx->ucomp =  positive_component(cyx[j]);
                                    crx->lcomp =  negative_component(cyx[j]);
                                    break;
                                }
                            }
                        }
                    }
                    /* first NORTH  */
                    ipn[0] = ip[0]+1;  ipn[1] = ip[1];
                    k = seg_index2d(ip,NORTH);
                    nc = seg_crx_count[k];
                    for (i = 0; i < nc; i++)
                    {
                        list = seg_crx_lists[k][i];
                        crx = &(crx_store[list]);
                        for(j = 0; j <num_of_bonds; j++)
                        {
                            if(point_on_bond2d1(crx->pt, byx[j]))
                            {
                                if(crx->crossing_direction ==  BELOW_TO_ABOVE)
                                {
                                    crx->ucomp =  positive_component(cyx[j]);
                                    crx->lcomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if( crx->crossing_direction == ABOVE_TO_BELOW)
                                {
                                    crx->lcomp =  positive_component(cyx[j]);
                                    crx->ucomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if(crx->crossing_direction == LEFT_TO_RIGHT)
                                {
                                    crx->lcomp =  positive_component(cyx[j]);
                                    crx->ucomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if(crx->crossing_direction == RIGHT_TO_LEFT)
                                {
                                    crx->ucomp =  positive_component(cyx[j]);
                                    crx->lcomp =  negative_component(cyx[j]);
                                    break;
                                }
                            }
                        }
                    }
                }
            }
        }

        /*  the top  */
        iy = smax[1];
        for (coyx = coy[smax[1]-1],
                 ix = smin[0]; ix < smax[0]; ix++)
        {
             ip[1] = smax[1]; ip[0] = ix; c = coyx[ix];
             if(c == ONFRONT)
             {
                    byx = T->bonds[iy-1][ix];
                    cyx = T->curves[iy-1][ix];
                    num_of_bonds = T->num_of_bonds[iy-1][ix];
                    /* first EAST  */
                    ipn[0] = ip[0]+1;  ipn[1] = ip[1];
                    k = seg_index2d(ip,EAST);
                    nc = seg_crx_count[k];
                    for (i = 0; i < nc; i++)
                    {
                        list = seg_crx_lists[k][i];
                        crx = &(crx_store[list]);
                        for(j = 0; j <num_of_bonds; j++)
                        {
                            if(point_on_bond2d1(crx->pt, byx[j]))
                            {
                                if(crx->crossing_direction ==  BELOW_TO_ABOVE)
                                {
                                    crx->ucomp =  positive_component(cyx[j]);
                                    crx->lcomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if( crx->crossing_direction == ABOVE_TO_BELOW)
                                {
                                    crx->lcomp =  positive_component(cyx[j]);
                                    crx->ucomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if(crx->crossing_direction == LEFT_TO_RIGHT)
                                {
                                    crx->lcomp =  positive_component(cyx[j]);
                                    crx->ucomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if(crx->crossing_direction == RIGHT_TO_LEFT)
                                {
                                    crx->ucomp =  positive_component(cyx[j]);
                                    crx->lcomp =  negative_component(cyx[j]);
                                    break;
                                }
                            }
                        }
                    }
             }
        }

        /* The right   */
        ix = smax[0];
        for (iy = smin[1]; iy < smax[1]; iy++)
        {
            coyx = coy[iy];
            ip[1] = iy; ip[0] = ix; c = coyx[ix-1];
            if(c == ONFRONT)
            {
                    byx = T->bonds[iy][ix-1];
                    cyx = T->curves[iy][ix-1];
                    num_of_bonds = T->num_of_bonds[iy][ix-1];
                    /* NORTH  */
                    ipn[0] = ip[0]+1;  ipn[1] = ip[1];
                    k = seg_index2d(ip,NORTH);
                    nc = seg_crx_count[k];
                    for (i = 0; i < nc; i++)
                    {
                        list = seg_crx_lists[k][i];
                        crx = &(crx_store[list]);
                        for(j = 0; j <num_of_bonds; j++)
                        {
                            if(point_on_bond2d1(crx->pt, byx[j]))
                            {
                                if(crx->crossing_direction ==  BELOW_TO_ABOVE)
                                {
                                    crx->ucomp =  positive_component(cyx[j]);
                                    crx->lcomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if( crx->crossing_direction == ABOVE_TO_BELOW)
                                {
                                    crx->lcomp =  positive_component(cyx[j]);
                                    crx->ucomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if(crx->crossing_direction == LEFT_TO_RIGHT)
                                {
                                    crx->lcomp =  positive_component(cyx[j]);
                                    crx->ucomp =  negative_component(cyx[j]);
                                    break;
                                }
                                if(crx->crossing_direction == RIGHT_TO_LEFT)
                                {
                                    crx->ucomp =  positive_component(cyx[j]);
                                    crx->lcomp =  negative_component(cyx[j]);
                                    break;
                                }
                            }
                        }
                    }
            }
        }

        DEBUG_LEAVE(set_comp_at_crossings2d1)
}

LOCAL   int     point_on_bond2d1(
        POINT            *pt,
        BOND             *b)
{
        if(b->start == pt)    return YES;
        if(b->end == pt)      return YES;
        return NO;
}


EXPORT void free_frac_state_storage(
        TRI_GRID      *ntg)
{
        int         i, j, max[MAXD], min[MAXD];
        int         icrds[MAXD];  

        Set_free(ntg,vol_rect_state_storage); ntg->vol_rect_state_storage = NULL; 
        for (i = 0; i < 2; i++)
        {
            min[i] = 0;
            max[i] = ntg->comp_grid.gmax[i];
            min[i] -= ntg->comp_grid.lbuf[i];
            max[i] += ntg->comp_grid.ubuf[i];
        }
        if(ntg->alloc.vol_states == 1)
        {
            for (i=min[1]; i<max[1]; i++)
            {
                for (j=min[0]; j<max[0]; j++)
                {
                    icrds[1] = i;
                    icrds[0] = j;
                    if(Comp_blk(icrds,ntg->vol_states,ntg) != NULL)
                        free(Comp_blk(icrds,ntg->vol_states,ntg)); 
                }
            }
        }
        Set_free(ntg,vol_states); ntg->vol_states = NULL;
}


EXPORT void free_comp_tri_storage(
        TRI_GRID      *ntg)
{
        Set_free(ntg,c_node_points); ntg->c_node_points = ntg->c_cg_npts = NULL;
        Set_free(ntg,c_components); ntg->c_components = ntg->c_cg_comps = NULL;
        Set_free(ntg,blk_type); ntg->blk_type = NULL;  
        free_frac_state_storage(ntg); 
}

EXPORT void free_comp_crx_lists(
	TRI_GRID      *ntg)
{
        Set_free(ntg,c_seg_crx_count); ntg->c_seg_crx_count = NULL;
        Set_free(ntg,c_seg_crx_lists); ntg->c_seg_crx_lists = NULL;
        Set_free(ntg,c_seg_crx_lists_store); ntg->c_seg_crx_lists_store = NULL;
        Set_free(ntg,c_crx_store); ntg->c_crx_store = NULL;
}

#endif /* if defined(CONSERVATIVE_ALG) */
#endif /* if defined(TWOD) */

