/*
*				tri3dv0.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Three dimensional specific tetrazation functions.
*/

#if defined(THREED)

#include <tri/tri3ddefs.h>

	/* LOCAL Function Declarations */
LOCAL	bool	set_comp_at_vertex(CRXING*,POINT*,TRI*,SURFACE*,int);
LOCAL	int	add_to_edge_list(TRI*,int,CRX_STORE*,int*,float*,int,int);
LOCAL	int	count_block_crossings(TRI_GRID*,TRI**,int,int*);
LOCAL	int	count_grid_crossings_on_intfc3dv0(TRI_GRID*);
LOCAL	int	set_components3d(TRI_GRID*,INTERFACE*,int);
LOCAL   void   	add_to_crx_list(int*,int,TRI_GRID*,TRI*,SURFACE*,CRXING*,
				CRX_STORE*,int*,int*,float*,int,int);
LOCAL	void	insert_block_crossings(TRI_GRID*,TRI**,SURFACE**,int,int*,int*);
LOCAL	void	insert_grid_crossings3dv0(TRI_GRID*,Front*);
LOCAL	void	interpolate_crx_pt_states_on_tri(TRI_GRID*,POINT*,TRI*,
				SURFACE*);
LOCAL	void	interpolate_crx_pt_states_on_edge(TRI_GRID*,POINT*,TRI*,
				SURFACE*,int);
LOCAL	void	linear_interp_coefs_3d_tri(float*,float*,TRI*);
LOCAL	void	set_crx_structure_storage3dv0(TRI_GRID*);
LOCAL	void	set_interpolation_storage3dv0(TRI_GRID*);
LOCAL	void	set_tri3d_tolerancesv0(TRI_GRID*);
LOCAL   void    print_random_scale(TRI_GRID*);

LOCAL	float 	crx_tol;

EXPORT	void	set_tri3dv0_tri_grid_hooks(
	TRI_GRID_HOOKS *tri_grid_hooks)
{
	tri_grid_hooks->_set_components = set_components3d;
	tri_grid_hooks->_set_crx_structure_storage3d =
	    set_crx_structure_storage3dv0;
	tri_grid_hooks->_count_grid_crossings_on_intfc3d = 
	    count_grid_crossings_on_intfc3dv0;
	tri_grid_hooks->_set_interpolation_storage3d =
	    set_interpolation_storage3dv0;
	tri_grid_hooks->_insert_grid_crossings3d = insert_grid_crossings3dv0;
	tri_grid_hooks->_set_tri3d_tolerances = set_tri3d_tolerancesv0;
}		/*end set_tri3dv0_tri_grid_hooks*/

LOCAL	int set_components3d(
	TRI_GRID	*ntg,
	INTERFACE	*intfc,
        int             which_grid)
{
	COMPONENT	*comp = ntg->components;
	COMPONENT	***coz, **cozy, *cozyx, c;
	INTERFACE	*ntg_intfc = ntg->grid_intfc;
	bool		status;
	int             xmax, ymax, zmax;
	int		*gmax = ntg->rect_grid.gmax;
	int		ix, iy, iz, i, n_reg_node;
	int		ixx,iyy,izz;
	int		smin[3], smax[3];
	DEBUG_ENTER(set_components3d)

	n_reg_node = (gmax[0]+1)*(gmax[1]+1)*(gmax[2]+1);

	for (i = 0; i < n_reg_node; ++i)
	    comp[i] = NO_COMP;

	if (ntg_intfc->surfaces == NULL)
	{
	    int istatus = set_untracked_components(ntg,intfc);
	    DEBUG_LEAVE(set_components3d)
	    return istatus;
        } 

	xmax = gmax[0];   /*                   */
	ymax = gmax[1];   /* global to tri3d.c */
	zmax = gmax[2];   /*                   */

			/* set lattice points of expanded_dual_grid */
	/*
	coz = (table_of_interface(ntg_intfc))->compon3d;
	for (iz = 0; iz < zmax; ++iz)
	{
	    for (cozy = coz[iz], iy = 0; iy < ymax; ++iy)
	    {
		for (cozyx = cozy[iy], ix = 0; ix < xmax; ++ix)
		{
		    c = cozyx[ix];
		    if (c != ONFRONT)
		    {
			for (izz = iz; izz <= iz + 1; ++izz)
			{
			    for (iyy = iy; iyy <= iy + 1; ++iyy)
			    {
				for (ixx = ix; ixx <= ix + 1; ++ixx)
				{
				    comp[d_index3d(ixx,iyy,izz,gmax)] = c;
				}
			    }
			}
		    }
		}
	    }
	}
	*/

	for (i = 0; i < 3; ++i)
	{
	    smin[i] = 0;
	    smax[i] = gmax[i];
	}
	status = track_comp_through_crxings3d(smin,smax,gmax,ntg,MULTIPLE);
	status = pp_min_status(status);
	if (status == FUNCTION_FAILED)
	{
	    screen("ERROR in set_components3d(), "
	           "track_comp_through_crxings3d() failed!\n");
	    print_interface(intfc);
	    clean_up(ERROR);
	}
	DEBUG_LEAVE(set_components3d)
	if (debugging("random_scale"))
            print_random_scale(ntg);
	return GOOD_STEP;
}		/*end set_components3d*/


LOCAL	void set_crx_structure_storage3dv0(
	TRI_GRID	*ntg)
{
	int		n_crx,n_segs;
	int             xmax, ymax, zmax;
	DEBUG_ENTER(set_crx_structure_storage3dv0)

	xmax = ntg->rect_grid.gmax[0];	
	ymax = ntg->rect_grid.gmax[1];	
	zmax = ntg->rect_grid.gmax[2];

	n_segs =     xmax   *(ymax+1)*(zmax+1) +
	            (xmax+1)* ymax   *(zmax+1) +
		    (xmax+1)*(ymax+1)* zmax;
	VECTOR(ntg,seg_crx_count,n_segs,INT);/* Note VECTOR initializes */
					     /* storage to zero */

	n_crx = count_grid_crossings_on_intfc3d(ntg);
	ntg->n_crx = n_crx;
	init_seg_crx_lists(ntg,n_crx,n_segs,ON_DUAL_GRID);

	ntg->n_bilin_els = xmax*ymax*zmax;
	ntg->n_node_points = (xmax+1)*(ymax+1)*(zmax+1);
	alloc_node_points(ntg,ntg->n_node_points);
	alloc_blk_els0(ntg,ntg->n_bilin_els);

#if defined(DEBUG_TRI_GRID)
	if (debugging("TRI_storage"))
	{
	    (void) printf("gmax %d %d %d\n",xmax,ymax,zmax);
	}
#endif /* defined(DEBUG_TRI_GRID) */
	DEBUG_LEAVE(set_crx_structure_storage3dv0)
}		/*end set_crx_structure_storage3dv0*/


LOCAL int count_grid_crossings_on_intfc3dv0(
	TRI_GRID	*ntg)
{
	struct Table	*T;
	TRI		*****tz, ****tzy, ***tzyx;
	int		i,j,k;
	int		***ntz, **ntzy, *ntzyx;
	int		icrds[MAXD];
	int		n_crx = 0;
	int             xmax, ymax, zmax;
	DEBUG_ENTER(count_grid_crossings_on_intfc3dv0)

        set_dual_interface_topology(ntg);
	T = table_of_interface(ntg->grid_intfc);

	xmax = ntg->rect_grid.gmax[0];
	ymax = ntg->rect_grid.gmax[1];
	zmax = ntg->rect_grid.gmax[2];

	for (k = 0, ntz = T->num_of_tris, tz = T->tris;
						k < zmax; ++k, ++ntz, ++tz)
	{
	    for (j = 0, ntzy = *ntz, tzy = *tz;
						j < ymax; ++j, ++ntzy, ++tzy)
	    {
		for (i = 0, ntzyx = *ntzy, tzyx = *tzy;
						i < xmax; ++i, ++ntzyx, ++tzyx)
		{
		    if (*ntzyx == 0)
			continue;
		    icrds[0] = i; icrds[1] = j; icrds[2] = k;
		    n_crx += count_block_crossings(ntg,*tzyx,*ntzyx,icrds);
		}
	    }
	}
	DEBUG_LEAVE(count_grid_crossings_on_intfc3dv0)
	return n_crx;
}		/*end count_grid_crossings_on_intfc3dv0*/

LOCAL void set_interpolation_storage3dv0(
        TRI_GRID *ntg)
{
        INTERFACE *intfc = ntg->grid_intfc;
        set_dual_interface_topology(ntg);
        if (interpolation_method(ntg) == COMPLETE_TRIANGULATION)
        {
            ntg->n_lin_els = max_num_3d_lin_els(intfc);
            VECTOR(ntg,bilin_els,ntg->n_bilin_els,sizeof(BILINEAR_ELEMENT));
            VECTOR(ntg,lin_els,ntg->n_lin_els,sizeof(LINEAR_ELEMENT));
            ntg->n_fr_pts = intfc->num_points;
        }
        if (interpolation_method(ntg) == ELEMENT_ON_THE_FLY)
        {
            ntg->n_pcs = count_num_pcs3d(ntg);
            VECTOR(ntg,pcs,ntg->n_pcs,sizeof(POINT_COMP_ST));
            ntg->n_fr_pts = intfc->num_points;
        }
        VECTOR(ntg,front_points,ntg->n_fr_pts,sizeof(TG_PT));
}               /*end set_interpolation_storage3dv0*/


/*ARGSUSED*/
LOCAL void insert_grid_crossings3dv0(
	TRI_GRID	*ntg,
	Front		*front)
{
	struct Table	*T;
	TRI		*****tz, ****tzy, ***tzyx;
	SURFACE		*****sz, ****szy, ***szyx;
	int		***ntz, **ntzy, *ntzyx;
	int		icrds[MAXD];
	int		i,j,k;
	int		crx_index = 0;
	int             xmax, ymax, zmax;
	DEBUG_ENTER(insert_grid_crossings3dv0)

	set_dual_interface_topology(ntg);
	T = table_of_interface(ntg->grid_intfc);

	xmax = ntg->rect_grid.gmax[0];
	ymax = ntg->rect_grid.gmax[1];
	zmax = ntg->rect_grid.gmax[2];

	for (k = 0, ntz = T->num_of_tris, tz = T->tris, sz = T->surfaces;
	    k < zmax; ++k, ++ntz, ++tz, ++sz)
	{
	    for (j = 0, ntzy = *ntz, tzy = *tz, szy = *sz; j < ymax;
		++j, ++ntzy, ++tzy, ++szy)
	    {
		for (i = 0, ntzyx = *ntzy, tzyx = *tzy, szyx = *szy; i < xmax;
		    ++i, ++ntzyx, ++tzyx, ++szyx)
		{
		    if (*ntzyx == 0)
			continue;
		    icrds[0] = i; icrds[1] = j; icrds[2] = k;
		    insert_block_crossings(ntg,*tzyx,*szyx,*ntzyx,icrds,
					   &crx_index);
		}
	    }
	}
	DEBUG_LEAVE(insert_grid_crossings3dv0)
}		/*end insert_grid_crossings3dv0*/


LOCAL void set_tri3d_tolerancesv0(
	TRI_GRID	*ntg)
{
	float	     *h = ntg->rect_grid.h;
	float	     hmin;
	int	     i;
	static	bool first = YES;

	hmin = h[0];
	for (i = 1; i < 3; ++i)
	{
	    if (hmin > h[i])
		hmin = h[i];
	}

	ltol(ntg) = 0.001*hmin*TOL; /*TOLERANCE*/
	stol(ntg) = ltol(ntg)*hmin;
	vtol(ntg) = stol(ntg)*hmin;
	if (first)
        {
            first = NO;
            crx_tol = 0.0001*hmin;
        }
        else
            crx_tol = 0.000001*hmin; /*TOLERANCE*/
}		/*end set_tri3d_tolerancesv0*/

enum { MAX_EDGE_CRX = 20 }; /*TOLERANCE*/

LOCAL int count_block_crossings(
	TRI_GRID	*ntg,
	TRI		**tris,
	int		num_tris,
	int		*ic)
{
	int		n_blk_crx;
	float		coords[MAXD],crds_crx[MAXD];
	float           *L, *h;
	int		i,n_ecrx;	/* number of crossings on edge */
	int		*seg_crx_count = ntg->seg_crx_count;
	int		index;
	int		iv,ie;
	int             xmax, ymax, zmax;
	static CRX_STORE	*crx_list;
	DEBUG_ENTER(count_block_crossings)

	xmax = ntg->rect_grid.gmax[0];
	ymax = ntg->rect_grid.gmax[1];
	zmax = ntg->rect_grid.gmax[2];
	set_seg_index3d_globals(ntg->rect_grid.gmax);
	L = ntg->rect_grid.L;
	h = ntg->rect_grid.h;
	if (crx_list == NULL)
	    vector(&crx_list,MAX_EDGE_CRX,sizeof(CRX_STORE));
	for (i = 0; i < 3; ++i)
	    coords[i] = L[i] + ic[i]*h[i];
	n_blk_crx = 0;

		/* Count EAST face crossings */

	n_ecrx = 0;
	for (i = 0; i < num_tris; ++i)
	{
	    if (tri_edge_crossing(tris[i],coords,crds_crx,0,&iv,&ie,h))
	    {
		n_blk_crx += add_to_edge_list(tris[i],0,crx_list,
				              &n_ecrx,crds_crx,iv,ie);
	    }
	}
	index = seg_index3d(ic[0],ic[1],ic[2],EAST);
	seg_crx_count[index] = n_ecrx;

		/* Count NORTH face crossings */

	n_ecrx = 0;
	for (i = 0; i < num_tris; ++i)
	{
	    if (tri_edge_crossing(tris[i],coords,crds_crx,1,&iv,&ie,h))
	    {
		n_blk_crx += add_to_edge_list(tris[i],1,crx_list,
				              &n_ecrx,crds_crx,iv,ie);
	    }
	}
	index = seg_index3d(ic[0],ic[1],ic[2],NORTH);
	seg_crx_count[index] = n_ecrx;

		/* Count UPPER face crossings */

	n_ecrx = 0;
	for (i = 0; i < num_tris; ++i)
	{
	    if (tri_edge_crossing(tris[i],coords,crds_crx,2,&iv,&ie,h))
	    {
		n_blk_crx += add_to_edge_list(tris[i],2,crx_list,
				              &n_ecrx,crds_crx,iv,ie);
	    }
	}
	index = seg_index3d(ic[0],ic[1],ic[2],UPPER);
	seg_crx_count[index] = n_ecrx;

	if (ic[0] != xmax-1 && ic[1] != ymax-1 && ic[2] != zmax-1)
	{
	    DEBUG_LEAVE(count_block_crossings)
	    return n_blk_crx;
	}

	if (ic[0] == xmax - 1)
	{
	    ic[0] += 1;
	    coords[0] += h[0];
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,1,&iv,&ie,h))
		{
		    n_blk_crx += add_to_edge_list(tris[i],1,crx_list,
				                  &n_ecrx,crds_crx,iv,ie);
		}
	    }
	    index = seg_index3d(ic[0],ic[1],ic[2],NORTH);
	    seg_crx_count[index] = n_ecrx;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,2,&iv,&ie,h))
		{
		    n_blk_crx += add_to_edge_list(tris[i],2,crx_list,
				                  &n_ecrx,crds_crx,iv,ie);
		}
	    }
	    index = seg_index3d(ic[0],ic[1],ic[2],UPPER);
	    seg_crx_count[index] = n_ecrx;
	    ic[0] -= 1;
	    coords[0] -= h[0];
	}
	if (ic[1] == ymax - 1)
	{
	    ic[1] += 1;
	    coords[1] += h[1];
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,0,&iv,&ie,h))
		{
		    n_blk_crx += add_to_edge_list(tris[i],0,crx_list,
				                  &n_ecrx,crds_crx,iv,ie);
		}
	    }
	    index = seg_index3d(ic[0],ic[1],ic[2],EAST);
	    seg_crx_count[index] = n_ecrx;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,2,&iv,&ie,h))
		{
		    n_blk_crx += add_to_edge_list(tris[i],2,crx_list,
				                  &n_ecrx,crds_crx,iv,ie);
		}
	    }
	    index = seg_index3d(ic[0],ic[1],ic[2],UPPER);
	    seg_crx_count[index] = n_ecrx;
	    coords[1] -= h[1];
	    ic[1] -= 1;
	}
	if (ic[2] == zmax - 1)
	{
	    ic[2] += 1;
	    coords[2] += h[2];
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,0,&iv,&ie,h))
		{
		    n_blk_crx += add_to_edge_list(tris[i],0,crx_list,
						  &n_ecrx,crds_crx,iv,ie);
		}
	    }
	    index = seg_index3d(ic[0],ic[1],ic[2],EAST);
	    seg_crx_count[index] = n_ecrx;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,1,&iv,&ie,h))
		{
		    n_blk_crx += add_to_edge_list(tris[i],1,crx_list,&n_ecrx,
						  crds_crx,iv,ie);
		}
	    }
	    index = seg_index3d(ic[0],ic[1],ic[2],NORTH);
	    seg_crx_count[index] = n_ecrx;
	    coords[2] -= h[2];
	    ic[2] -= 1;
	}
	if (ic[0] == xmax-1 && ic[1] == ymax-1)
	{
	    ic[0] += 1;
	    ic[1] += 1;
	    coords[0] += h[0];
	    coords[1] += h[1];
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,2,&iv,&ie,h))
		{
		    n_blk_crx += add_to_edge_list(tris[i],2,crx_list,
				                  &n_ecrx,crds_crx,iv,ie);
		}
	    }
	    index = seg_index3d(ic[0],ic[1],ic[2],UPPER);
	    seg_crx_count[index] = n_ecrx;
	    coords[0] -= h[0];
	    coords[1] -= h[1];
	    ic[0] -= 1;
	    ic[1] -= 1;
	}
	if (ic[0] == xmax-1 && ic[2] == zmax-1)
	{
	    ic[0] += 1;
	    ic[2] += 1;
	    coords[0] += h[0];
	    coords[2] += h[2];
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,1,&iv,&ie,h))
		{
		    n_blk_crx += add_to_edge_list(tris[i],1,crx_list,
				                  &n_ecrx,crds_crx,iv,ie);
		}
	    }
	    index = seg_index3d(ic[0],ic[1],ic[2],NORTH);
	    seg_crx_count[index] = n_ecrx;
	    coords[0] -= h[0];
	    coords[2] -= h[2];
	    ic[0] -= 1;
	    ic[2] -= 1;
	}
	if (ic[1] == ymax-1 && ic[2] == zmax-1)
	{
	    ic[1] += 1;
	    ic[2] += 1;
	    coords[1] += h[1];
	    coords[2] += h[2];
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,0,&iv,&ie,h))
		{
		    n_blk_crx += add_to_edge_list(tris[i],0,crx_list,
				                  &n_ecrx,crds_crx,iv,ie);
		}
	    }
	    index = seg_index3d(ic[0],ic[1],ic[2],EAST);
	    seg_crx_count[index] = n_ecrx;
	    coords[1] -= h[1];
	    coords[2] -= h[2];
	    ic[1] -= 1;
	    ic[2] -= 1;
	}
	DEBUG_LEAVE(count_block_crossings)
	return n_blk_crx;
}		/*end count_block_crossings*/

LOCAL int add_to_edge_list(
	TRI		*tri,
	int		ic,
	CRX_STORE	*crx_list,
	int		*nc,
	float		*crds_crx,
	int		iv,
	int		ie)
{
	int		i;

	if (*nc != 0)
	{
	    if (iv != ERROR)
	    {
	    	for (i = 0; i < *nc; ++i)
	    	{
		    if (crx_list[i].vertex == Point_of_tri(tri)[iv])
		    	return 0;
	    	}
	    }
	    else if (ie != ERROR)
	    {
		for (i = 0; i < *nc; ++i)
		{
		    if ((crx_list[i].edge[0] == Point_of_tri(tri)[ie] &&
			 crx_list[i].edge[1] == Point_of_tri(tri)[(ie+1)%3]) ||
		        (crx_list[i].edge[1] == Point_of_tri(tri)[ie] &&
			 crx_list[i].edge[0] == Point_of_tri(tri)[(ie+1)%3]))
		    {
		 	return 0;
		    }
		}
	    }
	    else
	    {
	    	for (i = 0; i < *nc; ++i)
	    	{
	    	    if ((crx_list[i].coords[ic] == crds_crx[ic]) &&
		        same_sign(Tri_normal(tri)[ic],crx_list[i].coords[3]))
		    	return 0;
	    	}
	    }
	}
	if (iv != ERROR)
	{
	    crx_list[*nc].vertex = Point_of_tri(tri)[iv];
	    crx_list[*nc].edge[0] = NULL;
	    crx_list[*nc].edge[1] = NULL;
	}
	else if (ie != ERROR)
	{
	    if (! is_side_bdry(tri,ie))
	    {
		TRI *nbtri = Tri_on_side(tri,ie);
		if (Tri_normal(tri)[ic] >= 0.0 && Tri_normal(nbtri)[ic] <= 0.0)
		    return 0;
	    }
	    crx_list[*nc].edge[0] = Point_of_tri(tri)[ie];
	    crx_list[*nc].edge[1] = Point_of_tri(tri)[(ie+1)%3];
	    crx_list[*nc].vertex = NULL;
	}
	else
	{
	    for (i = 0; i < 3; ++i)
	    {
	    	crx_list[*nc].coords[i] = crds_crx[i];
	    }
	    crx_list[*nc].coords[3] = Tri_normal(tri)[ic];
	    crx_list[*nc].vertex = NULL;
	    crx_list[*nc].edge[0] = NULL;
	    crx_list[*nc].edge[1] = NULL;
	}
	++(*nc);
	return 1;
}		/*end add_to_edge_list*/

LOCAL void insert_block_crossings(
	TRI_GRID	*ntg,
	TRI		**tris,             /* triangles         */
	SURFACE		**surfs,            /* surfs of tris     */
	int		num_tris,           /* num tris in block */
	int		*icrds,             /* block icoords     */
	int		*index )            /* crossing index    */
{
	float		coords[MAXD],crds_crx[MAXD];
	float           *L, *h;
	int		i,k,iv,ie,n_ecrx; /* number of crossings on edge */
	CRXING		*crx_list;
	CRXING		*crx_store = ntg->crx_store;
	int		**seg_crx_lists = ntg->seg_crx_lists;
	int		*seg_crx_count = ntg->seg_crx_count;
	int		*edge_list;
	int             xmax, ymax, zmax;
	static CRX_STORE	*crx_tmp_store;

	xmax = ntg->rect_grid.gmax[0];
	ymax = ntg->rect_grid.gmax[1];
	zmax = ntg->rect_grid.gmax[2];
	set_seg_index3d_globals(ntg->rect_grid.gmax);
	L = ntg->rect_grid.L;
	h = ntg->rect_grid.h;
	if (crx_tmp_store == NULL)
	    vector(&crx_tmp_store,MAX_EDGE_CRX,sizeof(CRX_STORE));
	for (i = 0; i < 3; ++i)
	    coords[i] = L[i] + icrds[i]*h[i];

		/* Count EAST face crossings */

	k = seg_index3d(icrds[0],icrds[1],icrds[2],EAST);
	edge_list = seg_crx_lists[k];
	crx_list = crx_store + *index;
	n_ecrx = 0;
	for (i = 0; i < num_tris; ++i)
	{
	    if (tri_edge_crossing(tris[i],coords,crds_crx,0,&iv,&ie,h))
	    {
		add_to_crx_list(index,0,ntg,tris[i],surfs[i],crx_list,
				crx_tmp_store,edge_list,&n_ecrx,crds_crx,iv,ie);
	    }
	}
	seg_crx_count[k] = n_ecrx;

		/* Count NORTH face crossings */

	k = seg_index3d(icrds[0],icrds[1],icrds[2],NORTH);
	edge_list = seg_crx_lists[k];
	crx_list = crx_store + *index;
	n_ecrx = 0;
	for (i = 0; i < num_tris; ++i)
	{
	    if (tri_edge_crossing(tris[i],coords,crds_crx,1,&iv,&ie,h))
	    {
		add_to_crx_list(index,1,ntg,tris[i],surfs[i],crx_list,
				crx_tmp_store,edge_list,&n_ecrx,crds_crx,iv,ie);
	    }
	}
	seg_crx_count[k] = n_ecrx;

		/* Count UPPER face crossings */

	k = seg_index3d(icrds[0],icrds[1],icrds[2],UPPER);
	edge_list = seg_crx_lists[k];
	crx_list = crx_store + *index;
	n_ecrx = 0;
	for (i = 0; i < num_tris; ++i)
	{
	    if (tri_edge_crossing(tris[i],coords,crds_crx,2,&iv,&ie,h))
	    {
		add_to_crx_list(index,2,ntg,tris[i],surfs[i],crx_list,
				crx_tmp_store,edge_list,&n_ecrx,crds_crx,iv,ie);
	    }
	}
	seg_crx_count[k] = n_ecrx;

	if (icrds[0] != xmax-1 && icrds[1] != ymax-1 && icrds[2] != zmax-1)
	    return;

	if (icrds[0] == xmax - 1)
	{
	    icrds[0] += 1;
	    coords[0] += h[0];
	    k = seg_index3d(icrds[0],icrds[1],icrds[2],NORTH);
	    edge_list = seg_crx_lists[k];
	    crx_list = crx_store + *index;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,1,&iv,&ie,h))
		{
		    add_to_crx_list(index,1,ntg,tris[i],surfs[i],crx_list,
				    crx_tmp_store,edge_list,&n_ecrx,
				    crds_crx,iv,ie);
		}
	    }
	    seg_crx_count[k] = n_ecrx;
	    k = seg_index3d(icrds[0],icrds[1],icrds[2],UPPER);
	    edge_list = seg_crx_lists[k];
	    crx_list = crx_store + *index;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,2,&iv,&ie,h))
		{
		    add_to_crx_list(index,2,ntg,tris[i],surfs[i],crx_list,
				    crx_tmp_store,edge_list,&n_ecrx,
				    crds_crx,iv,ie);
		}
	    }
	    seg_crx_count[k] = n_ecrx;
	    icrds[0] -= 1;
	    coords[0] -= h[0];
	}
	if (icrds[1] == ymax - 1)
	{
	    icrds[1] += 1;
	    coords[1] += h[1];
	    k = seg_index3d(icrds[0],icrds[1],icrds[2],EAST);
	    edge_list = seg_crx_lists[k];
	    crx_list = crx_store + *index;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,0,&iv,&ie,h))
		{
		    add_to_crx_list(index,0,ntg,tris[i],surfs[i],crx_list,
				    crx_tmp_store,edge_list,
				    &n_ecrx,crds_crx,iv,ie);
		}
	    }
	    seg_crx_count[k] = n_ecrx;
	    k = seg_index3d(icrds[0],icrds[1],icrds[2],UPPER);
	    edge_list = seg_crx_lists[k];
	    crx_list = crx_store + *index;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,2,&iv,&ie,h))
		{
		    add_to_crx_list(index,2,ntg,tris[i],surfs[i],crx_list,
				    crx_tmp_store,edge_list,
				    &n_ecrx,crds_crx,iv,ie);
		}
	    }
	    seg_crx_count[k] = n_ecrx;
	    coords[1] -= h[1];
	    icrds[1] -= 1;
	}
	if (icrds[2] == zmax - 1)
	{
	    icrds[2] += 1;
	    coords[2] += h[2];
	    k = seg_index3d(icrds[0],icrds[1],icrds[2],EAST);
	    edge_list = seg_crx_lists[k];
	    crx_list = crx_store + *index;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,0,&iv,&ie,h))
		{
		    add_to_crx_list(index,0,ntg,tris[i],surfs[i],crx_list,
				    crx_tmp_store,edge_list,&n_ecrx,
				    crds_crx,iv,ie);
		}
	    }
	    seg_crx_count[k] = n_ecrx;
	    k = seg_index3d(icrds[0],icrds[1],icrds[2],NORTH);
	    edge_list = seg_crx_lists[k];
	    crx_list = crx_store + *index;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,1,&iv,&ie,h))
		{
		    add_to_crx_list(index,1,ntg,tris[i],surfs[i],crx_list,
				    crx_tmp_store,edge_list,&n_ecrx,
				    crds_crx,iv,ie);
		}
	    }
	    seg_crx_count[k] = n_ecrx;
	    coords[2] -= h[2];
	    icrds[2] -= 1;
	}
	if (icrds[0] == xmax-1 && icrds[1] == ymax-1)
	{
	    icrds[0] += 1;
	    icrds[1] += 1;
	    coords[0] += h[0];
	    coords[1] += h[1];
	    k = seg_index3d(icrds[0],icrds[1],icrds[2],UPPER);
	    edge_list = seg_crx_lists[k];
	    crx_list = crx_store + *index;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,2,&iv,&ie,h))
		{
		    add_to_crx_list(index,2,ntg,tris[i],surfs[i],crx_list,
				    crx_tmp_store,edge_list,&n_ecrx,
				    crds_crx,iv,ie);
		}
	    }
	    seg_crx_count[k] = n_ecrx;
	    coords[0] -= h[0];
	    coords[1] -= h[1];
	    icrds[0] -= 1;
	    icrds[1] -= 1;
	}
	if (icrds[0] == xmax-1 && icrds[2] == zmax-1)
	{
	    icrds[0] += 1;
	    icrds[2] += 1;
	    coords[0] += h[0];
	    coords[2] += h[2];
	    k = seg_index3d(icrds[0],icrds[1],icrds[2],NORTH);
	    edge_list = seg_crx_lists[k];
	    crx_list = crx_store + *index;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,1,&iv,&ie,h))
		{
		    add_to_crx_list(index,1,ntg,tris[i],surfs[i],crx_list,
				    crx_tmp_store,edge_list,&n_ecrx,
				    crds_crx,iv,ie);
		}
	    }
	    seg_crx_count[k] = n_ecrx;
	    coords[0] -= h[0];
	    coords[2] -= h[2];
	    icrds[0] -= 1;
	    icrds[2] -= 1;
	}
	if (icrds[1] == ymax-1 && icrds[2] == zmax-1)
	{
	    icrds[1] += 1;
	    icrds[2] += 1;
	    coords[1] += h[1];
	    coords[2] += h[2];
	    k = seg_index3d(icrds[0],icrds[1],icrds[2],EAST);
	    edge_list = seg_crx_lists[k];
	    crx_list = crx_store + *index;
	    n_ecrx = 0;
	    for (i = 0; i < num_tris; ++i)
	    {
	    	if (tri_edge_crossing(tris[i],coords,crds_crx,0,&iv,&ie,h))
		{
		    add_to_crx_list(index,0,ntg,tris[i],surfs[i],crx_list,
				    crx_tmp_store,edge_list,&n_ecrx,
				    crds_crx,iv,ie);
		}
	    }
	    seg_crx_count[k] = n_ecrx;
	    coords[1] -= h[1];
	    coords[2] -= h[2];
	    icrds[1] -= 1;
	    icrds[2] -= 1;
	}
}		/*end insert_block_crossings*/

LOCAL	void add_to_crx_list(
	int *index,
	int ic,
	TRI_GRID *ntg,
	TRI *tri,
	SURFACE *surf,
	CRXING *crx_list,
	CRX_STORE *crx_tmp_store,
	int *edge_list,
	int *nc,
	float *crds_crx,
	int iv,
	int ie)
{
	int i;

	if (iv != ERROR)
	{
	    if (*nc != 0)
	    {
	    	for (i = 0; i < *nc; ++i)
		{
		    if (crx_tmp_store[i].vertex ==
			Point_of_tri(tri)[iv])/*vrtx already in list*/
			return;
		}
	    }
	    if (!set_comp_at_vertex(&crx_list[*nc],
		 Point_of_tri(tri)[iv],tri,surf,ic))
	    	return;
	    crx_tmp_store[*nc].vertex = Point_of_tri(tri)[iv];
	    if (crds_crx[(ic+1)%3] == Coords(Point_of_tri(tri)[iv])[(ic+1)%3]
		     &&
		crds_crx[(ic+2)%3] == Coords(Point_of_tri(tri)[iv])[(ic+2)%3])
	    {
	    	crx_list[*nc].pt = Point_of_tri(tri)[iv];
	    }
	    else
	    {
	    	crx_list[*nc].pt = copy_point(Point_of_tri(tri)[iv]);
		Coords(crx_list[*nc].pt)[(ic+1)%3] = crds_crx[(ic+1)%3];
		Coords(crx_list[*nc].pt)[(ic+2)%3] = crds_crx[(ic+2)%3];
	    }
	}
	else if (ie != ERROR)
	{
	    if (*nc != 0)
	    {
	    	for (i = 0; i < *nc; ++i)
		{
		    if ((crx_tmp_store[i].edge[0]==Point_of_tri(tri)[ie] &&
			 crx_tmp_store[i].edge[1]==Point_of_tri(tri)[(ie+1)%3])
			 ||
		        (crx_tmp_store[i].edge[1]==Point_of_tri(tri)[ie] &&
			 crx_tmp_store[i].edge[0]==Point_of_tri(tri)[(ie+1)%3]))
			return;
		}
	    }
	    if (! is_side_bdry(tri,ie))
	    {
		TRI *nbtri = Tri_on_side(tri,ie);
		if (Tri_normal(tri)[ic] >= 0.0 && Tri_normal(nbtri)[ic] <= 0.0)
		    return;
	    }
	    crx_list[*nc].pt = Point(crds_crx);
	    interpolate_crx_pt_states_on_edge(ntg,crx_list[*nc].pt,tri,surf,ie);
	    crx_tmp_store[*nc].edge[0] = Point_of_tri(tri)[ie];
	    crx_tmp_store[*nc].edge[1] = Point_of_tri(tri)[(ie+1)%3];
	    if (Tri_normal(tri)[ic] > 0.0)
	    {
		crx_list[*nc].lcomp = negative_component(surf);
		crx_list[*nc].ucomp = positive_component(surf);
	    }
	    else
	    {
		crx_list[*nc].lcomp = positive_component(surf);
		crx_list[*nc].ucomp = negative_component(surf);
	    }
	}
	else
	{
	    if (*nc != 0)
	    {
	    	for (i = 0; i < *nc; ++i)
		{
		    if ((Coords(crx_list[i].pt)[ic] == crds_crx[ic]) &&
			((Tri_normal(tri)[ic] > 0 && 
			  crx_list[i].lcomp == negative_component(surf)) ||
			 (Tri_normal(tri)[ic] < 0 &&
			 crx_list[i].lcomp == positive_component(surf))))
			return;
		}
	    }
	    crx_list[*nc].pt = Point(crds_crx);
	    interpolate_crx_pt_states_on_tri(ntg,crx_list[*nc].pt,tri,surf);
	    crx_tmp_store[*nc].edge[0] = NULL;
	    crx_tmp_store[*nc].edge[1] = NULL;
	    if (Tri_normal(tri)[ic] > 0.0)
	    {
		crx_list[*nc].lcomp = negative_component(surf);
		crx_list[*nc].ucomp = positive_component(surf);
	    }
	    else
	    {
		crx_list[*nc].lcomp = positive_component(surf);
		crx_list[*nc].ucomp = negative_component(surf);
	    }
	}

	crx_list[*nc].hs = Hyper_surf(surf);
	crx_list[*nc].tri = tri;
	crx_list[*nc].crx_num = 0;

	for (i = 0; i < *nc; ++i)
	{
	    if (Coords(crx_list[i].pt)[ic] > Coords(crx_list[*nc].pt)[ic])
	    {
		CRXING crx_tmp = crx_list[i];
		crx_list[i] = crx_list[*nc];
		crx_list[*nc] = crx_tmp;
	    }
	}
	edge_list[*nc] = *index;
	++(*nc);
	++(*index);
}		/* end add_to_crx */

LOCAL void interpolate_crx_pt_states_on_tri(
	TRI_GRID	*ntg,
	POINT		*pt,
	TRI		*tri,
	SURFACE		*surf)
{
	float		f[MAXD];
	float		*coords = Coords(pt);
	POINT		*p;
	int		i;
	Locstate	lstate[3], rstate[3];

	linear_interp_coefs_3d_tri(f,coords,tri);

	for (i=0; i< 3; ++i)
	{   
	    p = Point_of_tri(tri)[i];
	    slsr(p,Hyper_surf_element(tri),Hyper_surf(surf),lstate+i,rstate+i);
	}

	if ((tri_interpolate_intfc_states(ntg->grid_intfc,f[0],f[1],f[2],
		                          Coords(Point_of_tri(tri)[0]),
					  lstate[0],
					  Coords(Point_of_tri(tri)[1]),
					  lstate[1],
		                          Coords(Point_of_tri(tri)[2]),
					  lstate[2],
					  left_state(pt))
					      != FUNCTION_SUCCEEDED)
	    ||
	    (tri_interpolate_intfc_states(ntg->grid_intfc,f[0],f[1],f[2],
		                            Coords(Point_of_tri(tri)[0]),
					    rstate[0],
					    Coords(Point_of_tri(tri)[1]),
					    rstate[1],
		                            Coords(Point_of_tri(tri)[2]),
					    rstate[2],
					    right_state(pt))
						!= FUNCTION_SUCCEEDED))
	{
	    screen("ERROR in interpolate_crx_pt_states_on_tri(), "
		   "tri_interpolate_intfc_states() failed\n");
	    clean_up(ERROR);
	}
}		/*end interpolate_crx_pt_states_on_tri*/

LOCAL void linear_interp_coefs_3d_tri(
	float		*f,
	float		*coords,
	TRI		*t)
{
	float		*p0,*p1,*p2;
	float		v1[MAXD],v2[MAXD],v[MAXD];
	float		qd[MAXD],q1[MAXD],q2[MAXD];
	float		D,D2;
	int		i;

	p0 = Coords(Point_of_tri(t)[0]);
	p1 = Coords(Point_of_tri(t)[1]);
	p2 = Coords(Point_of_tri(t)[2]);

	for (i = 0; i < 3; ++i)
	{
	    v1[i] = p1[i] - p0[i];
	    v2[i] = p2[i] - p0[i];
	    v[i] = coords[i] - p0[i];
	}

	D = vector_product(v1,v2,qd,3);
	Cross3d(v,v2,q1);
	Cross3d(v,v1,q2);
	D2 = D*D;

	f[1] =  Dot3d(qd,q1)/D2;
	f[2] = -Dot3d(qd,q2)/D2;
	f[0] = 1.0 - f[1] - f[2];
}		/*end linear_interp_coefs_3d_tri*/

LOCAL void interpolate_crx_pt_states_on_edge(
	TRI_GRID	*ntg,
	POINT		*pt,
	TRI		*tri,
	SURFACE		*surf,
	int 		ie)
{
	float		alpha;
	float		*coords = Coords(pt);
	POINT		*p1,*p2;
	int		i;
	Locstate	lstate[2], rstate[2];

	p1 = Point_of_tri(tri)[ie];
	slsr(p1,Hyper_surf_element(tri),Hyper_surf(surf),lstate,rstate);
	p2 = Point_of_tri(tri)[(ie+1)%3];
	slsr(p2,Hyper_surf_element(tri),Hyper_surf(surf),lstate+1,rstate+1);
	alpha = 0.5;
	for (i = 0; i < 3; ++i)
	{
	    if (within_interval(Coords(p1)[i],Coords(p2)[i],coords[i]))
	    {
		if (fabs(Coords(p1)[i] - Coords(p2)[i]) < crx_tol)
		    continue;
		alpha = fabs(coords[i]-Coords(p2)[i])/
			fabs(Coords(p1)[i]-Coords(p2)[i]);
		break;
	    }
	}

	bi_interpolate_intfc_states(ntg->grid_intfc,alpha,1.0-alpha,
		Coords(p1),lstate[0],Coords(p2),lstate[1],left_state(pt));
	bi_interpolate_intfc_states(ntg->grid_intfc,alpha,1.0-alpha,
		Coords(p1),rstate[0],Coords(p2),rstate[1],right_state(pt));
}		/*end interpolate_crx_pt_states_on_tri*/

LOCAL bool set_comp_at_vertex(
	CRXING  *crx,
	POINT   *p,
	TRI     *tri,
	SURFACE *surf,
	int     dir)
{
	int i,v,num_tris;
	TRI **tri_list;
	float *p0,*p1,*p2,dp1[2],dp2[2];
	float dp1_len,dp2_len,sin_arg,cos_arg;
	float angle,accum_angle;
	float pi_2 = PI/2.0;

	num_tris = set_tri_list_around_point(p,tri,&tri_list,surf->interface);
	accum_angle = 0.0;
	for (i = 0; i < num_tris; ++i)
	{
	    v = Vertex_of_point(tri_list[i],p);
	    p0 = Coords(Point_of_tri(tri_list[i])[v]);
	    p1 = Coords(Point_of_tri(tri_list[i])[Next_m3(v)]);
	    p2 = Coords(Point_of_tri(tri_list[i])[Prev_m3(v)]);
	    dp1[0] = p1[Next_m3(dir)] - p0[Next_m3(dir)];
	    dp2[0] = p2[Next_m3(dir)] - p0[Next_m3(dir)];
	    dp1[1] = p1[Prev_m3(dir)] - p0[Prev_m3(dir)];
	    dp2[1] = p2[Prev_m3(dir)] - p0[Prev_m3(dir)];
	    dp1_len = sqrt(dp1[0]*dp1[0] + dp1[1]*dp1[1]);
	    dp2_len = sqrt(dp2[0]*dp2[0] + dp2[1]*dp2[1]);
	    if (dp1_len == 0.0 || dp2_len == 0.0) continue;
	    sin_arg = (dp1[0]*dp2[1] - dp1[1]*dp2[0])/dp1_len/dp2_len;
	    cos_arg = (dp1[0]*dp2[0] + dp1[1]*dp2[1]);
	    if (sin_arg >  1.0) sin_arg =  1.0;
	    if (sin_arg < -1.0) sin_arg = -1.0;
	    angle = asin(sin_arg);
	    if (cos_arg < 0.0) 
	    {
	        if (sin_arg > 0.0)
	    	angle =  PI - angle;
	        else
	    	angle = -PI - angle;
	    }
	    accum_angle += angle;
	}
	if (accum_angle > pi_2 - 0.01)
	{
	    crx->lcomp = negative_component(surf);
	    crx->ucomp = positive_component(surf);
	    return YES;
	}
	else if (accum_angle < -pi_2 + 0.01)
	{
	    crx->lcomp = positive_component(surf);
	    crx->ucomp = negative_component(surf);
	    return YES;
	}
	else
	{
	    crx->lcomp = crx->ucomp = NO_COMP;
	    return NO;
	}
}		/*end set_comp_at_vertex*/

LOCAL void print_random_scale(TRI_GRID *ntg)
{
        INTERFACE       *intfc = ntg->grid_intfc;
        COMPONENT       *comp = ntg->components;
        int             *gmax = ntg->rect_grid.gmax;
        int             *lbuf = computational_grid(intfc)->lbuf;
        int             *ubuf = computational_grid(intfc)->ubuf;
        int             xmin,ymin,zmin;
        int             xmax,ymax,zmax;
        int             ix,iy,iz;
        char            rs_name[100];
        FILE            *rs_file;

        sprintf(rs_name,"comp-%s",right_flush(pp_mynode(),3));
        xmin = (lbuf[0] == 0) ? 1 : lbuf[0];
        ymin = (lbuf[1] == 0) ? 1 : lbuf[1];
        zmin = (lbuf[2] == 0) ? 1 : lbuf[2];
        xmax = (ubuf[0] == 0) ? gmax[0] - 1 : gmax[0] - ubuf[0];
        ymax = (ubuf[1] == 0) ? gmax[1] - 1 : gmax[1] - ubuf[1];
        zmax = (ubuf[2] == 0) ? gmax[2] - 1 : gmax[2] - ubuf[2];
        rs_file = fopen(rs_name,"w");

        for (iz = zmin; iz <= zmax; ++iz)
        {
            for (iy = ymin; iy <= ymax; ++iy)
            {
                for (ix = xmin; ix <= xmax; ++ix)
                {
                    fprintf(rs_file,"%d ",comp[d_index3d(ix,iy,iz,gmax)]);
                }
                fprintf(rs_file,"\n");
            }
            fprintf(rs_file,"\n");
        }
}       /* end print_random_scale */
#endif /* defined(THREED) */
