/*
*				isub.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	This file contains elementary routines for the computation of
*	angles, areas, vector and scalar products.
*/

#include <intfc/iloc.h>

	/* LOCAL Function Prototypes */
#if defined(THREED)
LOCAL	BDRY_SIDE rect_bdry_side_for_surface(int*,int*,SURFACE*,RECT_GRID*);
LOCAL	void  set_area_weighted_normal(TRI**,int,float*);
LOCAL	void  debug_print_tri_list_around_point(POINT*,TRI*,INTERFACE*,TRI**,
	                                            int,const char*);
#endif /* defined(THREED) */
LOCAL   void  reflect(float,float,float,float,float*,float*,RECT_GRID*);
LOCAL   void  set_cross_position(float,float,int,float,float,int,POINT *);

#if (defined(ONED) && defined(TWOD)) || (defined(ONED) && defined(THREED)) || (defined(TWOD) && defined(THREED))
EXPORT	float separation(
	POINT		*p,
	POINT		*q,
	int		dim)
{
	int		i;
	float		sep;

	sep = 0.0;
	for (i = 0; i < dim; ++i)
	    sep += sqr(Coords(q)[i] - Coords(p)[i]);
	return sqrt(sep);
}		/*end separation*/
#endif /* (defined(ONED) && defined(TWOD)) || (defined(ONED) && defined(THREED)) || (defined(TWOD) && defined(THREED)) */


/*
*			vector_product_on_bonds():
*
*	This routine computes the vector product in 2D and 3D.
*	The answer is placed in Pout and is a vector for three dimensions and
*	a single float (scalar) in two dimensions. 
*
*	Pout = bonds1 X bonds2.
*
*/

EXPORT 	void 	vector_product_on_bonds(
	BOND		*b1,
	BOND		*b2,
	int		dim,
	float		*Pout)
{
	float		vect1[MAXD],vect2[MAXD];
	int		i;

	for (i = 0; i < dim; ++i)
	{
	    vect1[i] = Coords(b1->end)[i] - Coords(b1->start)[i];
	    vect2[i] = Coords(b2->end)[i] - Coords(b2->start)[i];
	}

	switch (dim)
	{
	case 2:
	    *Pout = vect1[0]*vect2[1]-vect1[1]*vect2[0] ;
	    break;
#if defined(THREED)
	case 3:
	    Pout[0] = vect1[1]*vect2[2]-vect1[2]*vect2[1];
	    Pout[1] = vect1[2]*vect2[0]-vect1[0]*vect2[2];
	    Pout[2] = vect1[0]*vect2[1]-vect1[1]*vect2[0];
	    break;
#endif /* defined(THREED) */
	default:
	    break;
	}
	return;
}		/*end vector_product_on_bonds*/


/*
*			scalar_product_on_bonds():
*
* 	This function returns the scalar product of two bonds 
*	The dimension must be specified when this function is called. 
*	
*/
EXPORT float scalar_product_on_bonds(
	BOND		*b1,
	BOND		*b2,
	int		dim)
{
	float		sp;
	int		i;

	sp = 0.0;
	for (i = 0; i < dim; ++i) 
	    sp += (Coords(b1->end)[i] - Coords(b1->start)[i])  *
		  (Coords(b2->end)[i] - Coords(b2->start)[i]);
	return sp;
}		/*end scalar_product_on_bonds*/

EXPORT	float scaled_bond_length(
	BOND		*b,
	float		*h,
	int		dim)
{
	float tmp, ans = 0.0;
	int   i;

	for (i = 0; i < dim; ++i) 
	{
	    tmp = (Coords(b->end)[i] - Coords(b->start)[i])/h[i];
	    ans += sqr(tmp);
	}
	return sqrt(ans);
}		/*end scaled_bond_length*/


#if defined(TWOD)

/*
*			area_of_closed_curve():
*
*       Returns the SIGNED area of a closed curve.  The
*       area will be negative if the inside is on the
*       right side of the curve.
*
*       Returns ERROR_FLOAT if the curve is not closed.
*/

EXPORT	float area_of_closed_curve(
	CURVE		*c)
{
	float		area = 0.0, cp[MAXD];
	BOND		*b;
	float		*p0;
	int		dim = c->interface->dim;

	if (!is_closed_curve(c))
	    return ERROR_FLOAT;

	if (c->num_points < 3)
	    return 0.0;

	p0 = Coords(c->start->posn);
	area = 0.0;
	for (b = c->first->next; b && (b != c->last) ; b = b->next)
	{
	    (void) vector_product_on_points(p0,Coords(b->start),
					    Coords(b->end),dim,cp);
	    area += cp[0];
	}

	area *= 0.5;
	return area;
}		/*end area_of_closed_curve*/
#endif /* defined(TWOD) */


EXPORT	P_LINK *add_to_hash_table(
	POINTER	pl,
	POINTER	pr,
	P_LINK	*hash_table,
	int	h_size)
{
	P_LINK   *link;
	uint64_t i, N, i_init;

	i_init = ptr2ull(pl) % h_size;
	link = hash_table+i_init;

	if ((link->pl == NULL) || (link->pl == pl))
	{
	    link->pl = pl;
	    link->pr = pr;
	    return link;
	}
	N = min(h_size-i_init,i_init+1);
	for (i = 1; i < N; ++i)
	{
	    link = hash_table + (i_init + i);
	    if ((link->pl == NULL) || (link->pl == pl))
	    {
	        link->pl = pl;
	        link->pr = pr;
	        return link;
	    }
	    link = hash_table + (i_init - i);
	    if ((link->pl == NULL) || (link->pl == pl))
	    {
	        link->pl = pl;
	        link->pr = pr;
	        return link;
	    }
	}
	for (i = i_init + N; i < h_size; ++i)
	{
	    link = hash_table + i;
	    if ((link->pl == NULL) || (link->pl == pl))
	    {
	        link->pl = pl;
	        link->pr = pr;
	        return link;
	    }
	}
	if (N <= i_init)
	{
	    for (i = 0; i <= i_init - N; ++i)
	    {
	        link = hash_table + i;
	        if ((link->pl == NULL) || (link->pl == pl))
	        {
	            link->pl = pl;
	            link->pr = pr;
	            return link;
	        }
	    }
	}
	return NULL;
}		/*end add_to_hash_table*/


EXPORT 	POINTER  find_from_hash_table(
	POINTER	pl,
	P_LINK	*hash_table,
	int	h_size)
{
	P_LINK   *link;
	uint64_t i, N, i_init;

	i_init = ptr2ull(pl) % h_size;
	link = hash_table+i_init;
	if (link->pl == pl)
	    return link->pr;
	if (link->pl == NULL)
	    return NULL;
	N = min(h_size-i_init,i_init+1);
	for (i = 1; i < N; ++i)
	{
	    link = hash_table + (i_init + i);
	    if (link->pl == pl)
	        return link->pr;
	    if (link->pl == NULL)
	        return NULL;
	    link = hash_table + (i_init - i);
	    if (link->pl == pl)
	        return link->pr;
	    if (link->pl == NULL)
	        return NULL;
	}
	for (i = i_init + N; i < h_size; ++i)
	{
	    link = hash_table + i;
	    if (link->pl == pl)
	        return link->pr;
	    if (link->pl == NULL)
	        return NULL;
	}
	if (N <= i_init)
	{
	    for (i = 0; i <= N - i_init; ++i)
	    {
	        link = hash_table + i;
	        if (link->pl == pl)
	            return link->pr;
	        if (link->pl == NULL)
	            return NULL;
	    }
	}
	return NULL;
}		/*end find_from_hash_table*/

EXPORT	void reset_hash_table(
	P_LINK		*hash_table,
	int		h_size)
{
	zero_scalar(hash_table,h_size*sizeof(P_LINK));
}		/*end reset_hash_table*/


EXPORT	BDRY_SIDE rect_bdry_side_for_curve(
	int		*idir,
	int		*iside,
	CURVE		*c,
	RECT_GRID	*gr)
{
	INTERFACE	*intfc = c->interface;
	float		*ps, *pe, d, m, min_d, p;
	int		i, dim = intfc->dim;
	int		lidir, liside;

	if (!is_bdry(c))
	    return NOT_A_BDRY;
	ps = Coords(c->start->posn);
	pe = Coords(c->end->posn);
	min_d = HUGE_VAL;
	for (i = 0; i < dim; ++i)
	{
	    d = fabs(ps[i] - pe[i]);
	    if (d < min_d)
	    {
	    	min_d = d;
	    	lidir = i;
	    }
	}
	p = 0.5*(ps[lidir] + pe[lidir]);
	m = 0.5*(gr->L[lidir] + gr->U[lidir]);
	liside = (p < m) ? 0 : 1;

	if (idir != NULL)
	    *idir = lidir;
	if (iside != NULL)
	    *iside = liside;
	switch (lidir)
	{
	case 0:
	    return (liside == 0) ? LEFT_BDRY : RIGHT_BDRY;
	case 1:
	    return (liside == 0) ? LOWER_BDRY : UPPER_BDRY;
	case 2:
	    return (liside == 0) ? ZMIN_BDRY : ZMAX_BDRY;
	}
	return NOT_A_BDRY;
}		/*end rect_bdry_side_for_curve*/

#if defined(THREED)
/*ARGSUSED*/
LOCAL	BDRY_SIDE rect_bdry_side_for_surface(
	int		*idir,
	int		*iside,
	SURFACE		*s,
	RECT_GRID	*gr)
{
	float		*L = gr->GL, *U = gr->GU;
	float		pbar[3];
	float		d, min_d;
	int		i;
	int		lidir, liside;

	average_position_of_surface(pbar,s);

	min_d = HUGE_VAL;
	liside = -1;
	for (i = 0; i < 3; ++i)
	{
	    d = fabs(L[i] - pbar[i]);
	    if (d < min_d)
	    {
	    	min_d = d;
	    	lidir = i;
		liside = 0;
	    }
	    d = fabs(U[i] - pbar[i]);
	    if (d < min_d)
	    {
	    	min_d = d;
	    	lidir = i;
		liside = 1;
	    }
	}

	if (idir != NULL)
	    *idir = lidir;
	if (iside != NULL)
	    *iside = liside;
	switch (lidir)
	{
	case 0:
	    return (liside == 0) ? LEFT_BDRY : RIGHT_BDRY;
	case 1:
	    return (liside == 0) ? LOWER_BDRY : UPPER_BDRY;
	case 2:
	    return (liside == 0) ? ZMIN_BDRY : ZMAX_BDRY;
	}
	return NOT_A_BDRY;
}		/*end rect_bdry_side_for_surface*/

EXPORT	void	average_position_of_surface(
	float	*pbar,
	SURFACE *s)
{
	TRI	*tri;
	float	*p0, *p1, *p2;
	int	i, nt;

	pbar[0] = pbar[1] = pbar[2] = 0.0;
	for (nt = 0, tri = first_tri(s); !at_end_of_tri_list(tri,s);
	     ++nt, tri = tri->next)
	{
	    p0 = Coords(Point_of_tri(tri)[0]);
	    p1 = Coords(Point_of_tri(tri)[1]);
	    p2 = Coords(Point_of_tri(tri)[2]);
	    for (i = 0; i < 3; ++i)
		pbar[i] += (p0[i] + p1[i] + p2[i])/3.0;
	}
	pbar[0] /= nt; pbar[1] /= nt; pbar[2] /= nt;
}		/*end average_position_of_surface */

/*
*		set_tri_list_around_point():
*
*	Locates a triangles directly connected to the give triangle
*	at the vertex v.  The list stops if a boundary bond is detected.
*/

EXPORT  int set_tri_list_around_point(
	POINT     *p,
	TRI       *tri,
	TRI       ***ptris,
	INTERFACE *intfc)
{
	static TRI	**tris = NULL;
	static int	max_num_tris = 0;
	TRI		*nbtri;
	int		j, n;

	if (tris == NULL)
	{
	    max_num_tris = 24;
	    vector(&tris,max_num_tris,sizeof(TRI *));
	}

	nbtri = tri;
	n = 0;
	do {
	  if ((n+1) >= max_num_tris)
	  {
	    TRI **tmp_tris;
	    int i, nmax = max_num_tris + 24;

	    vector(&tmp_tris,nmax,sizeof(TRI *));
	    for (i = 0; i < max_num_tris; ++i)
	      tmp_tris[i] = tris[i];
	    free(tris);
	    tris = tmp_tris;
	    max_num_tris = nmax;
	  }
	  tris[n++] = nbtri;
	  if ((nbtri = Prev_tri_at_vertex(nbtri,p)) != NULL)
	  {
	    if (Next_tri_at_vertex(nbtri,p) != tris[n-1])
	    {
	      screen("ERROR in set_tri_list_around_point(), "
		     "inconsistent neighboring tris\n");
	      debug_print_tri_list_around_point(p,tri,intfc,tris,n,
			                        "set_tri_list_around_point");
	      (void) printf("nbtri\n");
	      print_tri(nbtri,intfc);
	      clean_up(ERROR);
	    }
	    if (nbtri == tri)
	    {
	      tris[n] = NULL;
	      *ptris = tris;
	      return n;
	    }
	    for (j = 0; j < n; ++j)
	    {
	      if (nbtri == tris[j])
	      {
	        screen("ERROR in set_tri_list_around_point(), "
		       "nbtri reappeared in tri list before tri\n");
	        debug_print_tri_list_around_point(p,tri,intfc,tris,n,
			                          "set_tri_list_around_point");
		clean_up(ERROR);
	      }
	    }
	  }
	} while (nbtri != NULL);

	/* Hit boundary */
	for (nbtri = tri; Next_tri_at_vertex(nbtri,p) != NULL;
				nbtri = Next_tri_at_vertex(nbtri,p));

	if (nbtri == tri) /* Previously generated list okay */
	{
	    tris[n] = NULL;
	    *ptris = tris;
	    return n;
	}

	n = 0;
	do {
	  if ((n+1) >= max_num_tris)
	  {
	    TRI **tmp_tris;
	    int i, nmax = max_num_tris + 24;

	    vector(&tmp_tris,nmax,sizeof(TRI *));
	    for (i = 0; i < max_num_tris; ++i)
	      tmp_tris[i] = tris[i];
	    free(tris);
	    tris = tmp_tris;
	    max_num_tris = nmax;
	  }
	  tris[n++] = nbtri;
	  if ((nbtri = Prev_tri_at_vertex(nbtri,p)) != NULL)
	  {
	    if (Next_tri_at_vertex(nbtri,p) != tris[n-1])
	    {
	      screen("ERROR in set_tri_list_around_point(), "
		     "inconsistent neighboring tris\n");
	      debug_print_tri_list_around_point(p,tri,intfc,tris,n,
			                        "set_tri_list_around_point");
	      (void) printf("nbtri\n");
	      print_tri(nbtri,intfc);
	      clean_up(ERROR);
	    }
	    for (j = 0; j < n; ++j)
	    {
	      if (nbtri == tris[j])
	      {
		screen("ERROR in set_tri_list_around_point(), "
		       "nbtri reappeared in tri list before tri\n");
		debug_print_tri_list_around_point(p,tri,intfc,tris,n,
			                          "set_tri_list_around_point");
		clean_up(ERROR);
	      }
	    }
	  }
	} while (nbtri != NULL);

	tris[n] = NULL;
	*ptris = tris;
	return n;
}		/*end set_tri_list_around_point*/

LOCAL	void	debug_print_tri_list_around_point(
	POINT      *p,
	TRI        *tri,
	INTERFACE  *intfc,
	TRI        **tris,
	int        n,
	const char *func)
{
	float BBL[3], BBU[3];
	int   i;
	char  s[256];

	(void) printf("Current number of tris = %d\n",n);
	set_tri_list_bounding_box(tris,n,BBL,BBU,NO,YES);
	(void) sprintf(s,"%s-tris",func);
	gview_plot_triangle_list("",s,tris,n,
				 0.1,0.0,0.0,0.9,0.0,0.0,0.5,BBL,BBU);
	(void) sprintf(s,"%s-p",func);
	gview_plot_vertices("",s,&p,1,BBL,BBU);
	(void) printf("point %llu\n",point_number(p));
	print_general_vector("Coords(p) = ",Coords(p),3,"\n");
	(void) printf("tri\n");
	print_tri(tri,intfc);
	for (i = 0; i < n; ++i)
	{
	    (void) printf("tris[%d]\n",i);
	    print_tri(tris[i],intfc);
	}
}		/*end debug_print_tri_list*/

#endif /* defined(THREED) */

EXPORT	void rect_bdry_side_for_hyper_surf(
	int		*idir,
	int		*iside,
	HYPER_SURF	*hs,
	RECT_GRID	*gr)
{
	INTERFACE	*intfc = hs->interface;

	switch (intfc->dim)
	{
#if defined(ONED)
	case 1:
	{
	    float L = gr->L[0];
	    float U = gr->U[0];
	    POINT *p = (POINT *) hs;
	    *idir = 0;
	    if (fabs(Coords(p)[0] - L) < fabs(Coords(p)[0] - U))
	    {
	        *iside = 0;
	        return;
	    }
	    else
	    {
	        *iside = 1;
	        return;
	    }
	}
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    (void) rect_bdry_side_for_curve(idir,iside,Curve_of_hs(hs),gr);
	    return;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    (void) rect_bdry_side_for_surface(idir,iside,Surface_of_hs(hs),gr);
	    return;
#endif /* defined(THREED) */
	}
}		/*end rect_bdry_side_for_hyper_surf*/

EXPORT	void i_add_comp_to_list(
	COMPONENT	comp,
	COMP_LIST	*comp_list,
	INTERFACE	*intfc)
{
	INTERFACE	*cur_intfc;
	COMPONENT	*comps = comp_list->comps;
	int		ncomps = comp_list->ncomps;
	int		max_ncomps = comp_list->max_ncomps;
	int		i;
	static int	increment = 10;

	/* Check if already in the list */
	for (i = 0; i < ncomps; ++i)
		if (comp == comps[i]) return;

	if (comps == NULL)
	{
		cur_intfc = current_interface();
		if (cur_intfc != intfc) set_current_interface(intfc);
		max_ncomps = increment;
		comp_list->max_ncomps = max_ncomps;
		comp_list->comps = comps =
			(COMPONENT *) store(max_ncomps*sizeof(COMPONENT));
		if (cur_intfc != intfc)
			set_current_interface(cur_intfc);
	}
	else if (ncomps+1 > max_ncomps)
	{
		COMPONENT  *tmp_comps = comps;

		cur_intfc = current_interface();
		if (cur_intfc != intfc) set_current_interface(intfc);
		max_ncomps += increment;
		comp_list->max_ncomps = max_ncomps;
		comp_list->comps = comps =
			(COMPONENT *) store(max_ncomps*sizeof(COMPONENT));
		for (i = 0; i < ncomps; ++i) comps[i] = tmp_comps[i];
		if (cur_intfc != intfc)
			set_current_interface(cur_intfc);
	}
	comps[comp_list->ncomps++] = comp;
}		/*end i_add_to_comp_list*/

EXPORT	bool i_is_comp_in_list(
	COMPONENT	comp,
	COMP_LIST	*comp_list)
{
	int		i;

	for (i = 0; i < comp_list->ncomps; ++i)
	    if (comp == comp_list->comps[i])
		return YES;
	return NO;
}		/*end i_is_comp_in_list*/

EXPORT	int curve_in_curve_list(
	CURVE		*c,
	CURVE		**c_list)
{
	CURVE		**cc;

	for (cc = c_list; cc && *cc ; ++cc)
	    if (*cc == c) 
	    	return YES;
	return NO;
}		/*end curve_in_curve_list*/

EXPORT void reset_intfc_num_points(
	INTERFACE	*intfc)
{
	HYPER_SURF	*hs;
	HYPER_SURF_ELEMENT *hse;
	POINT		*p; 
	int		np = 0;

	(void) next_point(intfc,NULL,NULL,NULL);
	while (next_point(intfc,&p,&hse,&hs))
	    ++np;
	intfc->num_points = np;
}	/* end reset_intfc_num_points */

LIB_LOCAL	int	i_print_number_of_tangles(
	const char	*mesg,
	INTERFACE	*intfc,
	CROSS		*cross)
{
	int		num, dim = intfc->dim;
	CROSS		*cr;
	static const char *iname[] = {"","wave interaction","point","curve"};

	for (num = 0, cr = cross; cr != NULL; ++num, cr = cr->next);
	(void) printf("The %s%sinterface is tangled at ",
		      (pp_numnodes()>1)?"local ":"",mesg);
	(void) printf("%d %s%s\n",num,iname[dim],(num!=1) ? "s" : "");
	if (num > 0)
	{
	    if (debugging("intersect"))
	    	print_intersections(cross,intfc);
	    if (debugging("bond_cross"))
	    	print_crossing_elements(cross,intfc);
#if defined(TWOD)
	    if (debugging("rcl"))
	    	print_cross_list(cross);
#endif /* defined(TWOD) */
	}
	return num;
}		/*end i_print_number_of_tangles*/

#if defined(THREED)

struct	_TRI_LIST_AT_VERTEX
{
	POINT	*p;
	TRI	**tris;
	int	num_tris;
};
typedef struct  _TRI_LIST_AT_VERTEX TRI_LIST_AT_VERTEX;
LOCAL	TRI_LIST_AT_VERTEX Tri_list_at_vertex;


/*
*			plane_fit_normal3d():
*
*	Returns the Normal to the hypersurface hs at the point p on
*	hypersurface element hse of hs.  The normal points from the
*	negative side to the positive side of the hypersurface.
*
*	The normal is computed by a least squares plane fit to the point
*	p and the points adjacent to p.
*
*	NOTE: In some instances, such as when the point p is the vertex
*	of a cone-like structure,  it is desirable to omit the use of p
*	in the planar fit. The function omit_vertex_in_plane_fit() turns
*	on a flag that excludes the use of p in the fitting structure. This
*	flag is then reset to the standard value of including p in the plane
*	fit.
*/

LOCAL	bool	OmitVertexInPlaneFit = NO;

EXPORT	void	omit_vertex_in_plane_fit(void)
{
	OmitVertexInPlaneFit = YES;
}		/*end omit_vertex_in_plane_fit*/

EXPORT  void plane_fit_normal3d(
	POINT		   *p,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	   *hs,
	float		   *nor)
{
	INTERFACE    *intfc = hs->interface;
	POINT        *pp;
	TRI          *tri;
	int          n, i, v;
	float        pbar[3], lambda[3];
	static float **pts = NULL;
	static float **r = NULL;
	static int   num_pts = 0;

	if ((intfc->modified == NO) && (normal_at_point(p)[0] != HUGE_VAL))
	{
	    Tri_list_at_vertex.p = NULL;
	    Tri_list_at_vertex.num_tris = 0;
	    Tri_list_at_vertex.tris = NULL;
	    nor[0] = normal_at_point(p)[0];
	    nor[1] = normal_at_point(p)[1];
	    nor[2] = normal_at_point(p)[2];
	    return;
	}
	if (r == NULL)
	    matrix(&r,3,3,FLOAT);
	tri = Tri_of_hse(hse);
	Tri_list_at_vertex.p = p;
	Tri_list_at_vertex.num_tris =
	    set_tri_list_around_point(p,Tri_of_hse(hse),
	                              &Tri_list_at_vertex.tris,intfc);
	if ((Tri_list_at_vertex.num_tris+2) > num_pts)
	{
	    if (pts != NULL)
	        free(pts);
	    num_pts = 2*(Tri_list_at_vertex.num_tris+2);
	    vector(&pts,num_pts,sizeof(float*));
	}
	n = 0;
	if (OmitVertexInPlaneFit)
	    OmitVertexInPlaneFit = NO;
	else
	    pts[n++] = Coords(p);
	v = Vertex_of_point(Tri_list_at_vertex.tris[0],p);
	if (is_side_bdry(Tri_list_at_vertex.tris[0],Prev_m3(v)))
	{
	    pp = Point_of_tri(Tri_list_at_vertex.tris[0])[Prev_m3(v)];
	    pts[n++] = Coords(pp);
	}
	for (i = 0; i < Tri_list_at_vertex.num_tris; ++i)
	{
	    v = Vertex_of_point(Tri_list_at_vertex.tris[i],p);
	    pp = Point_of_tri(Tri_list_at_vertex.tris[i])[Next_m3(v)];
	    pts[n++] = Coords(pp);
	}
	affine_fit((const float* const*)pts,3,n,Tri_normal(tri),pbar,r,lambda);

	normal_at_point(p)[0] = nor[0] = r[2][0];
	normal_at_point(p)[1] = nor[1] = r[2][1];
	normal_at_point(p)[2] = nor[2] = r[2][2];
}		/*end plane_fit_normal3d*/


/*
*			area_weighted_normal3d():
*
*	Returns the Normal to the hypersurface hs at the point p on
*	hypersurface element hse of hs.  The normal points from the
*	negative side to the positive side of the hypersurface.
*
*	The normal is computed as the area weighted average of the normals
*	to the triangles that surround the point p. This normal vector has
*	the property that the sum of the signed areas of the triangled onto
*	a plane with this normal is a maxium.
*/

/*ARGSUSED*/
EXPORT  void area_weighted_normal3d(
	POINT		   *p,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF	   *hs,
	float		   *nor)
{
	INTERFACE    *intfc = hs->interface;
	if ((intfc->modified == NO) && (normal_at_point(p)[0] != HUGE_VAL))
	{
	    Tri_list_at_vertex.p = NULL;
	    Tri_list_at_vertex.num_tris = 0;
	    Tri_list_at_vertex.tris = NULL;
	    nor[0] = normal_at_point(p)[0];
	    nor[1] = normal_at_point(p)[1];
	    nor[2] = normal_at_point(p)[2];
	    return;
	}
	Tri_list_at_vertex.p = p;
	Tri_list_at_vertex.num_tris =
	    set_tri_list_around_point(p,Tri_of_hse(hse),
	                              &Tri_list_at_vertex.tris,intfc);
	set_area_weighted_normal(Tri_list_at_vertex.tris,
	                         Tri_list_at_vertex.num_tris,nor);
	normal_at_point(p)[0] = nor[0];
	normal_at_point(p)[1] = nor[1];
	normal_at_point(p)[2] = nor[2];
}		/*end area_weighted_normal3d*/


EXPORT	int	tri_list_computed_by_normal(
	POINT     *p,
	TRI       *tri,
	TRI       ***ptris,
	INTERFACE *intfc)
{
	if (Tri_list_at_vertex.p == NULL)
	{
	    Tri_list_at_vertex.p = p;
	    Tri_list_at_vertex.num_tris =
	        set_tri_list_around_point(p,tri,&Tri_list_at_vertex.tris,intfc);
	}

	*ptris = Tri_list_at_vertex.tris;
	return Tri_list_at_vertex.num_tris;
}		/*end tri_list_computed_by_normal*/

LOCAL	void	set_area_weighted_normal(
	TRI		**tris,
	int		num_tris,
	float		*nor)
{
	int	    i,j;
	float	    area_tri,length;
	const float *tnor;

	nor[0] = nor[1] = nor[2] = 0.0;
	for (i = 0; i < num_tris; ++i)
	{
	    tnor = Tri_normal(tris[i]);
	    area_tri = 0.5*Mag3d(tnor);
	    for (j = 0; j < 3; ++j)
	    {
	    	nor[j] += tnor[j]*area_tri;
	    }
	}
	length = Mag3d(nor);
	for (i = 0; i < 3; ++i)
	    nor[i] /= length;
}		/*end set_area_weighted_normal*/

/* TRI_Plus_normal and TRI_FullGeometry 
 * are moved to intfc/int.h */

LOCAL	TRI_STORAGE_TYPE tri_storage_type = TRI_PLUS_NORMAL;

#define Tri_normal_vector(_tri_)  (((TRI_Plus_normal*)(_tri_))->_nor)
#define sqr_normal_vector(_tri_)  (((TRI_Plus_normal*)(_tri_))->_sqr_norm)
#define fg_sv_store(_tri_)	  (((TRI_FullGeometry*)(_tri_))->_sv_store)
#define fg_side_vector(_tri_)     (((TRI_FullGeometry*)(_tri_))->_side_vector)
#define fg_length_side(_tri_)	  (((TRI_FullGeometry*)(_tri_))->_length_side)

LIB_LOCAL void	set_tri_storage_type(
	TRI_STORAGE_TYPE type)
{
	I_USER_INTERFACE *iuh = i_user_hook(3);
	switch (type)
	{
	case MIN_TRI_STORAGE:
	    tri_storage_type = type;
	    iuh->size_tri = sizeof(TRI);
	    break;
	case FULL_TRI_GEOMETRY:
	    tri_storage_type = type;
	    iuh->size_tri = sizeof(TRI_FullGeometry);
	    break;
	case TRI_PLUS_NORMAL:
	    tri_storage_type = type;
	    iuh->size_tri = sizeof(TRI_Plus_normal);
	    break;
	default:
	    screen("ERROR in set_tri_storage_type(), unknown value %d\n",type);
	    clean_up(ERROR);
	}
}		/*end set_tri_storage_type*/

EXPORT void reset_normal_on_intfc(
	INTERFACE	*intfc)
{
	SURFACE		**s;
	TRI		*t;


	switch (tri_storage_type)
	{
	case MIN_TRI_STORAGE:
	    break;
	case FULL_TRI_GEOMETRY:
	case TRI_PLUS_NORMAL:
	    for (s = intfc->surfaces; s && *s; ++s)
	        for (t = first_tri(*s); !at_end_of_tri_list(t,*s); t = t->next)
		    set_normal_of_tri(t);
	    break;
	default:
	    screen("ERROR in reset_normal_on_intfc(), unknown value %d\n",
		   tri_storage_type);
	    clean_up(ERROR);
	}
}		/*end reset_normal_on_intfc*/

/*ARGSUSED*/
EXPORT	void set_normal_of_tri(
	TRI	*tri)
{
	float *p[3];
	float *n;
	float **s, s0, s1, s2;
	int   i;

	if (tri_storage_type == MIN_TRI_STORAGE)
	    return;

	p[0] = Coords(Point_of_tri(tri)[0]);
	p[1] = Coords(Point_of_tri(tri)[1]);
	p[2] = Coords(Point_of_tri(tri)[2]);

	if (tri_storage_type == FULL_TRI_GEOMETRY)
	{
	    s = fg_side_vector(tri);
	    for (i = 0; i < 3; ++i)
	    {
	        s[i] = fg_sv_store(tri)[i];
	        s[i][0] = s0 = p[Next_m3(i)][0] - p[i][0];
	        s[i][1] = s1 = p[Next_m3(i)][1] - p[i][1];
	        s[i][2] = s2 = p[Next_m3(i)][2] - p[i][2];
	        fg_length_side(tri)[i] = sqrt(s0*s0 + s1*s1 + s2*s2);
	    }
	}
	else
	{
	    static float **sstore;
	    if (sstore == NULL)
		matrix(&sstore,3,3,FLOAT);
	    s = sstore;
	    for (i = 0; i < 3; ++i)
	    {
	        s[i][0] = p[Next_m3(i)][0] - p[i][0];
	        s[i][1] = p[Next_m3(i)][1] - p[i][1];
	        s[i][2] = p[Next_m3(i)][2] - p[i][2];
	    }
	}

	n = Tri_normal_vector(tri);
	n[0] = s[2][1]*s[0][2] - s[2][2]*s[0][1];
	n[1] = s[2][2]*s[0][0] - s[2][0]*s[0][2];
	n[2] = s[2][0]*s[0][1] - s[2][1]*s[0][0];
	sqr_normal_vector(tri) = n[0]*n[0] + n[1]*n[1] + n[2]*n[2];
}		/*end set_normal_of_tri*/

EXPORT	const float	*Tri_normal(
	const TRI *tri)
{
	if (tri_storage_type == MIN_TRI_STORAGE)
	{
	    const float  *p[3];
	    float        s[3][3];
	    static float n[3];
	    int   i;

	    p[0] = Coords(Point_of_tri(tri)[0]);
	    p[1] = Coords(Point_of_tri(tri)[1]);
	    p[2] = Coords(Point_of_tri(tri)[2]);

	    for (i = 0; i < 3; ++i)
	    {
	        s[i][0] = p[Next_m3(i)][0] - p[i][0];
	        s[i][1] = p[Next_m3(i)][1] - p[i][1];
	        s[i][2] = p[Next_m3(i)][2] - p[i][2];
	    }

	    n[0] = s[2][1]*s[0][2] - s[2][2]*s[0][1];
	    n[1] = s[2][2]*s[0][0] - s[2][0]*s[0][2];
	    n[2] = s[2][0]*s[0][1] - s[2][1]*s[0][0];
	    return n;
	}
	else
	    return Tri_normal_vector(tri);
}		/*end Tri_normal*/

EXPORT	float	sqr_norm(
	const TRI *tri)
{
	if (tri_storage_type == MIN_TRI_STORAGE)
	{
	    const float *n = Tri_normal(tri);
	    return n[0]*n[0] + n[1]*n[1] + n[2]*n[2];
	}
	else
	    return sqr_normal_vector(tri);
}		/*end sqr_norm*/

EXPORT	const float* const* side_vector(
	const TRI *tri)
{
	if (tri_storage_type == FULL_TRI_GEOMETRY)
	    return (const float* const*)fg_side_vector(tri);
	else
	{
	    float *p[3];
	    int   i;
	    static float *s[3];
	    static float ss[3][3];

	    p[0] = Coords(Point_of_tri(tri)[0]);
	    p[1] = Coords(Point_of_tri(tri)[1]);
	    p[2] = Coords(Point_of_tri(tri)[2]);

	    for (i = 0; i < 3; ++i)
	    {
	        s[i] = ss[i];
	        ss[i][0] = p[Next_m3(i)][0] - p[i][0];
	        ss[i][1] = p[Next_m3(i)][1] - p[i][1];
	        ss[i][2] = p[Next_m3(i)][2] - p[i][2];
	    }
	    return (const float* const*)s;
	}
}		/*end side_vector*/

EXPORT	const float *vector_on_tri_side(
	const TRI *tri,
	int       side,
	float     *v)
{
	if (v == NULL)
	{
	    if (tri_storage_type == FULL_TRI_GEOMETRY)
	        return fg_side_vector(tri)[side];
	    else
	    {
	        float *np = Coords(Point_of_tri(tri)[Next_m3(side)]);
	        float *p = Coords(Point_of_tri(tri)[side]);
	        static float sv[3];

	        sv[0] = np[0] - p[0];
	        sv[1] = np[1] - p[1];
	        sv[2] = np[2] - p[2];
	        return sv;
	    }
	}
	else
	{
	    if (tri_storage_type == FULL_TRI_GEOMETRY)
	    {
	        v[0] = fg_side_vector(tri)[side][0];
	        v[1] = fg_side_vector(tri)[side][1];
	        v[2] = fg_side_vector(tri)[side][2];
	    }
	    else
	    {
	        float *np = Coords(Point_of_tri(tri)[Next_m3(side)]);
	        float *p = Coords(Point_of_tri(tri)[side]);

	        v[0] = np[0] - p[0];
	        v[1] = np[1] - p[1];
	        v[2] = np[2] - p[2];
	    }
	    return v;
	}
}		/*end vector_on_tri_side*/

EXPORT	const float	*length_side(
	const TRI *tri)
{
	if (tri_storage_type == FULL_TRI_GEOMETRY)
	    return fg_length_side(tri);
	else
	{
	    static float l[3];
	    float        *p, *np;
	    int          side;

	    for (side = 0; side < 3; ++side)
	    {
	        p = Coords(Point_of_tri(tri)[side]);
	        np = Coords(Point_of_tri(tri)[Next_m3(side)]);
	        l[side] = sqrt(sqr(np[0]-p[0])+sqr(np[1]-p[1])+sqr(np[2]-p[2]));
	    }
	    return l;
	}
}		/*end length_side*/

EXPORT	float	length_of_tri_side(
	const TRI *tri,
	int       side)
{
	if (tri_storage_type == FULL_TRI_GEOMETRY)
	    return fg_length_side(tri)[side];
	else
	{
	    float *p = Coords(Point_of_tri(tri)[side]);
	    float *np = Coords(Point_of_tri(tri)[Next_m3(side)]);
	    return sqrt(sqr(np[0]-p[0])+sqr(np[1]-p[1])+sqr(np[2]-p[2]));
	}
}		/*end length_of_tri_side*/


#endif /* defined(THREED) */
#if defined(TWOD)

#define NO_ROBUST_CROSS_BONDS

#if defined(NO_ROBUST_CROSS_BONDS)

/*
*	               robust_cross_bonds():
*
*	Actually the non robust version of the robust_cross_bonds()
*	the robust version can be obtained by resetting the compiler
*	directive NO_ROBUST_CROSS_BONDS.  The only robustness in
*	this case is with respect to machine tolerance times a
*	a tolerance factor TOL_FAC(current_interface()).
*/

EXPORT	bool	robust_cross_bonds(
	BOND		*b1,
	int		on_b1,
	BOND		*b2,
	int		on_b2,
	float		*tcr1,
	float		*tcr2,
	RECT_GRID	*gr,
	POINT		*p)
{
	POINT		*p1s = b1->start,	*p1e = b1->end;
	POINT		*p2s = b2->start,	*p2e = b2->end;
	float		b1sx = Coords(p1s)[0],	b1sy = Coords(p1s)[1];
	float		b1ex = Coords(p1e)[0],	b1ey = Coords(p1e)[1];
	float		b2sx = Coords(p2s)[0],	b2sy = Coords(p2s)[1];
	float		b2ex = Coords(p2e)[0],	b2ey = Coords(p2e)[1];
	float		xx1, yy1, xx2, yy2;
	float		*h = gr->h;
	float		parallel = PARALLEL(current_interface());
	double		x0 = (double)b1sx - (double)b2sx;
	double		y0 = (double)b1sy - (double)b2sy;
	double		x1 = (double)b1ex - (double)b1sx;
	double		y1 = (double)b1ey - (double)b1sy;
	double		x2 = (double)b2ex - (double)b2sx;
	double		y2 = (double)b2ey - (double)b2sy;
	double		num1, num2, den, len;
	double		ux, uy, a0, a1, a2;
	double		lb1, lb2, lb12;
	double 		t1, t2;
	double		t1min, t1max, t2min, t2max;
	double		rcb_min_sc_sep = RCB_MIN_SC_SEP(current_interface());
	int		long_bond;
	int		dim = gr->dim;
	static bool	first = YES;
	static double	mac_tol;

#if defined(DEBUG_CROSS_BONDS)
	debug_print("rcb","Entered robust_cross_bonds()\n");
	if (debugging("rcb"))
	{
	    (void) printf("Bonds into robust_cross_bonds()\n");
	    (void) printf("b1 - ");	print_bond(b1);
	    (void) printf("b2 - ");	print_bond(b2);
	}
#endif /* defined(DEBUG_CROSS_BONDS) */

	if (first == YES)
	{
	    first = NO;

	    /*mac_tol = TOL_FAC(current_interface())*MACH_EPS;*/
	    mac_tol = RCB_MACH_TOL(current_interface());
	}

	/* Set robustness factor to zero except at nodes */

	t1min = (b1->prev == NULL) ?      -mac_tol : 0.0;
	t1max = (b1->next == NULL) ? 1.0 + mac_tol : 1.0;
	t2min = (b2->prev == NULL) ?      -mac_tol : 0.0;
	t2max = (b2->next == NULL) ? 1.0 + mac_tol : 1.0;

	den  = x2*y1  -  y2*x1;
	num1 = x0*y2  -  y0*x2;
	num2 = x0*y1  -  y0*x1;
	lb1  = (double) bond_length(b1);	lb2 = (double) bond_length(b2);

#if defined(DEBUG_CROSS_BONDS)
	if (debugging("rcb"))
	{
	    (void) printf("den %g num1 %g num2 %g lb1 %g lb2 %g\n",
	    		  den,num1,num2,lb1,lb2);
	}
#endif /* defined(DEBUG_CROSS_BONDS) */

	if (fabs(den) <= parallel*lb1*lb2) /* Parallel lines */
	{
	    lb12 = separation(p1s,p2s,gr->dim);

#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rcb"))
	    {
	        (void) printf("Bonds are parallel\n");
	        (void) printf("lb12 %g eps12 %g eps122 %g eps121 %g\n",
	        	      lb12,parallel*lb1*lb2,parallel*lb12*lb2,
	        	      parallel*lb12*lb1);
	    }
#endif /* defined(DEBUG_CROSS_BONDS) */

	    if ((fabs(num1) > parallel*lb12*lb2) 
	         || (fabs(num2) > parallel*lb12*lb1)   )
	    {
#if defined(DEBUG_CROSS_BONDS)
	        debug_print("rcb","Leaving robust_cross_bonds() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	        return NO;
	    }

	    if (max(lb1,lb2) < mac_tol)
	    {

	    /* both b1 and b2 have zero length, cross if p1s and p2s close */

	        if (scaled_separation(p1s,p2s,h,dim) > rcb_min_sc_sep)
	        {
#if defined(DEBUG_CROSS_BONDS)
	            debug_print("rcb","Leaving robust_cross_bonds() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	            return NO;
	        }
	        *tcr1 = *tcr2 = 0.0;
	        set_cross_position(b1sx,b1sy,on_b1,b2sx,b2sy,on_b2,p);
#if defined(DEBUG_CROSS_BONDS)
	        debug_print("rcb","Leaving robust_cross_bonds() BONDS CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	        return YES;
	    }

	    if (lb1 > lb2) 
	    {
	        long_bond = 1;
	        ux = x1;		uy = y1;
	        len = (double)bond_length(b1);
	        ux /= len;		uy /= len;
	    }
	    else
	    {
	        long_bond = 2;
	        ux = x2;		uy = y2;
	        len = (double)bond_length(b2);
	        ux /= len;		uy /= len;
	    }

	    a0 = x0*ux  +  y0*uy;
	    a1 = x1*ux  +  y1*uy;
	    a2 = x2*ux  +  y2*uy;

#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rcb"))
	    {
	        float A0, A1, A2;
	        (void) printf("long_bond %d ux %g uy %g a0 %g a1 %g a2 %g\n",
	        	      long_bond,ux,uy);

	        if (long_bond == 1)
	        {
	            A0 = a0/a1;	A2 = a2/a2;
	            (void) printf("a0 %g a2 %g\n",a0/a1,a2/a1);
	            (void) printf("tests: t1min %g t1max       %g\n",
	        		  t1min,t1max);
	            (void) printf("       -a0   %g t2min*a2-a0 %g\n",
    				  -A0,t2min*A2-A0);
	            (void) printf("       a2-a0 %g t2max*a2-a0 %g\n",
	        		  A2-A0,t2max*A2-A0);
	        }
	        else
	        {
	            A0 = a0/a2;	A1 = a1/a2;
	            (void) printf("a0 %g a1 %g\n",A0,A1);
	            (void) printf("tests: t2min %g t2max       %g\n",
	            	          t2min,t2max);
	            (void) printf("          a0 %g a0+t1min*a1 %g\n",
	            		  A0,A0+t1min*A1);
	            (void) printf("       a0+a1 %g a0+t1max*a1 %g\n",
	        		  A0+A1,A0+t1max*A1);
	        }
	    }
#endif /* defined(DEBUG_CROSS_BONDS) */

	    if (long_bond == 1) 
	    {
	        a0 /= a1;               a2 /= a1;
	        if ((t1min<-a0 && -a0<t1max) ||
	            (t2min<0.0 && (t1min<t2min*a2-a0 && t2min*a2-a0<t1max)))
	        {
	            t2 = 0.0;	t1 = -a0;
	        }
	        else if ((t1min<a2-a0 && a2-a0<t1max) ||
	        	 (t2max>1.0 &&
	        	     (t1min<t2max*a2-a0 && t2max*a2-a0 < t1max)))
	        {
	            t2 = 1.0;	t1 = a2 - a0;
	        }
	        else 
	        {
#if defined(DEBUG_CROSS_BONDS)
	            debug_print("rcb","Leaving robust_cross_bonds() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	            return NO;
	        }
	    }
	    else 
	    {
	        a0 /= a2;               a1 /= a2;
	        if ((t2min < a0 && a0 < t2max) ||
	            (t1min < 0.0 && (t2min < a0+t1min*a1
	        		      && a0+t1min*a1 < t2max)))
	        {
	            t1 = 0.0;	t2 = a0;
	        }
	        else if ((t2min < a0+a1 && a0+a1 < t2max) ||
	        	 (t1max > 1.0 &&
	        	     (t2min < a0+t1max*a1 && a0+t1max*a1 < t2max)))
	        {
	            t1 = 1.0;	t2 = a0 + a1;
	        }
	        else 
	        {
#if defined(DEBUG_CROSS_BONDS)
	            debug_print("rcb","Leaving robust_cross_bonds() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	            return NO;
	        }
	    }
	}

	else
	{
	    /* Should not have to worry that fabs(den) is small */
	    /* as this taken care of in previous if() test. */

	    t1 = num1/den;			t2 = num2/den;

#if defined(DEBUG_CROSS_BONDS)
	    if( debugging("rcb") )
	    {
	    	(void) printf("Bonds not parallel, t1 %g, t2 %g\n",t1,t2);
	    	(void) printf("t1min %g, t1max %g, t2min %g, t2max %g\n",
	    		      t1min,t1max,t2min,t2max);
	    }
#endif /* defined(DEBUG_CROSS_BONDS) */

	    if ((t1 < t1min || t1 > t1max) || (t2 < t2min || t2 > t2max))
	    {
#if defined(DEBUG_CROSS_BONDS)
	        debug_print("rcb","Leaving robust_cross_bonds() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    	return NO;
	    }
	}

	if (t1 < 0.0)
	    t1 = 0.0;
	if (t1 > 1.0)
	    t1 = 1.0;
	if (t2 < 0.0)
	    t2 = 0.0;
	if (t2 > 1.0)
	    t2 = 1.0;

#if defined(DEBUG_CROSS_BONDS)
	if (debugging("rcb"))
	    (void) printf("tcr: 1 %g  2 %g\n",t1,t2);
#endif /* defined(DEBUG_CROSS_BONDS) */

	*tcr1 = (float)t1;		*tcr2 = (float)t2;
	xx1 = b1sx + t1*x1;		yy1 = b1sy + t1*y1;
	xx2 = b2sx + t2*x2;		yy2 = b2sy + t2*y2;
	set_cross_position(xx1,yy1,on_b1,xx2,yy2,on_b2,p);

#if defined(DEBUG_CROSS_BONDS)
	debug_print("rcb","Leaving robust_cross_bonds() BONDS CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	return YES;
}		/*end robust_cross_bonds*/

#else /* defined(NO_ROBUST_CROSS_BONDS) */

#define DEBUG_CROSS_BONDS

/*
*			robust_cross_bonds():
*
*	Robust version of bond crossing function.  Extends each
*	bond by the scaled length factor ROBUST_FAC through nodes
*	and EPSILON*ROBUST_FAC elsewhere.
*/

EXPORT	bool	robust_cross_bonds(
	BOND		*b1,
	int		on_b1,
	BOND		*b2,
	int		on_b2,
	float		*tcr1,
	float		*tcr2,
	RECT_GRID	*gr,
	POINT		*p)
{
	POINT		*p1s = b1->start,	*p1e = b1->end;
	POINT		*p2s = b2->start,	*p2e = b2->end;
	float		b1sx = Coords(p1s)[0],	b1sy = Coords(p1s)[1];
	float		b1ex = Coords(p1e)[0],	b1ey = Coords(p1e)[1];
	float		b2sx = Coords(p2s)[0],	b2sy = Coords(p2s)[1];
	float		b2ex = Coords(p2e)[0],	b2ey = Coords(p2e)[1];
	float		xx1, yy1, xx2, yy2;
	float		*h = gr->h;
	double		x0 = (double)b1sx - (double)b2sx;
	double		y0 = (double)b1sy - (double)b2sy;
	double		x1 = (double)b1ex - (double)b1sx;
	double		y1 = (double)b1ey - (double)b1sy;
	double		x2 = (double)b2ex - (double)b2sx;
	double		y2 = (double)b2ey - (double)b2sy;
	double		t1, t2;
	double		num1, num2, den, len;
	double		ux, uy, a0, a1, a2;
	double		lb1, lb2, lb12;
	double		scaled_length;
	double		delta1, delta2, tcr1_min, tcr1_max, tcr2_min, tcr2_max;
	double		t1mid, dt1, t2mid, dt2;
	double		rcb_min_sc_sep = RCB_MIN_SC_SEP(current_interface());
	double		robust_fac = ROBUST_FAC(current_interface());
	int		long_bond;
	int		dim = gr->dim;

#if defined(DEBUG_CROSS_BONDS)
	debug_print("rcb","Entered robust_cross_bonds()\n");
	if (debugging("rcb"))
	{
	    (void) printf("Bonds into robust_cross_bonds()\n");
	    (void) printf("b1 - ");	print_bond(b1);
	    (void) printf("b2 - ");	print_bond(b2);
	}
#endif /* defined(DEBUG_CROSS_BONDS) */

	den  = x2*y1  -  y2*x1;
	num1 = x0*y2  -  y0*x2;
	num2 = x0*y1  -  y0*x1;
	lb1 = (double) bond_length(b1);	lb2 = (double) bond_length(b2);


	        /* Use a dimensionless delta */

	scaled_length = (double)scaled_bond_length(b1,h,dim);
	delta1 = (scaled_length < EPSILON) ? 0.0 : robust_fac*scaled_length;

	scaled_length = (double)scaled_bond_length(b2,h,dim);
	delta2 = (scaled_length < EPSILON) ? 0.0 : robust_fac*scaled_length;

	    /* delta is now stricter at curve ends */
	    /*  If this routine fails for a curve  */
	    /*    the robust_extend_() routines    */
	    /* 	  below should be called       */

	tcr1_min = (b1->prev == NULL) ? 0.0 - delta1 : 0.0 - EPSILON*delta1;
	tcr2_min = (b2->prev == NULL) ? 0.0 - delta2 : 0.0 - EPSILON*delta2;

	tcr1_max = (b1->next == NULL) ? 1.0 + delta1 : 1.0 + EPSILON*delta1;
	tcr2_max = (b2->next == NULL) ? 1.0 + delta2 : 1.0 + EPSILON*delta2;

	t1mid = 0.5*(tcr1_min + tcr1_max);
	t2mid = 0.5*(tcr2_min + tcr2_max);

	dt1 = 0.5*fabs(tcr1_max - tcr1_min);
	dt2 = 0.5*fabs(tcr2_max - tcr2_min);

#if defined(DEBUG_CROSS_BONDS)
	if (debugging("rcb"))
	{
	    (void) printf("den %g num1 %g num2 %g lb1 %g lb2 %g\n",
	    		  den,num1,num2,lb1,lb2);
	    (void) printf("bond 1: del %g min %g max %g mid %g dt %g\n",
	    		  delta2,tcr1_min,tcr1_max,t1mid,dt1);
	    (void) printf("bond 2: del %g min %g max %g mid %g dt %g\n",
	    		  delta1,tcr2_min,tcr2_max,t2mid,dt2);
	    (void) printf("tests: fabs(num1-t1mid*den) %g  dt1*fabs(den) %g\n",
	    		  fabs(num1-t1mid*den),dt1*fabs(den));
	    (void) printf("       fabs(num2-t2mid*den) %g  dt2*fabs(den) %g\n",
	    		  fabs(num2-t2mid*den),dt2*fabs(den));
	}
#endif /* defined(DEBUG_CROSS_BONDS) */


	if(    (fabs(num1 - t1mid*den) >= dt1*fabs(den))
	    || (fabs(num2 - t2mid*den) >= dt2*fabs(den))    )
	{
	    lb12 = separation(b1->start,b2->start,dim);

#if defined(DEBUG_CROSS_BONDS)
	    if (debug_print("rcb"))
	    	(void) printf("lb12 %g eps12 %g eps122 %g eps121 %g\n",
	    	              lb12,parallel*lb1*lb2,parallel*lb12*lb2,
	        	      parallel*lb12*lb1);
#endif /* defined(DEBUG_CROSS_BONDS) */

	    if ((fabs(den)  > parallel*lb1*lb2)  ||
		(fabs(num1) > parallel*lb12*lb2) ||
		(fabs(num2) > parallel*lb12*lb1))
	    {
#if defined(DEBUG_CROSS_BONDS)
	    	debug_print("rcb","Left robust_cross_bonds() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    	return NO;
	    }
	    if (lb1 > lb2) 
	    {
	    	long_bond = 1;
	    	ux = x1;		uy = y1;
	    	len = (double)bond_length(b1);
	    	ux /= len;		uy /= len;
	    }
	    else if (lb2 > 0.0)
	    {
	    	long_bond = 2;
	    	ux = x2;		uy = y2;
	    	len = (double)bond_length(b2);
	    	ux /= len;		uy /= len;
	    }
	    else
	    {

	    /* both b1 and b2 have zero length, cross if p1s and p2s close */

	    	if (scaled_separation(b1->start,b2->start,h,dim)>rcb_min_sc_sep)
	    	{
#if defined(DEBUG_CROSS_BONDS)
	    	    debug_print("rcb","Left robust_cross_bonds() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	                return NO;
	    	}
	    	*tcr1 = *tcr2 = 0.0;
	    	set_cross_position(b1sx,b1sy,on_b1,b2sx,b2sy,on_b2,p);
	    	return YES;
	    }
	    a0 = x0*ux  +  y0*uy;
	    a1 = x1*ux  +  y1*uy;
	    a2 = x2*ux  +  y2*uy;

#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rcb"))
	    {
	    	(void) printf("long_bond %d ux %g uy %g a0 %g a1 %g a2 %g\n",
	        	      long_bond,ux,uy,a0,a1,a2);

	    	(void) printf("tests: fabs(a0+t1mid*a1) %g  dt1*fabs(a1) %g\n",
	        	      fabs(a0+t1mid*a1),dt1*fabs(a1));
	    	(void) printf("       fabs(a0-a2+t1mid*a1) %g\n",
	    		      fabs(a0-a2+t1mid*a1));

	        (void) printf("       fabs(a0-t2mid*a2) %g  dt2*fabs(a2) %g\n",
	        	      fabs(a0-t2mid*a2),dt2*fabs(a2));
	        (void) printf("       fabs(a0+a1-t2mid*a2) %g\n",
	        	      fabs(a0+a1-t2mid*a2));

	    }
#endif /* defined(DEBUG_CROSS_BONDS) */
	    if (long_bond == 1) 
	    {
	    	if (fabs(a0 + t1mid*a1) <= dt1*fabs(a1))
	    	{
	    	    t2 = 0.0;	t1 = -a0/a1;
	    	}
	    	else if (fabs(a0 - a2 + t1mid*a1) <= dt1*fabs(a1))
	    	{
	    	    t2 = 1.0;	t1 = (a2 - a0)/a1;
	    	}
	    	else 
	    	{
#if defined(DEBUG_CROSS_BONDS)
	    	    debug_print("rcb","Left robust_cross_bonds() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    	    return NO;
	    	}
	    }
	    else 
	    {
	    	if (fabs(a0 - t2mid*a2) <= dt2*fabs(a2))
	    	{
	    	    t1 = 0.0;	t2 = a0/a2;
	    	}
	    	else if (fabs(a0 + a1 - t2mid*a2) <= dt2*fabs(a2))
	    	{
	    	    t1 = 1.0;	t2 = (a0 + a1)/a2;
	    	}
	    	else 
	    	{
#if defined(DEBUG_CROSS_BONDS)
	    	    debug_print("rcb","Left robust_cross_bonds() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    	    return NO;
	    	}
	    }
	}
	else 
	{
	    t1 = num1/den;		t2 = num2/den;
	}

#if defined(DEBUG_CROSS_BONDS)
	if (debugging("rcb"))
	    (void) printf("tcr: 1 %g  2 %g\n",t1,t2);
#endif /* defined(DEBUG_CROSS_BONDS) */

	if (t1 < 0.0)
	    t1 = 0.0;
	if (t1 > 1.0)
	    t1 = 1.0;
	if (t2 < 0.0)
	    t2 = 0.0;
	if (t2 > 1.0)
	    t2 = 1.0;

	*tcr1 = (float)t1;		*tcr2 = (float)t2;
	xx1 = b1sx + t1*x1;		yy1 = b1sy + t1*y1;
	xx2 = b2sx + t2*x2;		yy2 = b2sy + t2*y2;
	set_cross_position(xx1,yy1,on_b1,xx2,yy2,on_b2,p);

#if defined(DEBUG_CROSS_BONDS)
	debug_print("rcb","Left robust_cross_bonds() BONDS CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	return YES;
}		/*end robust_cross_bonds*/

#endif /* defined(NO_ROBUST_CROSS_BONDS) */


LOCAL	void set_cross_position(
	float		x1,
	float		y1,
	int		on_b1,
	float		x2,
	float		y2,
	int		on_b2,
	POINT		*pt)
{
	if (on_b1 == YES && on_b2 == NO)
	{
	    Coords(pt)[0] = x1;		Coords(pt)[1] = y1;
	}
	else if (on_b1 == NO && on_b2 == YES)
	{
	    Coords(pt)[0] = x2;		Coords(pt)[1] = y2;
	}
	else
	{
	    Coords(pt)[0] = 0.5*(x1 + x2);	Coords(pt)[1] = 0.5*(y1 + y2);
	}
}		/*end set_cross_position*/


EXPORT	int robust_extend_bond_to_cross_bond(
	BOND		*b1,
	ORIENTATION	extend_orient,
	BOND		*b2,
	float		*tcr1,
	float		*tcr2,
	POINT		*pc,
	float		*h,
	int		dim)
{
	double		p[MAXD],u[MAXD],v[MAXD],w[MAXD],vmw[MAXD];
	double		U_cross_V,U_cross_VMW, W_cross_V;
	double		U_dot_V, U_dot_W;
	double		scaled_length;
	double		ulen,para,alpha,delta;
	double		tcr2_min, tcr2_max;
	double		t2mid, dt2;
	double		dh[MAXD];
	int		i;

#if defined(DEBUG_CROSS_BONDS)
	debug_print("rebcb","Entered robust_extend_bond_to_cross_bond()\n");
	if (debugging("rebcb"))
	{
	    (void) printf("Checking for cross by extension of bond b1 %d ",b1);
	    (void) printf("with bond b2 %d\n",b2);
	    print_orientation("extend_orient = ",extend_orient,"\n");
	    (void) printf("b1 - ");		print_bond(b1);
	    (void) printf("b2 - ");		print_bond(b2);
	}
#endif /* defined(DEBUG_CROSS_BONDS) */

	if (b1 == NULL || b2 == NULL)
	{
#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rebcb"))
	    	(void) printf("NULL bond\n");
	    debug_print("rebcb",
	          "Left robust_extend_bond_to_cross_bond() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    return NO;	/* NULL bond */
	}
	for (i = 0; i < dim; ++i) dh[i] = (double) h[i];

	/* Find unit vector in direction of extension */

	if (extend_orient == POSITIVE_ORIENTATION) 
	{
	    for (i = 0; i < dim; ++i)
	    {
	    	p[i] = (double) Coords(b1->end)[i];
	    	u[i] = ((double) Coords(b1->start)[i]) - p[i];
	    }
	    *tcr1 = 0.0;
	}
	else 
	{
	    for (i = 0; i < dim; ++i)
	    {
	    	p[i] = (double) Coords(b1->start)[i];
	    	u[i] = ((double) Coords(b1->end)[i]) - p[i];
	    }
	    *tcr1 = 1.0;
	}
	if (dscaled_hypot(u,dh,dim) < 0.001)
	{
#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rebcb"))
	    	(void) printf("b1 is too short\n");
	    debug_print("rebcb",
	          "Left robust_extend_bond_to_cross_bond() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    return NO;
	}
	ulen = hypot(u[0],u[1]);

	for (i = 0; i < dim; ++i)
	{
	    u[i] /= ulen;
	    v[i] = ((double) Coords(b2->start)[i]) - p[i];
	    w[i] = ((double) Coords(b2->end)[i])   - p[i];
	    vmw[i] = v[i] - w[i];
	}

	scaled_length = dscaled_hypot(vmw,dh,dim);

	delta = (scaled_length < EPSILON) ? 0.0 : 0.01/scaled_length;
	tcr2_min = (b2->prev == NULL) ? -delta : -0.001*delta;
	tcr2_max = (b2->next == NULL) ? 1.0 + delta : 1.0 + 0.001*delta;

	t2mid = 0.5*(tcr2_min + tcr2_max);
	dt2 = 0.5*fabs(tcr2_max - tcr2_min);

	U_cross_VMW = u[0]*vmw[1] - u[1]*vmw[0];
	U_cross_V   = u[0]*v[1] - u[1]*v[0];
	W_cross_V   = w[0]*v[1] - w[1]*v[0];
	U_dot_V     = u[0]*v[0] + u[1]*v[1];
	U_dot_W     = u[0]*w[0] + u[1]*w[1];

#if defined(DEBUG_CROSS_BONDS)
	if (debugging("rebcb"))
	{
	    (void) printf("u = <%g, %g>, v = <%g, %g>\n",
	    	          u[0],u[1],v[0],v[1]);
	    (void) printf("w = <%g, %g>, vmw = <%g, %g>\n",
	    	          w[0],w[1],vmw[0],vmw[1]);
	    (void) printf("delta = %g, tcr2_min = %g, tcr2_max = %g\n",
	    	          delta,tcr2_min,tcr2_max);
	    (void) printf("t2mid = %g, dt2 = %g\n",t2mid,dt2);
	    (void) printf("U_cross_V = %g, U_cross_VMW = %g, W_cross_V = %g\n",
	    	          U_cross_V, U_cross_VMW,W_cross_V);
	    (void) printf("U_dot_V = %g, U_dot_W = %g\n",U_dot_V,U_dot_W);
	}
#endif /* defined(DEBUG_CROSS_BONDS) */

	if ((fabs(U_cross_V - t2mid*U_cross_VMW) > dt2*fabs(U_cross_VMW)) ||
	    	(W_cross_V*U_cross_VMW < 0.0) ||
	    	(fabs(W_cross_V) < fabs(ulen*U_cross_VMW)))
	{
#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rebcb"))	
	    	(void) printf("Cross by positive extension not on b2\n");
	    debug_print("rebcb",
	          "Left robust_extend_bond_to_cross_bond() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    return NO;
	}
	
	if (fabs(U_cross_VMW) > 0.0)
	{
	    alpha = U_cross_V/(U_cross_VMW);
	    para = (1.0 - alpha)*U_dot_V + alpha*U_dot_W;
	}
	else
	{
#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rebcb"))
	    	(void) printf("Parallel bonds\n");
#endif /* defined(DEBUG_CROSS_BONDS) */

	    if (U_dot_V < 0.0 && U_dot_W < 0.0)
	    {
#if defined(DEBUG_CROSS_BONDS)
	    	if (debugging("rebcb"))	
	    	{
	    	    (void) printf("b2 lies on opposite side of b1, "
	    	                  "with respect to extension direction\n");
	    	}
	    	debug_print("rebcb","Left robust_extend_bond_to_cross_bond() %s\n",
	              "NO CROSS");
#endif /* defined(DEBUG_CROSS_BONDS) */
	        return NO;
	    }
	    else if ((U_dot_W < 0.0) || (U_dot_V < U_dot_W))
	    {
	    	alpha = 0.0;
	    	para = U_dot_V;
	    }
	    else
	    {
	    	alpha = 1.0;
	    	para = U_dot_W;
	    }
	}
	if (para >= 0.0) 
	{
	    if (alpha < 0.0)
		alpha = 0.0;
	    if (alpha > 1.0)
		alpha = 1.0;
	    Coords(pc)[0] = Coords(b2->start)[0] + 
	    	alpha*(Coords(b2->end)[0] - Coords(b2->start)[0]);
	    Coords(pc)[1] = Coords(b2->start)[1] + 
	    	alpha*(Coords(b2->end)[1] - Coords(b2->start)[1]);
	    *tcr2 = (float) alpha;
#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rebcb"))	
	    {
	    	(void) printf("Cross by extension found at <%g, %g>\n",
	    		      Coords(pc)[0],Coords(pc)[1]);
	    	(void) printf("tcr1 = %g, tcr2 = %g\n",*tcr1,*tcr2);
	    }
	    debug_print("rebcb","Left robust_extend_bond_to_cross_bond() CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    return YES;
	}
	else
	{
#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rebcb"))	
	    {
	    	(void) printf("b2 lies on opposite side of b1, "
	    	              "with respect to extension direction\n");
	    }
	    debug_print("rebcb","Left robust_extend_bond_to_cross_bond() NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    return NO;
	}
}		/*end robust_extend_bond_to_cross_bond*/

EXPORT	int robust_extend_bonds_to_cross(
	BOND		*b1,
	ORIENTATION	c1_orient,
	int		on_b1,
	BOND		*b2,
	ORIENTATION	c2_orient,
	int		on_b2,
	POINT		*oldp,
	float		*tcr1,
	float		*tcr2,
	POINT		*p,
	RECT_GRID	*gr)
{
	double		hx = (double)(gr->h[0]);
	double		hy = (double)(gr->h[1]);
	double		ax = (double)(Coords(b1->start)[0] - 
	        		      Coords(b2->start)[0]);
	double		ay = (double)(Coords(b1->start)[1] - 
	        		      Coords(b2->start)[1]);
	double		bx = (double)(Coords(b1->end)[0] -
	        		      Coords(b1->start)[0]);
	double		by = (double)(Coords(b1->end)[1] -
	        		      Coords(b1->start)[1]);
	double		cx = (double)(Coords(b2->end)[0] -
	        		      Coords(b2->start)[0]);
	double		cy = (double)(Coords(b2->end)[1] -
	        		      Coords(b2->start)[1]);
	double		dx = (double)(0.5*(Coords(b1->start)[0] + 
	        			   Coords(b2->start)[0]));
	double		dy = (double)(0.5*(Coords(b1->start)[1] + 
	        			   Coords(b2->start)[1]));
	double		acc, acb, ccb, cdb;
	double		p10x, p10y, p11x, p11y;
	double		p20x, p20y, p21x, p21y;
	double		pmidx, pmidy;
	double		ex, ey, bce, bde, cce, cde;
	double		oldx = (double) Coords(oldp)[0];
	double		oldy = (double) Coords(oldp)[1];

#if defined(DEBUG_CROSS_BONDS)
	debug_print("rebcb","Entered robust_extend_bonds_to_cross()\n");
	if (debugging("rebcb"))
	{
	    (void) printf("Checking for cross by extension of bond b1 %d ",b1);
	    (void) printf("with bond b2 %d\n",b2);
	    print_orientation("c1_orient = ",c1_orient,"\n");
	    print_orientation("c2_orient = ",c2_orient,"\n");
	    (void) printf("b1 - ");		print_bond(b1);
	    (void) printf("b2 - ");		print_bond(b2);
	}
#endif /* defined(DEBUG_CROSS_BONDS) */

	if (robust_cross_bonds(b1,on_b1,b2,on_b2,tcr1,tcr2,gr,p))
	{
#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rebcb"))
	    	(void) printf("Bonds cross\n");
	    debug_print("rebcb","Left robust_extend_bonds_to_cross(), CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    return YES;
	}

	ccb = cx*by - cy*bx;	acc = ax*cy - ay*cx;	acb = ax*by - ay*bx;

	/* Check for cross in wrong direction */

	if ((c1_orient == POSITIVE_ORIENTATION && ccb*acc > 0.0)         ||
	    (c1_orient == NEGATIVE_ORIENTATION && ccb*(ccb - acc) > 0.0) ||
	    (c2_orient == POSITIVE_ORIENTATION && ccb*acb > 0.0)         ||
	    (c2_orient == NEGATIVE_ORIENTATION && ccb*(ccb - acb) > 0.0))
	{
#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rebcb")) 
	    	(void) printf("No cross in required direction\n");
	    debug_print("rebcb","Left robust_extend_bonds_to_cross(), NO CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    return NO;
	}

	/* Check for parallel bonds or cross far from oldp */

	if ((hx*fabs(ccb) <= fabs(ccb*(dx - oldx) + 0.5*(acc*bx + acb*cx))) ||
	    (hy*fabs(ccb) <= fabs(ccb*(dy - oldy) + 0.5*(acc*by + acb*cy))))
	{
#if defined(DEBUG_CROSS_BONDS)
	    if (debugging("rebcb"))
	    	(void) printf("parallel bonds\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	    cdb = cx*bx + cy*by;
	    if (fabs(ccb) > sqr(EPSILON)*fabs(cdb)) /* Vectors not || */
	    	return NO;
	    if (c1_orient == POSITIVE_ORIENTATION)
	    {
	    	p10x = (double) Coords(b1->start)[0];
	    	p10y = (double) Coords(b1->start)[1];
	    	p11x = (double) Coords(b1->end)[0];
	    	p11y = (double) Coords(b1->end)[1];
	    }
	    else
	    {
	    	p10x = (double) Coords(b1->end)[0];
	    	p10y = (double) Coords(b1->end)[1];
	    	p11x = (double) Coords(b1->start)[0];
	    	p11y = (double) Coords(b1->start)[1];
	    }
	    if (c2_orient == POSITIVE_ORIENTATION)
	    {
	    	p20x = (double) Coords(b2->start)[0];
	    	p20y = (double) Coords(b2->start)[1];
	    	p21x = (double) Coords(b2->end)[0];
	    	p21y = (double) Coords(b2->end)[1];
	    }
	    else
	    {
	    	p20x = (double) Coords(b2->end)[0];
	    	p20y = (double) Coords(b2->end)[1];
	    	p21x = (double) Coords(b2->start)[0];
	    	p21y = (double) Coords(b2->start)[1];
	    }
	    ex = p20x - p10x;	ey = p20y - p10y;
	    bce = bx*ey - by*ex;	bde = bx*ex + by*ey;
	    cce = cx*ey - cy*ex;	cde = cx*ex + cy*ey;
	    if ((fabs(bce) > EPSILON*fabs(bde)) || 
	    		(fabs(cce) > EPSILON*fabs(cde)))
	    	return NO;	/* Vectors not colinear */
	    pmidx = 0.5*(p11x + p21x);
	    pmidy = 0.5*(p11y + p21y);
	    if ((fabs(oldx - pmidx) > hx) || (fabs(oldy - pmidy) > hy))
	    	return NO;
	    Coords(p)[0] = (float) pmidx;	Coords(p)[1] = (float) pmidy;
	    *tcr1 = (c1_orient == POSITIVE_ORIENTATION) ? 0.0 : 1.0;
	    *tcr2 = (c2_orient == POSITIVE_ORIENTATION) ? 0.0 : 1.0;
	}
	else
	{
	    *tcr1 = acc/ccb;	*tcr2 = acb/ccb;
	    Coords(p)[0] = (float)dx + 0.5*(*tcr1*(float)bx + *tcr2*(float)cx);
	    Coords(p)[1] = (float)dy + 0.5*(*tcr1*(float)by + *tcr2*(float)cy);

	    *tcr1 = (c1_orient == POSITIVE_ORIENTATION) ?
	    		max(*tcr1,0.0) : min(*tcr1,1.0);
	    *tcr2 = (c2_orient == POSITIVE_ORIENTATION) ?
	    		max(*tcr2,0.0) : min(*tcr2,1.0);
	}
#if defined(DEBUG_CROSS_BONDS)
	if (debugging("rebcb")) 
	{
	    (void) printf("Cross by extension found\n");
	    (void) printf("at point <%g, %g>, tcr1 = %g, tcr2 = %g\n",
	    	          Coords(p)[0],Coords(p)[1],*tcr1,*tcr2);
	}
	debug_print("rebcb","Left robust_extend_bonds_to_cross(), CROSS\n");
#endif /* defined(DEBUG_CROSS_BONDS) */
	return YES;
}		/*end robust_extend_bonds_to_cross*/



EXPORT int robust_cross_bond_circle(
	BOND		*bond,
	POINT		*pcenter,
	float		Rsq,
	float		*tcr,
	POINT		*p)
{
	double		ax = (double) (Coords(bond->start)[0] -
	        		       Coords(pcenter)[0]);
	double		ay = (double) (Coords(bond->start)[1] -
	        		       Coords(pcenter)[1]);
	double		bx = (double) (Coords(bond->end)[0] -
	        		       Coords(bond->start)[0]);
	double		by = (double) (Coords(bond->end)[1] -
	        		       Coords(bond->start)[1]);
	double		a,b,c;
	double 		robust, hmin, hmax, h[2];
	double		rsq = (double) Rsq;
	double		rcbc_robust_fac = RCBC_ROBUST_FAC(current_interface());
	int		status = NO;

	static	double  epsilon;
	static	bool first = YES;

	if (first == YES)
	{
	    first = NO;
	    epsilon = TOL_FAC(current_interface())*MACH_EPS;
	}

	robust = rcbc_robust_fac*((double) bond_length(bond));
	hmin = -robust;	hmax = 1.0 + robust;

	a = sqr(bx) + sqr(by);
	b = 2.0*(ax*bx + ay*by);
	c = sqr(ax) + sqr(ay) - rsq;

	switch(robust_quad_roots_in_interval(h,a,b,c,hmin,hmax,epsilon))
	{
	case 1:
	    *tcr = h[0];
	    Coords(p)[0] = Coords(bond->start)[0] + h[0]*bx;
	    Coords(p)[1] = Coords(bond->start)[1] + h[0]*by;
	    status = YES;
	    break;
	case 2:
	    /*** Voluminous output
	    (void) printf("WARNING in robust_cross_bond_circle() - "
	                  "Circle crosses bond twice\n");
	    ***/
	    status = NO;
	    break;
	case 3:
	    /*** Voluminous output
	    (void) printf("WARNING in robust_cross_bond_circle() - "
	                  "Degenerate case, zero length bond\n"
	                  "With start of bond on circle\n"
	    ***/
	    *tcr = 0.5;
	    Coords(p)[0] = 0.5*(Coords(bond->start)[0] + Coords(bond->end)[0]);
	    Coords(p)[1] = 0.5*(Coords(bond->start)[1] + Coords(bond->end)[1]);
	    status = YES;
	    break;
	case 0:
	default:
	    /*** Voluminous output
	    (void) printf("WARNING in robust_cross_bond_circle() - "
	                  "No intersection of bond and circle\n");
	    ***/
	    status = NO;
	    break;
	}

	return status;

}		/*end robust_cross_bond_circle*/


/*
*				cross_sign():
*
*	Returns the sign of the cross product of the oriented bonds b1, b2.
*/


EXPORT int cross_sign(
	BOND		*b1,
	BOND		*b2)
{
	float		dx1 = Coords(b1->end)[0] - Coords(b1->start)[0];
	float		dy1 = Coords(b1->end)[1] - Coords(b1->start)[1];
	float		dx2 = Coords(b2->end)[0] - Coords(b2->start)[0];
	float		dy2 = Coords(b2->end)[1] - Coords(b2->start)[1];

	return (dx1*dy2 - dx2*dy1 > 0. ? 1 : -1);
}		/*end cross_sign*/

/*
*			c1_to_c2_direction():
*
*	Returns the direction (CLOCKWISE or COUNTER_CLOCK) of the 
*	interior angle (<= 180 deg) between the half lines defined
*	by two curves c1 and c2 with given orientation meeting at a common node.
*/


EXPORT ANGLE_DIRECTION c1_to_c2_direction(
	O_CURVE		*c1,
	O_CURVE		*c2)
{
	int		sign;

	sign = cross_sign(Bond_at_node_of_o_curve(c1),
	        	  Bond_at_node_of_o_curve(c2));
	if (c1->orient == NEGATIVE_ORIENTATION)
	    sign *= -1;
	if (c2->orient == NEGATIVE_ORIENTATION)
	    sign *= -1;
	return (sign == 1 ? COUNTER_CLOCK : CLOCKWISE);
}		/*end c1_to_c2_direction*/


/*
*			big_angle():
*
*	Computes angle between bonds b1,b2:  It is assumed that
*	the end of b1 coincides with the start of b2.
*	Special treatment is required for nodes and closed curves.
*	If the common point is a node, this node should be either a
*	boundary point or else the node of a closed curve.
*/

EXPORT void big_angle(
	BOND		*b1,
	CURVE		*c1,
	BOND		*b2,
	CURVE		*c2,
	float		*cos_angle,
	float		*sin_angle,
	RECT_GRID	*gr)
{
	float slopex,slopey,slopex_next,slopey_next;
	float distance,next_distance,xm,xl,xp,ym,yl,yp;

	if (b1==NULL && b2==NULL)
	{
	    screen("ERROR in big_angle(), null bonds\n");
	    clean_up(ERROR);
	}
	if (b1==NULL && (c1==c2) && is_closed_curve(c1))
	    b1 = c1->last;
	if (b2==NULL && (c1==c2) && is_closed_curve(c1))
	    b2 = c1->first;

	if (b1==NULL)				/* b2 starts curve */
	{
	    xl = Coords(b2->start)[0];	yl = Coords(b2->start)[1];
	    xp = Coords(b2->end)[0];	yp = Coords(b2->end)[1];
	    reflect(xl,yl,xp,yp,&xm,&ym,gr);
	}
	else if (b2==NULL)			/* b1 ends curve */
	{
	    xl = Coords(b1->end)[0];	yl = Coords(b1->end)[1];
	    xm = Coords(b1->start)[0];	ym = Coords(b1->start)[1];
	    reflect(xl,yl,xm,ym,&xp,&yp,gr);
	}
	else
	{
	    xm = Coords(b1->start)[0];	ym = Coords(b1->start)[1];
	    xl = Coords(b1->end)[0];	yl = Coords(b1->end)[1];
	    xp = Coords(b2->end)[0];	yp = Coords(b2->end)[1];
	}

	distance      = hypot(xm-xl,ym-yl);
	next_distance = hypot(xl-xp,yl-yp);
	slopex        = (xl - xm)/distance;
	slopey        = (yl - ym)/distance;
	slopex_next   = (xp - xl)/next_distance;
	slopey_next   = (yp - yl)/next_distance;
	*cos_angle    = slopex*slopex_next + slopey*slopey_next;
	*sin_angle    = slopex*slopey_next - slopey*slopex_next;
}		/*end big_angle*/

/*
*			reflect():
*			
*	Reflects the point x,y about the boundary, returning xr,yr.  The
*	boundary segment for reflection is determined by the condition 
*	that xl,yl lie of the boundary.
*/

LOCAL void reflect(
	float		xl,
	float		yl,
	float		x,
	float		y,
	float		*xr,
	float		*yr,
	RECT_GRID	*gr)
{
	float	rtol = RTOL(current_interface());
	if (xl < gr->L[0] + rtol*gr->h[0])
	{
	    *xr = 2*gr->L[0] - x;
	    *yr = y;
	}
	if (xl > gr->U[0] - rtol*gr->h[0])
	{ 
	    *xr = 2*gr->U[0] - x;
	    *yr = y;
	}
	if (yl < gr->L[1] + rtol*gr->h[1])
	{
	    *yr = 2*gr->L[1] - y;
	    *xr = x;
	}
	if (yl > gr->U[1] - rtol*gr->h[1])
	{
	    *yr = 2*gr->U[1] - y;
	    *xr = x;
	}
}		/*end reflect*/



/*
*		intersect_bond_with_curve_segment():
*
*	Finds the intersection if any of the bond b and the segment
*	of curve c from the bonds b1 and b2 inclusive.
*	It is assume that b2 follows b1 with respect to the given
*	orientation.
*	Returns YES if successful along with the point of intersection
*	pint and the bond bint on which the intersection lies.
*/

EXPORT	int intersect_bond_with_curve_segment(
	BOND		*b,
	BOND		*b1,
	BOND		*b2,
	O_CURVE		*oc,
	BOND		**bint,
	POINT		*pint,
	RECT_GRID	*gr)
{
	float		tcr1, tcr2;
	BOND		*bb;
	ORIENTATION	orient = oc->orient;

	for (bb = b1; bb != NULL; bb = Following_bond(bb,orient))
	{
	    if (robust_cross_bonds(b,YES,bb,YES,&tcr1,&tcr2,gr,pint))
	    {
	    	*bint = bb;
	    	return YES;
	    }
	    if (bb == b2)
		break;
	}

	return NO;
}		/*end intersect_bond_with_curve_segment*/

/*
*			is_short_curve():
*
*	Defines a curve as short on two criteria, one which is numerical
*	(gr->h[0], gr->h[1]) and the other which is physical (gr->X, gr->Y).
*	The physical criterion is relevent for extremely coarse grids.
*
*	The variable len scales the metric used to measure shortness.
*/

EXPORT int is_short_curve(
	CURVE		*c,
	ORIENTATION	c_orient,
	RECT_GRID	*gr,
	float		len)
{
	BOND		*b;
	float		*p, *p0;
	float		H[MAXD];
	int		i, dim = gr->dim;

	for (i = 0; i < dim; ++i)
	    H[i] = len*min(gr->h[i],0.25*(gr->U[i] - gr->L[i]));

	if (c->num_points <= SHORT_CURVE_NUM_POINTS(c->interface))
	    return YES;

	p0 = Coords(Node_of(c,c_orient)->posn);

	for (b = Bond_at_node(c,c_orient); b; b = Following_bond(b,c_orient))
	{
	    p = (c_orient == POSITIVE_ORIENTATION) ? Coords(b->end) :
	    					     Coords(b->start);

	    for (i = 0; i < dim; ++i)
	    	if (fabs(p[i] - p0[i]) >= H[i])
	            return NO;
	}
	return YES;
}		/*end is_short_curve*/

/*
*		robust_quad_roots_in_interval():
*
*	Find the real roots of the equation Ax**2 + Bx + C = 0
*	that lie in the interval [x0, x1].
*	The robustness comes from the condition that
*	quantities < epsilon are considered to be zero.
*	Returns the number of distinct real roots, unless
*	A, B and C are all zero, in which case it returns 3.
*/


EXPORT int robust_quad_roots_in_interval(
	double		*root,
	double		A,
	double		B,
	double		C,
	double		x0,
	double		x1,
	double		epsilon)
{
	double		xmid, len;
	double		maxabc, disc, sqrtd, Y;
	double		a, b, c;
	double		z;
	int		num_roots = 0;
	static bool	first = YES;
	static double	mac_tol;

	debug_print("quad_roots","Entered robust_quad_roots_in_interval()\n");
	if (debugging("quad_roots"))
	{
	    (void) printf("epsilon = %g\n",epsilon);
	    (void) printf("A = %.32g, B = %.32g, C = %.32g\n",A,B,C);
	    (void) printf("x0 = %.32g, x1 = %.32g\n",x0,x1);
	}
	if (first == YES)
	{
	    first = NO;

	    mac_tol = TOL_FAC(current_interface())*MACH_EPS;
	}
	len = fabs(x1 - x0);
	if (x1 < x0)
	{
	    double xtmp;

	    xtmp = x1;
	    x1 = x0;
	    x0 = xtmp;
	}
	x0 -= mac_tol*len; x1 += mac_tol*len;
	xmid = 0.5*(x0 + x1), len = 0.5*fabs(x1 - x0);
	a = A*len*len; b = (2.0*A*xmid + B)*len; c = (A*xmid + B)*xmid + C;
	if (b > 0.0)
	{
	    a = -a;
	    b = -b;
	    c = -c;
	}

	if (fabs(a) + fabs(b) + fabs(c) < epsilon)
	{
	    num_roots = 3;
	    if (debugging("quad_roots"))
	    {
	    	(void) printf("Degenerate quadratic\n"
	    	              "a = %.32g, b = %.32g, c = %.32g\n",a,b,c);
	    }
	    debug_print("quad_roots",
	          "Left robust_quad_roots_in_interval(), num_roots = %d\n",
	          num_roots);
	    return num_roots;
	}


	maxabc = max(fabs(b),fabs(c));
	maxabc = max(fabs(a),maxabc);

	a /= maxabc;	b /= maxabc;	c /= maxabc;
	disc = b*b - 4.0*a*c;
	sqrtd = sqrt(fabs(disc));
	if (debugging("quad_roots"))
	{
	    (void) printf("maxabc = %.32g\n",maxabc);
	    (void) printf("After normalization - ");
	    (void) printf("a = %.32g, b = %.32g, c = %.32g\n",a,b,c);
	    (void) printf("disc = %.32g\n",disc);
	}

	if (fabs(a + c) < fabs(b))
	{
	    if (debugging("quad_roots"))
	    	(void) printf("Single real root in interval\n");
	    z = 2.0*c/(-b + sqrtd);
	    root[0] = root[1] = xmid + len*z;
	    num_roots = 1;
	    if (debugging("quad_roots"))
	    	(void) printf("root = %.32g\n",root[0]);
	}
	else if (fabs(b) < 2.0*fabs(a) && sqrtd < 2.0*a*epsilon)
	{
	    if (debugging("quad_roots"))
	    	(void) printf("zero discriminant\n");
	    z = -0.5*b/a;
	    root[0] = root[1] = xmid + len*z;
	    num_roots = 1;
	    if (debugging("quad_roots"))
	    	(void) printf("root = %.32g\n",root[0]);
	}
	else if (disc > 0.0 && fabs(c) < fabs(a))
	{
	    if (debugging("quad_roots"))
	    	(void) printf("Two roots in interval\n");
	    num_roots = 2;
	    Y = -b + sqrtd;
	    if (a > 0.0)
	    {
	    	z = 2.0*c/Y;	root[0] = xmid + len*z;
	    	z = 0.5*Y/a;	root[1] = xmid + len*z;
	    }
	    else
	    {
	    	z = 0.5*Y/a;	root[0] = xmid + len*z;
	    	z = 2.0*c/Y;	root[1] = xmid + len*z;
	    }
	    if (debugging("quad_roots"))
	    	(void) printf("root[0] = %.32g, root[1] = %.32g\n",
	    	              root[0],root[1]);
	}
	debug_print("quad_roots",
	      "Left robust_quad_roots_in_interval(), num_roots = %d\n",
	      num_roots);
	return num_roots;
}		/*end robust_quad_roots_in_interval*/
#endif /* defined(TWOD) */
