/*
*				iscatter.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*/


#include <intfc/iloc.h>

/*
*		find_Cartesian_coordinates():
*
*	Identifies the coordinates of the subdomain with pp node id.  The
*	identification of the id and subdomain coordinates are given by
*
*	id = icoords[0] + icoords[1]*gmax[0] + icoords[2]*gmax[0]*gmax[1].
*
*	This function is the logical inverse of the function domain_id().
*
*	This defines the natural lexographical assignment of pp id numbers
*	to the subdomains, as illustrated for this 4x3 partition:
*
*	----------------------------------------
*	|        |         |         |         |
*	|   8    |    9    |   10    |   11    |
*	|        |         |         |         |
*	| (0,2)  |  (1,2)  |  (2,2)  |  (3,2)  |
*	----------------------------------------       -----------
*	|        |         |         |         |       |         |
*	|   4    |    5    |    6    |    7    |       |   id    | 
*	|        |         |         |         |       |         |
*	| (0,1)  |  (1,1)  |  (2,1)  |  (3,1)  |       | icoords |
*	----------------------------------------       -----------
*	|        |         |         |         |
*	|   0    |    1    |    2    |    3    |
*	|        |         |         |         |
*	| (0,0)  |  (1,0)  |  (2,0)  |  (3,0)  |
*	----------------------------------------
*
*/


EXPORT	void find_Cartesian_coordinates(
	int		id,
	PP_GRID		*pp_grid,
	int		*icoords)
{
	int 	dim =  pp_grid->Global_grid.dim;
	int 	d, G;

	for (d = 0; d < dim; d++)
	{
	    G = pp_grid->gmax[d];
	    icoords[d] = id % G;
	    id = (id - icoords[d])/G;
	}
}		/*end find_Cartesian_coordinates*/

/*
*			neighbor_id():
*
*	Finds the cartesian coordinate, him, and process id of the subdomain
*	on side, side, in the direction dir of the subdomain with
*	coordinates me.
*/

EXPORT	int	neighbor_id(
	int		*him,
	int		*me,
	int		dir,
	int		side,
	PP_GRID		*pp_grid)
{
	int		*G = pp_grid->gmax;
	int		i, dim = pp_grid->Global_grid.dim;

	for (i = 0; i < dim; i++)
	    him[i] = me[i];

	him[dir] = (me[dir] + 2*side - 1);
	if (him[dir] < 0)
	    him[dir] = G[dir] - 1;
	if (him[dir] >= G[dir])
	    him[dir] = 0;

	return domain_id(him,G,dim);
}		/*end neighbor_id*/

/*
*			domain_id():
*
*	Translates the integer coordinates of a subdomain into the pp id
*	of that domain.  This is the logical inverse of the function
*	find_Cartesian_coordinates().
*/

EXPORT  int domain_id(
	int		*icoords,
	int		*G,
	int		dim)
{
	int		tmpid;
	int		i;

	tmpid = icoords[dim-1];
	for (i = dim-2; i >= 0; i--)
	    tmpid = icoords[i] + G[i]*tmpid;
	return tmpid;
}		/*end domain_id*/


EXPORT	void print_PP_GRID_structure(
	PP_GRID		*pp_grid)
{
	int		i, j;
	int		dim;

	(void) printf("\n\t\tPP_GRID %p structure\n",(POINTER)pp_grid);
	if (pp_grid == NULL)
	{
		(void) printf("NULL PP_GRID");
		return;
	}
	dim = pp_grid->Global_grid.dim;
	(void) printf("\n\t1. Frame:\n\n");
	for (i = 0; i < dim; i++)
	{
		(void) printf("L[%d] = %g, U[%d] = %g, gmax[%d] = %d\n",
				i,pp_grid->dom[i][0],i,
				pp_grid->dom[i][pp_grid->gmax[i]],
				i,pp_grid->gmax[i]);
	}

	(void) printf("\n\t2. Subdomain Corners:\n\n");
	for (i = 0; i < pp_grid->Global_grid.dim; i++)
	{
		(void) printf("%d - direction:\n",i);
		for (j = 0; j < pp_grid->gmax[i]+1; j++)
			(void) printf("%-10g ",pp_grid->dom[i][j]);
		(void) printf("\n\n");
	}
 
 
	(void) printf("Global_grid\n");
	print_rectangular_grid(&pp_grid->Global_grid);
	(void) printf("Zoom_grid\n");
	print_rectangular_grid(&pp_grid->Zoom_grid);
 
	(void) printf("\t\tEnd PP_GRID %p structure\n\n",(POINTER)pp_grid);
}		/*end print_PP_GRID_structure*/
