/*
*				imkcurve.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Containing function of rebuilding interface within a mesh block.
*
*/


#define DEBUG_STRING "i_make_curve"

#include <intfc/int.h>

LOCAL	bool increment_theta(float*,float,float,float,ELLIP_PARAMS*,
			float*,float);


EXPORT CURVE *make_elliptic_curve(
	ELLIP_PARAMS *ellip,
	COMPONENT compin,
	COMPONENT compout,
	float space)
{
	int	    i;
	bool	    done;
	ORIENTATION nor_or = ellip->nor_orient;
	int	    closed = ellip->closed;
	float	    theta[MAXD-1];
	RECT_GRID   *gr = ellip->gr;
	float	    *h = gr->h, *L = gr->L, *U = gr->U;
	float	    coords[MAXD];
	float	    sgn;
	float	    ThetaS = ellip->ThetaS[0], ThetaE = ellip->ThetaE[0];
	CURVE	    *cur;
	NODE	    *ns, *ne;
	COMPONENT   l_comp,r_comp;

	if (nor_or == NEGATIVE_ORIENTATION)
	{
	    sgn = -1.0;
	    l_comp = compout;
	    r_comp = compin;
	}
	else
	{
	    sgn = 1.0;
	    l_comp = compin;
	    r_comp = compout;
	}
	if (closed == YES)
	{
	    if (nor_or == NEGATIVE_ORIENTATION)
	    {
		ThetaS = normalized_angle(ThetaS);
		ThetaE = ThetaS - 2.0*PI;
	    }
	    else
	    {
		ThetaS = normalized_angle(ThetaS);
		ThetaE = ThetaS + 2.0*PI;
	    }
	    coords_on_ellips(ellip->ThetaS,coords,ellip);
	    ns = make_node(Point(coords));
	    ne = ns;
	}
	else
	{
	}
	cur = make_curve(l_comp,r_comp,ns,ne);

	done = NO;
	theta[0] = ThetaS;
	while (done == NO)
	{
	    done = increment_theta(theta,ThetaS,ThetaE,sgn,ellip,h,space);
	    if (done == YES)
		break;
	    coords_on_ellips(theta,coords,ellip);
	    if (insert_point_in_bond(Point(coords),cur->last,cur) !=
		FUNCTION_SUCCEEDED)
	    {
	        screen("ERROR in g_make_ellipse(), "
		       "insert_point_in_bond() failed\n");
	        clean_up(ERROR);
	    }
	}
	return cur;
}	/* end make_elliptic_curve */


EXPORT	void coords_on_ellips(
	float		*theta,
	float		*coords,
	ELLIP_PARAMS	*ellip)
{
	float		*cen = ellip->cen, *rad = ellip->rad;
	FOURIER_POLY	*fpoly = ellip->fpoly;
	LEGENDRE_POLY	*lpoly = ellip->lpoly;
	int		dim = ellip->dim;
	float		Dr[MAXD];
	float		er[3];
	float		r;
	int		i, j;

	switch (dim)
	{
	case 2:
	    er[0] = cos(theta[0]);
	    er[1] = sin(theta[0]);
	    r = 1.0/hypot(er[0]/rad[0],er[1]/rad[1]);
	    break;
#if defined(THREED)
	case 3:
	    er[0] = cos(theta[0])*sin(theta[1]);
	    er[1] = sin(theta[0])*sin(theta[1]);
	    er[2] = cos(theta[1]);
	    r = 1.0/sqrt(sqr(er[0]/rad[0])+sqr(er[1]/rad[1])+sqr(er[2]/rad[2]));
	    break;
#endif /* defined(THREED) */
	default:
	    r = ERROR_FLOAT;
	    screen("ERROR in coords_on_ellips(), "
	           "invalid dimension %d\n",dim);
	    clean_up(ERROR);
	}
	if (fpoly != NULL)
	    r += fourier_poly(theta,fpoly);
	if ((dim == 2) && (lpoly != NULL))
	    r += legendre_poly(er[1],lpoly);

	for (i = 0; i < dim; i++)
            Dr[i] = 1.0*r*er[i];
	for (i = 0; i < dim; i++)
	    coords[i] = cen[i] + Dr[i];
}	/*end coords_on_ellips */


/*
*			increment_theta():
*
*	Computes the incremented theta in the polar coordinated
*	of the z = constant cross section of an ellipsoid.
*	The difference new_theta - theta is determined so that
*	the displacement of the two consecutive points on the
*	ellipsoid are approximately separated by the distance
*	space in the scaled metric with scale vector h.
*	This function assumes that if sgn and theta_e - theta_s
*	have the same algebraic sign.
*/

LOCAL	bool increment_theta(
	float		*theta,
	float		theta_s,
	float		theta_e,
	float		sgn,
	ELLIP_PARAMS	*ellip,
	float		*h,
	float		space)
{
	float		d_theta, dl;
	float		new_theta;
	float		rad[MAXD], r;
	float		er[2];
	bool		done;
	rad[0] = ellip->rad[0];	rad[1] = ellip->rad[1];
	dl = space*hypot(h[0],h[1]);
	er[0] = cos(*theta);	er[1] = sin(*theta);
	r = 1.0/hypot(er[0]/rad[0],er[1]/rad[1]);
	d_theta = dl/(r*r*r*hypot(er[0]/sqr(rad[0]),er[1]/sqr(rad[1])));

	new_theta = theta[0] + sgn*d_theta;
	done = NO;
	if (!Between(new_theta,theta_s,theta_e))
	{
	    new_theta = theta_e;
	    done = YES;
	}

	*theta = new_theta;
	return done;
}		/*end increment_theta*/
