/*
*				igview.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains printing routines for producing Geomview interface plots.
*/

#if defined(TWOD) || defined(THREED)

#include <intfc/iloc.h>

	/* LOCAL Function Prototypes */
#if defined(THREED)
LOCAL	void	gview_plot_cube(const char*,const char*,const float*,
	                        const float*,const float*,const float*);
LOCAL   void    gview_plot_curves(INTERFACE*,const float*,const float*,
	                          const char*,const char*,SURFACE_COLOR,int);
LOCAL	void	gview_plot_surfaces(INTERFACE*,RECT_GRID*,const float*,
	                            const float*,bool,
				    const char*,const char*,
				    bool,SURFACE_COLOR,SURFACE_COLOR);
LOCAL	void	print_polyline_description(FILE*,const char*,float* const*,int,
                                           SURFACE_COLOR color,float,int);
LOCAL	void	write_interpolated_color(FILE*,SURFACE_COLOR,
				         SURFACE_COLOR,float,float);
LOCAL   void 	write_color(FILE*,SURFACE_COLOR,float);
#endif /* if defined(THREED) */


#if defined(TWOD)
/* tmp for amr_overture 2d,infc geomview plot */

LOCAL   void    gview_plot_intfc2d(FILE*, INTERFACE*, RECT_GRID*);

EXPORT void set_default_clr_map(
        float  **clrmap,
        int    num_clr)
{
        int    i;
        i = 0;
        clrmap[i][0] = 0.098; clrmap[i][1] = 0.647;
        clrmap[i][2] = 0.400; clrmap[i][3] = 1.000;
        if((++i) == num_clr) return;

        clrmap[i][0] = 0.898; clrmap[i][1] = 0.400;
        clrmap[i][2] = 0.000; clrmap[i][3] = 1.000;
        if((++i) == num_clr) return;

        clrmap[i][0] = 0.500; clrmap[i][1] = 1.000;
        clrmap[i][2] = 0.500; clrmap[i][3] = 1.000;
        if((++i) == num_clr) return;

        clrmap[i][0] = 1.000; clrmap[i][1] = 0.000;
        clrmap[i][2] = 1.000; clrmap[i][3] = 1.000;
        if((++i) == num_clr) return;

        clrmap[i][0] = 0.000; clrmap[i][1] = 0.800;
        clrmap[i][2] = 1.000; clrmap[i][3] = 1.000;
        if((++i) == num_clr) return;

        clrmap[i][0] = 0.250; clrmap[i][1] = 0.808;
        clrmap[i][2] = 0.098; clrmap[i][3] = 1.000;
        i++;
}


#if defined(CONSERVATIVE_ALG)

EXPORT void gview_ts_mesh_tris_plot2d(
        const char   *dname,
        int          step, 
        TIME_SPACE_INTFC2D   **ts_intfc, 
        int          num,
        RECT_GRID    *gr,
        float        dt)
{
        static char   *fname = NULL, *ppfname = NULL;
        static size_t fname_len = 0, ppfname_len = 0;
        char          outname[256],outdir[256];
        int           i, myid, numnodes;
        const char    *nstep; 
        FILE          *fp;  
        static float  **clrmap = NULL;
        float         ccolor[4] = {0.0, 0.0, 0.0, 1.0}; 

        myid = pp_mynode(); numnodes = pp_numnodes();
        sprintf(outdir,"%s/%s",dname,"ts_intfc");
        ppfname = set_ppfname(ppfname,"ts_intfc",&ppfname_len);
        nstep = right_flush(step,7);
        sprintf(outname,"%s.ts%s",ppfname,nstep);

        if (create_directory(dname,YES) == FUNCTION_FAILED)
        {
            (void) printf("WARNING in gview_ts_mesh_tris_plot2d(), directory "
                          "%s doesn't exist and can't be created\n",dname);
            return;
        }
        if (create_directory(outdir,YES) == FUNCTION_FAILED)
        {
            (void) printf("WARNING in gview_ts_mesh_tris_plot2d(), directory "
                         "%s doesn't exist and can't be created\n",outdir);
            return;
        }

        fname = get_list_file_name(fname,outdir,outname,&fname_len);
        if ((fp = fopen(fname,"w")) == NULL)
        {
            (void) printf("WARNING in gview_ts_mesh_tris_plot2d(), "
                           "can't open %s\n",fname);
            return;
        }

        if(clrmap == NULL)
        {
            vector(&clrmap,6,sizeof(float*));
            for(i = 0; i < 6; i++)
                vector(&clrmap[i],4,sizeof(float));
            set_default_clr_map(clrmap, 6);
        }

        /* beginning of writting Vect to file */
        fprintf(fp,"{ LIST \n");

        gview_time_space_intfc_plot2d(fp, ts_intfc, gr, num, clrmap);
        gview_time_mesh_plot2d(fp, gr, dt, ccolor);

        /* end of LIST OBJ */
        fprintf(fp,"}\n");
 
        fclose(fp);  
}

EXPORT void gview_time_space_intfc_plot2d(
        FILE                 *fp, 
        TIME_SPACE_INTFC2D   **ts_intfc,
        RECT_GRID            *gr, 
        int                  num, 
        float                **clrmap)
{
        int           i, j, k;
        int           last = 0; 
        BOND_TRI      *btris; 
        TRI           *tri; 
        float         *dh = gr->h; 

        fprintf(fp,"{ LIST \n");

        for(k = 0; k < num; k++)
        {
            if(ts_intfc[k] == NULL)
                continue;  
            btris = ts_intfc[k]->btris; 
            for(j = 0; j < ts_intfc[k]->num_tris; j++)
            {
                i = 0;
                fprintf(fp,"{ OFF\n3 1 3\n\n");
                for(i = 0; i < 3; i++)
                {
                    if(j < ts_intfc[k]->num_old_tris)
                    {
                        if(i == 1)
                        {
                            fprintf(fp,"%f %f %f\n",
                              Coords(Point_of_tri(btris[j].tri)[i])[0],
                              Coords(Point_of_tri(btris[j].tri)[i])[1],
                                 dh[0]);
                        }
                        else
                        {
                            fprintf(fp,"%f %f %f\n",
                              Coords(Point_of_tri(btris[j].tri)[i])[0],
                              Coords(Point_of_tri(btris[j].tri)[i])[1],
                              Coords(Point_of_tri(btris[j].tri)[i])[2]);
                        }
                    }
                    else
                    {
                        if(i == 0 || i == 1)
                        {
                            fprintf(fp,"%f %f %f\n",
                              Coords(Point_of_tri(btris[j].tri)[i])[0],
                              Coords(Point_of_tri(btris[j].tri)[i])[1],
                                 dh[0]);
                        }
                        else
                        {
                            fprintf(fp,"%f %f %f\n",
                              Coords(Point_of_tri(btris[j].tri)[i])[0],
                              Coords(Point_of_tri(btris[j].tri)[i])[1],
                              Coords(Point_of_tri(btris[j].tri)[i])[2]);
                        }
                    }
                }
                if(last == 0)
                {
                   if(j < ts_intfc[k]->num_old_tris)
                       fprintf(fp,"\n3\t0 1 2 %f %f %f\n}\n",clrmap[0][0],clrmap[0][1],clrmap[0][2]);
                   else 
                       fprintf(fp,"\n3\t0 1 2 %f %f %f\n}\n",clrmap[3][0],clrmap[3][1],clrmap[3][2]);
                   last = 1;
                }
                else if(last == 1)
                {
                   if(j < ts_intfc[k]->num_old_tris)
                       fprintf(fp,"\n3\t0 1 2 %f %f %f\n}\n",clrmap[1][0],clrmap[1][1],clrmap[1][2]);
                   else 
                       fprintf(fp,"\n3\t0 1 2 %f %f %f\n}\n",clrmap[4][0],clrmap[4][1],clrmap[4][2]);
                   last = 2;
                }
                else
                {
                   if(j < ts_intfc[k]->num_old_tris)
                       fprintf(fp,"\n3\t0 1 2 %f %f %f\n}\n",clrmap[2][0],clrmap[2][1],clrmap[2][2]);
                   else 
                       fprintf(fp,"\n3\t0 1 2 %f %f %f\n}\n",clrmap[5][0],clrmap[5][1],clrmap[5][2]);
                   last = 0;
                }
            } 
        } 
        fprintf(fp,"}\n");  
}

EXPORT void gview_time_mesh_plot2d(
        FILE          *fp, 
        RECT_GRID     *gr,  
        float         dt,
        float         *color)
{
        int           i, j;
        int           imin[MAXD], imax[MAXD];
        int           *gmax = gr->gmax;
        int           n_x = 0, n_y = 0, n_z = 0;
        float         *L = gr->L;
        float         *U = gr->U;
        float         crds1[MAXD+1], crds2[MAXD+1];
        float         dh[MAXD];

        for (i = 0; i < gr->dim; i++)
        {
            imin[i] = 0;
            imax[i] = gmax[i];
            dh[i] = gr->h[i];
        }

        n_x = imax[0]-imin[0]+1;
        n_y = imax[1]-imin[1]+1;

        /* Draw all x-y plane */
        for(j = 0; j <= 1; j++)
        {
            crds1[2] = crds2[2] = 0.0 + j*dh[0];
            for(i=0; i<n_x; i++)
            {

                crds1[0] = crds2[0] = L[0]+(imin[0]+i)*dh[0];
                crds1[1] = L[1];
                crds2[1] = U[1];

                fprintf(fp,"{ VECT\n");
                fprintf(fp,"%1d %1d %1d\n", 1, 2, 1);
                fprintf(fp,"%1d %1d\n", 2, 1);

                fprintf(fp,"%f %f %f \t",crds1[0],crds1[1],crds1[2]);
                fprintf(fp,"%f %f %f \t",crds2[0],crds2[1],crds2[2]);
                fprintf(fp,"%f %f %f %f \t\n",
                  color[0], color[1], color[2], color[3]);
                fprintf(fp,"}\n");
            }

            for(i=0; i<n_y; i++)
            {
                crds1[0] = L[0];
                crds1[1] = crds2[1] = L[1]+(imin[1]+i)*dh[1];
                crds2[0] = U[0];

                fprintf(fp,"{ VECT\n");
                fprintf(fp,"%1d %1d %1d\n", 1, 2, 1);
                fprintf(fp,"%1d %1d\n", 2, 1);
                fprintf(fp,"%f %f %f \t",crds1[0],crds1[1],crds1[2]);
                fprintf(fp,"%f %f %f \t",crds2[0],crds2[1],crds2[2]);
                fprintf(fp,"%f %f %f %f \t\n",
                  color[0], color[1], color[2], color[3]);
                fprintf(fp,"}\n");
            }
        }

        /* Draw all x-z plane */
        for(j = 0; j <= gmax[1]; j++)
        {
            crds1[1] = crds2[1] = L[1] + j*dh[1];
            for(i=0; i<n_x; i++)
            {
                crds1[0] = crds2[0] = L[0]+(imin[0]+i)*dh[0];
                crds1[2] = 0.0;
                crds2[2] = dt;

                fprintf(fp,"{ VECT\n");
                fprintf(fp,"%1d %1d %1d\n", 1, 2, 1);
                fprintf(fp,"%1d %1d\n", 2, 1);

                fprintf(fp,"%f %f %f \t",crds1[0],crds1[1],crds1[2]);
                fprintf(fp,"%f %f %f \t",crds2[0],crds2[1],crds2[2]);
                fprintf(fp,"%f %f %f %f \t\n",
                  color[0], color[1], color[2], color[3]);
                fprintf(fp,"}\n");
            }
        }
}
#endif /* if defined(CONSERVATIVE_ALG) */

EXPORT void geomview_intfc_plot2d(
        const char    *dname,
        int           step,
        INTERFACE     *intfc,
        RECT_GRID     *gr)
{
        FILE          *fp;
        static char   *fname = NULL, *ppfname = NULL;
        static size_t fname_len = 0, ppfname_len = 0;
        char          outname[256],outdir[256];
        const char    *nstep;

        sprintf(outdir,"%s/%s",dname,"geomintfc");
        ppfname = set_ppfname(ppfname,"intfc",&ppfname_len);
        nstep = right_flush(step,7);        
        sprintf(outname,"%s.ts%s",ppfname,nstep);

        /*** TMP
        if (create_directory(dname,YES) == FUNCTION_FAILED)
        {
            (void) printf("WARNING in geomview_intfc_plot2d(), directory "
                          "%s doesn't exist and can't be created\n",dname);
            return;
        }
        if (create_directory(outdir,YES) == FUNCTION_FAILED)
        {
            (void) printf("WARNING in geomview_intfc_plot2d(), directory "
                         "%s doesn't exist and can't be created\n",outdir);
            return;
        }
        ***/

        fname = get_list_file_name(fname,outdir,outname,&fname_len);
        if ((fp = fopen(fname,"w")) == NULL)
        {
            (void) printf("WARNING in geomview_intfc_plot2d(), "
                           "can't open %s\n",fname);
            return;
        }

        /* Begin LIST OBJ */
        fprintf(fp,"{ LIST \n");

        gview_plot_intfc2d(fp,intfc,gr);

        /* end of LIST OBJ */
        fprintf(fp,"}\n");
        fclose(fp);
}

LOCAL void gview_plot_intfc2d(
        FILE          *fp,
        INTERFACE     *intfc,
        RECT_GRID     *gr)
{
        static const char *indent = "    ";
        static float **clrmap = NULL;
        float             ccolor[4] = {0.0, 0.0, 0.0, 1.0};
        float             coords[MAXD];
        CURVE             **c;
        CURVE             *oldc;
        BOND              *oldb;
        int               i; 
        float             *L = gr->L;
        float             *U = gr->U;
        float             *VL = gr->VL;
        float             *VU = gr->VU;

        if(clrmap == NULL)
        {
            vector(&clrmap,6,sizeof(float*));
            for(i = 0; i < 6; i++)
                vector(&clrmap[i],4,sizeof(float));
            set_default_clr_map(clrmap, 6);
        }

        /* beginning of writting Vect to file */
        for(i = 0, c = intfc->curves; c && *c;  c++, i++)
        {
            oldc = *c;
            oldb = oldc->first;
            fprintf(fp,"{ \n");
            fprintf(fp,"VECT\n");
            fprintf(fp,"%1d %1d %1d\n", 1, oldc->num_points, 1);
            fprintf(fp,"%1d\n%1d\n", oldc->num_points,1);

            while (oldb)
            {
                fprintf(fp,"%f %f %f \t",  Coords(oldb->start)[0],
                                Coords(oldb->start)[1] , 1.0);
                if (oldb == oldc->last)
                    break;
                oldb = oldb->next;
            }
            fprintf(fp,"%f %f %f \t\n",  Coords(oldb->end)[0],
                  Coords(oldb->end)[1] , 1.0);
            // fprintf(fp,"%f %f %f %f \t\n", 1.0, 0.2, 0.2, 0.8 );
            fprintf(fp,"%f %f %f %f \t\n", clrmap[i%6][0], clrmap[i%6][1],
                       clrmap[i%6][2], clrmap[i%6][3]);
            fprintf(fp,"}\n");
        }
        /* end of writting Vect to file     */

        /*  computational grid SKEL*/
        /* WHITE YELLOW */
        fprintf(fp,"{ VECT\n");
        fprintf(fp,"%1d %1d %1d\n", 1, 5, 1);
        fprintf(fp,"%1d\n%1d\n", 5,1);
        (void) fprintf(fp,"%f %f %f \t",L[0],L[1],1.0);
        (void) fprintf(fp,"%f %f %f \t",U[0],L[1],1.0);
        (void) fprintf(fp,"%f %f %f \t",U[0],U[1],1.0);
        (void) fprintf(fp,"%f %f %f \t\n",L[0],U[1],1.0);
        (void) fprintf(fp,"%f %f %f \t",L[0],L[1],1.0);
        fprintf(fp,"%f %f %f %f \t\n", 1.0, 1.0, 0.0, 1.0 );
        fprintf(fp,"}\n");

        /*  buffered computational grid SKEL*/
        /* WHITE COLOR   */
        fprintf(fp,"{ VECT\n");
        fprintf(fp,"%1d %1d %1d\n", 1, 5, 1);
        fprintf(fp,"%1d\n%1d\n", 5,1);
        (void) fprintf(fp,"%f %f %f \t",VL[0],VL[1],1.0);
        (void) fprintf(fp,"%f %f %f \t",VU[0],VL[1],1.0);
        (void) fprintf(fp,"%f %f %f \t",VU[0],VU[1],1.0);
        (void) fprintf(fp,"%f %f %f \t\n",VL[0],VU[1],1.0);
        (void) fprintf(fp,"%f %f %f \t",VL[0],VL[1],1.0);
        fprintf(fp,"%f %f %f %f \t\n", 1.0, 1.0, 1.0, 1.0 );
        fprintf(fp,"}\n");
}

#endif /* if defined(TWOD) */


#if defined(THREED)
EXPORT void geomview_interface_plot(
	const char *dname,
	INTERFACE  *intfc,
	RECT_GRID  *gr)
{
	float *BBL = topological_grid(intfc).GL;
	float *BBU = topological_grid(intfc).GU;

	if (intfc->dim != 3)
	    return;
	if (create_directory(dname,YES) == FUNCTION_FAILED)
	{
	    (void) printf("WARNING in geomview_interface_plot(), directory "
			  "%s doesn't exist and can't be created\n",dname);
	    return;
	}

	gview_plot_cube(dname,"grid",gr->L,gr->U,BBL,BBU);
	gview_plot_cube(dname,"vgrid",gr->VL,gr->VU,BBL,BBU);
	gview_plot_cube(dname,"ggrid",gr->GL,gr->GU,BBL,BBU);
	gview_plot_axes(dname,"axes",BBL,BBU,BBL,BBU);

	gview_plot_surfaces(intfc,gr,BBL,BBU,YES,dname,"surfs",
			    NO,pRED,pRED);

	gview_plot_surfaces(intfc,gr,BBL,BBU,YES,dname,"bdry",
			    YES,pBLUE,pGREEN);

	gview_plot_curves(intfc,BBL,BBU,dname,"curves",pYELLOW,5);
}		/*end geomview_interface_plot*/


LOCAL	void	gview_plot_surfaces(
	INTERFACE     *intfc,
	RECT_GRID     *gr,
	const float   *BBL,
	const float   *BBU,
	bool          clip,
	const char    *dname,
	const char    *name,
	bool	      bdry,
	SURFACE_COLOR color1,
	SURFACE_COLOR color2)
{
	FILE	          *file;
	POINT             *p;
	SURFACE	          **s;
	TRI	          *tri;
	bool              plot_surf;
	float 	          D, intensity = .5;
	float             L[MAXD],U[MAXD],tol[MAXD];
	float	          *crds;
	int	          num_surfs, num_tris, i, j, k, l;
	int               npts, ntris;
	static const char *indent = "    ";
	static float      *pts = NULL;
	static int        *verts = NULL;
	static int        alloc_len_verts = 0, alloc_len_pts = 0;
	static char       *fname = NULL;
	static size_t     fname_len = 0;

	fname = get_list_file_name(fname,dname,name,&fname_len);

	for (num_tris = 0, s = intfc->surfaces; s && *s; ++s)
	{
	    num_tris += (*s)->num_tri;
	    for (tri=first_tri(*s); !at_end_of_tri_list(tri,*s); tri=tri->next)
	    {
	        for (k = 0; k < 3; ++k)
		    Index_of_point(Point_of_tri(tri)[k]) = -1;
	    }
	}
	
	if (alloc_len_pts < 3*intfc->num_points)
	{
	    if (pts != NULL)
		free(pts);
	    alloc_len_pts = 3*intfc->num_points;
	    vector(&pts,alloc_len_pts,FLOAT);
	}
	if (alloc_len_verts < 4*num_tris)
	{
	    if (verts != NULL)
		free(verts);
	    alloc_len_verts = 4*num_tris;
	    vector(&verts,alloc_len_verts,INT);
	}
	for (i = 0; i < 3; i++)
	{
	    L[i] = gr->L[i] - 0.5*gr->h[i];
	    U[i] = gr->U[i] + 0.5*gr->h[i];
	    tol[i] = 0.00001*gr->h[i];
	}

        for (npts=0, ntris=0, num_surfs=0, s = intfc->surfaces; s && *s; ++s)
	{
	    if (bdry == YES  &&  !Boundary(*s))
		continue; 
	    if (bdry == NO  &&  Boundary(*s))
		continue;
	    if (clip == YES)
	    {
		plot_surf = NO;
	        for (tri = first_tri(*s); !at_end_of_tri_list(tri,*s); 
		     tri = tri->next)
	        {
	            for (k = 0; k < 3; ++k)
		    {
			crds = Coords(Point_of_tri(tri)[k]);
	                for (l = 0; l < 3; ++l)
			    if ((crds[l] < L[l] - tol[l]) || 
			        (U[l] + tol[l] < crds[l]))
				break;
			if (l < 3)
			    break;
		    }
		    if (k == 3)
		    {
			plot_surf = YES;
			for (k = 0; k < 3; ++k)
			{
		            p = Point_of_tri(tri)[k];
			    if (Index_of_point(p) == -1)
			    {
			        crds = Coords(p);
	                        for (l = 0; l < 3; ++l)
				    pts[3*npts+l] = crds[l];
				Index_of_point(p) = npts++;
			    }
			    verts[4*ntris+k] = Index_of_point(p);
			}
			verts[4*ntris+3] = num_surfs;
			++ntris;
		    }
		}
		if (plot_surf == YES)
		    ++num_surfs;
	    }
	    else
	    {
	        for (tri = first_tri(*s); !at_end_of_tri_list(tri,*s); 
		     tri = tri->next)
	        {
	            for (k = 0; k < 3; ++k)
		    {
		        p = Point_of_tri(tri)[k];
			if (Index_of_point(p) == -1)
			{
			    crds = Coords(p);
	                    for (l = 0; l < 3; ++l)
				pts[3*npts+l] = crds[l];
			    Index_of_point(p) = npts++;
			}
			verts[4*ntris+k] = Index_of_point(p);
		    }
		    verts[4*ntris+3] = num_surfs;
		    ++ntris;
		}
		++num_surfs;
	    }
	}
	if (num_surfs == 0)
	    return;

	if ((file = fopen(fname,"w")) == NULL)
	{
	    (void) printf("WARNING in gview_plot_surfaces(), "
	                  "can't open %s\n",fname);
	    return;
	}
	(void) fprintf(file,"{ LIST\n");

	gview_bounding_box(file,BBL,BBU,1,indent);

	(void) fprintf(file,"%s{\n%s%sOFF\n%s%s%6d %6d %6d\n",indent,
		       indent,indent,indent,indent,npts,ntris,0);
	for (i = 0; i < npts; ++i)
	{
	    (void) fprintf(file,"%s%s%-9g %-9g %-9g\n",indent,indent,
			   pts[3*i],pts[3*i+1],pts[3*i+2]);
	}
	D = (num_surfs == 1) ? 1.0 : 1/(num_surfs - 1.0);
	for (j = 0; j < ntris; ++j)
	{
	    (void) fprintf(file,"%s%s%-4d %-4d %-4d %-4d ",indent,indent,
			   3,verts[4*j],verts[4*j+1],verts[4*j+2]);
	    write_interpolated_color(file,color1,color2,verts[4*j+3]/D,
				     intensity);
	}
	(void) fprintf(file,"%s}\n",indent);
	(void) fprintf(file,"}\n");
	(void) fclose(file);
}		/*end gview_plot_surfaces*/

EXPORT	void	set_tri_list_bounding_box(
	TRI   **tris,
	int   nt,
	float *BBL,
	float *BBU,
	bool  preset,
	bool  cube)
{
    	const POINT *p;
	int   i, j, k;

	if (!preset)
	{
	    BBL[0] = BBL[1] = BBL[2] =  HUGE_VAL;
	    BBU[0] = BBU[1] = BBU[2] = -HUGE_VAL;
	}
	for (i = 0; i < nt; ++i)
	{
	    if (tris[i])
	    {
	        for (j = 0; j < 3; ++j)
	        {
	            p = Point_of_tri(tris[i])[j];
		    for (k = 0; k < 3; ++k)
		    {
	                if (Coords(p)[k] < BBL[k])
		            BBL[k] = Coords(p)[k];
		        if (Coords(p)[k] > BBU[k])
			    BBU[k] = Coords(p)[k];
		    }
	        }
	    }
	}
	if (cube)
	{
	    float len, mid;

	    len = BBU[0]-BBL[0];
	    for (j = 1; j < 3; ++j)
	    {
		if (len < (BBU[j]-BBL[j]))
		    len = BBU[j]-BBL[j];
	    }
	    for (j = 0; j < 3; ++j)
	    {
		if ((BBU[j]-BBL[j]) < len)
		{
		    mid = 0.5*(BBU[j]+BBL[j]);
		    BBL[j] = mid - 0.5*len;
		    BBU[j] = mid + 0.5*len;
		}
	    }
	}
}		/*end set_tri_list_boundary_box*/

EXPORT	void	set_point_list_bounding_box(
	POINT **pt,
	int   npt,
	float *BBL,
	float *BBU,
	bool  preset,
	bool  cube)
{
	const POINT *p;
	int i, j;

	if (!preset)
	{
	    BBL[0] = BBL[1] = BBL[2] =  HUGE_VAL;
	    BBU[0] = BBU[1] = BBU[2] = -HUGE_VAL;
	}
	for (i = 0; i < npt; ++i)
	{
	    p = pt[i];
	    for (j = 0; j < 3; ++j)
	    {
	        if (Coords(p)[j] < BBL[j])
	            BBL[j] = Coords(p)[j];
		if (Coords(p)[j] > BBU[j])
		    BBU[j] = Coords(p)[j];
	    }
	}
	if (cube)
	{
	    float len, mid;

	    len = BBU[0]-BBL[0];
	    for (j = 1; j < 3; ++j)
	    {
		if (len < (BBU[j]-BBL[j]))
		    len = BBU[j]-BBL[j];
	    }
	    for (j = 0; j < 3; ++j)
	    {
		if ((BBU[j]-BBL[j]) < len)
		{
		    mid = 0.5*(BBU[j]+BBL[j]);
		    BBL[j] = mid - 0.5*len;
		    BBU[j] = mid + 0.5*len;
		}
	    }
	}
}		/*end set_point_list_bounding_box*/

EXPORT	void set_vector_bounding_box(
	const float *p,
	const float *u,
	float       c,
	float       *BBL,
	float       *BBU,
	bool        preset,
	bool        cube)
{
	float x;
	int j;

	if (!preset)
	{
	    BBL[0] = BBL[1] = BBL[2] =  HUGE_VAL;
	    BBU[0] = BBU[1] = BBU[2] = -HUGE_VAL;
	}
	for (j = 0; j < 3; ++j)
	{
	    x = p[j] + c*u[j];
	    if (x < BBL[j])
		BBL[j] = x;
	    if (x > BBU[j])
		BBU[j] = x;
	}
	if (cube)
	{
	    float len, mid;

	    len = BBU[0]-BBL[0];
	    for (j = 1; j < 3; ++j)
	    {
		if (len < (BBU[j]-BBL[j]))
		    len = BBU[j]-BBL[j];
	    }
	    for (j = 0; j < 3; ++j)
	    {
		if ((BBU[j]-BBL[j]) < len)
		{
		    mid = 0.5*(BBU[j]+BBL[j]);
		    BBL[j] = mid - 0.5*len;
		    BBU[j] = mid + 0.5*len;
		}
	    }
	}
}		/*end set_vector_bounding_box*/

EXPORT	void	gview_plot_triangle_list(
	const char  *dname,
	const char  *name,
	TRI         **tris,
	int         num_tris,
	float       red_start,
	float       green_start,
	float       blue_start,
	float       red_end,
	float       green_end,
	float       blue_end,
	float       alpha,
	const float *BBL,
	const float *BBU)
{
	FILE              *file;
	POINT             **p;
	float             red, green, blue;
	float             D, x;
	int               nt, i, j;
	static char       *fname = NULL;
	static size_t     fname_len = 0;
	static const char *indent = "    ";

	fname = get_list_file_name(fname,dname,name,&fname_len);
	if ((file = fopen(fname,"w")) == NULL)
	{
	    (void) printf("WARNING in gview_plot_triangle_list(), "
	                  "can't open %s\n",fname);
	    return;
	}
	for (nt = 0, i = 0; i < num_tris; ++i)
	    if (tris[i] != NULL) ++nt;

	(void) fprintf(file,"{ LIST\n");
	gview_bounding_box(file,BBL,BBU,1,indent);

	(void) fprintf(file,"%s{\n%s%sOFF\n%s%s%6d %6d %6d\n",indent,
		       indent,indent,indent,indent,3*nt,nt,0);
	for (i = 0; i < num_tris; ++i)
	{
	    if (tris[i] != NULL)
	    {
	        p = Point_of_tri(tris[i]);
	        for (j = 0; j < 3; ++j)
	            (void) fprintf(file,"%s%s%-9g %-9g %-9g\n",indent,indent,
			           Coords(p[j])[0],Coords(p[j])[1],
				   Coords(p[j])[2]);
	    }
	}
	D = num_tris-1;
	for (i = 0; i < num_tris; ++i)
	{
	    x = i/D;
	    red = (1.0 - x)*red_start + x*red_end;
	    if (red < 0.0) red = 0.0; if (red > 1.0) red = 1.0;
	    green = (1.0 - x)*green_start + x*green_end;
	    if (green < 0.0) green = 0.0; if (green > 1.0) green = 1.0;
	    blue = (1.0 - x)*blue_start + x*blue_end;
	    if (blue < 0.0) blue = 0.0; if (blue > 1.0) blue = 1.0;
	    if (tris[i] != NULL)
	    {
	        (void) fprintf(file,"%s%s%-4d %-4d %-4d %-4d %g %g %g %g\n",
			       indent,indent,3,3*i,3*i+1,3*i+2,
			       red,green,blue,alpha);
	    }
	}
	(void) fprintf(file,"%s}\n",indent);

	(void) fprintf(file,"}\n");
	(void) fclose(file);
}		/*end gview_plot_triangle_list*/

EXPORT	void	gview_plot_polyline(
	const char  *dname,
	const char  *name,
	POINT       **v,
	int 	    nv,
	bool        closed,
	float       red,
	float       green,
	float       blue,
	float       alpha,
	const float *BBL,
	const float *BBU)
{
	FILE              *file;
	int               i;
	static char       *fname = NULL;
	static size_t     fname_len = 0;
	static const char *indent = "    ";

	fname = get_list_file_name(fname,dname,name,&fname_len);
	if ((file = fopen(fname,"w")) == NULL)
	{
	    (void) printf("WARNING in gview_plot_polyline(), "
	                  "can't open %s\n",fname);
	    return;
	}

	(void) fprintf(file,"{ LIST\n");
	gview_bounding_box(file,BBL,BBU,1,indent);

	(void) fprintf(file,"%s{\n%s%sVECT\n",indent,indent,indent);
	(void) fprintf(file,"%s%s%6d %6d %6d\n",indent,indent,
		       1,(closed)?nv+1:nv,1);
	(void) fprintf(file,"%s%s%6d\n",indent,indent,(closed)?nv+1:nv);
	(void) fprintf(file,"%s%s%6d\n",indent,indent,1);
	for (i = 0; i < nv; ++i)
	    (void) fprintf(file,"%s%s%g %g %g\n",indent,indent,
			   Coords(v[i])[0],Coords(v[i])[1],Coords(v[i])[2]);
	if (closed)
	    (void) fprintf(file,"%s%s%g %g %g\n",indent,indent,
			   Coords(v[0])[0],Coords(v[0])[1],Coords(v[0])[2]);
	(void) fprintf(file,"%s%s%g %g %g %g\n",indent,indent,
		       red,green,blue,alpha);
	(void) fprintf(file,"%s}\n",indent);
	(void) fprintf(file,"}\n");
	(void) fclose(file);
}		/*end gview_plot_polyline*/

EXPORT	void	gview_plot_vertices(
	const char  *dname,
	const char  *name,
	POINT       **v,
	int 	    nv,
	const float *BBL,
	const float *BBU)
{
	FILE              *file;
	float             r;
	int               i;
	static char       *fname = NULL;
	static size_t     fname_len = 0;
	static const char *indent = "    ";

	if ((v == NULL) || (nv <= 0))
	    return;
	fname = get_list_file_name(fname,dname,name,&fname_len);
	if ((file = fopen(fname,"w")) == NULL)
	{
	    (void) printf("WARNING in gview_plot_vertices(), "
	                  "can't open %s\n",fname);
	    return;
	}

	r = 0.01*sqrt((BBU[0]-BBL[0])*(BBU[0]-BBL[0]) +
	              (BBU[1]-BBL[1])*(BBU[1]-BBL[1]) +
	              (BBU[2]-BBL[2])*(BBU[2]-BBL[2]));
	(void) fprintf(file,"{ LIST\n");
	gview_bounding_box(file,BBL,BBU,1,indent);
	for (i = 0; i < nv; ++i)
	{
	    (void) fprintf(file,"%s{\n%s%sSPHERE\n",indent,indent,indent);
	    (void) fprintf(file,"%s%s%g\n",indent,indent,r);
	    (void) fprintf(file,"%s%s%g %g %g\n",indent,indent,
			   Coords(v[i])[0],Coords(v[i])[1],Coords(v[i])[2]);
	    (void) fprintf(file,"%s}\n",indent);
	}
	(void) fprintf(file,"}\n");
	(void) fclose(file);
}		/*end gview_plot_vertices*/

EXPORT	void	gview_plot_axes(
	const char  *dname,
	const char  *name,
	const float *L,
	const float *U,
	const float *BBL,
	const float *BBU)
{
	FILE              *file;
	static char       *fname = NULL;
	static size_t     fname_len = 0;
	static const char *indent = "    ";

	fname = get_list_file_name(fname,dname,name,&fname_len);

	if ((file = fopen(fname,"w")) == NULL)
	{
	    (void) printf("WARNING in gview_plot_axes(), "
	                  "can't open %s\n",fname);
	    return;
	}
	(void) fprintf(file,"{ LIST\n");

	gview_bounding_box(file,BBL,BBU,1,indent);

	/* Print Coordinate axes */
	(void) fprintf(file,"%s{\n",indent);
	(void) fprintf(file,"%s%sVECT\n",indent,indent);
	(void) fprintf(file,"%s%s%6d %6d %6d\n",indent,indent,3,6,3);
	(void) fprintf(file,"%s%s%6d %6d %6d\n",indent,indent,2,2,2);
	(void) fprintf(file,"%s%s%6d %6d %6d\n",indent,indent,1,1,1);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,L[0],L[1],L[2]);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,U[0],L[1],L[2]);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,L[0],L[1],L[2]);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,L[0],U[1],L[2]);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,L[0],L[1],L[2]);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,L[0],L[1],U[2]);
	(void) fprintf(file,"%s%s%6g %6g %6g %6g\n",
			    indent,indent,1.0,0.0,0.0,0.0);
	(void) fprintf(file,"%s%s%6g %6g %6g %6g\n",
			    indent,indent,0.0,1.0,0.0,0.0);
	(void) fprintf(file,"%s%s%6g %6g %6g %6g\n",
			    indent,indent,0.0,0.0,1.0,0.0);
	(void) fprintf(file,"%s}\n",indent);

	(void) fprintf(file,"}\n");
	(void) fclose(file);
}		/*end gview_plot_axes*/

EXPORT	void	gview_plot_coord_sys(
	const char  *dname,
	const char  *name,
	const float *o,
	const float *a0,
	const float *a1,
	const float *a2,
	const float *BBL,
	const float *BBU)
{
	FILE              *file;
	static char       *fname = NULL;
	static size_t     fname_len = 0;
	static const char *indent = "    ";

	fname = get_list_file_name(fname,dname,name,&fname_len);

	if ((file = fopen(fname,"w")) == NULL)
	{
	    (void) printf("WARNING in gview_plot_coord_sys(), "
	                  "can't open %s\n",fname);
	    return;
	}
	(void) fprintf(file,"{ LIST\n");

	gview_bounding_box(file,BBL,BBU,1,indent);

	/* Print Coordinate axes */
	(void) fprintf(file,"%s{\n",indent);
	(void) fprintf(file,"%s%sVECT\n",indent,indent);
	(void) fprintf(file,"%s%s%6d %6d %6d\n",indent,indent,3,6,3);
	(void) fprintf(file,"%s%s%6d %6d %6d\n",indent,indent,2,2,2);
	(void) fprintf(file,"%s%s%6d %6d %6d\n",indent,indent,1,1,1);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,o[0]-a0[0],o[1]-a0[1],o[2]-a0[2]);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,o[0]+a0[0],o[1]+a0[1],o[2]+a0[2]);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,o[0]-a1[0],o[1]-a1[1],o[2]-a1[2]);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,o[0]+a1[0],o[1]+a1[1],o[2]+a1[2]);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,o[0]-a2[0],o[1]-a2[1],o[2]-a2[2]);
	(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			    indent,indent,o[0]+a2[0],o[1]+a2[1],o[2]+a2[2]);
	(void) fprintf(file,"%s%s%6g %6g %6g %6g\n",
			    indent,indent,1.0,0.0,0.0,0.0);
	(void) fprintf(file,"%s%s%6g %6g %6g %6g\n",
			    indent,indent,0.0,1.0,0.0,0.0);
	(void) fprintf(file,"%s%s%6g %6g %6g %6g\n",
			    indent,indent,0.0,0.0,1.0,0.0);
	(void) fprintf(file,"%s}\n",indent);

	(void) fprintf(file,"}\n");
	(void) fclose(file);
}		/*end gview_plot_coord_sys*/

EXPORT	void	gview_bounding_box(
	FILE        *file,
	const float *BBL,
	const float *BBU,
	int         indent_level,
	const char  *indent)
{
	char bindent[256];
	char vfmt[256];
	int  i;

	bindent[0] = '\0';
	for (i = 0; i < indent_level; ++i)
	    (void) strcat(bindent,indent);

	(void) sprintf(vfmt,"%s%s%s\n",bindent,indent,"%-9g %-9g %-9g");
	/* Print bounding verticies */
	(void) fprintf(file,"%s{\n",bindent);
	(void) fprintf(file,"%s%sSKEL\n",bindent,indent);
	(void) fprintf(file,"%s%s%6d %6d\n",bindent,indent,8,8);
	(void) fprintf(file,vfmt,BBL[0],BBL[1],BBL[2]);
	(void) fprintf(file,vfmt,BBU[0],BBL[1],BBL[2]);
	(void) fprintf(file,vfmt,BBU[0],BBU[1],BBL[2]);
	(void) fprintf(file,vfmt,BBL[0],BBU[1],BBL[2]);
	(void) fprintf(file,vfmt,BBL[0],BBL[1],BBU[2]);
	(void) fprintf(file,vfmt,BBU[0],BBL[1],BBU[2]);
	(void) fprintf(file,vfmt,BBU[0],BBU[1],BBU[2]);
	(void) fprintf(file,vfmt,BBL[0],BBU[1],BBU[2]);
	for (i = 0; i < 8; ++i)
	    (void) fprintf(file,"%s%s%-9d %-9d %-9d\n",bindent,indent,2,i,i);
	(void) fprintf(file,"%s}\n",bindent);
}		/*end gview_bounding_box*/

LOCAL void write_interpolated_color(
	FILE          *file,
	SURFACE_COLOR color1,
	SURFACE_COLOR color2,
	float         d,
	float         intensity)
{
  	float color[8][4] =
	{
	    { 0.0, 0.0, 0.0, 0.75 }, /* black     */
	    { 1.0, 0.0, 0.0, 0.75 }, /* red       */
	    { 0.0, 1.0, 0.0, 0.75 }, /* green     */
	    { 1.0, 1.0, 0.0, 0.75 }, /* yellow    */
	    { 0.0, 0.0, 1.0, 0.75 }, /* blue      */ 
	    { 1.0, 0.0, 1.0, 0.75 }, /* magenta   */
	    { 0.0, 1.0, 1.0, 0.75 }, /* cyan      */
	    { 1.0, 1.0, 1.0, 0.75 }, /* white     */
	};

	float write_color;
	int i;

	for (i = 0 ; i < 4 ; ++i)
        {
	    write_color = ((1.0 - d)*color[color1][i] + d*color[color2][i])*
			  intensity;
	    (void) fprintf(file,"%7.5f ",write_color);
	}
	(void) fprintf(file,"\n");
}		/* end write_interpolated_color */

LOCAL void gview_plot_cube(
	const char *dname,
	const char *gname,
	const float *L,
	const float *U,
	const float *BBL,
	const float *BBU)
{
	FILE              *file;
	char              fmt[256];
	static const char *indent = "    ";
	static char       *fname = NULL;
	static size_t     fname_len = 0;

	fname = get_list_file_name(fname,dname,gname,&fname_len);

	if ((file = fopen(fname,"w")) == NULL)
	{
	    (void) printf("WARNING in gview_plot_cube(), "
	    	          "can't open %s\n",fname);
	    return;
	}
	(void) sprintf(fmt,"%s%s%s\n",indent,indent,
		                      "%-9g %-9g %-9g "
		                      "%-9g %-9g %-9g "
		                      "%-9g %-9g %-9g "
		                      "%-9g %-9g %-9g");
	(void) fprintf(file,"{ LIST\n");
	gview_bounding_box(file,BBL,BBU,1,indent);
	(void) fprintf(file,"%s{\n",indent);
	(void) fprintf(file,"%s%sQUAD\n",indent,indent);
	/* Lower face, z = L[2] */
	(void) fprintf(file,fmt,L[0],L[1],L[2],U[0],L[1],L[2],
				U[0],U[1],L[2],L[0],U[1],L[2]);
	/* Upper face, z = U[2] */
	(void) fprintf(file,fmt,L[0],L[1],U[2],U[0],L[1],U[2],
	                        U[0],U[1],U[2],L[0],U[1],U[2]);
	/* back face, x = L[0] */
	(void) fprintf(file,fmt,L[0],L[1],L[2],L[0],U[1],L[2],
	                        L[0],U[1],U[2],L[0],L[1],U[2]);
	/* front face, x = U[0] */
	(void) fprintf(file,fmt,U[0],L[1],L[2],U[0],U[1],L[2],
	                        U[0],U[1],U[2],U[0],L[1],U[2]);
	/* left face, y = L[1] */
	(void) fprintf(file,fmt,L[0],L[1],L[2],U[0],L[1],L[2],
	                        U[0],L[1],U[2],L[0],L[1],U[2]);
	/* right face, y = U[1] */
	(void) fprintf(file,fmt,L[0],U[1],L[2],U[0],U[1],L[2],
	                        U[0],U[1],U[2],L[0],U[1],U[2]);
	(void) fprintf(file,"%s}\n",indent);
	(void) fprintf(file,"}\n");

	(void) fclose(file);
}		/*end gview_plot_cube*/

LOCAL   void    gview_plot_curves(
        INTERFACE     *intfc,
	const float   *BBL,
	const float   *BBU,
        const char    *dname,
        const char    *name,
	SURFACE_COLOR color,
	int	      width)
{
        FILE              *file;
        POINT             *ps, *pe;
        CURVE             **c;
        BOND              *b;
        static const char *indent = "    ";
        int               num_bonds,i;
	static char       *fname = NULL;
	static size_t     fname_len = 0;

	fname = get_list_file_name(fname,dname,name,&fname_len);

        if ((file = fopen(fname,"w")) == NULL)
        {
	    screen("WARNING in gview_plot_curves(), "
		   "can't open %s\n",fname);
	    return;
        }

        (void) fprintf(file,"{ LIST\n");

	gview_bounding_box(file,BBL,BBU,1,indent);

        for (c = intfc->curves; c && *c; ++c)
        {
            num_bonds = (*c)->num_points - 1;
            (void) fprintf(file,"%s{appearance{*linewidth %d}\n"
			   "%s%sVECT\n%s%s%6d %6d %6d\n",
			   indent,width,indent,indent,indent,indent,
			   num_bonds,2*num_bonds,1);
            /* may make for very long lines! */
            (void) fprintf(file,"%s%s",indent,indent);
            for (i = 0; i < num_bonds; ++i)
            {
                (void) fprintf(file,"2 ");
            }
            (void) fprintf(file,"\n");
            (void) fprintf(file,"%s%s1 ",indent,indent);
            for (i = 0; i < num_bonds - 1; ++i)
            {
                (void) fprintf(file,"0 ");
            }
            (void) fprintf(file,"\n");
            for (b = (*c)->first; b; b = b->next)
            {
                ps = b->start;
                pe = b->end;
                (void) fprintf(file,"%s%s%-9g %-9g %-9g %-9g %-9g %-9g\n",
			       indent,indent,Coords(ps)[0], Coords(ps)[1],
			       Coords(ps)[2],Coords(pe)[0],Coords(pe)[1],
			       Coords(pe)[2]);
            }
	    (void) fprintf(file,"%s%s",indent,indent);
	    write_color(file,color,0.75);
            (void) fprintf(file,"%s}\n",indent);
        }
        (void) fprintf(file,"}\n");
        fclose(file);
}               /*end gview_plot_curves*/

EXPORT  void    gview_plot_curve(
        const CURVE   *c,
        const char    *dname,
        const char    *name,
	SURFACE_COLOR color,
	int	      width)
{
        FILE              *file;
        POINT             *ps,*pe;
        BOND              *b;
	BOND_TRI          **btris;
	INTERFACE         *intfc = c->interface;
	TRI	          **tri_list;
	float             *BBL = topological_grid(intfc).GL;
	float             *BBU = topological_grid(intfc).GU;
        int               num_bonds,i,j,tri_cnt = 0;
        static const char *indent = "    ";
	static char       *fname = NULL, *ppfname = NULL;
	static size_t     fname_len = 0, ppfname_len = 0;

	ppfname = set_ppfname(ppfname,name,&ppfname_len);

	if (create_directory(dname,YES) == FUNCTION_FAILED)
	{
	    screen("WARNING in gview_plot_curve(), "
		   "directory %s doesn't exist and can't be created\n",dname);
	    return;
        }

	fname = get_list_file_name(fname,dname,ppfname,&fname_len);

        if ((file = fopen(fname,"w")) == NULL)
        {
	    screen("WARNING in gview_plot_curve(), "
	           "can't open %s\n",fname);
	    return;
        }


        (void) fprintf(file,"{ LIST\n");

	gview_bounding_box(file,BBL,BBU,0,indent);

	num_bonds = c->num_points - 1;
	(void) fprintf(file,"%s{appearance{*linewidth %d}\n"
		       "%s%sVECT\n%s%s%6d %6d %6d\n",
		       indent,width,indent,indent,indent,indent,
		       num_bonds,2*num_bonds,1);

	(void) fprintf(file,"%s%s",indent,indent);
	for (i = 0; i < num_bonds; ++i)
	{
	    (void) fprintf(file,"2 ");
	}
	(void) fprintf(file,"\n");
	(void) fprintf(file,"%s%s1 ",indent,indent);
	for (i = 0; i < num_bonds - 1; ++i)
	{
	    (void) fprintf(file,"0 ");
	}
	(void) fprintf(file,"\n");
	for (b = c->first; b; b = b->next)
	{
	    ps = b->start;
	    pe = b->end;
	    (void) fprintf(file,"%s%s%-9g %-9g %-9g %-9g %-9g %-9g\n",
			   indent,indent,Coords(ps)[0], Coords(ps)[1],
			   Coords(ps)[2],Coords(pe)[0],Coords(pe)[1],
			   Coords(pe)[2]);
	}
	(void) fprintf(file,"%s",indent);
	write_color(file,color,0.75);
	(void) fprintf(file,"%s}\n",indent);

        (void) fprintf(file,"}\n");

        fclose(file);
	b = c->first;
	for (btris = Btris(b); btris && *btris; ++btris)
	    ++tri_cnt;

	vector(&tri_list,num_bonds,sizeof(TRI*));
	for (i = 0; i < tri_cnt; ++i)
	{
	    for (b = c->first, j = 0; b; b = b->next, ++j)
	        tri_list[j] = Btris(b)[i]->tri;
	    gview_plot_tri_list(dname,tri_list,num_bonds);
	}
	free(tri_list);
}               /*end gview_plot_curve*/

EXPORT  void    gview_polyline(
	const char    *dname,
        const char    *fname,
	float *const  *coords,
	int	      n_points,
	SURFACE_COLOR color,
	int	      width)
{
        FILE *file;
	char path[256];
	static char *ppfname = NULL;
	static size_t ppfname_len = 0;

	ppfname = set_ppfname(ppfname,fname,&ppfname_len);

	if (create_directory(dname,YES) == FUNCTION_FAILED)
	{
	    screen("WARNING in gview_polyline(), "
		   "directory %s doesn't exist and can't be created\n",dname);
	    return;
        }
	(void) sprintf(path,"%s/%s.vect",dname,fname);
        if ((file = fopen(path,"w")) == NULL)
        {
	    screen("WARNING in gview_polyline(), "
		   "can't open %s\n",path);
	    return;
        }

	print_polyline_description(file,"",coords,n_points,color,0.75,width);
        fclose(file);
}               /*end gview_polyline*/

LOCAL  void    print_polyline_description(
	FILE          *file,
	const char    *indent,
	float         *const *coords,
	int	      n_points,
	SURFACE_COLOR color,
	float         alpha,
	int	      width)
{
        int  i;

	if ((coords == NULL) || (n_points <= 0))
	    return;
        (void) fprintf(file,"%s{ appearance{*linewidth %d}\n",indent,width);
	(void) fprintf(file,"%s  VECT\n",indent);
	(void) fprintf(file,"%s    %d %d 1\n",indent,n_points-1,2*(n_points-1));

	(void) fprintf(file,"%s    ",indent);
        for (i = 0; i < n_points-1; ++i)
	    (void) fprintf(file,"2 ");
	(void) fprintf(file,"\n%s",indent);
	(void) fprintf(file,"    1 ");
        for (i = 0; i < n_points-2; ++i)
	    (void) fprintf(file,"0 ");

	(void) fprintf(file,"\n");
	for (i = 0; i < n_points-1; ++i)
	{
	    (void) fprintf(file,"%s    %13.8g %13.8g %13.8g"
	                        "    %13.8g %13.8g %13.8g\n",
			   indent,coords[i][0],coords[i][1],coords[i][2],
			   coords[i+1][0],coords[i+1][1],coords[i+1][2]);
	}
	
	(void) fprintf(file,"%s",indent);
	write_color(file,color,alpha);
	(void) fprintf(file,"%s}\n",indent);
}		/*end print_polyline_description*/

/*
*  			gview_local_surface()
*
*	draws the surface only within a ball with specified 
*	radius and center.  Warning, this function is not 
*	fully implemented: it only draws tris with at least one
*	point within the ball.  It is possible for a large 
*	triangle to contain points within the ball but not be drawn.
*/

EXPORT	void	gview_local_surface(
	SURFACE	      *surf,
	const char    *dname,
	const char    *fname,
	SURFACE_COLOR color,
	const float   *center,
	float         radius)
{
	FILE       *file;
	POINT	   *p;
	TRI	   *tri;
	static const char *indent = "    ";
	char   	   path[256];
	int	   num_tris, i, n_tris_written = 0;
	int	   *tri_near_point, tri_cnt;

	if (create_directory(dname,YES) == FUNCTION_FAILED)
	{
	    screen("WARNING in gview_local_surface(),"
		    "directory %s doesn't exist and can't be created\n",dname);
	    return;
	}

	(void) sprintf(path,"%s/%s.off",dname,fname);

	if ((file = fopen(path,"w")) == NULL)
	{
	    screen("WARNING in gview_local_surface(),"
		   "can't open %s\n",path);
	    return;
	}

	num_tris = surf->num_tri;
	vector(&tri_near_point,num_tris,sizeof(int));
	for (i = 0; i < num_tris; ++i)
	    tri_near_point[i] = NO;

	tri_cnt = 0;
	for (tri = first_tri(surf); ! at_end_of_tri_list(tri,surf);
	     tri = tri->next)
	{
	    for (i = 0; i < 3; ++i)
	    {
		p = Point_of_tri(tri)[i];
		if (distance_between_positions(Coords(p),center,3) <= radius)
	        {
		    tri_near_point[tri_cnt] = YES;
		    ++n_tris_written;
		    break;
		}
	    }
	    ++tri_cnt;
	}

	(void) fprintf(file,"%s{\n%s%sOFF\n%s%s%6d %6d %6d\n",
			indent,indent,indent,indent,indent,
		       3*n_tris_written,n_tris_written,0);

	tri_cnt = 0;
	for (tri = first_tri(surf); ! at_end_of_tri_list(tri,surf);
	     tri = tri->next)
        {
	    if (tri_near_point[tri_cnt] == YES)
	    {
	        for (i = 0; i < 3; ++i)
		{
		    p = Point_of_tri(tri)[i];
		    (void) fprintf(file,"%s%s%12.8f %12.8f %12.8f\n",
				   indent,indent,
				   Coords(p)[0],Coords(p)[1],Coords(p)[2]);
		}
	    }
	    ++tri_cnt;
	}

	tri_cnt = i = 0;
	for (tri = first_tri(surf); ! at_end_of_tri_list(tri,surf);
	     tri = tri->next)
        {
	    if (tri_near_point[tri_cnt] == YES)
	    {
		(void) fprintf(file,"%s%s%-4d %-4d %-4d %-4d ",
				indent,indent,3,i,i+1,i+2);
		write_color(file,color,0.75);
		i += 3;
	    }
	    ++tri_cnt;
	}
	(void) fprintf(file,"%s}\n",indent);
	(void) fclose(file);
}		/* end gview_local_surface */

LOCAL 	void 	write_color( 		/* for geomview files */
	FILE 	      *file,
	SURFACE_COLOR color,
	float         alpha)
{
  	switch (color)  /* color information  R G B opacity */
	{
	default:
	case pBLACK:
	    (void) fprintf(file,"0.0 0.0 0.0 %g\n",alpha);
	    break;
	case pRED:
	    (void) fprintf(file,"1.0 0.0 0.0 %g\n",alpha); 
	    break;
	case pGREEN:
	    (void) fprintf(file,"0.0 1.0 0.0 %g\n",alpha);
	    break;
	case pYELLOW:
	    (void) fprintf(file,"1.0 1.0 0.0 %g\n",alpha);
	    break;
	case pBLUE:
	    (void) fprintf(file,"0.0 0.0 1.0 %g\n",alpha);
	    break;
	case pMAGENTA:
	    (void) fprintf(file,"1.0 0.0 1.0 %g\n",alpha);
	    break;
	case pCYAN:
	    (void) fprintf(file,"0.0 1.0 1.0 %g\n",alpha);
	    break;
	case pWHITE:
	    (void) fprintf(file,"1.0 1.0 1.0 %g\n",alpha);
	    break;
	}
}  		/*end write_color*/

EXPORT	void	gview_surface(
	SURFACE	      *surface,
	const char    *fname,
	SURFACE_COLOR color)
{
	FILE 	   *file;
	POINT	   *p;
	TRI	   *tri;
	static const char *indent = "    ";
	char	   path[256];
	int	   num_tris, i;
	int	   tri_count = 0;

	if (create_directory("gv",YES) == FUNCTION_FAILED)
	{
	    screen("WARNING in gview_surface(), "
		   "can't open directory gv\n");
	    return;
	}

	if (create_directory("gv/surface",YES) == FUNCTION_FAILED)
	{
	    screen("WARNING in gview_surface(), "
		   "can't open directory gv/surface\n");
	    return;
	}

	(void) sprintf(path,"gv/surface/%s.off",fname);

	if ((file = fopen(path,"w")) == NULL)
	{
	    screen("WARNING in gview_surface(), "
		   "can't open file %s\n",path);
	    return;
	}

	num_tris = surface->num_tri;
	(void) fprintf(file,"%s{\n%s%sOFF\n%s%s%6d %6d %6d\n",indent,
		indent,indent,indent,indent,
		3*num_tris,num_tris,0);
	for (tri = first_tri(surface); !at_end_of_tri_list(tri,surface); 
	     tri = tri->next)
	{
	    ++tri_count;
	    for (i = 0; i < 3; ++i)
	    {
		p = Point_of_tri(tri)[i];
		(void) fprintf(file,"%s%s%-9g %-9g %-9g\n",
			       indent,indent,
			       Coords(p)[0],Coords(p)[1],Coords(p)[2]);
	    }
	}

	i = 0;
	for (tri = first_tri(surface); !at_end_of_tri_list(tri,surface); 
	     tri = tri->next)
	{
	    (void) fprintf(file,"%s%s%-4d %-4d %-4d %-4d ",
			   indent,indent,
			   3,i,i+1,i+2);
	    write_color(file,color,0.75);
	    i += 3;
	}
	(void) fprintf(file,"%s}\n",indent);
	(void) fclose(file);
}		/*end gview_surface*/


EXPORT 	void 	gview_cube(
	FILE        *file,
	const float *L,
	const float *U)
{
	/* Lower face, z = L[2] */
	(void) fprintf(file,"%-9g %-9g %-9g ", L[0],L[1],L[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", U[0],L[1],L[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", U[0],U[1],L[2]);
	(void) fprintf(file,"%-9g %-9g %-9g\n",L[0],U[1],L[2]);
	/* Upper face, z = U[2] */
	(void) fprintf(file,"%-9g %-9g %-9g ", L[0],L[1],U[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", U[0],L[1],U[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", U[0],U[1],U[2]);
	(void) fprintf(file,"%-9g %-9g %-9g\n",L[0],U[1],U[2]);
	/* back face, x = L[0] */
	(void) fprintf(file,"%-9g %-9g %-9g ", L[0],L[1],L[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", L[0],U[1],L[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", L[0],U[1],U[2]);
	(void) fprintf(file,"%-9g %-9g %-9g\n",L[0],L[1],U[2]);
	/* front face, x = U[0] */
	(void) fprintf(file,"%-9g %-9g %-9g ", U[0],L[1],L[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", U[0],U[1],L[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", U[0],U[1],U[2]);
	(void) fprintf(file,"%-9g %-9g %-9g\n",U[0],L[1],U[2]);
	/* left face, y = L[1] */
	(void) fprintf(file,"%-9g %-9g %-9g ", L[0],L[1],L[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", U[0],L[1],L[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", U[0],L[1],U[2]);
	(void) fprintf(file,"%-9g %-9g %-9g\n",L[0],L[1],U[2]);
	/* right face, y = U[1] */
	(void) fprintf(file,"%-9g %-9g %-9g ", L[0],U[1],L[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", U[0],U[1],L[2]);
	(void) fprintf(file,"%-9g %-9g %-9g ", U[0],U[1],U[2]);
	(void) fprintf(file,"%-9g %-9g %-9g\n",L[0],U[1],U[2]);
}		/*end gview_cube*/

EXPORT void gview_plot_tri_list(
	const char *dname,
	TRI        **tris,
	int        num_tris)
{
	static const char *indent = "    ";
	int        i,j;
	char       fname[256];
	FILE       *file;
	POINT      *p;

	if (create_directory(dname,YES) == FUNCTION_FAILED)
	{
	    screen("WARNING in gview_plot_tri_list(), "
	           "directory %s doesn't exist and can't be created\n",dname);
	    return;
	}
	(void) sprintf(fname,"%s/tri.list",dname);
	if ((file = fopen(fname,"w")) == NULL)
	{
	    screen("WARNING in gview_plot_tri_list(), "
	           "can't open %s\n",fname);
	    return;
	}
	(void) fprintf(file,"{ LIST\n");
	(void) fprintf(file,"%s{\n%s%sOFF\n%s%s%6d %6d %6d\n",
			indent,indent,indent,
			indent,indent,3*num_tris,num_tris,0);
	for (i = 0; i < num_tris; ++i)
	{
	    for (j = 0; j < 3; ++j)
	    {
		p = Point_of_tri(tris[i])[j];
		(void) fprintf(file, "%s%s%-9g %-9g %-9g\n",
			indent,indent,
			Coords(p)[0],Coords(p)[1],Coords(p)[2]);
	    }
	}
	for (i = 0; i < num_tris; ++i)
	{
	    (void) fprintf(file,"%s%s%-4d %-4d %-4d %-4d\n",indent,indent,
		3,3*i,3*i+1,3*i+2);
	}
	(void) fprintf(file,"%s}\n",indent);
	(void) fprintf(file,"}\n");
	(void) fclose(file);
}               /*end gview_plot_tri_list*/

EXPORT void gview_plot_tri_and_point_list(
	const char    *dname,
	TRI           **tris,
	const float   *color_tri,
	int           num_tris,
	float *const  *coords,
	SURFACE_COLOR lcolor,
	float         alpha,
	int           width,
	int           npts)
{
	static const char *indent = "    ";
	int               i,j;
	char              fname[256];
	FILE              *file;
	POINT             *p;

	if (create_directory(dname,YES) == FUNCTION_FAILED)
	{
	    screen("WARNING in gview_plot_tri_and_point_list(), "
	           "directory %s doesn't exist and can't be created\n",dname);
	    return;
	}
	(void) sprintf(fname,"%s/tri.list",dname);
	if ((file = fopen(fname,"w")) == NULL)
	{
	    screen("WARNING in gview_plot_tri_and_point_list(), "
	           "can't open %s\n",fname);
	    return;
	}
	(void) fprintf(file,"{ LIST\n");
	(void) fprintf(file,"%s{\n%s%sOFF\n%s%s%6d %6d %6d\n",
			indent,indent,indent,
			indent,indent,3*num_tris,num_tris,0);
	for (i = 0; i < num_tris; ++i)
	{
	    for (j = 0; j < 3; ++j)
	    {
		p = Point_of_tri(tris[i])[j];
		(void) fprintf(file, "%s%s%-9g %-9g %-9g\n",
			indent,indent,
			Coords(p)[0],Coords(p)[1],Coords(p)[2]);
	    }
	}
	for (i = 0; i < num_tris; ++i)
	{
	    (void) fprintf(file,"%s%s%-4d %-4d %-4d %-4d",indent,indent,
		           3,3*i,3*i+1,3*i+2);
	    if (color_tri)
	        (void) fprintf(file," %g %g %g %g",color_tri[4*i],
		                                   color_tri[4*i+1],
		                                   color_tri[4*i+2],
		                                   color_tri[4*i+3]);
	    (void) fprintf(file,"\n");
	}
	(void) fprintf(file,"%s}\n",indent);

	print_polyline_description(file,indent,coords,npts,lcolor,alpha,width);
	(void) fprintf(file,"}\n");
	(void) fclose(file);
}               /*end gview_plot_tri_and_point_list*/

EXPORT void gview_plot_c_curve(
	const C_CURVE *c_curve,
	int	      k,
	const char    *dname)
{
	C_BOND            *cb;
	FILE	          *file;
	POINT	          *p;
	TRI               *tri;
	char	          fname[256];
	const char        *color;
	float             *BBL = topological_grid(c_curve->interface).GL;
	float             *BBU = topological_grid(c_curve->interface).GU;
	int	          isurf,j, index, num_c_bonds;
	static const char *indent = "    ";
	static const char red[10] = " 200 0 0", green[10] = " 0 200 0";
	
	
	if (create_directory(dname,YES) == FUNCTION_FAILED)
	{
	    screen("WARNING in gview_plot_c_curve(), "
	           "directory %s doesn't exist and can't be created\n",dname);
	    return;
	}
	(void) sprintf(fname,"%s/curve%d.list",dname,k);
	if ((file = fopen(fname,"w")) == NULL)
	{
	    screen("WARNING in gview_plot_c_curve(), "
	           "can't open %s\n",fname);
	    return;
	}

	num_c_bonds=0;
	for (cb = c_curve->first; cb != NULL; cb = cb->next) 
	    ++num_c_bonds;

	(void) fprintf(file,"{ LIST\n");

	gview_bounding_box(file,BBL,BBU,1,indent);

	(void) fprintf(file,"%s{\n%s%sOFF\n%s%s%6d %6d %6d\n",
		       indent,indent,indent,
		       indent,indent,6*num_c_bonds,2*num_c_bonds,0);
	
	
	for (cb=c_curve->first; cb != NULL; cb=cb->next)
	{
	    for (isurf=0; isurf<2; ++isurf)
	    {
		tri = cb->s[isurf].t; 
	      
		for (j = 0; j < 3; ++j)
		{
		    p = Point_of_tri(tri)[j];
		    (void) fprintf(file, "%s%s%-9g %-9g %-9g\n",indent,indent,
				   Coords(p)[0],Coords(p)[1],Coords(p)[2]);
		}
	    }
	}
	   
	for (j=0; j < num_c_bonds; ++j)
	{
	    for (isurf=0; isurf<2; ++isurf)
	    {
		index = (6*j)+(3*isurf);
		color = (isurf)? red : green ;
	      
		(void) fprintf(file,"%s%s%-4d %-4d %-4d %-4d%s\n",indent,
			       indent,3,index,index+1,index+2,color);
	    }
	}
	
	(void) fprintf(file,"%s}\n",indent);
	(void) fprintf(file,"}\n");
	(void) fclose(file);
}		/*end gview_plot_c_curve*/

#endif /* if defined(THREED) */

EXPORT	char *set_ppfname(
	char       *ppfname,
	const char *fname,
	size_t     *ppfname_len)
{
	size_t len;

        if (pp_numnodes() > 1 )
	{
	    const char   *nd;

	    nd = right_flush(pp_mynode(),4);
	    if (fname == NULL)
	        fname = "";
	    len = strlen(fname)+1+strlen(nd)+1;
	    if (*ppfname_len < len)
	    {
	        *ppfname_len = len;
	        if (ppfname != NULL)
		    free(ppfname);
	        vector(&ppfname,*ppfname_len,CHAR);
	    }
	    (void) sprintf(ppfname,"%s.%s",fname,nd);
	}
	else
	{
	    if (fname == NULL)
	        fname = "";
	    len = strlen(fname)+1;
	    if (*ppfname_len < len)
	    {
	        *ppfname_len = len;
	        if (ppfname != NULL)
		    free(ppfname);
	        vector(&ppfname,*ppfname_len,CHAR);
	    }
	    (void) strcpy(ppfname,fname);
	}
	return ppfname;
}		/*end set_ppfname*/


EXPORT   char   *get_list_file_name(
        char       *fname,
        const char *dname,
        const char *name,
        size_t     *fname_len)
{
        size_t len;
        if (dname == NULL)
            dname = "";
        len = strlen(dname)+1+strlen(name)+6;
        if (*fname_len < len)
        {
            *fname_len = len;
            if (fname != NULL)
                free(fname);
            vector(&fname,*fname_len,CHAR);
        }
        if (strlen(dname) != 0)
            (void) sprintf(fname,"%s/%s.list",dname,name);
        else
            (void) sprintf(fname,"%s.list",name);
        return fname;
}               /*end get_list_file_name*/


#endif /* defined(TWOD) || defined(THREED) */
