/*
*				iblkc2.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Containing function of rebuilding interface within a mesh block.
*
*/

#if defined(THREED)

#include <intfc/int.h>

#define DEBUG_STRING "blk_intfc"

LOCAL	BBI_POINT *crx_in_idir(const BLK_CRX*,int,int);
LOCAL	BBI_POINT *crx_in_jdir(const BLK_CRX*,int,int);
LOCAL	BBI_POINT *crx_in_kdir(const BLK_CRX*,int,int);
LOCAL	BBI_POINT *crxing_in_between(const int*,const int*,const BLK_CRX*);
LOCAL   int compare_comp(COMPONENT***,COMPONENT****,int);
LOCAL	ORIENTATION curve_surface_orientation(SURFACE*,CURVE*);
LOCAL	void copy_blk_crx(const BLK_CRX*,BLK_CRX*);
LOCAL	void remove_null_pair_of_surface(BLK_TRI*,BLK_TRI*,int,int,int);
LOCAL	void set_prime_components(COMPONENT****);
LOCAL   void rot24(BLK_CRX*, int);
LOCAL   void x_rotation(BLK_CRX*);
LOCAL   void y_rotation(BLK_CRX*);
LOCAL   void z_rotation(BLK_CRX*);
LOCAL   void blk_case01_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case02_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case03_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case04_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case05_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case06_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case07_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case08_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case09_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case10_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case11_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case12_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case13_comp2(BLK_CRX*, BLK_TRI*);
LOCAL   void blk_case14_comp2(BLK_CRX*, BLK_TRI*);


EXPORT	int construct_comp2_blk(
	BLK_CRX *blk_crx,
	BLK_TRI *blk_mem)
{
	int       i,j,k;
	COMPONENT ***comp = blk_crx->comp;
	int       num_crx, case_found;
	BBI_POINT *crxs[12];
	static COMPONENT ****prime_comp;
	static BLK_CRX *bc_rot;
	void (*blk_intfc_comp2[14])(BLK_CRX*, BLK_TRI*) =
	{
	    blk_case01_comp2,
	    blk_case02_comp2,
	    blk_case03_comp2,
	    blk_case04_comp2,
	    blk_case05_comp2,
	    blk_case06_comp2,
	    blk_case07_comp2,
	    blk_case08_comp2,
	    blk_case09_comp2,
	    blk_case10_comp2,
	    blk_case11_comp2,
	    blk_case12_comp2,
	    blk_case13_comp2,
	    blk_case14_comp2,
	};
	if (prime_comp == NULL)
	{
	    quad_array(&prime_comp,14,2,2,2,sizeof(COMPONENT));
	    set_prime_components(prime_comp);
	}
	blk_mem->num_surfaces = 1;
	blk_mem->num_curves = 0;
	num_crx = 0;
	for (j = 0; j < 2; ++j)
	{
	    for (k = 0; k < 2; ++k)
	    {
	        if (comp[0][j][k] != comp[1][j][k])
	        {
	            crxs[num_crx] = crx_in_idir(blk_crx,j,k);
	            if (crxs[num_crx] == NULL)
		    {
	                return FUNCTION_FAILED;
		    }
	            ++num_crx;
	        }
	    }
	}
	for (k = 0; k < 2; ++k)
	{
	    for (i = 0; i < 2; ++i)
	    {
	        if (comp[i][0][k] != comp[i][1][k])
	        {
	            crxs[num_crx] = crx_in_jdir(blk_crx,k,i);
	            if (crxs[num_crx] == NULL)
		    {
	                return FUNCTION_FAILED;
		    }
	            ++num_crx;
	        }
	    }
	}
	for (i = 0; i < 2; ++i)
	{
	    for (j = 0; j < 2; ++j)
	    {
	        if (comp[i][j][0] != comp[i][j][1])
	        {
	            crxs[num_crx] = crx_in_kdir(blk_crx,i,j);
	            if (crxs[num_crx] == NULL)
		    {
	                return FUNCTION_FAILED;
		    }
	            ++num_crx;
	        }
	    }
	}
	if (num_crx == 0)
	{
	        /* No interface, but ONFRONT, this happens */
	        blk_mem->num_tris[0] = 0;
	        return FUNCTION_SUCCEEDED;
	}
	for (i = 1; i < num_crx; ++i)
	{
	    if (crxs[i]->s != crxs[0]->s)
	    {
	        screen("ERROR in reconstruct_blk_intfc(), more than "
	               "one surface in a block, code needed\n");
		(void) printf("crx[%d]->hs = %p  crx[0] = %p\n",
		              i,crxs[i]->s,crxs[0]->s);
		clean_up(ERROR);
	    }
	}
	if (blk_crx->nv[0] > 4)
	{
	    screen("ERROR: in construct_comp2_blk(), no such case!\n");
	    clean_up(ERROR);
	}
	blk_mem->first[0] = NULL;
	blk_mem->num_tris[0] = 0;
	blk_mem->surfs[0] = crxs[0]->s;
	if (bc_rot == NULL) bc_rot = alloc_blk_crx(NO);
	copy_blk_crx(blk_crx, bc_rot);
	
	/* begin 24-rotation */
	
	case_found = NO;
	for (j = 0; j <= 24; j++)
	{
	    for (i = 0; i < 14; i++)
	    {
	        if (compare_comp(bc_rot->comp, prime_comp, i)) 
	        { 
	            blk_intfc_comp2[i](bc_rot, blk_mem);
		    case_found = YES;
	            break;
	        }
	    }
	    if (case_found == YES) break;
	    rot24(bc_rot, j);
	}
	if (case_found == NO)
	{
	    for (i = 0; i < 2; i++)
	    for (j = 0; j < 2; j++) 
	    for (k = 0; k < 2; k++)
	    {
	        (void)printf("ERROR: No case is found\ncomp[%d][%d][%d] = %d\n", 
			      i, j, k, comp[i][j][k]);
	        clean_up(ERROR);
	    }
	}

	blk_mem->num_null_sides[0] = 3*blk_mem->num_tris[0];

	if (debugging("print_blk_tri"))
	{
	    (void) printf("debugging(print_blk_tri), "
			  "printing blk_mem BEFORE stitching inside.\n" );
	    print_blk_tri(blk_mem);
	}
	stitch_inside_blk(blk_mem);
	if (debugging("print_blk_tri"))
	{
	    (void) printf("debugging(print_blk_tri), "
			  "printing blk_mem AFTER stitching inside.\n" );
	    print_blk_tri(blk_mem);
	}

	return FUNCTION_SUCCEEDED;
}	/* end reconstruct_blk_intfc */

EXPORT	void stitch_inside_blk(
	BLK_TRI *bm)
{
	const int *num_tris = bm->num_tris;
	int       is, ic, i, j, side1, side2;
	TRI       *tri1, *tri2;
	POINT     **p1, **p2;
	BOND      *b;
	SURFACE   *s;
	CURVE     *c;
	POINT 	  *ptmp;
	ORIENTATION orient;

	if (bm->num_curves != 0)
	{
	    for (ic = 0; ic < bm->num_curves; ++ic)
	    {
		b = bm->bonds[ic];
		c = bm->curves[ic];
		if (b->start == c->start->posn)
		    c->first = b;
		else if (b->end == c->start->posn)
		{
		    ptmp = b->start;
		    b->start = b->end;
		    b->end = ptmp;
		    c->first = b;
		}
	    	for (is = 0; is < bm->num_surfaces; ++is)
		{
		    s = bm->surfs[is];
		    orient = curve_surface_orientation(s,c);
	    	    for (i = 0, tri1 = bm->first[is]; i < num_tris[is]; 
	    			++i, tri1 = tri1->next)
	    	    {
	        	p1 = Point_of_tri(tri1);
	            	for (side1 = 0; side1 < 3; ++side1)
	            	{
			    if (Neighbor_on_side(tri1,side1) != NULL)
			    	continue;
			    if (b->start == p1[side1] &&
			    	b->end == p1[Next_m3(side1)])
			    {
				if (orient == NEGATIVE_ORIENTATION)
				{
		    		    ptmp = b->start;
		    		    b->start = b->end;
		    		    b->end = ptmp;
				}
			    	link_tri_to_bond(NULL,tri1,s,b,c);
				bm->num_null_sides[is] -= 1;
			    }
			    else if (b->start == p1[Next_m3(side1)] &&
                                b->end == p1[side1])
			    {
				if (orient == POSITIVE_ORIENTATION)
				{
		    		    ptmp = b->start;
		    		    b->start = b->end;
		    		    b->end = ptmp;
				}
			    	link_tri_to_bond(NULL,tri1,s,b,c);
				bm->num_null_sides[is] -= 1;
			    }
			}
		    }
		}
	    }
	}
	for (is = 0; is < bm->num_surfaces; ++is)
	{
	    for (i = 0, tri1 = bm->first[is]; i < num_tris[is]; 
	    		++i, tri1 = tri1->next)
	    {
	        p1 = Point_of_tri(tri1);
	        for (j = i+1, tri2 = tri1->next; j < num_tris[is]; 
				++j, tri2 = tri2->next)
	        {
	            p2 = Point_of_tri(tri2);
	            for (side1 = 0; side1 < 3; ++side1)
	            {
		        if (Neighbor_on_side(tri1,side1) != NULL)
		    	    continue;
	                for (side2 = 0; side2 < 3; ++side2)
	                {
		    	    if (Neighbor_on_side(tri2,side2) != NULL)
		    	        continue;
	                    if (p1[side1] == p2[Next_m3(side2)] &&
	                        p1[(side1+1)%3] == p2[side2])
	                    {
	                        Tri_on_side(tri1,side1) = tri2;
	                        Tri_on_side(tri2,side2) = tri1;
	                        bm->num_null_sides[is] -= 2;
	                    }
	                }
	            }
	        }
	    }
	}
}	/*end stitch_tris_inside_block*/

EXPORT	void stitch_adj_blk(
	BLK_TRI *bm1,
	BLK_TRI *bm2)
{
	int   n1, n2;
	int   i, j;
	int   ic1,ic2,is1,is2;
	TRI   *tri1, *tri2;
	BOND  *b1,*b2;
	CURVE *c1,*c2;

	if (bm1 == NULL || bm2 == NULL)
	{
	    return;
	}
	for (ic1 = 0; ic1 < bm1->num_curves; ++ic1)
	{
	    b1 = bm1->bonds[ic1];
	    c1 = bm1->curves[ic1];
	    for (ic2 = 0; ic2 < bm2->num_curves; ++ic2)
	    {
	    	b2 = bm2->bonds[ic2];
	    	c2 = bm2->curves[ic2];
		if (c1 != c2) continue;
	    	if (b1->start == b2->start ||
		    b1->start == b2->end ||
		    b1->end == b2->start ||
		    b1->end == b2->end)
		{
		    if (b1->prev == NULL && b1 != c1->first)
		    	b1->prev = b2;
		    else if (b1->next == NULL)
		    	b1->next = b2;
		    else
		    {
		    	screen("ERROR: in stitch_adj_blk(),");
			screen("b1 connection is full!\n");
			clean_up(ERROR);
		    }
		    if (b2->prev == NULL && b2 != c2->first)
		    	b2->prev = b1;
		    else if (b2->next == NULL)
		    	b2->next = b1;
		    else
		    {
		    	screen("ERROR: in stitch_adj_blk(),");
			screen("b1 connection is full!\n");
			clean_up(ERROR);
		    }
		}
	    }
	}
	for (is1 = 0; is1 < bm1->num_surfaces; ++is1)
	{
	    for (is2 = 0; is2 < bm2->num_surfaces; ++is2)
	    {
		if (bm1->surfs[is1] != bm2->surfs[is2])
		    continue;
		n1 = bm1->num_tris[is1];
		n2 = bm2->num_tris[is2];

		for (i = 0, tri1 = bm1->first[is1]; i < n1; 
				++i, tri1 = tri1->next)
		{
	    	    for (j = 0, tri2 = bm2->first[is2]; j < n2; 
		    		++j, tri2 = tri2->next)
	    	    {
		    	if (link_neighbor_tris(tri1,tri2))
			{
	                    bm1->num_null_sides[is1]--;
	                    bm2->num_null_sides[is2]--;
			}
	    	    }
		}
	    }
	}
}	/*end stitch_adj_blk*/

EXPORT	void i_reorder_curve_link_list(
	CURVE *c)
{
	BOND *b1,*b2;

	b1 = c->first;
	c->num_points = 1;
	while (b1->next != NULL)
	{
	    b2 = b1->next;
	    if (b1->end == b2->end)
		reverse_bond(b2);
	    if (b2->prev != b1)
	    {
	    	BOND *btmp = b2->next;
		b2->next = b2->prev;
		b2->prev = btmp;
	    }
	    if (b2->prev != b1)
	    {
	    	screen("ERROR in order_curve_link_list(),");
		screen("b2 is not linked to b1!\n");
		clean_up(ERROR);
	    }
	    b1 = b2;
	    ++c->num_points;
	}
	c->last = b1;
	++c->num_points;
}	/* end order_curve_link_list */

EXPORT	void remove_null_pair(
	BLK_TRI *bm1,
	BLK_TRI *bm2,
	int     dir)
{
	int is1, is2;

	if (bm2 == NULL)
	    return;

	for (is1 = 0; is1 < bm1->num_surfaces; ++is1)
	{
	    for (is2 = 0; is2 < bm2->num_surfaces; ++is2)
	    {
	    	if (bm1->surfs[is1] != bm2->surfs[is2])
		    continue;
	    	remove_null_pair_of_surface(bm1,bm2,dir,is1,is2);
	    }
	}	
}	/* end remove_null_pair*/

LOCAL	void remove_null_pair_of_surface(
        BLK_TRI *bm1,
        BLK_TRI *bm2,
        int     dir,
	int     is1,
	int     is2)
{
	POINT    *pts[4];
	POINT    *p1[3], *p2[3];
	int      it, jt, i, j, sides[4];
	int      search_step;
	TRI      *tri1, *tri2, *tris[4];
	BLK_INFO *blk_info = bm1->blk_info;

	if (bm1->num_null_sides[is1] < 2 || bm2->num_null_sides[is2] < 2) 
	{ 
	    return;
	}

	search_step = 0;
	for (tri1=bm1->first[is1], it=0; it < bm1->num_tris[is1]; 
			tri1=tri1->next, ++it)
	{
	    p1[0] = Point_of_tri(tri1)[0];
	    p1[1] = Point_of_tri(tri1)[1];
	    p1[2] = Point_of_tri(tri1)[2];
	    for (tri2=bm2->first[is2], jt=0; jt<bm2->num_tris[is2]; 
	    			tri2=tri2->next, ++jt)
	    {
	        p2[0] = Point_of_tri(tri2)[0];
	        p2[1] = Point_of_tri(tri2)[1];
	        p2[2] = Point_of_tri(tri2)[2];
	        for (i = 0; i < 3; ++i)
	        {
		    if (Tri_on_side(tri1,i) != NULL || 
		        Coords(p1[i])[dir] != Coords(p1[(i+1)%3])[dir])
			continue;
	            for (j = 0; j < 3; ++j)
	            {
		    	if (Tri_on_side(tri2,j) != NULL || 
		            Coords(p2[j])[dir] != Coords(p2[(j+1)%3])[dir])
			    continue;
	                if (Coords(p1[i])[dir] == Coords(p2[j])[dir])
	                {
	                    if (search_step == 0)
	                    {
	                        if (p1[i] == p2[(j+1)%3])
	                        {
	                            pts[0] = p2[j];
	                            pts[1] = p1[i];
	                            pts[2] = p1[(i+1)%3];
	                            tris[0] = tri2;
	                            tris[1] = tri1;
	                            sides[0] = j;
	                            sides[1] = i;
	                            search_step = 1;
	                        }
	                        else if (p1[(i+1)%3] == p2[j])
	                        {
	                            pts[0] = p1[i];
	                            pts[1] = p2[j];
	                            pts[2] = p2[(j+1)%3];
	                            tris[0] = tri1;
	                            tris[1] = tri2;
	                            sides[0] = i;
	                            sides[1] = j;
	                            search_step = 1;
	                        }
	                    }
	                    else if (search_step == 1)
	                    {
	                        if (p1[i] == p2[(j+1)%3] &&
	                            p1[(i+1)%3] == pts[0] &&
	                            p2[j] == pts[2]) 
	                        {
	                            pts[3] = p1[i];
	                            tris[2] = tri2;
	                            tris[3] = tri1;
	                            sides[2] = j;
	                            sides[3] = i;
	                            search_step = 2;
	                        }
	                        else if (p1[(i+1)%3] == p2[j] &&
	                                 p2[(j+1)%3] == pts[0] &&
	                                 p1[i] == pts[2]) 
	                        {
	                            pts[3] = p2[j];
	                            tris[2] = tri1;
	                            tris[3] = tri2;
	                            sides[2] = i;
	                            sides[3] = j;
	                            search_step = 2;
	                        }
	                    }
	                }
	            }
	        }
	    }
	}
	if (search_step == 2)
	{
	    SURFACE *s = Surface_of_tri(tris[0]);
	    for (i = 0; i < blk_info->num_surfs; ++i)
	    {
	        if (s == blk_info->surfs[i])
		    break;
	    }
	    if (blk_info->cur_tris[i] == last_tri(s))
	    {
	        tri1 = make_tri(pts[2],pts[1],pts[0],NULL,NULL,NULL,0);
		tri1->prev = bm1->first[is1];
		tri1->next = bm1->first[is1]->next;
		if (bm1->first[is1] == last_tri(s))
		{
		    last_tri(s) = tri1;
	    	    blk_info->cur_tris[i] = tri1;
		}
		else
		    bm1->first[is1]->next->prev = tri1;
		bm1->first[is1]->next = tri1;
	    }
	    else
	    {
	        tri1 = blk_info->cur_tris[i]->next;
		if (tri1 != last_tri(s))
		{
		    tri1->next->prev = blk_info->cur_tris[i];
		    blk_info->cur_tris[i]->next = tri1->next;
		}
		else
		    last_tri(s) = blk_info->cur_tris[i];
	        Point_of_tri(tri1)[2] = pts[0];
	        Point_of_tri(tri1)[1] = pts[1];
	        Point_of_tri(tri1)[0] = pts[2];
	        Boundary_tri(tri1) = 0;
		tri1->prev = bm1->first[is1];
		tri1->next = bm1->first[is1]->next;
                if (bm1->first[is1] == last_tri(s))
                    last_tri(s) = tri1;
                else
                    bm1->first[is1]->next->prev = tri1;
                bm1->first[is1]->next = tri1;
	    }
	    tri1->surf = s;
	    ++(bm1->num_tris[is1]);
	    ++(s->num_tri);

	    if (blk_info->cur_tris[i] == last_tri(s))
	    {
		tri2 = make_tri(pts[0],pts[3],pts[2],NULL,NULL,NULL,0);
		tri2->prev = bm1->first[is1];
		tri2->next = bm1->first[is1]->next;
		if (bm1->first[is1] == last_tri(s))
		{
		    last_tri(s) = tri2;
	    	    blk_info->cur_tris[i] = tri2;
		}
		else
		    bm1->first[is1]->next->prev = tri2;
		bm1->first[is1]->next = tri2;
	    }
	    else
	    {
	        tri2 = blk_info->cur_tris[i]->next;
		if (tri2 != last_tri(s))
		{
		    tri2->next->prev = blk_info->cur_tris[i];
		    blk_info->cur_tris[i]->next = tri2->next;
		}
		else
		    last_tri(s) = blk_info->cur_tris[i];
		Point_of_tri(tri2)[2] = pts[2];
                Point_of_tri(tri2)[1] = pts[3];
                Point_of_tri(tri2)[0] = pts[0];
	        Boundary_tri(tri2) = 0;
		tri2->prev = bm1->first[is1];
		tri2->next = bm1->first[is1]->next;
                if (bm1->first[is1] == last_tri(s))
                    last_tri(s) = tri2;
                else
                    bm1->first[is1]->next->prev = tri2;
                bm1->first[is1]->next = tri2;
	    }
	    tri2->surf = s;
	    ++(bm1->num_tris[is1]);
	    ++(s->num_tri);

	    Tri_on_side12(tri1) = tris[0];
	    Tri_on_side01(tri1) = tris[1];
	    Tri_on_side20(tri1) = tri2;
	    Tri_on_side(tris[0],sides[0]) = tri1;
	    Tri_on_side(tris[1],sides[1]) = tri1;

	    Tri_on_side12(tri2) = tris[2];
	    Tri_on_side01(tri2) = tris[3];
	    Tri_on_side20(tri2) = tri1;
	    Tri_on_side(tris[2],sides[2]) = tri2;
	    Tri_on_side(tris[3],sides[3]) = tri2;
	    bm1->num_null_sides[is1] -= 2;
	    bm2->num_null_sides[is2] -= 2;
	    set_normal_of_tri(tri1);
	    set_normal_of_tri(tri2);
	}
}	/* end remove_null_pair_of_surface */

LOCAL	BBI_POINT *crx_in_idir(
	const BLK_CRX *blk_crx,
	int j,
	int k)
{
	int ip[3],ipn[3];
	int ***ix = blk_crx->ix;
	int ***iy = blk_crx->iy;
	int ***iz = blk_crx->iz;
	BBI_POINT *bbi;

	ip[0] = ix[0][j][k];
	ip[1] = iy[0][j][k];
	ip[2] = iz[0][j][k];
	ipn[0] = ix[1][j][k];
	ipn[1] = iy[1][j][k];
	ipn[2] = iz[1][j][k];
	bbi = crxing_in_between(ip,ipn,blk_crx);
	return bbi;
}	/* end crx_in_idir */

LOCAL	BBI_POINT *crx_in_jdir(
	const BLK_CRX *blk_crx,
	int k,
	int i)
{
	int ip[3],ipn[3];
	int ***ix = blk_crx->ix;
	int ***iy = blk_crx->iy;
	int ***iz = blk_crx->iz;
	BBI_POINT *bbi;

	ip[0] = ix[i][0][k];
	ip[1] = iy[i][0][k];
	ip[2] = iz[i][0][k];
	ipn[0] = ix[i][1][k];
	ipn[1] = iy[i][1][k];
	ipn[2] = iz[i][1][k];
	bbi = crxing_in_between(ip,ipn,blk_crx);
        return bbi;
}	/* end crx_in_jdir */

LOCAL	BBI_POINT *crx_in_kdir(
	const BLK_CRX *blk_crx,
	int i,
	int j)
{
	int ip[3],ipn[3];
	int ***ix = blk_crx->ix;
	int ***iy = blk_crx->iy;
	int ***iz = blk_crx->iz;
	BBI_POINT *bbi;

	ip[0] = ix[i][j][0];
	ip[1] = iy[i][j][0];
	ip[2] = iz[i][j][0];
	ipn[0] = ix[i][j][1];
	ipn[1] = iy[i][j][1];
	ipn[2] = iz[i][j][1];
	bbi = crxing_in_between(ip,ipn,blk_crx);
	return bbi; 
}	/* end crx_in_kdir */

LOCAL	BBI_POINT *crxing_in_between(
	const int *ip,
	const int *ipn,
	const BLK_CRX *blk_crx)
{
	if (ip[0] != ipn[0])
	{
	    if (blk_crx->crx[0][ip[1]][ip[2]]->p == NULL)
	    {
		screen("ERROR in crxing_in_between(), "
		       "no crossing point between:\n"
		       "(%d %d %d) and (%d %d %d)\n",
			ip[0],ip[1],ip[2],ipn[0],ipn[1],ipn[2]);
		clean_up(ERROR);
	    }
	    return blk_crx->crx[0][ip[1]][ip[2]];
	}
	else if (ip[1] != ipn[1])
	{
	    if (blk_crx->crx[1][ip[2]][ip[0]]->p == NULL)
	    {
		screen("ERROR in crxing_in_between(), "
		       "no crossing point between:\n"
		       "(%d %d %d) and (%d %d %d)\n",
			ip[0],ip[1],ip[2],ipn[0],ipn[1],ipn[2]);
		clean_up(ERROR);
	    }
	    return blk_crx->crx[1][ip[2]][ip[0]];
	}
	else if (ip[2] != ipn[2])
	{
	    if (blk_crx->crx[2][ip[0]][ip[1]]->p == NULL)
	    {
		screen("ERROR in crxing_in_between(), "
		       "no crossing point between:\n"
		       "(%d %d %d) and (%d %d %d)\n",
			ip[0],ip[1],ip[2],ipn[0],ipn[1],ipn[2]);
		clean_up(ERROR);
	    }
	    return blk_crx->crx[2][ip[0]][ip[1]];
	}
	screen("ERROR in crxing_in_between(), "
	       "Inconsistent values of ip and ipn arrays\n");
	clean_up(ERROR);
	return NULL;
}	/* end crxing_in_between */

EXPORT	BLK_CRX *alloc_blk_crx(
	bool alloc_BBI_POINT)
{
	BLK_CRX *blk_crx;

	scalar(&blk_crx,sizeof(BLK_CRX));
	tri_array(&blk_crx->comp,2,2,2,sizeof(COMPONENT));
	tri_array(&blk_crx->ix,2,2,2,sizeof(int));
	tri_array(&blk_crx->iy,2,2,2,sizeof(int));
	tri_array(&blk_crx->iz,2,2,2,sizeof(int));
	tri_array(&blk_crx->crx,3,2,2,sizeof(BBI_POINT*));
	matrix(&blk_crx->curve_crx,3,2,sizeof(BBI_POINT*));
	if (alloc_BBI_POINT)
	{
	    int i,j,k,num_crx = 0;
	    vector(&blk_crx->crx_store,19,sizeof(BBI_POINT));
	    for (i = 0; i < 3; ++i)
	    for (j = 0; j < 2; ++j)
	    for (k = 0; k < 2; ++k)
	    	blk_crx->crx[i][j][k] = &blk_crx->crx_store[num_crx++];
	    for (i = 0; i < 3; ++i)
	    for (j = 0; j < 2; ++j)
	    	blk_crx->curve_crx[i][j] = &blk_crx->crx_store[num_crx++];
	    blk_crx->node_crx = &blk_crx->crx_store[num_crx++];
	}
	return blk_crx;
}	/*end alloc_blk_crx*/

EXPORT	void  create_triangle(
	BLK_TRI *blk_mem,
	POINT   *p1,
	POINT   *p2,
	POINT   *p3,
	SURFACE *s)
{
	TRI      *tri;
	int      i,is;
	BLK_INFO *blk_info = blk_mem->blk_info;

	for (i = 0; i < blk_info->num_surfs; ++i)
	{
	    if (s == blk_info->surfs[i])
		break;
	}
	for (is = 0; is < blk_mem->num_surfaces; ++is)
	{
	    if (s == blk_mem->surfs[is])
		break;
	}
	if (first_tri(s) == NULL)
	{
	    tri = make_tri(p3,p2,p1,NULL,NULL,NULL,0);
	    first_tri(s) = last_tri(s) = tri;
	}
	else if (blk_info->cur_tris[i] == last_tri(s))
	{
	    tri = make_tri(p3,p2,p1,NULL,NULL,NULL,0);
	    tri->prev = last_tri(s);
	    tri->next = last_tri(s)->next;
	    last_tri(s)->next = tri;
	    last_tri(s) = tri;
	}
	else
	{
	    tri = (blk_info->cur_tris[i] == NULL) ?
		first_tri(s) : blk_info->cur_tris[i]->next;
	    Point_of_tri(tri)[2] = p1;
	    Point_of_tri(tri)[1] = p2;
	    Point_of_tri(tri)[0] = p3;
	    Neighbor_on_side01(tri) = NULL;
	    Neighbor_on_side12(tri) = NULL;
	    Neighbor_on_side20(tri) = NULL;
	    Boundary_tri(tri) = 0;
	}
	tri->surf = s;
	set_normal_of_tri(tri);
	blk_info->cur_tris[i] = tri;
	++(blk_mem->num_tris[is]);
	++(s->num_tri);
	if (blk_mem->first[is] == NULL)
	    blk_mem->first[is] = tri;
}	/*end create_triangle*/

LOCAL	ORIENTATION curve_surface_orientation(
	SURFACE *s,
	CURVE *c)
{
	if (pointer_is_in_array(s,c->pos_surfaces))
	    return POSITIVE_ORIENTATION;
	else if (pointer_is_in_array(s,c->neg_surfaces))
	    return NEGATIVE_ORIENTATION;
	else
	    return ORIENTATION_NOT_SET;
}	/* end curve_surface_orientation */

LOCAL   int compare_comp(
	COMPONENT ***comp,
	COMPONENT ****p_comp,
	int ii)
{
	int i,j,k;
        
	for (i = 0; i < 2; i++)
	for (j = 0; j < 2; j++)
        for (k = 0; k < 2; k++) 
	if (comp[i][j][k] != p_comp[ii][i][j][k])
	{
            return NO; 
	}
	return YES; 
}       /* end compare_comp */

LOCAL	void set_prime_components(
	COMPONENT ****pcomp)
{
	/* Case 1: */

	pcomp[0][0][0][0] = 0;
	pcomp[0][0][0][1] = 0;
	pcomp[0][0][1][0] = 0;
	pcomp[0][0][1][1] = 0;
	pcomp[0][1][0][0] = 1;
	pcomp[0][1][0][1] = 1;
	pcomp[0][1][1][0] = 1;
	pcomp[0][1][1][1] = 1;

	/* Case 2: */

	pcomp[1][0][0][0] = 0;
	pcomp[1][0][0][1] = 0;
	pcomp[1][0][1][0] = 0;
	pcomp[1][0][1][1] = 0;
	pcomp[1][1][0][0] = 0;
	pcomp[1][1][0][1] = 0;
	pcomp[1][1][1][0] = 1;
	pcomp[1][1][1][1] = 1;

	/* Case 3: */

	pcomp[2][0][0][0] = 0;
	pcomp[2][0][0][1] = 0;
	pcomp[2][0][1][0] = 0;
	pcomp[2][0][1][1] = 0;
	pcomp[2][1][0][0] = 0;
	pcomp[2][1][0][1] = 0;
	pcomp[2][1][1][0] = 0;
	pcomp[2][1][1][1] = 1;

	/* Case 4: */

	pcomp[3][0][0][0] = 0;
	pcomp[3][0][0][1] = 0;
	pcomp[3][0][1][0] = 0;
	pcomp[3][0][1][1] = 0;
	pcomp[3][1][0][0] = 0;
	pcomp[3][1][0][1] = 1;
	pcomp[3][1][1][0] = 1;
	pcomp[3][1][1][1] = 1;

	/* Case 5: */

	pcomp[4][0][0][0] = 0;
	pcomp[4][0][0][1] = 0;
	pcomp[4][0][1][0] = 0;
	pcomp[4][0][1][1] = 1;
	pcomp[4][1][0][0] = 0;
	pcomp[4][1][0][1] = 1;
	pcomp[4][1][1][0] = 1;
	pcomp[4][1][1][1] = 1;

	/* Case 6: */

	pcomp[5][0][0][0] = 0;
	pcomp[5][0][0][1] = 0;
	pcomp[5][0][1][0] = 0;
	pcomp[5][0][1][1] = 0;
	pcomp[5][1][0][0] = 0;
	pcomp[5][1][0][1] = 1;
	pcomp[5][1][1][0] = 1;
	pcomp[5][1][1][1] = 0;

	/* Case 7: */

	pcomp[6][0][0][0] = 0;
	pcomp[6][0][0][1] = 0;
	pcomp[6][0][1][0] = 0;
	pcomp[6][0][1][1] = 1;
	pcomp[6][1][0][0] = 1;
	pcomp[6][1][0][1] = 1;
	pcomp[6][1][1][0] = 0;
	pcomp[6][1][1][1] = 1;

	/* Case 8: */

	pcomp[7][0][0][0] = 0;
	pcomp[7][0][0][1] = 0;
	pcomp[7][0][1][0] = 0;
	pcomp[7][0][1][1] = 1;
	pcomp[7][1][0][0] = 1;
	pcomp[7][1][0][1] = 0;
	pcomp[7][1][1][0] = 0;
	pcomp[7][1][1][1] = 1;

	/* Case 9: */

	pcomp[8][0][0][0] = 0;
	pcomp[8][0][0][1] = 0;
	pcomp[8][0][1][0] = 0;
	pcomp[8][0][1][1] = 1;
	pcomp[8][1][0][0] = 1;
	pcomp[8][1][0][1] = 0;
	pcomp[8][1][1][0] = 0;
	pcomp[8][1][1][1] = 0;

	/* Case 10: */

	pcomp[9][0][0][0] = 0;
	pcomp[9][0][0][1] = 0;
	pcomp[9][0][1][0] = 0;
	pcomp[9][0][1][1] = 1;
	pcomp[9][1][0][0] = 1;
	pcomp[9][1][0][1] = 0;
	pcomp[9][1][1][0] = 1;
	pcomp[9][1][1][1] = 1;

	/* Case 11: */

	pcomp[10][0][0][0] = 0;
	pcomp[10][0][0][1] = 0;
	pcomp[10][0][1][0] = 1;
	pcomp[10][0][1][1] = 1;
	pcomp[10][1][0][0] = 1;
	pcomp[10][1][0][1] = 1;
	pcomp[10][1][1][0] = 0;
	pcomp[10][1][1][1] = 0;

	/* Case 12: */

	pcomp[11][0][0][0] = 0;
	pcomp[11][0][0][1] = 0;
	pcomp[11][0][1][0] = 0;
	pcomp[11][0][1][1] = 1;
	pcomp[11][1][0][0] = 0;
	pcomp[11][1][0][1] = 1;
	pcomp[11][1][1][0] = 1;
	pcomp[11][1][1][1] = 0;

	/* Case 13: */

	pcomp[12][0][0][0] = 0;
	pcomp[12][0][0][1] = 0;
	pcomp[12][0][1][0] = 0;
	pcomp[12][0][1][1] = 1;
	pcomp[12][1][0][0] = 1;
	pcomp[12][1][0][1] = 1;
	pcomp[12][1][1][0] = 1;
	pcomp[12][1][1][1] = 0;

	/* Case 14: */

	pcomp[13][0][0][0] = 0;
	pcomp[13][0][0][1] = 1;
	pcomp[13][0][1][0] = 1;
	pcomp[13][0][1][1] = 0;
	pcomp[13][1][0][0] = 1;
	pcomp[13][1][0][1] = 0;
	pcomp[13][1][1][0] = 0;
	pcomp[13][1][1][1] = 1;
}	/* end set_prime_components */

EXPORT	void copy_blk_crx(
	const BLK_CRX *blk_crx1,
	BLK_CRX       *blk_crx2)
{
	int i,j,k;
	
	for (i = 0; i < 3; ++i)
	for (j = 0; j < 2; ++j)
	for (k = 0; k < 2; ++k)
	     blk_crx2->crx = blk_crx1->crx;
	for (i = 0; i < 2; i++)
	for (j = 0; j < 2; j++)
        for (k = 0; k < 2; k++)
	{
	    if (blk_crx1->comp[i][j][k] == blk_crx1->comps[0])
	        blk_crx2->comp[i][j][k] = 1;
	    else
	        blk_crx2->comp[i][j][k] = 0;
	    blk_crx2->ix[i][j][k] = blk_crx1->ix[i][j][k];
	    blk_crx2->iy[i][j][k] = blk_crx1->iy[i][j][k];
	    blk_crx2->iz[i][j][k] = blk_crx1->iz[i][j][k];
	}
	for (i = 0; i < 8; i++) 
	    blk_crx2->comps[i] = blk_crx1->comps[i];
}      /*end copy_blk_crx*/

EXPORT  void rot24(
        BLK_CRX *blk_crx,
	int n)
{       
        if (n < 16)
        {
            x_rotation(blk_crx);
            if ((n+1)%4 == 0)
                y_rotation(blk_crx);
        }
        else if (n == 16)
        {   
            z_rotation(blk_crx); 
            x_rotation(blk_crx); 
        }   
        else if (n == 20)
        {   
	    z_rotation(blk_crx);
	    z_rotation(blk_crx);
	    x_rotation(blk_crx);
	}
	else if (n < 24)
	{
	    x_rotation(blk_crx);
	}
}       /* end rot24 */

LOCAL   void x_rotation(BLK_CRX *blk_crx)
{
        int i, temp;
	int ***ix = blk_crx->ix;
	int ***iy = blk_crx->iy;
	int ***iz = blk_crx->iz;
	COMPONENT ***comp = blk_crx->comp;
	COMPONENT comp_tmp;
	
	for (i = 0; i < 2; i++)
	{ 
	    comp_tmp = comp[i][0][0];
	    comp[i][0][0] = comp[i][1][0];
	    comp[i][1][0] = comp[i][1][1];
	    comp[i][1][1] = comp[i][0][1];
	    comp[i][0][1] = comp_tmp;
		
	    temp = ix[i][0][0];
	    ix[i][0][0] = ix[i][1][0];
	    ix[i][1][0] = ix[i][1][1];
	    ix[i][1][1] = ix[i][0][1]; 
	    ix[i][0][1] = temp;
		
	    temp = iy[i][0][0];
	    iy[i][0][0] = iy[i][1][0];
	    iy[i][1][0] = iy[i][1][1];
	    iy[i][1][1] = iy[i][0][1];    
	    iy[i][0][1] = temp;
		
	    temp = iz[i][0][0];
	    iz[i][0][0] = iz[i][1][0];
	    iz[i][1][0] = iz[i][1][1];
	    iz[i][1][1] = iz[i][0][1];    
	    iz[i][0][1] = temp;
	}
}       /* end x_rotation */

LOCAL   void y_rotation(BLK_CRX *blk_crx)
{
        int j, temp;
	int ***ix = blk_crx->ix;
	int ***iy = blk_crx->iy;
	int ***iz = blk_crx->iz;
	COMPONENT ***comp = blk_crx->comp;
	COMPONENT comp_tmp;

	for (j = 0; j < 2; j++)
	{
            comp_tmp = comp[0][j][0];
	    comp[0][j][0] = comp[0][j][1];
	    comp[0][j][1] = comp[1][j][1];
	    comp[1][j][1] = comp[1][j][0];
	    comp[1][j][0] = comp_tmp;

	    temp = ix[0][j][0];
	    ix[0][j][0] = ix[0][j][1];
	    ix[0][j][1] = ix[1][j][1];
	    ix[1][j][1] = ix[1][j][0];
	    ix[1][j][0] = temp;

	    temp = iy[0][j][0];
	    iy[0][j][0] = iy[0][j][1];
	    iy[0][j][1] = iy[1][j][1];
	    iy[1][j][1] = iy[1][j][0];
	    iy[1][j][0] = temp;

	    temp = iz[0][j][0];
	    iz[0][j][0] = iz[0][j][1];
	    iz[0][j][1] = iz[1][j][1];
	    iz[1][j][1] = iz[1][j][0];
	    iz[1][j][0] = temp;
	}
}       /* z_rotation */

LOCAL   void z_rotation(BLK_CRX *blk_crx)
{
        int k, temp;
	int ***ix = blk_crx->ix;
	int ***iy = blk_crx->iy;
	int ***iz = blk_crx->iz;
	COMPONENT ***comp = blk_crx->comp;
	COMPONENT comp_tmp;

	for (k = 0; k < 2; k++)
	{
	    comp_tmp = comp[0][0][k];
	    comp[0][0][k] = comp[1][0][k];
	    comp[1][0][k] = comp[1][1][k];
	    comp[1][1][k] = comp[0][1][k];
	    comp[0][1][k] = comp_tmp;

	    temp = ix[0][0][k];
	    ix[0][0][k] = ix[1][0][k];
	    ix[1][0][k] = ix[1][1][k];
	    ix[1][1][k] = ix[0][1][k];
	    ix[0][1][k] = temp;

	    temp = iy[0][0][k];
	    iy[0][0][k] = iy[1][0][k];
	    iy[1][0][k] = iy[1][1][k];
	    iy[1][1][k] = iy[0][1][k];
	    iy[0][1][k] = temp;
		
	    temp = iz[0][0][k];
	    iz[0][0][k] = iz[1][0][k];
	    iz[1][0][k] = iz[1][1][k];
	    iz[1][1][k] = iz[0][1][k];
	    iz[0][1][k] = temp;
	}
}       /* end z_rotation */

LOCAL void blk_case01_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3, *p4;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* one plan_float */

        p1 = crx_in_idir(blk_crx,1,0)->p; 
        p2 = crx_in_idir(blk_crx,1,1)->p; 
        p3 = crx_in_idir(blk_crx,0,0)->p; 
	p4 = crx_in_idir(blk_crx,0,1)->p; 

	if (blk_crx->comps[0] != positive_component(s))
	{
	    create_triangle(blk_mem,p1,p2,p3,s);
	    create_triangle(blk_mem,p2,p4,p3,s);
	}
	else
	{
	    create_triangle(blk_mem,p1,p3,p2,s);
	    create_triangle(blk_mem,p2,p3,p4,s);
	}
}       /* end blk_case01_comp2 */

LOCAL void blk_case02_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3, *p4;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* one float_tri */

	p1 = crx_in_idir(blk_crx,1,1)->p; 
        p2 = crx_in_jdir(blk_crx,1,1)->p; 
        p3 = crx_in_idir(blk_crx,1,0)->p;
	p4 = crx_in_jdir(blk_crx,0,1)->p;
	if (blk_crx->comps[0] != positive_component(s))
	{
	    create_triangle(blk_mem,p1,p2,p3,s);
	    create_triangle(blk_mem,p3,p2,p4,s);
	}
	else
	{
	    create_triangle(blk_mem,p1,p3,p2,s);
	    create_triangle(blk_mem,p3,p4,p2,s);
	}
}       /* end blk_case02_comp2 */

LOCAL   void blk_case03_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* one corner_tri */

        p1 = crx_in_idir(blk_crx,1,1)->p; 
        p2 = crx_in_jdir(blk_crx,1,1)->p; 
        p3 = crx_in_kdir(blk_crx,1,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);
}       /* end blk_case03_comp2 */

LOCAL void blk_case04_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3, *p4, *p5;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* one ceiling_corner */
	
        p1 = crx_in_idir(blk_crx,1,0)->p; 
        p2 = crx_in_idir(blk_crx,1,1)->p; 
        p3 = crx_in_jdir(blk_crx,0,1)->p; 
	p4 = crx_in_kdir(blk_crx,1,0)->p; 
	p5 = crx_in_idir(blk_crx,0,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	{
	    create_triangle(blk_mem,p1,p2,p3,s);
	    create_triangle(blk_mem,p3,p2,p4,s);
	    create_triangle(blk_mem,p2,p5,p4,s);
	}
	else
	{
	    create_triangle(blk_mem,p1,p3,p2,s);
	    create_triangle(blk_mem,p3,p4,p2,s);
	    create_triangle(blk_mem,p2,p4,p5,s);
	}
}       /* end blk_case04_comp2 */

LOCAL void blk_case05_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3, *p4, *p5, *p6;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* one corner_star */

        p1 = crx_in_idir(blk_crx,1,0)->p; 
        p2 = crx_in_kdir(blk_crx,1,0)->p;
        p3 = crx_in_jdir(blk_crx,0,1)->p;
	p4 = crx_in_kdir(blk_crx,0,1)->p;
	p5 = crx_in_idir(blk_crx,0,1)->p;
	p6 = crx_in_jdir(blk_crx,1,0)->p;
	if (blk_crx->comps[0] != positive_component(s))
	{
	    create_triangle(blk_mem,p1,p2,p3,s);
	    create_triangle(blk_mem,p1,p4,p2,s);
	    create_triangle(blk_mem,p4,p5,p2,s);
	    create_triangle(blk_mem,p4,p6,p5,s);
	}
	else
	{
	    create_triangle(blk_mem,p1,p3,p2,s);
	    create_triangle(blk_mem,p1,p2,p4,s);
	    create_triangle(blk_mem,p4,p2,p5,s);
	    create_triangle(blk_mem,p4,p5,p6,s);
	}
}       /* end blk_case05_comp2 */

LOCAL void blk_case06_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/*  the first corner_tri  */

        p1 = crx_in_idir(blk_crx,1,0)->p; 
        p2 = crx_in_kdir(blk_crx,1,1)->p; 
        p3 = crx_in_jdir(blk_crx,0,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);

	/*  the second corner_tri  */
        
	p1 = crx_in_jdir(blk_crx,1,1)->p; 
        p2 = crx_in_idir(blk_crx,0,1)->p; 
        p3 = crx_in_kdir(blk_crx,1,0)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);
}       /* end blk_case06_comp2 */


LOCAL void blk_case07_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3, *p4, *p5, *p6;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* one ij_twist */

        p1 = crx_in_kdir(blk_crx,0,1)->p; 
        p2 = crx_in_jdir(blk_crx,1,0)->p;
        p3 = crx_in_kdir(blk_crx,1,1)->p;
	p4 = crx_in_idir(blk_crx,0,1)->p; 
	p5 = crx_in_jdir(blk_crx,0,1)->p; 
	p6 = crx_in_idir(blk_crx,0,0)->p;
	if (blk_crx->comps[0] != positive_component(s))
	{
	    create_triangle(blk_mem,p1,p2,p3,s);
	    create_triangle(blk_mem,p3,p2,p4,s);
	    create_triangle(blk_mem,p3,p4,p5,s);
	    create_triangle(blk_mem,p5,p4,p6,s);
	}
	else
	{
	    create_triangle(blk_mem,p1,p3,p2,s);
	    create_triangle(blk_mem,p3,p4,p2,s);
	    create_triangle(blk_mem,p3,p5,p4,s);
	    create_triangle(blk_mem,p5,p6,p4,s);
	}
}       /* end blk_case07_comp2 */

LOCAL void blk_case08_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3, *p4;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* one cornor_tri */

        p1 = crx_in_jdir(blk_crx,0,1)->p; 
        p2 = crx_in_kdir(blk_crx,1,0)->p; 
        p3 = crx_in_idir(blk_crx,0,0)->p;
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);

	/* one float_tri */

        p1 = crx_in_kdir(blk_crx,0,1)->p; 
        p2 = crx_in_jdir(blk_crx,1,0)->p; 
        p3 = crx_in_kdir(blk_crx,1,1)->p; 
	p4 = crx_in_jdir(blk_crx,1,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	{
	    create_triangle(blk_mem,p1,p2,p3,s);
	    create_triangle(blk_mem,p3,p2,p4,s);
	}
	else
	{
	    create_triangle(blk_mem,p1,p3,p2,s);
	    create_triangle(blk_mem,p3,p4,p2,s);
	}
}       /* end blk_case08_comp2 */

LOCAL void blk_case09_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* the first corner_tri */

        p1 = crx_in_jdir(blk_crx,0,1)->p; 
        p2 = crx_in_kdir(blk_crx,1,0)->p;
        p3 = crx_in_idir(blk_crx,0,0)->p;
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);

	/* the second corner_tri */

        p1 = crx_in_kdir(blk_crx,0,1)->p; 
        p2 = crx_in_jdir(blk_crx,1,0)->p; 
        p3 = crx_in_idir(blk_crx,1,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);
}       /* end blk_case09_comp2 */

LOCAL void blk_case10_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3, *p4, *p5, *p6;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* one ji_twist */

        p1 = crx_in_idir(blk_crx,1,0)->p; 
        p2 = crx_in_kdir(blk_crx,1,0)->p; 
        p3 = crx_in_idir(blk_crx,0,0)->p; 
	p4 = crx_in_jdir(blk_crx,1,1)->p; 
	p5 = crx_in_kdir(blk_crx,0,1)->p; 
	p6 = crx_in_jdir(blk_crx,1,0)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	{
	    create_triangle(blk_mem,p1,p2,p3,s);
	    create_triangle(blk_mem,p1,p4,p2,s);
	    create_triangle(blk_mem,p1,p5,p4,s);
	    create_triangle(blk_mem,p5,p6,p4,s);
	}
	else
	{
	    create_triangle(blk_mem,p1,p3,p2,s);
	    create_triangle(blk_mem,p1,p2,p4,s);
	    create_triangle(blk_mem,p1,p4,p5,s);
	    create_triangle(blk_mem,p5,p4,p6,s);
	}
}       /* end blk_case10_comp2 */


LOCAL void blk_case11_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3, *p4;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* the first float_tri */

        p1 = crx_in_idir(blk_crx,1,0)->p; 
        p2 = crx_in_jdir(blk_crx,0,0)->p; 
        p3 = crx_in_jdir(blk_crx,1,0)->p; 
	p4 = crx_in_idir(blk_crx,1,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	{
	    create_triangle(blk_mem,p1,p2,p3,s);
	    create_triangle(blk_mem,p4,p1,p3,s);
	}
	else
	{
	    create_triangle(blk_mem,p1,p3,p2,s);
	    create_triangle(blk_mem,p4,p3,p1,s);
	}

	/* the second float_tri */

        p1 = crx_in_jdir(blk_crx,0,1)->p; 
        p2 = crx_in_idir(blk_crx,0,1)->p; 
        p3 = crx_in_idir(blk_crx,0,0)->p; 
	p4 = crx_in_jdir(blk_crx,1,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	{
	    create_triangle(blk_mem,p1,p2,p3,s);
	    create_triangle(blk_mem,p1,p4,p2,s);
	}
	else
	{
	    create_triangle(blk_mem,p1,p3,p2,s);
	    create_triangle(blk_mem,p1,p2,p4,s);
	}
}       /* end blk_case11_comp2 */

LOCAL void blk_case12_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem) 
{
        POINT *p1, *p2, *p3;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* the first corner_tri */

        p1 = crx_in_idir(blk_crx,1,0)->p;
        p2 = crx_in_kdir(blk_crx,1,1)->p; 
        p3 = crx_in_jdir(blk_crx,0,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);

	/* the second corner_tri */

        p1 = crx_in_kdir(blk_crx,0,1)->p;
        p2 = crx_in_jdir(blk_crx,1,0)->p; 
        p3 = crx_in_idir(blk_crx,1,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);

	/* the third corner_tri */
	
        p1 = crx_in_jdir(blk_crx,1,1)->p; 
        p2 = crx_in_idir(blk_crx,0,1)->p; 
        p3 = crx_in_kdir(blk_crx,1,0)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);
}       /* end blk_case12_comp2 */

LOCAL void blk_case13_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3, *p4, *p5;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* one ceiling_corner */

        p1 = crx_in_idir(blk_crx,1,0)->p; 
        p2 = crx_in_kdir(blk_crx,1,1)->p;
        p3 = crx_in_idir(blk_crx,0,0)->p; 
	p4 = crx_in_jdir(blk_crx,1,1)->p; 
	p5 = crx_in_idir(blk_crx,0,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	{
	    create_triangle(blk_mem,p1,p2,p3,s);
	    create_triangle(blk_mem,p2,p4,p3,s);
	    create_triangle(blk_mem,p4,p5,p3,s);
	}
	else
	{
	    create_triangle(blk_mem,p1,p3,p2,s);
	    create_triangle(blk_mem,p2,p3,p4,s);
	    create_triangle(blk_mem,p4,p3,p5,s);
	}

	/* one corner_tri */

        p1 = crx_in_kdir(blk_crx,0,1)->p; 
        p2 = crx_in_jdir(blk_crx,1,0)->p; 
        p3 = crx_in_idir(blk_crx,1,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);
}       /* end blk_case13_comp2 */

LOCAL void blk_case14_comp2(
        BLK_CRX *blk_crx,
        BLK_TRI *blk_mem)
{
        POINT *p1, *p2, *p3;
	SURFACE *s;

        s = blk_mem->surfs[0];

	/* the first corner_tri */

        p1 = crx_in_idir(blk_crx,1,0)->p; 
        p2 = crx_in_jdir(blk_crx,0,0)->p; 
        p3 = crx_in_kdir(blk_crx,0,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);

	/* the second corner_tri */

        p1 = crx_in_kdir(blk_crx,1,1)->p; 
        p2 = crx_in_idir(blk_crx,1,1)->p; 
        p3 = crx_in_jdir(blk_crx,1,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);

	/* the third corner_tri */

        p1 = crx_in_jdir(blk_crx,0,1)->p; 
        p2 = crx_in_kdir(blk_crx,1,0)->p; 
        p3 = crx_in_idir(blk_crx,0,0)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);

	/* the fourth corner_tri */

        p1 = crx_in_jdir(blk_crx,1,0)->p; 
        p2 = crx_in_kdir(blk_crx,0,0)->p; 
        p3 = crx_in_idir(blk_crx,0,1)->p; 
	if (blk_crx->comps[0] != positive_component(s))
	    create_triangle(blk_mem,p1,p2,p3,s);
	else
	    create_triangle(blk_mem,p1,p3,p2,s);
}       /* end blk_case14_comp2 */

#endif /* defined(THREED) */
