/*
*                               hunsplitvec.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*       Contains drivers for the hyperbolic library.
*/

#define DEBUG_STRING    "hypunsplit"

#include <hyp/hlocaldecs.h>

#if defined(CONSERVATIVE_ALG) && defined(TWOD)

#define DEBUG 

LOCAL  void      sweep_3l_comp_segments(int,int*,float*,Wave*,Wave*,
                    Front*,Front*,int*,int,int,float,float*,int*);
LOCAL  void      vol_top_and_btm_stat_with_tri(Wave*,Front*,Wave*); 


LIB_LOCAL void hyp_unsplit_reg_vec(
        int        *iperm,
        float      *dh,
        float      dt,
        Wave       *wv,
        Wave       *nwv,
        Front      *fr,
        Front      *nfr,
        COMPONENT  max_comp)
{
        int        nrad = vsten_radius(wv); //nrad = 2
        int        num_pts = wv->npts_sten;  //5 
        int        num_vpts = wv->npts_vsten; // 5 
        int        i1, dim;  
        int        ic[MAXD]; 
        RECT_GRID  *gr = fr->rect_grid;
        Wave       *swap_wv = NULL;   
        float      dir[MAXD];
        int        imin[3], imax[3];
        int        i0min, i0max;
        int        vsize;
        int        icoords[MAXD];
        int        i, k, idirs[MAXD];
        static char     err[] = "ERROR in hyp_unsplit_split_driver()"; 
 
        DEBUG_ENTER(hyp_unsplit_reg_vec)

        dim = gr->dim; 
        for(k = 0; k < dim; k++)
        {
            for (i = 0; i < dim; ++i)
            {
                idirs[i] = iperm[(i+k)%dim];
                dir[i] = 0.0;
            }
            dir[idirs[0]] = 1.0;

            // If not enough, call set_unsplit_sweep_limits()
            set_sweep_limits(wv, k, idirs, imin, imax);  
            if (gr->lbuf[idirs[0]] > 0)
                imin[0] -= nrad;
            if (gr->ubuf[idirs[0]] > 0)
                imax[0] += nrad;
            i0min = imin[0];        i0max = imax[0];

            if(k == 1)
            {
                // Ensure the 2nd swp is in the area made by the first swp.
                if (gr->lbuf[idirs[1]] > 0)
                    imin[1] += 2;  // if call set_unsplit_sweep_limits,
                                   //  should be 1 here. 
                if (gr->ubuf[idirs[1]] > 0)
                    imax[1] -= 2;  // if call set_unsplit_sweep_limits, 
                                   // should be 1 here.
            }
            for (i = 1; i < dim; ++i)
            {
                imin[i] += 1;
                imax[i] -= 1;
            }
            vsize = i0max - i0min;

#if defined(DEBUG)
            /*
            {    
                static const char *var[] = {"ix","iy","iz"};
                int kk;
 
                (void) printf("hyp_unsplit_reg_vec() Sweep limits\n");
                (void) printf("\tdir = %d, ",idirs[0]);
                print_int_vector("idirs = ",idirs,dim," ");
                print_int_vector("iperm = ",iperm,dim," ");
                (void) printf(", swp_num = %d\n",k);
                for (kk = 0; kk < dim; ++kk)
                    (void) printf("\t\t%2d <= %s < %d\n",
                        imin[kk],var[idirs[kk]],imax[kk]);

                for(kk = 0; kk < dim; ++kk)
                    printf("\tGmax[%d] = %d, lbuf[%d], ubuf[%d]\n", 
                       iperm[kk], gr->gmax[iperm[kk]], 
                       gr->lbuf[iperm[kk]], gr->ubuf[iperm[kk]]); 
            }
            */ 
#endif /* if defined(DEBUG) */ 
       
            // g_unsplit_muscl_alloc_phys_vecs;  
            alloc_unsplit_phys_vecs(wv,vsize);

            icoords[idirs[0]] = 0;  /* value not used */
            for(i1 = imin[1]; i1 < imax[1]; i1++)
            {
                icoords[idirs[1]] = i1; 
                sweep_3l_comp_segments(k,iperm,dir,wv,nwv,fr,
                        nfr,icoords,i0min,i0max,dt,dh,idirs);
            }
            // g_unsplit_muscl_free_phys_vecs;  
            free_unsplit_phys_vecs(wv);

            /*** GHOST CELL UPDATE THE IRREGULAR GRID ***/

            hyp_unsplit_npt(k,iperm,dh,dt,wv,nwv,swap_wv,fr,nfr,max_comp);

            if(k == 0)
            {
                swap_wv = copy_wave(nwv);  
                // h_assign_copy_wave_pointers
                assign_wave_parameters(swap_wv,nwv);
                if( !copy_hyp_solution_function(nwv,swap_wv) )
                {
                    screen("%s, copy_hyp_solution_function() failed\n",err);
                    free_wave(swap_wv);
                    DEBUG_LEAVE(hyp_unsplit_reg_vec)
                    clean_up(ERROR); 
                    return;
                }
                if( !copy_hyp_soln_state(nwv, swap_wv))
                {
                    screen("%s, copy_hyp_soln_state() failed\n",err);
                    free_wave(swap_wv);
                    DEBUG_LEAVE(hyp_unsplit_reg_vec)
                    clean_up(ERROR); 
                }
            }
            if(k == 1)
            {
                // h_free_wave
                // h_free_copy_wave_pointers(swap_wv);
                free_copy_wave_pointers(swap_wv);
                free_wave(swap_wv); 
            }
        }  

#if defined(DEBUG)
        /***
        {
            printf("After updating by unspliting FD, grid states\n"); 
            for(i = 0; i < 2; i++)
            {
                ic[0] = 0; ic[1] = 154+i; 
                printf("<%f, %f> <%d, %d>\n", Rect_coords(ic,nwv)[0], 
	           Rect_coords(ic,nwv)[1], ic[0], ic[1]);
                (*nfr->print_state)(Rect_state(ic,nwv));
                ic[0] = -1; ic[1] = 154+i; 
                printf("<%f, %f> <%d, %d>\n", Rect_coords(ic,nwv)[0], 
	           Rect_coords(ic,nwv)[1], ic[0], ic[1]);
                (*nfr->print_state)(Rect_state(ic,nwv));
            }
        }
        ***/
#endif /* if defined(DEBUG) */

        /* CONSERVATIVE FINITE VOLUME UPDATE STEP */

        if(interface_reconstructed(fr->interf) == YES &&
           interface_reconstructed(nfr->interf) == YES)
        {
            vol_top_and_btm_stat_with_tri(wv,fr,nwv);  
            hyp_consv_npt(k,iperm,dh,dt,wv,nwv,fr,nfr,max_comp); 
        }

        /* parallel part for interior states */
        for(k = 0; k < dim; k++)
        {
            if (!scatter_states(nwv,nfr,iperm,k))
            {
                screen("%s, scatter_states() failed\n",err);
                clean_up(ERROR);
            }
        }
 
        if(wave_tri_soln(wv)->tri_grid->alloc.c_seg_crx_count == 1)
        {
            // printf("Step[%d], DOES scatter_frac_cell_states\n", fr->step);
            scatter_frac_cell_states(nwv, nfr);
        }

        DEBUG_LEAVE(hyp_unsplit_reg_vec)
}

/*** Changed the volume state storage method.
 *** Instead of using Comp_blk(ic,ntg->vol_states,ntg)
 *** for every fractional top or buttom state face,
 *** the reg cell state is used if the regular cell center
 *** is contained inside this face. This will
 *** save the putting volume state back to wave step.
 ***/
LOCAL void vol_top_and_btm_stat_with_tri(
        Wave         *wv,
        Front        *fr,
        Wave         *nwv)
{
        TRI_GRID      *otg = wave_tri_soln(wv)->tri_grid;
        TRI_GRID      *ntg = wave_tri_soln(nwv)->tri_grid;
        int              i, j, smin[MAXD], smax[MAXD];
        int              ic[MAXD];
        CSG_BLK_CRX      *blk_crx;
        CSG_Solid        *s;

        DEBUG_ENTER(vol_top_and_btm_stat_with_tri)

        for (i = 0; i < 2; i++)
        {
            smin[i] = 0;
            smax[i] = wv->rect_grid->gmax[i];
            smin[i] -= wv->rect_grid->lbuf[i];
            smax[i] += wv->rect_grid->ubuf[i];
        }


        for(i= smin[0]; i < smax[0]; i++)
            for(j = smin[1]; j < smax[1]; j++)
        {
            ic[0] = i; ic[1] = j;
            blk_crx = Comp_blk(ic,otg->Volume.blk,otg);
            if(blk_crx)
            {
                s = blk_crx->s;
                while(s)
                {
                    top_and_btm_stat_with_tri(ic, s, 
                       ntg, otg, 1, fr->print_state);
                    s = s->nexts;
                }
            }
        }

        DEBUG_LEAVE(vol_top_and_btm_stat_with_tri)
}


/*
LIB_LOCAL void    set_unsplit_sweep_limits(
        Wave            *wv,
        int             swp_num,
        int             *idirs,
        int             *imin,
        int             *imax)
{
        RECT_GRID       *rgr = wv->rect_grid;
        int             *gmax = rgr->gmax;
        int             dim = rgr->dim;
        int             *lbuf = rgr->lbuf;
        int             *ubuf = rgr->ubuf;
        int             i;
        int             nrad = dim - swp_num;

        for (i = 0; i < dim; ++i)
        {
            imin[i] = 0;
            imax[i] = gmax[idirs[i]];
        }
        for (i = dim; i < 3; ++i)
        {
            imin[i] = 0;
            imax[i] = 1;
        }
        for (i = 1; i < dim; ++i)
        {
            if (lbuf[idirs[i]] > 0)
                imin[i] -= lbuf[idirs[i]];
            if (ubuf[idirs[i]] > 0)
                imax[i] += ubuf[idirs[i]];
        }

        if (lbuf[idirs[0]] > 0)
            imin[0] -= nrad;
        if (ubuf[idirs[0]] > 0)
            imax[0] += nrad;


        if (debugging("sweep_limits"))
        {
            static const char *var[] = {"ix","iy","iz"};
            int k;

            (void) printf("\nSweep limits\n");
            (void) printf("\tdir = %d, ",idirs[0]);
            print_int_vector("idirs = ",idirs,dim," ");
            (void) printf(", swp_num = %d\n",swp_num);
            for (k = 0; k < dim; ++k)
                (void) printf("\t\t%2d <= %s < %d\n",
                        imin[k],var[idirs[k]],imax[k]);
            (void) printf("\n");
        }
}  
*/        /*end set_unsplit_sweep_limits*/


LOCAL void sweep_3l_comp_segments(
        int             swp_num,
        int             *iperm,
        float           *dir,
        Wave            *wv,
        Wave            *newwv,
        Front           *fr,
        Front           *newfr,
        int             *icoords,
        int             i0min,
        int             i0max,
        float           dt,
        float           *dh,
        int             *idirs)
{
        COMPONENT       comp;
        int             seg_min,seg_max;
        int             seg_min0,seg_max0;
        int             seg_min1,seg_max1;
        int             icrds1[MAXD], icrds0[MAXD]; 

        seg_min = i0min;
        while (seg_min != i0max)
        {
            icoords[idirs[0]] = seg_min;
            icrds0[idirs[0]] = icrds1[idirs[0]] = seg_min;
            icrds0[idirs[1]] = icoords[idirs[1]] - 1;
            icrds1[idirs[1]] = icoords[idirs[1]] + 1;

            comp = Rect_comp(icoords,wv);
            if(Rect_comp(icoords,wv) != Rect_comp(icrds0,wv) ||
               Rect_comp(icoords,wv) != Rect_comp(icrds1,wv) ||
               Rect_comp(icrds0,wv) != Rect_comp(icrds1,wv))
            {
                seg_min++; 
                continue;
            }

            for (seg_max = seg_min+1; seg_max < i0max; seg_max++)
            {
                icoords[idirs[0]] = seg_max;
                if (comp != Rect_comp(icoords,wv))
                    break;
            }
            
            for (seg_max0 = seg_min+1; seg_max0 < i0max; seg_max0++)
            {
                icrds0[idirs[0]] = seg_max0;
                if (comp != Rect_comp(icrds0,wv))
                    break;
            }

            for (seg_max1 = seg_min+1; seg_max1 < i0max; seg_max1++)
            {
                icrds1[idirs[0]] = seg_max1;
                if (comp != Rect_comp(icrds1,wv))
                    break;
            }
           
            seg_max = min(min(seg_max0, seg_max1), seg_max); 
            icoords[idirs[0]] = seg_max; 

            // ghyp_unsplit_vec
            unsplit_vec_solver(swp_num,iperm,dir,wv,newwv,fr,newfr,
                        icoords,seg_min,seg_max,dt,dh);

            seg_min = seg_max;
        }
}

LIB_LOCAL int copy_hyp_soln_state(
        Wave *owave,
        Wave *nwave)
{
        TRI_GRID  *ngrid;
        TRI_GRID  *ogrid;
        register Locstate   *ostate, *nstate;
        size_t    sizest = owave->sizest; 
        int       *gmax; 
        int       dim = owave->rect_grid->dim; 
        int       xmax, ymax;
        int ix, iy;
        
        ogrid = wave_tri_soln(owave)->tri_grid; 
        ngrid = wave_tri_soln(nwave)->tri_grid; 
        gmax = ngrid->rect_grid.gmax; 

        switch(dim)
        {
        case 2:
            xmax = gmax[0], ymax = gmax[1];
            nstate = ngrid->states; 
            ostate = ogrid->states; 
            for (iy = 0;  iy <= ymax;  ++iy)
            {
                for (ix = 0;  ix <= xmax;  ++ix)
                {
                    assign((*nstate),(*ostate),sizest); 
                    nstate++; ostate++;  
                }
            }
        break;
        default:
           printf("ERROR: implement copy_hyp_soln_state for %D case\n", dim);
           clean_up(ERROR);  
        }
        return YES; 
}

#endif /* if defined(CONSERVATIVE_ALG) && defined(TWOD) */

