/*
*				hscatter.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains function for copying states across distinct processors
*/

#define DEBUG_STRING	"hscatter"
#include <hyp/hlocaldecs.h>

#define state_id(i)     (STATE_ID + (i+1))
LOCAL size_t BLOCK_SIZE = 0; /*TOLERANCE - TODO: what is a good value*/

	/* LOCAL Function Declarations */
LOCAL	POINTER	buffer_storage(size_t,int*,int*,int,POINTER,size_t*,size_t*);
LOCAL	void	copy_states_across_domain(int*,int,int,Front*,Wave*);
LOCAL	void	pp_receive_interior_states(int*,int,int*,int,Front*,Wave*);
LOCAL	void	pp_send_interior_states(int*,int,int*,int,Front*,Wave*);
LOCAL	void	set_receive_domain(int*,int*,int*,int,int,RECT_GRID*);
LOCAL	void	set_send_domain(int*,int*,int*,int,int,RECT_GRID*);

#if defined(CONSERVATIVE_ALG) && defined(TWOD)
LOCAL   void    pp_send_frac_interior_states(int*,int,int*,int,Front*,Wave*); 
LOCAL   void    pp_receive_frac_interior_states(int*,int,int*,int,Front*,Wave*); 
LOCAL   void    reflect_frac_states_across_domain(int*,int,int,Front*,Wave*); 
LOCAL   void    copy_frac_states_across_domain(int*,int,int,Front*,Wave*);
LOCAL   byte    *frac_buffer_storage(size_t,int*,int*,int,size_t*); 
#endif /* if defined(CONSERVATIVE_ALG) && defined(TWOD) */

EXPORT	void	SetHypPPBlockSize(
	size_t	new_BLOCK_SIZE)
{
	BLOCK_SIZE = new_BLOCK_SIZE;
	EnsureSufficientMessageBufferSize(BLOCK_SIZE);
}		/*end SetHypPPBlockSize*/

EXPORT	void	SetDefaultHypPPBlockSize(
	int	dim)
{
	if (BLOCK_SIZE != 0) /*BLOCK_SIZE explicitly set on command line*/
	    return;

	switch (dim)
	{
	case 1:
	case 2:
	    BLOCK_SIZE = 9992;
	    break;
	case 3:
	    BLOCK_SIZE = 1000000;
	    break;
	default:
	    screen("ERROR in SetDefaultHypPPBlockSize(), "
		   "invalid dimension %d\n",dim);
	    clean_up(ERROR);
	}
	EnsureSufficientMessageBufferSize(BLOCK_SIZE);
}		/*end SetDefaultHypPPBlockSize*/

/*
*			h_scatter_states():
*
*	Main control function for parallel communication of rectangular
*	lattice states.  The states in the buffer zone regions are shipped
*	to adjacent subdomains in this function.  This is conducted in
*	dim*2 passes.  As state information in each direction is transmitted,
*	the domain of validity is increased.  This removes the need to
*	transmit corner information.  The basic scheme for two dimensions
*	and iperm = [0 1],  is outlined below.
*
*	gr = front->rect_grid
*	Vl = -gr->lbuf	Vu = gr->gmax + gr->ubuf
*	l = 0,  u = gr->gmax
*
*	i = 0
*
*	                 l[0]+lbuf[0]     u[0]-ubuf[0]
*                            |                |
*			    \|/              \|/
*	  l[1] |------|------------------------------|------|
*	       |      |      |                |      |      |
*	       |  R   |      |                |      |  R   |
*	       |  E   |      |                |      |  E   |
*	       |  C   |  S   |                |  S   |  C   |
*	       |  E   |  E   |                |  E   |  E   |
*	       |  I   |  N   |                |  N   |  I   |
*	       |  V   |  D   |                |  D   |  V   |
*	       |  E   |      |                |      |  E   |
*	       |      |      |                |      |      |
*	       |      |      |                |      |      |
*	       |      |      |                |      |      |
*	   l[1]|------|------------------------------|------|
*         Vl[0]      l[0]                           u[0]  Vu[0]
*
*	i = 1
*
*         Vu[1]----------------------------------------------
*	       |                                            |
*	       |             RECEIVE                        |
*	  l[1] |--------------------------------------------|
*	       |                                            |
*	       |               SEND                         |
*	       |--------------------------------------------|
*	       |                                            |
*	       |                                            |
*	       |                                            |
*	       |                                            |
*	       |                                            |
*	       |                                            |
*	       |--------------------------------------------|
*	       |                                            |
*	       |               SEND                         |
*	   l[1]|--------------------------------------------|
*	       |                                            |
*	       |             RECEIVE                        |
*         Vl[1]----------------------------------------------
*         Vl[0]                                           Vu[0]
*/

EXPORT 	bool	h_scatter_states(
	Wave		*wave,
	Front		*front,
	int             *iperm,
	int		swp)
{
	PP_GRID		*pp_grid = wave->pp_grid;
	int		myid;
	int		me[MAXD];
	int		i,side, dim;

	DEBUG_ENTER(h_scatter_states)

	if (wave->sizest == 0)
	{
	    DEBUG_LEAVE(h_scatter_states)
	    return FUNCTION_SUCCEEDED;
	}

	start_clock("h_scatter_states");

	myid = pp_mynode();
	dim = wave->rect_grid->dim;

	find_Cartesian_coordinates(myid,pp_grid,me);

	for (side = 0; side < 2; ++side)
	{
	    pp_gsync();
	    pp_send_interior_states(me,swp,iperm,side,front,wave);
	    pp_receive_interior_states(me,swp,iperm,(side+1)%2,front,wave);
	}

	stop_clock("h_scatter_states");
	DEBUG_LEAVE(h_scatter_states)
	return FUNCTION_SUCCEEDED;
}		/*end h_scatter_states*/

/*
*			pp_send_interior_states():
*
*	Sends state information in a single buffer region.
*
*/

LOCAL	void	pp_send_interior_states(
	int		*me,
	int		swp,
	int		*iperm,
	int		side,
	Front		*front,
	Wave		*wave)
{
	INTERFACE     *intfc = front->interf;
	PP_GRID	      *pp_grid = front->pp_grid;
        RECT_GRID     *gr = front->rect_grid;
        byte	      *ps;
	int	      L[MAXD], U[MAXD];
	int	      i;
	int	      him[MAXD];
	int	      myid, dst_id;
        int	      dim = gr->dim;
	size_t        len;
	static byte   *storage = NULL;
	static size_t alloc_len = 0;

	DEBUG_ENTER(pp_send_interior_states)

	if (rect_boundary_type(intfc,iperm[swp],side) == REFLECTION_BOUNDARY)
	{
	    reflect_states_across_domain(iperm,side,swp,front,wave);
	    DEBUG_LEAVE(pp_send_interior_states)
	    return;
	}
	if (rect_boundary_type(intfc,iperm[swp],side) != SUBDOMAIN_BOUNDARY)
	{
	    DEBUG_LEAVE(pp_send_interior_states)
	    return;
	}

	myid = pp_mynode();
	dst_id = neighbor_id(him,me,iperm[swp],side,pp_grid);
	if (myid == dst_id)
	{
	    copy_states_across_domain(iperm,side,swp,front,wave);
	    DEBUG_LEAVE(pp_send_interior_states)
	    return;
	}

	set_send_domain(L,U,iperm,side,swp,gr);
	storage = (byte*) buffer_storage(front->sizest,L,U,dim,
					 (POINTER)storage,&len,&alloc_len);
	(*wave->bundle_states)(L,U,wave,storage);
	start_clock("pp_send_states");
	for (ps = storage, i = 0; len >= BLOCK_SIZE; 
				len -= BLOCK_SIZE, ps += BLOCK_SIZE, ++i)
	{
	    pp_send(state_id(i),(POINTER)ps,BLOCK_SIZE,dst_id);
	}
	if (len != 0)
	    pp_send(state_id(i),(POINTER)ps,len,dst_id);
	stop_clock("pp_send_states");
	DEBUG_LEAVE(pp_send_interior_states)
}		/*end pp_send_interior_states*/

/*
*			pp_receive_interior_states():
*
*	Receives state information in a single buffer region.
*
*/

LOCAL	void	pp_receive_interior_states(
	int		*me,
	int		swp,
	int		*iperm,
	int		side,
	Front		*front,
	Wave		*wave)
{
	INTERFACE     *intfc = front->interf;
	PP_GRID	      *pp_grid = front->pp_grid;
        RECT_GRID     *gr = front->rect_grid;
        byte	      *ps;
	int	      L[MAXD], U[MAXD];
	int	      him[MAXD];
	int	      myid, src_id;
        int	      dim = gr->dim;
	int	      i;
	size_t        len;
	static byte   *storage = NULL;
	static size_t alloc_len = 0;

	DEBUG_ENTER(pp_receive_interior_states)

	if (rect_boundary_type(intfc,iperm[swp],side) != SUBDOMAIN_BOUNDARY)
	{
	    DEBUG_LEAVE(pp_receive_interior_states)
	    return;
	}

	myid = pp_mynode();
	src_id = neighbor_id(him,me,iperm[swp],side,pp_grid);
	if (myid == src_id)
	{
	    DEBUG_LEAVE(pp_receive_interior_states)
	    return; /* Already done */
	}

	set_receive_domain(L,U,iperm,side,swp,gr);
	storage = (byte*) buffer_storage(front->sizest,L,U,dim,
					 (POINTER)storage,&len,&alloc_len);
	start_clock("pp_receive_states");
	for (ps = storage, i = 0; len >= BLOCK_SIZE;
				len -= BLOCK_SIZE, ps += BLOCK_SIZE,++i)
	{
	    pp_recv(state_id(i),src_id,(POINTER)ps,BLOCK_SIZE);
	}
	if (len != 0)
	    pp_recv(state_id(i),src_id,(POINTER)ps,len);
	stop_clock("pp_receive_states");
	(*wave->unbundle_states)(L,U,wave,storage);
	DEBUG_LEAVE(pp_receive_interior_states)
}		/*end pp_receive_interior_states*/

/*
*		set_send_domain():
*
*	Sets the domain limits for a buffer zone to be transmitted.
*
*/

LOCAL	void	set_send_domain(
	int		*L,
	int		*U,
	int		*iperm,
	int		side,
	int		swp,
	RECT_GRID	*gr)
{
        int		dim = gr->dim;
        int		*lbuf = gr->lbuf;
        int		*ubuf = gr->ubuf;
	int		*gmax = gr->gmax;
	int		j;

	DEBUG_ENTER(set_send_domain)
	for (j = 0; j < swp; ++j)
	{
	    L[iperm[j]] = -lbuf[iperm[j]];
	    U[iperm[j]] = gmax[iperm[j]] + ubuf[iperm[j]];
	}
	if (side == 0)
	{
	    L[iperm[swp]] =  0;
	    U[iperm[swp]] = lbuf[iperm[swp]];
	}
	else
	{
	    L[iperm[swp]] = gmax[iperm[swp]] - ubuf[iperm[swp]];
	    U[iperm[swp]] = gmax[iperm[swp]];
	}
	for (j = swp+1; j < dim; ++j)
	{
	    L[iperm[j]] = -lbuf[iperm[j]];
	    U[iperm[j]] = gmax[iperm[j]] + ubuf[iperm[j]];
	}
	DEBUG_LEAVE(set_send_domain)
}		/*end set_send_domain*/

/*
*		set_receive_domain():
*
*	Sets the domain limits for a buffer zone to be received.
*
*/

LOCAL	void	set_receive_domain(
	int		*L,
	int		*U,
	int		*iperm,
	int		side,
	int		swp,
	RECT_GRID	*gr)
{
        int		dim = gr->dim;
        int		*lbuf = gr->lbuf;
        int		*ubuf = gr->ubuf;
	int		*gmax = gr->gmax;
	int		j;

	DEBUG_ENTER(set_receive_domain)
	for (j = 0; j < swp; ++j)
	{
	    L[iperm[j]] = -lbuf[iperm[j]];
	    U[iperm[j]] = gmax[iperm[j]] + ubuf[iperm[j]];
	}
	if (side == 0)
	{
	    L[iperm[swp]] = -lbuf[iperm[swp]];
	    U[iperm[swp]] = 0;
	}
	else
	{
	    L[iperm[swp]] = gmax[iperm[swp]];
	    U[iperm[swp]] = gmax[iperm[swp]] + ubuf[iperm[swp]];
	}
	for (j = swp+1; j < dim; ++j)
	{
	    L[iperm[j]] = -lbuf[iperm[j]];
	    U[iperm[j]] = gmax[iperm[j]] + ubuf[iperm[j]];
	}
	if (DEBUG)
	{
	    (void) printf("swp = %d, side = %d, ",swp,side);
	    print_int_vector("iperm = ",iperm,dim,"\n");
	    print_int_vector("L = ",L,dim,", ");
	    print_int_vector("U = ",U,dim,"\n");
	}
	DEBUG_LEAVE(set_receive_domain)
}		/*end set_receive_domain*/

/*
*			buffer_storage():
*
*	Allocates storage buffer for buffer zone communication.
*/

LOCAL	POINTER buffer_storage(
	size_t  sizest,
	int     *L,
	int     *U,
	int     dim,
	POINTER storage,
	size_t  *plen,
	size_t  *palloc_len)
{
	int    i;
	size_t len;

	DEBUG_ENTER(buffer_storage)

	for (i = 0, len = 1; i < dim; ++i)
	    len *= U[i] - L[i];
	len *= sizest;
	*plen = len;

	if (len > *palloc_len)
	{
	    if (storage != NULL)
		free(storage);
	    scalar(&storage,len);
	    *palloc_len = len;
	}
	DEBUG_LEAVE(buffer_storage)
	return storage;
}		/*end buffer_storage*/

/*
*			copy_states_across_domain():
*
*	This special purpose function is provided to improve performance
*	in the case where the sending and receiving processors are the
*	same.  It copies the state in one region of the rectangular
*	lattice into another region.
*/

LOCAL	void	copy_states_across_domain(
	int		*iperm,
	int		side,
	int		swp,
	Front		*front,
	Wave		*wave)
{
        RECT_GRID	*gr = front->rect_grid;
	int		src_ic[MAXD], dst_ic[MAXD];
	int		src_L[MAXD], src_U[MAXD];
	int		dst_L[MAXD], dst_U[MAXD];
	size_t		sizest = front->sizest;

	DEBUG_ENTER(copy_states_across_domain)
	set_send_domain(src_L,src_U,iperm,side,swp,gr);
	set_receive_domain(dst_L,dst_U,iperm,(side+1)%2,swp,gr);

	switch (gr->dim)
	{
#if defined(ONED)
	case 1:
		{
		    int src_ix, dst_ix;
		    for (src_ix = src_L[0], dst_ix = dst_L[0];
					src_ix < src_U[0]; ++src_ix, ++dst_ix)
		    {
		        src_ic[0] = src_ix;
		        dst_ic[0] = dst_ix;
		        assign(Rect_state(dst_ic,wave),
				Rect_state(src_ic,wave),sizest);
		    }
		}
		break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
		{
		    int src_ix, dst_ix, src_iy, dst_iy;
		    for (src_iy = src_L[1], dst_iy = dst_L[1];
					src_iy < src_U[1]; ++src_iy, ++dst_iy)
		    {
		        src_ic[1] = src_iy;
		        dst_ic[1] = dst_iy;
		        for (src_ix = src_L[0], dst_ix = dst_L[0];
					src_ix < src_U[0]; ++src_ix, ++dst_ix)
		        {
		            src_ic[0] = src_ix;
		            dst_ic[0] = dst_ix;
		            assign(Rect_state(dst_ic,wave),
				Rect_state(src_ic,wave),sizest);
		        }
		    }
		}
		break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
		{
		    int src_ix, dst_ix, src_iy, dst_iy, src_iz, dst_iz;
		    for (src_iz = src_L[2], dst_iz = dst_L[2];
					src_iz < src_U[2]; ++src_iz, ++dst_iz)
		    {
		        src_ic[2] = src_iz;
		        dst_ic[2] = dst_iz;
		        for (src_iy = src_L[1], dst_iy = dst_L[1];
					src_iy < src_U[1]; ++src_iy, ++dst_iy)
		        {
		            src_ic[1] = src_iy;
		            dst_ic[1] = dst_iy;
		            for (src_ix = src_L[0], dst_ix = dst_L[0];
					src_ix < src_U[0]; ++src_ix, ++dst_ix)
		            {
		                src_ic[0] = src_ix;
		                dst_ic[0] = dst_ix;
		                assign(Rect_state(dst_ic,wave),
				    Rect_state(src_ic,wave),sizest);
		            }
		        }
		    }	
		}
		break;
#endif /* defined(THREED) */
	}
	DEBUG_LEAVE(copy_states_across_domain)
}		/*end copy_states_across_domain*/

/*
*		reflect_states_across_domain():
*
*	Copies the rect state storage across a reflection boundary.
*	States locations are reflected with respect to the reflection
*	boundary and the corresponding states are reflected.
*/

EXPORT	void	reflect_states_across_domain(
	int		*iperm,
	int		side,
	int		swp,
	Front		*front,
	Wave		*wave)
{
        RECT_GRID	*gr = front->rect_grid;
	Locstate	dst_st, src_st;
	float		*n;
	float		*dst_crds, *src_crds, p[MAXD];
	int		i, dim = gr->dim;
	int		dir = iperm[swp];
	int		src_ic[MAXD], dst_ic[MAXD];
	int		src_L[MAXD], src_U[MAXD];
	int		dst_L[MAXD], dst_U[MAXD];
	size_t		sizest = front->sizest;
	static float	nors[] = {  1.0,  0.0,  0.0,
				    0.0,  1.0,  0.0,
				    0.0,  0.0,  1.0,
				   -1.0,  0.0,  0.0,
				    0.0, -1.0,  0.0,
				    0.0,  0.0, -1.0};

	DEBUG_ENTER(reflect_states_across_domain)
	set_send_domain(src_L,src_U,iperm,side,swp,gr);
	set_receive_domain(dst_L,dst_U,iperm,side,swp,gr);

#define refl_ic(ic,dir,sL,dU)	((dU)[dir] - 1 + ((sL)[dir] - (ic)[dir]))

	n = nors+3*dir+9*side;
	switch (dim)
	{
#if defined(ONED)
	case 1:
	    {
	        int src_ix;
	        for (src_ix = src_L[0]; src_ix < src_U[0]; ++src_ix)
	        {
	            src_ic[0] = src_ix;
	    	    dst_ic[0] = refl_ic(src_ic,0,src_L,dst_U);
	    	    dst_st = Rect_state(dst_ic,wave);
	    	    src_st = Rect_state(src_ic,wave);
	    	    dst_crds = Rect_coords(dst_ic,wave);
	    	    src_crds = Rect_coords(src_ic,wave);
	    	    for (i = 0; i < dim; ++i)
	    		p[i] = 0.5*(dst_crds[i]+src_crds[i]);
	            assign(dst_st,src_st,sizest);
	    	    reflect_state(dst_st,front->interf,src_crds,p,n);
	        }
	    }
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    {
	        int src_ix, dst_ix, src_iy, dst_iy;
	        for (src_iy = src_L[1], dst_iy = dst_L[1];
	    			src_iy < src_U[1]; ++src_iy, ++dst_iy)
	        {
	            src_ic[1] = src_iy;
	            dst_ic[1] = dst_iy;
	            for (src_ix = src_L[0], dst_ix = dst_L[0];
	    			src_ix < src_U[0]; ++src_ix, ++dst_ix)
	            {
	                src_ic[0] = src_ix;
	                dst_ic[0] = dst_ix;
	    	        dst_ic[dir] = refl_ic(src_ic,dir,src_L,dst_U);
	    	        dst_st = Rect_state(dst_ic,wave);
	    	        src_st = Rect_state(src_ic,wave);
	    	        dst_crds = Rect_coords(dst_ic,wave);
	    	        src_crds = Rect_coords(src_ic,wave);
                        if(Rect_comp(src_ic,wave) != Rect_comp(dst_ic,wave))
                        {
                            static Locstate tmpst = NULL;
                            float  crds[MAXD];
                            if(tmpst == NULL)
                                alloc_state(front->interf, &tmpst, front->sizest);
                            printf("WARNING: reflect_states_across_domain\n");
                            printf("Diff comp: src_ic[%d %d], dst_ic[%d %d]\n",
                                 src_ic[0],src_ic[1], dst_ic[0], dst_ic[1]);
                            printf("src_comp[%d], dst_comp[%d]\n",
                                Rect_comp(src_ic,wave), Rect_comp(dst_ic,wave));

                            for (i = 0; i < dim; ++i)
                                crds[i] = src_crds[i];
                            if (! nearest_intfc_state(crds,Rect_comp(dst_ic,wave),
                                    front->interf, tmpst,crds,NULL))
                            {
                                printf("ERROR in reflect_states_across_domain\n");
                                printf("dst_ic[%d,%d] is unfilled\n",
                                  dst_ic[0], dst_ic[1]);
                                clean_up(ERROR);
                            }
                            for (i = 0; i < dim; ++i)
                                p[i] = 0.5*(dst_crds[i]+src_crds[i]);
                            assign(dst_st,tmpst,sizest);
                            reflect_state(dst_st,front->interf,src_crds,p,n);
                        }
                        else
                        {
	    	            for (i = 0; i < dim; ++i)
	    		        p[i] = 0.5*(dst_crds[i]+src_crds[i]);
	                    assign(dst_st,src_st,sizest);
	    	            reflect_state(dst_st,front->interf,src_crds,p,n);
                        } 
	            }
	        }
	    }
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    {
	        int src_ix, dst_ix, src_iy, dst_iy, src_iz, dst_iz;
	        for (src_iz = src_L[2], dst_iz = dst_L[2];
	    			src_iz < src_U[2]; ++src_iz, ++dst_iz)
	        {
	            src_ic[2] = src_iz;
	            dst_ic[2] = dst_iz;
	            for (src_iy = src_L[1], dst_iy = dst_L[1];
	    			src_iy < src_U[1]; ++src_iy, ++dst_iy)
	            {
	                src_ic[1] = src_iy;
	                dst_ic[1] = dst_iy;
	                for (src_ix = src_L[0], dst_ix = dst_L[0];
	    			src_ix < src_U[0]; ++src_ix, ++dst_ix)
	                {
	                    src_ic[0] = src_ix;
	                    dst_ic[0] = dst_ix;
	    	            dst_ic[dir] = refl_ic(src_ic,dir,src_L,dst_U);
	    	            dst_st = Rect_state(dst_ic,wave);
	    	            src_st = Rect_state(src_ic,wave);
	    	            dst_crds = Rect_coords(dst_ic,wave);
	    	            src_crds = Rect_coords(src_ic,wave);
	                    for (i = 0; i < dim; ++i)
	        	        p[i] = 0.5*(dst_crds[i]+src_crds[i]);
	                    assign(dst_st,src_st,sizest);
	    	            reflect_state(dst_st,front->interf,src_crds,p,n);
	                }
	            }
	        }	
	    }
	    break;
#endif /* defined(THREED) */
	}
#undef refl_ic
	DEBUG_LEAVE(reflect_states_across_domain)
}		/*end reflect_states_across_domain*/

#if defined(__MPI__)
EXPORT 	bool 	h_iscatter_states(
	Wave		*wave,
	Front		*front,
	int             *iperm,
        int             swp)
{
	INTERFACE     *intfc = front->interf;
	MPI_Request    ireq[2], oreq[2];
	PP_GRID	       *pp_grid = wave->pp_grid;
        RECT_GRID      *gr = front->rect_grid;
	bool           received[2], sent[2];
	int	       iL[2][3], iU[2][3];
	int	       oL[2][3], oU[2][3];
	int            itag[2], otag[2];
	int	       myid;
	int	       me[3], him[3];
	int	       i, j, dim;
	int            src_id, dst_id;
	size_t         ilen[2], olen[2];
        static POINTER ibuf[2], obuf[2];
	static size_t  alloc_ilen[2], alloc_olen[2];

	DEBUG_ENTER(h_iscatter_states)

	if (wave->sizest == 0)
	{
	    DEBUG_LEAVE(h_iscatter_states)
	    return YES;
	}

	start_clock("h_iscatter_states");

	myid = pp_mynode();
	dim = wave->rect_grid->dim;

	find_Cartesian_coordinates(myid,pp_grid,me);

	if (DEBUG)
	{
	    (void) printf("step = %d, ",front->step);
	    print_int_vector("iperm = ",iperm,dim,", ");
	    print_int_vector("me = ",me,dim,"\n");
	}


	/* Set domain sizes */
	i = iperm[swp];
	for (j = 0; j < 2; ++j)
	{
	    itag[j] = state_id(2*i+j);
	    otag[j] = state_id(2*i+(j+1)%2);
	    if (rect_boundary_type(intfc,i,j) == REFLECTION_BOUNDARY)
	    {
	        reflect_states_across_domain(iperm,j,swp,front,wave);
		received[j] = YES;
		sent[j] = YES;
	    }
	    else if (rect_boundary_type(intfc,i,j) != SUBDOMAIN_BOUNDARY)
	    {
		received[j] = YES;
		sent[j] = YES;
	    }
	    else
	    {
		received[j] = NO;
	        set_receive_domain(iL[j],iU[j],iperm,j,swp,gr);
		sent[j] = NO;
	        set_send_domain(oL[j],oU[j],iperm,j,swp,gr);
	        ibuf[j] = buffer_storage(front->sizest,iL[j],iU[j],dim,
					     ibuf[j],ilen+j,alloc_ilen+j);
	        obuf[j] = buffer_storage(front->sizest,oL[j],oU[j],dim,
					     obuf[j],olen+j,alloc_olen+j);
	        (*wave->bundle_states)(oL[j],oU[j],wave,(byte*)obuf[j]);
	        src_id = neighbor_id(him,me,i,j,pp_grid);
		if (DEBUG)
		{
			(void) printf("Receiving data in direction %d side %d "
				      "from processor %d with tag %d (%d)\n",
				      i,j,src_id,itag[j],itag[j]-state_id(0));
			(void) printf("iL[%d] = ",j);
			print_int_vector("",iL[j],dim,", ");
			(void) printf("iU[%d] = ",j);
			print_int_vector("",iU[j],dim,"\n");
			(void) printf("ilen[%d] = %d, %d states\n",j,
				      (int)ilen[j],
				      (int)(ilen[j]/front->sizest));
		}
	        pp_irecv(itag[j],src_id,ibuf[j],ilen[j],ireq+j);
	    }
	}

	pp_gsync();

	/* Submit nonblocking sends */
	for (j = 0; j < 2; ++j)
	{
	    if (sent[j] == NO)
	    {
	        dst_id = neighbor_id(him,me,i,j,pp_grid);
		if (DEBUG)
		{
		    (void) printf("Sending data in direction %d side %d "
				      "to processor %d with tag %d (%d)\n",
				      i,j,dst_id,otag[j],otag[j]-state_id(0));
		    (void) printf("oL[%d] = ",j);
		    print_int_vector("",oL[j],dim,", ");
		    (void) printf("oU[%d] = ",j);
		    print_int_vector("",oU[j],dim,"\n");
		    (void) printf("olen[%d] = %d, %d states\n",
				      j,(int)olen[j],
				      (int)(olen[j]/front->sizest));
		}
	        pp_isend(otag[j],obuf[j],olen[j],dst_id,oreq+j);
	    }
	}

	/* Process received state data */
	for (j = 0; j < 2; ++j)
	{
	    if (received[j] == NO)
	    {
		pp_wait(ireq+j);
		received[j] = YES;
	        (*wave->unbundle_states)(iL[j],iU[j],wave,(byte*)ibuf[j]);
	    }
	}

	/* Wait for sends to complete */
	for (j = 0; j < 2; ++j)
	{
	    if (sent[j] == NO)
	    {
		pp_wait(oreq+j);
		sent[j] = YES;
	    }
	}
	pp_gsync();

	stop_clock("h_iscatter_states");
	DEBUG_LEAVE(h_iscatter_states)
	return YES;
}		/*end h_iscatter_states*/
#endif /* defined(__MPI__) */

EXPORT void pp_send_large_data(
        byte          *storage,
        size_t        len,
        int           dst_id)
{
        byte          *ps;
        int           i;

        DEBUG_ENTER(pp_send_large_data)

        for (ps = storage, i = 0; len >= BLOCK_SIZE;
                   len -= BLOCK_SIZE, ps += BLOCK_SIZE, ++i)
        {
            pp_send(state_id(i),(POINTER)ps,BLOCK_SIZE,dst_id);
        }
        if (len != 0)
            pp_send(state_id(i),(POINTER)ps,len,dst_id);

        DEBUG_LEAVE(pp_send_large_data)
}


EXPORT void pp_receive_large_data(
        byte          *storage,
        size_t        len,
        int           src_id)
{
        byte          *ps;
        int           i;

        DEBUG_ENTER(pp_send_large_data)

        for (ps = storage, i = 0; len >= BLOCK_SIZE;
                   len -= BLOCK_SIZE, ps += BLOCK_SIZE, ++i)
        {
            pp_recv(state_id(i),src_id,(POINTER)ps,BLOCK_SIZE);
        }
        if (len != 0)
            pp_recv(state_id(i),src_id,(POINTER)ps,len);

        DEBUG_LEAVE(pp_send_large_data)
}

#if defined(CONSERVATIVE_ALG) && defined(TWOD)

LIB_LOCAL int scatter_frac_cell_states(
        Wave            *wave,
        Front           *front)
{
        PP_GRID         *pp_grid = wave->pp_grid;
        int             *iperm; /* Controls order of communication sweeps */
        int             myid;
        int             me[MAXD];
        int             i, side, dim;

        DEBUG_ENTER(scatter_frac_cell_states)

        if(wave->rect_grid->dim != 2 || wave->sizest == 0)
        {
            DEBUG_LEAVE(scatter_frac_cell_states)
            return YES;
        }

        myid = pp_mynode();
        dim = wave->rect_grid->dim;
        iperm = set_iperm(front->step,dim);

        find_Cartesian_coordinates(myid,pp_grid,me);
        for (i = 0; i < dim; i++)
        {
            for (side = 0; side < 2; side++)
            {
                pp_gsync();
                pp_send_frac_interior_states(me,i,iperm,side,front,wave);
                pp_receive_frac_interior_states(me,i,iperm,(side+1)%2,front,wave);
            }
        }


        DEBUG_LEAVE(scatter_frac_cell_states)
        return YES;  
}

LOCAL   void    pp_send_frac_interior_states(
        int             *me,
        int             swp,
        int             *iperm,
        int             side,
        Front           *front,
        Wave            *wave)
{
        INTERFACE       *intfc = front->interf;
        PP_GRID         *pp_grid = front->pp_grid;
        RECT_GRID       *gr = front->rect_grid;
        byte            *ps, *storage;
        int             L[MAXD], U[MAXD];
        int             i;
        size_t          len;
        int             him[MAXD];
        int             myid, dst_id;
        int             dim = gr->dim;

        DEBUG_ENTER(pp_send_frac_interior_states)

        if (rect_boundary_type(intfc,iperm[swp],side) == REFLECTION_BOUNDARY)
        {
            reflect_frac_states_across_domain(iperm,side,swp,front,wave);
            DEBUG_LEAVE(pp_send_frac_interior_states)
            return;
        }

        if (rect_boundary_type(intfc,iperm[swp],side) != SUBDOMAIN_BOUNDARY)
        {
            DEBUG_LEAVE(pp_send_frac_interior_states)
            return;
        }

        myid = pp_mynode();
        dst_id = neighbor_id(him,me,iperm[swp],side,pp_grid);
        if (myid == dst_id)
        {
            copy_frac_states_across_domain(iperm,side,swp,front,wave);
            DEBUG_LEAVE(pp_frac_send_interior_states)
            return;
        }

        set_send_domain(L,U,iperm,side,swp,gr);

        storage = frac_buffer_storage(front->sizest,L,U,dim,&len); 
        // g_bundle_frac_states
        (*wave->bundle_frac_states)(L,U,wave,storage);

        start_clock("pp_send_frac_states");
        for (ps = storage, i = 0; len >= BLOCK_SIZE;
                                len -= BLOCK_SIZE, ps += BLOCK_SIZE, i++)
        {
            pp_send(state_id(i),(POINTER)ps,BLOCK_SIZE,dst_id);
        }
        if (len != 0)
            pp_send(state_id(i),(POINTER)ps,len,dst_id);

        stop_clock("pp_send_frac_states");
        DEBUG_LEAVE(pp_send_frac_interior_states)
        return;

}

LOCAL   void    reflect_frac_states_across_domain(
        int             *iperm,
        int             side,
        int             swp,
        Front           *front,
        Wave            *wave)
{
        RECT_GRID       *gr = front->rect_grid;
        Locstate        dst_st, src_st;
        float           *n;
        float           *dst_crds, *src_crds, p[MAXD];
        int             dim = gr->dim;
        int             dir = iperm[swp];
        int             src_ic[MAXD], dst_ic[MAXD];
        int             src_L[MAXD], src_U[MAXD];
        int             dst_L[MAXD], dst_U[MAXD];
        size_t          sizest = front->sizest;
        static float    nors[] = {  1.0,  0.0,  0.0,
                                    0.0,  1.0,  0.0,
                                    0.0,  0.0,  1.0,
                                   -1.0,  0.0,  0.0,
                                    0.0, -1.0,  0.0,
                                    0.0,  0.0, -1.0};
        COMPONENT       comp;
        float           *crds;
        float           c_crds[MAXD+1];
        int             s_ic[MAXD], src_ic_end[MAXD], d_ic[MAXD];
        int             dst_end[2]; 
        int             i, j, jj, ii;
        int             max[MAXD], min[MAXD];
        int             lbuf[MAXD], ubuf[MAXD];
        float           dh[MAXD];
        CRXING          *crx0[4], *crx1[4], *crx2[4], *crx3[4];
        CRXING          *dcrx0[4], *dcrx1[4], *dcrx2[4], *dcrx3[4]; 
        int             nc[4], dnc[4]; /* number of CRXING */
        TRI_GRID        *ntg;
        register Locstate *src_state, *dst_state;

        DEBUG_ENTER(reflect_frac_states_across_domain)
        set_send_domain(src_L,src_U,iperm,side,swp,gr);
        set_receive_domain(dst_L,dst_U,iperm,side,swp,gr);

#define refl_ic(ic,d,sL,dU)     ((dU)[dir] - 1 + ((sL)[dir] - (ic)[dir]))

        n = nors+3*dir+9*side;
        switch (dim)
        {
#if defined(ONED)
        case 1:
                {
                    printf("ERROR: In reflect_consv_states_across_domain()"
                           " NEED to implement for 1d case\n");
                    clean_up(ERROR);
                }
                break;
#endif /* defined(ONED) */
#if defined(TWOD)
        case 2:
                {
                    int src_ix, dst_ix, src_iy, dst_iy;
                    /***
                    if(NULL == crx[0])
                    {
                        for(i = 0; i < 4; i++)
                            vector(&crx[i],4,sizeof(CRXING *));
                    }
                    ***/

                    ntg = wave_tri_soln(wave)->tri_grid;

                    for (src_iy = src_L[1], dst_iy = dst_L[1];
                                        src_iy < src_U[1]; src_iy++, dst_iy++)
                    {
                        src_ic[1] = src_iy;
                        dst_ic[1] = dst_iy;
                        for (src_ix = src_L[0], dst_ix = dst_L[0];
                                        src_ix < src_U[0]; src_ix++, dst_ix++)
                        {
                            src_ic[0] = src_ix;
                            dst_ic[0] = dst_ix;
                            dst_ic[dir] = refl_ic(src_ic,dir,src_L,dst_U);

                            src_ic_end[0] = src_ic[0] + 1;
                            src_ic_end[1] = src_ic[1] + 1;
                            dst_end[0] = dst_ic[0] + 1;
                            dst_end[1] = dst_ic[1] + 1;

                            // if(Comp_blk(src_ic,ntg->blk_type,ntg) == F_NO_VOL)
                            if(is_complex_blk(Comp_blk(src_ic,ntg->blk_type,ntg)))
                            {
                                // Comp_blk(dst_ic,ntg->vol_states,ntg) = NULL;
                                continue;
                            }

                            nc[0] = comp_crossings_in_direction(crx0,src_ic,EAST,ntg);
                            nc[2] = comp_crossings_in_direction(crx2,src_ic,NORTH,ntg);
                            nc[1] = comp_crossings_in_direction(crx1,src_ic_end,WEST,ntg);
                            nc[3] = comp_crossings_in_direction(crx3,src_ic_end,SOUTH,ntg);

                            dnc[0] = comp_crossings_in_direction(dcrx0,dst_ic,EAST,ntg);
                            dnc[2] = comp_crossings_in_direction(dcrx2,dst_ic,NORTH,ntg);
                            dnc[1] = comp_crossings_in_direction(dcrx1,dst_end,WEST,ntg);
                            dnc[3] = comp_crossings_in_direction(dcrx3,dst_end,SOUTH,ntg);
 
                            /** Due to the tolerance etc. the micro cell structure on the 
                             ** two reflection side might not be the same. The current
                             ** fix is to skip reflecting states when this is the case.
                             **/ 
                            if(dir == 0 && (nc[3] != dnc[2] ||
                               nc[1] != dnc[1] || nc[2] != dnc[3] ||
                               nc[0] != dnc[0]))
                            {
                                printf("WARNING: reflect_frac_states_across_domain\n"); 
                                printf("inconsistent detected for src[%d %d], dst[%d %d]\n",
                                          src_ic[0], src_ic[1], dst_ic[0], dst_ic[1]);
                                continue; 
                            }
                            if(dir == 1 && (nc[3] != dnc[3] ||
                               nc[1] != dnc[0] || nc[2] != dnc[2] ||
                               nc[0] != dnc[1]))
                            {
                                printf("WARNING: reflect_frac_states_across_domain\n"); 
                                printf("inconsistent detected for src[%d %d], dst[%d %d]\n",
                                          src_ic[0], src_ic[1], dst_ic[0], dst_ic[1]);
                                continue; 
                            }


                            dst_state = Comp_blk(dst_ic,ntg->vol_states,ntg);
                            src_state = Comp_blk(src_ic,ntg->vol_states,ntg);
                            dst_crds = Rect_coords(dst_ic,wave);
                            src_crds = Rect_coords(src_ic,wave);
                            for (i = 0; i < dim; i++)
                                p[i] = 0.5*(dst_crds[i]+src_crds[i]);

                            if(nc[1] == 0 && nc[0] == 0 &&
                               nc[2] == 0 && nc[3] == 0)
                            {
                                Comp_blk(dst_ic,ntg->vol_states,ntg) = NULL;
                            }
                            else if(nc[0] == 1 && nc[2] == 1 &&
                                    nc[1] == 0 && nc[3] == 0)
                            {
                                assign(dst_state[0],src_state[1],sizest);
                                assign(dst_state[1],src_state[0],sizest);
                                if(dir == 0)
                                {
                                    d_ic[0] = dst_ic[0]+1; d_ic[1] = dst_ic[1];
                                    /*
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic_end, ntg));
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    */
                                }
                                if(dir == 1 && side == 0)
                                {
                                    d_ic[0] = dst_ic[0]; d_ic[1] = dst_ic[1]+1;
                                    /*
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic_end, ntg));
                                    */
                                }
                                if(dir == 1 && side == 1)
                                {
                                    d_ic[0] = dst_ic[0]; d_ic[1] = dst_ic[1]+1;
                                    s_ic[0] = src_ic[0]; s_ic[1] = src_ic[1]+1;
                                    /*
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    assert(Regular_comp_grid_comp(s_ic, ntg) ==
                                           Regular_comp_grid_comp(dst_ic, ntg));
                                    */
                                }
                                reflect_state(dst_state[0],front->interf,src_crds,p,n);
                                reflect_state(dst_state[1],front->interf,src_crds,p,n);
                            }
                            else if(nc[0] == 1 && nc[3] == 1 &&
                                    nc[1] == 0 && nc[2] == 0)
                            {
                                if(dir == 0)
                                {
                                    assign(dst_state[0],src_state[1],sizest);
                                    assign(dst_state[1],src_state[0],sizest);

                                    d_ic[0] = dst_ic[0]+1; d_ic[1] = dst_ic[1];
                                    s_ic[0] = src_ic[0]+1; s_ic[1] = src_ic[1];
                                    /*
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(s_ic, ntg));
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    */
                                }
                                if(dir == 1)
                                {
                                    assign(dst_state[0],src_state[0],sizest);
                                    assign(dst_state[1],src_state[1],sizest);

                                    d_ic[0] = dst_ic[0]+1; d_ic[1] = dst_ic[1]+1;
                                    s_ic[0] = src_ic[0]+1; s_ic[1] = src_ic[1];
                                    /*
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(s_ic, ntg));
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    */

                                }
                                reflect_state(dst_state[0],front->interf,src_crds,p,n);
                                reflect_state(dst_state[1],front->interf,src_crds,p,n);
                            }
                            else if(nc[3] == 1 && nc[1] == 1 &&
                                    nc[2] == 0 && nc[0] == 0)
                            {
                                assign(dst_state[0],src_state[0],sizest);
                                assign(dst_state[1],src_state[1],sizest);
                                if(dir == 0)
                                {
                                    d_ic[0] = dst_ic[0]; d_ic[1] = dst_ic[1]+1;
                                    s_ic[0] = src_ic[0]+1; s_ic[1] = src_ic[1]+1;
                                    /*
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(s_ic, ntg));
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    */
                                }
                                if(dir == 1)
                                {
                                    d_ic[0] = dst_ic[0]+1; d_ic[1] = dst_ic[1];
                                    s_ic[0] = src_ic[0]+1; s_ic[1] = src_ic[1]+1;
                                    /*
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(s_ic, ntg));
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    */
                                }
                                reflect_state(dst_state[0],front->interf,src_crds,p,n);
                                reflect_state(dst_state[1],front->interf,src_crds,p,n);
                            }
                            else if(nc[1] == 1 && nc[2] == 1 &&
                                    nc[0] == 0 && nc[3] == 0)
                            {
                                if(dir == 0)
                                {
                                    assign(dst_state[0],src_state[0],sizest);
                                    assign(dst_state[1],src_state[1],sizest);
                                    d_ic[0] = dst_ic[0]+1; d_ic[1] = dst_ic[1]+1;
                                    s_ic[0] = src_ic[0]; s_ic[1] = src_ic[1]+1;
                                    /*
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(s_ic, ntg));
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    */
                                }
                                if(dir == 1)
                                {
                                    assign(dst_state[0],src_state[1],sizest);
                                    assign(dst_state[1],src_state[0],sizest);
                                    d_ic[0] = dst_ic[0]; d_ic[1] = dst_ic[1]+1;
                                    s_ic[0] = src_ic[0]; s_ic[1] = src_ic[1]+1;
                                    /*
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(s_ic, ntg));
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    */
                                }
                                reflect_state(dst_state[0],front->interf,src_crds,p,n);
                                reflect_state(dst_state[1],front->interf,src_crds,p,n);
                            }
                            else if(nc[2] == 1 && nc[3] == 1 &&
                                    nc[0] == 0 && nc[1] == 0)
                            {
                                if(dir == 0)
                                {
                                    assign(dst_state[0],src_state[0],sizest);
                                    assign(dst_state[1],src_state[1],sizest);
                                    d_ic[0] = dst_ic[0]; d_ic[1] = dst_ic[1]+1;
                                    s_ic[0] = src_ic[0]; s_ic[1] = src_ic[1]+1;
                                    /*
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(s_ic, ntg));
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    */
                                }
                                if(dir == 1)
                                {
                                    assign(dst_state[0],src_state[1],sizest);
                                    assign(dst_state[1],src_state[0],sizest);
                                    d_ic[0] = dst_ic[0]; d_ic[1] = dst_ic[1]+1;
                                    s_ic[0] = src_ic[0]; s_ic[1] = src_ic[1]+1;
                                    /*
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(s_ic, ntg));
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    */
                                }
                                reflect_state(dst_state[0],front->interf,src_crds,p,n);
                                reflect_state(dst_state[1],front->interf,src_crds,p,n);
                            }
                            else if(nc[1] == 1 && nc[0] == 1 &&
                                    nc[2] == 0 && nc[3] == 0)
                            {
                                if(dir == 0)
                                {
                                    assign(dst_state[0],src_state[1],sizest);
                                    assign(dst_state[1],src_state[0],sizest);
                                    d_ic[0] = dst_ic[0]+1; d_ic[1] = dst_ic[1]+1;
                                    s_ic[0] = src_ic[0]+1; s_ic[1] = src_ic[1]+1;
                                    /*
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(s_ic, ntg));
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    */
                                }
                                if(dir == 1)
                                {
                                    assign(dst_state[0],src_state[0],sizest);
                                    assign(dst_state[1],src_state[1],sizest);
                                    d_ic[0] = dst_ic[0]+1; d_ic[1] = dst_ic[1]+1;
                                    s_ic[0] = src_ic[0]+1; s_ic[1] = src_ic[1]+1;
                                    /*  
                                    assert(Regular_comp_grid_comp(d_ic, ntg) ==
                                           Regular_comp_grid_comp(s_ic, ntg));
                                    assert(Regular_comp_grid_comp(dst_ic, ntg) ==
                                           Regular_comp_grid_comp(src_ic, ntg));
                                    */
                                }
                                reflect_state(dst_state[0],front->interf,src_crds,p,n);
                                reflect_state(dst_state[1],front->interf,src_crds,p,n);
                            }
                            else if(nc[1] == 1 && nc[0] == 1 &&
                                    nc[2] == 1 && nc[3] == 1)
                            {
                                printf("ERROR: In reflect_frac_states_across_domain()\n"); 
                                printf("3pt case for assigining memory for states\n");
                                clean_up(ERROR);
                            }
                            /*
                            printf("src_ic[%d,%d], dst_ic[%d,%d], dir = %d\n",
                                    src_ic[0],src_ic[1], dst_ic[0], dst_ic[1], dir);
                            printf("src_L[0,1]=[%d, %d], src_U[0,1]=[%d, %d]\n",
                                    src_L[0],src_L[1],src_U[0],src_U[1]);
                            printf("src state:\n");
                            (*front->print_state)(Rect_state(src_ic,wave));
                            printf("dst state:\n");
                            (*front->print_state)(Rect_state(dst_ic,wave));
                            */
                        }
                    }
                }
                break;
#endif /* defined(TWOD) */
#if defined(THREED)
        case 3:
                {
                    printf("ERROR: In reflect_frac_states_across_domain()"
                           " NEED to implement for 3d case\n");
                    clean_up(ERROR);
                }
                break;
#endif /* defined(THREED) */
        }
#undef refl_ic
        DEBUG_LEAVE(reflect_frac_states_across_domain)
}

LOCAL   void    copy_frac_states_across_domain(
        int             *iperm,
        int             side,
        int             swp,
        Front           *front,
        Wave            *wave)
{
        RECT_GRID       *gr = front->rect_grid;
        int             src_ic[MAXD], dst_ic[MAXD];
        int             src_L[MAXD], src_U[MAXD];
        int             dst_L[MAXD], dst_U[MAXD];
        size_t          sizest = front->sizest;

        COMPONENT       comp;
        float           *crds;
        float           c_crds[MAXD+1];
        int             s_ic[MAXD], src_ic_end[MAXD], d_ic[MAXD];
        int             i, j, jj, ii;
        int             max[MAXD], min[MAXD];
        int             lbuf[MAXD], ubuf[MAXD];
        float           dh[MAXD];
        static CRXING   **crx[4] = {NULL,NULL,NULL,NULL};
        int             nc[4]; /* number of CRXING */
        TRI_GRID        *ntg;
        register Locstate *src_state, *dst_state;

        DEBUG_ENTER(copy_frac_states_across_domain)

        set_send_domain(src_L,src_U,iperm,side,swp,gr);
        set_receive_domain(dst_L,dst_U,iperm,(side+1)%2,swp,gr);
        switch (gr->dim)
        {
#if defined(ONED)
        case 1:
                {
                    printf("ERROR: In consv_copy_states_across_domain()"
                           " NEED to implement for 1d case\n");
                    clean_up(ERROR);
                }
                break;
#endif /* defined(ONED) */
#if defined(TWOD)
        case 2:
                {
                    int src_ix, dst_ix, src_iy, dst_iy;
                    if(NULL == crx[0])
                    {
                        for(i = 0; i < 4; i++)
                            vector(&crx[i],4,sizeof(CRXING *));
                    }
                    ntg = wave_tri_soln(wave)->tri_grid;

                    for (src_iy = src_L[1], dst_iy = dst_L[1];
                                        src_iy < src_U[1]; src_iy++, dst_iy++)
                    {
                        src_ic[1] = src_iy;
                        dst_ic[1] = dst_iy;
                        for (src_ix = src_L[0], dst_ix = dst_L[0];
                                        src_ix < src_U[0]; src_ix++, dst_ix++)
                        {
                            src_ic[0] = src_ix;
                            dst_ic[0] = dst_ix;

                            src_ic_end[0] = src_ic[0] + 1;
                            src_ic_end[1] = src_ic[1] + 1;

                            // if(Comp_blk(src_ic,ntg->blk_type,ntg) == F_NO_VOL)
                            if(is_complex_blk(Comp_blk(src_ic,ntg->blk_type,ntg)))
                            {
                                // Comp_blk(dst_ic,ntg->vol_states,ntg) = NULL;
                                continue;
                            }

                            nc[0] = comp_crossings_in_direction(crx[0],src_ic,EAST, ntg);
                            nc[2] = comp_crossings_in_direction(crx[2],src_ic,NORTH, ntg);
                            nc[1] = comp_crossings_in_direction(crx[1],src_ic_end,WEST, ntg);
                            nc[3] = comp_crossings_in_direction(crx[3],src_ic_end,SOUTH, ntg);
                            dst_state = Comp_blk(dst_ic,ntg->vol_states,ntg);
                            src_state = Comp_blk(src_ic,ntg->vol_states,ntg);

                            if(nc[1] == 0 && nc[0] == 0 &&
                               nc[2] == 0 && nc[3] == 0)
                            {
                                Comp_blk(dst_ic,ntg->vol_states,ntg) = NULL;
                            }
                            else if(nc[0] == 1 && nc[2] == 1 &&
                                    nc[1] == 0 && nc[3] == 0)
                            {
                                assign(dst_state[0],src_state[0],sizest);
                                assign(dst_state[1],src_state[1],sizest);
                            }
                            else if(nc[0] == 1 && nc[3] == 1 &&
                                    nc[1] == 0 && nc[2] == 0)
                            {
                                assign(dst_state[0],src_state[0],sizest);
                                assign(dst_state[1],src_state[1],sizest);
                            }
                            else if(nc[3] == 1 && nc[1] == 1 &&
                                    nc[2] == 0 && nc[0] == 0)
                            {
                                assign(dst_state[0],src_state[0],sizest);
                                assign(dst_state[1],src_state[1],sizest);
                            }
                            else if(nc[1] == 1 && nc[2] == 1 &&
                                    nc[0] == 0 && nc[3] == 0)
                            {
                                assign(dst_state[0],src_state[0],sizest);
                                assign(dst_state[1],src_state[1],sizest);
                            }
                            else if(nc[2] == 1 && nc[3] == 1 &&
                                    nc[0] == 0 && nc[1] == 0)
                            {
                                assign(dst_state[0],src_state[0],sizest);
                                assign(dst_state[1],src_state[1],sizest);
                            }
                            else if(nc[1] == 1 && nc[0] == 1 &&
                                    nc[2] == 0 && nc[3] == 0)
                            {
                                assign(dst_state[0],src_state[0],sizest);
                                assign(dst_state[1],src_state[1],sizest);
                            }
                            else if(nc[1] == 1 && nc[0] == 1 &&
                                    nc[2] == 1 && nc[3] == 1)
                            {
                                printf("ERROR: In consv_copy_states_across_domain()\n");
                                printf("3pt case for assigining memory for states\n");
                                clean_up(ERROR);
                            }
                            else
                            {
                                printf("ERROR: In consv_copy_states_across_domain()\n");
                                printf("unknown case for assigining memory for states\n");
                                printf("nc[0,1,2,3] = <%d, %d, %d, %d>\n",
                                        nc[0], nc[1], nc[2], nc[3]);
                                printf("src_ic[0,1] = <%d, %d>, <%g, %g>\n",
                                        src_ic[0], src_ic[1],
                                        Rect_coords(src_ic,wave)[0],
                                        Rect_coords(src_ic,wave)[1]);
                                print_crxings(crx[0][0], NO);
                                print_crxings(crx[1][0], NO);
                                print_crxings(crx[2][0], NO);
                                print_crxings(crx[3][0], NO);
                                {
                                    CURVE **c;
                                    for (c = ntg->grid_intfc->curves;
                                         c && *c; c++)
                                    {
                                        if(!is_bdry(*c))
                                            print_curve(*c);
                                    }
                                }
                                clean_up(ERROR);
                            }
                        }
                    }
                }
                break;
#endif /* if defined(TWOD) */
#if defined(THREED)
        case 3:
                {
                    printf("ERROR: In consv_copy_states_across_domain()"
                           " NEED to implement for 1d case\n");
                    clean_up(ERROR);
                }
                break;
#endif /* defined(THREED) */
        }
}

/*
*                       frac_buffer_storage():
*
*       Allocates storage buffer for buffer zone communication.
*/

LOCAL   byte    *frac_buffer_storage(
        size_t          sizest,
        int             *L,
        int             *U,
        int             dim,
        size_t          *plen)
{
        int             i;
        size_t          len;
        static size_t   cur_len;
        static byte     *storage;

        DEBUG_ENTER(frac_buffer_storage)
        for (i = 0, len = 1; i < dim; i++)
            len *= U[i] - L[i];

        len *= 3;  /* each cell, at most 3 pieces */
        if (len > cur_len)
        {
            cur_len = len;
            if (storage != NULL)
                free(storage);
            scalar(&storage,sizest*cur_len);
        }
        *plen = len*sizest;
        DEBUG_LEAVE(frac_buffer_storage)
        return storage;
}               /*end frac_buffer_storage*/

LOCAL   void    pp_receive_frac_interior_states(
        int             *me,
        int             swp,
        int             *iperm,
        int             side,
        Front           *front,
        Wave            *wave)
{
        INTERFACE       *intfc = front->interf;
        PP_GRID         *pp_grid = front->pp_grid;
        RECT_GRID       *gr = front->rect_grid;
        byte            *ps, *storage;
        int             L[MAXD], U[MAXD];
        int             him[MAXD];
        int             myid, src_id;
        int             dim = gr->dim;
        int             i;
        size_t          len;

        DEBUG_ENTER(pp_receive_frac_interior_states)

        if (rect_boundary_type(intfc,iperm[swp],side) != SUBDOMAIN_BOUNDARY)
        {
            DEBUG_LEAVE(pp_receive_frac_interior_states)
            return;
        }


        myid = pp_mynode();
        src_id = neighbor_id(him,me,iperm[swp],side,pp_grid);
        if (myid == src_id)
        {
            DEBUG_LEAVE(pp_receive_frac_interior_states)
            return; /* Already done */
        }

        set_receive_domain(L,U,iperm,side,swp,gr);
        storage = frac_buffer_storage(front->sizest,L,U,dim,&len);
        start_clock("pp_receive_frac_states");
        for (ps = storage, i = 0; len >= BLOCK_SIZE;
                                len -= BLOCK_SIZE, ps += BLOCK_SIZE,i++)
        {
            pp_recv(state_id(i),src_id,(POINTER)ps,BLOCK_SIZE);
        }
        if (len != 0)
            pp_recv(state_id(i),src_id,(POINTER)ps,len);
        stop_clock("pp_receive_frac_states");
 
        // g_unbundle_frac_states
        (*wave->unbundle_frac_states)(L,U,wave,storage);
        DEBUG_LEAVE(pp_receive_frac_interior_states)
        return;

}


#endif /* if defined(CONSERVATIVE_ALG) && defined(TWOD) */




