/*
*                               hconsvnpt.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*       Contains n-point stencil code for finite difference solution
*       of the hyperbolic equations within a single component of a tracked
*       interface problem.
*       Point sources in the region are found and passed appropriately.
*/


#include <hyp/hlocaldecs.h>

#define DEBUG_HYP_CNPT

/* Copied from trigrid1.c */
#if !defined(_NTOL)
#if defined(float)
#define _NTOL 1.0e-5
#define _TOL  1.0e-6
#else /* defined(float) */
#define _NTOL 1.0e-3
#define _TOL  1.0e-4
#endif /* defined(float) */

LOCAL const float NTOL     = _NTOL;     /*TOLERANCE*/
LOCAL const float TOL      = _TOL;      /*TOLERANCE*/
LOCAL const float ONEMNTOL = 1.0 - _NTOL; /*TOLERANCE*/
LOCAL const float ONEMTOL  = 1.0 - _TOL;  /*TOLERANCE*/
#endif /* if !defined(_NTOL) */


#if defined(CONSERVATIVE_ALG) && defined(TWOD)

LOCAL   int     gmax[MAXD], ggmax[MAXD], ggmin[MAXD], vsten_rad, is_vec_method;
LOCAL   int     lbuffed[MAXD], ubuffed[MAXD]; 
LOCAL   int     dim;
LOCAL   int     lbuf[MAXD], ubuf[MAXD];
LOCAL   float   h[MAXD];
LOCAL   float   L[MAXD], VL[MAXD], VU[MAXD];

/*
*       The following globals are shared between hyp_npt() and
*       the subroutine update_reg_grid_state().  They are used
*       to pass quantities that do not change inside the inner loop
*       of the update step.  The are passed as globals for efficiency.
*/

LOCAL   COMPONENT       ext_comp;
LOCAL   float           hdir, dir[MAXD];
LOCAL   GRID_DIRECTION  prev_side, next_side;
LOCAL   INTERFACE       *intfc = NULL;
LOCAL   int             endpt;

#undef cell_center
#define cell_center(indx,i,gr)  (L[i] + ((indx) + 0.5)*h[i])

LOCAL void     set_consv_npt_global(Wave*); 
LOCAL void     update_volume_state_2d(int*,CSG_BLK_CRX*,CSG_Solid*,
                     Wave*,Wave*,Front*,Front*,float,float*); 
LOCAL int      buffer_boundary_volume(int*,CSG_BLK_CRX*,CSG_Solid*,Wave*); 
LOCAL int      face_on_buffer_boundary(CSG_Face*); 
LOCAL int      is_interior_side_face(CSG_Face*);
LOCAL void     set_flow_specified_volume_state(CSG_Solid*,Wave*,Front*,FlowSpecifiedRegion*); 
LOCAL int      is_trivial_top_vol(CSG_Solid*); 


/* NEW FUNCTION */
LOCAL void     update_volume_intfc_state_2d(int*,CSG_BLK_CRX*,CSG_Solid*,
                     Wave*,Wave*,Front*,Front*,float,float*); 
LOCAL void     update_volume_new_state_2d(int*,CSG_BLK_CRX*,CSG_Solid*,
                     Wave*,Wave*,Front*,Front*,float,float*); 

LOCAL void set_consv_npt_global(
        Wave  *wave)
{
        RECT_GRID       *r_grid;
        int             i;

        r_grid = wave->rect_grid;
        dim = r_grid->dim;
        for (i = 0; i < dim; ++i)
        {
            gmax[i] = r_grid->gmax[i];
            h[i] = r_grid->h[i];
            L[i] = r_grid->L[i];
            VL[i] = r_grid->VL[i];
            VU[i] = r_grid->VU[i];
            lbuf[i] = r_grid->lbuf[i];
            ubuf[i] = r_grid->ubuf[i];
            ggmin[i] = 0; 
            ggmax[i] = r_grid->gmax[i];  
            ggmin[i] -= r_grid->lbuf[i]; 
            ggmax[i] += r_grid->ubuf[i];  
            if(lbuf[i] != 0)
                lbuffed[i] = YES;
            else
                lbuffed[i] = NO;
            if(ubuf[i] != 0)
                ubuffed[i] = YES;
            else
                ubuffed[i] = NO;  
        }
        vsten_rad = vsten_radius(wave);
        is_vec_method = is_vector_method(wave);
}

LIB_LOCAL void hyp_consv_npt(
        int             swp_num,
        int             *iperm,     /* sweep based permutation of coord dir */
        float           *dh,
        float           dt,
        Wave            *wave,
        Wave            *newwave,
        Front           *fr,
        Front           *newfr,     /* newfr needed if hlw is to support */
                                    /*  changing topologies          */
        COMPONENT       max_comp)
{
        POINT           Basep;
        int             i, idirs[MAXD];
        int             i0, i1, ic[MAXD];
        static  Stencil *sten = NULL;
        static  int     num_pts;
        static  int     **icoords = NULL;
        RECT_GRID       *gr = wave->rect_grid;
        CSG_BLK_CRX     *blk;
        CSG_Solid       *s; 
        TRI_GRID        *ntg = wave_tri_soln(wave)->tri_grid;

        debug_print("hyp", "Entered hyp_consv_npt\n"); 

        /* SET LOCAL GLOBAL */
        set_consv_npt_global(wave); 
        intfc = fr->interf;
        ext_comp = exterior_component(intfc);

        /* ALLOCATE S_T_INTFC mid_st storage HERE */

        // g_alloc_stintfc_st_pool(); 
        (*wave->_alloc_stintfc_st_pool)(wave); 
      
        for(i1 = ggmin[1]; i1 < ggmax[1]; i1++)
        {
            for(i0 = ggmin[0]; i0 < ggmax[0]; i0++)
            {
                ic[0] = i0; ic[1] = i1; 
                blk = Comp_blk(ic,ntg->Volume.blk,ntg); 
                if(blk == NULL)
                    continue;  
                if((s = blk->s) == NULL)
                    continue;  

                while(s)
                {
                    /***
                    if(fr->step>= 619 
                       && s->solidno == 56) 
                    {
                        s->debug = YES;  
                        // print_solid(blk);  
                        solidls(s, 2); 
                    }
                    ***/

                    update_volume_intfc_state_2d(ic,blk,s,wave,
                         newwave,fr,newfr,dt,dh); 
                    s = s->nexts;   
                }
            }
        } 
        for(i1 = ggmin[1]; i1 < ggmax[1]; i1++)
        {
            for(i0 = ggmin[0]; i0 < ggmax[0]; i0++)
            {
                ic[0] = i0; ic[1] = i1; 
                blk = Comp_blk(ic,ntg->Volume.blk,ntg); 
                if(blk == NULL)
                    continue;  
                if((s = blk->s) == NULL)
                    continue;  
                while(s)
                {
                    update_volume_new_state_2d(ic,blk,s,wave,
                         newwave,fr,newfr,dt,dh); 
                    s = s->nexts;   
                }
            }
        } 
        /** FREE S_T_INTFC mid_st storage HERE **/
        // g_free_stintfc_st_pool(); 
        (*wave->_free_stintfc_st_pool)(); 
       
        /**************************************/
        /************ OLD ALG.
        for(i1 = ggmin[1]; i1 < ggmax[1]; i1++)
        {
            for(i0 = ggmin[0]; i0 < ggmax[0]; i0++)
            {
                ic[0] = i0; ic[1] = i1; 
                blk = Comp_blk(ic,ntg->Volume.blk,ntg); 
                if(blk == NULL)
                    continue;  
                if((s = blk->s) == NULL)
                    continue;  

                while(s)
                {
                    update_volume_state_2d(ic,blk,s,wave,
                         newwave,fr,newfr,dt,dh); 
                    s = s->nexts;   
                }
            }
        } 
        *******************************/

        debug_print("hyp", "Left hyp_consv_npt\n"); 
}

LOCAL void update_volume_new_state_2d(
        int             *ic, 
        CSG_BLK_CRX     *blk,
        CSG_Solid       *s,
        Wave            *wave,
        Wave            *newwave,
        Front           *front,
        Front           *newfront,
        float           dt,
        float           *dh)
{
        FlowSpecifiedRegion *fsr;

        if(YES == buffer_boundary_volume(ic,blk,s,wave))
            return;

        if(YES == is_trivial_top_vol(s))
            return;

        for (fsr = Fsr_list(front); fsr != NULL; fsr = fsr->next)
            if (ComponentsMatch(fsr,s->comp,fsr->comp,front->interf))
        {
            set_flow_specified_volume_state(s, wave, front, fsr);
            return;
        }
        
        // contrl_vol_update
        (*wave->_consv_vol_update)(ic,blk,s,wave,newwave,
			   front,newfront,dt,dh);        
}

LOCAL void update_volume_intfc_state_2d(
        int             *ic, 
        CSG_BLK_CRX     *blk,
        CSG_Solid       *s,
        Wave            *wave,
        Wave            *newwave,
        Front           *front,
        Front           *newfront,
        float           dt,
        float           *dh)
{
        FlowSpecifiedRegion *fsr;

        if(YES == buffer_boundary_volume(ic,blk,s,wave))
        {
#if defined(DEBUG)
            if(s->debug == YES)
            {
                printf("Solid[%d] is buffer_boundary_volume",
                   s->solidno); 
                printf(" do not update volume intfc state\n"); 
            }
#endif /* if defined(DEBUG) */
            return;  
        }
        if(YES == is_trivial_top_vol(s))
            return; 

        for (fsr = Fsr_list(front); fsr != NULL; fsr = fsr->next)
            if (ComponentsMatch(fsr,s->comp,fsr->comp,front->interf))
            return;

        // contrl_vol_FD_ver2
        (*wave->_consv_vol_solver2)(ic,blk,s,wave,newwave,
				    front,newfront,dt,dh); 
}


LOCAL void update_volume_state_2d(
        int             *ic, 
        CSG_BLK_CRX     *blk,
        CSG_Solid       *s,
        Wave            *wave,
        Wave            *newwave,
        Front           *front,
        Front           *newfront,
        float           dt,
        float           *dh)
{
        FlowSpecifiedRegion *fsr;

        if(YES == buffer_boundary_volume(ic,blk,s,wave))
            return;  

        for (fsr = Fsr_list(front); fsr != NULL; fsr = fsr->next)
            if (ComponentsMatch(fsr,s->comp,fsr->comp,front->interf))
        {
            set_flow_specified_volume_state(s, wave, front, fsr);
            return;
        }

        // contrl_vol_FD
        (*wave->_consv_vol_solver)(ic,blk,s,wave,newwave,
				   front,newfront,dt,dh); 
}


LOCAL void set_flow_specified_volume_state(
        CSG_Solid       *s,
        Wave            *wave,
        Front           *fr,
        FlowSpecifiedRegion  *fsr)
{
        CSG_Face        *f, *btm_f;
        int             where[3];

        f = s->sfaces;
        while(f)
        {
            if(YES == is_top_or_bottom_face(f,where))
            {
                if(where[0] == 0)
                {
                    btm_f = f;
                    break;
                }
            }
            f = f->nextf;
        }

        f = s->sfaces;
        while(f)
        {
            if(YES == is_top_or_bottom_face(f,where))
            {
                if(where[0] == 1)
                {
                    SetFlowSpecifiedState(f->i_stat,
                         btm_f->i_stat,f->centroid,s->comp,fsr,fr);
                }
            }
            f = f->nextf;
        }
}


LOCAL int buffer_boundary_volume(
        int             *ic,
        CSG_BLK_CRX     *blk,
        CSG_Solid       *s,
        Wave            *wave)
{
        CSG_Face        *f;
        float           nor[3], *fnor; 
        int             where[3]; 

        f = s->sfaces;
        while(f)
        {
            /*
            if(fabs(f->face_eqn[2]) >= 0.1*TOL)
            {
                f = f->nextf; 
                continue; 
            }
            */
            if(! is_interior_side_face(f))
            {
                f = f->nextf; 
                continue; 
            }
            if(face_on_buffer_boundary(f) == YES)
                return YES; 

            f = f->nextf; 
        } 
        return NO;  
}

LOCAL int is_trivial_top_vol(
        CSG_Solid *s)
{
        CSG_Face        *f;
        int             found_top = NO; 
        int             where[3];

        f = s->sfaces;
        while(f)
        {
            if(YES == long_is_top_or_bottom_face(f,where))
            {
                if(where[2] == 1)
                {
                    found_top = YES;
                    return NO; 
                }
            }
            f = f->nextf;
        }

        if(NO == found_top)
            return YES; 
}

/* The front face must only be composed of crxing points */
LOCAL int is_interior_side_face(
        CSG_Face  *f)
{
        CSG_HalfEdge  *lh;
        int           i;
        float         *dir, prod;
        int           all_crx = YES;
        float         fnor[2][3] = {0.0, 0.0, 1.0,
                                    0.0, 0.0, -1.0};

        lh = f->floops->ledg;
        do
        {
            if(lh->vtx->pt->type == IS_ICRDS)
            {
                all_crx = NO;
                break;
            }
        }while((lh = lh->nxt) != f->floops->ledg);

        if(all_crx == YES)
            return NO;

        for( i = 0; i < 2; i++)
        {
           dir = fnor[i];
           prod = scalar_product(dir,f->face_eqn,3);
           if(fabs(prod - 1.0) <  0.0001*TOL || 
              fabs(prod + 1.0) <  0.0001*TOL)
               return NO;
        }
        return YES;
}

/* This function only applies to the side space-time faces */
/* Also depends on some local globals */
LOCAL int face_on_buffer_boundary(
        CSG_Face *f)
{

        CSG_POINT     *pt;
        CSG_HalfEdge  *h;
        int           on_buf = NO; 

        h = f->floops->ledg;

        do
        {
            on_buf = NO; 
            pt = h->vtx->pt; 
            if(pt->type == IS_ICRDS)
            {
                if(lbuffed[0] == YES)
                {
                    if(pt->csg_pt.icrds[0] == ggmin[0])
                        on_buf = YES; 
                }
                if(ubuffed[0] == YES)
                {
                    if(pt->csg_pt.icrds[0] == ggmax[0])
                        on_buf = YES; 
                }
                if(lbuffed[1] == YES)
                {
                    if(pt->csg_pt.icrds[1] == ggmin[1])
                        on_buf = YES; 
                }
                if(ubuffed[1] == YES)
                {
                    if(pt->csg_pt.icrds[1] == ggmax[1])
                        on_buf = YES; 
                }
                if(on_buf == NO)
                    return NO;  
            }
            else
            {
                if(lbuffed[0] == YES )
                {
                    if(fabs(Coords(pt->csg_pt.crx->pt)[0] - VL[0]) < 0.00001*TOL)
                        on_buf = YES; 
                }
                if(ubuffed[0] == YES )
                {
                    if(fabs(Coords(pt->csg_pt.crx->pt)[0] - VU[0]) < 0.00001*TOL)
                        on_buf = YES; 
                }
                if(lbuffed[1] == YES )
                {
                    if(fabs(Coords(pt->csg_pt.crx->pt)[1] - VL[1]) < 0.00001*TOL)
                        on_buf = YES; 
                }
                if(ubuffed[1] == YES )
                {
                    if(fabs(Coords(pt->csg_pt.crx->pt)[1] - VU[1]) < 0.00001*TOL)
                        on_buf = YES; 
                }
                if(on_buf == NO)
                    return NO;  
            }      
        }while((h = h->nxt) != f->floops->ledg); 

        return YES; 
}
        
         

#endif /* #if defined(CONSERVATIVE_ALG) && defined(TWOD) */


