#ifdef MHD
/*
*				mstate.c:
*
*	Contains routines for the evaluation of magnetic fields and states
*
*/

#include <gdecs/gdecs.h>

LOCAL float m_rect_coil_dBZ_dz(COIL*,float*,float);
LOCAL void mhd_modify_front_state(Front*);
LOCAL void mhd_modify_interior_state(Wave*,Front*);

EXPORT	int	mhd_modify_state(
	float	dt,
	float	*dt_frac,
	Wave	*wave,
	Front	*front)
{
	mhd_modify_front_state(front);
	mhd_modify_interior_state(wave,front);
	return true;
}	/* end mhd_modify_state() */

EXPORT  void         mhd_modify_state(
	Locstate     state,
	float        *coords,
	float        time,
	float,
	float        dt)
{
	float   B[3], velocity[3], ME;
	int     i;
	float	*J = Current(state);
	float	*lf = LorentzForce(state);	
	
	for (i = 0; i < 3; i++) 
		velocity[i] = vel(i,state); 

	m_params.magnetic_field(&m_params,coords,time,B);	
	compute_J(coords,velocity,time,0,B,&m_params,J);
	Lorentz_force(J,B,m_params.c,lf);
	ME = magnetic_energy(J,B,velocity,&m_params);
	Energy(state) += dt*ME;
	for (i = 0; i < 3; i++) 
	    Mom(state)[i] += dt*lf[i];	   
	if(debugging("current"))
	{
	    print_vector("Velocity = ",3,velocity,"%10.3e");
	    print_vector("Magnetic field = ",3,B,"%10.3e");
	    print_vector("Lorentz force = ",3,lf,"%10.3e");
	    printf("Magnetic Energy = %g\n",ME);
	    printf("Total Energy = %g\n",Energy(state));		
	    print_vector("Momentum = ",3,Mom(state),"%10.3e");
	    fflush(stdout);
	}
} /* end mhd_modify_state() */


LOCAL	void	mhd_modify_interior_state(
	Wave		*wave,
	Front		*front)
{
	COMPONENT	comp;
	Locstate	state;
	float		*coords;
	int		icoords[MAXD];
	int		dim = wave->rect_grid->dim;
	int		status;

	switch (dim)
	{
#if defined(ONED)
	case 1:
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	{
	    int		ix, iy, iz;
	    int		xmax, ymax, zmax;

	    xmax = wave->rect_grid->gmax[0];
	    ymax = wave->rect_grid->gmax[1];
	    zmax = wave->rect_grid->gmax[2];
	    for (iz = 0; iz < zmax; iz++)
	    {
	    	icoords[2] = iz;
	    	for (iy = 0; iy < ymax; iy++)
	    	{
	    	    icoords[1] = iy;
	    	    for (ix = 0; ix < xmax; ix++)
	    	    {
	    	    	icoords[0] = ix;
	    	    	coords = Rect_coords(icoords,wave);
	    	    	comp = Rect_comp(icoords,wave);
	    	    	state = Rect_state(icoords,wave);
			if(comp==2)
			mhd_modify_state(state,coords,front->time,0.0,front->dt);
	    	    }
	    	}
	    }
	    break;
	}
#endif /* defined(THREED) */
	}

	debug_print("init","Left init_interior_states()\n");
}		/*end init_interior_states*/


LOCAL	void	mhd_modify_front_state(
	Front 	*front)
{
	INTERFACE 		*intfc = front->interf;
	HYPER_SURF		*hs;
	HYPER_SURF_ELEMENT 	*hse;
	POINT			*pt;
	Locstate		st;
	
	(void) next_point(intfc,NULL,NULL,NULL);
	while (next_point(intfc,&pt,&hse,&hs))
	{	
	    if (positive_component(hs)==2)
		    st=right_state(pt);
	    if (negative_component(hs)==2)
		    st=left_state(pt);		    
	    mhd_modify_state(st,Coords(pt),front->time,0.0,front->dt);
	}
} /* end mhd_modify_front_state() */


/* Calculates the Lorentz force: (J x B)/c  */

EXPORT	void	Lorentz_force(
	float   *J,
	float   *B,
	float   c,
	float   *L_force)
{
	float   Lf[3];
	int     i;

	(void) vector_product(J,B,Lf,3);
	for (i = 0; i < 3; i++)
	    L_force[i] = Lf[i]/c;
	return;
} /* end Lorentz_force() */ 


/*  Calculates the term in the energy equation corresponding 
 *  to the magnetic energy
 */

EXPORT	float		magnetic_energy(
	float		*J,
	float   	*B,
	float   	*vel,
	M_PARAMS	*m_params)
{
	float	ME;
	float	conduct = m_params->conduct;
       
//	float	c = m_params->c;
//	float	rel_vel[3];
//	for (int i = 0; i < 3; i++)
//		rel_vel[i] = vel[i] - m_params->coil_vel[i];
//	(void) vector_product(J,B,jxb,3);
//	(void) vector_product(rel_vel,B,uxb,3);
//	ME = - scalar_product(rel_vel,jxb,3)/c
//		- scalar_product(J,uxb,3)/c
//		+ scalar_product(J,J,3)/conduct;
	ME = Dot3d(J,J)/conduct;
	return ME;
} /* end magnetic_energy() */


/*    This function gives an analytical approximation for the
 *    azimutal current density distribution in a thin jet moving 
 *    along the axis of a magnetic field in z-direction
 */ 

EXPORT	void	analytical_J(
	float		*coords,
	float		*vel,
	float		time,
	float		dz,
	float		*B,
	M_PARAMS	*m_params,
	float		*J)
{
	float        BXz,BYz,BZz, j_theta,v_r;
	float        rr,theta;
	float        c = m_params->c;
	float        conduct = m_params->conduct;
	float        coords2[3],B2[3];

	coords2[0] = coords[0];
	coords2[1] = coords[1];
	coords2[2] = coords[2] + dz/10;
	rr = sqrt(sqr(coords[0]) + sqr(coords[1]));
	theta = acos(fabs(coords[0]/rr));
	if (coords[0] < 0.0  && coords[1] >= 0.0) theta = PI - theta;
	if (coords[0] < 0.0  && coords[1] < 0.0) theta = PI + theta;
	if (coords[0] >= 0.0 && coords[1] < 0.0) theta = 2.0*PI - theta;

	v_r = vel[0]*cos(theta) + vel[1]*sin(theta);
//      m_rect_coil_B(m_params->coil,coords2,time,B2);
	m_params->magnetic_field(m_params,coords2,time,B2);       

	BXz = (B2[0] - B[0])*10/dz;
	BYz = (B2[1] - B[1])*10/dz;
	BZz = (B2[2] - B[2])*10/dz;
       
	j_theta = - conduct*rr*(vel[2] - m_params->coil_vel[2])*BZz/(2*c) 
		- conduct*v_r*B[2]/c;

	J[0] = - j_theta*sin(theta);
	J[1] = j_theta*cos(theta);
	if (m_params->coil->alpha == 0)
		J[2] = 0;
	else
	    J[2] = - (coords[0]*(vel[2]- m_params->coil_vel[2])*conduct*BYz)/c
		  - (coords[1]*(vel[2]- m_params->coil_vel[2])*conduct*BXz)/c; 
      
	return;
} /* end analytical_J() */


// This will replace the above function.
// It has to be renamed when it is stable.

EXPORT	void	compute_J(
	float		*coords,
	float		*vel,
	float		time,
	float,
	float		*B,
	M_PARAMS	*m_params,
	float		*J)
{
	float c = m_params->c;
/*
	{
	    float DB[9];
	    m_params->magnetic_field_deriv_1(m_params,coords,time,DB);
	    J[0]=4.0*PI/c*(DB[7]-DB[5]);
	    J[1]=4.0*PI/c*(DB[2]-DB[6]);
	    J[2]=4.0*PI/c*(DB[3]-DB[1]);
	}
*/	
	float conduct = m_params->conduct;
	float u[3];
	u[0] = vel[0] - m_params->coil_vel[0];
	u[1] = vel[1] - m_params->coil_vel[1];
	u[2] = vel[2] - m_params->coil_vel[2];

	REAL uxB[3];
	vector_product(u,B,uxB,3);
	
	COMPONENT comp = 2;
	REAL E[3];

	compute_E(coords,&comp,E);

	J[0] = conduct/c*(E[0]+uxB[0]);
	J[1] = conduct/c*(E[1]+uxB[1]);
	J[2] = conduct/c*(E[2]+uxB[2]);
	if(debugging("current"))
	{
	    print_vector("u = ",3,u,"%10.3e");
	    print_vector("B = ",3,B,"%10.3e");
	    print_vector("uxB = ",3,uxB,"%10.3e");
	    print_vector("E = ",3,E,"%10.3e");
//	    printf("conductivity = %10.3e\n",m_params->conduct);
//	    printf("speed of light = %10.3e\n",m_params->c);
	}	    
	
	if(debugging("current"))
	{
	    print_vector("J = ",3,J,"%10.3e");
	}
}	/* end compute_J() */


/*   m_rect_coil_B()  evaluates components of the magnetic field 
 *   in the point (x,y,z) 
 *   created by a rectangular coil located in the point center[] and
 *   moving with velocity coil_vel[]  
 */  

EXPORT	void	m_rect_coil_B(
	const M_PARAMS	*m_params,
	const float	*coords,
	float		time,
	float		*B)
{
	COIL	*coil = m_params->coil;	
	float   xm, xp, ym, yp, zm, zp;
	float   xm2, xp2, ym2, yp2, zm2, zp2;
	float   rmmm, rmmp, rmpm, rmpp;
	float   rpmm, rpmp, rppm, rppp;
	float   coord_prime[3], B_prime[3];
	int     i;
	float  	Lx = coil->Lx;
	float  	Ly = coil->Ly;
	float  	Lz = coil->Lz;
	float   alpha = coil->alpha;
	float  	I = coil->I;

	coord_prime[0] = coords[0];
	coord_prime[1] = cos(alpha)*coords[1] + 
	    sin(alpha)*(coords[2] - coil->center[2] - m_params->coil_vel[2]*time); 
	coord_prime[2] =-sin(alpha)*coords[1] + 
	    cos(alpha)*(coords[2] - coil->center[2] - m_params->coil_vel[2]*time);

	xm = coord_prime[0]  - Lx;
	xp = coord_prime[0]  + Lx;
	ym = coord_prime[1]  - Ly;
	yp = coord_prime[1]  + Ly;
	zm = coord_prime[2]  - Lz;
	zp = coord_prime[2]  + Lz;

	xm2 = xm*xm;
	xp2 = xp*xp;
	ym2 = ym*ym;
	yp2 = yp*yp;
	zm2 = zm*zm;
	zp2 = zp*zp;
      
	rmmm = 1/sqrt(xm2 + ym2 + zm2);
	rmmp = 1/sqrt(xm2 + ym2 + zp2);
	rmpm = 1/sqrt(xm2 + yp2 + zm2);
	rmpp = 1/sqrt(xm2 + yp2 + zp2);
	rpmm = 1/sqrt(xp2 + ym2 + zm2);
	rpmp = 1/sqrt(xp2 + ym2 + zp2);
	rppm = 1/sqrt(xp2 + yp2 + zm2);
	rppp = 1/sqrt(xp2 + yp2 + zp2);

	B_prime[0] = I*(atanh(ym*rmmp) - atanh(ym*rmmm) -
			atanh(yp*rmpp) + atanh(yp*rmpm) -
			atanh(ym*rpmp) + atanh(ym*rpmm) +
			atanh(yp*rppp) - atanh(yp*rppm));

	B_prime[1] = I*(atanh(xm*rmmp) - atanh(xm*rmmm) -
			atanh(xm*rmpp) + atanh(xm*rmpm) -
			atanh(xp*rpmp) + atanh(xp*rpmm) +
			atanh(xp*rppp) - atanh(xp*rppm));

	B_prime[2] = I*(atan(zp/(xm*ym*rmmp)) - atan(zm/(xm*ym*rmmm)) -
			atan(zp/(xm*yp*rmpp)) + atan(zm/(xm*yp*rmpm)) -
			atan(zp/(xp*ym*rpmp)) + atan(zm/(xp*ym*rpmm)) +
			atan(zp/(xp*yp*rppp)) - atan(zm/(xp*yp*rppm)));

	B[0] = B_prime[0];
	B[1] = cos(alpha)*B_prime[1] - sin(alpha)*B_prime[2]; 
	B[2] = sin(alpha)*B_prime[1] + cos(alpha)*B_prime[2];

	return;
} /* end  m_rect_coil_B() */




/*   m_rect_coil_B_deriv_1()  evaluates components of the first 
 *   derivative of the external magnetic field in the point (x,y,z) 
 *   created by a rectangular coil located in the point center[] and
 *   moving with velocity coil_vel[]  
 */  

EXPORT	void		m_rect_coil_B_deriv_1(
	const M_PARAMS	*m_params,
	const float	*coords,
	float		time,
	float         	*DB)
{
	float	xm, xp, ym, yp, zm, zp;
	float   xm2, xp2, ym2, yp2, zm2, zp2;
	float   rmmm, rmmp, rmpm, rmpp;
	float   rpmm, rpmp, rppm, rppp;
	float	rmmm2, rmmp2, rmpm2, rmpp2;
	float   rpmm2, rpmp2, rppm2, rppp2;
	float   rmmm3, rmmp3, rmpm3, rmpp3;
	float   rpmm3, rpmp3, rppm3, rppp3;
	float	rxmm, rxmp, rxpm, rxpp;
	float	rymm, rymp, rypm, rypp;    
	int	i;
	const float *coil_vel=m_params->coil_vel; 
	const COIL  *coil  = m_params->coil;
	const float *center=coil->center;
	float	Lx = coil->Lx;
	float	Ly = coil->Ly;
	float	Lz = coil->Lz;
	float	I =  coil->I;

	xm = coords[0] - center[0] - coil_vel[0]*time - Lx;
	xp = coords[0] - center[0] - coil_vel[0]*time + Lx;
	ym = coords[1] - center[1] - coil_vel[1]*time - Ly;
	yp = coords[1] - center[1] - coil_vel[1]*time + Ly;
	zm = coords[2] - center[2] - coil_vel[2]*time - Lz;
	zp = coords[2] - center[2] - coil_vel[2]*time + Lz;
	xm2 = xm * xm;
	xp2 = xp * xp;
	ym2 = ym * ym;
	yp2 = yp * yp;
	zm2 = zm * zm;
	zp2 = zp * zp;
	rmmm2 = 1 / (xm2 + ym2 + zm2);
	rmmp2 = 1 / (xm2 + ym2 + zp2);
	rmpm2 = 1 / (xm2 + yp2 + zm2);
	rmpp2 = 1 / (xm2 + yp2 + zp2);
	rpmm2 = 1 / (xp2 + ym2 + zm2);
	rpmp2 = 1 / (xp2 + ym2 + zp2);
	rppm2 = 1 / (xp2 + yp2 + zm2);
	rppp2 = 1 / (xp2 + yp2 + zp2);
	rmmm = sqrt (rmmm2);
	rmmp = sqrt (rmmp2);
	rmpm = sqrt (rmpm2);
	rmpp = sqrt (rmpp2);
	rpmm = sqrt (rpmm2);
	rpmp = sqrt (rpmp2);
	rppm = sqrt (rppm2);
	rppp = sqrt (rppp2);
	rxmm = 1 / (xm2 + zm2);
	rxmp = 1 / (xm2 + zp2);
	rxpm = 1 / (xp2 + zm2);
	rxpp = 1 / (xp2 + zp2);
	rymm = 1 / (ym2 + zm2);
	rymp = 1 / (ym2 + zp2);
	rypm = 1 / (yp2 + zm2);
	rypp = 1 / (yp2 + zp2);
	rmmm3 = rmmm * rmmm2;
	rmmp3 = rmmp * rmmp2;
	rmpm3 = rmpm * rmpm2;
	rmpp3 = rmpp * rmpp2;
	rpmm3 = rpmm * rpmm2;
	rpmp3 = rpmp * rpmp2;
	rppm3 = rppm * rppm2;
	rppp3 = rppp * rppp2;

	/*BXx*/ DB[0] = I * (xm * ym * (rxmm * rmmm - rxmp * rmmp)
			     - xm * yp * (rxmm * rmpm - rxmp * rmpp)
			     - xp * ym * (rxpm * rpmm - rxpp * rpmp)
			     + xp * yp * (rxpm * rppm - rxpp * rppp));
	/*BXy*/ DB[1] = I * (- rmmm + rmmp + rmpm - rmpp
			     + rpmm - rpmp - rppm + rppp);
	/*BXz*/ DB[2] = I * (ym * zm * (rxmm * rmmm - rxpm * rpmm)
			     - ym * zp * (rxmp * rmmp - rxpp * rpmp)
			     - yp * zm * (rxmm * rmpm - rxpm * rppm)
			     + yp * zp * (rxmp * rmpp - rxpp * rppp));

	/*BYx*/ DB[3] = I * (- rmmm + rmmp + rmpm - rmpp
			     + rpmm - rpmp - rppm + rppp);
	/*BYy*/ DB[4] = I * (xm * ym * (rymm * rmmm - rymp * rmmp)
			     - xm * yp * (rypm * rmpm - rypp * rmpp)
			     - xp * ym * (rymm * rpmm - rymp * rpmp)
			     + xp * yp * (rypm * rppm - rypp * rppp));
	/*BYz*/ DB[5] = I * (xm * zm * (rymm * rmmm - rypm * rmpm)
			     - xm * zp * (rymp * rmmp - rypp * rmpp)
			     - xp * zm * (rymm * rpmm - rypm * rppm)
			     + xp * zp * (rymp * rpmp - rypp * rppp));

	/*BZx*/ DB[6] = I * (ym * zm * (rxmm * rmmm - rxpm * rpmm)
			     - ym * zp * (rxmp * rmmp - rxpp * rpmp)
			     - yp * zm * (rxmm * rmpm - rxpm * rppm)
			     + yp * zp * (rxmp * rmpp - rxpp * rppp));
	/*BZy*/ DB[7] = I * (xm * zm * (rymm * rmmm - rypm * rmpm)
			     - xm * zp * (rymp * rmmp - rypp * rmpp)
			     - xp * zm * (rymm * rpmm - rypm * rppm)
			     + xp * zp * (rymp * rpmp - rypp * rppp));
	/*BZz*/ DB[8] = I *
		(xm * ym * ((rxmp + rymp) * rmmp - (rxmm + rymm) * rmmm)
		 - xm * yp * ((rxmp + rypp) * rmpp - (rxmm + rypm) * rmpm)   
		 - xp * ym * ((rxpp + rymp) * rpmp - (rxpm + rymm) * rpmm)
		 + xp * yp * ((rxpp + rypp) * rppp - (rxpm + rypm) * rppm));


	return;
} /* end m_rect_coil_B_deriv_1() */



/*   m_rect_coil_dBZ_dz()  evaluates \partial B_z\partial z
 *   of the external magnetic field in the point (x,y,z) 
 *   created by a rectangular coil located in the point center[] and
 *   moving with velocity coil_vel[]  
 */  

LOCAL	float	m_rect_coil_dBZ_dz(
	COIL	*coil,
	float	*coords,
	float	time)
{
	float	xm, xp, ym, yp, zm, zp;
	float   xm2, xp2, ym2, yp2, zm2, zp2;
	float	rmmm, rmmp, rmpm, rmpp;
	float   rpmm, rpmp, rppm, rppp;
	float   rmmm2, rmmp2, rmpm2, rmpp2;
	float   rpmm2, rpmp2, rppm2, rppp2;
	float   rmmm3, rmmp3, rmpm3, rmpp3;
	float   rpmm3, rpmp3, rppm3, rppp3;
	float   rxmm, rxmp, rxpm, rxpp;
	float   rymm, rymp, rypm, rypp;    
	int     i;
	float  	Lx = coil->Lx;
	float  	Ly = coil->Ly;
	float  	Lz = coil->Lz;
	float  	I = coil->I;
	float    coil_vel[3]; 
	float    center[3];
	float    BZz;

	for (i = 0; i < 3; i++)
	{
	    coil_vel[i] = m_params.coil_vel[i];
	    center[i] = coil->center[i];
	}
	xm = coords[0] - center[0] - coil_vel[0]*time - Lx;
	xp = coords[0] - center[0] - coil_vel[0]*time + Lx;
	ym = coords[1] - center[1] - coil_vel[1]*time - Ly;
	yp = coords[1] - center[1] - coil_vel[1]*time + Ly;
	zm = coords[2] - center[2] - coil_vel[2]*time - Lz;
	zp = coords[2] - center[2] - coil_vel[2]*time + Lz;
	xm2 = xm * xm;
	xp2 = xp * xp;
	ym2 = ym * ym;
	yp2 = yp * yp;
	zm2 = zm * zm;
	zp2 = zp * zp;
	rmmm2 = 1 / (xm2 + ym2 + zm2);
	rmmp2 = 1 / (xm2 + ym2 + zp2);
	rmpm2 = 1 / (xm2 + yp2 + zm2);
	rmpp2 = 1 / (xm2 + yp2 + zp2);
	rpmm2 = 1 / (xp2 + ym2 + zm2);
	rpmp2 = 1 / (xp2 + ym2 + zp2);
	rppm2 = 1 / (xp2 + yp2 + zm2);
	rppp2 = 1 / (xp2 + yp2 + zp2);
	rmmm = sqrt (rmmm2);
	rmmp = sqrt (rmmp2);
	rmpm = sqrt (rmpm2);
	rmpp = sqrt (rmpp2);
	rpmm = sqrt (rpmm2);
	rpmp = sqrt (rpmp2);
	rppm = sqrt (rppm2);
	rppp = sqrt (rppp2);
	rxmm = 1 / (xm2 + zm2);
	rxmp = 1 / (xm2 + zp2);
	rxpm = 1 / (xp2 + zm2);
	rxpp = 1 / (xp2 + zp2);
	rymm = 1 / (ym2 + zm2);
	rymp = 1 / (ym2 + zp2);
	rypm = 1 / (yp2 + zm2);
	rypp = 1 / (yp2 + zp2);
	rmmm3 = rmmm * rmmm2;
	rmmp3 = rmmp * rmmp2;
	rmpm3 = rmpm * rmpm2;
	rmpp3 = rmpp * rmpp2;
	rpmm3 = rpmm * rpmm2;
	rpmp3 = rpmp * rpmp2;
	rppm3 = rppm * rppm2;
	rppp3 = rppp * rppp2;

	BZz = I * (xm * ym * ((rxmp + rymp) * rmmp - (rxmm + rymm) * rmmm)
		- xm * yp * ((rxmp + rypp) * rmpp - (rxmm + rypm) * rmpm)   
		- xp * ym * ((rxpp + rymp) * rpmp - (rxpm + rymm) * rpmm)
		+ xp * yp * ((rxpp + rypp) * rppp - (rxpm + rypm) * rppm));

	return BZz;
} /* end m_rect_coil_dBZ_dz() */


/*   m_rect_coil_B_deriv_2()  evaluates components of the second 
 *   derivative of the external magnetic field in the point (x,y,z) 
 *   created by a rectangular coil located in the point center[] and
 *   moving with velocity coil_vel[]  
 */  

EXPORT	void	m_rect_coil_B_deriv_2(
	const M_PARAMS	*m_params,
	float         *coords,
	float         time,
	float         *DB)
{
	float         xm, xp, ym, yp, zm, zp;
	float         xm2, xp2, ym2, yp2, zm2, zp2;
	float         rmmm, rmmp, rmpm, rmpp;
	float         rpmm, rpmp, rppm, rppp;
	float         rmmm2, rmmp2, rmpm2, rmpp2;
	float         rpmm2, rpmp2, rppm2, rppp2;
	float         rmmm3, rmmp3, rmpm3, rmpp3;
	float         rpmm3, rpmp3, rppm3, rppp3;
	float         rxmm, rxmp, rxpm, rxpp;
	float         rymm, rymp, rypm, rypp;    
	int           i;
	const float *coil_vel = m_params->coil_vel;
	const COIL  *coil=m_params->coil;
	const float *center = coil->center;
	float  	Lx = coil->Lx;
	float  	Ly = coil->Ly;
	float  	Lz = coil->Lz;
	float  	I = coil->I;

	for (i = 0; i < 3; i++)
	{
	}
	xm = coords[0] - center[0] - coil_vel[0]*time - Lx;
	xp = coords[0] - center[0] - coil_vel[0]*time + Lx;
	ym = coords[1] - center[1] - coil_vel[1]*time - Ly;
	yp = coords[1] - center[1] - coil_vel[1]*time + Ly;
	zm = coords[2] - center[2] - coil_vel[2]*time - Lz;
	zp = coords[2] - center[2] - coil_vel[2]*time + Lz;
	xm2 = xm * xm;
	xp2 = xp * xp;
	ym2 = ym * ym;
	yp2 = yp * yp;
	zm2 = zm * zm;
	zp2 = zp * zp;
	rmmm2 = 1 / (xm2 + ym2 + zm2);
	rmmp2 = 1 / (xm2 + ym2 + zp2);
	rmpm2 = 1 / (xm2 + yp2 + zm2);
	rmpp2 = 1 / (xm2 + yp2 + zp2);
	rpmm2 = 1 / (xp2 + ym2 + zm2);
	rpmp2 = 1 / (xp2 + ym2 + zp2);
	rppm2 = 1 / (xp2 + yp2 + zm2);
	rppp2 = 1 / (xp2 + yp2 + zp2);
	rmmm = sqrt (rmmm2);
	rmmp = sqrt (rmmp2);
	rmpm = sqrt (rmpm2);
	rmpp = sqrt (rmpp2);
	rpmm = sqrt (rpmm2);
	rpmp = sqrt (rpmp2);
	rppm = sqrt (rppm2);
	rppp = sqrt (rppp2);
	rxmm = 1 / (xm2 + zm2);
	rxmp = 1 / (xm2 + zp2);
	rxpm = 1 / (xp2 + zm2);
	rxpp = 1 / (xp2 + zp2);
	rymm = 1 / (ym2 + zm2);
	rymp = 1 / (ym2 + zp2);
	rypm = 1 / (yp2 + zm2);
	rypp = 1 / (yp2 + zp2);
	rmmm3 = rmmm * rmmm2;
	rmmp3 = rmmp * rmmp2;
	rmpm3 = rmpm * rmpm2;
	rmpp3 = rmpp * rmpp2;
	rpmm3 = rpmm * rpmm2;
	rpmp3 = rpmp * rpmp2;
	rppm3 = rppm * rppm2;
	rppp3 = rppp * rppp2;

/*BXxx*/DB[0] = I * (ym * rxmm * rmmm * (1 - xm2 * (2 * rxmm + rmmm2))
		     - ym * rxmp * rmmp * (1 - xm2 * (2 * rxmp + rmmp2))
		     - yp * rxmm * rmpm * (1 - xm2 * (2 * rxmm + rmpm2))
		     + yp * rxmp * rmpp * (1 - xm2 * (2 * rxmp + rmpp2)) 
		     - ym * rxpm * rpmm * (1 - xp2 * (2 * rxpm + rpmm2))
		     + ym * rxpp * rpmp * (1 - xp2 * (2 * rxpp + rpmp2))
		     + yp * rxpm * rppm * (1 - xp2 * (2 * rxpm + rppm2))
		     - yp * rxpp * rppp * (1 - xp2 * (2 * rxpp + rppp2)))/2;
/*BXxy*/DB[1] = I * (xm * (rmmm3 - rmmp3 - rmpm3 + rmpp3)
		     - xp * (rpmm3 - rpmp3 - rppm3 + rppp3));
/*BXxz*/DB[2] = I * (- xm * ym * zm * rxmm * rmmm * (2 * rxmm + rmmm2)
                     + xm * ym * zp * rxmp * rmmp * (2 * rxmp + rmmp2) 
                     + xm * yp * zm * rxmm * rmpm * (2 * rxmm + rmpm2)
                     - xm * yp * zp * rxmp * rmpp * (2 * rxmp + rmpp2)
                     + xp * ym * zm * rxpm * rpmm * (2 * rxpm + rpmm2)
                     - xp * ym * zp * rxpp * rpmp * (2 * rxpp + rpmp2)
                     - xp * yp * zm * rxpm * rppm * (2 * rxpm + rppm2) 
                     + xp * yp * zp * rxpp * rppp * (2 * rxpp + rppp2));
/*BXyy*/DB[3] = I * (ym * (rmmm3 - rmmp3 - rpmm3 + rpmp3)
		     - yp * (rmpm3 - rmpp3 - rppm3 + rppp3))/2;
/*BXyz*/DB[4] = I * (zm * (rmmm3 - rmpm3 - rpmm3 + rppm3)
		     - zp * (rmmp3 - rmpp3 - rpmp3 + rppp3));
/*BXzz*/DB[5] = I * (ym * rxmm * rmmm * (1 - zm2 * (2 * rxmm + rmmm2))
		     - ym * rxmp * rmmp * (1 - zp2 * (2 * rxmp + rmmp2))
		     - yp * rxmm * rmpm * (1 - zm2 * (2 * rxmm + rmpm2))
		     + yp * rxmp * rmpp * (1 - zp2 * (2 * rxmp + rmpp2))
		     - ym * rxpm * rpmm * (1 - zm2 * (2 * rxpm + rpmm2))
		     + ym * rxpp * rpmp * (1 - zp2 * (2 * rxpp + rpmp2))
		     + yp * rxpm * rppm * (1 - zm2 * (2 * rxpm + rppm2))
		     - yp * rxpp * rppp * (1 - zp2 * (2 * rxpp + rppp2)))/2;
/*BYxx*/DB[6] = I * (xm * (rmmm3 - rmmp3 - rmpm3 + rmpp3)
		     - xp * (rpmm3 - rpmp3 - rppm3 + rppp3))/2;
/*BYxy*/DB[7] = I * (ym * (rmmm3 - rmmp3 - rpmm3 + rpmp3)
		     - yp * (rmpm3 - rmpp3 - rppm3 + rppp3));
/*BYxz*/DB[8] = I * (zm * (rmmm3 - rmpm3 - rpmm3 + rppm3)
		     - zp * (rmmp3 - rmpp3 - rpmp3 + rppp3));
/*BYyy*/DB[9] = I * (xm * rymm * rmmm * (1 - ym2 * (2 * rymm + rmmm2))
		     - xm * rymp * rmmp * (1 - ym2 * (2 * rymp + rmmp2))
		     - xm * rypm * rmpm * (1 - yp2 * (2 * rypm + rmpm2))
		     + xm * rypp * rmpp * (1 - yp2 * (2 * rypp + rmpp2)) 
		     - xp * rymm * rpmm * (1 - ym2 * (2 * rymm + rpmm2))
		     + xp * rymp * rpmp * (1 - ym2 * (2 * rymp + rpmp2))
		     + xp * rypm * rppm * (1 - yp2 * (2 * rypm + rppm2))
		     - xp * rypp * rppp * (1 - yp2 * (2 * rypp + rppp2)))/2;
/*BYyz*/ DB[10] = I * (- xm * ym * zm * rymm * rmmm * (2 * rymm + rmmm2)
                     + xm * ym * zp * rymp * rmmp * (2 * rymp + rmmp2)
                     + xm * yp * zm * rypm * rmpm * (2 * rypm + rmpm2)
                     - xm * yp * zp * rypp * rmpp * (2 * rypp + rmpp2)
                     + xp * ym * zm * rymm * rpmm * (2 * rymm + rpmm2)
                     - xp * ym * zp * rymp * rpmp * (2 * rymp + rpmp2)
                     - xp * yp * zm * rypm * rppm * (2 * rypm + rppm2) 
                     + xp * yp * zp * rypp * rppp * (2 * rypp + rppp2));
/*BYzz*/ DB[11] = I * (xm * rymm * rmmm * (1 - zm2 * (2 * rymm + rmmm2))
		       - xm * rymp * rmmp * (1 - zp2 * (2 * rymp + rmmp2))
		       - xm * rypm * rmpm * (1 - zm2 * (2 * rypm + rmpm2))
		       + xm * rypp * rmpp * (1 - zp2 * (2 * rypp + rmpp2)) 
		       - xp * rymm * rpmm * (1 - zm2 * (2 * rymm + rpmm2))
		       + xp * rymp * rpmp * (1 - zp2 * (2 * rymp + rpmp2))
		       + xp * rypm * rppm * (1 - zm2 * (2 * rypm + rppm2))
		       - xp * rypp * rppp * (1 - zp2 * (2 * rypp + rppp2)))/2;

/* 
another BZxx:  DB[12] = I * (- xm * ym * zm * rxmm * rmmm * (2 * rxmm + rmmm2)
                     + xm * ym * zp * rxmp * rmmp * (2 * rxmp + rmmp2) 
                     + xm * yp * zm * rxmm * rmpm * (2 * rxmm + rmpm2)
                     - xm * yp * zp * rxmp * rmpp * (2 * rxmp + rmpp2)
                     + xp * ym * zm * rxpm * rpmm * (2 * rxpm + rpmm2)
                     - xp * ym * zp * rxpp * rpmp * (2 * rxpp + rpmp2)
                     - xp * yp * zm * rxpm * rppm * (2 * rxpm + rppm2)
                     + xp * yp * zp * rxpp * rppp * (2 * rxpp + rppp2))/2;  
*/
/*BZxx*/DB[12] = I * (- xm * ym * zm * rxmm * rmmm * (2 * rxmm + rmmm2)
		      + xm * ym * zp * rxmp * rmmp * (2 * rxmp + rmmp2)
		      + xm * yp * zm * rxpm * rmpm * (2 * rxpm + rmpm2)
		      - xm * yp * zp * rxpp * rmpp * (2 * rxpp + rmpp2)
		      + xp * ym * zm * rxmm * rpmm * (2 * rxmm + rpmm2)
		      - xp * ym * zp * rxmp * rpmp * (2 * rxmp + rpmp2)
		      - xp * yp * zm * rxpm * rppm * (2 * rxpm + rppm2) 
		      + xp * yp * zp * rxpp * rppp * (2 * rxpp + rppp2))/2;  
/*BZxy*/DB[13] = I * (zm * (rmmm3 - rmpm3 - rpmm3 + rppm3)
		      - zp * (rmmp3 - rmpp3 - rpmp3 + rppp3));
/*BZxz*/DB[14] = I * (ym * rxmm * rmmm * (1 - zm2 * (2 * rxmm + rmmm2))
		      - ym * rxmp * rmmp * (1 - zp2 * (2 * rxmp + rmmp2))
		      - yp * rxmm * rmpm * (1 - zm2 * (2 * rxmm + rmpm2))
		      + yp * rxmp * rmpp * (1 - zp2 * (2 * rxmp + rmpp2)) 
		      - ym * rxpm * rpmm * (1 - zm2 * (2 * rxpm + rpmm2))
		      + ym * rxpp * rpmp * (1 - zp2 * (2 * rxpp + rpmp2))
		      + yp * rxpm * rppm * (1 - zm2 * (2 * rxpm + rppm2))
		      - yp * rxpp * rppp * (1 - zp2 * (2 * rxpp + rppp2)));
/*BZyy*/DB[15] = I * (- xm * ym * zm * rymm * rmmm * (2 * rymm + rmmm2)
		      + xm * ym * zp * rymp * rmmp * (2 * rymp + rmmp2) 
		      + xm * yp * zm * rypm * rmpm * (2 * rypm + rmpm2)
		      - xm * yp * zp * rypp * rmpp * (2 * rypp + rmpp2)
		      + xp * ym * zm * rymm * rpmm * (2 * rymm + rpmm2)
		      - xp * ym * zp * rymp * rpmp * (2 * rymp + rpmp2)
		      - xp * yp * zm * rypm * rppm * (2 * rypm + rppm2) 
		      + xp * yp * zp * rypp * rppp * (2 * rypp + rppp2))/2;  
/*BZyz*/DB[16] = I * (xm * rymm * rmmm * (1 - zm2 * (2 * rymm + rmmm2))
		      - xm * rymp * rmmp * (1 - zp2 * (2 * rymp + rmmp2))
		      - xm * rypm * rmpm * (1 - zm2 * (2 * rypm + rmpm2))
		      + xm * rypp * rmpp * (1 - zp2 * (2 * rypp + rmpp2)) 
		      - xp * rymm * rpmm * (1 - zm2 * (2 * rymm + rpmm2))
		      + xp * rymp * rpmp * (1 - zp2 * (2 * rymp + rpmp2))
		      + xp * rypm * rppm * (1 - zm2 * (2 * rypm + rppm2))
		      - xp * rypp * rppp * (1 - zp2 * (2 * rypp + rppp2)));
/*BZzz*/DB[17] = I * (xm * ym * zm * rmmm * (2 * (rxmm * rxmm + rymm * rymm)
	          + (rxmm + rymm) * rmmm2)
                   - xm * ym * zp * rmmp * (2 * (rxmp * rxmp + rymp * rymp)
                  + (rxmp + rymp) * rmmp2)
                   - xm * yp * zm * rmpm * (2 * (rxmm * rxmm + rypm * rypm)
                  + (rxmm + rypm) * rmpm2)
                   + xm * yp * zp * rmpp * (2 * (rxmp * rxmp + rypp * rypp)
                  + (rxmp + rypp) * rmpp2)
                   - xp * ym * zm * rpmm * (2 * (rxpm * rmpm + rymm * rymm)
                  + (rxpm + rymm) * rpmm2)
                   + xp * ym * zp * rpmp * (2 * (rxpp * rxpp + rymp * rymp)
                  + (rxpp + rymp) * rpmp2)
                   + xp * yp * zm * rppm * (2 * (rxpm * rxpm + rypm * rypm)
                  + (rxpm + rypm) * rppm2)
                   - xp * yp * zp * rppp * (2 * (rxpp * rxpp + rypp * rypp) 
                  + (rxpp + rypp) * rppp2))/2;

        return;
}/* end  m_rect_coil_B_deriv_2() */


inline void FUNCTION_NOT_IMPLEMENTED(const char *fname,const char*arch)
{
	screen("ERROR: %s() is not implemented for %s\n",fname,arch);
	clean_up(ERROR);
}

inline void FUNCTION_NOT_IMPLEMENTED(const char *fname)
{
	screen("ERROR: %s() is not implemented.\n",fname);
	clean_up(ERROR);
}


inline void FUNCTION_TESTING(const char *fname)
{
	screen("%s() is being tested.\n",fname);
}

EXPORT	void	m_transverse_B(
	const M_PARAMS	*m_params,
	const float     *coords,
	float		time,
	float		*B)
{
	float z = coords[2]-m_params->coil_vel[2]*time;
	
	B[0] = 0.0;
	B[1] = m_params->Bmax*sqrt((1.0 - tanh((z - 1.5)/0.62))/2);
	B[2] = 0.0;
	return;
}



/*   m_transvere_B_deriv_1()  evaluates components of the first 
 *   derivative of the external magnetic field in the point (x,y,z) 
 *   created by a rectangular coil located in the point center[] and
 *   moving with velocity coil_vel[]  
 */  

EXPORT	void	m_transverse_B_deriv_1(
	const M_PARAMS	*m_params,
	const float     *coords,
	float		time,
	float		*DB)
{

//	B[0] = 0.0;
//	B[1] = m_params->Bmax*sqrt((1.0 - tanh((coords[2] - 1.5)/0.62))/2);
//	B[2] = 0.0;

	float z=coords[2]-m_params->coil_vel[2]*time;
	float arg = (z-1.5)/0.62;
	
	DB[0]=0.0;	DB[1]=0.0;	DB[2]=0.0;
	DB[3]=0.0;	DB[4]=0.0;
	DB[5]=-m_params->Bmax/(4.0*0.62*sqrt((1.0 - tanh(arg))/2.0)*cosh(arg)*cosh(arg));
	DB[6]=0.0;	DB[7]=0.0;	DB[8]=0.0;
	return;
}

#endif
