/*
*				gbstate.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains routine for the evaluation of boundary states, in particular
*	the evaluation of states to be used in the imposition
*	of boundary conditions.
*
*/


#include <gdecs/gdecs.h>

/* LOCAL Function prototypes */
LOCAL	void	nearest_state_and_normal_on_hypersurface(float*,float*,
				 COMPONENT,HYPER_SURF*,Locstate,float*,Front*);
LOCAL   void    connect_by_riemann_wave(Locstate, const Locstate, int, const float*);
/*
*			flow_through_boundary_state():
*	
*	Assigns the boundary state at the exterior point coords to be the state
*	at the nearest interface point to coords..
*
*/

/*ARGSUSED*/
EXPORT	void flow_through_boundary_state(
	float		*coords,
	HYPER_SURF	*hs,
	Front		*front,
	POINTER		p2wave,
	Locstate	state)
{
	INTERFACE	*intfc = hs->interface;
	COMPONENT	ext_comp = exterior_component(intfc);
	HYPER_SURF	*hs_on = NULL;
	HYPER_SURF_ELEMENT *hse_on = NULL;
	bool pc_excluded = is_excluded_comp(positive_component(hs),intfc); 
	bool nc_excluded = is_excluded_comp(negative_component(hs),intfc); 
	float		coords_on[MAXD];
	float		t[MAXD];

	debug_print("flow_through","Entered flow_through_boundary_state()\n");

        if(!debugging("old_flowthrough"))
        {
            // Assigns the boundary state at the exterior point coords
            // to be the state at the nearest interior block when it is
            // not an abstacle state

            const RECT_GRID *rgr = ((Wave*)p2wave)->rect_grid;
            int i;
            int icoords[MAXD];
            rect_in_which(coords,icoords,rgr);
            for(i=0;i<rgr->dim;++i)
            {
                if (icoords[i] >= rgr->gmax[i])
                    icoords[i] = rgr->gmax[i]-1;
                if (icoords[i] < 0)
                    icoords[i] = 0;
            }
            if (!is_obstacle_state(Rect_state(icoords,(Wave*)p2wave)))
            {
                copy_state(state,Rect_state(icoords,(Wave*)p2wave));
                return;
            }
            // obstacle_state(front->interf,state,
            //         size_of_state(front->interf));
        }

	if (pc_excluded && !nc_excluded)
	    ext_comp = positive_component(hs);
	else if (!pc_excluded && nc_excluded)
	    ext_comp = negative_component(hs);
	else
	    ext_comp = exterior_component(intfc);

	if (intfc->modified)
	{
	    HYPER_SURF *chs = NULL;
	    if ((intfc != front->interf) && (!front->interf->modified))
	    {
		chs = find_correspond_hyper_surface(hs,NULL,NULL,front,
			                            front->interf);
		if (chs != NULL) 
		{ 
		    if (!nearest_interface_point(coords,ext_comp,
		                    front->interf,INCLUDE_BOUNDARIES,
				    chs,coords_on,t,&hse_on,&hs_on))
		    	chs = NULL;
		    else if (!equivalent_comps(positive_component(hs),
		                          positive_component(chs),intfc) ||
		        !equivalent_comps(negative_component(hs),
		                          negative_component(chs),intfc))
			chs = NULL;
		}
	    }
	    if ((chs == NULL) && !long_nearest_interface_point(coords,ext_comp,
			                     intfc,INCLUDE_BOUNDARIES,hs,
					     coords_on,t,&hse_on,&hs_on))
	    {
	    	screen("ERROR in flow_through_boundary_state(), "
	    	       "long_nearest_interface_point() failed\n");
	    	clean_up(ERROR);
	    }
	}
	else 
	{
	    if (!nearest_interface_point(coords,ext_comp,intfc,
					INCLUDE_BOUNDARIES,hs,coords_on,
					t,&hse_on,&hs_on))
	    {
	    	screen("ERROR in flow_through_boundary_state(), "
	    	       "nearest_interface_point() failed\n");
	    	clean_up(ERROR);
	    }
	}
	if (is_excluded_comp(negative_component(hs),intfc))
	    state_along_hypersurface_element(positive_component(hs),
	    				     t,hse_on,hs_on,state);
	else if (is_excluded_comp(positive_component(hs),intfc))
	    state_along_hypersurface_element(negative_component(hs),
					     t,hse_on,hs_on,state);
	else 
	{
	    screen("ERROR: in flow_through_boundary_state\n"
	           "No boundary component on boundary\n");
	    clean_up(ERROR);
	}

	if (is_gravity() == YES)
	{
	    int    i, dim = front->rect_grid->dim;
	    float  v[MAXD], d, g, c, u, tmp, dens;
	    const float  eps = 10.0*MACH_EPS; /*TOLERANCE*/

	    for (i = 0; i < dim; ++i)
		v[i] = coords[i] - coords_on[i];
	    d = mag_vector(v,dim);
	    g = scalar_product(gravity(coords,front->time),v,dim);
	    dens = Dens(state);
	    u = scalar_product(Mom(state),v,dim)/Dens(state);
	    c = d*sound_speed(state);
	    tmp = c*c - u*u;
	    if (fabs(tmp) < eps)
	    {
	    	(void) printf("WARNING in flow_through_boundary_state(), "
			      "velocity equals sound speed.\n");
	    	tmp = tmp > 0 ? eps : -eps;
	    }
	    dens += 3.0*d*d*dens*g/tmp;
	    for (i = 0; i < dim; ++i)
	    	v[i] = vel(i,state);
	    state_on_adiabat_with_dens(state,dens,state,GAS_STATE);
	    add_velocity_to_state(state,v);
	}

	if (debugging("flow_through"))
	{
	    (void) printf("State returned by flow_through_boundary_state()\n");
	    fprint_raw_gas_data(stdout,state,front->rect_grid->dim);
	}

	debug_print("flow_through","Left flow_through_boundary_state()\n");
}		/*end flow_through_boundary_state*/

/*
*			constant_pressure_flow_through_boundary_state():
*	
*	Assigns the boundary state at the exterior point coords to be the state
*	at the nearest interface point to coords..
*
*/

/*ARGSUSED*/
EXPORT	void constant_pressure_flow_through_boundary_state(
	float		*coords,
	HYPER_SURF	*hs,
	Front		*front,
	POINTER		p2wave,
	Locstate	state)
{
	Locstate bst = boundary_state(hs);
	int      st_type;

	debug_print("flow_through",
		"Entered constant_pressure_flow_through_boundary_state()\n");
        if(debugging("old_flowthrough"))
        {
            flow_through_boundary_state(coords,hs,front,p2wave,state);
            st_type = state_type(state);
            set_state(state,TGAS_STATE,state);
            Dens(state) = Dens(bst);
            Press(state) = pressure(bst);
        }
        else
        {
            int dim = front->rect_grid->dim;
            static Locstate tmp=0;
            float nor[MAXD];
            CURVE *c;
            if(dim!=2)
            {
                printf("Not impelemented in %s(%d).\n",__FILE__,__LINE__);
                clean_up(ERROR);
            }
            if(!tmp)
                scalar(&tmp,size_of_state(front->interf));
            flow_through_boundary_state(coords,hs,front,p2wave,tmp);
            st_type = state_type(tmp);
            Set_params(state,tmp);
            set_type_of_state(state,TGAS_STATE);
            set_state(state,TGAS_STATE,state);
            Dens(state) = density(bst);
            Press(state) = pressure(bst);
//          fprintf(stdout,"FP: Nearest interior state\n");
//          fprint_raw_gas_data(stdout,tmp,front->rect_grid->dim);
//          fflush(stdout);


            nor[MAXD];
            c = Curve_of_hs(hs);
            if (is_bdry(c)) /*Rectangular boundary*/
            {
                float t[MAXD], length;
                int i;
                for (i = 0; i < dim; ++i)
                    t[i] = Coords(c->end->posn)[i] - Coords(c->start->posn)[i];
                length = mag_vector(t,dim);
                for (i = 0; i < dim; ++i)
                    t[i] /= length;
                nor[0] = t[1];
                nor[1] = -t[0];
                // should be (0,-1,0) for jet run
            }
            else
            {
                printf("Not impelemented in %s(%d).\n",__FILE__,__LINE__);
                clean_up(ERROR);
            }

            connect_by_riemann_wave(state,tmp,2,nor);
//          fprintf(stdout,"FP: Updated state\n");
//          fprint_raw_gas_data(stdout,state,front->rect_grid->dim);
//          fflush(stdout);
        }
        set_state(state,st_type,state);
        if (debugging("flow_through"))
        {
            (void) printf("State returned by ");
            (void) printf("constant_pressure_flow_through_boundary_state()\n");
            fprint_raw_gas_data(stdout,state,front->rect_grid->dim);
        }
	debug_print("flow_through",
		"Left constant_pressure_flow_through_boundary_state()\n");
}		/*end constant_pressure_flow_through_boundary_state*/

/*
*			time_dep_pressure_flow_through_boundary_state():
*	
*	Assigns the boundary state at the exterior point coords to be the state
*	at the nearest interface point to coords..
*
*/

/*ARGSUSED*/
/*	
 *	 time_dep_pressure_flow_through_boundary_state();
 *
 *      Return state whose pressure at the time:
 *
 *        __________         peak_pressure (pr2)
 *       /          \
 *      /            \
 *      --------------       base_pressure (pr1)
 *     0 t1        t2 t3
 *
 *     tw - 0  = time for warming-up   (tw)
 *     tp - tw = time for peak         (tp)
 *     tc - tp = time for cooling-down (tc)
 *
 *
 * ==========================================================
 *
 *     The above scheme is generalized:
 *      fd_data->time[] stores time when the pressure is recorded
 *      fd_data->pressure[] stores the pressure at corresponding time
 *      fd_data->size is the size of the arrays
 *
 *      above pressure ramp corresponds size=4
*/

EXPORT	void time_dep_pressure_flow_through_boundary_state(
	float		*coords,
	HYPER_SURF	*hs,
	Front		*front,
	POINTER		p2wave,
	Locstate	state)
{
	int      st_type;
	FD_DATA *fd_data = (FD_DATA *)boundary_state_data(hs);
	float	tr = fd_data->tr;
	float	tp = fd_data->tp;
	float	ts = fd_data->ts;
	float	pr_a = fd_data->pr_a;
	float	pr_p = fd_data->pr_p;
	float	time = front->time;
        float   p;

	debug_print("flow_through",
		"Entered time_dep_pressure_flow_through_boundary_state()\n");
        if (time < tr)  p = (pr_p-pr_a)/tr*time + pr_a;
        else if (time <= tr+tp) p = pr_p;
        else if (time < tr+tp+ts) p = -(pr_p-pr_a)/ts *(time-tr-tp) + pr_p;
        else p = pr_a;

        st_type = state_type(state);
        if(debugging("old_flowthrough"))
        {
            flow_through_boundary_state(coords,hs,front,p2wave,state);
            set_state(state,TGAS_STATE,state);
            Press(state) = p;
            Dens(state) = density(state);
        }
        else
        {
            //fprintf(stdout,"FD: Boundary state, press=%g\n",Press(state));
            //fprint_raw_gas_data(stdout,state,front->rect_grid->dim);
            //fflush(stdout);
            static Locstate tmp=0;
            int dim = front->rect_grid->dim;
            float nor[MAXD];
            CURVE *c;
            if(dim!=2)
            {
                printf("Not impelemented in %s(%d).\n",__FILE__,__LINE__);
                clean_up(ERROR);
            }
            if(!tmp)
                scalar(&tmp,size_of_state(front->interf));
            flow_through_boundary_state(coords,hs,front,p2wave,tmp);
            st_type = state_type(tmp);
            Set_params(state,tmp);
            set_type_of_state(state,TGAS_STATE);
            set_state(state,TGAS_STATE,state);
            Press(state) = p;
            c=Curve_of_hs(hs);
            if (is_bdry(c)) /*Rectangular boundary*/
            {
                float t[MAXD], length;
                int i;
                for (i = 0; i < dim; ++i)
                    t[i] = Coords(c->end->posn)[i] - Coords(c->start->posn)[i];
                length = mag_vector(t,dim);
                for (i = 0; i < dim; ++i)
                    t[i] /= length;
                nor[0] = t[1];
                nor[1] = -t[0];
                // should be (0,1,0) for jet run
            }
            else
            {
                printf("Not impelemented in %s(%d).\n",__FILE__,__LINE__);
                clean_up(ERROR);
            }

            connect_by_riemann_wave(state,tmp,2,nor);
            //fprintf(stdout,"FD: Updated state\n");
            //fprint_raw_gas_data(stdout,state,front->rect_grid->dim);
            //fflush(stdout);
        }

        set_state(state,st_type,state);

	if (debugging("flow_through"))
	{
	    (void) printf("State returned by ");
	    (void) printf("time_dep_pressure_flow_through_boundary_state()\n");
	    fprint_raw_gas_data(stdout,state,front->rect_grid->dim);
	}
	debug_print("flow_through",
		"Left time_dep_pressure_flow_through_boundary_state()\n");
}	/*end time_dep_pressure_flow_through_boundary_state */

/*
*			g_neumann_bdry_state():
*
*	Imposes a reflecting boundary condition at points laying outside of a 
*	neumann boundary.  The point (x,y) is reflected through the neumann 
*	boundary curve Nbdry, the state at this reflected point (xref, yref) 
*	is evaluated using hyp_solution and the state at (x, y) is then set to
*	be the state with the same density, pressure and tangential (wrt 
*	Nbdry) component of velocity as the state at (xref, yref) and opposite
*	normal component of velocity.  It is assumed that the reflected point 
*	should lie in the component int_comp.
*
*	The use of this function may be problem dependent and it is accessed
*	through the function pointer front->neumann_bdry_state.
*
*       The method mentioned above is one of several methods implemented
*       in this function, and it is the default.  The other methods are
*       accessible by setting an appropriate debug string.
*/

/*ARGSUSED*/
EXPORT	int g_neumann_bdry_state(
	float		*coords,
	COMPONENT	int_comp,
	POINT		*pt,
	HYPER_SURF	*Nbdry,
	Front		*front,
	POINTER		p2wave,
	Locstate	state)
{
	Wave		*wave = (Wave*)p2wave;
	float		nor[MAXD];
	float		coords_ref[MAXD];
	float		coords_tmp[MAXD];

	int		i, dim = front->rect_grid->dim;

	float           tmp_vel[MAXD], vn;

        DEBUG_ENTER(g_neumann_bdry_state) 

	/* coords_ref is the location of the reflection of the point coords
	   across the Neumann boundary. */

	if (debugging("no_nbdry") 
	    || !reflect_pt_about_Nbdry(coords,coords_ref,nor,
					  int_comp,Nbdry,front))
	    return NO;

	/* Get the state at coords_ref. */
	
	hyp_solution(coords_ref,int_comp,Nbdry,UNKNOWN_SIDE,
		     front,wave,state,NULL);

	set_state(state,TGAS_STATE,state);
	vn = 0.0;
	for (i = 0; i < dim; i ++)
	{
	    tmp_vel[i] = Vel(state)[i];
	    vn += tmp_vel[i] * nor[i];
	}

	if (no_slip(Nbdry))
        {
	    float alpha = adherence_coeff(Nbdry);
            for (i = 0; i < dim; i ++)
	    	Vel(state)[i] = (1-2*alpha)*Vel(state)[i];
	}

	zero_normal_velocity(state,nor,dim);
	for (i = 0; i < dim; i ++)
	    Vel(state)[i] += - vn * nor[i];
	set_state(state,GAS_STATE,state);
	return YES;

	/* multiple versions */

	if (debugging("nbdry1"))
	{
	    /* This version extrapolates the density and pressure of the wall
	       state at coords_tmp, but reflects the velocity of the state at
	       coords_ref. */

	    float v_dot_n;
	    static Locstate tmpst = NULL;
	    if (tmpst == NULL)
		alloc_state(front->interf,&tmpst,front->sizest);
	

	    /* Reflect the velocity of the state at coords_ref. */
	    
	    set_state(state,TGAS_STATE,state);

	    v_dot_n = 2.0*scalar_product(Vel(state),nor,dim);
	    for (i = 0; i < dim; i ++) 
		Vel(state)[i] -= v_dot_n*nor[i];

	    /* Get the wall state at coords_tmp, which is the midpoint of the
	       line segment connecting coords to coords_ref. */

	    for (i = 0; i < dim; ++i) 
		coords_tmp[i] = 0.5*(coords[i]+coords_ref[i]);

	    hyp_solution(coords_tmp,int_comp,Nbdry,UNKNOWN_SIDE,front,
			 wave,tmpst,NULL);

	    /* Copy the density and pressure of tmpst to state. */

	    Dens(state) = Dens(tmpst);
	    Press(state) = pressure(tmpst);
	    set_state(state,GAS_STATE,state);
	}
	else if (debugging("nbdry2"))
	{
	    /* This version linearly extrapolates the TGAS_STATEs at the
	       reflection point and boundary (coords_ref and coords_tmp,
	       respectively). */

	    /* Get the wall state at coords_tmp, which is the midpoint of the
	       line segment connecting coords to coords_ref. */

	    static Locstate tmpst = NULL;
	    if (tmpst == NULL)
		alloc_state(front->interf,&tmpst,front->sizest);

	    for (i = 0; i < dim; ++i) 
		coords_tmp[i] = 0.5*(coords[i]+coords_ref[i]);

	    hyp_solution(coords_tmp,int_comp,Nbdry,UNKNOWN_SIDE,front,
			 wave,tmpst,NULL);
	    
	    /* Linearly extrapolate the density, pressure, and velocity. */

	    set_state(state,TGAS_STATE,state);
	    set_state(tmpst,TGAS_STATE,tmpst);
	    
	    Dens(state) = 2.0*Dens(tmpst) - Dens(state);
	    Press(state) = 2.0*Press(tmpst) - Press(state);
	    for (i = 0; i < dim; ++i) 
		Vel(state)[i] = 2.0*Vel(tmpst)[i] - Vel(state)[i];
	    
	    set_state(state,GAS_STATE,state);
	}
	else if (debugging("nbdry3"))
	{
	    /* This version reflects the velocity of the state at coords_ref
	       and linearly extrapolates the density and pressure at coords_ref
	       and at one cell further in.  The state at the wall is not
	       used. */

	    float v_dot_n;
	    static Locstate tmpst = NULL;

	    if (tmpst == NULL)
		alloc_state(front->interf,&tmpst,front->sizest);

	    /* Reflect the velocity of the state at coords_ref. */
	    
	    set_state(state,TGAS_STATE,state);

	    v_dot_n = 2.0*scalar_product(Vel(state),nor,dim);
	    for (i = 0; i < dim; i ++) 
		Vel(state)[i] -= v_dot_n*nor[i];

	    /* Get the location and state one cell further in. */

	    for (i = 0; i < dim; ++i) 
		coords_tmp[i] = 2.0*coords_ref[i] - coords[i];
	    
	    hyp_solution(coords_tmp,int_comp,Nbdry,UNKNOWN_SIDE,front,
			 wave,tmpst,NULL);
	    
	    /* Linearly extrapolate the density and pressure. */

	    Dens(state) = 2.0*Dens(state) - Dens(tmpst);
	    Press(state) = 2.0*Press(state) - pressure(tmpst);
	    
	    set_state(state,GAS_STATE,state);
	}
	else if (debugging("nbdry4") || debugging("nbdry4a"))
	{
	    /* This version does a quadratic extrapolation using the states
	       at coords_ref and at one and two cells further in.  The state
	       at the wall is not used. */

	    static Locstate tmpst = NULL, tmpst1 = NULL;
	    if (tmpst1 == NULL)
	    {
		alloc_state(front->interf,&tmpst,front->sizest);
		alloc_state(front->interf,&tmpst1,front->sizest);
	    }

	    /* Get the location and state one cell further in. */

	    for (i = 0; i < dim; ++i) 
		coords_tmp[i] = 2.0*coords_ref[i] - coords[i];
	    
	    hyp_solution(coords_tmp,int_comp,Nbdry,UNKNOWN_SIDE,front,
			 wave,tmpst,NULL);
	    
	    /* Get the location and state two cells further in. */

	    for (i = 0; i < dim; ++i) 
		coords[i] = 2.0*coords_tmp[i] - coords_ref[i];
	    
	    hyp_solution(coords,int_comp,Nbdry,UNKNOWN_SIDE,front,
			 wave,tmpst1,NULL);
	    
	    /* Check to see if the densities are monotonic.
	    
	    if ((Dens(state) > Dens(tmpst) && Dens(tmpst) < Dens(tmpst1))
	     || (Dens(state) < Dens(tmpst) && Dens(tmpst) > Dens(tmpst1)))
	    {
		(void) printf("WARNING in g_neumann_bdry_state() --\n");
		(void) printf("Dens(state) = %g\n", Dens(state));
		(void) printf("Dens(tmpst) = %g\n", Dens(tmpst));
		(void) printf("Dens(tmpst1) = %g\n", Dens(tmpst1));
		(void) printf("Densities are not monotonic near wall.\n");
	    }
	    */

	    set_state(state,TGAS_STATE,state);

	    /* Yet another variant:  Reflect or extrapolate the velocity. */

	    if (debugging("nbdry4a"))  /* reflect the velocity */
	    {
		float v_dot_n = 2.0*scalar_product(Vel(state),nor,dim);
		for (i = 0; i < dim; i ++) 
		    Vel(state)[i] -= v_dot_n*nor[i];
	    }
	    else                       /* extrapolate the velocity */
	    {
		for (i = 0; i < dim; ++i)
		    Vel(state)[i] = vel(i,tmpst1) - 3.0*vel(i,tmpst)
			+ 3.0*Vel(state)[i];
	    }

	    /* Extrapolate the density and pressure. */

	    Dens(state) = Dens(tmpst1) - 3.0*Dens(tmpst) + 3.0*Dens(state);
	    Press(state) = pressure(tmpst1) - 3.0*pressure(tmpst)
		+ 3.0*Press(state);

	    set_state(state,GAS_STATE,state);
	}
	else if (debugging("nbdry5"))
	{
	    /* This version does a quadratic extrapolation using the states
	       at the wall, at coords_ref and at one cell further in. */

	    static Locstate tmpst = NULL, tmpst1 = NULL;
	    if (tmpst == NULL)
	    {
		alloc_state(front->interf,&tmpst,front->sizest);
		alloc_state(front->interf,&tmpst1,front->sizest);
	    }

	    /* Get the location and state one cell further in. */

	    for (i = 0; i < dim; ++i) 
		coords_tmp[i] = 2.0*coords_ref[i] - coords[i];
	    
	    hyp_solution(coords_tmp,int_comp,Nbdry,UNKNOWN_SIDE,front,
			 wave,tmpst1,NULL);
	    
	    /* Get the location and state at the wall. */

	    for (i = 0; i < dim; ++i) 
		coords_tmp[i] = 0.5*(coords[i]+coords_ref[i]);

	    hyp_solution(coords,int_comp,Nbdry,UNKNOWN_SIDE,front,
			 wave,tmpst,NULL);
	    
	    set_state(state,TGAS_STATE,state);

	    /* Extrapolate the density, pressure, and velocity. */

	    Dens(state) = (Dens(tmpst1) - 6.0*Dens(state) 
			   + 8.0*Dens(tmpst))/3.0;
	    Press(state) = (pressure(tmpst1) - 6.0*pressure(state)
			    + 8.0*pressure(tmpst))/3.0;
	    for (i = 0; i < dim; ++i)
		Vel(state)[i] = (vel(i,tmpst1) - 6.0*Vel(state)[i]
		    + 8.0*vel(i,tmpst))/3.0;
	    
	    set_state(state,GAS_STATE,state);
	}
	else
	{

	    /* Default method: Determine the state according to the comment above
	       this function.  If there is gravity, then reflect_state() calls
	       g_reflect_and_stratify_state(), which stratifies the density and
	       pressure with respect to the state being reflected. */
    
	    /* coords_tmp is the point on the reflection plane. */

	    for (i = 0; i < dim; ++i) 
	        coords_tmp[i] = 0.5*(coords[i]+coords_ref[i]);

	    reflect_state(state,front->interf,coords,coords_tmp,nor);

	}
        DEBUG_LEAVE(g_neumann_bdry_state) 
	return YES;
}		/*end g_neumann_bdry_state*/



/*
*		g_fixed_boundary_state():
*	
*	Sets the state to the boundary_state of the given hypersurface
*	to the value of the fixed state boundary_state(hs).
*	If the boundary state params are different from params on
*	the hypersurface (an indication that a contact has broken
*	through on the boundary) then the boundary state is filtered
*	through a Riemann problem so the the params of the returned
*	state is consistent with those in the adjacent interior region.
*/

/*ARGSUSED*/
EXPORT	void g_fixed_boundary_state(
	float		*coords,
	HYPER_SURF	*hs,
	Front		*front,
	POINTER		p2wave,
	Locstate	state)
{
	Locstate		bs = boundary_state(hs);
	float			W[3];
	float			nor[3];
	float			coords_on[3];
	int			dim = front->rect_grid->dim;
	static	Locstate	st = NULL, tmpst = NULL;

	if (is_exterior_comp(negative_component(hs),hs->interface))
	{
		switch (dim)
		{
#if defined(ONED)
	    case 1:
	    	if (Same_params(right_state(Point_of_hs(hs)),bs))
	    	{
	    		assign(state,bs,front->sizest);
	    		return;
	    	}
	    	break;
#endif /* defined(ONED) */
#if defined(TWOD)
	    case 2:
	    	if (Same_params(right_start_state(Curve_of_hs(hs)),bs))
	    	{
	    	    assign(state,bs,front->sizest);
	    	    return;
	    	}
		break;
#endif /* defined(TWOD) */
#if defined(THREED)
	    case 3:
	    {
	        SURFACE	*s = Surface_of_hs(hs);
	        Locstate	sl, sr;
	        TRI	*tri = first_tri(s);
	        slsr(Point_of_tri(tri)[0],Hyper_surf_element(tri),hs,&sl,&sr);
	        if (Same_params(sr,bs))
	        {
	            assign(state,bs,front->sizest);
	            return;
	        }
	    }
	        break;
#endif /* defined(THREED) */
	    }
	    if (st == NULL)
	    {
	    	alloc_state(front->interf,&st,front->sizest);
	    	alloc_state(front->interf,&tmpst,front->sizest);
	    }
	    nearest_state_and_normal_on_hypersurface(coords,coords_on,
			positive_component(hs),hs,st,nor,front);
	    w_speed(coords_on,bs,st,tmpst,state,W,0.0,nor,CONTACT,front);
	    return;
	}
	else if (is_exterior_comp(positive_component(hs),hs->interface))
	{
	    switch (dim)
	    {
#if defined(ONED)
	    case 1:
	    	if (Same_params(left_state(Point_of_hs(hs)),bs))
	    	{
	    	    assign(state,bs,front->sizest);
	    	    return;
	    	}
	    	break;
#endif /* defined(ONED) */
#if defined(TWOD)
	    case 2:
	    	if (Same_params(left_start_state(Curve_of_hs(hs)),bs))
		{
		    assign(state,bs,front->sizest);
		    return;
		}
		break;
#endif /* defined(TWOD) */
#if defined(THREED)
	    case 3:
	    {
	        SURFACE	*s = Surface_of_hs(hs);
	    	Locstate	sl, sr;
	    	TRI	*tri = first_tri(s);
	    	slsr(Point_of_tri(tri)[0],Hyper_surf_element(tri),hs,&sl,&sr);
	    	if (Same_params(sl,bs))
	    	{
	    	    assign(state,bs,front->sizest);
	    	    return;
	    	}
	    }
	    	break;
#endif /* defined(THREED) */
	    }
	    if (st == NULL)
	    {
	    	alloc_state(front->interf,&st,front->sizest);
	    	alloc_state(front->interf,&tmpst,front->sizest);
	    }
	    nearest_state_and_normal_on_hypersurface(coords,coords_on,
			negative_component(hs),hs,st,nor,front);
	    w_speed(coords_on,st,bs,state,tmpst,W,0.0,nor,CONTACT,front);
	    return;
	}
	assign(state,bs,front->sizest);
}		/*end g_fixed_boundary_state*/

LOCAL	void	nearest_state_and_normal_on_hypersurface(
	float		*coords,
	float		*coords_on,
	COMPONENT	comp,
	HYPER_SURF	*hs,
	Locstate	state,
	float		*nor,
	Front		*front)
{
	HYPER_SURF_ELEMENT	*hse;
	HYPER_SURF		*hs_on;
	float			t[3];
	int			dim = front->rect_grid->dim;
	static	float		**nor_p = NULL;
#if defined(TWOD) || defined(THREED)
	float			mag;
#endif /* defined(TWOD) || defined(THREED) */

	if (nor_p == NULL)
		matrix(&nor_p,3,3,FLOAT);
	if (nearest_interface_point(coords,comp,hs->interface,
				    INCLUDE_BOUNDARIES,hs,coords_on,t,&hse,
				    &hs_on) != YES)
	{
	   screen("ERROR in nearest_state_and_normal_on_hypersurface(), "
	          "nearest_interface_point() failed\n");
	   clean_up(ERROR);
	}
	switch (dim)
	{
#if defined(ONED)
	case 1:
	    normal(Point_of_hs(hs),hse,hs,nor,front);
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    normal(Bond_of_hse(hse)->start,hse,hs,nor_p[0],front);
	    normal(Bond_of_hse(hse)->end,hse,hs,nor_p[1],front);
	    t[1] = 1.0 - t[0];
	    nor[0] = t[0]*nor_p[0][0] + t[1]*nor_p[1][0];
	    nor[1] = t[0]*nor_p[0][1] + t[1]*nor_p[1][1];
	    mag = mag_vector(nor,dim);
	    nor[0] /= mag;
	    nor[1] /= mag;
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    normal(Point_of_tri(Tri_of_hse(hse))[0],hse,hs,nor_p[0],front);
	    normal(Point_of_tri(Tri_of_hse(hse))[1],hse,hs,nor_p[1],front);
	    normal(Point_of_tri(Tri_of_hse(hse))[2],hse,hs,nor_p[2],front);
	    nor[0] = t[0]*nor_p[0][0] + t[1]*nor_p[1][0] + t[2]*nor_p[2][0];
	    nor[1] = t[0]*nor_p[0][1] + t[1]*nor_p[1][1] + t[2]*nor_p[2][1];
	    nor[2] = t[0]*nor_p[0][2] + t[1]*nor_p[1][2] + t[2]*nor_p[2][2];
	    mag = mag_vector(nor,dim);
	    nor[0] /= mag;
	    nor[1] /= mag;
	    nor[2] /= mag;
	    break;
#endif /* defined(THREED) */
	}
	state_along_hypersurface_element(comp,t,hse,hs,state);
}		/*end nearest_state_and_normal_on_hypersurface*/

LOCAL void connect_by_riemann_wave(
        Locstate        lstate,
        const Locstate  rstate,
        int             dim,
        const float     *nor)
{
        float v, vn, c;
        float v_adj=0.0;
        int i; 

        set_state(lstate,TGAS_STATE,lstate);
        set_state(rstate,TGAS_STATE,rstate);

        vn = scalar_product(Vel(rstate),nor,2);
        c = sound_speed(rstate);

        if( vn>=c ) // supersonic
        {
            v_adj = 0.0;
        }
        else
        {
            v_adj = riemann_wave_curve(rstate,Press(lstate));
        }
        v = vn + v_adj;
/*
        printf("pl = %g, pr = %g, vn = %g, v_adj = %g\n",
               Press(lstate),Press(rstate),vn,v);
*/
        for(i=0;i<dim;++i)
        {
            // Tangential velocity of lstate and rstate the same?

            Vel(lstate)[i] = (v-vn)*nor[i] + Vel(rstate)[i];
            /*
               equivalently,
               take the tangential velocity of rstate and
               the computed normal velocity
               float v_tan[MAXD];
               v_tan[i] = Vel(rstate)[i] - vn*nor[i];
               Vel(lstate)[i] = v*nor[i] + v_tan[i];
            */
            //Vel(lstate)[i] = v*nor[i]; // setting tangential velocity zero
        }
        Dens(lstate) = density(lstate);
}

