/*
*				gintext.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains diagnostic routines for interface extrema
*
*	TODO: Generalize to more than 2 distinct materials
*/

#if defined(TWOD) || defined(THREED)

#include <gprt/glayer.h>

	/* LOCAL Function Prototypes */

LOCAL	bool	accumulate_fractions_in_layer(Front*,float,
					      float*,float,const float*);
LOCAL	bool	height_at_fraction(Front*,float*,float,int,
				   float,float*,int,float,const float*);
LOCAL	bool	is_on_local_grid(const RECT_GRID*,const float*);
LOCAL	void 	add_state_to_totals(const Locstate,Big_State*,
				    Turbulence_moments*);
LOCAL 	void 	convert_to_spherical(float*,const float*,const float*,int);
LOCAL	void 	copy_Big_State(const Big_State*,Big_State*);
LOCAL	void 	copy_state_to_Big_State(const Locstate,Big_State*);
LOCAL   void 	print_data_at_height(const Grid*,const Front*,OUTPUT_DATA*,
				     const Big_State*,const Big_State*,
				     FILE*,FILE*,FILE*,float,float,bool);
LOCAL   void    print_data_at_height_headers(FILE*,FILE*,FILE*,int);
LOCAL   void    print_intfc_extrema(const Grid*,const Front*,
	                            OUTPUT_DATA*,const Big_State[2],
				    const Big_State[2],bool);
LOCAL   void    print_intfc_extrema_headers(const Intfc_extrema*,int);
LOCAL	void 	record_intfc_extrema(Grid*,Wave*,Front*,Printplot*,
				     OUTPUT_DATA*,bool);
LOCAL	void    spherical_unit_vectors(float*,float*,float*,
				       const float*,const float*,int);
LOCAL	void	zero_state_totals(Big_State*,Turbulence_moments*,int);

EXPORT	void init_intfc_extrema(
	INIT_DATA	*init,
	Front		*front,
	Grid		*grid,
	Printplot	*prt)
{
	Gas_param       **prms_list;
	Intfc_extrema	*iext;
	const int	dim = front->rect_grid->dim;
	char 		s[Gets_BUF_SIZE];
	char		*fname, *dname;
	int 		result;
	int             i, nprms;

	if (dim == 1)
	    return;

	screen("Type 'y' to request interface extrema data: ");
	(void) Gets(s);
	if ((s[0] != 'Y') && (s[0] != 'y'))
	    return;

	scalar(&iext,sizeof(Intfc_extrema));

	/* defaults */
	Output_mode(&iext->odata) = MESH_TIME;
	Output_step_freq(&iext->odata) = 1;
	Output_start_step(&iext->odata) = 0;
	Output_in_binary(&iext->odata) = NO;

	init_output_data(init,&iext->odata,grid,prt,NULL,NO,NO,YES);
	add_user_output_function(record_intfc_extrema,&iext->odata,prt);

	iext->geom = planar;	/* default */
	screen("Compute interface extrema for planar ('p'%s) or"
	       " radial geometry ('r'%s): ",
	       (iext->geom == planar)    ? ", default" : "",
	       (iext->geom == spherical) ? ", default" : "");
	(void) Gets(s);
	if (s[0] == 'R' || s[0] == 'r')
	    iext->geom = spherical;

	if (iext->geom == spherical)
	{
	    float *origin = iext->origin;
	    origin[0] = origin[1] = origin[0] = 0.0;
	    sprint_general_vector(s,"(default = ",origin,dim,")");
	    screen("Enter the coordinates of the origin %s: ",s);
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
#if defined(float)
		const char *fmt = "%lf %lf %lf";
#else /*defined(float)*/
		const char *fmt = "%f %f %f";
#endif /*defined(float)*/
	        result = sscanf(s,fmt,origin,origin+1,origin+2);
	        if (result < dim)
	        {
		    screen("ERROR in init_intfc_extrema(), "
		           "Insufficient number of coordinates.\n");
		    clean_up(ERROR);
	        }
	    }
	}

	iext->rfactor = 2.0;		/* default */
	screen("Enter a sub-grid refinement factor for the averaging of the\n"
	       " ambient state at the interface extrema"
	       " (default = %g): ",iext->rfactor);
	(void) Gets(s);
	if (s[0] !='\0')
	    (void) sscan_float(s,&iext->rfactor);

	screen("Current gas param list\n");
	nprms = return_params_list(&prms_list);
	screen("Number of params = %d\n\n",nprms);
	for (i = 0; i < nprms; i++)
	{
	    IMPORT bool suppress_prompts;
	    screen("Param[%d]\n",i);
	    fprint_Gas_param(stdout,prms_list[i]);
	    if (suppress_prompts == NO)
	        fprint_Gas_param(stderr,prms_list[i]);
	    screen("\n");
	}
	screen("Enter the EOS indices of the %s and %s materials, "
	       "respectively: ",
	       (iext->geom == spherical ? "inner" : "lower"),
	       (iext->geom == spherical ? "outer" : "upper"));
	Scanf("%d %d\n",&iext->index_at_max,&iext->index_at_min);

	set_default_data_file_names("intfc_extrema",".min",&dname,&fname,init);
	iext->out_min = open_data_file(front,"interface minimum",YES,YES,dname,
				       &iext->min_dname,fname,&iext->min_fname);

	set_default_data_file_names("intfc_extrema",".max",&dname,&fname,init);
	iext->out_max = open_data_file(front,"interface maximum",YES,YES,dname,
				       &iext->max_dname,fname,&iext->max_fname);

	set_default_data_file_names("intfc_extrema",".amp",&dname,&fname,init);
	iext->out_amp = open_data_file(front,"interface amplitude",YES,YES,
	                               dname,&iext->amp_dname,
	                               fname,&iext->amp_fname);

	screen("Type 'y' to get data for 1%%-99%% levels: ");
	(void) Gets(s);
	if (s[0] == 'Y' || s[0] == 'y')
	{
	    char fname[Gets_BUF_SIZE];
	    char *tmpdname, *tmpfname;

	    iext->do_01 = YES;

	    (void) sprintf(fname,"%s%s",
			   (iext->min_fname != NULL)?iext->min_fname:"","01");
	    iext->out_min1 = open_data_file(front,"interface 1% minimum",
	                                    YES,YES,iext->min_dname,
	                                    &tmpdname,fname,&tmpfname);

	    (void) sprintf(fname,"%s%s",
			   (iext->max_fname != NULL)?iext->max_fname:"","01");
	    iext->out_max1 = open_data_file(front,"interface 1% maximum",
	                                    YES,YES,iext->max_dname,
	                                    &tmpdname,fname,&tmpfname);

	    (void) sprintf(fname,"%s%s",
			   (iext->amp_fname != NULL)?iext->amp_fname:"","01");
	    iext->out_amp1 = open_data_file(front,"interface 1% amplitude",
	                                    YES,YES,iext->amp_dname,
	                                    &tmpdname,fname,&tmpfname);
	}
	screen("Type 'y' to get data for 5%%-95%% levels: ");
	(void) Gets(s);
	if (s[0] == 'Y' || s[0] == 'y')
	{
	    char fname[Gets_BUF_SIZE];
	    char *tmpdname, *tmpfname;

	    iext->do_05 = YES;

	    strcpy(fname,iext->min_fname);
	    strcat(fname,"05");
	    iext->out_min5 = open_data_file(front,"interface 5% minimum",
	                                    YES,YES,iext->min_dname,
	                                    &tmpdname,fname,&tmpfname);
	    strcpy(fname,iext->max_fname);
	    strcat(fname,"05");
	    iext->out_max5 = open_data_file(front,"interface 5% maximum",
	                                    YES,YES,iext->max_dname,
	                                    &tmpdname,fname,&tmpfname);
	    strcpy(fname,iext->amp_fname);
	    strcat(fname,"05");
	    iext->out_amp5 = open_data_file(front,"interface 5% amplitude",
	                                    YES,YES,iext->amp_dname,
	                                    &tmpdname,fname,&tmpfname);
	}
}		/*end init_intfc_extrema*/


LOCAL void convert_to_spherical(
	float        *p_sph,
	const float  *p_rect,
	const float  *origin,
	int          dim)
{
	int   i;
	float pr[3];

	for (i = 0; i < dim; i++)
	    pr[i] = p_rect[i];
	if (origin != NULL)
	    for (i = 0; i < dim; i++)
	        pr[i] -= origin[i];
	for (; i < 3; i++)
	    pr[i] = 0.0;

	p_sph[0] = mag_vector(pr,dim);
	if (dim == 3)
	    p_sph[1] = acos(pr[2]/p_sph[0]);
	p_sph[dim-1] = atan2(pr[1],pr[0]);
}		/*end convert_to_spherical*/

/*
*			spherical_unit_vectors():
*
*	Compute rectangular components of standard orthonormal vectors ("r hat",
*	"theta hat", "phi hat") for polar (dim == 2) or spherical (dim == 3)
*	coordinates given a point p_rect in rectangular coordinates and an
*	origin.
*/

LOCAL	void	spherical_unit_vectors(
	float       *rhat,
	float       *thhat,
	float       *phhat,
	const float *p_rect,
	const float *origin,
	int         dim)
{
	float p_sph[3], snph, csph;

	convert_to_spherical(p_sph,p_rect,origin,dim);
	switch(dim)
	{
	case 1:
	    screen("ERROR in spherical_unit_vectors(), 1D not supported\n");
	    clean_up(ERROR);
	    break;
#if defined(TWOD)
	case 2:  /* polar coordinates */
	    snph = sin(p_sph[1]);
	    csph = cos(p_sph[1]);
	     rhat[0] =  csph;  rhat[1] = snph;
	    phhat[0] = -snph; phhat[1] = csph;
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    {
	        float snth, csth;
	        snph = sin(p_sph[2]);
	        csph = cos(p_sph[2]);
	        snth = sin(p_sph[1]);
	        csth = cos(p_sph[1]);
	         rhat[0] = snth*csph;  rhat[1] = snth*snph;  rhat[2] =  csth;
	        thhat[0] = csth*csph; thhat[1] = csth*snph; thhat[2] = -snth;
	        phhat[0] = -snph;     phhat[1] = csph;      phhat[2] =  0.0;
	    }
	    break;
#endif /* defined(THREED) */
	}
}		/*end spherical_unit_vectors*/

LOCAL	bool	is_on_local_grid(
	const RECT_GRID *rgrid,
	const float     *coords)
{
	int   i;
	const int   dim = rgrid->dim;
	const float *L = rgrid->L, *U = rgrid->U;
	for (i = 0; i < dim; i++)
	    if (coords[i] < L[i] || coords[i] > U[i])
		return NO;
	return YES;
}		/*end is_on_local_grid*/

LOCAL	void 	copy_state_to_Big_State(
	const Locstate	st,
	Big_State	*bst)
{
	int	i;

	bst->d = Dens(st);
	for (i = 0; i < MAXD; i++)
	    bst->v[i] = vel(i,st);
	bst->p = pressure(st);
	bst->k = kinetic_energy(st);
	bst->e = internal_energy(st);
}		/*end copy_state_to_Big_State*/

LOCAL	void 	copy_Big_State(
	const Big_State	*bst0,
	Big_State	*bst)
{
	int	i;

	bst->d = bst0->d;
	for (i = 0; i < MAXD; i++)
	    bst->v[i] = bst0->v[i];
	bst->p = bst0->p;
	bst->k = bst0->k;
	bst->e = bst0->e;
	bst->frac = bst0->frac;
	bst->count = bst0->count;
}		/*end copy_Big_State*/

LOCAL	bool accumulate_fractions_in_layer(
	Front	    *front,
	float	    height,
	float	    *frac,
	float	    rfactor,
	const float *origin)
{
	INTERFACE		  *intfc = front->interf;
	const RECT_GRID		  *rgrid = front->rect_grid;
	const enum intfc_geometry geom = (origin == NULL ? planar : spherical);
	const int		  dim = rgrid->dim;
	const int		  n_params = num_gas_params(intfc);
	const int		  nn = pp_numnodes();
	const int		  zdir = dim - 1;
	float			  dx, dphi;
	float			  nor_fac, coords[3];
	float			  x_left, phi;
	int			  Nx, Nphi;
	int			  p;
	register int 		  i;
	COMPONENT		  comp;
	const float		  dh = (dim == 2) ?
				      min(rgrid->h[0],rgrid->h[1])/rfactor
				      : min(min(rgrid->h[0],rgrid->h[1]),
					    rgrid->h[2])/rfactor;

	static long int		*count = NULL;

	debug_print("glayer","Entered accumulate_fractions_in_layer(), h = %g\n",
		       height);

	if (count == NULL)
	    vector(&count,n_params,sizeof(long int));

        zero_scalar(count,n_params*sizeof(long int));
 
	for (i = 0; i < 3; i++)
	    coords[i] = 0.0;

	switch(geom)
	{
	case planar:
	    coords[zdir] = height;
	    if (height < rgrid->L[zdir] || height >= rgrid->U[zdir])
	      break;
	    Nx = irint(rgrid->gmax[0]*rfactor);
	    dx = (rgrid->U[0] - rgrid->L[0])/Nx;
	    x_left = rgrid->L[0] + 0.5*dx;
	    switch(dim)
	    {
	    case 1:
	        screen("ERROR in accumulate_fractions_in_layer(), "
		       "1D not supported\n");
	        clean_up(ERROR);
	        break;
#if defined(TWOD)
	    case 2:
	        for (i = 0; i < Nx; i++)
		{
		    coords[0] = x_left + i*dx;
		    comp = component(coords,intfc);
		    count[index_of_Gas_param(
				      gas_params_for_comp(comp,intfc))]++;
		}
		break;
#endif /* defined(TWOD) */
#if defined(THREED)
	    case 3:
		{
		    float dy, y_left;
		    int j, Ny;
		    Ny = irint(rgrid->gmax[1]*rfactor);
		    dy = (rgrid->U[1] - rgrid->L[1])/Ny;
		    y_left = rgrid->L[1] + 0.5*dy;
		    for (j = 0; j < Ny; j++)
		    {
		        coords[1] = y_left + j*dy;
		        for (i = 0; i < Nx; i++)
		        {
			    coords[0] = x_left + i*dx;
			    comp = component(coords,intfc);
			    count[index_of_Gas_param(
					  gas_params_for_comp(comp,intfc))]++;
		        }
		    }
		}
		break;
#endif /* defined(THREED) */
	    }
	    break;

	case spherical:
	    switch(dim)
	    {
	    case 1:
	        screen("ERROR in accumulate_fractions_in_layer(), "
		       "1D not supported\n");
	        clean_up(ERROR);
	        break;
#if defined(TWOD)
	    case 2:
	        Nphi = irint(2.0*PI*height/dh);
		dphi = 2.0*PI/Nphi;
		for (i = 0; i < Nphi; i++)
		{
		    phi = i*dphi;
		    coords[0] = height*cos(phi) + origin[0];
		    coords[1] = height*sin(phi) + origin[1];
		    if (is_on_local_grid(rgrid,coords) != YES)
		      continue;
		    comp = component(coords,intfc);
		    count[index_of_Gas_param(
				      gas_params_for_comp(comp,intfc))]++;
		}
	        break;
#endif /* defined(TWOD) */
#if defined(THREED)
	    case 3:
		{
	            float dth, th, rpolar;
		    int   j, Nth;
		    Nth = irint(PI*height/dh);
		    dth = PI/Nth;
		    for (j = 0; j < Nth; j++)
		    {
		        th = j*dth;
		        rpolar = height*sin(th);
		        Nphi = irint(2.0*PI*rpolar/dh);
		        dphi = 2.0*PI/Nphi;
		        for (i = 0; i < Nphi; i++)
		        {
			    phi = i*dphi;
			    coords[0] = rpolar*cos(phi) + origin[0];
			    coords[1] = rpolar*sin(phi) + origin[1];
			    coords[2] = height*cos(th) + origin[2];
			    if (is_on_local_grid(rgrid,coords) != YES)
			        continue;
			    comp = component(coords,intfc);
			    count[index_of_Gas_param(
				      gas_params_for_comp(comp,intfc))]++;
		        }
		    }
		}
		break;
#endif /* defined(THREED) */
	    }
	    break;
	}

	if (debugging("glayer"))
	{
	    (void) printf("\tLocal accumulation of fractions at "
			  "h = %g completed.\n",height);
	    for (p = 0; p < n_params; p++)
	        (void) printf("\t\tcount[%d] = %ld\t",p,count[p]);
	    (void) printf("\n");
	}

	if (nn > 1)
	    pp_global_lsum(count,(long int)n_params);

	if (debugging("glayer"))
	{
	    (void) printf("\tGlobal accumulation of fractions at "
			  "h = %g completed.\n",height);
	    for (p = 0; p < n_params; p++)
	        (void) printf("\t\tcount[%d] = %ld\t",p,count[p]);
	    (void) printf("\n");
	}

	for (p = 0; p < n_params && count[p] == 0; p++);
	if (p == n_params)
	{
	    (void) printf("WARNING in accumulate_fractions_in_layer(), "
			  "No points found in layer at h = %g\n",height);
	    return FUNCTION_FAILED;
	}

	for (p = 0, nor_fac = 0.0; p < n_params; p++)
	    nor_fac += (float) count[p];
	for (p = 0; p < n_params; p++)
	    frac[p] = ((float) count[p])/nor_fac;

	debug_print("glayer","Left accumulate_fractions_in_layer()\n");

	return FUNCTION_SUCCEEDED;
}		/*end accumulate_fractions_in_layer*/

/*
*			height_at_fraction():
*
*	Find the height (z coordinate or radius) closest to initial height
*	h0 in the given direction dir (+/- 1) where the given layer fraction
*	occurs.  This fraction corresponds to the material whose params
*	has the given index.
*/

LOCAL 	bool	height_at_fraction(
	Front	    *front,
	float	    *h0,
	float	    dh,
	int	    dir,
	float	    fraction,
	float	    *frac,
	int	    index,
	float	    rfactor,
	const float *origin)
{
	bool             status;
	const int        n_params = num_gas_params(front->interf);
	float            height = *h0, hlo, hhi, ftol = 1.0e-10;
	int              p, iter2;
	static const int max_num_iters = 3;
	static float     *new_frac = NULL, *frac_hi = NULL, *frac_lo = NULL;

	debug_print("gfrac","Entered height_at_fraction()\n");

	if (new_frac == NULL)
	{
	    vector(&new_frac,n_params,FLOAT);
	    vector(&frac_hi,n_params,FLOAT);
	    vector(&frac_lo,n_params,FLOAT);
	}

	status = accumulate_fractions_in_layer(front,height,frac,
                                               rfactor,origin);
        if ((status == FUNCTION_FAILED))
        {
            (void) printf("WARNING in height_at_fraction(), "
                          "unable to locate height at which frac[%d] = %g\n",
                          index,fraction);
            (void) printf("\tcurrent height = %g, direction = %d\n",*h0,dir);
            debug_print("gfrac","Left height_at_fraction()\n");
            return FUNCTION_FAILED;
        }

	if (frac[index] >= fraction)
	{
	    height += dir*dh*fraction/frac[index];
	    *h0 = height;
	    (void) printf("WARNING in height_at_fraction(), "
			  "frac >= %g occurred at initial height of %g.\n",
			  fraction,height);
	    debug_print("gfrac","Left height_at_fraction()\n");
	    return FUNCTION_SUCCEEDED;
	}

	if (debugging("gfrac"))
	{
	    (void) printf("\theight = %g, fraction = %g, index = %d\n",
			  height,fraction,index);
	    (void) printf("\tAfter initialization, "
			  "frac[%d] = %g, frac[%d] = %g\n",
			  index,frac[index],1-index,frac[1-index]);
	}

	do
	{
	    status = accumulate_fractions_in_layer(front,height+dir*dh,
						   new_frac,rfactor,origin);
	    if (status == FUNCTION_SUCCEEDED)
	    {
                if ((  (frac[index] <= fraction-ftol) &&
                       (fraction+ftol <= new_frac[index])) ||
                     ( (new_frac[index] <= fraction-ftol) &&
                       (fraction+ftol <= frac[index])))
	        {   /* we've bracketed the height, so interpolate */
	            height += dir*dh*(fraction-frac[index]) /
			             (new_frac[index]-frac[index]);
		    break;
	        }
	        else
	        {
	            for (p = 0; p < n_params; p++)
		        frac[p] = new_frac[p];
		    height += dir*dh;
	        }
	    }
	} while (status == FUNCTION_SUCCEEDED);

	if (status == FUNCTION_FAILED)
	{
	    (void) printf("WARNING in height_at_fraction(), "
	                  "unable to locate height at which frac[%d] = %g\n",
			  index,fraction);
	    (void) printf("\toriginal height = %g, direction = %d\n",*h0,dir);
	    (void) printf("\tcurrent height = %g\n",height);
	    debug_print("gfrac","Left height_at_fraction()\n");
	    return FUNCTION_FAILED;
	}

	if ((frac[index] <= fraction) && (fraction <= new_frac[index]))
	{
	    hlo = height - dir*dh;
	    hhi = height;
	    for (p = 0; p < n_params; p++)
	    {
	        frac_lo[p] = frac[p];
		frac_hi[p] = new_frac[p];
	    }
	}
	else
	{
	    hlo = height;
	    hhi = height - dir*dh;
	    for (p = 0; p < n_params; p++)
	    {
	        frac_lo[p] = new_frac[p];
		frac_hi[p] = frac[p];
	    }
	}

	if (debugging("gfrac"))
	{
	    (void) printf("\theight at frac = %g has been framed.\n",fraction);
	    (void) printf("\thlo = %g; frac_lo[%d] = %g\n",
			  hlo,index,frac_lo[index]);
	    (void) printf("\thhi = %g; frac_hi[%d] = %g\n",
			  hhi,index,frac_hi[index]);
	}

	for (iter2 = 0; iter2 < max_num_iters; iter2++)
	{
	    height = 0.5*(hlo + hhi);
	    status = accumulate_fractions_in_layer(front,height,
						   new_frac,rfactor,origin);
	    if (status == FUNCTION_FAILED)
	    {
	        screen("ERROR in height_at_fraction(), "
		       "\taccumulate_fractions_in_layer() failed during"
		       "bisection loop.\n");
		clean_up(ERROR);
	    }
	    if (fraction <= new_frac[index])
	    {
	        hhi = height;
		for (p = 0; p < n_params; p++)
		    frac_hi[p] = new_frac[p];
	    }
	    else
	    {
	        hlo = height;
		for (p = 0; p < n_params; p++)
		    frac_lo[p] = new_frac[p];
	    }
	}

        if( fabs(frac_hi[index]-frac_lo[index]) >= ftol)
            height = ((fraction-frac_lo[index])*hhi+(frac_hi[index]-fraction)
                                        *hlo)/(frac_hi[index]-frac_lo[index]);
        else
            height = hhi;

	(void) accumulate_fractions_in_layer(front,height,frac,rfactor,origin);

	if (frac[index] == 0.0 || frac[index] == 1.0)
	{
	    (void) printf("WARNING in height_at_fraction(), "
	                  "Interpolated height of %g has fraction = %g\n",
			  height,frac[index]);
	    if (frac[index] == 0.0)
	    {
	        height = hhi;
		for (p = 0; p < n_params; p++)
		    frac[p] = frac_hi[p];
		(void) printf("\tWill use h = %g, frac = %g instead.\n",
			      hhi, frac[index]);
	    }
	    else
	    {
	        height = hlo;
		for (p = 0; p < n_params; p++)
		    frac[p] = frac_lo[p];
		(void) printf("\tWill use h = %g, frac = %g instead.\n",
			      hlo, frac[index]);
	    }
	}

	if (debugging("gfrac"))
	{
	    (void) printf("\tAfter %d bisection iterations,\n",max_num_iters);
	    (void) printf("\theight = %0.8e; frac[%d] = %0.8e\n",
		          height,index,frac[index]);
	    (void) printf("\thlo = %0.8e; frac_lo[%d] = %0.8e\n",
		          hlo,index,frac_lo[index]);
	    (void) printf("\thhi = %0.8e; frac_hi[%d] = %0.8e\n",
		          hhi,index,frac_hi[index]);
	}

	*h0 = height;
	debug_print("gfrac","Left height_at_fraction()\n");
	return FUNCTION_SUCCEEDED;
}		/*end height_at_fraction*/

/*			record_intfc_extrema()
*
*	Find the extremal points of all contact surfaces, the states at these
*	points, and the ambient states (averaged across the layer at the same
*	height or radius), and print the results.
*
*	NOTE:  The performance of this function in situations where one or more
*	extremal points occur on pieces of the contact surface that have broken
*	off of the main branch is untested.
*/

/*ARGSUSED*/
LOCAL	void record_intfc_extrema(
	Grid		*grid,
	Wave		*wave,
	Front		*front,
	Printplot	*prt,
	OUTPUT_DATA	*out,
	bool		about_to_stop)
{
	Big_State	   bst[2], amb_st[2];
	HYPER_SURF	   *hs, *hs_max, *hs_min;
	HYPER_SURF_ELEMENT *hse, *hse_max, *hse_min;
	INTERFACE	   *intfc = front->interf;
	Intfc_extrema 	   *iext = Intfc_extrema_data(out);
	Locstate	   sminl, sminr, smaxl, smaxr;
	POINT		   *p, *p_max, *p_min;
	const float	   *origin = (iext->geom == spherical) ?
				      iext->origin : NULL;
	const int	   dim = front->rect_grid->dim;
	const int 	   myid = pp_mynode();
	const int	   nn = pp_numnodes();
	const int	   zdir = dim - 1;
	const float        *L = front->rect_grid->L;
	const float        *U = front->rect_grid->U;
	float		   g_h_at_min, g_h_at_max;
	float		   h, h_at_min, h_at_max;
	float		   nor_at_max[MAXD], nor_at_min[MAXD];
	int                index_at_max = iext->index_at_max;
	int                index_at_min = iext->index_at_min;
	int		   i;
	static const size_t size_bst = sizeof(Big_State);
	static float 	    **min_by_node = NULL;
	static float 	    **max_by_node = NULL;

	debug_print("gintext","Entered record_intfc_extrema()\n");
	start_clock("record_intfc_extrema");

	if (min_by_node == NULL)
	{
	    matrix(&min_by_node,nn,MAXD,FLOAT);
	    matrix(&max_by_node,nn,MAXD,FLOAT);
	    if (is_io_node(myid))
	    {
	        print_intfc_extrema_headers(iext,dim);
		if (iext->do_01 == YES)
		    print_data_at_height_headers(iext->out_min1,iext->out_max1,
						 iext->out_amp1,dim);
		if (iext->do_05 == YES)
		    print_data_at_height_headers(iext->out_min5,iext->out_max5,
						 iext->out_amp5,dim);
	    }
	}

	h_at_min =  HUGE_VAL;
	h_at_max = -HUGE_VAL;
	for (i = 0; i < dim; i++)
	{
            /*  old 082603
	    iext->pos[0][i] = -HUGE_VAL;
            */
            iext->pos[0][i] = HUGE_VAL;
	    iext->pos[1][i] = -HUGE_VAL;
	}

	/*
	*	Loop over all points in every interface, to find the
	*	extremal points of the contact surfaces.
	*
	* 	The method used here is a little wasteful, as it loops
	*	over all points of irrelevant interfaces.
	*/

	start_clock("find_interface_extrema");

	p_max = p_min = NULL;
	(void) next_point(intfc,NULL,NULL,NULL);
	while (next_point(intfc,&p,&hse,&hs))
	{
	    if (!is_scalar_wave(wave_type(hs)))
		continue;

	    /* Make sure p is on the actual grid */
	    for (i = 0; i < dim; i++)
		if (Coords(p)[i] < L[i] || Coords(p)[i] >= U[i])
		    break;
	    if (i < dim)
		continue;

	    h = (iext->geom == planar) ? Coords(p)[zdir] :
	                                 distance_between_positions(Coords(p),
								    origin,dim);

	    if (h < h_at_min)
	    {
	        h_at_min = h;
	        p_min = p;
	        hs_min = hs;
	        hse_min = hse;
	    }
	    if (h > h_at_max)
	    {
	        h_at_max = h;
	        p_max = p;
	        hs_max = hs;
	        hse_max = hse;
	    }
	}

	/*  Now find normal vector and states at min & max */

	if ((p_max != NULL) && (p_min != NULL))
	{
	    normal(p_max,hse_max,hs_max,nor_at_max,front);
	    normal(p_min,hse_min,hs_min,nor_at_min,front);
	    slsr(p_max,hse_max,hs_max,&smaxl,&smaxr);
	    slsr(p_min,hse_min,hs_min,&sminl,&sminr);

	    for (i = 0; i < dim; i++)
	    {
		iext->pos[1][i] = Coords(p_max)[i];
		iext->pos[0][i] = Coords(p_min)[i];
	    }

	    if (index_of_Gas_param(gas_params_for_comp(hs_max->neg_comp,intfc))
			    == index_at_max)
	        copy_state_to_Big_State(smaxl,&bst[1]);
	    else
		copy_state_to_Big_State(smaxr,&bst[1]);

	    if (index_of_Gas_param(gas_params_for_comp(hs_min->pos_comp,intfc))
		            == index_at_min)
		copy_state_to_Big_State(sminr,&bst[0]);
	    else
		copy_state_to_Big_State(sminl,&bst[0]);

	}  /* if ((p_max != NULL) and ... ) */

	/* Compute global max and min */

	if (nn > 1)
	{
	    int i_min, i_max;

	    for (i = 0; i < dim; i++)
	    {
	        min_by_node[myid][i] = iext->pos[0][i];
		max_by_node[myid][i] = iext->pos[1][i];
	    }

	    /* I/O node collects each of the other nodes'
	       minimum and maximum position. */

	    if (is_io_node(myid))
	    {
	        for (i = 0; i < nn; i++)
		{
		    if (i == myid)
			continue;
		    pp_recv(L_MIN_MAX_ID+i,i,min_by_node[i],MAXD*FLOAT);
		    pp_recv(L_MIN_MAX_ID+nn+i,i,max_by_node[i],MAXD*FLOAT);
		}
	    }
	    else
	    {
	        pp_send(L_MIN_MAX_ID+myid,min_by_node[myid],
			MAXD*FLOAT,IO_NODE_ID);
	        pp_send(L_MIN_MAX_ID+nn+myid,max_by_node[myid],
			MAXD*FLOAT,IO_NODE_ID);
	    }

	    if (debugging("gintext"))
	        (void) printf("Local extrema have been communicated.\n");

	    /* I/O node has a list of all the local extrema, arranged by
	       by node index, so now it can find the global extrema. */

	    if (is_io_node(myid))
	    {
	        switch(iext->geom)
		{
		case planar:
		    g_h_at_min = min_by_node[myid][zdir];
		    g_h_at_max = max_by_node[myid][zdir];
		    break;
		case spherical:
		    g_h_at_min = distance_between_positions(min_by_node[myid],
							    origin,dim);
		    g_h_at_max = distance_between_positions(max_by_node[myid],
							    origin,dim);
		    break;
		}

		i_max = i_min = myid;

		for (i = 0; i < nn; i++)
		{
		    switch(iext->geom)
		    {
		    case planar:
			h_at_min = min_by_node[i][zdir];
			h_at_max = max_by_node[i][zdir];
			break;
		    case spherical:
			h_at_min = distance_between_positions(min_by_node[i],
							      origin,dim);
			h_at_max = distance_between_positions(max_by_node[i],
							      origin,dim);
			break;
		    }
		    if (h_at_min < g_h_at_min)
		    {
			g_h_at_min = h_at_min;
			i_min = i;
		    }
		    if (h_at_max > g_h_at_max)
		    {
			g_h_at_max = h_at_max;
			i_max = i;
		    }
		}
		pp_send_all(G_IMIN_ID,&i_min,INT);
		pp_send_all(G_IMAX_ID,&i_max,INT);
		pp_send_all(G_HMIN_ID,&g_h_at_min,FLOAT);
		pp_send_all(G_HMAX_ID,&g_h_at_max,FLOAT);
	    }
	    else
	    {
	        pp_recv(G_IMIN_ID,IO_NODE_ID,&i_min,INT);
		pp_recv(G_IMAX_ID,IO_NODE_ID,&i_max,INT);
		pp_recv(G_HMIN_ID,IO_NODE_ID,&g_h_at_min,FLOAT);
		pp_recv(G_HMAX_ID,IO_NODE_ID,&g_h_at_max,FLOAT);
	    }

	    /* See if this node has a global extremum, and if so,
	       send the associated information to the I/O node. */

	    if (myid == i_min)
	    {
	        if (debugging("gintext"))
		    (void) printf("I have the global minimum.\n");
		pp_send_all(MIN_POS_ID,iext->pos[0],dim*FLOAT);
		pp_send_all(MIN_STATE_ID,&bst[0],size_bst);
	    }
	    else
	    {
	        if (debugging("gintext"))
		    (void) printf("Node %d has the global minimum.\n",i_min);
		pp_recv(MIN_POS_ID,i_min,iext->pos[0],dim*FLOAT);
		pp_recv(MIN_STATE_ID,i_min,&bst[0],size_bst);
	    }
	    if (myid == i_max)
	    {
	        if (debugging("gintext"))
		    (void) printf("I have the global maximum.\n");
		pp_send_all(MAX_POS_ID,iext->pos[1],dim*FLOAT);
		pp_send_all(MAX_STATE_ID,&bst[1],size_bst);
	    }
	    else
	    {
	        if (debugging("gintext"))
		    (void) printf("Node %d has the global maximum.\n",i_max);
		pp_recv(MAX_POS_ID,i_max,iext->pos[1],dim*FLOAT);
		pp_recv(MAX_STATE_ID,i_max,&bst[1],size_bst);
	    }

	}  /* if (nn > 1) */
	else
	{
	    g_h_at_min = h_at_min;
	    g_h_at_max = h_at_max;
	}

	/* Get the ambient states by a global layer average. */

	accumulate_state_in_layer(wave,front,g_h_at_min,&amb_st[0],NULL,
				  iext->rfactor,origin,YES,YES,YES);
	accumulate_state_in_layer(wave,front,g_h_at_max,&amb_st[1],NULL,
				  iext->rfactor,origin,YES,YES,YES);

	stop_clock("find_interface_extrema");

	if (is_io_node(myid))
	    print_intfc_extrema(grid,front,out,bst,amb_st,about_to_stop);

	/* Now that we're finished with intfc extrema, we work on the
	   volume fraction levels, if any were specified.

	   We need to find the height NEAREST THE EDGE corresponding to a given
	   volume fraction.  Bisection is not robust because the volume
	   fraction may not be monotone; instead we need to start at the edge
	   and then move into the mixing zone. */

	if (iext->do_01 == YES)     /* 1% specified? */
	{
	    Big_State tmp_bst[2];
	    bool   min_result, max_result;
	    float     dh = (g_h_at_max - g_h_at_min)/100.0;
	    float     frac_at_min[2], frac_at_max[2];

	    start_clock("find_1%%_levels");

	    h_at_min = g_h_at_min - dh;
	    h_at_max = g_h_at_max + dh;

	    min_result = height_at_fraction(front,&h_at_min,
					    dh,1,0.01,frac_at_min,index_at_min,
					    iext->rfactor,origin);
	    max_result = height_at_fraction(front,&h_at_max,
					    dh,-1,0.01,frac_at_max,index_at_max,
					    iext->rfactor,origin);

	    if (debugging("gfrac"))
	    {
	        (void) printf("After calls to height_at_fraction():\n");
		(void) printf("\th_at_min = %g; frac[%d] = %g\n",
		              h_at_min,index_at_min,frac_at_min[index_at_min]);
		(void) printf("\th_at_max = %g; frac[%d] = %g\n",
		              h_at_max,index_at_max,frac_at_max[index_at_max]);
	    }

	    accumulate_state_in_layer(wave,front,h_at_min,tmp_bst,NULL,
				      iext->rfactor,origin,NO,YES,YES);

	    copy_Big_State(&tmp_bst[index_at_min],&bst[0]);
	    copy_Big_State(&tmp_bst[1-index_at_min],&amb_st[0]);

	    accumulate_state_in_layer(wave,front,h_at_max,tmp_bst,NULL,
				      iext->rfactor,origin,NO,YES,YES);

	    copy_Big_State(&tmp_bst[index_at_max],&bst[1]);
	    copy_Big_State(&tmp_bst[1-index_at_max],&amb_st[1]);

	    if (debugging("gfrac"))
	    {
	        (void) printf("After calls to accumulate_state_in_layer():\n");
		(void) printf("\t\tAt min, frac[%d] = %g\n",
			      index_at_min,bst[0].frac);
		(void) printf("\t\tAt max, frac[%d] = %g\n",
			      index_at_max,bst[1].frac);
	    }

	    if (is_io_node(myid) &&
		(min_result == FUNCTION_SUCCEEDED) &&
		(max_result == FUNCTION_SUCCEEDED) &&
		(bst[0].count > 0) &&
		(bst[1].count > 0))
	        print_data_at_height(grid,front,out,bst,amb_st,
				     iext->out_min1,iext->out_max1,
				     iext->out_amp1,h_at_min,
				     h_at_max,about_to_stop);

	    stop_clock("find_1%%_levels");

	} /* if (iext->do_01 == YES) */

	if (iext->do_05 == YES)     /* 5% specified? */
	{
	    float dh = (g_h_at_max - g_h_at_min)/100.0;
	    float frac_at_min[2], frac_at_max[2];
	    Big_State tmp_bst[2];
	    bool min_result, max_result;

	    start_clock("find_1%%_levels");

	    h_at_min = g_h_at_min - dh;
	    h_at_max = g_h_at_max + dh;

	    min_result = height_at_fraction(front,&h_at_min,
					    dh,1,0.05,frac_at_min,index_at_min,
					    iext->rfactor,origin);
	    max_result = height_at_fraction(front,&h_at_max,
					    dh,-1,0.05,frac_at_max,index_at_max,
					    iext->rfactor,origin);

	    if (debugging("gfrac"))
	    {
	        printf("After calls to height_at_fraction():\n");
		printf("\th_at_min = %g; frac[%d] = %g\n",
		       h_at_min,index_at_min,frac_at_min[index_at_min]);
		printf("\th_at_max = %g; frac[%d] = %g\n",
		       h_at_max,index_at_max,frac_at_max[index_at_max]);
	    }

	    accumulate_state_in_layer(wave,front,h_at_min,
				      tmp_bst,NULL,
				      iext->rfactor,origin,
				      NO,YES,YES);

	    copy_Big_State(&tmp_bst[index_at_min],&bst[0]);
	    copy_Big_State(&tmp_bst[1-index_at_min],&amb_st[0]);

	    accumulate_state_in_layer(wave,front,h_at_max,
				      tmp_bst,NULL,
				      iext->rfactor,origin,
				      NO,YES,YES);

	    copy_Big_State(&tmp_bst[index_at_max],&bst[1]);
	    copy_Big_State(&tmp_bst[1-index_at_max],&amb_st[1]);

	    if (debugging("gfrac"))
	    {
	        (void) printf("After calls to accumulate_state_in_layer():\n");
		(void) printf("\t\tAt min, frac[%d] = %g\n",
			      index_at_min,bst[0].frac);
		(void) printf("\t\tAt max, frac[%d] = %g\n",
			      index_at_max,bst[1].frac);
	    }

	    if (is_io_node(myid) &&
		(min_result == FUNCTION_SUCCEEDED) &&
		(max_result == FUNCTION_SUCCEEDED) &&
		(bst[0].count > 0) &&
		(bst[1].count > 0))
	        print_data_at_height(grid,front,out,bst,amb_st,
				     iext->out_min5,iext->out_max5,
				     iext->out_amp5,h_at_min,
				     h_at_max,about_to_stop);

	    stop_clock("find_1%%_levels");

	} /* if (iext->do_05 == YES) */

	stop_clock("record_intfc_extrema");
	debug_print("gintext","Left record_intfc_extrema()\n");
}		/*end record_intfc_extrema*/

LOCAL   void print_intfc_extrema(
	const Grid      *grid,
	const Front     *front,
	OUTPUT_DATA     *out,
	const Big_State bst[2],
	const Big_State amb_st[2],
	bool         about_to_stop)
{
	const Intfc_extrema *iext = (Intfc_extrema*)out;
	const int 	    dim = front->rect_grid->dim;
	float		    vel_at_max[MAXD], vel_at_min[MAXD];
	float		    amb_vel_at_max[3], amb_vel_at_min[3];
	float		    sph_pos_at_min[3], sph_pos_at_max[3];
	float		    sph_vel_at_min[3], sph_vel_at_max[3];
	float		    rhat_min[3], thhat_min[3], phhat_min[3];
	float		    rhat_max[3], thhat_max[3], phhat_max[3];
	register int	    i;
	static int	    n_min, n_max, n_amp;
	static float	    *fmax = NULL, *fmin = NULL, *famp = NULL;
	FILE                *omin = iext->out_min;
	FILE                *omax = iext->out_max;
	FILE                *oamp = iext->out_amp;

	debug_print("gintext","Entered print_intfc_extrema()\n");
	start_clock("print_intfc_extrema");

	if (fmin == NULL)
	{
	    n_min = n_max = 10;
	    n_amp = 3+4*dim;
	    vector(&fmin,n_min,FLOAT);
	    vector(&fmax,n_max,FLOAT);
	    vector(&famp,n_amp,FLOAT);
	}

	for (i = 0; i < dim; i++)
	{
	    vel_at_min[i] = bst[0].v[i];
	    vel_at_max[i] = bst[1].v[i];
	    amb_vel_at_min[i] = amb_st[0].v[i];
	    amb_vel_at_max[i] = amb_st[1].v[i];
	}

	if (iext->geom == spherical)
	{
	    convert_to_spherical(sph_pos_at_min,iext->pos[0],iext->origin,dim);
	    convert_to_spherical(sph_pos_at_max,iext->pos[1],iext->origin,dim);
	}

	fmin[0] = fmax[0] = famp[0] = grid->time;

	switch(iext->geom)
	{
	case planar:
	    fmin[1] = iext->pos[0][dim-1];
	    fmax[1] = iext->pos[1][dim-1];
  	    fmin[2] = vel_at_min[dim-1];
	    fmax[2] = vel_at_max[dim-1];
	    fmin[3] = amb_vel_at_min[dim-1];
	    fmax[3] = amb_vel_at_max[dim-1];
	    famp[1] = 0.5*(iext->pos[1][dim-1] - iext->pos[0][dim-1]);
	    famp[2] = 0.5*(vel_at_max[dim-1] - vel_at_min[dim-1]);

	    for (i = 1; i <= dim; i++)
	    {
	        famp[2+i] = iext->pos[0][i-1];
		famp[2+dim+i] = iext->pos[1][i-1];
		famp[2+2*dim+i] = vel_at_min[i-1];
		famp[2+3*dim+i] = vel_at_max[i-1];
	    }
	    break;

	case spherical:
	    spherical_unit_vectors(rhat_min,thhat_min,phhat_min,
				   iext->pos[0],iext->origin,dim);
	    spherical_unit_vectors(rhat_max,thhat_max,phhat_max,
				   iext->pos[1],iext->origin,dim);
	    sph_vel_at_min[0] = scalar_product(vel_at_min,rhat_min,dim);
	    sph_vel_at_max[0] = scalar_product(vel_at_max,rhat_max,dim);
	    sph_vel_at_min[1] = scalar_product(vel_at_min,phhat_min,dim);
	    sph_vel_at_max[1] = scalar_product(vel_at_max,phhat_max,dim);
	    if (dim == 3)
	    {
	        sph_vel_at_min[2] = scalar_product(vel_at_min,thhat_min,dim);
	        sph_vel_at_max[2] = scalar_product(vel_at_max,thhat_max,dim);
	    }
	    fmin[1] = sph_pos_at_min[0];
	    fmax[1] = sph_pos_at_max[0];
	    fmin[2] = sph_vel_at_min[0];
	    fmax[2] = sph_vel_at_max[0];
	    fmin[3] = amb_vel_at_min[0];
	    fmax[3] = amb_vel_at_max[0];
	    famp[1] = 0.5*(sph_pos_at_max[0] - sph_pos_at_min[0]);
	    famp[2] = 0.5*(fmax[2] - fmin[2]);

	    for (i = 1; i <= dim; i++)
	    {
	        famp[2+i] = sph_pos_at_min[i-1];
		famp[2+dim+i] = sph_pos_at_max[i-1];
		famp[2+2*dim+i] = sph_vel_at_min[i-1];
		famp[2+3*dim+i] = sph_vel_at_max[i-1];
	    }
	    break;
	}

	fmin[4] = bst[0].d;
	fmax[4] = bst[1].d;
	fmin[5] = amb_st[0].d;
	fmax[5] = amb_st[1].d;
	fmin[6] = bst[0].p;
	fmax[6] = bst[1].p;
	fmin[7] = amb_st[0].p;
	fmax[7] = amb_st[1].p;
	fmin[8] = bst[0].e;
	fmax[8] = bst[1].e;
	fmin[9] = amb_st[0].e;
	fmax[9] = amb_st[1].e;

	for (i = 0; i < n_min; i++)
	    (void) fprintf(omin,"%15.5e",fmin[i]);
	if (debugging("gintext"))
	    (void) fprintf(omin,"%10d",amb_st[0].count);
	(void) fprintf(omin,"\n");

	for (i = 0; i < n_max; i++)
	    (void) fprintf(omax,"%15.5e",fmax[i]);
	if (debugging("gintext"))
	    (void) fprintf(omax,"%10d",amb_st[1].count);
	(void) fprintf(omax,"\n");

	for (i = 0; i < n_amp; i++)
	    (void) fprintf(oamp,"%15.5e",famp[i]);
	(void) fprintf(oamp,"\n");

	if (about_to_stop == YES)
	{
	    (void) fprintf(omin,"\n");
	    (void) fprintf(omax,"\n");
	    (void) fprintf(oamp,"\n");
	}

	stop_clock("print_intfc_extrema");
	debug_print("gintext","Left print_intfc_extrema()\n");
}		/*end print_intfc_extrema*/

LOCAL   void    print_intfc_extrema_headers(
	const Intfc_extrema*    iext,
	int                     dim)
{
	FILE *omax = iext->out_max;
	FILE *omin = iext->out_min;
	FILE *oamp = iext->out_amp;
	char xyz[3] = {'x', 'y', 'z'};
	char rtp[3] = {'r', 't', 'p'};
	int  i;

	(void) foutput(omin);
	(void) foutput(omax);
	(void) foutput(oamp);
	(void) fprintf(omin,"\n# %13s","time");
	(void) fprintf(omax,"\n# %13s","time");
	(void) fprintf(oamp,"\n# %13s","time");

	(void) fprintf(oamp," %14s %14s", "a", "adot");

	switch(iext->geom)
	{
	case planar:
	    for (i = 0; i < dim; i++)
	        (void) fprintf(oamp,"          %c_%3s",xyz[i],"min");
	    for (i = 0; i < dim; i++)
	        (void) fprintf(oamp,"          %c_%3s",xyz[i],"max");
	    for (i = 0; i < dim; i++)
	        (void) fprintf(oamp,"         v%c_%3s",xyz[i],"min");
	    for (i = 0; i < dim; i++)
	        (void) fprintf(oamp,"         v%c_%3s",xyz[i],"max");
	    break;
	case spherical:
	    for (i = 0; i < dim; i++)
	        (void) fprintf(oamp,"          %c_%3s",rtp[i],"min");
	    for (i = 0; i < dim; i++)
	        (void) fprintf(oamp,"          %c_%3s",rtp[i],"max");
	    for (i = 0; i < dim; i++)
	        (void) fprintf(oamp,"         v%c_%3s",rtp[i],"min");
	    for (i = 0; i < dim; i++)
	        (void) fprintf(oamp,"         v%c_%3s",rtp[i],"max");
	    break;
	}

	(void) fprintf(omin,"          %c_%3s",'h',"min");
	(void) fprintf(omax,"          %c_%3s",'h',"max");

	(void) fprintf(omin," %14s %14s %14s %14s",
		       "vel", "amb_vel", "den", "amb_den");
	(void) fprintf(omin," %14s %14s", "pre", "amb_pre");
	(void) fprintf(omax," %14s %14s %14s %14s",
		       "vel", "amb_vel", "den", "amb_den");
	(void) fprintf(omax," %14s %14s", "pre", "amb_pre");

	(void) fprintf(omin," %14s %14s", "ien", "amb_ien");
	(void) fprintf(omax," %14s %14s", "ien", "amb_ien");

	(void) fprintf(omin,"\n");
	(void) fprintf(omax,"\n");
	(void) fprintf(oamp,"\n");
}		/*end print_intfc_extrema_headers*/

LOCAL   void print_data_at_height(
	const Grid      *grid,
	const Front     *front,
	OUTPUT_DATA     *out,
	const Big_State bst[2],
	const Big_State amb_st[2],
	FILE		*omin,
	FILE		*omax,
	FILE		*oamp,
	float		h_at_min,
	float		h_at_max,
	bool         about_to_stop)
{
	const Intfc_extrema *iext = (Intfc_extrema*)out;
	const int 	dim = front->rect_grid->dim;
	float		vel_at_max[3], vel_at_min[3];
	float		amb_vel_at_max[3], amb_vel_at_min[3];
	register int	i;
	static int	n_min, n_max, n_amp;
	static float	*fmax = NULL, *fmin = NULL, *famp = NULL;

	debug_print("gintext","Entered print_data_at_height()\n");
	start_clock("print_data_at_height");

	if (fmin == NULL)
	{
	    n_min = n_max = 10;
	    n_amp = 5 + 2*dim;
	    vector(&fmin,n_min,FLOAT);
	    vector(&fmax,n_max,FLOAT);
	    vector(&famp,n_amp,FLOAT);
	}

	for (i = 0; i < dim; i++)
	{
	    vel_at_min[i] = bst[0].v[i];
	    vel_at_max[i] = bst[1].v[i];
	    amb_vel_at_min[i] = amb_st[0].v[i];
	    amb_vel_at_max[i] = amb_st[1].v[i];
	}

	fmin[0] = fmax[0] = famp[0] = grid->time;
	fmin[1] = h_at_min;
	fmax[1] = h_at_max;

	switch(iext->geom)
	{
	case planar:
  	    fmin[2] = vel_at_min[dim-1];
	    fmax[2] = vel_at_max[dim-1];
	    fmin[3] = amb_vel_at_min[dim-1];
	    fmax[3] = amb_vel_at_max[dim-1];
	    break;

	case spherical:
	    fmin[2] = vel_at_min[0];
	    fmax[2] = vel_at_max[0];
	    fmin[3] = amb_vel_at_min[0];
	    fmax[3] = amb_vel_at_max[0];
	    break;
	}

	famp[1] = 0.5*(h_at_max - h_at_min);
	famp[2] = 0.5*(fmax[2] - fmin[2]);

	fmin[4] = bst[0].d;
	fmax[4] = bst[1].d;
	fmin[5] = amb_st[0].d;
	fmax[5] = amb_st[1].d;
	fmin[6] = bst[0].p;
	fmax[6] = bst[1].p;
	fmin[7] = amb_st[0].p;
	fmax[7] = amb_st[1].p;
	fmin[8] = bst[0].e;
	fmax[8] = bst[1].e;
	fmin[9] = amb_st[0].e;
	fmax[9] = amb_st[1].e;

	for (i = 1; i <= dim; i++)
	{
	    famp[2+i] = vel_at_min[i-1];
	    famp[2+dim+i] = vel_at_max[i-1];
	}
	famp[3+2*dim] = bst[0].frac;
	famp[4+2*dim] = bst[1].frac;

	for (i = 0; i < n_min; i++)
	    (void) fprintf(omin,"%15.5e",fmin[i]);
	if (debugging("gintext"))
	    (void) fprintf(omin,"%15.5e",bst[0].frac);
	(void) fprintf(omin,"\n");

	for (i = 0; i < n_max; i++)
	    (void) fprintf(omax,"%15.5e",fmax[i]);
	if (debugging("gintext"))
	    (void) fprintf(omax,"%15.5e",bst[1].frac);
	(void) fprintf(omax,"\n");

	for (i = 0; i < n_amp; i++)
	    (void) fprintf(oamp,"%15.5e",famp[i]);
	(void) fprintf(oamp,"\n");

	if (about_to_stop == YES)
	{
	    (void) fprintf(omin,"\n");
	    (void) fprintf(omax,"\n");
	    (void) fprintf(oamp,"\n");
	}

	stop_clock("print_data_at_height");
	debug_print("gintext","Left print_data_at_height()\n");
}		/*end print_data_at_height*/


LOCAL   void    print_data_at_height_headers(
	FILE *omin,
	FILE *omax,
	FILE *oamp,
	int  dim)
{
	int i;

	char xyz[3] = {'x', 'y', 'z'};

	(void) foutput(omin);
	(void) foutput(omax);
	(void) foutput(oamp);

	(void) fprintf(omin,"\n# %13s","time");
	(void) fprintf(omax,"\n# %13s","time");
	(void) fprintf(oamp,"\n# %13s","time");

	(void) fprintf(omin,"          %c_%3s",'h',"min");
	(void) fprintf(omax,"          %c_%3s",'h',"max");

	(void) fprintf(omin," %14s %14s %14s %14s",
		       "vel", "amb_vel", "den", "amb_den");
	(void) fprintf(omin," %14s %14s", "pre", "amb_pre");
	(void) fprintf(omax," %14s %14s %14s %14s",
		       "vel", "amb_vel", "den", "amb_den");
	(void) fprintf(omax," %14s %14s", "pre", "amb_pre");

	(void) fprintf(oamp," %14s %14s", "a", "adot");

	for (i = 0; i < dim; i++)
	    (void) fprintf(oamp,"         v%c_%3s",xyz[i],"min");
	for (i = 0; i < dim; i++)
	    (void) fprintf(oamp,"         v%c_%3s",xyz[i],"max");

	(void) fprintf(oamp," %14s %14s", "lower_frac", "upper_frac");

	(void) fprintf(omin," %14s %14s", "ien", "amb_ien");
	(void) fprintf(omax," %14s %14s", "ien", "amb_ien");

	(void) fprintf(omin,"\n");
	(void) fprintf(omax,"\n");
	(void) fprintf(oamp,"\n");
}		/*end print_data_at_height_headers*/

LOCAL	void	zero_state_totals(
	Big_State          *bst,
	Turbulence_moments *turb,
	int                n_params)
{
	const size_t 	size_bst = sizeof(Big_State);
	const size_t	size_turb = sizeof(Turbulence_moments);
	int 		p;

	if (bst != NULL)
	    for (p = 0; p < n_params; p++)
	        zero_scalar(&bst[p],size_bst);
	if (turb != NULL)
	    for (p = 0; p < n_params; p++)
	        zero_scalar(&turb[p],size_turb);
}		/*end zero_state_totals*/

LOCAL	void 	add_state_to_totals(
	const Locstate     st,
	Big_State          *bst,
	Turbulence_moments *turb)
{
	int	i, j;
	float	d, p, k, e;
	float	v[MAXD];

	bst->count++;
	if (is_obstacle_state(st))
	    return;

	d = Dens(st);
	p = pressure(st);
	k = kinetic_energy(st);
	e = internal_energy(st);
	bst->d += d;
	for (i = 0; i < MAXD; i++)
	{
	    v[i] = vel(i,st);
	    bst->v[i] += v[i];
	}
	bst->p += p;
	bst->k += k;
	bst->e += e;
	if (turb != NULL)
	{
	    turb->dd += d*d;
	    turb->de += d*e;
	    for (i = 0; i < MAXD; i++)
	    {
	        turb->dv[i] += d*v[i];
		turb->dev[i] += d*e*v[i];
		turb->dkv[i] += d*k*v[i];
		turb->pv[i] += p*v[i];
		for (j = i; j < MAXD; j++)
		{
		    turb->dvv[sym_index(i,j)] += d*v[i]*v[j];
		    turb->vv[sym_index(i,j)] += v[i]*v[j];
		}
	    }
	}
}		/*end add_state_to_totals*/

EXPORT	void 	accumulate_state_totals(
	const Big_State          *bst1,
	const Turbulence_moments *turb1,
	Big_State                *bst2,
	Turbulence_moments       *turb2)
{
	int	i, j;

	if (bst2 != NULL && bst1 != NULL)
	{
	    bst2->count += bst1->count;
	    bst2->d += bst1->d;
	    for (i = 0; i < MAXD; i++)
	        bst2->v[i] += bst1->v[i];
	    bst2->p += bst1->p;
	    bst2->k += bst1->k;
	    bst2->e += bst1->e;
	}

	if (turb2 != NULL && turb1 != NULL)
	{
	    turb2->dd += turb1->dd;
	    turb2->de += turb1->de;
	    for (i = 0; i < MAXD; i++)
	    {
	        turb2->dv[i] += turb1->dv[i];
		turb2->dev[i] += turb1->dev[i];
		turb2->dkv[i] += turb1->dkv[i];
		turb2->pv[i] += turb1->pv[i];
		for (j = i; j < MAXD; j++)
		{
		    turb2->vv[sym_index(i,j)] += turb1->vv[sym_index(i,j)];
		    turb2->dvv[sym_index(i,j)] += turb1->dvv[sym_index(i,j)];
		}
	    }
	}
}		/*end accumulate_state_totals*/

EXPORT	void	normalize_state_totals(
	Big_State          *bst,
	Turbulence_moments *turb,
	int                n_points)
{
	int i, j;
	float nf;

	if (bst->count == 0)
	    return;

	nf = (float)bst->count;

	bst->d /= nf;
	bst->p /= nf;
	bst->e /= nf;
	bst->k /= nf;
	for (i = 0; i < MAXD; i++)
	    bst->v[i] /= nf;
	bst->frac = nf/n_points;

	if (turb != NULL)
	{
	    turb->dd /= nf;
	    turb->de /= nf;
	    for (i = 0; i < MAXD; i++)
	    {
	        turb->dv[i] /= nf;
		turb->dkv[i] /= nf;
		turb->dev[i] /= nf;
		turb->pv[i] /= nf;
		for (j = i; j < MAXD; j++)
		{
		    turb->vv[sym_index(i,j)] /= nf;
		    turb->dvv[sym_index(i,j)] /= nf;
		}
	    }
	}
}		/*end normalize_state_totals*/

/*
*			accumulate_state_in_layer():
*
*   Sum the gas states in the planar or spherical layer at the given height.
*
*   origin == NULL means that the layer is a plane of given height; otherwise
*   the layer is a spherical shell whose radius (height) is measured with
*   respect to origin[].
*
*   ignore_params == YES means to combine the sums for the different materials
*   into a single global sum (otherwise sums are kept for the individual
*   materials).
*
*   normalize == YES means convert the raw totals to averages prior to exiting.
*
*   communicate == YES means that, in the case of parallel computation, to
*   combine the totals from the different nodes.
*/

EXPORT	void 	accumulate_state_in_layer(
	Wave			*wave,
	Front			*front,
	float			height,
	Big_State		*bst,
	Turbulence_moments	*turb,
	float			rfactor,
	const float		*origin,
	bool			ignore_params,
	bool			normalize,
	bool			communicate)
{
	INTERFACE		*intfc = front->interf;
	const RECT_GRID		*rgrid = front->rect_grid;
	const int		n_params = num_gas_params(intfc);
	const int		nn = pp_numnodes();
	const int		myid = pp_mynode();
	const int		dim = rgrid->dim;
	const int		zdir = dim - 1;
	const size_t		size_bst = sizeof(Big_State);
	const size_t		size_turb = sizeof(Turbulence_moments);
	const enum intfc_geometry  geom = (origin == NULL ? planar : spherical);
	const float		dh = (dim == 2 ?
				      min(rgrid->h[0],rgrid->h[1])/rfactor
				      : min(min(rgrid->h[0],rgrid->h[1]),
					    rgrid->h[2])/rfactor);
	int			Nx, Ny, Nphi;
	float			dx, dy, dphi;
	float			x_left, y_left, phi;
	register int 		i, j;
	int			k, n, p, n_points;
	float			coords[3], vtmp[3];
	float			rhat[MAXD],  phhat[MAXD];
	COMPONENT		comp;

	static Locstate		st = NULL;
	static Big_State	*commbst = NULL, *tmpbst = NULL;
	static Turbulence_moments *commturb = NULL, *tmpturb = NULL;

	debug_print("glayer","Entered accumulate_state_in_layer(), h = %g\n",height);

	if (st == NULL)
	{
	    alloc_state(intfc,&st,front->sizest);
	    vector(&commbst,n_params,size_bst);
	    vector(&tmpbst,n_params,size_bst);
	    vector(&commturb,n_params,size_turb);
	    vector(&tmpturb,n_params,size_turb);
	}

	if ((normalize == YES) && (communicate == NO))
	{
	    (void) printf("WARNING in accumulate_state_in_layer(), "
	                  "Normalization with no communication will give\n"
	                  "\tincorrect results in parallel computation.\n");
	}

	zero_state_totals(tmpbst,tmpturb,n_params);
	n_points = 0;

	switch(geom)
	{
	case planar:
	    coords[zdir] = height;
	    if (height < rgrid->L[zdir] || height >= rgrid->U[zdir])
	        break;
	    Nx = irint(rgrid->gmax[0]*rfactor);
	    dx = (rgrid->U[0] - rgrid->L[0])/Nx;
	    x_left = rgrid->L[0] + 0.5*dx;
	    switch(dim)
	    {
	    case 1:
	        screen("ERROR in accumulate_state_in_layer(), "
		       "1D not supported\n");
	        clean_up(ERROR);
	        break;
#if defined(TWOD)
	    case 2:
	        for (i = 0; i < Nx; i++)
		{
		    n_points++;
		    coords[0] = x_left + i*dx;
		    comp = component(coords,intfc);
		    hyp_solution(coords,comp,NULL,UNKNOWN_SIDE,
				 front,wave,st,NULL);
		    add_state_to_totals(st,
				      &tmpbst[index_of_Gas_param(Params(st))],
				      &tmpturb[index_of_Gas_param(Params(st))]);
		}
		break;
#endif /* defined(TWOD) */
#if defined(THREED)
	    case 3:
		Ny = irint(rgrid->gmax[1]*rfactor);
		dy = (rgrid->U[1] - rgrid->L[1])/Ny;
		y_left = rgrid->L[1] + 0.5*dy;
		for (j = 0; j < Ny; j++)
		{
		    coords[1] = y_left + j*dy;
		    for (i = 0; i < Nx; i++)
		    {
		        n_points++;
			coords[0] = x_left + i*dx;
			comp = component(coords,intfc);
			hyp_solution(coords,comp,NULL,UNKNOWN_SIDE,
				     front,wave,st,NULL);
			add_state_to_totals(st,
				      &tmpbst[index_of_Gas_param(Params(st))],
				      &tmpturb[index_of_Gas_param(Params(st))]);
		    }
		}
		break;
#endif /* defined(THREED) */
	    }
	    break;

	case spherical:
	    switch(dim)
	    {
	    case 1:
	        screen("ERROR in accumulate_state_in_layer(), "
		       "1D not supported\n");
	        clean_up(ERROR);
	        break;
#if defined(TWOD)
	    case 2:
	        Nphi = irint(2.0*PI*height/dh);
		dphi = 2.0*PI/Nphi;
		for (i = 0; i < Nphi; i++)
		{
		    phi = i*dphi;
		    coords[0] = height*cos(phi) + origin[0];
		    coords[1] = height*sin(phi) + origin[1];
		    if (is_on_local_grid(rgrid,coords) != YES)
		      continue;
		    n_points++;
		    comp = component(coords,intfc);
		    hyp_solution(coords,comp,NULL,UNKNOWN_SIDE,
				 front,wave,st,NULL);

		    /* Get components of velocity vector along
		       polar unit directions */

		    set_state(st,TGAS_STATE,st);
		    spherical_unit_vectors(rhat,NULL,phhat,coords,origin,dim);
		    vtmp[0] = scalar_product(Vel(st),rhat,dim);
		    vtmp[1] = scalar_product(Vel(st),phhat,dim);
		    for (k = 0; k < dim; k++)
		        Vel(st)[k] = vtmp[k];

		    add_state_to_totals(st,
				    &tmpbst[index_of_Gas_param(Params(st))],
				    &tmpturb[index_of_Gas_param(Params(st))]);
		}
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	    case 3:
		{
		    float th, dth, rpolar, thhat[3];
	            int j, Nth;
		    Nth = irint(PI*height/dh);
		    dth = PI/Nth;
		    for (j = 0; j < Nth; j++)
		    {
		        th = j*dth;
		        rpolar = height*sin(th);
		        Nphi = irint(2.0*PI*rpolar/dh);
		        dphi = 2.0*PI/Nphi;
		        for (i = 0; i < Nphi; i++)
		        {
			    phi = i*dphi;
			    coords[0] = rpolar*cos(phi) + origin[0];
			    coords[1] = rpolar*sin(phi) + origin[1];
			    coords[2] = height*cos(th) + origin[2];
			    if (is_on_local_grid(rgrid,coords) != YES)
			        continue;
			    n_points++;
			    comp = component(coords,intfc);
			    hyp_solution(coords,comp,NULL,UNKNOWN_SIDE,
				         front,wave,st,NULL);

			    /* Get components of velocity vector along
			       spherical unit directions */

			    set_state(st,TGAS_STATE,st);
			    spherical_unit_vectors(rhat,thhat,phhat,
					           coords,origin,dim);
			    vtmp[0] = scalar_product(Vel(st),rhat,dim);
			    vtmp[1] = scalar_product(Vel(st),thhat,dim);
			    vtmp[2] = scalar_product(Vel(st),phhat,dim);
			    for (k = 0; k < dim; k++)
			        Vel(st)[k] = vtmp[k];

			    add_state_to_totals(st,
				    &tmpbst[index_of_Gas_param(Params(st))],
				    &tmpturb[index_of_Gas_param(Params(st))]);
		        }
		    }
		}
		break;
#endif /* defined(THREED) */
	    }
	    break;
	}

	if (debugging("glayer"))
	{
	    (void) printf("\tI found %d points in this layer.\n"
	                  "\th = %g; geom = %s\n",n_points,height,
		          (geom == spherical ? "spherical" : "planar"));
	}

	if ((communicate == YES) && (nn > 1))
	{
	    if (myid == IO_NODE_ID)
	    {
	        for (n = 0; n < nn; n++)
		{
		    if (n != myid)
		    {
		        pp_recv(LAYER_SUM_ID,n,(POINTER)commbst,
				n_params*size_bst);
			pp_recv(LAYER_SUM_ID+1,n,(POINTER)commturb,
				n_params*size_turb);
			for (p = 0; p < n_params; p++)
			    accumulate_state_totals(&commbst[p],&commturb[p],
						    &tmpbst[p],&tmpturb[p]);
		    }
		}
		pp_send_all(LAYER_SUM_ID+2,(POINTER)tmpbst,n_params*size_bst);
		pp_send_all(LAYER_SUM_ID+3,(POINTER)tmpturb,n_params*size_turb);
	    }
	    else
	    {
	        pp_send(LAYER_SUM_ID,(POINTER)tmpbst,
			n_params*size_bst,IO_NODE_ID);
		pp_send(LAYER_SUM_ID+1,(POINTER)tmpturb,
			n_params*size_turb,IO_NODE_ID);
		pp_recv(LAYER_SUM_ID+2,IO_NODE_ID,(POINTER)tmpbst,
			n_params*size_bst);
		pp_recv(LAYER_SUM_ID+3,IO_NODE_ID,(POINTER)tmpturb,
			n_params*size_turb);
	    }

	    pp_global_isum(&n_points,1L);
	}

	if (ignore_params == YES)
	{
	    zero_state_totals(bst,turb,1);
	    for (p = 0; p < n_params; p++)
	        accumulate_state_totals(&tmpbst[p],&tmpturb[p],bst,turb);
	    if (normalize == YES)
	        normalize_state_totals(bst,turb,n_points);
	}
	else
	{
	    zero_state_totals(bst,turb,n_params);
	    if (turb == NULL)
	    {
		for (p = 0; p < n_params; p++)
		{
		    accumulate_state_totals(&tmpbst[p],NULL,&bst[p],NULL);
		    if (normalize == YES)
		        normalize_state_totals(&bst[p],NULL,n_points);
		}
	    }
	    else
	    {
		for (p = 0; p < n_params; p++)
		{
		    accumulate_state_totals(&tmpbst[p],&tmpturb[p],
					    &bst[p],&turb[p]);
		    if (normalize == YES)
		        normalize_state_totals(&bst[p],&turb[p],n_points);
		}
	    }
	}
	debug_print("glayer","Left accumulate_state_in_layer()\n");
}			/*end accumulate_state_in_layer*/
#endif  /* defined(TWOD) || defined(THREED) */
