/*
*
*				gwspeed.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	g_w_speed() and g_npt_w_speed() calculate the wave speeds and 
*	states for a tracked wave.  The first-order computation is done
*	in g_w_speed().  The function g_npt_w_speed() calculates the
*	second-order corrections.
*
*	Source-term corrections to front states due to gravitation or 
*	cylindrical symmetry are included in g_npt_w_speed() through
*	the method of characteristics or throught the function include_source().
*	Also the characteristic equations in
*	g_npt_w_speed() are modified due to cylindrical source terms,
*	which become important near the axis of symmetry.
*/

#include <gdecs/gdecs.h>

	/* LOCAL Function Prototypes */
LOCAL	bool    invalid_shock(float,float,float,float);
LOCAL	bool	strong_interaction(Locstate,Locstate,int,
	                           float,float,float,float);
LOCAL	bool	strong_wave(Locstate,float,float);
LOCAL	bool	fC(float,float*,POINTER);
LOCAL   bool    pressure_switch_on(Locstate,Locstate,Locstate);
LOCAL	float	wall_limiter(WSSten*,SIDE,NptWSpeedOpts*);
LOCAL	void	apply_bc(Locstate,Wave*,HYPER_SURF*);
LOCAL	void	IncomingStatesAtContact(WSSten*,float,float,
	                                Locstate,Locstate,float*,float*,
	                                NptWSpeedOpts*);
LOCAL   void    bdry_npt_w_speed(WSSten*,Locstate,Locstate,NptWSpeedOpts*);
LOCAL	void	bdry_wspeed(int,float*,Locstate,Locstate,Locstate,Locstate,
	                    int,Front*);
LOCAL	void	contact_cheap_moc(float*,Locstate,Locstate,Locstate,Locstate,
				  Locstate,Locstate,float,float,float,float,
				  float,float*,float*,Front*);
LOCAL	void	contact_filter_outgoing_wave(Locstate,Locstate,Locstate,float,
	                                     WSSten*,SIDE,NptWSpeedOpts*);
LOCAL	void	contact_moc_plus_rh(float*,float,Locstate,Locstate,Locstate,
			            Locstate,Locstate,Locstate,float,float,
				    float,float*,float*,Front*);
LOCAL	void	contact_riemann(Locstate,Locstate,Locstate,Locstate,
	                        Locstate,float,Locstate,float,Locstate,Locstate,
	                        WSSten*,float*,NptWSpeedOpts*,float*);
LOCAL	void	g_Left_w_speed(const char*,Locstate,Locstate,float*,int);
LOCAL	void	neumann_cheap_moc(float*,Locstate,float,float,Locstate,
	                          SIDE,Locstate,float,float*,Front*);
LOCAL	void	onedw_speed(Locstate,Locstate,int,Locstate,Locstate,Locstate,
	                    Locstate,float*,int,float,Front*);
LOCAL	void	shock_ahead_state_cheap_moc(float*,Locstate,Locstate,Locstate,
	                                    Locstate,Locstate,float,float,float,
					    float,float*,float*,int,float,
					    Front*);
LOCAL	void	wspeed_neumann_riem_inv_moc(float*,Locstate,float,float,
	                                    Locstate,SIDE,Locstate,float,float*,
	                                    Front*);
LOCAL	void	wspeed_shock_ahead_state_riem_inv_moc(float*,Locstate,Locstate,
	                                              Locstate,Locstate,
						      Locstate,float,float,
						      float,float,float*,
						      float*,int,float,Front*);

#if defined(COMBUSTION_CODE)
LOCAL	void	CJ_state_behind_curved_wave(Locstate,Locstate,float,float*,
	                                    int,int);
#endif /* defined(COMBUSTION_CODE) */

#if !defined(__INTEL_COMPILER)
#pragma	noinline	strong_interaction
#pragma	noinline	fC
#pragma noinline        bdry_npt_w_speed
#pragma	noinline	bdry_wspeed
#pragma	noinline	contact_cheap_moc
#pragma	noinline	contact_moc_plus_rh
#pragma	noinline	midstate
#pragma	noinline	neumann_cheap_moc
#pragma	noinline	contact_riemann
#pragma	noinline	shock_ahead_state_cheap_moc
#if defined(COMBUSTION_CODE)
#pragma	noinline	CJ_state_behind_curved_wave
#endif /* defined(COMBUSTION_CODE) */
#endif /*!defined(__INTEL_COMPILER)*/

LOCAL	NptWSpeedOpts	G3ptOpts = {
	-1.0,		/*Mach_tol*/
	-1.0,		/*A_tol*/
	-1.0,		/*Wall_limiter*/
	MOC_TYPE_UNSET,	/*vector_moc*/
	MOC_TYPE_UNSET,	/*scalar_moc*/
	NO,		/*_scalar_filter_outgoing_waves_at_contact*/
	NO,		/*use_cheap_ahead_state_moc*/
	NO,		/*use_neumann_cheap_moc*/
	NULL,
	NULL,
	IDENTITY_REMAP,
	MODIFIED_EULER
};

EXPORT	void	set_npt_wspeed_options(
	NptWSpeedOpts	*opts)
{
	if (opts->vector_ahead_state_moc == NULL)
	{
	    opts->vector_ahead_state_moc =
		(opts->use_cheap_ahead_state_moc == YES) ?
		shock_ahead_state_cheap_moc :
		wspeed_shock_ahead_state_riem_inv_moc;
	}
	if (opts->neumann_moc == NULL)
	{
	    opts->neumann_moc = (opts->use_neumann_cheap_moc == YES) ?
		                neumann_cheap_moc :
		                wspeed_neumann_riem_inv_moc;
	}
	G3ptOpts = *opts;
}		/*end set_npt_wspeed_options*/

#if !defined(UNRESTRICTED_THERMODYNAMICS)
#define pressure_is_near_vacuum(p,st) ((p) < 2.0*Min_pressure(st)) /*TOLERANCE*/
#else /* !defined(UNRESTRICTED_THERMODYNAMICS) */
#define pressure_is_near_vacuum(p,st)	(NO)
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */



#if defined(DEBUG_W_SPEED)

LOCAL	bool debug_w_speed = NO;

#define	Left_w_speed(f,ansl,ansr,w,dim) g_Left_w_speed(f,ansl,ansr,w,dim)

EXPORT void g_set_debug_w_speed(bool y_or_n)
{
	debug_w_speed = y_or_n;
}	/*end g_set_debug_w_speed*/

LOCAL	void	g_Left_w_speed(
	const char	*func,
	Locstate	ansl,
	Locstate	ansr,
	float		*w,
	int		dim)
{
	float		len;
	int		i;

	if (debug_w_speed == YES) 
	{
	    (void) printf("Final left, right states:\n");
	    verbose_print_state("ansl",ansl);
	    verbose_print_state("ansr",ansr);
	}
	if (debugging("wave_speeds")) 
	{
	    if (w != NULL)
	    {
	        (void) printf("\nFinal wave velocity in %s() = ",func);
	        print_general_vector("",w,dim,"");
	        for (len = 0.0, i = 0; i < dim; ++i)
	            len += sqr(w[i]);
	        len = sqrt(len);
	        (void) printf("speed = %g\n",len);
	    }
	}
	if ((debug_w_speed == YES) && (!debugging("w_speed")))
	    (void) printf("Left %s()\n",func);
	debug_print("w_speed","Left %s()\n",func);
}		/*end g_Left_w_speed*/


#else /* defined(DEBUG_W_SPEED) */

	/*   Make Left_w_speed a macro call so that  */
	/* there is no function call when not debugging */

#define	Left_w_speed(func,ansl,ansr,w,dim)

#endif /* defined(DEBUG_W_SPEED) */

EXPORT	void	print_g_npt_w_speed_opts(
	NptWSpeedOpts	*opts)
{
	screen("Current values for options for g_npt_w_speed\n");
	screen("\tA wave is defined to be strong if "
	       "|1 - (1/(rho*c)*|dp/du|| > Mach_tol or\n"
	       "\t\t|rhol - rhor|/(rhol+rhor) > A_tol\n");
	screen("\tMach_tol = %g\n",opts->Mach_tol);
	screen("\tA_tol = %g\n",opts->A_tol);
	screen("\tNeumann boundary states are computed by an average "
	       "of a reflection\n"
	       "\t\tsymmetry contact propagation and a method of characterics\n"
	       "\t\tcalculation.  The weight of the symmetry contact result\n"
	       "\t\tis proportional to the flow gradient.  The wall limiter\n"
	       "\t\tvalue gives this proportionality constant.\n");
	screen("\tWall_limiter = %g\n",opts->Wall_limiter);
	screen("\tvector_moc = ");
	switch (opts->vector_moc)
	{
	case MOC_PLUS_RIEMANN:
	    screen("MOC_PLUS_RIEMANN\n");
	    break;
	case MOC_PLUS_RH:
	    screen("MOC_PLUS_RH\n");
	    break;
	case RIEMANN:
	    screen("RIEMANN\n");
	    break;
	default:
	    screen("ERROR in print_g_npt_w_speed_opts(), "
	           "Invalid value %d for vector_moc\n",opts->vector_moc);
	    clean_up(ERROR);
	}
	screen("\tscalar_moc = ");
	switch (opts->scalar_moc)
	{
	case MOC_PLUS_RIEMANN:
	    screen("MOC_PLUS_RIEMANN\n");
	    break;
	case MOC_PLUS_RH:
	    screen("MOC_PLUS_RH\n");
	    break;
	case RIEMANN:
	    screen("RIEMANN "
	           "%s filtering of outgoing waves\n",
	           (opts->_scalar_filter_outgoing_waves_at_contact == YES) ?
	           "with" : "without");
	    break;
	default:
	    screen("ERROR in print_g_npt_w_speed_opts(), "
	           "Invalid value %d for scalar_moc\n",opts->scalar_moc);
	    clean_up(ERROR);
	}
	if (opts->vector_ahead_state_moc == NULL)
	{
	    opts->vector_ahead_state_moc =
		(opts->use_cheap_ahead_state_moc == YES) ?
		shock_ahead_state_cheap_moc :
		wspeed_shock_ahead_state_riem_inv_moc;
	}
	screen("\tvector_ahead_state_moc = %p ",opts->vector_ahead_state_moc);
	if (opts->vector_ahead_state_moc ==
	                                wspeed_shock_ahead_state_riem_inv_moc)
	    screen("shock_ahead_state_riem_inv_moc\n");
	else if (opts->vector_ahead_state_moc ==
	                                shock_ahead_state_cheap_moc)
	    screen("shock_ahead_state_cheap_moc\n");
	else
	    screen("user defined\n");

	if (opts->neumann_moc == NULL)
	{
	    opts->neumann_moc = (opts->use_neumann_cheap_moc == YES) ?
		                neumann_cheap_moc :
		                wspeed_neumann_riem_inv_moc;
	}
	screen("\tneumann_moc = %p ",opts->neumann_moc);
	if (opts->neumann_moc == wspeed_neumann_riem_inv_moc)
	    screen("neumann_riem_inv_moc\n");
	else if (opts->neumann_moc == neumann_cheap_moc)
	    screen("neumann_cheap_moc\n");
	else
	    screen("user defined\n");
#if defined(ROTATIONAL_SYMMETRY)
	if (opts->remap != IDENTITY_REMAP)
	{
	    screen("\tgeom_source_method = ");
	    switch (opts->geom_source_method)
	    {
	    case MODIFIED_EULER:
	        screen("MODIFIED_EULER");
	        break;
	    case ANALYTIC_SOLUTION:
	        screen("ANALYTIC_SOLUTION");
	        break;
	    case BACKWARD_EULER:
	        screen("BACKWARD_EULER");
	        break;
	    }
	    screen("\n");
	}
#endif /* defined(ROTATIONAL_SYMMETRY) */
	screen("End current values for options for g_npt_w_speed\n");
}		/*end print_g_npt_w_speed_opts*/



/*
*			g_w_speed():
*
*	Computes wave speeds and states for a tracked wave to first order.
*
*	Input:	sl,sr   - left and right states close to oriented front
*		nor     - unit normal to front (left->right)
*		w_type  - type of tracked wave
*		dt	- time step, used for the inclusion of source terms.
*		pjump   - For surface tension,  the pressure on the negative
*                         side of a contact is equal to the pressure on the
*                         positive side plus pjump.
*
*	Output:	ansl,ansr - updated left and right states.
*		W         - calculated wave velocity
*
*/

/*ARGSUSED*/
EXPORT	void g_w_speed(
	float		*pt,
	Locstate	sl,
	Locstate	sr,
	Locstate	ansl,
	Locstate	ansr,
	float		*W,
	float		pjump,
	float		*nor,
	int		w_type,
	Front		*fr)
{
	int		dim = fr->rect_grid->dim;
	static size_t	sizest = 0;
	static Locstate Tsl = NULL, Tsr = NULL, left = NULL, right = NULL;

#if defined(COMBUSTION_CODE)
	static Locstate CJ = NULL;
#endif /* defined(COMBUSTION_CODE) */

	float s;		/* normal wave speed */
	float vtanl[MAXD], vtanr[MAXD];	/* tangential velocity */
	float vnorl, vnorr;     /* normal velocity              */
	int i;

	if (is_obstacle_state(sl) && is_obstacle_state(sr)) 
	{
	    obstacle_state(fr->interf,ansl,fr->sizest);
	    obstacle_state(fr->interf,ansr,fr->sizest);
	    for (i = 0; i < dim; ++i)
	        W[i] = 0.0;
	    return;
	}
	if (right == NULL)
	{
	    Gas_param *params=(!is_obstacle_state(sl)) ? Params(sl):Params(sr);
	    sizest = params->sizest;
	    (*params->_alloc_state)(&Tsl,sizest);
	    (*params->_alloc_state)(&Tsr,sizest);
	    (*params->_alloc_state)(&left,max(sizeof(VGas),sizest));
	    (*params->_alloc_state)(&right,max(sizeof(VGas),sizest));
#if defined(COMBUSTION_CODE)
	    (*params->_alloc_state)(&CJ,sizest);
#endif /* defined(COMBUSTION_CODE) */
	}

#if defined(DEBUG_W_SPEED)
	debug_print("w_speed","Entered g_w_speed()\n");
	if (debugging("w_speed"))
	{
	    debug_w_speed = YES;
	}
	else if (debug_w_speed == YES)
	    (void) printf("Entered g_w_speed()\n");
	if (debug_w_speed == YES) 
	{
	    (void) printf("INPUT DATA TO W_SPEED()\n");
	    fprint_wave_type(stdout,"wave type = ",w_type,"\n",fr->interf);
	    print_general_vector("nor = ",nor,dim,"\n");
	    verbose_print_state("sl",sl);
	    verbose_print_state("sr",sr);
	    (void) printf("END INPUT DATA TO W_SPEED()\n");
	}
#endif /* defined(DEBUG_W_SPEED) */

	if (w_type < FIRST_PHYSICS_WAVE_TYPE)
	{

	    for (i = 0; i < dim; ++i)
	        W[i] = 0.0;
	    bdry_wspeed(w_type,nor,sl,sr,ansl,ansr,GAS_STATE,fr);
	    Left_w_speed("g_w_speed",ansl,ansr,W,dim);
	    return;
	}


	set_state(Tsl,TGAS_STATE,sl);
	set_state(Tsr,TGAS_STATE,sr);
	vnorl = vnorr = 0.0;
	for (i = 0; i < dim; ++i)
	{
	    vnorl += nor[i] * Vel(Tsl)[i];
	    vnorr += nor[i] * Vel(Tsr)[i];
	}
	for (i = 0; i < dim; ++i)
	{
	    vtanl[i] = Vel(Tsl)[i] - nor[i] * vnorl;
	    vtanr[i] = Vel(Tsr)[i] - nor[i] * vnorr;
	}
	Vel(Tsl)[0] = vnorl;
	Vel(Tsr)[0] = vnorr;
	for (i = 1; i < dim; ++i)
	    Vel(Tsl)[i] =  Vel(Tsr)[i] = 0.0;

	onedw_speed(Tsl,Tsr,w_type,left,right,ansl,ansr,&s,TGAS_STATE,pjump,fr);
	
	for (i = dim - 1; i >= 0; --i)
	{
	    W[i] = nor[i] * s;
	    Vel(ansl)[i] =  vtanl[i] + nor[i] * Vel(ansl)[0];
	    Vel(ansr)[i] =  vtanr[i] + nor[i] * Vel(ansr)[0];
	}

	set_state(ansl,GAS_STATE,ansl);
	if (ansr != ansl)
	    set_state(ansr,GAS_STATE,ansr);

#if defined(DEBUG_W_SPEED)
	Left_w_speed("g_w_speed",ansl,ansr,W,dim);
#endif /* defined(DEBUG_W_SPEED) */
}	/*end g_w_speed*/



LOCAL	void bdry_wspeed(
	int		w_type,
	float		*nor,
	Locstate	sl,
	Locstate	sr,
	Locstate	ansl,
	Locstate	ansr,
	int		state_type,
	Front		*fr)
{
	size_t		sizest = fr->sizest;
	float		mnor;
	int		i, dim;

#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES)
	    fprint_wave_type(stdout,"wave_type = ",w_type,"\n",fr->interf);
#endif /* defined(DEBUG_W_SPEED) */
	set_type_of_state(ansl,GAS_STATE);
	set_type_of_state(ansr,GAS_STATE);
	switch (w_type)
	{
	case PASSIVE_BOUNDARY:
	case SUBDOMAIN_BOUNDARY:
	    /* PASSIVE and SUBDOMAIN states not set or used */
	    obstacle_state(fr->interf,ansl,sizest);
	    obstacle_state(fr->interf,ansr,sizest);
	    break;

	case DIRICHLET_BOUNDARY:
	case TIME_DIRICHLET_BOUNDARY:
	    copy_state(ansl,sl);
	    copy_state(ansr,sr);
	    break;

	case NEUMANN_BOUNDARY:
	    if (is_obstacle_state(sl))
	        obstacle_state(fr->interf,ansl,sizest);
	    else
	    {
	        Dens(ansl) = Dens(sl);
	        Energy(ansl) = energy(sl);
	        mnor = 0.0;
	        dim = Params(sl)->dim;
	        for (i = 0; i < dim; ++i)
	            mnor +=  nor[i] * mom(i,sl);
	        for (i = 0; i < dim; ++i)
	            Mom(ansl)[i] = mom(i,sl) - nor[i] * mnor;
	        /* QUESTION?  Should the Energy be adjusted to
	        *  offset the change in kinetic energy? */
	        /* ANSWER: Yes, if any thermodynamic call is
	        *  made, even through a subroutine, before
	        *  tangential momentum is readded to ansl.
	        *  This change has to be done everywhere, in a
	        *  consistent fashion, when it is done */

#if defined(COMBUSTION_CODE)
	        if (Composition_type(sl) == ZND)
	            Prod(ansl) = Prod(sl); 
#endif /* defined(COMBUSTION_CODE) */

	        Set_params(ansl,sl);
	        if (state_type != state_type(ansl)) 
	            set_state(ansl,state_type,ansl);
	    }

	    if (is_obstacle_state(sr))
	        obstacle_state(fr->interf,ansr,sizest);
	    else
	    {
	        Dens(ansr) = Dens(sr);
	        Energy(ansr) = energy(sr);
	        mnor = 0.0;
	        dim = Params(sr)->dim;
	        for (i = 0; i < dim; ++i)
	            mnor +=  nor[i] * mom(i,sr);
	        for (i = 0; i < dim; ++i)
	            Mom(ansr)[i] = mom(i,sr) - nor[i] * mnor;
	        /* QUESTION?  Should the Energy be adjusted to
	        *  offset the change in kinetic energy? */

#if defined(COMBUSTION_CODE)
	        if (Composition_type(sr) == ZND)
	            Prod(ansr) = Prod(sr); 
#endif /* defined(COMBUSTION_CODE) */

	        Set_params(ansr,sr);
	        if (state_type != state_type(ansr)) 
	            set_state(ansr,state_type,ansr);
	    }
	    break;
	
	default:
	    screen("ERROR in  bdry_wspeed(), ");
	    fprint_wave_type(stdout,"unknown wave_type = ",w_type,"\n",
	                     fr->interf);
	    clean_up(ERROR);
	}
}	/*end bdry_wspeed*/


/*
*			onedw_speed():
*
*	One dimensional wave speed operator.
*
*	Input	sl, sr		- States at time t near curve
*		w_type		- type of tracked wave
*		pwall		- Wall information (Unused if nonwall)
*
*	Output	left/right	- States corrected for wall interactions
*		Wx		- pointer to wave velocity
*		ansl, ansr	- States at time t + dt near curve
*		state_type	- type of state representation
*		pjump		- Pressure jump across contact
*
*/

LOCAL	void onedw_speed(
	Locstate	sl,
	Locstate	sr,
	int		w_type,
	Locstate	left,
	Locstate	right,
	Locstate	ansl,
	Locstate	ansr,
	float		*Wx,
	int		state_type,
	float		pjump,
	Front		*fr)
{
	RIEMANN_SOLVER_WAVE_TYPE l_wave,r_wave;
#if defined(ONED)
	int		dim;
#endif /* defined(ONED) */
	float		pml, pmr, uml, umr ,mr,ml; /* midstate quantities */
#if defined(COMBUSTION_CODE)
	static		Locstate CJ = NULL;
#endif /* defined(COMBUSTION_CODE) */
	static	size_t sizest = 0;
	
	if (is_obstacle_state(sl) && is_obstacle_state(sr)) 
	{
	    obstacle_state(fr->interf,ansl,fr->sizest);
	    obstacle_state(fr->interf,left,fr->sizest);
	    obstacle_state(fr->interf,ansr,fr->sizest);
	    obstacle_state(fr->interf,right,fr->sizest);
	    *Wx = 0.0;
	    return;
	}
	if (sizest == 0)
	{
	    Gas_param *params = (!is_obstacle_state(sl))?Params(sl):Params(sr);

	    sizest = params->sizest;
#if defined(COMBUSTION_CODE)
	    (*params->_alloc_state)(&CJ,sizest);
#endif /* defined(COMBUSTION_CODE) */

	}

#if defined(DEBUG_W_SPEED)
	debug_print("w_speed","Entered onedw_speed()\n");
	if (debugging("w_speed"))
	{
	    debug_w_speed = YES;
	}
	else if (debug_w_speed == YES)
	    (void) printf("Entered onedw_speed()\n");

	if (debug_w_speed == YES)
	{
	    fprint_wave_type(stdout,"wave_type = ",w_type,"\n",fr->interf);
	    verbose_print_state("sl",sl);
	    verbose_print_state("sr",sr);
	}
#endif /* defined(DEBUG_W_SPEED) */

	
#if defined(ONED)
	dim = (is_obstacle_state(sl)) ? Params(sr)->dim : Params(sl)->dim;
	if ((dim == 1) && ((w_type < FIRST_PHYSICS_WAVE_TYPE) || 
	                    (w_type == TIME_DIRICHLET_BOUNDARY)) )
	{
	    static float nor[3] = {1.0, 0.0, 0.0};
	    *Wx = 0.0;

	    bdry_wspeed(w_type,nor,sl,sr,ansl,ansr,state_type,fr);
	    copy_state(left,sl);
	    copy_state(right,sr);
	    Left_w_speed("onedw_speed",ansl,ansr,Wx,dim);
	    return;
	}
#endif /* defined(ONED) */


	set_type_of_state(ansl,GAS_STATE);
	set_type_of_state(ansr,GAS_STATE);
	set_state_for_find_mid_state(left,sl);
	set_state_for_find_mid_state(right,sr);


	if (find_mid_state(left,right,pjump,&pml,&pmr,&uml,&umr,&ml,&mr,
	                       &l_wave,&r_wave) != FUNCTION_SUCCEEDED)
	{
	    (void) printf("WARNING in onedw_speed(), "
	                  "find_mid_state() did not converge\n");
	    verbose_print_state("left",left);
	    verbose_print_state("right",right);
	    (void) printf("pjump = %g\n",pjump);
	    (void) printf("pml = %g, pmr = %g\n",pml,pmr);
	    (void) printf("uml = %g, umr = %g\n",uml,umr);
	    (void) printf("ml = %g, mr = %g\n",ml,mr);
	    print_rsoln_wave("l_wave = ",l_wave,", ");
	    print_rsoln_wave("r_wave = ",r_wave,"\n");
	}

#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES)
	{
	    (void) printf("l_wave = %s, r_wave = %s\n",rsoln_wave_name(l_wave),
	                  rsoln_wave_name(r_wave));
	    (void) printf("midstate pressures = %g %g\n",pml,pmr);
	    (void) printf("midstate velocity = %g %g\n",uml,umr);
	}
#endif /* defined(DEBUG_W_SPEED) */

	if (is_backward_wave(w_type))
	{
	    state_behind_sound_wave(left,ansr,NULL,Wx,pjump,ml,uml,
	                            pml,state_type,w_type,l_wave,LEFT_FAMILY);
#if defined(COMBUSTION_CODE)
	    if (r_wave == SHOCK || r_wave == RAREFACTION)
	        Set_params(ansr,left);
#endif /* defined(COMBUSTION_CODE) */
#if DONT_COMPILE /*OLD VERSION*/
	    if ((l_wave==RAREFACTION) && is_rarefaction_trailing_edge(w_type))
#endif /*DONT_COMPILE*/ /*OLD VERSION*/
	    if (is_rarefaction_trailing_edge(w_type))
	        copy_state(ansl,ansr);
	    else
	        set_state(ansl,state_type,sl);
	}

	else if (is_scalar_wave(w_type)) 
	{
	    midstate(left,ansl,ml,uml,pml,state_type,l_wave,LEFT_FAMILY);
	    midstate(right,ansr,mr,umr,pmr,state_type,r_wave,RIGHT_FAMILY);
	    *Wx = 0.5*(uml+umr);
	}

	else if (is_forward_wave(w_type)) 
	{
	    state_behind_sound_wave(right,ansl,NULL,Wx,pjump,mr,umr,
	                            pmr,state_type,w_type,r_wave,RIGHT_FAMILY);
#if defined(COMBUSTION_CODE)
	    if (r_wave == SHOCK || r_wave == RAREFACTION)
	        Set_params(ansl,right);
#endif /* defined(COMBUSTION_CODE) */

#if DONT_COMPILE /*OLD VERSION*/
	    if ((r_wave==RAREFACTION) && is_rarefaction_trailing_edge(w_type))
#endif /*DONT_COMPILE*/ /*OLD VERSION*/
	    if (is_rarefaction_trailing_edge(w_type))
	        copy_state(ansr,ansl);
	    else
	        set_state(ansr,state_type,sr);
	}
	else
	{
	    screen("ERROR: unknown wave_type %d in w_speed\n",w_type);
	    clean_up(ERROR);
	}

#if defined(ONED)
	if ((dim == 1) && (state_type != TGAS_STATE))
	{
	    set_state(left,state_type,left);
	    set_state(right,state_type,right);
	}
#endif /* defined(ONED) */

	Left_w_speed("onedw_speed",ansl,ansr,Wx,1);
	return;
}	/*end onedw_speed*/

/*ARGSUSED*/
EXPORT	void state_behind_sound_wave(
	Locstate	         ahead,
	Locstate	         ans,
	float		         *cans,
	float		         *Wx,
	float		         pjump,
	float		         m,
	float		         um,
	float		         pm,
	int		         st_type_ans,
	int		         w_type,
	RIEMANN_SOLVER_WAVE_TYPE wave,
	int		         family)
{
	float	s, c;
	float	sgn = (family == LEFT_FAMILY) ? -1.0 : 1.0;
	float   pa;
	int	i, dim = Params(ahead)->dim;

#if defined(DEBUG_W_SPEED)
	debug_print("w_speed","Entered state_behind_sound_wave()\n");
	if (debug_w_speed == YES)
	{
	    if (!debugging("w_speed"))
	        (void) printf("Entered state_behind_sound_wave()\n");
	    switch (wave) 
	    {
	    case SHOCK:
	        (void) printf("shock\n");
	        break;

	    case RAREFACTION:
	        (void) printf("rarefaction\n");
	        break;

#if defined(COMBUSTION_CODE)
	    case STRONG_DET:
	        (void) printf("strong detonation\n");
	        break;

	    case CJ_DET:	
	        (void) printf("CJ-detonation\n");
	        (void) printf("CJ pressure = %g \n",pressure(ans));
	        (void) printf("CJ density = %g \n",Dens(ans));
	        (void) printf("CJ velocity = %g \n",um);
	        break;
#endif /* defined(COMBUSTION_CODE) */
	    
	    default:
	        screen("ERROR - unknown wave %d "
	               "in state_behind_sound_wave()\n",wave);
	        clean_up(ERROR);
	        break;
	    }
	}
#endif /* defined(DEBUG_W_SPEED) */
#if defined(COMBUSTION_CODE)
	if ((wave == CJ_DET) || (wave == STRONG_DET))
	{
	    CJ_state_behind_curved_wave(ahead,ans,pjump,Wx,st_type_ans,family);
	    return;
	}
#endif /* defined(COMBUSTION_CODE) */

	for (i = 0; i < dim; ++i)
	    Vel(ans)[i] = 0.0;
	switch (wave) 
	{
	case SHOCK:
	    s = Vel(ahead)[0] + sgn*m/Dens(ahead);
	    Dens(ans) = -sgn*m/(um - s);	
	    pa = pressure(ahead);
	    if (invalid_shock(pa,Dens(ahead),pm,Dens(ans)))
		Dens(ans) = dens_Hugoniot(pm,ahead);
	    Set_params(ans,ahead);
	    switch (st_type_ans)
	    {
	    case EGAS_STATE:
	        set_type_of_state(ans,EGAS_STATE);
	        Vel(ans)[0] = um;
	        Energy(ans) = specific_internal_energy(ahead) +
	                      0.5*(pa+pm)*(1.0/Dens(ahead) - 1.0/Dens(ans));
#if defined(COMBUSTION_CODE)
	        if (Composition_type(ans) == ZND)
	            React(ans) = React(ahead);
#endif /* defined(COMBUSTION_CODE) */
	        break;
	    case GAS_STATE:
	        set_type_of_state(ans,GAS_STATE);
	        Mom(ans)[0] = Dens(ans)*um;
	        Energy(ans) = Dens(ans)*(0.5*sqr(um) +
	                      specific_internal_energy(ahead) + 
	                      0.5*(pa+pm)*(1.0/Dens(ahead) - 1.0/Dens(ans)));
#if defined(COMBUSTION_CODE)
	        if (Composition_type(ans) == ZND)
	            Prod(ans) = Prod(ahead);
#endif /* defined(COMBUSTION_CODE) */
	        break;
	    case TGAS_STATE:
	        set_type_of_state(ans,TGAS_STATE);
	        Vel(ans)[0] = um;
	        Press(ans) = pm;
#if defined(COMBUSTION_CODE)
	        if (Composition_type(ans) == ZND)
	            React(ans) = React(ahead);
#endif /* defined(COMBUSTION_CODE) */
	        break;
	    default:
	        set_type_of_state(ans,TGAS_STATE);
	        Vel(ans)[0] = um;
	        Press(ans) = pm;
#if defined(COMBUSTION_CODE)
	        if (Composition_type(ans) == ZND)
	            React(ans) = React(ahead);
#endif /* defined(COMBUSTION_CODE) */
	        set_state(ans,st_type_ans,ans);
	        break;
	    }
	    if (cans != NULL)
	        c = sound_speed(ans);
	    break;

	case RAREFACTION:
	    if (is_rarefaction_trailing_edge(w_type))
	    {
	        switch (st_type_ans)
	        {
	        case EGAS_STATE:
	            state_on_adiabat_with_pr(ahead,pm,ans,EGAS_STATE);
	            Vel(ans)[0] = um;
	            break;
	        case GAS_STATE:
	            state_on_adiabat_with_pr(ahead,pm,ans,GAS_STATE);
	            Mom(ans)[0] = Dens(ans)*um;
	            Energy(ans) += 0.5*Mom(ans)[0]*um;
		    break;
	        case TGAS_STATE:
	            state_on_adiabat_with_pr(ahead,pm,ans,TGAS_STATE);
	            Vel(ans)[0] = um;
		    break;
		default:
	            state_on_adiabat_with_pr(ahead,pm,ans,TGAS_STATE);
	            Vel(ans)[0] = um;
	            set_state(ans,st_type_ans,ans);
		    break;
	        }
	        c = sound_speed(ans);
	        s = um + sgn*c;
	    }
	    else
	    {
	        Dens(ans) = Dens(ahead);
	        Set_params(ans,ahead);
	        switch (st_type_ans)
	        {
	        case EGAS_STATE:
	            set_type_of_state(ans,EGAS_STATE);
	            Vel(ans)[0] = vel(0,ahead);
	            Energy(ans) = specific_internal_energy(ahead);
	            break;
	        case GAS_STATE:
	            set_type_of_state(ans,GAS_STATE);
	            Mom(ans)[0] = mom(0,ahead);
	            Energy(ans) = energy(ahead);
	            break;
	        case TGAS_STATE:
	            set_type_of_state(ans,TGAS_STATE);
	            Vel(ans)[0] = vel(0,ahead);
	            Press(ans) = pressure(ahead);
	            break;
		default:
	            set_type_of_state(ans,TGAS_STATE);
	            Vel(ans)[0] = vel(0,ahead);
	            Press(ans) = pressure(ahead);
	            set_state(ans,st_type_ans,ans);
	            break;
	        }
	        c = sound_speed(ahead);
	        s = vel(0,ahead) + sgn*c;
	    }
	    break;

#if defined(COMBUSTION_CODE)
	case STRONG_DET:
	    Set_other_params(ans,ahead);
	    s = Vel(ahead)[0] + m/Dens(ahead);
	    Dens(ans) = -sgn*m/(um - s);
	    switch (st_type_ans)
	    {
	    case EGAS_STATE:
	        set_type_of_state(ans,EGAS_STATE);
	        Vel(ans)[0] = um;
	        Energy(ans) = specific_internal_energy(ahead) +
	                      0.5*(pressure(ahead)+pm)*
	                      (1.0/Dens(ahead) - 1.0/Dens(ans));
	        break;
	    case GAS_STATE:
	        set_type_of_state(ans,GAS_STATE);
	        Mom(ans)[0] = Dens(ans)*um;
	        /* Is this correct for Stong Det?*/
	        Energy(ans) = Dens(ans)*(0.5*sqr(um) +
	                      specific_internal_energy(ahead) + 
	                      0.5*(pressure(ahead)+pm)*
	                      (1.0/Dens(ahead) - 1.0/Dens(ans)));
	    case TGAS_STATE:
	        set_type_of_state(ans,TGAS_STATE);
	        Vel(ans)[0] = um;
	        Press(ans) = pm;
	        break;
	    default:
	        set_type_of_state(ans,TGAS_STATE);
	        Vel(ans)[0] = um;
	        Press(ans) = pm;
	        set_state(ans,st_type_ans,ans);
	        break;
	    }
	    if (cans != NULL)
	        c = sound_speed(ans);
	    break;

	case CJ_DET:	
	    /*Is this right? */
	    s = CJ_det(ans,st_type_ans,ahead,family);
	    Set_other_params(ans,ahead);
	    /*
	    c = sound_speed(ans);
	    s = vel(0,ans) + sgn*c;
	    */
	    break;
#endif /* defined(COMBUSTION_CODE) */
	
	default:
	    screen("ERROR - unknown wave %d "
	           "in state_behind_sound_wave()\n",wave);
	    clean_up(ERROR);
	    break;
	}

	if (cans != NULL)
	    *cans = c;
	*Wx = s;

#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES)
	{
	    verbose_print_state("Answer from state_behind_sound_wave",ans);
	    (void) printf("Wave speed = %g, sound speed = %g\n",
	                  s,sound_speed(ans));

	}
	if ((debug_w_speed == YES) && (!debugging("w_speed")))
	    (void) printf("Left state_behind_sound_wave()\n");
	debug_print("w_speed","Left state_behind_sound_wave()\n");
#endif /* defined(DEBUG_W_SPEED) */
	return;
}	/*end state_behind_sound_wave*/

EXPORT	void midstate(
	Locstate	         ahead,
	Locstate	         ans,
	float		         m,
	float		         um,
	float		         pm,
	int		         st_type_ans,
	RIEMANN_SOLVER_WAVE_TYPE wave,
	int		         family)
{
	float sgn, pa;
#if defined(COMBUSTION_CODE)
	static Locstate CJ = NULL;
#endif /* defined(COMBUSTION_CODE) */
	

	sgn = (family == LEFT_FAMILY) ? 1.0 : -1.0;
	switch (wave) 
	{
	case SHOCK:
	    pa = pressure(ahead);
	    Set_params(ans,ahead);
	    set_type_of_state(ans,st_type_ans);
	    Dens(ans) = sgn*m/(um - vel(0,ahead) + sgn*m/Dens(ahead));
	    if (invalid_shock(pa,Dens(ahead),pm,Dens(ans)))
		Dens(ans) = dens_Hugoniot(pm,ahead);
	    switch (st_type_ans)
	    {
	    case TGAS_STATE:
	        Vel(ans)[0] = um;
	        Press(ans) = pm;
#if defined(COMBUSTION_CODE)
	        if (Composition_type(ans) == ZND)
	            React(ans) = React(ahead);
#endif /* defined(COMBUSTION_CODE) */
	        break;
	    case EGAS_STATE:
	        Vel(ans)[0] = um;
	        Energy(ans) = specific_internal_energy(ahead) +
	                      0.5*(pa+pm)*(1.0/Dens(ahead) - 1.0/Dens(ans));
#if defined(COMBUSTION_CODE)
	        if (Composition_type(ans) == ZND)
	            React(ans) = React(ahead);
#endif /* defined(COMBUSTION_CODE) */
	        break;
	    case GAS_STATE:
	        Mom(ans)[0] = Dens(ans)*um;
	        Energy(ans) = Dens(ans)*(0.5*sqr(um) +
	                      specific_internal_energy(ahead) + 
	                      0.5*(pa+pm)*(1.0/Dens(ahead) - 1.0/Dens(ans)));
#if defined(COMBUSTION_CODE)
	        if (Composition_type(ans) == ZND)
	            Prod(ans) = Prod(ahead);
#endif /* defined(COMBUSTION_CODE) */
	    }
	    break;

	case RAREFACTION:
	    state_on_adiabat_with_pr(ahead,pm,ans,st_type_ans);
	    if (pressure_is_near_vacuum(pm,ahead))
	    {
	        um = vel(0,ahead) - sgn*riemann_wave_curve(ahead,pm);
	    }
	    switch (st_type_ans)
	    {
	    case TGAS_STATE:
	    case EGAS_STATE:
	        Vel(ans)[0] = um;
	        break;
	    case GAS_STATE:
	        Mom(ans)[0] = Dens(ans)*um;
	        Energy(ans) += 0.5*Mom(ans)[0]*um;
	    }
	    break;

#if defined(COMBUSTION_CODE)
	case STRONG_DET:
	    pa = pressure(ahead);
	    Set_other_params(ans,ahead);
	    Dens(ans) = sgn*m/(um - Vel(ahead)[0] + sgn*m/Dens(ahead));
	    set_type_of_state(ans,st_type_ans);
	    switch (st_type_ans)
	    {
	    case TGAS_STATE:
	        Vel(ans)[0] = um;
	        Press(ans) = pm;
	        break;
	    case EGAS_STATE:
	        Vel(ans)[0] = um;
                      Energy(ans) = specific_internal_energy(ahead) +
	                          0.5*(pa+pm)*(1.0/Dens(ahead) - 1.0/Dens(ans));
	        break;
	    case GAS_STATE:
	        Mom(ans)[0] = Dens(ans)*um;
	        Energy(ans) = Dens(ans)*(0.5*sqr(um) +
	                      specific_internal_energy(ahead) + 
	                      0.5*(pa+pm)*(1.0/Dens(ahead) - 1.0/Dens(ans)));
	    }
	    break;

	case CJ_DET:
	    if (CJ == NULL)
	    {
	        (*Params(ahead)->_alloc_state)(&CJ,Params(ahead)->sizest);
	    }
	    CJ_det(CJ,st_type_ans,ahead,family);
	    Set_other_params(ans,ahead);
	    Set_params(CJ,ans);
#if defined(DEBUG_W_SPEED)
	    if (debug_w_speed == YES)
	    {
	        (void) printf("CJ pressure = %g \n",Press(CJ));
	        (void) printf("CJ density = %g \n",Dens(CJ));
	        (void) printf("CJ velocity = %g \n",Vel(CJ)[0]);
	    }
#endif /* defined(DEBUG_W_SPEED) */
	    state_on_adiabat_with_pr(CJ,pm,ans,st_type_ans);
	    if (st_type_ans == TGAS_STATE || st_type_ans == EGAS_STATE)
	        Vel(ans)[0] = um;
	    else if (st_type_ans == GAS_STATE)
	    {
	        Mom(ans)[0] = Dens(ans)*um;
	        Energy(ans) += 0.5*Mom(ans)[0]*um;
	    }
	    break;
#endif /* defined(COMBUSTION_CODE) */

	default:
	    screen("ERROR: unknown wave %d in midstate()\n",wave);
	    clean_up(ERROR);
	    break;
	}
}	/*end midstate*/





/*
*			g_npt_w_speed():
*
*	Finds the states and wave speed for the solution of a non-local Riemann
*	problem.  Source-term corrections to front states and (in the case of 
*	cylindrical symmetry) to the method of characteristics are included.
*
*
*	Input
*	sten		- WSSten structure containing all input data
*
*	Output
*	ansl, ansr	- left/right propagated states
*	W		- wave velocity
*	
*/

EXPORT	void g_npt_w_speed(
	WSSten		*sten,
	Locstate	ansl,
	Locstate	ansr,
	float		*W)
{
	float		*pt = sten->coords;
	Locstate	 sll;
	Locstate	  sl;
	Locstate	  sr;
	Locstate	 srr;
	float		*nor = sten->nor;
	float		dn = sten->dn;
	float		dt = sten->dt;
	float		pjump = sten->pjump;
	int		w_type = sten->w_type;
	Front		*front = sten->front;
	RIEMANN_SOLVER_WAVE_TYPE l_wave,r_wave;
	RECT_GRID	*gr = front->rect_grid;
	NptWSpeedOpts	Opts;
	int		i, dim = front->interf->dim;
	size_t		sizest = front->sizest;
	float		rr,rl,r;		/* densities */
	float		pml, pmr, pl, pr;	/* pressures */
	float		vtanr[MAXD], vtanl[MAXD], V[MAXD];     /* velocities */
	float		ul,ur,uml,umr,vl,vr;	/* normal velocities */
	float		s;			/* wave speed	*/
	float		cl,cr,cm;		/* sound speeds */
	float		dn1,dn2,dn3;		/* origins of characteristics
	                                           in mesh units */
	float		mr,ml;			/* miscellaneous variables */
	float           dnl, dnr;
	static float	g[MAXD];
	static Locstate left = NULL, right = NULL, mid = NULL,
	                mid_l = NULL, mid_r = NULL, Tsl = NULL, Tsr = NULL;
	static Locstate	st_l1 = NULL, st_l2 = NULL, st_l3 = NULL,
	                st_r1 = NULL, st_r2 = NULL, st_r3 = NULL;
#if defined(COMBUSTION_CODE)
	static Locstate CJ = NULL;
#endif /* defined(COMBUSTION_CODE) */

	Opts = G3ptOpts;

#if defined(DEBUG_W_SPEED)
	debug_print("w_speed","Entered g_npt_w_speed()\n");
	if (debugging("w_speed"))
	    debug_w_speed = YES;
	else if (debug_w_speed == YES)
	    (void) printf("Entered g_npt_w_speed()\n");
#endif /* defined(DEBUG_W_SPEED) */

	if (Tsr == NULL) 
	{
	    alloc_state(front->interf,&st_l1,max(sizeof(VGas),sizest));
	    alloc_state(front->interf,&st_l2,max(sizeof(VGas),sizest));
	    alloc_state(front->interf,&st_l3,max(sizeof(VGas),sizest));
	    alloc_state(front->interf,&st_r1,max(sizeof(VGas),sizest));
	    alloc_state(front->interf,&st_r2,max(sizeof(VGas),sizest));
	    alloc_state(front->interf,&st_r3,max(sizeof(VGas),sizest));
	    alloc_state(front->interf,&mid,sizest);
	    alloc_state(front->interf,&mid_l,sizest);
	    alloc_state(front->interf,&mid_r,sizest);
	    alloc_state(front->interf,&left,max(sizeof(VGas),sizest));
	    alloc_state(front->interf,&right,max(sizeof(VGas),sizest));
	    alloc_state(front->interf,&Tsl,max(sizeof(VGas),sizest));
	    alloc_state(front->interf,&Tsr,max(sizeof(VGas),sizest));

#if defined(COMBUSTION_CODE)
	    alloc_state(front->interf,&CJ,sizest);
#endif /* defined(COMBUSTION_CODE) */

	}
	eval_gravity(pt,front->time,g);

	set_ws_slopes(sten);
	sll  =  sten->sl[1];
	sl   =  sten->sl[0];
	sr   =  sten->sr[0];
	srr  =  sten->sr[1];

#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES)
	{
	    (void) printf("INPUT DATA\n");
	    PrintWSSten(sten);
	    (void) printf("END INPUT DATA\n");
            print_g_npt_w_speed_opts(&Opts);
	}
#endif /* defined(DEBUG_W_SPEED) */

	if (w_type < FIRST_PHYSICS_WAVE_TYPE)
	{
	    for (i = 0; i < dim; ++i)
	        W[i] = 0.0;
	    bdry_npt_w_speed(sten,ansl,ansr,&Opts);
	    Left_w_speed("g_npt_w_speed",ansl,ansr,W,dim);
	    return;
	}

	    /* Set left,right state variables */

	set_type_of_state(ansl,GAS_STATE);
	set_type_of_state(ansr,GAS_STATE);
	set_state_for_find_mid_state(right,sr);
	set_state_for_find_mid_state(left,sl);
	copy_state(Tsr,right);
	copy_state(Tsl,left);

	rr  = Dens(right);		rl  = Dens(left);
	ur = ul = 0.0;
	for ( i = 0; i < dim; ++i)
	{
	    ur += nor[i] * Vel(right)[i];
	    ul += nor[i] * Vel(left)[i];
	}
	for ( i = 0; i < dim; ++i)
	{
	    vtanr[i] = Vel(right)[i] - nor[i] * ur;
	    vtanl[i] = Vel(left)[i]  - nor[i] * ul;
	}


	Vel(right)[0] = ur;	Vel(left)[0] = ul;
	for ( i = 1; i < dim; ++i)
	    Vel(right)[i] = Vel(left)[i] = Vel(mid)[i] = 0.0;

	if (find_mid_state(left,right,pjump,&pml,&pmr,&uml,&umr,&ml,&mr,
	                       &l_wave,&r_wave) != FUNCTION_SUCCEEDED)
	{
	    (void) printf("WARNING in g_npt_w_speed(), find_mid_state() "
	                  "did not converge for left/right RP\n");
	}
	if (strong_interaction(left,right,w_type,pml,pmr,uml,umr) == YES)
	{
#if defined(DEBUG_W_SPEED)
	    if (debug_w_speed == YES)
	    {
	        (void) printf("WARNING in g_npt_w_speed(), "
	                      "strong interaction detected\n");
	    }
#endif /* defined(DEBUG_W_SPEED) */
	    Opts.vector_moc = RIEMANN;
	    Opts.scalar_moc = RIEMANN;
	}
#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES)
	{
	    (void) printf("Answer from find_mid_state()\n"
	                  "pml = %g, pmr = %g\n"
	                  "uml = %g, umr = %g\n"
	                  "ml = %g, mr = %g\n",
	                  pml,pmr,uml,umr,ml,mr);
	    print_rsoln_wave("l_wave = ",l_wave,", ");
	    print_rsoln_wave("r_wave = ",r_wave,"\n");
	}
#endif /* defined(DEBUG_W_SPEED) */

	if (is_backward_wave(w_type))
	{
#if defined(DEBUG_W_SPEED)
	    if (debug_w_speed == YES)
	        (void) printf("Backward wave\n");
#endif /* defined(DEBUG_W_SPEED) */
	    state_behind_sound_wave(left,mid,&cm,&s,pjump,ml,uml,pml,
	                TGAS_STATE,w_type,l_wave,LEFT_FAMILY);

	    uml = Vel(mid)[0];
	    cl = sound_speed(Tsl);
	    for (i = 0; i < dim; ++i)
	        W[i] = nor[i] * s;

	    /* Find feet of characteristics from wave location */

	    dn2 = 1.0 + (s - ul) * dt / dn;
	    dn3 = 1.0 + (s - ul - cl) * dt / dn;

	    dn1 = (s + sound_speed(sll));
	    for (i = 0; i < dim; ++i)
	        dn1 -= (nor[i] * vel(i,sll));
	    dn1 = 1.0 + dn1 * dt / dn;
	    dn1 = min(dn1,1.0 + ((s - uml + cm) * dt / dn));
	    dn1 = max(dn1,dn2);
	    dn1 = min(dn1,1.0);

	    /* Find states ahead of shock at feet of characteristics */

	    ws_interpolate(st_l3,dn3,NEGATIVE_SIDE,TGAS_STATE,sten);
	    ws_interpolate(st_l2,dn2,NEGATIVE_SIDE,TGAS_STATE,sten);
	    ws_interpolate(st_l1,dn1,NEGATIVE_SIDE,TGAS_STATE,sten);

#if defined(DEBUG_W_SPEED)
	    if (debug_w_speed == YES) 
	    {
	        (void) printf("Interpolated states\n");
	        (void) printf("dn1 = %g, dn2 = %g, dn3 = %g\n",dn1,dn2,dn3);
	        verbose_print_state("st_l1",st_l1);
	        verbose_print_state("st_l2",st_l2);
	        verbose_print_state("st_l3",st_l3);
	    }
#endif /* defined(DEBUG_W_SPEED) */

	    if (l_wave == RAREFACTION)
	    {
	        if (w_type==BACKWARD_SOUND_WAVE_TE)
	        {
	            (*Opts.vector_ahead_state_moc)(pt,Tsl,st_l1,st_l2,st_l3,
						   left,-dn,1.0-dn1,1.0-dn2,
						   1.0-dn3,nor,W,YES,dt,front);
	            w_speed(pt,sr,srr,right,Tsr,V,pjump,nor,w_type,front);
	            set_state(left,GAS_STATE,left);
	            w_speed(pt,left,right,ansl,ansr,V,pjump,nor,w_type,front);
	            for (i = 0; i < dim; ++i)
	                W[i] = 0.5*(W[i] + V[i]);
	            copy_state(ansl,ansr);
	        }
		else
	        {
	            (*Opts.vector_ahead_state_moc)(pt,Tsl,st_l1,st_l2,st_l3,
						   left,-dn,1.0-dn1,1.0-dn2,
						   1.0-dn3,nor,W,YES,dt,front);


	            s = - sound_speed(left);
	            for (i = 0; i < dim; ++i)
	                s += nor[i]*Vel(left)[i];
	            for (i = 0; i < dim; ++i)
	            {
	                V[i] = nor[i] * s;
	                W[i] = 0.5*(W[i] + V[i]);
	            }

	            set_state(ansl,GAS_STATE,left);
	            copy_state(ansr,ansl);
	        }
	    }
	    else
	    {

	        /* Find state behind shock at foot of characteristic */
	        /*   Find foot of characteristic from wave location  */

	        dn1 = (s + sound_speed(srr));
	        for (i = 0; i < dim; ++i)
	            dn1 -= (nor[i] * vel(i,srr));
	        dn1 *= dt / dn;
	        dn1 = max(dn1,((s - umr + cm) * dt / dn));
	        dn1 = min(dn1,1.0);
	        dn1 = max(dn1,0.0);


	        ws_interpolate(st_r1,dn1,POSITIVE_SIDE,TGAS_STATE,sten);
#if defined(DEBUG_W_SPEED)
	        if (debug_w_speed == YES) 
	        {
	            (void) printf("Interpolated right state\n");
	            (void) printf("dn1 = %g\n",dn1);
	            verbose_print_state("st_r1",st_r1);
	        }
#endif /* defined(DEBUG_W_SPEED) */

	        switch (Opts.vector_moc)
	        {
	        case RIEMANN:
	            /* Find left state after time dt */
	            /* Solve characteristic PDEs to get left state */

	            (*Opts.vector_ahead_state_moc)(pt,Tsl,st_l1,st_l2,st_l3,
						   left,
						   -dn,1.0-dn1,1.0-dn2,1.0-dn3,
						   nor,W,NO,0.0,front);

	            set_state(ansl,GAS_STATE,left);

#if defined(DEBUG_W_SPEED)
	            if (debug_w_speed == YES) 
	            {
	                (void) printf("left state after dt\n");
	                verbose_print_state("ansl",ansl);
	            }
#endif /* defined(DEBUG_W_SPEED) */

	        /* Separate out back mode at foot of characteristic */
	        /* by solving Riemann problem between sr, s1 */

	            set_state_for_find_mid_state(st_r1,st_r1);
	            Vel(st_r1)[0] = scalar_product(Vel(st_r1),nor,dim);
	            Set_params(st_r1,sr);

	            if (find_mid_state(right,st_r1,0.0,&pl,&pr,&vl,&vr,&ml,&mr,
				       &l_wave,&r_wave) != FUNCTION_SUCCEEDED)
	            {
	                (void) printf("WARNING in g_npt_w_speed(), "
	                              "find_mid_state() did not converge for "
	                              "backward wave right/st_r1 RP\n");
	            }

	            switch (l_wave) 
	            {
	            case SHOCK:
	                Set_params(st_r1,sr);
	                r = ml/(vl - ur + ml/rr);
	                if (invalid_shock(pressure(Tsr),Dens(Tsr),pl,r))
		            r = dens_Hugoniot(pl,Tsr);
	                break;

	            case RAREFACTION:
	                Set_params(st_r1,sr);
	                r = dens_rarefaction(pl,Tsr);
	                break;

#if defined(COMBUSTION_CODE)
	            case STRONG_DET:
	                Set_other_params(st_r1,sr);
	                r = ml/(vl - ur + ml/rr);
	                break;

	            case CJ_DET:
	                CJ_det(CJ,TGAS_STATE,right,LEFT_FAMILY);
	                Set_other_params(st_r1,sr);
	                r = Dens(CJ);
	                vl = Vel(CJ)[0];
	                break;
#endif /* defined(COMBUSTION_CODE) */
	        
	            default:
	                screen("ERROR: unknown wave ");
	                screen("%d in g_npt_w_speed\n",l_wave);
	                clean_up(ERROR);
	            }

	            Dens(st_r1) = r;
	            Press(st_r1) = pl;
	            set_type_of_state(st_r1,TGAS_STATE);
	            for (i = 0; i < dim; ++i)
	                Vel(st_r1)[i] = vtanr[i] + nor[i] * vl;
#if defined(COMBUSTION_CODE)
	            if (Composition_type(sr) == ZND)
	                React(st_r1) = React(Tsr);
#endif /* defined(COMBUSTION_CODE) */
	        
	        /* Solve Riemann problem to find right state */

	            set_state(st_r1,GAS_STATE,st_r1);
	            copy_state(left,ansl);
	            w_speed(pt,left,st_r1,ansl,ansr,V,pjump,nor,w_type,front);
#if defined(DEBUG_W_SPEED)
	            if (debug_w_speed == YES) 
	            {
	                (void) printf("States after w_speed\n");
	                verbose_print_state("left",left);
	                verbose_print_state("st_r1",st_r1);
	                verbose_print_state("ansl",ansl);
	                verbose_print_state("ansr",ansr);
	            }
#endif /* defined(DEBUG_W_SPEED) */

	        /* Use centered difference in time for wave speed */

	            for (i = 0; i < dim; ++i)
	                W[i] = 0.5*(W[i] + V[i]);

	            include_source(pt,ansl,ul,dt,nor,W,g,gr,w_type);
	            include_source(pt,ansr,uml,dt,nor,W,g,gr,w_type);
	            break;

	        case MOC_PLUS_RIEMANN:

	        /* Find left state after time dt */
	        /* Solve characteristic PDEs to get left state */

	            (*Opts.vector_ahead_state_moc)(pt,Tsl,st_l1,st_l2,st_l3,
						   left,
						   -dn,1.0-dn1,1.0-dn2,1.0-dn3,
						   nor,W,NO,0.0,front);

	            set_state(ansl,GAS_STATE,left);
	            set_state(st_r1,GAS_STATE,st_r1);

#if defined(DEBUG_W_SPEED)
	            if (debug_w_speed == YES) 
	            {
	                (void) printf("left state after dt\n");
	                verbose_print_state("ansl",ansl);
	            }
#endif /* defined(DEBUG_W_SPEED) */

	            /* Solve Riemann problem to find right state */

	            copy_state(left,ansl);
	            w_speed(pt,left,st_r1,ansl,ansr,V,pjump,nor,w_type,front);
#if defined(DEBUG_W_SPEED)
	            if (debug_w_speed == YES) 
	            {
	                (void) printf("States after w_speed\n");
	                verbose_print_state("left ",left);
	                verbose_print_state("st_r1",st_r1);
	                verbose_print_state("ansl ",ansl);
	                verbose_print_state("ansr ",ansr);
	            }
#endif /* defined(DEBUG_W_SPEED) */

	        /* Use centered difference in time for wave speed */

	            for (i = 0; i < dim; ++i)
	                W[i] = 0.5*(W[i] + V[i]);

	            include_source(pt,ansl,ul,dt,nor,W,g,gr,w_type);
	            include_source(pt,ansr,uml,dt,nor,W,g,gr,w_type);
	            break;
	        
	        case MOC_PLUS_RH:

	        /* Find left state after time dt */
	        /* Solve characteristic PDEs to get left state */

	            (*Opts.vector_ahead_state_moc)(pt,Tsl,st_l1,st_l2,st_l3,
						   left,
						   -dn,1.0-dn1,1.0-dn2,1.0-dn3,
						   nor,W,YES,dt,front);

	            set_state(ansl,GAS_STATE,left);

#if defined(DEBUG_W_SPEED)
	            if (debug_w_speed == YES) 
	            {
	                (void) printf("left state after dt\n");
	                verbose_print_state("ansl",ansl);
	            }
#endif /* defined(DEBUG_W_SPEED) */

	    /* Use characteristic equations and two Riemann invariants */

	            set_state(st_r1,TGAS_STATE,st_r1);
	            if (!shock_moc_plus_rh(pt,left,mid,st_r1,ansr,
	                                   -dn1*dn,nor,W,w_type,front))
		    {
			MOC_TYPE vector_moc = G3ptOpts.vector_moc;
			G3ptOpts.vector_moc = RIEMANN;
			g_npt_w_speed(sten,ansl,ansr,W);
			G3ptOpts.vector_moc = vector_moc;
			return;
		    }
	        }
	    }    
	}
	else if (is_scalar_wave(w_type))
	{
#if defined(DEBUG_W_SPEED)
	    if (debug_w_speed == YES)
	        (void) printf("Contact\n");
#endif /* defined(DEBUG_W_SPEED) */
	    set_type_of_state(mid_l,TGAS_STATE);
	    /* Use mid as temporary storage to compute sound speed on left */
	    Press(mid_l) = pml;
	    Vel(mid_l)[0] = uml;
	    switch (l_wave) 
	    {
	    case SHOCK:
	        Set_params(mid_l,sl);
	        Dens(mid_l) = ml/(uml - ul + ml/rl);
	        if (invalid_shock(pressure(Tsl),Dens(Tsl),pml,Dens(mid_l)))
		    Dens(mid_l) = dens_Hugoniot(pml,Tsl);
	        break;

	    case RAREFACTION:
	        Set_params(mid_l,sl);
	        Dens(mid_l)= dens_rarefaction(Press(mid_l),Tsl);
	        if (pressure_is_near_vacuum(pml,left))
	            Vel(mid_l)[0] = ul - riemann_wave_curve(left,Press(mid_l));
	        break;

#if defined(COMBUSTION_CODE)
	    case STRONG_DET:
	        Set_other_params(mid_l,sl);
	        Dens(mid_l) = ml/(uml - ul + ml/rl);
	        break;

	    case CJ_DET:
	        CJ_det(CJ,TGAS_STATE,left,LEFT_FAMILY);
	        Set_other_params(mid_l,sl);
	        Set_params(CJ,mid_l);
	        Dens(mid_l) = dens_rarefaction(Press(mid_l),CJ);
	        break;
#endif /* defined(COMBUSTION_CODE) */
	    
	    default:
	        screen("ERROR: unknown wave %d in w_speed\n",l_wave);
	        clean_up(ERROR);
	        break;
	    }
#if defined(COMBUSTION_CODE)
	    if (Composition_type(sl) == ZND)
	        React(mid_l) = React(Tsl); 
#endif /* defined(COMBUSTION_CODE) */
	    cl = sound_speed(mid_l);

	    set_type_of_state(mid_r,TGAS_STATE);
	    Press(mid_r) = pmr;	/* Store pm in mid */
	    Vel(mid_r)[0] = umr;
	    switch (r_wave) 
	    {
	    case SHOCK:
	        Set_params(mid_r,sr);
	        Dens(mid_r) = -mr/(umr - ur - mr/rr);
	        if (invalid_shock(pressure(Tsr),Dens(Tsr),pmr,Dens(mid_r)))
		    Dens(mid_r) = dens_Hugoniot(pmr,Tsr);
	        break;

	    case RAREFACTION:
	        Set_params(mid_r,sr);
	        Dens(mid_r)= dens_rarefaction(pmr,Tsr);
	        if (pressure_is_near_vacuum(pmr,right))
	            Vel(mid_r)[0] = ur + riemann_wave_curve(right,Press(mid_r));
	        break;

#if defined(COMBUSTION_CODE)
	    case STRONG_DET:
	        Set_other_params(mid_r,sr);
	        Dens(mid_r) = -mr/(umr - ur - mr/rr);
	        break;

	    case CJ_DET:
	        CJ_det(CJ,TGAS_STATE,right,RIGHT_FAMILY);
	        Set_other_params(mid_r,sr);
	        Set_params(CJ,mid_r);
	        Dens(mid_r) = dens_rarefaction(Press(mid_r),CJ);
	        break;
#endif /* defined(COMBUSTION_CODE) */

	    default:
	        screen("ERROR: unknown wave %d in w_speed\n",r_wave);
	        clean_up(ERROR);
	        break;
	    }
	    Set_params(mid_r,sr);
#if defined(COMBUSTION_CODE)
	    if (Composition_type(sl) == ZND)
	        React(mid_r) = React(Tsr); 
#endif /* defined(COMBUSTION_CODE) */
	    cr = sound_speed(mid_r);

	    for (i = 0; i < dim; ++i)
	        W[i] = nor[i] * 0.5 * (uml + umr);

	    /* Find right and left states after time dt */

	    /* Find far left and right states */
	    IncomingStatesAtContact(sten,cl,cr,st_l3,st_r1,&dnl,&dnr,&Opts);

#if defined(DEBUG_W_SPEED)
	    if (debug_w_speed == YES) 
	    {
	        (void) printf("Interpolated states\n");
	        verbose_print_state("st_r1",st_r1);
	        verbose_print_state("st_l3",st_l3);
	    }
#endif /* defined(DEBUG_W_SPEED) */

#if !defined(COMBUSTION_CODE)
	    Set_params(st_r1,sr);    Set_params(st_l3,sl);
#endif /* !defined(COMBUSTION_CODE) */

	    switch (Opts.scalar_moc)
	    {
	    case RIEMANN:
	        contact_riemann(left,Tsl,right,Tsr,st_l3,dnl,st_r1,dnr,ansl,ansr,
			        sten,W,&Opts,g);
	        break;

	    case MOC_PLUS_RIEMANN:
	        set_state(st_r1,TGAS_STATE,st_r1);
	        set_state(st_l3,TGAS_STATE,st_l3);
	        /* Find corrected left and right states */
	        /* by solving characteristic equations */
	        contact_cheap_moc(pt,Tsl,Tsr,st_l3,st_r1,ansl,ansr,
				  cl,cr,pjump,dnl,dnr,nor,W,front);
	        break;


	    case MOC_PLUS_RH:
	        set_state(st_r1,TGAS_STATE,st_r1);
	        set_state(st_l3,TGAS_STATE,st_l3);
	        /* Use characteristic equations and */
	        /* two Riemann invariants */
	        contact_moc_plus_rh(pt,pmr,Tsl,Tsr,st_l3,st_r1,
	                            ansl,ansr,pjump,dnl,dnr,nor,W,front);
	        break;
	    }
	}
	else if (is_forward_wave(w_type)) 
	{
#if defined(DEBUG_W_SPEED)
	    if (debug_w_speed == YES)
	        (void) printf("Forward Wave\n");
#endif /* defined(DEBUG_W_SPEED) */
	    state_behind_sound_wave(right,mid,&cm,&s,pjump,mr,umr,pmr,
	                TGAS_STATE,w_type,r_wave,RIGHT_FAMILY);

	    umr = Vel(mid)[0];
	    cr = sound_speed(Tsr);

	    for (i = 0; i < dim; ++i)
	        W[i] = nor[i] * s;

	        /* Find feet of characteristics ahead */
	        /*  of the shock from wave location   */

	    dn1 = (s - ur + cr) * dt / dn;
	    dn2 = (s - ur) * dt / dn;

	    dn3 = s - sound_speed(srr);
	    for (i = 0; i < dim; ++i)
	        dn3 -= nor[i] * vel(i,srr);
	    dn3 *= dt / dn;
	    dn3 = max(dn3,((s - umr - cm) * dt / dn));
	    dn3 = min(dn3,dn2);
	    dn3 = max(dn3,0.0);

	    /* Find states ahead of the shock at feet of characteristics */

	    ws_interpolate(st_r1,dn1,POSITIVE_SIDE,TGAS_STATE,sten);
	    ws_interpolate(st_r2,dn2,POSITIVE_SIDE,TGAS_STATE,sten);
	    ws_interpolate(st_r3,dn3,POSITIVE_SIDE,TGAS_STATE,sten);

#if defined(DEBUG_W_SPEED)
	    if (debug_w_speed == YES) 
	    {
	        (void) printf("Interpolated states\n");
	        (void) printf("dn1 = %g, dn2 = %g, dn3 = %g\n",dn1,dn2,dn3);
	        verbose_print_state("st_r1",st_r1);
	        verbose_print_state("st_r2",st_r2);
	        verbose_print_state("st_r3",st_r3);
	    }
#endif /* defined(DEBUG_W_SPEED) */

	    if (r_wave==RAREFACTION)
	    {
	        if (w_type==FORWARD_SOUND_WAVE_TE)
	        {
	            (*Opts.vector_ahead_state_moc)(pt,Tsr,st_r1,st_r2,st_r3,
						   right,dn,dn1,dn2,dn3,
					           nor,W,YES,dt,front);

	            w_speed(pt,sll,sr,Tsl,left,V,pjump,nor,w_type,front);
	            set_state(right,GAS_STATE,right);
	            w_speed(pt,left,right,ansl,ansr,V,pjump,nor,w_type,front);
	            for (i = 0; i < dim; ++i)
	                W[i] = 0.5*(W[i] + V[i]);
	            copy_state(ansr,ansl);
	        }
		else
	        {
	            (*Opts.vector_ahead_state_moc)(pt,Tsr,st_r1,st_r2,st_r3,
						   right,dn,dn1,dn2,dn3,
					           nor,W,YES,dt,front);


	            s = sound_speed(right);
	            for (i = 0; i < dim; ++i)
	                s += nor[i] * Vel(right)[i];
	            for (i = 0; i < dim; ++i)
	            {
	                V[i] = nor[i] * s;
	                W[i] = 0.5*(W[i] + V[i]);
	            }

	            set_state(ansr,GAS_STATE,right);
	            copy_state(ansl,ansr);
	        }
	    }
	    else
	    {
	        /* Find foot of characteristic behind */
	        /*    shock from wave location      */

	        dn3 = s - sound_speed(sll);
	        for (i = 0; i < dim; ++i)
	            dn3 -= nor[i] * vel(i,sll);
	        dn3 = 1.0 + dn3 * dt / dn;
	        dn3 = min(dn3,(1.0 + (s - uml - cm) * dt / dn));
	        dn3 = min(dn3,1.0);
	        dn3 = max(dn3,0.0);

	        /* Find state behind shock at foot of characteristic */

	        ws_interpolate(st_l3,dn3,NEGATIVE_SIDE,TGAS_STATE,sten);
#if defined(DEBUG_W_SPEED)
	        if (debug_w_speed == YES) 
	        {
	            (void) printf("Left interpolated state\n""dn3 = %g\n",dn3);
	            verbose_print_state("st_l3",st_l3);
	        }
#endif /* defined(DEBUG_W_SPEED) */


	        switch (Opts.vector_moc)
	        {
	        case RIEMANN:

	            /* Find right state after time dt */
	            /* Solve characteristic PDEs for right state */

	            (*Opts.vector_ahead_state_moc)(pt,Tsr,st_r1,st_r2,st_r3,
	                                           right,dn,dn1,dn2,dn3,
						   nor,W,NO,0.0,front);
	            
	            set_state(ansr,GAS_STATE,right);

#if defined(DEBUG_W_SPEED)
	            if (debug_w_speed == YES) 
	            {
	                (void) printf("right state after dt\n");
	                verbose_print_state("ansr",ansr);
	            }
#endif /* defined(DEBUG_W_SPEED) */

	                /*   Separate out forward mode at foot  */
	                /* of characteristic by solving Riemann */
	                /*    problem between st_l3, sl       */

	            set_state_for_find_mid_state(st_l3,st_l3);
	            Vel(st_l3)[0] = scalar_product(Vel(st_l3),nor,dim);
	            Set_params(st_l3,sl);

	            if (find_mid_state(st_l3,left,0.0,&pl,&pr,&vl,&vr,&ml,&mr,
				       &l_wave,&r_wave) != FUNCTION_SUCCEEDED)
	            {
	                (void) printf("WARNING in g_npt_w_speed(), "
	                              "find_mid_state() did not converge for "
	                              "forward wave st_l3/left RP\n");
	            }

	            switch (r_wave) 
	            {
	            case SHOCK:
	                r = -mr/(vr - ul - mr/rl);
	                if (invalid_shock(pressure(Tsl),Dens(Tsl),pr,r))
		            r = dens_Hugoniot(pr,Tsl);
	                Set_params(st_l3,sl);
	                break;

	            case RAREFACTION:
	                r = dens_rarefaction(pr,Tsl);
	                Set_params(st_l3,sl);
	                break;

#if defined(COMBUSTION_CODE)
	            case STRONG_DET:
	                Set_other_params(st_l3,sl);
	                r = -mr/(vr - ul - mr/rl);
	                break;

	            case CJ_DET:
	                CJ_det(CJ,TGAS_STATE,left,RIGHT_FAMILY);
	                Set_other_params(st_l3,sl);
	                r = Dens(CJ);
	                vr = Vel(CJ)[0];
	                break;
#endif /* defined(COMBUSTION_CODE) */
	            
	            default:
	                screen("ERROR: unknown wave ");
	                screen("%d in g_npt_w_speed\n",r_wave);
	                clean_up(ERROR);
	                break;
	            }

	            set_type_of_state(st_l3,TGAS_STATE);
	            Dens(st_l3) = r;
	            Press(st_l3) = pr;
	            for ( i = 0; i < dim; ++i)
	                Vel(st_l3)[i] = vtanl[i] + nor[i] * vr;


#if defined(COMBUSTION_CODE)
	            if (Composition_type(sl) == ZND)
	                React(st_l3) = React(Tsl);
#endif /* defined(COMBUSTION_CODE) */
	            
	            /* Solve Riemann problem to find right state */

	            set_state(st_l3,GAS_STATE,st_l3);
	            copy_state(right,ansr);
	            w_speed(pt,st_l3,right,ansl,ansr,V,pjump,nor,w_type,front);
#if defined(DEBUG_W_SPEED)
	            if (debug_w_speed == YES) 
	            {
	                (void) printf("States after w_speed\n");
	                verbose_print_state("st_l3",st_l3);
	                verbose_print_state("right",right);
	                verbose_print_state("ansl ",ansl);
	                verbose_print_state("ansr ",ansr);
	            }
#endif /* defined(DEBUG_W_SPEED) */

	            /* Use centered in time difference for wave speed */

	            for ( i= 0; i < dim; ++i)
	                W[i] = 0.5*(W[i] + V[i]);

	            include_source(pt,ansl,umr,dt,nor,W,g,gr,w_type);
	            include_source(pt,ansr,ur,dt,nor,W,g,gr,w_type);
	            break;

	        case MOC_PLUS_RIEMANN:

	            /* Find right state after time dt */
	            /* Solve characteristic PDEs for right state */

	            (*Opts.vector_ahead_state_moc)(pt,Tsr,st_r1,st_r2,st_r3,
						   right,dn,dn1,dn2,dn3,
						   nor,W,NO,0.0,front);
	            
	            set_state(ansr,GAS_STATE,right);
	            set_state(st_l3,GAS_STATE,st_l3);

#if defined(DEBUG_W_SPEED)
	            if (debug_w_speed == YES) 
	            {
	                (void) printf("right state after dt\n");
	                verbose_print_state("ansr",ansr);
	            }
#endif /* defined(DEBUG_W_SPEED) */

	            /* Solve Riemann problem to find left state */

	            copy_state(right,ansr);
	            w_speed(pt,st_l3,right,ansl,ansr,V,pjump,nor,w_type,front);

#if defined(DEBUG_W_SPEED)
	            if (debug_w_speed == YES) 
	            {
	                (void) printf("States after w_speed\n");
	                verbose_print_state("st_l3",st_l3);
	                verbose_print_state("right",right);
	                verbose_print_state("ansl",ansl);
	                verbose_print_state("ansr",ansr);
	            }
#endif /* defined(DEBUG_W_SPEED) */

	            /* Use centered in time difference for wave speed */

	            for (i = 0; i < dim; ++i)
	                W[i] = 0.5*(W[i] + V[i]);

	            include_source(pt,ansl,umr,dt,nor,W,g,gr,w_type);
	            include_source(pt,ansr,ur,dt,nor,W,g,gr,w_type);
	            break;

	        case MOC_PLUS_RH:

	            /* Find right state after time dt */
	            /* Solve characteristic PDEs for right state */

	            (*Opts.vector_ahead_state_moc)(pt,Tsr,st_r1,st_r2,st_r3,
						   right,dn,dn1,dn2,dn3,
						   nor,W,YES,dt,front);
	            
	            set_state(ansr,GAS_STATE,right);

#if defined(DEBUG_W_SPEED)
	            if (debug_w_speed == YES) 
	            {
	                (void) printf("right state after dt\n");
	                verbose_print_state("ansr",ansr);
	            }
#endif /* defined(DEBUG_W_SPEED) */

	                /* Use characteristic equations and */
	                /*    two Riemann invariants        */

	            set_state(st_l3,TGAS_STATE,st_l3);
	            if (!shock_moc_plus_rh(pt,right,mid,st_l3,ansl,
				           (1.0 - dn3)*dn,nor,W,w_type,front))
		    {
			MOC_TYPE vector_moc = G3ptOpts.vector_moc;
			G3ptOpts.vector_moc = RIEMANN;
			g_npt_w_speed(sten,ansl,ansr,W);
			G3ptOpts.vector_moc = vector_moc;
			return;
		    }
	        }
	    }
	}
	else
	{
	    screen("ERROR: unknown wave_type %d in g_npt_w_speed\n",w_type);
	    clean_up(ERROR);
	}

	Left_w_speed("g_npt_w_speed",ansl,ansr,W,dim);
}	/*end g_npt_w_speed*/


LOCAL	void bdry_npt_w_speed(
	WSSten		*wssten,
	Locstate	ansl,
	Locstate	ansr,
	NptWSpeedOpts   *opts)
{
	Locstate	 sll =  wssten->sl[1];
	Locstate	  sl =  wssten->sl[0];
	Locstate	  sr =  wssten->sr[0];
	Locstate	 srr =  wssten->sr[1];
	COMPONENT	l_comp = wssten->ncomp;
	COMPONENT	r_comp = wssten->pcomp;
	COMPONENT       comp;
	float		*nor = wssten->nor;
	float		dn = wssten->dn;
	float		dt = wssten->dt;
	float           W[3];
	int		w_type = wssten->w_type;
	Front		*front = wssten->front;
	Wave		*wave = wssten->wave;
	SIDE            side;
	HYPER_SURF	*hs = wssten->hs;
	float		*pt = wssten->coords;
	float           alpha;
	static	int	nrad;
	static	Tan_stencil *sten = NULL;
	static	Locstate left = NULL, right = NULL, Tsl = NULL, Tsr = NULL;
	static	Locstate st_l3 = NULL, st_r1 = NULL;
	static	Locstate stemp = NULL;
	static  Locstate ansl_1, ansl_2, ansr_1, ansr_2;

	float		ul, ur;		/* normal velocities */
	float		cl, cr;		/* sound speeds */
	float		dn1, dn3;	/* origins of characteristics
	                                   in mesh units */
	size_t		sizest = front->sizest;
	INTERFACE       *intfc = front->interf;
	int             i, j, k, dim = intfc->dim;

#if defined(DEBUG_W_SPEED)
	debug_print("w_speed","Entered bdry_npt_w_speed()\n");
	if (debugging("w_speed"))
	    debug_w_speed = YES;
	else if (debug_w_speed == YES)
	    (void) printf("Entered bdry_npt_w_speed()\n");
	if (debug_w_speed == YES)
	    print_wave_type("wave type = ",w_type,"\n",intfc);
#endif /* defined(DEBUG_W_SPEED) */

	if (Tsr == NULL) 
	{
	    nrad = front->npts_tan_sten/2;
	    sten = alloc_tan_stencil(front,nrad);
	    set_default_tan_stencil(sten);
	    alloc_state(intfc,&st_l3,max(sizeof(VGas),sizest));
	    alloc_state(intfc,&st_r1,max(sizeof(VGas),sizest));
	    alloc_state(intfc,&left,max(sizeof(VGas),sizest));
	    alloc_state(intfc,&right,max(sizeof(VGas),sizest));
	    alloc_state(intfc,&Tsl,sizest);
	    alloc_state(intfc,&Tsr,sizest);
	    alloc_state(intfc,&stemp,sizest);
	    alloc_state(intfc,&ansl_1,sizest);
	    alloc_state(intfc,&ansl_2,sizest);
	    alloc_state(intfc,&ansr_1,sizest);
	    alloc_state(intfc,&ansr_2,sizest);
	}

#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES) 
	{
	    print_general_vector("nor = ",nor,dim,"");
	    (void) printf(", dn = %g, dt = %g\n",dn,dt);
	    (void) printf("Initial data -- sll, sl, sr, srr:\n");
	    verbose_print_state("sll",sll);
	    verbose_print_state("sl ",sl );
	    verbose_print_state("sr ",sr );
	    verbose_print_state("srr",srr);
	}
#endif /* defined(DEBUG_W_SPEED) */

	set_type_of_state(ansl,GAS_STATE);
	set_type_of_state(ansr,GAS_STATE);
	switch (w_type)
	{
	case PASSIVE_BOUNDARY: 
	case SUBDOMAIN_BOUNDARY:
	    /* PASSIVE and SUBDOMAIN states not set or used */
	    obstacle_state(intfc,ansl,sizest);
	    obstacle_state(intfc,ansr,sizest);
	    break;

	case DIRICHLET_BOUNDARY:
	    k = min(wssten->nsts-1,nrad);
	    for (i = 0; i < dim; ++i)
	        Coords(sten->p[0])[i] = pt[i];
	    for (j = 1; j <= nrad; ++j)
	    {
	        for (i = 0; i < dim; ++i)
	        {
	            Coords(sten->p[j])[i]  = pt[i] + j*dn*nor[i];
	            Coords(sten->p[-j])[i] = pt[i] - j*dn*nor[i];
	        }
	    }
	    if (is_obstacle_state(sl)) /*left side is exterior*/
	    {
		comp = r_comp;
	        for (j = 0; j <= nrad; ++j)
	            copy_state(sten->rightst[j],sr);
	        for (j = 1; j <= nrad; ++j)
	            hyp_solution(Coords(sten->p[-j]),comp,hs,UNKNOWN_SIDE,
			         front,wave,sten->rightst[-j],
				 sten->rightst[-j+1]);
		for (j = -nrad; j <= nrad; ++j)
	            Set_params(sten->rightst[j],sr);
	    }
	    else
	    {
		comp = l_comp;
		for (j = 0; j <= k; ++j)
		    copy_state(sten->rightst[-j],wssten->sl[j]);
		for (; j <= nrad; ++j)
		    hyp_solution(Coords(sten->p[-j]),comp,hs,UNKNOWN_SIDE,
				 front,wave,sten->rightst[-j],
				 sten->rightst[-j+1]);
		for (j = 1; j <= nrad; ++j)
		    copy_state(sten->rightst[j],sl);
		for (j = -nrad; j <= nrad; ++j)
	            Set_params(sten->rightst[j],sl);
	    }
	    sten->newhs = NULL;
	    sten->comp = comp;
	    sten->states = sten->rightst;
	    sten->dir = nor;
	    one_side_npt_tang_solver(dn,dt,sten,ansl,front);

	    if (is_obstacle_state(sr)) /*right side is exterior*/
	    {
		comp = l_comp;
	        for (j = 0; j <= nrad; ++j)
	            copy_state(sten->rightst[-j],sl);
	        for (j = 1; j <= nrad; ++j)
	            hyp_solution(Coords(sten->p[j]),comp,hs,UNKNOWN_SIDE,
			         front,wave,sten->rightst[j],
				 sten->rightst[j-1]);
		for (j = -nrad; j <= nrad; ++j)
	            Set_params(sten->rightst[j],sl);
	    }
	    else
	    {
		comp = r_comp;
	        for (j = 0; j <= k; ++j)
	            copy_state(sten->rightst[j],wssten->sr[j]);
	        for (; j <= nrad; ++j)
	            hyp_solution(Coords(sten->p[j]),comp,hs,UNKNOWN_SIDE,
			         front,wave,sten->rightst[j],
				 sten->rightst[j-1]);
	        for (j = 1; j <= nrad; ++j)
	            copy_state(sten->rightst[-j],sr);
		for (j = -nrad; j <= nrad; ++j)
	            Set_params(sten->rightst[j],sr);
	    }
	    sten->newhs = NULL;
	    sten->comp = comp;
	    sten->states = sten->rightst;
	    sten->dir = nor;
	    one_side_npt_tang_solver(dn,dt,sten,ansr,front);

#if defined(DEBUG_W_SPEED)
	    if (debug_w_speed == YES) 
	    {
	        (void) printf("Intermediate left, right states:\n");
	        verbose_print_state("ansl",ansl);
	        verbose_print_state("ansr",ansr);
	    }
#endif /* defined(DEBUG_W_SPEED) */

	        /*  Source terms are included above in the two  */
	        /* calls to one_side_npt_tang_solver(). */

	    if (!is_obstacle_state(sl))
	    {
		// apply_bc(ansr,wave,hs);
                evaluate_dirichlet_boundary_state(Coords(sten->p[0]),hs,front,wave,ansr); 
	        riemann_solution(0.0,nor,ansl,ansr,ansl,GAS_STATE);
#if !defined(COMBUSTION_CODE)
	        Set_params(ansl,sl);
#endif /* !defined(COMBUSTION_CODE) */
	    }
	    if (!is_obstacle_state(sr))
	    {
		// apply_bc(ansl,wave,hs);
                evaluate_dirichlet_boundary_state(Coords(sten->p[0]),hs,front,wave,ansl); 
	        riemann_solution(0.0,nor,ansl,ansr,ansr,GAS_STATE);
#if !defined(COMBUSTION_CODE)
	        Set_params(ansr,sr);
#endif /* !defined(COMBUSTION_CODE) */
	    }

	    if (is_obstacle_state(sl))
	        obstacle_state(intfc,ansl,sizest);
	    if (is_obstacle_state(sr))
	        obstacle_state(intfc,ansr,sizest);

	    break;

	case NEUMANN_BOUNDARY:
	    /* Use the Method of Characteristics in the normal direction */


	    if (is_obstacle_state(sr)) 
		side = NEGATIVE_SIDE;
	    else if (is_obstacle_state(sl)) 
		side = POSITIVE_SIDE;
	    else
	    {
	        side = UNKNOWN_SIDE;
		screen("ERROR in bdry_npt_w_speed(), invalid state "
		       "on NEUMANN_BOUNDARY,  no obstacle state found\n");
		clean_up(ERROR);
	    }
	    reflect_wssten(wssten,side,front);
	    wssten->w_type = CONTACT;
	    npt_w_speed(wssten,ansl_1,ansr_1,W);
	    alpha = wall_limiter(wssten,side,opts);
	    wssten->w_type = NEUMANN_BOUNDARY;

	    if (side == POSITIVE_SIDE) /* Right Side */
	    {
	        obstacle_state(intfc,ansl,sizest);
		for (i = 0; i < wssten->nsts; ++i)
		{
	            obstacle_state(intfc,wssten->sl[i],sizest);
	            obstacle_state(intfc,wssten->tsl[i],sizest);
	            obstacle_state(intfc,wssten->dsl[i],sizest);
		}

	        set_state(Tsr,TGAS_STATE,sr);
	        ur = 0.0;
	        if (include_wall_normal_velocity(front) == YES)
	            ur += scalar_product(nor,Vel(Tsr),dim);
	        cr = sound_speed(Tsr);
	        dn1 = cr * dt / dn;
	        ws_interpolate(st_r1,dn1,POSITIVE_SIDE,TGAS_STATE,wssten);
#if defined(DEBUG_W_SPEED)
	        if (debug_w_speed == YES)
	        {
	            (void) printf("Interpolated state, dn1 = %g\n",dn1);
	            verbose_print_state("st_r1",st_r1);
		    (void) printf("Calling neumann_moc()\n");
	        }
#endif /* defined(DEBUG_W_SPEED) */
	        (*opts->neumann_moc)(pt,Tsr,ur,cr,st_r1,side,
				     right,cr*dt,nor,front);
	        set_state(ansr_2,GAS_STATE,right);

	        if (Params(Tsr)->avisc.heat_cond != 0.0)
	        {
	            float heat_cond = Params(Tsr)->avisc.heat_cond;
		    float Tw = temperature(Tsr);
		    float Gam = gruneisen_gamma(Tsr);
		    float dS = heat_cond * (temperature(st_r1)/Tw - 1.0);
		    Dens(ansr_1) *= exp(-Gam*Tw*dS/sqr(cr));
	        }
		interpolate_states(front,alpha,1.0-alpha,pt,
		                   ansr_1,pt,ansr_2,ansr);
		if (no_slip(hs))
		{
		    float alpha = 1.0 - adherence_coeff(hs);
		    alpha_state_velocity(alpha,ansr,dim);
		}
		zero_normal_velocity(ansr,nor,dim);
	    }
	    else if (side == NEGATIVE_SIDE) /* Left Side */
	    {
	        obstacle_state(intfc,ansr,sizest);
		for (i = 0; i < wssten->nsts; ++i)
		{
	            obstacle_state(intfc,wssten->sr[i],sizest);
	            obstacle_state(intfc,wssten->tsr[i],sizest);
	            obstacle_state(intfc,wssten->dsr[i],sizest);
		}

	        set_state(Tsl,TGAS_STATE,sl);
	        ul = 0.0;
	        if (include_wall_normal_velocity(front) == YES)
	            ul += scalar_product(nor,Vel(Tsl),dim);
	        cl = sound_speed(Tsl);
	        dn3 = 1.0 - cl * dt / dn;
	        ws_interpolate(st_l3,dn3,NEGATIVE_SIDE,TGAS_STATE,wssten);
#if defined(DEBUG_W_SPEED)
	        if (debug_w_speed == YES)
	        {
	            (void) printf("Interpolated state, dn3 = %g\n",dn3);
	            verbose_print_state("st_l3",st_l3);
		    (void) printf("Calling neumann_moc()\n");
	        }
#endif /* defined(DEBUG_W_SPEED) */
	        (*opts->neumann_moc)(pt,Tsl,ul,cl,st_l3,side,
				     left,-cl*dt,nor,front);
	        set_state(ansl_2,GAS_STATE,left);

	        if (Params(Tsl)->avisc.heat_cond != 0.0)
	        {
	            float heat_cond = Params(Tsl)->avisc.heat_cond;
		    float Tw = temperature(Tsl);
		    float Gam = gruneisen_gamma(Tsl);
		    float dS = heat_cond * (temperature(st_l3)/Tw - 1.0);
		    Dens(ansl_1) *= exp(-Gam*Tw*dS/sqr(cl));
	        }
		interpolate_states(front,alpha,1.0-alpha,pt,
		                   ansl_1,pt,ansl_2,ansl);
		if (no_slip(hs))
		{
		    float alpha = 1.0 - adherence_coeff(hs);
		    alpha_state_velocity(alpha,ansl,dim);
		}
		zero_normal_velocity(ansl,nor,dim);
	    }
	    break;

	default:
	    screen("ERROR in bdry_npt_w_speed(), ");
	    print_wave_type("unknown wave_type = ",w_type,"\n",intfc);
	    clean_up(ERROR);
	}
	Left_w_speed("bdry_npt_w_speed",ansl,ansr,NULL,dim);
	return;
}		/*end bdry_npt_w_speed*/


LOCAL	void	apply_bc(
	Locstate   st,
	Wave       *wv,
	HYPER_SURF *hs)
{
	if (boundary_state_function(hs) ==
		constant_pressure_flow_through_boundary_state)
	{
            Locstate bst;
            int      st_type;
	    st_type = state_type(st);
	    bst = boundary_state(hs);
	    set_state(st,TGAS_STATE,st);
	    Dens(st) = Dens(bst);
	    Press(st) = pressure(bst);
	    set_state(st,st_type,st);
	}
	else if(boundary_state_function(hs) ==
		time_dep_pressure_flow_through_boundary_state)
	{
	    /* For time dependent pressure Dirichlet boundary */
            Locstate    bst;
            int         st_type;
            FD_DATA     *fd_data;
            float       tr;
            float       tp;
            float       ts;
            float       pr_a;
            float       pr_p;
            float       time;

            bst = boundary_state(hs);
            st_type = state_type(st);
            fd_data = (FD_DATA*)boundary_state_data(hs);
            tr = fd_data->tr;
            tp = fd_data->tp;
            ts = fd_data->ts;
            pr_a = fd_data->pr_a;
            pr_p = fd_data->pr_p;
            time = wv->time;

	    set_state(bst,TGAS_STATE,bst);
	    set_state(st,TGAS_STATE,st);

	    if (time < tr) Press(st) = (pr_p-pr_a)/tr*time + pr_a;
	    else if (time <= tr+tp) Press(st) = pr_p;
	    else if (time <= tr+tp+ts) Press(st) = -(pr_p-pr_a)/ts
	    			*(time-tr-tp) + pr_p;
	    else Press(st) = pr_a;

	    Dens(bst) = Dens(st) = density(st);
	    Press(st) = Press(bst);
	    set_state(st,st_type,st);
	}
}		/*end apply_bc*/

/*
*			include_source():
*
*	This function incorporates the source terms introduced by gravity
*	and 3-D cylindrical symmetry.  Operator splitting (in time) is
*	used.
*
*	Input:	pt      - starting location of point being updated
*               ans 	- state to be updated
*		vnor0   - normal component of velocity at start of step
*		dt	- time step size
*		nor	- unit normal to front
*		W       - front velocity
*		g       - gravity vector
*		gr      - computational grid
*		w_type  - wave type for hypersurface being propagated
*
*	Output:	ans	- updated state
*			  obstacle states are not updated.	
*
*	The operator split algorithm updates the equation
*
*	rho_t + alpha*rho*nor[0]*(u,nor)/r(t) = 0
*	u_t = (g,nor)*nor
*	S_t = 0
*
*	where nor is the normal vector to the surface, S is the entropy,
*	u is the vector velocity, rho is the mass density, and g is the body
*	source. The geometry factor, alpha,  is zero for rectangular geometry,
*	one for cylindrical, and two for spherical geometry.  The position
*	r moves with velocity W[0],  r(t) = r0 + W[0]*t.
*/		

/*ARGSUSED*/
EXPORT	void include_source(
	float		*pt,
	Locstate	ans,
	float           vnor0,
	float		dt,
	float		*nor,
	float		*W,
	float		*g,
	RECT_GRID	*gr,
	int             w_type)
{
	static bool	first = YES;
	static bool	is_grav;
	static bool	add_source_terms;
	float		g_factor;
	int		dim = gr->dim;
	int		stype;
	int		i;

#if defined(ROTATIONAL_SYMMETRY)

	static GEOMETRY	geom;
	static float	alpha;
	GEOM_SOURCE_METHOD geom_source_method = G3ptOpts.geom_source_method;

#endif /* defined(ROTATIONAL_SYMMETRY) */

	if (first == YES) 
	{
	    first = NO;
	    is_grav = is_gravity();
	    add_source_terms = is_grav;

#if defined(ROTATIONAL_SYMMETRY)
	    geom = Geometry(&alpha);
	    if (geom != RECTANGULAR)
	        add_source_terms = YES;
#endif /* defined(ROTATIONAL_SYMMETRY) */
	}

	if (!add_source_terms )
	    return;

	if (is_obstacle_state(ans))
	    return;

	stype = state_type(ans);
	set_state(ans,TGAS_STATE,ans);
	if ((is_grav) && (g != NULL))
	{
	    g_factor = scalar_product(nor,g,dim)*dt;
	    for (i = 0; i < dim; ++i)
	        Vel(ans)[i] += nor[i]*g_factor;
	}
#if defined(ROTATIONAL_SYMMETRY)
	if (geom != RECTANGULAR)
	{
	    float    rmin, r0, r1;
	    float    v[3];

	    rmin = pos_radius(0.0,gr);
	    r0 = pos_radius(pt[0],gr);
	    r1 = pos_radius(pt[0] + W[0]*dt,gr);

	    if ((fabs(r0) > rmin) && (fabs(r1) > rmin))
	    {
	        float rho, c2, pm, p;
		float rho_fac;
		float pa, pb, p0;
	        float vn;
		float rm = (geom == SPHERICAL) ?
	            2.0*(r0*r0+r0*r1+r1*r1)/(3.0*(r0+r1)) : 0.5*(r0+r1);
                int number_of_interations;
		const int MAX_NUM_ITERATIONS = 20;

		switch (geom_source_method)
		{
		case ANALYTIC_SOLUTION:
		    if (geom == SPHERICAL)
			rho_fac = (r0*r0)/(r1*r1);
	            else if (geom == CYLINDRICAL)
			rho_fac = r0/r1;
		    else
		    {
			rho_fac = 1.0;
			screen("ERROR in include_source(), unknown geometry\n");
			clean_up(ERROR);
		    }
		    if (!is_scalar_wave(w_type))
		    {
	                vn = nor[0]*0.5*(vnor0+scalar_product(Vel(ans),nor,dim));
		        rho_fac *= exp(alpha*(W[0]-vn)*dt/rm);
		    }
		    rho = Dens(ans)*rho_fac;
		    Press(ans) = pressure_rarefaction(rho,ans);
		    Dens(ans) = rho;
		    break;

		case  BACKWARD_EULER:
	            vn = nor[0]*0.5*(vnor0+scalar_product(Vel(ans),nor,dim));
                    pa = p0 = Press(ans);
                    number_of_interations = 0;
	            for (i = 0; i < dim; ++i)
		        v[i] = Vel(ans)[i];
                    do
                    {
                        pb = pa;
                        c2 = sound_speed_squared(ans);
                        pa = p0 - alpha*Dens(ans)*c2*vn*dt/rm;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
                        pa = max(Min_pressure(ans),pa);
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
                        state_on_adiabat_with_pr(ans,pa,ans,state_type(ans));
                    } while ((fabs(pb - pa) > EPSILON*pa) &&
                         (++number_of_interations < MAX_NUM_ITERATIONS));
	            for (i = 0; i < dim; ++i)
		        Vel(ans)[i] = v[i];
		    break;

		case MODIFIED_EULER:
		default:
	            vn = nor[0]*0.5*(vnor0+scalar_product(Vel(ans),nor,dim));
	            p = Press(ans);
	            c2 = sound_speed_squared(ans);
	            rho = Dens(ans);
	            pm = p - 0.5*alpha*rho*c2*vn*dt/r0;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
	            pm = max(Min_pressure(ans),pm);
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
	            Dens(ans) = dens_rarefaction(pm,ans);
	            Press(ans) = pm;
	            c2 = sound_speed_squared(ans);

	            p -= alpha*Dens(ans)*c2*vn*dt/rm;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
	            p = max(Min_pressure(ans),p);
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
	            Dens(ans) = dens_rarefaction(p,ans);
	            Press(ans) = p;
		    break;
		}
	    }
	}
#endif /* defined(ROTATIONAL_SYMMETRY) */

	set_state(ans,stype,ans);
}	/*end include_source*/

LOCAL	bool	strong_wave(
	Locstate	s0,
	float		pm,
	float		um)
{
	float	Mach_tol = G3ptOpts.Mach_tol;
	float	i0 = acoustic_impedance(s0);
	float   p0 = pressure(s0);
	float	idu = i0*fabs(um - vel(0,s0));
	float	dp = fabs(pm - p0);

	if (Mach_tol == HUGE_VAL)
	    return NO;
	
	if ((dp + idu) < EPSILON*p0) /*TOLERANCE*/
	    return NO;
	if (fabs(dp - idu) <= Mach_tol*max(idu,dp))
	    return NO;
	return YES;
}		/*end strong_wave*/

LOCAL	bool	strong_interaction(
	Locstate	sl,
	Locstate	sr,
	int		w_type,
	float		pml,
	float		pmr,
	float		uml,
	float		umr)
{
	float	rhol, rhor;
	float	A_tol = G3ptOpts.A_tol;

	if (debugging("no_si"))
	    return NO;

	if (is_forward_wave(w_type))
	{
	    if (strong_wave(sl,pml,uml))
	        return YES;
	    rhol = dens_Hugoniot(pml,sl);
	    rhor = dens_Hugoniot(pmr,sr);
	    if (fabs(rhol - rhor)/(rhol+rhor) > A_tol)
	    {
#if defined(DEBUG_W_SPEED)
	        if (debug_w_speed == YES)
	            (void) printf("strong density jump for forward wave "
				  "rhol = %g, rhor = %g, "
				  "A = %g, A_tol = %g\n",rhol,rhor,
				  (rhol - rhor)/(rhol+rhor),A_tol);
#endif /* defined(DEBUG_W_SPEED) */
	        return YES;
	    }
	    if (is_rarefaction_wave(w_type))
	    {
	        if (strong_wave(sr,pmr,umr))
	            return YES;
	    }
	}
	else if (is_scalar_wave(w_type))
	{
	    if ((strong_wave(sl,pml,uml)) || (strong_wave(sr,pmr,umr)))
	        return YES;
	}
	else if (is_backward_wave(w_type))
	{
	    if (strong_wave(sr,pmr,umr))
	        return YES;
	    rhol = dens_Hugoniot(pml,sl);
	    rhor = dens_Hugoniot(pmr,sr);
	    if (fabs(rhol - rhor)/(rhol+rhor) > A_tol)
	    {
#if defined(DEBUG_W_SPEED)
	        if (debug_w_speed == YES)
	            (void) printf("strong density jump for backward wave "
				  "rhol = %g, rhor = %g, "
				  "A = %g, A_tol = %g\n",rhol,rhor,
				  (rhol - rhor)/(rhol+rhor),A_tol);
#endif /* defined(DEBUG_W_SPEED) */
	        return YES;
	    }
	    if (is_rarefaction_wave(w_type))
	    {
	        if (strong_wave(sl,pml,uml))
	            return YES;
	    }
	}
	return NO;
}		/*end strong_interaction*/

/*
*			cheap method of characteristics
*
*	The following functions solve the Euler equations in characteristic
*	form.  The particular equations solved are
*
*		d P           d N
*              ----   - rho*c ---  = 0		where l- = N - c
*               d l-          d l-
*
*		d T
*              ----   = 0			where l0 = N
*	        d l0
*
*		d P          d rho
*              ----  -  c^2 ------   = 0	where l0 = N
*		d l0         d l0
*
*		d P           d N
*              ----   + rho*c ---  = 0		where l+ = N + c
*               d l+          d l+
*
*	Here P is the pressure, rho the density, and N and T are the
*	normal and tangential components of velocity respectively.
*/


/*
*			neumann_cheap_moc():
*
*	Performs a simple method of characteristic update of the
*	state near a neumann boundary.  The method is to use
*	a finite difference approximation to the characteristic
*	equations for the pressure and velocity.
*
*	Assumes that all of the states are in TGas format.
*/

/*ARGSUSED*/
LOCAL	void neumann_cheap_moc(
	float    *pt,		/* Wall point position           */
	Locstate sw,		/* State at the wall             */
	float    uw,		/* Wall velocity                 */
	float    cw,		/* Sound speed at the wall       */
	Locstate sfoot,		/* Incoming characteristic state */
	SIDE     side,		/* Interior side of the wall     */
	Locstate ans,		/* Output answer state           */
	float    dn,		/* Normal direction mesh spacing */
	float    *nor,		/* Wall normal                   */
	Front    *front)	/* Front structure               */
{
	RECT_GRID *gr = front->rect_grid;
	float     ufoot, p, rw, pw, du, vnor;
	float     dp, heat_cond;
	float     dS, Tw, Gam;
	float     time = front->time;
	float     dt = front->dt;
	float     g1[3], g;
	float     pt1[3];
	int       i, dim;

#if defined(ROTATIONAL_SYMMETRY)
	static	float	alpha;
	static	bool	first = YES;

	if (first == YES)
	{
	    first = NO;
	    alpha = rotational_symmetry();
	}
#endif /* defined(ROTATIONAL_SYMMETRY) */

	debug_print("w_speed","Entered neumann_cheap_moc()\n");

	dim = gr->dim;
	for (i = 0; i < dim; ++i)
	    pt1[i] = pt[i]+dn*nor[i];
	eval_gravity(pt1,time,g1);
	g = scalar_product(g1,nor,dim);
	ufoot = scalar_product(nor,Vel(sfoot),dim);
	rw = Dens(sw);
	pw = Press(sw);
	du = uw - ufoot;

	    /* source term correction to MOC: */

	if (is_gravity() == YES)
	    du -= g*dt;
	if (side == NEGATIVE_SIDE)
	    du = -du;

#if defined(ROTATIONAL_SYMMETRY)
	if (alpha > 0.0)
	{
	    float rmin = pos_radius(0.0,gr);
	    float rad = pos_radius(pt1[0],gr);
	    if (fabs(rad) > fabs(rmin))
	        du -= alpha*nor[0]*cw*ufoot*dt/rad;
	}
#endif /* defined(ROTATIONAL_SYMMETRY) */

	Press(ans) = p = Press(sfoot) + rw * cw * du;
	heat_cond = Params(sw)->avisc.heat_cond;
	if (heat_cond != 0.0)
	{
	    Tw = temperature(sw);
	    Gam = gruneisen_gamma(sw);
	    dS = heat_cond * (temperature(sfoot)/Tw - 1.0);
	    dp = p - pw - rw*Tw*Gam*dS;
	}
	else
	    dp = p - pw;
	Dens(ans) = rw + dp / sqr(cw);
	vnor = scalar_product(nor,Vel(sw),dim);
	for (i = 0; i < dim; ++i)
	    Vel(ans)[i] = Vel(sw)[i] - nor[i] * vnor;
	Set_params(ans,sw);
	set_type_of_state(ans,TGAS_STATE);
	debug_print("w_speed","Left neumann_cheap_moc()\n");
}	/*end neumann_cheap_moc*/

LOCAL	void	wspeed_neumann_riem_inv_moc(
	float    *pt,
	Locstate sw,
	float	 uw,
	float	 cw,
	Locstate sfoot,
	SIDE     side,
	Locstate ans,
	float    dn,
	float	 *nor,
	Front    *fr)
{
	debug_print("w_speed","Entered wspeed_neumann_riem_inv_moc()\n");
	neumann_riem_inv_moc(pt,sw,uw,cw,sfoot,side,ans,dn,nor,fr);
	debug_print("w_speed","Left wspeed_neumann_riem_inv_moc()\n");
}	/*end wspeed_neumann_riem_inv_moc*/




/*
*		shock_ahead_state_cheap_moc():
*
*	Performs a simple method of characteristics update of the state
*	ahead of a shock wave, by using a simple finite difference version
*	of the characteristic equations for the pressure and velocity.
*
*		Assumes that all state are in TGas format.
*	
*/

LOCAL	void shock_ahead_state_cheap_moc(
	float     *pt,
	Locstate  st0,
	Locstate  st1,
	Locstate  st2,
	Locstate  st3,
	Locstate  ans,
	float     dn,
	float     f1,
	float     f2,
	float     f3,
	float     *nor,
	float     *W,
	int       add_source,
	float     dt,
	Front     *front)
{
	RECT_GRID *gr = front->rect_grid;
	float	  time = front->time;
	float	  u1, u2, u3;
	float	  u, vtan[MAXD];
	float	  r2, c2, p1, p2, p3;
	float	  p;
	float     g, pt2[3];
	int	  i, dim;
#if defined(ROTATIONAL_SYMMETRY)
	float	  alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */

#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES)
	    (void) printf("Entered shock_ahead_state_cheap_moc()\n");
#endif /* defined(DEBUG_W_SPEED) */

	dim = Params(st1)->dim;
	u1 = u2 = u3 = 0.0;
	for (i = 0; i < dim; ++i)
	{
	    u1 += nor[i]*Vel(st1)[i];
	    u2 += nor[i]*Vel(st2)[i];
	    u3 += nor[i]*Vel(st3)[i];
	    pt2[i] = pt[i] + f2*dn*nor[i];
	}
	if (add_source)
	{
	    g = scalar_product(gravity(pt2,time),nor,dim);
	}
	else
	    g = 0.0;
#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES)
	    (void) printf("u1 = %g, u2 = %g, u3 = %g\n",u1,u2,u3);
#endif /* defined(DEBUG_W_SPEED) */
	p1 = Press(st1);
	r2 = Dens(st2);
	p2 = Press(st2);
	c2 = sound_speed(st2);
	p3 = Press(st3);
	Set_params(ans,st1);
	set_type_of_state(ans,TGAS_STATE);
	p = 0.5 * ( p3 + p1  + r2*c2*(u3 - u1) );

#if defined(ROTATIONAL_SYMMETRY)
	if (add_source && alpha > 0.0)
	{
	    float rmin = pos_radius(0.0,gr);
	    float rad = pos_radius(pt[0],gr);
	    if (fabs(rad) > fabs(rmin))
	        p -= alpha*nor[0]*r2*sqr(c2)*u2*dt/rad;
	}
#endif /* defined(ROTATIONAL_SYMMETRY) */

	Press(ans) = p;
	Dens(ans) = r2 + (p - p2) / sqr(c2);
	u = 0.5 * ( (u3 + u1) + (p3 - p1) / (r2 * c2));
	if (add_source && (is_gravity() == YES))
	    u += g*dt;
	for (i = 0; i < dim; ++i)
	{
	    vtan[i] = Vel(st2)[i] - nor[i]*u2;
	    /*Vel(ans)[i] = vtan[i]*r2/Dens(ans) + nor[i]*u;*/
	    Vel(ans)[i] = vtan[i] + nor[i]*u;
	}

#if !defined(UNRESTRICTED_THERMODYNAMICS)
	if (Dens(ans) < Vacuum_dens(ans))
	{
	    shock_ahead_state_riem_inv_moc(pt,st0,st1,st2,st3,ans,dn,f1,f2,f3,
					   nor,W,add_source,dt,front);
	}
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES)
	{
	    verbose_print_state("Answer from shock_ahead_state_cheap_moc()",
	                        ans);
	    (void) printf("Left shock_ahead_state_cheap_moc()\n");
	}
#endif /* defined(DEBUG_W_SPEED) */

}	/*end shock_ahead_state_cheap_moc*/

LOCAL	void wspeed_shock_ahead_state_riem_inv_moc(
	float     *pt,
	Locstate  st0,
	Locstate  st1,
	Locstate  st2,
	Locstate  st3,
	Locstate  ans,
	float     dn,
	float     f1,
	float     f2,
	float     f3,
	float     *nor,
	float     *W,
	int       add_source,
	float     dt,
	Front     *front)
{
	shock_ahead_state_riem_inv_moc(pt,st0,st1,st2,st3,ans,dn,f1,f2,f3,
				       nor,W,add_source,dt,front);
}	/*end wspeed_shock_ahead_state_riem_inv_moc*/


LOCAL	void	contact_riemann(
	Locstate	left,
	Locstate	Tsl,
	Locstate	right,
	Locstate	Tsr,
	Locstate	st_l3,
	float           dnl,
	Locstate	st_r1,
	float           dnr,
	Locstate	ansl,
	Locstate	ansr,
	WSSten          *sten,
	float		*W,
	NptWSpeedOpts	*opts,
	float		*grav)
{
	float	*pt = sten->coords;
	float	V[3];
	float   pjump = sten->pjump;
	float	*nor = sten->nor;
	float	dt = sten->dt;
	float   vnor0;
	int	w_type = sten->w_type;
	Front	*front = sten->front;
	int	i, dim = front->rect_grid->dim;

	contact_filter_outgoing_wave(right,Tsr,st_r1,dnr,
		                     sten,POSITIVE_SIDE,opts);
#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES) 
	{
	    verbose_print_state("st_r1 after separation of back "
	                        "mode at foot of characteristic",st_r1);
	}
#endif /* defined(DEBUG_W_SPEED) */

	contact_filter_outgoing_wave(left,Tsl,st_l3,dnl,sten,NEGATIVE_SIDE,opts);

#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES) 
	{
	    verbose_print_state("st_l3 after separation of back "
	                        "mode at foot of characteristic",st_l3);
	}
#endif /* defined(DEBUG_W_SPEED) */

	/* Solve Riemann problem to find left and right states */

	set_state(st_r1,GAS_STATE,st_r1);
	set_state(st_l3,GAS_STATE,st_l3);
	w_speed(pt,st_l3,st_r1,ansl,ansr,V,pjump,nor,w_type,front);

#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES) 
	{
	    (void) printf("States after w_speed\n");
	    verbose_print_state("st_l3",st_l3);
	    verbose_print_state("st_r1",st_r1);
	    verbose_print_state("ansl ",ansl);
	    verbose_print_state("ansr ",ansr);
	}
#endif /* defined(DEBUG_W_SPEED) */

	if (is_gravity() == YES)
	{
	    float g = scalar_product(grav,nor,dim);

	    /*
	    *  source terms:  include_source() takes care of the left
	    *  and right states, but we also need to accelerate the
	    *  contact if there's an external force (which for now is
	    *  only gravity).
	    */

	    for (i = 0; i < dim; ++i)
	        V[i] += nor[i] * g * dt;
	}

	/* Use centered in time difference for wave speed */

	vnor0 = scalar_product(W,nor,dim);
	for (i = 0; i < dim; ++i)
	    W[i] = 0.5*(W[i] + V[i]);

	include_source(pt,ansl,vnor0,dt,nor,W,grav,front->rect_grid,w_type);
	include_source(pt,ansr,vnor0,dt,nor,W,grav,front->rect_grid,w_type);
}		/*end contact_riemann*/

#if DONT_COMPILE
LOCAL	void	filter_wave(Locstate,Locstate,Locstate,float*,SIDE);
LOCAL	void	filter_wave(
	Locstate	sl,
	Locstate	sr,
	Locstate	sfilter,
	float		*nor,
	SIDE		side)
{
	RIEMANN_SOLVER_WAVE_TYPE l_wave, r_wave, wave;
	float	 *vahead, vl[3], vr[3];
	float	 pl, pr, ul, ur, ml, mr, p, u, m;
	int	 state_type = state_type(sfilter);
	int	 i, dim = Params(sl)->dim;
	int      family;
	Locstate ahead;
	static	Locstate	left = NULL, right = NULL;

	if (left == NULL)
	{
	    (*Params(sl)->_alloc_state)(&left,max(sizeof(VGas),
	                                Params(sl)->sizest));
	    (*Params(sr)->_alloc_state)(&right,max(sizeof(VGas),
	                                Params(sr)->sizest));
	}
	set_state_for_find_mid_state(left,sl);
	set_state_for_find_mid_state(right,sr);
	Vel(left)[0] = scalar_product(VelocityVector(left,vl),nor,dim);
	Vel(right)[0] = scalar_product(VelocityVector(right,vr),nor,dim);

	if (find_mid_state(left,right,0.0,&pl,&pr,&ul,&ur,&ml,&mr,
	                   &l_wave,&r_wave) != FUNCTION_SUCCEEDED)
	{
	    (void) printf("WARNING in filter_wave(), "
	                  "find_mid_state() did not converge for ");
	                  "contact wave RP\n");
	}
	if (side == POSITIVE_SIDE)
	{
	    ahead = right;
	    p = pr;
	    u = ur;
	    m = mr;
	    wave = r_wave;
	    vahead = vr;
	    family = RIGHT_FAMILY;
	}
	else if (side == NEGATIVE_SIDE)
	{
	    ahead = left;
	    p = pl;
	    u = ul;
	    m = ml;
	    wave = l_wave;
	    vahead = vl;
	    family = LEFT_FAMILY;
	}
	else
	{
	    screen("ERROR in filter_wave, unknown side %d\n",side);
	    clean_up(ERROR);
	}
	midstate(ahead,sfilter,m,u,p,TGAS_STATE,wave,family);
	for (i = 0; i < dim; ++i)
	    Vel(sfilter)[i] = vahead[i] + (u - Vel(ahead)[0])*nor[i];
	set_state(sfilter,state_type,sfilter);
}		/*end filter_wave*/
#endif /* DONT_COMPILE */


/*ARGSUSED*/
LOCAL	void	contact_filter_outgoing_wave(
	Locstate	scontact,
	Locstate	Tscontact,
	Locstate	sfilter,
	float           dsfilter,
	WSSten		*sten,
	SIDE		side,
	NptWSpeedOpts	*opts)
{
	RIEMANN_SOLVER_WAVE_TYPE l_wave, r_wave, wave;
	Locstate	sl, sr;
	float	*nor = sten->nor;
	float	pl, pr, ul, ur, ml, mr, p, u, m;
	float	r;
	float	rc, uc = Vel(scontact)[0];
	float	sgn;
	float   vnor_sfilter, vtan[3];
	Front   *front = sten->front;
	int	dim = front->rect_grid->dim;
	int	i;
	static	Locstate tmpst = NULL;
#if defined(COMBUSTION_CODE)
	int	family;
#endif /* defined(COMBUSTION_CODE) */

	if (tmpst == NULL)
	    alloc_state(front->interf,&tmpst,front->sizest);

	set_state(sfilter,TGAS_STATE,sfilter);
	vnor_sfilter = scalar_product(Vel(sfilter),nor,dim);
	for (i = 0; i < dim; ++i)
	    vtan[i] = Vel(Tscontact)[i] - nor[i]*uc;
	if (opts->_scalar_filter_outgoing_waves_at_contact == NO)
	{
	    for (i = 0; i < dim; ++i)
	        Vel(sfilter)[i] = vtan[i] + vnor_sfilter*nor[i];
	    return;
	}

	rc = Dens(scontact);

	/* Load normal component of velociy of sfilter in FRONT local coords */
	/* The other components are not needed here. */
	Vel(sfilter)[0] = vnor_sfilter;

	if (side == POSITIVE_SIDE)
	{
	    sl = scontact;
	    sr = sfilter;
	    sgn = -1.0;
#if defined(COMBUSTION_CODE)
	    family = LEFT_FAMILY;
#endif /* defined(COMBUSTION_CODE) */
	}
	else
	{
	    sl = sfilter;
	    sr = scontact;
	    sgn = 1.0;
#if defined(COMBUSTION_CODE)
	    family = RIGHT_FAMILY;
#endif /* defined(COMBUSTION_CODE) */
	}

	/* Separate out back mode at foot of characteristic */
	/* by solving Riemann problem between scontact, sfilter */

	set_state_for_find_mid_state(sfilter,sfilter);

	if (!find_mid_state(sl,sr,0.0,&pl,&pr,&ul,&ur,&ml,&mr,&l_wave,&r_wave))
	{
	    (void) printf("WARNING in contact_filter_outgoing_wave(), "
	                  "find_mid_state() did not converge for "
	                  "contact wave RP\n");
	}

	if (side == POSITIVE_SIDE)
	{
	    p = pl;
	    u = ul;
	    m = ml;
	    wave = l_wave;
	}
	else
	{
	    p = pr;
	    u = ur;
	    m = -mr;
	    wave = r_wave;
	}

	switch (wave) 
	{
	case SHOCK:
	    r = m/(u - uc + m/rc);
	    if (invalid_shock(pressure(Tscontact),Dens(Tscontact),p,r))
		r = dens_Hugoniot(p,Tscontact);
	    Set_params(sfilter,Tscontact);
	    break;

	case RAREFACTION:
	    r = dens_rarefaction(p,Tscontact);
	    Set_params(sfilter,Tscontact);
	    if (pressure_is_near_vacuum(p,Tscontact))
	        u = uc + sgn*riemann_wave_curve(scontact,p);
	    break;

#if defined(COMBUSTION_CODE)
	case STRONG_DET:
	    Set_other_params(sfilter,Tscontact);
	    r = m/(u - uc + m/rc);
	    break;

	case CJ_DET:
	    CJ_det(tmpst,TGAS_STATE,scontact,family);
	    Set_other_params(sfilter,Tscontact);
	    r = Dens(tmpst);
	    u = Vel(tmpst)[0];
	    break;
#endif /* defined(COMBUSTION_CODE) */

	default:
	    screen("ERROR in contact_filter_outgoing_wave(), "
	           "unknown wave %d\n",wave);
	    clean_up(ERROR);
	    break;
	}

	Dens(sfilter) = r;	Press(sfilter) = p;
	set_type_of_state(sfilter,TGAS_STATE);
	for (i = 0; i < dim; ++i)
	    Vel(sfilter)[i] = vtan[i] + u*nor[i];

#if defined(COMBUSTION_CODE)
	if (Composition_type(scontact) == ZND)
	    React(sfilter) = React(Tscontact); 
#endif /* defined(COMBUSTION_CODE) */
}		/*end contact_filter_outgoing_wave*/

/*
*		contact_cheap_moc():
*
*	Updates the states on a contact discontinuity using
*	a simple finite difference approximation to the
*	characteristic equations for the pressure and normal 
*	velocity.
*
*	Assumes the input states are in TGas format.
*/

LOCAL	void contact_cheap_moc(
	float	  *pt,
	Locstate  sl,
	Locstate  sr,
	Locstate  sl3,
	Locstate  sr1,
	Locstate  ansl,
	Locstate  ansr,
	float	  cl,
	float	  cr,
	float	  pjump,
	float	  dnl,
	float     dnr,
	float	  *nor,
	float	  *W,
	Front     *front)
{
	RECT_GRID *gr = front->rect_grid;
	float     u1, u3, p1, p3;		/* 1, 3 state nor vel, press */
	float     pansl, pansr;		/* answer press */
	float     uansl, uansr, u;	/* answer normal velocity */
	float     ul, ur;
	float     rr, pr, rl, pl;		/* left, right density, press */
	float     den, rcr, rcl;
	float     pt3[3], pt1[3];
	float     dgdt, gv3[3], gv1[3], g1dt, g3dt;
	float     dt = front->dt;
	float     time = front->time;
	float     pfac = 0.0, ulfac = 0.0, urfac = 0.0, ufac = 0.0;
	int       i, dim = gr->dim;

#if defined(ROTATIONAL_SYMMETRY)
	float	alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */

#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES)
	    (void) printf("Entered contact_cheap_moc()\n");
#endif /* defined(DEBUG_W_SPEED) */

	rr = Dens(sr);		pr = Press(sr);
	rl = Dens(sl);		pl = Press(sl);
	p1 = Press(sr1);	p3 = Press(sl3);
	u1 = u3 = ul = ur = 0.0;
	for (i = 0; i < dim; ++i)
	{
	    u1 += nor[i] * Vel(sr1)[i];
	    u3 += nor[i] * Vel(sl3)[i];
	    ul += nor[i] * Vel(sl)[i];
	    ur += nor[i] * Vel(sr)[i];
	    pt3[i] = pt[i] + dnl*nor[i];
	    pt1[i] = pt[i] + dnr*nor[i];
	}
	eval_gravity(pt3,time,gv3);
	eval_gravity(pt1,time,gv1);
	g3dt = scalar_product(gv3,nor,dim)*dt;
	g1dt = scalar_product(gv1,nor,dim)*dt;
	dgdt = g1dt - g3dt;
	/*ur = ul = 0.5 * (ul + ur);*/

	rcr = rr*cr;	rcl = rl*cl;	den = rcr + rcl;
#if defined(ROTATIONAL_SYMMETRY)
	if (alpha > 0.0) 
	{
	    float rmin = pos_radius(0.0,gr);
	    float radr = pos_radius(pt1[0],gr);
	    float radl = pos_radius(pt3[0],gr);
	    if ((fabs(radr) > fabs(rmin)) && (fabs(radl) > fabs(rmin)))
	    {
	        pfac = -nor[0]*dt*rcl*rcr*alpha*(u1*cr/radr + u3*cl/radl)/den;
		ulfac = (nor[0]*dt*alpha*cl*rcl*u3/radl)/den;
		urfac = (nor[0]*dt*alpha*cr*rcr*u1/radr)/den;
		ufac = urfac - ulfac;
	    }
	}
#endif /* defined(ROTATIONAL_SYMMETRY) */

	pansr = (rcr*p3+rcl*p1 + rcl*rcr*(u3-u1-dgdt) - rcr*pjump)/den + pfac;
	pansl = (rcr*p3+rcl*p1 + rcl*rcr*(u3-u1-dgdt) + rcl*pjump)/den + pfac;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
	pansr = max(Min_pressure(sr),pansr);
	pansl = max(Min_pressure(sl),pansl);
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
	u = (rcl*(u3+g3dt) + rcr*(u1+g1dt) + p3 - p1 - pjump)/den + ufac;

	uansl = (pressure_is_near_vacuum(pansl,sl)) ? u3+g3dt+p3/rcl-ulfac : u;
	uansr = (pressure_is_near_vacuum(pansr,sr)) ? u1+g1dt-p1/rcr+urfac : u;

	for (i = 0; i < dim; ++i)
	{
	    Vel(ansl)[i] = Vel(sl)[i]  + (uansl - ul) * nor[i];
	    Vel(ansr)[i] = Vel(sr)[i]  + (uansr - ur) * nor[i];
	}
	Dens(ansr) = rr + (pansr - pr) / (cr * cr);
	Press(ansr) = pansr;
	Set_params(ansr,sr);
	set_type_of_state(ansr,TGAS_STATE);

	set_state(ansr,GAS_STATE,ansr);

	Dens(ansl) = rl + (pansl - pl) / (cl * cl);
	Press(ansl) = pansl;
	Set_params(ansl,sl);
	set_type_of_state(ansl,TGAS_STATE);

	set_state(ansl,GAS_STATE,ansl);

	    /* Use centered in time difference for wave speed */

	for (i = 0; i < dim; ++i)
	    W[i] = 0.5*(W[i] + nor[i]*u);

#if defined(DEBUG_W_SPEED)
	if (debug_w_speed == YES)
	{
	    verbose_print_state("Answer right from contact_cheap_moc()",ansr);
	    verbose_print_state("Answer left from contact_cheap_moc()",ansl);
	    (void) printf("Left contact_cheap_moc()\n");
	}
#endif /* defined(DEBUG_W_SPEED) */
}	/*end contact_cheap_moc*/

/*
*			contact_moc_plus_rh():
*
*	The states on opposite sides of a contact discontinuity
*	are updated using the method of characteristics and
*	the Rankine-Hugoniot relations across the contact.
*
*	All input states are assumed to be in TGas format.
*
*	The answer states ansl and ansr are returned in Gas format.
*
*/

typedef struct {
	Locstate st3l, st1r;
	Locstate st2l, st2r;
	Locstate ansl, ansr;
	float rc3l, rc1r;
	float u3l, u1r;
	float p3l, p1r;
	float pjump;
	float p_min;
#if defined(ROTATIONAL_SYMMETRY)
	float alpha;
	float rfac;
#endif /* defined(ROTATIONAL_SYMMETRY) */
} C_MPRH;


LOCAL void contact_moc_plus_rh(
	float	 *pt,
	float	 pm,
	Locstate st2l,
	Locstate st2r,
	Locstate st3l,
	Locstate st1r,
	Locstate ansl,
	Locstate ansr,
	float	 pjump,
	float	 dnl,
	float    dnr,
	float	 *nor,
	float	 *W,
	Front	*front)
{
	RECT_GRID *gr = front->rect_grid;
	const float meps = MACH_EPS;/*TOLERANCE*/
	float     time = front->time;
	float     dt = front->dt;
	float     pt1[3], pt3[3], pta[3];
	float     g1, g3, gv1[3], gv3[3], gva[3];
	float	  u1r, u2r, u2l, u3l;
	float	  ul, ur;
	float	  pi, u;
	float	  x0, a, b;
	float	  delta, epsilon;
	float	  p_min = max(Min_pressure(st3l),Min_pressure(st1r));
	int	  i, dim = gr->dim;
	static const int mnth = 10; /*TOLERANCE*/
	C_MPRH	  Cmprh;
#if defined(ROTATIONAL_SYMMETRY)
	float     alpha = rotational_symmetry();
	Cmprh.alpha = alpha;
#endif /* defined(ROTATIONAL_SYMMETRY) */

#if defined(DEBUG_W_SPEED)
	debug_print("cmph","Entered contact_moc_plus_rh()\n");
#endif /* defined(DEBUG_W_SPEED) */

	Cmprh.pjump = pjump;
	Cmprh.p_min = p_min;

	Cmprh.ansr = ansr;
	Cmprh.st2r = st2r;
	Cmprh.st1r = st1r;
	Cmprh.rc1r = acoustic_impedance(st1r);
	Cmprh.p1r = pressure(st1r);

	Cmprh.ansl = ansl;
	Cmprh.st2l = st2l;
	Cmprh.st3l = st3l;
	Cmprh.rc3l = acoustic_impedance(st3l);
	Cmprh.p3l = pressure(st3l);

	u1r = u2r = u2l = u3l = 0.0;
	for (i = 0; i < dim; ++i)
	{
	    u1r += nor[i] * Vel(st1r)[i];
	    u2r += nor[i] * Vel(st2r)[i];
	    u2l += nor[i] * Vel(st2l)[i];
	    u3l += nor[i] * Vel(st3l)[i];
	    pt1[i] = pt[i] + dnr*nor[i];
	    pt3[i] = pt[i] + dnl*nor[i];
	    pta[i] = pt[i] + W[i]*dt;
	}
	eval_gravity(pt3,time,gv3);
	eval_gravity(pt1,time,gv1);
	eval_gravity(pta,time+dt,gva);
	for (g1 = 0.0, g3 = 0.0, i = 0; i < dim; ++i)
	{
	    g1 += 0.5*(gva[i]+gv1[i])*nor[i];
	    g3 += 0.5*(gva[i]+gv3[i])*nor[i];
	}
	u1r += g1*dt;
	u3l += g3*dt;

#if defined(DEBUG_W_SPEED)
	if (debugging("cmph"))
	{
	    (void) printf("Input data into contact_moc_plus_rh()\n");
	    verbose_print_state("st3l",st3l);
	    verbose_print_state("st2l",st2l);
	    verbose_print_state("st2r",st2r);
	    verbose_print_state("st1r",st1r);
	    (void) printf("pm = %g, pjump = %g\n",pm,pjump);
	    (void) printf("dnl = %g, dnr = %g, dt = %g\n",dnl,dnr,dt);
	    (void) printf("g1 = %g, g3 = %g\n",g1,g3);
	    print_general_vector("nor = ",nor,dim,"\n");
	}
#endif /* defined(DEBUG_W_SPEED) */


#if defined(ROTATIONAL_SYMMETRY)
	Cmprh.rfac = 0.0;
	if (alpha > 0.0)
	{
	    float c3l, c1r;
	    float rmin, rad, radl, radr;

	    rmin = pos_radius(0.0,gr);
	    radl = pos_radius(pt3[0],gr);
	    radr = pos_radius(pt1[0],gr);
	    rad = pos_radius(pta[0],gr);

	    c3l = sound_speed(st3l);
	    c1r = sound_speed(st1r);

	    rmin = fabs(rmin);
	    if ((fabs(rad) > rmin) && (fabs(radl) > rmin) && 
	    	(fabs(radr) > rmin))
	    {
	        u3l *= (1.0 - 0.5*nor[0]*dt*c3l*alpha/radl );
	        u1r *= (1.0 + 0.5*nor[0]*dt*c1r*alpha/radr );
	        Cmprh.rfac = 0.5*nor[0]*dt*alpha/rad;
	    }
	}
#endif /* defined(ROTATIONAL_SYMMETRY) */
	Cmprh.u1r = u1r;
	Cmprh.u3l = u3l;

	x0 = pm;
	delta = max(meps, x0*EPS);
	(void) fC(x0, &epsilon, (POINTER) &Cmprh);
	epsilon = fabs(epsilon)*EPS;
	epsilon = max(epsilon, meps);
	a = 0.5*x0;/*TOLERANCE*/
	b = 1.5*x0;/*TOLERANCE*/
#if defined(DEBUG_W_SPEED)
	if (debugging("cmph"))
	    (void) printf("Initial search interval = (%g, %g)\n",a,b);
#endif /* defined(DEBUG_W_SPEED) */
	if (find_root(fC,(POINTER) &Cmprh,0.0,&pi,a,b,epsilon,delta) ==
	                                        FUNCTION_FAILED)
	{
	    float du;
	    /* Check for vacuum production */

#if defined(DEBUG_W_SPEED)
	    if (debugging("cmph"))
	        (void) printf("First find_root() failed\n");
#endif /* defined(DEBUG_W_SPEED) */

	    (void) fC(p_min,&du,(POINTER) &Cmprh);

	    a = p_min;
	    b = 10.0*max(Cmprh.p3l,Cmprh.p1r);
	    if (du <= 0.0)
	    {
	        ul = vel(0,ansl);
	        ur = vel(0,ansr);
	        u = (Dens(ansr) > Dens(ansl)) ? ur : ul;
	    }
	    else if ( (find_root(fC,(POINTER) &Cmprh,0.0,&pi,
	                         a,b,epsilon,delta) == FUNCTION_SUCCEEDED)
	        ||
	              (search_harder_for_root(fC,(POINTER) &Cmprh,0.0,&pi,a,b,
					      &a,&b,p_min,HUGE_VAL,mnth,epsilon,
					      delta) == FUNCTION_SUCCEEDED))
	    {
#if defined(DEBUG_W_SPEED)
	        if (debugging("cmph"))
	            (void) printf("Second find_root() succeeded\n");
#endif /* defined(DEBUG_W_SPEED) */
	        ul = vel(0,ansl);
	        ur = vel(0,ansr);
	        if (pressure(ansl) <= p_min || pressure(ansr) <= p_min)
	        {
	            u = (Dens(ansr) > Dens(ansl)) ? ur : ul;
	        }
	        else
	        {
	            u = ul = ur = 0.5*(ul + ur);
	        }
	    }
	    else
	    {
	        ul = vel(0,ansl);
	        ur = vel(0,ansr);
	        u = ul = ur = 0.5*(ul + ur);
	        if (debugging("cmph"))
	        {
	            (void) printf("WARNING in contact_moc_plus_rh(), "
	                          "Failure in fC() for contact\n");
	            (void) printf("x0 = %g, a = %g, b = %g\n",x0,a,b);
	            (void) fC(pi,&du,(POINTER) &Cmprh);
	            (void) printf("pi = %g, du(pi) = %g\n",pi,du);
	            (void) printf("Velocities at vacuum, ");
	            (void) printf("ul = %g, ur = %g, ur - ul = %g\n",
	                          ul,ur,ur-ul);
	            print_function_values(fC,(POINTER) &Cmprh,0.0,a,b,100,
	                                  "fC",stdout);
	        }
	    }
	} 	
	else
	{
	    ul = vel(0,ansl);
	    ur = vel(0,ansr);
	    if (pressure(ansl) <= p_min || pressure(ansr) <= p_min)
	    {
	        u = (Dens(ansr) > Dens(ansl)) ? ur : ul;
	    }
	    else
	    {
	        u = ul = ur = 0.5*(ul + ur);
	    }
	}

	/* This assumes state is such as velocities are stored */
	for (i = 0; i < dim; ++i)
	{
	    Vel(ansl)[i] = Vel(st2l)[i] + (ul - u2l)*nor[i];
	    Vel(ansr)[i] = Vel(st2r)[i] + (ur - u2r)*nor[i];
	}
	set_state(ansl,GAS_STATE,ansl);
	set_state(ansr,GAS_STATE,ansr);

	/* Use centered in time difference for wave speed */

	for (i = 0; i < dim; ++i)
	    W[i] = 0.5*(W[i] + nor[i] * u);

#if defined(DEBUG_W_SPEED)
	if (debugging("cmph"))
	{
	    verbose_print_state("ansl ",ansl);
	    verbose_print_state("ansr ",ansr);
	    print_general_vector("W = ",W,dim,"\n");
	}
	debug_print("cmph","Left contact_moc_plus_rh()\n");
#endif /* defined(DEBUG_W_SPEED) */
}	/*end contact_moc_plus_rh*/

/* function for contact */

LOCAL bool fC(
	float		pr,
	float		*fans,
	POINTER		prm)
{
	C_MPRH		*cmprh = (C_MPRH *) prm;
	float		rcr, rcl;
	float		pl, ul, ur, kl, kr;
	Locstate	ansl = cmprh->ansl, ansr = cmprh->ansr;

	state_on_adiabat_with_pr(cmprh->st2r,pr,ansr,TGAS_STATE);
	rcr = 0.5 * (cmprh->rc1r + acoustic_impedance(ansr));

	pl = floor_pressure(pr + cmprh->pjump,cmprh->p_min);
	state_on_adiabat_with_pr(cmprh->st2l,pl,ansl,TGAS_STATE);
	rcl = 0.5 * (cmprh->rc3l + acoustic_impedance(ansl));
	
	kl = kr = 1.0;
#if defined(ROTATIONAL_SYMMETRY)
	if (cmprh->alpha > 0.0)
	{
	    float cl, cr;
	    float rfac = cmprh->rfac;

	    cl = sound_speed(ansl);
	    kl = 1.0/(1.0 + cl*rfac);
	    cr = sound_speed(ansr);
	    kr = 1.0/(1.0 - cr*rfac);
	}
#endif /* defined(ROTATIONAL_SYMMETRY) */

	Vel(ansl)[0] = ul = kl*(cmprh->u3l - (pl - cmprh->p3l)/rcl);
	Vel(ansr)[0] = ur = kr*(cmprh->u1r + (pr - cmprh->p1r)/rcr);

	*fans = ul - ur;

#if defined(DEBUG_W_SPEED)
	if (debugging("cmph"))
	{
	    (void) printf("In fC(), pl = %g, pr = %g, ",pl,pr);
	    (void) printf("ul = %g, ur = %g, ul - ur = %g\n",ul,ur,ul-ur);
	}
#endif /* defined(DEBUG_W_SPEED) */

	return FUNCTION_SUCCEEDED;
}	/*end fC*/


LOCAL	void	IncomingStatesAtContact(
	WSSten	      *sten,
	float	      cl,
	float	      cr,
	Locstate      st_l3,
	Locstate      st_r1,
	float	      *dnl,
	float         *dnr,
	NptWSpeedOpts *opts)
{
	float		*pt = sten->coords;
	Locstate	 sll =  sten->sl[1];
	Locstate	  sl =  sten->sl[0];
	Locstate	  sr =  sten->sr[0];
	Locstate	 srr =  sten->sr[1];
	float		dn = sten->dn;
	float		dt = sten->dt;
	float		dn1, dn3;
	Locstate	slll, srrr;

	dn1 = cr * dt / dn;
	*dnr = cr * dt;
	dn3 = 1.0 - cl * dt / dn;
	*dnl = -cl*dt;
	if (sten->nsts > 2)
	{
	    /* Find states at feet of characteristic */

	    slll = sten->sl[2];
	    srrr = sten->sr[2];
	    ws_interpolate(st_r1,dn1,POSITIVE_SIDE,TGAS_STATE,sten);
	    ws_interpolate(st_l3,dn3,NEGATIVE_SIDE,TGAS_STATE,sten);
	    if (opts->scalar_moc != RIEMANN)
	    {
	        if ((pressure_switch_on(sr,srr,srrr) == NO) ||
	            (pressure_switch_on(sl,sll,slll) == NO))
	            opts->scalar_moc = RIEMANN;
	    }
	}
	else
	{
	    static Locstate wk_slll = NULL, wk_srrr = NULL;
	    Front	    *front = sten->front;
	    Wave	    *wave = sten->wave;
	    float	    *crdsll = sten->lcrds[1];
	    float	    *crdsrr = sten->rcrds[1];
	    float           crdslll[3], crdsrrr[3];
	    float	    *nor = sten->nor;
	    COMPONENT	    pcomp = sten->pcomp;
	    COMPONENT	    ncomp = sten->ncomp;
	    HYPER_SURF	    *hs = sten->hs;
	    size_t	    sizest = front->sizest;
	    int             i, dim = front->rect_grid->dim;

	    if (wk_slll == NULL)
	    {
		scalar(&wk_slll,sizest);
		scalar(&wk_srrr,sizest);
	    }
	    slll = wk_slll;
	    srrr = wk_srrr;
	    for (i = 0; i < dim; ++i)
	    {
		crdslll[i] = pt[i] - nor[i] * 2.0 * dn;
		crdsrrr[i] = pt[i] + nor[i] * 2.0 * dn;
	    }
	    hyp_solution(crdslll,ncomp,hs,NEGATIVE_SIDE,front,wave,slll,sl);
	    hyp_solution(crdsrrr,pcomp,hs,POSITIVE_SIDE,front,wave,srrr,sr);
	    if (pressure_switch_on(sr,srr,srrr))
		interpolate_states(front,1.0-dn1,dn1,pt,sr,crdsrr,srr,st_r1);
	    else
	    {
	        copy_state(st_r1,srr);
		opts->scalar_moc = RIEMANN;
	    }
	    set_state(st_r1,TGAS_STATE,st_r1);
	    if (pressure_switch_on(sl,sll,slll))
		interpolate_states(front,1.0-dn3,dn3,crdsll,sll,pt,sl,st_l3);
	    else
	    {
		copy_state(st_l3,sll);
		opts->scalar_moc = RIEMANN;
	    }
	    set_state(st_l3,TGAS_STATE,st_l3);
	}
}		/*end IncomingStatesAtContact*/

LOCAL   bool pressure_switch_on(
	Locstate s1,
	Locstate s2,
	Locstate s3)
{
	float p1,p2,p3;
	p1 = pressure(s1);
	p2 = pressure(s2);
	p3 = pressure(s3);

	return (fabs(p2-p1) > fabs(p3-p2) || (p1-p2)*(p2-p3) < 0.0) ? NO : YES;
}		/* end pressure_switch_on */


LOCAL	float	wall_limiter(
	WSSten        *wssten,
	SIDE          side,
	NptWSpeedOpts *opts)
{
	float    alpha;
	float    *nor = wssten->nor;
	const float    eps = 10.0*MACH_EPS;
	float    wl = opts->Wall_limiter;
	int      dim = wssten->front->rect_grid->dim;
	int      i, j, nsts = wssten->nsts;
	Locstate *st = (side == POSITIVE_SIDE) ? wssten->sr : wssten->sl;
	static   float *p, *rho, *e, *c, **v, *vn, *s;
	static   int   N = 0;

	if (wl == HUGE_VAL)
	    alpha = 1.0;
	else if (wl <= 0.0)
	    alpha = 0.0;
	else
	{
	    if ((p == NULL) || (N < nsts))
	    {
	        if (p != NULL)
	            free_these(7,p,rho,e,c,v,vn,s);
	        N = nsts;
	        vector(&p,N,FLOAT);
	        vector(&rho,N,FLOAT);
	        vector(&e,N,FLOAT);
	        vector(&c,N,FLOAT);
	        matrix(&v,N,3,FLOAT);
	        vector(&vn,N,FLOAT);
	        vector(&s,N,FLOAT);
	    }
    
	    for (i = 0; i < nsts; ++i)
	    {
	        p[i] = pressure(st[i]);
	        rho[i] = Dens(st[i]);
	        e[i] = specific_internal_energy(st[i]);
	        c[i] = sound_speed(st[i]);
	        (void) VelocityVector(st[i],v[i]); 
	        s[i] = mag_vector(v[i],dim);
	        vn[i] = scalar_product(v[i],nor,dim);
	    }
	    alpha = 0.0;
	    for (i = 0; i < (nsts-1); ++i)
	    {
	        for (j = i+1; j < nsts; ++j)
	        {
	            float x;
    
		    x = fabs(p[j] - p[i])/((j-i)*(p[i] + eps));
		    alpha = max(alpha,x);
		    x = fabs(rho[j] - rho[i])/((j-i)*(rho[i] + eps));
		    alpha = max(alpha,x);
		    x = fabs(e[j] - e[i])/((j-i)*(e[i] + eps));
		    alpha = max(alpha,x);
		    x = fabs(vn[j] - vn[i])/((j-i)*(c[i] + eps));
		    alpha = max(alpha,x);
		    x = fabs(s[j] - s[i])/((j-i)*(c[i] + eps));
		    alpha = max(alpha,x);
		    x = vector_product(v[i],v[j],NULL,dim)/(s[i]*s[j] + eps);
		    alpha = max(alpha,x);
	        }
	    }
	    alpha *= wl;
	    alpha = min(alpha,1.0);
	}
	return alpha;
}		/*end wall_limiter */

LOCAL	bool invalid_shock(
	float pa,
	float rhoa,
	float pb,
	float rhob)
{
	return (((pb - pa)*(rhob - rhoa) < 0.0) || (rhob < 0.0)) ? YES : NO;
}

#if defined(COMBUSTION_CODE)

LOCAL	void CJ_state_behind_curved_wave(
	Locstate	ahead,
	Locstate	ans,
	float		pjump,
	float		*Wx,
	int		st_type_ans,
	int		family)
{
	float	    pCJ, uCJ, rhoCJ, speedCJ;
	float	    corr_fac=1.0;
	static const float BFF	= 0.3;	/* TOLERANCE */

	debug_print("w_speed","Entered CJ_state_behind_curved_wave()\n");
	speedCJ = CJ_det(ans,st_type_ans,ahead,family);
	pCJ = pressure(ans);
	rhoCJ = Dens(ans);
	uCJ = vel(0,ans);

	if ((Params(ans)->speed_corr * fabs(pjump)) > (BFF * speedCJ))
	{
	    corr_fac = BFF * speedCJ / (Params(ans)->speed_corr * pjump);
	}
	pjump *= corr_fac;
	pCJ -= Params(ans)->p_corr * pjump;
	uCJ -= Params(ans)->u_corr * pjump;
	rhoCJ -= Params(ans)->rho_corr * pjump;
	speedCJ -= Params(ans)->speed_corr * pjump;

	Dens(ans) = rhoCJ;
	Vel(ans)[0] = uCJ;
	Press(ans) = pCJ;
	set_type_of_state(ans,TGAS_STATE);

	set_state(ans,st_type_ans,ans);

	Set_other_params(ans,ahead);
	*Wx = speedCJ;
	debug_print("w_speed","Left CJ_state_behind_curved_wave()\n");
	return;
}	/*end CJ_state_behind_curved_wave*/
#endif /* defined(COMBUSTION_CODE) */
